/*
    w32loader
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "w32private.h"
#include "kernel32.h"


static LRESULT WINAPI
w32edit_DefWindowProc (HWND   hWnd,
                       UINT   Msg,
                       WPARAM wParam,
                       LPARAM lParam)
{
  GtkWidget *edit;
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (hWnd), "user_data");
  if (!wd)
    return 0;
  edit = GTK_IS_SCROLLED_WINDOW (hWnd) ? gtk_bin_get_child (GTK_BIN (hWnd))
                                       : hWnd;
  switch (Msg)
    {
      case EM_GETSEL:
        {
          gint st = 0, ed = 0;

          if (GTK_IS_TEXT_VIEW (edit))
            {
              GtkTextIter start, end;

              if (gtk_text_buffer_get_selection_bounds
                            (gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit)),
                                                                &start, &end))
                {
                  st = gtk_text_iter_get_offset (&start);
                  st = gtk_text_iter_get_offset (&end);
                }
            }
          else
            {
              gtk_editable_get_selection_bounds (GTK_EDITABLE (edit),
                                                                    &st, &ed);
            }
          if (wParam)
            *((LPDWORD)wParam) = st;
          if (lParam)
            *((LPDWORD)lParam) = ed;
          return MAKELRESULT (st, ed);
        }
      case EM_SETSEL:
        if (!GTK_IS_TEXT_VIEW (edit))
          gtk_editable_select_region (GTK_EDITABLE (edit), wParam, lParam);
        return 0;
      case EM_GETRECT:
        return 0;
      case EM_SETRECT:
        return 0;
      case EM_SETRECTNP:
        return 0;
      case EM_SCROLL:
        return FALSE;
      case EM_LINESCROLL:
        return GTK_IS_TEXT_VIEW (edit);
      case EM_SCROLLCARET:
        if (GTK_IS_TEXT_VIEW (edit))
          gtk_text_view_place_cursor_onscreen (GTK_TEXT_VIEW (edit));
        return 1;
      case EM_GETMODIFY:
        return GTK_IS_TEXT_VIEW (edit) ? gtk_text_buffer_get_modified
                    (gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit))) : TRUE;
      case EM_SETMODIFY:
        if (GTK_IS_TEXT_VIEW (edit))
          gtk_text_buffer_set_modified
                    (gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit)), wParam);
        return 0;
      case EM_GETLINECOUNT:
        return GTK_IS_TEXT_VIEW (edit) ? gtk_text_buffer_get_line_count
                        (gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit))) : 1;
      case EM_LINEINDEX:
        if (GTK_IS_TEXT_VIEW (edit))
          {
            GtkTextIter iter;

            gtk_text_buffer_get_iter_at_line (gtk_text_view_get_buffer
                                        (GTK_TEXT_VIEW (edit)), &iter, wParam);
            return gtk_text_iter_get_offset (&iter);
          }
        return 0;
      case EM_SETHANDLE:
        return 0;
      case EM_GETHANDLE:
        return 0;
      case EM_GETTHUMB:
        return 0;
      case EM_LINELENGTH:
        {
          gchar *utf8str;
          LRESULT lResult = 0;

          if (GTK_IS_TEXT_VIEW (edit))
            {
              GtkTextIter start, end;
              GtkTextBuffer *buffer;

              buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit));
              gtk_text_buffer_get_iter_at_line (buffer, &start, wParam);
              gtk_text_buffer_get_iter_at_line_offset (buffer, &end, wParam,
                                    gtk_text_iter_get_chars_in_line (&start));
              utf8str = gtk_text_buffer_get_text (buffer, &start, &end, TRUE);
            }
          else
            {
              utf8str = g_strdup (gtk_entry_get_text (GTK_ENTRY (edit)));
            }
          if (utf8str)
            {
              if (wd->widechar)
                {
                  gunichar2 *wc;

                  wc = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
                  lResult = lstrlenW (wc);
                  g_free (wc);
                }
              else
                {
                  gchar *mb;

                  mb = w32ldr_utf8_to_mb (utf8str);
                  lResult = g_strlen (mb);
                  g_free (mb);
                }
              g_free (utf8str);
            }
          return lResult;
        }
      case EM_REPLACESEL:
        {
          gchar *utf8str;

          utf8str = lParam ? wd->widechar
                           ? g_utf16_to_utf8 (GINT_TO_POINTER (lParam),
                                                        -1, NULL, NULL, NULL)
                           : w32ldr_utf8_from_mb (GINT_TO_POINTER (lParam))
                           : g_strdup ("");
          if (GTK_IS_TEXT_VIEW (edit))
            {
              GtkTextIter start, end;
              GtkTextBuffer *buffer;

              buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit));
              if (gtk_text_buffer_get_selection_bounds (buffer, &start, &end)){
                gtk_text_buffer_delete (buffer, &start, &end);}
              if (utf8str)
                gtk_text_buffer_insert_at_cursor (buffer, utf8str, -1);
            }
          else
            {
              gint pos;

              gtk_editable_delete_selection (GTK_EDITABLE (edit));
              pos = gtk_editable_get_position (GTK_EDITABLE (edit));
              if (utf8str)
                gtk_editable_insert_text (GTK_EDITABLE (edit),
                                            utf8str, g_strlen (utf8str), &pos);
            }
          g_free (utf8str);
        }
        return 0;
      case EM_GETLINE:
        {
          gchar *utf8str;
          LRESULT lResult = 0;

          if (GTK_IS_TEXT_VIEW (edit))
            {
              GtkTextIter start, end;
              GtkTextBuffer *buffer;

              buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit));
              gtk_text_buffer_get_iter_at_line (buffer, &start, wParam);
              gtk_text_buffer_get_iter_at_line_offset (buffer, &end, wParam,
                                    gtk_text_iter_get_chars_in_line (&start));
              utf8str = gtk_text_buffer_get_text (buffer, &start, &end, TRUE);
            }
          else
            {
              utf8str = g_strdup (gtk_entry_get_text (GTK_ENTRY (edit)));
            }
          if (utf8str)
            {
              if (wd->widechar)
                {
                  gunichar2 *wc;

                  wc = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
                  lResult = lstrlenW (wc);
                  if (*((LPWORD)lParam) < lResult)
                    lstrcpyW (GINT_TO_POINTER (lParam), wc);
                  else
                    lResult = 0;
                  g_free (wc);
                }
              else
                {
                  gchar *mb;

                  mb = w32ldr_utf8_to_mb (utf8str);
                  lResult = g_strlen (mb);
                  if (*((LPWORD)lParam) < lResult)
                    g_strcpy (GINT_TO_POINTER (lParam), mb);
                  else
                    lResult = 0;
                  g_free (mb);
                }
              g_free (utf8str);
            }
          return lResult;
        }
      case EM_CANUNDO:
        return FALSE;
      case EM_UNDO:
        return GTK_IS_ENTRY (edit);
      case EM_FMTLINES:
        return wParam;
      case EM_LINEFROMCHAR:
        if (GTK_IS_TEXT_VIEW (edit))
          {
            GtkTextIter iter;

            gtk_text_buffer_get_iter_at_offset (gtk_text_view_get_buffer
                                        (GTK_TEXT_VIEW (edit)), &iter, wParam);
            return gtk_text_iter_get_line (&iter);
          }
        return 0;
      case EM_SETTABSTOPS:
        return FALSE;
      case EM_SETPASSWORDCHAR:
        if (GTK_IS_ENTRY (edit))
          {
            gtk_entry_set_visibility (GTK_ENTRY (edit), wParam > 0);
            if (wParam > 0)
              gtk_entry_set_invisible_char (GTK_ENTRY (edit), wParam);
          }
        return 0;
      case EM_EMPTYUNDOBUFFER:
        return 0;
      case EM_GETFIRSTVISIBLELINE:
        if (GTK_IS_TEXT_VIEW (edit))
          {
            GtkTextIter iter;

            if (gtk_text_view_forward_display_line (GTK_TEXT_VIEW (edit),
                                                                        &iter))
              return gtk_text_iter_get_line (&iter);
          }
        return 0;
      case EM_SETREADONLY:
        if (wParam)
          wd->dwStyle |= ES_READONLY;
        else
          wd->dwStyle &= ~ES_READONLY;
        if (GTK_IS_TEXT_VIEW (edit))
          gtk_text_view_set_editable (GTK_TEXT_VIEW (edit),
                                                !(ES_READONLY & wd->dwStyle));
        else
          gtk_editable_set_editable (GTK_EDITABLE (edit),
                                                !(ES_READONLY & wd->dwStyle));
      case EM_SETWORDBREAKPROC:
        return 0;
      case EM_GETWORDBREAKPROC:
        return 0;
      case EM_GETPASSWORDCHAR:
        return GTK_IS_ENTRY (edit)
                    ? gtk_entry_get_invisible_char (GTK_ENTRY (edit)) : '*';
      case EM_SETMARGINS:
        if (GTK_IS_TEXT_VIEW (edit))
          {
            if (wParam & EC_USEFONTINFO)
              {
                gtk_text_view_set_left_margin (GTK_TEXT_VIEW (edit), 0);
                gtk_text_view_set_right_margin (GTK_TEXT_VIEW (edit), 0);
              }
            else
              {
                if (wParam & EC_LEFTMARGIN)
                  gtk_text_view_set_left_margin (GTK_TEXT_VIEW (edit),
                                                            LOWORD (lParam));
                if (wParam & EC_RIGHTMARGIN)
                  gtk_text_view_set_right_margin (GTK_TEXT_VIEW (edit),
                                                            HIWORD (lParam));
              }
          }
        return 0;
      case EM_GETMARGINS:
        return GTK_IS_TEXT_VIEW (edit) ? MAKELRESULT
                   (gtk_text_view_get_left_margin (GTK_TEXT_VIEW (edit)),
                    gtk_text_view_get_right_margin (GTK_TEXT_VIEW (edit))) : 0;
      case EM_SETLIMITTEXT:
        if (GTK_IS_ENTRY (edit))
          gtk_entry_set_max_length (GTK_ENTRY (edit),
                                            wParam > 0 ? wParam : G_MAXINT16);
        return 0;
      case EM_GETLIMITTEXT:
        return GTK_IS_ENTRY (edit)
                    ? gtk_entry_get_max_length (GTK_ENTRY (edit)) : G_MAXINT16;
      case EM_POSFROMCHAR:
        if (GTK_IS_TEXT_VIEW (edit))
          {
            gint x, y;
            GtkTextIter iter;

            gtk_text_buffer_get_iter_at_offset (gtk_text_view_get_buffer
                                        (GTK_TEXT_VIEW (edit)), &iter, wParam);
            gtk_text_view_buffer_to_window_coords (GTK_TEXT_VIEW (edit),
                                    GTK_TEXT_WINDOW_WIDGET,
                                    gtk_text_iter_get_line_offset (&iter),
                                    gtk_text_iter_get_line (&iter), &x, &y);
            return MAKELRESULT (x, y);
          }
        return -1;
      case EM_CHARFROMPOS:
        if (GTK_IS_TEXT_VIEW (edit))
          {
            gint x, y;
            GtkTextIter iter;

            gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (edit),
                                    GTK_TEXT_WINDOW_WIDGET,
                                    LOWORD (lParam), HIWORD (lParam), &x, &y);
            gtk_text_view_get_iter_at_location (GTK_TEXT_VIEW (edit), &iter,
                                            LOWORD (lParam), HIWORD (lParam));
            return MAKELRESULT (gtk_text_iter_get_line_offset (&iter),
                                gtk_text_iter_get_line (&iter));
          }
        return -1;
      case EM_SETIMESTATUS:
        return 0;
      case EM_GETIMESTATUS:
        return 0;
      case WM_GETTEXT:
        {
          gchar *utf8str;

          if (GTK_IS_TEXT_VIEW (edit))
            {
              GtkTextBuffer *buffer;
              GtkTextIter start, end;

              buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit));
              gtk_text_buffer_get_bounds (buffer, &start, &end);
              utf8str = gtk_text_buffer_get_text (buffer, &start, &end, TRUE);
            }
          else
            {
              utf8str = g_strdup (gtk_entry_get_text (GTK_ENTRY (edit)));
            }
          if (wParam > 0 && lParam && utf8str)
            {
              if (wd->widechar)
                {
                  gunichar2 *wc;

                  wc = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
                  g_free (utf8str);
                  lstrcpynW (GINT_TO_POINTER (lParam), wc, wParam);
                  g_free (wc);
                  return lstrlenW (GINT_TO_POINTER (lParam));
                }
              else
                {
                  gchar *mb;

                  mb = w32ldr_utf8_to_mb (utf8str);
                  g_free (utf8str);
                  g_strncpy (GINT_TO_POINTER (lParam), mb, wParam);
                  g_free (mb);
                  return g_strlen (GINT_TO_POINTER (lParam));
                }
            }
        }
        return 0;
      case WM_GETTEXTLENGTH:
        {
          gchar *utf8str;

          if (GTK_IS_TEXT_VIEW (edit))
            {
              GtkTextBuffer *buffer;
              GtkTextIter start, end;

              buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit));
              gtk_text_buffer_get_start_iter (buffer, &start);
              gtk_text_buffer_get_end_iter (buffer, &end);
              utf8str = gtk_text_buffer_get_text (buffer, &start, &end, TRUE);
            }
          else
            {
              utf8str = g_strdup (gtk_entry_get_text (GTK_ENTRY (edit)));
            }
          if (utf8str)
            {
              LRESULT lResult;

              if (wd->widechar)
                {
                  gunichar2 *wc;

                  wc = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
                  lResult = lstrlenW (wc);
                  g_free (wc);
                }
              else
                {
                  gchar *mb;

                  mb = w32ldr_utf8_to_mb (utf8str);
                  lResult = g_strlen (mb);
                  g_free (mb);
                }
              g_free (utf8str);
              return lResult;
            }
        }
        return 0;
      case WM_SETTEXT:
        {
          gchar *utf8str;

          utf8str = lParam ? wd->widechar
                           ? g_utf16_to_utf8 (GINT_TO_POINTER (lParam),
                                                        -1, NULL, NULL, NULL)
                           : w32ldr_utf8_from_mb (GINT_TO_POINTER (lParam))
                           : g_strdup ("");
          if (GTK_IS_TEXT_VIEW (edit))
            gtk_text_buffer_set_text
                (gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit)), utf8str, -1);
          else
            gtk_entry_set_text (GTK_ENTRY (edit), utf8str);
          g_free (utf8str);
        }
        return TRUE;
    }
  return wd->widechar ? DefWindowProcW (hWnd, Msg, wParam, lParam)
                       : DefWindowProcA (hWnd, Msg, wParam, lParam);
}


/* ja:バッファの内容が変化したとき */
static void
w32edit_buffer_changed (GtkTextBuffer *buffer,
                        GtkWidget     *widget)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->widechar)
    {
      SendMessageW (wd->hWndParent, WM_COMMAND,
                    MAKEWPARAM (wd->wID, EN_CHANGE), GPOINTER_TO_INT (widget));
      SendMessageW (wd->hWndParent, WM_COMMAND,
                    MAKEWPARAM (wd->wID, EN_UPDATE), GPOINTER_TO_INT (widget));
    }
  else
    {
      SendMessageA (wd->hWndParent, WM_COMMAND,
                    MAKEWPARAM (wd->wID, EN_CHANGE), GPOINTER_TO_INT (widget));
      SendMessageA (wd->hWndParent, WM_COMMAND,
                    MAKEWPARAM (wd->wID, EN_UPDATE), GPOINTER_TO_INT (widget));
    }
}


/* ja:エントリーの内容が変化したとき */
static void
w32edit_entry_changed (GtkEditable *editable,
                       gpointer     user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (editable), "user_data");
  if (wd->widechar)
    {
      SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, EN_CHANGE), GPOINTER_TO_INT (editable));
      SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, EN_UPDATE), GPOINTER_TO_INT (editable));
    }
  else
    {
      SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, EN_CHANGE), GPOINTER_TO_INT (editable));
      SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, EN_UPDATE), GPOINTER_TO_INT (editable));
    }
}


/* ja:フォーカス取得 */
static gboolean
w32edit_focus_in (GtkWidget     *widget,
                  GdkEventFocus *event,
                  gpointer       user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, EN_SETFOCUS), GPOINTER_TO_INT (widget));
  else
    SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, EN_SETFOCUS), GPOINTER_TO_INT (widget));
  return FALSE;
}


/* ja:フォーカス喪失 */
static gboolean
w32edit_focus_out (GtkWidget     *widget,
                   GdkEventFocus *event,
                   gpointer       user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, EN_KILLFOCUS), GPOINTER_TO_INT (widget));
  else
    SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, EN_KILLFOCUS), GPOINTER_TO_INT (widget));
  return FALSE;
}


GtkWidget *
w32ldr_control_create_edit (const gchar    *windowname,
                            const guint32   style,
                            const guint32   exstyle,
                            const gint      width,
                            const gint      height,
                            const guint16   id,
                            HWND            hWndParent,
                            HINSTANCE       hInstance,
                            const gboolean  widechar)
{
  GtkWidget *edit, *widget;
  W32LdrWindowData *wd;
  LPARAM lParam;

  wd = g_malloc (sizeof (W32LdrWindowData));
  wd->dwStyle = style;
  wd->dwExStyle = exstyle;
  wd->hInstance = hInstance;
  wd->lpfnWndProc = w32edit_DefWindowProc;
  wd->hWndParent = hWndParent;
  wd->wID = id;
  wd->dwUserData = 0;
  wd->lpDialogFunc = NULL;
  wd->lResult = -1;
  wd->dwUser = 0;
  wd->csa = NULL;
  wd->csw = NULL;
  wd->dwInitParam = 0;
  wd->widechar = widechar;
  wd->resizable = TRUE;
  wd->classname = g_strdup ("EDIT");
  wd->focus = NULL;
  wd->child = NULL;
  if (style & ES_MULTILINE)
    {
      /* ja:テキストビュー */
      GtkTextBuffer *buffer;
      GtkTextIter iter;
      GtkWidget *scroll;

      edit = gtk_text_view_new ();
      gtk_text_view_set_justification (GTK_TEXT_VIEW (edit),
                                      style & ES_CENTER ? GTK_JUSTIFY_CENTER
                                    : style & ES_RIGHT  ? GTK_JUSTIFY_RIGHT
                                                        : GTK_JUSTIFY_LEFT);
      gtk_text_view_set_editable (GTK_TEXT_VIEW (edit),
                                                    !(ES_READONLY & style));
      buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit));
      gtk_text_buffer_set_text (buffer, "", -1);
      gtk_text_buffer_get_iter_at_offset (buffer, &iter, 0);
      gtk_text_buffer_place_cursor (buffer, &iter);
      gtk_widget_show (edit);
      /* ja:スクロールウインドウ */
      scroll = gtk_scrolled_window_new (NULL, NULL);
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
      gtk_container_add (GTK_CONTAINER (scroll), edit);
      g_signal_connect (G_OBJECT (buffer), "changed",
                                G_CALLBACK (w32edit_buffer_changed), scroll);
      wd->focus = edit;
      widget = scroll;
    }
  else
    {
      /* ja:エントリー */
      edit = gtk_entry_new ();
#if GTK_CHECK_VERSION(2,4,0)
      gtk_entry_set_alignment (GTK_ENTRY (edit),
                        style & ES_CENTER ? 0.5 : style & ES_RIGHT ? 1 : 0);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      gtk_editable_set_editable (GTK_EDITABLE (edit), !(ES_READONLY & style));
      gtk_entry_set_visibility (GTK_ENTRY (edit), !(ES_PASSWORD & style));
      g_signal_connect (G_OBJECT (edit), "changed",
                                    G_CALLBACK (w32edit_entry_changed), NULL);
      wd->focus = edit;
      widget = edit;
    }
  g_signal_connect (G_OBJECT (edit), "focus-in-event",
                                        G_CALLBACK (w32edit_focus_in), NULL);
  g_signal_connect (G_OBJECT (edit), "focus-out-event",
                                        G_CALLBACK (w32edit_focus_out), NULL);
  gtk_widget_add_events (edit, GDK_FOCUS_CHANGE_MASK);
  g_object_set_data (G_OBJECT (widget), "user_data", wd);
  /* ja:テキスト */
  lParam = widechar ? GPOINTER_TO_INT (w32ldr_atom_get_string_wc (windowname))
                    : GPOINTER_TO_INT (w32ldr_atom_get_string_mb (windowname));
  w32edit_DefWindowProc (widget, WM_SETTEXT, 0, lParam);
  g_free (GINT_TO_POINTER (lParam));
  return widget;
}
