/*
    avicore
    copyright (c) 1998-2008 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "aviclip.h"
#include "misc/pixbuf.h"
#ifdef G_OS_WIN32
# include <gdk/gdkwin32.h>
# include <tchar.h>
#endif /* G_OS_WIN32 */


enum
{
  PASTE_SIGNAL,
  LAST_SIGNAL
};
#ifndef G_OS_WIN32
enum
{
# if GTK_CHECK_VERSION(2,6,0)
  TARGET_VMAID,
  TARGET_PIXBUF
# else /* not GTK_CHECK_VERSION(2,6,0) */
  TARGET_VMAID
# endif /* not GTK_CHECK_VERSION(2,6,0) */
};
#endif /* not G_OS_WIN32 */


static gint avi_clip_signals[LAST_SIGNAL] = {0};
#ifdef G_OS_WIN32
static UINT uFormat;            /* ja:クリップボード */
#else /* not G_OS_WIN32 */
static GdkAtom atom_scenario;   /* ja:シナリオオブジェクト */
#endif /* not G_OS_WIN32 */


/******************************************************************************
*                                                                             *
******************************************************************************/
G_DEFINE_TYPE (AviClip, avi_clip, G_TYPE_OBJECT);
static void avi_clip_dispose (GObject *object);


static void
avi_clip_class_init (AviClipClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->dispose = avi_clip_dispose;

  klass->paste = NULL;

  avi_clip_signals[PASTE_SIGNAL]
        = g_signal_new ("paste",
                G_TYPE_FROM_CLASS (klass),
                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                G_STRUCT_OFFSET (AviClipClass, paste),
                NULL, NULL,
                g_cclosure_marshal_VOID__POINTER,
                G_TYPE_NONE, 1,
                G_TYPE_POINTER);

#ifdef G_OS_WIN32
  uFormat = RegisterClipboardFormat (_T("Video maid Scenario Object"));
#else /* not G_OS_WIN32 */
  atom_scenario = gdk_atom_intern ("Video maid Scenario Object", FALSE);
#endif /* not G_OS_WIN32 */
}


/*  ja:新規作成
    RET,オブジェクト                                                        */
static void
avi_clip_init (AviClip *clip)
{
  clip->handler_destroy  = 0;
  clip->window           = NULL;
#ifndef G_OS_WIN32
  clip->clipboard        = NULL;
#endif /* not G_OS_WIN32 */
}


static void
avi_clip_dispose (GObject *object)
{
  AviClip *clip;

  clip = AVI_CLIP (object);
  if (clip->handler_destroy)
    g_signal_handler_disconnect (G_OBJECT (clip->window),
                                                        clip->handler_destroy);
  if (G_OBJECT_CLASS (avi_clip_parent_class)->dispose)
    G_OBJECT_CLASS (avi_clip_parent_class)->dispose (object);
}


/******************************************************************************
*                                                                             *
* ja:クリップボード関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    window,ウィジェット
       RET,オブジェクト                                                     */
GObject *
avi_clip_new (GtkWidget *window)
{
  AviClip *clip;

  if (!window)
    return NULL;
  clip = AVI_CLIP (g_object_new (AVI_TYPE_CLIP, NULL));
  clip->window = window;
  clip->handler_destroy = g_signal_connect_swapped (G_OBJECT (window),
                    "destroy", G_CALLBACK (g_object_unref), G_OBJECT (clip));
#ifndef G_OS_WIN32
# if GTK_CHECK_VERSION(2,2,0)
  clip->clipboard = gtk_widget_get_clipboard (window, GDK_SELECTION_CLIPBOARD);
# else /* not GTK_CHECK_VERSION(2,2,0) */
  clip->clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
# endif /* not GTK_CHECK_VERSION(2,2,0) */
#endif /* not G_OS_WIN32 */
  return G_OBJECT (clip);
}


#ifndef G_OS_WIN32
# if GTK_CHECK_VERSION(2,6,0)
struct _AviClipData
{
  gchar     *scenario;
  GdkPixbuf *pixbuf;
};
typedef struct _AviClipData AviClipData;
# endif /* GTK_CHECK_VERSION(2,6,0) */


static void
avi_clip_get_func (GtkClipboard     *clipboard,
                   GtkSelectionData *selection_data,
                   guint             info,
                   gpointer          user_data)
{
# if GTK_CHECK_VERSION(2,6,0)
  if (selection_data->selection == GDK_SELECTION_CLIPBOARD)
    {
      AviClipData *clip_data;

      clip_data = user_data;
      switch (info)
        {
          case TARGET_VMAID:
            gtk_selection_data_set (selection_data, atom_scenario, 8,
                                            (guchar *)clip_data->scenario,
                                            g_strlen (clip_data->scenario));
            break;
          case TARGET_PIXBUF:
            gtk_selection_data_set_pixbuf (selection_data, clip_data->pixbuf);
        }
    }
# else /* not GTK_CHECK_VERSION(2,6,0) */
  if (selection_data->selection == GDK_SELECTION_CLIPBOARD
                                                    && info == TARGET_VMAID)
    gtk_selection_data_set (selection_data, atom_scenario, 8,
                                            user_data, g_strlen (user_data));
# endif /* not GTK_CHECK_VERSION(2,6,0) */
}


static void
avi_clip_clear_func (GtkClipboard *clipboard,
                     gpointer      user_data)
{
# if GTK_CHECK_VERSION(2,6,0)
  AviClipData *clip_data;

  clip_data = user_data;
  g_free (clip_data->scenario);
  g_object_unref (clip_data->pixbuf);
  g_free (clip_data);
# else /* not GTK_CHECK_VERSION(2,6,0) */
  g_free (user_data);
# endif /* not GTK_CHECK_VERSION(2,6,0) */
}
#endif /* not G_OS_WIN32 */


/*  ja:クリップボードにコピーする
        clip,オブジェクト
    avi_edit,AVI編集ハンドル                                                */
void
avi_clip_copy (AviClip  *clip,
               AviEdit **avi_edit)
{
#ifdef G_OS_WIN32
  if (clip)
    {
      gchar *scenario;
      gint i;
      HGLOBAL hGlobalScenario = NULL, hGlobalBitmap = NULL;

      scenario = avi_edit_to_scenario (avi_edit, FALSE);
      if (scenario)
        {
          hGlobalScenario = GlobalAlloc (GMEM_MOVEABLE,
                                (g_strlen (scenario) + 1) * sizeof (gchar));
          if (hGlobalScenario)
            {
              LPSTR lpScenario;

              lpScenario = GlobalLock (hGlobalScenario);
              if (lpScenario)
                {
                  g_strcpy (lpScenario, scenario);
                  GlobalUnlock (hGlobalScenario);
                }
              else
                {
                  GlobalFree (hGlobalScenario);
                  hGlobalScenario = NULL;
                }
            }
          g_free (scenario);
        }
      for (i = 0; avi_edit[i]; i++)
        if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
          {
            AviFrame *avi_frame;

            avi_frame = avi_frame_open (avi_edit[i]);
            if (avi_frame)
              {
                BitmapInfoHeader *bmih;

                bmih = avi_frame_get_bitmap (avi_frame, 0,
                                        avi_edit_get_width (avi_edit[i]),
                                        avi_edit_get_height (avi_edit[i]),
                                        avi_edit_get_bit_count (avi_edit[i]));
                if (bmih)
                  {
                    hGlobalBitmap = GlobalAlloc (GMEM_MOVEABLE,
                                                        bm_all_bytes (bmih));
                    if (hGlobalBitmap)
                      {
                        LPBITMAPINFOHEADER lpBmih;

                        lpBmih = GlobalLock (hGlobalBitmap);
                        if (lpBmih)
                          {
                            g_memmove (lpBmih, bmih, bm_all_bytes (bmih));
                            GlobalUnlock (hGlobalBitmap);
                          }
                        else
                          {
                            GlobalFree (hGlobalBitmap);
                            hGlobalBitmap = NULL;
                          }
                      }
                  }
                avi_frame_close (avi_frame);
                if (hGlobalBitmap)
                  break;
              }
          }
      if (hGlobalScenario || hGlobalBitmap)
        {
          if (OpenClipboard (GDK_WINDOW_HWND (clip->window->window)))
            {
              if (EmptyClipboard ())
                {
                  if (hGlobalScenario
                                && SetClipboardData (uFormat, hGlobalScenario))
                    hGlobalScenario = NULL;
                  if (hGlobalBitmap
                                && SetClipboardData (CF_DIB, hGlobalBitmap))
                    hGlobalBitmap = NULL;
                }
              CloseClipboard ();
            }
          if (hGlobalScenario)
            GlobalFree (hGlobalScenario);
          if (hGlobalBitmap)
            GlobalFree (hGlobalBitmap);
        }
    }
#else /* not G_OS_WIN32 */
  if (clip)
    {
      gchar *scenario;
# if GTK_CHECK_VERSION(2,6,0)
      gint i, n_targets;
      GdkPixbuf *pixbuf = NULL;
      GtkTargetList *list;
      GtkTargetEntry *targets;
# endif /* GTK_CHECK_VERSION(2,6,0) */
      /* ja:提供可能なセレクション */
      GtkTargetEntry target[1]
                        = {{"Video maid Scenario Object", 0, TARGET_VMAID}};

      scenario = avi_edit_to_scenario (avi_edit, FALSE);
# if GTK_CHECK_VERSION(2,6,0)
      for (i = 0; avi_edit[i]; i++)
        if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
          {
            AviFrame *avi_frame;

            avi_frame = avi_frame_open (avi_edit[i]);
            if (avi_frame)
              {
                BitmapInfoHeader *bmih;

                bmih = avi_frame_get_bitmap (avi_frame, 0,
                                        avi_edit_get_width (avi_edit[i]),
                                        avi_edit_get_height (avi_edit[i]),
                                        avi_edit_get_bit_count (avi_edit[i]));
                if (bmih)
                  pixbuf = pixbuf_from_bitmap (bmih);
                avi_frame_close (avi_frame);
                if (pixbuf)
                  break;
              }
          }
      list = scenario ? gtk_target_list_new (target, G_N_ELEMENTS (target))
                      : gtk_target_list_new (NULL, 0);
      if (pixbuf)
        gtk_target_list_add_image_targets (list, TARGET_PIXBUF, TRUE);
#  if GTK_CHECK_VERSION(2,10,0)
      targets = gtk_target_table_new_from_list (list, &n_targets);
#  else /* not GTK_CHECK_VERSION(2,10,0) */
      n_targets = g_list_length (list->list);
      targets = g_malloc (n_targets * sizeof (GtkTargetEntry));
      for (i = 0; i < n_targets; i++)
        {
          GtkTargetPair *pair;

          pair = g_list_nth_data (list->list, i);
          targets[i].target = gdk_atom_name (pair->target);
          targets[i].flags  = pair->flags;
          targets[i].info   = pair->info;
        }
#  endif /* not GTK_CHECK_VERSION(2,10,0) */
      gtk_target_list_unref (list);
      if (targets)
        {
          AviClipData *clip_data;

          clip_data = g_malloc (sizeof (AviClipData));
          clip_data->scenario = scenario;
          clip_data->pixbuf = pixbuf;
          gtk_clipboard_clear (clip->clipboard);
          if (!gtk_clipboard_set_with_data (clip->clipboard,
                            targets, n_targets,
                            avi_clip_get_func, avi_clip_clear_func, clip_data))
            avi_clip_clear_func (clip->clipboard, clip_data);
#  if GTK_CHECK_VERSION(2,10,0)
          gtk_target_table_free (targets, n_targets);
#  else /* not GTK_CHECK_VERSION(2,10,0) */
          for (i = 0; i < n_targets; i++)
            g_free (targets[i].target);
          g_free (targets);
#  endif /* not GTK_CHECK_VERSION(2,10,0) */
        }
# else /* not GTK_CHECK_VERSION(2,6,0) */
      if (scenario)
        {
          gtk_clipboard_clear (clip->clipboard);
          if (!gtk_clipboard_set_with_data (clip->clipboard,
                            target, G_N_ELEMENTS (target),
                            avi_clip_get_func, avi_clip_clear_func, scenario))
            g_free (scenario);
        }
# endif /* not GTK_CHECK_VERSION(2,6,0) */
    }
#endif /* not G_OS_WIN32 */
}


#ifndef G_OS_WIN32
static void
avi_clip_received_func (GtkClipboard     *clipboard,
                        GtkSelectionData *selection_data,
                        gpointer          user_data)
{
  if (selection_data->selection == GDK_SELECTION_CLIPBOARD
                                    && selection_data->target == atom_scenario
                                    && selection_data->type == atom_scenario
                                    && selection_data->format == 8
                                    && selection_data->length > 0)
    {
      AviEdit **avi_edit;

      avi_edit = avi_edit_from_scenario ((gchar *)selection_data->data);
      if (avi_edit)
        {
          gint i;

          g_signal_emit_by_name (G_OBJECT (user_data), "paste", avi_edit);
          for (i = 0; avi_edit[i]; i++)
            avi_edit_close (avi_edit[i]);
          g_free (avi_edit);
        }
    }
}


# if GTK_CHECK_VERSION(2,6,0)
static void
avi_clip_image_received_func (GtkClipboard *clipboard,
                              GdkPixbuf    *pixbuf,
                              gpointer      user_data)
{
  AviEdit *avi_edit[2];

  avi_edit[0] = avi_edit_open_from_pixbuf (pixbuf);
  if (avi_edit[0])
    {
      avi_edit[1] = NULL;
      g_signal_emit_by_name (G_OBJECT (user_data), "paste", avi_edit);
      avi_edit_close (avi_edit[0]);
    }
}
# endif /* GTK_CHECK_VERSION(2,6,0) */
#endif /* not G_OS_WIN32 */


/*  ja:クリップボードから貼り付ける
    clip,オブジェクト                                                       */
void
avi_clip_paste (AviClip *clip)
{
#ifdef G_OS_WIN32
  if (clip && OpenClipboard (GDK_WINDOW_HWND (clip->window->window)))
    {
      AviEdit **avi_edit = NULL;

      if (IsClipboardFormatAvailable (uFormat))
        {
          HGLOBAL hGlobal;

          hGlobal = GetClipboardData (uFormat);
          if (hGlobal)
            {
              LPSTR lpScenario;

              lpScenario = GlobalLock (hGlobal);
              if (lpScenario)
                {
                  avi_edit = avi_edit_from_scenario (lpScenario);
                  GlobalUnlock (hGlobal);
                }
            }
        }
      else
        {
          AviEdit *avi_edit_video = NULL, *avi_edit_audio = NULL;

          if (IsClipboardFormatAvailable (CF_DIB))
            {
              HGLOBAL hGlobal;

              hGlobal = GetClipboardData (CF_DIB);
              if (hGlobal)
                {
                  LPBITMAPINFOHEADER lpBmih;

                  lpBmih = GlobalLock (hGlobal);
                  if (lpBmih)
                    {
                      avi_edit_video = avi_edit_open_from_memory (lpBmih);
                      GlobalUnlock (hGlobal);
                    }
                }
            }
#ifdef XXX
          if (IsClipboardFormatAvailable (CF_WAVE))
            {
              HGLOBAL hGlobal;

              hGlobal = GetClipboardData (CF_WAVE);
              if (hGlobal)
                {
                  LPVOID lpMem;

                  lpMem = GlobalLock (hGlobal);
                  if (lpMem)
                    {
                      avi_edit_audio = avi_edit_open_from_memory (lpMem);
                      GlobalUnlock (hGlobal);
                    }
                }
            }
#endif
          if (avi_edit_video || avi_edit_audio)
            {
              gint i = 0;

              avi_edit = g_malloc ((avi_edit_video && avi_edit_audio ? 3 : 2)
                                                        * sizeof (AviEdit *));
              if (avi_edit_video)
                avi_edit[i++] = avi_edit_video;
              if (avi_edit_audio)
                avi_edit[i++] = avi_edit_audio;
              avi_edit[i] = NULL;
            }
        }
      CloseClipboard ();
      if (avi_edit)
        {
          gint i;

          g_signal_emit_by_name (G_OBJECT (clip), "paste", avi_edit);
          for (i = 0; avi_edit[i]; i++)
            avi_edit_close (avi_edit[i]);
          g_free (avi_edit);
        }
    }
#else /* not G_OS_WIN32 */
# if GTK_CHECK_VERSION(2,6,0)
  if (clip)
    {
      if (gtk_clipboard_wait_is_target_available (clip->clipboard,
                                                                atom_scenario))
        gtk_clipboard_request_contents (clip->clipboard, atom_scenario,
                                                avi_clip_received_func, clip);
      else if (gtk_clipboard_wait_is_image_available (clip->clipboard))
        gtk_clipboard_request_image (clip->clipboard,
                                        avi_clip_image_received_func, clip);
    }
# else /* not GTK_CHECK_VERSION(2,6,0) */
  if (clip)
    gtk_clipboard_request_contents (clip->clipboard, atom_scenario,
                                                avi_clip_received_func, clip);
# endif /* not GTK_CHECK_VERSION(2,6,0) */
#endif /* not G_OS_WIN32 */
}
