
module Color
	RESET   = "\033]R"
	CRE     = "\033[K"
	CLEAR   = "\033c"
	NORMAL  = "\033[0;39m"
	RED     = "\033[1;31m"
	GREEN   = "\033[1;32m"
	YELLOW  = "\033[1;33m"
	BLUE    = "\033[1;34m"
	MAGENTA = "\033[1;35m"
	CYAN    = "\033[1;36m"
	WHITE   = "\033[1;37m"
end

class IgnoreWriter
	def <<(other)
	end
	def flush
	end
end

class Logger
	DEFAULT_LOGFILTER	= 3

	DEBUG0			= 0
	DEBUG				= 1
	DEBUGERROR	= 2
	INFO				= 3
	SUCCESS			= 4
	WARN				= 5
	ERROR				= 6

	def initialize
		@logfilter = DEFAULT_LOGFILTER
		@debug_mode = false
		@filewriter = IgnoreWriter.new
	end
	attr_accessor :debug_mode

	def setFilterLevel(level)
		@logfilter = level
		if level <= DEBUGERROR
			@debug_mode = true
		else
			@debug_mode = false
		end
	end

	def setLogFile(path)
		@filewriter = File.open(path, "w")
	end

	def logEnd
		if @filewriter.class != IgnoreWriter
			@filewriter.close
			@filewriter = IgnoreWriter.new
		end
	end

	def writelog(level, msg)
		begin
			if level < @logfilter
				return
			end

			color  = Color::NORMAL
			prefix = ""
			postfix = ""
			info = ""

			case level
			when DEBUG0
				color = Color::NORMAL
				prefix = "#{caller(2)[0]}:  "
				#postfix = "   :<=#{caller(2).join("  <=")}"
			when DEBUG
				color = Color::CYAN
				prefix = "#{caller(2)[0]}:  "
				#postfix = "   :<=#{caller(2).join("  <=")}"
			when DEBUGERROR
				color = Color::WHITE
				prefix = "#{caller(2)[0]}:  "
				#info = "#{caller(2).join("\n    from ")}"
			when INFO
				color = Color::YELLOW
				@debug_mode && prefix = "#{caller(2)[0]}:  "
			when SUCCESS
				color = Color::GREEN
				@debug_mode && prefix = "#{caller(2)[0]}:  "
			when WARN
				color = Color::MAGENTA
				#@debug_mode && info = "#{caller(2).join("\n    from ")}"
			when ERROR
				color = Color::RED
				#@debug_mode && info = "#{caller(2).join("\n    from ")}"
			end

			# format
			puts "#{color}#{prefix}#{msg}#{postfix}#{Color::NORMAL}"
			@filewriter << "#{prefix}#{msg}#{postfix}\n"
			if !info.empty?
				puts "#{color}#{info}#{Color::NORMAL}"
				@filewriter << "#{info}\n"
			end

		rescue
			puts "#{Color::RED}Log Error: #{$!}#{Color::NORMAL}"
		end
	end

	def flush
		@filewriter.flush
	end

	def debug0(msg)
		writelog(DEBUG0, msg)
	end

	def debug(msg)
		writelog(DEBUG,  msg)
	end

	def debugError(msg)
		writelog(DEBUGERROR,  msg)
	end

	def info(msg)
		writelog(INFO,   msg)
	end

	def success(msg)
		writelog(SUCCESS,   msg)
	end

	def warn(msg)
		writelog(WARN,   msg)
	end

	def error(msg)
		writelog(ERROR,  msg)
	end
end

$log = Logger.new

$log.debug "#{File.basename(__FILE__, ".*")} loaded"

