#ifndef VFIELD_AUTOSOCK_H__
#define VFIELD_AUTOSOCK_H__

#include "finalizer.h"
#include "exception.h"
#include "tools.h"
#include <cstdio>
#include <fcntl.h>
#include <boost/bind.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/scoped_ptr.hpp>

namespace VFIELD {

////
// Exception
//
struct SocketErrorException : public SystemCallException {
	SocketErrorException(int errno_, const std::string& message) :
		SystemCallException(errno_, message) {}
};
struct SocketClosedException : public std::runtime_error {
	SocketClosedException() :
		std::runtime_error("Stream socket is closed unexpectedly") {}
};
struct SocketTimedOutException : public std::runtime_error {
	SocketTimedOutException() :
		std::runtime_error("Stream connection timed out") {}
};


class AutoSockFinalizer {
public:
	AutoSockFinalizer() {}
	virtual ~AutoSockFinalizer() {}
	virtual void operator() (int sock) throw()
	{  // デフォルトのファイナライザ
		::close(sock);
	}
};


class AutoSockHolder {
public:
	AutoSockHolder(int sock) :
		m_sock(sock) {}

	AutoSockHolder(int sock, AutoSockFinalizer* new_finalizer) :
		m_sock(sock), m_finalizer(new_finalizer) {}

	~AutoSockHolder()
	{
		if( m_sock > 0 && m_finalizer ) {
			(*m_finalizer)(m_sock);
		}
	}

	void setFinalizer(AutoSockFinalizer* new_finalizer)
		{ m_finalizer.reset(new_finalizer); }
public:
	inline int sock(void) { return m_sock; }
	inline int release(void) throw()
	{
		int tmp = m_sock;
		m_sock = -1;
		return tmp;
	}
private:
	int m_sock;
	boost::scoped_ptr<AutoSockFinalizer> m_finalizer;
};


////
// AutoSock
//
class AutoSock {  // copyable, shared resource
public:
	AutoSock(int sock) :
		m_holder( new AutoSockHolder(sock) ) {}

	~AutoSock() {}

	template <typename Finalizer>
	void setFinalizer(Finalizer func)
	{
		m_holder->setFinalizer(new Finalizer(func));
	}
public:
	inline void close(void) throw()
	{
		::close( m_holder->release() );
	}

	inline int release(void) throw()
	{
		return m_holder->release();
	}

	inline size_t read(void* buf, size_t count)
	{
		ssize_t len = ::read(m_holder->sock(), buf, count);
		if( likely(len > 0) ) {
			return len;
		} else if( len == 0 ) {
			close();
			throw SocketClosedException();
		} else if( errno == EAGAIN || errno == EWOULDBLOCK ) {  // len < 0
			// タイムアウト
			close();
			throw SocketTimedOutException();
		} else {
			// EINTRもエラーとみなす
			int err = errno;
			close();
			throw SocketErrorException(err, "Can't read from socket");
		}
	}

	inline size_t write(const void* buf, size_t count)
	{
		ssize_t len = ::write(m_holder->sock(), buf, count);
		if( likely(len > 0) ) {
			return len;
		} else if( len == 0 ) {
			close();
			throw SocketClosedException();
		} else if( errno == EAGAIN || errno == EWOULDBLOCK ) {  // len < 0
			// タイムアウト
			close();
			throw SocketTimedOutException();
		} else {
			// EINTRもエラーとみなす
			int err = errno;
			close();
			throw SocketErrorException(err, "Can't write to socket");
		}
	}

	inline size_t timed_read(void* buf, size_t count, long sec, long usec)
	{
		fd_set fds;
		struct timeval tv;
		int select_val;
		while(1) {
			FD_ZERO(&fds);
			FD_SET(m_holder->sock(), &fds);

			tv.tv_sec = sec;
			tv.tv_usec = usec;

			select_val = select(m_holder->sock()+1, &fds, NULL, &fds, &tv);
			if( likely(select_val > 0) ) {
				break;
			} else if( select_val == 0 ) {
				// タイムアウト
				close();
				throw SocketTimedOutException();
			} else if( select_val < 0 && errno != EINTR ) {
				int err = errno;
				close();
				throw SocketErrorException(err, "Can't wait socket");
			}
			// select_val < 0 && errno == EINTR
		}

		return read(buf, count);
	}

	inline size_t timed_write(const void* buf, size_t count, long sec, long usec)
	{
		fd_set fds;
		struct timeval tv;
		int select_val;
		while(1) {
			FD_ZERO(&fds);
			FD_SET(m_holder->sock(), &fds);

			tv.tv_sec = sec;
			tv.tv_usec = usec;

			select_val = select(m_holder->sock()+1, NULL, &fds, &fds, &tv);
			if( likely(select_val > 0) ) {
				break;
			} else if( select_val == 0 ) {
				// タイムアウト
				close();
				throw SocketTimedOutException();
			} else if( select_val < 0 && errno != EINTR ) {
				int err = errno;
				close();
				throw SocketErrorException(err, "Can't wait socket");
			}
			// select_val < 0 && errno == EINTR (リトライする)
		}

		return write(buf, count);
	}

	inline void setNonBlock(void)
	{
		if( ::fcntl(m_holder->sock(), F_SETFL, O_NONBLOCK) < 0 ) {
			int err = errno;
			close();
			throw SocketErrorException(err, "Can't set socket non-blocking mode");
		}
	}

	inline void setBlock(void)
	{
		if( ::fcntl(m_holder->sock(), F_SETFL, 0) < 0 ) {
			int err = errno;
			close();
			throw SocketErrorException(err, "Can't set socket blocking mode");
		}
	}
private:
	boost::shared_ptr<AutoSockHolder> m_holder;
private:
	AutoSock();
};


// helper function
template <typename Finalizer>
inline AutoSock makeAutoSock(int sock, Finalizer func)
{
	AutoSock asock(sock);
	asock.setFinalizer(func);
	return asock;
}


}  // namespace VFIELD

#endif /* autosock.h */
