#include "engine.h"
#include "commander.h"
#include "log.h"
#include "vfield.h"
#include "node.h"
#include "request.h"
#include "bufferedreq.h"
#include "stream/stream.h"
#include "rpc/rpcclient.h"
#include "vtable/vtable.h"
#include "threadpool.h"
#include "rw_lock.h"
#include <list>
#include <algorithm>
#include <functional>
#include <boost/bind.hpp>
#include <boost/ptr_container/ptr_vector.hpp>

namespace VFIELD {


namespace {


class WaitChildThread : private boost::noncopyable {
public:
	WaitChildThread() : m_end_flag(true) {}
	~WaitChildThread() throw()
	{
		try {
			wait();
			LogDebug0(". One child thread is joined");
		} catch (...) {
			LogWarn("An error occurred while waiting child searching engine");
		}
	}
public:
	inline void startchild(void) throw()
	{
		// コンストラクタでm_end_flagをfalseにしてしまうと、
		// WaitChildThreadNotifierの生成時に例外が発生した
		// ときにwait()が返ってこなくなってしまう。
		//boost::mutex::scoped_lock lk(m_mutex);
		m_end_flag = false;
	}
	void notify(void)  // FIXME: throwしてはいけない for ~WaitChildThreadNotifier
	{
		boost::mutex::scoped_lock lk(m_mutex);
		m_end_flag = true;
		m_condition.notify_all();
	}
private:
	void wait(void)
	{
		boost::mutex::scoped_lock lk(m_mutex);
		while( !m_end_flag ) {
			m_condition.wait(lk);
		}
	}
private:
	boost::mutex m_mutex;
	boost::condition m_condition;
	volatile bool m_end_flag;
};

class WaitChildThreadNotifier {
public:
	WaitChildThreadNotifier(WaitChildThread* wct) : m_wct(wct)
	{
		// スレッドが本当に開始したときだけ呼ばれる
		m_wct->startchild();
	}
	~WaitChildThreadNotifier()
	{
		m_wct->notify();
	}
private:
	WaitChildThread* m_wct;
};


typedef VTable::matches_type  table_type;


class EngineChildWorker {
public:
	EngineChildWorker(EngineToolSet& tools, EngineCommander& commander) :
		m_tools(tools), m_commander(commander) {}
	~EngineChildWorker() {}
public:
	void tryConnect(const NodeIdentity* node, const DataRange* range, WaitChildThread* wct);
	void streamDownload(	std::list<DataRange>::iterator progress_ite,
				char* buffer,
				const NodeIdentity node,
				WaitChildThread* wct  );
private:
	EngineToolSet m_tools;
	EngineCommander& m_commander;
private:
	EngineChildWorker();
};


void EngineChildWorker::tryConnect(const NodeIdentity* node, const DataRange* range, WaitChildThread* wct)
{
	WaitChildThreadNotifier wctn( wct );

	if( m_tools.smgr.tryConnect(*node) ) {
		// 接続成功
		m_commander.connectSuccess(*node, *range);
	} else {
		// 接続失敗/タイムアウト
		m_commander.connectFailure(*node, *range);
	}
}



void EngineChildWorker::streamDownload(
		std::list<DataRange>::iterator progress_ite,
		char* buffer,
		const NodeIdentity node,
		WaitChildThread* wct  )
{
	try {
		WaitChildThreadNotifier wctn( wct );


		const DataRange& range(*progress_ite);

		LogDebug( Log::format("Requesting Stream Get Data %1% to %2%") % range % addr46_t(node) );

		// リクエストバッファを作成
		char reqbuf[STREAM_HEADER_SIZE + STREAM_GET_DATA_SIZE];
		reqbuf[0] = STREAM_GET_DATA_MAGIC;
		::memcpy(&reqbuf[STREAM_HEADER_SIZE+0], range.getNetRaw().get(), DataRange::raw_size);

		// コネクションを開始＆ネゴシエーション
		AutoSock asock( m_tools.smgr.getReadyConnection(
					node,
					reqbuf,
					sizeof(reqbuf),
					STREAM_GET_DATA_ACK_MAGIC
					) );

		// ネゴシエーション成立

		// データの受信を開始
		LogDebug( Log::format("Negotiation OK. Downloading data (%1% from %2%) ...")
				% range
				% addr46_t(node) );
		uint64_t pos_start = range.start();
		uint64_t pos_end = range.end();
		uint64_t cur = pos_start;

		do {
			cur += asock.read( buffer + (cur - pos_start), pos_end - cur + 1 );
				// 例外は呼出元がキャッチ
		} while(cur <= pos_end);

		LogDebug0( Log::format("...Download succeeded (%1% from %2%)") % range % addr46_t(node) );

		LogDebug( Log::format("Stream Get Data %1% from %2% succeeded")
				% (*progress_ite)
				% addr46_t(node) );
		m_commander.downloadSuccess(progress_ite);

		// getReadyConnectionはAutoReturnのAutoSockを返すので、asockは自動的に返却される
		// AutoSockの返却は、commanderに通知する後まで遅延させる

	// XXX: タイムアウトのときはリトライする？
	// XXX: タイムアウトのときはVTableから削除する？
	} catch (const std::runtime_error& ex) {
		LogDebugError( Log::format("Stream Get Data %1% from %2% failed: %3%")
				% (*progress_ite)
				% addr46_t(node)
				% ex.what() );
		m_commander.downloadFailure(progress_ite);
	} catch (...) {
		LogDebugError( Log::format("Stream Get Data %1% from %2% failed: unknown error")
				% (*progress_ite)
				% addr46_t(node) );
		m_commander.downloadFailure(progress_ite);
	}
}



inline std::list<DataRange>::iterator match_find(std::list<DataRange>& trying, const DataRange& vr)
{
	// tryingから線形検索部分一致を線形検索
	return std::find_if(
			trying.begin(),
			trying.end(),
			boost::bind(&DataRange::contains_part, &vr, _1)
			//std::bind1st( std::mem_fun_ref(&DataRange::contains_part), vr )  // reference to reference
			);
}


struct RequestedDataNotFoundException : public std::runtime_error {
	RequestedDataNotFoundException() :
		std::runtime_error("Can't find requested data") {}
};

}  // noname namespace



void SearchEngine(BufferedRequest& request, EngineToolSet tools, EngineCommander& commander)
{
	LogDebug0( Log::format("Searching data %1%") % request );

	EngineChildWorker worker(tools, commander);

	unsigned short retry = SEARCH_ENGINE_RETRY_LIMIT;
	while(1) {  // retry loop
	try {

	// retry_recastはretryごとにリセット
	unsigned short retry_recast = SEARCH_ENGINE_RECAST_LIMIT;


	// VTableから検索
	table_type nodes;
	tools.vtable.find(request, nodes);


	// このptr_vectorのデストラクタで子スレッドをすべてwaitする。
	// waitしないと子スレッドが持つ（nodesの中身への）ポインタが
	// 無効になって、Segmentation Faultしてしまう
	boost::ptr_vector<WaitChildThread> wct_group;

	// EngineToolSetの中身は全部リファレンス



	std::list<DataRange> remain;
	std::list<DataRange> trying;
	std::list<DataRange> progress;
	// listのイテレータは、そのポイント先が削除されない限り無効化されない
	typedef std::list<DataRange>::iterator progress_iterator;

	// 接続済みノードをマッチングしながらremainをセット
	{
		pos_type cur( request.start() );  // 同じデータ範囲を複数の接続済みノードにリクエストしないために使用
		pos_type cur_trying( cur );	  // remainのセットに使用

		for(table_type::iterator nd(nodes.begin()), nd_end(nodes.end()); nd != nd_end; ++nd ) {
			if( tools.smgr.isConnected(nd->node()) ) {
				////
				// 接続済みノード
				//
				LogDebug0( Log::format("Checking table entry %1% on connected %2%") % nd->range() % addr46_t(nd->node()) );

				// curが既にリクエストされた範囲より先に進んでいるか、
				// curが既に範囲の終わりより先に進んでいる
				if( cur > request.end() || cur > nd->range().end() ) continue;

				// ここから cur <= nd->range().end()

				// curが範囲の始めより小さい
				// + nodesは範囲の始めでソートされている
				if( cur < nd->range().start() ) {
					// curからnd->range().start-1まで接続済みノードなし
					// remainへの追加はcur_tryingの役割
					cur = nd->range().start();
				}
				// ここから nd->range().start() <= cur && cur <= nd->range.end()

				// curから nd->range().end() まで接続済みノード
				// ただしrequest.end()を越えないようにする

				// 接続済みなのでダウンロード開始
				LogDebug0( Log::format("Found data %1% on connected node %2%")
						% DataRange(
							cur,
							std::min(nd->range().end(), request.end())
							)
						% addr46_t(nd->node())
						);
				progress.push_front(	// progressに追加
						DataRange(
							cur,
							std::min(nd->range().end(), request.end())
							)
						);
					// push_frontしたイテレータでダウンロード終了時に通知してもらう
				progress_iterator progress_ite( progress.begin() );
					// wctに追加
				wct_group.push_back( new WaitChildThread() );
				tools.threadpool.submitTask(
						boost::bind(
							&EngineChildWorker::streamDownload,
							&worker,
				/* DataRange */		progress_ite,
				/* buffer    */		(request.data() + cur - request.start()),
				/* Node      */		nd->node(),
							&(*wct_group.rbegin())
							)
						);

				// curを進める
				cur = nd->range().end() + 1;

				// curまではremainに追加する必要が無い
				if( cur_trying < cur ) {
					cur_trying = cur;
				}

				// cur_tryingの方が残っているかもしれないので、breakしない


			} else {
				////
				// 非接続済みノード
				//
				LogDebug0( Log::format("Checking table entry %1% on non-connected %2%") % nd->range() % addr46_t(nd->node()) );

				// VTableにマッチした時点で部分一致はしているので接続開始
				// XXX: 接続開始はすべての接続済みノードにダウンロードを開始するまで遅延させる（ほとんどのノードは接続済みである）
					// wctに追加
				wct_group.push_back( new WaitChildThread() );
				tools.threadpool.submitTask(
						boost::bind(
							&EngineChildWorker::tryConnect,
							&worker,
							&nd->node(),
							&nd->range(),
							&(*wct_group.rbegin())
							)
						);

				// cur_tryingが既にリクエストされた範囲より先に進んでいるか、
				// cur_tryingが既に範囲の終わりより先に進んでいる
				if( cur_trying > request.end() || cur_trying > nd->range().end() ) continue;

				// cur_tringが範囲の始めより小さい
				// + nodesは範囲の始めでソートされている
				if( cur_trying < nd->range().start() ) {
					// curからnd->range().start-1までマッチするノードなし
					// remainに追加
					LogDebug0( Log::format("Unmatched data %1%") % DataRange(cur_trying, nd->range().start() - 1) );
					remain.push_back( DataRange(cur_trying, nd->range().start() - 1) );
					cur_trying = nd->range().start();
				}

				// FIXME: ↑の条件を満たしたときは↓は必ず成り立つ
				// cur_tryingが範囲の終わりより小さい
				// + cur_tyring >= nd->range().start()
				if( cur_trying < nd->range().end() ) {
					// nd->range().end()まではマッチするノードあり（=remainには入れない）
					// ただしrequest.end()を越えないようにする
					LogDebug0( Log::format("Found data %1% uniquely on non-connected node %2%")
							% DataRange(
								cur_trying,
								std::min(nd->range().end(),request.end())
								)
							% addr46_t(nd->node()) );
					trying.push_front(
							DataRange(
								cur_trying,
								std::min(nd->range().end(), request.end())
								)
							);
					cur_trying = nd->range().end() + 1;
				}

			}
		}  // for( nodes.begin() - nd - nodes.end() )

		// 端があればremainに追加
		if( cur_trying < request.end() ) {
			LogDebug0( Log::format("Unmatched edge data %1%") % DataRange(cur_trying, request.end()) );
			remain.push_back( DataRange(cur_trying, request.end() ) );
		}

	}  // マッチング終わり
	LogDebug0("Data matching end");


	if( !remain.empty() ) {
		// どのノードにもマッチしないrangeがあるので、Findをブロードキャストする
		LogDebug0( Log::format("Broadcasting RPC Find, number of remains: %1%") % remain.size() );
		tools.rpcc.bcastFind(request);

		// remainをtryingに移す
		trying.splice(trying.end(), remain, remain.begin(), remain.end());
	}


	boost::xtime xt;
	boost::xtime_get(&xt, boost::TIME_UTC);		// FIXME: xtime_getのオーバーヘッドは？キャッシュした方が速い？
	xt.sec += SEARCH_ENGINE_RECAST_FIND_TIME_SEC;
	xt.nsec += SEARCH_ENGINE_RECAST_FIND_TIME_NSEC;

	while( !progress.empty() || !trying.empty() ) {
		// Commanderを待つ
		EngineCommander::value_ptr cmd( commander.timed_pop(xt) );

		if( !cmd ) {
			// XXX: ダウンロード済み範囲も含めてFindを再送してしまう
			// XXX: remainがもったいない
			// XXX: ダウンロードできていないときは大抵全部できていない？そんなことは無いだろう…
			// タイムアウトした
			if( !trying.empty() ) {
				// ! trying.empty() のときのみ
				// 巨大な範囲を一度にリクエストされたとき（Join時など）は、
				// trying.empty()の状態で何度もタイムアウトすることになる
				// XXX: Join時には分割してリクエストしてもらう？
				LogDebugError("Searching process timed out");

				if( retry_recast <= 0 ) {
					// タイムアウト制限を超えた
					throw RequestedDataNotFoundException();
				}

				// tryingが残っていればRPC Findを全レンジでブロードキャスト
				tools.rpcc.bcastFind(request);

				--retry_recast;
			}

			// タイムアウト時間を更新
			boost::xtime_get(&xt, boost::TIME_UTC);
			xt.sec += SEARCH_ENGINE_RECAST_FIND_TIME_SEC;
			xt.nsec += SEARCH_ENGINE_RECAST_FIND_TIME_NSEC;

			continue;
		}

		switch( cmd->letter ) {
			case command_type::RPCNotifyUp:		// ほぼ確実にデータを持っているノードを発見
			case command_type::ConnectSuccess:
				// rangeをtryingから検索
				// 部分一致した場合:
				//    tryingから線形検索部分一致を線形検索（match_find）
				//    tryingから部分一致部分を削除
				//    fromに部分一致部分をsreamDownloadを要請
				//    progressに部分一致部分を追加
				for( std::list<DataRange>::iterator
						rmc( match_find(trying, cmd->range()) ),
						rmc_end( trying.end() );
							// trying.end()はループの中で変化しない

						rmc != rmc_end;

						rmc = match_find(trying, cmd->range()) ) {

					// XXX: 例外安全でない

					// 完全一致しなかった部分をtryingに追加
					if( cmd->range().end() < rmc->end() ) {
						DataRange rem_greater( cmd->range().end() + 1, rmc->end() );
						trying.push_front(rem_greater);	// push_backはNG
					}
					if( rmc->start() < cmd->range().start() ) {
						DataRange rem_less( rmc->start(), cmd->range().start() - 1 );
						trying.push_front(rem_less);	// push_backはNG
					}

					// XXX: 部分一致部分は↑の条件から算出可能
					DataRange part(		// 部分一致部分
							std::max(cmd->range().start(), rmc->start()),
							std::min(cmd->range().end(),   rmc->end())
						      );
					LogDebug0( Log::format("Found data %1% on certained %2%") % part % addr46_t(cmd->node()) );

					// XXX: ↓↑ここはトランザクション管理しないといけない
					trying.erase(rmc);	// 部分一致部分を削除

					// ダウンロード開始
					progress.push_front( part );	// progressに追加
						// push_frontしたイテレータでダウンロード終了時に通知してもらう
					progress_iterator progress_ite( progress.begin() );
						// wctに追加
					wct_group.push_back( new WaitChildThread() );
					tools.threadpool.submitTask(
							boost::bind(
								&EngineChildWorker::streamDownload,
								&worker,
					/* DataRange */		progress_ite,
					/* buffer    */		(request.data() + part.start() - request.start()),
					/* Node      */		cmd->node(),  // この変数はすぐ削除されるのでコピー
								&(*wct_group.rbegin())
								)
							);
				}
				break;


			case command_type::DownloadSuccess:
				// ダウンロード成功
				// progressから削除
				// XXX: ここでSegmentation Faultすることがある
				progress.erase( cmd->progress_ite() );
				break;

			case command_type::DownloadFailure:
				// ダウンロード失敗
				// tryingに追加
				trying.push_front( *cmd->progress_ite() );
				// RPCFindを限定レンジでブロードキャスト
				tools.rpcc.bcastFind( *cmd->progress_ite() );
				// progressから削除
				progress.erase( cmd->progress_ite() );
				break;

			case command_type::ConnectFailure:
				// 接続失敗
				// RPCFindを限定レンジでブロードキャスト
				tools.rpcc.bcastFind( cmd->range() );
				break;
		}
	}

	// すべてダウンロード完了
	request.notifyTriger(BufferedRequest::SUCCESS);

	LogDebug( Log::format("Searching engine for %1% is end. Waiting child %2% threads...")
			% request
			% wct_group.size() );

	break;	// retry loopを抜ける
	// ここでwct_groupがデストラクトされて、子スレッドがすべてjoinされる


	}  // try
	catch (const std::runtime_error& ex) {
		LogWarn( Log::format("An error occurred while downloading data %1%, retrying: %2%") % request % ex.what() );
	} catch (...) {
		LogWarn( Log::format("Unknown error occurred while downloading data %1%, retrying") % request );
	}
	if( retry <= 0 ) {
		// データ取得失敗
		LogWarn( Log::format("Failed to download data %1%") % request );
		request.notifyTriger(BufferedRequest::FAILED);
		break;
	}
	--retry;
	}  // retry loop
}


}  // namespace VFIELD
