#ifndef VFIELD_DATARANGE_H__
#define VFIELD_DATARANGE_H__

#include "vfield.h"
#include "tools.h"
#include "exception.h"
#include <stdint.h>
#include <assert.h>
#include <cstring>
#include "auto_array"
#include <ostream>
#include <boost/lexical_cast.hpp>

namespace VFIELD {


////
// Exception
//
class InvalidDataRangeException : public std::runtime_error {
public:
	InvalidDataRangeException(const std::string& message, pos_type start, pos_type end) :
		std::runtime_error(
				message
				+ ": "
				+ boost::lexical_cast<std::string>(start)
				+ " - "
				+ boost::lexical_cast<std::string>(end)
				) {}
};


////
// DataRange
//
class DataRange {
friend class NoOverlapLess;
friend class NoOverlapGreater;
friend class CompareStartLess;
friend class CompareStartGreater;
friend class StartLessIfSameEndLess;
friend class SequenctialLess;
public:
	const static unsigned short raw_size = 16;
public:
	explicit DataRange(const char* net_data)			// throw( InvalidDataRangeException )
	{
		m_start = ntohll( *(uint64_t*)&net_data[0] );
		m_end   = ntohll( *(uint64_t*)&net_data[8] );
		if( unlikely(m_start > m_end) ) {
			throw( InvalidDataRangeException("Invalid Data Range", m_start, m_end) );
		}
	}

	DataRange(pos_type start, pos_type end) :	// throw( InvalidDataRangeException )
		m_start(start), m_end(end)
	{
		if( unlikely(m_start > m_end) ) {
			throw( InvalidDataRangeException("Invalid Data Range", m_start, m_end) );
		}
	}
	virtual ~DataRange() throw() {}
public:
	inline pos_type start(void)  const { return m_start; }
	inline pos_type end(void)    const { return m_end;   }
	typedef pos_type size_type;
	inline size_type length(void) const { return m_end - m_start + 1; }
	inline size_type size(void)   const { return length(); }
public:
	mem::auto_array<char> getNetRaw(void) const
	{
		mem::auto_array<char> data( new char[raw_size] );
		uint64_t start_n( htonll(m_start) );
		uint64_t end_n( htonll(m_end) );
		::memcpy(&data.get()[0], &start_n, 8);
		::memcpy(&data.get()[8], &end_n, 8);
		return data;
	}
public:
	inline bool operator== (const DataRange& rhl) const
		{ return m_start == rhl.m_start && m_end == rhl.m_end; }
	inline bool operator!= (const DataRange& rhl) const
		{ return ! operator==(rhl); }
	inline bool contains_part(const DataRange& rhl) const
		{ return rhl.m_start < m_end && m_start < rhl.m_end; }
	inline bool contains_all(const DataRange& rhl) const
		{ return m_start <= rhl.m_start && rhl.m_end <= m_end; }
friend  std::ostream& operator<< (std::ostream& stream, const DataRange& range)
		{ return stream << range.start() << " - " << range.end(); }
private:
	pos_type m_start;	// ホストバイトオーダ
	pos_type m_end;		// ホストバイトオーダ
private:
	DataRange();
};


struct NoOverlapLess {
	inline bool operator() (const DataRange& lhl, const DataRange& rhl) const
		{ return lhl.m_end < rhl.m_start; }
};
struct NoOverlapGreater {
	inline bool operator() (const DataRange& lhl, const DataRange& rhl) const
		{ return lhl.m_start > rhl.m_end; }
};

struct CompareStartLess {
	inline bool operator() (const DataRange& lhl, const DataRange& rhl) const
		{ return lhl.m_start < rhl.m_start; }
};
struct CompareStartGreater {
	inline bool operator() (const DataRange& lhl, const DataRange& rhl) const
		{ return lhl.m_start > rhl.m_start; }
};

struct StartLessIfSameEndLess {
	inline bool operator() (const DataRange& lhl, const DataRange& rhl) const
	{
		if( lhl.m_start != rhl.m_start ) {
			return lhl.m_start < rhl.m_start;
		} else {
			return lhl.m_end < rhl.m_end;
		}
	}
};

struct SequenctialLess {
	inline bool operator() (const DataRange& lhl, const DataRange& rhl) const
		{ return lhl.m_start <= rhl.m_start && lhl.m_end < rhl.m_end; }
};


}  // namespace VFIELD

#endif /* datarange.h */
