#ifndef VFIELD_LIMITED_CACHE_H__
#define VFIELD_LIMITED_CACHE_H__

#include "log.h"
#include "datarange.h"
#include "bufferedreq.h"
#include "tools.h"
#include <map>
#include <list>
#include <vector>
#include <string>
#include <algorithm>
#include <stdexcept>
#include <boost/shared_ptr.hpp>
#include <boost/weak_ptr.hpp>
//#include <boost/thread.hpp>

namespace VFIELD {


////
// Exception
//
struct InvalidLimitException : public std::runtime_error {
	InvalidLimitException(const std::string& message) :
		std::runtime_error(message) {}
};


class WeakCache {
public:
	typedef DataRange			key_type;
	typedef BufferedRequest			value_type;
	typedef boost::shared_ptr<value_type>	shared_type;
	typedef boost::weak_ptr<value_type>	weak_type;
	typedef std::list<shared_type>		link_type;
	typedef link_type::iterator		link_iterator;
public:
	WeakCache(shared_type& shared_value, link_iterator link) :
		m_weak(shared_value), m_link(link)  {}
	~WeakCache() {}
	WeakCache() {}	// コンテナに入れるために必要
public:
	bool		expired(void) const  { return m_weak.expired(); }
	shared_type	lock(void)    const  { return m_weak.lock();    }
	link_iterator	getLinkIterator(void) { return m_link; }
private:
	weak_type m_weak;
	link_iterator m_link;
};

// ロックしない
class LimitedCache {
public:
	typedef WeakCache::key_type		key_type;
	typedef WeakCache::value_type		value_type;
	typedef WeakCache::link_type		link_type;
	typedef WeakCache::link_iterator	link_iterator;
	typedef WeakCache::shared_type		shared_type;
	typedef NoOverlapLess					compare_type;
	typedef std::map<key_type, WeakCache, compare_type>	index_type;
	typedef index_type::iterator				index_iterator;
public:
	LimitedCache(pos_type hardlimit, pos_type softlimit);
	~LimitedCache();
public:
	template <typename MatchFunc>	// MatchFuncは、void func(const index_type::value_type&);
					// MatchFuncは例外を投げてはいけない
	void match(const key_type& key, MatchFunc& match_func);
	inline bool insert(const key_type& key, const value_type& value);
private:
	void checkHardLimit(void);
	//void checkSoftLimitThread(unsigned short interval_seconds);
private:
	index_type m_index;
	link_type  m_link;

	value_type::size_type m_cached_size;
	value_type::size_type m_hardlimit;
	value_type::size_type m_softlimit;	// XXX: softlimitは未実装  // XXX: 必要ない？
	//boost::thread m_softlimit_thread;	// スレッドは最後に初期化
};


template <typename MatchFunc>
void LimitedCache::match(const key_type& key, MatchFunc& match_func)
{
	// キャッシュ済みデータを検索
	std::vector<DataRange> uncached;

	std::pair<index_iterator, index_iterator> found( m_index.equal_range(key) );
	index_iterator& in(found.first);
	while( in != found.second ) {
		if( ! in->second.expired() ) {
			// 存在するデータ
			// m_link内の位置を繰り上げる
			m_link.splice(m_link.begin(), m_link, in->second.getLinkIterator());
			// match_funcを呼ぶ
			match_func( enconst(*in) );
			++in;
		} else {
			LogDebug0( Log::format("erased data: %1%") % in->first );
			// 削除済みデータ
			m_index.erase(in++);
		}
	}
}


bool LimitedCache::insert(const key_type& key, const value_type& value)
{
	LogDebug0( Log::format("inserting data: %1%") % key );
	m_link.push_front( shared_type(new value_type(value)) );
	std::pair<index_iterator, bool> ins = m_index.insert(
			index_type::value_type(
				key,
				WeakCache(*m_link.begin(), m_link.begin())
				)
			);
	if( likely(ins.second) ) {
		// データの挿入に成功
		m_cached_size += value.size();
		checkHardLimit();
		return true;
	} else {
		// データの挿入に失敗（DataRangeがオーバーラップしたデータがある）
		// ActiveCacheの構造上あり得ない
		LogDebug( Log::format("*** Failed to insert data %1% ***") % value );
		return false;
	}
}


}  // namespace VFIELD

#endif /* limited_cache.h */
