#include "active_cache.h"
#include "limited_cache.h"
#include "bufferedreq.h"
#include "limited_cache.h"
#include "vfield.h"
#include "log.h"
#include <algorithm>
#include <boost/thread.hpp>
#include <boost/weak_ptr.hpp>

namespace VFIELD {


////
// ActiveCacheIMPL
//
class ActiveCacheIMPL : private boost::noncopyable {
public:
	ActiveCacheIMPL(uint64_t hardlimit, uint64_t softlimit, SearchController& search);
	~ActiveCacheIMPL();
public:
	void setRequest(DataRequest& request, std::vector<DataReceptor>& result_receptors);
private:
	SearchController& m_search;
	boost::mutex m_mutex;
	LimitedCache m_cache;
};



////
// Constructor / Destructor
//
ActiveCache::ActiveCache(uint64_t hardlimit, uint64_t softlimit, SearchController& search) :
	impl( new ActiveCacheIMPL(hardlimit, softlimit, search) ) {}
ActiveCache::~ActiveCache() {}
ActiveCacheIMPL::ActiveCacheIMPL(uint64_t hardlimit, uint64_t softlimit, SearchController& search) :
	m_search(search), m_cache(hardlimit, softlimit)
{
	LogInfo( Log::format("Caching layer is initialized: %1% bytes hardlimit, %2% bytes softlimit") % hardlimit % softlimit); 
}
ActiveCacheIMPL::~ActiveCacheIMPL() {}

ActiveCacheIMPL* ActiveCache::operator-> (void) { return impl; }



class SetBuffer {
public:
	SetBuffer( DataRequest& request,
		   std::vector<DataReceptor>& result_receptors,
		   std::vector<DataRange>& result_uncached ) :
		m_current_pos(request.start()),
		m_request(request),
		m_result_receptors(result_receptors),
		m_result_uncached(result_uncached)
	{}
	~SetBuffer()
	{
		// データの端を処理
		if( m_current_pos < m_request.end() ) {
			LogDebug0( Log::format("Unmatched data: %1%") % DataRange(m_current_pos, m_request.end()) );
			m_result_uncached.push_back( DataRange(m_current_pos, m_request.end()) );
		}
	}
public:
	typedef WeakCache::link_type		link_type;
	void operator() (const LimitedCache::index_type::value_type& in)
	{
		if( m_current_pos > m_request.end() ) {
			LogDebug0("All data is already matched");
			return;
		}
		if( m_current_pos < in.first.start() ) {
			// m_current_posより、マッチしたデータの先頭の方が大きい
			// m_current_pos - in.fist.start()-1 まではキャッシュに存在しないデータ
			LogDebug0( Log::format("Unmatched data: %1%") % DataRange(m_current_pos, (in.first.start()-1)) );
			m_result_uncached.push_back( DataRange(m_current_pos, in.first.start()-1) );
			m_current_pos = in.first.start();
		}
		// m_current_posよりマッチしたデータの先頭の方が小さいか同じ
		if( m_current_pos <= in.first.end() ) {
			// m_current_posより、マッチしたデータの終端の方が大きいか同じ
			// m_current_pos - in.first.end() まではキャッシュにあるデータ
			LogDebug0( Log::format("Matched data: %1%") % DataRange(m_current_pos, in.first.end()) );
			m_result_receptors.push_back(
					DataReceptor(
						*in.second.lock(),
						m_request
						)
					);
			m_current_pos = in.first.end() + 1;
		}
	}
private:
	pos_type m_current_pos;
	DataRequest& m_request;
	std::vector<DataReceptor>& m_result_receptors;
	std::vector<DataRange>& m_result_uncached;
};


namespace {
struct cache_inserter {
	cache_inserter(LimitedCache& cache) : m_cache(cache) {}
	void operator() (const BufferedRequest& value) {
		// Keyはアップキャスト、Valueはコピー演算
		// 挿入失敗は無視する（あり得ない）
		m_cache.insert(value, value);
	}
private:
	LimitedCache& m_cache;
	cache_inserter();
};
struct result_inserter {
	result_inserter(std::vector<DataReceptor>& result_receptors, DataRequest& request) :
		m_result(result_receptors), m_request(request) {}
	void operator() (const BufferedRequest& part) {
		m_result.push_back( DataReceptor(part, m_request) );
	}
private:
	std::vector<DataReceptor>& m_result;
	DataRequest& m_request;
};
}  // noname namespace

void ActiveCache::setRequest(DataRequest& request, std::vector<DataReceptor>& result_receptors)
	{ return impl->setRequest(request, result_receptors); }
void ActiveCacheIMPL::setRequest(DataRequest& request, std::vector<DataReceptor>& result_receptors)
{
	// XXX: 例外安全でない
	LogDebug0( Log::format("Request: %1%") % request );

	boost::mutex::scoped_lock lk(m_mutex);

	// キャッシュ済みデータを検索
	std::vector<DataRange> uncached;
	{
		SetBuffer set_buf(request, result_receptors, uncached);
		m_cache.match(request, set_buf);
	}  // set_bufのデストラクト時にデータの端が処理される


	// 全てキャッシュされていたらここでreturn
	if( uncached.empty() ) return;


	// バッファ確保
	// XXX: bad_alloc
	// 暗黙の型変換時にバッファを確保
	std::vector<BufferedRequest> buffered(uncached.begin(), uncached.end());


	// bufferedをリクエスト
	m_search.submitRequests(buffered);
	// 別スレッドでデータをダウンロード中...


	// キャッシュにbufferedを追加
	std::for_each(	buffered.begin(),
			buffered.end(),
			cache_inserter(m_cache)
		     );

	// result_receptorsの後ろにリクエストしたbufferedを型変換しながら追加
	result_receptors.reserve( result_receptors.size() + buffered.size() );
	std::for_each(	buffered.begin(), buffered.end(),
			result_inserter(result_receptors, request)
			);
}


}  // namespace VFIELD
