/* Copyright 1999-2003 Red Hat, Inc.
 *
 * This software may be freely redistributed under the terms of the GNU
 * public license.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Portions of this code from pcmcia-cs, copyright David A. Hinds
 * <dahinds@users.sourceforge.net>
 * 
 */

#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/fd.h>

#include "misc.h"
#include "modules.h"

static void miscFreeDevice(struct miscDevice *dev)
{
	freeDevice((struct device *) dev);
}

static void miscWriteDevice(FILE *file, struct miscDevice *dev)
{
	writeDevice(file, (struct device *)dev);
}

static int miscCompareDevice(struct miscDevice *dev1, struct miscDevice *dev2)
{
	return compareDevice( (struct device *)dev1, (struct device *)dev2);
}

struct miscDevice *miscNewDevice(struct miscDevice *old)
{
	struct miscDevice *ret;

	ret = malloc(sizeof(struct miscDevice));
	memset(ret, '\0', sizeof(struct miscDevice));
	ret = (struct miscDevice *) newDevice((struct device *) old, (struct device *) ret);
	ret->bus = BUS_MISC;
	ret->newDevice = miscNewDevice;
	ret->freeDevice = miscFreeDevice;
	ret->writeDevice = miscWriteDevice;
	ret->compareDevice = miscCompareDevice;
	return ret;
}

#if defined(__i386__) || defined(__alpha__)

#include <sys/io.h>

unsigned short i365_base = 0x03e0;
unsigned short tcic_base = 0x240;
#define I365_IDENT		0x00
#define TCIC_SCTRL		0x06
#define TCIC_SCTRL_RESET	0x80
#define TCIC_ADDR		0x02
#define I365_REG(slot, reg)    (((slot) << 6) | (reg))

static unsigned char i365_get(unsigned short sock, unsigned short reg)
{
    unsigned char val = I365_REG(sock, reg);
    outb(val, i365_base); val = inb(i365_base+1);
    return val;
}

#endif

static char *getFloppyDesc(char *name) {
	int size;
	char *type;
	char desc[64];
	
	size = atoi(name+1);
	
	if (isupper(name[0]))
	  type = "3.5\"";
	else
	  type = "5.25\"";
	if (size > 1000)
	  snprintf(desc, 63, "%s %d.%dMB floppy drive", type, size / 1000, (size % 1000) / 10);
	else 
	  snprintf(desc, 64, "%s %dKB floppy drive", type, size);
	return strdup(desc);
}

struct device *miscProbe(enum deviceClass probeClass, int probeFlags,
			struct device *devlist)
{
	int fd, x, rc;
	char path[32], name[32];
	struct miscDevice *miscdev;
	struct floppy_drive_struct ds;
	
	if ((probeClass & CLASS_FLOPPY) || (probeClass & CLASS_SOCKET) ||
	    (probeClass & CLASS_CDROM)) {
		if (probeClass & CLASS_FLOPPY) {
			for (x=0; x<=3; x++) {
				snprintf(path, 31, "/dev/fd%d", x);
				fd = open(path, O_RDONLY|O_NONBLOCK);
				if (fd < 0)
					break;
				ioctl(fd, FDRESET, NULL);
				rc = ioctl(fd, FDGETDRVTYP, name);
				if (rc || !name || !strcmp(name, "(null)"))
					goto cont;
				rc = ioctl(fd, FDPOLLDRVSTAT, &ds);
				if (rc)
					goto cont;
				miscdev = miscNewDevice(NULL);
				miscdev->device = strdup(basename(path));
				miscdev->type = CLASS_FLOPPY;
				miscdev->desc = getFloppyDesc(name);
				miscdev->driver = strdup("unknown");
				if (ds.track < 0)
					miscdev->detached = 1;
				if (devlist)
					miscdev->next = devlist;
				devlist = (struct device *) miscdev;
				cont:
				close(fd);
			}
		}
#if defined(__i386__) || defined(__alpha__)
		if (probeClass & CLASS_SOCKET) {
			int val, sock, done, i;
			unsigned short old;
			
			sock = done = 0;
			i = ioperm(i365_base, 4, 1);
			if (i == 0)
				i = ioperm(0x80, 1, 1);

			if (i == 0)
				for (; sock < 2; sock++) {
					val = i365_get(sock, I365_IDENT);
					switch (val) {
					case 0x82:
					case 0x83:
					case 0x84:
					case 0x88:
					case 0x89: 
					case 0x8a:
					case 0x8b:
					case 0x8c:
						break;
					default:
						done = 1;
					}
					if (done) break;
				}
			
			if (sock) {
				miscdev = miscNewDevice(NULL);
				miscdev->type = CLASS_SOCKET;
				miscdev->desc = strdup("Generic i82365-compatible PCMCIA controller");
				miscdev->driver = strdup("i82365");
				if (devlist)
					miscdev->next = devlist;
				devlist = (struct device *) miscdev;
			}
			
			i = ioperm(tcic_base, 16, 1);
			if (i == 0)
				i = ioperm(0x80, 1, 1);
			if (i)
				return devlist;
			
			/* Anything there?? */
			for (i = 0; i < 0x10; i += 2)
				if (inw(tcic_base + i) == 0xffff)
					return devlist;

			/* Try to reset the chip */
			outw(TCIC_SCTRL_RESET, tcic_base + TCIC_SCTRL);
			outw(0, tcic_base + TCIC_SCTRL);
    
			/* Can we set the addr register? */
			old = inw(tcic_base + TCIC_ADDR);
			outw(0, tcic_base + TCIC_ADDR);
			if (inw(tcic_base + TCIC_ADDR) != 0) {
				outw(old, tcic_base + TCIC_ADDR);
				return devlist;
			}
    
			outw(0xc3a5, tcic_base + TCIC_ADDR);
			if (inw(tcic_base + TCIC_ADDR) != 0xc3a5)
				return devlist;

			miscdev = miscNewDevice(NULL);
			miscdev->type = CLASS_SOCKET;
			miscdev->desc = strdup("Generic TCIC-2 PCMCIA controller");
			miscdev->driver = strdup("tcic");
			if (devlist)
				miscdev->next = devlist;
			devlist = (struct device *) miscdev;
		}
#endif
#ifdef __powerpc__
		if (probeClass & CLASS_CDROM) {
			char *buf, *ptr, *b;
			int cdnum;
			int modloaded = 0;
			
			if (!loadModule("viocd"))
				modloaded = 1;
			
			if (access("/proc/iSeries/viocd", R_OK))
				return devlist;
			fd = open("/proc/iSeries/viocd", O_RDONLY);
			if (fd < 0)
				return devlist;
			b = buf = bufFromFd(fd);
			while (*buf) {
				ptr = buf;
				while (*ptr && *ptr != '\n')
					ptr++;
				if (*ptr) {
					*ptr = '\0';
					ptr++;
				}
				if (strncmp("viocd", buf, 5)) {
					buf = ptr;
					continue;
				}
				buf +=13;
				cdnum = atoi(buf);
				buf = strstr(buf,"type");
				miscdev = miscNewDevice(NULL);
				miscdev->device = malloc(20);
				snprintf(miscdev->device,19,"iseries/vcd%c", cdnum + 'a');
				miscdev->desc = malloc(64);
				if (buf)
					snprintf(miscdev->desc,63,"IBM Virtual CD-ROM %s",buf);
				else
					snprintf(miscdev->desc,63,"IBM Virtual CD-ROM");
				miscdev->type = CLASS_CDROM;
				miscdev->driver = strdup("ignore");
				if (devlist)
					miscdev->next = devlist;
				devlist = (struct device *) miscdev;
				buf = ptr;
				continue;
			}
			free(b);
			if (modloaded)
				removeModule("viocd");
		}
#endif
	}
	return devlist;
}
