(****************************************************************************\
** Copyright 2019 Levashev Ivan Aleksandrovich                              **
**                                                                          **
** Licensed under the Apache License, Version 2.0 (the "License");          **
** you may not use this file except in compliance with the License.         **
** You may obtain a copy of the License at                                  **
**                                                                          **
**     http://www.apache.org/licenses/LICENSE-2.0                           **
**                                                                          **
** Unless required by applicable law or agreed to in writing, software      **
** distributed under the License is distributed on an "AS IS" BASIS,        **
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. **
** See the License for the specific language governing permissions and      **
** limitations under the License.                                           **
\****************************************************************************)

library vdsoleaut;

{ Important note about DLL memory management: ShareMem must be the
  first unit in your library's USES clause AND your project's (select
  Project-View Source) USES clause if your DLL exports any procedures or
  functions that pass strings as parameters or function results. This
  applies to all strings passed to and from your DLL--even those that
  are nested in records and classes. ShareMem is the interface unit to
  the BORLNDMM.DLL shared memory manager, which must be deployed along
  with your DLL. To avoid using BORLNDMM.DLL, pass string information
  using PChar or ShortString parameters. }

uses
  SysUtils,
  Classes,
  Types,
  StrUtils,
  Dialogs,
  TypInfo,
  Windows,
  vdsoleaut.Impl in 'vdsoleaut.Impl.pas',
  vdsoleaut.Promises in 'vdsoleaut.Promises.pas';

{$R *.res}

const
  Buf_Size = 0;       { parameter buffer size (user-definable) }

var
  AHandle: THandle;   { application handle }
  ErrorCode: Integer;
  ResultBuffer: AnsiString;

function Init(Handle: THandle; Addr: vdsoleaut.Impl.TVDSExtEventProc; KeyString: PChar; var MaxPar, BufSize: Integer): PChar; export; cdecl; forward;
(*
This function is called when the extension is declared using the EXTERNAL
command.

Handle is the DialogScript application's handle, which is passed in case
it is needed by any of the DLL functions.

Addr is the address of the callback function which is used so that the
extension DLL can notify VDS of events.

KeyString can be anything, including null. It is the contents of the second
parameter to the EXTERNAL command. It could be used, for example, to pass
a serial number code so the DLL can check whether the calling program is
licensed to use it, or a version number so the DLL can check if it is
compatible. The string can be used for anything you like.

Any initialisation of the DLL should be performed during this function.

If everything is OK then the DLL should return a string containing the name
of the command/function which will be used to call the extension. If something
is wrong then a null string should be returned, which will cause VDS to halt
with an error.
*)

function CommandProc(Params: PAnsiChar): Integer; export; cdecl; forward;
(*
This function is called when the extension command is called. The parameters
on the command line are passed in Params as a series of concatenated null
terminated strings, and can be read using the NextParam function.

The return value should be:
  0   - everything OK: OK indicator left true
  -1  - non-fatal error: OK indicator set false;
  >0  - a standard VDS error code (errno is set to this)
*)

function FuncProc(Args: PAnsiChar): PAnsiChar; export; cdecl; forward;
(*
This function is called when the extension function is called. The arguments
are passed in Args as a series of concatenated null terminated strings, and
can be read using the NextParam function.

The return value is the string value that is substituted for the function.

The function StatProc is called immediately after FuncProc to get the value
of OK / errorcode.
*)

function StatProc: Integer; export; cdecl; forward;
(*
This function is called by VDS immediately after FuncProc to obtain the
value of OK and errorcode.
*)

{ begin utility functions (not exported) }
function ParamsToArray(Params: PAnsiChar): Types.TStringDynArray;
var
  InternalBuffer: AnsiString;
  ResultLength: Integer;
  Was_Zero, Next_Zero: Integer;
  ParamsEnd: PAnsiChar;
begin
  ResultLength := 0;
  Result := nil;
  if Params = nil then
  begin
    Exit;
  end;
  ParamsEnd := Params + Types.PInteger(Params - 4)^ - 1;
  while (ParamsEnd >= Params) and (ParamsEnd^ = #0) do
  begin
    Dec(ParamsEnd);
  end;
  if ParamsEnd + 1 = Params then
  begin
    Exit;
  end;
  SetString(InternalBuffer, Params, ParamsEnd - Params + 1);

  Was_Zero := 0;
  Next_Zero := Pos(#0, InternalBuffer);
  while Next_Zero > 0 do
  begin
    SetLength(Result, ResultLength + 1);
    Result[ResultLength] := Copy(InternalBuffer, Was_Zero + 1, Next_Zero - Was_Zero - 1);
    Inc(ResultLength);
    Was_Zero := Next_Zero;
    Next_Zero := StrUtils.PosEx(#0, InternalBuffer, Was_Zero + 1);
  end;

  if Was_Zero <> Length(InternalBuffer) then
  begin
    SetLength(Result, ResultLength + 1);
    Result[ResultLength] := Copy(InternalBuffer, Was_Zero + 1, Length(InternalBuffer) - Was_Zero);
  end;
end;

{ end utility functions }

{ begin exported functions }

var
  InitResult: AnsiString;

function Init(Handle: THandle; Addr: vdsoleaut.Impl.TVDSExtEventProc; KeyString: PAnsiChar; var MaxPar, BufSize: Integer): PAnsiChar;
var
  KeyStringAsString: AnsiString;
begin
  AHandle := Handle;
  vdsoleaut.Impl.EventProc := Addr;
  vdsoleaut.Impl.WHandle := Windows.PHandle(MaxPar);
  MaxPar := vdsoleaut.Impl.Max_Parameters;
  BufSize := Buf_Size;
  { start user-defined code }
  SetString(KeyStringAsString, KeyString, Types.PInteger(KeyString - 4)^);
  try
    InitResult := vdsoleaut.Impl.HLInit(KeyStringAsString);
    Result := PAnsiChar(InitResult);
  except
    on E: vdsoleaut.Impl.EVDSError do
    begin
      ErrorCode := E.ErrorCode;
      Result := '';
    end;
    on E: Exception do
    begin
      vdsoleaut.Impl.MessageException(E);
      ErrorCode := 32;
      Result := '';
    end;
  end;
  Exit;
  { end user-defined code (remember to set Result) }
end;

function CommandProc(Params: PAnsiChar): Integer;
begin
  Result := 0;
  try
    vdsoleaut.Impl.HLCommandProc(ParamsToArray(Params));
  except
    on E: vdsoleaut.Impl.EVDSError do
    begin
      Result := E.ErrorCode;
      Exit;
    end;
    on E: Exception do
    begin
      vdsoleaut.Impl.MessageException(E);
      Result := 32;
      Exit;
    end;
  end;
end;

var
  FuncResult: AnsiString;

function FuncProc(Args: PAnsiChar): PAnsiChar;
begin
  ErrorCode := 0;
  try
    FuncResult :=
      vdsoleaut.Impl.HLFunctionProc(ParamsToArray(Args));
    Result := PAnsiChar(FuncResult);
  except
    on E: vdsoleaut.Impl.EVDSError do
    begin
      ErrorCode := E.ErrorCode;
      Result := '';
      Exit;
    end;
    on E: Exception do
    begin
      vdsoleaut.Impl.MessageException(E);
      ErrorCode := 32;
      Result := '';
      Exit;
    end;
  end;
end;

function StatProc: Integer;
{ this function requires no modification }
begin
  Result := ErrorCode;
end;

exports
  Init, CommandProc, FuncProc, StatProc;

begin
end.
