(****************************************************************************\
** Copyright 2019 Levashev Ivan Aleksandrovich                              **
**                                                                          **
** Licensed under the Apache License, Version 2.0 (the "License");          **
** you may not use this file except in compliance with the License.         **
** You may obtain a copy of the License at                                  **
**                                                                          **
**     http://www.apache.org/licenses/LICENSE-2.0                           **
**                                                                          **
** Unless required by applicable law or agreed to in writing, software      **
** distributed under the License is distributed on an "AS IS" BASIS,        **
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. **
** See the License for the specific language governing permissions and      **
** limitations under the License.                                           **
\****************************************************************************)

unit vdsoleaut.Impl;

interface

uses
  SysUtils, Types, Windows;

const
  Max_Parameters = 6 + 2 * ({ComObj.MaxDispArgs} 64 + 1);
                            { maximum number of params/args (user-definable) }
                            { more than Max_Parameters are not visible }

type
  TVDSExtEventProc = procedure(EventType: PAnsiChar); cdecl;

  EVDSError = class(Exception)
  private
    FErrorCode: Integer;
  public
    property ErrorCode: Integer read FErrorCode;
    constructor Create(AErrorCode: Integer = 32);
  end;

var
  EventProc: TVDSExtEventProc;
  WHandle: Windows.PHandle;
  WM_VDSOLEASYNC: Windows.UINT = 0;

procedure MessageException(E: SysUtils.Exception);
function HLInit(const KeyString: AnsiString): AnsiString;
procedure HLCommandProc(const Params: Types.TStringDynArray);
function HLFunctionProc(const Args: Types.TStringDynArray): AnsiString;

implementation

uses
  Dialogs, Variants, ActiveX, ComObj, Classes, TypInfo,
  vdsoleaut.Promises;

// --------------------- BEGIN Utility

constructor EVDSError.Create(AErrorCode: Integer = 32);
begin
  FErrorCode := AErrorCode;
  inherited Create('VDS error #' + SysUtils.IntToStr(AErrorCode));
end;

var
  AllocatedOleVariants: array of OleVariant;
  AllocatedOleEnumerators: array of ActiveX.IEnumVARIANT;

function AllocateOleVariant(const Item: OleVariant): AnsiString;
var
  L: Integer;
begin
  L := Length(AllocatedOleVariants);
  SetLength(AllocatedOleVariants, L + 1);
  AllocatedOleVariants[L] := Item;
  Result := IntToStr(L + 1);
end;

function AllocateOleEnumerator(const Item: ActiveX.IEnumVARIANT): AnsiString;
var
  L: Integer;
begin
  L := Length(AllocatedOleEnumerators);
  SetLength(AllocatedOleEnumerators, L + 1);
  AllocatedOleEnumerators[L] := Item;
  Result := IntToStr(L + 1);
end;

function AllocatedOleVariant(Index: Integer): OleVariant;
begin
  if (Index < 1) or (Index > Length(AllocatedOleVariants)) then
  begin
    Result := Variants.Unassigned;
    Exit;
  end;

  Result := AllocatedOleVariants[Index - 1];
end;

function AllocatedOleEnumerator(Index: Integer): ActiveX.IEnumVARIANT;
begin
  if (Index < 1) or (Index > Length(AllocatedOleEnumerators)) then
  begin
    Result := nil;
    Exit;
  end;

  Result := AllocatedOleEnumerators[Index - 1];
end;

procedure DeallocateOleVariant(Index: Integer);
var
  L, I: Integer;
begin
  if (Index < 1) or (Index > Length(AllocatedOleVariants)) then
  begin
    Exit;
  end;

  System.VarClear(AllocatedOleVariants[Index - 1]);
  L := Length(AllocatedOleVariants);
  for I := L - 1 downto 0 do
  begin
    if not Variants.VarIsEmpty(AllocatedOleVariants[I]) then
    begin
      if I <> L - 1 then
      begin
        SetLength(AllocatedOleVariants, I + 1);
      end;

      Exit;
    end;
  end;

  AllocatedOleVariants := nil;
end;

procedure DeallocateOleEnumerator(Index: Integer);
var
  L, I: Integer;
begin
  if (Index < 1) or (Index > Length(AllocatedOleEnumerators)) then
  begin
    Exit;
  end;

  AllocatedOleEnumerators[Index - 1] := nil;
  L := Length(AllocatedOleEnumerators);
  for I := L - 1 downto 0 do
  begin
    if Assigned(AllocatedOleEnumerators[I]) then
    begin
      if I <> L - 1 then
      begin
        SetLength(AllocatedOleEnumerators, I + 1);
      end;

      Exit;
    end;
  end;

  AllocatedOleEnumerators := nil;
end;

procedure DeallocateOleVariants;
begin
  AllocatedOleVariants := nil;
end;

procedure DeallocateOleEnumerators;
begin
  AllocatedOleEnumerators := nil;
end;

var
  FormatSettings: SysUtils.TFormatSettings;

type
  PIStream = ^ActiveX.IStream;
  TMtaOleObjectCreator = class(Classes.TThread)
  private
    FName: WideString;
    FResultAsIStream: PIStream;
  protected
    procedure Execute; override;
  public
    constructor Create(const AName: WideString; AResultAsIStream: PIStream);
  end;

constructor TMtaOleObjectCreator.Create(const AName: WideString; AResultAsIStream: PIStream);
begin
  inherited Create(True);
  FName := AName;
  FResultAsIStream := AResultAsIStream;
  Resume;
end;

procedure TMtaOleObjectCreator.Execute;
var
  ResultAsIDispatch: IDispatch;
begin
  ComObj.OleCheck(ActiveX.CoInitializeEx(nil, ActiveX.COINIT_MULTITHREADED));
  try
    ResultAsIDispatch := ComObj.CreateOleObject(FName);
    ComObj.OleCheck(ActiveX.CoMarshalInterThreadInterfaceInStream(IDispatch, ResultAsIDispatch, FResultAsIStream^));
  except
    on E: SysUtils.Exception do
    begin
      MessageException(E);
    end;
  end;
  ActiveX.CoUninitialize;
end;

function CreateMtaOleObject(const Name: WideString): IDispatch;
var
  Creator: TMtaOleObjectCreator;
  ResultAsIStream: ActiveX.IStream;
begin
  Creator := TMtaOleObjectCreator.Create(Name, @ResultAsIStream);
  try
    Creator.WaitFor;

    if not Assigned(ResultAsIStream) then
    begin
      raise EVDSError.Create;
    end;

    ComObj.OleCheck(CoGetInterfaceAndReleaseStream(ResultAsIStream, IDispatch, Result));
    ResultAsIStream := nil;
  finally
    FreeAndNil(Creator);
  end;
end;

function CreateOleVariant(const Value, Kind: AnsiString): OleVariant;
begin
  if Kind = 'STRING' then
  begin
    Result := WideString(Value);
  end
  else if Kind = 'UTF' then
  begin
    Result := UTF8Decode(Value);
  end
  else if Kind = 'INTEGER' then
  begin
    Result := StrToInt64(Value);
  end
  else if Kind = 'BOOLEAN' then
  begin
    if (Value = 'TRUE') or (Value = 'True') or (Value = 'true') then
    begin
      Result := True;
    end
    else if (Value = 'FALSE') or (Value = 'False') or (Value = 'false') then
    begin
      Result := False;
    end
    else
    begin
      raise EVDSError.Create;
    end;
  end
  else if Kind = 'REAL' then
  begin
    Result := StrToFloat(Value, FormatSettings);
  end
  else if (Kind = 'EMPTY') or (Kind = '') then
  begin
    Result := Variants.EmptyParam;
  end
  else if Kind = 'OBJECT' then
  begin
    Result := ComObj.CreateOleObject(WideString(Value));
  end
  else if Kind = 'THREADEDOBJECT' then
  begin
    Result := CreateMtaOleObject(WideString(Value));
  end
  else if Kind = 'VARIANT' then
  begin
    Result := AllocatedOleVariant(StrToInt(Value));
  end
  else
  begin
    raise EVDSError.Create;
  end;
end;

type
  TStringArrayForSlice = array[0 .. MaxInt div SizeOf(string) - 1] of string;
  PStringArrayForSlice = ^TStringArrayForSlice;

function BeginSlice(const DynArray: Types.TStringDynArray; Start: Integer): PStringArrayForSlice;
begin
  Result :=
    PStringArrayForSlice(PAnsiChar(Pointer(DynArray)) + Start * SizeOf(string));
end;

// --------------------- END Utility

var
  InitializedOle: Boolean = False;

function HLInit(const KeyString: AnsiString): AnsiString;
begin
  DeallocateOleVariants;
  if not InitializedOle then
  begin
    ComObj.OleCheck(ActiveX.OleInitialize(nil));
    InitializedOle := True;
  end;

  if WM_VDSOLEASYNC = 0 then
  begin
    WM_VDSOLEASYNC := Windows.RegisterWindowMessageW('WM_VDSOLEASYNC');
    if WM_VDSOLEASYNC = 0 then
    begin
      RaiseLastOSError;
    end;
  end;

  Result := 'OLE|OLEASYNC|*' + IntToStr(WM_VDSOLEASYNC) + '.OLEASYNCEVENT!';
end;

function HLTypeOf(const Item: OleVariant): AnsiString;
begin
  case Variants.VarType(Item) of
  varEmpty: Result := 'UNDEFINED';
  varNull: Result := 'NULL';
  varSmallInt, varInteger, varShortInt,
  varByte, varWord, varLongWord,
  varInt64, $0015: Result := 'INTEGER';
  varSingle, varDouble, varCurrency: Result := 'REAL';
  varDate: Result := 'DATETIME';
  varOleStr: Result := 'STRING';
  varDispatch: Result := 'OBJECT';
  varBoolean: Result := 'BOOLEAN';
  else
    Result := 'UNKNOWN';
  end;
end;

function HLToString(const Item: OleVariant; Ansi: Boolean): AnsiString;
begin
  case Variants.VarType(Item) of
  varEmpty: Result := 'UNDEFINED';
  varNull: Result := 'NULL';
  varSmallInt, varInteger, varShortInt,
  varByte, varWord: Result := IntToStr(Integer(Item));
  varLongWord: Result := IntToStr(Int64(LongWord(Item)));
  varInt64, $0015: Result := IntToStr(System.TVarData(Item).VInt64);
  varSingle, varDouble: Result := FloatToStr(Double(Item), FormatSettings);
  varCurrency: Result := CurrToStr(System.TVarData(Item).VCurrency, FormatSettings);
  varDate: Result := DateTimeToStr(Variants.VarToDateTime(Item), FormatSettings);
  varOleStr:
    if Ansi then
    begin
      Result := AnsiString(WideString(Item));
    end
    else
    begin
      Result := UTF8Encode(WideString(Item));
    end;
  varDispatch: Result := 'OBJECT';
  varBoolean: Result := BoolToStr(Boolean(Item), True);
  else
    Result := 'UNKNOWN';
  end;
end;

function HLEnumerate(const Item: OleVariant): ActiveX.IEnumVARIANT; forward;

function ProduceResult(const Item: OleVariant; const Mode: AnsiString): AnsiString;
begin
  if Mode = 'variant' then
  begin
    Result := AllocateOleVariant(Item);
    Exit;
  end
  else if Mode = 'string' then
  begin
    Result := HLToString(Item, True);
    Exit;
  end
  else if Mode = 'utf' then
  begin
    Result := HLToString(Item, False);
    Exit;
  end
  else if Mode = 'enumerator' then
  begin
    Result := AllocateOleEnumerator(HLEnumerate(Item));
    Exit;
  end;

  raise EVDSError.Create;
end;

const
  IID_NULL: TGUID = '{00000000-0000-0000-0000-000000000000}';

type
  TWideStringDynArray = array of WideString;
  // WideString is important (OLE), it's not UnicodeString
  TDispIDDynArray = array of ActiveX.TDispID;
  TOleVariantDynArray = array of OleVariant;
  TDispParamsHolder = record
    MethodID: ActiveX.TDispID;
    DispIDs: TDispIDDynArray;
    ParametersForInvoke: TOleVariantDynArray;
    DispParams: ActiveX.TDispParams;
  end;

function PrepareInvoke(const Item: IDispatch; const MethodName: AnsiString;
  const Parameters: array of AnsiString;
  PutProperty: Boolean = False): TDispParamsHolder;
var
  Names: TWideStringDynArray;
  ParameterIndex: Integer;
  Kind: AnsiString;
  KindPos: Integer;
  PutPropertyShift: Integer;
begin
  SetLength(Names, 1);
  Names[0] := WideString(MethodName);

  PutPropertyShift := 0;
  if PutProperty then
  begin
    PutPropertyShift := 1;
  end;

  SetLength(Result.ParametersForInvoke, Length(Parameters) div 2);
  for ParameterIndex := 0 to Length(Result.ParametersForInvoke) - 1 - PutPropertyShift do
  begin
    Kind := Parameters[ParameterIndex * 2 + 1];
    KindPos := Pos(':', Kind);
    if KindPos = 0 then
    begin
      if Length(Names) > 1 then
      begin
        raise EVDSError.Create;
        // cannot have positional parameters after named ones
      end;
    end
    else
    begin
      if Length(Names) = 1 then
      begin
        SetLength(Names, Length(Result.ParametersForInvoke) + 1 - ParameterIndex - PutPropertyShift);
      end;

      Names[Length(Result.ParametersForInvoke) - ParameterIndex - PutPropertyShift] :=
        WideString(Trim(Copy(Kind, 1, KindPos - 1)));
      Kind := Trim(Copy(Kind, KindPos + 1, Length(Kind) - KindPos));
    end;

    Result.ParametersForInvoke
     [Length(Result.ParametersForInvoke) - ParameterIndex - 1] :=
     CreateOleVariant(Parameters[ParameterIndex * 2], Kind);
  end;
  Kind := '';

  SetLength(Result.DispIDs, Length(Names));
  ComObj.OleCheck(IDispatch(Item).GetIDsOfNames
   (IID_NULL, @(Names[0]), Length(Names), ActiveX.STDOLE_LCID, @(Result.DispIDs[0])));

  SetLength(Names, 0);

  Result.MethodID := Result.DispIDs[0];

  if PutProperty then
  begin
    Result.DispIDs[0] := ActiveX.DISPID_PROPERTYPUT;
    ParameterIndex := Length(Result.ParametersForInvoke) - 1;
    Result.ParametersForInvoke[0] := CreateOleVariant(Parameters[ParameterIndex * 2], Parameters[ParameterIndex * 2 + 1]);
  end;

  if Length(Result.ParametersForInvoke) > 0 then
  begin
    Result.DispParams.rgvarg := Pointer(@(Result.ParametersForInvoke[0]));
  end
  else
  begin
    Result.DispParams.rgvarg := nil;
  end;

  if Length(Result.DispIDs) > (1 - PutPropertyShift) then
  begin
    Result.DispParams.rgdispidNamedArgs := Pointer(@(Result.DispIDs[1 - PutPropertyShift]));
  end
  else
  begin
    Result.DispParams.rgdispidNamedArgs := nil;
  end;

  Result.DispParams.cArgs := Length(Result.ParametersForInvoke);
  Result.DispParams.cNamedArgs := Length(Result.DispIDs) + PutPropertyShift - 1;
end;

procedure DispatchCheck(Result: HResult; const ExcepInfo: ActiveX.TExcepInfo);
begin
  if not ActiveX.Succeeded(Result) then
  begin
    ComObj.DispatchInvokeError(Result, ExcepInfo);
  end;
end;

procedure HLInvokeDo(const Item: OleVariant; Flags: Word;
  const MethodName: AnsiString;
  const Parameters: array of AnsiString);
var
  DispParamsHolder: TDispParamsHolder;
  ExcepInfo: ActiveX.TExcepInfo;
begin
  if not Variants.VarIsType(Item, varDispatch) then
  begin
    raise EVDSError.Create;
  end;

  DispParamsHolder := PrepareInvoke(IDispatch(Item), MethodName, Parameters);

  DispatchCheck(IDispatch(Item).Invoke
   (DispParamsHolder.MethodID, IID_NULL, ActiveX.STDOLE_LCID,
    Flags, DispParamsHolder.DispParams, nil, @ExcepInfo, nil), ExcepInfo);
end;

function HLInvokeDoWithResult(const Item: OleVariant; Flags: Word;
  const MethodName: AnsiString;
  const Parameters: array of AnsiString): OleVariant;
var
  DispParamsHolder: TDispParamsHolder;
  ExcepInfo: ActiveX.TExcepInfo;
begin
  if not Variants.VarIsType(Item, varDispatch) then
  begin
    raise EVDSError.Create;
  end;

  DispParamsHolder := PrepareInvoke(IDispatch(Item), MethodName, Parameters);

  DispatchCheck(IDispatch(Item).Invoke
   (DispParamsHolder.MethodID, IID_NULL, ActiveX.STDOLE_LCID,
    Flags, DispParamsHolder.DispParams, @Result, @ExcepInfo, nil), ExcepInfo);

  if Variants.VarIsByRef(Result) then
  begin
    case ActiveX.TVariantArg(Result).vt of
      ActiveX.VT_BYREF or ActiveX.VT_UI1: Result := ActiveX.TVariantArg(Result).pbVal^;
      ActiveX.VT_BYREF or ActiveX.VT_I2: Result := ActiveX.TVariantArg(Result).piVal^;
      ActiveX.VT_BYREF or ActiveX.VT_I4: Result := ActiveX.TVariantArg(Result).plVal^;
      ActiveX.VT_BYREF or ActiveX.VT_R4: Result := ActiveX.TVariantArg(Result).pfltVal^;
      ActiveX.VT_BYREF or ActiveX.VT_R8: Result := ActiveX.TVariantArg(Result).pdblVal^;
      ActiveX.VT_BYREF or ActiveX.VT_BOOL: Result := ActiveX.TVariantArg(Result).pbool^;
      ActiveX.VT_BYREF or ActiveX.VT_ERROR: Result := Variants.VarAsError(ActiveX.TVariantArg(Result).pscode^);
      ActiveX.VT_BYREF or ActiveX.VT_CY: Result := ActiveX.TVariantArg(Result).pcyVal^;
      ActiveX.VT_BYREF or ActiveX.VT_DATE: Result := Variants.VarFromDateTime(ActiveX.TVariantArg(Result).pdate^);
      ActiveX.VT_BYREF or ActiveX.VT_BSTR: Result := ActiveX.TVariantArg(Result).pbstrVal^;
      ActiveX.VT_BYREF or ActiveX.VT_UNKNOWN: Result := ActiveX.TVariantArg(Result).punkVal^;
      ActiveX.VT_BYREF or ActiveX.VT_DISPATCH: Result := ActiveX.TVariantArg(Result).pdispVal^;
      // ActiveX.VT_BYREF or ActiveX.VT_ARRAY: Result := ActiveX.TVariantArg(Result).pparray^;
      ActiveX.VT_BYREF or ActiveX.VT_VARIANT: Result := ActiveX.TVariantArg(Result).pvarVal^;
      // ActiveX.VT_BYREF or ActiveX.VT_DECIMAL:  (pdecVal: PDecimal);
      ActiveX.VT_BYREF or ActiveX.VT_I1: Result := ShortInt(ActiveX.TVariantArg(Result).pcVal^);
      ActiveX.VT_BYREF or ActiveX.VT_UI2: Result := ActiveX.TVariantArg(Result).puiVal^;
      ActiveX.VT_BYREF or ActiveX.VT_UI4: Result := ActiveX.TVariantArg(Result).pulVal^;
      ActiveX.VT_BYREF or ActiveX.VT_INT: Result := ActiveX.TVariantArg(Result).pintVal^;
      ActiveX.VT_BYREF or ActiveX.VT_UINT: Result := ActiveX.TVariantArg(Result).puintVal^;
    else
      raise EVDSError.Create;
    end;
  end;

  if Variants.VarIsArray(Result) then
  begin
    raise EVDSError.Create;
  end;

  // TODO: arrays
  // TODO: try cast IUnknown to IDispatch?
end;

procedure MessageException(E: SysUtils.Exception);
var
  TI: TypInfo.PTypeInfo;
begin
  TI := E.ClassInfo;
  if Assigned(TI) then
  begin
    Dialogs.MessageDlg(TypInfo.GetTypeData(TI).UnitName + '.' +
      E.ClassName + ': ' + E.Message, Dialogs.mtError, [Dialogs.mbOK], 0, Dialogs.mbOK);
  end
  else
  begin
    Dialogs.MessageDlg(E.ClassName + ': ' + E.Message, Dialogs.mtError, [Dialogs.mbOK], 0, Dialogs.mbOK);
  end;
end;

type
  TIStreamDynArray = array of ActiveX.IStream;
  TMarshalOleVariantMode = (movmNone, movmIUnknown, movmIDispatch);
  TMarshalOleVariantDynArray = array of TMarshalOleVariantMode;
  TGenericInvokeThread = class(Classes.TThread)
  private
    FPromise: vdsoleaut.Promises.IAsyncCallPromise;
    FDispParamsHolder: TDispParamsHolder;
    FMarshalOleVariantModes: TMarshalOleVariantDynArray;
    FMarshalOleVariantStreams: TIStreamDynArray;
    FItemStream: ActiveX.IStream; // IDispatch;
    FFlags: Word;
  protected
    function GetItemOnce: IDispatch;
    procedure MarshalParams;
    procedure UnmarshalParams;
  public
    constructor Create
      (const APromise: vdsoleaut.Promises.IAsyncCallPromise;
       const ADispParamsHolder: TDispParamsHolder;
       const AItem: IDispatch;
       AFlags: Word);
  end;

constructor TGenericInvokeThread.Create
  (const APromise: vdsoleaut.Promises.IAsyncCallPromise;
   const ADispParamsHolder: TDispParamsHolder;
   const AItem: IDispatch;
   AFlags: Word);
begin
  inherited Create(True);
  FPromise := APromise;
  FDispParamsHolder := ADispParamsHolder;
  // FItem := AItem;
  ComObj.OleCheck(ActiveX.CoMarshalInterThreadInterfaceInStream(IDispatch, AItem, FItemStream));
  MarshalParams;
  FFlags := AFlags;
  FreeOnTerminate := True;
  Resume;
end;

function TGenericInvokeThread.GetItemOnce: IDispatch;
begin
  ComObj.OleCheck(CoGetInterfaceAndReleaseStream(FItemStream, IDispatch, Result));
  FItemStream := nil;
end;

procedure TGenericInvokeThread.MarshalParams;
var
  I, L: Integer;
begin
  L := Length(FDispParamsHolder.ParametersForInvoke);
  SetLength(FMarshalOleVariantModes, L);
  SetLength(FMarshalOleVariantStreams, L);
  for I := 0 to L - 1 do
  begin
    case TVarData(FDispParamsHolder.ParametersForInvoke[I]).VType of
      varDispatch:
      begin
        FMarshalOleVariantModes[I] := movmIDispatch;
        ComObj.OleCheck(ActiveX.CoMarshalInterThreadInterfaceInStream(IDispatch, IDispatch(FDispParamsHolder.ParametersForInvoke[I]), FMarshalOleVariantStreams[I]));
        FDispParamsHolder.ParametersForInvoke[I] := Variants.Unassigned;
      end;
      varUnknown:
      begin
        FMarshalOleVariantModes[I] := movmIUnknown;
        ComObj.OleCheck(ActiveX.CoMarshalInterThreadInterfaceInStream(IUnknown, IUnknown(FDispParamsHolder.ParametersForInvoke[I]), FMarshalOleVariantStreams[I]));
        FDispParamsHolder.ParametersForInvoke[I] := Variants.Unassigned;
      end;
    else
      FMarshalOleVariantModes[I] := movmNone;
    end;
  end;
end;

procedure TGenericInvokeThread.UnmarshalParams;
var
  I, L: Integer;
  TempD: IDispatch;
  TempU: IUnknown;
begin
  L := Length(FDispParamsHolder.ParametersForInvoke);
  for I := 0 to L - 1 do
  begin
    case FMarshalOleVariantModes[I] of
      movmIDispatch:
      begin
        ComObj.OleCheck(ActiveX.CoGetInterfaceAndReleaseStream(FMarshalOleVariantStreams[I], IDispatch, TempD));
        FDispParamsHolder.ParametersForInvoke[I] := TempD;
        TempD := nil;
        FMarshalOleVariantStreams[I] := nil;
      end;
      movmIUnknown:
      begin
        ComObj.OleCheck(ActiveX.CoGetInterfaceAndReleaseStream(FMarshalOleVariantStreams[I], IUnknown, TempU));
        FDispParamsHolder.ParametersForInvoke[I] := TempU;
        TempU := nil;
        FMarshalOleVariantStreams[I] := nil;
      end;
      movmNone: // nothing to do
    end;
  end;
  SetLength(FMarshalOleVariantModes, 0);
  SetLength(FMarshalOleVariantStreams, 0);
end;

type
  TInvokeDoThread = class(TGenericInvokeThread)
  protected
    procedure Execute; override;
  end;

procedure TInvokeDoThread.Execute;
var
  ResultAsVariant: OleVariant;
  ExcepInfo: ActiveX.TExcepInfo;
begin
  ComObj.OleCheck(ActiveX.CoInitializeEx(nil, ActiveX.COINIT_MULTITHREADED));
  try
    UnmarshalParams;
    DispatchCheck(GetItemOnce.Invoke
     (FDispParamsHolder.MethodID, IID_NULL, ActiveX.STDOLE_LCID,
      FFlags, FDispParamsHolder.DispParams, @ResultAsVariant, @ExcepInfo, nil), ExcepInfo);

    if Variants.VarIsByRef(ResultAsVariant) then
    begin
      case ActiveX.TVariantArg(ResultAsVariant).vt of
        ActiveX.VT_BYREF or ActiveX.VT_UI1: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pbVal^;
        ActiveX.VT_BYREF or ActiveX.VT_I2: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).piVal^;
        ActiveX.VT_BYREF or ActiveX.VT_I4: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).plVal^;
        ActiveX.VT_BYREF or ActiveX.VT_R4: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pfltVal^;
        ActiveX.VT_BYREF or ActiveX.VT_R8: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pdblVal^;
        ActiveX.VT_BYREF or ActiveX.VT_BOOL: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pbool^;
        ActiveX.VT_BYREF or ActiveX.VT_ERROR: ResultAsVariant := Variants.VarAsError(ActiveX.TVariantArg(ResultAsVariant).pscode^);
        ActiveX.VT_BYREF or ActiveX.VT_CY: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pcyVal^;
        ActiveX.VT_BYREF or ActiveX.VT_DATE: ResultAsVariant := Variants.VarFromDateTime(ActiveX.TVariantArg(ResultAsVariant).pdate^);
        ActiveX.VT_BYREF or ActiveX.VT_BSTR: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pbstrVal^;
        ActiveX.VT_BYREF or ActiveX.VT_UNKNOWN: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).punkVal^;
        ActiveX.VT_BYREF or ActiveX.VT_DISPATCH: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pdispVal^;
        // ActiveX.VT_BYREF or ActiveX.VT_ARRAY: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pparray^;
        ActiveX.VT_BYREF or ActiveX.VT_VARIANT: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pvarVal^;
        // ActiveX.VT_BYREF or ActiveX.VT_DECIMAL:  (pdecVal: PDecimal);
        ActiveX.VT_BYREF or ActiveX.VT_I1: ResultAsVariant := ShortInt(ActiveX.TVariantArg(ResultAsVariant).pcVal^);
        ActiveX.VT_BYREF or ActiveX.VT_UI2: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).puiVal^;
        ActiveX.VT_BYREF or ActiveX.VT_UI4: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pulVal^;
        ActiveX.VT_BYREF or ActiveX.VT_INT: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).pintVal^;
        ActiveX.VT_BYREF or ActiveX.VT_UINT: ResultAsVariant := ActiveX.TVariantArg(ResultAsVariant).puintVal^;
      else
        raise EVDSError.Create;
      end;
    end;

    if Variants.VarIsArray(ResultAsVariant) then
    begin
      raise EVDSError.Create;
    end;

    // TODO: arrays
    // TODO: try cast IUnknown to IDispatch?
  except
    on E: SysUtils.Exception do
    begin
      MessageException(E);
    end;
  end;
  FPromise.PostValue(ResultAsVariant);
  ActiveX.CoUninitialize;
end;

function HLInvokeDoWithResultAsync(const Item: OleVariant; Flags: Word;
  const MethodName: AnsiString;
  const Parameters: array of AnsiString):
  vdsoleaut.Promises.IAsyncCallPromise;
var
  DispParamsHolder: TDispParamsHolder;
begin
  if not Variants.VarIsType(Item, varDispatch) then
  begin
    raise EVDSError.Create;
  end;

  DispParamsHolder := PrepareInvoke(IDispatch(Item), MethodName, Parameters);
  Result := vdsoleaut.Promises.AllocatePromise;
  TInvokeDoThread.Create(Result, DispParamsHolder, IDispatch(Item), Flags);
end;

procedure HLPutProperty(const Item: OleVariant; Flags: Word;
  const MethodName: AnsiString;
  const Parameters: array of AnsiString);
var
  DispParamsHolder: TDispParamsHolder;
  ExcepInfo: ActiveX.TExcepInfo;
begin
  if not Variants.VarIsType(Item, varDispatch) then
  begin
    raise EVDSError.Create;
  end;

  DispParamsHolder := PrepareInvoke(IDispatch(Item), MethodName, Parameters, True);

  DispatchCheck(IDispatch(Item).Invoke
   (DispParamsHolder.MethodID, IID_NULL, ActiveX.STDOLE_LCID,
    Flags, DispParamsHolder.DispParams, nil, @ExcepInfo, nil), ExcepInfo);
end;

type
  TPutPropertyThread = class(TGenericInvokeThread)
  protected
    procedure Execute; override;
  end;

procedure TPutPropertyThread.Execute;
var
  ExcepInfo: ActiveX.TExcepInfo;
begin
  ComObj.OleCheck(ActiveX.CoInitializeEx(nil, ActiveX.COINIT_MULTITHREADED));
  try
    UnmarshalParams;
    DispatchCheck(GetItemOnce.Invoke
     (FDispParamsHolder.MethodID, IID_NULL, ActiveX.STDOLE_LCID,
      FFlags, FDispParamsHolder.DispParams, nil, @ExcepInfo, nil), ExcepInfo);
  except
    on E: SysUtils.Exception do
    begin
      MessageException(E);
    end;
  end;
  FPromise.PostValue(Variants.Unassigned);
  ActiveX.CoUninitialize;
end;

function HLPutPropertyAsync(const Item: OleVariant; Flags: Word;
  const MethodName: AnsiString;
  const Parameters: array of AnsiString):
  vdsoleaut.Promises.IAsyncCallPromise;
var
  DispParamsHolder: TDispParamsHolder;
begin
  if not Variants.VarIsType(Item, varDispatch) then
  begin
    raise EVDSError.Create;
  end;

  DispParamsHolder := PrepareInvoke(IDispatch(Item), MethodName, Parameters, True);

  Result := vdsoleaut.Promises.AllocatePromise;
  TPutPropertyThread.Create(Result, DispParamsHolder, IDispatch(Item), Flags);
end;

function HLEnumerate(const Item: OleVariant): ActiveX.IEnumVARIANT;
var
  ExcepInfo: ActiveX.TExcepInfo;
  ResultAsVariant: OleVariant;
  DispParams: ActiveX.TDispParams;
begin
  if not Variants.VarIsType(Item, varDispatch) then
  begin
    raise EVDSError.Create;
  end;

  DispParams.rgvarg := nil;
  DispParams.rgdispidNamedArgs := nil;
  DispParams.cArgs := 0;
  DispParams.cNamedArgs := 0;

  DispatchCheck(IDispatch(Item).Invoke
   (ActiveX.DISPID_NEWENUM, IID_NULL, ActiveX.STDOLE_LCID,
    ActiveX.DISPATCH_PROPERTYGET, DispParams, @ResultAsVariant, @ExcepInfo, nil), ExcepInfo);

  Result := IUnknown(ResultAsVariant) as ActiveX.IEnumVARIANT;
end;

function HLNext(const Item: ActiveX.IEnumVARIANT): AnsiString;
var
  Status: HRESULT;
  Fetched: LongWord;
  ResultAsVariant: OleVariant;
begin
  if not Assigned(Item) then
  begin
    raise EVDSError.Create;
  end;

  Status := Item.Next(1, ResultAsVariant, Fetched);
  if Status = S_FALSE then
  begin
    Result := '';
    Exit;
  end;

  ComObj.OleCheck(Status);

  if Fetched < 1 then
  begin
    Result := '';
    Exit;
  end
  else
  begin
    Result := AllocateOleVariant(ResultAsVariant);
    Exit;
  end;
end;

function HLIsCompleted(const Item: vdsoleaut.Promises.IAsyncCallPromise): AnsiString;
begin
  if not Assigned(Item) then
  begin
    raise EVDSError.Create;
  end;

  if Item.IsCompleted then
  begin
    Result := 'TRUE'
  end
  else
  begin
    Result := ''
  end;
end;

function HLIsRunning(const Item: vdsoleaut.Promises.IAsyncCallPromise): AnsiString;
begin
  if not Assigned(Item) then
  begin
    raise EVDSError.Create;
  end;

  if not Item.IsCompleted then
  begin
    Result := 'TRUE'
  end
  else
  begin
    Result := ''
  end;
end;

function HLPromiseValue(const Item: vdsoleaut.Promises.IAsyncCallPromise): OleVariant;
begin
  if not Assigned(Item) then
  begin
    raise EVDSError.Create;
  end;

  if not Item.IsCompleted then
  begin
    raise EVDSError.Create;
  end
  else
  begin
    Result := Item.Value;
  end;
end;

// ---------------------------------------

procedure HLCommandProc(const Params: Types.TStringDynArray);
begin
  if Length(Params) < 1 then
  begin
    raise EVDSError.Create;
  end;

  if Params[0] = 'OLE' then
  begin
    if Length(Params) < 2 then
    begin
      raise EVDSError.Create;
    end;

    if Params[1] = 'CLOSE' then
    begin
      if Length(Params) < 3 then
      begin
        raise EVDSError.Create;
      end;

      if Params[2] = 'variant' then
      begin
        if Length(Params) < 4 then
        begin
          raise EVDSError.Create;
        end;

        if Params[3] = 'ALL' then
        begin
          DeallocateOleVariants;
          Exit;
        end
        else
        begin
          DeallocateOleVariant(StrToInt(Params[3]));
          Exit;
        end;
      end
      else
      if Params[2] = 'enumerator' then
      begin
        if Length(Params) < 4 then
        begin
          raise EVDSError.Create;
        end;

        if Params[3] = 'ALL' then
        begin
          DeallocateOleEnumerators;
          Exit;
        end
        else
        begin
          DeallocateOleEnumerator(StrToInt(Params[3]));
          Exit;
        end;
      end
      else
      if Params[2] = 'promise' then
      begin
        if Length(Params) < 4 then
        begin
          raise EVDSError.Create;
        end;

        if Params[3] = 'ALL' then
        begin
          vdsoleaut.Promises.DeallocatePromises;
          Exit;
        end
        else
        begin
          vdsoleaut.Promises.DeallocatePromise(StrToInt(Params[3]));
          Exit;
        end;
      end
      else
      begin
        raise EVDSError.Create;
      end;
    end
    else if Params[1] = 'INVOKE' then
    begin
      if Length(Params) < 3 then
      begin
        raise EVDSError.Create;
      end;

      if Params[2] = 'variant' then
      begin
        if Length(Params) < 5 then
        begin
          raise EVDSError.Create;
        end;

        if Params[4] = 'DO' then
        begin
          if Length(Params) < 6 then
          begin
            raise EVDSError.Create;
          end;

          HLInvokeDo(AllocatedOleVariant(StrToInt(Params[3])),
                     ActiveX.DISPATCH_METHOD, Params[5],
                     Slice(BeginSlice(Params, 6)^, Length(Params) - 6));
          Exit;
        end
        else
        begin
          raise EVDSError.Create;
        end;
      end
      else
      begin
        raise EVDSError.Create;
      end;
    end else
    if Params[1] = 'MODIFY' then
    begin
      if Length(Params) < 6 + 2 then
      begin
        raise EVDSError.Create;
      end;

      if (Params[2] = 'variant') and (Params[4] = 'PUT') then
      begin
        HLPutProperty(AllocatedOleVariant(StrToInt(Params[3])),
                      ActiveX.DISPATCH_PROPERTYPUT, Params[5],
                      Slice(BeginSlice(Params, 6)^, Length(Params) - 6));
        Exit;
      end;

      raise EVDSError.Create;
    end else
    begin
      raise EVDSError.Create;
    end;
  end
  else
  begin
    raise EVDSError.Create;
  end;

  raise EVDSError.Create;
end;

function HLFunctionProc(const Args: Types.TStringDynArray): AnsiString;
begin
  if Length(Args) < 1 then
  begin
    raise EVDSError.Create;
  end;

  if Args[0] = 'OLE' then
  begin
    if Length(Args) < 2 then
    begin
      raise EVDSError.Create;
    end;

    if Args[1] = 'variant' then
    begin
      if Length(Args) < 4 then
      begin
        raise EVDSError.Create;
      end;

      if Args[3] = 'TYPEOF' then
      begin
        Result := HLTypeOf(AllocatedOleVariant(StrToInt(Args[2])));
        Exit;
      end
      else if Args[3] = 'TOUTF' then
      begin
        Result := HLToString(AllocatedOleVariant(StrToInt(Args[2])), False);
        Exit;
      end
      else if Args[3] = 'TOSTRING' then
      begin
        Result := HLToString(AllocatedOleVariant(StrToInt(Args[2])), True);
        Exit;
      end
      else if Args[3] = 'DO' then
      begin
        if Length(Args) < 6 then
        begin
          raise EVDSError.Create;
        end;

        if Args[4] = 'promise' then
        begin
          Result := IntToStr(HLInvokeDoWithResultAsync(AllocatedOleVariant
           (StrToInt(Args[2])), ActiveX.DISPATCH_METHOD, Args[5],
            Slice(BeginSlice(Args, 6)^, Length(Args) - 6)).Index);
          Exit;
        end
        else
        begin
          Result := ProduceResult(HLInvokeDoWithResult(AllocatedOleVariant
           (StrToInt(Args[2])), ActiveX.DISPATCH_METHOD, Args[5],
            Slice(BeginSlice(Args, 6)^, Length(Args) - 6)), Args[4]);
          Exit;
        end;
      end
      else if Args[3] = 'GET' then
      begin
        if Length(Args) < 6 then
        begin
          raise EVDSError.Create;
        end;

        if Args[4] = 'promise' then
        begin
          Result := IntToStr(HLInvokeDoWithResultAsync(AllocatedOleVariant
           (StrToInt(Args[2])), ActiveX.DISPATCH_PROPERTYGET, Args[5],
            Slice(BeginSlice(Args, 6)^, Length(Args) - 6)).Index);
          Exit;
        end
        else
        begin
          Result := ProduceResult(HLInvokeDoWithResult(AllocatedOleVariant
           (StrToInt(Args[2])), ActiveX.DISPATCH_PROPERTYGET, Args[5],
            Slice(BeginSlice(Args, 6)^, Length(Args) - 6)), Args[4]);
          Exit;
        end;
      end
      else if Args[3] = 'PUT' then
      begin
        if Length(Args) < 6 + 2 then
        begin
          raise EVDSError.Create;
        end;

        if Args[4] <> 'promise' then
        begin
          raise EVDSError.Create;
        end;

        Result := IntToStr(HLPutPropertyAsync(AllocatedOleVariant
         (StrToInt(Args[2])), ActiveX.DISPATCH_PROPERTYPUT, Args[5],
          Slice(BeginSlice(Args, 6)^, Length(Args) - 6)).Index);
        Exit;
      end
      else if Args[3] = 'ENUMERATE' then
      begin
        Result := AllocateOleEnumerator(HLEnumerate((AllocatedOleVariant(StrToInt(Args[2])))));
        Exit;
      end
      else if (Args[3] = 'STRING') or (Args[3] = 'UTF') or (Args[3] = 'INTEGER') or
        (Args[3] = 'BOOLEAN') or (Args[3] = 'REAL') or (Args[3] = 'OBJECT') or
        (Args[3] = 'THREADEDOBJECT') or (Args[3] = 'VARIANT') then
      begin
        Result := AllocateOleVariant(CreateOleVariant(Args[2], Args[3]));
        Exit;
      end
      else
      begin
        raise EVDSError.Create;
      end;
    end
    else if Args[1] = 'enumerator' then
    begin
      if Length(Args) < 4 then
      begin
        raise EVDSError.Create;
      end;

      if Args[3] = 'NEXT' then
      begin
        Result := HLNext(AllocatedOleEnumerator(StrToInt(Args[2])));
        Exit;
      end
      else
      begin
        raise EVDSError.Create;
      end;
    end
    else if Args[1] = 'promise' then
    begin
      if Length(Args) < 4 then
      begin
        raise EVDSError.Create;
      end;

      if Args[3] = 'ISCOMPLETED' then
      begin
        Result := HLIsCompleted(vdsoleaut.Promises.AllocatedPromise(StrToInt(Args[2])));
        Exit;
      end
      else if Args[3] = 'ISRUNNING' then
      begin
        Result := HLIsRunning(vdsoleaut.Promises.AllocatedPromise(StrToInt(Args[2])));
        Exit;
      end
      else if Args[3] = 'VALUE' then
      begin
        Result := AllocateOleVariant(HLPromiseValue(vdsoleaut.Promises.AllocatedPromise(StrToInt(Args[2]))));
        Exit;
      end
      else if Args[3] = 'TOSTRING' then
      begin
        Result := HLToString(HLPromiseValue(vdsoleaut.Promises.AllocatedPromise(StrToInt(Args[2]))), True);
        Exit;
      end
      else if Args[3] = 'TOUTF' then
      begin
        Result := HLToString(HLPromiseValue(vdsoleaut.Promises.AllocatedPromise(StrToInt(Args[2]))), False);
        Exit;
      end
      else
      begin
        raise EVDSError.Create;
      end;
    end
    else
    begin
      raise EVDSError.Create;
    end;
  end
  else
  begin
    raise EVDSError.Create;
  end;

  raise EVDSError.Create;
end;

initialization
  FormatSettings.DecimalSeparator := '.';
finalization
  if InitializedOle then
  begin
    ActiveX.OleUninitialize;
    InitializedOle := False;
  end;
end.
