package jp.sourceforge.masasa.architecture.framework.validation;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.masasa.architecture.framework.validation.annotation.ValidateResolver;
import jp.sourceforge.masasa.architecture.framework.validation.resource.PropertyManager;
import jp.sourceforge.masasa.architecture.framework.validation.util.MethodAccessUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.validator.ValidatorException;

/**
 * バリデーションを実行する抽象クラス.
 *
 * @author masasa.
 * @param <T>
 *            チェック対象のBean.
 */
public abstract class AbstractValidator<T> {

    /** logger. */
    protected static final Log LOGGER = LogFactory.getLog(AbstractValidator.class);
    /** チェック対象のBean. */
    protected T targetBean;
    /** 入力チェック結果(全結果格納). */
    protected ValidateResults results = new ValidateResults();
    /** 入力チェック結果(エラーのみ格納). */
    protected ValidateResults errors = new ValidateResults();
    /** 総行数. */
    protected int totalLine;

    /**
     * 単一Beanに対してバリデーションを実行します.
     *
     * @param t
     *            バリデーション対象オブジェクト.
     */
    public void validate(final T t) {
        List<T> list = new ArrayList<T>();
        list.add(t);
        validate(list);
    }

    /**
     * リストBeanに対してバリデーションを実行します.
     *
     * @param list
     *            バリデーション対象リスト.
     */
    public void validate(final List<T> list) {
        validate((T[]) list.toArray());
    }

    /**
     * 配列Beanに対してバリデーションを実行します.
     *
     * @param array
     *            バリデーション対象配列.
     */
    public void validate(final T[] array) {
        totalLine = array.length;
        if (this.totalLine > 0) this.targetBean = array[0];
        loggingStart();
        try {
            // バリデーションを実行する
            execute(array);
        } catch (ValidatorException e) {
            loggingTerminate(e);
        }
        loggingResult();
        loggingEnd();
    }

    /**
     * バリデーションの実処理を実行します.<br>
     * カスタマイズしたい場合は本メソッドをオーバーライドしてください.
     *
     * @param array
     *            バリデーション対象配列.
     * @throws ValidatorException
     *             ValidatorException.
     */
    protected void execute(final T[] array) throws ValidatorException {
        Object lineObject;
        ValidateResolver resolver;
        String targetValue;
        ValidateResult result;
        for (int cnt = 0; cnt < this.totalLine; cnt++) {
            lineObject = array[cnt];
            for (Field field : lineObject.getClass().getDeclaredFields()) {
                for (Annotation ann : field.getAnnotations()) {
                    resolver = ann.annotationType().getAnnotation(ValidateResolver.class);
                    if (resolver == null) {
                        // バリデーションチェックルールが存在しない
                        // または未定義の場合
                        throw new ValidatorException(
                                PropertyManager.getMessage(ValidateProperties.UNKNOWN_VALIDATION_RULE));
                    }
                    targetValue = MethodAccessUtil.doInvoke(lineObject, field);
                    boolean isValid = resolver.value().getInstance().validate(ann, targetValue, lineObject);
                    result = new ValidateResult(cnt, field.getName(), targetValue, resolver.value().getName(), isValid);
                    // 入力チェック結果(全結果格納)
                    results.add(result);
                    if (isValid) continue;
                    // 入力チェック結果(エラーのみ格納)
                    this.errors.add(result);
                    if (this.errors.isOverTerminate()) {
                        // 最大エラー件数を超えた場合
                        throw new ValidatorException(PropertyManager.getMessage(ValidateProperties.TERMINATE_ERROR));
                    }
                }
            }
        }
    }

    /**
     * 開始ログを出力します.
     */
    protected abstract void loggingStart();

    /**
     * 強制終了時ログを出力します.
     *
     * @param e
     *            ValidatorException.
     */
    protected abstract void loggingTerminate(ValidatorException e);

    /**
     * 入力チェック結果をログ出力します.
     */
    protected abstract void loggingResult();

    /**
     * 終了ログを出力します.
     */
    protected abstract void loggingEnd();

    /**
     * 入力チェック結果(全結果格納)を返却します.
     *
     * @return 入力チェック結果(全結果格納).
     */
    public ValidateResults getResults() {
        return this.results;
    }

    /**
     * 入力チェック結果(エラーのみ格納)を返却します.
     *
     * @return 入力チェック結果(エラーのみ格納).
     */
    public ValidateResults getErrors() {
        return this.errors;
    }

    /**
     * 入力チェック結果(エラーのみ格納)の存在状態を返却します.
     *
     * @return 入力チェック結果(エラーのみ格納)の存在状態.
     *         <ul>
     *         <li><code>true</code>:エラーなし.</li>
     *         <li><code>false</code>:エラーあり.</li>
     *         </ul>
     */
    public boolean isValid() {
        return this.errors.isEmpty();
    }

}
