package net.osdn.util.sql;

/** SQL実行ハンドラー
 * 
 * DataSourceにSQL実行ハンドラーを設定すると、DataSource または Transaction を通して実行されるSQLの
 * 各局面(onBeforeExecution, onAfterReturning, onAfterThrowing, onAfterExecution)でSQL実行ハンドラーが呼び出されるようになります。
 * SQL実行ハンドラーはログ出力するのに役立ちます。
 * 
 * DataSource.getConnectionメソッドなどで直接取得した接続を使ってSQLを実行した場合は、SQL実行ハンドラーは呼び出されないことに注意してください。
 *
 */
public interface SqlExecutionHandler {
	
	/** SQL実行前に呼び出されます。
	 * 
	 * SQL配列の要素数は通常 1 です。 内部で executeBatch が実行され、
	 * 複数のSQLが一度にデータベースに送信される場合、SQL配列の要素数は 2 以上になることがあります。
	 * DataSource.update(Object... objects) を呼び出した場合などがこれに該当します。
	 * 
	 * SQL実行前であるため、Sqlオブジェクトの getAffectedRows は -1 を返し、getResult は null を返します。
	 * 
	 * @param transaction トランザクション
	 * @param array SQL配列
	 */
	void onBeforeExecution(Transaction transaction, Sql[] array);
	
	/** SQL実行後に呼び出されます。
	 * SQL実行で例外がスローされた場合も、この onAfterExectuion は呼び出されます。
	 * 
	 * SQL配列の要素数は通常 1 です。 内部で executeBatch が実行され、
	 * 複数のSQLが一度にデータベースに送信される場合、SQL配列の要素数は 2 以上になることがあります。
	 * DataSource.update(Object... objects) を呼び出した場合などがこれに該当します。
	 * 
	 * @param transaction トランザクション
	 * @param array SQL配列
	 */
	void onAfterExecution(Transaction transaction, Sql[] array);
	
	/** SQL実行から復帰した後に呼び出されます。
	 * SQL実行で例外がスローされた場合、この onAfterReturning は呼び出されません。
	 * 
	 * Sqlオブジェクトの getAffectedRows はSQL実行によって影響を受けた行数を返します。
	 * getResult はSQL実行による結果セットを保持しています。ただし、INSERT/UPDATE/DELETEなど結果セットを返さない場合、getResult は null を返します。
	 * getResult が返す型は CachedRowSet ですが、これに対して操作をおこなうとSQL呼び出し元に返されるCachedRowSetに影響することに注意してください。
	 * 
	 * SQL配列の要素数は通常 1 です。 内部で executeBatch が実行され、
	 * 複数のSQLが一度にデータベースに送信される場合、SQL配列の要素数は 2 以上になることがあります。
	 * DataSource.update(Object... objects) を呼び出した場合などがこれに該当します。
	 * 
	 * @param transaction トランザクション
	 * @param array SQL配列
	 */
	void onAfterReturning(Transaction transaction, Sql[] array);
	
	/** SQL実行で例外がスローされた後に呼び出されます。
	 * 
	 * SQL実行は成功していないため、Sqlオブジェクトの getAffectedRows は -1 を返し、getResult は null を返します。
	 * 
	 * SQL配列の要素数は通常 1 です。 内部で executeBatch が実行され、
	 * 複数のSQLが一度にデータベースに送信される場合、SQL配列の要素数は 2 以上になることがあります。
	 * DataSource.update(Object... objects) を呼び出した場合などがこれに該当します。
	 * 
	 * @param transaction トランザクション
	 * @param array SQL配列
	 * @param e スローされた例外のインスタンス
	 */
	void onAfterThrowing(Transaction transaction, Sql[] array, Throwable e);
}
