package net.osdn.util.jersey;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.Date;
import java.util.TimeZone;

import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.ws.rs.ext.Provider;

import com.fasterxml.jackson.databind.util.ISO8601Utils;

import net.osdn.util.jersey.fastpack.BytePacker;

@Provider
@Produces(MediaType.APPLICATION_JSON)
public class ResultSetMessageBodyWriter implements MessageBodyWriter<Object> {
	
	@Context
	private HttpHeaders requestHeaders;
	
	@Override
	public long getSize(Object rs, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
		return -1;
	}

	@Override
	public boolean isWriteable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
		return ResultSet.class.isAssignableFrom(type);
	}

	@Override
	public void writeTo(Object obj, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, Object> httpHeaders, OutputStream entityStream) throws IOException, WebApplicationException {
		ResultSet rs = (ResultSet)obj;

		// リクエストに X-DirectSQLite ヘッダーが含まれている場合は JSON形式ではなく FastPack形式で出力します。
		boolean isDirectSQLite = (requestHeaders != null && requestHeaders.getHeaderString("X-DirectSQLite") != null);
		
		int rowCount = getRowCount(rs);
		if(rowCount >= 0) {
			httpHeaders.add("X-RowCount", rowCount);
		}
		
		try {
			int columnCount = rs.getMetaData().getColumnCount();
			httpHeaders.add("X-ColumnCount", columnCount);
			
			if(isDirectSQLite) {
				httpHeaders.add("Content-Type", "application/octet-stream");
				writeFastPack(entityStream, rs);
			} else {
				writeJson(entityStream, rs);
			}

			//CachedRowSetなどStatementがnullになっているケースもあります。 
			Statement st = rs.getStatement();
			Connection cn = null;
			if(st != null) {
				cn = st.getConnection();
			}
			rs.close();
			if(st != null) {
				st.close();
			}
			if(cn != null) {
				cn.close();
			}
		} catch(SQLException e) {
			e.printStackTrace();
		}
	}
	
	/** 結果セットをJSON形式で出力します。
	 * 
	 * @param writer
	 * @param rs
	 * @param columnCount
	 * @param columnNames
	 * @param columnTypes
	 * @throws IOException
	 * @throws SQLException
	 */
	private void writeJson(OutputStream entityStream, ResultSet rs) throws IOException, SQLException {
		ResultSetMetaData meta = rs.getMetaData();
		int columnCount = meta.getColumnCount();
		String[] columnNames = new String[columnCount + 1];
		int[] columnTypes = new int[columnCount + 1];
		for(int i = 1; i <= columnCount; i++) {
			columnNames[i] = meta.getColumnLabel(i);
			columnTypes[i] = meta.getColumnType(i);
		}
		BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(entityStream, "UTF-8"));
		writer.write("[\n");
		boolean isNotFirst = false;
		while(rs.next()) {
			if(isNotFirst) {
				writer.write(",\n");
			} else {
				isNotFirst = true;
			}
			writer.write("{ ");
			for(int i = 1; i <= columnCount; i++) {
				writer.write('"');
				writer.write(columnNames[i]);
				writer.write("\":\"");
				switch(columnTypes[i]) {
				case Types.DATE:
				case Types.TIME:
				case Types.TIMESTAMP:
					Date date = rs.getTimestamp(i);
					String s = ISO8601Utils.format(date, true, TimeZone.getDefault());
					writer.write(s);
					break;
				case Types.BIT:
				case Types.BOOLEAN:
					writer.write(rs.getBoolean(i) ? "true" : "false");
					break;
				default:
					writer.write(rs.getString(i));
				}
				writer.write("\"");
				if(i + 1 <= columnCount) {
					writer.write(", ");
				}
			}
			writer.write(" }");
		}
		writer.write("\n]");
		writer.close();
	}
	
	private void writeFastPack(OutputStream entityStream, ResultSet rs) throws IOException, SQLException {
		ResultSetMetaData meta = rs.getMetaData();
		int columnCount = meta.getColumnCount();
		String[] columnNames = new String[columnCount + 1];
		int[] columnTypes = new int[columnCount + 1];
		int columnType;
		for(int i = 1; i <= columnCount; i++) {
			columnNames[i] = meta.getColumnLabel(i);
			columnType = meta.getColumnType(i);
			switch(columnType) {
			case Types.BIT:
			case Types.BOOLEAN:
				columnTypes[i] = Types.BOOLEAN;
				break;
			case Types.TINYINT:
			case Types.SMALLINT:
			case Types.INTEGER:
			case Types.BIGINT:
				columnTypes[i] = Types.BIGINT;
				break;
			case Types.REAL:
			case Types.FLOAT:
			case Types.DOUBLE:
				columnTypes[i] = Types.DOUBLE;
				break;
			case Types.NUMERIC:
			case Types.DECIMAL:
				if(meta.getScale(i) == 0) {
					if(meta.getPrecision(i) <= 18) {
						columnTypes[i] = Types.BIGINT;
					} else {
						columnTypes[i] = Types.DOUBLE;
					}
				} else {
					columnTypes[i] = Types.DOUBLE;
				}
				break;
			case Types.DATE:
			case Types.TIME:
			case Types.TIMESTAMP:
				columnTypes[i] = Types.TIMESTAMP;
				break;
			default:
				columnTypes[i] = columnType;
			}
		}
		BytePacker packer = new BytePacker(entityStream);
		long l;
		double d;
		String s;
		boolean b;
		Date dt;

		//先頭の要素はlong型でカラム数です。
		packer.writeLong(columnCount);

		//続いてカラム名(String型)とカラムタイプ(long型)を交互に格納していきます。
		for(int i = 1; i <= columnCount; i++) {
			packer.writeString(columnNames[i]);
			switch(columnTypes[i]) {
			case Types.TIMESTAMP:
			case Types.BOOLEAN:
			case Types.BIGINT:
				packer.writeLong((long)'L');
				break;
			case Types.DOUBLE:
				packer.writeLong((long)'D');
				break;
			default:
				packer.writeLong((long)'S');
				break;
			}
		}

		//後は要素の値を連続で格納していきます。行区切りを示すデータなどはありません。
		while(rs.next()) {
			for(int i = 1; i <= columnCount; i++) {
				switch(columnTypes[i]) {
				case Types.TIMESTAMP:
					dt = rs.getTimestamp(i);
					if(rs.wasNull()) {
						packer.writeNull();
					} else {
						l = dt.getTime();
						packer.writeLong(l);
					}
					break;
				case Types.BOOLEAN:
					b = rs.getBoolean(i);
					if(rs.wasNull()) {
						packer.writeNull();
					} else {
						l = b ? 1L : 0L;
						packer.writeLong(l);
					}
					break;
				case Types.BIGINT:
					l = rs.getLong(i);
					if(rs.wasNull()) {
						packer.writeNull();
					} else {
						packer.writeLong(l);
					}
					break;
				case Types.DOUBLE:
					d = rs.getDouble(i);
					if(rs.wasNull()) {
						packer.writeNull();
					} else {
						packer.writeDouble(d);
					}
					break;
				default:
					s = rs.getString(i);
					if(rs.wasNull()) {
						packer.writeNull();
					} else {
						packer.writeString(s);
					}
					break;
				}
			}
		}
		entityStream.close();
	}
	
	/** 結果セットの行数を取得します。
	 * 
	 * @param rs 結果セット
	 * @return 結果セットの行数。サポートされていない場合は -1 が返されます。
	 */
	private static int getRowCount(ResultSet rs) {
		int rowCount = -1;
		try {
			rs.last();
			rowCount = rs.getRow();
			rs.beforeFirst();
		} catch (SQLException e) {}
		return rowCount;
	}
}
