<?php

  assert_options( ASSERT_BAIL, 1 );
  ini_set( 'session.name', 'sid' );
  ini_set( 'session.use_cookies', 0 );
  ini_set( 'session.cache_expire', 60*24*7 );

  require_once( 'browser_config.php' );

  session_start();
  init_openssl_certs();

  ///////////////////////////////////////////////////////////////////////////////
  // [parse_track_xml]
  ///////////////////////////////////////////////////////////////////////////////
  function parse_track_xml( $xmlstr ) {
    $dom = domxml_open_mem( $xmlstr );
    if( ! $dom ) { return NULL; }
    $root = $dom->document_element();
    if( $root->node_name() != 'track' ) { return NULL; }
    $ret = array();
    $ret['name'] = $root->get_attribute('name');
    $ret['comment'] = $root->get_attribute('comment');
    $ret['description_url'] = $root->get_attribute('description_url');
    $ret['optattr'] = $root->get_attribute('optattr');
    if( $root_child = $root->first_child() ) {
      do {
        if( $root_child->node_name() == 'layers' ) {
          $ret['layers'] = array();
          if( $layers_child = $root_child->first_child() ) {
            do {
              if( $layers_child->node_name() == 'layer' ) {
                $layer =  array();
                $layer['kind'] = $layers_child->get_attribute('kind');
                $layer['url'] = expand_metaurl( $layers_child->get_attribute('url'), $track_base_url );
                $ret['layers'][$layer['kind']] = $layer;
              }
            } while( $layers_child = $layers_child->next_sibling() );
          }
        }
        elseif( $root_child->node_name() == 'species' ) {
          $ret['species'] = array();
          if( $species_child = $root_child->first_child() ) {
            do {
              if( $species_child->node_name() == 'species' ) {
                $species =  array();
                $species['name'] = $species_child->get_attribute('name');
                if( $species_grandchild = $species_child->first_child() ) {
                  do {
                    if( $species_grandchild->node_name() == 'revisions' ) {
                      $species['revisions'] = array();
                      if( $revisions_child = $species_grandchild->first_child() ) {
                        do {
                          if( $revisions_child->node_name() == 'revision' ) {
                            $species['revisions'][] = $revisions_child->get_attribute('name');
                          }
                        } while( $revisions_child = $revisions_child->next_sibling() );
                      }
                    }
                  } while( $species_grandchild = $species_grandchild->next_sibling() );
                }
                $ret['species'][] = $species;
              }
            } while( $species_child = $species_child->next_sibling() );
          }
        }
      } while( $root_child = $root_child->next_sibling() );
    }
    return $ret;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [get_first_text_value]
  ///////////////////////////////////////////////////////////////////////////////
  function get_first_text_value( $domelem ) {
    $child = $domelem->first_child();
    if( ! $child ) { return NULL; }
    do {
      if( $child->node_name() == '#text' ) {
        return $child->node_value();
      }
    } while( $child = $child->next_sibling() );
    return NULL;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [http_get]
  ///////////////////////////////////////////////////////////////////////////////
  function http_get( $url, $timeout=1 ) {
    $url = expand_metaurl($url, null);
    $ch = curl_init( $url );
    curl_setopt( $ch, CURLOPT_USERAGENT, 'UTGB' );
    curl_setopt( $ch, CURLOPT_TIMEOUT, $timeout );
    ob_start();
    curl_exec( $ch );
    if( ! curl_errno( $ch ) ) {
      curl_close( $ch );
      $retrievedhtml = ob_get_contents();
      ob_end_clean();
      return $retrievedhtml;
    }
    else {
      curl_close( $ch );
      ob_end_clean();
      return NULL;
    }
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [http_get_with_info]
  ///////////////////////////////////////////////////////////////////////////////
  function http_get_with_info( $url, $tileout=1 ) {
    $ch = curl_init( $url );
    curl_setopt( $ch, CURLOPT_USERAGENT, 'UTGB' );
    curl_setopt( $ch, CURLOPT_HEADER, $tileout );
    ob_start();
    curl_exec( $ch );
    if( ! curl_errno( $ch ) ) {
      $info = curl_getinfo( $ch );
      curl_close( $ch );
      $retrieved = ob_get_contents();
      ob_end_clean();
      $separated = explode( "\r\n\r\n", $retrieved );
      $separatedheader = explode( "\r\n", array_shift($separated) );
      $retrievedstatus = array_shift( $separatedheader );
      $retrievedheader = array();
      foreach( $separatedheader as $sh ) {
        $sep = explode( ': ', $sh );
        $name = array_shift( $sep );
        $retrievedheader[$name] = implode( ': ', $sep );
      }
      $retrievedhtml   = implode( "\r\n\r\n", $separated );
      return array( 'status' => $retrievedstatus, 'header' => $retrievedheader, 'contents' => $retrievedhtml, 'info' => $info );
    }
    else {
      curl_close( $ch );
      ob_end_clean();
      return NULL;
    }
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [query2string]
  //   Tv : Azz`̃NG[𕶎ɕϊ
  //      :  ; array( 'a' => 'b', 'c' => 'd' )
  //          o ; 'a=b&c=d'
  ///////////////////////////////////////////////////////////////////////////////
  function query2string( $query ) {
    $ret = '';
    foreach( $query as $key => $val ) {
      $ret .= $key.'='.$val.'&';
    }
    return substr( $ret, 0, -1 );
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [getmovedquery]
  ///////////////////////////////////////////////////////////////////////////////
  function getmovedquery( $query, $distance, $thislen ) {
    if( is_numeric($query['start']) && is_numeric($query['end']) ) {
      if( $query['start'] <= $query['end'] ) {
        $query['start'] += $distance;
        $query['end']   += $distance;
      }
      else {
        $query['start'] -= $distance;
        $query['end']   -= $distance;
      }
      if( $query['start'] < 0 ) {
        $query['end'] -= $query['start'];
        $query['start'] = 0;
      }
      if( $query['end'] < 0 ) {
        $query['start'] -= $query['end'];
        $query['end'] = 0;
      }
      if( $thislen-1 < $query['end'] ) {
        $query['start'] -= $query['end'] - ($thislen - 1);
        $query['end'] = $thislen-1;
      }
      if( $thislen-1 < $query['start'] ) {
        $query['end'] -= $query['start'] - ($thislen - 1);
        $query['start'] = $thislen-1;
      }
      $query['start'] = round($query['start']);
      $query['end']   = round($query['end']);
    }
    return $query;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [getzoomedquery]
  ///////////////////////////////////////////////////////////////////////////////
  function getzoomedquery( $query, $zoomrate, $thislen ) {
    if( is_numeric($query['start']) && is_numeric($query['end']) && is_numeric($query['width']) ) {
      $center = ( $query['start'] + $query['end'] ) / 2;
      $width = $query['width'];
      if( $query['start'] <= $query['end'] ) {
        $query['start'] = $center - $width/$zoomrate/2;
        $query['end']   = $center + $width/$zoomrate/2;
      }
      else {
        $query['start'] = $center + $width/$zoomrate/2;
        $query['end']   = $center - $width/$zoomrate/2;
      }
      if( $query['start'] < 0 ) {
        $query['end'] -= $query['start'];
        $query['start'] = 0;
      }
      if( $query['end'] < 0 ) {
        $query['start'] -= $query['end'];
        $query['end'] = 0;
      }
      $query['start'] = round($query['start']);
      $query['end']   = round($query['end']);
    }
    return $query;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [getreversedquery]
  ///////////////////////////////////////////////////////////////////////////////
  function getreversedquery( $query ) {
    if( is_numeric($query['start']) && is_numeric($query['end']) ) {
      $tmp            = $query['start'];
      $query['start'] = $query['end'];
      $query['end']   = $tmp;
    }
    return $query;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [parse_optattr]
  //  : optattr 
  //        Ex. 'fromSpecies/select[medaka,fugu,zebrafish](disp,color,sort),matchRate/real[0.7,1.0](color,ubound,lbound)'
  // o : optattr 
  //        Ex. array( array( 'name' => 'fromSpecies',
  //                          'type' => 'select',
  //                          'candidates' => array('medaka','fugu','zebrafish'),
  //                          'operation' => array('disp','color','sort') ),
  //                   array( 'matchRate' => 'fromSpecies',
  //                          'type' => 'real',
  //                          'range' => array( 0.7, 1.0 ),
  //                          'operation' => array('color','ubound','lbound') ) )
  ///////////////////////////////////////////////////////////////////////////////
  function parse_optattr( $optattr_str ) {
    $ret = array();
    while( preg_match( '/^([^\/]+)\/([^\[\]]+)\[([^\[\]]+)\]\(([^\[\]]+)\)\,?/', $optattr_str, $matches ) ) {
      $optattr = array();
      $optattr['name'] = $matches[1];
      $optattr['type'] = $matches[2];
      if( $optattr['type'] == 'select' ) {
        $optattr['candidates'] = explode( ',', $matches[3] );
      }
      elseif( $optattr['type'] == 'real' ) {
        $optattr['range'] = explode( ',', $matches[3] );
      }
      $optattr['operation'] = explode( ',', $matches[4] );
      $ret[] = $optattr;
      $optattr_str = substr( $optattr_str, strlen( $matches[0] ) );
    }
    if( count($ret) == 0 ) {
      return null;
    }
    else {
      return $ret;
    }
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [print_color_selecter]
  ///////////////////////////////////////////////////////////////////////////////
  function print_color_selecter( $name, $color=null, $id=nulll, $disabled=null ) {

    $colors = array( array( 'color' => '(0,0,0)',       'name' => 'Black' ),
                     array( 'color' => '(255,0,0)',     'name' => 'Red' ),
                     array( 'color' => '(255,255,0)',   'name' => 'Yellow' ),
                     array( 'color' => '(0,255,0)',     'name' => 'Green' ),
                     array( 'color' => '(0,255,255)',   'name' => 'Cyan' ),
                     array( 'color' => '(0,0,255)',     'name' => 'Blue' ),
                     array( 'color' => '(255,0,255)',   'name' => 'Violet' ),
                     array( 'color' => '(255,255,255)', 'name' => 'White' ),
                     array( 'color' => '(153,102,0)',   'name' => 'Chr1' ),
                     array( 'color' => '(102,102,0)',   'name' => 'Chr2' ),
                     array( 'color' => '(153,153,0)',   'name' => 'Chr3' ),
                     array( 'color' => '(204,0,0)',     'name' => 'Chr4' ),
                     array( 'color' => '(254,0,0)',     'name' => 'Chr5' ),
                     array( 'color' => '(254,0,204)',   'name' => 'Chr6' ),
                     array( 'color' => '(254,204,204)', 'name' => 'Chr7' ),
                     array( 'color' => '(254,153,0)',   'name' => 'Chr8' ),
                     array( 'color' => '(254,204,0)',   'name' => 'Chr9' ),
                     array( 'color' => '(254,254,0)',   'name' => 'Chr10' ),
                     array( 'color' => '(204,254,0)',   'name' => 'Chr11' ),
                     array( 'color' => '(0,254,0)',     'name' => 'Chr12' ),
                     array( 'color' => '(53,128,0)',    'name' => 'Chr13' ),
                     array( 'color' => '(0,0,204)',     'name' => 'Chr14' ),
                     array( 'color' => '(102,153,254)', 'name' => 'Chr15' ),
                     array( 'color' => '(153,204,254)', 'name' => 'Chr16' ),
                     array( 'color' => '(0,254,254)',   'name' => 'Chr17' ),
                     array( 'color' => '(204,254,254)', 'name' => 'Chr18' ),
                     array( 'color' => '(153,0,204)',   'name' => 'Chr19' ),
                     array( 'color' => '(204,51,254)',  'name' => 'Chr20' ),
                     array( 'color' => '(204,153,254)', 'name' => 'Chr21' ),
                     array( 'color' => '(102,102,102)', 'name' => 'Chr22' ),
                     array( 'color' => '(153,153,153)', 'name' => 'ChrX' ),
                     array( 'color' => '(204,204,204)', 'name' => 'ChrY' ),
                     array( 'color' => '(204,204,153)', 'name' => 'ChrM' ),
                     array( 'color' => '(121,204,61)',  'name' => 'ChrUn' ) );

	if ( is_null($disabled) ) {
	    print "<SELECT name=\"$name\" id=\"$id\">";
	} 
	else {
	    print "<SELECT name=\"$name\" id=\"$id\" disabled >";
	}
    if( is_null($color) ) {
		print '<OPTION selected value="_default">Default</OPTION>';
	}
    else { 
		print '<OPTION value="_default">Default</OPTION>';
	}
    foreach( $colors as $col ) {
      $c = $col['color'];
      $n = $col['name'];
      preg_match( '/^\((\d+)\,(\d+)\,(\d+)\)$/', $c, $match );	  
      assert( count($match) == 4 );
      $rrr = sprintf( "%02x", $match[1] );    $irrr = sprintf( "%02x", 255 - $match[1] );
      $ggg = sprintf( "%02x", $match[2] );    $iggg = sprintf( "%02x", 255 - $match[2] );
      $bbb = sprintf( "%02x", $match[3] );    $ibbb = sprintf( "%02x", 255 - $match[3] );
      if( $color == $c ) {
	  		print '<OPTION style="color:#'.$irrr.$iggg.$ibbb.'; background:#'.$rrr.$ggg.$bbb.';" selected value="'.$c.'">'.$n.'</OPTION>'; 
	  }
	  else {
	  		print '<OPTION style="color:#'.$irrr.$iggg.$ibbb.'; background:#'.$rrr.$ggg.$bbb.';" value="'.$c.'">'.$n.'</OPTION>';
	  }
    }
    print '</SELECT>';
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [colorstr2color]
  ///////////////////////////////////////////////////////////////////////////////
  function colorstr2color( $image, $colorstr ) {
    preg_match( '/^\((\d+)\,(\d+)\,(\d+)\)$/', $colorstr, $match );
    assert( count($match) == 4 );
    return imagecolorallocate( $image, $match[1], $match[2], $match[3] );
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [gen_track_query]
  //   Tv : Track ɓnKvXg𐶐
  ///////////////////////////////////////////////////////////////////////////////
  function gen_track_query( $track, $dispparam ) {
    return array_merge( $dispparam, $track['optparam'] );
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [urlize]
  ///////////////////////////////////////////////////////////////////////////////
  function urlize( $str ) {
    $before = array( '+', '/', '=' );
    $after  = array( '-', '_', '.' );
    return str_replace( $before, $after, base64_encode( $str ) );
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [deurlize]
  ///////////////////////////////////////////////////////////////////////////////
  function deurlize( $str ) {
    $before = array( '+', '/', '=' );
    $after  = array( '-', '_', '.' );
    return base64_decode( str_replace( $after, $before,  $str ) );
  }

  assert( deurlize(urlize('6pCNr17qoHamd-Y6cClFMTF8YG5')) == '6pCNr17qoHamd-Y6cClFMTF8YG5' );
  assert( urlize('Tmr36_OM4dsa6aswbsG6Asg') == urlencode(urlize('Tmr36_OM4dsa6aswbsG6Asg')) );

  ///////////////////////////////////////////////////////////////////////////////
  // [urlpack]
  // Tv : $str  URL Ƃėp\ȕŴ݂ňk
  ///////////////////////////////////////////////////////////////////////////////
  function urlpack( $str ) {
    $compressed = gzcompress( $str );
    return urlize( $compressed );
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [urlunpack]
  // Tv : urlpack ̌ʂɖ߂
  ///////////////////////////////////////////////////////////////////////////////
  function urlunpack( $str ) {
    $converted    = deurlize( $str );
    $uncompressed = gzuncompress( $converted );
    return $uncompressed;
  }

#  assert( urlunpack(urlpack('6pCNr17qoHamd-Y6cClFMTF8YG5')) == '6pCNr17qoHamd-Y6cClFMTF8YG5' );
#  assert( urlpack('Tmr36_OM4dsa6aswbsG6Asg') == urlencode(urlpack('Tmr36_OM4dsa6aswbsG6Asg')) );

  ///////////////////////////////////////////////////////////////////////////////
  // [req_urldecode]
  //   Tv : NGXgz key  urldecode ꂽԂɂ
  ///////////////////////////////////////////////////////////////////////////////
  function req_urldecode( $req ) {
    $ret = array();
    foreach( $req as $key => $val ) {
      $ret[urldecode($key)] = $val;
    }
    return $ret;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [parse_operation_xml]
  // o : array(
  //          'type' => 'operation_layer',
  //          'areas' => array(
  //            array(
  //              'type' => 'rect_area',
  //              'rect' => '(0,0,100,100)',
  //              'events' => array (
  //                array(
  //                  'type' => 'mouseclick_event',
  //                  'operations' => array (
  //                    array(
  //                      'type' => 'link_operation',
  //                      'url' => 'http://www.google.com/',
  //                      'target' => '_blank',
  //                    ),
  //                  ),
  //                ),
  //              ),
  //            ),
  //          ),
  //        )
  ///////////////////////////////////////////////////////////////////////////////
  function parse_operation_xml( $xmlstr ) {
    $dom = domxml_open_mem( $xmlstr );
    if( ! $dom ) { return NULL; }
    $root = $dom->document_element();
    return parse_operation_xml_root( $root );
  }

  function parse_operation_xml_root( $root ) {
    if( $root->node_name() == 'operation_layer' ) {
      $areas = array();
      if( $root_child = $root->first_child() ) {
        do {
          $ret = parse_operation_xml_area( $root_child );
          if( $ret ) { $areas[] = $ret; }
        } while( $root_child = $root_child->next_sibling() );
      }
      return array( 'type' => $root->node_name(), 'areas' => $areas );
    }
    else {
      return NULL;
    }
  }

  function parse_operation_xml_area( $area ) {
    if( $area->node_name() == 'rect_area' ) {
      $events = array();
      if( $area_child = $area->first_child() ) {
        do {
          $ret = parse_operation_xml_event( $area_child );
          if( $ret ) { $events[] = $ret; }
        } while( $area_child = $area_child->next_sibling() );
      }
      return array( 'type' => $area->node_name(), 'rect' => $area->get_attribute('rect'), 'events' => $events );
    }
    else {
      return NULL;
    }
  }

  function parse_operation_xml_event( $event ) {
    if( $event->node_name() == 'mouseclick_event' || $event->node_name() == 'mousestop_event' || $event->node_name() == 'suboperation_layer' ) {
      $operations = array();
      if( $event_child = $event->first_child() ) {
        do {
          $ret = parse_operation_xml_operation( $event_child );
          if( $ret ) { $operations[] = $ret; }
        } while( $event_child = $event_child->next_sibling() );
      }
      return array( 'type' => $event->node_name(), 'operations' => $operations );
    }
    else {
      return NULL;
    }
  }

  function parse_operation_xml_operation( $operation ) {
    if( $operation->node_name() == 'link_operation' ) {
      return array( 'type' => $operation->node_name(), 'url' => $operation->get_attribute('url'), 'target' => $operation->get_attribute('target') );
    }
    elseif( $operation->node_name() == 'suboperation' ) {
      return array( 'type' => $operation->node_name(), 'url' => $operation->get_attribute('url') );
    }
    elseif( $operation->node_name() == 'frame_operation' ) {
      $commands = array();
      if( $operation_child = $operation->first_child() ) {
        do {
          if( $operation_child->node_name() == 'command' ) {
            $commands[] = array( 'type' => $operation_child->node_name(), 'name' => $operation_child->get_attribute('name'), 'value' => $operation_child->get_attribute('value') );
          }
        } while( $operation_child = $operation_child->next_sibling() );
      }
      return array( 'type' => $operation->node_name(), 'commands' => $commands );
    }
    elseif( $operation->node_name() == 'menu_operation' ) {
      $items = array();
      if( $operation_child = $operation->first_child() ) {
        do {
          if( $operation_child->node_name() == 'menu_item' ) {
            $opers = array();
            if( $menu_child = $operation_child->first_child() ) {
              do {
                $ret = parse_operation_xml_operation( $menu_child );
                if( $ret ) { $opers[] = $ret; }
              } while( $menu_child = $menu_child->next_sibling() );
            }
            $items[] = array( 'type' => $operation_child->node_name(), 'caption' => $operation_child->get_attribute('caption'), 'operations' => $opers );
          }
        } while( $operation_child = $operation_child->next_sibling() );
      }
      return array( 'type' => $operation->node_name(), 'title' => $operation->get_attribute('title'), 'items' => $items );
    }
    else {
      return NULL;
    }
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [parse_suboperation_xml]
  ///////////////////////////////////////////////////////////////////////////////
  function parse_suboperation_xml( $xmlstr ) {
    $dom = domxml_open_mem( $xmlstr );
    if( ! $dom ) { return NULL; }
    $root = $dom->document_element();
    return parse_operation_xml_event( $root );
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [get_targetlen]
  ///////////////////////////////////////////////////////////////////////////////
  function get_targetlen( $species, $revision, $target ) {

    global $acceptspecies_connection_string;
    $dbconn = pg_connect( $acceptspecies_connection_string );

    assert( $dbconn );
    $res = pg_query( $dbconn, "SELECT length FROM targetlen WHERE ksid=(SELECT ksid FROM knownspecies
                               WHERE species='$species' and revision='$revision') and target='$target';" );
    $col = pg_fetch_assoc( $res );
    if( $col ) {
      assert( is_numeric($col['length']) );
      return (int)$col['length'];
    }
    else {
      return NULL;
    }
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [correct_dispparam]
  ///////////////////////////////////////////////////////////////////////////////
  function correct_dispparam( $dispparam ) {
    $thislen = get_targetlen( $dispparam['species'], $dispparam['revision'], $dispparam['target'] );
    assert( is_numeric($thislen) );
    if( $dispparam['start'] < $dispparam['end']  ) {
      if( isset($thislen) && $thislen <= abs($dispparam['start']-$dispparam['end'])+1 ) {
        $dispparam['start'] = 1;
        $dispparam['end']   = $thislen;
      }
      elseif( $dispparam['start'] < 1 ) {
        $dispparam['end'] -= $dispparam['start'] - 1;
        $dispparam['start'] = 1;
      }
      elseif( isset($thislen) && $thislen < $dispparam['end'] ) {
        $dispparam['start'] -= $dispparam['end'] - $thislen;
        $dispparam['end']    = $thislen;
      }
    }
    else {
      if( isset($thislen) && $thislen <= abs($dispparam['start']-$dispparam['end'])+1 ) {
        $dispparam['start'] = $thislen;
        $dispparam['end']   = 1;
      }
      elseif( $dispparam['end'] < 1 ) {
        $dispparam['start'] -= $dispparam['end'] - 1;
        $dispparam['end'] = 1;
      }
      elseif( isset($thislen) && $thislen-1 < $dispparam['start'] ) {
        $dispparam['end'] -= $dispparam['start'] - $thislen;
        $dispparam['start']    = $thislen;
      }
    }
    $dispparam['start'] = round( $dispparam['start'] );
    $dispparam['end']   = round( $dispparam['end'] );
    $dispparam['width'] = round( $dispparam['width'] );

    assert( is_numeric($dispparam['start']) );
    assert( is_numeric($dispparam['end']) );
    assert( is_numeric($dispparam['width']) );
    assert( 1 <= $dispparam['start'] );
    assert( 1 <= $dispparam['end'] );
    assert( is_null($thislen) || $dispparam['start'] <= $thislen );
    assert( is_null($thislen) || $dispparam['end'] <= $thislen );
    assert( is_null($thislen) || abs($dispparam['start']-$dispparam['end'])+1 <= $thislen );

    return $dispparam;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [additional_attributes_to_str]
  ///////////////////////////////////////////////////////////////////////////////
  function additional_attributes_to_str( $additional_attr ) {
    if( is_array($additional_attr) ) {
      $ret = '';
      foreach( $additional_attr as $key => $val ) {
        $ret .= $key.'="'.$val.'" ';
      }
      return substr( $ret, 0, -1 );
    }
    else {
      return '';
    }
  }
  assert( additional_attributes_to_str(null) == '' );
  assert( additional_attributes_to_str( array('a'=>'b') ) == 'a="b"' );
  assert( additional_attributes_to_str( array('a'=>'b','c'=>'d') ) == 'a="b" c="d"' );

  ///////////////////////////////////////////////////////////////////////////////
  // [print_input_text]
  ///////////////////////////////////////////////////////////////////////////////
  function print_input_text( $name, $default=null, $size=null, $additional_attr=null ) {
    assert( isset($name) );
    assert( is_null($size) || is_numeric($size) );
    $additional_attr_str = additional_attributes_to_str( $additional_attr );
    if( strlen($additional_attr_str) > 0 ) { $additional_attr_str = ' '.$additional_attr_str; }
    if( isset($default) && isset($size) ) {
      print "<INPUT type=\"text\" name=\"$name\" value=\"$default\" size=\"$size\"$additional_attr_str>";
    }
    elseif( isset($default) && is_null($size) ) {
      print "<INPUT type=\"text\" name=\"$name\" value=\"$default\"$additional_attr_str>";
    }
    elseif( is_null($default) && isset($size) ) {
      print "<INPUT type=\"text\" name=\"$name\" size=\"$size\"$additional_attr_str>";
    }
    else {
      assert( is_null($default) && is_null($size) );
      print "<INPUT type=\"text\" name=\"$name\"$additional_attr_str>";
    }
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [print_input_textarea]
  ///////////////////////////////////////////////////////////////////////////////
  function print_input_textarea( $name, $default=null, $size=null, $additional_attr=null ) {
    assert( isset($name) );
    assert( is_null($size) || is_array($size) );
    assert( is_null($size) || count($size) == 2 );
    assert( is_null($size) || is_numeric($size['rows']) );
    assert( is_null($size) || is_numeric($size['cols']) );
    $additional_attr_str = additional_attributes_to_str( $additional_attr );
    if( strlen($additional_attr_str) > 0 ) { $additional_attr_str = ' '.$additional_attr_str; }
    if( isset($size) ) {
      $r = $size['rows'];
      $c = $size['cols'];
      print "<TEXTAREA name=\"$name\" rows=\"$r\" cols=\"$c\"$additional_attr_str>";
    }
    else {
      if( is_null($size) );
      print "<TEXTAREA name=\"$name\"$additional_attr_str>";
    }
    if( isset($default) ) {
      print $default;
    }
    print "</TEXTAREA>";
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [print_input_select]
  ///////////////////////////////////////////////////////////////////////////////
  function print_input_select( $name, $options, $default=null, $additional_attr=null ) {
    assert( isset($name) );
    assert( isset($options) && is_array($options) );
    $additional_attr_str = additional_attributes_to_str( $additional_attr );
    if( strlen($additional_attr_str) > 0 ) { $additional_attr_str = ' '.$additional_attr_str; }
    print "<SELECT name=\"$name\"$additional_attr_str>";
    foreach( $options as $opt ) {
      if( $opt == $default ) {
        print "<OPTION value=\"$opt\" selected>$opt</OPTION>";
      }
      else {
        print "<OPTION value=\"$opt\">$opt</OPTION>";
      }
    }
    print "</SELECT>";
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [print_input_checkbox]
  ///////////////////////////////////////////////////////////////////////////////
  function print_input_checkbox( $name, $options, $default=null, $separator=null, $additional_attr=null ) {
    assert( isset($name) );
    assert( isset($options) && is_array($options) );
    assert( is_null($default) || is_array($default) );
    assert( is_null($separator) || is_string($separator) );
    $additional_attr_str = additional_attributes_to_str( $additional_attr );
    if( strlen($additional_attr_str) > 0 ) { $additional_attr_str = ' '.$additional_attr_str; }
    foreach( $options as $opt ) {
      if( isset($default) && in_array($opt,$default) ) {
        print "<INPUT type=\"checkbox\" name=\"${name}[]\" value=\"$opt\" checked$additional_attr_str> $opt";
      }
      else {
        print "<INPUT type=\"checkbox\" name=\"${name}[]\" value=\"$opt\"$additional_attr_str> $opt";
      }
      if( isset($separator) ) { print $separator; }
    }
  }

  function framecommand_2_querystr( $commands ) {
    $querystr = '';
    foreach( $commands as $cmd ) {
      $name  = $cmd['name'];
      $value = $cmd['value'];
      if( strlen($value) > 0 ) {
        $querystr .= "$name=$value&";
      }
      else {
        $querystr .= "$name&";
      }
    }
    return substr( $querystr, 0, -1 );
  }


  ///////////////////////////////////////////////////////////////////////////////
  // [get_targetseq]
  ///////////////////////////////////////////////////////////////////////////////
  function get_targetseq( $species, $revision, $target ) {
    global $targetsequence_connection_string;
    $dbconn = pg_connect( $targetsequence_connection_string );

    assert( $dbconn );
    $res = pg_query( $dbconn, "SELECT sequence FROM targetseq WHERE ksid=(SELECT ksid FROM knownspecies WHERE species='$species' and revision='$revision') and target='$target';" );
    $col = pg_fetch_assoc($res);
    $seq = $col['sequence'];
    $seq = preg_replace( '/[^ATGCN]/', '', $seq );
    assert( preg_match( '/^[ATGCN]*$/', $seq ) );
    return $seq;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [inv_nseq]
  ///////////////////////////////////////////////////////////////////////////////
  function inv_nseq( $seq ) {
    assert( preg_match( '/^[ATGCN]*$/', $seq ) );
    $inv_tbl = array( 'A' => 'T', 'T' => 'A', 'G' => 'C', 'C' => 'G', 'N' => 'N' );
    $inv_seq = '';
    $len     = strlen( $seq );
    for( $i=0 ; $i<$len ; $i++ ) {
      $inv_seq .= $inv_tbl[substr($seq,-1-$i,1)];
    }
    return $inv_seq;
  }
  assert( inv_nseq('A') == 'T' );
  assert( inv_nseq('T') == 'A' );
  assert( inv_nseq('G') == 'C' );
  assert( inv_nseq('C') == 'G' );
  assert( inv_nseq('ATGCN') == 'NGCAT' );
  assert( inv_nseq('GCATN') == 'NATGC' );

  ///////////////////////////////////////////////////////////////////////////////
  // [parse_viewpoint_xml]
  //   Tv : XML `ŋLqꂽ Viewpoint ǂݍ
  ///////////////////////////////////////////////////////////////////////////////
  function parse_viewpoint_xml( $xmlstr ) {
    $dom = domxml_open_mem( $xmlstr );
    if( ! $dom ) { return NULL; }
    $root = $dom->document_element();
    return parse_viewpoint_xml_root( $root );
  }

  function parse_viewpoint_xml_root( $viewpoint ) {
    if( $viewpoint->node_name() == 'viewpoint' ) {
      $dispparam = NULL;
      $tracks    = NULL;
      if( $viewpoint_child = $viewpoint->first_child() ) {
        do {
          if( $viewpoint_child->node_name() == 'dispparam' ) {
            $dispparam = parse_viewpoint_xml_dispparam( $viewpoint_child );
            assert( isset($dispparam) );
          }
          elseif( $viewpoint_child->node_name() == 'tracks' ) {
            $s = $viewpoint_child->get_attribute('species');
            $r = $viewpoint_child->get_attribute('revision');
            $tracks[$s][$r] = parse_viewpoint_xml_tracks( $viewpoint_child );
            assert( isset($tracks) );
          }
        } while( $viewpoint_child = $viewpoint_child->next_sibling() );
      }
      return array( 'dispparam' => $dispparam, 'tracks' => $tracks );
    }
    else {
      return NULL;
    }
  }

  function parse_viewpoint_xml_dispparam( $dispparam ) {
    if( $dispparam->node_name() == 'dispparam' ) {
      return array(
        'species'  => $dispparam->get_attribute('species'),
        'revision' => $dispparam->get_attribute('revision'),
        'target'   => $dispparam->get_attribute('target'),
        'start'    => $dispparam->get_attribute('start'),
        'end'      => $dispparam->get_attribute('end'),
        'width'    => $dispparam->get_attribute('width'),
      );
    }
    else {
      return NULL;
    }
  }

  function parse_viewpoint_xml_tracks( $tracks ) {
    if( $tracks->node_name() == 'tracks' ) {
      $track = array();
      if( $tracks_child = $tracks->first_child() ) {
        do {
          if( $tracks_child->node_name() == 'track' ) {
            $track[] = parse_viewpoint_xml_track( $tracks_child );
          }
        } while( $tracks_child = $tracks_child->next_sibling() );
      }
      return $track;
    }
    else {
      return NULL;
    }
  }

  function parse_viewpoint_xml_track( $track ) {
    if( $track->node_name() == 'track' ) {
      $optparams = NULL;
      if( $track_child = $track->first_child() ) {
        do {
          if( $track_child->node_name() == 'optparams' ) {
            $optparams = parse_viewpoint_xml_optparams( $track_child );
          }
        } while( $track_child = $track_child->next_sibling() );
      }
      return array(
        'url'  => $track->get_attribute('url'),
        'removed' => ($track->get_attribute('removed') == 'yes'),
        'optparams' => $optparams
      );
    }
    else {
      return NULL;
    }
  }

  function parse_viewpoint_xml_optparams( $optparams ) {
    if( $optparams->node_name() == 'optparams' ) {
      $oa = array();
      if( $optparams_child = $optparams->first_child() ) {
        do {
          if( $optparams_child->node_name() == 'optparam' ) {
            $oa[$optparams_child->get_attribute('key')] = $optparams_child->get_attribute('val');
          }
        } while( $optparams_child = $optparams_child->next_sibling() );
      }
      return $oa;
    }
    else {
      return NULL;
    }
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // [load_viewpoint_xml]
  //   Tv : XML `ŋLqꂽ Viewpoint ǂݍ
  //          Ɋe Track ǂݍ
	//
	//	2005.5.19 modify t.shishiki utracksvƁuremovedtracksvf[^x[Xitrackinformationj擾悤ɕύX
	//
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  function load_viewpoint_xml( $url ) {
    $viewpoint_xmlstr = http_get( $url );
    return load_viewpoint_xml_by_content( $viewpoint_xmlstr );
  }

  function load_viewpoint_xml_by_content( $content ) {
    $viewpoint = parse_viewpoint_xml( $content );

    assert( isset($viewpoint['dispparam']) );
//    assert( isset($viewpoint['tracks']) );
    $tracks = array();
    $removedtracks = array();
//    foreach( $viewpoint['tracks'] as $s => $revs ) {
//      foreach( $revs as $r => $trks ) {
//        foreach( $trks as $trk ) {
//          $track_url = $trk['url'];
//          $xmlstr = http_get( $track_url );
//          if( is_null($xmlstr) ) { continue; }
//          $parsed_track = parse_track_xml( $xmlstr );
//          if( is_null($parsed_track) ) { continue; }
//          $track = array();
//          $track['url']             = $track_url;
//          $track['name']            = $parsed_track['name'];
//          $track['description_url'] = $parsed_track['description_url'];
//          $track['comment']         = $parsed_track['comment'];
//          $track['layers']          = $parsed_track['layers'];
//          $track['optattr']         = parse_optattr( $parsed_track['optattr'] );
//          $track['optparam']        = $trk['optparams'];
//          if( $trk['removed'] ) { $removedtracks[$s][$r][] = $track; }
//          else                  { $tracks[$s][$r][] = $track; }
//        }
//      }
//    }
//    $viewpoint['tracks']       = $tracks;
//    $viewpoint['removedtracks'] = $removedtracks;
    return $viewpoint;
  }

  ////////////////////////////////////////////////////////////////////////////////
  // [load_default_viewpoint]
  ////////////////////////////////////////////////////////////////////////////////
  function load_default_viewpoint () {
    global $default_viewpoint_xml_path;
    $xml_content = file_get_contents( $default_viewpoint_xml_path );

    return load_viewpoint_xml_by_content( $xml_content );
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [_species]
  // Tv : ̃VXeΉĂ AcceptSpecies  DB ǂ
  // o : array(
  //          'medaka' => array(
  //             '200406',
  //             '200409',
  //           ),
  //          'test' => array(
  //             'test1',
  //             'test2',
  //           ),
  //        )
  ///////////////////////////////////////////////////////////////////////////////

  function load_accept_species() {
    global $acceptspecies_connection_string;
    $ignode_revision = '';
    $ret = array();
    $dbconn = pg_connect( $acceptspecies_connection_string );
    assert( $dbconn );
    $res = pg_query( $dbconn, "SELECT * FROM knownspecies WHERE revision != '$ignode_revision';" );
    while( $col = pg_fetch_assoc($res) ) {
      $s = $col['species'];
      $r = $col['revision'];
      if( isset($ret[$s]) ) {
        assert( is_array($ret[$s]) );
        $ret[$s][] = $r;
      }
      else {
        $ret[$s] = array( $r );
      }
    }
    return $ret;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [create_image_from_url]
  // Tv : w肳ꂽ URL ɂC[W𐶐ĕԂ
  ///////////////////////////////////////////////////////////////////////////////
  function create_image_from_url( $url ) {
    $got = http_get_with_info( $url );
    $type = $got['header']['Content-Type'];
    if( strpos( $type, 'image/' ) === 0 ) {
      $tmpname = tempnam( '/tmp', 'utgb_imagefromurl' );
      $fh = fopen( $tmpname, 'wb' );
      fwrite( $fh, $got['contents'] );
      fclose( $fh );
      $img = NULL;
      if( $type == 'image/png' ) {
        $img = array();
        $img['image'] = imagecreatefrompng( $tmpname );
        $img['size']  = getimagesize( $tmpname );
      }
      elseif( $type == 'image/jpeg' || $type == 'image/jpg' ) {
         $img = array();
        $img['image'] = imagecreatefromjpeg( $tmpname );
        $img['size']  = getimagesize( $tmpname );
     }
      unlink( $tmpname );
      return $img;
    }
    else {
      return NULL;
    }
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [print_revisions_by_species_javascript_code]
  // Tv : revisionsBySpecies ϐpӂ JavaScript R[hf
  ///////////////////////////////////////////////////////////////////////////////
  function print_revisions_by_species_javascript_code( $accept_species ) {
    print 'var revisionsBySpecies = new Array(); ';
    foreach( $accept_species as $species => $revisions ) {
      $new_array_str = 'new Array( ';
      foreach( $revisions as $revision ) {
        $new_array_str .= '\''. $revision . '\', ';
      }
      $new_array_str = substr( $new_array_str, 0, -2 );
      $new_array_str .= ' )';
      print "revisionsBySpecies['$species'] = $new_array_str;  ";
    }
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [load_keyword_kinds]
  // Tv : ̃VXeΉĂ KeywordSearch  DB ǂ
  // o : array(
  //          'medaka' => array(
  //             '200406' => array( 'Genbank Acc', 'Clone ID', 'Ensemble ID', 'GI Number' ),
  //             '200409' => array( 'Genbank Acc', 'LocusLink ID', 'Symbol', 'Unigene Cluster ID' ),
  //           )
  //        )
  ///////////////////////////////////////////////////////////////////////////////
  function load_keyword_kinds() {
    global $keywordsearch_connection_string;
    $ret = array();
    $dbconn = pg_connect( $keywordsearch_connection_string );
		assert( $dbconn );
    $res = pg_query( $dbconn, "SELECT * FROM knownkeykind;" );
    while( $col = pg_fetch_assoc($res) ) {
      $s = $col['species'];
      $r = $col['revision'];
      $k = $col['keykind'];
      if( isset($ret[$s]) ) {
        assert( is_array($ret[$s]) );
        $ret[$s][$r][] = $k;
      }
      else {
        $ret[$s] = array();
        $ret[$s][$r] = array( $k );
      }
    }
    return $ret;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [print_keywkinds_javascript_code]
  // Tv : keywkinds ϐpӂ JavaScript R[hf
  ///////////////////////////////////////////////////////////////////////////////
  function print_keywkinds_javascript_code( $keykinds ) {
    $species_allocated = array();
    print 'var keywkinds = new Array(); ';
    foreach( $keykinds as $species => $revisions ) {
      foreach( $revisions as $revision => $keykinds ) {
        $new_array_str_2 = 'new Array( ';
        foreach( $keykinds as $keykind ) {
          $new_array_str_2 .= '\''. $keykind . '\', ';
        }
        $new_array_str_2 = substr( $new_array_str_2, 0, -2 ) . ' )';
        if( ! array_key_exists( $species, $species_allocated ) ) {
          print "keywkinds['$species'] = new Array(); ";
          $species_allocated[$species] = 1;
        }
        print "keywkinds['$species']['$revision'] = $new_array_str_2; ";
      }
    }
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [self_script_file_url]
  ///////////////////////////////////////////////////////////////////////////////
  function self_script_file_url() {
    return 'http://' . $_SERVER['SERVER_NAME'] . $_SERVER['SCRIPT_NAME'];
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [self_script_directory_url]
  ///////////////////////////////////////////////////////////////////////////////
  function self_script_directory_url() {
    return 'http://' . $_SERVER['SERVER_NAME'] . dirname($_SERVER['SCRIPT_NAME']);
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [encrypt]
  ///////////////////////////////////////////////////////////////////////////////
  function encrypt( $data ) {
    global $openssl_publickey;
    openssl_public_encrypt( $data, $crypted, $openssl_publickey );
    return $crypted;
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [decrypt]
  ///////////////////////////////////////////////////////////////////////////////
  function decrypt( $crypted ) {
    global $openssl_privatekey;
    openssl_private_decrypt( $crypted, $data, $openssl_privatekey );
    return $data;
  }

#  assert( 0 == decrypt( encrypt( 0 ) ) );
#  assert( '' == decrypt( encrypt( '' ) ) );
#  assert( 12 == decrypt( encrypt( 12 ) ) );
#  assert( 1234 == decrypt( encrypt( 1234 ) ) );
#  assert( '1234' == decrypt( encrypt( '1234' ) ) );
#  assert( 'asdf' == decrypt( deurlize( urlize( encrypt( 'asdf' ) ) ) ) );

  ///////////////////////////////////////////////////////////////////////////////
  // [get_synteny]
  ///////////////////////////////////////////////////////////////////////////////
  function get_synteny( $fromspe, $fromrev, $tospe, $torev, $fromtarget, $fromstart, $fromend ) {
		
    global $synteny_connection_string;
    $ret = array();
    $dbconn = pg_connect( $synteny_connection_string );	
    $sql = "SELECT * FROM synteny WHERE ssid=(SELECT ssid FROM syntenicspecies WHERE fromspecies=(SELECT ksid FROM knownspecies WHERE species='$fromspe' AND revision='$fromrev') AND tospecies=(SELECT ksid FROM knownspecies WHERE species='$tospe' AND revision='$torev')) AND from_target='$fromtarget' AND from_start BETWEEN $fromstart AND $fromend AND from_end BETWEEN $fromstart AND $fromend ORDER BY from_start;";
    $res = pg_query( $dbconn, $sql );
    while( $col = pg_fetch_assoc($res) ) { $ret[] = $col; }
    return $ret;
  }

  function create_track_images( $dispparam, $tracks, $ignodetracknames ) {
    $width = $dispparam['width'];

    $fontsize  = 10;
    $fontfile  = './fonts/Abadi MT Condensed Light.ttf';

    $indeximages   = array();
    $graphicimages = array();
    $maxtitlewidth = 0;
    $maxindexwidth = 0;
    $totalheight = 0;
    foreach( $tracks as $tracknum => $track ) {
      if( in_array( $track['name'], $ignodetracknames ) ) { continue; }
      $query4thistrack_str = query2string( gen_track_query( $track, $dispparam ) );
      if( $track['layers']['index'] ) {
        $index_layer_url = $track['layers']['index']['url'].'?'.$query4thistrack_str;
        $image = create_image_from_url( $index_layer_url );
        if( isset($image) ) {
          $indeximages[$tracknum] = $image;
          $maxindexwidth = max( $maxindexwidth, $image['size'][0] );
        }
      }
      if( strlen($track['name']) > 0 ) {
        $strbox = imagettfbbox( $fontsize, 0, $fontfile, $track['name'] );
        $maxtitlewidth = max( $maxtitlewidth, $strbox[2] - $strbox[0] );
      }
      $graphic_layer_url = $track['layers']['graphic']['url'].'?'.$query4thistrack_str;
      $image = create_image_from_url( $graphic_layer_url );
      if( isset($image) ) {
        $graphicimages[$tracknum] = $image;
        if( $image['size'][0] == $width ) {
          $totalheight += $image['size'][1];
        }
        else {
          $totalheight += $image['size'][1] * $width / $image['size'][0];
        }
      }
    }

    return array(
      'graphicimages' => $graphicimages,
      'indeximages'   => $indeximages,
      'maxtitlewidth' => $maxtitlewidth,
      'maxindexwidth' => $maxindexwidth,
      'totalheight'   => $totalheight,
    );
  }

  ///////////////////////////////////////////////////////////////////////////////
  // [pick_menuoper]
  // Tv : parse_operation_xml œǂ operation ɊÂ
  //        \ߗpӂĂׂ menu 𐮗B
  ///////////////////////////////////////////////////////////////////////////////
  function pick_menuoper( &$operation, $prefix='' ) {
    if( $operation['type'] == 'operation_layer' ) {
      $menuopers = array();
      pick_menuoper_sub( $operation['areas'], $menuopers );
      foreach( $menuopers as $num => $menuoper ) {
        $menuopers[$num]['__id'] = $prefix.$num;
      }
      return $menuopers;
    }
    else {
      return NULL;
    }
  }

  function pick_menuoper_sub( &$operation, &$menuopers ) {
    if( is_array( $operation ) ) {
      foreach( $operation as $i => $oper ) {
        switch( $oper['type'] ) {
        case 'rect_area':
          pick_menuoper_sub( $operation[$i]['events'], $menuopers );
          break;
        case 'mouseclick_event':
          pick_menuoper_sub( $operation[$i]['operations'], $menuopers );
          break;
        case 'menu_operation':
          $menuopers[] =& $operation[$i];
          pick_menuoper_sub( $operation[$i]['items'], $menuopers );
          break;
        case 'menu_item':
          pick_menuoper_sub( $operation[$i]['operations'], $menuopers );
          break;
        }
      }
    }
  }

  function print_picked_menus_html( $picked_menus ) {
    if( ! is_array( $picked_menus ) ) { return; }
    foreach( $picked_menus as $picked_menu ) {
      if( $picked_menu['type'] != 'menu_operation' ) { continue; }
      $title = $picked_menu['title'];
      $id    = $picked_menu['__id'];
      print "<DIV id=\"$id\" style=\"position:absolute; display:none;\">\n";
      print "  <DIV id=\"$id-title\" class=\"menutitle\" onDblClick=\"\"><TABLE border=\"0\" width=\"100%\"><TBODY><TR><TD align=\"left\">$title</TD><TD align=\"right\"><A href=\"JavaScript: document.getElementById('$id').style.display='none'; void(0);\">[X]</TD></TR></TBODY></TABLE></DIV>\n";
      foreach( $picked_menu['items'] as $itemnum => $menuitem ) {
        if( $menuitem['type'] != 'menu_item' ) { continue; }
        $caption = $menuitem['caption'];
        $operation = $menuitem['operations'][0];
        switch( $operation['type'] ) {
        case 'menu_operation':
          $nextmenuid = $operation['__id'];
          print "  <DIV id=\"$id-$itemnum\" class=\"menuitem\"><A href=\"JavaScript: menuPopup('$nextmenuid');\">$caption</A></DIV>\n";
          break;
        case 'link_operation':
          $linkurl    = $operation['url'];
          $linktarget = $operation['target'];
          print "  <DIV id=\"$id-$itemnum\" class=\"menuitem\"><A href=\"$linkurl\" target=\"$linktarget\">$caption</A></DIV>\n";
          break;
        }
      }
      print "</DIV>\n";
    }
  }

  function init_openssl_certs() {
    global $openssl_certs_dir, $openssl_privatekey_filename, $openssl_certificate_filename, $openssl_certreq_filename, $openssl_privatekey_password, $openssl_expiration, $openssl_dn;
    global $openssl_privatekey, $openssl_publickey, $openssl_certificate, $openssl_certreq;
    if( ! ( file_exists($openssl_certs_dir.$openssl_privatekey_filename) && file_exists($openssl_certs_dir.$openssl_certificate_filename) && file_exists($openssl_certs_dir.$openssl_certreq_filename) ) ) {
      assert( is_writable($openssl_certs_dir) );
      $openssl_privatekey  = openssl_pkey_new();
      $openssl_certreq     = openssl_csr_new( $openssl_dn, $openssl_privatekey );
      $openssl_certificate = openssl_csr_sign( $openssl_certreq, null, $openssl_privatekey, $openssl_expiration );
      openssl_pkey_export( $openssl_privatekey, $privatekeyStr, $openssl_privatekey_password );
      openssl_x509_export( $openssl_certificate, $certificateStr );
      openssl_csr_export( $openssl_certreq, $certreqStr );
      putfile( $openssl_certs_dir.$openssl_privatekey_filename, $privatekeyStr );
      putfile( $openssl_certs_dir.$openssl_certificate_filename, $certificateStr );
      putfile( $openssl_certs_dir.$openssl_certreq_filename, $certreqStr );
      clearstatcache();
    }
    assert( is_readable($openssl_certs_dir.$openssl_privatekey_filename) );
    assert( is_readable($openssl_certs_dir.$openssl_certificate_filename) );
    assert( is_readable($openssl_certs_dir.$openssl_certreq_filename) );
    $openssl_privatekey  = openssl_get_privatekey( getfile($openssl_certs_dir.$openssl_privatekey_filename), $openssl_privatekey_password );
    $openssl_publickey   = openssl_get_publickey( getfile($openssl_certs_dir.$openssl_certificate_filename) );
    $openssl_certificate = getfile($openssl_certs_dir.$openssl_certificate_filename);
    $openssl_certreq     = getfile($openssl_certs_dir.$openssl_certreq_filename);
  }

  function putfile( $filemane, $data ) {
    $fh = fopen( $filemane, 'w' );
    if( ! $fh ) { return; }
    fwrite( $fh, $data );
    fclose( $fh );
  }

  function getfile( $filemane ) {
    $fh = fopen( $filemane, 'r' );
    if( ! $fh ) { return; }
    $contents = '';
    for(;;) {
      $data = fread( $fh, 8192 );
      if( strlen($data) == 0 ) { break; }
      $contents .= $data;
    }
    fclose( $fh );
    return $contents;
  }

  function expand_metaurl( $murl, $track_base_url ) {
    global $base_url;
    $ret = $murl;
    $ret = str_replace( '$BROWSER_BASE_URL$', $base_url, $ret );
#    if( isset($track_base_url) ) { $ret = str_replace('$TRACK_BASE_URL$', $track_base_url, $ret ); }
    return $ret;
  }
		
?>
