/*
 * Asprintf
 *
 * Copyright (c) 2014 Akira Sasaki
 *
 * This software is released under the MIT License.
 *
 * http://opensource.org/licenses/mit-license.php
 */

/**
 * @file
 * @brief Implementation of @asprintf and @vasprintf.
 *
 * This is an implementation of @asprintf and @vasprintf.
 *
 * The function @asprintf and @vasprintf are similar to @sprintf and
 * @vsprintf respectively, except that they allocate memory to store
 * output string, and return a pointer to it via the first parameter.
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>

#include "asprintf.h"

#ifdef UNIT_TEST
int (*mockable_vsnprintf1)(
    char * restrict str, size_t size,
    const char * restrict format, va_list ap) = vsnprintf;
int (*mockable_vsnprintf2)(
    char * restrict str, size_t size,
    const char * restrict format, va_list ap) = vsnprintf;
void *(*mockable_malloc)(size_t size) = malloc;
void (*mockable_free)(void *ptr) = free;
#else
/**
 * @brief Function pointer for mocking first @vsnprintf call.
 *
 * If macro @c UNIT_TEST is defined, this is a function pointer for mocking
 * first @vsnprintf call, whose type is as same as @vsnprintf and initial
 * value is @c vsnprintf.
 * Otherwise, this is replaced with @c vsnprintf simply.
 */
#define mockable_vsnprintf1 vsnprintf
/**
 * @brief Function pointer for mocking second @vsnprintf call.
 *
 * If macro @c UNIT_TEST is defined, this is a function pointer for mocking
 * second @vsnprintf call, whose type is as same as @vsnprintf and initial
 * value is @c vsnprintf.
 * Otherwise, this is replaced with @c vsnprintf simply.
 */
#define mockable_vsnprintf2 vsnprintf
/**
 * @brief Function pointer for mocking @malloc call.
 *
 * If macro @c UNIT_TEST is defined, this is a function pointer for mocking
 * @malloc call, whose type is as same as @malloc and initial value is
 * @c malloc.
 * Otherwise, this is replaced with @c malloc simply.
 */
#define mockable_malloc malloc
/**
 * @brief Function pointer for mocking @free call.
 *
 * If macro @c UNIT_TEST is defined, this is a function pointer for mocking
 * @free call, whose type is as same as @free and initial value is @c free.
 * Otherwise, this is replaced with @c free simply.
 */
#define mockable_free free
#endif

int vasprintf(char **strp, const char *format, va_list ap)
{
    *strp = NULL;

    va_list ap_copy;
    va_copy(ap_copy, ap);
    int print1_size = mockable_vsnprintf1(NULL, 0, format, ap_copy);
    va_end(ap_copy);
    if (print1_size == -1) {
        return -1;
    }

    int alloc_size = print1_size + 1;
    char *alloc_area = mockable_malloc(alloc_size);
    if (alloc_area == NULL) {
        return -1;
    }

    int print2_size = mockable_vsnprintf2(alloc_area, alloc_size, format, ap);
    if (print2_size != print1_size) {
        mockable_free(alloc_area);
        return -1;
    }

    *strp = alloc_area;
    return print1_size;
}

int asprintf(char **strp, const char *format, ...)
{
    va_list ap;

    va_start(ap, format);
    int size = vasprintf(strp, format, ap);
    va_end(ap);

    return size;
}
