static char sccsid[]="%Z% %M% %I% %E% %U%";
/*******************************************/
/*								*/
/*      coded by A.Kobayashi 2010.5.20   */
/*	XV : 2018.09.09 		*/
/*								*/
/******************************************/
#include "akxcommon.h"

#if defined(AIX) || defined(WIN) || defined(HPUX) || defined(CYGWIN)
#define SJIS
#else
#define EUC
#endif

#ifndef SS2
#define SS2		0x8e	/* Hankaku Kana */
#define SS3		0x8f	/* G3 AREA      */
#endif

#ifdef EUC
static uchar  gucCodeType = CD_TYPE_EUC;
#else
static uchar  gucCodeType = CD_TYPE_SJIS;
#endif

static uchar kigo[]=
{' '
,'!','"','#','$','%','&','\'','(',')','*'
,'+',',','-','.','/',':',';','<','=','>'
,'?','@','[','\\',']','^','_','`','{','|'
,'}','~','\'','"','\'','"','\0'};

static ushort dkigo[]=
{0x2121
,0x212a,0x2149,0x2174,0x2170,0x2173,0x2175,0x2147,0x214a,0x214b,0x2176
,0x215c,0x2124,0x215d,0x2125,0x213f,0x2127,0x2128,0x2163,0x2161,0x2164
,0x2129,0x2177,0x214e,0x216f,0x214f,0x2130,0x2132,0x212e,0x2150,0x2143
,0x2151,0x2141,0x216c,0x216d,0x212d,0x212f,0x0000};

typedef struct {
	short flag;
	short hkey;
	short next;
	short data;
} HaslCell;

#define HASL_MSO	29

static HaslCell gCell[41]=
{0,   0, 0,  0
,1,8526, 0, 91
,1,8527,35, 93
,1,8528,39,123
,1,8529, 0,125
,1,8559, 0, 92
,1,8560, 0, 36
,0,   0, 0,  0
,0,   0, 0,  0
,1,8563, 0, 37
,1,8564, 0, 35
,1,8565, 0, 38
,1,8566, 0, 42
,1,8567, 0, 64
,1,8481, 0, 32
,1,8540,31, 43
,1,8541, 0, 45
,1,8484,37, 44
,1,8485, 0, 46
,1,8515, 0,124
,1,8487,32, 58
,1,8488, 0, 59
,1,8547,34, 60
,1,8490,30, 33
,0,   0, 0,  0
,1,8521, 0, 34
,1,8522,40, 40
,1,8523,36, 41
,1,8495, 0, 34
,1,8496, 0, 94
,1,8519,33, 39
,1,8511, 0, 47
,1,8545, 0, 61
,1,8548, 0, 62
,1,8489, 0, 63
,1,8498,38, 95
,1,8494, 0, 96
,1,8513, 0,126
,1,8556, 0, 39
,1,8557, 0, 34
,1,8493, 0, 39};

typedef struct {
	ushort knj_s;
	ushort knj_e;
	char   step;
	uchar  kna_s;
	uchar  kna_e;
	uchar  dakuten;
} tdtKanaKan;

static tdtKanaKan kanakan[]=
{0x2522,0x252a,2,0xb1,0xb5,0
,0x2521,0x2529,2,0xa7,0xab,0
,0x252b,0x2533,2,0xb6,0xba,0
,0x252c,0x2534,2,0xb6,0xba,0xde
,0x2535,0x253d,2,0xbb,0xbf,0
,0x2536,0x253e,2,0xbb,0xbf,0xde
,0x253f,0x2541,2,0xc0,0xc1,0
,0x2543,0x2543,1,0xaf,0xaf,0
,0x2544,0x2544,1,0xc2,0xc2,0
,0x2546,0x2548,2,0xc3,0xc4,0
,0x2540,0x2542,2,0xc0,0xc1,0xde
,0x2545,0x2545,1,0xc2,0xc2,0xde
,0x2547,0x2549,2,0xc3,0xc4,0xde
,0x254a,0x254e,1,0xc5,0xc9,0
,0x254f,0x255b,3,0xca,0xce,0
,0x2550,0x255c,3,0xca,0xce,0xde
,0x2551,0x255d,3,0xca,0xce,0xdf
,0x255e,0x2562,1,0xcf,0xd3,0
,0x2564,0x2568,2,0xd4,0xd6,0
,0x2563,0x2567,2,0xac,0xae,0
,0x2569,0x256d,1,0xd7,0xdb,0
,0x256f,0x256f,1,0xdc,0xdc,0
,0x2570,0x2570,1,0xb2,0xb2,0
,0x2571,0x2571,1,0xb4,0xb4,0
,0x2572,0x2572,1,0xa6,0xa6,0
,0x2573,0x2573,1,0xdd,0xdd,0
,0x2574,0x2574,1,0xb3,0xb3,0xde
,0x2575,0x2575,1,0xb6,0xb6,0
,0x2576,0x2576,1,0xb9,0xb9,0
,0x213c,0x213c,1,0xb0,0xb0,0
,0x213d,0x213e,1,0x2d,0x2d,0
,0,0,0,0,0,0};

static ushort _kigo2zen(uc)
uchar uc;
{
	ushort us;

	if (uc == ' ') us = 0x2121;
	else if (uc>='!' && uc<='/') us = dkigo[uc-'!'+1];
	else if (uc>=':' && uc<='@') us = dkigo[uc-':'+16];
	else if (uc>='[' && uc<='`') us = dkigo[uc-'['+23];
	else if (uc>='{' && uc<='~') us = dkigo[uc-'{'+29];
	else us = 0;

	return us;
}

static uchar _zen2kigo(us)
short us;
{
	int i,n;
	HaslCell *cell,*ce;

	cell = gCell;
	i = (n=us)%HASL_MSO+1;
L10:
	ce = &cell[i];
	if (ce->hkey == us) return (uchar)ce->data;
	else if ((i=ce->next)>0) goto L10;

	return 0;
}

static int _kanjikana(us,ucp)
ushort us;
uchar  *ucp;
{
	tdtKanaKan *p;
	ushort wus,wus_e;
	uchar uc;
	int byte,k,kk;

	byte = 0;
	p = kanakan;
	while (p->knj_s) {
		uc = p->kna_s;
		wus_e = p->knj_e;
		k = p->step;
		kk = 0;
		if (uc < p->kna_e) kk++;
		for (wus=p->knj_s; wus<=wus_e; wus+=k,uc+=kk) {
			if (us == wus) {
				*ucp++ = uc;
				byte++;
				if (uc = p->dakuten) {
					*ucp++ = uc;
					byte++;
				}
				break;
			}
		}
		if (byte) break;
		p++;
	}
	return byte;
}

static int _kanakanji(uc,uc1,usp)
uchar uc,uc1;
ushort *usp;
{
	tdtKanaKan *p;
	uchar uc_s,uc_e;
	ushort us;
	int byte,k;

	us = byte = 0;
	p = kanakan;
	if (uc1!=0xde && uc1!=0xdf) uc1 = 0;
	while (p->knj_s) {
		if (uc1 == p->dakuten) {
			uc_s = p->kna_s;
			uc_e = p->kna_e;
			if (uc>=uc_s && uc<=uc_e) {
				us = p->knj_s + (uc-uc_s)*p->step;
				byte++;
				if (uc1) byte++;
				break;
			}
		}
		p++;
	}
	if (!us && uc1) byte = _kanakanji(uc,0,usp);
	else if (usp) *usp = us;
	return byte;
}

int akxt_set_code_type(ucCodeType)
uchar ucCodeType;
{
	return gucCodeType = ucCodeType;
}

int akxt_get_code_type()
{
	return gucCodeType;
}

int akxqissjis(c)
uchar c;
{
	if ((c>=0x81 && c<=0x9f) || (c>=0xe0 && c<=0xfc))
		return 2;
	else
		return 0;
}

int akxqkanjilen1(c1)
uchar c1;
{
	return akxqmbslen1(gucCodeType,c1);
}

int akxqmbslen1(code_type,c1)
uchar code_type,c1;
{
	uchar p[4];

	p[0] = c1;
	if (code_type == CD_TYPE_EUC) {
		/************************/
		/*	EUC					*/
		/************************/
		p[1] = 0xa1;
	}
	else if (code_type == CD_TYPE_SJIS) {
		/************************/
		/*	SJIS				*/
		/************************/
		p[1] = 0x40;
	}
	else if (code_type == CD_TYPE_UTF8) {
		/************************/
		/*	UTF-8				*/
		/************************/
	}
	return akxqmbslen(code_type,p);
}

int akxqkanjilen(p)
uchar *p;
{
	return akxqmbslen(gucCodeType,p);
}

int akxqkanjilen2(p,len)
uchar *p;
int   len;
{
	return akxqmbsnlen(gucCodeType,p,len);
}

int akxqmbsnlen(code_type,p,len)
uchar code_type,*p;
int len;
{
	uchar c1,c2;

	if (!p || len<=0) return 0;
	if (!code_type) code_type = gucCodeType;

	if (code_type == CD_TYPE_EUC) {
		/**************************/
		/*    EUC                 */
		/**************************/
		c1 = *p++;
		c2 = *p;
		if (c1==SS3) return 3;
		else if ((c1==SS2) ||
			((c1>=0xa1) && (c1<=0xfe) && (c2>=0xa1) && (c2<=0xfe))) return 2;
	}
	else if (code_type == CD_TYPE_SJIS) {
		/**************************/
		/*    SJIS                */
		/**************************/
		c2 = *(p+1);
		if (akxqissjis(*p) &&
		    ((c2>=0x40 && c2<=0x7e)||(c2>=0x80 && c2<=0xfc))) return 2;
	}
	else if (code_type == CD_TYPE_UTF8) {
		/**************************/
		/*    UTF-8               */
		/**************************/
		return akxqu8nlen(*p,len);
	}
	return 1;
}

int akxqmbslen(code_type,p)
uchar code_type,*p;
{
	int len;

	if ((len=akxqmbsnlen(code_type,p,4)) < 0) len = -len;
	return len;
}

int akxqiskanji1(c1)
uchar c1;
{
	return akxqismbs1(gucCodeType,c1);
}

int akxqismbs1(code_type,c1)
uchar code_type,c1;
{
	int len;

	if ((len=akxqmbslen1(code_type,c1)) == 1) len = 0;
	return len;
}

int akxqiskanji(p)
uchar *p;
{
	return akxqismbs(gucCodeType,p);
}

int akxqismbs(code_type,p)
uchar code_type,*p;
{
	int len;

	if ((len=akxqmbslen(code_type,p)) == 1) len = 0;
	return len;
}

#if 1
int akxqisank_type(int type, char c)
{
	int rc;
	uchar uc;

	rc = 0;
	uc = (uchar)c;
	if (type == CD_TYPE_SJIS) {
		if ((uc>=0x20 && uc<=0x7e) || (uc>=0xa1 && uc<=0xdf)) rc = 1;
	}
/*	else if (type==CD_TYPE_EUC I1 type==CD_TYPE_UTF8) I */
	else {
		if (uc>=0x20 && uc<=0x7e) rc = 1;
	}
	return rc;
}

int akxqisank(char c)
{
	return akxqisank_type(gucCodeType, c);
}

char akxctoank_type(int type, char c)
{
	if (!akxqisank_type(gucCodeType, c)) c = '.';
	return c;
}

char akxctoank(char c)
{
	return akxctoank_type(gucCodeType, c);
}
#else
int akxqisank(c)
uchar c;
{
	if (gucCodeType==CD_TYPE_EUC || gucCodeType==CD_TYPE_UTF8) {
		/************************/
		/*	EUC or UTF-8		*/
		/************************/
		if ((c>=0x20 && c<=0x7e))
			return (1);
		else
			return (0);
	}
	else if (gucCodeType == CD_TYPE_SJIS) {
		/************************/
		/*	SJIS				*/
		/************************/
		if ((c>=0x20 && c<=0x7e) ||
		    (c>=0xa1 && c<=0xdf) )
			return (1);
		else
			return (0);
	}
}

char akxctoank(c)
uchar c;
{
	if (akxqisank(c))
		return ((char)c);
	else
		return ('.');
}
#endif

#define SO	0x0e
#define SI	0x0f

int akxqmbsnalen(p,len,attr)
int len;
uchar *p,attr[];
{
	uchar c1,c2,code_type;
	int type,mode,nb,bits;

	if (!p || len<0) return -1;
	if (!len) return 0;
	if (attr) {
		if (!(code_type = attr[0])) {
			attr[0] = code_type = gucCodeType;
		}
	}
	else code_type = gucCodeType;
	type = code_type;

	if ((type == CD_TYPE_EUC) || (type == CD_TYPE_SJIS) ||
	    (type == CD_TYPE_UTF8)) return akxqmbsnlen(code_type,p,len);
	else if (type == CD_TYPE_JIS) {
		if (!attr) return -2;
		if (len == 1) {
			mode = 1;
			nb = 1;
			bits = 8;
		}
		else {
			mode = attr[1];
			c1 = *p++;
			if (c1 == SO || c1 == SI) {
				mode = 1;
				if (c1 == SI) bits = 7;
				else if (c1 == SO) bits = 8;
			}
			else if (len>=3 && c1 == 0x1b) {
				c1 = *p++;
				c2 = *p;
				if (c1 == '$') {
					mode = 2;
					bits = 16;
					if (c2=='@' || c2=='B') nb = 5;
					else {
						mode = 0;
						nb = 2;
					}
				}
				else if (c1 == '(') {
					mode = 1;
					bits = 8;
					if (c2=='B' || c2=='J') nb = 4;
					else if (c2=='I') {
						nb = 4;
						bits = 7;
					}
					else {
						mode = 0;
						nb = 2;
					}
				}
				else {
					mode = 0;
					nb = 1;
				}
			}
			else {
				if (mode <= 1) {
					mode = nb = 1;
				}
				else {
					mode = nb = 2;
				}
			}
		}
		attr[1] = mode;
		attr[2] = nb;
		attr[3] = bits;
	}
	else return -3;
	return nb;
}

int akxtmget(p,len,ssp)
char *p;
int len;
SSP_S *ssp;	/* ssp.attr[0]:kanji code kind    */
			/*     attr[1]:mode 1=ANK,2=KANJI */
			/*     attr[2]:got byte */
			/*     attr[3]:7/8/16 bit */
{
	int m,mode,nb,sp;
	char *wd;
	uchar *attr;

	if (!p || !ssp) return -1;
	sp = ssp->sp;
	wd = ssp->wd;
	attr = ssp->attr;
	p += sp;
	len -= sp;
	m = akxqmbsnalen(p,len,attr);
	if (m <= 0) return m;
	if (attr[0] == CD_TYPE_JIS) {
		mode = attr[1];
		nb   = attr[2];
		sp += nb;
		while (!mode) {
			if (!p[nb-1]) return 0;
			p += nb;
			len -= nb;
			m = akxqmbsnalen(p,len,attr);
			if (m < 0) return m;
			mode = attr[1];
			nb   = attr[2];
			sp += nb;
		}
		if(wd) memcpy(wd,p,nb);
		m = nb;
	}
	else {
		if (m <= 0) m = 1;
		if(wd) memcpy(wd,p,m);
		sp += m;
	}
	ssp->sp = sp;
	return m;
}

int akxtmput(p,len,ssp)
char *p;
int len;
SSP_S *ssp;	/* ssp.attr[0]:ESC mode*/
			/*     attr[1]:mode 1=ANK,2=KANJI */
			/*     attr[2]:put len */
			/*     attr[3]:before mode */
{
	int m,mode,bmode,sp,esc_mode,nb;
	char *wd;
	uchar *attr;

	if (!p || !ssp) return -1;
	sp = ssp->sp;
	len -= sp;
	nb = 0;
	if (len > 0) {
		wd = ssp->wd;
		attr = ssp->attr;
		esc_mode = attr[0];
		mode = attr[1];
		m = attr[2];
		bmode = attr[3];
		p += sp;
		if (mode == 2) {
			if (bmode == 2) {
				memcpy(p,wd,m);
				nb = m;
			}
			else if (len < 5) nb = 0;
			else {
				*p++ = 0x1b;
				*p++ = '$';
				if (esc_mode) *p++ = 'B';
				else          *p++ = '@';
				memcpy(p,wd,m);
				nb = m + 3;
			}
		}
		else {
			if (bmode <= 1) {
				if (len < 4) nb = 0;
				else {
					*p++ = 0x1b;
					*p++ = '(';
					if (esc_mode) *p++ = 'J';
					else          *p++ = 'B';
					memcpy(p,wd,m);
					nb = m + 3;
				}
			}
			else if (len < 2) nb = 0;
			else {
				memcpy(p,wd,2);
				nb = 2;
			}
		}
		attr[3] = mode;
		ssp->sp = sp + nb;
	}
	return nb;
}

/****************************/
/*	S-JIS --> JIS			*/
/****************************/
int akxcstoj(n, inc, outc)
int		n;
uchar *inc, *outc;
{
	int m,len=n,nout,nb;
	SSP_S ssp;
	uchar *pi=inc,w[4];

	ssp.sp = 1;
	ssp.attr[0] = 0;
	ssp.attr[3] = 0;
	while (len>0) {
		m = akxqmbsnlen(CD_TYPE_SJIS,pi,len);
		ssp.attr[1] = m;
		ssp.attr[2] = m;
		if (m == 2) {
			akxcstoj1(pi,w);
			ssp.wd = w;
		}
		else ssp.wd = pi;
		pi += m;
		len -= m;
		nb = akxtmput(outc,INT_MAX,&ssp);
	}
	return ssp.sp;
}

/****************************/
/*	JIS --> S-JIS			*/
/****************************/
int akxcjtos(n, inc, outc)
int		n;
uchar *inc, *outc;
{
	int m,len,nb;
	SSP_S ssp;
	uchar *po=outc,*w;

	ssp.sp = 1;
	ssp.attr[0] = CD_TYPE_JIS;
	len = 0;
	while ((nb=akxtmget(inc,n,&ssp)) > 0) {
		m = ssp.attr[1];
		w = ssp.wd+(nb-m);
		if (m == 2) akxcjtos1(w,po);
		else *po = *w;
		po += m;
		len += m;
	}
	return len;
}

/****************************/
/*	EUC --> S-JIS			*/
/****************************/
static ushort sj_to_euc();
static ushort jis_to_sj();

#define	Z_BEGIN		0xa1	/* Zenkaku Start */
#define	Z_END		0xfe	/* Zenkaku End */
#define AKX_DEFAULT_CODE	0x3f3f	/* '??' */;

int akxcetos(n, inc, outc)
int		n;
unsigned char *inc, *outc;
{
	int i, j, stat, max=n;
	ushort c,c1,c2;
	ushort s1,s2;
	uchar *pi=inc,*po=outc;

	stat = j = 0;
	for (i=0;i<max;i++) {
		c = *pi++;
		switch (stat) {
		case 0:
			if (Z_BEGIN <= c  && c <= Z_END) { /* Zenkaku */
				c1 = c & 0x7f;
				stat = 1;
			}
			else if (c == SS2) {	 /* Hankaku Kana */
				stat = 2;
			}
			else {	/* ASCII */
				*po++ = c;
				j++;
			}
			break;
		case 1:
			stat = 0;
			if (!(c & 0x80)) {
				*po++ = '?';
				*po++ = '?';
				j += 2;
				break;
			}
		 	c2 = c & 0x7f;
			jis_to_sj(c1,c2,&s1,&s2); /* Convert JIS to Sift_JIS */
			*po++ = s1;
			*po++ = s2;
			j += 2;
			break;
		case 2:
			*po++ = c;
			j++;
			stat = 0;
			break;
		}
	}
	if (stat) j = -j;
	return j;
}

static ushort jis_to_sj(jis1,jis2,ibm1,ibm2)
ushort jis1,jis2,*ibm1,*ibm2;
{
	ushort sj1,sj2;

	if(jis1 >= 0x5f) jis1 += 0x80;

	if(jis1 & 0x01) {
		if(jis2 >= 0x60) jis2 = jis2 + 0x01;
		sj1 = (jis1 - 0x31)/2 + 0x89;
		sj2 = jis2 + 0x1f;
	}
	else {
		sj1 = (jis1 - 0x30)/2 + 0x88;
		sj2 = jis2 + 0x7e;
	}
	sj1 &= 0xff;
	sj2 &= 0xff;
	*ibm1 = sj1;
	*ibm2 = sj2;
	return ntohs(sj1<<8 + sj2);
}

/****************************************/
/*	S-JIS --> EUC  Convert Routine		*/
/*	opt=0:only replace					*/
/*	rep=NULL:replace default			*/
/*	   =""  :suppress					*/
/****************************************/
int akxcstoe2(n, inc, outc, opt, rep)
uchar *inc,*outc,*rep;
int n,opt;
{
	int i, j, stat, replen=-1,max=n;
	ushort c,c1;
	ushort s1,s2,us;
	uchar w[3];
	uchar *pi=inc,*po=outc;

	if (rep) replen = strlen(rep);
	stat = j = 0;
	for (i=0;i<max;i++) {
		c = *pi++;
		switch (stat) {
		case 0:
			if((c>=0x81 && c<=0x9f) || (c>=0xe0 && c<=0xfc)) {
				c1 = c;			/* ZENKAKU */
				stat = 1;
			}
			else if((0xa1 <= c) && (c <= 0xdf)) { /* Hankaku Kana */
				*po++ = SS2;
				*po++ = c;
				j += 2;
			}
			else { 				/* ASCII */
				*po++ = c;
				j++;
			}
			break;
		case 1:
			stat = 0;
			if (c<0x40 || c==0x7f || c>0xfc) {	/* invalid */
				*po++ = '?';
				j++;
				i--;
			}
			else {
				if (c1<0xf0) {
					sj_to_euc(c1,c,&s1,&s2);/* Change Shift_JIS into EUC */
				}
				else {
					if (replen<0) {
						us = AKX_DEFAULT_CODE;
						s1 = us>>8;
						s2 = us & 0x00ff;
					}
					else {
						if (replen>0) {
							memcpy(po,rep,replen);
							j += replen;
							po += replen;
						}
						continue;
					}
				}
				*po++ = s1;
				*po++ = s2;
				j += 2;
			}
			break;
		}
	}
	return j;
}

int akxcstoe(n, inc, outc)
int				n;
uchar	*inc, *outc;
{
	return akxcstoe2(n, inc, outc, 0, NULL);
}

static ushort sj_to_euc(sj1,sj2,euc1,euc2)
ushort sj1,sj2,*euc1,*euc2;
{
	ushort eu1,eu2;

	if (sj1 >= 0xe0) sj1 -=  0x40;	/* The 2nd standard of JIS */

	if (sj2 >= 0x9f) {
		eu1 = (sj1 - 0x88)*2 + 0xb0;
		eu2 = sj2 + 0x02;
	}
	else {
		if (sj2 >= 0x7f) sj2 -= 0x01;
		eu1 = (sj1 - 0x89)*2 + 0xb1;
		eu2 = sj2 + 0x61;
	}
	*euc1 = eu1;
	*euc2 = eu2;
	return ntohs(eu1<<8 + eu2);
}

int akxcetos1(euc,sjis)
uchar *sjis,*euc;
{
	int ret;

	ret = jis_to_sj(*euc & 0x7f,*(euc+1) & 0x7f,sjis1,sjis2);
	*sjis++ = sjis1;
	*sjis   = sjis2;
	return ret;
}

int akxcstoe1(sjis,euc)
uchar *sjis,*jis;
{
	int ret;

	ret = sj_to_euc(*sjis,*(sjis+1),euc1,euc2);
	*euc++ = euc1;
	*euc   = euc2;
	return ret;
}

int akxcjtos1(jis,sjis)
uchar *jis,*sjis;
{
	ushort c1,c2;

	c1 = *jis++;
	c2 = *jis;

	if (c1 & 1) {
		c1 = (c1 >> 1) + 0x71;
		c2 += 0x1f;
		if (c2 >= 0x7f) c2++;
	}
	else {
		c1 = (c1 >> 1) + 0x70;
		c2 += 0x7e;
	}
	if (c1 > 0x9f) c1 += 0x40;

	c1 &= 0xff;
	c2 &= 0xff;
	*sjis++ = c1;
	*sjis   = c2;

	return ntohs(c1<<8 | c2);
}

int akxcstoj1(sjis,jis)
uchar *sjis,*jis;
{
	ushort c1,c2;

	c1 = *sjis++;
	c2 = *sjis;

	c1 -= (c1 <= 0x9f) ? 0x70 : 0xb0;
	c1 <<= 1;
	if (c2 < 0x9f) {
		c2 -= (c2 < 0x7f) ? 0x1f : 0x20;
		c1--;
	}
	else
		c2 -= 0x7e;

	c1 &= 0xff;
	c2 &= 0xff;
	*jis++ = c1;
	*jis   = c2;

	return ntohs(c1<<8 | c2);
}

int akxcstou81(sjis,utf8)
uchar *sjis,*utf8;
{
	ushort us,usw;

	memcpy(&usw,sjis,2);
	us = ntohs(usw);
	return akxc_sj_to_utf8(us,utf8,NULL);
}

int akxcu8tos1(utf8,sjis)
uchar *utf8,*sjis;
{
	int len;
	ushort us,usw;

	len = akxc_utf8_to_sj(utf8,&us,NULL);
	if (len == 2) {
		usw = htons(us);
		memcpy(sjis,&usw,2);
	}
	else {
		*sjis = us;
	}
	return len;
}

/****************************************/
/*	Sp JIS --> type					*/
/****************************************/
int akxcfromjis(pd,ps,type)
char *ps,*pd;
int type;
{
	int ret;
	ushort us,usw;
	char w[4];

	if (type == CD_TYPE_UTF8) {
		akxcjtos1(ps,w);
		ret = akxcstou81(w,pd);
	}
	else {
		if (type == CD_TYPE_EUC) {
			pd[0] = ps[0] | 0x80;
			pd[1] = ps[1] | 0x80;
		}
		else if (type == CD_TYPE_SJIS) {
			akxcjtos1(ps,pd);
		}
		memcpy(&usw,pd,2);
		ret = ntohs(usw);
	}
	return ret;
}

/****************************************/
/*	Sp type --> JIS					*/
/****************************************/
ushort akxctojis(pd,ps,type)
char *ps,*pd;
int type;
{
	ushort us,usw;
	char w[4];

	if (type == CD_TYPE_EUC) {
		pd[0] = ps[0] & 0x7f;
		pd[1] = ps[1] & 0x7f;
	}
	else if (type == CD_TYPE_SJIS) {
		akxcstoj1(ps,pd);
	}
	else if (type == CD_TYPE_UTF8) {
		akxcu8tos1(ps,w);
		akxcstoj1(w,pd);
	}
	memcpy(&usw,pd,2);
	us = ntohs(usw);
	return us;
}

/****************************************/
/*	Zenkaku --> Hankaku					*/
/****************************************/
int akxctohan(n, inc, outc)
int  n;
char *inc,*outc;
{
	return akxctohan_opt(n, inc, outc, 0);
}

int akxctohan_opt(n, inc, outc, opt)
int  n,opt;
char *inc,*outc;
{
	int  len1,type,i,byte;
	char *p1,*p,w[5];
	uchar  uc;
	ushort us,usw;

	len1 = n;
	p1 = inc;
	p  = outc;
	type = akxt_get_code_type();
	for (i=0;i<len1;i++) {
		if (akxqiskanji(p1)) {
#if 1
			us = akxctojis(w,p1,type);
#else
			memcpy((char *)&usw,p1,2);
			if (type == CD_TYPE_EUC)
				usw &= 0x7f7f;
			else if (type == CD_TYPE_SJIS) {
				akxcstoj1(p1,&usw);
			}
			us = ntohs(usw);
#endif
			byte = 0;
			uc = 0;
			if (!(opt & 0x01) && us >= 0x2330 && us <= 0x2339) { /* 0-9 */
				uc = (us - 0x2330) + '0';
			}
			else if (!(opt & 0x02) && us >= 0x2341 && us <= 0x235A) { /* A-Z */
				uc = (us - 0x2341) + 'A';
			}
			else if (!(opt & 0x04) && us >= 0x2361 && us <= 0x237a) { /* a-z */
				uc = (us - 0x2361) + 'a';
			}
			else if (!(opt & 0x10) &&
			         ((us >= 0x2501 && us <= 0x2576) || /* @-  */
			          (us >= 0x213c && us <= 0x213e))) { /* [ \ ]  */
				byte = _kanjikana(us,p);
			}
			else if (!(opt & 0x08)) uc = _zen2kigo(us);
			if (byte) {
				p  += byte;
				p1 += 2;
			}
			else if (uc) {
				*p++ = uc;
				p1 += 2;
			}
			else {
				*p++ = *p1++;
				*p++ = *p1++;
			}
			i++;
		}
		else {
			*p++ = *p1++;
		}
	}
	*p = '\0';
	return strlen(outc);
}

/****************************************/
/*	Hankaku --> Zenkaku					*/
/****************************************/
int akxctozen(n, inc, outc)
int  n;
char *inc,*outc;
{
	int  len1,type,i,byte,out_len,nn[3];
	char *p1,*p;
	uchar  uc,uc1;
	ushort us,usw;

	len1 = n;
	p1 = inc;
	p  = outc;
	out_len = 0;
	type = akxt_get_code_type();
#if 1
	while (len1 > 0) {
		byte = akxctozen1_type(len1, p1, p, nn, type);
		if (outc) p += byte;
		p1 += nn[0];
		len1 -= nn[0];
		out_len += byte;
	}
	if (outc) outc[out_len] = '\0';
#else
	for (i=0;i<len1;i++) {
		if ((byte=akxqkanjilen(p1)) >= 2) {
			if (outc) {
				memcpy(p,p1,byte);
				p += byte;
				p1 += byte;
			}
			i += byte-1;
			out_len += byte;
		}
		else {
			uc = *p1++;
			us = 0;
			if ((uc >= '0' && uc <= '9') ||
			    (uc >= 'A' && uc <= 'Z') ||
			    (uc >= 'a' && uc <= 'z')) {
				if (outc) {
					if (!(opt & 0x01) && uc >= '0' && uc <= '9') {
						us = (uc - '0') + 0x2330;
					}
					else if (!(opt & 0x02) && uc >= 'A' && uc <= 'Z') {
						us = (uc - 'A') + 0x2341;
					}
					else if (!(opt & 0x04) && uc >= 'a' && uc <= 'z') {
						us = (uc - 'a') + 0x2361;
					}
				}
				else us = 0x2121;
			}
			else if (uc >= 0xa6 && uc <= 0xdd) {
				if (i < len1) uc1 = *p1;
				else uc1 = 0;
				byte = _kanakanji(uc,uc1,&us);
				if (byte == 2) {
					i++;
					p1++;
				}
			}
			else us = _kigo2zen(uc);

			if (us) {
				if (outc) {
					usw = htons(us);
					us = akxcfromjis(p,&usw,type);
					p += 2;
				}
				out_len += 2;
			}
			else {
				if (outc) *p++ = uc;
				out_len++;
			}
		}
	}
	if (outc) *p = '\0';
#endif
	return out_len;
}

int akxctozen1(n, inc, outc, nn)
int  n,nn[];
char *inc,*outc;
{
	int out_len;

	out_len = akxctozen1_type(n, inc, outc, nn, -1);
	if (outc) outc[out_len] = '\0';
	return out_len;
}

int akxctozen1_type(n, inc, outc, nn, type)
int  n,nn[],type;
char *inc,*outc;
{
	return akxctozen1_type_opt(n, inc, outc, nn, type, 0);
}

int akxctozen1_type_opt(n, inc, outc, nn, type, opt)
int  n,nn[],type,opt;
char *inc,*outc;
{
	int  len1,i,byte,out_len;
	char *p1,*p;
	uchar  uc,uc1;
	ushort us,usw;

	len1 = n;
	p1 = inc;
	p  = outc;
	out_len = 0;
	if (type < 0) type = akxt_get_code_type();
	if (len1 >= 2) {
		if ((byte=akxqkanjilen(p1)) >= 2) {
			nn[0] = byte;
			nn[1] = byte;
			nn[2] = 1;
			if (outc) {
				memcpy(outc,inc,byte);
				outc[byte] = '\0';
			}
			return byte;
		}
	}
	us = 0;
	byte = 1;
	uc = *p1++;
	if ((uc >= '0' && uc <= '9') ||
	    (uc >= 'A' && uc <= 'Z') ||
	    (uc >= 'a' && uc <= 'z')) {
		if (outc) {
			if (!(opt & 0x01) && uc >= '0' && uc <= '9') {
				us = (uc - '0') + 0x2330;
			}
			else if (!(opt & 0x02) && uc >= 'A' && uc <= 'Z') {
				us = (uc - 'A') + 0x2341;
			}
			else if (!(opt & 0x04) && uc >= 'a' && uc <= 'z') {
				us = (uc - 'a') + 0x2361;
			}
		}
		else us = 0x2121;
	}
	else if (!(opt & 0x10) && uc >= 0xa6 && uc <= 0xdd) {
		if (len1 > 1) uc1 = *p1;
		else uc1 = 0;
		byte = _kanakanji(uc,uc1,&us);
	}
	else if (!(opt & 0x08)) us = _kigo2zen(uc);

	if (us) {
		if (outc) {
			usw = htons(us);
			us = akxcfromjis(p,&usw,type);
			p += 2;
		}
		out_len += 2;
	}
	else {
		if (outc) *p++ = uc;
		out_len++;
	}
/*	if (outc) *p = '\0';	*/
	nn[0] = byte;
	nn[1] = out_len;
	nn[2] = byte;
	return nn[1];
}

int akxt_chk_lang_type(lang)
char *lang;
{
	int type=0;

	if (!strcmp(lang,"ja_JP") || !strcmp(lang,"ujis"))
		type = CD_TYPE_EUC;
	else if (!strcmp(lang,"Ja_JP")) type = CD_TYPE_SJIS;
	else if (!strcmp(lang,"JA_JP")) type = CD_TYPE_UTF8;
	else if (!strcmp(lang,"ja")) type = akxt_get_code_type();
	else {
		if (stristr(lang,"EUC")) type = CD_TYPE_EUC;
		else if (stristr(lang,"SJIS") || stristr(lang,"S-JIS"))
			type = CD_TYPE_SJIS;
		else if (stristr(lang,"UTF8") || stristr(lang,"UTF-8"))
			type = CD_TYPE_UTF8;
	}
	return type;
}

int akxt_get_lang_type(key,opt,buf,buflen)
char *key,*buf;
int  opt,buflen;
{
	static struct {
		char *lang;
		int   type;
	} reg[]={NULL,0,NULL,0};

	char *lang=NULL,*p;
	int  type=-1,i=-1;

	if (!key) key = "LANG";

	if (!strcmp(key,"LANG")) i = 0;
	else if (!strcmp(key,"NLS_LANG")) i = 1;

	if (!opt && i>=0) {
		if (lang = reg[i].lang) {
			type = reg[i].type;
			i = -2;
		}
	}

	if (!lang) {
		if (lang=getenv(key)) {
#if 1
			if (!(type=akxt_chk_lang_type(lang))) type = -1;
#else
			if (!strcmp(lang,"ja_JP") || !strcmp(lang,"ujis"))
				type = CD_TYPE_EUC;
			else if (!strcmp(lang,"Ja_JP")) type = CD_TYPE_SJIS;
			else if (!strcmp(lang,"JA_JP")) type = CD_TYPE_UTF8;
			else if (!strcmp(lang,"ja")) type = akxt_get_code_type();
			else {
				if (stristr(lang,"EUC")) type = CD_TYPE_EUC;
				else if (stristr(lang,"SJIS") || stristr(lang,"S-JIS"))
					type = CD_TYPE_SJIS;
				else if (stristr(lang,"UTF8") || stristr(lang,"UTF-8"))
					type = CD_TYPE_UTF8;
			}
#endif
		}
		else lang = "";
	}

	if (buf) strnzcpy(buf,lang,buflen-1);
	if (i >= 0) {
		if (p=reg[i].lang) Free(p);
		reg[i].lang = Strdup(lang);
		reg[i].type = type;
	}
	return type;
}

/****************************************/
/*										*/
/****************************************/
int akxt_code_trans(pParm, pParmd)
tdtGeneralData *pParm, *pParmd;
{
	int rc,len_s,len_d;
	uchar type_s,type_d,ds_d;
	char  *up_s,*up_d;

	type_s = pParm->gd_code;
	type_d = pParmd->gd_code;
	up_s = pParm->gd_data;
	up_d = pParmd->gd_data;
	len_s = pParm->gd_dlen;
	pParmd->gd_attr = AKX_ZOK_CHAR;
	if (up_d) ds_d = pParmd->gd_scale;
	else pParmd->gd_scale = ds_d = 0;
	pParmd->gd_dlen  = 0;
	if (len_s<=0 || type_s==type_d) {
		if (len_s>=0 && up_d) {
			memnzcpy(up_d,up_s,len_s,len_s);
		}
		return pParmd->gd_dlen=len_s;
	}
	if (type_s == CD_TYPE_SJIS) {
		if (type_d == CD_TYPE_EUC) {
			if (!up_d) {
				if (!(up_d=Malloc(len_s+1))) return -1;
				ds_d = AKX_DATA_MALLOC;
			}
			len_d = akxcstoe(len_s,up_s,up_d);
		}
		else if (type_d == CD_TYPE_UTF8) {
			if (!up_d) {
				if (!(up_d=Malloc(len_s*3+1))) return -1;
				ds_d = AKX_DATA_MALLOC;
			}
			len_d = akxcstou8(len_s,up_s,up_d);
		}
		else if (type_d == CD_TYPE_JIS) {
			if (!up_d) {
				if (!(up_d=Malloc(len_s*3+4))) return -1;
				ds_d = AKX_DATA_MALLOC;
			}
			len_d = akxcstoj(len_s,up_s,up_d);
		}
		else {
			return -1;
		}
	}
	else if (type_s == CD_TYPE_EUC) {
		if (type_d == CD_TYPE_SJIS) {
			if (!up_d) {
				if (!(up_d=Malloc(len_s+1))) return -1;
				ds_d = AKX_DATA_MALLOC;
			}
			len_d = akxcetos(len_s,up_s,up_d);
		}
		else {
			return -1;
		}
	}
	else if (type_s == CD_TYPE_UTF8) {
		if (type_d == CD_TYPE_SJIS) {
			if (!up_d) {
				if (!(up_d=Malloc(len_s+1))) return -1;
				ds_d = AKX_DATA_MALLOC;
			}
/*
printf("akxt_code_trans: len_s=%d up_s=%08x up_d=%08x\n",len_s,up_s,up_d);
*/
			len_d = akxcu8tos(len_s,up_s,up_d);
/*
printf("akxt_code_trans: len_d=%d\n",len_d);
*/
		}
		else {
			return -1;
		}
	}
	else if (type_s == CD_TYPE_JIS) {
		if (type_d == CD_TYPE_SJIS) {
			if (!up_d) {
				if (!(up_d=Malloc(len_s+1))) return -1;
				ds_d = AKX_DATA_MALLOC;
			}
			len_d = akxcjtos(len_s,up_s,up_d);
		}
		else {
			return -1;
		}
	}
	else {
		return -1;
	}
	pParmd->gd_data = up_d;
	pParmd->gd_scale = ds_d;
	if (len_d >= 0) {
		up_d[len_d] = '\0';
	}
	else if (ds_d & AKX_DATA_MALLOC) {
		Free(up_d);
		pParmd->gd_data = NULL;
		pParmd->gd_scale = 0;
	}
	return pParmd->gd_dlen=len_d;
}

/****************************************/
/*										*/
/****************************************/
int akxc_code_trans1(type_s,type_d,up_s,len_s,up_d)
int type_s,type_d,len_s;
uchar *up_s,*up_d;
{
	int i,ret;
	ushort us;
	uchar uc1,uc2;

	ret = len_s;
	if (!up_s || !up_d)
		return -1;
	else if (type_s == type_d) {
		memcpy(up_d,up_s,len_s);
		return len_s;
	}
	else if (type_s == CD_TYPE_UTF8)
		ret = akxcu8tos1(up_s,up_d);
	else if (len_s == 1) {
		*up_d = *up_s;
		return len_s;
	}
	else if (type_s == CD_TYPE_EUC) {
		if (type_d == CD_TYPE_JIS) {
			uc1 = (*up_s++) & 0x7f;
			uc2 =  *up_s    & 0x7f;
			*up_d++ = uc1;
			*up_d   = uc2;
			return len_s;
		}
		else
			us = akxcetos1(up_s,up_d);
	}
	else if (type_s == CD_TYPE_JIS) {
		if (type_d == CD_TYPE_EUC) {
			uc1 = (*up_s++) | 0x80;
			uc2 =  *up_s    | 0x80;
			*up_d++ = uc1;
			*up_d   = uc2;
			return len_s;
		}
		else
			us = akxcjtos1(up_s,up_d);
	}
	else
		return -1;

	if (type_d == CD_TYPE_EUC) {
		us = akxcstoe1(up_s,up_d);
	else if (type_d == CD_TYPE_UTF8)
		ret = akxcstou81(up_s,up_d);
	else if (type_d == CD_TYPE_JIS)
		us = akxcstoj1(up_s,up_d);
	else
		return -1;
	return ret;
}

/****************************************/
/*										*/
/****************************************/
int akxqmlen(s,slen)
char *s;
{
	char *p=s;
	int len=slen;
	int count,m;

	count = 0;
	while (len > 0) {
		if (len >= 2) {
			m = akxqkanjilen(p);
			len -= m;
			p += m;
		}
		else {
			len--;
			p++;
		}
		count++;
	}
	return count;
}

/****************************************/
/*										*/
/****************************************/
int akxqm2len(s,slen,mlen)
char *s;
{
	char *p=s;
	int len=slen;
	int i,count,m;

	count = 0;
	for (i=0;i<mlen && len>0;i++) {
		if (len >= 2) {
			m = akxqkanjilen(p);
			len -= m;
			p += m;
			count += m;
		}
		else {
			len--;
			p++;
			count++;
		}
	}
	return count;
}

/****************************************/
/*										*/
/****************************************/
int akxqstrlen(s)
char *s;
{
	return akxqmlen(s,strlen(s));
}

/********************************************************/
/*														*/
/********************************************************/
int akxcuppern(d,s,len)
char *d,*s;
int len;
{
	return  akxcuplwn(0,d,s,len);
}

/********************************************************/
/*														*/
/********************************************************/
int akxclowern(d,s,len)
char *d,*s;
int len;
{
	return  akxcuplwn(1,d,s,len);
}

/********************************************************/
/*														*/
/********************************************************/
int akxcuplwn(opt,d,s,len)
int opt;	/* 0:upper, 1:lower */
char *d,*s;
int len;
{
	char *pd,*ps;
	int  n,i;

	if (d && s) {
		pd = d;
		ps = s;
	}
	else {
		if (s) {
			pd = ps = s;
		}
		else if (d) {
			pd = ps = d;
		}
		else return -1;
	}
	for (i=0;i<len;) {
		n = akxcuplw(pd,ps,opt);
		ps += n;
		pd += n;
		i += n;
	}
	*pd = '\0';
	return len;
}

/********************************************************/
/*														*/
/********************************************************/
int akxcuplw(pd,ps,opt)
char *pd,*ps;
int opt;	/* 0:upper, 1:lower */
{
	int n,type;
	ushort us,usw;
	char c,cc,w[5];

	if (n=akxqiskanji(ps)) {
		type = akxt_get_code_type();
		us = akxctojis(w,ps,type);
/*
printf("akxcuplw: us=0x%02x\n",us);
*/
		usw = 0;
		if (us >= 0x2341 && us <= 0x235A) { /* A-Z */
			if (opt) usw = 0x2361 + us - 0x2341;
		}
		else if (us >= 0x2361 && us <= 0x237a) { /* a-z */
			if (!opt) usw = 0x2341 + us - 0x2361;
		}
		if (usw) {
/*
printf("akxcuplw: usw=0x%02x\n",usw);
*/
			us = htons(usw);
			memcpy(w,(char *)&us,2);
			us = akxcfromjis(pd,w,type);
		}
		else if (pd != ps) memcpy(pd,ps,n);
	}
	else {
		c = *ps;
		if (opt) cc = tolower(c);
		else cc = toupper(c);
		*pd = cc;
		n = 1;
	}
	return n;
}

/****************************/
/*	UTF-8					*/
/****************************/
#define AKX_SJIS_START		0x8140
#define AKX_SJIS_OFFSET		(AKX_SJIS_START-256)
#ifdef UTF8_SRC
#define AKX_SJIS_AKI_START	0xA000
#define AKX_SJIS_START2		0xE040
#define AKX_SJIS_OFFSET2	(AKX_SJIS_START2-AKX_SJIS_AKI_START)
#define AKX_SJIS_MAX		(65536-AKX_SJIS_OFFSET-AKX_SJIS_OFFSET2)
#define AKX_SJIS_HASH_MAX	11500
#define AKX_SJIS_HASH_PRE	11467
#endif

static int gexe_set_utf8=0;
static UINT4 *gucs4=NULL;
static UINT4 *gutf8=NULL;
static HASHB *ha=NULL;
static HASHB *ha4=NULL;
#ifdef UTF8_SRC

static int _init_sj_utf8();
static int _Unihaslr();

int akxc_set_utf8_file(file)
char *file;
{
	return 0;
}

int akxc_set_sj_utf8(file)
char *file;
{
	return 0;
}
#else
static char *hkey[2],*h4key[2];
static char *utf8_file;

int akxc_set_utf8_file(file)
char *file;
{
	if (!(utf8_file=Strdup(file))) return -1;
	return 0;
}

int akxc_set_sj_utf8(file)
char *file;
{
	FILE *fp;
	char buf[128],parm[256],*argv[3],*p,c,*hkey[2],*h4key[2];
	int  len,n,ret,offset,iha4;
	ushort ucs2;
	UINT4  val,sj,utf8,ucs4,ix;

	if (!file) return -1;
	if (!(fp=fopen(file,"r"))) {
		XERROROUTL5(200,"%s open error",file,0,0,0,0);
		return -2;
	}

	gexe_set_utf8 = 1;
	len = AKX_SJIS_MAX*sizeof(UINT4)*2;
	if (!gucs4) {
		if (!(gucs4=(UINT4 *)Malloc(len))) {
			XERROROUTL5(200,"gucs4 malloc error",0,0,0,0,0);
			ret = -3;
			goto Err;
		}
	}
	memset(gucs4,0,len);
	gutf8 = gucs4 + AKX_SJIS_MAX;

	if (ha) akxs_hasl_free(ha);
	ha = akxs_hasl_new(4,AKX_SJIS_HASH_MAX,0);
	if (!ha) {
		XERROROUTL5(200,"akxs_hasl_new errno=%d",errno,0,0,0,0);
		ret = -3;
		goto Err;
	}
/*
printf("UTF8: id=%c%c len=%d maxreg=%d mso=%d\n",
ha->ha_id[0],ha->ha_id[1],ha->ha_keylen,ha->ha_maxreg,ha->ha_prereg);
*/
	ha->ha_key = (char *)hkey;
	if (ha4) akxs_hasl_free(ha4);
	ha4 = akxs_hasl_new(4,AKX_SJIS_HASH_MAX,0);
	if (!ha4) {
		XERROROUTL5(200,"akxs_hasl_new errno=%d",errno,0,0,0,0);
		ret = -4;
		goto Err;
	}
/*
printf("UCS4: id=%c%c len=%d maxreg=%d mso=%d\n",
ha->ha_id[0],ha->ha_id[1],ha->ha_keylen,ha->ha_maxreg,ha->ha_prereg);
*/
	ha4->ha_key = (char *)h4key;

printf("--sjis-- --ucs4-- --utf8-- - ix- -iha- -iha4\n");

	while ((len = akxa_read_line(buf,sizeof(buf),fp)) > 0) {
		if ((c=*buf) == '#') continue;
		else if (c == '\t') offset = 1;
		else offset = 8;
		n = akxtgetargv2(buf+offset,argv,3,parm,sizeof(parm),1);
		if (n >= 2) {
			p = argv[0] + 2;
			ret = akxccvx(p,strlen(p),&val);
			sj = val;
			if (n>=3 && *argv[2]=='(') p = argv[2] + 1;
			else p = argv[1] + 2;
			ret = akxccvx(p,strlen(p),&val);
			ucs4 = val;
			if (ucs4 <= 0x7f) {
				utf8 = ucs4<<24;
			}
			else if (ucs4 <= 0x7ff) {
				utf8 = ((ucs4&0x07c0)<<2 | (ucs4&0x3f) | 0xc080)<<16;
			}
			else if (ucs4 <= 0xffff) {
				utf8 = ((ucs4&0xf000)<<4 | (ucs4&0x0fc0)<<2 | (ucs4&0x3f) |
				        0xe08080)<<8;
			}
			else {
				utf8 = (ucs4&0x1c0000)<<6 | (ucs4&0x3f000)<<4 |
				       (ucs4&0xfc0)<<2 | (ucs4&0x3f) | 0xf0808080;
			}
			if ((ix=sj) >= AKX_SJIS_START) ix = sj - AKX_SJIS_OFFSET;
			gucs4[ix] = ucs4;
			gutf8[ix] = utf8;
			hkey[1] = (char *)sj;
			ret = akxshasls(ha,utf8);
			h4key[1] = (char *)sj;
			iha4 = akxshasls(ha4,ucs4);

printf("%08x %08x %08x %5d %5d %5d\n",sj,ucs4,utf8,ix,ret,iha4);

		}
	}
	ret = 0;
Err:
	fclose(fp);
	return ret;
}
#endif

int akxc_sj_to_utf8(sjis,utf8c,ucs4)
ushort  sjis;
uchar  *utf8c;
UINT4  *ucs4;
{
	int len,ret;
	UINT4 w,utf8,ix,sj,ucs4w;

	if (!gexe_set_utf8) {
#ifdef UTF8_SRC
		if (ret=_init_sj_utf8()) return ret;
#else
		if (ret=akxc_set_sj_utf8(utf8_file)) return ret;
#endif
	}
	if (!gutf8) return -1;
	sj = sjis;
#ifdef UTF8_SRC
	if ((ix=sj) >= AKX_SJIS_START) {
		ix = sj - AKX_SJIS_OFFSET;
		if (sj>=AKX_SJIS_START2) ix -= AKX_SJIS_OFFSET2;
	}
#else
	if ((ix=sj) >= AKX_SJIS_START) ix = sj - AKX_SJIS_OFFSET;
#endif
/*
printf("akxc_sj_to_utf8: %04x %08x\n",sjis,utf8);
*/
	if (utf8c) {
		utf8 = gutf8[ix];
		len = akxqu8len((utf8>>24) & 0xff);
		w = htonl(utf8);
		memcpy(utf8c,&w,len);
	}
	else len = 0;
	if (ucs4) *ucs4 = gucs4[ix];
	return len;
}

int akxcstou8(n, inc, outc)
int   n;
uchar *inc, *outc;
{
	int i,j,len;
	uchar  c1;
	ushort sjis;

	j = 0;
	for (i=0;i<n;i++,inc++) {
		sjis = c1 = *inc;
		if (akxqiskanji1(c1)) {
			if (i+1 < n) {
				if (akxqiskanji(inc)) {
					i++;
					sjis = sjis<<8 | *(++inc);
				}
			}
			else break;
		}
		len = akxc_sj_to_utf8(sjis,outc,NULL);
		outc += len;
		j    += len;
	}
	return j;
}

int akxc_utf8_to_sj(utf8c,sjis,ucs4)
uchar  *utf8c;
ushort *sjis;
UINT4  *ucs4;
{
	int len,ret;
	UINT4 w,utf8,ix,sj,ucs4w;
	char **argv;

	if (!gexe_set_utf8) {
#ifdef UTF8_SRC
		if (ret=_init_sj_utf8()) return ret;
#else
		if (ret=akxc_set_sj_utf8(utf8_file)) return ret;
#endif
	}
	if (!ha) return -1;
	len = akxqu8len(*utf8c);
	sj = ucs4w = w = 0;
	memcpy(&w,utf8c,len);
	utf8 = ntohl(w);
#ifdef UTF8_SRC
	if ((ret = _Unihaslr(ha,utf8)) > 0) {
#else
	if ((ret = akxshaslr(ha,utf8)) > 0) {
#endif
/*
printf("akxc_utf8_to_sj: ha->ha_key=%08x hkey=%08x\n",ha->ha_key,hkey);
*/
#ifdef UTF8_SRC
		sj = (UINT4)ha->ha_next;
		if ((ix=sj) >= AKX_SJIS_START) {
			ix = sj - AKX_SJIS_OFFSET;
			if (sj>=AKX_SJIS_START2) ix -= AKX_SJIS_OFFSET2;
		}
#else
		if (!(argv=(char **)ha->ha_key)) return -1;
		sj = (UINT4)argv[1];
		if ((ix=sj) >= AKX_SJIS_START) ix = sj - AKX_SJIS_OFFSET;
#endif
		ucs4w = gucs4[ix];
	}
	if (sjis) *sjis = sj;
	if (ucs4) *ucs4 = ucs4w;
/*
printf("akxc_utf8_to_sj: utf8=%08x len=%d ih=%5d sj=%04x ucs4=%04x\n",utf8,len,ret,sj,ucs4w);
*/
	return len;
}

int akxcu8tos(n, inc, outc)
int   n;
uchar *inc, *outc;
{
	int i,j,len;
	ushort sjis;

	j = 0;
	for (i=0;i<n;i+=len) {
		if ((len = akxc_utf8_to_sj(inc,&sjis,NULL)) < 0) return len;
		inc += len;
		if (sjis) {
			*outc = sjis;
			if (sjis & 0xff00) {
				outc[1] = *outc;
				*outc++ = sjis>>8;
				j++;
			}
			outc++;
			j++;
		}
	}
	return j;
}

#if 1
int akxqu8len(c)
uchar c;
{
	UINT4 u1;
	int mlen;

	mlen = 1;
	u1 = (UINT4)c;
	if (u1 < 0xc2) ;
	else if (u1 < 0xe0) {
		if (u1 <= 0xdf) mlen = 2;
	}
	else if (u1 < 0xf0) {
		if (u1 <= 0xed) mlen = 3;
	}
	else if (u1 <= 0xf4) mlen = 4;
	return mlen;
}

int akxqu8nlen(p_str, len)
char *p_str;
int len;
{
	UINT4 u1,u2,u3,u4;
	int mlen;
	char *p;

	if (!(p=p_str)) return -1;
	if (len <= 0) return akxqu8len(*p);
/*	else if (len == 1) return 1; */

	mlen = 1;
	u1 = (UINT4)*p & 0xff;
/*
printf("akxqu8nlen: len=%d u1=%08x\nh,len,u1);
*/
	if (u1 < 0xc2) ;
	else if (u1 < 0xe0) {
		if (u1<=0xdf) {
			if (len >= 2) {
				u2 = (UINT4)*(++p) & 0xff;
				if (u2>=0x80 && u2<=0xbf) mlen = 2;
			}
			else mlen = -2;
		}
	}
	else if (u1 < 0xf0) {
		if (len >= 3) {
			u2 = (UINT4)*(++p) & 0xff;
			if (u1==0xe0) {
				if (u2>=0xa0 && u2<=0xbf) mlen = 3;
			}
			else if ((u1>=0xe1 && u1<=0xec) || (u1>=0xee && u1<=0xef)) {
				if (u2>=0x80 && u2<=0xbf) mlen = 3;
			}
			else if (u1==0xed) {
				if (u2>=0x80 && u2<=0x9f) mlen = 3;
			}
			if (mlen>1) {
				u3 = (UINT4)*(++p) & 0xff;
				if (u1==0xef && u2==0xbb && u3==0xbf) mlen = 103; /* BOM */
				else if (u3>=0x80 && u3<=0xbf) ;
				else mlen = 1;
			}
		}
		else mlen = -3;
	}
	else {
		if (len >= 4) {
			u2 = (UINT4)*(++p) & 0xff;
			if (u1==0xf0) {
				if (u2>=0x90 && u2<=0xbf) mlen = 4;
			}
			else if (u1<0xf4) {
				if (u2>=0x80 && u2<=0xbf) mlen = 4;
			}
			else if (u1==0xf4) {
				if (u2>=0x80 && u2<=0x8f) mlen = 4;
			}
			if (mlen>1) {
				u3 = (UINT4)*(++p) & 0xff;
				if (u3>=0x80 && u3<=0xbf) {
					u4 = (int)*(++p) & 0xff;
					if (u4>=0x80 && u4<=0xbf) ;
					else mlen = 1;
				}
				else mlen = 1;
			}
		}
		else mlen = -4;
	}
	return mlen;
}
#else
int akxqu8len(c1)
uchar c1;
{
	UINT4 utf8;
	int len;

	utf8 = c1;
	if (!(utf8 & 0x80)) len = 1;
	else if ((utf8 & 0xe0) == 0xc0) len = 2;
	else if ((utf8 & 0xf0) == 0xe0) len = 3;
	else len = 4;
	return len;
}
#endif

#ifdef UTF8_SRC
static int _Unihaslr(hp,key)
HASHB *hp;
long key;
{
	int i;
	tdtHaslCell *cell,*ce;

	if (!hp) return -1;
	i=(ulong)key%hp->ha_prereg+1;

	cell=(tdtHaslCell *)hp->ha_reg-1;
  L10:
	ce=&cell[i];
	if (ce->hc_flag && (ce->hc_hkey==key))
		hp->ha_next=ce->hc_datp;
	else if ((i=ce->hc_next)>0) goto L10;

	return i;
}

static INT4 gutf8_tbl[AKX_SJIS_MAX]={
 0,0x1000000,0x2000000,0x3000000,0x4000000,0x5000000,0x6000000,0x7000000
,0x8000000,0x9000000,0xa000000,0xb000000,0xc000000,0xd000000,0xe000000,0xf000000
,0x10000000,0x11000000,0x12000000,0x13000000,0x14000000,0x15000000,0x16000000,0x17000000
,0x18000000,0x19000000,0x1a000000,0x1b000000,0x1c000000,0x1d000000,0x1e000000,0x1f000000
,0x20000000,0x21000000,0x22000000,0x23000000,0x24000000,0x25000000,0x26000000,0x27000000
,0x28000000,0x29000000,0x2a000000,0x2b000000,0x2c000000,0x2d000000,0x2e000000,0x2f000000
,0x30000000,0x31000000,0x32000000,0x33000000,0x34000000,0x35000000,0x36000000,0x37000000
,0x38000000,0x39000000,0x3a000000,0x3b000000,0x3c000000,0x3d000000,0x3e000000,0x3f000000
,0x40000000,0x41000000,0x42000000,0x43000000,0x44000000,0x45000000,0x46000000,0x47000000
,0x48000000,0x49000000,0x4a000000,0x4b000000,0x4c000000,0x4d000000,0x4e000000,0x4f000000
,0x50000000,0x51000000,0x52000000,0x53000000,0x54000000,0x55000000,0x56000000,0x57000000
,0x58000000,0x59000000,0x5a000000,0x5b000000,0xc2a50000,0x5d000000,0x5e000000,0x5f000000
,0x60000000,0x61000000,0x62000000,0x63000000,0x64000000,0x65000000,0x66000000,0x67000000
,0x68000000,0x69000000,0x6a000000,0x6b000000,0x6c000000,0x6d000000,0x6e000000,0x6f000000
,0x70000000,0x71000000,0x72000000,0x73000000,0x74000000,0x75000000,0x76000000,0x77000000
,0x78000000,0x79000000,0x7a000000,0x7b000000,0x7c000000,0x7d000000,0xe280be00,0x7f000000
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0xefbda100,0xefbda200,0xefbda300,0xefbda400,0xefbda500,0xefbda600,0xefbda700
,0xefbda800,0xefbda900,0xefbdaa00,0xefbdab00,0xefbdac00,0xefbdad00,0xefbdae00,0xefbdaf00
,0xefbdb000,0xefbdb100,0xefbdb200,0xefbdb300,0xefbdb400,0xefbdb500,0xefbdb600,0xefbdb700
,0xefbdb800,0xefbdb900,0xefbdba00,0xefbdbb00,0xefbdbc00,0xefbdbd00,0xefbdbe00,0xefbdbf00
,0xefbe8000,0xefbe8100,0xefbe8200,0xefbe8300,0xefbe8400,0xefbe8500,0xefbe8600,0xefbe8700
,0xefbe8800,0xefbe8900,0xefbe8a00,0xefbe8b00,0xefbe8c00,0xefbe8d00,0xefbe8e00,0xefbe8f00
,0xefbe9000,0xefbe9100,0xefbe9200,0xefbe9300,0xefbe9400,0xefbe9500,0xefbe9600,0xefbe9700
,0xefbe9800,0xefbe9900,0xefbe9a00,0xefbe9b00,0xefbe9c00,0xefbe9d00,0xefbe9e00,0xefbe9f00
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe3808000,0xe3808100,0xe3808200,0xefbc8c00,0xefbc8e00,0xe383bb00,0xefbc9a00,0xefbc9b00
,0xefbc9f00,0xefbc8100,0xe3829b00,0xe3829c00,0xc2b40000,0xefbd8000,0xc2a80000,0xefbcbe00
,0xefbfa300,0xefbcbf00,0xe383bd00,0xe383be00,0xe3829d00,0xe3829e00,0xe3808300,0xe4bb9d00
,0xe3808500,0xe3808600,0xe3808700,0xe383bc00,0xe2809400,0xe2809000,0xefbc8f00,0x5c000000
,0xe3809c00,0xe2809600,0xefbd9c00,0xe280a600,0xe280a500,0xe2809800,0xe2809900,0xe2809c00
,0xe2809d00,0xefbc8800,0xefbc8900,0xe3809400,0xe3809500,0xefbcbb00,0xefbcbd00,0xefbd9b00
,0xefbd9d00,0xe3808800,0xe3808900,0xe3808a00,0xe3808b00,0xe3808c00,0xe3808d00,0xe3808e00
,0xe3808f00,0xe3809000,0xe3809100,0xefbc8b00,0xe2889200,0xc2b10000,0xc3970000,0
,0xc3b70000,0xefbc9d00,0xe289a000,0xefbc9c00,0xefbc9e00,0xe289a600,0xe289a700,0xe2889e00
,0xe288b400,0xe2998200,0xe2998000,0xc2b00000,0xe280b200,0xe280b300,0xe2848300,0xefbfa500
,0xefbc8400,0xc2a20000,0xc2a30000,0xefbc8500,0xefbc8300,0xefbc8600,0xefbc8a00,0xefbca000
,0xc2a70000,0xe2988600,0xe2988500,0xe2978b00,0xe2978f00,0xe2978e00,0xe2978700,0xe2978600
,0xe296a100,0xe296a000,0xe296b300,0xe296b200,0xe296bd00,0xe296bc00,0xe280bb00,0xe3809200
,0xe2869200,0xe2869000,0xe2869100,0xe2869300,0xe3809300,0xefbc8700,0xefbc8200,0xefbc8d00
,0x7e000000,0xe380b300,0xe380b400,0xe380b500,0xe380bb00,0xe380bc00,0xe380bd00,0xe3878000
,0xe2888800,0xe2888b00,0xe28a8600,0xe28a8700,0xe28a8200,0xe28a8300,0xe288aa00,0xe288a900
,0xe28a8400,0xe28a8500,0xe28a8a00,0xe28a8b00,0xe2888900,0xe2888500,0xe28c8500,0xe28c8600
,0xe288a700,0xe288a800,0xc2ac0000,0xe2879200,0xe2879400,0xe2888000,0xe2888300,0xe28a9500
,0xe28a9600,0xe28a9700,0xe288a500,0xe288a600,0xe2819000,0xe2819100,0xe3809800,0xe3809900
,0xe3809600,0xe3809700,0xe288a000,0xe28aa500,0xe28c9200,0xe2888200,0xe2888700,0xe289a100
,0xe2899200,0xe289aa00,0xe289ab00,0xe2889a00,0xe288bd00,0xe2889d00,0xe288b500,0xe288ab00
,0xe288ac00,0xe289a200,0xe2898300,0xe2898500,0xe2898800,0xe289b600,0xe289b700,0xe2869400
,0xe284ab00,0xe280b000,0xe299af00,0xe299ad00,0xe299aa00,0xe280a000,0xe280a100,0xc2b60000
,0xe299ae00,0xe299ab00,0xe299ac00,0xe299a900,0xe297af00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe296b700,0xe296b600,0xe2978100,0xe2978000,0xe2869700,0xe2869800,0xe2869600,0xe2869900
,0xe2878400,0xe287a800,0xe287a600,0xe287a700,0xe287a900,0xe287b400,0xe287b500,0xefbc9000
,0xefbc9100,0xefbc9200,0xefbc9300,0xefbc9400,0xefbc9500,0xefbc9600,0xefbc9700,0xefbc9800
,0xefbc9900,0xe2819200,0xe2978900,0xe3878100,0xe3878200,0xe3878300,0xe297a600,0xe280a200
,0xefbca100,0xefbca200,0xefbca300,0xefbca400,0xefbca500,0xefbca600,0xefbca700,0xefbca800
,0xefbca900,0xefbcaa00,0xefbcab00,0xefbcac00,0xefbcad00,0xefbcae00,0xefbcaf00,0xefbcb000
,0xefbcb100,0xefbcb200,0xefbcb300,0xefbcb400,0xefbcb500,0xefbcb600,0xefbcb700,0xefbcb800
,0xefbcb900,0xefbcba00,0xe2889300,0xe284b500,0xe2848f00,0xe38f8b00,0xe2849300,0
,0xe284a700,0xefbd8100,0xefbd8200,0xefbd8300,0xefbd8400,0xefbd8500,0xefbd8600,0xefbd8700
,0xefbd8800,0xefbd8900,0xefbd8a00,0xefbd8b00,0xefbd8c00,0xefbd8d00,0xefbd8e00,0xefbd8f00
,0xefbd9000,0xefbd9100,0xefbd9200,0xefbd9300,0xefbd9400,0xefbd9500,0xefbd9600,0xefbd9700
,0xefbd9800,0xefbd9900,0xefbd9a00,0xe2818e00,0xe2809300,0xe28e9b00,0xe28e9c00,0xe3818100
,0xe3818200,0xe3818300,0xe3818400,0xe3818500,0xe3818600,0xe3818700,0xe3818800,0xe3818900
,0xe3818a00,0xe3818b00,0xe3818c00,0xe3818d00,0xe3818e00,0xe3818f00,0xe3819000,0xe3819100
,0xe3819200,0xe3819300,0xe3819400,0xe3819500,0xe3819600,0xe3819700,0xe3819800,0xe3819900
,0xe3819a00,0xe3819b00,0xe3819c00,0xe3819d00,0xe3819e00,0xe3819f00,0xe381a000,0xe381a100
,0xe381a200,0xe381a300,0xe381a400,0xe381a500,0xe381a600,0xe381a700,0xe381a800,0xe381a900
,0xe381aa00,0xe381ab00,0xe381ac00,0xe381ad00,0xe381ae00,0xe381af00,0xe381b000,0xe381b100
,0xe381b200,0xe381b300,0xe381b400,0xe381b500,0xe381b600,0xe381b700,0xe381b800,0xe381b900
,0xe381ba00,0xe381bb00,0xe381bc00,0xe381bd00,0xe381be00,0xe381bf00,0xe3828000,0xe3828100
,0xe3828200,0xe3828300,0xe3828400,0xe3828500,0xe3828600,0xe3828700,0xe3828800,0xe3828900
,0xe3828a00,0xe3828b00,0xe3828c00,0xe3828d00,0xe3828e00,0xe3828f00,0xe3829000,0xe3829100
,0xe3829200,0xe3829300,0xe3829400,0xe3829500,0xe3829600,0xe3879000,0xe3879100,0xe3879200
,0xe3879300,0xe3879400,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe382a100,0xe382a200,0xe382a300,0xe382a400,0xe382a500,0xe382a600,0xe382a700,0xe382a800
,0xe382a900,0xe382aa00,0xe382ab00,0xe382ac00,0xe382ad00,0xe382ae00,0xe382af00,0xe382b000
,0xe382b100,0xe382b200,0xe382b300,0xe382b400,0xe382b500,0xe382b600,0xe382b700,0xe382b800
,0xe382b900,0xe382ba00,0xe382bb00,0xe382bc00,0xe382bd00,0xe382be00,0xe382bf00,0xe3838000
,0xe3838100,0xe3838200,0xe3838300,0xe3838400,0xe3838500,0xe3838600,0xe3838700,0xe3838800
,0xe3838900,0xe3838a00,0xe3838b00,0xe3838c00,0xe3838d00,0xe3838e00,0xe3838f00,0xe3839000
,0xe3839100,0xe3839200,0xe3839300,0xe3839400,0xe3839500,0xe3839600,0xe3839700,0xe3839800
,0xe3839900,0xe3839a00,0xe3839b00,0xe3839c00,0xe3839d00,0xe3839e00,0xe3839f00,0
,0xe383a000,0xe383a100,0xe383a200,0xe383a300,0xe383a400,0xe383a500,0xe383a600,0xe383a700
,0xe383a800,0xe383a900,0xe383aa00,0xe383ab00,0xe383ac00,0xe383ad00,0xe383ae00,0xe383af00
,0xe383b000,0xe383b100,0xe383b200,0xe383b300,0xe383b400,0xe383b500,0xe383b600,0xe387a000
,0xe387a100,0xe387a200,0xe387a300,0xe387a400,0xe387a500,0xe387a600,0xe387a700,0xce910000
,0xce920000,0xce930000,0xce940000,0xce950000,0xce960000,0xce970000,0xce980000,0xce990000
,0xce9a0000,0xce9b0000,0xce9c0000,0xce9d0000,0xce9e0000,0xce9f0000,0xcea00000,0xcea10000
,0xcea30000,0xcea40000,0xcea50000,0xcea60000,0xcea70000,0xcea80000,0xcea90000,0xe299a400
,0xe299a000,0xe299a200,0xe299a600,0xe299a100,0xe299a500,0xe299a700,0xe299a300,0xceb10000
,0xceb20000,0xceb30000,0xceb40000,0xceb50000,0xceb60000,0xceb70000,0xceb80000,0xceb90000
,0xceba0000,0xcebb0000,0xcebc0000,0xcebd0000,0xcebe0000,0xcebf0000,0xcf800000,0xcf810000
,0xcf830000,0xcf840000,0xcf850000,0xcf860000,0xcf870000,0xcf880000,0xcf890000,0xcf820000
,0xe28eac00,0xe28ead00,0xe28eae00,0xe28eaf00,0xe28eb000,0xe28eb100,0xe28eb200,0xe28eb300
,0xe28eb400,0xe28eb500,0xe2989600,0xe2989700,0xe380a000,0xe2988e00,0xe2988000,0xe2988100
,0xe2988200,0xe2988300,0xe299a800,0xe296b100,0xe387a800,0xe387a900,0xe387aa00,0xe387ab00
,0xe387ac00,0xe387ad00,0xe387ae00,0xe387af00,0xe387b000,0xe387b100,0xe387b200,0xe387b300
,0xe387b400,0xe387b500,0xe387b600,0xe387b700,0xe387b800,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xd0900000,0xd0910000,0xd0920000,0xd0930000,0xd0940000,0xd0950000,0xd0810000,0xd0960000
,0xd0970000,0xd0980000,0xd0990000,0xd09a0000,0xd09b0000,0xd09c0000,0xd09d0000,0xd09e0000
,0xd09f0000,0xd0a00000,0xd0a10000,0xd0a20000,0xd0a30000,0xd0a40000,0xd0a50000,0xd0a60000
,0xd0a70000,0xd0a80000,0xd0a90000,0xd0aa0000,0xd0ab0000,0xd0ac0000,0xd0ad0000,0xd0ae0000
,0xd0af0000,0xe28e9d00,0xe28e9e00,0xe28e9f00,0xe28ea000,0xe28ea100,0xe28ea200,0xe28ea300
,0xe28ea400,0xe28ea500,0xe28ea600,0xe28ea700,0xe28ea800,0xe28ea900,0xe28eaa00,0xe28eab00
,0xd0b00000,0xd0b10000,0xd0b20000,0xd0b30000,0xd0b40000,0xd0b50000,0xd1910000,0xd0b60000
,0xd0b70000,0xd0b80000,0xd0b90000,0xd0ba0000,0xd0bb0000,0xd0bc0000,0xd0bd0000,0
,0xd0be0000,0xd0bf0000,0xd1800000,0xd1810000,0xd1820000,0xd1830000,0xd1840000,0xd1850000
,0xd1860000,0xd1870000,0xd1880000,0xd1890000,0xd18a0000,0xd18b0000,0xd18c0000,0xd18d0000
,0xd18e0000,0xd18f0000,0xe383b700,0xe383b800,0xe383b900,0xe383ba00,0xe28b9a00,0xe28b9b00
,0xe2859300,0xe2859400,0xe2859500,0xe29c9300,0xe28c9800,0xe290a300,0xe2989800,0xe2948000
,0xe2948200,0xe2948c00,0xe2949000,0xe2949800,0xe2949400,0xe2949c00,0xe294ac00,0xe294a400
,0xe294b400,0xe294bc00,0xe2948100,0xe2948300,0xe2948f00,0xe2949300,0xe2949b00,0xe2949700
,0xe294a300,0xe294b300,0xe294ab00,0xe294bb00,0xe2958b00,0xe294a000,0xe294af00,0xe294a800
,0xe294b700,0xe294bf00,0xe2949d00,0xe294b000,0xe294a500,0xe294b800,0xe2958200,0xe29f9000
,0xe29f9100,0xe29f9200,0xe29f9300,0xe29f9400,0xe29f9500,0xe29f9600,0xe29f9700,0xe29f9800
,0xe29f9900,0xe29f9a00,0xe29f9b00,0xe29f9c00,0xe29f9d00,0xe29f9e00,0xe29f9f00,0xe29fa000
,0xe29fa100,0xe29fa200,0xe29fa300,0xe29fa400,0xe29fa500,0xe29fa600,0xe29fa700,0xe29fa800
,0xe29fa900,0xe29faa00,0xe29fab00,0xe29fac00,0xe29fad00,0,0,0
,0,0,0,0,0,0xe2979000,0xe2979100,0xe2979200
,0xe2979300,0xe280bc00,0xe2818700,0xe2818800,0xe2818900,0xc78d0000,0xc78e0000,0xc7900000
,0xe1b8be00,0xe1b8bf00,0xc7b80000,0xc7b90000,0xc7910000,0xc7920000,0xc7940000,0xc7960000
,0xc7980000,0xc79a0000,0xc79c0000,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe282ac00,0xc2a00000,0xc2a10000,0xc2a40000,0xc2a60000,0xc2a90000,0xc2aa0000,0xc2ab0000
,0xc2ad0000,0xc2ae0000,0xc2af0000,0xc2b20000,0xc2b30000,0xc2b70000,0xc2b80000,0xc2b90000
,0xc2ba0000,0xc2bb0000,0xc2bc0000,0xc2bd0000,0xc2be0000,0xc2bf0000,0xc3800000,0xc3810000
,0xc3820000,0xc3830000,0xc3840000,0xc3850000,0xc3860000,0xc3870000,0xc3880000,0xc3890000
,0xc38a0000,0xc38b0000,0xc38c0000,0xc38d0000,0xc38e0000,0xc38f0000,0xc3900000,0xc3910000
,0xc3920000,0xc3930000,0xc3940000,0xc3950000,0xc3960000,0xc3980000,0xc3990000,0xc39a0000
,0xc39b0000,0xc39c0000,0xc39d0000,0xc39e0000,0xc39f0000,0xc3a00000,0xc3a10000,0xc3a20000
,0xc3a30000,0xc3a40000,0xc3a50000,0xc3a60000,0xc3a70000,0xc3a80000,0xc3a90000,0
,0xc3aa0000,0xc3ab0000,0xc3ac0000,0xc3ad0000,0xc3ae0000,0xc3af0000,0xc3b00000,0xc3b10000
,0xc3b20000,0xc3b30000,0xc3b40000,0xc3b50000,0xc3b60000,0xc3b80000,0xc3b90000,0xc3ba0000
,0xc3bb0000,0xc3bc0000,0xc3bd0000,0xc3be0000,0xc3bf0000,0xc4800000,0xc4aa0000,0xc5aa0000
,0xc4920000,0xc58c0000,0xc4810000,0xc4ab0000,0xc5ab0000,0xc4930000,0xc58d0000,0xc4840000
,0xcb980000,0xc5810000,0xc4bd0000,0xc59a0000,0xc5a00000,0xc59e0000,0xc5a40000,0xc5b90000
,0xc5bd0000,0xc5bb0000,0xc4850000,0xcb9b0000,0xc5820000,0xc4be0000,0xc59b0000,0xcb870000
,0xc5a10000,0xc59f0000,0xc5a50000,0xc5ba0000,0xcb9d0000,0xc5be0000,0xc5bc0000,0xc5940000
,0xc4820000,0xc4b90000,0xc4860000,0xc48c0000,0xc4980000,0xc49a0000,0xc48e0000,0xc5830000
,0xc5870000,0xc5900000,0xc5980000,0xc5ae0000,0xc5b00000,0xc5a20000,0xc5950000,0xc4830000
,0xc4ba0000,0xc4870000,0xc48d0000,0xc4990000,0xc49b0000,0xc48f0000,0xc4910000,0xc5840000
,0xc5880000,0xc5910000,0xc5990000,0xc5af0000,0xc5b10000,0xc5a30000,0xcb990000,0xc4880000
,0xc49c0000,0xc4a40000,0xc4b40000,0xc59c0000,0xc5ac0000,0xc4890000,0xc49d0000,0xc4a50000
,0xc4b50000,0xc59d0000,0xc5ad0000,0xc9b10000,0xca8b0000,0xc9be0000,0xca830000,0xca920000
,0xc9ac0000,0xc9ae0000,0xc9b90000,0xca880000,0xc9960000,0xc9b30000,0xc9bd0000,0xca820000
,0xca900000,0xc9bb0000,0xc9ad0000,0xc99f0000,0xc9b20000,0xca9d0000,0xca8e0000,0xc9a10000
,0xc58b0000,0xc9b00000,0xca810000,0xc4a70000,0xca950000,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xca940000,0xc9a60000,0xca980000,0xc7820000,0xc9930000,0xc9970000,0xca840000,0xc9a00000
,0xc6930000,0xc5930000,0xc5920000,0xc9a80000,0xca890000,0xc9980000,0xc9b50000,0xc9990000
,0xc99c0000,0xc99e0000,0xc9900000,0xc9af0000,0xca8a0000,0xc9a40000,0xca8c0000,0xc9940000
,0xc9910000,0xc9920000,0xca8d0000,0xc9a50000,0xcaa20000,0xcaa10000,0xc9950000,0xca910000
,0xc9ba0000,0xc9a70000,0xc99a0000,0xc8b40000,0xc7bd0000,0xe1bdb000,0xe1bdb100,0xc8b50000
,0xc8b60000,0xc8b70000,0xc8b80000,0xc8b90000,0xc8ba0000,0xc8bb0000,0xc8bc0000,0xe1bdb200
,0xe1bdb300,0xcda10000,0xcb880000,0xcb8c0000,0xcb900000,0xcb910000,0xcc860000,0xe280bf00
,0xcc8b0000,0xcc810000,0xcc840000,0xcc800000,0xcc8f0000,0xcc8c0000,0xcc820000,0
,0xcba50000,0xcba60000,0xcba70000,0xcba80000,0xcba90000,0xcbaf0000,0xcbb00000,0xcca50000
,0xccac0000,0xccb90000,0xcc9c0000,0xcc9f0000,0xcca00000,0xcc880000,0xccbd0000,0xcca90000
,0xccaf0000,0xcb9e0000,0xcca40000,0xccb00000,0xccbc0000,0xccb40000,0xcc9d0000,0xcc9e0000
,0xcc980000,0xcc990000,0xccaa0000,0xccba0000,0xccbb0000,0xcc830000,0xcc9a0000,0xe29db600
,0xe29db700,0xe29db800,0xe29db900,0xe29dba00,0xe29dbb00,0xe29dbc00,0xe29dbd00,0xe29dbe00
,0xe29dbf00,0xe29fae00,0xe29faf00,0xe29fb000,0xe29fb100,0xe29fb200,0xe29fb300,0xe29fb400
,0xe29fb500,0xe29fb600,0xe29fb700,0xe285b000,0xe285b100,0xe285b200,0xe285b300,0xe285b400
,0xe285b500,0xe285b600,0xe285b700,0xe285b800,0xe285b900,0xe285ba00,0xe285bb00,0xe2939000
,0xe2939100,0xe2939200,0xe2939300,0xe2939400,0xe2939500,0xe2939600,0xe2939700,0xe2939800
,0xe2939900,0xe2939a00,0xe2939b00,0xe2939c00,0xe2939d00,0xe2939e00,0xe2939f00,0xe293a000
,0xe293a100,0xe293a200,0xe293a300,0xe293a400,0xe293a500,0xe293a600,0xe293a700,0xe293a800
,0xe293a900,0xe38b9000,0xe38b9100,0xe38b9200,0xe38b9300,0xe38b9400,0xe38b9500,0xe38b9600
,0xe38b9700,0xe38b9800,0xe38b9900,0xe38b9a00,0xe38b9b00,0xe38b9c00,0xe38b9d00,0xe38b9e00
,0xe38b9f00,0xe38ba000,0xe38ba100,0xe38ba200,0xe38ba300,0xe38bba00,0xe38ba900,0xe38ba500
,0xe38bad00,0xe38bac00,0,0,0,0,0,0
,0,0,0,0xe2819300,0xe2818200,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe291a000,0xe291a100,0xe291a200,0xe291a300,0xe291a400,0xe291a500,0xe291a600,0xe291a700
,0xe291a800,0xe291a900,0xe291aa00,0xe291ab00,0xe291ac00,0xe291ad00,0xe291ae00,0xe291af00
,0xe291b000,0xe291b100,0xe291b200,0xe291b300,0xe285a000,0xe285a100,0xe285a200,0xe285a300
,0xe285a400,0xe285a500,0xe285a600,0xe285a700,0xe285a800,0xe285a900,0xe285aa00,0xe38d8900
,0xe38c9400,0xe38ca200,0xe38d8d00,0xe38c9800,0xe38ca700,0xe38c8300,0xe38cb600,0xe38d9100
,0xe38d9700,0xe38c8d00,0xe38ca600,0xe38ca300,0xe38cab00,0xe38d8a00,0xe38cbb00,0xe38e9c00
,0xe38e9d00,0xe38e9e00,0xe38e8e00,0xe38e8f00,0xe38f8400,0xe38ea100,0xe285ab00,0
,0,0,0,0,0,0,0xe38dbb00,0
,0xe3809d00,0xe3809f00,0xe2849600,0xe38f8d00,0xe284a100,0xe38aa400,0xe38aa500,0xe38aa600
,0xe38aa700,0xe38aa800,0xe388b100,0xe388b200,0xe388b900,0xe38dbe00,0xe38dbd00,0xe38dbc00
,0,0,0,0xe288ae00,0,0,0,0
,0xe2889f00,0xe28abf00,0,0,0,0xe29d9600,0xe2989e00,0xe4bfb100
,0xeaaaa200,0xe3908200,0xe4b8a800,0xe4b8af00,0xe4b8b000,0xe4ba8d00,0xe4bba100,0xe4bbbd00
,0xe4bbbf00,0xe4bc8300,0xe4bc8b00,0xe4bda000,0xe4bd8800,0xe4bd8900,0xe4bd9600,0xe4bd9f00
,0xe4bdaa00,0xe4bdac00,0xe4bdbe00,0xe4be8a00,0xe4be9400,0xe4be9700,0xefa8b000,0xe4bf8900
,0xe4bfa000,0xe5808100,0xe5808200,0xe5808e00,0xe5809800,0xe580a700,0xe580ae00,0xe5818000
,0xe580bb00,0xe5818100,0xe5829400,0xe5838c00,0xe583b200,0xe5839000,0xe583a600,0xefa8b100
,0xe5848600,0xe5848300,0xe5848b00,0xe5849e00,0xe584b500,0xe5858a00,0xefa8b200,0xe5859500
,0xe5859700,0xe392b500,0xe5869d00,0xe5878300,0xe5878a00,0xe5879e00,0xe587a200,0xe587ae00
,0xe5888100,0xe3939b00,0xe5889300,0xe5889500,0xe5898900,0xe5899700,0xe589a100,0xe58a9300
,0xe58b8800,0xefa8b300,0xe58b8c00,0xe58b9000,0xe58b9600,0xe58b9b00,0xefa8b400,0xe58bb000
,0xe58bbb00,0xe58c8000,0xe58c8700,0xe58c9c00,0xefa8b500,0xe58da100,0xe58da300,0xe58dbd00
,0xe58e9300,0xe58e9d00,0xe58eb200,0xe5909200,0xe590a700,0xe5918d00,0xe5929c00,0xe591ab00
,0xe591b400,0xe591bf00,0xe5928800,0xe5929600,0xe592a100,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe592a900,0xe5938600,0xe593bf00,0xe5948e00,0xe594ab00,0xe594b500,0xe5959000,0xe5959e00
,0xe5968100,0xe5968600,0xe5968e00,0xefa8b600,0xe596ad00,0xe5978e00,0xefa8b700,0xe5988800
,0xe5988e00,0xe598bb00,0xe5998900,0xe599b600,0xe599a600,0xefa8b800,0xe599af00,0xe599b100
,0xe599b200,0xe59a9900,0xe59a9e00,0xe59aa900,0xe59aac00,0xe59ab300,0xe59b8900,0xe59b8a00
,0xe59c8a00,0xeaaabe00,0xe59ca100,0xe59caf00,0xe59cb300,0xe59cb400,0xe59db000,0xe59db700
,0xe59dbc00,0xe59e9c00,0xefa88f00,0xeaab8400,0xe59eb800,0xe59f8700,0xe59f8800,0xe59f8f00
,0xe59fa400,0xe59fad00,0xe59fb500,0xe59fb600,0xe59fbf00,0xe5a08900,0xefa89000,0xe5a1a100
,0xe5a1a400,0xefa8b900,0xe5a1bc00,0xe5a28900,0xe5a29e00,0xefa8ba00,0xe5a2a900,0
,0xeaab8900,0xe5a39200,0xe5a38e00,0xe5a39400,0xe5a39a00,0xe5a3a000,0xe5a3a900,0xe5a48c00
,0xe8998100,0xe5a59d00,0xe5a5ad00,0xe5a68b00,0xe5a69200,0xe5a6a400,0xe5a78300,0xe5a79200
,0xe5a79d00,0xe5a89300,0xe5a8a300,0xe5a9a700,0xe5a9ad00,0xe5a9b700,0xe5a9be00,0xe5aa8400
,0xe5aa9e00,0xe5aaa700,0xe5ab8400,0xeaab9000,0xe5ac9900,0xe5aca500,0xe5899d00,0xe4ba9c00
,0xe5949600,0xe5a88300,0xe998bf00,0xe5938000,0xe6849b00,0xe68ca800,0xe5a7b600,0xe980a200
,0xe891b500,0xe88c9c00,0xe7a99000,0xe682aa00,0xe68fa100,0xe6b8a500,0xe697ad00,0xe891a600
,0xe88aa600,0xe9afb500,0xe6a29300,0xe59ca700,0xe696a100,0xe689b100,0xe5ae9b00,0xe5a79000
,0xe899bb00,0xe9a3b400,0xe7b5a200,0xe7b6be00,0xe9ae8e00,0xe6889600,0xe7b29f00,0xe8a2b700
,0xe5ae8900,0xe5bab500,0xe68c8900,0xe69a9700,0xe6a18800,0xe9978700,0xe99e8d00,0xe69d8f00
,0xe4bba500,0xe4bc8a00,0xe4bd8d00,0xe4be9d00,0xe5818900,0xe59bb200,0xe5a4b700,0xe5a79400
,0xe5a88100,0xe5b08900,0xe6839f00,0xe6848f00,0xe685b000,0xe6989300,0xe6a48500,0xe782ba00
,0xe7958f00,0xe795b000,0xe7a7bb00,0xe7b6ad00,0xe7b7af00,0xe8838300,0xe8908e00,0xe8a1a300
,0xe8ac8200,0xe9819500,0xe981ba00,0xe58cbb00,0xe4ba9500,0xe4baa500,0xe59f9f00,0xe882b200
,0xe9838100,0xe7a3af00,0xe4b88000,0xe5a3b100,0xe6baa200,0xe980b800,0xe7a8b200,0xe88ca800
,0xe88a8b00,0xe9b0af00,0xe5858100,0xe58db000,0xe592bd00,0xe593a100,0xe59ba000,0xe5a7bb00
,0xe5bc9500,0xe9a3b200,0xe6b7ab00,0xe883a400,0xe894ad00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe999a200,0xe999b000,0xe99aa000,0xe99fbb00,0xe5908b00,0xe58fb300,0xe5ae8700,0xe7838f00
,0xe7bebd00,0xe8bf8200,0xe99ba800,0xe58daf00,0xe9b59c00,0xe7aaba00,0xe4b89100,0xe7a29300
,0xe887bc00,0xe6b8a600,0xe5989800,0xe5948400,0xe6ac9d00,0xe8949a00,0xe9b0bb00,0xe5a7a500
,0xe58ea900,0xe6b5a600,0xe7939c00,0xe9968f00,0xe5998200,0xe4ba9100,0xe9818b00,0xe99bb200
,0xe88d8f00,0xe9a48c00,0xe58fa100,0xe596b600,0xe5acb000,0xe5bdb100,0xe698a000,0xe69bb300
,0xe6a08400,0xe6b0b800,0xe6b3b300,0xe6b4a900,0xe7919b00,0xe79b8800,0xe7a98e00,0xe9a0b400
,0xe88bb100,0xe8a19b00,0xe8a9a000,0xe98bad00,0xe6b6b200,0xe796ab00,0xe79b8a00,0xe9a78500
,0xe682a600,0xe8ac8100,0xe8b68a00,0xe996b200,0xe6a68e00,0xe58ead00,0xe5868600,0
,0xe59c9200,0xe5a0b000,0xe5a58400,0xe5aeb400,0xe5bbb600,0xe680a800,0xe68ea900,0xe68fb400
,0xe6b2bf00,0xe6bc9400,0xe7828e00,0xe7849400,0xe7859900,0xe7879500,0xe78cbf00,0xe7b88100
,0xe889b600,0xe88b9100,0xe8969700,0xe981a000,0xe9899b00,0xe9b49b00,0xe5a1a900,0xe696bc00
,0xe6b19a00,0xe794a500,0xe587b900,0xe5a4ae00,0xe5a5a500,0xe5be8000,0xe5bf9c00,0xe68abc00
,0xe697ba00,0xe6a8aa00,0xe6aca700,0xe6aeb400,0xe78e8b00,0xe7bf8100,0xe8a59600,0xe9b4ac00
,0xe9b48e00,0xe9bb8400,0xe5b2a100,0xe6b29600,0xe88dbb00,0xe5848400,0xe5b18b00,0xe686b600
,0xe8878600,0xe6a1b600,0xe789a100,0xe4b99900,0xe4bfba00,0xe58db800,0xe681a900,0xe6b8a900
,0xe7a98f00,0xe99fb300,0xe4b88b00,0xe58c9600,0xe4bbae00,0xe4bd9500,0xe4bcbd00,0xe4bea100
,0xe4bdb300,0xe58aa000,0xe58faf00,0xe5988900,0xe5a48f00,0xe5ab8100,0xe5aeb600,0xe5afa100
,0xe7a79100,0xe69a8700,0xe69e9c00,0xe69eb600,0xe6ad8c00,0xe6b2b300,0xe781ab00,0xe78f8200
,0xe7a68d00,0xe7a6be00,0xe7a8bc00,0xe7ae8700,0xe88ab100,0xe88b9b00,0xe88c8400,0xe88db700
,0xe88faf00,0xe88f9300,0xe89da600,0xe8aab200,0xe598a900,0xe8b2a800,0xe8bfa600,0xe9818e00
,0xe99c9e00,0xe89a8a00,0xe4bf8400,0xe5b3a800,0xe6889100,0xe7899900,0xe794bb00,0xe887a500
,0xe88abd00,0xe89bbe00,0xe8b38000,0xe99b8500,0xe9a49300,0xe9a79500,0xe4bb8b00,0xe4bc9a00
,0xe8a7a300,0xe59b9e00,0xe5a18a00,0xe5a38a00,0xe5bbbb00,0xe5bfab00,0xe680aa00,0xe6829400
,0xe681a200,0xe6879000,0xe6889200,0xe68b9000,0xe694b900,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe9ad8100,0xe699a600,0xe6a2b000,0xe6b5b700,0xe781b000,0xe7958c00,0xe79a8600,0xe7b5b500
,0xe88aa500,0xe89fb900,0xe9968b00,0xe99a8e00,0xe8b29d00,0xe587b100,0xe58abe00,0xe5a49600
,0xe592b300,0xe5aeb300,0xe5b49600,0xe685a800,0xe6a68200,0xe6b6af00,0xe7a28d00,0xe8938b00
,0xe8a19700,0xe8a9b200,0xe98ea700,0xe9aab800,0xe6b5ac00,0xe9a6a800,0xe89b9900,0xe59ea300
,0xe69fbf00,0xe89b8e00,0xe9888e00,0xe58a8300,0xe59a8700,0xe5908400,0xe5bb9300,0xe68ba100
,0xe692b900,0xe6a0bc00,0xe6a0b800,0xe6aebb00,0xe78db200,0xe7a2ba00,0xe7a9ab00,0xe8a69a00
,0xe8a79200,0xe8b5ab00,0xe8bc8300,0xe983ad00,0xe996a300,0xe99a9400,0xe99da900,0xe5ada600
,0xe5b2b300,0xe6a5bd00,0xe9a18d00,0xe9a18e00,0xe68e9b00,0xe7aca000,0xe6a8ab00,0
,0xe6a9bf00,0xe6a2b600,0xe9b08d00,0xe6bd9f00,0xe589b200,0xe5969d00,0xe681b000,0xe68bac00
,0xe6b4bb00,0xe6b88700,0xe6bb9100,0xe8919b00,0xe8a49000,0xe8bd8400,0xe4b89400,0xe9b0b900
,0xe58fb600,0xe6a49b00,0xe6a8ba00,0xe99e8400,0xe6a0aa00,0xe5859c00,0xe7ab8300,0xe892b200
,0xe9879c00,0xe98e8c00,0xe5999b00,0xe9b4a800,0xe6a0a200,0xe88c8500,0xe890b100,0xe7b2a500
,0xe5888800,0xe88b8500,0xe793a600,0xe4b9be00,0xe4be8300,0xe586a000,0xe5af9200,0xe5888a00
,0xe58b9800,0xe58ba700,0xe5b7bb00,0xe5969a00,0xe5a0aa00,0xe5a7a600,0xe5ae8c00,0xe5ae9800
,0xe5af9b00,0xe5b9b200,0xe5b9b900,0xe682a300,0xe6849f00,0xe685a300,0xe686be00,0xe68f9b00
,0xe695a200,0xe69f9100,0xe6a19300,0xe6a3ba00,0xe6acbe00,0xe6ad9300,0xe6b19700,0xe6bca200
,0xe6be9700,0xe6bd8500,0xe792b000,0xe7949800,0xe79ba300,0xe79c8b00,0xe7abbf00,0xe7aea100
,0xe7b0a100,0xe7b7a900,0xe7bcb600,0xe7bfb000,0xe8829d00,0xe889a600,0xe88e9e00,0xe8a6b300
,0xe8ab8c00,0xe8b2ab00,0xe9828400,0xe9919100,0xe9969300,0xe9969100,0xe996a200,0xe999a500
,0xe99f9300,0xe9a4a800,0xe8889800,0xe4b8b800,0xe590ab00,0xe5b2b800,0xe5b78c00,0xe78ea900
,0xe7998c00,0xe79cbc00,0xe5b2a900,0xe7bfab00,0xe8b48b00,0xe99b8100,0xe9a09100,0xe9a19400
,0xe9a19800,0xe4bc8100,0xe4bc8e00,0xe58db100,0xe5969c00,0xe599a800,0xe59fba00,0xe5a58700
,0xe5ac8900,0xe5af8400,0xe5b29000,0xe5b88c00,0xe5b9be00,0xe5bf8c00,0xe68fae00,0xe69cba00
,0xe6979700,0xe697a200,0xe69c9f00,0xe6a38b00,0xe6a38400,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe6a99f00,0xe5b8b000,0xe6af8500,0xe6b09700,0xe6b1bd00,0xe795bf00,0xe7a58800,0xe5ada300
,0xe7a88000,0xe7b48000,0xe5bebd00,0xe8a68f00,0xe8a89800,0xe8b2b400,0xe8b5b700,0xe8bb8c00
,0xe8bc9d00,0xe9a3a200,0xe9a88e00,0xe9acbc00,0xe4ba8000,0xe581bd00,0xe5848000,0xe5a69300
,0xe5ae9c00,0xe688af00,0xe68a8000,0xe693ac00,0xe6acba00,0xe78aa000,0xe7969100,0xe7a58700
,0xe7bea900,0xe89fbb00,0xe8aabc00,0xe8adb000,0xe68eac00,0xe88f8a00,0xe99ea000,0xe5908900
,0xe5908300,0xe596ab00,0xe6a19400,0xe6a99800,0xe8a9b000,0xe7a0a700,0xe69db500,0xe9bb8d00
,0xe58db400,0xe5aea200,0xe8849a00,0xe8999000,0xe9808600,0xe4b89800,0xe4b98500,0xe4bb8700
,0xe4bc9100,0xe58f8a00,0xe590b800,0xe5aeae00,0xe5bc9300,0xe680a500,0xe6959100,0
,0xe69cbd00,0xe6b18200,0xe6b1b200,0xe6b3a300,0xe781b800,0xe7908300,0xe7a9b600,0xe7aaae00
,0xe7ac8800,0xe7b49a00,0xe7b3be00,0xe7b5a600,0xe697a700,0xe7899b00,0xe58ebb00,0xe5b18500
,0xe5b7a800,0xe68b9200,0xe68ba000,0xe68c9900,0xe6b8a000,0xe8999a00,0xe8a8b100,0xe8b79d00
,0xe98bb800,0xe6bc8100,0xe7a6a600,0xe9ad9a00,0xe4baa800,0xe4baab00,0xe4baac00,0xe4be9b00
,0xe4bea000,0xe5839100,0xe5858700,0xe7abb600,0xe585b100,0xe587b600,0xe58d9400,0xe58ca100
,0xe58dbf00,0xe58fab00,0xe596ac00,0xe5a28300,0xe5b3a100,0xe5bcb700,0xe5bd8a00,0xe680af00
,0xe6819000,0xe681ad00,0xe68c9f00,0xe6959900,0xe6a98b00,0xe6b38100,0xe78b8200,0xe78bad00
,0xe79faf00,0xe883b800,0xe8848500,0xe8888800,0xe8958e00,0xe983b700,0xe98fa100,0xe99fbf00
,0xe9a59700,0xe9a99a00,0xe4bbb000,0xe5879d00,0xe5b0ad00,0xe69a8100,0xe6a5ad00,0xe5b18000
,0xe69bb200,0xe6a5b500,0xe78e8900,0xe6a19000,0xe7b28100,0xe5838500,0xe58ba400,0xe59d8700
,0xe5b7be00,0xe98ca600,0xe696a400,0xe6aca300,0xe6acbd00,0xe790b400,0xe7a68100,0xe7a6bd00
,0xe7ad8b00,0xe7b78a00,0xe88ab900,0xe88f8c00,0xe8a1bf00,0xe8a59f00,0xe8acb900,0xe8bf9100
,0xe9879100,0xe5909f00,0xe98a8000,0xe4b99d00,0xe580b600,0xe58fa500,0xe58cba00,0xe78b9700
,0xe78e9600,0xe79fa900,0xe88ba600,0xe8baaf00,0xe9a78600,0xe9a78800,0xe9a79200,0xe585b700
,0xe6849a00,0xe8999e00,0xe596b000,0xe7a9ba00,0xe581b600,0xe5af9300,0xe9818700,0xe99a8500
,0xe4b8b200,0xe6ab9b00,0xe987a700,0xe5b19100,0xe5b18800,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe68e9800,0xe7aa9f00,0xe6b29300,0xe99db400,0xe8bda100,0xe7aaaa00,0xe7868a00,0xe99a8800
,0xe7b28200,0xe6a09700,0xe7b9b000,0xe6a19100,0xe98dac00,0xe58bb200,0xe5909b00,0xe896ab00
,0xe8a89300,0xe7bea400,0xe8bb8d00,0xe983a100,0xe58da600,0xe8a28800,0xe7a58100,0xe4bf8200
,0xe582be00,0xe5889100,0xe5858400,0xe5959300,0xe59cad00,0xe78faa00,0xe59e8b00,0xe5a59100
,0xe5bda200,0xe5be8400,0xe681b500,0xe685b600,0xe685a700,0xe686a900,0xe68eb200,0xe690ba00
,0xe695ac00,0xe699af00,0xe6a18200,0xe6b89300,0xe795a600,0xe7a8bd00,0xe7b3bb00,0xe7b58c00
,0xe7b69900,0xe7b98b00,0xe7bdab00,0xe88c8e00,0xe88d8a00,0xe89b8d00,0xe8a88800,0xe8a9a300
,0xe8ada600,0xe8bbbd00,0xe9a09a00,0xe9b68f00,0xe88ab800,0xe8bf8e00,0xe9afa800,0
,0xe58a8700,0xe6889f00,0xe6928300,0xe6bf8000,0xe99a9900,0xe6a18100,0xe5829100,0xe6aca000
,0xe6b1ba00,0xe6bd9400,0xe7a9b400,0xe7b59000,0xe8a18000,0xe8a8a300,0xe69c8800,0xe4bbb600
,0xe580b900,0xe580a600,0xe581a500,0xe585bc00,0xe588b800,0xe589a300,0xe596a700,0xe59c8f00
,0xe5a08500,0xe5ab8c00,0xe5bbba00,0xe686b200,0xe687b800,0xe68bb300,0xe68db200,0xe6a49c00
,0xe6a8a900,0xe789bd00,0xe78aac00,0xe78cae00,0xe7a09400,0xe7a1af00,0xe7b5b900,0xe79c8c00
,0xe882a900,0xe8a68b00,0xe8ac9900,0xe8b3a200,0xe8bb9200,0xe981a300,0xe98db500,0xe999ba00
,0xe9a19500,0xe9a89300,0xe9b9b800,0xe5858300,0xe58e9f00,0xe58eb300,0xe5b9bb00,0xe5bca600
,0xe6b89b00,0xe6ba9000,0xe78e8400,0xe78fbe00,0xe7b58300,0xe888b700,0xe8a88000,0xe8abba00
,0xe9999000,0xe4b98e00,0xe5808b00,0xe58fa400,0xe591bc00,0xe59bba00,0xe5a79100,0xe5ada400
,0xe5b7b100,0xe5baab00,0xe5bca700,0xe688b800,0xe6958500,0xe69eaf00,0xe6b99600,0xe78b9000
,0xe7b38a00,0xe8a2b400,0xe882a100,0xe883a100,0xe88fb000,0xe8998e00,0xe8aa8700,0xe8b7a800
,0xe988b700,0xe99b8700,0xe9a1a700,0xe9bc9300,0xe4ba9400,0xe4ba9200,0xe4bc8d00,0xe58d8800
,0xe5918900,0xe590be00,0xe5a8af00,0xe5be8c00,0xe5bea100,0xe6829f00,0xe6a2a700,0xe6aa8e00
,0xe7919a00,0xe7a28100,0xe8aa9e00,0xe8aaa400,0xe8adb700,0xe9869000,0xe4b99e00,0xe9af8900
,0xe4baa400,0xe4bdbc00,0xe4beaf00,0xe5809900,0xe5809600,0xe5858900,0xe585ac00,0xe58a9f00
,0xe58ab900,0xe58bbe00,0xe58e9a00,0xe58fa300,0xe5909100,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe5908e00,0xe5968900,0xe59d9100,0xe59ea200,0xe5a5bd00,0xe5ad9400,0xe5ad9d00,0xe5ae8f00
,0xe5b7a500,0xe5b7a700,0xe5b7b700,0xe5b9b800,0xe5ba8300,0xe5ba9a00,0xe5bab700,0xe5bc9800
,0xe6819200,0xe6858c00,0xe68a9700,0xe68b9800,0xe68ea700,0xe694bb00,0xe6988200,0xe6998300
,0xe69bb400,0xe69dad00,0xe6a0a100,0xe6a29700,0xe6a78b00,0xe6b19f00,0xe6b4aa00,0xe6b5a900
,0xe6b8af00,0xe6ba9d00,0xe794b200,0xe79a8700,0xe7a1ac00,0xe7a8bf00,0xe7b3a000,0xe7b48500
,0xe7b49800,0xe7b59e00,0xe7b6b100,0xe8809500,0xe8808300,0xe882af00,0xe882b100,0xe8859400
,0xe8868f00,0xe888aa00,0xe88d9200,0xe8a18c00,0xe8a1a100,0xe8ac9b00,0xe8b2a200,0xe8b3bc00
,0xe9838a00,0xe985b500,0xe989b100,0xe7a0bf00,0xe98bbc00,0xe996a400,0xe9998d00,0
,0xe9a08500,0xe9a69900,0xe9ab9800,0xe9b4bb00,0xe5899b00,0xe58aab00,0xe58fb700,0xe5908800
,0xe5a39500,0xe68bb700,0xe6bfa000,0xe8b1aa00,0xe8bd9f00,0xe9bab900,0xe5858b00,0xe588bb00
,0xe5918a00,0xe59bbd00,0xe7a98000,0xe985b700,0xe9b5a000,0xe9bb9200,0xe78d8400,0xe6bc8900
,0xe885b000,0xe7949100,0xe5bfbd00,0xe6839a00,0xe9aaa800,0xe78b9b00,0xe8bebc00,0xe6ada400
,0xe9a08300,0xe4bb8a00,0xe59bb000,0xe59da400,0xe5a2be00,0xe5a99a00,0xe681a800,0xe6878700
,0xe6988f00,0xe6988600,0xe6a0b900,0xe6a2b100,0xe6b7b700,0xe7979500,0xe7b4ba00,0xe889ae00
,0xe9ad8200,0xe4ba9b00,0xe4bd9000,0xe58f8900,0xe5948600,0xe5b5af00,0xe5b7a600,0xe5b7ae00
,0xe69fbb00,0xe6b29900,0xe791b300,0xe7a08200,0xe8a99000,0xe98e9600,0xe8a39f00,0xe59d9000
,0xe5baa700,0xe68cab00,0xe582b500,0xe582ac00,0xe5868d00,0xe69c8000,0xe5938900,0xe5a19e00
,0xe5a6bb00,0xe5aeb000,0xe5bda900,0xe6898d00,0xe68ea100,0xe6a0bd00,0xe6adb300,0xe6b88800
,0xe781bd00,0xe9878700,0xe78a8000,0xe7a09500,0xe7a0a600,0xe7a5ad00,0xe6968e00,0xe7b4b000
,0xe88f9c00,0xe8a38100,0xe8bc8900,0xe99a9b00,0xe589a400,0xe59ca800,0xe69d9000,0xe7bdaa00
,0xe8b2a100,0xe586b400,0xe59d8200,0xe998aa00,0xe5a0ba00,0xe6a68a00,0xe882b400,0xe592b200
,0xe5b48e00,0xe59fbc00,0xe7a29500,0xe9b7ba00,0xe4bd9c00,0xe5898a00,0xe5928b00,0xe690be00
,0xe698a800,0xe69c9400,0xe69fb500,0xe7aa8400,0xe7ad9600,0xe7b4a200,0xe98caf00,0xe6a19c00
,0xe9aead00,0xe7acb900,0xe58c9900,0xe5868a00,0xe588b700,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe5af9f00,0xe68bb600,0xe692ae00,0xe693a600,0xe69cad00,0xe6aeba00,0xe896a900,0xe99b9100
,0xe79a9000,0xe9af9600,0xe68d8c00,0xe98c8600,0xe9aeab00,0xe79abf00,0xe6999200,0xe4b88900
,0xe5829800,0xe58f8200,0xe5b1b100,0xe683a800,0xe6929200,0xe695a300,0xe6a19f00,0xe787a600
,0xe78f8a00,0xe794a300,0xe7ae9700,0xe7ba8200,0xe89a9500,0xe8ae8300,0xe8b39b00,0xe985b800
,0xe9a49000,0xe696ac00,0xe69aab00,0xe6ae8b00,0xe4bb9500,0xe4bb9400,0xe4bcba00,0xe4bdbf00
,0xe588ba00,0xe58fb800,0xe58fb200,0xe597a300,0xe59b9b00,0xe5a3ab00,0xe5a78b00,0xe5a78900
,0xe5a7bf00,0xe5ad9000,0xe5b18d00,0xe5b88200,0xe5b8ab00,0xe5bf9700,0xe6809d00,0xe68c8700
,0xe694af00,0xe5ad9c00,0xe696af00,0xe696bd00,0xe697a800,0xe69e9d00,0xe6ada200,0
,0xe6adbb00,0xe6b08f00,0xe78d8500,0xe7a58900,0xe7a78100,0xe7b3b800,0xe7b49900,0xe7b4ab00
,0xe882a200,0xe8848200,0xe887b300,0xe8a69600,0xe8a99e00,0xe8a9a900,0xe8a9a600,0xe8aa8c00
,0xe8abae00,0xe8b38700,0xe8b39c00,0xe99b8c00,0xe9a3bc00,0xe6adaf00,0xe4ba8b00,0xe4bcbc00
,0xe4be8d00,0xe5859000,0xe5ad9700,0xe5afba00,0xe6858800,0xe68c8100,0xe6998200,0xe6aca100
,0xe6bb8b00,0xe6b2bb00,0xe788be00,0xe792bd00,0xe7979400,0xe7a38100,0xe7a4ba00,0xe8808c00
,0xe880b300,0xe887aa00,0xe8929400,0xe8be9e00,0xe6b19000,0xe9b9bf00,0xe5bc8f00,0xe8ad9800
,0xe9b4ab00,0xe7abba00,0xe8bbb800,0xe5ae8d00,0xe99bab00,0xe4b88300,0xe58fb100,0xe59fb700
,0xe5a4b100,0xe5ab8900,0xe5aea400,0xe6828900,0xe6b9bf00,0xe6bc8600,0xe796be00,0xe8b3aa00
,0xe5ae9f00,0xe8948000,0xe7afa000,0xe581b200,0xe69fb400,0xe88a9d00,0xe5b1a100,0xe8958a00
,0xe7b89e00,0xe8888e00,0xe5869900,0xe5b08400,0xe68da800,0xe8b5a600,0xe6969c00,0xe785ae00
,0xe7a4be00,0xe7b49700,0xe8808500,0xe8ac9d00,0xe8bb8a00,0xe981ae00,0xe89b8700,0xe982aa00
,0xe5809f00,0xe58bba00,0xe5b0ba00,0xe69d9300,0xe781bc00,0xe788b500,0xe9858c00,0xe9878800
,0xe98cab00,0xe88ba500,0xe5af8200,0xe5bcb100,0xe683b900,0xe4b8bb00,0xe58f9600,0xe5ae8800
,0xe6898b00,0xe69cb100,0xe6ae8a00,0xe78ba900,0xe78fa000,0xe7a8ae00,0xe885ab00,0xe8b6a300
,0xe9859200,0xe9a69600,0xe5849200,0xe58f9700,0xe591aa00,0xe5afbf00,0xe68e8800,0xe6a8b900
,0xe7b6ac00,0xe99c8000,0xe59b9a00,0xe58f8e00,0xe591a800,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe5ae9700,0xe5b0b100,0xe5b79e00,0xe4bfae00,0xe6848100,0xe68bbe00,0xe6b4b200,0xe7a78000
,0xe7a78b00,0xe7b58200,0xe7b98d00,0xe7bf9200,0xe887ad00,0xe8889f00,0xe8929000,0xe8a18600
,0xe8a5b200,0xe8ae9000,0xe8b9b400,0xe8bcaf00,0xe980b100,0xe9858b00,0xe985ac00,0xe99b8600
,0xe9869c00,0xe4bb8000,0xe4bd8f00,0xe5858500,0xe58d8100,0xe5be9300,0xe6888e00,0xe69f9400
,0xe6b18100,0xe6b88b00,0xe78da300,0xe7b8a600,0xe9878d00,0xe98a8300,0xe58f9400,0xe5a49900
,0xe5aebf00,0xe6b79100,0xe7a59d00,0xe7b8ae00,0xe7b29b00,0xe5a1be00,0xe7869f00,0xe587ba00
,0xe8a19300,0xe8bfb000,0xe4bf8a00,0xe5b3bb00,0xe698a500,0xe79eac00,0xe7aba300,0xe8889c00
,0xe9a7bf00,0xe5878600,0xe5beaa00,0xe697ac00,0xe6a5af00,0xe6ae8900,0xe6b7b300,0
,0xe6ba9600,0xe6bda400,0xe79bbe00,0xe7b49400,0xe5b7a100,0xe981b500,0xe9868700,0xe9a08600
,0xe587a600,0xe5889d00,0xe6898000,0xe69a9100,0xe69b9900,0xe6b89a00,0xe5bab600,0xe7b79200
,0xe7bdb200,0xe69bb800,0xe896af00,0xe897b700,0xe8abb800,0xe58aa900,0xe58f9900,0xe5a5b300
,0xe5ba8f00,0xe5be9000,0xe6819500,0xe98ba400,0xe999a400,0xe582b700,0xe5849f00,0xe58b9d00
,0xe58ca000,0xe58d8700,0xe58fac00,0xe593a800,0xe5958600,0xe594b100,0xe5989700,0xe5a5a800
,0xe5a6be00,0xe5a8bc00,0xe5aeb500,0xe5b08600,0xe5b08f00,0xe5b09100,0xe5b09a00,0xe5ba8400
,0xe5ba8a00,0xe5bba000,0xe5bdb000,0xe689bf00,0xe68a8400,0xe68b9b00,0xe68e8c00,0xe68db700
,0xe6988700,0xe6988c00,0xe698ad00,0xe699b600,0xe69dbe00,0xe6a2a200,0xe6a89f00,0xe6a8b500
,0xe6b2bc00,0xe6b68800,0xe6b88900,0xe6b99800,0xe784bc00,0xe784a600,0xe785a700,0xe7978700
,0xe79c8100,0xe7a19d00,0xe7a48100,0xe7a5a500,0xe7a7b000,0xe7aba000,0xe7ac9100,0xe7b2a700
,0xe7b4b900,0xe8829600,0xe88f9600,0xe8928b00,0xe8958900,0xe8a19d00,0xe8a3b300,0xe8a89f00
,0xe8a8bc00,0xe8a99400,0xe8a9b300,0xe8b1a100,0xe8b39e00,0xe986a400,0xe989a600,0xe98dbe00
,0xe9909800,0xe99a9c00,0xe99e9800,0xe4b88a00,0xe4b88800,0xe4b89e00,0xe4b99700,0xe5869700
,0xe589b000,0xe59f8e00,0xe5a0b400,0xe5a38c00,0xe5aca200,0xe5b8b800,0xe6838500,0xe693be00
,0xe69da100,0xe69d9600,0xe6b58400,0xe78ab600,0xe795b300,0xe7a9a300,0xe892b800,0xe8adb200
,0xe986b800,0xe98ca000,0xe598b100,0xe59fb400,0xe9a3be00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe68bad00,0xe6a48d00,0xe6ae9600,0xe787ad00,0xe7b99400,0xe881b700,0xe889b200,0xe8a7a600
,0xe9a39f00,0xe89d9500,0xe8beb100,0xe5b0bb00,0xe4bcb800,0xe4bfa100,0xe4beb500,0xe5948700
,0xe5a8a000,0xe5af9d00,0xe5afa900,0xe5bf8300,0xe6858e00,0xe68caf00,0xe696b000,0xe6998b00
,0xe6a3ae00,0xe6a69b00,0xe6b5b800,0xe6b7b100,0xe794b300,0xe796b900,0xe79c9f00,0xe7a59e00
,0xe7a7a600,0xe7b4b300,0xe887a300,0xe88aaf00,0xe896aa00,0xe8a6aa00,0xe8a8ba00,0xe8baab00
,0xe8be9b00,0xe980b200,0xe9879d00,0xe99c8700,0xe4baba00,0xe4bb8100,0xe5888300,0xe5a1b500
,0xe5a3ac00,0xe5b08b00,0xe7949a00,0xe5b0bd00,0xe8858e00,0xe8a88a00,0xe8bf8500,0xe999a300
,0xe99dad00,0xe7aca500,0xe8ab8f00,0xe9a08800,0xe985a200,0xe59bb300,0xe58ea800,0
,0xe9809700,0xe590b900,0xe59e8200,0xe5b8a500,0xe68ea800,0xe6b0b400,0xe7828a00,0xe79da100
,0xe7b28b00,0xe7bfa000,0xe8a1b000,0xe9818200,0xe9859400,0xe98c9000,0xe98c9800,0xe99a8f00
,0xe7919e00,0xe9ab8400,0xe5b48700,0xe5b5a900,0xe695b000,0xe69ea200,0xe8b6a800,0xe99b9b00
,0xe68dae00,0xe69d8900,0xe6a49900,0xe88f8500,0xe9a09700,0xe99b8000,0xe8a3be00,0xe6be8400
,0xe691ba00,0xe5afb800,0xe4b89600,0xe780ac00,0xe7959d00,0xe698af00,0xe5878400,0xe588b600
,0xe58ba200,0xe5a79300,0xe5be8100,0xe680a700,0xe6889000,0xe694bf00,0xe695b400,0xe6989f00
,0xe699b400,0xe6a3b200,0xe6a09600,0xe6ada300,0xe6b88500,0xe789b200,0xe7949f00,0xe79b9b00
,0xe7b2be00,0xe8819600,0xe5a3b000,0xe8a3bd00,0xe8a5bf00,0xe8aaa000,0xe8aa9300,0xe8ab8b00
,0xe9809d00,0xe9869200,0xe99d9200,0xe99d9900,0xe6968900,0xe7a88e00,0xe8848600,0xe99abb00
,0xe5b8ad00,0xe6839c00,0xe6889a00,0xe696a500,0xe6989400,0xe69e9000,0xe79fb300,0xe7a98d00
,0xe7b18d00,0xe7b8be00,0xe8848a00,0xe8b2ac00,0xe8b5a400,0xe8b7a100,0xe8b99f00,0xe7a2a900
,0xe5888700,0xe68b9900,0xe68ea500,0xe6918200,0xe68a9800,0xe8a8ad00,0xe7aa8300,0xe7af8000
,0xe8aaac00,0xe99baa00,0xe7b5b600,0xe8888c00,0xe89d8900,0xe4bb9900,0xe5858800,0xe58d8300
,0xe58da000,0xe5aea300,0xe5b08200,0xe5b09600,0xe5b79d00,0xe688a600,0xe6898700,0xe692b000
,0xe6a09300,0xe6a0b400,0xe6b38900,0xe6b58500,0xe6b49700,0xe69f9300,0xe6bd9c00,0xe7858e00
,0xe785bd00,0xe6978b00,0xe7a9bf00,0xe7aead00,0xe7b79a00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe7b98a00,0xe7bea800,0xe885ba00,0xe8889b00,0xe888b900,0xe896a600,0xe8a9ae00,0xe8b38e00
,0xe8b7b500,0xe981b800,0xe981b700,0xe98aad00,0xe98a9100,0xe9968300,0xe9aeae00,0xe5898d00
,0xe5968400,0xe6bcb800,0xe784b600,0xe585a800,0xe7a68500,0xe7b99500,0xe886b300,0xe7b38e00
,0xe5998c00,0xe5a19100,0xe5b2a800,0xe68eaa00,0xe69bbe00,0xe69bbd00,0xe6a59a00,0xe78b9900
,0xe7968f00,0xe7968e00,0xe7a48e00,0xe7a59600,0xe7a79f00,0xe7b29700,0xe7b4a000,0xe7b58400
,0xe8988700,0xe8a8b400,0xe998bb00,0xe981a100,0xe9bca000,0xe583a700,0xe589b500,0xe58f8c00
,0xe58fa200,0xe5808900,0xe596aa00,0xe5a3ae00,0xe5a58f00,0xe788bd00,0xe5ae8b00,0xe5b1a400
,0xe58c9d00,0xe683a300,0xe683b300,0xe68d9c00,0xe68e8300,0xe68cbf00,0xe68ebb00,0
,0xe6938d00,0xe697a900,0xe69bb900,0xe5b7a300,0xe6a78d00,0xe6a7bd00,0xe6bc9500,0xe787a500
,0xe4ba8900,0xe797a900,0xe79bb800,0xe7aa9300,0xe7b39f00,0xe7b78f00,0xe7b69c00,0xe881a100
,0xe88d8900,0xe88d9800,0xe891ac00,0xe892bc00,0xe897bb00,0xe8a38500,0xe8b5b000,0xe9808100
,0xe981ad00,0xe98e9700,0xe99c9c00,0xe9a89200,0xe5838f00,0xe5a29700,0xe6868e00,0xe8879300
,0xe894b500,0xe8b48800,0xe980a000,0xe4bf8300,0xe581b400,0xe5898700,0xe58db300,0xe681af00
,0xe68d8900,0xe69d9f00,0xe6b8ac00,0xe8b6b300,0xe9809f00,0xe4bf9700,0xe5b19e00,0xe8b38a00
,0xe6978f00,0xe7b69a00,0xe58d9200,0xe8a29600,0xe585b600,0xe68f8300,0xe5ad9800,0xe5adab00
,0xe5b08a00,0xe6908d00,0xe69d9100,0xe9819c00,0xe4bb9600,0xe5a49a00,0xe5a4aa00,0xe6b1b000
,0xe8a99100,0xe594be00,0xe5a09500,0xe5a6a500,0xe683b000,0xe6899300,0xe69f8100,0xe888b500
,0xe6a59500,0xe9998000,0xe9a78400,0xe9a8a800,0xe4bd9300,0xe5a08600,0xe5afbe00,0xe8809000
,0xe5b2b100,0xe5b8af00,0xe5be8500,0xe680a000,0xe6858b00,0xe688b400,0xe69bbf00,0xe6b3b000
,0xe6bb9e00,0xe8838e00,0xe885bf00,0xe88b9400,0xe8a28b00,0xe8b2b800,0xe9808000,0xe980ae00
,0xe99a8a00,0xe9bb9b00,0xe9af9b00,0xe4bba300,0xe58fb000,0xe5a4a700,0xe7acac00,0xe9868d00
,0xe9a18c00,0xe9b7b900,0xe6bb9d00,0xe780a700,0xe58d9300,0xe5958400,0xe5ae8500,0xe6899800
,0xe68a9e00,0xe68b9300,0xe6b2a200,0xe6bfaf00,0xe790a200,0xe8a89700,0xe990b800,0xe6bf8100
,0xe8abbe00,0xe88cb800,0xe587a700,0xe89bb800,0xe58faa00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe58fa900,0xe4bd8600,0xe9819400,0xe8beb000,0xe5a5aa00,0xe884b100,0xe5b7bd00,0xe7abaa00
,0xe8bebf00,0xe6a39a00,0xe8b0b700,0xe78bb800,0xe9b18800,0xe6a8bd00,0xe8aab000,0xe4b8b900
,0xe58d9800,0xe5988600,0xe59da600,0xe68b8500,0xe68ea200,0xe697a600,0xe6ad8e00,0xe6b7a100
,0xe6b99b00,0xe782ad00,0xe79fad00,0xe7abaf00,0xe7aeaa00,0xe7b6bb00,0xe880bd00,0xe8838600
,0xe89b8b00,0xe8aa9500,0xe98d9b00,0xe59ba300,0xe5a38700,0xe5bcbe00,0xe696ad00,0xe69a9600
,0xe6aa8000,0xe6aeb500,0xe794b700,0xe8ab8700,0xe580a400,0xe79fa500,0xe59cb000,0xe5bc9b00
,0xe681a500,0xe699ba00,0xe6b1a000,0xe797b400,0xe7a89a00,0xe7bdae00,0xe887b400,0xe89c9800
,0xe9818500,0xe9a6b300,0xe7af8900,0xe7959c00,0xe7abb900,0xe7ad9100,0xe8938400,0
,0xe9809000,0xe7a7a900,0xe7aa9200,0xe88cb600,0xe5aba100,0xe79d8000,0xe4b8ad00,0xe4bbb200
,0xe5ae9900,0xe5bfa000,0xe68abd00,0xe698bc00,0xe69fb100,0xe6b3a800,0xe899ab00,0xe8a1b700
,0xe8a8bb00,0xe9858e00,0xe98bb300,0xe9a79000,0xe6a89700,0xe780a600,0xe78caa00,0xe88ba700
,0xe8919700,0xe8b2af00,0xe4b88100,0xe5858600,0xe5878b00,0xe5968b00,0xe5afb500,0xe5b89600
,0xe5b8b300,0xe5ba8100,0xe5bc9400,0xe5bcb500,0xe5bdab00,0xe5beb400,0xe687b200,0xe68c9100
,0xe69aa200,0xe69c9d00,0xe6bdae00,0xe7899200,0xe794ba00,0xe79cba00,0xe881b400,0xe884b900
,0xe885b800,0xe89db600,0xe8aabf00,0xe8ab9c00,0xe8b68500,0xe8b7b300,0xe98a9a00,0xe995b700
,0xe9a08200,0xe9b3a500,0xe58b8500,0xe68d9700,0xe79bb400,0xe69c9500,0xe6b28800,0xe78f8d00
,0xe8b38300,0xe98eae00,0xe999b300,0xe6b4a500,0xe5a29c00,0xe6a48e00,0xe6a78c00,0xe8bfbd00
,0xe98e9a00,0xe7979b00,0xe9809a00,0xe5a19a00,0xe6a08200,0xe68eb400,0xe6a7bb00,0xe4bd8300
,0xe6bcac00,0xe69f9800,0xe8bebb00,0xe894a600,0xe7b6b400,0xe98d9400,0xe6a4bf00,0xe6bdb000
,0xe59daa00,0xe5a3b700,0xe5acac00,0xe7b4ac00,0xe788aa00,0xe5908a00,0xe987a300,0xe9b6b400
,0xe4baad00,0xe4bd8e00,0xe5819c00,0xe581b500,0xe5898300,0xe8b29e00,0xe5918800,0xe5a0a400
,0xe5ae9a00,0xe5b89d00,0xe5ba9500,0xe5baad00,0xe5bbb700,0xe5bc9f00,0xe6828c00,0xe68ab500
,0xe68cba00,0xe68f9000,0xe6a2af00,0xe6b18000,0xe7a28700,0xe7a68e00,0xe7a88b00,0xe7b7a000
,0xe8898700,0xe8a88200,0xe8aba600,0xe8b98400,0xe9809300,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe982b800,0xe984ad00,0xe9879800,0xe9bc8e00,0xe6b3a500,0xe6919800,0xe693a200,0xe695b500
,0xe6bbb400,0xe79a8400,0xe7ac9b00,0xe981a900,0xe98f9100,0xe6baba00,0xe593b200,0xe5beb900
,0xe692a400,0xe8bd8d00,0xe8bfad00,0xe9898400,0xe585b800,0xe5a1ab00,0xe5a4a900,0xe5b19500
,0xe5ba9700,0xe6b7bb00,0xe7ba8f00,0xe7949c00,0xe8b2bc00,0xe8bba200,0xe9a19b00,0xe782b900
,0xe4bc9d00,0xe6aebf00,0xe6beb100,0xe794b000,0xe99bbb00,0xe5858e00,0xe5909000,0xe5a0b500
,0xe5a19700,0xe5a6ac00,0xe5b1a000,0xe5be9200,0xe6969700,0xe69d9c00,0xe6b8a100,0xe799bb00
,0xe88f9f00,0xe8b3ad00,0xe9809400,0xe983bd00,0xe98d8d00,0xe7a0a500,0xe7a0ba00,0xe58aaa00
,0xe5baa600,0xe59c9f00,0xe5a5b400,0xe6809200,0xe5809200,0xe5859a00,0xe586ac00,0
,0xe5878d00,0xe5888000,0xe5949000,0xe5a19400,0xe5a19800,0xe5a59700,0xe5ae9500,0xe5b3b600
,0xe5b68b00,0xe682bc00,0xe68a9500,0xe690ad00,0xe69db100,0xe6a18300,0xe6a2bc00,0xe6a39f00
,0xe79b9700,0xe6b79800,0xe6b9af00,0xe6b69b00,0xe781af00,0xe7878800,0xe5bd9300,0xe7979800
,0xe7a5b700,0xe7ad8900,0xe7ad9400,0xe7ad9200,0xe7b39600,0xe7b5b100,0xe588b000,0xe891a300
,0xe895a900,0xe897a400,0xe8a88e00,0xe8ac8400,0xe8b18600,0xe8b88f00,0xe9808300,0xe9808f00
,0xe9909900,0xe999b600,0xe9a0ad00,0xe9a8b000,0xe9979800,0xe5838d00,0xe58b9500,0xe5908c00
,0xe5a08200,0xe5b08e00,0xe686a700,0xe6929e00,0xe6b49e00,0xe79eb300,0xe7aba500,0xe883b400
,0xe8908400,0xe9819300,0xe98a8500,0xe5b3a000,0xe9b48700,0xe58cbf00,0xe5be9700,0xe5beb300
,0xe6b69c00,0xe789b900,0xe79da300,0xe7a6bf00,0xe7afa400,0xe6af9200,0xe78bac00,0xe8aaad00
,0xe6a08300,0xe6a9a100,0xe587b800,0xe7aa8100,0xe6a4b400,0xe5b18a00,0xe9b3b600,0xe88bab00
,0xe5af8500,0xe9858900,0xe7809e00,0xe599b800,0xe5b1af00,0xe6838700,0xe695a600,0xe6b28c00
,0xe8b19a00,0xe9818100,0xe9a09300,0xe5919100,0xe69b8700,0xe9888d00,0xe5a58800,0xe982a300
,0xe5868500,0xe4b98d00,0xe587aa00,0xe8969900,0xe8ac8e00,0xe7819800,0xe68dba00,0xe98d8b00
,0xe6a5a200,0xe9a6b400,0xe7b88400,0xe795b700,0xe58d9700,0xe6a5a000,0xe8bb9f00,0xe99ba300
,0xe6b19d00,0xe4ba8c00,0xe5b0bc00,0xe5bc9000,0xe8bfa900,0xe58c8200,0xe8b39100,0xe8828900
,0xe899b900,0xe5bbbf00,0xe697a500,0xe4b9b300,0xe585a500,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe5a68200,0xe5b0bf00,0xe99fae00,0xe4bbbb00,0xe5a68a00,0xe5bf8d00,0xe8aa8d00,0xe6bfa100
,0xe7a6b000,0xe7a5a200,0xe5afa700,0xe891b100,0xe78cab00,0xe786b100,0xe5b9b400,0xe5bfb500
,0xe68dbb00,0xe6929a00,0xe7878300,0xe7b29800,0xe4b98300,0xe5bbbc00,0xe4b98b00,0xe59f9c00
,0xe59aa200,0xe682a900,0xe6bf8300,0xe7b48d00,0xe883bd00,0xe884b300,0xe886bf00,0xe8beb200
,0xe8a69700,0xe89aa400,0xe5b7b400,0xe68a8a00,0xe692ad00,0xe8a68700,0xe69db700,0xe6b3a200
,0xe6b4be00,0xe790b600,0xe7a0b400,0xe5a98600,0xe7bdb500,0xe88aad00,0xe9a6ac00,0xe4bfb300
,0xe5bb8300,0xe68b9d00,0xe68e9200,0xe6959700,0xe69daf00,0xe79b8300,0xe7898c00,0xe8838c00
,0xe882ba00,0xe8bca900,0xe9858d00,0xe5808d00,0xe59fb900,0xe5aa9200,0xe6a28500,0
,0xe6a5b300,0xe785a400,0xe78bbd00,0xe8b2b700,0xe5a3b200,0xe8b3a000,0xe999aa00,0xe9809900
,0xe89dbf00,0xe7a7a400,0xe79fa700,0xe890a900,0xe4bcaf00,0xe589a500,0xe58d9a00,0xe68b8d00
,0xe69f8f00,0xe6b38a00,0xe799bd00,0xe7ae9400,0xe7b29500,0xe888b600,0xe8968400,0xe8bfab00
,0xe69b9d00,0xe6bca000,0xe7888600,0xe7b89b00,0xe88eab00,0xe9a78100,0xe9baa600,0xe587bd00
,0xe7aeb100,0xe7a1b200,0xe7aeb800,0xe8828700,0xe7ad8800,0xe6aba800,0xe5b9a100,0xe8828c00
,0xe7959100,0xe795a000,0xe585ab00,0xe989a200,0xe6ba8c00,0xe799ba00,0xe9869700,0xe9abaa00
,0xe4bc9000,0xe7bdb000,0xe68a9c00,0xe7ad8f00,0xe996a500,0xe9b3a900,0xe599ba00,0xe5a19900
,0xe89ba400,0xe99abc00,0xe4bcb400,0xe588a400,0xe58d8a00,0xe58f8d00,0xe58f9b00,0xe5b88600
,0xe690ac00,0xe6969100,0xe69dbf00,0xe6b0be00,0xe6b18e00,0xe7898800,0xe78aaf00,0xe78fad00
,0xe7959400,0xe7b98100,0xe888ac00,0xe897a900,0xe8b2a900,0xe7af8400,0xe9878600,0xe785a900
,0xe9a09200,0xe9a3af00,0xe68cbd00,0xe699a900,0xe795aa00,0xe79ba400,0xe7a39000,0xe8958300
,0xe89bae00,0xe58caa00,0xe58d9100,0xe590a600,0xe5a68300,0xe5ba8700,0xe5bdbc00,0xe682b200
,0xe6898900,0xe689b900,0xe68aab00,0xe6969000,0xe6af9400,0xe6b38c00,0xe796b200,0xe79aae00
,0xe7a29100,0xe7a79800,0xe7b78b00,0xe7bdb700,0xe882a500,0xe8a2ab00,0xe8aab900,0xe8b2bb00
,0xe981bf00,0xe99d9e00,0xe9a39b00,0xe6a88b00,0xe7b0b800,0xe5829900,0xe5b0be00,0xe5beae00
,0xe69e8700,0xe6af9800,0xe790b500,0xe79c8900,0xe7be8e00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe9bcbb00,0xe69f8a00,0xe7a89700,0xe58cb900,0xe7968b00,0xe9abad00,0xe5bda600,0xe8869d00
,0xe88fb100,0xe8829800,0xe5bcbc00,0xe5bf8500,0xe795a200,0xe7ad8600,0xe980bc00,0xe6a1a700
,0xe5a7ab00,0xe5aa9b00,0xe7b49000,0xe799be00,0xe8acac00,0xe4bfb500,0xe5bdaa00,0xe6a89900
,0xe6b0b700,0xe6bc8200,0xe793a200,0xe7a5a800,0xe8a1a800,0xe8a99500,0xe8b1b900,0xe5bb9f00
,0xe68f8f00,0xe7978500,0xe7a79200,0xe88b9700,0xe98ca800,0xe98bb200,0xe8929c00,0xe89bad00
,0xe9b0ad00,0xe5938100,0xe5bdac00,0xe6968c00,0xe6b59c00,0xe7809500,0xe8b2a700,0xe8b39300
,0xe9a0bb00,0xe6958f00,0xe793b600,0xe4b88d00,0xe4bb9800,0xe59fa000,0xe5a4ab00,0xe5a9a600
,0xe5af8c00,0xe586a800,0xe5b88300,0xe5ba9c00,0xe6809600,0xe689b600,0xe695b700,0
,0xe696a700,0xe699ae00,0xe6b5ae00,0xe788b600,0xe7aca600,0xe8859000,0xe8869a00,0xe88a9900
,0xe8ad9c00,0xe8b2a000,0xe8b3a600,0xe8b5b400,0xe9989c00,0xe9998400,0xe4beae00,0xe692ab00
,0xe6ada600,0xe8889e00,0xe891a100,0xe895aa00,0xe983a800,0xe5b08100,0xe6a59300,0xe9a2a800
,0xe891ba00,0xe8959700,0xe4bc8f00,0xe589af00,0xe5bea900,0xe5b98500,0xe69c8d00,0xe7a68f00
,0xe885b900,0xe8a48700,0xe8a68600,0xe6b7b500,0xe5bc9700,0xe6899500,0xe6b2b800,0xe4bb8f00
,0xe789a900,0xe9ae9200,0xe5888600,0xe590bb00,0xe599b400,0xe5a2b300,0xe686a400,0xe689ae00
,0xe7849a00,0xe5a5ae00,0xe7b28900,0xe7b39e00,0xe7b49b00,0xe99bb000,0xe6968700,0xe8819e00
,0xe4b89900,0xe4bdb500,0xe585b500,0xe5a18000,0xe5b9a300,0xe5b9b300,0xe5bc8a00,0xe69f8400
,0xe4b8a600,0xe894bd00,0xe9968900,0xe9999b00,0xe7b1b300,0xe9a08100,0xe583bb00,0xe5a38100
,0xe7999600,0xe7a2a700,0xe588a500,0xe79ea500,0xe8949100,0xe7ae8600,0xe5818f00,0xe5a48900
,0xe7898700,0xe7af8700,0xe7b7a800,0xe8beba00,0xe8bf9400,0xe9818d00,0xe4bebf00,0xe58b8900
,0xe5a8a900,0xe5bc8100,0xe99ead00,0xe4bf9d00,0xe8889700,0xe98baa00,0xe59c8300,0xe68d9500
,0xe6ada900,0xe794ab00,0xe8a39c00,0xe8bc9400,0xe7a98200,0xe58b9f00,0xe5a29300,0xe6859500
,0xe6888a00,0xe69aae00,0xe6af8d00,0xe7b0bf00,0xe88fa900,0xe580a300,0xe4bfb800,0xe58c8500
,0xe5918600,0xe5a0b100,0xe5a58900,0xe5ae9d00,0xe5b3b000,0xe5b3af00,0xe5b4a900,0xe5ba9600
,0xe68ab100,0xe68da700,0xe694be00,0xe696b900,0xe69c8b00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe6b39500,0xe6b3a100,0xe783b900,0xe7a0b200,0xe7b8ab00,0xe8839e00,0xe88ab300,0xe8908c00
,0xe893ac00,0xe89c8200,0xe8a49200,0xe8a8aa00,0xe8b18a00,0xe982a600,0xe98b9200,0xe9a3bd00
,0xe9b3b300,0xe9b5ac00,0xe4b98f00,0xe4baa100,0xe5828d00,0xe5899600,0xe59d8a00,0xe5a6a800
,0xe5b8bd00,0xe5bf9800,0xe5bf9900,0xe688bf00,0xe69ab400,0xe69c9b00,0xe69f9000,0xe6a39200
,0xe5869200,0xe7b4a100,0xe882aa00,0xe886a800,0xe8ac8000,0xe8b28c00,0xe8b2bf00,0xe989be00
,0xe998b200,0xe590a000,0xe9a0ac00,0xe58c9700,0xe5839500,0xe58d9c00,0xe5a2a800,0xe692b200
,0xe69cb400,0xe789a700,0xe79da600,0xe7a98600,0xe987a600,0xe58b8300,0xe6b2a100,0xe6ae8600
,0xe5a08000,0xe5b98c00,0xe5a59400,0xe69cac00,0xe7bfbb00,0xe587a100,0xe79b8600,0
,0xe691a900,0xe7a3a800,0xe9ad9400,0xe9babb00,0xe59f8b00,0xe5a6b900,0xe698a700,0xe69e9a00
,0xe6af8e00,0xe593a900,0xe6a79900,0xe5b99500,0xe8869c00,0xe69e9500,0xe9aeaa00,0xe69fbe00
,0xe9b19200,0xe6a19d00,0xe4baa600,0xe4bfa300,0xe58f8800,0xe68ab900,0xe69cab00,0xe6b2ab00
,0xe8bf8400,0xe4bead00,0xe7b9ad00,0xe9babf00,0xe4b88700,0xe685a200,0xe6ba8000,0xe6bcab00
,0xe8949300,0xe591b300,0xe69caa00,0xe9ad8500,0xe5b7b300,0xe7ae9500,0xe5b2ac00,0xe5af8600
,0xe89c9c00,0xe6b98a00,0xe8939100,0xe7a89400,0xe8848800,0xe5a69900,0xe7b28d00,0xe6b09100
,0xe79ca000,0xe58b9900,0xe5a4a200,0xe784a100,0xe7899f00,0xe79f9b00,0xe99ca700,0xe9b5a100
,0xe6a48b00,0xe5a9bf00,0xe5a89800,0xe586a500,0xe5908d00,0xe591bd00,0xe6988e00,0xe79b9f00
,0xe8bfb700,0xe98a9800,0xe9b3b400,0xe5a7aa00,0xe7899d00,0xe6bb8500,0xe5858d00,0xe6a38900
,0xe7b6bf00,0xe7b7ac00,0xe99da200,0xe9baba00,0xe691b800,0xe6a8a100,0xe88c8200,0xe5a68400
,0xe5ad9f00,0xe6af9b00,0xe78c9b00,0xe79bb200,0xe7b6b200,0xe8809700,0xe8929900,0xe584b200
,0xe69ca800,0xe9bb9900,0xe79bae00,0xe69da200,0xe58bbf00,0xe9a48500,0xe5b0a400,0xe688bb00
,0xe7b1be00,0xe8b2b000,0xe5958f00,0xe682b600,0xe7b48b00,0xe9968000,0xe58c8100,0xe4b99f00
,0xe586b600,0xe5a49c00,0xe788ba00,0xe880b600,0xe9878e00,0xe5bca500,0xe79fa200,0xe58e8400
,0xe5bdb900,0xe7b48400,0xe896ac00,0xe8a8b300,0xe8ba8d00,0xe99d9600,0xe69fb300,0xe896ae00
,0xe9919300,0xe6848900,0xe6848800,0xe6b2b900,0xe7999200,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe8abad00,0xe8bcb800,0xe594af00,0xe4bd9100,0xe584aa00,0xe58b8700,0xe58f8b00,0xe5aea500
,0xe5b9bd00,0xe682a000,0xe6868200,0xe68f9600,0xe69c8900,0xe69f9a00,0xe6b9a700,0xe6b68c00
,0xe78cb600,0xe78cb700,0xe794b100,0xe7a59000,0xe8a39500,0xe8aa9800,0xe9818a00,0xe9829100
,0xe983b500,0xe99b8400,0xe89e8d00,0xe5a49500,0xe4ba8800,0xe4bd9900,0xe4b88e00,0xe8aa8900
,0xe8bcbf00,0xe9a09000,0xe582ad00,0xe5b9bc00,0xe5a69600,0xe5aeb900,0xe5bab800,0xe68f9a00
,0xe68fba00,0xe6938100,0xe69b9c00,0xe6a58a00,0xe6a79800,0xe6b48b00,0xe6bab600,0xe7869400
,0xe794a800,0xe7aaaf00,0xe7be8a00,0xe8808000,0xe8918900,0xe8938900,0xe8a68100,0xe8aca100
,0xe8b88a00,0xe981a500,0xe999bd00,0xe9a48a00,0xe685be00,0xe68a9100,0xe6acb200,0
,0xe6b28300,0xe6b5b400,0xe7bf8c00,0xe7bfbc00,0xe6b78000,0xe7be8500,0xe89eba00,0xe8a3b800
,0xe69da500,0xe88eb100,0xe9a0bc00,0xe99bb700,0xe6b49b00,0xe7b5a100,0xe890bd00,0xe985aa00
,0xe4b9b100,0xe58db500,0xe5b59000,0xe6ac8400,0xe6bfab00,0xe8978d00,0xe898ad00,0xe8a6a700
,0xe588a900,0xe5908f00,0xe5b1a500,0xe69d8e00,0xe6a2a800,0xe7908600,0xe7928300,0xe797a200
,0xe8a38f00,0xe8a3a100,0xe9878c00,0xe99ba200,0xe999b800,0xe5be8b00,0xe78e8700,0xe7ab8b00
,0xe8918e00,0xe68ea000,0xe795a500,0xe58a8900,0xe6b58100,0xe6ba9c00,0xe7908900,0xe7959900
,0xe7a1ab00,0xe7b29200,0xe99a8600,0xe7ab9c00,0xe9be8d00,0xe4beb600,0xe685ae00,0xe6978500
,0xe8999c00,0xe4ba8600,0xe4baae00,0xe5839a00,0xe4b8a100,0xe5878c00,0xe5afae00,0xe6969900
,0xe6a28100,0xe6b6bc00,0xe78c9f00,0xe7998200,0xe79ead00,0xe7a89c00,0xe7b3a700,0xe889af00
,0xe8ab9200,0xe981bc00,0xe9878f00,0xe999b500,0xe9a09800,0xe58a9b00,0xe7b79100,0xe580ab00
,0xe58e9800,0xe69e9700,0xe6b78b00,0xe7879000,0xe790b300,0xe887a800,0xe8bcaa00,0xe99aa300
,0xe9b19700,0xe9ba9f00,0xe791a000,0xe5a18100,0xe6b69900,0xe7b4af00,0xe9a19e00,0xe4bba400
,0xe4bcb600,0xe4be8b00,0xe586b700,0xe58ab100,0xe5b6ba00,0xe6809c00,0xe78eb200,0xe7a4bc00
,0xe88b9300,0xe988b400,0xe99ab700,0xe99bb600,0xe99c8a00,0xe9ba9700,0xe9bda200,0xe69aa600
,0xe6adb400,0xe5889700,0xe58aa300,0xe7838800,0xe8a38200,0xe5bb8900,0xe6818b00,0xe6869000
,0xe6bca300,0xe7858900,0xe7b0be00,0xe7b7b400,0xe881af00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe893ae00,0xe980a300,0xe98cac00,0xe5918200,0xe9adaf00,0xe6ab9300,0xe7828900,0xe8b38200
,0xe8b7af00,0xe99cb200,0xe58ab400,0xe5a98100,0xe5bb8a00,0xe5bc8400,0xe69c9700,0xe6a5bc00
,0xe6a69400,0xe6b5aa00,0xe6bc8f00,0xe789a200,0xe78bbc00,0xe7afad00,0xe8808100,0xe881be00
,0xe89d8b00,0xe9838e00,0xe585ad00,0xe9ba9300,0xe7a68400,0xe8828b00,0xe98cb200,0xe8ab9600
,0xe580ad00,0xe5928c00,0xe8a9b100,0xe6adaa00,0xe8b38400,0xe8848700,0xe6839100,0xe69ea000
,0xe9b7b200,0xe4ba9900,0xe4ba9800,0xe9b09000,0xe8a9ab00,0xe8978100,0xe895a800,0xe6a48000
,0xe6b9be00,0xe7a29700,0xe8859500,0xf0a0ae9f,0xe5ad8100,0xe5ad9600,0xe5adbd00,0xe5ae9300
,0xe5af9800,0xe5afac00,0xe5b09200,0xe5b09e00,0xe5b0a300,0xe5b0ab00,0xe39e8d00,0
,0xe5b1a200,0xefa8bb00,0xefa8bc00,0xeaab9400,0xe5b1ba00,0xe5b28f00,0xe5b29f00,0xe5b2a300
,0xe5b2aa00,0xe5b2ba00,0xe5b38b00,0xe5b39000,0xe5b39200,0xe5b3b400,0xeaab9d00,0xe39fa200
,0xe5b48d00,0xe5b4a700,0xefa89100,0xe5b58600,0xe5b58700,0xe5b59300,0xe5b58a00,0xe5b5ad00
,0xe5b68100,0xe5b6a000,0xe5b6a400,0xe5b6a700,0xe5b6b800,0xe5b78b00,0xe5909e00,0xe5bc8c00
,0xe4b89000,0xe4b89500,0xe4b8aa00,0xe4b8b100,0xe4b8b600,0xe4b8bc00,0xe4b8bf00,0xe4b98200
,0xe4b99600,0xe4b99800,0xe4ba8200,0xe4ba8500,0xe8b1ab00,0xe4ba8a00,0xe8889200,0xe5bc8d00
,0xe4ba8e00,0xe4ba9e00,0xe4ba9f00,0xe4baa000,0xe4baa200,0xe4bab000,0xe4bab300,0xe4bab600
,0xe4bb8e00,0xe4bb8d00,0xe4bb8400,0xe4bb8600,0xe4bb8200,0xe4bb9700,0xe4bb9e00,0xe4bbad00
,0xe4bb9f00,0xe4bbb700,0xe4bc8900,0xe4bd9a00,0xe4bcb000,0xe4bd9b00,0xe4bd9d00,0xe4bd9700
,0xe4bd8700,0xe4bdb600,0xe4be8800,0xe4be8f00,0xe4be9800,0xe4bdbb00,0xe4bda900,0xe4bdb000
,0xe4be9100,0xe4bdaf00,0xe4be8600,0xe4be9600,0xe5849800,0xe4bf9400,0xe4bf9f00,0xe4bf8e00
,0xe4bf9800,0xe4bf9b00,0xe4bf9100,0xe4bf9a00,0xe4bf9000,0xe4bfa400,0xe4bfa500,0xe5809a00
,0xe580a800,0xe5809400,0xe580aa00,0xe580a500,0xe5808500,0xe4bc9c00,0xe4bfb600,0xe580a100
,0xe580a900,0xe580ac00,0xe4bfbe00,0xe4bfaf00,0xe5809100,0xe5808600,0xe5818300,0xe5818700
,0xe69c8300,0xe5819500,0xe5819000,0xe5818800,0xe5819a00,0xe5819600,0xe581ac00,0xe581b800
,0xe5828000,0xe5829a00,0xe5828500,0xe582b400,0xe582b200,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe5838900,0xe5838a00,0xe582b300,0xe5838200,0xe5839600,0xe5839e00,0xe583a500,0xe583ad00
,0xe583a300,0xe583ae00,0xe583b900,0xe583b500,0xe5848900,0xe5848100,0xe5848200,0xe5849600
,0xe5849500,0xe5849400,0xe5849a00,0xe584a100,0xe584ba00,0xe584b700,0xe584bc00,0xe584bb00
,0xe584bf00,0xe5858000,0xe5859200,0xe5858c00,0xe5859400,0xe585a200,0xe7abb800,0xe585a900
,0xe585aa00,0xe585ae00,0xe5868000,0xe5868200,0xe59b9800,0xe5868c00,0xe5868900,0xe5868f00
,0xe5869100,0xe5869300,0xe5869500,0xe5869600,0xe586a400,0xe586a600,0xe586a200,0xe586a900
,0xe586aa00,0xe586ab00,0xe586b300,0xe586b100,0xe586b200,0xe586b000,0xe586b500,0xe586bd00
,0xe5878500,0xe5878900,0xe5879b00,0xe587a000,0xe8999500,0xe587a900,0xe587ad00,0
,0xe587b000,0xe587b500,0xe587be00,0xe5888400,0xe5888b00,0xe5889400,0xe5888e00,0xe588a700
,0xe588aa00,0xe588ae00,0xe588b300,0xe588b900,0xe5898f00,0xe5898400,0xe5898b00,0xe5898c00
,0xe5899e00,0xe5899400,0xe589aa00,0xe589b400,0xe589a900,0xe589b300,0xe589bf00,0xe589bd00
,0xe58a8d00,0xe58a9400,0xe58a9200,0xe589b100,0xe58a8800,0xe58a9100,0xe8bea800,0xe8bea700
,0xe58aac00,0xe58aad00,0xe58abc00,0xe58ab500,0xe58b8100,0xe58b8d00,0xe58b9700,0xe58b9e00
,0xe58ba300,0xe58ba600,0xe9a3ad00,0xe58ba000,0xe58bb300,0xe58bb500,0xe58bb800,0xe58bb900
,0xe58c8600,0xe58c8800,0xe794b800,0xe58c8d00,0xe58c9000,0xe58c8f00,0xe58c9500,0xe58c9a00
,0xe58ca300,0xe58caf00,0xe58cb100,0xe58cb300,0xe58cb800,0xe58d8000,0xe58d8600,0xe58d8500
,0xe4b89700,0xe58d8900,0xe58d8d00,0xe5879600,0xe58d9e00,0xe58da900,0xe58dae00,0xe5a49800
,0xe58dbb00,0xe58db700,0xe58e8200,0xe58e9600,0xe58ea000,0xe58ea600,0xe58ea500,0xe58eae00
,0xe58eb000,0xe58eb600,0xe58f8300,0xe7b09200,0xe99b9900,0xe58f9f00,0xe69bbc00,0xe787ae00
,0xe58fae00,0xe58fa800,0xe58fad00,0xe58fba00,0xe5908100,0xe590bd00,0xe5918000,0xe590ac00
,0xe590ad00,0xe590bc00,0xe590ae00,0xe590b600,0xe590a900,0xe5909d00,0xe5918e00,0xe5928f00
,0xe591b500,0xe5928e00,0xe5919f00,0xe591b100,0xe591b700,0xe591b000,0xe5929200,0xe591bb00
,0xe5928000,0xe591b600,0xe5928400,0xe5929000,0xe5928600,0xe5938700,0xe592a200,0xe592b800
,0xe592a500,0xe592ac00,0xe5938400,0xe5938800,0xe592a800,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe592ab00,0xe5938200,0xe592a400,0xe592be00,0xe592bc00,0xe5939800,0xe593a500,0xe593a600
,0xe5948f00,0xe5949400,0xe593bd00,0xe593ae00,0xe593ad00,0xe593ba00,0xe593a200,0xe594b900
,0xe5958000,0xe595a300,0xe5958c00,0xe594ae00,0xe5959c00,0xe5958500,0xe5959600,0xe5959700
,0xe594b800,0xe594b300,0xe5959d00,0xe5969900,0xe5968000,0xe592af00,0xe5968a00,0xe5969f00
,0xe595bb00,0xe595be00,0xe5969800,0xe5969e00,0xe596ae00,0xe595bc00,0xe5968300,0xe596a900
,0xe5968700,0xe596a800,0xe5979a00,0xe5978500,0xe5979f00,0xe5978400,0xe5979c00,0xe597a400
,0xe5979400,0xe5989400,0xe597b700,0xe5989600,0xe597be00,0xe597bd00,0xe5989b00,0xe597b900
,0xe5998e00,0xe5999000,0xe7879f00,0xe598b400,0xe598b600,0xe598b200,0xe598b800,0
,0xe599ab00,0xe599a400,0xe598af00,0xe599ac00,0xe599aa00,0xe59a8600,0xe59a8000,0xe59a8a00
,0xe59aa000,0xe59a9400,0xe59a8f00,0xe59aa500,0xe59aae00,0xe59ab600,0xe59ab400,0xe59b8200
,0xe59abc00,0xe59b8100,0xe59b8300,0xe59b8000,0xe59b8800,0xe59b8e00,0xe59b9100,0xe59b9300
,0xe59b9700,0xe59bae00,0xe59bb900,0xe59c8000,0xe59bbf00,0xe59c8400,0xe59c8900,0xe59c8800
,0xe59c8b00,0xe59c8d00,0xe59c9300,0xe59c9800,0xe59c9600,0xe5978700,0xe59c9c00,0xe59ca600
,0xe59cb700,0xe59cb800,0xe59d8e00,0xe59cbb00,0xe59d8000,0xe59d8f00,0xe59da900,0xe59f8000
,0xe59e8800,0xe59da100,0xe59dbf00,0xe59e8900,0xe59e9300,0xe59ea000,0xe59eb300,0xe59ea400
,0xe59eaa00,0xe59eb000,0xe59f8300,0xe59f8600,0xe59f9400,0xe59f9200,0xe59f9300,0xe5a08a00
,0xe59f9600,0xe59fa300,0xe5a08b00,0xe5a09900,0xe5a09d00,0xe5a1b200,0xe5a0a100,0xe5a1a200
,0xe5a18b00,0xe5a1b000,0xe6af8000,0xe5a19200,0xe5a0bd00,0xe5a1b900,0xe5a28500,0xe5a2b900
,0xe5a29f00,0xe5a2ab00,0xe5a2ba00,0xe5a39e00,0xe5a2bb00,0xe5a2b800,0xe5a2ae00,0xe5a38500
,0xe5a39300,0xe5a39100,0xe5a39700,0xe5a39900,0xe5a39800,0xe5a3a500,0xe5a39c00,0xe5a3a400
,0xe5a39f00,0xe5a3af00,0xe5a3ba00,0xe5a3b900,0xe5a3bb00,0xe5a3bc00,0xe5a3bd00,0xe5a48200
,0xe5a48a00,0xe5a49000,0xe5a49b00,0xe6a2a600,0xe5a4a500,0xe5a4ac00,0xe5a4ad00,0xe5a4b200
,0xe5a4b800,0xe5a4be00,0xe7ab9200,0xe5a59500,0xe5a59000,0xe5a58e00,0xe5a59a00,0xe5a59800
,0xe5a5a200,0xe5a5a000,0xe5a5a700,0xe5a5ac00,0xe5a5a900,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe5a5b800,0xe5a68100,0xe5a69d00,0xe4bd9e00,0xe4beab00,0xe5a6a300,0xe5a6b200,0xe5a78600
,0xe5a7a800,0xe5a79c00,0xe5a68d00,0xe5a79900,0xe5a79a00,0xe5a8a500,0xe5a89f00,0xe5a89100
,0xe5a89c00,0xe5a88900,0xe5a89a00,0xe5a98000,0xe5a9ac00,0xe5a98900,0xe5a8b500,0xe5a8b600
,0xe5a9a200,0xe5a9aa00,0xe5aa9a00,0xe5aabc00,0xe5aabe00,0xe5ab8b00,0xe5ab8200,0xe5aabd00
,0xe5aba300,0xe5ab9700,0xe5aba600,0xe5aba900,0xe5ab9600,0xe5abba00,0xe5abbb00,0xe5ac8c00
,0xe5ac8b00,0xe5ac9600,0xe5acb200,0xe5ab9000,0xe5acaa00,0xe5acb600,0xe5acbe00,0xe5ad8300
,0xe5ad8500,0xe5ad8000,0xe5ad9100,0xe5ad9500,0xe5ad9a00,0xe5ad9b00,0xe5ada500,0xe5ada900
,0xe5adb000,0xe5adb300,0xe5adb500,0xe5adb800,0xe6968800,0xe5adba00,0xe5ae8000,0
,0xe5ae8300,0xe5aea600,0xe5aeb800,0xe5af8300,0xe5af8700,0xe5af8900,0xe5af9400,0xe5af9000
,0xe5afa400,0xe5afa600,0xe5afa200,0xe5af9e00,0xe5afa500,0xe5afab00,0xe5afb000,0xe5afb600
,0xe5afb300,0xe5b08500,0xe5b08700,0xe5b08800,0xe5b08d00,0xe5b09300,0xe5b0a000,0xe5b0a200
,0xe5b0a800,0xe5b0b800,0xe5b0b900,0xe5b18100,0xe5b18600,0xe5b18e00,0xe5b19300,0xe5b19000
,0xe5b18f00,0xe5adb100,0xe5b1ac00,0xe5b1ae00,0xe4b9a200,0xe5b1b600,0xe5b1b900,0xe5b28c00
,0xe5b29100,0xe5b29400,0xe5a69b00,0xe5b2ab00,0xe5b2bb00,0xe5b2b600,0xe5b2bc00,0xe5b2b700
,0xe5b38500,0xe5b2be00,0xe5b38700,0xe5b39900,0xe5b3a900,0xe5b3bd00,0xe5b3ba00,0xe5b3ad00
,0xe5b68c00,0xe5b3aa00,0xe5b48b00,0xe5b49500,0xe5b49700,0xe5b59c00,0xe5b49f00,0xe5b49b00
,0xe5b49100,0xe5b49400,0xe5b4a200,0xe5b49a00,0xe5b49900,0xe5b49800,0xe5b58c00,0xe5b59200
,0xe5b58e00,0xe5b58b00,0xe5b5ac00,0xe5b5b300,0xe5b5b600,0xe5b68700,0xe5b68400,0xe5b68200
,0xe5b6a200,0xe5b69d00,0xe5b6ac00,0xe5b6ae00,0xe5b6bd00,0xe5b69000,0xe5b6b700,0xe5b6bc00
,0xe5b78900,0xe5b78d00,0xe5b79300,0xe5b79200,0xe5b79600,0xe5b79b00,0xe5b7ab00,0xe5b7b200
,0xe5b7b500,0xe5b88b00,0xe5b89a00,0xe5b89900,0xe5b89100,0xe5b89b00,0xe5b8b600,0xe5b8b700
,0xe5b98400,0xe5b98300,0xe5b98000,0xe5b98e00,0xe5b99700,0xe5b99400,0xe5b99f00,0xe5b9a200
,0xe5b9a400,0xe5b98700,0xe5b9b500,0xe5b9b600,0xe5b9ba00,0xe9babc00,0xe5b9bf00,0xe5baa000
,0xe5bb8100,0xe5bb8200,0xe5bb8800,0xe5bb9000,0xe5bb8f00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe5bb9600,0xe5bba300,0xe5bb9d00,0xe5bb9a00,0xe5bb9b00,0xe5bba200,0xe5bba100,0xe5bba800
,0xe5bba900,0xe5bbac00,0xe5bbb100,0xe5bbb300,0xe5bbb000,0xe5bbb400,0xe5bbb800,0xe5bbbe00
,0xe5bc8300,0xe5bc8900,0xe5bd9d00,0xe5bd9c00,0xe5bc8b00,0xe5bc9100,0xe5bc9600,0xe5bca900
,0xe5bcad00,0xe5bcb800,0xe5bd8100,0xe5bd8800,0xe5bd8c00,0xe5bd8e00,0xe5bcaf00,0xe5bd9100
,0xe5bd9600,0xe5bd9700,0xe5bd9900,0xe5bda100,0xe5bdad00,0xe5bdb300,0xe5bdb700,0xe5be8300
,0xe5be8200,0xe5bdbf00,0xe5be8a00,0xe5be8800,0xe5be9100,0xe5be8700,0xe5be9e00,0xe5be9900
,0xe5be9800,0xe5bea000,0xe5bea800,0xe5bead00,0xe5bebc00,0xe5bf9600,0xe5bfbb00,0xe5bfa400
,0xe5bfb800,0xe5bfb100,0xe5bf9d00,0xe682b300,0xe5bfbf00,0xe680a100,0xe681a000,0
,0xe6809900,0xe6809000,0xe680a900,0xe6808e00,0xe680b100,0xe6809b00,0xe6809500,0xe680ab00
,0xe680a600,0xe6808f00,0xe680ba00,0xe6819a00,0xe6818100,0xe681aa00,0xe681b700,0xe6819f00
,0xe6818a00,0xe6818600,0xe6818d00,0xe681a300,0xe6818300,0xe681a400,0xe6818200,0xe681ac00
,0xe681ab00,0xe6819900,0xe6828100,0xe6828d00,0xe683a700,0xe6828300,0xe6829a00,0xe6828400
,0xe6829b00,0xe6829600,0xe6829700,0xe6829200,0xe682a700,0xe6828b00,0xe683a100,0xe682b800
,0xe683a000,0xe6839300,0xe682b400,0xe5bfb000,0xe682bd00,0xe6838600,0xe682b500,0xe6839800
,0xe6858d00,0xe6849500,0xe6848600,0xe683b600,0xe683b700,0xe6848000,0xe683b400,0xe683ba00
,0xe6848300,0xe684a100,0xe683bb00,0xe683b100,0xe6848d00,0xe6848e00,0xe6858700,0xe684be00
,0xe684a800,0xe684a700,0xe6858a00,0xe684bf00,0xe684bc00,0xe684ac00,0xe684b400,0xe684bd00
,0xe6858200,0xe6858400,0xe685b300,0xe685b700,0xe6859800,0xe6859900,0xe6859a00,0xe685ab00
,0xe685b400,0xe685af00,0xe685a500,0xe685b100,0xe6859f00,0xe6859d00,0xe6859300,0xe685b500
,0xe6869900,0xe6869600,0xe6868700,0xe686ac00,0xe6869400,0xe6869a00,0xe6868a00,0xe6869100
,0xe686ab00,0xe686ae00,0xe6878c00,0xe6878a00,0xe6878900,0xe687b700,0xe6878800,0xe6878300
,0xe6878600,0xe686ba00,0xe6878b00,0xe7bdb900,0xe6878d00,0xe687a600,0xe687a300,0xe687b600
,0xe687ba00,0xe687b400,0xe687bf00,0xe687bd00,0xe687bc00,0xe687be00,0xe6888000,0xe6888800
,0xe6888900,0xe6888d00,0xe6888c00,0xe6889400,0xe6889b00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe6889e00,0xe688a100,0xe688aa00,0xe688ae00,0xe688b000,0xe688b200,0xe688b300,0xe6898100
,0xe6898e00,0xe6899e00,0xe689a300,0xe6899b00,0xe689a000,0xe689a800,0xe689bc00,0xe68a8200
,0xe68a8900,0xe689be00,0xe68a9200,0xe68a9300,0xe68a9600,0xe68b9400,0xe68a8300,0xe68a9400
,0xe68b9700,0xe68b9100,0xe68abb00,0xe68b8f00,0xe68bbf00,0xe68b8600,0xe6939400,0xe68b8800
,0xe68b9c00,0xe68b8c00,0xe68b8a00,0xe68b8200,0xe68b8700,0xe68a9b00,0xe68b8900,0xe68c8c00
,0xe68bae00,0xe68bb100,0xe68ca700,0xe68c8200,0xe68c8800,0xe68baf00,0xe68bb500,0xe68d9000
,0xe68cbe00,0xe68d8d00,0xe6909c00,0xe68d8f00,0xe68e9600,0xe68e8e00,0xe68e8000,0xe68eab00
,0xe68db600,0xe68ea300,0xe68e8f00,0xe68e8900,0xe68e9f00,0xe68eb500,0xe68dab00,0
,0xe68da900,0xe68ebe00,0xe68fa900,0xe68f8000,0xe68f8600,0xe68fa300,0xe68f8900,0xe68f9200
,0xe68fb600,0xe68f8400,0xe6909600,0xe690b400,0xe6908600,0xe6909300,0xe690a600,0xe690b600
,0xe6949d00,0xe6909700,0xe690a800,0xe6908f00,0xe691a700,0xe691af00,0xe691b600,0xe6918e00
,0xe694aa00,0xe6929500,0xe6929300,0xe692a500,0xe692a900,0xe6928800,0xe692bc00,0xe6939a00
,0xe6939200,0xe6938500,0xe6938700,0xe692bb00,0xe6939800,0xe6938200,0xe693b100,0xe693a700
,0xe8888900,0xe693a000,0xe693a100,0xe68aac00,0xe693a300,0xe693af00,0xe694ac00,0xe693b600
,0xe693b400,0xe693b200,0xe693ba00,0xe6948000,0xe693bd00,0xe6949800,0xe6949c00,0xe6948500
,0xe694a400,0xe694a300,0xe694ab00,0xe694b400,0xe694b500,0xe694b700,0xe694b600,0xe694b800
,0xe7958b00,0xe6958800,0xe6959600,0xe6959500,0xe6958d00,0xe6959800,0xe6959e00,0xe6959d00
,0xe695b200,0xe695b800,0xe6968200,0xe6968300,0xe8ae8a00,0xe6969b00,0xe6969f00,0xe696ab00
,0xe696b700,0xe6978300,0xe6978600,0xe6978100,0xe6978400,0xe6978c00,0xe6979200,0xe6979b00
,0xe6979900,0xe697a000,0xe697a100,0xe697b100,0xe69db200,0xe6988a00,0xe6988300,0xe697bb00
,0xe69db300,0xe698b500,0xe698b600,0xe698b400,0xe6989c00,0xe6998f00,0xe6998400,0xe6998900
,0xe6998100,0xe6999e00,0xe6999d00,0xe699a400,0xe699a700,0xe699a800,0xe6999f00,0xe699a200
,0xe699b000,0xe69a8300,0xe69a8800,0xe69a8e00,0xe69a8900,0xe69a8400,0xe69a9800,0xe69a9d00
,0xe69b8100,0xe69ab900,0xe69b8900,0xe69abe00,0xe69abc00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe69b8400,0xe69ab800,0xe69b9600,0xe69b9a00,0xe69ba000,0xe698bf00,0xe69ba600,0xe69ba900
,0xe69bb000,0xe69bb500,0xe69bb700,0xe69c8f00,0xe69c9600,0xe69c9e00,0xe69ca600,0xe69ca700
,0xe99cb800,0xe69cae00,0xe69cbf00,0xe69cb600,0xe69d8100,0xe69cb800,0xe69cb700,0xe69d8600
,0xe69d9e00,0xe69da000,0xe69d9900,0xe69da300,0xe69da400,0xe69e8900,0xe69db000,0xe69ea900
,0xe69dbc00,0xe69daa00,0xe69e8c00,0xe69e8b00,0xe69ea600,0xe69ea100,0xe69e8500,0xe69eb700
,0xe69faf00,0xe69eb400,0xe69fac00,0xe69eb300,0xe69fa900,0xe69eb800,0xe69fa400,0xe69f9e00
,0xe69f9d00,0xe69fa200,0xe69fae00,0xe69eb900,0xe69f8e00,0xe69f8600,0xe69fa700,0xe6aa9c00
,0xe6a09e00,0xe6a18600,0xe6a0a900,0xe6a18000,0xe6a18d00,0xe6a0b200,0xe6a18e00,0
,0xe6a2b300,0xe6a0ab00,0xe6a19900,0xe6a1a300,0xe6a1b700,0xe6a1bf00,0xe6a29f00,0xe6a28f00
,0xe6a2ad00,0xe6a29400,0xe6a29d00,0xe6a29b00,0xe6a28300,0xe6aaae00,0xe6a2b900,0xe6a1b400
,0xe6a2b500,0xe6a2a000,0xe6a2ba00,0xe6a48f00,0xe6a28d00,0xe6a1be00,0xe6a48100,0xe6a38a00
,0xe6a48800,0xe6a39800,0xe6a4a200,0xe6a4a600,0xe6a3a100,0xe6a48c00,0xe6a38d00,0xe6a39400
,0xe6a3a700,0xe6a39500,0xe6a4b600,0xe6a49200,0xe6a48400,0xe6a39700,0xe6a3a300,0xe6a4a500
,0xe6a3b900,0xe6a3a000,0xe6a3af00,0xe6a4a800,0xe6a4aa00,0xe6a49a00,0xe6a4a300,0xe6a4a100
,0xe6a38600,0xe6a5b900,0xe6a5b700,0xe6a59c00,0xe6a5b800,0xe6a5ab00,0xe6a59400,0xe6a5be00
,0xe6a5ae00,0xe6a4b900,0xe6a5b400,0xe6a4bd00,0xe6a59900,0xe6a4b000,0xe6a5a100,0xe6a59e00
,0xe6a59d00,0xe6a68100,0xe6a5aa00,0xe6a6b200,0xe6a6ae00,0xe6a79000,0xe6a6bf00,0xe6a78100
,0xe6a79300,0xe6a6be00,0xe6a78e00,0xe5afa800,0xe6a78a00,0xe6a79d00,0xe6a6bb00,0xe6a78300
,0xe6a6a700,0xe6a8ae00,0xe6a69100,0xe6a6a000,0xe6a69c00,0xe6a69500,0xe6a6b400,0xe6a79e00
,0xe6a7a800,0xe6a88200,0xe6a89b00,0xe6a7bf00,0xe6ac8a00,0xe6a7b900,0xe6a7b200,0xe6a7a700
,0xe6a88500,0xe6a6b100,0xe6a89e00,0xe6a7ad00,0xe6a89400,0xe6a7ab00,0xe6a88a00,0xe6a89200
,0xe6ab8100,0xe6a8a300,0xe6a89300,0xe6a98400,0xe6a88c00,0xe6a9b200,0xe6a8b600,0xe6a9b800
,0xe6a98700,0xe6a9a200,0xe6a99900,0xe6a9a600,0xe6a98800,0xe6a8b800,0xe6a8a200,0xe6aa9000
,0xe6aa8d00,0xe6aaa000,0xe6aa8400,0xe6aaa200,0xe6aaa300,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe6aa9700,0xe8989700,0xe6aabb00,0xe6ab8300,0xe6ab8200,0xe6aab800,0xe6aab300,0xe6aaac00
,0xe6ab9e00,0xe6ab9100,0xe6ab9f00,0xe6aaaa00,0xe6ab9a00,0xe6abaa00,0xe6abbb00,0xe6ac8500
,0xe8989600,0xe6abba00,0xe6ac9200,0xe6ac9600,0xe9acb100,0xe6ac9f00,0xe6acb800,0xe6acb700
,0xe79b9c00,0xe6acb900,0xe9a3ae00,0xe6ad8700,0xe6ad8300,0xe6ad8900,0xe6ad9000,0xe6ad9900
,0xe6ad9400,0xe6ad9b00,0xe6ad9f00,0xe6ada100,0xe6adb800,0xe6adb900,0xe6adbf00,0xe6ae8000
,0xe6ae8400,0xe6ae8300,0xe6ae8d00,0xe6ae9800,0xe6ae9500,0xe6ae9e00,0xe6aea400,0xe6aeaa00
,0xe6aeab00,0xe6aeaf00,0xe6aeb200,0xe6aeb100,0xe6aeb300,0xe6aeb700,0xe6aebc00,0xe6af8600
,0xe6af8b00,0xe6af9300,0xe6af9f00,0xe6afac00,0xe6afab00,0xe6afb300,0xe6afaf00,0
,0xe9babe00,0xe6b08800,0xe6b09300,0xe6b09400,0xe6b09b00,0xe6b0a400,0xe6b0a300,0xe6b19e00
,0xe6b19500,0xe6b1a200,0xe6b1aa00,0xe6b28200,0xe6b28d00,0xe6b29a00,0xe6b28100,0xe6b29b00
,0xe6b1be00,0xe6b1a800,0xe6b1b300,0xe6b29200,0xe6b29000,0xe6b38400,0xe6b3b100,0xe6b39300
,0xe6b2bd00,0xe6b39700,0xe6b38500,0xe6b39d00,0xe6b2ae00,0xe6b2b100,0xe6b2be00,0xe6b2ba00
,0xe6b39b00,0xe6b3af00,0xe6b39900,0xe6b3aa00,0xe6b49f00,0xe8a18d00,0xe6b4b600,0xe6b4ab00
,0xe6b4bd00,0xe6b4b800,0xe6b49900,0xe6b4b500,0xe6b4b300,0xe6b49200,0xe6b48c00,0xe6b5a300
,0xe6b69300,0xe6b5a400,0xe6b59a00,0xe6b5b900,0xe6b59900,0xe6b68e00,0xe6b69500,0xe6bfa400
,0xe6b68500,0xe6b7b900,0xe6b89500,0xe6b88a00,0xe6b6b500,0xe6b78700,0xe6b7a600,0xe6b6b800
,0xe6b78600,0xe6b7ac00,0xe6b79e00,0xe6b78c00,0xe6b7a800,0xe6b79200,0xe6b78500,0xe6b7ba00
,0xe6b79900,0xe6b7a400,0xe6b79500,0xe6b7aa00,0xe6b7ae00,0xe6b8ad00,0xe6b9ae00,0xe6b8ae00
,0xe6b89900,0xe6b9b200,0xe6b99f00,0xe6b8be00,0xe6b8a300,0xe6b9ab00,0xe6b8ab00,0xe6b9b600
,0xe6b98d00,0xe6b89f00,0xe6b98300,0xe6b8ba00,0xe6b98e00,0xe6b8a400,0xe6bbbf00,0xe6b89d00
,0xe6b8b800,0xe6ba8200,0xe6baaa00,0xe6ba9800,0xe6bb8900,0xe6bab700,0xe6bb9300,0xe6babd00
,0xe6baaf00,0xe6bb8400,0xe6bab200,0xe6bb9400,0xe6bb9500,0xe6ba8f00,0xe6baa500,0xe6bb8200
,0xe6ba9f00,0xe6bd8100,0xe6bc9100,0xe7818c00,0xe6bbac00,0xe6bbb800,0xe6bbbe00,0xe6bcbf00
,0xe6bbb200,0xe6bcb100,0xe6bbaf00,0xe6bcb200,0xe6bb8c00,0,0,0
,0xe6bcbe00,0xe6bc9300,0xe6bbb700,0xe6be8600,0xe6bdba00,0xe6bdb800,0xe6be8100,0xe6be8000
,0xe6bdaf00,0xe6bd9b00,0xe6bfb300,0xe6bdad00,0xe6be8200,0xe6bdbc00,0xe6bd9800,0xe6be8e00
,0xe6be9100,0xe6bf8200,0xe6bda600,0xe6beb300,0xe6bea300,0xe6bea100,0xe6bea400,0xe6beb900
,0xe6bf8600,0xe6beaa00,0xe6bf9f00,0xe6bf9500,0xe6bfac00,0xe6bf9400,0xe6bf9800,0xe6bfb100
,0xe6bfae00,0xe6bf9b00,0xe7808900,0xe7808b00,0xe6bfba00,0xe7809100,0xe7808100,0xe7808f00
,0xe6bfbe00,0xe7809b00,0xe7809a00,0xe6bdb400,0xe7809d00,0xe7809800,0xe7809f00,0xe780b000
,0xe780be00,0xe780b200,0xe7819100,0xe781a300,0xe7829900,0xe7829200,0xe782af00,0xe783b100
,0xe782ac00,0xe782b800,0xe782b300,0xe782ae00,0xe7839f00,0xe7838b00,0xe7839d00,0
,0xe7839900,0xe7848900,0xe783bd00,0xe7849c00,0xe7849900,0xe785a500,0xe7859500,0xe7868800
,0xe785a600,0xe785a200,0xe7858c00,0xe7859600,0xe785ac00,0xe7868f00,0xe787bb00,0xe7868400
,0xe7869500,0xe786a800,0xe786ac00,0xe7879700,0xe786b900,0xe786be00,0xe7879200,0xe7878900
,0xe7879400,0xe7878e00,0xe787a000,0xe787ac00,0xe787a700,0xe787b500,0xe787bc00,0xe787b900
,0xe787bf00,0xe7888d00,0xe7889000,0xe7889b00,0xe788a800,0xe788ad00,0xe788ac00,0xe788b000
,0xe788b200,0xe788bb00,0xe788bc00,0xe788bf00,0xe7898000,0xe7898600,0xe7898b00,0xe7899800
,0xe789b400,0xe789be00,0xe78a8200,0xe78a8100,0xe78a8700,0xe78a9200,0xe78a9600,0xe78aa200
,0xe78aa700,0xe78ab900,0xe78ab200,0xe78b8300,0xe78b8600,0xe78b8400,0xe78b8e00,0xe78b9200
,0xe78ba200,0xe78ba000,0xe78ba100,0xe78bb900,0xe78bb700,0xe5808f00,0xe78c9700,0xe78c8a00
,0xe78c9c00,0xe78c9600,0xe78c9d00,0xe78cb400,0xe78caf00,0xe78ca900,0xe78ca500,0xe78cbe00
,0xe78d8e00,0xe78d8f00,0xe9bb9800,0xe78d9700,0xe78daa00,0xe78da800,0xe78db000,0xe78db800
,0xe78db500,0xe78dbb00,0xe78dba00,0xe78f8800,0xe78eb300,0xe78f8e00,0xe78ebb00,0xe78f8000
,0xe78fa500,0xe78fae00,0xe78f9e00,0xe792a200,0xe7908500,0xe791af00,0xe790a500,0xe78fb800
,0xe790b200,0xe790ba00,0xe7919500,0xe790bf00,0xe7919f00,0xe7919900,0xe7918100,0xe7919c00
,0xe791a900,0xe791b000,0xe791a300,0xe791aa00,0xe791b600,0xe791be00,0xe7928b00,0xe7929e00
,0xe792a700,0xe7938a00,0xe7938f00,0xe7939400,0xe78fb100,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe793a000,0xe793a300,0xe793a700,0xe793a900,0xe793ae00,0xe793b200,0xe793b000,0xe793b100
,0xe793b800,0xe793b700,0xe7948400,0xe7948300,0xe7948500,0xe7948c00,0xe7948e00,0xe7948d00
,0xe7949500,0xe7949300,0xe7949e00,0xe794a600,0xe794ac00,0xe794bc00,0xe7958400,0xe7958d00
,0xe7958a00,0xe7958900,0xe7959b00,0xe7958600,0xe7959a00,0xe795a900,0xe795a400,0xe795a700
,0xe795ab00,0xe795ad00,0xe795b800,0xe795b600,0xe7968600,0xe7968700,0xe795b400,0xe7968a00
,0xe7968900,0xe7968200,0xe7969400,0xe7969a00,0xe7969d00,0xe796a500,0xe796a300,0xe7978200
,0xe796b300,0xe7978300,0xe796b500,0xe796bd00,0xe796b800,0xe796bc00,0xe796b100,0xe7978d00
,0xe7978a00,0xe7979200,0xe7979900,0xe797a300,0xe7979e00,0xe797be00,0xe797bf00,0
,0xe797bc00,0xe7988100,0xe797b000,0xe797ba00,0xe797b200,0xe797b300,0xe7988b00,0xe7988d00
,0xe7988900,0xe7989f00,0xe798a700,0xe798a000,0xe798a100,0xe798a200,0xe798a400,0xe798b400
,0xe798b000,0xe798bb00,0xe7998700,0xe7998800,0xe7998600,0xe7999c00,0xe7999800,0xe799a100
,0xe799a200,0xe799a800,0xe799a900,0xe799aa00,0xe799a700,0xe799ac00,0xe799b000,0xe799b200
,0xe799b600,0xe799b800,0xe799bc00,0xe79a8000,0xe79a8300,0xe79a8800,0xe79a8b00,0xe79a8e00
,0xe79a9600,0xe79a9300,0xe79a9900,0xe79a9a00,0xe79ab000,0xe79ab400,0xe79ab800,0xe79ab900
,0xe79aba00,0xe79b8200,0xe79b8d00,0xe79b9600,0xe79b9200,0xe79b9e00,0xe79ba100,0xe79ba500
,0xe79ba700,0xe79baa00,0xe898af00,0xe79bbb00,0xe79c8800,0xe79c8700,0xe79c8400,0xe79ca900
,0xe79ca400,0xe79c9e00,0xe79ca500,0xe79ca600,0xe79c9b00,0xe79cb700,0xe79cb800,0xe79d8700
,0xe79d9a00,0xe79da800,0xe79dab00,0xe79d9b00,0xe79da500,0xe79dbf00,0xe79dbe00,0xe79db900
,0xe79e8e00,0xe79e8b00,0xe79e9100,0xe79ea000,0xe79e9e00,0xe79eb000,0xe79eb600,0xe79eb900
,0xe79ebf00,0xe79ebc00,0xe79ebd00,0xe79ebb00,0xe79f8700,0xe79f8d00,0xe79f9700,0xe79f9a00
,0xe79f9c00,0xe79fa300,0xe79fae00,0xe79fbc00,0xe7a08c00,0xe7a09200,0xe7a4a600,0xe7a0a000
,0xe7a4aa00,0xe7a18500,0xe7a28e00,0xe7a1b400,0xe7a28600,0xe7a1bc00,0xe7a29a00,0xe7a28c00
,0xe7a2a300,0xe7a2b500,0xe7a2aa00,0xe7a2af00,0xe7a39100,0xe7a38600,0xe7a38b00,0xe7a39400
,0xe7a2be00,0xe7a2bc00,0xe7a38500,0xe7a38a00,0xe7a3ac00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe7a3a700,0xe7a39a00,0xe7a3bd00,0xe7a3b400,0xe7a48700,0xe7a49200,0xe7a49100,0xe7a49900
,0xe7a4ac00,0xe7a4ab00,0xe7a58000,0xe7a5a000,0xe7a59700,0xe7a59f00,0xe7a59a00,0xe7a59500
,0xe7a59300,0xe7a5ba00,0xe7a5bf00,0xe7a68a00,0xe7a69d00,0xe7a6a700,0xe9bd8b00,0xe7a6aa00
,0xe7a6ae00,0xe7a6b300,0xe7a6b900,0xe7a6ba00,0xe7a78900,0xe7a79500,0xe7a7a700,0xe7a7ac00
,0xe7a7a100,0xe7a7a300,0xe7a88800,0xe7a88d00,0xe7a89800,0xe7a89900,0xe7a8a000,0xe7a89f00
,0xe7a68000,0xe7a8b100,0xe7a8bb00,0xe7a8be00,0xe7a8b700,0xe7a98300,0xe7a99700,0xe7a98900
,0xe7a9a100,0xe7a9a200,0xe7a9a900,0xe9be9d00,0xe7a9b000,0xe7a9b900,0xe7a9bd00,0xe7aa8800
,0xe7aa9700,0xe7aa9500,0xe7aa9800,0xe7aa9600,0xe7aaa900,0xe7ab8800,0xe7aab000,0
,0xe7aab600,0xe7ab8500,0xe7ab8400,0xe7aabf00,0xe9828300,0xe7ab8700,0xe7ab8a00,0xe7ab8d00
,0xe7ab8f00,0xe7ab9500,0xe7ab9300,0xe7ab9900,0xe7ab9a00,0xe7ab9d00,0xe7aba100,0xe7aba200
,0xe7aba600,0xe7abad00,0xe7abb000,0xe7ac8200,0xe7ac8f00,0xe7ac8a00,0xe7ac8600,0xe7acb300
,0xe7ac9800,0xe7ac9900,0xe7ac9e00,0xe7acb500,0xe7aca800,0xe7acb600,0xe7ad9000,0xe7adba00
,0xe7ac8400,0xe7ad8d00,0xe7ac8b00,0xe7ad8c00,0xe7ad8500,0xe7adb500,0xe7ada500,0xe7adb400
,0xe7ada700,0xe7adb000,0xe7adb100,0xe7adac00,0xe7adae00,0xe7ae9d00,0xe7ae9800,0xe7ae9f00
,0xe7ae8d00,0xe7ae9c00,0xe7ae9a00,0xe7ae8b00,0xe7ae9200,0xe7ae8f00,0xe7ad9d00,0xe7ae9900
,0xe7af8b00,0xe7af8100,0xe7af8c00,0xe7af8f00,0xe7aeb400,0xe7af8600,0xe7af9d00,0xe7afa900
,0xe7b09100,0xe7b09400,0xe7afa600,0xe7afa500,0xe7b1a000,0xe7b08000,0xe7b08700,0xe7b09300
,0xe7afb300,0xe7afb700,0xe7b09700,0xe7b08d00,0xe7afb600,0xe7b0a300,0xe7b0a700,0xe7b0aa00
,0xe7b09f00,0xe7b0b700,0xe7b0ab00,0xe7b0bd00,0xe7b18c00,0xe7b18300,0xe7b19400,0xe7b18f00
,0xe7b18000,0xe7b19000,0xe7b19800,0xe7b19f00,0xe7b1a400,0xe7b19600,0xe7b1a500,0xe7b1ac00
,0xe7b1b500,0xe7b28300,0xe7b29000,0xe7b2a400,0xe7b2ad00,0xe7b2a200,0xe7b2ab00,0xe7b2a100
,0xe7b2a800,0xe7b2b300,0xe7b2b200,0xe7b2b100,0xe7b2ae00,0xe7b2b900,0xe7b2bd00,0xe7b38000
,0xe7b38500,0xe7b38200,0xe7b39800,0xe7b39200,0xe7b39c00,0xe7b3a200,0xe9acbb00,0xe7b3af00
,0xe7b3b200,0xe7b3b400,0xe7b3b600,0xe7b3ba00,0xe7b48600,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe7b48200,0xe7b49c00,0xe7b49500,0xe7b48a00,0xe7b58500,0xe7b58b00,0xe7b4ae00,0xe7b4b200
,0xe7b4bf00,0xe7b4b500,0xe7b58600,0xe7b5b300,0xe7b59600,0xe7b58e00,0xe7b5b200,0xe7b5a800
,0xe7b5ae00,0xe7b58f00,0xe7b5a300,0xe7b69300,0xe7b68900,0xe7b59b00,0xe7b68f00,0xe7b5bd00
,0xe7b69b00,0xe7b6ba00,0xe7b6ae00,0xe7b6a300,0xe7b6b500,0xe7b78700,0xe7b6bd00,0xe7b6ab00
,0xe7b8bd00,0xe7b6a200,0xe7b6af00,0xe7b79c00,0xe7b6b800,0xe7b69f00,0xe7b6b000,0xe7b79800
,0xe7b79d00,0xe7b7a400,0xe7b79e00,0xe7b7bb00,0xe7b7b200,0xe7b7a100,0xe7b88500,0xe7b88a00
,0xe7b8a300,0xe7b8a100,0xe7b89200,0xe7b8b100,0xe7b89f00,0xe7b88900,0xe7b88b00,0xe7b8a200
,0xe7b98600,0xe7b9a600,0xe7b8bb00,0xe7b8b500,0xe7b8b900,0xe7b98300,0xe7b8b700,0
,0xe7b8b200,0xe7b8ba00,0xe7b9a700,0xe7b99d00,0xe7b99600,0xe7b99e00,0xe7b99900,0xe7b99a00
,0xe7b9b900,0xe7b9aa00,0xe7b9a900,0xe7b9bc00,0xe7b9bb00,0xe7ba8300,0xe7b79500,0xe7b9bd00
,0xe8beae00,0xe7b9bf00,0xe7ba8800,0xe7ba8900,0xe7ba8c00,0xe7ba9200,0xe7ba9000,0xe7ba9300
,0xe7ba9400,0xe7ba9600,0xe7ba8e00,0xe7ba9b00,0xe7ba9c00,0xe7bcb800,0xe7bcba00,0xe7bd8500
,0xe7bd8c00,0xe7bd8d00,0xe7bd8e00,0xe7bd9000,0xe7bd9100,0xe7bd9500,0xe7bd9400,0xe7bd9800
,0xe7bd9f00,0xe7bda000,0xe7bda800,0xe7bda900,0xe7bda700,0xe7bdb800,0xe7be8200,0xe7be8600
,0xe7be8300,0xe7be8800,0xe7be8700,0xe7be8c00,0xe7be9400,0xe7be9e00,0xe7be9d00,0xe7be9a00
,0xe7bea300,0xe7beaf00,0xe7beb200,0xe7beb900,0xe7beae00,0xe7beb600,0xe7beb800,0xe8adb100
,0xe7bf8500,0xe7bf8600,0xe7bf8a00,0xe7bf9500,0xe7bf9400,0xe7bfa100,0xe7bfa600,0xe7bfa900
,0xe7bfb300,0xe7bfb900,0xe9a39c00,0xe8808600,0xe8808400,0xe8808b00,0xe8809200,0xe8809800
,0xe8809900,0xe8809c00,0xe880a100,0xe880a800,0xe880bf00,0xe880bb00,0xe8818a00,0xe8818600
,0xe8819200,0xe8819800,0xe8819a00,0xe8819f00,0xe881a200,0xe881a800,0xe881b300,0xe881b200
,0xe881b000,0xe881b600,0xe881b900,0xe881bd00,0xe881bf00,0xe8828400,0xe8828600,0xe8828500
,0xe8829b00,0xe8829300,0xe8829a00,0xe882ad00,0xe5869000,0xe882ac00,0xe8839b00,0xe883a500
,0xe8839900,0xe8839d00,0xe8838400,0xe8839a00,0xe8839600,0xe8848900,0xe883af00,0xe883b100
,0xe8849b00,0xe884a900,0xe884a300,0xe884af00,0xe8858b00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe99a8b00,0xe8858600,0xe884be00,0xe8859300,0xe8859100,0xe883bc00,0xe885b100,0xe885ae00
,0xe885a500,0xe885a600,0xe885b400,0xe8868300,0xe8868800,0xe8868a00,0xe8868000,0xe8868200
,0xe886a000,0xe8869500,0xe886a400,0xe886a300,0xe8859f00,0xe8869300,0xe886a900,0xe886b000
,0xe886b500,0xe886be00,0xe886b800,0xe886bd00,0xe8878000,0xe8878200,0xe886ba00,0xe8878900
,0xe8878d00,0xe8879100,0xe8879900,0xe8879800,0xe8878800,0xe8879a00,0xe8879f00,0xe887a000
,0xe887a700,0xe887ba00,0xe887bb00,0xe887be00,0xe8888100,0xe8888200,0xe8888500,0xe8888700
,0xe8888a00,0xe8888d00,0xe8889000,0xe8889600,0xe888a900,0xe888ab00,0xe888b800,0xe888b300
,0xe8898000,0xe8899900,0xe8899800,0xe8899d00,0xe8899a00,0xe8899f00,0xe889a400,0
,0xe889a200,0xe889a800,0xe889aa00,0xe889ab00,0xe888ae00,0xe889b100,0xe889b700,0xe889b800
,0xe889be00,0xe88a8d00,0xe88a9200,0xe88aab00,0xe88a9f00,0xe88abb00,0xe88aac00,0xe88ba100
,0xe88ba300,0xe88b9f00,0xe88b9200,0xe88bb400,0xe88bb300,0xe88bba00,0xe88e9300,0xe88c8300
,0xe88bbb00,0xe88bb900,0xe88b9e00,0xe88c8600,0xe88b9c00,0xe88c8900,0xe88b9900,0xe88cb500
,0xe88cb400,0xe88c9600,0xe88cb200,0xe88cb100,0xe88d8000,0xe88cb900,0xe88d9000,0xe88d8500
,0xe88caf00,0xe88cab00,0xe88c9700,0xe88c9800,0xe88e8500,0xe88e9a00,0xe88eaa00,0xe88e9f00
,0xe88ea200,0xe88e9600,0xe88ca300,0xe88e8e00,0xe88e8700,0xe88e8a00,0xe88dbc00,0xe88eb500
,0xe88db300,0xe88db500,0xe88ea000,0xe88e8900,0xe88ea800,0xe88fb400,0xe8909300,0xe88fab00
,0xe88f8e00,0xe88fbd00,0xe8908300,0xe88f9800,0xe8908b00,0xe88f8100,0xe88fb700,0xe8908700
,0xe88fa000,0xe88fb200,0xe8908d00,0xe890a200,0xe890a000,0xe88ebd00,0xe890b800,0xe8948600
,0xe88fbb00,0xe891ad00,0xe890aa00,0xe890bc00,0xe8959a00,0xe8928400,0xe891b700,0xe891ab00
,0xe892ad00,0xe891ae00,0xe8928200,0xe891a900,0xe8918600,0xe890ac00,0xe891af00,0xe891b900
,0xe890b500,0xe8938a00,0xe891a200,0xe892b900,0xe892bf00,0xe8929f00,0xe8939900,0xe8938d00
,0xe892bb00,0xe8939a00,0xe8939000,0xe8938100,0xe8938600,0xe8939600,0xe892a100,0xe894a100
,0xe893bf00,0xe893b400,0xe8949700,0xe8949800,0xe894ac00,0xe8949f00,0xe8949500,0xe8949400
,0xe893bc00,0xe8958000,0xe895a300,0xe8959800,0xe8958800,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe8958100,0xe8988200,0xe8958b00,0xe8959500,0xe8968000,0xe896a400,0xe8968800,0xe8969100
,0xe8968a00,0xe896a800,0xe895ad00,0xe8969400,0xe8969b00,0xe897aa00,0xe8968700,0xe8969c00
,0xe895b700,0xe895be00,0xe8969000,0xe8978900,0xe896ba00,0xe8978f00,0xe896b900,0xe8979000
,0xe8979500,0xe8979d00,0xe897a500,0xe8979c00,0xe897b900,0xe8988a00,0xe8989300,0xe8988b00
,0xe897be00,0xe897ba00,0xe8988600,0xe898a200,0xe8989a00,0xe898b000,0xe898bf00,0xe8998d00
,0xe4b99500,0xe8999400,0xe8999f00,0xe899a700,0xe899b100,0xe89a9300,0xe89aa300,0xe89aa900
,0xe89aaa00,0xe89a8b00,0xe89a8c00,0xe89ab600,0xe89aaf00,0xe89b8400,0xe89b8600,0xe89ab000
,0xe89b8900,0xe8a0a300,0xe89aab00,0xe89b9400,0xe89b9e00,0xe89ba900,0xe89bac00,0
,0xe89b9f00,0xe89b9b00,0xe89baf00,0xe89c9200,0xe89c8600,0xe89c8800,0xe89c8000,0xe89c8300
,0xe89bbb00,0xe89c9100,0xe89c8900,0xe89c8d00,0xe89bb900,0xe89c8a00,0xe89cb400,0xe89cbf00
,0xe89cb700,0xe89cbb00,0xe89ca500,0xe89ca900,0xe89c9a00,0xe89da000,0xe89d9f00,0xe89db800
,0xe89d8c00,0xe89d8e00,0xe89db400,0xe89d9700,0xe89da800,0xe89dae00,0xe89d9900,0xe89d9300
,0xe89da300,0xe89daa00,0xe8a08500,0xe89ea200,0xe89e9f00,0xe89e8200,0xe89eaf00,0xe89f8b00
,0xe89ebd00,0xe89f8000,0xe89f9000,0xe99b9600,0xe89eab00,0xe89f8400,0xe89eb300,0xe89f8700
,0xe89f8600,0xe89ebb00,0xe89faf00,0xe89fb200,0xe89fa000,0xe8a08f00,0xe8a08d00,0xe89fbe00
,0xe89fb600,0xe89fb700,0xe8a08e00,0xe89f9200,0xe8a09100,0xe8a09600,0xe8a09500,0xe8a0a200
,0xe8a0a100,0xe8a0b100,0xe8a0b600,0xe8a0b900,0xe8a0a700,0xe8a0bb00,0xe8a18400,0xe8a18200
,0xe8a19200,0xe8a19900,0xe8a19e00,0xe8a1a200,0xe8a1ab00,0xe8a28100,0xe8a1be00,0xe8a29e00
,0xe8a1b500,0xe8a1bd00,0xe8a2b500,0xe8a1b200,0xe8a28200,0xe8a29700,0xe8a29200,0xe8a2ae00
,0xe8a29900,0xe8a2a200,0xe8a28d00,0xe8a2a400,0xe8a2b000,0xe8a2bf00,0xe8a2b100,0xe8a38300
,0xe8a38400,0xe8a39400,0xe8a39800,0xe8a39900,0xe8a39d00,0xe8a3b900,0xe8a48200,0xe8a3bc00
,0xe8a3b400,0xe8a3a800,0xe8a3b200,0xe8a48400,0xe8a48c00,0xe8a48a00,0xe8a49300,0xe8a58300
,0xe8a49e00,0xe8a4a500,0xe8a4aa00,0xe8a4ab00,0xe8a58100,0xe8a58400,0xe8a4bb00,0xe8a4b600
,0xe8a4b800,0xe8a58c00,0xe8a49d00,0xe8a5a000,0xe8a59e00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe8a5a600,0xe8a5a400,0xe8a5ad00,0xe8a5aa00,0xe8a5af00,0xe8a5b400,0xe8a5b700,0xe8a5be00
,0xe8a68300,0xe8a68800,0xe8a68a00,0xe8a69300,0xe8a69800,0xe8a6a100,0xe8a6a900,0xe8a6a600
,0xe8a6ac00,0xe8a6af00,0xe8a6b200,0xe8a6ba00,0xe8a6bd00,0xe8a6bf00,0xe8a78000,0xe8a79a00
,0xe8a79c00,0xe8a79d00,0xe8a7a700,0xe8a7b400,0xe8a7b800,0xe8a88300,0xe8a89600,0xe8a89000
,0xe8a88c00,0xe8a89b00,0xe8a89d00,0xe8a8a500,0xe8a8b600,0xe8a98100,0xe8a99b00,0xe8a99200
,0xe8a98600,0xe8a98800,0xe8a9bc00,0xe8a9ad00,0xe8a9ac00,0xe8a9a200,0xe8aa8500,0xe8aa8200
,0xe8aa8400,0xe8aaa800,0xe8aaa100,0xe8aa9100,0xe8aaa500,0xe8aaa600,0xe8aa9a00,0xe8aaa300
,0xe8ab8400,0xe8ab8d00,0xe8ab8200,0xe8ab9a00,0xe8abab00,0xe8abb300,0xe8aba700,0
,0xe8aba400,0xe8abb100,0xe8ac9400,0xe8aba000,0xe8aba200,0xe8abb700,0xe8ab9e00,0xe8ab9b00
,0xe8ac8c00,0xe8ac8700,0xe8ac9a00,0xe8aba100,0xe8ac9600,0xe8ac9000,0xe8ac9700,0xe8aca000
,0xe8acb300,0xe99eab00,0xe8aca600,0xe8acab00,0xe8acbe00,0xe8aca800,0xe8ad8100,0xe8ad8c00
,0xe8ad8f00,0xe8ad8e00,0xe8ad8900,0xe8ad9600,0xe8ad9b00,0xe8ad9a00,0xe8adab00,0xe8ad9f00
,0xe8adac00,0xe8adaf00,0xe8adb400,0xe8adbd00,0xe8ae8000,0xe8ae8c00,0xe8ae8e00,0xe8ae9200
,0xe8ae9300,0xe8ae9600,0xe8ae9900,0xe8ae9a00,0xe8b0ba00,0xe8b18100,0xe8b0bf00,0xe8b18800
,0xe8b18c00,0xe8b18e00,0xe8b19000,0xe8b19500,0xe8b1a200,0xe8b1ac00,0xe8b1b800,0xe8b1ba00
,0xe8b28200,0xe8b28900,0xe8b28500,0xe8b28a00,0xe8b28d00,0xe8b28e00,0xe8b29400,0xe8b1bc00
,0xe8b29800,0xe6889d00,0xe8b2ad00,0xe8b2aa00,0xe8b2bd00,0xe8b2b200,0xe8b2b300,0xe8b2ae00
,0xe8b2b600,0xe8b38800,0xe8b38100,0xe8b3a400,0xe8b3a300,0xe8b39a00,0xe8b3bd00,0xe8b3ba00
,0xe8b3bb00,0xe8b48400,0xe8b48500,0xe8b48a00,0xe8b48700,0xe8b48f00,0xe8b48d00,0xe8b49000
,0xe9bd8e00,0xe8b49300,0xe8b38d00,0xe8b49400,0xe8b49600,0xe8b5a700,0xe8b5ad00,0xe8b5b100
,0xe8b5b300,0xe8b68100,0xe8b69900,0xe8b78200,0xe8b6be00,0xe8b6ba00,0xe8b78f00,0xe8b79a00
,0xe8b79600,0xe8b78c00,0xe8b79b00,0xe8b78b00,0xe8b7aa00,0xe8b7ab00,0xe8b79f00,0xe8b7a300
,0xe8b7bc00,0xe8b88800,0xe8b88900,0xe8b7bf00,0xe8b89d00,0xe8b89e00,0xe8b89000,0xe8b89f00
,0xe8b98200,0xe8b8b500,0xe8b8b000,0xe8b8b400,0xe8b98a00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe8b98700,0xe8b98900,0xe8b98c00,0xe8b99000,0xe8b98800,0xe8b99900,0xe8b9a400,0xe8b9a000
,0xe8b8aa00,0xe8b9a300,0xe8b99500,0xe8b9b600,0xe8b9b200,0xe8b9bc00,0xe8ba8100,0xe8ba8700
,0xe8ba8500,0xe8ba8400,0xe8ba8b00,0xe8ba8a00,0xe8ba9300,0xe8ba9100,0xe8ba9400,0xe8ba9900
,0xe8baaa00,0xe8baa100,0xe8baac00,0xe8bab000,0xe8bb8600,0xe8bab100,0xe8babe00,0xe8bb8500
,0xe8bb8800,0xe8bb8b00,0xe8bb9b00,0xe8bba300,0xe8bbbc00,0xe8bbbb00,0xe8bbab00,0xe8bbbe00
,0xe8bc8a00,0xe8bc8500,0xe8bc9500,0xe8bc9200,0xe8bc9900,0xe8bc9300,0xe8bc9c00,0xe8bc9f00
,0xe8bc9b00,0xe8bc8c00,0xe8bca600,0xe8bcb300,0xe8bcbb00,0xe8bcb900,0xe8bd8500,0xe8bd8200
,0xe8bcbe00,0xe8bd8c00,0xe8bd8900,0xe8bd8600,0xe8bd8e00,0xe8bd9700,0xe8bd9c00,0
,0xe8bda200,0xe8bda300,0xe8bda400,0xe8be9c00,0xe8be9f00,0xe8bea300,0xe8bead00,0xe8beaf00
,0xe8beb700,0xe8bf9a00,0xe8bfa500,0xe8bfa200,0xe8bfaa00,0xe8bfaf00,0xe9828700,0xe8bfb400
,0xe9808500,0xe8bfb900,0xe8bfba00,0xe9809100,0xe9809500,0xe980a100,0xe9808d00,0xe9809e00
,0xe9809600,0xe9808b00,0xe980a700,0xe980b600,0xe980b500,0xe980b900,0xe8bfb800,0xe9818f00
,0xe9819000,0xe9819100,0xe9819200,0xe9808e00,0xe9818900,0xe980be00,0xe9819600,0xe9819800
,0xe9819e00,0xe981a800,0xe981af00,0xe981b600,0xe99aa800,0xe981b200,0xe9828200,0xe981bd00
,0xe9828100,0xe9828000,0xe9828a00,0xe9828900,0xe9828f00,0xe982a800,0xe982af00,0xe982b100
,0xe982b500,0xe983a200,0xe983a400,0xe6898800,0xe9839b00,0xe9848200,0xe9849200,0xe9849900
,0xe984b200,0xe984b000,0xe9858a00,0xe9859600,0xe9859800,0xe985a300,0xe985a500,0xe985a900
,0xe985b300,0xe985b200,0xe9868b00,0xe9868900,0xe9868200,0xe986a200,0xe986ab00,0xe986af00
,0xe986aa00,0xe986b500,0xe986b400,0xe986ba00,0xe9878000,0xe9878100,0xe9878900,0xe9878b00
,0xe9879000,0xe9879600,0xe9879f00,0xe987a100,0xe9879b00,0xe987bc00,0xe987b500,0xe987b600
,0xe9889e00,0xe987bf00,0xe9889400,0xe988ac00,0xe9889500,0xe9889100,0xe9899e00,0xe9899700
,0xe9898500,0xe9898900,0xe989a400,0xe9898800,0xe98a9500,0xe988bf00,0xe9898b00,0xe9899000
,0xe98a9c00,0xe98a9600,0xe98a9300,0xe98a9b00,0xe9899a00,0xe98b8f00,0xe98ab900,0xe98ab700
,0xe98ba900,0xe98c8f00,0xe98bba00,0xe98d8400,0xe98cae00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe98c9900,0xe98ca200,0xe98c9a00,0xe98ca300,0xe98cba00,0xe98cb500,0xe98cbb00,0xe98d9c00
,0xe98da000,0xe98dbc00,0xe98dae00,0xe98d9600,0xe98eb000,0xe98eac00,0xe98ead00,0xe98e9400
,0xe98eb900,0xe98f9600,0xe98f9700,0xe98fa800,0xe98fa500,0xe98f9800,0xe98f8300,0xe98f9d00
,0xe98f9000,0xe98f8800,0xe98fa400,0xe9909a00,0xe9909400,0xe9909300,0xe9908300,0xe9908700
,0xe9909000,0xe990b600,0xe990ab00,0xe990b500,0xe990a100,0xe990ba00,0xe9918100,0xe9919200
,0xe9918400,0xe9919b00,0xe991a000,0xe991a200,0xe9919e00,0xe991aa00,0xe988a900,0xe991b000
,0xe991b500,0xe991b700,0xe991bd00,0xe9919a00,0xe991bc00,0xe991be00,0xe9928100,0xe991bf00
,0xe9968200,0xe9968700,0xe9968a00,0xe9969400,0xe9969600,0xe9969800,0xe9969900,0
,0xe996a000,0xe996a800,0xe996a700,0xe996ad00,0xe996bc00,0xe996bb00,0xe996b900,0xe996be00
,0xe9978a00,0xe6bfb600,0xe9978300,0xe9978d00,0xe9978c00,0xe9979500,0xe9979400,0xe9979600
,0xe9979c00,0xe997a100,0xe997a500,0xe997a200,0xe998a100,0xe998a800,0xe998ae00,0xe998af00
,0xe9998200,0xe9998c00,0xe9998f00,0xe9998b00,0xe999b700,0xe9999c00,0xe9999e00,0xe9999d00
,0xe9999f00,0xe999a600,0xe999b200,0xe999ac00,0xe99a8d00,0xe99a9800,0xe99a9500,0xe99a9700
,0xe99aaa00,0xe99aa700,0xe99ab100,0xe99ab200,0xe99ab000,0xe99ab400,0xe99ab600,0xe99ab800
,0xe99ab900,0xe99b8e00,0xe99b8b00,0xe99b8900,0xe99b8d00,0xe8a58d00,0xe99b9c00,0xe99c8d00
,0xe99b9500,0xe99bb900,0xe99c8400,0xe99c8600,0xe99c8800,0xe99c9300,0xe99c8e00,0xe99c9100
,0xe99c8f00,0xe99c9600,0xe99c9900,0xe99ca400,0xe99caa00,0xe99cb000,0xe99cb900,0xe99cbd00
,0xe99cbe00,0xe99d8400,0xe99d8600,0xe99d8800,0xe99d8200,0xe99d8900,0xe99d9c00,0xe99da000
,0xe99da400,0xe99da600,0xe99da800,0xe58b9200,0xe99dab00,0xe99db100,0xe99db900,0xe99e8500
,0xe99dbc00,0xe99e8100,0xe99dba00,0xe99e8600,0xe99e8b00,0xe99e8f00,0xe99e9000,0xe99e9c00
,0xe99ea800,0xe99ea600,0xe99ea300,0xe99eb300,0xe99eb400,0xe99f8300,0xe99f8600,0xe99f8800
,0xe99f8b00,0xe99f9c00,0xe99fad00,0xe9bd8f00,0xe99fb200,0xe7ab9f00,0xe99fb600,0xe99fb500
,0xe9a08f00,0xe9a08c00,0xe9a0b800,0xe9a0a400,0xe9a0a100,0xe9a0b700,0xe9a0bd00,0xe9a18600
,0xe9a18f00,0xe9a18b00,0xe9a1ab00,0xe9a1af00,0xe9a1b000,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe9a1b100,0xe9a1b400,0xe9a1b300,0xe9a2aa00,0xe9a2af00,0xe9a2b100,0xe9a2b600,0xe9a38400
,0xe9a38300,0xe9a38600,0xe9a3a900,0xe9a3ab00,0xe9a48300,0xe9a48900,0xe9a49200,0xe9a49400
,0xe9a49800,0xe9a4a100,0xe9a49d00,0xe9a49e00,0xe9a4a400,0xe9a4a000,0xe9a4ac00,0xe9a4ae00
,0xe9a4bd00,0xe9a4be00,0xe9a58200,0xe9a58900,0xe9a58500,0xe9a59000,0xe9a58b00,0xe9a59100
,0xe9a59200,0xe9a58c00,0xe9a59500,0xe9a69700,0xe9a69800,0xe9a6a500,0xe9a6ad00,0xe9a6ae00
,0xe9a6bc00,0xe9a79f00,0xe9a79b00,0xe9a79d00,0xe9a79800,0xe9a79100,0xe9a7ad00,0xe9a7ae00
,0xe9a7b100,0xe9a7b200,0xe9a7bb00,0xe9a7b800,0xe9a88100,0xe9a88f00,0xe9a88500,0xe9a7a200
,0xe9a89900,0xe9a8ab00,0xe9a8b700,0xe9a98500,0xe9a98200,0xe9a98000,0xe9a98300,0
,0xe9a8be00,0xe9a99500,0xe9a98d00,0xe9a99b00,0xe9a99700,0xe9a99f00,0xe9a9a200,0xe9a9a500
,0xe9a9a400,0xe9a9a900,0xe9a9ab00,0xe9a9aa00,0xe9aaad00,0xe9aab000,0xe9aabc00,0xe9ab8000
,0xe9ab8f00,0xe9ab9100,0xe9ab9300,0xe9ab9400,0xe9ab9e00,0xe9ab9f00,0xe9aba200,0xe9aba300
,0xe9aba600,0xe9abaf00,0xe9abab00,0xe9abae00,0xe9abb400,0xe9abb100,0xe9abb700,0xe9abbb00
,0xe9ac8600,0xe9ac9800,0xe9ac9a00,0xe9ac9f00,0xe9aca200,0xe9aca300,0xe9aca500,0xe9aca700
,0xe9aca800,0xe9aca900,0xe9acaa00,0xe9acae00,0xe9acaf00,0xe9acb200,0xe9ad8400,0xe9ad8300
,0xe9ad8f00,0xe9ad8d00,0xe9ad8e00,0xe9ad9100,0xe9ad9800,0xe9adb400,0xe9ae9300,0xe9ae8300
,0xe9ae9100,0xe9ae9600,0xe9ae9700,0xe9ae9f00,0xe9aea000,0xe9aea800,0xe9aeb400,0xe9af8000
,0xe9af8a00,0xe9aeb900,0xe9af8600,0xe9af8f00,0xe9af9100,0xe9af9200,0xe9afa300,0xe9afa200
,0xe9afa400,0xe9af9400,0xe9afa100,0xe9b0ba00,0xe9afb200,0xe9afb100,0xe9afb000,0xe9b09500
,0xe9b09400,0xe9b08900,0xe9b09300,0xe9b08c00,0xe9b08600,0xe9b08800,0xe9b09200,0xe9b08a00
,0xe9b08400,0xe9b0ae00,0xe9b09b00,0xe9b0a500,0xe9b0a400,0xe9b0a100,0xe9b0b000,0xe9b18700
,0xe9b0b200,0xe9b18600,0xe9b0be00,0xe9b19a00,0xe9b1a000,0xe9b1a700,0xe9b1b600,0xe9b1b800
,0xe9b3a700,0xe9b3ac00,0xe9b3b000,0xe9b48900,0xe9b48800,0xe9b3ab00,0xe9b48300,0xe9b48600
,0xe9b4aa00,0xe9b4a600,0xe9b6af00,0xe9b4a300,0xe9b49f00,0xe9b58400,0xe9b49500,0xe9b49200
,0xe9b58100,0xe9b4bf00,0xe9b4be00,0xe9b58600,0xe9b58800,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe9b59d00,0xe9b59e00,0xe9b5a400,0xe9b59100,0xe9b59000,0xe9b59900,0xe9b5b200,0xe9b68900
,0xe9b68700,0xe9b6ab00,0xe9b5af00,0xe9b5ba00,0xe9b69a00,0xe9b6a400,0xe9b6a900,0xe9b6b200
,0xe9b78400,0xe9b78100,0xe9b6bb00,0xe9b6b800,0xe9b6ba00,0xe9b78600,0xe9b78f00,0xe9b78200
,0xe9b79900,0xe9b79300,0xe9b7b800,0xe9b7a600,0xe9b7ad00,0xe9b7af00,0xe9b7bd00,0xe9b89a00
,0xe9b89b00,0xe9b89e00,0xe9b9b500,0xe9b9b900,0xe9b9bd00,0xe9ba8100,0xe9ba8800,0xe9ba8b00
,0xe9ba8c00,0xe9ba9200,0xe9ba9500,0xe9ba9100,0xe9ba9d00,0xe9baa500,0xe9baa900,0xe9bab800
,0xe9baaa00,0xe9baad00,0xe99da100,0xe9bb8c00,0xe9bb8e00,0xe9bb8f00,0xe9bb9000,0xe9bb9400
,0xe9bb9c00,0xe9bb9e00,0xe9bb9d00,0xe9bba000,0xe9bba500,0xe9bba800,0xe9bbaf00,0
,0xe9bbb400,0xe9bbb600,0xe9bbb700,0xe9bbb900,0xe9bbbb00,0xe9bbbc00,0xe9bbbd00,0xe9bc8700
,0xe9bc8800,0xe79ab700,0xe9bc9500,0xe9bca100,0xe9bcac00,0xe9bcbe00,0xe9bd8a00,0xe9bd9200
,0xe9bd9400,0xe9bda300,0xe9bd9f00,0xe9bda000,0xe9bda100,0xe9bda600,0xe9bda700,0xe9bdac00
,0xe9bdaa00,0xe9bdb700,0xe9bdb200,0xe9bdb600,0xe9be9500,0xe9be9c00,0xe9bea000,0xe5a0af00
,0xe6a78700,0xe9819900,0xe791a400,0xe5879c00,0xe7869900,0xe5999300,0xe5b7a200,0xe5b89400
,0xe5b89800,0xe5b99800,0xe5b99e00,0xe5babe00,0xefa4a800,0xe5bb8b00,0xe5bbb900,0xe5bc8000
,0xe5bc8200,0xe5bc8700,0xe5bc9d00,0xe5bca300,0xe5bcb400,0xe5bcb600,0xe5bcbd00,0xe5bd8000
,0xe5bd8500,0xe5bd9400,0xe5bd9800,0xe5bda400,0xe5bda700,0xe5bdbd00,0xe5be8900,0xe5be9c00
,0xe5bea700,0xe5beaf00,0xe5beb500,0xe5beb700,0xe5bf8900,0xe5bf9e00,0xe5bfa100,0xe5bfa900
,0xe6808d00,0xe6809400,0xe6809800,0xe680b300,0xe680b500,0xe6818700,0xefa8bd00,0xe6829d00
,0xe6829e00,0xe6838b00,0xe6839400,0xe6839500,0xe6839d00,0xe683b800,0xe6849c00,0xe684ab00
,0xe684b000,0xe684b700,0xefa8be00,0xe6868d00,0xefa8bf00,0xe686bc00,0xe686b900,0xefa98000
,0xe688a200,0xe688be00,0xe6898300,0xe6899600,0xe6899a00,0xe689af00,0xe68a8500,0xe68b8400
,0xe68b9600,0xe68bbc00,0xe68c8a00,0xe68c9800,0xe68cb900,0xe68d8300,0xe68da500,0xe68dbc00
,0xe68fa500,0xe68fad00,0xe68fb500,0xe6909000,0xe6909400,0xe690a200,0xe691b900,0xe6919100
,0xe691a000,0xe691ad00,0xe6938e00,0xe692be00,0xe692bf00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe6938400,0xe6938a00,0xe6939000,0xe693b700,0xe693bb00,0xe694a200,0xe694a900,0xefa98100
,0xe695a700,0xe6969d00,0xefa98200,0xe6988000,0xe6988900,0xe6989500,0xe6989e00,0xe698ba00
,0xe698a200,0xe698a400,0xe698ab00,0xe698b000,0xe698b100,0xe698b300,0xe69bbb00,0xe6998800
,0xe6998c00,0xeaabb100,0xe6999900,0xe6999a00,0xe699a100,0xe699a500,0xe699b300,0xe699b700
,0xe699b800,0xe69a8d00,0xefa98300,0xe69aa000,0xe69ab200,0xe69abb00,0xe69b8600,0xe69b8800
,0xe3aca200,0xe69b9b00,0xe69ba800,0xe69bba00,0xe69c9300,0xefa4a900,0xe69cb300,0xe69da600
,0xe69d8700,0xe69d8800,0xe69dbb00,0xe69e8100,0xe69e9300,0xe69e9800,0xe69e9b00,0xe69ebb00
,0xe69fb900,0xe69f8000,0xe69f9700,0xe69fbc00,0xe6a08100,0xe6a19200,0xe6a09d00,0
,0xe6a0ac00,0xe6a0b100,0xe6a19b00,0xe6a1b200,0xe6a1b500,0xefa98400,0xe6a2a300,0xe6a2a500
,0xe6a2b200,0xe6a38800,0xe6a39000,0xe6a3a800,0xe6a3ad00,0xe6a3b000,0xe6a3b100,0xe6a3bc00
,0xe6a48a00,0xe6a58900,0xeaac8300,0xe6a4b500,0xe6a58200,0xe6a59700,0xe6a5a300,0xe6a5a400
,0xe6a5a800,0xe6a68000,0xefa89400,0xe6a6a500,0xe6a6ad00,0xe6a78f00,0xe3aeb600,0xe3af8300
,0xe6a7a200,0xe6a7a900,0xefa98500,0xe6a7b500,0xe6a7b600,0xe6a88f00,0xe6a89500,0xeaac8900
,0xe6a8bb00,0xe6a8be00,0xe6a98500,0xe6a99000,0xe6a99600,0xe6a99b00,0xe6a9ab00,0xe6a9b300
,0xeaac8b00,0xe6aa8900,0xe6aa9400,0xe6aa9d00,0xe6aa9e00,0xe6aaa500,0xe6aba400,0xe6aba700
,0xe3b08f00,0xefa49d00,0xe6ac9b00,0xe6ac9e00,0xe6acac00,0xe6acb500,0xe6ad8600,0xe6ad9600
,0xe6ada000,0xe6ada500,0xe6ada700,0xe6adb700,0xe6ae8200,0xe6aea900,0xe6aead00,0xefa5b000
,0xe6af8f00,0xe6af9600,0xe6af9700,0xe6afbf00,0xe6b08500,0xe6b09000,0xe6b0b300,0xe6b19900
,0xe6b19c00,0xe6b2aa00,0xe6b1b400,0xe6b1b600,0xe6b28500,0xe6b28600,0xe6b29800,0xe6b29c00
,0xe6b3bb00,0xe6b38600,0xe6b39400,0xe6b3a000,0xe6b3ab00,0xe6b3ae00,0xeaac9400,0xe6b48400
,0xe6b48e00,0xe6b4ae00,0xe6b4b100,0xe6b4b900,0xe6b4bf00,0xe6b59800,0xe6b5a500,0xefa98600
,0xe6b68200,0xe6b68700,0xe6b68900,0xe6b69400,0xe6b6aa00,0xe6b6ac00,0xe6b6bf00,0xe6b78400
,0xe6b79600,0xe6b79a00,0xe6b79b00,0xe6b79d00,0xe6b7bc00,0xefa98700,0xe6b8b400,0xe6b98400
,0xe6b99c00,0xe6b99e00,0xe6baab00,0xe6bab100,0xe6bb8100,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe6bb8700,0xe6bb8e00,0xe6bc9000,0xe6bc9a00,0xefa98800,0xe6bcaa00,0xe6bcaf00,0xe6bcb300
,0xe6bd9100,0xe6bd9900,0xe6bd9e00,0xe6bda100,0xe6bda200,0xe6bdbe00,0xe6be8800,0xe6be8c00
,0xe6be8d00,0xe6be9400,0xe6bea000,0xe6bea700,0xe6beb600,0xe6bebc00,0xe6bf8700,0xe6bf8a00
,0xe6bfb900,0xe6bfb000,0xe6bfb500,0xe7808500,0xe7808600,0xe780a800,0xe7818a00,0xe7819d00
,0xe7819e00,0xe7818e00,0xe781a400,0xe781b500,0xe7828500,0xe782a400,0xe782ab00,0xe782b700
,0xe7839400,0xe7839800,0xe783a400,0xe7848f00,0xe784ab00,0xe7849e00,0xe784a000,0xe784ae00
,0xe784b000,0xe7858600,0xe7858700,0xe7859100,0xefa98900,0xe7859200,0xe7859c00,0xe785a000
,0xe785a800,0xefa89500,0xe7868500,0xe7868700,0xe7869200,0xe7878100,0xe786ba00,0
,0xe7878400,0xe787be00,0xe7888000,0xe7889500,0xe7899500,0xe7899600,0xe3b8bf00,0xe78a8d00
,0xe78a9b00,0xe78abe00,0xe78b8000,0xe78bbb00,0xeaaca200,0xe78ca700,0xe78ca800,0xefa89600
,0xe78d9000,0xe78da600,0xe78dbc00,0xe78e9500,0xe78e9f00,0xe78ea000,0xe78ea200,0xe78ea600
,0xe78eab00,0xe78f8900,0xe78f8f00,0xe78f9600,0xe78f9900,0xe78fa300,0xe78fa900,0xe7908700
,0xe7908a00,0xe7909a00,0xe7909b00,0xefa98b00,0xe790a600,0xe790a800,0xe790aa00,0xe790ab00
,0xe790ac00,0xe790ae00,0xe790af00,0xe790b000,0xe7918400,0xe7918600,0xe7918700,0xe7918b00
,0xe7919700,0xe791a200,0xe791ab00,0xe791ad00,0xe7928600,0xe7928700,0xe7928900,0xe7929800
,0xe7929c00,0xe7929f00,0xe792a300,0xe7929000,0xe792a600,0xe792a800,0xe792a900,0xe792b500
,0xe792bf00,0xe7938800,0xe7938900,0xe7939a00,0xe793bf00,0xe7948100,0xe7949700,0xe794af00
,0xe795af00,0xe795b900,0xe7969200,0xe3bdb200,0xe7978e00,0xe797a400,0xe7988000,0xe7988200
,0xe7988800,0xe7989500,0xe7989600,0xe7989900,0xe7989e00,0xe798ad00,0xe798b500,0xe7998300
,0xe7998b00,0xe799a400,0xe799a500,0xe799ad00,0xe799af00,0xe799b100,0xe79a8100,0xe79a9b00
,0xe79a9d00,0xe79a9e00,0xe79aa600,0xe79aaa00,0xe79ab600,0xe79b8500,0xe79b8c00,0xe79b8e00
,0xe79b9400,0xe79ba600,0xe79bb100,0xe79bbc00,0xe79c8a00,0xe79c9900,0xe79cb400,0xe79cb600
,0xe79d8600,0xe79d8d00,0xe79d8e00,0xe79d9c00,0xe79d9f00,0xe79da200,0xe79dba00,0xe79e8000
,0xe79e9400,0xe79eaa00,0xe79fa000,0xe7a0ad00,0xeaacb600,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe7a18300,0xe7a18e00,0xe7a18f00,0xe7a19100,0xe7a1a800,0xe7a1ae00,0xefa98c00,0xe7a2b000
,0xeaacb800,0xe7a2ad00,0xe7a3a400,0xe7a3b200,0xe7a48000,0xe7a3b700,0xe7a49c00,0xe7a4ae00
,0xe7a4b100,0xe7a4b400,0xefa98d00,0xefa98e00,0xe7a58500,0xe7a58600,0xefa98f00,0xefa99000
,0xefa99100,0xe7a59c00,0xefa99200,0xefa89900,0xefa89a00,0xe7a5b900,0xefa99300,0xefa99400
,0xefa89b00,0xe7a69800,0xe7a6b100,0xe7a6b800,0xe7a78800,0xe7a78a00,0xeaacba00,0xe7a79400
,0xe7a79e00,0xe7a7ab00,0xe7a7ad00,0xe7a88300,0xefa99500,0xe7a8b900,0xe7a99d00,0xe7a9ad00
,0xefa99600,0xe7aa8500,0xe7aaa000,0xeaacbd00,0xe7aab300,0xe7aabb00,0xe7ab8e00,0xe7abab00
,0xe7abbd00,0xe7ac9200,0xe7acad00,0xe7acbb00,0xe7ad8700,0xe7ad8e00,0xe7ada000,0
,0xe7adad00,0xe7adaf00,0xe7adb200,0xe7ae9e00,0xefa99700,0xe7af9700,0xe7af9900,0xe7b08100
,0xe7b0b100,0xe7b09e00,0xe7b0a000,0xe7b0b300,0xe7b0b600,0xe489a400,0xeaad8500,0xe7b19900
,0xe7b1ad00,0xe7b1b900,0xe7b28f00,0xe7b29400,0xe7b2a000,0xe7b2bc00,0xe7b39500,0xe7b39900
,0xe7b39d00,0xe7b48700,0xe7b48800,0xe7b49300,0xe7b49d00,0xe7b4a300,0xe7b4b100,0xe7b58100
,0xe7b58800,0xe7b59300,0xe7b59c00,0xe7b5ba00,0xe7b68300,0xe7b68b00,0xe7b6a000,0xe7b6a600
,0xe7b78200,0xe7b78c00,0xe7b79600,0xe7b7a300,0xefa99800,0xe7b8a800,0xe7b88800,0xe7b89100
,0xe7b89500,0xefa99a00,0xe7b98700,0xe7b99200,0xe7b9a100,0xe7ba8a00,0xe7ba8d00,0xe7bd8700
,0xefa99b00,0xe7be9100,0xe7be9700,0xe7bebf00,0xe7bf8e00,0xe7bf9b00,0xe7bf9f00,0xe7bfac00
,0xe7bfae00,0xe7bfba00,0xefa99c00,0xe8809400,0xe880a600,0xe880b500,0xe880b700,0xe880bc00
,0xe8838a00,0xe8839700,0xe883a000,0xe883b300,0xe8849800,0xe8858a00,0xe885a000,0xe885a700
,0xe885a800,0xe885ad00,0xe886bb00,0xe8878a00,0xe8878f00,0xe8879700,0xefa99d00,0xe4919300
,0xe4919b00,0xe889a000,0xe889b400,0xeaada800,0xe88a8e00,0xe88aa100,0xe88aa300,0xe88aa400
,0xe88aa900,0xe88aae00,0xe88ab700,0xe88abe00,0xe88abf00,0xe88b8600,0xe88b9500,0xe88bbd00
,0xe88bbe00,0xe88c8000,0xe88c8100,0xe88da200,0xe88ca200,0xe88cad00,0xe88cba00,0xe88d8300
,0xe88d8700,0xe88d9100,0xe88d9500,0xe88dbd00,0xe88e8600,0xe88e9200,0xe88e9800,0xe88ea700
,0xe88ea900,0xe88ebf00,0xe88f8000,0xe88f8700,0xe88f8f00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe88f9100,0xe88fa100,0xe88faa00,0xe8908100,0xe8908600,0xe8908a00,0xefa99f00,0xe8918800
,0xe8919f00,0xe891b000,0xe891b300,0xe8928500,0xe8929e00,0xe892af00,0xe892b400,0xe892ba00
,0xe8938000,0xe8938200,0xeaadab00,0xe894b200,0xe8949e00,0xe894a300,0xe894af00,0xe8959900
,0xe895a400,0xefa89f00,0xe896ad00,0xe895ba00,0xe8968c00,0xe8968f00,0xe896a200,0xe896b000
,0xe8978b00,0xe8978e00,0xe897ad00,0xe8989200,0xe897bf00,0xe8988400,0xe8988500,0xe8989000
,0xeaadb100,0xe8989800,0xe898a900,0xe898b800,0xe8999700,0xe8999b00,0xefa4b600,0xe899a200
,0xe4969d00,0xe899ac00,0xe899b500,0xe89a9800,0xe89ab800,0xe89bba00,0xe89bbc00,0xe89bbd00
,0xe89c8b00,0xe89db100,0xe89e8700,0xe89e8800,0xe89eac00,0xe89ead00,0xe89eb500,0
,0xe497aa00,0xe89f9600,0xe89fac00,0xe8a08600,0xe8a08a00,0xe8a09000,0xe8a09400,0xe8a09f00
,0xe8a29800,0xe8a2aa00,0xe8a38a00,0xe8a38e00,0xeaadbc00,0xe8a3b500,0xe8a49c00,0xefa9a000
,0xe8a49800,0xe8a49900,0xe8a49a00,0xe8a4a700,0xe8a4b000,0xe8a4b200,0xe8a4b900,0xe8a58000
,0xe8a69400,0xefa9a100,0xe8a79400,0xe8a7a500,0xe8a7b600,0xe8a89200,0xe8a89500,0xe8a8a200
,0xe8a8b700,0xe8a98700,0xe8a98e00,0xe8a99d00,0xe8a9a100,0xe8a9b500,0xe8a9b900,0xe8aaa700
,0xe8ab9000,0xe8ab9f00,0xe8abb400,0xe8abb600,0xefa8a200,0xefa9a200,0xefa9a300,0xe8ad8600
,0xe8ad9400,0xe8ad9900,0xe8ada900,0xe8ae9d00,0xe8b18900,0xe8b1a800,0xefa9a400,0xe8b3a100
,0xe8b3b400,0xe8b3b800,0xe8b3be00,0xefa9a500,0xe8b49200,0xe8b49b00,0xe8b6af00,0xe8b78e00
,0xe8b79100,0xe8b79700,0xe8b8a000,0xe8b8a300,0xe8b8bd00,0xe8b9b000,0xe8b9bb00,0xeaae8c00
,0xe8bb8000,0xe4a18400,0xe8bbba00,0xe8bc9e00,0xe8bcad00,0xe8bcb600,0xe8bd9400,0xeaae8f00
,0xe8bea600,0xe8beb500,0xe8bfa400,0xe8bfa800,0xe8bfae00,0xe9808800,0xe980ad00,0xefa9a700
,0xe9828800,0xe9829500,0xe9829700,0xe9829900,0xe9829b00,0xe982a200,0xe982b300,0xe982be00
,0xe9838400,0xe9838500,0xe9838700,0xe9839700,0xe9839d00,0xe9839e00,0xe983af00,0xe983b400
,0xefa8a600,0xe9849400,0xe9849500,0xe9849600,0xe984a200,0xe984a300,0xe984a700,0xe984af00
,0xe984b100,0xe984b400,0xe984bd00,0xe9858800,0xe9859b00,0xe9868300,0xe9869e00,0xe986ac00
,0xe986b100,0xe986bc00,0xe9879700,0xe987bb00,0xe987a400,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe987a500,0xe987ad00,0xe987b100,0xe9888700,0xe9889000,0xe988b800,0xe988b900,0xe988ba00
,0xe988bc00,0xe9898000,0xe9898300,0xe9898f00,0xe989b800,0xe98a8800,0xe98b8200,0xe98b8b00
,0xe98b8c00,0xe98b9300,0xe98ba000,0xe98bbf00,0xe98c8400,0xe98c9f00,0xe98ca100,0xe98ca500
,0xe98d8800,0xe98d8900,0xe98d8a00,0xe98da400,0xe98da500,0xe98daa00,0xe98db000,0xe98e9b00
,0xe98ea300,0xe98eba00,0xe98f8600,0xe98f9e00,0xe98f9f00,0xe9908400,0xe98fbd00,0xe990b300
,0xe9918a00,0xe991a300,0xe991ab00,0xe991b100,0xe991b200,0xe9968e00,0xe9969f00,0xe996a600
,0xe996a900,0xe996ac00,0xe996b600,0xe996bd00,0xe9978b00,0xe9979000,0xe9979300,0xe4a6b000
,0xe9979a00,0xe9979e00,0xe9999800,0xe99a8400,0xefa79c00,0xe99a9d00,0xe99aa400,0
,0xe99aa500,0xe99b9200,0xe99b9e00,0xefa9a800,0xe99ba900,0xe99baf00,0xe99cb300,0xe99cbb00
,0xe99d8d00,0xe99d8e00,0xe99d8f00,0xe99d9a00,0xe99dae00,0xe99db300,0xe99e9500,0xe99eae00
,0xe99eba00,0xe99f8100,0xe99f8900,0xe99f9e00,0xe99f9b00,0xe99fb400,0xefa9a900,0xe9a08a00
,0xe9a09e00,0xe9a0ab00,0xe9a0b000,0xefa9aa00,0xe9a19200,0xe9a19300,0xe9a19600,0xe9a19700
,0xe9a19900,0xe9a19a00,0xefa79000,0xe9a1a500,0xe9a1ac00,0xe9a2ba00,0xe9a38800,0xe9a3a700
,0xe9a59800,0xe9a69e00,0xe9a88200,0xe9a88300,0xe9a8a400,0xe9a8ad00,0xe9a8ae00,0xe9a8b800
,0xe9a98a00,0xe9a98e00,0xe9a99200,0xe9aab600,0xe9ab8100,0xe9ab8300,0xe9ab8e00,0xe9ab9600
,0xe9abb900,0xe9ac8200,0xe9ac8800,0xe9aca000,0xe4b09700,0xe9acad00,0xe9ad9e00,0xe9adb900
,0xe9ada600,0xe9adb200,0xe9adb500,0xe9ae8400,0xe9ae8a00,0xe9ae8f00,0xe9ae9e00,0xe9aea700
,0xe9af8100,0xe9af8e00,0xe9afa500,0xe9afb800,0xe9afbd00,0xe9b08000,0xe9b0a300,0xe9b18100
,0xe9b18f00,0xe9b19000,0xe9b19300,0xe9b1a300,0xe9b1a500,0xe9b1b700,0xe9b49d00,0xe9b49e00
,0xe9b58300,0xe9b58700,0xe9b59200,0xe9b5a300,0xe9b5b000,0xe9b5bc00,0xe9b68a00,0xe9b69600
,0xe9b78000,0xe9b6ac00,0xe9b6bc00,0xe9b79700,0xeaaf8800,0xe9b7a700,0xe9b88700,0xe9b89500
,0xe9b9bc00,0xe9ba9e00,0xe9baa400,0xe9baac00,0xe9baaf00,0xe9bab400,0xe9bab500,0xe9bb8300
,0xe9bb9100,0xe9bc9000,0xe9bcb900,0xe9bd9700,0xe9be9000,0xe9be9400,0xe9be9700,0xe9bea200
,0xe5a7b800,0xe5b19b00,0xe5b9b700,0xe798a600,0xe7b9ab00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xeaaaa100,0xe4b88200,0xe4b88f00,0xe4b89200,0xe4b8a900,0xe4b8ab00,0xe4b8ae00,0xe4b98000
,0xe4b98700,0xe4b98800,0xeaaaa300,0xe4b99100,0xe3908600,0xeaaaa400,0xe4b99a00,0xe4b9a900
,0xe4ba9d00,0xe390ac00,0xe390ae00,0xe4bab900,0xe4babb00,0xeaaaa500,0xe4babc00,0xe4bb8300
,0xe4bb8800,0xe4bb9000,0xe4bbab00,0xe4bb9a00,0xe4bbb100,0xe4bbb500,0xe4bc8000,0xe4bc9600
,0xe4bda400,0xe4bcb700,0xe4bcbe00,0xe4bd9400,0xe4bd9800,0xeaaaa600,0xe4bdb700,0xe4bdb800
,0xe4bdba00,0xe4bdbd00,0xe4be8200,0xe4be8500,0xe4be9200,0xe4be9a00,0xe4bfa600,0xe4beb200
,0xe4bebe00,0xe4bf8500,0xe4bf8b00,0xe4bf8f00,0xe4bf9200,0xe391aa00,0xe4bfb200,0xe5808000
,0xe5809000,0xe5809300,0xe5809c00,0xe5809e00,0xe580a200,0xe391a800,0xe5818200,0
,0xe5818600,0xe5818e00,0xe5819300,0xe5819700,0xe581a300,0xe581a600,0xe581aa00,0xe581b000
,0xe582a300,0xe5828800,0xe5829200,0xe5829300,0xe5829500,0xe5829600,0xe5829c00,0xe582aa00
,0xeaaaa700,0xe582b100,0xe582ba00,0xe582bb00,0xe5838400,0xe5838700,0xe583b300,0xeaaaa900
,0xe5838e00,0xeaaaa800,0xe5839400,0xe5839900,0xe583a100,0xe583a900,0xe3929200,0xe5ae9600
,0xe5aeac00,0xe39da100,0xe5af8000,0xe39da200,0xe5af8e00,0xe5af9600,0xe39dac00,0xe39dab00
,0xe5afb100,0xe5afbd00,0xe39db500,0xe5b08300,0xe5b0a900,0xe5b0b000,0xeaab9200,0xe5b19f00
,0xe5b1a300,0xe5b1a700,0xe5b1a800,0xe5b1a900,0xe5b1b000,0xeaab9300,0xeaab9500,0xe5b1bc00
,0xeaab9600,0xeaab9700,0xe5b28800,0xe5b28a00,0xe39f8100,0xeaab9900,0xeaab9a00,0xe5b2a000
,0xe5b2a200,0xe5b2a600,0xe5b2a700,0xeaab9800,0xe5b2ad00,0xe5b2b500,0xeaab9b00,0xe5b38900
,0xeaab9c00,0xeaab9e00,0xe5b48600,0xe5b49000,0xe5b4ab00,0xe5b49d00,0xe5b4a000,0xe5b4a400
,0xe5b4a600,0xe5b4b100,0xe5b4b900,0xe5b58200,0xe39fa800,0xe5b5a100,0xe5b5aa00,0xe39fb400
,0xe5b5b000,0xeaab9f00,0xe39fbd00,0xe5b68800,0xe3a08000,0xe5b69200,0xe5b69400,0xe5b69700
,0xe5b69900,0xe5b6b000,0xe5b6b200,0xe5b6b400,0xeaaba000,0xe5b6b900,0xe5b79100,0xe5b79700
,0xe5b79800,0xe5b7a000,0xeaaba100,0xe5b7a400,0xe5b7a900,0xe3a0af00,0xe5b88000,0xe3a0b600
,0xe5b89200,0xe5b89500,0xe3a18000,0xe5b89f00,0xe5b8ae00,0xe5b8be00,0xe5b98900,0xe3a19c00
,0xe5b99600,0xe3a1a100,0xe5b9ab00,0xe5b9ac00,0xe5b9ad00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe5848800,0xeaaaaa00,0xe5849700,0xe5849b00,0xeaaaab00,0xe585a000,0xeaaaac00,0xe585b300
,0xe5868300,0xe5868b00,0xe392bc00,0xe5869800,0xe586a300,0xe586ad00,0xe3938700,0xe586bc00
,0xeaaaad00,0xeaaaae00,0xe587b300,0xe587b400,0xe5888200,0xe5889200,0xe5889600,0xeaaaaf00
,0xe5899500,0xe5899c00,0xe589ac00,0xe589b700,0xe58a8400,0xe58a8200,0xeaaab000,0xe58a9800
,0xeaaab100,0xe58aa400,0xe58aa600,0xe58aaf00,0xe58aba00,0xe58abb00,0xe58b8a00,0xe3949f00
,0xe58b9100,0xeaaab200,0xe58bb700,0xe58c8a00,0xe58c8b00,0xe58ca400,0xe58cb500,0xe58cbe00
,0xe58d8200,0xeaaab300,0xeaaab400,0xe58da700,0xe58dac00,0xe58dba00,0xe58ea400,0xe58eb400
,0xeaaab500,0xe58eb700,0xe58f8000,0xeaaab600,0xe3959d00,0xe3959e00,0xe58f9500,0
,0xe58f9a00,0xe395a300,0xe58fb400,0xe58fb500,0xe5919500,0xe590a400,0xe590a800,0xe395ae00
,0xe5918300,0xe591a200,0xe591a600,0xe591ac00,0xe5928a00,0xe5928d00,0xe5929500,0xe592a000
,0xe592a600,0xe592ad00,0xe592ae00,0xe592b700,0xe592ba00,0xe592bf00,0xe5938300,0xeaaab700
,0xe593ac00,0xe593af00,0xe593b100,0xe593b300,0xe5948000,0xe5948100,0xe5948900,0xe594bc00
,0xe5958100,0xe396a600,0xe5958700,0xe5958a00,0xe396a800,0xe595a000,0xe595a100,0xe595a400
,0xeaaab800,0xe595bd00,0xe5968200,0xe5968800,0xe5969100,0xe3978500,0xe5979200,0xeaaaba00
,0xeaaabb00,0xe596bf00,0xe5978900,0xe5978c00,0xe5979100,0xe5979d00,0xe3979a00,0xe597a200
,0xeaaab900,0xe597a900,0xe598a800,0xeaaabc00,0xe5988700,0xe5989000,0xe598b000,0xe598b700
,0xe397b400,0xe598bd00,0xe598bf00,0xe5998000,0xe5998700,0xe5999e00,0xe599a000,0xe599ad00
,0xe3988500,0xe59a8800,0xe59a8c00,0xe59a9500,0xe59a9a00,0xe59a9d00,0xe59aa800,0xe59aad00
,0xe59ab200,0xe59b8500,0xe59b8d00,0xe59b9f00,0xe59ba800,0xe59bb600,0xe59bb700,0xeaaabd00
,0xe59c9500,0xe59ca300,0xeaaabf00,0xe59ca900,0xeaab8100,0xe59d8500,0xe59d8600,0xe59d8c00
,0xe59d8d00,0xeaab8000,0xe59da800,0xe59daf00,0xe59db300,0xe59db400,0xe59db500,0xe59dbb00
,0xeaab8200,0xeaab8300,0xe59eac00,0xe59e9a00,0xe59e9d00,0xe59e9e00,0xe59ea800,0xe59f9700
,0xeaab8500,0xe59f8c00,0xeaab8600,0xeaab8700,0xe59f9e00,0xe59fa600,0xe59fb000,0xe3998a00
,0xe59fb800,0xe59fbb00,0xe59fbd00,0xe5a08400,0xe5a09e00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe5a0a000,0xe5a0a700,0xe5a0b200,0xe5a0b900,0xeaab8800,0xe5a18900,0xe5a18c00,0xe5a1a700
,0xe5a28a00,0xe5a28b00,0xe5a28d00,0xe5a28f00,0xe5a29000,0xe5a29400,0xe5a29d00,0xe5a2aa00
,0xe5a2b100,0xeaab8a00,0xe5a38300,0xe5a38d00,0xe5a3a200,0xe5a3b300,0xe5a3b400,0xe5a48500
,0xe5a48600,0xe5a48b00,0xe5a48d00,0xe5a49400,0xe5a4a400,0xeaab8b00,0xe39a9100,0xe5a4bd00
,0xe39a9900,0xe5a58600,0xe39a9600,0xeaab8c00,0xe5a59b00,0xe5a59f00,0xeaab8d00,0xe5a5b500
,0xe5a5b600,0xe5a5bc00,0xe5a69f00,0xe5a6ae00,0xe5a6bc00,0xe5a78800,0xe5a78d00,0xe5a79e00
,0xe5a7a300,0xe5a7a400,0xe5a7a700,0xe5a7ae00,0xeaab8e00,0xeaab8f00,0xe39b8f00,0xe5a88c00
,0xe5a88d00,0xe5a89700,0xe5a8a700,0xe5a8ad00,0xe5a99500,0xe5a9a500,0xe5a9ba00,0
,0xe5aa8b00,0xe5aa9c00,0xe5aa9f00,0xe5aaa000,0xe5aaa200,0xe5aab100,0xe5aab300,0xe5aab500
,0xe5aaba00,0xe5aabf00,0xe5ab9a00,0xe5ab9c00,0xe5aba000,0xe5aba500,0xe5abb000,0xe5abae00
,0xe5abb500,0xe5ac8000,0xe5ac8800,0xe5ac9700,0xe5acb400,0xe5acad00,0xe5ad8c00,0xe5ad9200
,0xe5ada800,0xe5adaf00,0xe5adbc00,0xe5adbf00,0xe5ae8100,0xe5ae8400,0xeaab9100,0xe5b9ae00
,0xeaaba200,0xe5baa500,0xe5baaa00,0xe5baac00,0xe5bab900,0xe5babf00,0xe5bb8600,0xe5bb9200
,0xe5bb9900,0xeaaba300,0xe5bbbd00,0xe5bc8800,0xe5bc8e00,0xe5bc9c00,0xeaaba400,0xe5bc9e00
,0xe5bd8700,0xe5bda300,0xe5bdb200,0xe5bdbe00,0xe5be8f00,0xe5bea200,0xe5bea400,0xe5beb800
,0xe5bf8400,0xe3a3ba00,0xe5bf8700,0xe5bf8b00,0xe5bf9200,0xe5bf9300,0xe5bf9400,0xe5bfa200
,0xe5bfae00,0xe5bfaf00,0xe5bfb300,0xe5bfbc00,0xe3a49700,0xe6809700,0xe680a200,0xe680a400
,0xe3a49a00,0xe6818c00,0xe681bf00,0xe6828a00,0xe6829500,0xe682a800,0xeaaba500,0xe682b000
,0xe682b100,0xe682be00,0xe6838800,0xe6839900,0xe6839b00,0xe683ae00,0xe683b200,0xe683b500
,0xe6849000,0xe6849200,0xe6849300,0xe6849900,0xe6849e00,0xe684ba00,0xe3a5af00,0xe6858100
,0xe6858600,0xe685a000,0xe685bc00,0xeaaba600,0xe6869200,0xe6869300,0xe6869700,0xe6869800
,0xe686a500,0xe686a800,0xe686ad00,0xeaaba700,0xe6879500,0xe6879d00,0xe6879f00,0xe687b500
,0xeaaba800,0xe6889500,0xe688a300,0xe688a900,0xe6898600,0xe6898c00,0xe6899100,0xe6899200
,0xe689a100,0xe689a400,0xe689bb00,0xe689ad00,0xe689b300,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe68a9900,0xe68aa600,0xe68b9500,0xeaaba900,0xe68bbd00,0xe68c8300,0xe68c8d00,0xe68c9000
,0xeaabab00,0xeaabac00,0xe68cb200,0xe68cb500,0xe68cbb00,0xe68cbc00,0xe68d8100,0xe68d8400
,0xe68d8e00,0xeaabaa00,0xe68d9900,0xeaabad00,0xeaabae00,0xe68dac00,0xe68e8400,0xe68e9900
,0xeaabaf00,0xe68e9400,0xe68ebd00,0xe68fb700,0xe68f9400,0xe68f9500,0xe68f9c00,0xe68fa000
,0xe68fab00,0xe68fac00,0xe68fb200,0xe6908900,0xe6909e00,0xe690a500,0xe690a900,0xe690af00
,0xe6919a00,0xe6919b00,0xe6919d00,0xe691b300,0xe691bd00,0xe6928700,0xe6929100,0xe6929d00
,0xe6929f00,0xe6938b00,0xe6938c00,0xe6939500,0xe6939700,0xeaabb000,0xe693a400,0xe693a500
,0xe693bf00,0xe6948400,0xe3a9ae00,0xe6948f00,0xe6949400,0xe6949600,0xe3a9b300,0
,0xe6949e00,0xe694b200,0xe6958400,0xe6959400,0xe695ab00,0xe695ba00,0xe6968100,0xe6968400
,0xe6968500,0xe6968a00,0xe696b200,0xe696b500,0xe696b800,0xe696bf00,0xe6978200,0xe6978900
,0xe6979400,0xe3ab9600,0xe697b200,0xe697b900,0xe697bc00,0xe6988400,0xe6988800,0xe698a100
,0xe698aa00,0xe6998500,0xe6999100,0xe6998e00,0xe3abaa00,0xeaabb200,0xe6999700,0xe6999b00
,0xe699a300,0xeaabb400,0xeaabb300,0xe699aa00,0xe699ab00,0xe699ac00,0xe699ad00,0xe699bb00
,0xe69a8000,0xe69a9000,0xe69a9200,0xe69a9900,0xe3ac8e00,0xe69aad00,0xe69ab100,0xe69ab500
,0xe3ac9a00,0xe69abf00,0xe3ac9c00,0xe69bac00,0xe3ab9700,0xe69c8100,0xe69c8500,0xe69c9200
,0xeaabb500,0xe69c9900,0xeaabb700,0xeaabb800,0xe69d8c00,0xe69d8d00,0xe69d9400,0xe69d9d00
,0xeaabb600,0xeaabbc00,0xeaabb900,0xe69db400,0xe69db600,0xeaabba00,0xe69e9200,0xeaabbb00
,0xe88da300,0xe6a09000,0xe69eb000,0xe69eb200,0xe69f8300,0xe69f8800,0xe69f9200,0xe69f9900
,0xe69f9b00,0xe69fb000,0xe69fb700,0xeaabbe00,0xeaabbf00,0xeaac8000,0xe6a09800,0xe6a09f00
,0xe6a0ad00,0xeaabbd00,0xe6a0b300,0xe6a0bb00,0xe6a0be00,0xe6a18400,0xe6a18500,0xe6a18900
,0xe6a18c00,0xe6a19500,0xe6a19700,0xe3adb700,0xe6a1ab00,0xe6a1ae00,0xe6a1ba00,0xe6a1bc00
,0xe6a28200,0xe6a29000,0xe6a29600,0xe3adad00,0xe6a29800,0xe6a29900,0xe6a29a00,0xe6a29c00
,0xe6a2aa00,0xe6a2ab00,0xe6a2b400,0xe6a2bb00,0xe6a3bb00,0xeaac8100,0xeaac8200,0xefa89300
,0xe6a38300,0xe6a38500,0xe6a38c00,0xe6a38f00,0xe6a39600,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe6a39900,0xe6a3a400,0xe6a3a500,0xe6a3ac00,0xe6a3b700,0xe6a48300,0xe6a48700,0xe3ae8700
,0xe3ae8800,0xeaac8400,0xe6a4bb00,0xe3ae8d00,0xe6a58600,0xe6a5a900,0xe6a5ac00,0xe6a5b200
,0xe6a5ba00,0xe6a5bf00,0xe6a69200,0xe3aea400,0xe6a69600,0xe6a69800,0xe6a6a600,0xe6a6b000
,0xe6a6b700,0xe6a6ba00,0xe6a6bc00,0xe6a78000,0xe6a79100,0xe6a79600,0xeaac8500,0xeaac8600
,0xe6a8b000,0xeaac8700,0xeaac8800,0xe6a7a300,0xe6a7ae00,0xe6a7af00,0xe6a7b300,0xe3af8d00
,0xe6a7b400,0xe6a7be00,0xe6a89100,0xe6a89a00,0xe6a89d00,0xeaac8a00,0xe6a8b200,0xe6a8b300
,0xe6a8b400,0xe6a8bf00,0xe6a98600,0xe6a98900,0xe6a9ba00,0xe6a98e00,0xe6a99200,0xe6a9a400
,0xeaac8c00,0xe6a9be00,0xe6aa8300,0xe6aa8b00,0xe3afb000,0xe6aa9100,0xe6aa9f00,0
,0xe6aaa100,0xeaac8d00,0xe6aaab00,0xe6aabd00,0xe6ab8600,0xe6ab9400,0xe6ab9000,0xe6ab9c00
,0xe6ab9d00,0xeaac8e00,0xeaac8f00,0xe6abac00,0xe6abb100,0xe6abb200,0xe6abb300,0xe6abbd00
,0xeaac9000,0xe6ac8b00,0xe6ac8f00,0xe6ac9000,0xe6ac9100,0xeaac9100,0xe6ac9700,0xe3b0a600
,0xe6acaf00,0xe6ad8a00,0xe6ad9800,0xe6adac00,0xe6adb500,0xe6adba00,0xe6ae8100,0xe6ae9b00
,0xe6aeae00,0xeaac9200,0xe6aebd00,0xe6aebe00,0xe6af8700,0xe6af8800,0xe6af8900,0xe6af9a00
,0xe6afa600,0xe6afa700,0xe6afae00,0xe6afb100,0xe6b08200,0xe6b08a00,0xe6b08e00,0xe6b0b500
,0xe6b0b600,0xe6b0ba00,0xeaac9300,0xe6b0bf00,0xe6b18d00,0xe6b19b00,0xe6b1ad00,0xe6b28400
,0xe6b28900,0xe3b38300,0xe6b29400,0xe6b29500,0xe6b29700,0xe6b2ad00,0xe6b38200,0xe6b39000
,0xe3b39200,0xe6b39600,0xe6b39a00,0xe6b39c00,0xe6b3a900,0xe6b3ac00,0xe6b3ad00,0xeaac9500
,0xe6b48000,0xe6b48a00,0xe6b4a400,0xe6b4a600,0xe6b4a700,0xe6b1a700,0xe6b4af00,0xe6b4bc00
,0xe6b59b00,0xe6b59e00,0xe6b5a000,0xe6b5b000,0xe6b68000,0xe6b68100,0xe6b68a00,0xe6b68d00
,0xe6b69100,0xe6b69800,0xeaac9600,0xe6b89700,0xeaac9700,0xeaac9800,0xeaac9900,0xe6b6ab00
,0xe6b6ae00,0xe6b6b400,0xe6b78200,0xe6b4b400,0xe6b78800,0xe6b78e00,0xe6b78f00,0xe6b79000
,0xe6b79f00,0xe6b7a900,0xe6b7b600,0xe6b8b600,0xe6b89e00,0xe6b8a200,0xe6b8a700,0xe3b49100
,0xe6b8b200,0xe6b8bc00,0xe6b98800,0xe6b98900,0xe6b98b00,0xe6b98c00,0xe6b98f00,0xe6b99100
,0xe6b99300,0xe6b99400,0xe6b99700,0xe6b9a300,0xe3b49e00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe6ba9300,0xe6baa700,0xe6bab400,0xe6babf00,0xe6bb8300,0xe6bb8a00,0xe6bb9900,0xe6bcb500
,0xe6bbab00,0xe6bbb900,0xe6bbbb00,0xe6bc8a00,0xe6bc8c00,0xe6bc9800,0xe6bca500,0xe6bcb600
,0xe6bcbc00,0xeaac9a00,0xe6bd9200,0xe6bd9700,0xe6bd9a00,0xe6bda000,0xe6bda800,0xe6be9800
,0xe6bdbd00,0xe6be9000,0xe6be9600,0xe6bebe00,0xe6be9f00,0xe6bea500,0xe6beaf00,0xe3b5a400
,0xe6beb500,0xe6bf8800,0xe6bf8900,0xe6bf9a00,0xe6bf9e00,0xe6bfa900,0xeaac9b00,0xe6bfbc00
,0xe7808000,0xe7808700,0xe7808a00,0xe780a300,0xeaac9c00,0xe780b900,0xe780ba00,0xe780bc00
,0xe7818300,0xe7818700,0xe7818b00,0xe3b69a00,0xe7819400,0xe781a500,0xe781a900,0xe781ac00
,0xe781ae00,0xe781b600,0xe781be00,0xe7828100,0xe7828600,0xe7829500,0xe7829700,0
,0xe782bb00,0xeaac9d00,0xe7829f00,0xe782b100,0xeaac9e00,0xe783ac00,0xe7838a00,0xe7839100
,0xe7839300,0xe7839c00,0xe7848300,0xe7848400,0xe7848600,0xe7848700,0xe7848800,0xe7848c00
,0xe3b78000,0xe784af00,0xe784b100,0xe7859000,0xe7858a00,0xe7859300,0xe7859e00,0xe3b79400
,0xe7869600,0xe7868000,0xe7869b00,0xe786a000,0xe786a200,0xe786ae00,0xe786af00,0xe786b300
,0xeaac9f00,0xe7878b00,0xe7879300,0xe7879900,0xe7879c00,0xe7888700,0xe3b88500,0xefa98a00
,0xe788ab00,0xe788b400,0xe788b800,0xe788b900,0xe4b8ac00,0xe7898200,0xe7899300,0xe7899700
,0xe789a300,0xeaaca000,0xe789ae00,0xe789af00,0xe789b800,0xe789bf00,0xe78a8e00,0xeaaca100
,0xe78aad00,0xe78aae00,0xe78ab000,0xe78ab100,0xe78b8100,0xe3b9a000,0xe78b8c00,0xe3b9a600
,0xe3b9a800,0xe78bb300,0xe78bba00,0xe78c8700,0xe78c9200,0xe78c9800,0xe78c9900,0xe3ba8300
,0xe78cb900,0xe78cac00,0xe78cb100,0xe78cb300,0xe78cbd00,0xe78d9200,0xe3ba9400,0xe78dab00
,0xe78dac00,0xeaaca300,0xe78dae00,0xe78daf00,0xe78db100,0xe78db700,0xe78e8100,0xe78e8500
,0xe78e8a00,0xe78e9400,0xe78e9800,0xe78e9c00,0xe78e9e00,0xe78ea500,0xe78ea800,0xe78eb500
,0xe78eb700,0xe78eb900,0xe78ebc00,0xe78ebf00,0xe78f8500,0xe78f8b00,0xe78fa100,0xe78fa700
,0xe78fb900,0xe7909300,0xe78fba00,0xe7908100,0xe790a400,0xe790b100,0xe790b900,0xe7919300
,0xe7918000,0xe7918300,0xe7918d00,0xe7919200,0xe7919d00,0xe791b100,0xe7928100,0xe7928500
,0xe7928800,0xeaaca400,0xe7929200,0xe7929700,0xe7929900,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe792a000,0xe792a100,0xe792a500,0xe792aa00,0xe792ab00,0xe792b900,0xe792bb00,0xe792ba00
,0xe7939600,0xe7939800,0xe7939e00,0xe793af00,0xe793ab00,0xeaaca500,0xe793ba00,0xeaaca600
,0xe794a000,0xe794a400,0xe794aa00,0xe3bd9700,0xeaaca700,0xe794bd00,0xe794be00,0xe7958000
,0xe7958800,0xe7958e00,0xe7959000,0xe7959200,0xe795ac00,0xe795b200,0xe795b100,0xe795ba00
,0xe795bd00,0xe795be00,0xe7968100,0xeaaca800,0xe7968c00,0xe3bdb500,0xe796a200,0xe3bdb700
,0xe796b000,0xe796b700,0xe796bf00,0xe7978000,0xe7978600,0xe7978f00,0xe7979300,0xe7979d00
,0xe7979f00,0xe797a000,0xe797a700,0xe797ac00,0xe797ae00,0xe797b100,0xe797b900,0xe7988300
,0xe7989800,0xe7988700,0xe7988f00,0xe3beae00,0xeaaca900,0xe7989300,0xe7989b00,0
,0xe7989c00,0xeaacaa00,0xe798a500,0xe798a800,0xe798bc00,0xe798b300,0xeaacab00,0xe3bf8900
,0xe7998100,0xeaacac00,0xe7998900,0xe7999500,0xe3bf9700,0xe799ae00,0xe79a9500,0xe79a9c00
,0xe79aa100,0xe79aa000,0xe79aa700,0xe79aa800,0xe79aaf00,0xeaacad00,0xe79b8900,0xeaacae00
,0xe79ba800,0xe79bac00,0xeaacaf00,0xe79c9700,0xe79c9a00,0xe79cad00,0xe79cb500,0xeaacb000
,0xe480b900,0xeaacb200,0xeaacb100,0xe79d9800,0xe79da000,0xe79daa00,0xeaacb400,0xe79db200
,0xe79dbc00,0xe79dbd00,0xeaacb300,0xe4819800,0xe79e9a00,0xe79e9f00,0xe79ea200,0xe79ea400
,0xe79ea900,0xe79f9e00,0xe79f9f00,0xe79fa400,0xe79fa600,0xe79faa00,0xe79fac00,0xe4829300
,0xe79fb000,0xe79fb400,0xe79fbb00,0xeaacb500,0xe7a08500,0xe7a08600,0xe7a08900,0xe7a08d00
,0xe7a09900,0xe7a0a100,0xe7a0ac00,0xe7a18700,0xe7a1a400,0xe7a1aa00,0xeaacb700,0xe7a28a00
,0xe7a29400,0xe7a2a400,0xe7a29d00,0xe7a29e00,0xe7a29f00,0xe7a2bb00,0xe7a38800,0xe7a38c00
,0xe7a38e00,0xe7a39500,0xe7a3a000,0xe7a3a100,0xe7a3a600,0xe7a3b900,0xe7a3ba00,0xe7a3bb00
,0xe7a3be00,0xeaacb900,0xe7a49000,0xe7a49b00,0xe7a4b000,0xe7a4a500,0xe7a4bb00,0xe7a58a00
,0xe7a59800,0xe7a59b00,0xe4848500,0xe7a5a700,0xe7a5b200,0xe7a69400,0xe7a69500,0xe7a69600
,0xe7a69b00,0xe7a6a100,0xe7a6a900,0xe7a6b400,0xe7a6bb00,0xe7a78200,0xe7a78700,0xe7a78c00
,0xe7a78d00,0xe7a79600,0xe4858800,0xeaacbb00,0xeaacbc00,0xe4858f00,0xe7a88a00,0xe7a89100
,0xe7a89500,0xe7a89b00,0xe7a89e00,0xe485a300,0xe7a8ad00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe7a8b800,0xe7a98700,0xe7a98c00,0xe7a99600,0xe7a99900,0xe7a99c00,0xe7a99f00,0xe7a9a000
,0xe7a9a700,0xe7a9aa00,0xe7a9b500,0xe7a9b800,0xe7aa8200,0xe7aa8a00,0xe7aa9000,0xe7aaa300
,0xe7aaac00,0xeaacbe00,0xe486b400,0xe7aab900,0xe7aabc00,0xe7aabe00,0xe486bf00,0xe7ab8c00
,0xe7ab9100,0xe7aba700,0xe7aba800,0xe7abb400,0xeaacbf00,0xeaad8000,0xe7ac8700,0xeaad8100
,0xe7acbd00,0xe7aca700,0xe7acaa00,0xe7acae00,0xe7acaf00,0xe7acb100,0xe487a600,0xe487b300
,0xe7adbf00,0xe7ad8100,0xe487ae00,0xe7ad9500,0xe7adb900,0xe7ada400,0xe7ada600,0xe7ada900
,0xe7adb300,0xeaad8200,0xe4888700,0xe7ae9000,0xe7ae9100,0xe7ae9b00,0xe4888e00,0xe7aeaf00
,0xe7aeb500,0xe7aebc00,0xe7af8500,0xe7af8a00,0xeaad8300,0xeaad8400,0xe7af9400,0
,0xe7af9600,0xe7af9a00,0xe7afaa00,0xe7afb000,0xe7b08300,0xe7b08b00,0xe7b08e00,0xe7b08f00
,0xe7b0a600,0xe7b18500,0xe7b18a00,0xe7b19100,0xe7b19700,0xe7b19e00,0xe7b1a100,0xe7b1a900
,0xe7b1ae00,0xe7b1af00,0xe7b1b000,0xeaad8600,0xeaad8700,0xeaad8900,0xe7b2a600,0xeaad8800
,0xe7b2b600,0xe7b2b700,0xe7b2bf00,0xeaad8b00,0xe7b38400,0xeaad8a00,0xe7b38800,0xe7b38d00
,0xeaad8c00,0xe7b39700,0xeaad8d00,0xe7b3a600,0xe7b3ab00,0xeaad8e00,0xe7b3b500,0xe7b48300
,0xe7b48900,0xe48b8600,0xe7b49200,0xe7b49e00,0xeaad8f00,0xeaad9000,0xe7b4bd00,0xe7b4be00
,0xe7b58000,0xe7b58700,0xeaad9100,0xeaad9200,0xe48b9600,0xe7b59900,0xe7b59a00,0xe7b5aa00
,0xe7b5b000,0xe48b9d00,0xe7b5bf00,0xeaad9300,0xe7b68600,0xe7b68800,0xe7b68c00,0xe7b69700
,0xeaad9400,0xe7b69d00,0xe7b6a700,0xe7b6aa00,0xe7b6b600,0xe7b6b700,0xe7b78000,0xe7b79700
,0xe7b79900,0xe7b7a600,0xe7b7b100,0xe7b7b900,0xe48c8200,0xeaad9500,0xefa99900,0xe7b89000
,0xe7b89700,0xe7b89d00,0xe7b8a000,0xe7b8a700,0xe7b8ac00,0xe7b98500,0xe7b9b300,0xe7b9b500
,0xe7b9be00,0xe7ba8600,0xe7ba8700,0xe48cab00,0xe7ba9100,0xe7ba9800,0xe7ba9a00,0xe48d8300
,0xe7bcbc00,0xe7bcbb00,0xe7bcbe00,0xe7bd8300,0xe7bd8400,0xe7bd8f00,0xe3938100,0xeaad9600
,0xe7bd9200,0xeaad9700,0xe7bda100,0xe7bda300,0xe7bda400,0xe7bdad00,0xe7bdbd00,0xe7bdbe00
,0xeaad9800,0xe7be9000,0xe585bb00,0xeaad9900,0xe7be9600,0xe7be9c00,0xe7bead00,0xeaad9a00
,0xe7bf8300,0xe7bf8f00,0xe7bfa300,0xe7bfa500,0xe7bfaf00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe7bfb200,0xe8808200,0xe8808a00,0xe8808800,0xe8808e00,0xe8809100,0xe8809600,0xe880a400
,0xe880ac00,0xe880b000,0xe8818300,0xe881a600,0xe881b100,0xe881b500,0xe881bb00,0xe8829900
,0xe8829c00,0xe882a400,0xe882a700,0xe882b800,0xeaad9b00,0xe8838500,0xe8839500,0xe8839800
,0xe883a600,0xeaad9c00,0xe8848d00,0xe883b500,0xe883bb00,0xe48fae00,0xe884b500,0xe8849600
,0xe8849e00,0xe48fb000,0xe884a400,0xe884a700,0xe884ac00,0xeaad9d00,0xe884bd00,0xe4908800
,0xe885a900,0xe4909700,0xe8868100,0xe4909c00,0xe8868400,0xe8868500,0xe490a200,0xe8869800
,0xe886b200,0xe8878100,0xe8878300,0xe8879600,0xe8879b00,0xeaad9e00,0xe887a400,0xeaad9f00
,0xe887ac00,0xeaada000,0xe887bd00,0xe887bf00,0xeaada100,0xe8888400,0xeaada200,0
,0xe8889900,0xe888a100,0xe888a200,0xeaada300,0xe888b200,0xe888b400,0xe888bc00,0xe8898600
,0xe8898900,0xe8898500,0xeaada400,0xe8898b00,0xe491b600,0xe8898f00,0xe491ba00,0xe8899700
,0xeaada500,0xe8899c00,0xe889a300,0xeaada600,0xeaada700,0xefa99e00,0xe889b900,0xe4929100
,0xe889bd00,0xe889bf00,0xe88a8300,0xe88a8a00,0xe88a9300,0xe88aa700,0xe88aa800,0xe88ab200
,0xe88ab400,0xe88aba00,0xe88abc00,0xe88ba200,0xe88ba800,0xe88bb700,0xe88c8700,0xe88c8800
,0xe88c8c00,0xe88d9400,0xe88c9b00,0xe88c9d00,0xe88cb000,0xe88cbc00,0xe88d8400,0xe88d9700
,0xe492be00,0xe88dbf00,0xe4939400,0xe492b300,0xe88e8d00,0xe88e9400,0xe88e9500,0xe88e9b00
,0xe88e9d00,0xe88f8900,0xe88f9000,0xe88f9400,0xe88f9d00,0xe88fa500,0xe88fb900,0xe8908f00
,0xe8909100,0xe8909500,0xeaada900,0xe8909700,0xe890b900,0xe8918a00,0xe8918f00,0xe8919100
,0xe8919200,0xe8919900,0xe8919a00,0xe8919c00,0xeaadaa00,0xe891a500,0xe891b600,0xe891b800
,0xe891bc00,0xe8928100,0xe4948d00,0xe8939c00,0xe8929700,0xe892a600,0xe892be00,0xe4948800
,0xe8938e00,0xe8938f00,0xe8939300,0xeaadac00,0xe893a700,0xe893aa00,0xe893af00,0xe893b000
,0xe893b100,0xe893ba00,0xe893bd00,0xe8948c00,0xe8949b00,0xe894a400,0xe894a500,0xe894ab00
,0xe894b400,0xe8958f00,0xe895af00,0xe494a500,0xe4958300,0xe894be00,0xe8959100,0xe8959300
,0xe8959e00,0xe895a100,0xe895a200,0xeaadad00,0xe895bb00,0xe895bd00,0xe895bf00,0xe8968100
,0xe8968600,0xe8969300,0xe8969d00,0xe8969f00,0xeaadae00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xeaadaf00,0xeaadb000,0xe896b700,0xe896bc00,0xe8978700,0xe8978a00,0xe8979800,0xe8979900
,0xe8979f00,0xe897a100,0xe897a600,0xe897b600,0xe8988000,0xe8989100,0xe8989e00,0xe898a100
,0xe898a400,0xe898a700,0xeaadb200,0xe898b900,0xe898bc00,0xeaadb300,0xe8998000,0xefa8a000
,0xe8999300,0xe8999600,0xe899af00,0xe899b700,0xe899ba00,0xe89a8700,0xe89a8900,0xe89a8d00
,0xe89a9100,0xe89a9c00,0xe89a9d00,0xe89aa800,0xefa8a100,0xe89ab100,0xe89ab300,0xe89b8100
,0xe89b8300,0xe89b9100,0xe89b9500,0xe89b9700,0xe89ba300,0xe89ba600,0xe496b800,0xe89c8500
,0xe89c8700,0xe89c8e00,0xe89c9000,0xe89c9300,0xe89c9900,0xe89c9f00,0xe89ca100,0xe89ca300
,0xe89cb100,0xe89cba00,0xe89cbe00,0xe89d8000,0xe89d8300,0xe89d9100,0xe89d9800,0
,0xe89da400,0xe89da500,0xe89db200,0xe89dbc00,0xeaadb500,0xeaadb600,0xe89ea700,0xe89e8900
,0xe89e8b00,0xe89e9300,0xe89ea000,0xeaadb400,0xe497a500,0xe89ebe00,0xeaadb700,0xe89f8100
,0xe89f8e00,0xe89fb500,0xe89f9f00,0xeaadb800,0xe89fa300,0xe89fa500,0xe89fa600,0xe89faa00
,0xe89fab00,0xe89fad00,0xe8a08100,0xe8a08300,0xe8a08b00,0xe8a09300,0xe8a0a800,0xe8a0ae00
,0xe8a0b200,0xe8a0bc00,0xe4988f00,0xe8a18a00,0xe8a19800,0xe8a19f00,0xe8a1a400,0xeaadb900
,0xeaadba00,0xe8a1a900,0xeaadbb00,0xe8a1af00,0xe8a2a000,0xe8a2bc00,0xe8a2bd00,0xe8a2be00
,0xe8a38000,0xe8a39200,0xeaadbd00,0xe8a39100,0xe8a39300,0xe8a39b00,0xe8a3b000,0xe8a3b100
,0xe4998100,0xe8a48100,0xeaadbe00,0xe8a4b700,0xeaadbf00,0xe8a58200,0xe8a58500,0xe8a58900
,0xeaae8000,0xe499a500,0xe8a5a200,0xe8a68000,0xe8a68900,0xe8a69000,0xe8a69f00,0xe8a6b000
,0xe8a6b700,0xe8a79600,0xe8a79800,0xe8a7ab00,0xe49aa100,0xe8a7b100,0xe8a7b300,0xe8a7bd00
,0xe8a7bf00,0xe49aaf00,0xe8a89100,0xe8a89400,0xeaae8100,0xe8a8a100,0xe8a8b500,0xe8a8be00
,0xe8a98500,0xe8a98d00,0xe8a99800,0xe8aaae00,0xe8aa9000,0xe8aab700,0xe8aabe00,0xe8ab9700
,0xe8abbc00,0xeaae8200,0xe8ac8a00,0xe8ac8500,0xe8ac8d00,0xe8ac9c00,0xe8ac9f00,0xe8acad00
,0xe8ad8300,0xe49c8c00,0xe8ad9100,0xe8ad9e00,0xe8adb600,0xe8adbf00,0xe8ae8100,0xe8ae8b00
,0xe8ae9400,0xe8ae9500,0xe8ae9c00,0xe8ae9e00,0xe8b0b900,0xeaae8300,0xe8b0bd00,0xeaae8400
,0xeaae8500,0xe8b18500,0xe8b18700,0xe8b18f00,0xe8b19400,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe8b19700,0xe8b1a900,0xe8b1ad00,0xe8b1b300,0xeaae8600,0xe8b29300,0xe8b29200,0xe8b29900
,0xe49da400,0xe8b29b00,0xe8b2a400,0xe8b39600,0xe8b39500,0xe8b39900,0xeaae8700,0xe8b3b000
,0xe8b3b100,0xeaae8800,0xe8b48900,0xe8b48e00,0xe8b5ac00,0xe8b68400,0xe8b69500,0xe8b6a600
,0xeaae8900,0xe8b78600,0xe8b78800,0xe8b79900,0xe8b7ac00,0xe8b88c00,0xe49fbd00,0xe8b7bd00
,0xe8b88600,0xeaae8a00,0xe8b89400,0xe8b89600,0xe8b8a100,0xe8b8a200,0xe8b8a700,0xeaae8b00
,0xe4a09600,0xe8b8b600,0xe8b8b900,0xe8b98b00,0xe8b99400,0xe8b9a200,0xe8b9ac00,0xe8b9ad00
,0xe8b9af00,0xe8ba9800,0xe8ba9e00,0xe8baae00,0xe8bab300,0xe8bab500,0xe8bab600,0xe8babb00
,0xeaae8d00,0xe8bb9100,0xe8bb9400,0xe4a18e00,0xe8bbb900,0xeaae8e00,0xe8bc8000,0
,0xe8bc8800,0xe8bc9700,0xe8bcab00,0xe8bd8000,0xe8bd8a00,0xe8bd9800,0xeaae9000,0xe8bea400
,0xe8beb400,0xefa9a600,0xe8beb600,0xeaae9100,0xe8bf8100,0xe8bf8600,0xefa8a400,0xe8bf8a00
,0xe8bf8d00,0xe8bf9300,0xe8bf9500,0xe8bfa000,0xe8bfb100,0xe8bfb500,0xe8bfbb00,0xe9808200
,0xe9808c00,0xe980b700,0xeaae9200,0xe9818300,0xe9818400,0xe9819d00,0xeaae9300,0xeaae9400
,0xe9828500,0xe9828c00,0xe9829000,0xe9989d00,0xe982a100,0xe4a2b500,0xe982b000,0xe982b600
,0xe9838300,0xe9838800,0xeaae9500,0xe9839c00,0xe9839f00,0xeaae9600,0xe983b600,0xe983b200
,0xe9848000,0xe983ab00,0xe983be00,0xe983bf00,0xe9848400,0xe9848600,0xe9849800,0xe9849c00
,0xe9849e00,0xe984b700,0xe984b900,0xe984ba00,0xe9858600,0xe9858700,0xe9859700,0xe9859900
,0xe985a100,0xe985a400,0xe985b400,0xe985b900,0xe9868500,0xe9868e00,0xe986a800,0xe986ae00
,0xe986b300,0xe986b600,0xe9878300,0xe9878400,0xe9879a00,0xeaae9700,0xeaae9800,0xe987ac00
,0xe987ae00,0xe9888100,0xe9888a00,0xe9889600,0xe9889700,0xeaae9900,0xe988b300,0xe9898200
,0xe9898700,0xe9898a00,0xe9898e00,0xe9899100,0xe9899600,0xe9899900,0xe989a000,0xe989a100
,0xe989a500,0xe989a700,0xe989a800,0xeaae9a00,0xeaae9b00,0xe989bc00,0xe989bd00,0xe989bf00
,0xe98a8900,0xe98a8d00,0xe98a9700,0xe98a9900,0xe98a9f00,0xe98aa700,0xe98aab00,0xeaae9c00
,0xeaae9d00,0xe98ab200,0xe98abf00,0xe98b8000,0xe98b8600,0xe98b8e00,0xe98b9000,0xe98b9700
,0xe98b9900,0xe98ba500,0xe98ba700,0xe98c9100,0xeaae9e00,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xeaae9f00,0xe98bb700,0xe98bb900,0xe98bbb00,0xe98c8200,0xe98c8d00,0xe98c9500,0xe98c9d00
,0xe98c9e00,0xe98ca700,0xe98ca900,0xeaaea000,0xeaaea100,0xe98d8700,0xe98d9100,0xe98d9700
,0xe98d9a00,0xe98dab00,0xe98db100,0xe98db300,0xe98ea100,0xeaaea200,0xeaaea300,0xe98e8800
,0xe98e8b00,0xe98e8f00,0xe98e9e00,0xe98fb500,0xeaaea400,0xeaaea500,0xe98fb100,0xe98f8100
,0xe98f8700,0xe98f9c00,0xe98fa200,0xe98fa700,0xe9908900,0xe9908f00,0xe9909600,0xe9909700
,0xe98fbb00,0xe990b200,0xe990b400,0xe990bb00,0xe9918500,0xeaaea600,0xeaaea700,0xe991ad00
,0xe991af00,0xe995b800,0xe995b900,0xe9968600,0xe9968c00,0xe9968d00,0xeaaea800,0xe996ab00
,0xe996b400,0xeaaea900,0xe9978800,0xeaaeaa00,0xeaaeab00,0xe998ac00,0xe998b300,0
,0xe998b400,0xeaaeac00,0xe998bc00,0xe9998100,0xe999a100,0xeaaead00,0xe99a8200,0xeaaeae00
,0xe99a9a00,0xeaaeaf00,0xe4a7a700,0xe99aa900,0xe99aaf00,0xe99ab300,0xe99aba00,0xe99abd00
,0xe4a7ba00,0xeaaeb000,0xe99b9800,0xe99b9a00,0xe99b9d00,0xe4a88400,0xe99c9400,0xe99ca300
,0xe4a8a900,0xe99cb600,0xe99d8100,0xe99d8700,0xe99d9500,0xe99d9700,0xe99d9b00,0xe99daa00
,0xeaaeb100,0xeaaeb200,0xe99e9600,0xe99e9a00,0xe99e9e00,0xe99ea200,0xe99eb100,0xe99eb200
,0xe99ebe00,0xe99f8c00,0xe99f9100,0xe99f9400,0xe99f9800,0xe99f9900,0xe99fa100,0xe99fb100
,0xe9a08400,0xe9a08d00,0xe9a08e00,0xe9a09400,0xe9a09600,0xe4aabc00,0xeaaeb300,0xe9a0a300
,0xe9a0b200,0xe9a0b300,0xe9a0a500,0xe9a18700,0xe9a1a600,0xe9a2ab00,0xe9a2ad00,0xe9a2b000
,0xeaaeb400,0xe9a2b700,0xe9a2b800,0xe9a2bb00,0xe9a2bc00,0xe9a2bf00,0xe9a38200,0xe9a38700
,0xe9a38b00,0xe9a3a000,0xeaaeb500,0xe9a3a100,0xe9a3a300,0xe9a3a500,0xe9a3aa00,0xe9a3b000
,0xe9a3b100,0xe9a3b300,0xe9a48800,0xe4acbb00,0xeaaeb600,0xe9a49600,0xe9a49700,0xeaaeb700
,0xe9a49a00,0xe9a49b00,0xe9a49c00,0xeaaeb800,0xe9a4b100,0xe9a4b200,0xe9a4b300,0xe9a4ba00
,0xe9a4bb00,0xe9a4bc00,0xe9a58000,0xe9a58100,0xe9a58600,0xe9a58d00,0xe9a58e00,0xe9a59c00
,0xe9a59f00,0xe9a5a000,0xe9a6a300,0xe9a6a600,0xe9a6b900,0xe9a6bd00,0xe9a6bf00,0xe9a78300
,0xe9a78900,0xe9a79400,0xe9a79900,0xe9a79e00,0xeaaeb900,0xe9a7b000,0xe9a7b900,0xe9a7bc00
,0xe9a88a00,0xe9a89100,0xe9a89600,0xe9a89a00,0xe9a8a000,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xe9a8b100,0xe9a8b600,0xe9a98400,0xe9a98c00,0xe9a99800,0xe4af8200,0xe9aaaf00,0xe4af8a00
,0xe9aab700,0xe4af9200,0xe9aab900,0xeaaeba00,0xe9ab8600,0xe9ab9000,0xe9ab9200,0xe9ab9500
,0xe4afa800,0xe9ab9c00,0xe9aba000,0xe9aba500,0xe9aba900,0xe9ac8300,0xe9ac8c00,0xe9ac9000
,0xe9ac9200,0xe9ac9600,0xe9ac9c00,0xe9acab00,0xe9acb300,0xe9acbd00,0xe4b0a000,0xe9ad8b00
,0xe9ada300,0xe9ada500,0xe9adab00,0xe9adac00,0xe9adb300,0xe9adb600,0xe9adb700,0xe9aea600
,0xe9aeac00,0xe9aeb100,0xeaaebb00,0xeaaebc00,0xe9aeb200,0xe9aeb800,0xe9aebe00,0xe9af8700
,0xe9afb300,0xe9af9800,0xe9af9d00,0xe9afa700,0xe9afaa00,0xe9afab00,0xe9afaf00,0xe9afae00
,0xeaaebd00,0xe9afba00,0xeaaebf00,0xe9afb700,0xeaaebe00,0xe9b09600,0xe9b09800,0
,0xe9b09900,0xe9b09a00,0xe9b09d00,0xe9b0a200,0xe9b0a700,0xe9b0a900,0xe9b0aa00,0xeaaf8000
,0xe9b0b100,0xe9b0b600,0xe9b0b700,0xe9b18500,0xe9b19c00,0xeaaf8200,0xe9b18900,0xe9b18a00
,0xeaaf8100,0xe9b19400,0xe9b19800,0xe9b19b00,0xe9b19d00,0xe9b19f00,0xe9b1a900,0xe9b1aa00
,0xe9b1ab00,0xe9b1ad00,0xe9b1ae00,0xe9b1b000,0xe9b1b200,0xe9b1b500,0xe9b1ba00,0xe9b3a600
,0xe9b3b200,0xe9b48b00,0xe9b48200,0xeaaf8300,0xe9b49100,0xe9b49700,0xe9b49800,0xeaaf8500
,0xe4b38400,0xeaaf8400,0xe9b4b200,0xe4b39100,0xe9b58200,0xe9b58a00,0xe9b59f00,0xe9b5a200
,0xeaaf8600,0xe9b5a900,0xe9b5ab00,0xeaaf8700,0xe9b5b300,0xe9b5b600,0xe9b5b700,0xe9b5be00
,0xe9b68400,0xe9b68d00,0xe9b69900,0xe9b6a100,0xe9b6bf00,0xe9b6b500,0xe9b6b900,0xe9b6bd00
,0xe9b78300,0xe9b78700,0xe9b78900,0xe9b79600,0xe9b79a00,0xe9b79f00,0xe9b7a000,0xe9b7a300
,0xe9b7b400,0xe4b48700,0xe9b88a00,0xe9b88200,0xe9b88d00,0xe9b89900,0xe9b89c00,0xe9b89d00
,0xe9b9bb00,0xeaaf8900,0xe9ba8000,0xe9ba8500,0xe9ba9b00,0xe9baa800,0xeaaf8a00,0xe9babd00
,0xeaaf8b00,0xe9bb9f00,0xe9bba700,0xe9bbae00,0xe9bbbf00,0xe9bc8200,0xe4b5b700,0xe9bc8300
,0xe9bc9700,0xe9bc9900,0xe9bcaf00,0xe9bcb700,0xe9bcba00,0xe9bcbd00,0xe9bd8100,0xe9bd8500
,0xe9bd8600,0xe9bd9300,0xe9bd9500,0xe9bd9800,0xeaaf8c00,0xe9bd9d00,0xeaaf8d00,0xe9bda900
,0xeaaf8e00,0xe9bdad00,0xe9bdb000,0xe9bdb500,0xeaaf8f00
};
static INT4 gucs4_tbl[AKX_SJIS_MAX]={
 0,0x1,0x2,0x3,0x4,0x5,0x6,0x7
,0x8,0x9,0xa,0xb,0xc,0xd,0xe,0xf
,0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17
,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f
,0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27
,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f
,0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37
,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f
,0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47
,0x48,0x49,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f
,0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57
,0x58,0x59,0x5a,0x5b,0xa5,0x5d,0x5e,0x5f
,0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67
,0x68,0x69,0x6a,0x6b,0x6c,0x6d,0x6e,0x6f
,0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77
,0x78,0x79,0x7a,0x7b,0x7c,0x7d,0x203e,0x7f
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0xff61,0xff62,0xff63,0xff64,0xff65,0xff66,0xff67
,0xff68,0xff69,0xff6a,0xff6b,0xff6c,0xff6d,0xff6e,0xff6f
,0xff70,0xff71,0xff72,0xff73,0xff74,0xff75,0xff76,0xff77
,0xff78,0xff79,0xff7a,0xff7b,0xff7c,0xff7d,0xff7e,0xff7f
,0xff80,0xff81,0xff82,0xff83,0xff84,0xff85,0xff86,0xff87
,0xff88,0xff89,0xff8a,0xff8b,0xff8c,0xff8d,0xff8e,0xff8f
,0xff90,0xff91,0xff92,0xff93,0xff94,0xff95,0xff96,0xff97
,0xff98,0xff99,0xff9a,0xff9b,0xff9c,0xff9d,0xff9e,0xff9f
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x3000,0x3001,0x3002,0xff0c,0xff0e,0x30fb,0xff1a,0xff1b
,0xff1f,0xff01,0x309b,0x309c,0xb4,0xff40,0xa8,0xff3e
,0xffe3,0xff3f,0x30fd,0x30fe,0x309d,0x309e,0x3003,0x4edd
,0x3005,0x3006,0x3007,0x30fc,0x2014,0x2010,0xff0f,0x5c
,0x301c,0x2016,0xff5c,0x2026,0x2025,0x2018,0x2019,0x201c
,0x201d,0xff08,0xff09,0x3014,0x3015,0xff3b,0xff3d,0xff5b
,0xff5d,0x3008,0x3009,0x300a,0x300b,0x300c,0x300d,0x300e
,0x300f,0x3010,0x3011,0xff0b,0x2212,0xb1,0xd7,0
,0xf7,0xff1d,0x2260,0xff1c,0xff1e,0x2266,0x2267,0x221e
,0x2234,0x2642,0x2640,0xb0,0x2032,0x2033,0x2103,0xffe5
,0xff04,0xa2,0xa3,0xff05,0xff03,0xff06,0xff0a,0xff20
,0xa7,0x2606,0x2605,0x25cb,0x25cf,0x25ce,0x25c7,0x25c6
,0x25a1,0x25a0,0x25b3,0x25b2,0x25bd,0x25bc,0x203b,0x3012
,0x2192,0x2190,0x2191,0x2193,0x3013,0xff07,0xff02,0xff0d
,0x7e,0x3033,0x3034,0x3035,0x303b,0x303c,0x303d,0x31c0
,0x2208,0x220b,0x2286,0x2287,0x2282,0x2283,0x222a,0x2229
,0x2284,0x2285,0x228a,0x228b,0x2209,0x2205,0x2305,0x2306
,0x2227,0x2228,0xac,0x21d2,0x21d4,0x2200,0x2203,0x2295
,0x2296,0x2297,0x2225,0x2226,0x2050,0x2051,0x3018,0x3019
,0x3016,0x3017,0x2220,0x22a5,0x2312,0x2202,0x2207,0x2261
,0x2252,0x226a,0x226b,0x221a,0x223d,0x221d,0x2235,0x222b
,0x222c,0x2262,0x2243,0x2245,0x2248,0x2276,0x2277,0x2194
,0x212b,0x2030,0x266f,0x266d,0x266a,0x2020,0x2021,0xb6
,0x266e,0x266b,0x266c,0x2669,0x25ef,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x25b7,0x25b6,0x25c1,0x25c0,0x2197,0x2198,0x2196,0x2199
,0x21c4,0x21e8,0x21e6,0x21e7,0x21e9,0x21f4,0x21f5,0xff10
,0xff11,0xff12,0xff13,0xff14,0xff15,0xff16,0xff17,0xff18
,0xff19,0x2052,0x25c9,0x31c1,0x31c2,0x31c3,0x25e6,0x2022
,0xff21,0xff22,0xff23,0xff24,0xff25,0xff26,0xff27,0xff28
,0xff29,0xff2a,0xff2b,0xff2c,0xff2d,0xff2e,0xff2f,0xff30
,0xff31,0xff32,0xff33,0xff34,0xff35,0xff36,0xff37,0xff38
,0xff39,0xff3a,0x2213,0x2135,0x210f,0x33cb,0x2113,0
,0x2127,0xff41,0xff42,0xff43,0xff44,0xff45,0xff46,0xff47
,0xff48,0xff49,0xff4a,0xff4b,0xff4c,0xff4d,0xff4e,0xff4f
,0xff50,0xff51,0xff52,0xff53,0xff54,0xff55,0xff56,0xff57
,0xff58,0xff59,0xff5a,0x204e,0x2013,0x239b,0x239c,0x3041
,0x3042,0x3043,0x3044,0x3045,0x3046,0x3047,0x3048,0x3049
,0x304a,0x304b,0x304c,0x304d,0x304e,0x304f,0x3050,0x3051
,0x3052,0x3053,0x3054,0x3055,0x3056,0x3057,0x3058,0x3059
,0x305a,0x305b,0x305c,0x305d,0x305e,0x305f,0x3060,0x3061
,0x3062,0x3063,0x3064,0x3065,0x3066,0x3067,0x3068,0x3069
,0x306a,0x306b,0x306c,0x306d,0x306e,0x306f,0x3070,0x3071
,0x3072,0x3073,0x3074,0x3075,0x3076,0x3077,0x3078,0x3079
,0x307a,0x307b,0x307c,0x307d,0x307e,0x307f,0x3080,0x3081
,0x3082,0x3083,0x3084,0x3085,0x3086,0x3087,0x3088,0x3089
,0x308a,0x308b,0x308c,0x308d,0x308e,0x308f,0x3090,0x3091
,0x3092,0x3093,0x3094,0x3095,0x3096,0x31d0,0x31d1,0x31d2
,0x31d3,0x31d4,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x30a1,0x30a2,0x30a3,0x30a4,0x30a5,0x30a6,0x30a7,0x30a8
,0x30a9,0x30aa,0x30ab,0x30ac,0x30ad,0x30ae,0x30af,0x30b0
,0x30b1,0x30b2,0x30b3,0x30b4,0x30b5,0x30b6,0x30b7,0x30b8
,0x30b9,0x30ba,0x30bb,0x30bc,0x30bd,0x30be,0x30bf,0x30c0
,0x30c1,0x30c2,0x30c3,0x30c4,0x30c5,0x30c6,0x30c7,0x30c8
,0x30c9,0x30ca,0x30cb,0x30cc,0x30cd,0x30ce,0x30cf,0x30d0
,0x30d1,0x30d2,0x30d3,0x30d4,0x30d5,0x30d6,0x30d7,0x30d8
,0x30d9,0x30da,0x30db,0x30dc,0x30dd,0x30de,0x30df,0
,0x30e0,0x30e1,0x30e2,0x30e3,0x30e4,0x30e5,0x30e6,0x30e7
,0x30e8,0x30e9,0x30ea,0x30eb,0x30ec,0x30ed,0x30ee,0x30ef
,0x30f0,0x30f1,0x30f2,0x30f3,0x30f4,0x30f5,0x30f6,0x31e0
,0x31e1,0x31e2,0x31e3,0x31e4,0x31e5,0x31e6,0x31e7,0x391
,0x392,0x393,0x394,0x395,0x396,0x397,0x398,0x399
,0x39a,0x39b,0x39c,0x39d,0x39e,0x39f,0x3a0,0x3a1
,0x3a3,0x3a4,0x3a5,0x3a6,0x3a7,0x3a8,0x3a9,0x2664
,0x2660,0x2662,0x2666,0x2661,0x2665,0x2667,0x2663,0x3b1
,0x3b2,0x3b3,0x3b4,0x3b5,0x3b6,0x3b7,0x3b8,0x3b9
,0x3ba,0x3bb,0x3bc,0x3bd,0x3be,0x3bf,0x3c0,0x3c1
,0x3c3,0x3c4,0x3c5,0x3c6,0x3c7,0x3c8,0x3c9,0x3c2
,0x23ac,0x23ad,0x23ae,0x23af,0x23b0,0x23b1,0x23b2,0x23b3
,0x23b4,0x23b5,0x2616,0x2617,0x3020,0x260e,0x2600,0x2601
,0x2602,0x2603,0x2668,0x25b1,0x31e8,0x31e9,0x31ea,0x31eb
,0x31ec,0x31ed,0x31ee,0x31ef,0x31f0,0x31f1,0x31f2,0x31f3
,0x31f4,0x31f5,0x31f6,0x31f7,0x31f8,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x410,0x411,0x412,0x413,0x414,0x415,0x401,0x416
,0x417,0x418,0x419,0x41a,0x41b,0x41c,0x41d,0x41e
,0x41f,0x420,0x421,0x422,0x423,0x424,0x425,0x426
,0x427,0x428,0x429,0x42a,0x42b,0x42c,0x42d,0x42e
,0x42f,0x239d,0x239e,0x239f,0x23a0,0x23a1,0x23a2,0x23a3
,0x23a4,0x23a5,0x23a6,0x23a7,0x23a8,0x23a9,0x23aa,0x23ab
,0x430,0x431,0x432,0x433,0x434,0x435,0x451,0x436
,0x437,0x438,0x439,0x43a,0x43b,0x43c,0x43d,0
,0x43e,0x43f,0x440,0x441,0x442,0x443,0x444,0x445
,0x446,0x447,0x448,0x449,0x44a,0x44b,0x44c,0x44d
,0x44e,0x44f,0x30f7,0x30f8,0x30f9,0x30fa,0x22da,0x22db
,0x2153,0x2154,0x2155,0x2713,0x2318,0x2423,0x2618,0x2500
,0x2502,0x250c,0x2510,0x2518,0x2514,0x251c,0x252c,0x2524
,0x2534,0x253c,0x2501,0x2503,0x250f,0x2513,0x251b,0x2517
,0x2523,0x2533,0x252b,0x253b,0x254b,0x2520,0x252f,0x2528
,0x2537,0x253f,0x251d,0x2530,0x2525,0x2538,0x2542,0x27d0
,0x27d1,0x27d2,0x27d3,0x27d4,0x27d5,0x27d6,0x27d7,0x27d8
,0x27d9,0x27da,0x27db,0x27dc,0x27dd,0x27de,0x27df,0x27e0
,0x27e1,0x27e2,0x27e3,0x27e4,0x27e5,0x27e6,0x27e7,0x27e8
,0x27e9,0x27ea,0x27eb,0x27ec,0x27ed,0,0,0
,0,0,0,0,0,0x25d0,0x25d1,0x25d2
,0x25d3,0x203c,0x2047,0x2048,0x2049,0x1cd,0x1ce,0x1d0
,0x1e3e,0x1e3f,0x1f8,0x1f9,0x1d1,0x1d2,0x1d4,0x1d6
,0x1d8,0x1da,0x1dc,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x20ac,0xa0,0xa1,0xa4,0xa6,0xa9,0xaa,0xab
,0xad,0xae,0xaf,0xb2,0xb3,0xb7,0xb8,0xb9
,0xba,0xbb,0xbc,0xbd,0xbe,0xbf,0xc0,0xc1
,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9
,0xca,0xcb,0xcc,0xcd,0xce,0xcf,0xd0,0xd1
,0xd2,0xd3,0xd4,0xd5,0xd6,0xd8,0xd9,0xda
,0xdb,0xdc,0xdd,0xde,0xdf,0xe0,0xe1,0xe2
,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0
,0xea,0xeb,0xec,0xed,0xee,0xef,0xf0,0xf1
,0xf2,0xf3,0xf4,0xf5,0xf6,0xf8,0xf9,0xfa
,0xfb,0xfc,0xfd,0xfe,0xff,0x100,0x12a,0x16a
,0x112,0x14c,0x101,0x12b,0x16b,0x113,0x14d,0x104
,0x2d8,0x141,0x13d,0x15a,0x160,0x15e,0x164,0x179
,0x17d,0x17b,0x105,0x2db,0x142,0x13e,0x15b,0x2c7
,0x161,0x15f,0x165,0x17a,0x2dd,0x17e,0x17c,0x154
,0x102,0x139,0x106,0x10c,0x118,0x11a,0x10e,0x143
,0x147,0x150,0x158,0x16e,0x170,0x162,0x155,0x103
,0x13a,0x107,0x10d,0x119,0x11b,0x10f,0x111,0x144
,0x148,0x151,0x159,0x16f,0x171,0x163,0x2d9,0x108
,0x11c,0x124,0x134,0x15c,0x16c,0x109,0x11d,0x125
,0x135,0x15d,0x16d,0x271,0x28b,0x27e,0x283,0x292
,0x26c,0x26e,0x279,0x288,0x256,0x273,0x27d,0x282
,0x290,0x27b,0x26d,0x25f,0x272,0x29d,0x28e,0x261
,0x14b,0x270,0x281,0x127,0x295,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x294,0x266,0x298,0x1c2,0x253,0x257,0x284,0x260
,0x193,0x153,0x152,0x268,0x289,0x258,0x275,0x259
,0x25c,0x25e,0x250,0x26f,0x28a,0x264,0x28c,0x254
,0x251,0x252,0x28d,0x265,0x2a2,0x2a1,0x255,0x291
,0x27a,0x267,0x25a,0x234,0x1fd,0x1f70,0x1f71,0x235
,0x236,0x237,0x238,0x239,0x23a,0x23b,0x23c,0x1f72
,0x1f73,0x361,0x2c8,0x2cc,0x2d0,0x2d1,0x306,0x203f
,0x30b,0x301,0x304,0x300,0x30f,0x30c,0x302,0
,0x2e5,0x2e6,0x2e7,0x2e8,0x2e9,0x2ef,0x2f0,0x325
,0x32c,0x339,0x31c,0x31f,0x320,0x308,0x33d,0x329
,0x32f,0x2de,0x324,0x330,0x33c,0x334,0x31d,0x31e
,0x318,0x319,0x32a,0x33a,0x33b,0x303,0x31a,0x2776
,0x2777,0x2778,0x2779,0x277a,0x277b,0x277c,0x277d,0x277e
,0x277f,0x27ee,0x27ef,0x27f0,0x27f1,0x27f2,0x27f3,0x27f4
,0x27f5,0x27f6,0x27f7,0x2170,0x2171,0x2172,0x2173,0x2174
,0x2175,0x2176,0x2177,0x2178,0x2179,0x217a,0x217b,0x24d0
,0x24d1,0x24d2,0x24d3,0x24d4,0x24d5,0x24d6,0x24d7,0x24d8
,0x24d9,0x24da,0x24db,0x24dc,0x24dd,0x24de,0x24df,0x24e0
,0x24e1,0x24e2,0x24e3,0x24e4,0x24e5,0x24e6,0x24e7,0x24e8
,0x24e9,0x32d0,0x32d1,0x32d2,0x32d3,0x32d4,0x32d5,0x32d6
,0x32d7,0x32d8,0x32d9,0x32da,0x32db,0x32dc,0x32dd,0x32de
,0x32df,0x32e0,0x32e1,0x32e2,0x32e3,0x32fa,0x32e9,0x32e5
,0x32ed,0x32ec,0,0,0,0,0,0
,0,0,0,0x2053,0x2042,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x2460,0x2461,0x2462,0x2463,0x2464,0x2465,0x2466,0x2467
,0x2468,0x2469,0x246a,0x246b,0x246c,0x246d,0x246e,0x246f
,0x2470,0x2471,0x2472,0x2473,0x2160,0x2161,0x2162,0x2163
,0x2164,0x2165,0x2166,0x2167,0x2168,0x2169,0x216a,0x3349
,0x3314,0x3322,0x334d,0x3318,0x3327,0x3303,0x3336,0x3351
,0x3357,0x330d,0x3326,0x3323,0x332b,0x334a,0x333b,0x339c
,0x339d,0x339e,0x338e,0x338f,0x33c4,0x33a1,0x216b,0
,0,0,0,0,0,0,0x337b,0
,0x301d,0x301f,0x2116,0x33cd,0x2121,0x32a4,0x32a5,0x32a6
,0x32a7,0x32a8,0x3231,0x3232,0x3239,0x337e,0x337d,0x337c
,0,0,0,0x222e,0,0,0,0
,0x221f,0x22bf,0,0,0,0x2756,0x261e,0x4ff1
,0xaaa2,0x3402,0x4e28,0x4e2f,0x4e30,0x4e8d,0x4ee1,0x4efd
,0x4eff,0x4f03,0x4f0b,0x4f60,0x4f48,0x4f49,0x4f56,0x4f5f
,0x4f6a,0x4f6c,0x4f7e,0x4f8a,0x4f94,0x4f97,0xfa30,0x4fc9
,0x4fe0,0x5001,0x5002,0x500e,0x5018,0x5027,0x502e,0x5040
,0x503b,0x5041,0x5094,0x50cc,0x50f2,0x50d0,0x50e6,0xfa31
,0x5106,0x5103,0x510b,0x511e,0x5135,0x514a,0xfa32,0x5155
,0x5157,0x34b5,0x519d,0x51c3,0x51ca,0x51de,0x51e2,0x51ee
,0x5201,0x34db,0x5213,0x5215,0x5249,0x5257,0x5261,0x5293
,0x52c8,0xfa33,0x52cc,0x52d0,0x52d6,0x52db,0xfa34,0x52f0
,0x52fb,0x5300,0x5307,0x531c,0xfa35,0x5361,0x5363,0x537d
,0x5393,0x539d,0x53b2,0x5412,0x5427,0x544d,0x549c,0x546b
,0x5474,0x547f,0x5488,0x5496,0x54a1,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x54a9,0x54c6,0x54ff,0x550e,0x552b,0x5535,0x5550,0x555e
,0x5581,0x5586,0x558e,0xfa36,0x55ad,0x55ce,0xfa37,0x5608
,0x560e,0x563b,0x5649,0x5676,0x5666,0xfa38,0x566f,0x5671
,0x5672,0x5699,0x569e,0x56a9,0x56ac,0x56b3,0x56c9,0x56ca
,0x570a,0xaabe,0x5721,0x572f,0x5733,0x5734,0x5770,0x5777
,0x577c,0x579c,0xfa0f,0xaac4,0x57b8,0x57c7,0x57c8,0x57cf
,0x57e4,0x57ed,0x57f5,0x57f6,0x57ff,0x5809,0xfa10,0x5861
,0x5864,0xfa39,0x587c,0x5889,0x589e,0xfa3a,0x58a9,0
,0xaac9,0x58d2,0x58ce,0x58d4,0x58da,0x58e0,0x58e9,0x590c
,0x8641,0x595d,0x596d,0x598b,0x5992,0x59a4,0x59c3,0x59d2
,0x59dd,0x5a13,0x5a23,0x5a67,0x5a6d,0x5a77,0x5a7e,0x5a84
,0x5a9e,0x5aa7,0x5ac4,0xaad0,0x5b19,0x5b25,0x525d,0x4e9c
,0x5516,0x5a03,0x963f,0x54c0,0x611b,0x6328,0x59f6,0x9022
,0x8475,0x831c,0x7a50,0x60aa,0x63e1,0x6e25,0x65ed,0x8466
,0x82a6,0x9bf5,0x6893,0x5727,0x65a1,0x6271,0x5b9b,0x59d0
,0x867b,0x98f4,0x7d62,0x7dbe,0x9b8e,0x6216,0x7c9f,0x88b7
,0x5b89,0x5eb5,0x6309,0x6697,0x6848,0x95c7,0x978d,0x674f
,0x4ee5,0x4f0a,0x4f4d,0x4f9d,0x5049,0x56f2,0x5937,0x59d4
,0x5a01,0x5c09,0x60df,0x610f,0x6170,0x6613,0x6905,0x70ba
,0x754f,0x7570,0x79fb,0x7dad,0x7def,0x80c3,0x840e,0x8863
,0x8b02,0x9055,0x907a,0x533b,0x4e95,0x4ea5,0x57df,0x80b2
,0x90c1,0x78ef,0x4e00,0x58f1,0x6ea2,0x9038,0x7a32,0x8328
,0x828b,0x9c2f,0x5141,0x5370,0x54bd,0x54e1,0x56e0,0x59fb
,0x5f15,0x98f2,0x6deb,0x80e4,0x852d,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x9662,0x9670,0x96a0,0x97fb,0x540b,0x53f3,0x5b87,0x70cf
,0x7fbd,0x8fc2,0x96e8,0x536f,0x9d5c,0x7aba,0x4e11,0x7893
,0x81fc,0x6e26,0x5618,0x5504,0x6b1d,0x851a,0x9c3b,0x59e5
,0x53a9,0x6d66,0x74dc,0x958f,0x5642,0x4e91,0x904b,0x96f2
,0x834f,0x990c,0x53e1,0x55b6,0x5b30,0x5f71,0x6620,0x66f3
,0x6804,0x6c38,0x6cf3,0x6d29,0x745b,0x76c8,0x7a4e,0x9834
,0x82f1,0x885b,0x8a60,0x92ed,0x6db2,0x75ab,0x76ca,0x99c5
,0x60a6,0x8b01,0x8d8a,0x95b2,0x698e,0x53ad,0x5186,0
,0x5712,0x5830,0x5944,0x5bb4,0x5ef6,0x6028,0x63a9,0x63f4
,0x6cbf,0x6f14,0x708e,0x7114,0x7159,0x71d5,0x733f,0x7e01
,0x8276,0x82d1,0x8597,0x9060,0x925b,0x9d1b,0x5869,0x65bc
,0x6c5a,0x7525,0x51f9,0x592e,0x5965,0x5f80,0x5fdc,0x62bc
,0x65fa,0x6a2a,0x6b27,0x6bb4,0x738b,0x7fc1,0x8956,0x9d2c
,0x9d0e,0x9ec4,0x5ca1,0x6c96,0x837b,0x5104,0x5c4b,0x61b6
,0x81c6,0x6876,0x7261,0x4e59,0x4ffa,0x5378,0x6069,0x6e29
,0x7a4f,0x97f3,0x4e0b,0x5316,0x4eee,0x4f55,0x4f3d,0x4fa1
,0x4f73,0x52a0,0x53ef,0x5609,0x590f,0x5ac1,0x5bb6,0x5be1
,0x79d1,0x6687,0x679c,0x67b6,0x6b4c,0x6cb3,0x706b,0x73c2
,0x798d,0x79be,0x7a3c,0x7b87,0x82b1,0x82db,0x8304,0x8377
,0x83ef,0x83d3,0x8766,0x8ab2,0x5629,0x8ca8,0x8fe6,0x904e
,0x971e,0x868a,0x4fc4,0x5ce8,0x6211,0x7259,0x753b,0x81e5
,0x82bd,0x86fe,0x8cc0,0x96c5,0x9913,0x99d5,0x4ecb,0x4f1a
,0x89e3,0x56de,0x584a,0x58ca,0x5efb,0x5feb,0x602a,0x6094
,0x6062,0x61d0,0x6212,0x62d0,0x6539,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x9b41,0x6666,0x68b0,0x6d77,0x7070,0x754c,0x7686,0x7d75
,0x82a5,0x87f9,0x958b,0x968e,0x8c9d,0x51f1,0x52be,0x5916
,0x54b3,0x5bb3,0x5d16,0x6168,0x6982,0x6daf,0x788d,0x84cb
,0x8857,0x8a72,0x93a7,0x9ab8,0x6d6c,0x99a8,0x86d9,0x57a3
,0x67ff,0x86ce,0x920e,0x5283,0x5687,0x5404,0x5ed3,0x62e1
,0x64b9,0x683c,0x6838,0x6bbb,0x7372,0x78ba,0x7a6b,0x899a
,0x89d2,0x8d6b,0x8f03,0x90ed,0x95a3,0x9694,0x9769,0x5b66
,0x5cb3,0x697d,0x984d,0x984e,0x639b,0x7b20,0x6a2b,0
,0x6a7f,0x68b6,0x9c0d,0x6f5f,0x5272,0x559d,0x6070,0x62ec
,0x6d3b,0x6e07,0x6ed1,0x845b,0x8910,0x8f44,0x4e14,0x9c39
,0x53f6,0x691b,0x6a3a,0x9784,0x682a,0x515c,0x7ac3,0x84b2
,0x91dc,0x938c,0x565b,0x9d28,0x6822,0x8305,0x8431,0x7ca5
,0x5208,0x82c5,0x74e6,0x4e7e,0x4f83,0x51a0,0x5bd2,0x520a
,0x52d8,0x52e7,0x5dfb,0x559a,0x582a,0x59e6,0x5b8c,0x5b98
,0x5bdb,0x5e72,0x5e79,0x60a3,0x611f,0x6163,0x61be,0x63db
,0x6562,0x67d1,0x6853,0x68fa,0x6b3e,0x6b53,0x6c57,0x6f22
,0x6f97,0x6f45,0x74b0,0x7518,0x76e3,0x770b,0x7aff,0x7ba1
,0x7c21,0x7de9,0x7f36,0x7ff0,0x809d,0x8266,0x839e,0x89b3
,0x8acc,0x8cab,0x9084,0x9451,0x9593,0x9591,0x95a2,0x9665
,0x97d3,0x9928,0x8218,0x4e38,0x542b,0x5cb8,0x5dcc,0x73a9
,0x764c,0x773c,0x5ca9,0x7feb,0x8d0b,0x96c1,0x9811,0x9854
,0x9858,0x4f01,0x4f0e,0x5371,0x559c,0x5668,0x57fa,0x5947
,0x5b09,0x5bc4,0x5c90,0x5e0c,0x5e7e,0x5fcc,0x63ee,0x673a
,0x65d7,0x65e2,0x671f,0x68cb,0x68c4,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x6a5f,0x5e30,0x6bc5,0x6c17,0x6c7d,0x757f,0x7948,0x5b63
,0x7a00,0x7d00,0x5fbd,0x898f,0x8a18,0x8cb4,0x8d77,0x8ecc
,0x8f1d,0x98e2,0x9a0e,0x9b3c,0x4e80,0x507d,0x5100,0x5993
,0x5b9c,0x622f,0x6280,0x64ec,0x6b3a,0x72a0,0x7591,0x7947
,0x7fa9,0x87fb,0x8abc,0x8b70,0x63ac,0x83ca,0x97a0,0x5409
,0x5403,0x55ab,0x6854,0x6a58,0x8a70,0x7827,0x6775,0x9ecd
,0x5374,0x5ba2,0x811a,0x8650,0x9006,0x4e18,0x4e45,0x4ec7
,0x4f11,0x53ca,0x5438,0x5bae,0x5f13,0x6025,0x6551,0
,0x673d,0x6c42,0x6c72,0x6ce3,0x7078,0x7403,0x7a76,0x7aae
,0x7b08,0x7d1a,0x7cfe,0x7d66,0x65e7,0x725b,0x53bb,0x5c45
,0x5de8,0x62d2,0x62e0,0x6319,0x6e20,0x865a,0x8a31,0x8ddd
,0x92f8,0x6f01,0x79a6,0x9b5a,0x4ea8,0x4eab,0x4eac,0x4f9b
,0x4fa0,0x50d1,0x5147,0x7af6,0x5171,0x51f6,0x5354,0x5321
,0x537f,0x53eb,0x55ac,0x5883,0x5ce1,0x5f37,0x5f4a,0x602f
,0x6050,0x606d,0x631f,0x6559,0x6a4b,0x6cc1,0x72c2,0x72ed
,0x77ef,0x80f8,0x8105,0x8208,0x854e,0x90f7,0x93e1,0x97ff
,0x9957,0x9a5a,0x4ef0,0x51dd,0x5c2d,0x6681,0x696d,0x5c40
,0x66f2,0x6975,0x7389,0x6850,0x7c81,0x50c5,0x52e4,0x5747
,0x5dfe,0x9326,0x65a4,0x6b23,0x6b3d,0x7434,0x7981,0x79bd
,0x7b4b,0x7dca,0x82b9,0x83cc,0x887f,0x895f,0x8b39,0x8fd1
,0x91d1,0x541f,0x9280,0x4e5d,0x5036,0x53e5,0x533a,0x72d7
,0x7396,0x77e9,0x82e6,0x8eaf,0x99c6,0x99c8,0x99d2,0x5177
,0x611a,0x865e,0x55b0,0x7a7a,0x5076,0x5bd3,0x9047,0x9685
,0x4e32,0x6adb,0x91e7,0x5c51,0x5c48,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x6398,0x7a9f,0x6c93,0x9774,0x8f61,0x7aaa,0x718a,0x9688
,0x7c82,0x6817,0x7e70,0x6851,0x936c,0x52f2,0x541b,0x85ab
,0x8a13,0x7fa4,0x8ecd,0x90e1,0x5366,0x8888,0x7941,0x4fc2
,0x50be,0x5211,0x5144,0x5553,0x572d,0x73ea,0x578b,0x5951
,0x5f62,0x5f84,0x6075,0x6176,0x6167,0x61a9,0x63b2,0x643a
,0x656c,0x666f,0x6842,0x6e13,0x7566,0x7a3d,0x7cfb,0x7d4c
,0x7d99,0x7e4b,0x7f6b,0x830e,0x834a,0x86cd,0x8a08,0x8a63
,0x8b66,0x8efd,0x981a,0x9d8f,0x82b8,0x8fce,0x9be8,0
,0x5287,0x621f,0x6483,0x6fc0,0x9699,0x6841,0x5091,0x6b20
,0x6c7a,0x6f54,0x7a74,0x7d50,0x8840,0x8a23,0x6708,0x4ef6
,0x5039,0x5026,0x5065,0x517c,0x5238,0x5263,0x55a7,0x570f
,0x5805,0x5acc,0x5efa,0x61b2,0x61f8,0x62f3,0x6372,0x691c
,0x6a29,0x727d,0x72ac,0x732e,0x7814,0x786f,0x7d79,0x770c
,0x80a9,0x898b,0x8b19,0x8ce2,0x8ed2,0x9063,0x9375,0x967a
,0x9855,0x9a13,0x9e78,0x5143,0x539f,0x53b3,0x5e7b,0x5f26
,0x6e1b,0x6e90,0x7384,0x73fe,0x7d43,0x8237,0x8a00,0x8afa
,0x9650,0x4e4e,0x500b,0x53e4,0x547c,0x56fa,0x59d1,0x5b64
,0x5df1,0x5eab,0x5f27,0x6238,0x6545,0x67af,0x6e56,0x72d0
,0x7cca,0x88b4,0x80a1,0x80e1,0x83f0,0x864e,0x8a87,0x8de8
,0x9237,0x96c7,0x9867,0x9f13,0x4e94,0x4e92,0x4f0d,0x5348
,0x5449,0x543e,0x5a2f,0x5f8c,0x5fa1,0x609f,0x68a7,0x6a8e
,0x745a,0x7881,0x8a9e,0x8aa4,0x8b77,0x9190,0x4e5e,0x9bc9
,0x4ea4,0x4f7c,0x4faf,0x5019,0x5016,0x5149,0x516c,0x529f
,0x52b9,0x52fe,0x539a,0x53e3,0x5411,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x540e,0x5589,0x5751,0x57a2,0x597d,0x5b54,0x5b5d,0x5b8f
,0x5de5,0x5de7,0x5df7,0x5e78,0x5e83,0x5e9a,0x5eb7,0x5f18
,0x6052,0x614c,0x6297,0x62d8,0x63a7,0x653b,0x6602,0x6643
,0x66f4,0x676d,0x6821,0x6897,0x69cb,0x6c5f,0x6d2a,0x6d69
,0x6e2f,0x6e9d,0x7532,0x7687,0x786c,0x7a3f,0x7ce0,0x7d05
,0x7d18,0x7d5e,0x7db1,0x8015,0x8003,0x80af,0x80b1,0x8154
,0x818f,0x822a,0x8352,0x884c,0x8861,0x8b1b,0x8ca2,0x8cfc
,0x90ca,0x9175,0x9271,0x783f,0x92fc,0x95a4,0x964d,0
,0x9805,0x9999,0x9ad8,0x9d3b,0x525b,0x52ab,0x53f7,0x5408
,0x58d5,0x62f7,0x6fe0,0x8c6a,0x8f5f,0x9eb9,0x514b,0x523b
,0x544a,0x56fd,0x7a40,0x9177,0x9d60,0x9ed2,0x7344,0x6f09
,0x8170,0x7511,0x5ffd,0x60da,0x9aa8,0x72db,0x8fbc,0x6b64
,0x9803,0x4eca,0x56f0,0x5764,0x58be,0x5a5a,0x6068,0x61c7
,0x660f,0x6606,0x6839,0x68b1,0x6df7,0x75d5,0x7d3a,0x826e
,0x9b42,0x4e9b,0x4f50,0x53c9,0x5506,0x5d6f,0x5de6,0x5dee
,0x67fb,0x6c99,0x7473,0x7802,0x8a50,0x9396,0x88df,0x5750
,0x5ea7,0x632b,0x50b5,0x50ac,0x518d,0x6700,0x54c9,0x585e
,0x59bb,0x5bb0,0x5f69,0x624d,0x63a1,0x683d,0x6b73,0x6e08
,0x707d,0x91c7,0x7280,0x7815,0x7826,0x796d,0x658e,0x7d30
,0x83dc,0x88c1,0x8f09,0x969b,0x5264,0x5728,0x6750,0x7f6a
,0x8ca1,0x51b4,0x5742,0x962a,0x583a,0x698a,0x80b4,0x54b2
,0x5d0e,0x57fc,0x7895,0x9dfa,0x4f5c,0x524a,0x548b,0x643e
,0x6628,0x6714,0x67f5,0x7a84,0x7b56,0x7d22,0x932f,0x685c
,0x9bad,0x7b39,0x5319,0x518a,0x5237,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x5bdf,0x62f6,0x64ae,0x64e6,0x672d,0x6bba,0x85a9,0x96d1
,0x7690,0x9bd6,0x634c,0x9306,0x9bab,0x76bf,0x6652,0x4e09
,0x5098,0x53c2,0x5c71,0x60e8,0x6492,0x6563,0x685f,0x71e6
,0x73ca,0x7523,0x7b97,0x7e82,0x8695,0x8b83,0x8cdb,0x9178
,0x9910,0x65ac,0x66ab,0x6b8b,0x4ed5,0x4ed4,0x4f3a,0x4f7f
,0x523a,0x53f8,0x53f2,0x55e3,0x56db,0x58eb,0x59cb,0x59c9
,0x59ff,0x5b50,0x5c4d,0x5e02,0x5e2b,0x5fd7,0x601d,0x6307
,0x652f,0x5b5c,0x65af,0x65bd,0x65e8,0x679d,0x6b62,0
,0x6b7b,0x6c0f,0x7345,0x7949,0x79c1,0x7cf8,0x7d19,0x7d2b
,0x80a2,0x8102,0x81f3,0x8996,0x8a5e,0x8a69,0x8a66,0x8a8c
,0x8aee,0x8cc7,0x8cdc,0x96cc,0x98fc,0x6b6f,0x4e8b,0x4f3c
,0x4f8d,0x5150,0x5b57,0x5bfa,0x6148,0x6301,0x6642,0x6b21
,0x6ecb,0x6cbb,0x723e,0x74bd,0x75d4,0x78c1,0x793a,0x800c
,0x8033,0x81ea,0x8494,0x8f9e,0x6c50,0x9e7f,0x5f0f,0x8b58
,0x9d2b,0x7afa,0x8ef8,0x5b8d,0x96eb,0x4e03,0x53f1,0x57f7
,0x5931,0x5ac9,0x5ba4,0x6089,0x6e7f,0x6f06,0x75be,0x8cea
,0x5b9f,0x8500,0x7be0,0x5072,0x67f4,0x829d,0x5c61,0x854a
,0x7e1e,0x820e,0x5199,0x5c04,0x6368,0x8d66,0x659c,0x716e
,0x793e,0x7d17,0x8005,0x8b1d,0x8eca,0x906e,0x86c7,0x90aa
,0x501f,0x52fa,0x5c3a,0x6753,0x707c,0x7235,0x914c,0x91c8
,0x932b,0x82e5,0x5bc2,0x5f31,0x60f9,0x4e3b,0x53d6,0x5b88
,0x624b,0x6731,0x6b8a,0x72e9,0x73e0,0x7a2e,0x816b,0x8da3
,0x9152,0x9996,0x5112,0x53d7,0x546a,0x5bff,0x6388,0x6a39
,0x7dac,0x9700,0x56da,0x53ce,0x5468,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x5b97,0x5c31,0x5dde,0x4fee,0x6101,0x62fe,0x6d32,0x79c0
,0x79cb,0x7d42,0x7e4d,0x7fd2,0x81ed,0x821f,0x8490,0x8846
,0x8972,0x8b90,0x8e74,0x8f2f,0x9031,0x914b,0x916c,0x96c6
,0x919c,0x4ec0,0x4f4f,0x5145,0x5341,0x5f93,0x620e,0x67d4
,0x6c41,0x6e0b,0x7363,0x7e26,0x91cd,0x9283,0x53d4,0x5919
,0x5bbf,0x6dd1,0x795d,0x7e2e,0x7c9b,0x587e,0x719f,0x51fa
,0x8853,0x8ff0,0x4fca,0x5cfb,0x6625,0x77ac,0x7ae3,0x821c
,0x99ff,0x51c6,0x5faa,0x65ec,0x696f,0x6b89,0x6df3,0
,0x6e96,0x6f64,0x76fe,0x7d14,0x5de1,0x9075,0x9187,0x9806
,0x51e6,0x521d,0x6240,0x6691,0x66d9,0x6e1a,0x5eb6,0x7dd2
,0x7f72,0x66f8,0x85af,0x85f7,0x8af8,0x52a9,0x53d9,0x5973
,0x5e8f,0x5f90,0x6055,0x92e4,0x9664,0x50b7,0x511f,0x52dd
,0x5320,0x5347,0x53ec,0x54e8,0x5546,0x5531,0x5617,0x5968
,0x59be,0x5a3c,0x5bb5,0x5c06,0x5c0f,0x5c11,0x5c1a,0x5e84
,0x5e8a,0x5ee0,0x5f70,0x627f,0x6284,0x62db,0x638c,0x6377
,0x6607,0x660c,0x662d,0x6676,0x677e,0x68a2,0x6a1f,0x6a35
,0x6cbc,0x6d88,0x6e09,0x6e58,0x713c,0x7126,0x7167,0x75c7
,0x7701,0x785d,0x7901,0x7965,0x79f0,0x7ae0,0x7b11,0x7ca7
,0x7d39,0x8096,0x83d6,0x848b,0x8549,0x885d,0x88f3,0x8a1f
,0x8a3c,0x8a54,0x8a73,0x8c61,0x8cde,0x91a4,0x9266,0x937e
,0x9418,0x969c,0x9798,0x4e0a,0x4e08,0x4e1e,0x4e57,0x5197
,0x5270,0x57ce,0x5834,0x58cc,0x5b22,0x5e38,0x60c5,0x64fe
,0x6761,0x6756,0x6d44,0x72b6,0x7573,0x7a63,0x84b8,0x8b72
,0x91b8,0x9320,0x5631,0x57f4,0x98fe,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x62ed,0x690d,0x6b96,0x71ed,0x7e54,0x8077,0x8272,0x89e6
,0x98df,0x8755,0x8fb1,0x5c3b,0x4f38,0x4fe1,0x4fb5,0x5507
,0x5a20,0x5bdd,0x5be9,0x5fc3,0x614e,0x632f,0x65b0,0x664b
,0x68ee,0x699b,0x6d78,0x6df1,0x7533,0x75b9,0x771f,0x795e
,0x79e6,0x7d33,0x81e3,0x82af,0x85aa,0x89aa,0x8a3a,0x8eab
,0x8f9b,0x9032,0x91dd,0x9707,0x4eba,0x4ec1,0x5203,0x5875
,0x58ec,0x5c0b,0x751a,0x5c3d,0x814e,0x8a0a,0x8fc5,0x9663
,0x976d,0x7b25,0x8acf,0x9808,0x9162,0x56f3,0x53a8,0
,0x9017,0x5439,0x5782,0x5e25,0x63a8,0x6c34,0x708a,0x7761
,0x7c8b,0x7fe0,0x8870,0x9042,0x9154,0x9310,0x9318,0x968f
,0x745e,0x9ac4,0x5d07,0x5d69,0x6570,0x67a2,0x8da8,0x96db
,0x636e,0x6749,0x6919,0x83c5,0x9817,0x96c0,0x88fe,0x6f84
,0x647a,0x5bf8,0x4e16,0x702c,0x755d,0x662f,0x51c4,0x5236
,0x52e2,0x59d3,0x5f81,0x6027,0x6210,0x653f,0x6574,0x661f
,0x6674,0x68f2,0x6816,0x6b63,0x6e05,0x7272,0x751f,0x76db
,0x7cbe,0x8056,0x58f0,0x88fd,0x897f,0x8aa0,0x8a93,0x8acb
,0x901d,0x9192,0x9752,0x9759,0x6589,0x7a0e,0x8106,0x96bb
,0x5e2d,0x60dc,0x621a,0x65a5,0x6614,0x6790,0x77f3,0x7a4d
,0x7c4d,0x7e3e,0x810a,0x8cac,0x8d64,0x8de1,0x8e5f,0x78a9
,0x5207,0x62d9,0x63a5,0x6442,0x6298,0x8a2d,0x7a83,0x7bc0
,0x8aac,0x96ea,0x7d76,0x820c,0x8749,0x4ed9,0x5148,0x5343
,0x5360,0x5ba3,0x5c02,0x5c16,0x5ddd,0x6226,0x6247,0x64b0
,0x6813,0x6834,0x6cc9,0x6d45,0x6d17,0x67d3,0x6f5c,0x714e
,0x717d,0x65cb,0x7a7f,0x7bad,0x7dda,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x7e4a,0x7fa8,0x817a,0x821b,0x8239,0x85a6,0x8a6e,0x8cce
,0x8df5,0x9078,0x9077,0x92ad,0x9291,0x9583,0x9bae,0x524d
,0x5584,0x6f38,0x7136,0x5168,0x7985,0x7e55,0x81b3,0x7cce
,0x564c,0x5851,0x5ca8,0x63aa,0x66fe,0x66fd,0x695a,0x72d9
,0x758f,0x758e,0x790e,0x7956,0x79df,0x7c97,0x7d20,0x7d44
,0x8607,0x8a34,0x963b,0x9061,0x9f20,0x50e7,0x5275,0x53cc
,0x53e2,0x5009,0x55aa,0x58ee,0x594f,0x723d,0x5b8b,0x5c64
,0x531d,0x60e3,0x60f3,0x635c,0x6383,0x633f,0x63bb,0
,0x64cd,0x65e9,0x66f9,0x5de3,0x69cd,0x69fd,0x6f15,0x71e5
,0x4e89,0x75e9,0x76f8,0x7a93,0x7cdf,0x7dcf,0x7d9c,0x8061
,0x8349,0x8358,0x846c,0x84bc,0x85fb,0x88c5,0x8d70,0x9001
,0x906d,0x9397,0x971c,0x9a12,0x50cf,0x5897,0x618e,0x81d3
,0x8535,0x8d08,0x9020,0x4fc3,0x5074,0x5247,0x5373,0x606f
,0x6349,0x675f,0x6e2c,0x8db3,0x901f,0x4fd7,0x5c5e,0x8cca
,0x65cf,0x7d9a,0x5352,0x8896,0x5176,0x63c3,0x5b58,0x5b6b
,0x5c0a,0x640d,0x6751,0x905c,0x4ed6,0x591a,0x592a,0x6c70
,0x8a51,0x553e,0x5815,0x59a5,0x60f0,0x6253,0x67c1,0x8235
,0x6955,0x9640,0x99c4,0x9a28,0x4f53,0x5806,0x5bfe,0x8010
,0x5cb1,0x5e2f,0x5f85,0x6020,0x614b,0x6234,0x66ff,0x6cf0
,0x6ede,0x80ce,0x817f,0x82d4,0x888b,0x8cb8,0x9000,0x902e
,0x968a,0x9edb,0x9bdb,0x4ee3,0x53f0,0x5927,0x7b2c,0x918d
,0x984c,0x9df9,0x6edd,0x7027,0x5353,0x5544,0x5b85,0x6258
,0x629e,0x62d3,0x6ca2,0x6fef,0x7422,0x8a17,0x9438,0x6fc1
,0x8afe,0x8338,0x51e7,0x86f8,0x53ea,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x53e9,0x4f46,0x9054,0x8fb0,0x596a,0x8131,0x5dfd,0x7aea
,0x8fbf,0x68da,0x8c37,0x72f8,0x9c48,0x6a3d,0x8ab0,0x4e39
,0x5358,0x5606,0x5766,0x62c5,0x63a2,0x65e6,0x6b4e,0x6de1
,0x6e5b,0x70ad,0x77ed,0x7aef,0x7baa,0x7dbb,0x803d,0x80c6
,0x86cb,0x8a95,0x935b,0x56e3,0x58c7,0x5f3e,0x65ad,0x6696
,0x6a80,0x6bb5,0x7537,0x8ac7,0x5024,0x77e5,0x5730,0x5f1b
,0x6065,0x667a,0x6c60,0x75f4,0x7a1a,0x7f6e,0x81f4,0x8718
,0x9045,0x99b3,0x7bc9,0x755c,0x7af9,0x7b51,0x84c4,0
,0x9010,0x79e9,0x7a92,0x8336,0x5ae1,0x7740,0x4e2d,0x4ef2
,0x5b99,0x5fe0,0x62bd,0x663c,0x67f1,0x6ce8,0x866b,0x8877
,0x8a3b,0x914e,0x92f3,0x99d0,0x6a17,0x7026,0x732a,0x82e7
,0x8457,0x8caf,0x4e01,0x5146,0x51cb,0x558b,0x5bf5,0x5e16
,0x5e33,0x5e81,0x5f14,0x5f35,0x5f6b,0x5fb4,0x61f2,0x6311
,0x66a2,0x671d,0x6f6e,0x7252,0x753a,0x773a,0x8074,0x8139
,0x8178,0x8776,0x8abf,0x8adc,0x8d85,0x8df3,0x929a,0x9577
,0x9802,0x9ce5,0x52c5,0x6357,0x76f4,0x6715,0x6c88,0x73cd
,0x8cc3,0x93ae,0x9673,0x6d25,0x589c,0x690e,0x69cc,0x8ffd
,0x939a,0x75db,0x901a,0x585a,0x6802,0x63b4,0x69fb,0x4f43
,0x6f2c,0x67d8,0x8fbb,0x8526,0x7db4,0x9354,0x693f,0x6f70
,0x576a,0x58f7,0x5b2c,0x7d2c,0x722a,0x540a,0x91e3,0x9db4
,0x4ead,0x4f4e,0x505c,0x5075,0x5243,0x8c9e,0x5448,0x5824
,0x5b9a,0x5e1d,0x5e95,0x5ead,0x5ef7,0x5f1f,0x608c,0x62b5
,0x633a,0x63d0,0x68af,0x6c40,0x7887,0x798e,0x7a0b,0x7de0
,0x8247,0x8a02,0x8ae6,0x8e44,0x9013,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x90b8,0x912d,0x91d8,0x9f0e,0x6ce5,0x6458,0x64e2,0x6575
,0x6ef4,0x7684,0x7b1b,0x9069,0x93d1,0x6eba,0x54f2,0x5fb9
,0x64a4,0x8f4d,0x8fed,0x9244,0x5178,0x586b,0x5929,0x5c55
,0x5e97,0x6dfb,0x7e8f,0x751c,0x8cbc,0x8ee2,0x985b,0x70b9
,0x4f1d,0x6bbf,0x6fb1,0x7530,0x96fb,0x514e,0x5410,0x5835
,0x5857,0x59ac,0x5c60,0x5f92,0x6597,0x675c,0x6e21,0x767b
,0x83df,0x8ced,0x9014,0x90fd,0x934d,0x7825,0x783a,0x52aa
,0x5ea6,0x571f,0x5974,0x6012,0x5012,0x515a,0x51ac,0
,0x51cd,0x5200,0x5510,0x5854,0x5858,0x5957,0x5b95,0x5cf6
,0x5d8b,0x60bc,0x6295,0x642d,0x6771,0x6843,0x68bc,0x68df
,0x76d7,0x6dd8,0x6e6f,0x6d9b,0x706f,0x71c8,0x5f53,0x75d8
,0x7977,0x7b49,0x7b54,0x7b52,0x7cd6,0x7d71,0x5230,0x8463
,0x8569,0x85e4,0x8a0e,0x8b04,0x8c46,0x8e0f,0x9003,0x900f
,0x9419,0x9676,0x982d,0x9a30,0x95d8,0x50cd,0x52d5,0x540c
,0x5802,0x5c0e,0x61a7,0x649e,0x6d1e,0x77b3,0x7ae5,0x80f4
,0x8404,0x9053,0x9285,0x5ce0,0x9d07,0x533f,0x5f97,0x5fb3
,0x6d9c,0x7279,0x7763,0x79bf,0x7be4,0x6bd2,0x72ec,0x8aad
,0x6803,0x6a61,0x51f8,0x7a81,0x6934,0x5c4a,0x9cf6,0x82eb
,0x5bc5,0x9149,0x701e,0x5678,0x5c6f,0x60c7,0x6566,0x6c8c
,0x8c5a,0x9041,0x9813,0x5451,0x66c7,0x920d,0x5948,0x90a3
,0x5185,0x4e4d,0x51ea,0x8599,0x8b0e,0x7058,0x637a,0x934b
,0x6962,0x99b4,0x7e04,0x7577,0x5357,0x6960,0x8edf,0x96e3
,0x6c5d,0x4e8c,0x5c3c,0x5f10,0x8fe9,0x5302,0x8cd1,0x8089
,0x8679,0x5eff,0x65e5,0x4e73,0x5165,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x5982,0x5c3f,0x97ee,0x4efb,0x598a,0x5fcd,0x8a8d,0x6fe1
,0x79b0,0x7962,0x5be7,0x8471,0x732b,0x71b1,0x5e74,0x5ff5
,0x637b,0x649a,0x71c3,0x7c98,0x4e43,0x5efc,0x4e4b,0x57dc
,0x56a2,0x60a9,0x6fc3,0x7d0d,0x80fd,0x8133,0x81bf,0x8fb2
,0x8997,0x86a4,0x5df4,0x628a,0x64ad,0x8987,0x6777,0x6ce2
,0x6d3e,0x7436,0x7834,0x5a46,0x7f75,0x82ad,0x99ac,0x4ff3
,0x5ec3,0x62dd,0x6392,0x6557,0x676f,0x76c3,0x724c,0x80cc
,0x80ba,0x8f29,0x914d,0x500d,0x57f9,0x5a92,0x6885,0
,0x6973,0x7164,0x72fd,0x8cb7,0x58f2,0x8ce0,0x966a,0x9019
,0x877f,0x79e4,0x77e7,0x8429,0x4f2f,0x5265,0x535a,0x62cd
,0x67cf,0x6cca,0x767d,0x7b94,0x7c95,0x8236,0x8584,0x8feb
,0x66dd,0x6f20,0x7206,0x7e1b,0x83ab,0x99c1,0x9ea6,0x51fd
,0x7bb1,0x7872,0x7bb8,0x8087,0x7b48,0x6ae8,0x5e61,0x808c
,0x7551,0x7560,0x516b,0x9262,0x6e8c,0x767a,0x9197,0x9aea
,0x4f10,0x7f70,0x629c,0x7b4f,0x95a5,0x9ce9,0x567a,0x5859
,0x86e4,0x96bc,0x4f34,0x5224,0x534a,0x53cd,0x53db,0x5e06
,0x642c,0x6591,0x677f,0x6c3e,0x6c4e,0x7248,0x72af,0x73ed
,0x7554,0x7e41,0x822c,0x85e9,0x8ca9,0x7bc4,0x91c6,0x7169
,0x9812,0x98ef,0x633d,0x6669,0x756a,0x76e4,0x78d0,0x8543
,0x86ee,0x532a,0x5351,0x5426,0x5983,0x5e87,0x5f7c,0x60b2
,0x6249,0x6279,0x62ab,0x6590,0x6bd4,0x6ccc,0x75b2,0x76ae
,0x7891,0x79d8,0x7dcb,0x7f77,0x80a5,0x88ab,0x8ab9,0x8cbb
,0x907f,0x975e,0x98db,0x6a0b,0x7c38,0x5099,0x5c3e,0x5fae
,0x6787,0x6bd8,0x7435,0x7709,0x7f8e,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x9f3b,0x67ca,0x7a17,0x5339,0x758b,0x9aed,0x5f66,0x819d
,0x83f1,0x8098,0x5f3c,0x5fc5,0x7562,0x7b46,0x903c,0x6867
,0x59eb,0x5a9b,0x7d10,0x767e,0x8b2c,0x4ff5,0x5f6a,0x6a19
,0x6c37,0x6f02,0x74e2,0x7968,0x8868,0x8a55,0x8c79,0x5edf
,0x63cf,0x75c5,0x79d2,0x82d7,0x9328,0x92f2,0x849c,0x86ed
,0x9c2d,0x54c1,0x5f6c,0x658c,0x6d5c,0x7015,0x8ca7,0x8cd3
,0x983b,0x654f,0x74f6,0x4e0d,0x4ed8,0x57e0,0x592b,0x5a66
,0x5bcc,0x51a8,0x5e03,0x5e9c,0x6016,0x6276,0x6577,0
,0x65a7,0x666e,0x6d6e,0x7236,0x7b26,0x8150,0x819a,0x8299
,0x8b5c,0x8ca0,0x8ce6,0x8d74,0x961c,0x9644,0x4fae,0x64ab
,0x6b66,0x821e,0x8461,0x856a,0x90e8,0x5c01,0x6953,0x98a8
,0x847a,0x8557,0x4f0f,0x526f,0x5fa9,0x5e45,0x670d,0x798f
,0x8179,0x8907,0x8986,0x6df5,0x5f17,0x6255,0x6cb8,0x4ecf
,0x7269,0x9b92,0x5206,0x543b,0x5674,0x58b3,0x61a4,0x626e
,0x711a,0x596e,0x7c89,0x7cde,0x7d1b,0x96f0,0x6587,0x805e
,0x4e19,0x4f75,0x5175,0x5840,0x5e63,0x5e73,0x5f0a,0x67c4
,0x4e26,0x853d,0x9589,0x965b,0x7c73,0x9801,0x50fb,0x58c1
,0x7656,0x78a7,0x5225,0x77a5,0x8511,0x7b86,0x504f,0x5909
,0x7247,0x7bc7,0x7de8,0x8fba,0x8fd4,0x904d,0x4fbf,0x52c9
,0x5a29,0x5f01,0x97ad,0x4fdd,0x8217,0x92ea,0x5703,0x6355
,0x6b69,0x752b,0x88dc,0x8f14,0x7a42,0x52df,0x5893,0x6155
,0x620a,0x66ae,0x6bcd,0x7c3f,0x83e9,0x5023,0x4ff8,0x5305
,0x5446,0x5831,0x5949,0x5b9d,0x5cf0,0x5cef,0x5d29,0x5e96
,0x62b1,0x6367,0x653e,0x65b9,0x670b,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x6cd5,0x6ce1,0x70f9,0x7832,0x7e2b,0x80de,0x82b3,0x840c
,0x84ec,0x8702,0x8912,0x8a2a,0x8c4a,0x90a6,0x92d2,0x98fd
,0x9cf3,0x9d6c,0x4e4f,0x4ea1,0x508d,0x5256,0x574a,0x59a8
,0x5e3d,0x5fd8,0x5fd9,0x623f,0x66b4,0x671b,0x67d0,0x68d2
,0x5192,0x7d21,0x80aa,0x81a8,0x8b00,0x8c8c,0x8cbf,0x927e
,0x9632,0x5420,0x982c,0x5317,0x50d5,0x535c,0x58a8,0x64b2
,0x6734,0x7267,0x7766,0x7a46,0x91e6,0x52c3,0x6ca1,0x6b86
,0x5800,0x5e4c,0x5954,0x672c,0x7ffb,0x51e1,0x76c6,0
,0x6469,0x78e8,0x9b54,0x9ebb,0x57cb,0x59b9,0x6627,0x679a
,0x6bce,0x54e9,0x69d9,0x5e55,0x819c,0x6795,0x9baa,0x67fe
,0x9c52,0x685d,0x4ea6,0x4fe3,0x53c8,0x62b9,0x672b,0x6cab
,0x8fc4,0x4fad,0x7e6d,0x9ebf,0x4e07,0x6162,0x6e80,0x6f2b
,0x8513,0x5473,0x672a,0x9b45,0x5df3,0x7b95,0x5cac,0x5bc6
,0x871c,0x6e4a,0x84d1,0x7a14,0x8108,0x5999,0x7c8d,0x6c11
,0x7720,0x52d9,0x5922,0x7121,0x725f,0x77db,0x9727,0x9d61
,0x690b,0x5a7f,0x5a18,0x51a5,0x540d,0x547d,0x660e,0x76df
,0x8ff7,0x9298,0x9cf4,0x59ea,0x725d,0x6ec5,0x514d,0x68c9
,0x7dbf,0x7dec,0x9762,0x9eba,0x6478,0x6a21,0x8302,0x5984
,0x5b5f,0x6bdb,0x731b,0x76f2,0x7db2,0x8017,0x8499,0x5132
,0x6728,0x9ed9,0x76ee,0x6762,0x52ff,0x9905,0x5c24,0x623b
,0x7c7e,0x8cb0,0x554f,0x60b6,0x7d0b,0x9580,0x5301,0x4e5f
,0x51b6,0x591c,0x723a,0x8036,0x91ce,0x5f25,0x77e2,0x5384
,0x5f79,0x7d04,0x85ac,0x8a33,0x8e8d,0x9756,0x67f3,0x85ae
,0x9453,0x6109,0x6108,0x6cb9,0x7652,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x8aed,0x8f38,0x552f,0x4f51,0x512a,0x52c7,0x53cb,0x5ba5
,0x5e7d,0x60a0,0x6182,0x63d6,0x6709,0x67da,0x6e67,0x6d8c
,0x7336,0x7337,0x7531,0x7950,0x88d5,0x8a98,0x904a,0x9091
,0x90f5,0x96c4,0x878d,0x5915,0x4e88,0x4f59,0x4e0e,0x8a89
,0x8f3f,0x9810,0x50ad,0x5e7c,0x5996,0x5bb9,0x5eb8,0x63da
,0x63fa,0x64c1,0x66dc,0x694a,0x69d8,0x6d0b,0x6eb6,0x7194
,0x7528,0x7aaf,0x7f8a,0x8000,0x8449,0x84c9,0x8981,0x8b21
,0x8e0a,0x9065,0x967d,0x990a,0x617e,0x6291,0x6b32,0
,0x6c83,0x6d74,0x7fcc,0x7ffc,0x6dc0,0x7f85,0x87ba,0x88f8
,0x6765,0x83b1,0x983c,0x96f7,0x6d1b,0x7d61,0x843d,0x916a
,0x4e71,0x5375,0x5d50,0x6b04,0x6feb,0x85cd,0x862d,0x89a7
,0x5229,0x540f,0x5c65,0x674e,0x68a8,0x7406,0x7483,0x75e2
,0x88cf,0x88e1,0x91cc,0x96e2,0x9678,0x5f8b,0x7387,0x7acb
,0x844e,0x63a0,0x7565,0x5289,0x6d41,0x6e9c,0x7409,0x7559
,0x786b,0x7c92,0x9686,0x7adc,0x9f8d,0x4fb6,0x616e,0x65c5
,0x865c,0x4e86,0x4eae,0x50da,0x4e21,0x51cc,0x5bee,0x6599
,0x6881,0x6dbc,0x731f,0x7642,0x77ad,0x7a1c,0x7ce7,0x826f
,0x8ad2,0x907c,0x91cf,0x9675,0x9818,0x529b,0x7dd1,0x502b
,0x5398,0x6797,0x6dcb,0x71d0,0x7433,0x81e8,0x8f2a,0x96a3
,0x9c57,0x9e9f,0x7460,0x5841,0x6d99,0x7d2f,0x985e,0x4ee4
,0x4f36,0x4f8b,0x51b7,0x52b1,0x5dba,0x601c,0x73b2,0x793c
,0x82d3,0x9234,0x96b7,0x96f6,0x970a,0x9e97,0x9f62,0x66a6
,0x6b74,0x5217,0x52a3,0x70c8,0x88c2,0x5ec9,0x604b,0x6190
,0x6f23,0x7149,0x7c3e,0x7df4,0x806f,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x84ee,0x9023,0x932c,0x5442,0x9b6f,0x6ad3,0x7089,0x8cc2
,0x8def,0x9732,0x52b4,0x5a41,0x5eca,0x5f04,0x6717,0x697c
,0x6994,0x6d6a,0x6f0f,0x7262,0x72fc,0x7bed,0x8001,0x807e
,0x874b,0x90ce,0x516d,0x9e93,0x7984,0x808b,0x9332,0x8ad6
,0x502d,0x548c,0x8a71,0x6b6a,0x8cc4,0x8107,0x60d1,0x67a0
,0x9df2,0x4e99,0x4e98,0x9c10,0x8a6b,0x85c1,0x8568,0x6900
,0x6e7e,0x7897,0x8155,0x20b9f,0x5b41,0x5b56,0x5b7d,0x5b93
,0x5bd8,0x5bec,0x5c12,0x5c1e,0x5c23,0x5c2b,0x378d,0
,0x5c62,0xfa3b,0xfa3c,0xaad4,0x5c7a,0x5c8f,0x5c9f,0x5ca3
,0x5caa,0x5cba,0x5ccb,0x5cd0,0x5cd2,0x5cf4,0xaadd,0x37e2
,0x5d0d,0x5d27,0xfa11,0x5d46,0x5d47,0x5d53,0x5d4a,0x5d6d
,0x5d81,0x5da0,0x5da4,0x5da7,0x5db8,0x5dcb,0x541e,0x5f0c
,0x4e10,0x4e15,0x4e2a,0x4e31,0x4e36,0x4e3c,0x4e3f,0x4e42
,0x4e56,0x4e58,0x4e82,0x4e85,0x8c6b,0x4e8a,0x8212,0x5f0d
,0x4e8e,0x4e9e,0x4e9f,0x4ea0,0x4ea2,0x4eb0,0x4eb3,0x4eb6
,0x4ece,0x4ecd,0x4ec4,0x4ec6,0x4ec2,0x4ed7,0x4ede,0x4eed
,0x4edf,0x4ef7,0x4f09,0x4f5a,0x4f30,0x4f5b,0x4f5d,0x4f57
,0x4f47,0x4f76,0x4f88,0x4f8f,0x4f98,0x4f7b,0x4f69,0x4f70
,0x4f91,0x4f6f,0x4f86,0x4f96,0x5118,0x4fd4,0x4fdf,0x4fce
,0x4fd8,0x4fdb,0x4fd1,0x4fda,0x4fd0,0x4fe4,0x4fe5,0x501a
,0x5028,0x5014,0x502a,0x5025,0x5005,0x4f1c,0x4ff6,0x5021
,0x5029,0x502c,0x4ffe,0x4fef,0x5011,0x5006,0x5043,0x5047
,0x6703,0x5055,0x5050,0x5048,0x505a,0x5056,0x506c,0x5078
,0x5080,0x509a,0x5085,0x50b4,0x50b2,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x50c9,0x50ca,0x50b3,0x50c2,0x50d6,0x50de,0x50e5,0x50ed
,0x50e3,0x50ee,0x50f9,0x50f5,0x5109,0x5101,0x5102,0x5116
,0x5115,0x5114,0x511a,0x5121,0x513a,0x5137,0x513c,0x513b
,0x513f,0x5140,0x5152,0x514c,0x5154,0x5162,0x7af8,0x5169
,0x516a,0x516e,0x5180,0x5182,0x56d8,0x518c,0x5189,0x518f
,0x5191,0x5193,0x5195,0x5196,0x51a4,0x51a6,0x51a2,0x51a9
,0x51aa,0x51ab,0x51b3,0x51b1,0x51b2,0x51b0,0x51b5,0x51bd
,0x51c5,0x51c9,0x51db,0x51e0,0x8655,0x51e9,0x51ed,0
,0x51f0,0x51f5,0x51fe,0x5204,0x520b,0x5214,0x520e,0x5227
,0x522a,0x522e,0x5233,0x5239,0x524f,0x5244,0x524b,0x524c
,0x525e,0x5254,0x526a,0x5274,0x5269,0x5273,0x527f,0x527d
,0x528d,0x5294,0x5292,0x5271,0x5288,0x5291,0x8fa8,0x8fa7
,0x52ac,0x52ad,0x52bc,0x52b5,0x52c1,0x52cd,0x52d7,0x52de
,0x52e3,0x52e6,0x98ed,0x52e0,0x52f3,0x52f5,0x52f8,0x52f9
,0x5306,0x5308,0x7538,0x530d,0x5310,0x530f,0x5315,0x531a
,0x5323,0x532f,0x5331,0x5333,0x5338,0x5340,0x5346,0x5345
,0x4e17,0x5349,0x534d,0x51d6,0x535e,0x5369,0x536e,0x5918
,0x537b,0x5377,0x5382,0x5396,0x53a0,0x53a6,0x53a5,0x53ae
,0x53b0,0x53b6,0x53c3,0x7c12,0x96d9,0x53df,0x66fc,0x71ee
,0x53ee,0x53e8,0x53ed,0x53fa,0x5401,0x543d,0x5440,0x542c
,0x542d,0x543c,0x542e,0x5436,0x5429,0x541d,0x544e,0x548f
,0x5475,0x548e,0x545f,0x5471,0x5477,0x5470,0x5492,0x547b
,0x5480,0x5476,0x5484,0x5490,0x5486,0x54c7,0x54a2,0x54b8
,0x54a5,0x54ac,0x54c4,0x54c8,0x54a8,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x54ab,0x54c2,0x54a4,0x54be,0x54bc,0x54d8,0x54e5,0x54e6
,0x550f,0x5514,0x54fd,0x54ee,0x54ed,0x54fa,0x54e2,0x5539
,0x5540,0x5563,0x554c,0x552e,0x555c,0x5545,0x5556,0x5557
,0x5538,0x5533,0x555d,0x5599,0x5580,0x54af,0x558a,0x559f
,0x557b,0x557e,0x5598,0x559e,0x55ae,0x557c,0x5583,0x55a9
,0x5587,0x55a8,0x55da,0x55c5,0x55df,0x55c4,0x55dc,0x55e4
,0x55d4,0x5614,0x55f7,0x5616,0x55fe,0x55fd,0x561b,0x55f9
,0x564e,0x5650,0x71df,0x5634,0x5636,0x5632,0x5638,0
,0x566b,0x5664,0x562f,0x566c,0x566a,0x5686,0x5680,0x568a
,0x56a0,0x5694,0x568f,0x56a5,0x56ae,0x56b6,0x56b4,0x56c2
,0x56bc,0x56c1,0x56c3,0x56c0,0x56c8,0x56ce,0x56d1,0x56d3
,0x56d7,0x56ee,0x56f9,0x5700,0x56ff,0x5704,0x5709,0x5708
,0x570b,0x570d,0x5713,0x5718,0x5716,0x55c7,0x571c,0x5726
,0x5737,0x5738,0x574e,0x573b,0x5740,0x574f,0x5769,0x57c0
,0x5788,0x5761,0x577f,0x5789,0x5793,0x57a0,0x57b3,0x57a4
,0x57aa,0x57b0,0x57c3,0x57c6,0x57d4,0x57d2,0x57d3,0x580a
,0x57d6,0x57e3,0x580b,0x5819,0x581d,0x5872,0x5821,0x5862
,0x584b,0x5870,0x6bc0,0x5852,0x583d,0x5879,0x5885,0x58b9
,0x589f,0x58ab,0x58ba,0x58de,0x58bb,0x58b8,0x58ae,0x58c5
,0x58d3,0x58d1,0x58d7,0x58d9,0x58d8,0x58e5,0x58dc,0x58e4
,0x58df,0x58ef,0x58fa,0x58f9,0x58fb,0x58fc,0x58fd,0x5902
,0x590a,0x5910,0x591b,0x68a6,0x5925,0x592c,0x592d,0x5932
,0x5938,0x593e,0x7ad2,0x5955,0x5950,0x594e,0x595a,0x5958
,0x5962,0x5960,0x5967,0x596c,0x5969,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x5978,0x5981,0x599d,0x4f5e,0x4fab,0x59a3,0x59b2,0x59c6
,0x59e8,0x59dc,0x598d,0x59d9,0x59da,0x5a25,0x5a1f,0x5a11
,0x5a1c,0x5a09,0x5a1a,0x5a40,0x5a6c,0x5a49,0x5a35,0x5a36
,0x5a62,0x5a6a,0x5a9a,0x5abc,0x5abe,0x5acb,0x5ac2,0x5abd
,0x5ae3,0x5ad7,0x5ae6,0x5ae9,0x5ad6,0x5afa,0x5afb,0x5b0c
,0x5b0b,0x5b16,0x5b32,0x5ad0,0x5b2a,0x5b36,0x5b3e,0x5b43
,0x5b45,0x5b40,0x5b51,0x5b55,0x5b5a,0x5b5b,0x5b65,0x5b69
,0x5b70,0x5b73,0x5b75,0x5b78,0x6588,0x5b7a,0x5b80,0
,0x5b83,0x5ba6,0x5bb8,0x5bc3,0x5bc7,0x5bc9,0x5bd4,0x5bd0
,0x5be4,0x5be6,0x5be2,0x5bde,0x5be5,0x5beb,0x5bf0,0x5bf6
,0x5bf3,0x5c05,0x5c07,0x5c08,0x5c0d,0x5c13,0x5c20,0x5c22
,0x5c28,0x5c38,0x5c39,0x5c41,0x5c46,0x5c4e,0x5c53,0x5c50
,0x5c4f,0x5b71,0x5c6c,0x5c6e,0x4e62,0x5c76,0x5c79,0x5c8c
,0x5c91,0x5c94,0x599b,0x5cab,0x5cbb,0x5cb6,0x5cbc,0x5cb7
,0x5cc5,0x5cbe,0x5cc7,0x5cd9,0x5ce9,0x5cfd,0x5cfa,0x5ced
,0x5d8c,0x5cea,0x5d0b,0x5d15,0x5d17,0x5d5c,0x5d1f,0x5d1b
,0x5d11,0x5d14,0x5d22,0x5d1a,0x5d19,0x5d18,0x5d4c,0x5d52
,0x5d4e,0x5d4b,0x5d6c,0x5d73,0x5d76,0x5d87,0x5d84,0x5d82
,0x5da2,0x5d9d,0x5dac,0x5dae,0x5dbd,0x5d90,0x5db7,0x5dbc
,0x5dc9,0x5dcd,0x5dd3,0x5dd2,0x5dd6,0x5ddb,0x5deb,0x5df2
,0x5df5,0x5e0b,0x5e1a,0x5e19,0x5e11,0x5e1b,0x5e36,0x5e37
,0x5e44,0x5e43,0x5e40,0x5e4e,0x5e57,0x5e54,0x5e5f,0x5e62
,0x5e64,0x5e47,0x5e75,0x5e76,0x5e7a,0x9ebc,0x5e7f,0x5ea0
,0x5ec1,0x5ec2,0x5ec8,0x5ed0,0x5ecf,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x5ed6,0x5ee3,0x5edd,0x5eda,0x5edb,0x5ee2,0x5ee1,0x5ee8
,0x5ee9,0x5eec,0x5ef1,0x5ef3,0x5ef0,0x5ef4,0x5ef8,0x5efe
,0x5f03,0x5f09,0x5f5d,0x5f5c,0x5f0b,0x5f11,0x5f16,0x5f29
,0x5f2d,0x5f38,0x5f41,0x5f48,0x5f4c,0x5f4e,0x5f2f,0x5f51
,0x5f56,0x5f57,0x5f59,0x5f61,0x5f6d,0x5f73,0x5f77,0x5f83
,0x5f82,0x5f7f,0x5f8a,0x5f88,0x5f91,0x5f87,0x5f9e,0x5f99
,0x5f98,0x5fa0,0x5fa8,0x5fad,0x5fbc,0x5fd6,0x5ffb,0x5fe4
,0x5ff8,0x5ff1,0x5fdd,0x60b3,0x5fff,0x6021,0x6060,0
,0x6019,0x6010,0x6029,0x600e,0x6031,0x601b,0x6015,0x602b
,0x6026,0x600f,0x603a,0x605a,0x6041,0x606a,0x6077,0x605f
,0x604a,0x6046,0x604d,0x6063,0x6043,0x6064,0x6042,0x606c
,0x606b,0x6059,0x6081,0x608d,0x60e7,0x6083,0x609a,0x6084
,0x609b,0x6096,0x6097,0x6092,0x60a7,0x608b,0x60e1,0x60b8
,0x60e0,0x60d3,0x60b4,0x5ff0,0x60bd,0x60c6,0x60b5,0x60d8
,0x614d,0x6115,0x6106,0x60f6,0x60f7,0x6100,0x60f4,0x60fa
,0x6103,0x6121,0x60fb,0x60f1,0x610d,0x610e,0x6147,0x613e
,0x6128,0x6127,0x614a,0x613f,0x613c,0x612c,0x6134,0x613d
,0x6142,0x6144,0x6173,0x6177,0x6158,0x6159,0x615a,0x616b
,0x6174,0x616f,0x6165,0x6171,0x615f,0x615d,0x6153,0x6175
,0x6199,0x6196,0x6187,0x61ac,0x6194,0x619a,0x618a,0x6191
,0x61ab,0x61ae,0x61cc,0x61ca,0x61c9,0x61f7,0x61c8,0x61c3
,0x61c6,0x61ba,0x61cb,0x7f79,0x61cd,0x61e6,0x61e3,0x61f6
,0x61fa,0x61f4,0x61ff,0x61fd,0x61fc,0x61fe,0x6200,0x6208
,0x6209,0x620d,0x620c,0x6214,0x621b,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x621e,0x6221,0x622a,0x622e,0x6230,0x6232,0x6233,0x6241
,0x624e,0x625e,0x6263,0x625b,0x6260,0x6268,0x627c,0x6282
,0x6289,0x627e,0x6292,0x6293,0x6296,0x62d4,0x6283,0x6294
,0x62d7,0x62d1,0x62bb,0x62cf,0x62ff,0x62c6,0x64d4,0x62c8
,0x62dc,0x62cc,0x62ca,0x62c2,0x62c7,0x629b,0x62c9,0x630c
,0x62ee,0x62f1,0x6327,0x6302,0x6308,0x62ef,0x62f5,0x6350
,0x633e,0x634d,0x641c,0x634f,0x6396,0x638e,0x6380,0x63ab
,0x6376,0x63a3,0x638f,0x6389,0x639f,0x63b5,0x636b,0
,0x6369,0x63be,0x63e9,0x63c0,0x63c6,0x63e3,0x63c9,0x63d2
,0x63f6,0x63c4,0x6416,0x6434,0x6406,0x6413,0x6426,0x6436
,0x651d,0x6417,0x6428,0x640f,0x6467,0x646f,0x6476,0x644e
,0x652a,0x6495,0x6493,0x64a5,0x64a9,0x6488,0x64bc,0x64da
,0x64d2,0x64c5,0x64c7,0x64bb,0x64d8,0x64c2,0x64f1,0x64e7
,0x8209,0x64e0,0x64e1,0x62ac,0x64e3,0x64ef,0x652c,0x64f6
,0x64f4,0x64f2,0x64fa,0x6500,0x64fd,0x6518,0x651c,0x6505
,0x6524,0x6523,0x652b,0x6534,0x6535,0x6537,0x6536,0x6538
,0x754b,0x6548,0x6556,0x6555,0x654d,0x6558,0x655e,0x655d
,0x6572,0x6578,0x6582,0x6583,0x8b8a,0x659b,0x659f,0x65ab
,0x65b7,0x65c3,0x65c6,0x65c1,0x65c4,0x65cc,0x65d2,0x65db
,0x65d9,0x65e0,0x65e1,0x65f1,0x6772,0x660a,0x6603,0x65fb
,0x6773,0x6635,0x6636,0x6634,0x661c,0x664f,0x6644,0x6649
,0x6641,0x665e,0x665d,0x6664,0x6667,0x6668,0x665f,0x6662
,0x6670,0x6683,0x6688,0x668e,0x6689,0x6684,0x6698,0x669d
,0x66c1,0x66b9,0x66c9,0x66be,0x66bc,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x66c4,0x66b8,0x66d6,0x66da,0x66e0,0x663f,0x66e6,0x66e9
,0x66f0,0x66f5,0x66f7,0x670f,0x6716,0x671e,0x6726,0x6727
,0x9738,0x672e,0x673f,0x6736,0x6741,0x6738,0x6737,0x6746
,0x675e,0x6760,0x6759,0x6763,0x6764,0x6789,0x6770,0x67a9
,0x677c,0x676a,0x678c,0x678b,0x67a6,0x67a1,0x6785,0x67b7
,0x67ef,0x67b4,0x67ec,0x67b3,0x67e9,0x67b8,0x67e4,0x67de
,0x67dd,0x67e2,0x67ee,0x67b9,0x67ce,0x67c6,0x67e7,0x6a9c
,0x681e,0x6846,0x6829,0x6840,0x684d,0x6832,0x684e,0
,0x68b3,0x682b,0x6859,0x6863,0x6877,0x687f,0x689f,0x688f
,0x68ad,0x6894,0x689d,0x689b,0x6883,0x6aae,0x68b9,0x6874
,0x68b5,0x68a0,0x68ba,0x690f,0x688d,0x687e,0x6901,0x68ca
,0x6908,0x68d8,0x6922,0x6926,0x68e1,0x690c,0x68cd,0x68d4
,0x68e7,0x68d5,0x6936,0x6912,0x6904,0x68d7,0x68e3,0x6925
,0x68f9,0x68e0,0x68ef,0x6928,0x692a,0x691a,0x6923,0x6921
,0x68c6,0x6979,0x6977,0x695c,0x6978,0x696b,0x6954,0x697e
,0x696e,0x6939,0x6974,0x693d,0x6959,0x6930,0x6961,0x695e
,0x695d,0x6981,0x696a,0x69b2,0x69ae,0x69d0,0x69bf,0x69c1
,0x69d3,0x69be,0x69ce,0x5be8,0x69ca,0x69dd,0x69bb,0x69c3
,0x69a7,0x6a2e,0x6991,0x69a0,0x699c,0x6995,0x69b4,0x69de
,0x69e8,0x6a02,0x6a1b,0x69ff,0x6b0a,0x69f9,0x69f2,0x69e7
,0x6a05,0x69b1,0x6a1e,0x69ed,0x6a14,0x69eb,0x6a0a,0x6a12
,0x6ac1,0x6a23,0x6a13,0x6a44,0x6a0c,0x6a72,0x6a36,0x6a78
,0x6a47,0x6a62,0x6a59,0x6a66,0x6a48,0x6a38,0x6a22,0x6a90
,0x6a8d,0x6aa0,0x6a84,0x6aa2,0x6aa3,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x6a97,0x8617,0x6abb,0x6ac3,0x6ac2,0x6ab8,0x6ab3,0x6aac
,0x6ade,0x6ad1,0x6adf,0x6aaa,0x6ada,0x6aea,0x6afb,0x6b05
,0x8616,0x6afa,0x6b12,0x6b16,0x9b31,0x6b1f,0x6b38,0x6b37
,0x76dc,0x6b39,0x98ee,0x6b47,0x6b43,0x6b49,0x6b50,0x6b59
,0x6b54,0x6b5b,0x6b5f,0x6b61,0x6b78,0x6b79,0x6b7f,0x6b80
,0x6b84,0x6b83,0x6b8d,0x6b98,0x6b95,0x6b9e,0x6ba4,0x6baa
,0x6bab,0x6baf,0x6bb2,0x6bb1,0x6bb3,0x6bb7,0x6bbc,0x6bc6
,0x6bcb,0x6bd3,0x6bdf,0x6bec,0x6beb,0x6bf3,0x6bef,0
,0x9ebe,0x6c08,0x6c13,0x6c14,0x6c1b,0x6c24,0x6c23,0x6c5e
,0x6c55,0x6c62,0x6c6a,0x6c82,0x6c8d,0x6c9a,0x6c81,0x6c9b
,0x6c7e,0x6c68,0x6c73,0x6c92,0x6c90,0x6cc4,0x6cf1,0x6cd3
,0x6cbd,0x6cd7,0x6cc5,0x6cdd,0x6cae,0x6cb1,0x6cbe,0x6cba
,0x6cdb,0x6cef,0x6cd9,0x6cea,0x6d1f,0x884d,0x6d36,0x6d2b
,0x6d3d,0x6d38,0x6d19,0x6d35,0x6d33,0x6d12,0x6d0c,0x6d63
,0x6d93,0x6d64,0x6d5a,0x6d79,0x6d59,0x6d8e,0x6d95,0x6fe4
,0x6d85,0x6df9,0x6e15,0x6e0a,0x6db5,0x6dc7,0x6de6,0x6db8
,0x6dc6,0x6dec,0x6dde,0x6dcc,0x6de8,0x6dd2,0x6dc5,0x6dfa
,0x6dd9,0x6de4,0x6dd5,0x6dea,0x6dee,0x6e2d,0x6e6e,0x6e2e
,0x6e19,0x6e72,0x6e5f,0x6e3e,0x6e23,0x6e6b,0x6e2b,0x6e76
,0x6e4d,0x6e1f,0x6e43,0x6e3a,0x6e4e,0x6e24,0x6eff,0x6e1d
,0x6e38,0x6e82,0x6eaa,0x6e98,0x6ec9,0x6eb7,0x6ed3,0x6ebd
,0x6eaf,0x6ec4,0x6eb2,0x6ed4,0x6ed5,0x6e8f,0x6ea5,0x6ec2
,0x6e9f,0x6f41,0x6f11,0x704c,0x6eec,0x6ef8,0x6efe,0x6f3f
,0x6ef2,0x6f31,0x6eef,0x6f32,0x6ecc,0,0,0
,0x6f3e,0x6f13,0x6ef7,0x6f86,0x6f7a,0x6f78,0x6f81,0x6f80
,0x6f6f,0x6f5b,0x6ff3,0x6f6d,0x6f82,0x6f7c,0x6f58,0x6f8e
,0x6f91,0x6fc2,0x6f66,0x6fb3,0x6fa3,0x6fa1,0x6fa4,0x6fb9
,0x6fc6,0x6faa,0x6fdf,0x6fd5,0x6fec,0x6fd4,0x6fd8,0x6ff1
,0x6fee,0x6fdb,0x7009,0x700b,0x6ffa,0x7011,0x7001,0x700f
,0x6ffe,0x701b,0x701a,0x6f74,0x701d,0x7018,0x701f,0x7030
,0x703e,0x7032,0x7051,0x7063,0x7099,0x7092,0x70af,0x70f1
,0x70ac,0x70b8,0x70b3,0x70ae,0x70df,0x70cb,0x70dd,0
,0x70d9,0x7109,0x70fd,0x711c,0x7119,0x7165,0x7155,0x7188
,0x7166,0x7162,0x714c,0x7156,0x716c,0x718f,0x71fb,0x7184
,0x7195,0x71a8,0x71ac,0x71d7,0x71b9,0x71be,0x71d2,0x71c9
,0x71d4,0x71ce,0x71e0,0x71ec,0x71e7,0x71f5,0x71fc,0x71f9
,0x71ff,0x720d,0x7210,0x721b,0x7228,0x722d,0x722c,0x7230
,0x7232,0x723b,0x723c,0x723f,0x7240,0x7246,0x724b,0x7258
,0x7274,0x727e,0x7282,0x7281,0x7287,0x7292,0x7296,0x72a2
,0x72a7,0x72b9,0x72b2,0x72c3,0x72c6,0x72c4,0x72ce,0x72d2
,0x72e2,0x72e0,0x72e1,0x72f9,0x72f7,0x500f,0x7317,0x730a
,0x731c,0x7316,0x731d,0x7334,0x732f,0x7329,0x7325,0x733e
,0x734e,0x734f,0x9ed8,0x7357,0x736a,0x7368,0x7370,0x7378
,0x7375,0x737b,0x737a,0x73c8,0x73b3,0x73ce,0x73bb,0x73c0
,0x73e5,0x73ee,0x73de,0x74a2,0x7405,0x746f,0x7425,0x73f8
,0x7432,0x743a,0x7455,0x743f,0x745f,0x7459,0x7441,0x745c
,0x7469,0x7470,0x7463,0x746a,0x7476,0x747e,0x748b,0x749e
,0x74a7,0x74ca,0x74cf,0x74d4,0x73f1,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x74e0,0x74e3,0x74e7,0x74e9,0x74ee,0x74f2,0x74f0,0x74f1
,0x74f8,0x74f7,0x7504,0x7503,0x7505,0x750c,0x750e,0x750d
,0x7515,0x7513,0x751e,0x7526,0x752c,0x753c,0x7544,0x754d
,0x754a,0x7549,0x755b,0x7546,0x755a,0x7569,0x7564,0x7567
,0x756b,0x756d,0x7578,0x7576,0x7586,0x7587,0x7574,0x758a
,0x7589,0x7582,0x7594,0x759a,0x759d,0x75a5,0x75a3,0x75c2
,0x75b3,0x75c3,0x75b5,0x75bd,0x75b8,0x75bc,0x75b1,0x75cd
,0x75ca,0x75d2,0x75d9,0x75e3,0x75de,0x75fe,0x75ff,0
,0x75fc,0x7601,0x75f0,0x75fa,0x75f2,0x75f3,0x760b,0x760d
,0x7609,0x761f,0x7627,0x7620,0x7621,0x7622,0x7624,0x7634
,0x7630,0x763b,0x7647,0x7648,0x7646,0x765c,0x7658,0x7661
,0x7662,0x7668,0x7669,0x766a,0x7667,0x766c,0x7670,0x7672
,0x7676,0x7678,0x767c,0x7680,0x7683,0x7688,0x768b,0x768e
,0x7696,0x7693,0x7699,0x769a,0x76b0,0x76b4,0x76b8,0x76b9
,0x76ba,0x76c2,0x76cd,0x76d6,0x76d2,0x76de,0x76e1,0x76e5
,0x76e7,0x76ea,0x862f,0x76fb,0x7708,0x7707,0x7704,0x7729
,0x7724,0x771e,0x7725,0x7726,0x771b,0x7737,0x7738,0x7747
,0x775a,0x7768,0x776b,0x775b,0x7765,0x777f,0x777e,0x7779
,0x778e,0x778b,0x7791,0x77a0,0x779e,0x77b0,0x77b6,0x77b9
,0x77bf,0x77bc,0x77bd,0x77bb,0x77c7,0x77cd,0x77d7,0x77da
,0x77dc,0x77e3,0x77ee,0x77fc,0x780c,0x7812,0x7926,0x7820
,0x792a,0x7845,0x788e,0x7874,0x7886,0x787c,0x789a,0x788c
,0x78a3,0x78b5,0x78aa,0x78af,0x78d1,0x78c6,0x78cb,0x78d4
,0x78be,0x78bc,0x78c5,0x78ca,0x78ec,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x78e7,0x78da,0x78fd,0x78f4,0x7907,0x7912,0x7911,0x7919
,0x792c,0x792b,0x7940,0x7960,0x7957,0x795f,0x795a,0x7955
,0x7953,0x797a,0x797f,0x798a,0x799d,0x79a7,0x9f4b,0x79aa
,0x79ae,0x79b3,0x79b9,0x79ba,0x79c9,0x79d5,0x79e7,0x79ec
,0x79e1,0x79e3,0x7a08,0x7a0d,0x7a18,0x7a19,0x7a20,0x7a1f
,0x7980,0x7a31,0x7a3b,0x7a3e,0x7a37,0x7a43,0x7a57,0x7a49
,0x7a61,0x7a62,0x7a69,0x9f9d,0x7a70,0x7a79,0x7a7d,0x7a88
,0x7a97,0x7a95,0x7a98,0x7a96,0x7aa9,0x7ac8,0x7ab0,0
,0x7ab6,0x7ac5,0x7ac4,0x7abf,0x9083,0x7ac7,0x7aca,0x7acd
,0x7acf,0x7ad5,0x7ad3,0x7ad9,0x7ada,0x7add,0x7ae1,0x7ae2
,0x7ae6,0x7aed,0x7af0,0x7b02,0x7b0f,0x7b0a,0x7b06,0x7b33
,0x7b18,0x7b19,0x7b1e,0x7b35,0x7b28,0x7b36,0x7b50,0x7b7a
,0x7b04,0x7b4d,0x7b0b,0x7b4c,0x7b45,0x7b75,0x7b65,0x7b74
,0x7b67,0x7b70,0x7b71,0x7b6c,0x7b6e,0x7b9d,0x7b98,0x7b9f
,0x7b8d,0x7b9c,0x7b9a,0x7b8b,0x7b92,0x7b8f,0x7b5d,0x7b99
,0x7bcb,0x7bc1,0x7bcc,0x7bcf,0x7bb4,0x7bc6,0x7bdd,0x7be9
,0x7c11,0x7c14,0x7be6,0x7be5,0x7c60,0x7c00,0x7c07,0x7c13
,0x7bf3,0x7bf7,0x7c17,0x7c0d,0x7bf6,0x7c23,0x7c27,0x7c2a
,0x7c1f,0x7c37,0x7c2b,0x7c3d,0x7c4c,0x7c43,0x7c54,0x7c4f
,0x7c40,0x7c50,0x7c58,0x7c5f,0x7c64,0x7c56,0x7c65,0x7c6c
,0x7c75,0x7c83,0x7c90,0x7ca4,0x7cad,0x7ca2,0x7cab,0x7ca1
,0x7ca8,0x7cb3,0x7cb2,0x7cb1,0x7cae,0x7cb9,0x7cbd,0x7cc0
,0x7cc5,0x7cc2,0x7cd8,0x7cd2,0x7cdc,0x7ce2,0x9b3b,0x7cef
,0x7cf2,0x7cf4,0x7cf6,0x7cfa,0x7d06,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x7d02,0x7d1c,0x7d15,0x7d0a,0x7d45,0x7d4b,0x7d2e,0x7d32
,0x7d3f,0x7d35,0x7d46,0x7d73,0x7d56,0x7d4e,0x7d72,0x7d68
,0x7d6e,0x7d4f,0x7d63,0x7d93,0x7d89,0x7d5b,0x7d8f,0x7d7d
,0x7d9b,0x7dba,0x7dae,0x7da3,0x7db5,0x7dc7,0x7dbd,0x7dab
,0x7e3d,0x7da2,0x7daf,0x7ddc,0x7db8,0x7d9f,0x7db0,0x7dd8
,0x7ddd,0x7de4,0x7dde,0x7dfb,0x7df2,0x7de1,0x7e05,0x7e0a
,0x7e23,0x7e21,0x7e12,0x7e31,0x7e1f,0x7e09,0x7e0b,0x7e22
,0x7e46,0x7e66,0x7e3b,0x7e35,0x7e39,0x7e43,0x7e37,0
,0x7e32,0x7e3a,0x7e67,0x7e5d,0x7e56,0x7e5e,0x7e59,0x7e5a
,0x7e79,0x7e6a,0x7e69,0x7e7c,0x7e7b,0x7e83,0x7dd5,0x7e7d
,0x8fae,0x7e7f,0x7e88,0x7e89,0x7e8c,0x7e92,0x7e90,0x7e93
,0x7e94,0x7e96,0x7e8e,0x7e9b,0x7e9c,0x7f38,0x7f3a,0x7f45
,0x7f4c,0x7f4d,0x7f4e,0x7f50,0x7f51,0x7f55,0x7f54,0x7f58
,0x7f5f,0x7f60,0x7f68,0x7f69,0x7f67,0x7f78,0x7f82,0x7f86
,0x7f83,0x7f88,0x7f87,0x7f8c,0x7f94,0x7f9e,0x7f9d,0x7f9a
,0x7fa3,0x7faf,0x7fb2,0x7fb9,0x7fae,0x7fb6,0x7fb8,0x8b71
,0x7fc5,0x7fc6,0x7fca,0x7fd5,0x7fd4,0x7fe1,0x7fe6,0x7fe9
,0x7ff3,0x7ff9,0x98dc,0x8006,0x8004,0x800b,0x8012,0x8018
,0x8019,0x801c,0x8021,0x8028,0x803f,0x803b,0x804a,0x8046
,0x8052,0x8058,0x805a,0x805f,0x8062,0x8068,0x8073,0x8072
,0x8070,0x8076,0x8079,0x807d,0x807f,0x8084,0x8086,0x8085
,0x809b,0x8093,0x809a,0x80ad,0x5190,0x80ac,0x80db,0x80e5
,0x80d9,0x80dd,0x80c4,0x80da,0x80d6,0x8109,0x80ef,0x80f1
,0x811b,0x8129,0x8123,0x812f,0x814b,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x968b,0x8146,0x813e,0x8153,0x8151,0x80fc,0x8171,0x816e
,0x8165,0x8166,0x8174,0x8183,0x8188,0x818a,0x8180,0x8182
,0x81a0,0x8195,0x81a4,0x81a3,0x815f,0x8193,0x81a9,0x81b0
,0x81b5,0x81be,0x81b8,0x81bd,0x81c0,0x81c2,0x81ba,0x81c9
,0x81cd,0x81d1,0x81d9,0x81d8,0x81c8,0x81da,0x81df,0x81e0
,0x81e7,0x81fa,0x81fb,0x81fe,0x8201,0x8202,0x8205,0x8207
,0x820a,0x820d,0x8210,0x8216,0x8229,0x822b,0x8238,0x8233
,0x8240,0x8259,0x8258,0x825d,0x825a,0x825f,0x8264,0
,0x8262,0x8268,0x826a,0x826b,0x822e,0x8271,0x8277,0x8278
,0x827e,0x828d,0x8292,0x82ab,0x829f,0x82bb,0x82ac,0x82e1
,0x82e3,0x82df,0x82d2,0x82f4,0x82f3,0x82fa,0x8393,0x8303
,0x82fb,0x82f9,0x82de,0x8306,0x82dc,0x8309,0x82d9,0x8335
,0x8334,0x8316,0x8332,0x8331,0x8340,0x8339,0x8350,0x8345
,0x832f,0x832b,0x8317,0x8318,0x8385,0x839a,0x83aa,0x839f
,0x83a2,0x8396,0x8323,0x838e,0x8387,0x838a,0x837c,0x83b5
,0x8373,0x8375,0x83a0,0x8389,0x83a8,0x83f4,0x8413,0x83eb
,0x83ce,0x83fd,0x8403,0x83d8,0x840b,0x83c1,0x83f7,0x8407
,0x83e0,0x83f2,0x840d,0x8422,0x8420,0x83bd,0x8438,0x8506
,0x83fb,0x846d,0x842a,0x843c,0x855a,0x8484,0x8477,0x846b
,0x84ad,0x846e,0x8482,0x8469,0x8446,0x842c,0x846f,0x8479
,0x8435,0x84ca,0x8462,0x84b9,0x84bf,0x849f,0x84d9,0x84cd
,0x84bb,0x84da,0x84d0,0x84c1,0x84c6,0x84d6,0x84a1,0x8521
,0x84ff,0x84f4,0x8517,0x8518,0x852c,0x851f,0x8515,0x8514
,0x84fc,0x8540,0x8563,0x8558,0x8548,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x8541,0x8602,0x854b,0x8555,0x8580,0x85a4,0x8588,0x8591
,0x858a,0x85a8,0x856d,0x8594,0x859b,0x85ea,0x8587,0x859c
,0x8577,0x857e,0x8590,0x85c9,0x85ba,0x85cf,0x85b9,0x85d0
,0x85d5,0x85dd,0x85e5,0x85dc,0x85f9,0x860a,0x8613,0x860b
,0x85fe,0x85fa,0x8606,0x8622,0x861a,0x8630,0x863f,0x864d
,0x4e55,0x8654,0x865f,0x8667,0x8671,0x8693,0x86a3,0x86a9
,0x86aa,0x868b,0x868c,0x86b6,0x86af,0x86c4,0x86c6,0x86b0
,0x86c9,0x8823,0x86ab,0x86d4,0x86de,0x86e9,0x86ec,0
,0x86df,0x86db,0x86ef,0x8712,0x8706,0x8708,0x8700,0x8703
,0x86fb,0x8711,0x8709,0x870d,0x86f9,0x870a,0x8734,0x873f
,0x8737,0x873b,0x8725,0x8729,0x871a,0x8760,0x875f,0x8778
,0x874c,0x874e,0x8774,0x8757,0x8768,0x876e,0x8759,0x8753
,0x8763,0x876a,0x8805,0x87a2,0x879f,0x8782,0x87af,0x87cb
,0x87bd,0x87c0,0x87d0,0x96d6,0x87ab,0x87c4,0x87b3,0x87c7
,0x87c6,0x87bb,0x87ef,0x87f2,0x87e0,0x880f,0x880d,0x87fe
,0x87f6,0x87f7,0x880e,0x87d2,0x8811,0x8816,0x8815,0x8822
,0x8821,0x8831,0x8836,0x8839,0x8827,0x883b,0x8844,0x8842
,0x8852,0x8859,0x885e,0x8862,0x886b,0x8881,0x887e,0x889e
,0x8875,0x887d,0x88b5,0x8872,0x8882,0x8897,0x8892,0x88ae
,0x8899,0x88a2,0x888d,0x88a4,0x88b0,0x88bf,0x88b1,0x88c3
,0x88c4,0x88d4,0x88d8,0x88d9,0x88dd,0x88f9,0x8902,0x88fc
,0x88f4,0x88e8,0x88f2,0x8904,0x890c,0x890a,0x8913,0x8943
,0x891e,0x8925,0x892a,0x892b,0x8941,0x8944,0x893b,0x8936
,0x8938,0x894c,0x891d,0x8960,0x895e,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x8966,0x8964,0x896d,0x896a,0x896f,0x8974,0x8977,0x897e
,0x8983,0x8988,0x898a,0x8993,0x8998,0x89a1,0x89a9,0x89a6
,0x89ac,0x89af,0x89b2,0x89ba,0x89bd,0x89bf,0x89c0,0x89da
,0x89dc,0x89dd,0x89e7,0x89f4,0x89f8,0x8a03,0x8a16,0x8a10
,0x8a0c,0x8a1b,0x8a1d,0x8a25,0x8a36,0x8a41,0x8a5b,0x8a52
,0x8a46,0x8a48,0x8a7c,0x8a6d,0x8a6c,0x8a62,0x8a85,0x8a82
,0x8a84,0x8aa8,0x8aa1,0x8a91,0x8aa5,0x8aa6,0x8a9a,0x8aa3
,0x8ac4,0x8acd,0x8ac2,0x8ada,0x8aeb,0x8af3,0x8ae7,0
,0x8ae4,0x8af1,0x8b14,0x8ae0,0x8ae2,0x8af7,0x8ade,0x8adb
,0x8b0c,0x8b07,0x8b1a,0x8ae1,0x8b16,0x8b10,0x8b17,0x8b20
,0x8b33,0x97ab,0x8b26,0x8b2b,0x8b3e,0x8b28,0x8b41,0x8b4c
,0x8b4f,0x8b4e,0x8b49,0x8b56,0x8b5b,0x8b5a,0x8b6b,0x8b5f
,0x8b6c,0x8b6f,0x8b74,0x8b7d,0x8b80,0x8b8c,0x8b8e,0x8b92
,0x8b93,0x8b96,0x8b99,0x8b9a,0x8c3a,0x8c41,0x8c3f,0x8c48
,0x8c4c,0x8c4e,0x8c50,0x8c55,0x8c62,0x8c6c,0x8c78,0x8c7a
,0x8c82,0x8c89,0x8c85,0x8c8a,0x8c8d,0x8c8e,0x8c94,0x8c7c
,0x8c98,0x621d,0x8cad,0x8caa,0x8cbd,0x8cb2,0x8cb3,0x8cae
,0x8cb6,0x8cc8,0x8cc1,0x8ce4,0x8ce3,0x8cda,0x8cfd,0x8cfa
,0x8cfb,0x8d04,0x8d05,0x8d0a,0x8d07,0x8d0f,0x8d0d,0x8d10
,0x9f4e,0x8d13,0x8ccd,0x8d14,0x8d16,0x8d67,0x8d6d,0x8d71
,0x8d73,0x8d81,0x8d99,0x8dc2,0x8dbe,0x8dba,0x8dcf,0x8dda
,0x8dd6,0x8dcc,0x8ddb,0x8dcb,0x8dea,0x8deb,0x8ddf,0x8de3
,0x8dfc,0x8e08,0x8e09,0x8dff,0x8e1d,0x8e1e,0x8e10,0x8e1f
,0x8e42,0x8e35,0x8e30,0x8e34,0x8e4a,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x8e47,0x8e49,0x8e4c,0x8e50,0x8e48,0x8e59,0x8e64,0x8e60
,0x8e2a,0x8e63,0x8e55,0x8e76,0x8e72,0x8e7c,0x8e81,0x8e87
,0x8e85,0x8e84,0x8e8b,0x8e8a,0x8e93,0x8e91,0x8e94,0x8e99
,0x8eaa,0x8ea1,0x8eac,0x8eb0,0x8ec6,0x8eb1,0x8ebe,0x8ec5
,0x8ec8,0x8ecb,0x8edb,0x8ee3,0x8efc,0x8efb,0x8eeb,0x8efe
,0x8f0a,0x8f05,0x8f15,0x8f12,0x8f19,0x8f13,0x8f1c,0x8f1f
,0x8f1b,0x8f0c,0x8f26,0x8f33,0x8f3b,0x8f39,0x8f45,0x8f42
,0x8f3e,0x8f4c,0x8f49,0x8f46,0x8f4e,0x8f57,0x8f5c,0
,0x8f62,0x8f63,0x8f64,0x8f9c,0x8f9f,0x8fa3,0x8fad,0x8faf
,0x8fb7,0x8fda,0x8fe5,0x8fe2,0x8fea,0x8fef,0x9087,0x8ff4
,0x9005,0x8ff9,0x8ffa,0x9011,0x9015,0x9021,0x900d,0x901e
,0x9016,0x900b,0x9027,0x9036,0x9035,0x9039,0x8ff8,0x904f
,0x9050,0x9051,0x9052,0x900e,0x9049,0x903e,0x9056,0x9058
,0x905e,0x9068,0x906f,0x9076,0x96a8,0x9072,0x9082,0x907d
,0x9081,0x9080,0x908a,0x9089,0x908f,0x90a8,0x90af,0x90b1
,0x90b5,0x90e2,0x90e4,0x6248,0x90db,0x9102,0x9112,0x9119
,0x9132,0x9130,0x914a,0x9156,0x9158,0x9163,0x9165,0x9169
,0x9173,0x9172,0x918b,0x9189,0x9182,0x91a2,0x91ab,0x91af
,0x91aa,0x91b5,0x91b4,0x91ba,0x91c0,0x91c1,0x91c9,0x91cb
,0x91d0,0x91d6,0x91df,0x91e1,0x91db,0x91fc,0x91f5,0x91f6
,0x921e,0x91ff,0x9214,0x922c,0x9215,0x9211,0x925e,0x9257
,0x9245,0x9249,0x9264,0x9248,0x9295,0x923f,0x924b,0x9250
,0x929c,0x9296,0x9293,0x929b,0x925a,0x92cf,0x92b9,0x92b7
,0x92e9,0x930f,0x92fa,0x9344,0x932e,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x9319,0x9322,0x931a,0x9323,0x933a,0x9335,0x933b,0x935c
,0x9360,0x937c,0x936e,0x9356,0x93b0,0x93ac,0x93ad,0x9394
,0x93b9,0x93d6,0x93d7,0x93e8,0x93e5,0x93d8,0x93c3,0x93dd
,0x93d0,0x93c8,0x93e4,0x941a,0x9414,0x9413,0x9403,0x9407
,0x9410,0x9436,0x942b,0x9435,0x9421,0x943a,0x9441,0x9452
,0x9444,0x945b,0x9460,0x9462,0x945e,0x946a,0x9229,0x9470
,0x9475,0x9477,0x947d,0x945a,0x947c,0x947e,0x9481,0x947f
,0x9582,0x9587,0x958a,0x9594,0x9596,0x9598,0x9599,0
,0x95a0,0x95a8,0x95a7,0x95ad,0x95bc,0x95bb,0x95b9,0x95be
,0x95ca,0x6ff6,0x95c3,0x95cd,0x95cc,0x95d5,0x95d4,0x95d6
,0x95dc,0x95e1,0x95e5,0x95e2,0x9621,0x9628,0x962e,0x962f
,0x9642,0x964c,0x964f,0x964b,0x9677,0x965c,0x965e,0x965d
,0x965f,0x9666,0x9672,0x966c,0x968d,0x9698,0x9695,0x9697
,0x96aa,0x96a7,0x96b1,0x96b2,0x96b0,0x96b4,0x96b6,0x96b8
,0x96b9,0x96ce,0x96cb,0x96c9,0x96cd,0x894d,0x96dc,0x970d
,0x96d5,0x96f9,0x9704,0x9706,0x9708,0x9713,0x970e,0x9711
,0x970f,0x9716,0x9719,0x9724,0x972a,0x9730,0x9739,0x973d
,0x973e,0x9744,0x9746,0x9748,0x9742,0x9749,0x975c,0x9760
,0x9764,0x9766,0x9768,0x52d2,0x976b,0x9771,0x9779,0x9785
,0x977c,0x9781,0x977a,0x9786,0x978b,0x978f,0x9790,0x979c
,0x97a8,0x97a6,0x97a3,0x97b3,0x97b4,0x97c3,0x97c6,0x97c8
,0x97cb,0x97dc,0x97ed,0x9f4f,0x97f2,0x7adf,0x97f6,0x97f5
,0x980f,0x980c,0x9838,0x9824,0x9821,0x9837,0x983d,0x9846
,0x984f,0x984b,0x986b,0x986f,0x9870,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x9871,0x9874,0x9873,0x98aa,0x98af,0x98b1,0x98b6,0x98c4
,0x98c3,0x98c6,0x98e9,0x98eb,0x9903,0x9909,0x9912,0x9914
,0x9918,0x9921,0x991d,0x991e,0x9924,0x9920,0x992c,0x992e
,0x993d,0x993e,0x9942,0x9949,0x9945,0x9950,0x994b,0x9951
,0x9952,0x994c,0x9955,0x9997,0x9998,0x99a5,0x99ad,0x99ae
,0x99bc,0x99df,0x99db,0x99dd,0x99d8,0x99d1,0x99ed,0x99ee
,0x99f1,0x99f2,0x99fb,0x99f8,0x9a01,0x9a0f,0x9a05,0x99e2
,0x9a19,0x9a2b,0x9a37,0x9a45,0x9a42,0x9a40,0x9a43,0
,0x9a3e,0x9a55,0x9a4d,0x9a5b,0x9a57,0x9a5f,0x9a62,0x9a65
,0x9a64,0x9a69,0x9a6b,0x9a6a,0x9aad,0x9ab0,0x9abc,0x9ac0
,0x9acf,0x9ad1,0x9ad3,0x9ad4,0x9ade,0x9adf,0x9ae2,0x9ae3
,0x9ae6,0x9aef,0x9aeb,0x9aee,0x9af4,0x9af1,0x9af7,0x9afb
,0x9b06,0x9b18,0x9b1a,0x9b1f,0x9b22,0x9b23,0x9b25,0x9b27
,0x9b28,0x9b29,0x9b2a,0x9b2e,0x9b2f,0x9b32,0x9b44,0x9b43
,0x9b4f,0x9b4d,0x9b4e,0x9b51,0x9b58,0x9b74,0x9b93,0x9b83
,0x9b91,0x9b96,0x9b97,0x9b9f,0x9ba0,0x9ba8,0x9bb4,0x9bc0
,0x9bca,0x9bb9,0x9bc6,0x9bcf,0x9bd1,0x9bd2,0x9be3,0x9be2
,0x9be4,0x9bd4,0x9be1,0x9c3a,0x9bf2,0x9bf1,0x9bf0,0x9c15
,0x9c14,0x9c09,0x9c13,0x9c0c,0x9c06,0x9c08,0x9c12,0x9c0a
,0x9c04,0x9c2e,0x9c1b,0x9c25,0x9c24,0x9c21,0x9c30,0x9c47
,0x9c32,0x9c46,0x9c3e,0x9c5a,0x9c60,0x9c67,0x9c76,0x9c78
,0x9ce7,0x9cec,0x9cf0,0x9d09,0x9d08,0x9ceb,0x9d03,0x9d06
,0x9d2a,0x9d26,0x9daf,0x9d23,0x9d1f,0x9d44,0x9d15,0x9d12
,0x9d41,0x9d3f,0x9d3e,0x9d46,0x9d48,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x9d5d,0x9d5e,0x9d64,0x9d51,0x9d50,0x9d59,0x9d72,0x9d89
,0x9d87,0x9dab,0x9d6f,0x9d7a,0x9d9a,0x9da4,0x9da9,0x9db2
,0x9dc4,0x9dc1,0x9dbb,0x9db8,0x9dba,0x9dc6,0x9dcf,0x9dc2
,0x9dd9,0x9dd3,0x9df8,0x9de6,0x9ded,0x9def,0x9dfd,0x9e1a
,0x9e1b,0x9e1e,0x9e75,0x9e79,0x9e7d,0x9e81,0x9e88,0x9e8b
,0x9e8c,0x9e92,0x9e95,0x9e91,0x9e9d,0x9ea5,0x9ea9,0x9eb8
,0x9eaa,0x9ead,0x9761,0x9ecc,0x9ece,0x9ecf,0x9ed0,0x9ed4
,0x9edc,0x9ede,0x9edd,0x9ee0,0x9ee5,0x9ee8,0x9eef,0
,0x9ef4,0x9ef6,0x9ef7,0x9ef9,0x9efb,0x9efc,0x9efd,0x9f07
,0x9f08,0x76b7,0x9f15,0x9f21,0x9f2c,0x9f3e,0x9f4a,0x9f52
,0x9f54,0x9f63,0x9f5f,0x9f60,0x9f61,0x9f66,0x9f67,0x9f6c
,0x9f6a,0x9f77,0x9f72,0x9f76,0x9f95,0x9f9c,0x9fa0,0x582f
,0x69c7,0x9059,0x7464,0x51dc,0x7199,0x5653,0x5de2,0x5e14
,0x5e18,0x5e58,0x5e5e,0x5ebe,0xf928,0x5ecb,0x5ef9,0x5f00
,0x5f02,0x5f07,0x5f1d,0x5f23,0x5f34,0x5f36,0x5f3d,0x5f40
,0x5f45,0x5f54,0x5f58,0x5f64,0x5f67,0x5f7d,0x5f89,0x5f9c
,0x5fa7,0x5faf,0x5fb5,0x5fb7,0x5fc9,0x5fde,0x5fe1,0x5fe9
,0x600d,0x6014,0x6018,0x6033,0x6035,0x6047,0xfa3d,0x609d
,0x609e,0x60cb,0x60d4,0x60d5,0x60dd,0x60f8,0x611c,0x612b
,0x6130,0x6137,0xfa3e,0x618d,0xfa3f,0x61bc,0x61b9,0xfa40
,0x6222,0x623e,0x6243,0x6256,0x625a,0x626f,0x6285,0x62c4
,0x62d6,0x62fc,0x630a,0x6318,0x6339,0x6343,0x6365,0x637c
,0x63e5,0x63ed,0x63f5,0x6410,0x6414,0x6422,0x6479,0x6451
,0x6460,0x646d,0x64ce,0x64be,0x64bf,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x64c4,0x64ca,0x64d0,0x64f7,0x64fb,0x6522,0x6529,0xfa41
,0x6567,0x659d,0xfa42,0x6600,0x6609,0x6615,0x661e,0x663a
,0x6622,0x6624,0x662b,0x6630,0x6631,0x6633,0x66fb,0x6648
,0x664c,0xaaf1,0x6659,0x665a,0x6661,0x6665,0x6673,0x6677
,0x6678,0x668d,0xfa43,0x66a0,0x66b2,0x66bb,0x66c6,0x66c8
,0x3b22,0x66db,0x66e8,0x66fa,0x6713,0xf929,0x6733,0x6766
,0x6747,0x6748,0x677b,0x6781,0x6793,0x6798,0x679b,0x67bb
,0x67f9,0x67c0,0x67d7,0x67fc,0x6801,0x6852,0x681d,0
,0x682c,0x6831,0x685b,0x6872,0x6875,0xfa44,0x68a3,0x68a5
,0x68b2,0x68c8,0x68d0,0x68e8,0x68ed,0x68f0,0x68f1,0x68fc
,0x690a,0x6949,0xab03,0x6935,0x6942,0x6957,0x6963,0x6964
,0x6968,0x6980,0xfa14,0x69a5,0x69ad,0x69cf,0x3bb6,0x3bc3
,0x69e2,0x69e9,0xfa45,0x69f5,0x69f6,0x6a0f,0x6a15,0xab09
,0x6a3b,0x6a3e,0x6a45,0x6a50,0x6a56,0x6a5b,0x6a6b,0x6a73
,0xab0b,0x6a89,0x6a94,0x6a9d,0x6a9e,0x6aa5,0x6ae4,0x6ae7
,0x3c0f,0xf91d,0x6b1b,0x6b1e,0x6b2c,0x6b35,0x6b46,0x6b56
,0x6b60,0x6b65,0x6b67,0x6b77,0x6b82,0x6ba9,0x6bad,0xf970
,0x6bcf,0x6bd6,0x6bd7,0x6bff,0x6c05,0x6c10,0x6c33,0x6c59
,0x6c5c,0x6caa,0x6c74,0x6c76,0x6c85,0x6c86,0x6c98,0x6c9c
,0x6cfb,0x6cc6,0x6cd4,0x6ce0,0x6ceb,0x6cee,0xab14,0x6d04
,0x6d0e,0x6d2e,0x6d31,0x6d39,0x6d3f,0x6d58,0x6d65,0xfa46
,0x6d82,0x6d87,0x6d89,0x6d94,0x6daa,0x6dac,0x6dbf,0x6dc4
,0x6dd6,0x6dda,0x6ddb,0x6ddd,0x6dfc,0xfa47,0x6e34,0x6e44
,0x6e5c,0x6e5e,0x6eab,0x6eb1,0x6ec1,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x6ec7,0x6ece,0x6f10,0x6f1a,0xfa48,0x6f2a,0x6f2f,0x6f33
,0x6f51,0x6f59,0x6f5e,0x6f61,0x6f62,0x6f7e,0x6f88,0x6f8c
,0x6f8d,0x6f94,0x6fa0,0x6fa7,0x6fb6,0x6fbc,0x6fc7,0x6fca
,0x6ff9,0x6ff0,0x6ff5,0x7005,0x7006,0x7028,0x704a,0x705d
,0x705e,0x704e,0x7064,0x7075,0x7085,0x70a4,0x70ab,0x70b7
,0x70d4,0x70d8,0x70e4,0x710f,0x712b,0x711e,0x7120,0x712e
,0x7130,0x7146,0x7147,0x7151,0xfa49,0x7152,0x715c,0x7160
,0x7168,0xfa15,0x7185,0x7187,0x7192,0x71c1,0x71ba,0
,0x71c4,0x71fe,0x7200,0x7215,0x7255,0x7256,0x3e3f,0x728d
,0x729b,0x72be,0x72c0,0x72fb,0xab22,0x7327,0x7328,0xfa16
,0x7350,0x7366,0x737c,0x7395,0x739f,0x73a0,0x73a2,0x73a6
,0x73ab,0x73c9,0x73cf,0x73d6,0x73d9,0x73e3,0x73e9,0x7407
,0x740a,0x741a,0x741b,0xfa4b,0x7426,0x7428,0x742a,0x742b
,0x742c,0x742e,0x742f,0x7430,0x7444,0x7446,0x7447,0x744b
,0x7457,0x7462,0x746b,0x746d,0x7486,0x7487,0x7489,0x7498
,0x749c,0x749f,0x74a3,0x7490,0x74a6,0x74a8,0x74a9,0x74b5
,0x74bf,0x74c8,0x74c9,0x74da,0x74ff,0x7501,0x7517,0x752f
,0x756f,0x7579,0x7592,0x3f72,0x75ce,0x75e4,0x7600,0x7602
,0x7608,0x7615,0x7616,0x7619,0x761e,0x762d,0x7635,0x7643
,0x764b,0x7664,0x7665,0x766d,0x766f,0x7671,0x7681,0x769b
,0x769d,0x769e,0x76a6,0x76aa,0x76b6,0x76c5,0x76cc,0x76ce
,0x76d4,0x76e6,0x76f1,0x76fc,0x770a,0x7719,0x7734,0x7736
,0x7746,0x774d,0x774e,0x775c,0x775f,0x7762,0x777a,0x7780
,0x7794,0x77aa,0x77e0,0x782d,0xab36,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x7843,0x784e,0x784f,0x7851,0x7868,0x786e,0xfa4c,0x78b0
,0xab38,0x78ad,0x78e4,0x78f2,0x7900,0x78f7,0x791c,0x792e
,0x7931,0x7934,0xfa4d,0xfa4e,0x7945,0x7946,0xfa4f,0xfa50
,0xfa51,0x795c,0xfa52,0xfa19,0xfa1a,0x7979,0xfa53,0xfa54
,0xfa1b,0x7998,0x79b1,0x79b8,0x79c8,0x79ca,0xab3a,0x79d4
,0x79de,0x79eb,0x79ed,0x7a03,0xfa55,0x7a39,0x7a5d,0x7a6d
,0xfa56,0x7a85,0x7aa0,0xab3d,0x7ab3,0x7abb,0x7ace,0x7aeb
,0x7afd,0x7b12,0x7b2d,0x7b3b,0x7b47,0x7b4e,0x7b60,0
,0x7b6d,0x7b6f,0x7b72,0x7b9e,0xfa57,0x7bd7,0x7bd9,0x7c01
,0x7c31,0x7c1e,0x7c20,0x7c33,0x7c36,0x4264,0xab45,0x7c59
,0x7c6d,0x7c79,0x7c8f,0x7c94,0x7ca0,0x7cbc,0x7cd5,0x7cd9
,0x7cdd,0x7d07,0x7d08,0x7d13,0x7d1d,0x7d23,0x7d31,0x7d41
,0x7d48,0x7d53,0x7d5c,0x7d7a,0x7d83,0x7d8b,0x7da0,0x7da6
,0x7dc2,0x7dcc,0x7dd6,0x7de3,0xfa58,0x7e28,0x7e08,0x7e11
,0x7e15,0xfa5a,0x7e47,0x7e52,0x7e61,0x7e8a,0x7e8d,0x7f47
,0xfa5b,0x7f91,0x7f97,0x7fbf,0x7fce,0x7fdb,0x7fdf,0x7fec
,0x7fee,0x7ffa,0xfa5c,0x8014,0x8026,0x8035,0x8037,0x803c
,0x80ca,0x80d7,0x80e0,0x80f3,0x8118,0x814a,0x8160,0x8167
,0x8168,0x816d,0x81bb,0x81ca,0x81cf,0x81d7,0xfa5d,0x4453
,0x445b,0x8260,0x8274,0xab68,0x828e,0x82a1,0x82a3,0x82a4
,0x82a9,0x82ae,0x82b7,0x82be,0x82bf,0x82c6,0x82d5,0x82fd
,0x82fe,0x8300,0x8301,0x8362,0x8322,0x832d,0x833a,0x8343
,0x8347,0x8351,0x8355,0x837d,0x8386,0x8392,0x8398,0x83a7
,0x83a9,0x83bf,0x83c0,0x83c7,0x83cf,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x83d1,0x83e1,0x83ea,0x8401,0x8406,0x840a,0xfa5f,0x8448
,0x845f,0x8470,0x8473,0x8485,0x849e,0x84af,0x84b4,0x84ba
,0x84c0,0x84c2,0xab6b,0x8532,0x851e,0x8523,0x852f,0x8559
,0x8564,0xfa1f,0x85ad,0x857a,0x858c,0x858f,0x85a2,0x85b0
,0x85cb,0x85ce,0x85ed,0x8612,0x85ff,0x8604,0x8605,0x8610
,0xab71,0x8618,0x8629,0x8638,0x8657,0x865b,0xf936,0x8662
,0x459d,0x866c,0x8675,0x8698,0x86b8,0x86fa,0x86fc,0x86fd
,0x870b,0x8771,0x8787,0x8788,0x87ac,0x87ad,0x87b5,0
,0x45ea,0x87d6,0x87ec,0x8806,0x880a,0x8810,0x8814,0x881f
,0x8898,0x88aa,0x88ca,0x88ce,0xab7c,0x88f5,0x891c,0xfa60
,0x8918,0x8919,0x891a,0x8927,0x8930,0x8932,0x8939,0x8940
,0x8994,0xfa61,0x89d4,0x89e5,0x89f6,0x8a12,0x8a15,0x8a22
,0x8a37,0x8a47,0x8a4e,0x8a5d,0x8a61,0x8a75,0x8a79,0x8aa7
,0x8ad0,0x8adf,0x8af4,0x8af6,0xfa22,0xfa62,0xfa63,0x8b46
,0x8b54,0x8b59,0x8b69,0x8b9d,0x8c49,0x8c68,0xfa64,0x8ce1
,0x8cf4,0x8cf8,0x8cfe,0xfa65,0x8d12,0x8d1b,0x8daf,0x8dce
,0x8dd1,0x8dd7,0x8e20,0x8e23,0x8e3d,0x8e70,0x8e7b,0xab8c
,0x8ec0,0x4844,0x8efa,0x8f1e,0x8f2d,0x8f36,0x8f54,0xab8f
,0x8fa6,0x8fb5,0x8fe4,0x8fe8,0x8fee,0x9008,0x902d,0xfa67
,0x9088,0x9095,0x9097,0x9099,0x909b,0x90a2,0x90b3,0x90be
,0x90c4,0x90c5,0x90c7,0x90d7,0x90dd,0x90de,0x90ef,0x90f4
,0xfa26,0x9114,0x9115,0x9116,0x9122,0x9123,0x9127,0x912f
,0x9131,0x9134,0x913d,0x9148,0x915b,0x9183,0x919e,0x91ac
,0x91b1,0x91bc,0x91d7,0x91fb,0x91e4,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x91e5,0x91ed,0x91f1,0x9207,0x9210,0x9238,0x9239,0x923a
,0x923c,0x9240,0x9243,0x924f,0x9278,0x9288,0x92c2,0x92cb
,0x92cc,0x92d3,0x92e0,0x92ff,0x9304,0x931f,0x9321,0x9325
,0x9348,0x9349,0x934a,0x9364,0x9365,0x936a,0x9370,0x939b
,0x93a3,0x93ba,0x93c6,0x93de,0x93df,0x9404,0x93fd,0x9433
,0x944a,0x9463,0x946b,0x9471,0x9472,0x958e,0x959f,0x95a6
,0x95a9,0x95ac,0x95b6,0x95bd,0x95cb,0x95d0,0x95d3,0x49b0
,0x95da,0x95de,0x9658,0x9684,0xf9dc,0x969d,0x96a4,0
,0x96a5,0x96d2,0x96de,0xfa68,0x96e9,0x96ef,0x9733,0x973b
,0x974d,0x974e,0x974f,0x975a,0x976e,0x9773,0x9795,0x97ae
,0x97ba,0x97c1,0x97c9,0x97de,0x97db,0x97f4,0xfa69,0x980a
,0x981e,0x982b,0x9830,0xfa6a,0x9852,0x9853,0x9856,0x9857
,0x9859,0x985a,0xf9d0,0x9865,0x986c,0x98ba,0x98c8,0x98e7
,0x9958,0x999e,0x9a02,0x9a03,0x9a24,0x9a2d,0x9a2e,0x9a38
,0x9a4a,0x9a4e,0x9a52,0x9ab6,0x9ac1,0x9ac3,0x9ace,0x9ad6
,0x9af9,0x9b02,0x9b08,0x9b20,0x4c17,0x9b2d,0x9b5e,0x9b79
,0x9b66,0x9b72,0x9b75,0x9b84,0x9b8a,0x9b8f,0x9b9e,0x9ba7
,0x9bc1,0x9bce,0x9be5,0x9bf8,0x9bfd,0x9c00,0x9c23,0x9c41
,0x9c4f,0x9c50,0x9c53,0x9c63,0x9c65,0x9c77,0x9d1d,0x9d1e
,0x9d43,0x9d47,0x9d52,0x9d63,0x9d70,0x9d7c,0x9d8a,0x9d96
,0x9dc0,0x9dac,0x9dbc,0x9dd7,0xabc8,0x9de7,0x9e07,0x9e15
,0x9e7c,0x9e9e,0x9ea4,0x9eac,0x9eaf,0x9eb4,0x9eb5,0x9ec3
,0x9ed1,0x9f10,0x9f39,0x9f57,0x9f90,0x9f94,0x9f97,0x9fa2
,0x59f8,0x5c5b,0x5e77,0x7626,0x7e6b,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xaaa1,0x4e02,0x4e0f,0x4e12,0x4e29,0x4e2b,0x4e2e,0x4e40
,0x4e47,0x4e48,0xaaa3,0x4e51,0x3406,0xaaa4,0x4e5a,0x4e69
,0x4e9d,0x342c,0x342e,0x4eb9,0x4ebb,0xaaa5,0x4ebc,0x4ec3
,0x4ec8,0x4ed0,0x4eeb,0x4eda,0x4ef1,0x4ef5,0x4f00,0x4f16
,0x4f64,0x4f37,0x4f3e,0x4f54,0x4f58,0xaaa6,0x4f77,0x4f78
,0x4f7a,0x4f7d,0x4f82,0x4f85,0x4f92,0x4f9a,0x4fe6,0x4fb2
,0x4fbe,0x4fc5,0x4fcb,0x4fcf,0x4fd2,0x346a,0x4ff2,0x5000
,0x5010,0x5013,0x501c,0x501e,0x5022,0x3468,0x5042,0
,0x5046,0x504e,0x5053,0x5057,0x5063,0x5066,0x506a,0x5070
,0x50a3,0x5088,0x5092,0x5093,0x5095,0x5096,0x509c,0x50aa
,0xaaa7,0x50b1,0x50ba,0x50bb,0x50c4,0x50c7,0x50f3,0xaaa9
,0x50ce,0xaaa8,0x50d4,0x50d9,0x50e1,0x50e9,0x3492,0x5b96
,0x5bac,0x3761,0x5bc0,0x3762,0x5bce,0x5bd6,0x376c,0x376b
,0x5bf1,0x5bfd,0x3775,0x5c03,0x5c29,0x5c30,0xaad2,0x5c5f
,0x5c63,0x5c67,0x5c68,0x5c69,0x5c70,0xaad3,0xaad5,0x5c7c
,0xaad6,0xaad7,0x5c88,0x5c8a,0x37c1,0xaad9,0xaada,0x5ca0
,0x5ca2,0x5ca6,0x5ca7,0xaad8,0x5cad,0x5cb5,0xaadb,0x5cc9
,0xaadc,0xaade,0x5d06,0x5d10,0x5d2b,0x5d1d,0x5d20,0x5d24
,0x5d26,0x5d31,0x5d39,0x5d42,0x37e8,0x5d61,0x5d6a,0x37f4
,0x5d70,0xaadf,0x37fd,0x5d88,0x3800,0x5d92,0x5d94,0x5d97
,0x5d99,0x5db0,0x5db2,0x5db4,0xaae0,0x5db9,0x5dd1,0x5dd7
,0x5dd8,0x5de0,0xaae1,0x5de4,0x5de9,0x382f,0x5e00,0x3836
,0x5e12,0x5e15,0x3840,0x5e1f,0x5e2e,0x5e3e,0x5e49,0x385c
,0x5e56,0x3861,0x5e6b,0x5e6c,0x5e6d,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x5108,0xaaaa,0x5117,0x511b,0xaaab,0x5160,0xaaac,0x5173
,0x5183,0x518b,0x34bc,0x5198,0x51a3,0x51ad,0x34c7,0x51bc
,0xaaad,0xaaae,0x51f3,0x51f4,0x5202,0x5212,0x5216,0xaaaf
,0x5255,0x525c,0x526c,0x5277,0x5284,0x5282,0xaab0,0x5298
,0xaab1,0x52a4,0x52a6,0x52af,0x52ba,0x52bb,0x52ca,0x351f
,0x52d1,0xaab2,0x52f7,0x530a,0x530b,0x5324,0x5335,0x533e
,0x5342,0xaab3,0xaab4,0x5367,0x536c,0x537a,0x53a4,0x53b4
,0xaab5,0x53b7,0x53c0,0xaab6,0x355d,0x355e,0x53d5,0
,0x53da,0x3563,0x53f4,0x53f5,0x5455,0x5424,0x5428,0x356e
,0x5443,0x5462,0x5466,0x546c,0x548a,0x548d,0x5495,0x54a0
,0x54a6,0x54ad,0x54ae,0x54b7,0x54ba,0x54bf,0x54c3,0xaab7
,0x54ec,0x54ef,0x54f1,0x54f3,0x5500,0x5501,0x5509,0x553c
,0x5541,0x35a6,0x5547,0x554a,0x35a8,0x5560,0x5561,0x5564
,0xaab8,0x557d,0x5582,0x5588,0x5591,0x35c5,0x55d2,0xaaba
,0xaabb,0x55bf,0x55c9,0x55cc,0x55d1,0x55dd,0x35da,0x55e2
,0xaab9,0x55e9,0x5628,0xaabc,0x5607,0x5610,0x5630,0x5637
,0x35f4,0x563d,0x563f,0x5640,0x5647,0x565e,0x5660,0x566d
,0x3605,0x5688,0x568c,0x5695,0x569a,0x569d,0x56a8,0x56ad
,0x56b2,0x56c5,0x56cd,0x56df,0x56e8,0x56f6,0x56f7,0xaabd
,0x5715,0x5723,0xaabf,0x5729,0xaac1,0x5745,0x5746,0x574c
,0x574d,0xaac0,0x5768,0x576f,0x5773,0x5774,0x5775,0x577b
,0xaac2,0xaac3,0x57ac,0x579a,0x579d,0x579e,0x57a8,0x57d7
,0xaac5,0x57cc,0xaac6,0xaac7,0x57de,0x57e6,0x57f0,0x364a
,0x57f8,0x57fb,0x57fd,0x5804,0x581e,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x5820,0x5827,0x5832,0x5839,0xaac8,0x5849,0x584c,0x5867
,0x588a,0x588b,0x588d,0x588f,0x5890,0x5894,0x589d,0x58aa
,0x58b1,0xaaca,0x58c3,0x58cd,0x58e2,0x58f3,0x58f4,0x5905
,0x5906,0x590b,0x590d,0x5914,0x5924,0xaacb,0x3691,0x593d
,0x3699,0x5946,0x3696,0xaacc,0x595b,0x595f,0xaacd,0x5975
,0x5976,0x597c,0x599f,0x59ae,0x59bc,0x59c8,0x59cd,0x59de
,0x59e3,0x59e4,0x59e7,0x59ee,0xaace,0xaacf,0x36cf,0x5a0c
,0x5a0d,0x5a17,0x5a27,0x5a2d,0x5a55,0x5a65,0x5a7a,0
,0x5a8b,0x5a9c,0x5a9f,0x5aa0,0x5aa2,0x5ab1,0x5ab3,0x5ab5
,0x5aba,0x5abf,0x5ada,0x5adc,0x5ae0,0x5ae5,0x5af0,0x5aee
,0x5af5,0x5b00,0x5b08,0x5b17,0x5b34,0x5b2d,0x5b4c,0x5b52
,0x5b68,0x5b6f,0x5b7c,0x5b7f,0x5b81,0x5b84,0xaad1,0x5e6e
,0xaae2,0x5ea5,0x5eaa,0x5eac,0x5eb9,0x5ebf,0x5ec6,0x5ed2
,0x5ed9,0xaae3,0x5efd,0x5f08,0x5f0e,0x5f1c,0xaae4,0x5f1e
,0x5f47,0x5f63,0x5f72,0x5f7e,0x5f8f,0x5fa2,0x5fa4,0x5fb8
,0x5fc4,0x38fa,0x5fc7,0x5fcb,0x5fd2,0x5fd3,0x5fd4,0x5fe2
,0x5fee,0x5fef,0x5ff3,0x5ffc,0x3917,0x6017,0x6022,0x6024
,0x391a,0x604c,0x607f,0x608a,0x6095,0x60a8,0xaae5,0x60b0
,0x60b1,0x60be,0x60c8,0x60d9,0x60db,0x60ee,0x60f2,0x60f5
,0x6110,0x6112,0x6113,0x6119,0x611e,0x613a,0x396f,0x6141
,0x6146,0x6160,0x617c,0xaae6,0x6192,0x6193,0x6197,0x6198
,0x61a5,0x61a8,0x61ad,0xaae7,0x61d5,0x61dd,0x61df,0x61f5
,0xaae8,0x6215,0x6223,0x6229,0x6246,0x624c,0x6251,0x6252
,0x6261,0x6264,0x627b,0x626d,0x6273,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x6299,0x62a6,0x62d5,0xaae9,0x62fd,0x6303,0x630d,0x6310
,0xaaeb,0xaaec,0x6332,0x6335,0x633b,0x633c,0x6341,0x6344
,0x634e,0xaaea,0x6359,0xaaed,0xaaee,0x636c,0x6384,0x6399
,0xaaef,0x6394,0x63bd,0x63f7,0x63d4,0x63d5,0x63dc,0x63e0
,0x63eb,0x63ec,0x63f2,0x6409,0x641e,0x6425,0x6429,0x642f
,0x645a,0x645b,0x645d,0x6473,0x647d,0x6487,0x6491,0x649d
,0x649f,0x64cb,0x64cc,0x64d5,0x64d7,0xaaf0,0x64e4,0x64e5
,0x64ff,0x6504,0x3a6e,0x650f,0x6514,0x6516,0x3a73,0
,0x651e,0x6532,0x6544,0x6554,0x656b,0x657a,0x6581,0x6584
,0x6585,0x658a,0x65b2,0x65b5,0x65b8,0x65bf,0x65c2,0x65c9
,0x65d4,0x3ad6,0x65f2,0x65f9,0x65fc,0x6604,0x6608,0x6621
,0x662a,0x6645,0x6651,0x664e,0x3aea,0xaaf2,0x6657,0x665b
,0x6663,0xaaf4,0xaaf3,0x666a,0x666b,0x666c,0x666d,0x667b
,0x6680,0x6690,0x6692,0x6699,0x3b0e,0x66ad,0x66b1,0x66b5
,0x3b1a,0x66bf,0x3b1c,0x66ec,0x3ad7,0x6701,0x6705,0x6712
,0xaaf5,0x6719,0xaaf7,0xaaf8,0x674c,0x674d,0x6754,0x675d
,0xaaf6,0xaafc,0xaaf9,0x6774,0x6776,0xaafa,0x6792,0xaafb
,0x8363,0x6810,0x67b0,0x67b2,0x67c3,0x67c8,0x67d2,0x67d9
,0x67db,0x67f0,0x67f7,0xaafe,0xaaff,0xab00,0x6818,0x681f
,0x682d,0xaafd,0x6833,0x683b,0x683e,0x6844,0x6845,0x6849
,0x684c,0x6855,0x6857,0x3b77,0x686b,0x686e,0x687a,0x687c
,0x6882,0x6890,0x6896,0x3b6d,0x6898,0x6899,0x689a,0x689c
,0x68aa,0x68ab,0x68b4,0x68bb,0x68fb,0xab01,0xab02,0xfa13
,0x68c3,0x68c5,0x68cc,0x68cf,0x68d6,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x68d9,0x68e4,0x68e5,0x68ec,0x68f7,0x6903,0x6907,0x3b87
,0x3b88,0xab04,0x693b,0x3b8d,0x6946,0x6969,0x696c,0x6972
,0x697a,0x697f,0x6992,0x3ba4,0x6996,0x6998,0x69a6,0x69b0
,0x69b7,0x69ba,0x69bc,0x69c0,0x69d1,0x69d6,0xab05,0xab06
,0x6a30,0xab07,0xab08,0x69e3,0x69ee,0x69ef,0x69f3,0x3bcd
,0x69f4,0x69fe,0x6a11,0x6a1a,0x6a1d,0xab0a,0x6a32,0x6a33
,0x6a34,0x6a3f,0x6a46,0x6a49,0x6a7a,0x6a4e,0x6a52,0x6a64
,0xab0c,0x6a7e,0x6a83,0x6a8b,0x3bf0,0x6a91,0x6a9f,0
,0x6aa1,0xab0d,0x6aab,0x6abd,0x6ac6,0x6ad4,0x6ad0,0x6adc
,0x6add,0xab0e,0xab0f,0x6aec,0x6af1,0x6af2,0x6af3,0x6afd
,0xab10,0x6b0b,0x6b0f,0x6b10,0x6b11,0xab11,0x6b17,0x3c26
,0x6b2f,0x6b4a,0x6b58,0x6b6c,0x6b75,0x6b7a,0x6b81,0x6b9b
,0x6bae,0xab12,0x6bbd,0x6bbe,0x6bc7,0x6bc8,0x6bc9,0x6bda
,0x6be6,0x6be7,0x6bee,0x6bf1,0x6c02,0x6c0a,0x6c0e,0x6c35
,0x6c36,0x6c3a,0xab13,0x6c3f,0x6c4d,0x6c5b,0x6c6d,0x6c84
,0x6c89,0x3cc3,0x6c94,0x6c95,0x6c97,0x6cad,0x6cc2,0x6cd0
,0x3cd2,0x6cd6,0x6cda,0x6cdc,0x6ce9,0x6cec,0x6ced,0xab15
,0x6d00,0x6d0a,0x6d24,0x6d26,0x6d27,0x6c67,0x6d2f,0x6d3c
,0x6d5b,0x6d5e,0x6d60,0x6d70,0x6d80,0x6d81,0x6d8a,0x6d8d
,0x6d91,0x6d98,0xab16,0x6e17,0xab17,0xab18,0xab19,0x6dab
,0x6dae,0x6db4,0x6dc2,0x6d34,0x6dc8,0x6dce,0x6dcf,0x6dd0
,0x6ddf,0x6de9,0x6df6,0x6e36,0x6e1e,0x6e22,0x6e27,0x3d11
,0x6e32,0x6e3c,0x6e48,0x6e49,0x6e4b,0x6e4c,0x6e4f,0x6e51
,0x6e53,0x6e54,0x6e57,0x6e63,0x3d1e,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x6e93,0x6ea7,0x6eb4,0x6ebf,0x6ec3,0x6eca,0x6ed9,0x6f35
,0x6eeb,0x6ef9,0x6efb,0x6f0a,0x6f0c,0x6f18,0x6f25,0x6f36
,0x6f3c,0xab1a,0x6f52,0x6f57,0x6f5a,0x6f60,0x6f68,0x6f98
,0x6f7d,0x6f90,0x6f96,0x6fbe,0x6f9f,0x6fa5,0x6faf,0x3d64
,0x6fb5,0x6fc8,0x6fc9,0x6fda,0x6fde,0x6fe9,0xab1b,0x6ffc
,0x7000,0x7007,0x700a,0x7023,0xab1c,0x7039,0x703a,0x703c
,0x7043,0x7047,0x704b,0x3d9a,0x7054,0x7065,0x7069,0x706c
,0x706e,0x7076,0x707e,0x7081,0x7086,0x7095,0x7097,0
,0x70bb,0xab1d,0x709f,0x70b1,0xab1e,0x70ec,0x70ca,0x70d1
,0x70d3,0x70dc,0x7103,0x7104,0x7106,0x7107,0x7108,0x710c
,0x3dc0,0x712f,0x7131,0x7150,0x714a,0x7153,0x715e,0x3dd4
,0x7196,0x7180,0x719b,0x71a0,0x71a2,0x71ae,0x71af,0x71b3
,0xab1f,0x71cb,0x71d3,0x71d9,0x71dc,0x7207,0x3e05,0xfa4a
,0x722b,0x7234,0x7238,0x7239,0x4e2c,0x7242,0x7253,0x7257
,0x7263,0xab20,0x726e,0x726f,0x7278,0x727f,0x728e,0xab21
,0x72ad,0x72ae,0x72b0,0x72b1,0x72c1,0x3e60,0x72cc,0x3e66
,0x3e68,0x72f3,0x72fa,0x7307,0x7312,0x7318,0x7319,0x3e83
,0x7339,0x732c,0x7331,0x7333,0x733d,0x7352,0x3e94,0x736b
,0x736c,0xab23,0x736e,0x736f,0x7371,0x7377,0x7381,0x7385
,0x738a,0x7394,0x7398,0x739c,0x739e,0x73a5,0x73a8,0x73b5
,0x73b7,0x73b9,0x73bc,0x73bf,0x73c5,0x73cb,0x73e1,0x73e7
,0x73f9,0x7413,0x73fa,0x7401,0x7424,0x7431,0x7439,0x7453
,0x7440,0x7443,0x744d,0x7452,0x745d,0x7471,0x7481,0x7485
,0x7488,0xab24,0x7492,0x7497,0x7499,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x74a0,0x74a1,0x74a5,0x74aa,0x74ab,0x74b9,0x74bb,0x74ba
,0x74d6,0x74d8,0x74de,0x74ef,0x74eb,0xab25,0x74fa,0xab26
,0x7520,0x7524,0x752a,0x3f57,0xab27,0x753d,0x753e,0x7540
,0x7548,0x754e,0x7550,0x7552,0x756c,0x7572,0x7571,0x757a
,0x757d,0x757e,0x7581,0xab28,0x758c,0x3f75,0x75a2,0x3f77
,0x75b0,0x75b7,0x75bf,0x75c0,0x75c6,0x75cf,0x75d3,0x75dd
,0x75df,0x75e0,0x75e7,0x75ec,0x75ee,0x75f1,0x75f9,0x7603
,0x7618,0x7607,0x760f,0x3fae,0xab29,0x7613,0x761b,0
,0x761c,0xab2a,0x7625,0x7628,0x763c,0x7633,0xab2b,0x3fc9
,0x7641,0xab2c,0x7649,0x7655,0x3fd7,0x766e,0x7695,0x769c
,0x76a1,0x76a0,0x76a7,0x76a8,0x76af,0xab2d,0x76c9,0xab2e
,0x76e8,0x76ec,0xab2f,0x7717,0x771a,0x772d,0x7735,0xab30
,0x4039,0xab32,0xab31,0x7758,0x7760,0x776a,0xab34,0x7772
,0x777c,0x777d,0xab33,0x4058,0x779a,0x779f,0x77a2,0x77a4
,0x77a9,0x77de,0x77df,0x77e4,0x77e6,0x77ea,0x77ec,0x4093
,0x77f0,0x77f4,0x77fb,0xab35,0x7805,0x7806,0x7809,0x780d
,0x7819,0x7821,0x782c,0x7847,0x7864,0x786a,0xab37,0x788a
,0x7894,0x78a4,0x789d,0x789e,0x789f,0x78bb,0x78c8,0x78cc
,0x78ce,0x78d5,0x78e0,0x78e1,0x78e6,0x78f9,0x78fa,0x78fb
,0x78fe,0xab39,0x7910,0x791b,0x7930,0x7925,0x793b,0x794a
,0x7958,0x795b,0x4105,0x7967,0x7972,0x7994,0x7995,0x7996
,0x799b,0x79a1,0x79a9,0x79b4,0x79bb,0x79c2,0x79c7,0x79cc
,0x79cd,0x79d6,0x4148,0xab3b,0xab3c,0x414f,0x7a0a,0x7a11
,0x7a15,0x7a1b,0x7a1e,0x4163,0x7a2d,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x7a38,0x7a47,0x7a4c,0x7a56,0x7a59,0x7a5c,0x7a5f,0x7a60
,0x7a67,0x7a6a,0x7a75,0x7a78,0x7a82,0x7a8a,0x7a90,0x7aa3
,0x7aac,0xab3e,0x41b4,0x7ab9,0x7abc,0x7abe,0x41bf,0x7acc
,0x7ad1,0x7ae7,0x7ae8,0x7af4,0xab3f,0xab40,0x7b07,0xab41
,0x7b3d,0x7b27,0x7b2a,0x7b2e,0x7b2f,0x7b31,0x41e6,0x41f3
,0x7b7f,0x7b41,0x41ee,0x7b55,0x7b79,0x7b64,0x7b66,0x7b69
,0x7b73,0xab42,0x4207,0x7b90,0x7b91,0x7b9b,0x420e,0x7baf
,0x7bb5,0x7bbc,0x7bc5,0x7bca,0xab43,0xab44,0x7bd4,0
,0x7bd6,0x7bda,0x7bea,0x7bf0,0x7c03,0x7c0b,0x7c0e,0x7c0f
,0x7c26,0x7c45,0x7c4a,0x7c51,0x7c57,0x7c5e,0x7c61,0x7c69
,0x7c6e,0x7c6f,0x7c70,0xab46,0xab47,0xab49,0x7ca6,0xab48
,0x7cb6,0x7cb7,0x7cbf,0xab4b,0x7cc4,0xab4a,0x7cc8,0x7ccd
,0xab4c,0x7cd7,0xab4d,0x7ce6,0x7ceb,0xab4e,0x7cf5,0x7d03
,0x7d09,0x42c6,0x7d12,0x7d1e,0xab4f,0xab50,0x7d3d,0x7d3e
,0x7d40,0x7d47,0xab51,0xab52,0x42d6,0x7d59,0x7d5a,0x7d6a
,0x7d70,0x42dd,0x7d7f,0xab53,0x7d86,0x7d88,0x7d8c,0x7d97
,0xab54,0x7d9d,0x7da7,0x7daa,0x7db6,0x7db7,0x7dc0,0x7dd7
,0x7dd9,0x7de6,0x7df1,0x7df9,0x4302,0xab55,0xfa59,0x7e10
,0x7e17,0x7e1d,0x7e20,0x7e27,0x7e2c,0x7e45,0x7e73,0x7e75
,0x7e7e,0x7e86,0x7e87,0x432b,0x7e91,0x7e98,0x7e9a,0x4343
,0x7f3c,0x7f3b,0x7f3e,0x7f43,0x7f44,0x7f4f,0x34c1,0xab56
,0x7f52,0xab57,0x7f61,0x7f63,0x7f64,0x7f6d,0x7f7d,0x7f7e
,0xab58,0x7f90,0x517b,0xab59,0x7f96,0x7f9c,0x7fad,0xab5a
,0x7fc3,0x7fcf,0x7fe3,0x7fe5,0x7fef,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x7ff2,0x8002,0x800a,0x8008,0x800e,0x8011,0x8016,0x8024
,0x802c,0x8030,0x8043,0x8066,0x8071,0x8075,0x807b,0x8099
,0x809c,0x80a4,0x80a7,0x80b8,0xab5b,0x80c5,0x80d5,0x80d8
,0x80e6,0xab5c,0x810d,0x80f5,0x80fb,0x43ee,0x8135,0x8116
,0x811e,0x43f0,0x8124,0x8127,0x812c,0xab5d,0x813d,0x4408
,0x8169,0x4417,0x8181,0x441c,0x8184,0x8185,0x4422,0x8198
,0x81b2,0x81c1,0x81c3,0x81d6,0x81db,0xab5e,0x81e4,0xab5f
,0x81ec,0xab60,0x81fd,0x81ff,0xab61,0x8204,0xab62,0
,0x8219,0x8221,0x8222,0xab63,0x8232,0x8234,0x823c,0x8246
,0x8249,0x8245,0xab64,0x824b,0x4476,0x824f,0x447a,0x8257
,0xab65,0x825c,0x8263,0xab66,0xab67,0xfa5e,0x8279,0x4491
,0x827d,0x827f,0x8283,0x828a,0x8293,0x82a7,0x82a8,0x82b2
,0x82b4,0x82ba,0x82bc,0x82e2,0x82e8,0x82f7,0x8307,0x8308
,0x830c,0x8354,0x831b,0x831d,0x8330,0x833c,0x8344,0x8357
,0x44be,0x837f,0x44d4,0x44b3,0x838d,0x8394,0x8395,0x839b
,0x839d,0x83c9,0x83d0,0x83d4,0x83dd,0x83e5,0x83f9,0x840f
,0x8411,0x8415,0xab69,0x8417,0x8439,0x844a,0x844f,0x8451
,0x8452,0x8459,0x845a,0x845c,0xab6a,0x8465,0x8476,0x8478
,0x847c,0x8481,0x450d,0x84dc,0x8497,0x84a6,0x84be,0x4508
,0x84ce,0x84cf,0x84d3,0xab6c,0x84e7,0x84ea,0x84ef,0x84f0
,0x84f1,0x84fa,0x84fd,0x850c,0x851b,0x8524,0x8525,0x852b
,0x8534,0x854f,0x856f,0x4525,0x4543,0x853e,0x8551,0x8553
,0x855e,0x8561,0x8562,0xab6d,0x857b,0x857d,0x857f,0x8581
,0x8586,0x8593,0x859d,0x859f,0xab6e,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xab6f,0xab70,0x85b7,0x85bc,0x85c7,0x85ca,0x85d8,0x85d9
,0x85df,0x85e1,0x85e6,0x85f6,0x8600,0x8611,0x861e,0x8621
,0x8624,0x8627,0xab72,0x8639,0x863c,0xab73,0x8640,0xfa20
,0x8653,0x8656,0x866f,0x8677,0x867a,0x8687,0x8689,0x868d
,0x8691,0x869c,0x869d,0x86a8,0xfa21,0x86b1,0x86b3,0x86c1
,0x86c3,0x86d1,0x86d5,0x86d7,0x86e3,0x86e6,0x45b8,0x8705
,0x8707,0x870e,0x8710,0x8713,0x8719,0x871f,0x8721,0x8723
,0x8731,0x873a,0x873e,0x8740,0x8743,0x8751,0x8758,0
,0x8764,0x8765,0x8772,0x877c,0xab75,0xab76,0x87a7,0x8789
,0x878b,0x8793,0x87a0,0xab74,0x45e5,0x87be,0xab77,0x87c1
,0x87ce,0x87f5,0x87df,0xab78,0x87e3,0x87e5,0x87e6,0x87ea
,0x87eb,0x87ed,0x8801,0x8803,0x880b,0x8813,0x8828,0x882e
,0x8832,0x883c,0x460f,0x884a,0x8858,0x885f,0x8864,0xab79
,0xab7a,0x8869,0xab7b,0x886f,0x88a0,0x88bc,0x88bd,0x88be
,0x88c0,0x88d2,0xab7d,0x88d1,0x88d3,0x88db,0x88f0,0x88f1
,0x4641,0x8901,0xab7e,0x8937,0xab7f,0x8942,0x8945,0x8949
,0xab80,0x4665,0x8962,0x8980,0x8989,0x8990,0x899f,0x89b0
,0x89b7,0x89d6,0x89d8,0x89eb,0x46a1,0x89f1,0x89f3,0x89fd
,0x89ff,0x46af,0x8a11,0x8a14,0xab81,0x8a21,0x8a35,0x8a3e
,0x8a45,0x8a4d,0x8a58,0x8aae,0x8a90,0x8ab7,0x8abe,0x8ad7
,0x8afc,0xab82,0x8b0a,0x8b05,0x8b0d,0x8b1c,0x8b1f,0x8b2d
,0x8b43,0x470c,0x8b51,0x8b5e,0x8b76,0x8b7f,0x8b81,0x8b8b
,0x8b94,0x8b95,0x8b9c,0x8b9e,0x8c39,0xab83,0x8c3d,0xab84
,0xab85,0x8c45,0x8c47,0x8c4f,0x8c54,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x8c57,0x8c69,0x8c6d,0x8c73,0xab86,0x8c93,0x8c92,0x8c99
,0x4764,0x8c9b,0x8ca4,0x8cd6,0x8cd5,0x8cd9,0xab87,0x8cf0
,0x8cf1,0xab88,0x8d09,0x8d0e,0x8d6c,0x8d84,0x8d95,0x8da6
,0xab89,0x8dc6,0x8dc8,0x8dd9,0x8dec,0x8e0c,0x47fd,0x8dfd
,0x8e06,0xab8a,0x8e14,0x8e16,0x8e21,0x8e22,0x8e27,0xab8b
,0x4816,0x8e36,0x8e39,0x8e4b,0x8e54,0x8e62,0x8e6c,0x8e6d
,0x8e6f,0x8e98,0x8e9e,0x8eae,0x8eb3,0x8eb5,0x8eb6,0x8ebb
,0xab8d,0x8ed1,0x8ed4,0x484e,0x8ef9,0xab8e,0x8f00,0
,0x8f08,0x8f17,0x8f2b,0x8f40,0x8f4a,0x8f58,0xab90,0x8fa4
,0x8fb4,0xfa66,0x8fb6,0xab91,0x8fc1,0x8fc6,0xfa24,0x8fca
,0x8fcd,0x8fd3,0x8fd5,0x8fe0,0x8ff1,0x8ff5,0x8ffb,0x9002
,0x900c,0x9037,0xab92,0x9043,0x9044,0x905d,0xab93,0xab94
,0x9085,0x908c,0x9090,0x961d,0x90a1,0x48b5,0x90b0,0x90b6
,0x90c3,0x90c8,0xab95,0x90dc,0x90df,0xab96,0x90f6,0x90f2
,0x9100,0x90eb,0x90fe,0x90ff,0x9104,0x9106,0x9118,0x911c
,0x911e,0x9137,0x9139,0x913a,0x9146,0x9147,0x9157,0x9159
,0x9161,0x9164,0x9174,0x9179,0x9185,0x918e,0x91a8,0x91ae
,0x91b3,0x91b6,0x91c3,0x91c4,0x91da,0xab97,0xab98,0x91ec
,0x91ee,0x9201,0x920a,0x9216,0x9217,0xab99,0x9233,0x9242
,0x9247,0x924a,0x924e,0x9251,0x9256,0x9259,0x9260,0x9261
,0x9265,0x9267,0x9268,0xab9a,0xab9b,0x927c,0x927d,0x927f
,0x9289,0x928d,0x9297,0x9299,0x929f,0x92a7,0x92ab,0xab9c
,0xab9d,0x92b2,0x92bf,0x92c0,0x92c6,0x92ce,0x92d0,0x92d7
,0x92d9,0x92e5,0x92e7,0x9311,0xab9e,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0xab9f,0x92f7,0x92f9,0x92fb,0x9302,0x930d,0x9315,0x931d
,0x931e,0x9327,0x9329,0xaba0,0xaba1,0x9347,0x9351,0x9357
,0x935a,0x936b,0x9371,0x9373,0x93a1,0xaba2,0xaba3,0x9388
,0x938b,0x938f,0x939e,0x93f5,0xaba4,0xaba5,0x93f1,0x93c1
,0x93c7,0x93dc,0x93e2,0x93e7,0x9409,0x940f,0x9416,0x9417
,0x93fb,0x9432,0x9434,0x943b,0x9445,0xaba6,0xaba7,0x946d
,0x946f,0x9578,0x9579,0x9586,0x958c,0x958d,0xaba8,0x95ab
,0x95b4,0xaba9,0x95c8,0xabaa,0xabab,0x962c,0x9633,0
,0x9634,0xabac,0x963c,0x9641,0x9661,0xabad,0x9682,0xabae
,0x969a,0xabaf,0x49e7,0x96a9,0x96af,0x96b3,0x96ba,0x96bd
,0x49fa,0xabb0,0x96d8,0x96da,0x96dd,0x4a04,0x9714,0x9723
,0x4a29,0x9736,0x9741,0x9747,0x9755,0x9757,0x975b,0x976a
,0xabb1,0xabb2,0x9796,0x979a,0x979e,0x97a2,0x97b1,0x97b2
,0x97be,0x97cc,0x97d1,0x97d4,0x97d8,0x97d9,0x97e1,0x97f1
,0x9804,0x980d,0x980e,0x9814,0x9816,0x4abc,0xabb3,0x9823
,0x9832,0x9833,0x9825,0x9847,0x9866,0x98ab,0x98ad,0x98b0
,0xabb4,0x98b7,0x98b8,0x98bb,0x98bc,0x98bf,0x98c2,0x98c7
,0x98cb,0x98e0,0xabb5,0x98e1,0x98e3,0x98e5,0x98ea,0x98f0
,0x98f1,0x98f3,0x9908,0x4b3b,0xabb6,0x9916,0x9917,0xabb7
,0x991a,0x991b,0x991c,0xabb8,0x9931,0x9932,0x9933,0x993a
,0x993b,0x993c,0x9940,0x9941,0x9946,0x994d,0x994e,0x995c
,0x995f,0x9960,0x99a3,0x99a6,0x99b9,0x99bd,0x99bf,0x99c3
,0x99c9,0x99d4,0x99d9,0x99de,0xabb9,0x99f0,0x99f9,0x99fc
,0x9a0a,0x9a11,0x9a16,0x9a1a,0x9a20,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0,0,0,0,0,0,0,0
,0x9a31,0x9a36,0x9a44,0x9a4c,0x9a58,0x4bc2,0x9aaf,0x4bca
,0x9ab7,0x4bd2,0x9ab9,0xabba,0x9ac6,0x9ad0,0x9ad2,0x9ad5
,0x4be8,0x9adc,0x9ae0,0x9ae5,0x9ae9,0x9b03,0x9b0c,0x9b10
,0x9b12,0x9b16,0x9b1c,0x9b2b,0x9b33,0x9b3d,0x4c20,0x9b4b
,0x9b63,0x9b65,0x9b6b,0x9b6c,0x9b73,0x9b76,0x9b77,0x9ba6
,0x9bac,0x9bb1,0xabbb,0xabbc,0x9bb2,0x9bb8,0x9bbe,0x9bc7
,0x9bf3,0x9bd8,0x9bdd,0x9be7,0x9bea,0x9beb,0x9bef,0x9bee
,0xabbd,0x9bfa,0xabbf,0x9bf7,0xabbe,0x9c16,0x9c18,0
,0x9c19,0x9c1a,0x9c1d,0x9c22,0x9c27,0x9c29,0x9c2a,0xabc0
,0x9c31,0x9c36,0x9c37,0x9c45,0x9c5c,0xabc2,0x9c49,0x9c4a
,0xabc1,0x9c54,0x9c58,0x9c5b,0x9c5d,0x9c5f,0x9c69,0x9c6a
,0x9c6b,0x9c6d,0x9c6e,0x9c70,0x9c72,0x9c75,0x9c7a,0x9ce6
,0x9cf2,0x9d0b,0x9d02,0xabc3,0x9d11,0x9d17,0x9d18,0xabc5
,0x4cc4,0xabc4,0x9d32,0x4cd1,0x9d42,0x9d4a,0x9d5f,0x9d62
,0xabc6,0x9d69,0x9d6b,0xabc7,0x9d73,0x9d76,0x9d77,0x9d7e
,0x9d84,0x9d8d,0x9d99,0x9da1,0x9dbf,0x9db5,0x9db9,0x9dbd
,0x9dc3,0x9dc7,0x9dc9,0x9dd6,0x9dda,0x9ddf,0x9de0,0x9de3
,0x9df4,0x4d07,0x9e0a,0x9e02,0x9e0d,0x9e19,0x9e1c,0x9e1d
,0x9e7b,0xabc9,0x9e80,0x9e85,0x9e9b,0x9ea8,0xabca,0x9ebd
,0xabcb,0x9edf,0x9ee7,0x9eee,0x9eff,0x9f02,0x4d77,0x9f03
,0x9f17,0x9f19,0x9f2f,0x9f37,0x9f3a,0x9f3d,0x9f41,0x9f45
,0x9f46,0x9f53,0x9f55,0x9f58,0xabcc,0x9f5d,0xabcd,0x9f69
,0xabce,0x9f6d,0x9f70,0x9f75,0xabcf
};
static struct {
	long hc_flag;
	long hc_hkey;
	long hc_next;
	long hc_datp;
} gutf8_cell_tbl[AKX_SJIS_HASH_MAX]={
 1,0x0,9495,0x0,1,0xe2939a00,657,0x86c9,1,0xe293a200,1060,0x86d1
,1,0xe38b9900,0,0x86e2,1,0xe38bad00,1128,0x86f0,1,0xe285a300,635,0x8757
,1,0xe38cab00,1779,0x876c,1,0xe38aa400,198,0x8785,1,0xe288ae00,4518,0x8793
,1,0xeaaaa200,236,0x87a0,1,0xe4b8a800,3000,0x87a2,1,0xe5968600,0,0x8849
,1,0xe5968e00,0,0x884a,1,0xe596ad00,0,0x884c,1,0xe5978e00,1399,0x884d
,1,0xe5988800,43,0x884f,1,0xe5988e00,985,0x8850,1,0xe2939200,2436,0x86c1
,1,0xeaaabe00,69,0x8861,1,0xe29db800,2207,0x86a1,1,0xeaab8400,791,0x886b
,1,0xe5a1a100,0,0x8877,1,0xeaab8900,3429,0x8880,1,0xeaab9000,0,0x889b
,1,0xe88c9c00,339,0x88a9,1,0xe7b6be00,0,0x88bb,1,0xe7b29f00,793,0x88be
,1,0xe5bab500,2191,0x88c1,1,0xe69d8f00,821,0x88c7,1,0xe383a200,8296,0x8382
,1,0xe6848f00,986,0x88d3,1,0xe7958f00,3035,0x88d8,1,0xe7a7bb00,0,0x88da
,1,0xe7b6ad00,2635,0x88db,1,0xe8a1a300,0,0x88df,1,0xc2bd0000,0,0x8553
,1,0xc9950000,92,0x865e,1,0xe6baa200,144,0x88ec,1,0xe6b7ab00,146,0x88fa
,1,0xe894ad00,0,0x88fc,1,0xe7838f00,1607,0x8947,1,0xe9b59c00,2831,0x894c
,1,0xe4b89100,132,0x894e,1,0xe5989800,831,0x8952,1,0xe8949a00,995,0x8955
,1,0xe7939c00,157,0x895a,1,0xe596b600,1629,0x8963,1,0xe79b8800,89,0x896d
,1,0xe8a19b00,1933,0x8971,1,0xe79b8a00,1771,0x8976,1,0xe5bbb600,148,0x8984
,1,0xe7859900,545,0x898c,1,0xe78cbf00,369,0x898e,1,0xe8969700,731,0x8992
,1,0xe7bf8100,5558,0x89a5,1,0xe59ab300,5252,0x885d,1,0xe5b2a100,0,0x89aa
,1,0xe686b600,135,0x89af,1,0xe8878600,1977,0x89b0,1,0xe6b8a900,395,0x89b7
,1,0xe4b88b00,3142,0x89ba,1,0xe4bcbd00,645,0x89be,1,0xe4bea100,462,0x89bf
,1,0xe5988900,1013,0x89c3,1,0xe5afa100,1077,0x89c7,1,0xe6ad8c00,237,0x89cc
,1,0xe685b000,0,0x88d4,1,0xe88ab100,624,0x89d4,1,0xe88b9b00,1605,0x89d5
,1,0xe28eac00,826,0x83d8,1,0xe7a6be00,75,0x89d1,1,0xe89da600,0,0x89da
,1,0xe7899900,1644,0x89e5,1,0xe887a500,10371,0x89e7,1,0xe8b38000,2446,0x89ea
,1,0xe8a7a300,722,0x89f0,1,0xce930000,0,0x83a1,1,0xe5bfab00,547,0x89f5
,1,0xe7b5b500,0,0x8a47,1,0xe88aa500,127,0x8a48,1,0xe88ca800,8025,0x88ef
,1,0xe8938b00,0,0x8a57,1,0xe8a19700,0,0x8a58,1,0xe69fbf00,0,0x8a60
,1,0xe8a69a00,10719,0x8a6f,1,0xe7aca000,1168,0x8a7d,1,0xe6bd9f00,5931,0x8a83
,1,0xe5969d00,6052,0x8a85,1,0xe6bb9100,584,0x8a8a,1,0xe4b89400,442,0x8a8e
,1,0xe29fa500,2221,0x84d4,1,0xe6a0a200,0,0x8a9c,1,0xe890b100,3401,0x8a9e
,1,0xe7b2a500,627,0x8a9f,1,0xe4b9be00,590,0x8aa3,1,0xe5969a00,0,0x8aab
,1,0xe5ae9800,1232,0x8aaf,1,0xe6bca200,759,0x8abf,1,0xe7949800,0,0x8ac3
,1,0xe79c8b00,1877,0x8ac5,1,0xe7aea100,4584,0x8ac7,1,0xe7b0a100,230,0x8ac8
,1,0xe996a200,178,0x8ad6,1,0xe5818100,3507,0x87c1,1,0xe5b2a900,0,0x8ae2
,1,0xe58ba700,0,0x8aa9,1,0xe7bfab00,3716,0x8ae3,1,0xe5969c00,1601,0x8aec
,1,0xc99c0000,3894,0x8650,1,0xe5af8400,3926,0x8af1,1,0xe5b9be00,799,0x8af4
,1,0xe8a89800,2803,0x8b4c,1,0xe9a3a200,3108,0x8b51,1,0xe4ba8000,10030,0x8b54
,1,0xe58c8700,0,0x87ea,1,0xccb90000,0,0x8689,1,0xefbcab00,14,0x826a
,1,0xe7bea900,1635,0x8b60,1,0xe8aabc00,0,0x8b62,1,0xe596ab00,4218,0x8b69
,1,0xe6a99800,492,0x8b6b,1,0xe8849a00,4766,0x8b72,1,0xe4b89800,2901,0x8b75
,1,0xe6b3a300,498,0x8b83,1,0xe7a9b600,0,0x8b86,1,0xefbd8b00,0,0x828b
,1,0xe7aaae00,4375,0x8b87,1,0xe7b49a00,667,0x8b89,1,0xe7b5a600,0,0x8b8b
,1,0xe2859300,174,0x8498,1,0xe7a6a600,460,0x8b9a,1,0xe5858700,200,0x8ba2
,1,0xe596ac00,2930,0x8baa,1,0xe5b3a100,1464,0x8bac,1,0xe6b38100,6871,0x8bb5
,1,0xe4bd8900,689,0x87ad,1,0xe8958e00,672,0x8bbc,1,0xe5a7a600,6268,0x8aad
,1,0xe99fbf00,559,0x8bbf,1,0xe69a8100,304,0x8bc5,1,0xc4a50000,0,0x85df
,1,0xe5b18000,3558,0x8bc7,1,0xe69bb200,2554,0x8bc8,1,0xe6a5b500,1873,0x8bc9
,1,0xe6aca300,3362,0x8bd3,1,0xe7b78a00,820,0x8bd9,1,0xe8a59f00,3078,0x8bdd
,1,0xe9879100,1413,0x8be0,1,0xce9a0000,718,0x83a8,1,0xe596b000,0,0x8bf2
,1,0xe4b8b200,6276,0x8bf8,1,0xe68e9800,4413,0x8c40,1,0xe896ab00,476,0x8c4f
,1,0xe7bea400,0,0x8c51,1,0xe59e8b00,0,0x8c5e,1,0xe690ba00,3470,0x8c67
,1,0xe6a18200,485,0x8c6a,1,0xe7b3bb00,0,0x8c6e,1,0xe7b58c00,600,0x8c6f
,1,0xe7b69900,2664,0x8c70,1,0xe8a9a300,508,0x8c77,1,0xe3818900,0,0x82a7
,1,0xe99a9900,4660,0x8c84,1,0xe5829100,3525,0x8c86,1,0xe7b59000,0,0x8c8b
,1,0xe8a8a300,0,0x8c8d,1,0xe4bbb600,705,0x8c8f,1,0xe596a700,0,0x8c96
,1,0xe789bd00,496,0x8ca1,1,0xe7a1af00,7217,0x8ca5,1,0xe7b5b900,0,0x8ca6
,1,0xe8b3a200,0,0x8cab,1,0xe981a300,9150,0x8cad,1,0xe6b89b00,9759,0x8cb8
,1,0xe7b38a00,2602,0x8cd0,1,0xe5888100,0,0x87d8,1,0xe58d9400,0,0x8ba6
,1,0xe8a2b400,0,0x8cd1,1,0xefbdb800,5607,0xb8,1,0xe5bea100,1900,0x8ce4
,1,0xe7919a00,0,0x8ce8,1,0xe5968900,0,0x8d41,1,0xe5b9b800,2815,0x8d4b
,1,0xe5ba8300,0,0x8d4c,1,0xe5bc9800,1550,0x8d4f,1,0xe68b9800,446,0x8d53
,1,0xe6a0a100,1006,0x8d5a,1,0xefbe9800,5972,0xd8,1,0xe6a29700,922,0x8d5b
,1,0xe6b8af00,5191,0x8d60,1,0xe794b200,1959,0x8d62,1,0xe79a8700,0,0x8d63
,1,0xe7a1ac00,0,0x8d64,1,0xe7b3a000,2431,0x8d66,1,0xe7b49800,0,0x8d68
,1,0xe7b59e00,2633,0x8d69,1,0xe7b6b100,556,0x8d6a,1,0xe8809500,926,0x8d6b
,1,0xe5a3a000,0,0x8885,1,0xe882af00,1388,0x8d6d,1,0xe882b100,0,0x8d6e
,1,0xe8868f00,1816,0x8d70,1,0xe291a600,7847,0x8746,1,0xe8b2a200,835,0x8d76
,1,0xe8bb8d00,789,0x8c52,1,0xe6bfa000,346,0x8d8a,1,0xe9bab900,10196,0x8d8d
,1,0xe9bb9200,0,0x8d95,1,0xe7949100,0,0x8d99,1,0xe6878700,1180,0x8da7
,1,0xe6988f00,3683,0x8da8,1,0xe7979500,1741,0x8dad,1,0xe7b4ba00,2157,0x8dae
,1,0xe4bd9000,0,0x8db2,1,0xe381b600,402,0x82d4,1,0xe7a08200,0,0x8dbb
,1,0xe68cab00,5884,0x8dc1,1,0xe582b500,3499,0x8dc2,1,0xefa8b900,719,0x8879
,1,0xe5a6bb00,1216,0x8dc8,1,0xcea10000,6111,0x83af,1,0xe69eb600,0,0x89cb
,1,0xe5aeb000,8105,0x8dc9,1,0xe3829600,196,0x82f4,1,0xe387a900,9043,0x83ed
,1,0xe89fbb00,0,0x8b61,1,0xe7b4b000,5133,0x8dd7,1,0xe8a38100,7587,0x8dd9
,1,0x7f000000,969,0x7f,1,0xe7bdaa00,1527,0x8ddf,1,0xe7b4a200,0,0x8df5
,1,0xe896a900,740,0x8e46,1,0xe5829800,4124,0x8e50,1,0xe683a800,0,0x8e53
,1,0xe695a300,1219,0x8e55,1,0xe597a300,2511,0x8e6b,1,0xe6809d00,11016,0x8e76
,1,0xe697a800,0,0x8e7c,1,0xe6adbb00,0,0x8e80,1,0xe7a78100,6542,0x8e84
,1,0xe7b49900,3876,0x8e86,1,0xe7b4ab00,1591,0x8e87,1,0xe9a3bc00,1683,0x8e94
,1,0xe4bcbc00,0,0x8e97,1,0xe6858800,9802,0x8e9c,1,0xe6aca100,2876,0x8e9f
,1,0xe6bb8b00,2039,0x8ea0,1,0xe788be00,2621,0x8ea2,1,0xe8808c00,1010,0x8ea7
,1,0xefbfa500,7538,0x818f,1,0xe887aa00,0,0x8ea9,1,0xe6b08f00,816,0x8e81
,1,0xe8929400,825,0x8eaa,1,0xe8ad9800,5805,0x8eaf,1,0xe8ac8100,0,0x8979
,1,0xe7abba00,9418,0x8eb1,1,0xe5aea400,0,0x8eba,1,0xe8948000,4102,0x8ec1
,1,0xe59a8700,8152,0x8a64,1,0xe69fb400,712,0x8ec4,1,0xe5b1a100,486,0x8ec6
,1,0xe8958a00,3988,0x8ec7,1,0xe7b89e00,1780,0x8ec8,1,0xe7b49700,8467,0x8ed1
,1,0xe8bb8a00,4589,0x8ed4,1,0xe781bc00,8713,0x8edc,1,0xe98cab00,3283,0x8ee0
,1,0xe5af8200,301,0x8ee2,1,0xe5ae8800,0,0x8ee7,1,0xe8b6a300,1346,0x8eef
,1,0xe68a9700,5858,0x8d52,1,0xe5afbf00,1191,0x8ef5,1,0xe7b6ac00,0,0x8ef8
,1,0xe2939300,0,0x86c2,1,0xe99c8000,531,0x8ef9,1,0xe29db900,606,0x86a2
,1,0xe5b0b100,1454,0x8f41,1,0x68000000,0,0x68,1,0xe7a78000,1536,0x8f47
,1,0xe7a78b00,2452,0x8f48,1,0xe887ad00,1193,0x8f4c,1,0xe8929000,1492,0x8f4e
,1,0xe980b100,6120,0x8f54,1,0xe5be9300,0,0x8f5d,1,0xe98a8300,2213,0x8f65
,1,0xe383a300,2305,0x8383,1,0xe5a49900,6231,0x8f67,1,0xe7a59d00,7394,0x8f6a
,1,0xe7b29b00,0,0x8f6c,1,0xe5a1be00,1669,0x8f6d,1,0xe5beaa00,1574,0x8f7a
,1,0xe6b7b300,6131,0x8f7e,1,0xcea80000,1821,0x83b5,1,0xe6ba9600,0,0x8f80
,1,0xe79bbe00,604,0x8f82,1,0xe7b49400,0,0x8f83,1,0xe5b7a100,621,0x8f84
,1,0xe69a9100,0,0x8f8b,1,0xe896af00,363,0x8f92,1,0xe6819500,841,0x8f9a
,1,0xe582b700,994,0x8f9d,1,0xe5989700,507,0x8fa6,1,0xe5aeb500,6699,0x8faa
,1,0xe5ba8a00,1808,0x8fb0,1,0xe69dbe00,845,0x8fbc,1,0xe8a28800,0,0x8c55
,1,0xe6a2a200,1838,0x8fbd,1,0xe6b88900,4250,0x8fc2,1,0xe6b99800,0,0x8fc3
,1,0xe5908e00,5104,0x8d40,1,0xe7a19d00,5560,0x8fc9,1,0xe7b2a700,1068,0x8fcf
,1,0xe7b4b900,0,0x8fd0,1,0xe8928b00,3949,0x8fd3,1,0xe6b6af00,0,0x8a55
,1,0xe8958900,0,0x8fd4,1,0xe8a19d00,424,0x8fd5,1,0xe8b2a100,905,0x8de0
,1,0xe8b1a100,0,0x8fdb,1,0xe5968100,3299,0x8848,1,0xe4bb9d00,9688,0x8157
,1,0xe4b89e00,901,0x8fe5,1,0xe5aca200,1361,0x8fec,1,0xe6838500,6992,0x8fee
,1,0xc9b10000,98,0x85e3,1,0xe28ead00,193,0x83d9,1,0x51000000,534,0x51
,1,0xe889b200,9529,0x9046,1,0xe4bcb800,605,0x904c,1,0xe4bfa100,3354,0x904d
,1,0xe79c9f00,0,0x905e,1,0xe7b4b300,1372,0x9061,1,0xe896aa00,0,0x9064
,1,0xe8baab00,0,0x9067,1,0xe980b200,2101,0x9069,1,0xe294a000,384,0x84b5
,1,0xe9879d00,0,0x906a,1,0xe7949a00,8832,0x9072,1,0xe7aca500,893,0x9079
,1,0xe7828a00,5080,0x9086,1,0xe7bfa000,10721,0x9089,1,0xe98c9000,0,0x908d
,1,0xe99a8f00,495,0x908f,1,0xe69ea200,0,0x9095,1,0xe6a49900,378,0x909a
,1,0xe9a09700,400,0x909c,1,0xe29fa600,94,0x84d5,1,0xe4b89600,5809,0x90a2
,1,0xe7959d00,10589,0x90a4,1,0xe698af00,3461,0x90a5,1,0xe680a700,5313,0x90ab
,1,0xe6889000,660,0x90ac,1,0xe6ada300,1231,0x90b3,1,0xe5bca600,458,0x8cb7
,1,0xe8819600,0,0x90b9,1,0xe6889a00,2236,0x90ca,1,0xc4ba0000,0,0x85c8
,1,0xe7bdab00,0,0x8c72,1,0xe68ea500,2799,0x90da,1,0xe6918200,3109,0x90db
,1,0xe68a9800,964,0x90dc,1,0xe7b5b600,2652,0x90e2,1,0xe5858800,651,0x90e6
,1,0xe58d8300,6649,0x90e7,1,0xe688a600,6723,0x90ed,1,0xe6b38900,0,0x90f2
,1,0xe69f9300,2747,0x90f5,1,0xe7a9bf00,780,0x90fa,1,0xe7b98a00,3200,0x9140
,1,0xe7bea800,2746,0x9141,1,0xe896a600,982,0x9145,1,0xe5968400,0,0x9150
,1,0xefbcac00,1022,0x826b,1,0xe7b38e00,0,0x9157,1,0xe68eaa00,3928,0x915b
,1,0x3a000000,7498,0x3a,1,0xe69bbe00,1997,0x915c,1,0xe69bbd00,2845,0x915d
,1,0xe78b9900,0,0x915f,1,0xe7968f00,6073,0x9160,1,0xe7a48e00,0,0x9162
,1,0xefbd8c00,15,0x828c,1,0xe59ca100,0,0x8862,1,0xe7a79f00,1308,0x9164
,1,0xe686be00,366,0x8ab6,1,0xe2859400,7618,0x8499,1,0xe7b29700,698,0x9165
,1,0xe7b4a000,2611,0x9166,1,0xe596aa00,9488,0x9172,1,0xe788bd00,4451,0x9175
,1,0xe683a300,2883,0x9179,1,0xe9a3be00,0,0x8ffc,1,0xe68e8300,1318,0x917c
,1,0xe6bc9500,5825,0x9186,1,0xe79bb800,2000,0x918a,1,0xe7b39f00,2604,0x918c
,1,0xe7b69c00,8938,0x918e,1,0xd0900000,0,0x8440,1,0xe881a100,2971,0x918f
,1,0xe88d8900,516,0x9190,1,0xe8879300,3366,0x919f,1,0xe894b500,8235,0x91a0
,1,0xe6b8ac00,0,0x91aa,1,0xe4bf9700,1170,0x91ad,1,0xe6978f00,489,0x91b0
,1,0xe7b69a00,2643,0x91b1,1,0xe5ad9800,1415,0x91b6,1,0xe6908d00,2195,0x91b9
,1,0xe8809000,2242,0x91cf,1,0xe69bbf00,9001,0x91d6,1,0xe88b9400,2172,0x91db
,1,0xe9808000,3473,0x91de,1,0xe9bb9b00,0,0x91e1,1,0xe4bba300,3799,0x91e3
,1,0xe7acac00,2550,0x91e6,1,0xe9868d00,1279,0x91e7,1,0xe6899800,0,0x91ef
,1,0xe3818a00,8224,0x82a8,1,0xe68a9e00,829,0x91f0,1,0xe6b2a200,2371,0x91f2
,1,0xe8a89700,3497,0x91f5,1,0xe89bb800,0,0x91fb,1,0xe587a200,3599,0x87d6
,1,0xe9819400,1246,0x9242,1,0x23000000,284,0x23,1,0xe884b100,4071,0x9245
,1,0xe5988600,5202,0x9251,1,0xe6ad8e00,1930,0x9256,1,0xe6b7a100,512,0x9257
,1,0xcb990000,3073,0x85d6,1,0xe6b99b00,768,0x9258,1,0xe782ad00,5402,0x9259
,1,0xe7abaf00,0,0x925b,1,0xe7b6bb00,1009,0x925d,1,0xefbdb900,5580,0xb9
,1,0xe797b400,668,0x9273,1,0xceb60000,0,0x83c4,1,0xe7bdae00,9113,0x9275
,1,0xe89c9800,9263,0x9277,1,0xe8aa9500,0,0x9261,1,0xe9818500,4986,0x9278
,1,0xe7ad9100,5763,0x927d,1,0xe8938400,1008,0x927e,1,0xefbe9900,1047,0xd9
,1,0xe7aa9200,5288,0x9282,1,0xe5aba100,1320,0x9284,1,0xe2818e00,1966,0x829b
,1,0xe79d8000,0,0x9285,1,0xe6b3a800,529,0x928d,1,0xe8a1b700,563,0x928f
,1,0xe5968b00,7762,0x929d,1,0xe5bcb500,704,0x92a3,1,0xe5bdab00,749,0x92a4
,1,0xe4be9700,0,0x87b5,1,0xc3850000,1329,0x855b,1,0xe687b200,0,0x92a6
,1,0xe69aa200,2025,0x92a8,1,0xe69c9d00,0,0x92a9,1,0xe291a700,2092,0x8747
,1,0xe6bdae00,615,0x92aa,1,0xe99b8500,0,0x89eb,1,0xe8b7b300,840,0x92b5
,1,0xe98a9a00,2860,0x92b6,1,0xe9a08200,8381,0x92b8,1,0xe6898b00,0,0x8ee8
,1,0xe79bb400,0,0x92bc,1,0xe98e9a00,1211,0x92c8,1,0xe9809a00,2711,0x92ca
,1,0xe6a08200,0,0x92cc,1,0xd0970000,0,0x8448,1,0xe381b700,2261,0x82d5
,1,0xe69f9800,3731,0x92d1,1,0xe894a600,846,0x92d3,1,0xc000000,10839,0xc
,1,0xefa8ba00,1185,0x887d,1,0xe7b6b400,1214,0x92d4,1,0xe6a4bf00,1520,0x92d6
,1,0xe59daa00,1575,0x92d8,1,0xe5acac00,2560,0x92da,1,0xe7b4ac00,4619,0x92db
,1,0xe387aa00,2700,0x83ee,1,0xe987a300,0,0x92de,1,0xe5b89d00,0,0x92e9
,1,0xe8898700,3529,0x92f8,1,0xe6b3a500,3629,0x9344,1,0xe6919800,1129,0x9345
,1,0xe5bfa000,3862,0x9289,1,0xe6bbb400,726,0x9348,1,0xe692a400,1904,0x9350
,1,0xe5a1ab00,2710,0x9355,1,0xe9809300,0,0x92fc,1,0xe6b7bb00,1994,0x9359
,1,0xe7949c00,3139,0x935b,1,0xe8bba200,970,0x935d,1,0xe5b1a000,9566,0x936a
,1,0xe5be9200,1568,0x936b,1,0xe6b8a100,794,0x936e,1,0xe8b3ad00,3915,0x9371
,1,0xe5a19800,7443,0x9384,1,0xe794ba00,0,0x92ac,1,0xe8a68f00,0,0x8b4b
,1,0xe5a59700,2872,0x9385,1,0xe5ae9500,0,0x9386,1,0xe58f8200,0,0x8e51
,1,0xe68a9500,0,0x938a,1,0xe599a800,810,0x8aed,1,0xe6a18300,5959,0x938d
,1,0xe6a2bc00,1847,0x938e,1,0xcebd0000,1728,0x83cb,1,0xe6b79800,796,0x9391
,1,0xe8ac8200,0,0x88e0,1,0xe7979800,2905,0x9397,1,0xe8b6a800,4995,0x9096
,1,0xe99bb200,2995,0x895f,1,0xe4ba9100,5637,0x895d,1,0xe7ad9200,669,0x939b
,1,0xe5a4ae00,0,0x899b,1,0xe7b39600,2600,0x939c,1,0xe7b5b100,882,0x939d
,1,0xe891a300,742,0x939f,1,0xe895a900,2275,0x93a0,1,0xe88abd00,9503,0x89e8
,1,0xe9808300,3004,0x93a6,1,0xe9808f00,5070,0x93a7,1,0xe9909900,0,0x93a8
,1,0xe9979800,1137,0x93ac,1,0xc38c0000,361,0x8562,1,0xe5a08200,6360,0x93b0
,1,0xe79eb300,681,0x93b5,1,0xe2939400,2437,0x86c3,1,0xe7aba500,2177,0x93b6
,1,0xe29dba00,0,0x86a3,1,0xe9819300,0,0x93b9,1,0xe7a6bf00,2921,0x93c3
,1,0xe9b3b600,10680,0x93ce,1,0xe5ab8100,0,0x89c5,1,0xe88bab00,7995,0x93cf
,1,0xe5af8500,1491,0x93d0,1,0xe695a600,6533,0x93d6,1,0xe8b19a00,2181,0x93d8
,1,0xe982a300,4428,0x93df,1,0xe383a400,11476,0x8384,1,0xe8969900,2306,0x93e3
,1,0xe7819800,10690,0x93e5,1,0xe6a5a200,9377,0x93e8,1,0xe697a500,0,0x93fa
,1,0xe5b0bf00,1220,0x9441,1,0xe4bbbb00,1257,0x9443,1,0xe78cab00,1819,0x944c
,1,0xe785ae00,1620,0x8ecf,1,0xe5bfb500,1578,0x944f,1,0xe7878300,0,0x9452
,1,0xe981a000,9216,0x8993,1,0xe7b29800,9813,0x9453,1,0xe5858900,0,0x8cf5
,1,0xe4b98300,2647,0x9454,1,0xe58faf00,0,0x89c2,1,0xe5bbbc00,6626,0x9455
,1,0xe4b98b00,6660,0x9456,1,0xe59aa200,1654,0x9458,1,0xe790b400,0,0x8bd5
,1,0xe7b48d00,3058,0x945b,1,0xe883bd00,2840,0x945c,1,0xe98ca600,0,0x8bd1
,1,0xe886bf00,0,0x945e,1,0xe89aa400,814,0x9461,1,0xe6b3a200,0,0x9467
,1,0xe7bdb500,9139,0x946c,1,0xe4bfb300,1637,0x946f,1,0xe6b19d00,0,0x93f0
,1,0xe68e9200,0,0x9472,1,0xe9858d00,0,0x947a,1,0xe785a400,2890,0x9481
,1,0xcba70000,204,0x8682,1,0xe8b7b500,0,0x9148,1,0xe7a7a400,5563,0x9489
,1,0xe7b29500,915,0x9494,1,0xe888b600,0,0x9495,1,0xe8968400,0,0x9496
,1,0xe7978700,6993,0x8fc7,1,0xe9baa600,862,0x949e,1,0xe28eae00,0,0x83da
,1,0xe7a1b200,868,0x94a1,1,0xe6aba800,991,0x94a5,1,0xe5b9a100,0,0x94a6
,1,0xe8828c00,0,0x94a7,1,0xe795a000,0,0x94a9,1,0xe6ba8c00,1204,0x94ac
,1,0xe7bdb000,5881,0x94b1,1,0xe9b3a900,939,0x94b5,1,0xe7a28d00,2400,0x8a56
,1,0xe5a19900,4529,0x94b7,1,0xe5b88600,2182,0x94bf,1,0xe69dbf00,0,0x94c2
,1,0xe8b2a900,0,0x94cc,1,0xe7af8400,0,0x94cd,1,0xe699a900,0,0x94d3
,1,0xe79ba400,0,0x94d5,1,0xc3930000,261,0x8569,1,0xe8958300,1498,0x94d7
,1,0xe7a88000,377,0x8b48,1,0xe88d8a00,0,0x8c74,1,0xe29fa700,8163,0x84d6
,1,0xe7b78b00,8319,0x94ea,1,0xe99d9e00,3400,0x94f1,1,0xe7b0b800,0,0x94f4
,1,0xe69e8700,677,0x94f8,1,0xe585b600,0,0x91b4,1,0xe6978b00,0,0x90f9
,1,0xe5bca700,1557,0x8cca,1,0xe6af9800,4149,0x94f9,1,0xe790b500,0,0x94fa
,1,0xe7a89700,4648,0x9542,1,0xd0a50000,8,0x8456,1,0xe8869d00,2076,0x9547
,1,0xe8829800,942,0x9549,1,0xe795a200,965,0x954c,1,0xe7b49000,0,0x9552
,1,0xe5bdaa00,1003,0x9556,1,0xe793a200,3971,0x955a,1,0xe8a1a800,0,0x955c
,1,0xe78cae00,1656,0x8ca3,1,0xe68f8f00,891,0x9560,1,0xe7978500,6986,0x9561
,1,0xe7a79200,2455,0x9562,1,0xe88b9700,10036,0x9563,1,0xe8929c00,3460,0x9566
,1,0xe89bad00,0,0x9567,1,0xefbcad00,0,0x826c,1,0xe5bdac00,1565,0x956a
,1,0xe6958f00,1923,0x9571,1,0xe4b88d00,3956,0x9573,1,0xe284b500,436,0x827b
,1,0xe5af8c00,3761,0x9578,1,0xe689b600,729,0x957d,1,0xe7aca600,8737,0x9584
,1,0xe8859000,0,0x9585,1,0xefbd8d00,0,0x828d,1,0xe8869a00,0,0x9586
,1,0xe5a1b500,1306,0x906f,1,0xe6b38a00,0,0x9491,1,0xe2859500,1759,0x849a
,1,0xe6bdb000,0,0x92d7,1,0xe7a2ba00,201,0x8a6d,1,0xe8b2a000,2929,0x9589
,1,0xe895aa00,2266,0x9593,1,0xe5b98500,4856,0x959d,1,0xe59d8200,0,0x8de2
,1,0xe6828c00,0,0x92ee,1,0xe7a68f00,1145,0x959f,1,0xe590bb00,2314,0x95ab
,1,0xe5a2b300,7102,0x95ad,1,0xe689ae00,2107,0x95af,1,0xe7b39e00,7917,0x95b3
,1,0xe88db700,0,0x89d7,1,0xe7b49b00,9658,0x95b4,1,0xe99bb000,9597,0x95b5
,1,0xe8819e00,0,0x95b7,1,0xe4b89900,5618,0x95b8,1,0xe585b500,1171,0x95ba
,1,0xe69f8400,1457,0x95bf,1,0xe4b8a600,5992,0x95c0,1,0xe7a98d00,0,0x90cf
,1,0xe894bd00,0,0x95c1,1,0xe583bb00,8854,0x95c6,1,0xe7999600,1749,0x95c8
,1,0xe8949100,1303,0x95cc,1,0xe7898700,3907,0x95d0,1,0xe8bf9400,935,0x95d4
,1,0xe5bc8100,1461,0x95d9,1,0xc39a0000,0,0x856f,1,0xe5a29300,6207,0x95e6
,1,0xe7af8000,0,0x90df,1,0xe3818b00,4441,0x82a9,1,0xe6888a00,2327,0x95e8
,1,0xe69aae00,2367,0x95e9,1,0xe580a300,3510,0x95ed,1,0xe5a0b100,0,0x95f1
,1,0xe5b3af00,0,0x95f5,1,0xe5ba9600,3835,0x95f7,1,0xe694be00,1573,0x95fa
,1,0xe69c8b00,0,0x95fc,1,0xe6b3a100,1889,0x9641,1,0xe8839e00,1929,0x9645
,1,0xe88ab300,2756,0x9646,1,0xd0ac0000,0,0x845d,1,0xe8908c00,0,0x9647
,1,0xe5888300,0,0x906e,1,0xe9a3bd00,1096,0x964f,1,0xe592a900,0,0x8840
,1,0xefbdba00,5585,0xba,1,0xe7898800,0,0x94c5,1,0xe59d8a00,1673,0x9656
,1,0xe5bf9800,1951,0x9659,1,0xe69f9000,2809,0x965e,1,0xe7b4a100,2612,0x9661
,1,0xe5a2a800,0,0x966e,1,0xe79da600,1802,0x9672,1,0xe5938900,0,0x8dc6
,1,0xefbe9a00,5991,0xda,1,0xe5b98c00,5294,0x9679,1,0xe5a59400,0,0x967a
,1,0xe6b49700,1965,0x90f4,1,0xe69cac00,2012,0x967b,1,0xe587a100,3594,0x967d
,1,0xe79b8600,0,0x967e,1,0xe698a700,0,0x9686,1,0xe8baaf00,0,0x8beb
,1,0xe4b98500,0,0x8b76,1,0xe6a79900,844,0x968a,1,0xe8869c00,5770,0x968c
,1,0xe4bfa300,1155,0x9693,1,0xe6bcab00,4270,0x969f,1,0xe8949300,2252,0x96a0
,1,0xe291a800,8532,0x8748,1,0xe7a9ba00,2500,0x8bf3,1,0xe99b8600,8053,0x8f57
,1,0xe5b2ac00,0,0x96a6,1,0xe5af8600,5349,0x96a7,1,0xe6b98a00,1123,0x96a9
,1,0xe8939100,5848,0x96aa,1,0xe5a69900,861,0x96ad,1,0xe5b3bb00,1476,0x8f73
,1,0xe7a58700,0,0x8b5f,1,0xe79ca000,0,0x96b0,1,0xe784a100,869,0x96b3
,1,0xe381b800,0,0x82d6,1,0xe9b5a100,0,0x96b7,1,0xe5a9bf00,1753,0x96b9
,1,0xe5a89800,3500,0x96ba,1,0xe5908d00,897,0x96bc,1,0xe79b9f00,1213,0x96bf
,1,0xe8bfb700,3832,0x96c0,1,0xe7b6bf00,1585,0x96c8,1,0xe8908400,2204,0x93b8
,1,0xe8959700,0,0x9599,1,0xe387ab00,2703,0x83ef,1,0xe7b7ac00,0,0x96c9
,1,0xe9b19200,0,0x9690,1,0xe7b6b200,4080,0x96d4,1,0xe588b000,0,0x939e
,1,0xe8809700,2093,0x96d5,1,0xc3a10000,472,0x8576,1,0xe8929900,2223,0x96d6
,1,0xe584b200,6761,0x96d7,1,0xe69ca800,0,0x96d8,1,0xe9809400,0,0x9372
,1,0xe9bb9900,0,0x96d9,1,0xe69da200,0,0x96db,1,0xe7b1be00,0,0x96e0
,1,0x74000000,0,0x74,1,0xe7b48b00,0,0x96e4,1,0xe586b600,0,0x96e8
,1,0xe880b600,2887,0x96eb,1,0xe9878e00,6842,0x96ec,1,0xe794bb00,0,0x89e6
,1,0xe9868700,9286,0x8f86,1,0xd0b30000,0,0x8473,1,0xe5bca500,4290,0x96ed
,1,0xe79fa200,812,0x96ee,1,0xe5949600,0,0x88a0,1,0xe896ac00,2299,0x96f2
,1,0xe99d9600,0,0x96f5,1,0xe6b09100,0,0x96af,1,0xe896ae00,1729,0x96f7
,1,0xe79aae00,1767,0x94e7,1,0xe6b2b900,0,0x96fb,1,0xe7999200,4772,0x96fc
,1,0xe8abad00,3215,0x9740,1,0xe5b9bd00,2993,0x9748,1,0xe4ba9200,8170,0x8cdd
,1,0xe59f9c00,936,0x9457,1,0xe682a000,0,0x9749,1,0xe689b900,0,0x94e1
,1,0xe6868200,4279,0x974a,1,0xc5820000,2385,0x85ac,1,0xe885ab00,0,0x8eee
,1,0xe68f9600,4555,0x974b,1,0xe78cb600,3002,0x9750,1,0xe8a39500,2940,0x9754
,1,0xe8aa9800,10545,0x9755,1,0xe5a58f00,6245,0x9174,1,0xe9818a00,5859,0x9756
,1,0xe5afb500,6365,0x929e,1,0xe694bf00,0,0x90ad,1,0xe2939500,7860,0x86c4
,1,0xe983b500,2521,0x9758,1,0xe29dbb00,93,0x86a4,1,0xe99b8400,0,0x9759
,1,0xe4bd9900,10562,0x975d,1,0xe4b88e00,6729,0x975e,1,0xe8aa8900,1164,0x975f
,1,0xe5bab800,0,0x9766,1,0xe6a58a00,7610,0x976b,1,0xe6a79800,3841,0x976c
,1,0xe6bab600,7589,0x976e,1,0xe7aaaf00,8806,0x9771,1,0xe383a500,34,0x8385
,1,0xe7be8a00,0,0x9772,1,0x5d000000,0,0x5d,1,0xe8938900,2239,0x9775
,1,0xe584aa00,877,0x9744,1,0xe8aca100,979,0x9777,1,0xe999bd00,1258,0x977a
,1,0xe6b5b400,1974,0x9781,1,0xe6b78000,1163,0x9784,1,0xe9a0bc00,0,0x978a
,1,0xe6b49b00,0,0x978c,1,0xe981a100,0,0x916b,1,0xe4b9b100,2833,0x9790
,1,0xe5858a00,0,0x87cd,1,0xe6bfab00,988,0x9794,1,0xe58fb000,0,0x91e4
,1,0xe5908f00,6627,0x9799,1,0xc3a80000,1875,0x857d,1,0xe7908600,1911,0x979d
,1,0xe6b0be00,612,0x94c3,1,0xe38f8b00,3055,0x827d,1,0xe7928300,2841,0x979e
,1,0xe797a200,879,0x979f,1,0xe8918e00,3717,0x97a8,1,0xe5909000,0,0x9366
,1,0xe795a500,1011,0x97aa,1,0xe6ba9c00,1066,0x97ad,1,0xe7959900,6796,0x97af
,1,0xe7a1ab00,1691,0x97b0,1,0xe7b29200,0,0x97b1,1,0xe9be8d00,2051,0x97b4
,1,0xe8999c00,5796,0x97b8,1,0xd0ba0000,0,0x847b,1,0xe4b8a100,855,0x97bc
,1,0xefbc8100,1024,0x8149,1,0xe7b3a700,917,0x97c6,1,0xe680a000,0,0x91d3
,1,0xe889af00,1347,0x97c7,1,0xe6b79100,0,0x8f69,1,0xe999b500,2069,0x97cb
,1,0xe28eaf00,4321,0x83db,1,0xe8b38300,2449,0x92c0,1,0xe9a09800,9531,0x97cc
,1,0xe58a9b00,3308,0x97cd,1,0xe580ab00,0,0x97cf,1,0xe6b78b00,1978,0x97d2
,1,0xe7879000,923,0x97d3,1,0xe790b300,0,0x97d4,1,0xe7b4af00,3418,0x97dd
,1,0xe4bba400,2098,0x97df,1,0xe4bcb600,2134,0x97e0,1,0x46000000,2540,0x46
,1,0xe4be8b00,0,0x97e1,1,0xe586b700,0,0x97e2,1,0xe78eb200,0,0x97e6
,1,0xe7a4bc00,5107,0x97e7,1,0xe88b9300,0,0x97e8,1,0xe9bda200,895,0x97ee
,1,0xe696ac00,0,0x8e61,1,0xe2958200,410,0x84be,1,0xe7ad9400,0,0x939a
,1,0xe29fa800,312,0x84d7,1,0xe6818b00,0,0x97f6,1,0xe6869000,1430,0x97f7
,1,0xe6bca300,7686,0x97f8,1,0xe580a400,3503,0x926c,1,0xe585b700,0,0x8bef
,1,0xe7b0be00,1501,0x97fa,1,0xe69c9f00,0,0x8afa,1,0xe980a300,3868,0x9841
,1,0xe8b38200,0,0x9847,1,0xe7afad00,1614,0x9855,1,0xe38aa500,8409,0x8786
,1,0xe8a2b700,0,0x88bf,1,0xe9ba9300,0,0x985b,1,0xe5869700,0,0x8fe7
,1,0xe6839100,0,0x9866,1,0xe69ea000,997,0x9867,1,0xe7828900,0,0x9846
,1,0xe9b7b200,7298,0x9868,1,0xe8a9ab00,1655,0x986c,1,0xe8978100,10004,0x986d
,1,0xe9838e00,0,0x9859,1,0xe8a8aa00,0,0x964b,1,0xe895a800,2041,0x986e
,1,0xe6a48000,1045,0x986f,1,0xe6b9be00,0,0x9870,1,0xefbcae00,150,0x826d
,1,0xe8859500,1641,0x9872,1,0xc3af0000,1924,0x8585,1,0xe6b2ab00,1932,0x9697
,1,0xf0a0ae9f,6678,0x9873,1,0xe5ad8100,0,0x9874,1,0xe5af9800,2358,0x9878
,1,0xe5afac00,1434,0x9879,1,0xe5b09200,6369,0x987a,1,0xefbd8e00,0,0x828e
,1,0xe5b1a200,5672,0x9880,1,0xe681ad00,0,0x8bb1,1,0xefa8bb00,4464,0x9881
,1,0x2f000000,4252,0x2f,1,0xefa8bc00,1187,0x9882,1,0xe882b200,7821,0x88e7
,1,0xeaab9400,2931,0x9883,1,0xe5b2a300,2967,0x9887,1,0xe99ead00,9631,0x95da
,1,0xeaab9d00,0,0x988e,1,0xe39fa200,1311,0x988f,1,0xe5b58600,0,0x9893
,1,0xe5b6a000,0,0x9899,1,0xe5b6b800,1809,0x989c,1,0xe4b89000,5616,0x98a0
,1,0xe7a8ae00,0,0x8eed,1,0xe4b89500,5793,0x98a1,1,0xe4b8aa00,6054,0x98a2
,1,0xe8b1ab00,3195,0x98ac,1,0xe4baa200,8418,0x98b4,1,0xe4bb9e00,1148,0x98be
,1,0xe5ad9c00,0,0x8e79,1,0xe4bb9f00,9729,0x98c0,1,0xe4bd9a00,6473,0x98c3
,1,0xe7a98e00,8642,0x896e,1,0xc5900000,101,0x85c1,1,0xe4bcb000,2243,0x98c4
,1,0xe4bdb600,10505,0x98c9,1,0xe4be9800,6260,0x98cc,1,0xe4bdbb00,0,0x98cd
,1,0xe4bdaf00,0,0x98d1,1,0xe4be9600,2739,0x98d3,1,0xe4bf9400,0,0x98d5
,1,0xcf850000,0,0x83d2,1,0xe4bf9800,2198,0x98d8,1,0xe3818c00,296,0x82aa
,1,0xe8999e00,2329,0x8bf1,1,0xe4bf9100,0,0x98da,1,0xe4bf9000,3485,0x98dc
,1,0xefa88f00,164,0x886a,1,0xe4bfa400,2089,0x98dd,1,0xe4bfa500,0,0x98de
,1,0xe4bfb600,1572,0x98e6,1,0xe6a39f00,0,0x938f,1,0xe580a100,6739,0x98e7
,1,0xe580ac00,4128,0x98e9,1,0xe4bfbe00,0,0x98ea,1,0xe5818300,10916,0x98ee
,1,0xe69c8300,0,0x98f0,1,0xe5819a00,968,0x98f4,1,0xe58d9700,0,0x93ec
,1,0xe5829a00,10956,0x98f9,1,0x18000000,11468,0x18,1,0xe582b400,5151,0x98fb
,1,0xe582b200,3528,0x98fc,1,0xe280b000,7526,0x81f1,1,0xe582b300,2816,0x9942
,1,0xe5839e00,4506,0x9945,1,0xe98fa100,0,0x8bbe,1,0xe583b900,0,0x994a
,1,0xe5849a00,0,0x9952,1,0xefbe9b00,662,0xdb,1,0xe59db000,0,0x8866
,1,0xe6af8500,0,0x8b42,1,0xe2819000,11486,0x81d4,1,0xca8e0000,1752,0x85f6
,1,0xe584ba00,7107,0x9954,1,0xe584bf00,0,0x9958,1,0xe59b9800,9824,0x9964
,1,0xe5869500,2989,0x996a,1,0xe5869600,0,0x996b,1,0xe586b000,0,0x9975
,1,0xe6839a00,0,0x8d9b,1,0xe5888400,1156,0x9983,1,0xe588ae00,3628,0x9989
,1,0xe5898f00,990,0x998c,1,0xe291a900,0,0x8749,1,0xe296bc00,328,0x81a5
,1,0xe99b8700,0,0x8cd9,1,0xe9a09a00,0,0x8c7a,1,0xe5898b00,0,0x998e
,1,0xe589b300,0,0x9995,1,0xe6898d00,0,0x8dcb,1,0xe68ea000,0,0x97a9
,1,0xe589bd00,6037,0x9997,1,0xe7a58800,0,0x8b46,1,0xe58bb800,2510,0x99ae
,1,0xe58c8800,2308,0x99b1,1,0xe381b900,0,0x82d7,1,0xe58c9a00,0,0x99b7
,1,0xe4b89700,5814,0x99c0,1,0xe5a49800,0,0x99c7,1,0xe582be00,899,0x8c58
,1,0xe58f8300,0,0x99d2,1,0xe699a600,0,0x8a41,1,0xe99b9900,1588,0x99d4
,1,0xe69bbc00,8108,0x99d6,1,0xe7b5a100,2623,0x978d,1,0xe387ac00,2714,0x83f0
,1,0xe58fae00,2775,0x99d8,1,0xe58fad00,4445,0x99da,1,0x1000000,952,0x1
,1,0xe590bd00,1272,0x99dd,1,0xe5ba8f00,0,0x8f98,1,0xc8b40000,6176,0x8663
,1,0xe5918000,0,0x99de,1,0xe6a9bf00,0,0x8a80,1,0xe89b8b00,0,0x9260
,1,0xe590bc00,3018,0x99e1,1,0xe591b500,0,0x99e8,1,0xe591b600,3540,0x99f1
,1,0xe5938700,1825,0x99f5,1,0xe592b800,5455,0x99f7,1,0xe5938800,8190,0x99fb
,1,0xe592a800,0,0x99fc,1,0xe5949400,1907,0x9a49,1,0xe595a300,4159,0x9a51
,1,0xe5969900,1970,0x9a5b,1,0xe5968000,3298,0x9a5c,1,0xe5968a00,2871,0x9a5e
,1,0xe8b0b700,0,0x924a,1,0xe5969f00,0,0x9a5f,1,0xe5969800,0,0x9a62
,1,0xe4b9b300,1059,0x93fb,1,0xe5969e00,0,0x9a63,1,0xe7908900,0,0x97ae
,1,0xe7959c00,3986,0x927b,1,0xe596ae00,7674,0x9a64,1,0xe8ac8400,0,0x93a3
,1,0xe5968300,9419,0x9a66,1,0xe596a900,0,0x9a67,1,0xe8bbbd00,0,0x8c79
,1,0xe5968700,1062,0x9a68,1,0xe596a800,5305,0x9a69,1,0xe5979a00,1343,0x9a6a
,1,0xe5978500,1733,0x9a6b,1,0xc3bd0000,0,0x8592,1,0xca950000,2386,0x85fc
,1,0xe5979f00,0,0x9a6c,1,0xe5978400,4558,0x9a6d,1,0xe8bc9d00,10226,0x8b50
,1,0xe9a1a700,0,0x8cda,1,0xe5979c00,0,0x9a6e,1,0xe597a400,1134,0x9a6f
,1,0xe5979400,0,0x9a70,1,0xe5989400,1189,0x9a71,1,0xe597b700,5464,0x9a72
,1,0xe2939600,7866,0x86c5,1,0xe5989600,2516,0x9a73,1,0xe29dbc00,0,0x86a5
,1,0xe597be00,5438,0x9a74,1,0xe597bd00,5613,0x9a75,1,0xe5989b00,5106,0x9a76
,1,0xe597b900,4663,0x9a77,1,0xe5b09600,408,0x90eb,1,0xe5b5a900,0,0x9093
,1,0xe5998e00,6774,0x9a78,1,0xe7ac8800,0,0x8b88,1,0xe5999000,6784,0x9a79
,1,0xe383a600,2295,0x8386,1,0xe388b900,422,0x878c,1,0xe598b600,3444,0x9a7c
,1,0xe598b200,3829,0x9a7d,1,0xe598af00,6333,0x9a82,1,0xe599aa00,7617,0x9a84
,1,0xe59a8600,4425,0x9a85,1,0xe59a8000,1824,0x9a86,1,0xe59a8a00,1619,0x9a87
,1,0xe59a8f00,8213,0x9a8a,1,0xc59e0000,439,0x85a5,1,0xe59aa500,2795,0x9a8b
,1,0xe59ab600,2033,0x9a8d,1,0xe5858b00,0,0x8d8e,1,0xe59ab400,3667,0x9a8e
,1,0xe58fb100,0,0x8eb6,1,0xe59b8200,1593,0x9a8f,1,0xc8bb0000,0,0x866d
,1,0xe59b8100,6786,0x9a91,1,0xe790b600,6862,0x9469,1,0xe8a28b00,3054,0x91dc
,1,0xe59b8e00,5435,0x9a95,1,0xe98ca800,0,0x9564,1,0xe59b9300,9674,0x9a97
,1,0xe5909100,0,0x8cfc,1,0xe59bae00,5561,0x9a99,1,0xe59c8000,9123,0x9a9b
,1,0xe59c8400,10925,0x9a9d,1,0xe6b19f00,0,0x8d5d,1,0xe6b6b200,0,0x8974
,1,0xe288bd00,4284,0x81e4,1,0xe59c8800,4965,0x9a9f,1,0xe59c9300,9447,0x9aa2
,1,0xe5978700,0,0x9aa5,1,0xefbc8200,372,0x81ae,1,0xe59b9700,9742,0x9a98
,1,0xe5a0aa00,1291,0x8aac,1,0xe5a5bd00,1356,0x8d44,1,0xe59ca600,1742,0x9aa7
,1,0xe59d8e00,7078,0x9aaa,1,0xe28eb000,170,0x83dc,1,0xe8b38400,0,0x9864
,1,0xe59d8000,7901,0x9aac,1,0xe59f8000,3603,0x9aaf,1,0xe59e8800,5270,0x9ab0
,1,0xe5a18a00,1302,0x89f2,1,0xe59da100,1142,0x9ab1,1,0xe59ea400,5877,0x9ab7
,1,0xe6bd8500,156,0x8ac1,1,0xe59f9400,2402,0x9abc,1,0xe294a300,2460,0x84b0
,1,0xe59f9200,2186,0x9abd,1,0xe59f9600,3316,0x9ac0,1,0xe5a1b200,0,0x9ac5
,1,0xe5a0a100,3737,0x9ac6,1,0xe5a1a200,1309,0x9ac7,1,0xe68c8700,0,0x8e77
,1,0xe6af8000,1328,0x9aca,1,0xe696ad00,0,0x9266,1,0xe5a1b900,0,0x9acd
,1,0xe5a2bb00,8039,0x9ad4,1,0xe29fa900,2230,0x84d8,1,0xe5a39900,2320,0x9adb
,1,0xe5a39800,0,0x9adc,1,0xe5a3a500,0,0x9add,1,0xe580a500,0,0x98e3
,1,0xe585b800,0,0x9354,1,0xe5a3b900,1233,0x9ae3,1,0xe5a3bc00,1270,0x9ae5
,1,0xe5a48200,7645,0x9ae7,1,0xe3808000,8186,0x8140,1,0xe7b89b00,4642,0x949b
,1,0xe38aa600,438,0x8787,1,0xe5a48a00,3068,0x9ae8,1,0xe5a4ac00,4698,0x9aed
,1,0xe5a4ad00,1531,0x9aee,1,0xe5a4be00,0,0x9af1,1,0xe590be00,0,0x8ce1
,1,0xe6a29300,340,0x88b2,1,0xe5a59500,0,0x9af3,1,0xe5a59000,1340,0x9af4
,1,0xe7be8e00,0,0x94fc,1,0xc5a50000,2804,0x85b2,1,0xe5a58e00,0,0x9af5
,1,0xe98db500,11075,0x8cae,1,0xe5a59800,5442,0x9af7,1,0xe5a5ac00,0,0x9afb
,1,0xefbcaf00,0,0x826e,1,0xe5a69d00,1365,0x9b42,1,0xe4beab00,1665,0x9b44
,1,0xe5a6b200,4219,0x9b46,1,0xe5a7a800,6275,0x9b48,1,0xe5a79900,9846,0x9b4b
,1,0xe5a89f00,9401,0x9b4e,1,0xe5a88900,1383,0x9b51,1,0xe998bb00,0,0x916a
,1,0xefbd8f00,5594,0x828f,1,0xe5a89a00,0,0x9b52,1,0xe5a9ac00,1394,0x9b54
,1,0xe6b38c00,0,0x94e5,1,0xe5a98900,1754,0x9b55,1,0xe5a8b500,0,0x9b56
,1,0xe5a8b600,2104,0x9b57,1,0xe5aa9a00,0,0x9b5a,1,0xe9999b00,2962,0x95c3
,1,0xe5aabc00,3011,0x9b5b,1,0xe4bd8d00,0,0x88ca,1,0xe5a29700,0,0x919d
,1,0xe5a7aa00,0,0x96c3,1,0xe5aabe00,1621,0x9b5c,1,0xe5ab8200,0,0x9b5e
,1,0xe5aba600,0,0x9b62,1,0xe5aba900,0,0x9b63,1,0xe5ab9600,2974,0x9b64
,1,0xe5ac8c00,2543,0x9b67,1,0xe5ad9500,3360,0x9b73,1,0xe5ad9a00,0,0x9b74
,1,0xe5ad9b00,0,0x9b75,1,0xe5ad9d00,0,0x8d46,1,0xe5ada500,0,0x9b76
,1,0xe697ba00,0,0x89a0,1,0xe7a98f00,8634,0x89b8,1,0xe5adb000,0,0x9b78
,1,0xe893ac00,8214,0x9648,1,0xe5ae8000,2684,0x9b7e,1,0xe5ae8300,0,0x9b80
,1,0xe5aea600,10765,0x9b81,1,0xe581b200,0,0x8ec3,1,0xe5b39000,1463,0x988b
,1,0xe5af8300,0,0x9b83,1,0xe69dad00,0,0x8d59,1,0xe5af8700,4070,0x9b84
,1,0xe3818d00,2254,0x82ab,1,0xe5af8900,1433,0x9b85,1,0xe5afa400,0,0x9b88
,1,0xe5afb600,3778,0x9b8f,1,0xefa89000,5334,0x8876,1,0xe5afb300,1439,0x9b90
,1,0xe58cb800,1245,0x99bc,1,0xe5b09300,0,0x9b95,1,0xe5b18600,1613,0x9b9c
,1,0xe788aa00,6740,0x92dc,1,0xe3878000,2339,0x81b7,1,0xe5adb100,2776,0x9ba1
,1,0xe5b1ac00,2811,0x9ba2,1,0xe5b28c00,0,0x9ba7,1,0xe9bb8d00,0,0x8b6f
,1,0xe58d9800,0,0x9250,1,0xe592ab00,0,0x9a40,1,0xefbdbc00,883,0xbc
,1,0xe5a69b00,2030,0x9baa,1,0xe5b2bb00,0,0x9bac,1,0xe5b2b600,1956,0x9bad
,1,0xe5b2b700,9270,0x9baf,1,0xc5ac0000,781,0x85dc,1,0xcc840000,10074,0x867a
,1,0xe5b2be00,0,0x9bb1,1,0xe5b3ba00,1466,0x9bb6,1,0xefbe9c00,904,0xdc
,1,0xe5b49b00,5297,0x9bbf,1,0xe5b49a00,1677,0x9bc3,1,0xe2819100,1723,0x81d5
,1,0xe5b49900,0,0x9bc4,1,0xe5b49800,2082,0x9bc5,1,0xe5b58e00,0,0x9bc8
,1,0xe5b5ac00,0,0x9bca,1,0xe5b68700,1444,0x9bcd,1,0xe99fbb00,5216,0x8943
,1,0xe5b6a200,5301,0x9bd0,1,0xe5a3a400,6227,0x9adf,1,0xe5b6ac00,1389,0x9bd2
,1,0xe2878400,588,0x8248,1,0xe5b6ae00,1814,0x9bd3,1,0xe291aa00,0,0x874a
,1,0xe296bd00,0,0x81a4,1,0xe5b69000,9590,0x9bd5,1,0xe5b78900,0,0x9bd8
,1,0xe5b79300,0,0x9bda,1,0xe5b79200,1427,0x9bdb,1,0xe5b79600,1392,0x9bdc
,1,0xe68ea100,0,0x8dcc,1,0xe5b3bd00,1475,0x9bb5,1,0xe7a58900,0,0x8e83
,1,0xe5b7ab00,8700,0x9bde,1,0xe5b7b200,0,0x9bdf,1,0xe381ba00,0,0x82d8
,1,0xe5b7b500,0,0x9be0,1,0xc78d0000,0,0x84ed,1,0xe5b88b00,0,0x9be1
,1,0xe5b8b600,3742,0x9be6,1,0xe5b98400,6003,0x9be8,1,0xe5b98300,1521,0x9be9
,1,0xe5b98000,2895,0x9bea,1,0xe5b99400,4594,0x9bed,1,0xe7b5a200,4625,0x88ba
,1,0xe387ad00,11479,0x83f1,1,0xe5b9b500,6520,0x9bf2,1,0xe5b9bf00,10856,0x9bf6
,1,0xe5baa000,4532,0x9bf7,1,0xe5bb8100,0,0x9bf8,1,0xe69a8700,0,0x89c9
,1,0xe69f9a00,0,0x974d,1,0xe5bb9000,0,0x9bfb,1,0xe5bb9a00,10264,0x9c43
,1,0xe5bb9b00,0,0x9c44,1,0xe5bba200,3024,0x9c45,1,0xe8a5b200,3087,0x8f50
,1,0xe5bba100,11018,0x9c46,1,0xe5bbac00,0,0x9c49,1,0xe58ea500,0,0x99ce
,1,0xe5bc8300,0,0x9c50,1,0xe5bc8900,0,0x9c51,1,0xe5bd9c00,1647,0x9c53
,1,0xe78faa00,1682,0x8c5d,1,0xe5bc9100,7551,0x9c55,1,0xe5bc9600,3093,0x9c56
,1,0xe5bca900,0,0x9c57,1,0xe5bd8100,0,0x9c5a,1,0xe5bd8800,3631,0x9c5b
,1,0x69000000,0,0x69,1,0xe599ab00,7632,0x9a80,1,0xe5bd9700,0,0x9c61
,1,0xe5bd9900,1687,0x9c62,1,0xe6b5a600,349,0x8959,1,0xe5bda100,2037,0x9c63
,1,0xe5bdb300,1570,0x9c65,1,0xe5be8200,0,0x9c68,1,0xcc8b0000,103,0x8678
,1,0xe5be8a00,2997,0x9c6a,1,0xe4ba9400,8184,0x8cdc,1,0xe5be8800,5723,0x9c6b
,1,0xe5a4b100,1342,0x8eb8,1,0xe5be8700,6285,0x9c6d,1,0xe5be9e00,7057,0x9c6e
,1,0xe5be9900,0,0x9c6f,1,0xe5bea000,4520,0x9c71,1,0xe5bf9d00,2949,0x9c7a
,1,0xe680a100,4846,0x9c7d,1,0xe681a000,0,0x9c7e,1,0xe6809000,0,0x9c81
,1,0xe5a59100,245,0x8c5f,1,0xe680b100,1404,0x9c84,1,0xe68fae00,0,0x8af6
,1,0xe6809b00,4312,0x9c85,1,0xe2939700,7870,0x86c6,1,0xe680ab00,0,0x9c87
,1,0xe29dbd00,0,0x86a6,1,0xe6808f00,0,0x9c89,1,0xe6818100,3382,0x9c8c
,1,0xe681aa00,3433,0x9c8d,1,0xe5ab8400,0,0x889a,1,0xe6818200,0,0x9c96
,1,0xe681ac00,6021,0x9c97,1,0xe69ab400,0,0x965c,1,0xe681ab00,4566,0x9c98
,1,0xe6819900,0,0x9c99,1,0xe383a700,371,0x8387,1,0xe6828d00,2723,0x9c9b
,1,0xe683a700,3448,0x9c9c,1,0xc7940000,0,0x84f6,1,0xe6829600,11039,0x9ca1
,1,0xe589bf00,6432,0x9996,1,0xe5bb9d00,0,0x9c42,1,0xe7809e00,0,0x93d2
,1,0xe6828b00,5756,0x9ca5,1,0xe7b78f00,0,0x918d,1,0xe682b800,9091,0x9ca7
,1,0xd1890000,173,0x848b,1,0x52000000,2837,0x52,1,0xe5858c00,0,0x995b
,1,0xe58a9f00,3668,0x8cf7,1,0xe58fb200,0,0x8e6a,1,0xe683a000,0,0x9ca8
,1,0xe6839300,0,0x9ca9,1,0xe682b500,5935,0x9cae,1,0xe6839800,0,0x9caf
,1,0xe38f8d00,2164,0x8783,1,0xe6848600,3104,0x9cb2,1,0xe6848000,10010,0x9cb5
,1,0xe683b400,3204,0x9cb6,1,0xe5909200,0,0x87f3,1,0xe6848300,0,0x9cb8
,1,0xe684a100,9147,0x9cb9,1,0xe683b100,0,0x9cbb,1,0xe6b1a000,4191,0x9272
,1,0xe6848d00,0,0x9cbc,1,0xe6858700,1660,0x9cbe,1,0xe684be00,4968,0x9cbf
,1,0xe684a800,0,0x9cc0,1,0xe684a700,0,0x9cc1,1,0xefbc8300,0,0x8194
,1,0xe4bba100,963,0x87a6,1,0xe6858400,6194,0x9cc9,1,0xe685b300,1363,0x9cca
,1,0xe6859800,1985,0x9ccc,1,0xe6859900,0,0x9ccd,1,0xe28eb100,0,0x83dd
,1,0xe685ab00,2453,0x9ccf,1,0xc5ba0000,0,0x85b3,1,0xe99da200,0,0x96ca
,1,0xe4bc8100,0,0x8ae9,1,0xe5a18b00,1305,0x9ac8,1,0xe685b400,0,0x9cd0
,1,0xe685af00,0,0x9cd1,1,0xe685a500,0,0x9cd2,1,0xe8828700,0,0x94a3
,1,0xe294a400,2135,0x84a7,1,0xe685b100,2189,0x9cd3,1,0xe6859f00,0,0x9cd4
,1,0xe685b500,5816,0x9cd7,1,0xe9a8a800,0,0x91cb,1,0xe6869400,0,0x9cdc
,1,0xe6869a00,1411,0x9cdd,1,0xe5b1a400,6378,0x9177,1,0xe5b6b700,0,0x9bd6
,1,0xe6868a00,4383,0x9cde,1,0xe7ad9600,0,0x8df4,1,0x3b000000,11481,0x3b
,1,0xe6869100,0,0x9cdf,1,0xe6878a00,0,0x9ce3,1,0xe6878900,0,0x9ce4
,1,0xe580a600,0,0x8c91,1,0xe6878800,0,0x9ce6,1,0xe6878b00,6058,0x9cea
,1,0xe687b600,0,0x9cef,1,0xe781ab00,1604,0x89ce,1,0xe3808100,1878,0x8141
,1,0xe687bf00,2675,0x9cf2,1,0xe38aa700,72,0x8788,1,0xe6888000,9476,0x9cf6
,1,0xe9b48e00,5319,0x89a8,1,0xe5869900,0,0x8eca,1,0xe6888800,0,0x9cf7
,1,0xe5bd8a00,1560,0x8bae,1,0xe688aa00,1432,0x9d42,1,0xe688ae00,1639,0x9d43
,1,0xe6acba00,7374,0x8b5c,1,0xe688b300,0,0x9d46,1,0xe6898e00,2805,0x9d48
,1,0xe6899e00,0,0x9d49,1,0xe689a300,3577,0x9d4a,1,0xe68a8900,0,0x9d50
,1,0xe5919f00,0,0x99ea,1,0xefbcb000,0,0x826f,1,0xe68a9200,0,0x9d52
,1,0xe68a9600,5857,0x9d54,1,0xe68b8f00,6883,0x9d5b,1,0xe68b8200,0,0x9d63
,1,0xe68b8700,4945,0x9d64,1,0xe98e9600,0,0x8dbd,1,0xe68a9b00,0,0x9d65
,1,0xe68bb100,10149,0x9d69,1,0xefbd9000,5575,0x8290,1,0xe68ca700,0,0x9d6a
,1,0xe681af00,0,0x91a7,1,0xe68c8200,0,0x9d6b,1,0xe6b8a000,0,0x8b94
,1,0xe68c8800,7711,0x9d6c,1,0xe882b400,0,0x8de6,1,0xe68d9000,2838,0x9d6f
,1,0xe68d8d00,3253,0x9d71,1,0xe68d8f00,7929,0x9d73,1,0xe4bd8e00,5677,0x92e1
,1,0xc2a40000,0,0x8543,1,0xe5a7ab00,0,0x9550,1,0xe5acbe00,6320,0x9b6e
,1,0x24000000,0,0x24,1,0xe68e9600,0,0x9d74,1,0xe68e8e00,0,0x9d75
,1,0xe68e8000,2973,0x9d76,1,0xe68eab00,0,0x9d77,1,0xcc990000,0,0x8699
,1,0xe68ea300,5908,0x9d79,1,0xe68e8f00,6469,0x9d7a,1,0xe68d9500,0,0x95df
,1,0xe5b2b100,0,0x91d0,1,0xe68e9f00,1507,0x9d7c,1,0xe7a99000,484,0x88aa
,1,0xe68fa900,1602,0x9d82,1,0xe68f8000,2070,0x9d83,1,0xe68fa300,3372,0x9d85
,1,0xe68f8900,0,0x9d86,1,0xe9afa800,0,0x8c7e,1,0xe9b4bb00,0,0x8d83
,1,0xe68f9200,0,0x9d87,1,0xe68fb600,3601,0x9d88,1,0xe5bdb700,0,0x9c66
,1,0xe690b400,1846,0x9d8b,1,0xe3818e00,0,0x82ac,1,0xe6908600,2337,0x9d8c
,1,0xe690b600,1587,0x9d8f,1,0xe983bd00,3491,0x9373,1,0xefa89100,5338,0x9892
,1,0xe587a600,0,0x8f88,1,0xe58cb900,1244,0x9543,1,0xe5be9700,0,0x93be
,1,0xc4850000,1609,0x85aa,1,0xe6909700,6742,0x9d91,1,0xe3878100,2343,0x825b
,1,0xe38c9400,0,0x8760,1,0xe690a800,0,0x9d92,1,0xe6908f00,4774,0x9d93
,1,0xe5888600,3606,0x95aa,1,0xe692a500,0,0x9d9b,1,0xe592ac00,0,0x99f9
,1,0xefbdbd00,19,0xbd,1,0xe6928800,1902,0x9d9d,1,0xe6938500,7424,0x9da1
,1,0xe280b200,1957,0x818c,1,0xe6938700,0,0x9da2,1,0xe6939800,2966,0x9da4
,1,0xe6938200,0,0x9da5,1,0xe693b100,0,0x9da6,1,0xe693a700,0,0x9da7
,1,0xefbe9d00,11,0xdd,1,0xd000000,924,0xd,1,0xe682bc00,6829,0x9389
,1,0xe2819200,209,0x8259,1,0xe693a000,4265,0x9da9,1,0xe693a100,0,0x9daa
,1,0xe693a300,3184,0x9dac,1,0xe693b400,0,0x9db0,1,0xe693b200,10172,0x9db1
,1,0xe6948000,5218,0x9db3,1,0xe4be9b00,0,0x8b9f,1,0xe6839c00,1118,0x90c9
,1,0xe688af00,0,0x8b59,1,0xe79a8400,0,0x9349,1,0xe28c9800,206,0x849c
,1,0xe291ab00,2094,0x874b,1,0xe6949800,2154,0x9db5,1,0xe8bb9200,3274,0x8cac
,1,0xe6948500,0,0x9db7,1,0xe694a300,0,0x9db9,1,0xe6959800,5780,0x9dc5
,1,0xc2ab0000,0,0x8547,1,0xe68ea200,0,0x9254,1,0xe6959e00,5348,0x9dc6
,1,0xe6959d00,7849,0x9dc7,1,0xe6968200,0,0x9dca,1,0xe6968300,8099,0x9dcb
,1,0xe381bb00,0,0x82d9,1,0xe6969b00,0,0x9dcd,1,0xcca00000,3422,0x868c
,1,0xe696ab00,6053,0x9dcf,1,0xe6978300,3759,0x9dd1,1,0xe6978400,0,0x9dd4
,1,0xe6978c00,1523,0x9dd5,1,0xe6979200,3859,0x9dd6,1,0xe69db200,2565,0x9ddc
,1,0xe3829b00,2281,0x814a,1,0xe387ae00,0,0x83f2,1,0xe6988300,5101,0x9dde
,1,0xe697bb00,9822,0x9ddf,1,0xe69db300,8668,0x9de0,1,0xe588b300,3633,0x998a
,1,0xe698b400,1528,0x9de3,1,0xe5bfa400,6613,0x9c77,1,0xe6989c00,0,0x9de4
,1,0xe6998f00,3685,0x9de5,1,0xe89b8d00,0,0x8c75,1,0xe9809700,0,0x9080
,1,0xe985aa00,0,0x978f,1,0xe5848000,0,0x8b56,1,0xe6998100,0,0x9de8
,1,0xe58ea600,0,0x99cd,1,0xe699a400,6114,0x9deb,1,0xe699a700,0,0x9dec
,1,0xc48c0000,6768,0x85bb,1,0xe699a800,2583,0x9ded,1,0xe8a18000,7556,0x8c8c
,1,0xe699b000,2827,0x9df0,1,0xe8aba600,0,0x92fa,1,0xe69a8300,9403,0x9df1
,1,0xe69a8800,0,0x9df2,1,0xe69a8e00,0,0x9df3,1,0xe599ac00,7642,0x9a83
,1,0xe69a8400,1642,0x9df5,1,0xe69a9800,1688,0x9df6,1,0xe69a9d00,0,0x9df7
,1,0xe6baba00,0,0x934d,1,0xe69b8100,0,0x9df8,1,0xe69abe00,3432,0x9dfb
,1,0xe996a300,0,0x8a74,1,0xe99bb600,0,0x97eb,1,0xe4ba9500,5644,0x88e4
,1,0xe59f9f00,0,0x88e6,1,0xe5a4b200,0,0x9aef,1,0xe689bc00,3991,0x9d4e
,1,0xe2889200,0,0x817c,1,0xe69b9600,0,0x9e42,1,0xe7a5b700,2435,0x9398
,1,0xe69b9a00,2947,0x9e43,1,0xe69ba000,3003,0x9e44,1,0xe698bf00,0,0x9e45
,1,0xe69bb500,3378,0x9e49,1,0xe69c8f00,4983,0x9e4b,1,0xe68a9c00,0,0x94b2
,1,0xe5afb800,0,0x90a1,1,0xe69c9600,0,0x9e4c,1,0xe2939800,0,0x86c7
,1,0xe7abaa00,2514,0x9247,1,0xe29dbe00,2212,0x86a7,1,0xe69c9e00,0,0x9e4d
,1,0xe69ca600,0,0x9e4e,1,0xe69cae00,3434,0x9e51,1,0xe69d8100,0,0x9e54
,1,0xc2b20000,1455,0x854b,1,0xe695a200,0,0x8ab8,1,0xe69da000,3687,0x9e59
,1,0xe69d9900,2180,0x9e5a,1,0xe69da300,0,0x9e5b,1,0xe383a800,2316,0x8388
,1,0xe69da400,4681,0x9e5c,1,0xe69e8900,3443,0x9e5d,1,0xe69db000,0,0x9e5e
,1,0xe69daa00,9368,0x9e61,1,0xe5b68b00,1493,0x9388,1,0xe69e8c00,0,0x9e62
,1,0xe69e8b00,1826,0x9e63,1,0xe69ea100,0,0x9e65,1,0xe69eb700,8954,0x9e67
,1,0xe69faf00,2724,0x9e68,1,0xe69eb400,0,0x9e69,1,0xe69fac00,4467,0x9e6a
,1,0xe5858d00,5166,0x96c6,1,0xe58aa000,0,0x89c1,1,0xe58fb300,0,0x8945
,1,0xe6a18800,0,0x88c4,1,0xe6a69b00,4094,0x9059,1,0xe69eb300,0,0x9e6b
,1,0xe69eb800,8393,0x9e6d,1,0xe9828400,2802,0x8ad2,1,0xe69fa200,1782,0x9e71
,1,0xe69eb900,2879,0x9e73,1,0xe69f8e00,3201,0x9e74,1,0xe69fa700,0,0x9e76
,1,0xe6aa9c00,0,0x9e77,1,0xe6a09e00,9155,0x9e78,1,0xe6a18600,9334,0x9e79
,1,0xc4930000,0,0x859d,1,0xe796ab00,6973,0x8975,1,0xe8a88000,295,0x8cbe
,1,0xe6a18000,4969,0x9e7b,1,0xe6a19900,0,0x9e82,1,0xe6a1a300,0,0x9e83
,1,0xefbc8400,12,0x8190,1,0xe6a28f00,7189,0x9e87,1,0xe6a29400,2547,0x9e89
,1,0xe685b600,1701,0x8c63,1,0xe6a1b400,0,0x9e8f,1,0xe6a48100,2060,0x9e96
,1,0xe28eb200,2394,0x83de,1,0xe6a39800,3042,0x9e99,1,0xe6a4a200,1858,0x9e9a
,1,0xe6a3a100,4072,0x9e9c,1,0xe6a3a700,3359,0x9ea0,1,0xe6818300,5739,0x9c94
,1,0xe6869600,5817,0x9cd9,1,0xe6a3a000,0,0x9ea9,1,0xe6a4aa00,8974,0x9eac
,1,0xe7a29100,0,0x94e8,1,0xe294a500,576,0x84bc,1,0xe6a4a300,0,0x9eae
,1,0xe6a4a100,0,0x9eaf,1,0xe6a5b900,8161,0x9eb1,1,0xe6a5b700,0,0x9eb2
,1,0xe6a5b800,0,0x9eb4,1,0xe68c8900,0,0x88c2,1,0xe5b1a500,0,0x979a
,1,0xe696af00,908,0x8e7a,1,0xe6a59900,7262,0x9ebc,1,0xe6a5a100,0,0x9ebe
,1,0xe29fab00,2606,0x84da,1,0xe6a6bf00,0,0x9ec6,1,0xe6a69500,0,0x9ed5
,1,0xe6a79e00,0,0x9ed7,1,0xe580a700,0,0x87bd,1,0xe6a7a800,0,0x9ed8
,1,0xc2b90000,405,0x854f,1,0xc9910000,0,0x8658,1,0xe6a88200,3869,0x9ed9
,1,0xe3808200,175,0x8142,1,0xe6ac8a00,8352,0x9edc,1,0xe38aa800,589,0x8789
,1,0xe6a7a700,3070,0x9edf,1,0xe5818700,0,0x98ef,1,0xe6a88500,0,0x9ee0
,1,0xe6ab8100,5382,0x9ee8,1,0xe6a9b200,0,0x9eed,1,0xe6a9a200,0,0x9ef1
,1,0xe7879f00,1632,0x9a7a,1,0xe6a9a600,0,0x9ef3,1,0xe6aaa000,4694,0x9ef9
,1,0xe6aaa200,0,0x9efb,1,0xe8a8ad00,5488,0x90dd,1,0xe6aaa300,0,0x9efc
,1,0xe58c8d00,0,0x99b3,1,0xe6aab800,0,0x9f45,1,0xefbcb100,0,0x8270
,1,0xe6abaa00,1913,0x9f4d,1,0xe6acb800,1667,0x9f56,1,0xe6acb700,1710,0x9f57
,1,0xe79b9c00,0,0x9f58,1,0xe9898400,9341,0x9353,1,0xe98e9700,0,0x9199
,1,0xe6acb900,5219,0x9f59,1,0xe5928000,0,0x99f0,1,0xefbd9100,0,0x8291
,1,0xe4bcaf00,1092,0x948c,1,0xe681b000,1113,0x8a86,1,0xe6ad9900,4125,0x9f5f
,1,0xc49a0000,497,0x85bd,1,0xe6ad9b00,1823,0x9f61,1,0xe6ada100,2103,0x9f63
,1,0xe6ae8000,2932,0x9f67,1,0xe6ae8d00,3257,0x9f6a,1,0xe6ae9800,0,0x9f6b
,1,0xe4bd8f00,5679,0x8f5a,1,0xe6aea400,0,0x9f6e,1,0xe687a300,0,0x9cee
,1,0xe6aeab00,4138,0x9f70,1,0xe6aeb200,2080,0x9f72,1,0xe6af8600,8510,0x9f77
,1,0xe6af8b00,2141,0x9f78,1,0xe88dbb00,0,0x89ac,1,0xe6af9300,6541,0x9f79
,1,0xe6afab00,0,0x9f7c,1,0xe6afb300,0,0x9f7d,1,0xe6afaf00,5835,0x9f7e
,1,0xe5ad9f00,0,0x96d0,1,0xe692a900,0,0x9d9c,1,0xe9babe00,6474,0x9f80
,1,0xe6b09300,4163,0x9f82,1,0xe6b09400,8759,0x9f83,1,0xe893ae00,0,0x9840
,1,0xe6b0a300,4167,0x9f86,1,0xe6b19e00,1935,0x9f87,1,0xe6b1a200,1937,0x9f89
,1,0xe581b400,0,0x91a4,1,0xe5b39200,0,0x988c,1,0xe5b8a500,1512,0x9083
,1,0xe69daf00,0,0x9474,1,0xe782b900,596,0x935f,1,0xe3818f00,264,0x82ad
,1,0xe6b28200,2336,0x9f8b,1,0xe7bebd00,0,0x8948,1,0xe6b28100,3482,0x9f8e
,1,0xe5829400,42,0x87c2,1,0xe587a700,5173,0x91fa,1,0xe58cba00,647,0x8be6
,1,0xe5be9800,0,0x9c70,1,0xc9980000,1849,0x864d,1,0xe6a8b500,1895,0x8fbf
,1,0xe3878200,4490,0x825c,1,0xe6b38500,0,0x9f9a,1,0xe6b2ae00,4736,0x9f9c
,1,0xe989b100,0,0x8d7a,1,0xe5888700,3607,0x90d8,1,0xe58d9a00,0,0x948e
,1,0xe6b39b00,5521,0x9fa0,1,0xefbdbe00,5674,0xbe,1,0xe7898c00,0,0x9476
,1,0xe6b3af00,4134,0x9fa1,1,0xe280b300,1955,0x818d,1,0xe6b3aa00,0,0x9fa3
,1,0xe98a9100,2858,0x914c,1,0xe8a18d00,3175,0x9fa5,1,0xe6b4ab00,0,0x9fa7
,1,0xe6b49900,0,0x9faa,1,0xefbe9e00,5205,0xde,1,0xe4bdbc00,5687,0x8cf1
,1,0xe682bd00,6832,0x9cac,1,0xe2819300,808,0x86fb,1,0xe6b5a300,1745,0x9faf
,1,0xe6b5a400,1972,0x9fb1,1,0xe6b59a00,4987,0x9fb2,1,0xe6b5b900,0,0x9fb3
,1,0xe6bfa400,3264,0x9fb7,1,0xe6b68500,5625,0x9fb8,1,0xe59e9300,0,0x9ab4
,1,0xe6b88a00,1989,0x9fbb,1,0xe688b000,0,0x9d44,1,0xe6b78700,0,0x9fbd
,1,0xe6bfa100,0,0x9447,1,0xe291ac00,2121,0x874c,1,0xe6b78600,0,0x9fc0
,1,0xe6b79e00,4343,0x9fc2,1,0xe6b79200,0,0x9fc5,1,0xe6b7a400,0,0x9fc9
,1,0xe6b8ad00,2004,0x9fcd,1,0xe6b9ae00,1973,0x9fce,1,0x75000000,11471,0x75
,1,0xe693b600,0,0x9daf,1,0xe6b8ae00,7633,0x9fcf,1,0xe6b99f00,0,0x9fd2
,1,0xe6b8be00,4264,0x9fd3,1,0xe381bc00,889,0x82da,1,0xe6b9ab00,2018,0x9fd5
,1,0xe6b8ab00,2003,0x9fd6,1,0xe6b89f00,4254,0x9fd9,1,0xe68f8300,634,0x91b5
,1,0xe5b49f00,6421,0x9bbe,1,0xe5b9b200,597,0x8ab1,1,0xe6b8ba00,0,0x9fdb
,1,0xe6b89d00,2200,0x9fdf,1,0xe3829c00,0,0x814b,1,0xe387af00,0,0x83f3
,1,0xe6b8b800,10650,0x9fe0,1,0xe6ba8200,0,0x9fe1,1,0xe6ba9800,3530,0x9fe3
,1,0xe6bb8900,3634,0x9fe4,1,0xe69a8900,0,0x9df4,1,0xe6bab700,0,0x9fe5
,1,0xe784a600,0,0x8fc5,1,0xe789b900,0,0x93c1,1,0xe89b8e00,0,0x8a61
,1,0xe6bb9300,2780,0x9fe6,1,0xe392b500,2824,0x87d1,1,0xe5848100,11024,0x994d
,1,0xe5899400,0,0x9991,1,0xe6babd00,2038,0x9fe7,1,0xe593ba00,0,0x9a4d
,1,0xe6baaf00,4086,0x9fe8,1,0xc99f0000,1648,0x85f3,1,0xe6bb9400,2377,0x9feb
,1,0xe6bb9500,4278,0x9fec,1,0xe6ba8f00,3089,0x9fed,1,0xe6bd8100,2765,0x9ff1
,1,0xe6bbac00,2902,0x9ff4,1,0xe6bcbf00,4291,0x9ff7,1,0xccbc0000,0,0x8694
,1,0xe6bbb200,5633,0x9ff8,1,0xe6ab8200,0,0x9f44,1,0xe6bbaf00,4283,0x9ffa
,1,0xe6bcb200,2059,0x9ffb,1,0xe6bb8c00,0,0x9ffc,1,0xe6bcbe00,2875,0xe040
,1,0xe9919100,540,0x8ad3,1,0xe996a400,0,0x8d7d,1,0xe99bb700,0,0x978b
,1,0x5e000000,8195,0x5e,1,0xe59fa000,0,0x9575,1,0xe6bbb700,0,0xe042
,1,0xe6b68800,0,0x8fc1,1,0xe2889300,2042,0x827a,1,0xe7a0a500,2387,0x9375
,1,0xe6be8100,0,0xe046,1,0xe6bdaf00,7857,0xe048,1,0xe6bfb300,7753,0xe04a
,1,0xe6bdad00,0,0xe04b,1,0xe5a08000,10547,0x9678,1,0xe6858a00,0,0x9cc2
,1,0xe6bd9800,0,0xe04e,1,0xe6be8e00,1912,0xe04f,1,0xe6be9100,2096,0xe050
,1,0xe2939900,202,0x86c8,1,0xe6bea300,4399,0xe054,1,0xe29dbf00,865,0x86a8
,1,0xe6bf8600,0,0xe058,1,0xe6bf9f00,3383,0xe05a,1,0xe6bfae00,0,0xe060
,1,0xe6bf9b00,4373,0xe061,1,0xe7808f00,3781,0xe067,1,0xce9d0000,235,0x83ab
,1,0xe7809b00,6123,0xe069,1,0xe88c8200,2192,0x96ce,1,0xe7809a00,8130,0xe06a
,1,0xe383a900,197,0x8389,1,0xe6bdb400,1920,0xe06b,1,0xe7809d00,3345,0xe06c
,1,0xe7809800,11131,0xe06d,1,0xe7809f00,4053,0xe06e,1,0xe5b68c00,1494,0x9bb8
,1,0xe5bb9f00,1544,0x955f,1,0xe6a0a900,0,0x9e7a,1,0xe6a5bc00,0,0x984f
,1,0xe7b79100,0,0x97ce,1,0xe780b000,0,0xe06f,1,0xe981a500,0,0x9779
,1,0xe986b800,0,0x8ff8,1,0xe5858e00,0,0x9365,1,0xe780be00,2536,0xe070
,1,0xe781a300,5482,0xe073,1,0xe7829900,6632,0xe074,1,0xe6a69c00,0,0x9ed4
,1,0xe7829200,0,0xe075,1,0xe783b100,7457,0xe077,1,0xe782ac00,9695,0xe078
,1,0xe9879800,0,0x9342,1,0x47000000,267,0x47,1,0xe58b8100,3205,0x99a4
,1,0xe782b800,0,0xe079,1,0xe782b300,3133,0xe07a,1,0xe7838b00,9213,0xe07d
,1,0xe7839900,6798,0xe080,1,0xc9a60000,1692,0x8641,1,0xe6b6b500,0,0x9fbc
,1,0xe7848900,0,0xe081,1,0xe98d8b00,4795,0x93e7,1,0xe8b2a700,6017,0x956e
,1,0xe7849c00,0,0xe083,1,0xefbc8500,1027,0x8193,1,0xe59b9a00,414,0x8efa
,1,0xe7849900,6195,0xe084,1,0xe685b700,1699,0x9ccb,1,0xe6b79500,0,0x9fca
,1,0xe289a000,331,0x8182,1,0xe28eb300,583,0x83df,1,0xe8b38700,0,0x8e91
,1,0xe785a500,0,0xe085,1,0xe7859500,3309,0xe086,1,0xe4bc8300,10373,0x87a9
,1,0xe7868800,0,0xe087,1,0xe785a600,6261,0xe088,1,0xe785a200,0,0xe089
,1,0xe7858c00,0,0xe08a,1,0xe8828900,0,0x93f7,1,0xe7859600,0,0xe08b
,1,0xe785ac00,2036,0xe08c,1,0xe99e8400,0,0x8a93,1,0xe7868f00,6703,0xe08d
,1,0xe786a800,0,0xe091,1,0xe786b900,8182,0xe094,1,0xe786be00,3768,0xe095
,1,0xe7879400,6379,0xe098,1,0xe696b000,6056,0x9056,1,0xe7878e00,0,0xe099
,1,0xe88d8f00,0,0x8960,1,0xe29fac00,11488,0x84db,1,0xe787b500,6732,0xe09d
,1,0xe787b900,5089,0xe09f,1,0xe787bf00,2255,0xe0a0,1,0xe580a800,3506,0x98e0
,1,0xe7889000,8421,0xe0a2,1,0xe788a800,7525,0xe0a4,1,0xcea40000,0,0x83b1
,1,0xe6a1b600,0,0x89b1,1,0xe3808300,4608,0x8156,1,0x30000000,263,0x30
,1,0xe788ad00,2735,0xe0a5,1,0xe788ac00,0,0xe0a6,1,0xe5818800,0,0x98f3
,1,0xe788b000,0,0xe0a7,1,0xe788bf00,2627,0xe0ab,1,0xe5bd8c00,0,0x9c5c
,1,0xe7898600,3857,0xe0ad,1,0xe787a000,3895,0xe09a,1,0xe7898b00,7377,0xe0ae
,1,0xe7899800,4498,0xe0af,1,0xe789b400,5980,0xe0b0,1,0xe789be00,2284,0xe0b1
,1,0xe78a8700,0,0xe0b4,1,0xe78a9600,6766,0xe0b6,1,0xe78ab200,0,0xe0ba
,1,0xefbcb200,0,0x8271,1,0xe78b9200,4104,0xe0bf,1,0xe78ba200,0,0xe0c0
,1,0xe78ba000,4224,0xe0c1,1,0xe78ba100,7609,0xe0c2,1,0xe78bb900,3114,0xe0c3
,1,0xe78c9700,0,0xe0c6,1,0xe78c9600,4810,0xe0c9,1,0xe78cb400,0,0xe0cb
,1,0xefbd9200,1039,0x8292,1,0xe59ca700,921,0x88b3,1,0xe78caf00,2072,0xe0cc
,1,0xe78ca900,0,0xe0cd,1,0xc9ad0000,4256,0x85f2,1,0xe78d8e00,0,0xe0d0
,1,0xe78d8f00,6814,0xe0d1,1,0xe78daa00,2934,0xe0d4,1,0xe78db000,3255,0xe0d6
,1,0xe8beba00,10280,0x95d3,1,0xe59d8700,214,0x8bcf,1,0xe78dbb00,2105,0xe0d9
,1,0xe78dba00,0,0xe0da,1,0xe78eb300,0,0xe0dc,1,0xe792a200,0,0xe0e3
,1,0xe790b200,2417,0xe0e8,1,0xe7a8b200,2479,0x88ee,1,0xe7919500,2393,0xe0ea
,1,0xe7919f00,3012,0xe0ec,1,0xe7919900,3367,0xe0ed,1,0xe7918100,9082,0xe0ee
,1,0xe791a900,5842,0xe0f0,1,0xe68d9700,0,0x92bb,1,0xe5b2b300,0,0x8a78
,1,0x19000000,0,0x19,1,0xe791aa00,8640,0xe0f3,1,0xe7938a00,0,0xe0f9
,1,0xe7b3b800,8856,0x8e85,1,0xe7938f00,0,0xe0fa,1,0xe7939400,0,0xe0fb
,1,0xe793a000,6898,0xe140,1,0xe581b500,1178,0x92e3,1,0xe793a700,3806,0xe142
,1,0xe793ae00,0,0xe144,1,0xe5bdb900,1569,0x96f0,1,0xe782ba00,8079,0x88d7
,1,0xe3819000,195,0x82ae,1,0xe793b200,3962,0xe145,1,0xe793b000,4703,0xe146
,1,0xe793b100,3494,0xe147,1,0xe7948300,3519,0xe14b,1,0xe7948c00,0,0xe14d
,1,0xe58cbb00,0,0x88e3,1,0xe69e9000,1288,0x90cd,1,0xe6a3a300,0,0x9ea6
,1,0xe6a8b600,1894,0x9eee,1,0xe3878300,0,0x825d,1,0xe7948e00,0,0xe14e
,1,0xe7948d00,0,0xe14f,1,0xe7949500,0,0xe150,1,0xe5888800,3608,0x8aa0
,1,0xe794ac00,0,0xe154,1,0xe794bc00,5522,0xe155,1,0xefbdbf00,6676,0xbf
,1,0xe7959b00,3909,0xe15a,1,0xe7959a00,3934,0xe15c,1,0xe795ab00,0,0xe160
,1,0xe7968a00,0,0xe167,1,0xe7969a00,0,0xe16b,1,0xe796a500,3174,0xe16d
,1,0xe7978200,5738,0xe16f,1,0xe796b300,0,0xe170,1,0xefbe9f00,912,0xdf
,1,0xe7978300,5298,0xe171,1,0xe78f8000,6217,0xe0df,1,0xe7949300,0,0xe151
,1,0xe6b9af00,0,0x9392,1,0xe7978a00,2083,0xe178,1,0xe7979200,4010,0xe179
,1,0xe7979900,0,0xe17a,1,0xe797a300,0,0xe17b,1,0xe7988100,3749,0xe181
,1,0xe4be9d00,0,0x88cb,1,0x2000000,0,0x2,1,0xe797b000,0,0xe182
,1,0xe79a8600,1781,0x8a46,1,0xe797b200,2819,0xe184,1,0xe291ad00,2120,0x874d
,1,0xe889b600,368,0x8990,1,0xe797b300,3270,0xe185,1,0xe7989f00,5065,0xe189
,1,0xe798a700,0,0xe18a,1,0xe798a000,0,0xe18b,1,0xe5a99a00,6278,0x8da5
,1,0xe798a100,0,0xe18c,1,0xe798a200,3421,0xe18d,1,0xe798a400,5785,0xe18e
,1,0xe7aa9f00,0,0x8c41,1,0xe88fa900,0,0x95ec,1,0xe381bd00,6679,0x82db
,1,0xe798b400,0,0xe18f,1,0xe798b000,0,0xe190,1,0xe7999800,10843,0xe196
,1,0xe68f8400,0,0x9d89,1,0xe799a100,5025,0xe197,1,0xe5b9b300,6510,0x95bd
,1,0xe2988000,0,0x83e6,1,0xe7b09200,0,0x99d3,1,0xe3829d00,0,0x8154
,1,0xe387b000,2706,0x83f4,1,0xe799a200,0,0xe198,1,0xe9b19700,0,0x97d8
,1,0xe799a900,4735,0xe19a,1,0xe799ac00,3639,0xe19d,1,0xe799b600,0,0xe1a0
,1,0xe69f9d00,0,0x9e70,1,0xe6a4b000,0,0x9ebd,1,0xe79a8000,0,0xe1a3
,1,0xc4bd0000,0,0x85a2,1,0xe9809900,2778,0x9487,1,0xe985ac00,0,0x8f56
,1,0xe5848200,9940,0x994e,1,0xe79a8300,5412,0xe1a4,1,0xe58ea800,0,0x907e
,1,0xe79a9600,0,0xe1a8,1,0xe79a9300,3099,0xe1a9,1,0xceb20000,1650,0x83c0
,1,0xe78fad00,0,0x94c7,1,0xe79a9900,2539,0xe1aa,1,0xe79ab000,0,0xe1ac
,1,0xe79ab400,0,0xe1ad,1,0xe79b8200,3188,0xe1b1,1,0xe58f8800,0,0x9694
,1,0xe79b8d00,2900,0xe1b2,1,0xe79b9200,7676,0xe1b4,1,0xe6ab8300,0,0x9f43
,1,0xe79b9e00,0,0xe1b5,1,0xe6b5a900,587,0x8d5f,1,0xe287b400,0,0x824d
,1,0xe79ba500,0,0xe1b7,1,0xe79ba700,3194,0xe1b8,1,0xe996a500,0,0x94b4
,1,0xe79baa00,0,0xe1b9,1,0xe79bbb00,8202,0xe1bb,1,0xc3810000,0,0x8557
,1,0xe79c8800,3754,0xe1bc,1,0xe689be00,3993,0x9d51,1,0xe79c8700,4031,0xe1bd
,1,0xe7a0a600,0,0x8dd4,1,0xe885b000,4360,0x8d98,1,0xe79c8400,4824,0xe1be
,1,0xe79ca900,0,0xe1bf,1,0xe79c9e00,3347,0xe1c1,1,0xe79ca500,0,0xe1c2
,1,0xe6858b00,702,0x91d4,1,0xe5aaa700,419,0x8899,1,0xe5afba00,0,0x8e9b
,1,0xe79ca600,4035,0xe1c3,1,0xc9bb0000,2,0x85f1,1,0xd0930000,2515,0x8443
,1,0xe7b0bf00,0,0x95eb,1,0xe79c9b00,0,0xe1c4,1,0xe79da800,2244,0xe1c9
,1,0xe9acb100,0,0x9f54,1,0xe79dab00,2721,0xe1ca,1,0xe5b09a00,0,0x8fae
,1,0xe5b5ad00,0,0x9897,1,0xe79d9b00,0,0xe1cb,1,0xe79da500,2153,0xe1cc
,1,0xe79dbe00,2214,0xe1ce,1,0xe383aa00,232,0x838a,1,0xe79e8e00,4684,0xe1d0
,1,0xe79e8b00,3451,0xe1d1,1,0xe79eb000,0,0xe1d5,1,0xe79eb600,6415,0xe1d6
,1,0xe79eb900,6433,0xe1d7,1,0xe5bba000,0,0x8fb1,1,0xe6a0aa00,0,0x8a94
,1,0xe6a5bd00,0,0x8a79,1,0xe7b79200,2665,0x8f8f,1,0xe89c9c00,0,0x96a8
,1,0xe79f9700,3333,0xe1de,1,0xe79fa300,0,0xe1e1,1,0xe7a08c00,0,0xe1e4
,1,0xe7a4a600,3671,0xe1e6,1,0xe7a0a000,2218,0xe1e7,1,0xe7a4aa00,0,0xe1e8
,1,0xe7869400,0,0x976f,1,0xe7a18500,0,0xe1e9,1,0xe790ba00,2354,0xe0e9
,1,0xe7a28e00,2406,0xe1ea,1,0xe7a1b400,0,0xe1eb,1,0xe98cac00,0,0x9842
,1,0xe7a28c00,2786,0xe1ef,1,0xe7a2aa00,0,0xe1f2,1,0xe7a2be00,0,0xe1f8
,1,0xe7a2bc00,3605,0xe1f9,1,0xe7a3a700,6799,0xe240,1,0xceb90000,181,0x83c7
,1,0xe7a39a00,0,0xe241,1,0xe8a88200,0,0x92f9,1,0xe7a3bd00,5806,0xe242
,1,0xe8b2a800,0,0x89dd,1,0xe7a49100,0,0xe246,1,0xefbc8600,5568,0x8195
,1,0xe59b9b00,847,0x8e6c,1,0xe680a500,0,0x8b7d,1,0xe6b28300,0,0x9780
,1,0xe7978d00,4003,0xe177,1,0xe289a100,725,0x81df,1,0xe28eb400,0,0x83e0
,1,0xe7a49900,2450,0xe247,1,0xe7a4ac00,2425,0xe248,1,0xe7a4ab00,0,0xe249
,1,0xe7a59700,10381,0xe24c,1,0xe7a59f00,0,0xe24d,1,0xc3880000,6262,0x855e
,1,0xe7a69d00,0,0xe254,1,0xe690be00,448,0x8def,1,0xe7a29300,0,0x894f
,1,0xe7a7a600,0,0x9060,1,0xe7acb900,8738,0x8df9,1,0xe7a6a700,3315,0xe255
,1,0xe9bd8b00,0,0xe256,1,0xe7a7a700,3391,0xe25e,1,0xe7a7ac00,0,0xe25f
,1,0xe7a7a100,2461,0xe260,1,0xe7a7a300,2468,0xe261,1,0xe5b6ba00,0,0x97e4
,1,0xe7a89800,0,0xe264,1,0xe7a89900,2476,0xe265,1,0xd09a0000,11482,0x844b
,1,0xe7a8a000,8970,0xe266,1,0xe7a8b100,0,0xe269,1,0xe7a8be00,5113,0xe26b
,1,0xe580a900,0,0x98e8,1,0xe585bc00,0,0x8c93,1,0xe7a9a100,2488,0xe270
,1,0xe5bcad00,0,0x9c58,1,0xe6a1b700,7183,0x9e84,1,0xe9be9d00,0,0xe273
,1,0xe8989600,2326,0x9f50,1,0xe7bdb200,0,0x8f90,1,0xe7a9b000,4880,0xe274
,1,0xe5818900,10926,0x88cc,1,0xe7a9b900,2520,0xe275,1,0xe7a9bd00,4540,0xe276
,1,0xe7aa8800,0,0xe277,1,0xe7828e00,189,0x898a,1,0xe7aa9800,0,0xe27a
,1,0xe6acbd00,1917,0x8bd4,1,0xe7aab600,9125,0xe280,1,0xe8a39c00,0,0x95e2
,1,0xe7ab8500,0,0xe281,1,0xe7ab8400,0,0xe282,1,0xe58c8f00,0,0x99b5
,1,0xe9828300,5492,0xe284,1,0xefbcb300,0,0x8272,1,0xe6a88a00,0,0x9ee6
,1,0xe7ab8d00,0,0xe287,1,0xe792a700,0,0xe0f8,1,0xe797ba00,0,0xe183
,1,0xe7ab9500,0,0xe289,1,0xe7ab9a00,0,0xe28c,1,0xe7ab9d00,6075,0xe28d
,1,0xe998bf00,0,0x88a2,1,0xefbd9300,0,0x8293,1,0xe59ca800,0,0x8ddd
,1,0xe7aba100,0,0xe28e,1,0xe7aba600,2530,0xe290,1,0xe6b8a300,0,0x9fd4
,1,0xe7abb000,0,0xe292,1,0xe7ac8a00,4567,0xe295,1,0xe7ac8600,0,0xe296
,1,0xe7acb300,3258,0xe297,1,0xe8bebb00,10283,0x92d2,1,0xe4bd9100,0,0x9743
,1,0xe6829200,0,0x9ca3,1,0xe7acb500,5828,0xe29b,1,0xe6b98300,0,0x9fda
,1,0xe7aca800,7103,0xe29c,1,0xe7acb600,4508,0xe29d,1,0xe888aa00,0,0x8d71
,1,0xe7ac8400,4579,0xe2a0,1,0xe7ac8b00,6544,0xe2a2,1,0xe7ad8c00,4572,0xe2a3
,1,0xe7adac00,6600,0xe2ab,1,0xe7ae9f00,2548,0xe2af,1,0xc38f0000,0,0x8565
,1,0xe692ab00,0,0x958f,1,0xe7af8100,4324,0xe2b9,1,0xe7aeb400,0,0xe2bc
,1,0xe7b09100,8761,0xe2c0,1,0xe7afa500,8859,0xe2c3,1,0xe7b1a000,3375,0xe2c4
,1,0xe7b09700,0,0xe2ca,1,0xe7b08d00,10774,0xe2cb,1,0xe581b600,3517,0x8bf4
,1,0xe7b0a700,6390,0xe2ce,1,0xe7b0b700,3822,0xe2d1,1,0xe69db100,0,0x938c
,1,0xe7af8600,7200,0xe2bd,1,0xe3819100,241,0x82af,1,0xd0a10000,0,0x8452
,1,0xe7b0bd00,4501,0xe2d3,1,0xe7b19800,8799,0xe2da,1,0xe7b19f00,3522,0xe2db
,1,0xe587a900,0,0x997d,1,0xe5b98700,0,0x9bf1,1,0xe7b1a400,0,0xe2dc
,1,0xe7b1ac00,3698,0xe2df,1,0xe7b1b500,0,0xe2e0,1,0xe7b29000,2344,0xe2e2
,1,0xe7b2a400,4503,0xe2e3,1,0xe7b2ad00,5308,0xe2e4,1,0xe7b2a200,2847,0xe2e5
,1,0xe7b2ab00,5176,0xe2e6,1,0xe58d9c00,0,0x966d,1,0xe592af00,0,0x9a5d
,1,0xe6a48400,0,0x9ea4,1,0xe7b2a100,0,0xe2e7,1,0xe6aeaa00,0,0x9f6f
,1,0xe7b2a800,2595,0xe2e8,1,0xe7b2b300,9271,0xe2e9,1,0xe7b2b200,3129,0xe2ea
,1,0xe7b2b100,0,0xe2eb,1,0xe7b2ae00,0,0xe2ec,1,0xe7b38000,0,0xe2ef
,1,0xe7b38500,4100,0xe2f0,1,0xe4bdbe00,5691,0x87b2,1,0xe7b39800,8844,0xe2f2
,1,0xe7b39200,0,0xe2f3,1,0xe799a700,0,0xe19c,1,0xe7b39c00,8846,0xe2f4
,1,0xe7b3b200,7784,0xe2f8,1,0xe7b3b400,0,0xe2f9,1,0xe7b3b600,2692,0xe2fa
,1,0xe5998200,139,0x895c,1,0xe7b48600,0,0xe2fc,1,0xe6839f00,0,0x88d2
,1,0xe688b200,0,0x9d45,1,0xe6ba9000,192,0x8cb9,1,0xe79f9a00,0,0xe1df
,1,0xe291ae00,8542,0x874e,1,0xe7b49c00,2501,0xe341,1,0xe99b8c00,0,0x8e93
,1,0xe7b49500,8868,0xe342,1,0xe7b48a00,8755,0xe343,1,0xe7b58500,0,0xe344
,1,0xe7b58b00,6280,0xe345,1,0xe5aeae00,359,0x8b7b,1,0xe7b4ae00,8878,0xe346
,1,0xe7b4b200,0,0xe347,1,0xe7b4bf00,4389,0xe348,1,0xe7afb300,4592,0xe2c8
,1,0xe381be00,664,0x82dc,1,0xe7b4b500,2613,0xe349,1,0xe7b58600,10644,0xe34a
,1,0xe7b5b300,9562,0xe34b,1,0xe7b59600,5914,0xe34c,1,0x6a000000,11484,0x6a
,1,0xe5b9b400,6519,0x944e,1,0xe2988100,0,0x83e7,1,0xe7b09300,4595,0xe2c7
,1,0xe3829e00,129,0x8155,1,0xe387b100,2707,0x83f5,1,0xe7b58e00,0,0xe34d
,1,0xe7b5b200,10865,0xe34e,1,0xe583a300,3527,0x9948,1,0xe588b600,0,0x90a7
,1,0xe7b5ae00,0,0xe350,1,0xe69f9e00,0,0x9e6f,1,0xe7b58f00,0,0xe351
,1,0xe7b5a300,2573,0xe352,1,0xe7b69300,2644,0xe353,1,0xd0a80000,468,0x8459
,1,0xe7b68900,2634,0xe354,1,0xe5848300,0,0x87c9,1,0xe5899600,0,0x9655
,1,0xe58ea900,0,0x8958,1,0xe7b59b00,0,0xe355,1,0xe7b68f00,8935,0xe356
,1,0xe7b5bd00,4117,0xe357,1,0xe78fae00,0,0xe0e1,1,0xe7b69b00,0,0xe358
,1,0xe8a69600,416,0x8e8b,1,0xe7b6ba00,4958,0xe359,1,0xe7b6ae00,4989,0xe35a
,1,0xe58f8900,10611,0x8db3,1,0xe7b6a300,0,0xe35b,1,0xe599af00,7678,0x8856
,1,0xe7b6b500,7574,0xe35c,1,0xe6b09700,0,0x8b43,1,0xe6b5aa00,0,0x9851
,1,0xe287b500,1775,0x824e,1,0xe7b6bd00,5725,0xe35e,1,0xe9919300,0,0x96f8
,1,0xe7b6ab00,4637,0xe35f,1,0xe7b6a200,3278,0xe361,1,0xe4ba9800,1063,0x986a
,1,0xe7b6af00,8771,0xe362,1,0xe684ac00,0,0x9cc5,1,0xe689bf00,6933,0x8fb3
,1,0xe6bb9d00,4281,0x91ea,1,0xe7a0a700,0,0x8b6d,1,0xe7a5ba00,2433,0xe251
,1,0xe7b6b800,2941,0xe364,1,0xe7b69f00,3287,0xe365,1,0xe7b6b000,3891,0xe366
,1,0x53000000,528,0x53,1,0xe6858c00,1133,0x8d51,1,0xe7b79800,2669,0xe367
,1,0xe7b7bb00,6366,0xe36b,1,0xe7b7b200,7798,0xe36c,1,0xe2939b00,0,0x86ca
,1,0xe7abad00,0,0xe291,1,0xe7b7a100,3007,0xe36d,1,0xe7b8a100,4649,0xe371
,1,0xe7b89f00,3380,0xe374,1,0xe7b9a600,3436,0xe379,1,0xe7b8b500,0,0xe37b
,1,0xe7b9a700,0,0xe382,1,0xe7b99e00,0,0xe385,1,0xe69ab800,0,0x9e41
,1,0xe88c8400,0,0x89d6,1,0xe8919700,2202,0x9298,1,0xe383ab00,333,0x838b
,1,0xe7b99900,4687,0xe386,1,0xe7b9aa00,0,0xe389,1,0xe7b9a900,0,0xe38a
,1,0xe7ba8300,3554,0xe38d,1,0xe7ba8800,6435,0xe392,1,0xc39d0000,1261,0x8572
,1,0xe6a0ab00,0,0x9e81,1,0xe6a5be00,0,0x9eb7,1,0xe7ba8900,8957,0xe393
,1,0xe7ba8c00,0,0xe394,1,0xe8a1b000,0,0x908a,1,0xe7ba9300,3101,0xe397
,1,0xe5859000,0,0x8e99,1,0xe58aa300,0,0x97f2,1,0xe58fb600,670,0x8a90
,1,0xe7ba9600,9027,0xe399,1,0xe7869500,0,0xe090,1,0xe7ba9b00,6788,0xe39b
,1,0xe7ba9c00,2694,0xe39c,1,0xe7bd8500,3016,0xe39f,1,0xd0af0000,76,0x8460
,1,0xe7bd9400,0,0xe3a6,1,0xe58b8300,0,0x9675,1,0xe7be8c00,3166,0xe3b3
,1,0xe7be9e00,0,0xe3b5,1,0xe59abc00,9221,0x9a90,1,0xe7be9d00,6805,0xe3b6
,1,0xe7919b00,0,0x896c,1,0xe7be9a00,0,0xe3b7,1,0xe7bea300,2842,0xe3b8
,1,0xe98d8d00,2897,0x9374,1,0x3c000000,595,0x3c,1,0xe7beb900,2955,0xe3bb
,1,0xefbc8700,182,0x81ad,1,0xe4bba500,9908,0x88c8,1,0xe680a600,3677,0x9c88
,1,0xe7beb800,2750,0xe3be,1,0xe2848f00,4249,0x827c,1,0xe289a200,4034,0x81e9
,1,0xe28eb500,2099,0x83e1,1,0xe7bf9500,2447,0xe3c3,1,0xe7bfa100,0,0xe3c5
,1,0xe7bfb300,3310,0xe3c8,1,0xe8808b00,0,0xe3cd,1,0xe6818600,0,0x9c91
,1,0xe6869900,0,0x9cd8,1,0xe68bac00,5882,0x8a87,1,0xe28a8200,0,0x81bc
,1,0xe8828b00,7809,0x985d,1,0xe294a800,2132,0x84b7,1,0xe8809800,2535,0xe3cf
,1,0xe8809900,3319,0xe3d0,1,0xe8809c00,0,0xe3d1,1,0xe8819200,0,0xe3d8
,1,0xe8819800,0,0xe3d9,1,0xe68c8c00,0,0x9d67,1,0xe8819a00,5943,0xe3da
,1,0xe8819f00,4319,0xe3db,1,0xe881a200,2553,0xe3dc,1,0xe881a800,4577,0xe3dd
,1,0xe29fae00,2222,0x86a9,1,0xe897b700,0,0x8f93,1,0xe881b300,10590,0xe3de
,1,0xe881bf00,3466,0xe3e4,1,0xe580aa00,10915,0x98e2,1,0xe8828400,5967,0xe3e5
,1,0xe5b79b00,1502,0x9bdd,1,0xe8828600,0,0xe3e6,1,0xe781af00,0,0x9394
,1,0xe3808500,443,0x8158,1,0xe8989700,8353,0x9f41,1,0xe8828500,2743,0xe3e7
,1,0xe8829a00,9227,0xe3ea,1,0xe882ac00,3514,0xe3ed,1,0xe5869d00,0,0x87d2
,1,0xe58bb000,0,0x87e7,1,0xe5bd8e00,0,0x9c5d,1,0xc3a40000,6653,0x8579
,1,0xe6a7ab00,0,0x9ee5,1,0xe6acbe00,0,0x8abc,1,0x25000000,0,0x25
,1,0xe8839900,2807,0xe3f0,1,0xe8839600,3032,0xe3f4,1,0xe8848900,11080,0xe3f5
,1,0xe58c9000,0,0x99b4,1,0xe8849b00,0,0xe3f8,1,0xefbcb400,47,0x8273
,1,0xe6a88b00,0,0x94f3,1,0xe884a900,0,0xe3f9,1,0xe6b2b100,4228,0x9f9d
,1,0xe884a300,0,0xe3fa,1,0xe8a99000,9851,0x8dbc,1,0xd0b60000,467,0x8477
,1,0xe884be00,4806,0xe442,1,0xe8859100,0,0xe444,1,0xefbd9400,0,0x8294
,1,0xe883bc00,5675,0xe445,1,0xe5a1bc00,0,0x887a,1,0xe885b100,4236,0xe446
,1,0xe6b8a400,7629,0x9fdd,1,0xe885ae00,0,0xe447,1,0xe8868800,8500,0xe44c
,1,0xe8868a00,2939,0xe44d,1,0xe8869500,0,0xe451,1,0xe8bebc00,3323,0x8d9e
,1,0xe8859f00,0,0xe454,1,0xe5a29c00,0,0x92c4,1,0xe687a600,0,0x9ced
,1,0xe8869300,3771,0xe455,1,0xe6be9700,2079,0x8ac0,1,0xe886b000,7830,0xe457
,1,0xe886b500,2481,0xe458,1,0xe886be00,0,0xe459,1,0xe99f9300,0,0x8ad8
,1,0xe886b800,0,0xe45a,1,0xe886bd00,2920,0xe45b,1,0xe8878000,0,0xe45c
,1,0xe8878200,6330,0xe45d,1,0xe886ba00,5984,0xe45e,1,0xe7a48100,0,0x8fca
,1,0xe8878900,8644,0xe45f,1,0xe88e9e00,0,0x8ace,1,0xe7b3ba00,8860,0xe2fb
,1,0xe8878d00,7892,0xe460,1,0xe8879100,0,0xe461,1,0xe8879900,0,0xe462
,1,0xe8879800,3515,0xe463,1,0xe8878800,6393,0xe464,1,0xe6989f00,0,0x90af
,1,0xe000000,1503,0xe,1,0xe7af8700,0,0x95d1,1,0xe3819200,128,0x82b0
,1,0xe7b9ad00,0,0x969a,1,0xe8879a00,5939,0xe465,1,0xe887a000,3483,0xe467
,1,0xe887a700,4092,0xe468,1,0xe587aa00,0,0x93e2,1,0xe8888100,7906,0xe46c
,1,0xe8888200,6589,0xe46d,1,0xe8888500,6674,0xe46e,1,0xe6a8b800,0,0x9ef5
,1,0xe8889000,2538,0xe472,1,0xe38c9800,341,0x8763,1,0xe8889600,2812,0xe473
,1,0xe888a900,0,0xe474,1,0xe5888a00,208,0x8aa7,1,0xe888ab00,0,0xe475
,1,0xe888b300,0,0xe477,1,0xe6a48500,0,0x88d6,1,0xc3ab0000,121,0x8581
,1,0xca830000,1675,0x85e6,1,0xe8898000,0,0xe478,1,0xe8899900,0,0xe479
,1,0xe8899800,0,0xe47a,1,0xe8899a00,10076,0xe47c,1,0xe889a200,0,0xe480
,1,0xe889a800,0,0xe481,1,0xe889ab00,6119,0xe483,1,0xe4bdbf00,0,0x8e67
,1,0xe78f8200,1678,0x89cf,1,0xe6b49e00,1967,0x93b4,1,0xe799a800,0,0xe199
,1,0xe79ebb00,0,0xe1db,1,0xe888ae00,3186,0xe484,1,0xd0bd0000,0,0x847e
,1,0xe889b100,8038,0xe485,1,0xe889b700,6664,0xe486,1,0xe889be00,0,0xe488
,1,0xe5a3a900,1326,0x8886,1,0xe5a8bc00,1391,0x8fa9,1,0xe79a8800,0,0xe1a5
,1,0xe79f9b00,1732,0x96b5,1,0xe291af00,4325,0x874f,1,0xe889b800,0,0xe487
,1,0xe88a8d00,3276,0xe489,1,0xe88a9200,0,0xe48a,1,0xe88abb00,3084,0xe48d
,1,0xe5a48900,1332,0x95cf,1,0xe6899300,0,0x91c5,1,0xe88ba100,0,0xe48f
,1,0xe88ba300,0,0xe490,1,0xe88b9f00,5956,0xe491,1,0xc58c0000,0,0x8599
,1,0xe88e9300,2559,0xe496,1,0xe381bf00,8899,0x82dd,1,0xe88c8300,8754,0xe497
,1,0xe88b9e00,3426,0xe49a,1,0xe9b0b900,0,0x8a8f,1,0xe68f8600,0,0x9d84
,1,0xe5b4a200,0,0x9bc2,1,0xcf810000,1250,0x83cf,1,0xe2988200,0,0x83e8
,1,0xe7b09400,2562,0xe2c1,1,0xe88c8600,8913,0xe49b,1,0xe387b200,9053,0x83f6
,1,0xe88b9c00,3427,0xe49c,1,0xe88b9900,0,0xe49e,1,0xe88c9600,4786,0xe4a1
,1,0xe588b700,6016,0x8dfc,1,0xe5ba9500,0,0x92ea,1,0xe88cb100,7794,0xe4a3
,1,0xe88d8000,11162,0xe4a4,1,0xe88cb900,0,0xe4a5,1,0xe88d9000,4492,0xe4a6
,1,0xe88caf00,2791,0xe4a8,1,0xe88cab00,2825,0xe4a9,1,0xe5848400,0,0x89ad
,1,0xe5899700,0,0x87dd,1,0xe88c9700,6300,0xe4aa,1,0xe593bd00,0,0x9a4a
,1,0xe7858900,0,0x97f9,1,0xe88c9800,3922,0xe4ab,1,0xe88e9a00,5719,0xe4ad
,1,0xe88e9600,2787,0xe4b1,1,0xe8a69700,9806,0x9460,1,0xe88ca300,0,0xe4b2
,1,0xe88e8e00,9408,0xe4b3,1,0xe58f8a00,0,0x8b79,1,0xe88dbc00,2972,0xe4b6
,1,0xe88db500,7680,0xe4b9,1,0xe88e8900,0,0xe4bb,1,0xc3b20000,0,0x8588
,1,0xca8a0000,617,0x8654,1,0xe88ea800,0,0xe4bc,1,0xe88fab00,9365,0xe4bf
,1,0xe8908300,0,0xe4c2,1,0xe88f9800,3217,0xe4c3,1,0xe8908b00,0,0xe4c4
,1,0xe4ba9900,8226,0x9869,1,0xe59fa300,0,0x9ac1,1,0xe8908700,0,0xe4c7
,1,0xe7968200,0,0xe169,1,0xe6bb9e00,0,0x91d8,1,0xe88fa000,0,0xe4c8
,1,0xe890a200,0,0xe4cb,1,0xe9978700,0,0x88c5,1,0xe890a000,0,0xe4cc
,1,0xe8948600,0,0xe4cf,1,0xe8959a00,8273,0xe4d4,1,0xe6858d00,0,0x9cb0
,1,0xe8928400,0,0xe4d5,1,0xe892ad00,8172,0xe4d8,1,0xe8928200,0,0xe4da
,1,0xe2939c00,7874,0x86cb,1,0xe88ba500,0,0x8ee1,1,0xe890b800,0,0xe4ce
,1,0xe8918600,0,0xe4dc,1,0xe890ac00,0,0xe4dd,1,0xe890b500,9617,0xe4e0
,1,0xe5ab8900,0,0x8eb9,1,0xe6909300,0,0x9d8d,1,0xe5b5af00,539,0x8db5
,1,0xe69ab900,0,0x9df9,1,0xe88c8500,0,0x8a9d,1,0xc5930000,8131,0x8649
,1,0xe383ac00,153,0x838c,1,0xe89bbe00,0,0x89e9,1,0xe8938a00,0,0xe4e1
,1,0xe892b900,0,0xe4e3,1,0xe9b7b900,0,0x91e9,1,0xe8929f00,0,0xe4e5
,1,0xe69b9900,1259,0x8f8c,1,0xcf880000,3866,0x83d5,1,0xe7b28100,6691,0x8bcc
,1,0xe8939900,4640,0xe4e6,1,0xe8938d00,0,0xe4e7,1,0xe8939000,3043,0xe4ea
,1,0xe8938100,0,0xe4eb,1,0xe8938600,11052,0xe4ec,1,0xe8939600,3669,0xe4ed
,1,0xe58fb700,3702,0x8d86,1,0xe892a100,6634,0xe4ee,1,0xe894a100,7270,0xe4ef
,1,0xe78ba900,6789,0x8eeb,1,0xe893bf00,4493,0xe4f0,1,0xe893b400,4865,0xe4f1
,1,0xe8949700,3105,0xe4f2,1,0xe8949800,3143,0xe4f3,1,0xe894ac00,0,0xe4f4
,1,0xe8949f00,10689,0xe4f5,1,0xe8949500,8225,0xe4f6,1,0xe8949400,4444,0xe4f7
,1,0xe6ac9200,0,0x9f52,1,0xe7919c00,0,0xe0ef,1,0xe6b6b800,0,0x9fbf
,1,0xe893bc00,0,0xe4f8,1,0xe8958000,10599,0xe4f9,1,0xe895a300,2915,0xe4fa
,1,0xe8959800,9526,0xe4fb,1,0xefbc8800,1014,0x8169,1,0xe8958800,9937,0xe4fc
,1,0xe5a0b000,351,0x8981,1,0xe8958100,3958,0xe540,1,0xc3b90000,510,0x858e
,1,0xca910000,713,0x865f,1,0xe8988200,0,0xe541,1,0xe8b38a00,565,0x91af
,1,0xe8958b00,2956,0xe542,1,0xe8959500,5023,0xe543,1,0xe8968000,5075,0xe544
,1,0xe896a400,3556,0xe545,1,0xe5a6a300,1366,0x9b45,1,0xe68bad00,0,0x9040
,1,0xe28a8300,0,0x81bd,1,0xe7a29500,586,0x8dea,1,0xe8879f00,0,0xe466
,1,0xe88cb200,4408,0xe4a2,1,0xe8968800,0,0xe546,1,0xe8969100,0,0xe547
,1,0xe8968a00,5081,0xe548,1,0xe5a78300,0,0x888e,1,0xe5ac9600,6302,0x9b69
,1,0xe896a800,3788,0xe549,1,0xe5b6bc00,0,0x9bd7,1,0xe7a88800,0,0xe262
,1,0xe88d9200,0,0x8d72,1,0xe29faf00,2234,0x86aa,1,0xe895ad00,0,0xe54a
,1,0xe8969400,3404,0xe54b,1,0xe8969b00,3468,0xe54c,1,0xe9b3b300,842,0x9650
,1,0xe8968700,0,0xe54e,1,0xe8969c00,0,0xe54f,1,0xe5bcaf00,1558,0x9c5e
,1,0xe781b000,0,0x8a44,1,0xe3808600,374,0x8159,1,0xc59a0000,2325,0x85a3
,1,0xe895b700,3892,0xe550,1,0xe895be00,3072,0xe551,1,0xe8969000,6155,0xe552
,1,0xe8978900,0,0xe553,1,0xe896ba00,0,0xe554,1,0xe69d8600,0,0x9e57
,1,0xc8b70000,6654,0x8669,1,0xe8978f00,0,0xe555,1,0xe7b98100,779,0x94c9
,1,0xe7be9400,9127,0xe3b4,1,0xe896b900,0,0xe556,1,0xe8a8b100,0,0x8b96
,1,0xe8979500,0,0xe558,1,0xe8989300,4241,0xe55e,1,0xe898a200,0,0xe563
,1,0xefbcb500,0,0x8274,1,0xe6a88c00,0,0x9eec,1,0xe6ad9f00,0,0x9f62
,1,0xe8989a00,4229,0xe564,1,0xe797bc00,0,0xe180,1,0xe8a99100,3120,0x91c0
,1,0xe898b000,4978,0xe565,1,0xe898bf00,10511,0xe566,1,0xe5928400,0,0x99f2
,1,0xefbd9500,0,0x8295,1,0xe8998d00,0,0xe567,1,0xe4b99500,3912,0xe568
,1,0xe8999400,4239,0xe569,1,0xe6b8a500,7631,0x88ad,1,0xe6bdb800,0,0xe045
,1,0xe8999f00,9574,0xe56a,1,0xe899a700,0,0xe56b,1,0xe89a9300,10154,0xe56d
,1,0xe4b88000,3327,0x88ea,1,0xe4bd9300,693,0x91cc,1,0xe6829400,5759,0x89f7
,1,0xe89aa300,0,0xe56e,1,0xe68cba00,0,0x92f0,1,0xe89aa900,7097,0xe56f
,1,0xe290a300,11499,0x849d,1,0xe888ac00,0,0x94ca,1,0xe89aaa00,2976,0xe570
,1,0xe89a8b00,10291,0xe571,1,0xe89ab600,0,0xe573,1,0xe89aaf00,0,0xe574
,1,0xe89ab000,0,0xe577,1,0xe5ada300,0,0x8b47,1,0xe692ad00,1206,0x9464
,1,0xe89b8900,0,0xe578,1,0xe8a0a300,4344,0xe579,1,0xe88e9f00,0,0xe4af
,1,0xe380b300,158,0x81b1,1,0xe89aab00,0,0xe57a,1,0xe89bac00,0,0xe57e
,1,0xe89c8000,0,0xe586,1,0xe581b800,1179,0x98f7,1,0xe6938d00,0,0x9180
,1,0xe698a000,0,0x8966,1,0xe5bdbc00,1508,0x94de,1,0xe89bbb00,7386,0xe588
,1,0xe3819300,656,0x82b1,1,0xe89bb900,0,0xe58c,1,0xe89c9a00,0,0xe594
,1,0xe9b08d00,7548,0x8a82,1,0xe9b5a000,233,0x8d94,1,0xe89da000,0,0xe595
,1,0xe89d9f00,5447,0xe596,1,0xe89d8c00,3724,0xe598,1,0xe7839d00,0,0xe07e
,1,0xe6a8b900,1897,0x8ef7,1,0xe7ba8e00,0,0xe39a,1,0xc5a10000,2376,0x85b0
,1,0xe89d9700,2814,0xe59b,1,0xe89da800,0,0xe59c,1,0xe5888b00,0,0x9984
,1,0xe58d9e00,0,0x99c4,1,0xe89dae00,0,0xe59d,1,0xe89da300,0,0xe5a0
,1,0xe6a99900,0,0x9ef2,1,0x76000000,0,0x76,1,0xe793b600,0,0x9572
,1,0xe89daa00,9901,0xe5a1,1,0xe8aa9e00,3121,0x8cea,1,0xe89ea200,3171,0xe5a3
,1,0xe89e9f00,0,0xe5a4,1,0xe89e8200,0,0xe5a5,1,0xe4b8ad00,6135,0x9286
,1,0xe59db700,0,0x8867,1,0xe89eaf00,6875,0xe5a6,1,0xe6b49f00,0,0x9fa4
,1,0xe6b9b200,2020,0x9fd1,1,0xe79ebc00,0,0xe1d9,1,0xe89f8b00,0,0xe5a7
,1,0xe8b5a400,0,0x90d4,1,0xe89f9000,7969,0xe5aa,1,0xe4b98d00,3878,0x93e1
,1,0xe4bea000,0,0x8ba0,1,0xe683a100,0,0x9ca6,1,0xe688b400,0,0x91d5
,1,0xe99b9600,0,0xe5ab,1,0xe79f9c00,0,0xe1e0,1,0xe291b000,0,0x8750
,1,0xe89eab00,8669,0xe5ac,1,0xe89eb300,2870,0xe5ae,1,0xc7820000,3342,0x8643
,1,0xe89fb200,0,0xe5b3,1,0xe6848100,1127,0x8f44,1,0xe89fa000,0,0xe5b4
,1,0xe68ea700,223,0x8d54,1,0xe693ba00,0,0x9db2,1,0xe8858600,0,0xe441
,1,0xe88a9900,0,0x9587,1,0xe8a08f00,0,0xe5b5,1,0xe89fbe00,9249,0xe5b7
,1,0xe89f9200,4502,0xe5bb,1,0xe8a09600,3022,0xe5bd,1,0xe8a0a200,3478,0xe5bf
,1,0xe5af9000,0,0x9b87,1,0xe8a0a100,0,0xe5c0,1,0xe5b9b600,6521,0x9bf3
,1,0xe2988300,0,0x83e9,1,0xe29d9600,687,0x879d,1,0xe7b5a800,0,0xe34f
,1,0xe387b300,0,0x83f7,1,0xe8a0bb00,3211,0xe5c5,1,0xe8a18400,3501,0xe5c6
,1,0xe583a500,4793,0x9946,1,0xe588b800,0,0x8c94,1,0x5f000000,680,0x5f
,1,0xe8a18200,3732,0xe5c7,1,0xe8a19e00,0,0xe5ca,1,0xe3838000,169,0x835f
,1,0xe8a1a200,3678,0xe5cb,1,0xe8a1ab00,8643,0xe5cc,1,0xe8a28100,0,0xe5cd
,1,0xe8a1be00,0,0xe5ce,1,0xe9bca000,0,0x916c,1,0xe8a2b500,0,0xe5d2
,1,0xe8a1b200,0,0xe5d3,1,0xe6a59300,7263,0x9596,1,0xe8a28d00,0,0xe5da
,1,0xe8a39400,0,0xe5e1,1,0xe8a39800,0,0xe5e2,1,0xe8a39900,2828,0xe5e3
,1,0xcc800000,2864,0x867b,1,0xe8a39d00,3190,0xe5e4,1,0xe58f8b00,0,0x9746
,1,0xe8a58300,0,0xe5ef,1,0xe599b100,7691,0x8857,1,0xe8a4a500,6777,0xe5f1
,1,0xe8a4aa00,0,0xe5f2,1,0xe6b5ac00,0,0x8a5c,1,0xe8a4ab00,4273,0xe5f3
,1,0xe8a58100,9993,0xe5f4,1,0xe8a58400,0,0xe5f5,1,0xe8a4b600,3216,0xe5f7
,1,0xe99bbb00,0,0x9364,1,0xe8a5af00,8231,0xe644,1,0xe59fa400,0,0x8870
,1,0xe5a4b700,0,0x88ce,1,0xe6b68c00,0,0x974f,1,0xe79b9600,0,0xe1b3
,1,0xe8a5b400,0,0xe645,1,0xe8a5b700,0,0xe646,1,0xe8a5be00,4826,0xe647
,1,0xe8a69300,0,0xe64b,1,0xe8a69800,0,0xe64c,1,0xe8a6ac00,5727,0xe650
,1,0xe6858e00,502,0x9054,1,0xe8a6ba00,4924,0xe653,1,0xe68fb400,5920,0x8987
,1,0xe8a6bd00,3100,0xe654,1,0xe2939d00,2443,0x86cc,1,0xe88ba600,433,0x8bea
,1,0xe8bd8400,8798,0x8a8d,1,0xe8a6bf00,9815,0xe655,1,0xe8a79a00,4817,0xe657
,1,0x48000000,0,0x48,1,0xe8a79c00,0,0xe658,1,0xe8a79d00,3782,0xe659
,1,0xe8a88300,3812,0xe65d,1,0xe8a89600,0,0xe65e,1,0xe7ac8f00,2528,0xe294
,1,0xe8a98800,8817,0xe669,1,0xe383ad00,762,0x838d,1,0xe8a9bc00,3385,0xe66a
,1,0xe9ad9400,0,0x9682,1,0xe8a9ac00,0,0xe66c,1,0xe9b7ba00,745,0x8deb
,1,0xe6968700,1229,0x95b6,1,0xe5bba300,1546,0x9c41,1,0xe8a9a200,6622,0xe66d
,1,0xe7b28200,0,0x8c48,1,0xe7b79500,0,0xe38e,1,0xe8aaa100,9092,0xe672
,1,0xe981a900,2800,0x934b,1,0xe8aa9a00,4925,0xe676,1,0xe5859200,3562,0x995a
,1,0xe8aaa300,11166,0xe677,1,0xe58fb800,1264,0x8e69,1,0xe6a18d00,0,0x9e7c
,1,0xe6a6a000,0,0x9ed3,1,0xe8abab00,6790,0xe67c,1,0xe8abb300,0,0xe67d
,1,0xe8a29200,3062,0xe5d6,1,0xe9879c00,4753,0x8a98,1,0xe98caf00,0,0x8df6
,1,0xe58b8500,828,0x92ba,1,0xe8aba700,2991,0xe67e,1,0xe8aba400,0,0xe680
,1,0xe8aba200,6711,0xe684,1,0xe78c8a00,0,0xe0c7,1,0xe8ab9b00,7467,0xe687
,1,0xe8ac8700,7580,0xe689,1,0xe8aba100,7364,0xe68b,1,0xc5af0000,254,0x85d3
,1,0xe8b2ab00,0,0x8ad1,1,0xe8acb300,7121,0xe690,1,0xefbc8900,451,0x816a
,1,0xe59b9e00,9939,0x89f1,1,0xe680a800,678,0x8985,1,0xe99eab00,3960,0xe691
,1,0xe6b79900,0,0x9fc8,1,0xe6bcac00,0,0x92d0,1,0xe8acab00,3146,0xe693
,1,0xe8ad8100,3920,0xe696,1,0x31000000,2958,0x31,1,0xe8ad8c00,3314,0xe697
,1,0xe8ad8f00,10382,0xe698,1,0xe5a19100,0,0x9159,1,0xe5a6a400,0,0x888d
,1,0xe68bae00,0,0x9d68,1,0xe28a8400,0,0x81c0,1,0xe8ad8e00,5445,0xe699
,1,0xe7a7a900,2469,0x9281,1,0xe8ad9600,0,0xe69b,1,0xe8adab00,0,0xe69e
,1,0xe9a39b00,8571,0x94f2,1,0xe8adbd00,5082,0xe6a3,1,0xe8ae9a00,5831,0xe6ab
,1,0xe8b0ba00,5883,0xe6ac,1,0xe8b18c00,0,0xe6b0,1,0xe5b6bd00,0,0x9bd4
,1,0xe8b19500,0,0xe6b3,1,0xe8b1a200,0,0xe6b4,1,0xe29fb000,8840,0x86ab
,1,0xe897b900,0,0xe55c,1,0xe8b28500,5090,0xe6ba,1,0xc7900000,0,0x84ef
,1,0xe9b3b400,940,0x96c2,1,0xe8b28a00,3210,0xe6bb,1,0xe5b79d00,6470,0x90ec
,1,0xe69ca700,0,0x9e4f,1,0xe8b2aa00,6639,0xe6c3,1,0xe3808700,2603,0x815a
,1,0xe7b8a200,0,0xe377,1,0xd1850000,568,0x8487,1,0xe8a2bf00,5122,0xe5dd
,1,0xe8b38800,0,0xe6c9,1,0xe8b38100,3384,0xe6ca,1,0xe58bb200,0,0x8c4d
,1,0xe8b3a300,3853,0xe6cc,1,0xe8b48500,6656,0xe6d2,1,0xe6a7ad00,0,0x9ee3
,1,0xe78cb700,0,0x9751,1,0xe8b48700,0,0xe6d4,1,0xe8a39f00,0,0x8dbe
,1,0xe8b48f00,3196,0xe6d5,1,0xe8b48d00,3159,0xe6d6,1,0xe8b49000,5730,0xe6d7
,1,0xe8b49300,0,0xe6d9,1,0xefbcb600,0,0x8275,1,0xe8b49400,0,0xe6db
,1,0xe78d9700,3930,0xe0d3,1,0xe6b2b300,7512,0x89cd,1,0x1a000000,0,0x1a
,1,0xe8a99200,3115,0xe667,1,0xe8b49600,0,0xe6dc,1,0xe8b68100,4811,0xe6e1
,1,0xe8b78f00,6314,0xe6e6,1,0xefbd9600,0,0x8296,1,0xe4bcb400,0,0x94ba
,1,0xe681b500,290,0x8c62,1,0xe6b39300,1944,0x9f97,1,0xe6b8a600,0,0x8951
,1,0xe8b78c00,3226,0xe6e9,1,0xe882ba00,0,0x9478,1,0xe8b78b00,0,0xe6eb
,1,0xe999a200,0,0x8940,1,0xe4b88100,0,0x929a,1,0xe8b7ab00,3245,0xe6ed
,1,0xe5a29e00,6209,0x887c,1,0xe8b7a300,5015,0xe6ef,1,0xe8b7bc00,6323,0xe6f0
,1,0xe8b89d00,0,0xe6f4,1,0xe8839a00,0,0xe3f3,1,0xe8b89e00,0,0xe6f5
,1,0xe8b98900,2975,0xe741,1,0xe8b99900,0,0xe745,1,0xe9a4a800,0,0x8ad9
,1,0xe8b9a400,0,0xe746,1,0xe5a89100,1386,0x9b4f,1,0xe5ada400,0,0x8cc7
,1,0xe692ae00,1207,0x8e42,1,0xe8b9a000,5601,0xe747,1,0xe8b9a300,7930,0xe749
,1,0xe88ea000,0,0xe4ba,1,0xe380b400,0,0x81b2,1,0xe8b9bc00,7968,0xe74d
,1,0xe8ba8100,7805,0xe74e,1,0xe8ba8700,10782,0xe74f,1,0xe8ba8500,5905,0xe750
,1,0xe8ba8400,3792,0xe751,1,0xe8ba8b00,6478,0xe752,1,0xe8ba8a00,3722,0xe753
,1,0xe7af8900,4590,0x927a,1,0xe3819400,714,0x82b2,1,0xe8ba9300,0,0xe754
,1,0xe38bba00,7300,0x86ed,1,0xe8ba9100,0,0xe755,1,0xe5829900,728,0x94f5
,1,0xe8ba9400,5174,0xe756,1,0xe58cbf00,1518,0x93bd,1,0xe8baac00,0,0xe75a
,1,0x3000000,1610,0x3,1,0xe6a8ba00,0,0x8a92,1,0xe7ba8f00,0,0x935a
,1,0xe8bab000,3265,0xe75b,1,0xd18c0000,4905,0x848e,1,0xe8bab100,3052,0xe75d
,1,0xe8bb8800,3615,0xe760,1,0xe8bbbc00,0,0xe764,1,0xe592b200,0,0x8de7
,1,0xe8bbbb00,6677,0xe765,1,0xe8bbab00,0,0xe766,1,0xe8bbbe00,3855,0xe767
,1,0xe793b700,0,0xe149,1,0xe8a58c00,0,0xe5f9,1,0xe8bc9900,3127,0xe76c
,1,0xe8bc9300,0,0xe76d,1,0xe8bc9c00,3281,0xe76e,1,0xe8bc9f00,3285,0xe76f
,1,0xe8bc9b00,5207,0xe770,1,0xe8bcbb00,0,0xe774,1,0xe6af8d00,0,0x95ea
,1,0xe8bd8200,4242,0xe777,1,0xe799aa00,0,0xe19b,1,0xe79ebd00,0,0xe1da
,1,0xe8bda200,10078,0xe780,1,0xe8beb700,5050,0xe788,1,0xe8bf9a00,10178,0xe789
,1,0xe4b98e00,6682,0x8cc1,1,0xc2a00000,63,0x8541,1,0xe5a3ab00,0,0x8e6d
,1,0xe9828700,0,0xe78e,1,0xe9808500,7045,0xe790,1,0xe9809500,6153,0xe794
,1,0xe291b100,7840,0x8751,1,0xe8b68500,0,0x92b4,1,0xc5bd0000,0,0x85a8
,1,0xe9808d00,0,0xe796,1,0xe9809e00,0,0xe797,1,0xe9809600,4721,0xe798
,1,0xe6899500,0,0x95a5,1,0xe68ea800,0,0x9084,1,0xe980b600,3811,0xe79b
,1,0xe7a59000,0,0x9753,1,0xe9818f00,7963,0xe79f,1,0xe7afb600,0,0xe2cc
,1,0xe9819000,0,0xe7a0,1,0xe9819200,0,0xe7a2,1,0xe9818900,0,0xe7a4
,1,0xe9b0bb00,0,0x8956,1,0xe980be00,3982,0xe7a5,1,0xe9819800,6012,0xe7a7
,1,0xe699ae00,5189,0x9581,1,0xe7ab8300,0,0x8a96,1,0xe8908d00,0,0xe4ca
,1,0xe382a100,0,0x8340,1,0xe387b400,9055,0x83f8,1,0xe9819e00,4525,0xe7a8
,1,0xe981a800,4437,0xe7a9,1,0xe583a600,4808,0x87c6,1,0xe588b900,0,0x998b
,1,0xe5ba9700,1529,0x9358,1,0xc4810000,0,0x859a,1,0xe6a4b400,4077,0x93cc
,1,0xe3838100,1905,0x8360,1,0xe981b600,0,0xe7ab,1,0xe9809d00,0,0x90c0
,1,0xe99aa800,3266,0xe7ac,1,0xe5848600,0,0x87c8,1,0xe9828200,3652,0xe7ae
,1,0xe9828100,0,0xe7b0,1,0xe593bf00,3863,0x8842,1,0xe6a59400,0,0x9eb6
,1,0xe9828000,0,0xe7b1,1,0xe78fb100,0,0xe0fc,1,0xe8a18600,0,0x8f4f
,1,0xe9869000,0,0x8ced,1,0xe9828900,0,0xe7b3,1,0xe8b0bf00,3187,0xe6ae
,1,0xe58f8c00,0,0x916f,1,0xe9828f00,4724,0xe7b4,1,0xe599b200,7702,0x8858
,1,0xe982a800,0,0xe7b5,1,0xe982af00,0,0xe7b6,1,0xe795a400,0,0xe15e
,1,0xe982b100,3647,0xe7b7,1,0xe8ac8c00,0,0xe688,1,0xe982b500,3469,0xe7b8
,1,0xe9849900,4997,0xe7bf,1,0xe984b200,0,0xe7c0,1,0xe4ba9b00,8244,0x8db1
,1,0xe4bfae00,0,0x8f43,1,0xe5a4b800,0,0x9af0,1,0xe9859600,6935,0xe7c3
,1,0xe79b9700,0,0x9390,1,0xe880a100,0,0xe3d2,1,0xe885b400,0,0xe44a
,1,0xe985a300,0,0xe7c5,1,0xe99c9c00,0,0x919a,1,0xe985a500,3349,0xe7c6
,1,0xe5a08500,0,0x8c98,1,0xc2a70000,1139,0x8198,1,0xe985a900,0,0xe7c7
,1,0xe5afbe00,3779,0x91ce,1,0xe9868b00,4745,0xe7ca,1,0xe2939e00,2125,0x86cd
,1,0xe88ba700,2179,0x9297,1,0xe8bd8500,0,0xe776,1,0xe9868900,0,0xe7cb
,1,0xcc9c0000,3388,0x868a,1,0xe9868200,0,0xe7cc,1,0xe5ab8b00,1401,0x9b5d
,1,0xe5b09e00,0,0x987b,1,0xe986aa00,0,0xe7d0,1,0xe986b500,3842,0xe7d1
,1,0xe986b400,7549,0xe7d2,1,0xe986ba00,8819,0xe7d3,1,0xe383ae00,4630,0x838e
,1,0xe9878900,5078,0xe7d6,1,0xe9879600,0,0xe7d9,1,0xe9879b00,0,0xe7dc
,1,0xe987bc00,0,0xe7dd,1,0xe6968800,0,0x9b7c,1,0xe9889e00,3848,0xe7e0
,1,0xe987bf00,5315,0xe7e1,1,0xe7b28300,0,0xe2e1,1,0xe8978d00,4469,0x9795
,1,0xe9889400,5466,0xe7e2,1,0xe9899700,0,0xe7e7,1,0xe9898500,4771,0xe7e8
,1,0xe9898900,3565,0xe7e9,1,0xe989a400,5428,0xe7ea,1,0xe5bc8400,0,0x984d
,1,0xe6a18e00,0,0x9e7e,1,0xc4880000,0,0x85d7,1,0xe9898800,0,0xe7eb
,1,0xe98a9500,9105,0xe7ec,1,0xe9828a00,0,0xe7b2,1,0xe8a7a600,337,0x9047
,1,0xe8acb900,0,0x8bde,1,0xe9898b00,0,0xe7ee,1,0xe98a9c00,0,0xe7f0
,1,0xe98a9600,5998,0xe7f1,1,0xe6a78100,9251,0x9ec7,1,0xe98a9300,0,0xe7f2
,1,0xe7919e00,6914,0x9090,1,0xe796b100,0,0xe176,1,0xe98a9b00,0,0xe7f3
,1,0xe9899a00,3948,0xe7f4,1,0xe8b2ac00,0,0x90d3,1,0xe8b7bf00,0,0xe6f3
,1,0xefbc8a00,0,0x8196,1,0xe98b8f00,5664,0xe7f5,1,0xe680a900,5733,0x9c82
,1,0xe98ab900,3532,0xe7f6,1,0xe98c8f00,4006,0xe7f9,1,0xe79ca400,0,0xe1c0
,1,0xe98d8400,5676,0xe7fb,1,0xe98cae00,9918,0xe7fc,1,0xe8b89f00,0,0xe6f7
,1,0xe99da900,0,0x8a76,1,0xe98ca200,10383,0xe841,1,0xe5a19200,0,0x9acb
,1,0xe5a6a500,0,0x91c3,1,0xe68baf00,0,0x9d6d,1,0xe28a8500,0,0x81c1
,1,0xe7a29700,0,0x9871,1,0xe294ab00,0,0x84b2,1,0xe88cb400,8012,0xe4a0
,1,0xe98c9a00,0,0xe842,1,0xe9a39c00,6811,0xe3ca,1,0xe98ca300,0,0xe843
,1,0xe98cb500,0,0xe845,1,0xc2ae0000,0,0x8549,1,0xe98dbc00,0,0xe849
,1,0xe98dae00,6453,0xe84a,1,0xe2958b00,2472,0x84b4,1,0xe7ad9d00,8735,0xe2b6
,1,0xe29fb100,0,0x86ac,1,0xe897ba00,0,0xe561,1,0xe98d9600,0,0xe84b
,1,0xe98e9400,4803,0xe84f,1,0xe580ad00,0,0x9860,1,0xe98eb900,0,0xe850
,1,0xe5b79e00,0,0x8f42,1,0xe5bcb100,746,0x8ee3,1,0xe98fa800,0,0xe853
,1,0xe3808800,0,0x8171,1,0xe7b8a300,2680,0xe370,1,0xe98fa500,6963,0xe854
,1,0xe98fa400,9687,0xe85a,1,0xe9909a00,3513,0xe85b,1,0xe586a000,0,0x8aa5
,1,0xe58bb300,0,0x99ac,1,0xe5bd9100,3854,0x9c5f,1,0xe6a29b00,1840,0x9e8b
,1,0xe787a500,6725,0x9187,1,0xe7b98300,0,0xe37d,1,0xe89e8d00,8426,0x975a
,1,0xe9908300,4886,0xe85e,1,0xe8a8b300,0,0x96f3,1,0xe9908700,3579,0xe85f
,1,0xe990b600,11142,0xe861,1,0xe990ab00,5497,0xe862,1,0xefbcb700,0,0x8276
,1,0xe990b500,0,0xe863,1,0xc48f0000,1668,0x85cd,1,0xe9918100,7516,0xe866
,1,0xe797be00,0,0xe17d,1,0xe9919200,9861,0xe867,1,0xe991a200,3368,0xe86b
,1,0xe988a900,0,0xe86e,1,0xe5928600,0,0x99f4,1,0xefbd9700,0,0x8297
,1,0xe991b000,0,0xe86f,1,0xe991b700,6821,0xe871,1,0xe991bd00,3963,0xe872
,1,0xe9919a00,4262,0xe873,1,0xe6bdba00,0,0xe044,1,0xe991bc00,0,0xe874
,1,0xe991be00,0,0xe875,1,0xe999a300,9563,0x9077,1,0xe8bebf00,5198,0x9248
,1,0xe4bd9500,5680,0x89bd,1,0xe5a29f00,1314,0x9ad0,1,0xe9928100,5968,0xe876
,1,0xe991bf00,6324,0xe877,1,0xe79e9100,0,0xe1d2,1,0xe8839b00,4322,0xe3ee
,1,0xe7a8b700,2498,0xe26c,1,0xe996a000,0,0xe880,1,0xe996a800,0,0xe881
,1,0xe996a700,0,0xe882,1,0xe996be00,0,0xe887,1,0xe6888900,0,0x9cf8
,1,0xe68d9c00,1173,0x917b,1,0xe5b2b800,0,0x8add,1,0xe6bfb600,0,0xe889
,1,0xe7a99700,0,0xe26e,1,0xe7aeaa00,0,0x925c,1,0xe380b500,2610,0x81b3
,1,0xe9978300,3374,0xe88a,1,0xe9978d00,6187,0xe88b,1,0xe9979400,10778,0xe88e
,1,0xe5ae8500,0,0x91ee,1,0xe9979600,0,0xe88f,1,0xe5b8ab00,3823,0x8e74
,1,0xe69db500,6157,0x8b6e,1,0xe88f8100,8080,0xe4c5,1,0xe3819500,2616,0x82b3
,1,0xe7b9b000,2693,0x8c4a,1,0xe89eba00,0,0x9786,1,0xe997a100,0,0xe891
,1,0xccaa0000,946,0x869a,1,0xe587ad00,5175,0x997e,1,0xe6998200,0,0x8e9e
,1,0xe69e9500,1287,0x968d,1,0xe7839f00,3873,0xe07c,1,0xe788b200,3903,0xe0a8
,1,0xe7ba9000,9023,0xe396,1,0xe997a200,5283,0xe893,1,0xe998a100,0,0xe894
,1,0xe9998c00,0,0xe899,1,0xe9998f00,0,0xe89a,1,0xe58da000,0,0x90e8
,1,0xe592b300,0,0x8a50,1,0xe6a48800,0,0x9e98,1,0xe7899200,3910,0x92ab
,1,0xe9999c00,6824,0xe89d,1,0xe793b800,0,0xe148,1,0xe9999e00,3277,0xe89e
,1,0xe8aaa000,9922,0x90bd,1,0xe9999d00,0,0xe89f,1,0xe9999f00,0,0xe8a0
,1,0xe99a8d00,0,0xe8a4,1,0xe4b8af00,6214,0x87a3,1,0xe99a9800,0,0xe8a5
,1,0xe6af8e00,7432,0x9688,1,0x6b000000,99,0x6b,1,0xe99aaa00,9573,0xe8a8
,1,0xe99aa700,0,0xe8a9,1,0xe99b8e00,4929,0xe8b1,1,0xe8b5a600,0,0x8ecd
,1,0xe99ab000,0,0xe8ac,1,0xe4b98f00,6716,0x9652,1,0xe99b8b00,4015,0xe8b2
,1,0xe5a3ac00,391,0x9070,1,0xe6b58100,0,0x97ac,1,0xe79a8b00,0,0xe1a6
,1,0xe99b8900,7742,0xe8b3,1,0xe291b200,4327,0x8752,1,0xe99b8d00,8067,0xe8b4
,1,0xe8a58d00,0,0xe8b5,1,0xe99bb900,6652,0xe8b9,1,0xe4bf8200,0,0x8c57
,1,0xe5a48c00,1335,0x8887,1,0xe99c9300,3764,0xe8bd,1,0xe68ea900,0,0x8986
,1,0xe99c9100,0,0xe8bf,1,0xe99c8f00,0,0xe8c0,1,0xe99c9600,0,0xe8c1
,1,0xe7afb700,0,0xe2c9,1,0xe99c9900,6687,0xe8c2,1,0xe9a69600,0,0x8ef1
,1,0xe99caa00,10645,0xe8c4,1,0xe99cb000,0,0xe8c5,1,0xe5af9200,1435,0x8aa6
,1,0xe6949c00,0,0x9db6,1,0xe699af00,183,0x8c69,1,0xe2988500,2162,0x819a
,1,0xe8908e00,2203,0x88de,1,0xe382a200,8258,0x8341,1,0xe387b500,0,0x83f9
,1,0xe99cbd00,5203,0xe8c7,1,0xe99cbe00,10849,0xe8c8,1,0xe583a700,3511,0x916d
,1,0xe588ba00,0,0x8e68,1,0xc2bc0000,0,0x8552,1,0xc9940000,78,0x8657
,1,0xe99d8400,3882,0xe8c9,1,0xe3838200,7328,0x8361,1,0xe89b9400,0,0xe57b
,1,0xe8a0a700,3023,0xe5c4,1,0xe99da000,3663,0xe8cf,1,0xe99da400,3543,0xe8d0
,1,0xe99da600,9177,0xe8d1,1,0xe58ead00,0,0x897d,1,0x54000000,469,0x54
,1,0xe6a59500,1870,0x91c8,1,0xe99da800,0,0xe8d2,1,0xe99e8500,8223,0xe8d7
,1,0xe99e8100,0,0xe8d9,1,0xe3939b00,85,0x87d9,1,0xe98ba400,769,0x8f9b
,1,0xe99e8600,0,0xe8db,1,0xe58f8d00,0,0x94bd,1,0xe99ea800,0,0xe8e0
,1,0xe99ea600,7701,0xe8e1,1,0xe99eb300,0,0xe8e3,1,0xe6b09b00,0,0x9f84
,1,0xe6b5ae00,823,0x9582,1,0xe79ab800,0,0xe1ae,1,0xe99eb400,4959,0xe8e4
,1,0xe99f8800,0,0xe8e7,1,0xe8b6b300,0,0x91ab,1,0xe5958000,0,0x9a50
,1,0xe4ba9c00,8248,0x889f,1,0xe4bfaf00,5724,0x98eb,1,0xe99f8b00,3757,0xe8e8
,1,0xe6b68e00,0,0x9fb5,1,0xc49d0000,0,0x85de,1,0xe99f9c00,0,0xe8e9
,1,0xe9bd8f00,5970,0xe8eb,1,0xe9978a00,0,0xe888,1,0xe8bca600,0,0xe772
,1,0xe99fb600,3350,0xe8ee,1,0xe5a08600,0,0x91cd,1,0xe9a0a100,0,0xe8f4
,1,0xce920000,0,0x83a0,1,0xe6bc8100,272,0x8b99,1,0xe9a0bd00,0,0xe8f6
,1,0xe2939f00,2438,0x86ce,1,0xe9a1ab00,8720,0xe8fa,1,0xe8bd8600,0,0xe77b
,1,0xe9a1af00,9498,0xe8fb,1,0xe9a1b000,0,0xe8fc,1,0xe9a1b400,0,0xe941
,1,0xe5ab8c00,0,0x8c99,1,0xe6909600,0,0x9d8a,1,0xe9a2b100,0,0xe945
,1,0xe69abc00,0,0x9dfc,1,0xe7ac9100,8729,0x8fce,1,0xe8919b00,2217,0x8a8b
,1,0xe383af00,765,0x838f,1,0xe9a2b600,5079,0xe946,1,0xe9a4a100,0,0xe951
,1,0x3d000000,0,0x3d,1,0xe9a49d00,0,0xe952,1,0xe6968900,6042,0x90c4
,1,0xe69b9c00,0,0x976a,1,0xe780a600,0,0x9295,1,0xe9a49e00,0,0xe953
,1,0xe9a4a400,4641,0xe954,1,0xe9a4a000,9093,0xe955,1,0xe9a4be00,0,0xe959
,1,0xe5808100,0,0x87b9,1,0xe5859400,11058,0x995c,1,0xe9a58200,9474,0xe95a
,1,0xe58fba00,0,0x99db,1,0xe9a58900,0,0xe95b,1,0xe9a58500,3679,0xe95c
,1,0xe78bac00,0,0x93c6,1,0xe790bf00,0,0xe0eb,1,0xe9a6bc00,0,0xe968
,1,0xe8a7a700,0,0xe65a,1,0xe9a79f00,0,0xe969,1,0xe58b8700,0,0x9745
,1,0xe9a79b00,0,0xe96a,1,0xe9a79d00,0,0xe96b,1,0xe9a79800,9257,0xe96c
,1,0xe9a88100,0,0xe974,1,0xe6b1a800,1936,0x9f91,1,0xe796b200,0,0x94e6
,1,0xe9a7a200,0,0xe977,1,0xe8ad9a00,10017,0xe69d,1,0xe8b2ad00,0,0xe6c2
,1,0xe9a8ab00,0,0xe979,1,0xefbc8b00,0,0x817b,1,0xe59ba000,9983,0x88f6
,1,0xe680aa00,0,0x89f6,1,0xe6b28800,0,0x92be,1,0xe2849300,1986,0x827e
,1,0xe289a600,2061,0x8185,1,0xe881af00,0,0x97fc,1,0xe8b38d00,0,0xe6da
,1,0xe9a98500,4833,0xe97b,1,0xe9a98300,10263,0xe97e,1,0xe4bc8900,1080,0x98c2
,1,0xe6818a00,0,0x9c90,1,0xe9a98d00,8290,0xe982,1,0xe9a9a200,0,0xe986
,1,0xe28a8600,0,0x81ba,1,0xc4a40000,0,0x85d9,1,0xe294ac00,4380,0x84a6
,1,0xe88cb500,0,0xe49f,1,0x26000000,0,0x26,1,0xe9aabc00,0,0xe98e
,1,0xe9a8b000,0,0x93ab,1,0xe5a78600,0,0x9b47,1,0xe5ac9900,6305,0x889c
,1,0xce990000,1198,0x83a7,1,0xe9ab9400,0,0xe993,1,0xe7a88b00,2473,0x92f6
,1,0xe9ab9e00,4578,0xe994,1,0xe29fb200,2233,0x86ad,1,0xe897bb00,0,0x9194
,1,0xe9ab9f00,3425,0xe995,1,0xe9aba200,0,0xe996,1,0xe580ae00,535,0x87be
,1,0xe6928300,1199,0x8c82,1,0xe9aba300,3810,0xe997,1,0xe9aba600,5132,0xe998
,1,0xe9ac8600,6643,0xe9a0,1,0xe3808900,0,0x8172,1,0xe9aca900,0,0xe9a9
,1,0xe7bdb700,2738,0x94eb,1,0xe982b800,0,0x9340,1,0xe9acaa00,5323,0xe9aa
,1,0xe9acae00,0,0xe9ab,1,0xe9acaf00,0,0xe9ac,1,0xe69d8900,0,0x9099
,1,0xe9acb200,6657,0xe9ad,1,0xe787a600,6726,0x8e57,1,0xe9ad8400,7372,0xe9ae
,1,0xe9ad8300,4500,0xe9af,1,0xe8a3a100,0,0x97a1,1,0xe8a8b400,0,0x9169
,1,0xe98dbe00,0,0x8fdf,1,0xe9ad8f00,8275,0xe9b0,1,0xe9ad8d00,6564,0xe9b1
,1,0xefbcb800,0,0x8277,1,0xe7888600,3900,0x949a,1,0xe6ada200,0,0x8e7e
,1,0xe9ad8e00,0,0xe9b2,1,0xe797bf00,9250,0xe17e,1,0xe8a99400,3125,0x8fd9
,1,0xe9ad9100,4046,0xe9b3,1,0xe8b3ba00,0,0xe6cf,1,0xe9ae8300,0,0xe9b7
,1,0xefbd9800,1029,0x8298,1,0xe59cad00,848,0x8c5c,1,0xe681b700,3933,0x9c8e
,1,0xe6b39500,0,0x9640,1,0xe285a000,1998,0x8754,1,0xf000000,7715,0xf
,1,0xe9ae9600,0,0xe9b9,1,0xe9ae9700,0,0xe9ba,1,0xe999a400,9564,0x8f9c
,1,0xe4b88300,0,0x8eb5,1,0xe4bd9600,5681,0x87ae,1,0xe6829700,0,0x9ca2
,1,0xe9ae9f00,0,0xe9bb,1,0xe68cbd00,7014,0x94d2,1,0xe9aea000,0,0xe9bc
,1,0xe9af8000,9234,0xe9bf,1,0xe9afa300,7926,0xe9c6,1,0xe8ba8d00,5020,0x96f4
,1,0xe9afa200,9665,0xe9c7,1,0xe9afa400,0,0xe9c8,1,0xe9afa100,0,0xe9ca
,1,0xe5a89300,675,0x8891,1,0xe5ada600,0,0x8a77,1,0xe692b000,0,0x90ef
,1,0xe9b0ba00,0,0xe9cb,1,0xc4ab0000,8652,0x859b,1,0xe88ea200,0,0xe4b0
,1,0xe7b3be00,0,0x8b8a,1,0xe9b08900,9750,0xe9d1,1,0xe9b08c00,0,0xe9d3
,1,0xe9afb000,0,0xe9ce,1,0xe9b08600,0,0xe9d4,1,0xe5b39900,3807,0x9bb3
,1,0xcea00000,0,0x83ae,1,0xe5bdbf00,6586,0x9c69,1,0xe7af8b00,0,0xe2b8
,1,0xe3819600,2615,0x82b4,1,0xe9b08800,0,0xe9d5,1,0xe89ebb00,0,0xe5b1
,1,0xe9b09000,0,0x986b,1,0xe9b08a00,4351,0xe9d7,1,0xe587ae00,0,0x87d7
,1,0xe6998300,701,0x8d57,1,0xe9b08400,0,0xe9d8,1,0xe9b0a100,0,0xe9dd
,1,0xe9b0b000,5909,0xe9de,1,0xe9b0b200,8377,0xe9e0,1,0xe9b19a00,0,0xe9e3
,1,0xe9b3a700,0,0xe9e8,1,0xe9b3ac00,4948,0xe9e9,1,0xe5888e00,0,0x9986
,1,0xe58da100,0,0x87ed,1,0xe9b3b000,0,0xe9ea,1,0xe9b48900,0,0xe9eb
,1,0xe9b48800,7322,0xe9ec,1,0xe6aeaf00,4141,0x9f71,1,0xe9b3ab00,0,0xe9ed
,1,0xe9b6af00,4815,0xe9f2,1,0xe98a9800,2856,0x96c1,1,0xe9b49500,0,0xe9f6
,1,0xe9b49200,0,0xe9f7,1,0xe9b4bf00,0,0xe9f9,1,0xe4b8b000,6254,0x87a4
,1,0xe9b4be00,5903,0xe9fa,1,0xe9b58800,3936,0xe9fc,1,0xe9b59d00,5339,0xea40
,1,0xd0810000,2021,0x8446,1,0xe79ebf00,0,0xe1d8,1,0xe9b59e00,3888,0xea41
,1,0xe8b5a700,0,0xe6dd,1,0xe99ab100,0,0xe8aa,1,0xe9b59900,5626,0xea45
,1,0xe9b68900,5707,0xea47,1,0xe9b6ab00,4746,0xea49,1,0xe9b5ba00,7421,0xea4b
,1,0xe9b6a400,4484,0xea4d,1,0xe9b6a900,4597,0xea4e,1,0xe291b300,2423,0x8753
,1,0xe9b78400,6253,0xea50,1,0xe99b9100,0,0x8e47,1,0xe9a0a400,0,0xe8f3
,1,0xe4bf8300,0,0x91a3,1,0xe9b78100,5787,0xea51,1,0xe9b6ba00,0,0xea54
,1,0xe5aeb300,375,0x8a51,1,0xe693bd00,0,0x9db4,1,0xe9b78600,0,0xea55
,1,0xe297a600,0,0x825e,1,0xe88faf00,0,0x89d8,1,0xe9b78f00,0,0xea56
,1,0xe9a69700,0,0xe963,1,0xe9abaa00,0,0x94af,1,0xe9b78200,0,0xea57
,1,0xe5af9300,0,0x8bf5,1,0xe6949d00,6013,0x9d90,1,0xe5b9b900,1525,0x8ab2
,1,0xe2988600,2161,0x8199,1,0xe9b79300,8784,0xea59,1,0xe382a300,8922,0x8342
,1,0xe387b600,0,0x83fa,1,0xe9b7b800,0,0xea5a,1,0xe9b7a600,10875,0xea5b
,1,0xe9b7af00,3739,0xea5d,1,0xe588bb00,906,0x8d8f,1,0xe9b7bd00,6525,0xea5e
,1,0xcea70000,0,0x83b4,1,0xe6a4b600,0,0x9ea2,1,0xe3838300,4462,0x8362
,1,0xe9b89a00,9072,0xea5f,1,0xe9809f00,9192,0x91ac,1,0xe985b200,5381,0xe7c9
,1,0xe9b89b00,5374,0xea60,1,0xe5899b00,0,0x8d84,1,0xe58eae00,0,0x99cf
,1,0xe6a08300,0,0x93c8,1,0xe9b89e00,0,0xea61,1,0xe78aa000,0,0x8b5d
,1,0xe9b9b900,0,0xea63,1,0xe9b9bd00,0,0xea64,1,0xe9869200,0,0x90c1
,1,0xe8abae00,9968,0x8e90,1,0xe990b800,0,0x91f6,1,0xe58f8e00,0,0x8efb
,1,0xe9ba8100,9395,0xea65,1,0xe599b400,7724,0x95ac,1,0xe9ba8800,4118,0xea66
,1,0xe9ba8b00,7446,0xea67,1,0xe795a600,0,0x8c6c,1,0xe79ab900,0,0xe1af
,1,0xe8ac8e00,0,0x93e4,1,0xc6930000,319,0x8648,1,0xe9ba8c00,10136,0xea68
,1,0xe9ba9200,5540,0xea69,1,0xe59a9400,5236,0x9a89,1,0xe9ba9500,0,0xea6a
,1,0xe9ba9100,0,0xea6b,1,0xe7968600,0,0xe164,1,0xe2889a00,11498,0x81e3
,1,0xe9baa500,7795,0xea6d,1,0xe7a5bf00,0,0xe252,1,0xe9baa900,5005,0xea6e
,1,0xe99c9e00,0,0x89e0,1,0xe9a1b100,0,0xe940,1,0xe9bab800,0,0xea6f
,1,0xe5a59a00,0,0x9af6,1,0xe9baaa00,0,0xea70,1,0xe6bc8200,6343,0x9559
,1,0xe9baad00,8473,0xea71,1,0xe293a000,0,0x86cf,1,0xe99da100,8750,0xea72
,1,0xe890bc00,0,0xe4d3,1,0xe9bb8c00,9215,0xea73,1,0xe9bb8e00,8895,0xea74
,1,0xe9bb8f00,0,0xea75,1,0xe9bb9000,3765,0xea76,1,0xe5b0a000,1451,0x9b96
,1,0xe5b5b300,0,0x9bcb,1,0xe2948000,0,0x849f,1,0xe88c8900,4403,0xe49d
,1,0xe7b1a500,8825,0xe2de,1,0xe383b000,299,0x8390,1,0xe9bb9400,0,0xea77
,1,0xe9bb9c00,0,0xea78,1,0xe9bb9d00,0,0xea7a,1,0xe584b500,142,0x87cc
,1,0xe9bba500,6449,0xea7c,1,0xe69b9d00,0,0x9498,1,0xe780a700,0,0x91eb
,1,0xe9bbaf00,0,0xea7e,1,0xc4b90000,2319,0x85b9,1,0xcb910000,0,0x8675
,1,0xe8a1b500,0,0xe5d0,1,0xe5808200,0,0x87ba,1,0xe5859500,11061,0x87cf
,1,0xe9bbb600,11167,0xea81,1,0xe9bbb700,0,0xea82,1,0xe6a19000,7162,0x8bcb
,1,0xe9bbb900,0,0xea83,1,0xe78bad00,0,0x8bb7,1,0xe9bbbb00,0,0xea84
,1,0xe9bbbc00,9697,0xea85,1,0xe9879f00,0,0xe7da,1,0xe98cb200,0,0x985e
,1,0xe58b8800,5178,0x87e0,1,0xe5909b00,0,0x8c4e,1,0xe9bbbd00,11121,0xea86
,1,0xe6a78300,0,0x9ecf,1,0xe6ac9600,0,0x9f53,1,0xe791a000,0,0x97da
,1,0xe6b6bc00,1979,0x97c1,1,0xe8a88800,0,0x8c76,1,0xe8ad9b00,0,0xe69c
,1,0xe8b2ae00,0,0xe6c7,1,0xe79ab700,0,0xea89,1,0xefbc8c00,13,0x8143
,1,0xe9bc9500,10014,0xea8a,1,0xe5a0b400,1299,0x8fea,1,0xe685be00,6877,0x977c
,1,0xe9bca100,7590,0xea8b,1,0xe289a700,2065,0x8186,1,0xe881b000,0,0xe3e0
,1,0xe8b38e00,0,0x9147,1,0xe9bcac00,9534,0xea8c,1,0xe99dab00,0,0xe8d4
,1,0xe4bc8a00,10833,0x88c9,1,0xe5a19400,860,0x9383,1,0xe9bcbe00,0,0xea8d
,1,0xe5abba00,1405,0x9b65,1,0xe28a8700,4294,0x81bb,1,0xe9bda300,0,0xea91
,1,0xe887a300,2133,0x9062,1,0xe88cb600,0,0x9283,1,0xe99e8b00,0,0xe8dc
,1,0xe9bda600,4601,0xea95,1,0xe9bda700,10583,0xea96,1,0xe9bdac00,5426,0xea97
,1,0xe68c9100,0,0x92a7,1,0xe9bdaa00,0,0xea98,1,0xe696b700,0,0x9dd0
,1,0xe9bdb200,0,0xea9a,1,0xe9be9500,0,0xea9c,1,0xe29fb300,2232,0x86ae
,1,0xe9bea000,0,0xea9e,1,0xe5a0af00,0,0xea9f,1,0xe9819900,0,0xeaa1
,1,0xe7869900,0,0xeaa4,1,0xe5b7a200,0,0xeaa6,1,0xe6979700,6067,0x8af8
,1,0xe69caa00,0,0x96a2,1,0xe5b89800,0,0xeaa8,1,0xe3808a00,801,0x8173
,1,0xe5b99800,4461,0xeaa9,1,0xe7bdb800,0,0xe3ad,1,0xe5babe00,6536,0xeaab
,1,0xe5818f00,10928,0x95ce,1,0xe586a200,0,0x996e,1,0xe58bb500,6074,0x99ad
,1,0xe5bd9300,0,0x9396,1,0xe6a29d00,0,0x9e8a,1,0xe787a700,0,0xe09c
,1,0xe5bbb900,8359,0xeaae,1,0xe38b9000,0,0x86d9,1,0xcb980000,0,0x85a0
,1,0xe988ac00,9339,0xe7e3,1,0xe5bc8000,5484,0xeaaf,1,0xe58c9500,0,0x99b6
,1,0xe591a800,0,0x8efc,1,0xefbcb900,0,0x8278,1,0xe5bc8200,7299,0xeab0
,1,0xceb50000,353,0x83c3,1,0xe5bc9d00,0,0xeab2,1,0xe8a48200,0,0xe5e6
,1,0xe8a99500,0,0x955d,1,0xe5bcb400,0,0xeab4,1,0xe8b3bb00,0,0xe6d0
,1,0xe5928800,0,0x87fa,1,0xefbd9900,0,0x8299,1,0xe5bcb600,5276,0xeab5
,1,0xe6ae8300,0,0x9f69,1,0xe5bcbd00,6891,0xeab6,1,0xe285a100,60,0x8755
,1,0xe6bdbc00,0,0xe04d,1,0xe5bd8500,0,0xeab8,1,0xe5bd9800,4819,0xeaba
,1,0xe999a500,0,0x8ad7,1,0xe5bda700,0,0xeabc,1,0xe4bd9700,1093,0x98c7
,1,0xc3840000,0,0x855a,1,0xe5bdbd00,6585,0xeabd,1,0xe68cbe00,7017,0x9d70
,1,0xe5be9c00,3856,0xeabf,1,0xe8839d00,7832,0xe3f1,1,0xe5beaf00,4385,0xeac1
,1,0xe99a8500,0,0x8bf7,1,0xe5beb500,3847,0xeac2,1,0xe5bf8900,3861,0xeac4
,1,0xe5a38100,0,0x95c7,1,0xe5bfa100,0,0xeac6,1,0xe5bfa900,9653,0xeac7
,1,0xe5b2ba00,0,0x9889,1,0xe6809400,0,0xeac9,1,0xc9be0000,7934,0x85e5
,1,0x77000000,11473,0x77,1,0xe6809800,0,0xeaca,1,0xe9a58b00,0,0xe95e
,1,0xe6818700,9630,0xeacd,1,0xe9afb100,0,0xe9cd,1,0xe5ae8700,0,0x8946
,1,0xefa8bd00,0,0xeace,1,0xe5b8ad00,3827,0x90c8,1,0xe69db700,0,0x9466
,1,0xe7af8c00,0,0xe2ba,1,0xe3819700,0,0x82b5,1,0xe6838b00,0,0xead1
,1,0xe6839500,0,0xead3,1,0xe6839d00,6617,0xead4,1,0xe9b5a400,4353,0xea42
,1,0xe6849c00,0,0xead6,1,0xe6998400,0,0x9de6,1,0xe69e9700,1289,0x97d1
,1,0xe684ab00,0,0xead7,1,0xe6a8bd00,7310,0x924d,1,0xe7ba9200,9026,0xe395
,1,0xe684b000,5172,0xead8,1,0xe684b700,0,0xead9,1,0xefa8be00,10986,0xeada
,1,0xe6868d00,0,0xeadb,1,0xefa8bf00,4943,0xeadc,1,0xe686b900,0,0xeade
,1,0xefa98000,3875,0xeadf,1,0xe6899600,0,0xeae3,1,0xe68b8400,0,0xeae7
,1,0xe280bb00,7532,0x81a6,1,0xe68bbc00,9778,0xeae9,1,0xe68c8a00,4607,0xeaea
,1,0xe68c9800,0,0xeaeb,1,0xe58e8200,0,0x99ca,1,0xe68cb900,4020,0xeaec
,1,0xe4b8b100,5612,0x98a3,1,0xe68d8300,5699,0xeaed,1,0xe68fa500,0,0xeaf0
,1,0xcebc0000,338,0x83ca,1,0xe6b9b600,4269,0x9fd7,1,0xe8ab8200,0,0xe67a
,1,0xe68fad00,5953,0xeaf1,1,0xe68fb500,0,0xeaf2,1,0xe99ab200,8030,0xe8ab
,1,0xe6909000,5222,0xeaf3,1,0xe6909400,0,0xeaf4,1,0xe5a3ae00,0,0x9173
,1,0xe688b800,0,0x8ccb,1,0x60000000,294,0x60,1,0xe690a200,4303,0xeaf5
,1,0xe691b900,0,0xeaf6,1,0xe691a000,5948,0xeaf8,1,0xe8bb9b00,5027,0xe762
,1,0xe691ad00,10344,0xeaf9,1,0xe4bf8400,0,0x89e2,1,0xe6938e00,0,0xeafa
,1,0xc38b0000,417,0x8561,1,0xe5aeb400,1424,0x8983,1,0xe693be00,0,0x8fef
,1,0xe7a59300,4516,0xe250,1,0xe88a9d00,0,0x8ec5,1,0xe88fb000,0,0x8cd4
,1,0xe6938400,9841,0xeb40,1,0xe9a69800,0,0xe964,1,0xe9abab00,0,0xe99a
,1,0xe9b0be00,0,0xe9e2,1,0xe5af9400,0,0x9b86,1,0xe5b4a700,6423,0x9891
,1,0xe5b9ba00,0,0x9bf4,1,0xe6938a00,0,0xeb41,1,0xe6939000,0,0xeb42
,1,0xe382a400,11475,0x8343,1,0xe387b700,2705,0x83fb,1,0xe693b700,6929,0xeb43
,1,0xe693bb00,0,0xeb44,1,0xe695a700,4864,0xeb48,1,0xe5b58700,0,0x9894
,1,0xe5ba9a00,298,0x8d4d,1,0xe69fa400,0,0x9e6e,1,0xe6988000,4083,0xeb4b
,1,0xe3838400,8276,0x8363,1,0xe6989500,0,0xeb4d,1,0xe980a000,0,0x91a2
,1,0xe985b300,0,0xe7c8,1,0xe5848900,0,0x994c,1,0xe6989e00,5413,0xeb4e
,1,0xe698a200,0,0xeb50,1,0xe6a08400,0,0x8968,1,0xe7858e00,3887,0x90f7
,1,0xe6aaaa00,0,0x9f4b,1,0xe698a400,0,0xeb51,1,0xe6998c00,0,0xeb58
,1,0xeaabb100,0,0xeb59,1,0xe6999900,0,0xeb5a,1,0xe6999a00,0,0xeb5b
,1,0xe699a500,6118,0xeb5d,1,0xe699b300,0,0xeb5e,1,0x49000000,95,0x49
,1,0xe69a8d00,7340,0xeb61,1,0xefa98300,10988,0xeb62,1,0xe795a700,6923,0xe15f
,1,0xe79aba00,0,0xe1b0,1,0xe98c8600,0,0x8e4b,1,0xcba60000,2906,0x8681
,1,0xe69ab200,5002,0xeb64,1,0xe69abb00,4088,0xeb65,1,0xe4ba9e00,5645,0x98b1
,1,0xe4bfb100,0,0x879f,1,0xe69b8600,0,0xeb66,1,0xe7968700,0,0xe165
,1,0xe3aca200,10305,0xeb68,1,0xe69b9b00,4311,0xeb69,1,0xe8b28200,0,0xe6b8
,1,0xe9978c00,0,0xe88c,1,0xe69ba800,0,0xeb6a,1,0xe69cb300,0,0xeb6e
,1,0xe69d8700,0,0xeb70,1,0xe69d8800,5812,0xeb71,1,0xe69e8100,3935,0xeb73
,1,0xe69e9300,0,0xeb74,1,0xe69e9800,6651,0xeb75,1,0xe293a100,0,0x86d0
,1,0xe69ebb00,0,0xeb77,1,0xe890bd00,0,0x978e,1,0xe69f8000,0,0xeb79
,1,0xe69f9700,0,0xeb7a,1,0xe6a08100,0,0xeb7c,1,0xe68b8500,0,0x9253
,1,0xc3920000,0,0x8568,1,0xe6a0ac00,6429,0xeb80,1,0xe2948100,279,0x84aa
,1,0xe6a1b200,7178,0xeb83,1,0xe6a1b500,7179,0xeb84,1,0xe383b100,4639,0x8391
,1,0xe9a88500,0,0xe976,1,0xe9ad9800,0,0xe9b4,1,0xe6a3a800,0,0xeb8b
,1,0xe5b18100,0,0x9b9b,1,0xe6a48a00,5152,0xeb90,1,0xe6a58900,0,0xeb91
,1,0xe6a0b100,4060,0xeb81,1,0xeaac8300,0,0xeb92,1,0xe8979000,8313,0xe557
,1,0xd0a40000,9097,0x8455,1,0xe981ad00,0,0x9198,1,0xe6a4b500,0,0xeb93
,1,0x32000000,3569,0x32,1,0xe58aa900,0,0x8f95,1,0xe5bc8700,0,0xeab1
,1,0xe6a19100,0,0x8c4b,1,0xe6a59700,7272,0xeb95,1,0xefa89400,5342,0xeb9a
,1,0xe7bd8c00,9106,0xe3a0,1,0xe8a29600,0,0x91b3,1,0xe3aeb600,5072,0xeb9e
,1,0xe3af8300,0,0xeb9f,1,0xe58b8900,0,0x95d7,1,0xe6a7a200,7296,0xeba0
,1,0xe6a7a900,0,0xeba1,1,0xefa98500,5355,0xeba2,1,0xe6a7b500,6809,0xeba3
,1,0xe6b1aa00,0,0x9f8a,1,0xe6a7b600,0,0xeba4,1,0xe6a88f00,0,0xeba5
,1,0xe8ad9c00,0,0x9588,1,0xe8b2af00,0,0x9299,1,0xe6a8bb00,0,0xeba8
,1,0xefbc8d00,0,0x81af,1,0xe6a8be00,5265,0xeba9,1,0xe5a0b500,0,0x9367
,1,0xe6a99000,5123,0xebab,1,0xe7979400,4014,0x8ea4,1,0xe6a99600,8306,0xebac
,1,0xe6a99b00,8479,0xebad,1,0xe6a9ab00,0,0xebae,1,0xeaac8b00,9535,0xebb0
,1,0xe6aa8900,0,0xebb1,1,0xe4bc8b00,10386,0x87aa,1,0xe6aa9d00,4240,0xebb3
,1,0xe5a6a800,0,0x9657,1,0xe5abbb00,0,0x9b66,1,0xe79d8700,0,0xe1c7
,1,0xe7a29a00,0,0xe1ee,1,0xe6aa9e00,4381,0xebb4,1,0xe8b98200,0,0xe6f8
,1,0xe6aaa500,6471,0xebb5,1,0xe9a39f00,0,0x9048,1,0xe6aba400,0,0xebb6
,1,0xefa49d00,5819,0xebb9,1,0xe6acb500,7373,0xebbd,1,0xe5b1ae00,0,0x9ba3
,1,0xe6ad9600,6455,0xebbf,1,0xe7a88d00,8618,0xe263,1,0xe6ada000,4412,0xebc0
,1,0xe29fb400,2231,0x86af,1,0x1b000000,0,0x1b,1,0xe6ae8200,5091,0xebc4
,1,0xe6aea900,4136,0xebc5,1,0xe6aead00,0,0xebc6,1,0xe6af8f00,0,0xebc8
,1,0xc3990000,297,0x856e,1,0xe69cab00,3707,0x9696,1,0xe6a1be00,4067,0x9e95
,1,0xe3808b00,2592,0x8174,1,0xe7b8a600,0,0x8f63,1,0xe7bdb900,0,0x9ceb
,1,0xe6af9600,0,0xebc9,1,0xe5819000,10932,0x98f2,1,0xe6af9700,5383,0xebca
,1,0xe6b08500,0,0xebcc,1,0xe5bd9400,0,0xeab9,1,0xe6b09000,4162,0xebcd
,1,0xe6b19c00,4434,0xebd0,1,0xe7b98600,0,0xe378,1,0xe38b9100,0,0x86da
,1,0xd0ab0000,0,0x845c,1,0xe8a8b600,0,0xe664,1,0xe5878300,3580,0x87d3
,1,0xe58c9600,0,0x89bb,1,0xe6b2aa00,5703,0xebd1,1,0xefbcba00,0,0x8279
,1,0xe6b28500,6735,0xebd4,1,0xe6ada400,0,0x8d9f,1,0xe6b28600,9629,0xebd5
,1,0xe6b29800,0,0xebd6,1,0xe6b3bb00,4665,0xebd8,1,0xe6b39400,0,0xebda
,1,0xe8b3bc00,0,0x8d77,1,0xe6b3a000,0,0xebdb,1,0xefbd9a00,1037,0x829a
,1,0xe59caf00,329,0x8863,1,0xe6ae8400,0,0x9f68,1,0xe6b39700,0,0x9f99
,1,0xe285a200,2002,0x8756,1,0xe6b3ab00,7098,0xebdc,1,0xe6b3ae00,0,0xebdd
,1,0xeaac9400,0,0xebde,1,0xe999a600,9565,0xe8a1,1,0xe6b48e00,0,0xebe0
,1,0xe59d8f00,5285,0x9aad,1,0xe6b4ae00,0,0xebe1,1,0xe6b4b100,0,0xebe2
,1,0xe68cbf00,720,0x917d,1,0xe6b4b900,4301,0xebe3,1,0x4000000,684,0x4
,1,0xe6b59800,0,0xebe5,1,0xe99a8600,0,0x97b2,1,0xe8bfa200,10309,0xe78b
,1,0xe9a4ac00,0,0xe956,1,0xe6b5a500,0,0xebe6,1,0xe6888c00,0,0x9cfa
,1,0xe6b68200,4453,0xebe8,1,0xe692b200,1205,0x966f,1,0xe7a48700,0,0xe244
,1,0xe6b68700,0,0xebe9,1,0xe7aead00,0,0x90fb,1,0xe6b68900,0,0xebea
,1,0xe9a58c00,0,0xe961,1,0xe6b69400,0,0xebeb,1,0xe9afb200,0,0xe9cc
,1,0xe581bd00,269,0x8b55,1,0xe6939200,0,0x9da0,1,0xe698a500,6092,0x8f74
,1,0xe6b6ac00,0,0xebed,1,0xe6b78400,0,0xebef,1,0xe3819800,388,0x82b6
,1,0xe6b79600,8364,0xebf0,1,0xe89ebd00,0,0xe5a8,1,0xe9b09200,0,0xe9d6
,1,0xe6b79a00,4372,0xebf1,1,0xe587b000,3597,0x9980,1,0xe5b98e00,6488,0x9beb
,1,0xc3a00000,180,0x8575,1,0xe6b79b00,0,0xebf2,1,0xe788b500,3905,0x8edd
,1,0xe89a8a00,2346,0x89e1,1,0xe7bfa600,0,0xe3c6,1,0xe6b79d00,4737,0xebf3
,1,0xe6b7bc00,9595,0xebf4,1,0xe9bb9800,0,0xe0d2,1,0xe58da300,0,0x87ee
,1,0xefa98700,0,0xebf5,1,0xe6a48b00,0,0x96b8,1,0xe6b98400,4112,0xebf7
,1,0xe6aeb100,7429,0x9f73,1,0xe280bc00,0,0x84e9,1,0xe6b99c00,9779,0xebf8
,1,0xd0b20000,463,0x8472,1,0xe6b99e00,7658,0xebf9,1,0xe6bab100,7672,0xebfb
,1,0xe6bb8100,0,0xebfc,1,0xe598a900,151,0x89dc,1,0xe59dbc00,0,0x8868
,1,0xe78f8800,0,0xe0db,1,0xe6bc9000,6910,0xec42,1,0xe6bc9a00,7659,0xec43
,1,0xefa98800,0,0xec44,1,0xe6bcaa00,0,0xec45,1,0xe6bcaf00,0,0xec46
,1,0xe6bcb300,10182,0xec47,1,0xe5998900,6722,0x8852,1,0xe59e9c00,0,0x8869
,1,0xe5a3af00,0,0x9ae1,1,0xe6b58400,0,0x8ff2,1,0xe79a8e00,0,0xe1a7
,1,0xc5810000,0,0x85a1,1,0xe6bd9e00,4335,0xec4a,1,0xe9968000,2502,0x96e5
,1,0xe6bda200,0,0xec4c,1,0xe6be9400,10055,0xec51,1,0xe6bea700,5306,0xec53
,1,0xe5a48f00,1331,0x89c4,1,0xe5a9a200,0,0x9b58,1,0xe68eac00,303,0x8b64
,1,0xe6beb600,6408,0xec54,1,0xe8858b00,0,0xe3fc,1,0xe6bebc00,0,0xec55
,1,0xe88fb100,0,0x9548,1,0xe9a18600,0,0xe8f7,1,0xe9a69900,0,0x8d81
,1,0xe6bf8700,0,0xec56,1,0xe6bf8a00,0,0xec57,1,0xe6bfb000,0,0xec59
,1,0xe780a800,0,0xec5d,1,0xe5b9bb00,0,0x8cb6,1,0xe7ab8700,0,0xe285
,1,0xe7818a00,8788,0xec5e,1,0xe382a500,4629,0x8344,1,0xe387b800,0,0x83fc
,1,0xe7819e00,10657,0xec60,1,0xe781b500,10879,0xec63,1,0xe9b6b200,5028,0xea4f
,1,0xe782a400,0,0xec65,1,0xe782b700,6527,0xec67,1,0xe7839400,6177,0xec68
,1,0xe7839800,7255,0xec69,1,0xe3838500,579,0x8364,1,0xe783a400,9073,0xec6a
,1,0xe980a100,0,0xe795,1,0xe7848f00,7644,0xec6b,1,0xe784ab00,0,0xec6c
,1,0xe5899d00,0,0x889e,1,0xe58eb000,0,0x99d0,1,0xe784a000,4496,0xec6e
,1,0xc3a70000,0,0x857c,1,0xe78aa200,6776,0xe0b7,1,0xe784ae00,0,0xec6f
,1,0xe9818100,9209,0x93d9,1,0xe784b000,0,0xec70,1,0xe7858600,4271,0xec71
,1,0xe990ba00,11144,0xe865,1,0xe7858700,9119,0xec72,1,0xe7859100,0,0xec73
,1,0xe599b600,7767,0x8853,1,0xe78b8200,0,0x8bb6,1,0xefa89500,10955,0xec79
,1,0xe7868500,0,0xec7a,1,0xe7869200,0,0xec7c,1,0xe8ac9000,0,0xe68d
,1,0xd0b90000,3202,0x847a,1,0xe996ad00,0,0xe883,1,0xe7878100,0,0xec7d
,1,0xe4ba9f00,8334,0x98b2,1,0xe786ba00,5206,0xec7e,1,0xe7878400,0,0xec80
,1,0xe787be00,4845,0xec81,1,0xe79b9b00,0,0x90b7,1,0xe7888000,0,0xec82
,1,0xe885b800,0,0x92b0,1,0xe8b79600,3227,0xe6e8,1,0xe8bca900,0,0x9479
,1,0xe9a1b300,0,0xe942,1,0xe5a08900,0,0x8875,1,0xe6859300,0,0x9cd6
,1,0xe7889500,6289,0xec83,1,0xe7899500,0,0xec84,1,0xe7899600,4314,0xec85
,1,0xc5880000,3,0x85d0,1,0xe1b8be00,537,0x84f0,1,0xe8bd8900,0,0xe77a
,1,0xe78a9b00,0,0xec88,1,0xe78b8000,0,0xec8a,1,0xeaaca200,0,0xec8c
,1,0xe68b8600,0,0x9d5d,1,0xe5b0a200,0,0x9b97,1,0xe695ac00,0,0x8c68
,1,0xe2948200,240,0x84a0,1,0xefa89600,0,0xec8f,1,0xe78d9000,0,0xec90
,1,0xe383b200,2294,0x8392,1,0xe78da600,0,0xec91,1,0xe78dbc00,8372,0xec92
,1,0xe78e9500,0,0xec93,1,0xe584b700,0,0x9955,1,0xe6968c00,0,0x956b
,1,0xe5bba800,0,0x9c47,1,0xe6a0b200,0,0x9e7d,1,0xe78e9f00,0,0xec94
,1,0xe7b79a00,0,0x90fc,1,0xe78ea000,4266,0xec95,1,0xe981ae00,450,0x8ed5
,1,0xe78ea200,0,0xec96,1,0xe5859700,0,0x87d0,1,0xe58aaa00,3673,0x9377
,1,0xe78ea600,6140,0xec97,1,0xe6a19200,0,0xeb7d,1,0xe6a6a500,0,0xeb9b
,1,0xe78eab00,0,0xec98,1,0xe7bd8d00,0,0xe3a1,1,0xe8a29700,0,0xe5d5
,1,0xe987a100,0,0xe7db,1,0xe78f8900,0,0xec99,1,0xe78f8f00,5431,0xec9a
,1,0xe5909d00,0,0x99e5,1,0xe78f9600,0,0xec9b,1,0xe78f9900,0,0xec9c
,1,0xc3ae0000,0,0x8584,1,0xe78fa300,4197,0xec9d,1,0xe796b500,0,0xe172
,1,0xe8a88a00,9830,0x9075,1,0xe98d9400,0,0x92d5,1,0xe8b2b000,0,0x96e1
,1,0xe78fa900,0,0xec9e,1,0xefbc8e00,0,0x8144,1,0xe59ba300,10038,0x9263
,1,0xe7908700,4796,0xec9f,1,0xe7908a00,0,0xeca0,1,0xe2849600,212,0x8782
,1,0xe6bcb100,0,0x9ff9,1,0xe881b200,0,0xe3df,1,0xe7909a00,0,0xeca1
,1,0xe7909b00,4834,0xeca2,1,0xe99dad00,0,0x9078,1,0xe59c8300,10850,0x95de
,1,0xe6818d00,0,0x9c92,1,0xefa98b00,0,0xeca3,1,0xe68bb300,0,0x8c9d
,1,0xe6bd9100,0,0xec48,1,0xe790a800,0,0xeca5,1,0xe294af00,74,0x84b6
,1,0xe88cb800,0,0x91f9,1,0xe99e8d00,6552,0x88c6,1,0xe790aa00,10396,0xeca6
,1,0xe790ab00,0,0xeca7,1,0xe5a78900,0,0x8e6f,1,0xe790ae00,0,0xeca9
,1,0xe5b1af00,0,0x93d4,1,0xe696b900,0,0x95fb,1,0xe7a88e00,2474,0x90c5
,1,0xe88d9800,0,0x9191,1,0xe29fb500,0,0x86b0,1,0xe897be00,0,0xe560
,1,0xe790af00,0,0xecaa,1,0xe7918700,10722,0xecae,1,0xe7919700,0,0xecb0
,1,0xe5b28f00,0,0x9885,1,0xe6979900,3680,0x9dd8,1,0xcf840000,452,0x83d1
,1,0xe6a1bf00,7187,0x9e85,1,0xe3808c00,82,0x8175,1,0xe791a200,8356,0xecb1
,1,0xe791ab00,4692,0xecb2,1,0xe9af8600,0,0xe9c2,1,0xe791ad00,0,0xecb3
,1,0xe586a400,0,0x996c,1,0xe5b88200,5437,0x8e73,1,0xe7928700,6570,0xecb5
,1,0xe6a29f00,0,0x9e86,1,0xe6a7b200,0,0x9ede,1,0xe7928900,4485,0xecb6
,1,0xe38b9200,2363,0x86db,1,0xe9839b00,0,0xe7bc,1,0xe7929800,4762,0xecb7
,1,0xe5878400,3585,0x90a6,1,0xe58c9700,0,0x966b,1,0xe591aa00,0,0x8ef4
,1,0xefbcbb00,0,0x816d,1,0xe6a89200,0,0x9ee7,1,0xe6ada500,0,0xebc1
,1,0xe6b2b800,0,0x95a6,1,0xe8a48400,0,0xe5eb,1,0xe7929f00,9868,0xecb9
,1,0xe792a300,10037,0xecba,1,0xe8b3bd00,0,0xe6ce,1,0xe7929000,5499,0xecbb
,1,0xefbd9b00,5596,0x816f,1,0xe59cb000,0,0x926e,1,0xe792a600,0,0xecbc
,1,0xe2809000,11485,0x815d,1,0xca8d0000,6,0x865a,1,0xe6bdbe00,0,0xec4d
,1,0xe792a800,10070,0xecbd,1,0xe792a900,0,0xecbe,1,0xe792b500,4835,0xecbf
,1,0xe792bf00,5053,0xecc0,1,0xe59d9000,790,0x8dbf,1,0xe6829a00,0,0x9c9e
,1,0xe5a7b600,0,0x88a6,1,0xe7998200,7019,0x97c3,1,0xe7938800,0,0xecc1
,1,0xe7a3a800,0,0x9681,1,0xe7a8bb00,8631,0xe26a,1,0xe7938900,0,0xecc2
,1,0xe7939a00,6198,0xecc3,1,0xe7948100,6908,0xecc5,1,0xe7949700,5764,0xecc6
,1,0xe6888d00,0,0x9cf9,1,0xe5ada900,0,0x9b77,1,0xe5b2bc00,0,0x9bae
,1,0xe794af00,0,0xecc7,1,0xe795af00,0,0xecc8,1,0xe7969200,0,0xecca
,1,0xe7978e00,0,0xeccc,1,0xe7988000,4963,0xecce,1,0xe7988200,0,0xeccf
,1,0xe7989500,7005,0xecd1,1,0xe5ae8900,1423,0x88c0,1,0xe7989600,10842,0xecd2
,1,0xe5b8af00,0,0x91d1,1,0xe7989900,7006,0xecd3,1,0xe88f8500,0,0x909b
,1,0x6c000000,11470,0x6c,1,0xe899ab00,0,0x928e,1,0xe9ab8000,9146,0xe98f
,1,0xe9b09300,0,0xe9d2,1,0xe7989e00,7007,0xecd4,1,0xe587b100,207,0x8a4d
,1,0xe798b500,0,0xecd6,1,0xe799a400,6158,0xecd9,1,0xe799a500,6675,0xecda
,1,0xe788b600,4447,0x9583,1,0xe7ba9400,2704,0xe398,1,0xe799ad00,0,0xecdb
,1,0xe79a8100,0,0xecde,1,0xe79aa600,9976,0xece2,1,0xe5889100,748,0x8c59
,1,0xe79ab600,4904,0xece4,1,0xe79b8500,5526,0xece5,1,0xe6a48c00,7233,0x9e9d
,1,0xe6a99f00,4113,0x8b40,1,0xe78ea900,1676,0x8adf,1,0xe79b8c00,0,0xece6
,1,0xe79b8e00,4919,0xece7,1,0xe8aaa400,3132,0x8ceb,1,0xe79b9400,0,0xece8
,1,0xe58e8400,0,0x96ef,1,0xe79ba600,0,0xece9,1,0xe79bbc00,6290,0xeceb
,1,0xe79c8a00,0,0xecec,1,0xe6af9200,4148,0x93c5,1,0xe6b4a500,493,0x92c3
,1,0xe799af00,0,0xecdc,1,0xe8ab8400,5634,0xe678,1,0xe79c9900,0,0xeced
,1,0xe79cb600,0,0xecef,1,0xe99ab400,0,0xe8ad,1,0xeaac8900,0,0xeba7
,1,0xe79d8600,5708,0xecf0,1,0xe5a3b000,1312,0x90ba,1,0xe6b58500,0,0x90f3
,1,0xc3bc0000,1764,0x8591,1,0xca940000,760,0x8640,1,0xe79da200,8551,0xecf5
,1,0xe8b68a00,8684,0x897a,1,0xe79e9400,0,0xecf8,1,0xe79eaa00,0,0xecf9
,1,0xe79fa000,7129,0xecfa,1,0xe5a49000,0,0x9ae9,1,0xe6899a00,0,0xeae4
,1,0xe5aeb600,0,0x89c6,1,0x55000000,216,0x55,1,0xe7a59500,0,0xe24f
,1,0xe88a9f00,0,0xe48c,1,0xe88fb200,0,0xe4c9,1,0xe7a0ad00,8468,0xecfb
,1,0xe7a18300,0,0xed40,1,0xe9abad00,0,0x9545,1,0xe7a18e00,0,0xed41
,1,0xe7a18f00,5917,0xed42,1,0xe5b4a900,0,0x95f6,1,0xe5b9bc00,3833,0x9763
,1,0xe7ab8800,0,0xe27d,1,0xe7a1a800,6451,0xed44,1,0xe382a600,2278,0x8345
,1,0xe7a1ae00,4679,0xed45,1,0xe7a3a400,0,0xed4a,1,0xe9b1a000,0,0xe9e4
,1,0xe7a48000,5064,0xed4c,1,0xe7a4ae00,0,0xed4f,1,0xe5ba9c00,0,0x957b
,1,0xe7a4b100,6181,0xed50,1,0xe6a4b900,4085,0x9eb9,1,0xe3838600,0,0x8365
,1,0xc59d0000,0,0x85e1,1,0xe980a200,0,0x88a7,1,0xe985b500,2826,0x8d79
,1,0xe5848b00,3551,0x87ca,1,0xe5899e00,0,0x9990,1,0xe7a4b400,0,0xed51
,1,0xefa98d00,0,0xed52,1,0xc8ba0000,1625,0x866c,1,0xe6aaac00,0,0x9f47
,1,0xe6afbf00,0,0xebcb,1,0xe9818200,0,0x908b,1,0xefa98e00,0,0xed53
,1,0xe8abb100,0,0xe681,1,0xe7a58500,10099,0xed54,1,0xe7a58600,0,0xed55
,1,0xefa98f00,10990,0xed56,1,0xeaacb600,0,0xecfc,1,0xe78b8300,0,0xe0bb
,1,0xefa99000,0,0xed57,1,0xe795a900,0,0xe15d,1,0xe7a59c00,0,0xed59
,1,0xefa89900,5145,0xed5b,1,0xe9919b00,0,0xe869,1,0xefa89a00,5345,0xed5c
,1,0xe5958400,0,0x91ed,1,0xe4baa000,8399,0x98b3,1,0x3e000000,569,0x3e
,1,0xe684b400,5797,0x9cc6,1,0xe7968900,0,0xe168,1,0xe2889d00,1657,0x81e5
,1,0xe7a5b900,4507,0xed5d,1,0xe885b900,0,0x95a0,1,0xefa99300,4831,0xed5e
,1,0xe8bcaa00,0,0x97d6,1,0xe4bb8000,3351,0x8f59,1,0xe5a08a00,0,0x9abf
,1,0xe5a59d00,0,0x8889,1,0xefa99400,0,0xed5f,1,0xe68fba00,0,0x9768
,1,0xe8818600,4315,0xe3d7,1,0xe293a300,0,0x86d2,1,0xe1b8bf00,0,0x84f1
,1,0xefa89b00,9621,0xed60,1,0xe7a69800,0,0xed61,1,0xe7a6b100,0,0xed62
,1,0xe7a79e00,0,0xed68,1,0xe5ab9000,1402,0x9b6b,1,0xe5b0a300,0,0x987c
,1,0xe5b5b600,0,0x9bcc,1,0xe2948300,7878,0x84ab,1,0xe7a7ab00,8730,0xed69
,1,0xe7a7ad00,4599,0xed6a,1,0xe383b300,434,0x8393,1,0xe7a88300,0,0xed6b
,1,0xe9ad9a00,0,0x8b9b,1,0xe7a8b900,0,0xed6d,1,0xe7a9ad00,0,0xed6f
,1,0xe7aa8500,0,0xed71,1,0xe5bba900,1547,0x9c48,1,0xe7aaa000,6623,0xed72
,1,0xe785bd00,0,0x90f8,1,0xeaacbd00,0,0xed73,1,0xe89ca500,0,0xe592
,1,0xe981af00,0,0xe7aa,1,0xe5808500,0,0x98e4,1,0xe7ab8e00,0,0xed76
,1,0xe58aab00,0,0x8d85,1,0xe69c8000,1265,0x8dc5,1,0xe6a19300,4063,0x8aba
,1,0xe7abab00,6706,0xed77,1,0xe7abbd00,0,0xed78,1,0xe7bd8e00,0,0xe3a2
,1,0xc5a40000,3065,0x85a6,1,0xe7ac9200,9298,0xed79,1,0xe8acbe00,3157,0xe694
,1,0x27000000,0,0x27,1,0xe5909e00,0,0x989e,1,0xe7acad00,0,0xed7a
,1,0xe7acbb00,0,0xed7b,1,0xe7ad8e00,0,0xed7d,1,0xe791a300,0,0xe0f2
,1,0xe6b6bf00,0,0xebee,1,0xe7ada000,8749,0xed7e,1,0xe7ae9e00,8758,0xed83
,1,0xefa99700,5366,0xed84,1,0xe7af9700,0,0xed85,1,0xefbc8f00,5569,0x815e
,1,0xe4bbad00,10039,0x98bf,1,0xe7b0b100,0,0xed88,1,0xe6b28c00,819,0x93d7
,1,0xe7b0b300,6998,0xed8b,1,0xe289aa00,1789,0x81e1,1,0xe7a1bc00,2397,0xe1ed
,1,0xe8b39100,0,0x93f6,1,0xe7b0b600,0,0xed8c,1,0xeaad8500,5041,0xed8e
,1,0xe4bc8d00,1081,0x8cde,1,0xe5a19700,0,0x9368,1,0xe7b19900,0,0xed8f
,1,0xe7b28f00,0,0xed92,1,0xe28a8a00,7088,0x81c2,1,0xe8829300,0,0xe3e9
,1,0xe294b000,0,0x84bb,1,0xe8b98400,2537,0x92fb,1,0xe7b29400,7461,0xed93
,1,0xe7b2a000,8579,0xed94,1,0xe7b39500,0,0xed96,1,0xe7b39900,10870,0xed97
,1,0xe7b39d00,0,0xed98,1,0xe691a700,5951,0x9d94,1,0xe7b48700,0,0xed99
,1,0xe7b49300,0,0xed9b,1,0xcaa20000,0,0x865c,1,0xe29fb600,0,0x86b1
,1,0xe7b49d00,4644,0xed9c,1,0xe7b4a300,5135,0xed9d,1,0xe7b4b100,5117,0xed9e
,1,0xe7b58100,6749,0xed9f,1,0xe5b29000,6382,0x8af2,1,0xe5b7a300,0,0x9183
,1,0xe69cad00,3711,0x8e44,1,0xe7b58800,5019,0xeda0,1,0xe3808d00,4436,0x8176
,1,0xe7b59300,4476,0xeda1,1,0x10000000,9114,0x10,1,0xe7b59c00,8909,0xeda2
,1,0xe7b5ba00,5069,0xeda3,1,0xe586a500,0,0x96bb,1,0xe5b88300,981,0x957a
,1,0xe5bd9600,6572,0x9c60,1,0xe6a2a000,7195,0x9e91,1,0xe7b68300,6728,0xeda4
,1,0xe7b68b00,7384,0xeda5,1,0xe38b9300,0,0x86dc,1,0xe7b6a000,9730,0xeda6
,1,0xe7b6a600,8940,0xeda7,1,0xe5878500,0,0x9978,1,0xe7b78c00,0,0xeda9
,1,0xe591ab00,0,0x87f7,1,0xe7b79600,0,0xedaa,1,0xe6a89300,0,0x9eea
,1,0xe6ada600,0,0x9590,1,0xe792b000,767,0x8ac2,1,0xefa99800,0,0xedac
,1,0xc5ab0000,4947,0x859c,1,0xcc830000,0,0x869d,1,0xe7b8a800,4814,0xedad
,1,0xe5928b00,0,0x8dee,1,0xefbd9c00,0,0x8162,1,0xe4bcba00,0,0x8e66
,1,0xe6ae8600,0,0x9677,1,0xe6b39900,0,0x9fa2,1,0xe285a400,403,0x8758
,1,0xe7b88800,0,0xedae,1,0xefa99a00,0,0xedb1,1,0xe7b98700,0,0xedb2
,1,0xe7b9a100,0,0xedb4,1,0xe4b88700,0,0x969c,1,0xe59d9100,918,0x8d42
,1,0xe6829b00,0,0x9ca0,1,0xe7ba8a00,0,0xedb5,1,0xe7998300,7024,0xecd7
,1,0xe7ba8d00,7108,0xedb6,1,0xe7bd8700,4509,0xedb7,1,0xe7a8bc00,2482,0x89d2
,1,0xe99a8800,2981,0x8c47,1,0xefa99b00,4857,0xedb8,1,0xe9a4ae00,0,0xe957
,1,0xe7be9100,4696,0xedb9,1,0xe6888e00,5843,0x8f5e,1,0xe7be9700,0,0xedba
,1,0xe7bebf00,0,0xedbb,1,0xe7bf9b00,0,0xedbd,1,0xe7bf9f00,4709,0xedbe
,1,0xe7bfac00,8069,0xedbf,1,0xe7bfae00,9165,0xedc0,1,0xefa99c00,10519,0xedc2
,1,0xe880a600,0,0xedc4,1,0xe8839700,4571,0xedc9,1,0xe883a000,0,0xedca
,1,0xe6939400,0,0x9d5e,1,0xe5b8b000,707,0x8b41,1,0xe8849800,0,0xedcc
,1,0xe7af8f00,0,0xe2bb,1,0xe3819a00,231,0x82b8,1,0xd1810000,4667,0x8483
,1,0xe8858a00,5096,0xedcd,1,0xe9b09400,0,0xe9d0,1,0xe885a800,0,0xedd0
,1,0xe9baba00,0,0x96cb,1,0xe885ad00,0,0xedd1,1,0xe69e9a00,0,0x9687
,1,0xe6a3ad00,4074,0xeb8c,1,0xe7b58200,0,0x8f49,1,0xe89a8c00,0,0xe572
,1,0xe886bb00,5103,0xedd2,1,0xe8878a00,4739,0xedd3,1,0xe9b68700,9911,0xea48
,1,0xe8878f00,5403,0xedd4,1,0xe8879700,0,0xedd5,1,0xe5bf8300,1007,0x9053
,1,0xe6a48d00,7234,0x9041,1,0xefa99d00,7323,0xedd6,1,0xe6aeb300,0,0x9f74
,1,0xe280be00,0,0x7e,1,0xe9858900,0,0x93d1,1,0xe8aaa500,3126,0xe674
,1,0xe4919300,0,0xedd7,1,0xe889a000,0,0xedd9,1,0xe5939800,0,0x9a45
,1,0xe889b400,6329,0xedda,1,0xe6aa8000,0,0x9268,1,0xe78f8a00,1680,0x8e58
,1,0xeaada800,6912,0xeddb,1,0xe799b000,0,0xe19e,1,0xe88a8e00,0,0xeddc
,1,0xe88aa100,10079,0xeddd,1,0xe8b5ab00,0,0x8a71,1,0xe8babe00,0,0xe75e
,1,0xe88aa300,6738,0xedde,1,0xe88ab700,5709,0xede2,1,0xe5a3b100,234,0x88eb
,1,0xe688bb00,0,0x96df,1,0xe79a9000,0,0x8e48,1,0xe6bfac00,2090,0xe05c
,1,0xe88abe00,0,0xede3,1,0xe9968200,0,0xe878,1,0xe99b9500,0,0xe8b8
,1,0xe88abf00,0,0xede4,1,0xe88b8600,0,0xede5,1,0xe6848800,0,0x96fa
,1,0xe6899b00,0,0x9d4b,1,0xe88bbd00,0,0xede7,1,0xe88bbe00,0,0xede8
,1,0xe7a59600,0,0x9163,1,0xe7aaa900,0,0xe27c,1,0xe88c8000,8597,0xede9
,1,0xe88c8100,0,0xedea,1,0xe88ca200,0,0xedec,1,0xe9abae00,0,0xe99b
,1,0xe5aa8400,0,0x8897,1,0xe88cad00,7829,0xeded,1,0xe88d8300,0,0xedef
,1,0xe699b400,770,0x90b0,1,0xe88d8700,0,0xedf0,1,0xe8909300,0,0xe4be
,1,0xe382a700,0,0x8346,1,0xe88d9100,0,0xedf1,1,0xe88d9500,0,0xedf2
,1,0xe88dbd00,0,0xedf3,1,0xe9b6b400,5068,0x92df,1,0xe5b58a00,0,0x9896
,1,0xe88e8600,0,0xedf4,1,0xe5bfb000,1590,0x9cab,1,0xe88e9800,7256,0xedf6
,1,0xe3838700,2289,0x8366,1,0xe89b9900,0,0x8a5e,1,0xd1880000,867,0x848a
,1,0xe8a5bf00,0,0x90bc,1,0xe88ebf00,7858,0xedf9,1,0xe88f8000,0,0xedfa
,1,0xe58eb200,0,0x87f2,1,0xe88faa00,0,0xee42,1,0xe6a59a00,4089,0x915e
,1,0xe8908600,8109,0xee44,1,0xe8908a00,0,0xee45,1,0xe8a18c00,9684,0x8d73
,1,0xefa99f00,5368,0xee46,1,0xe98ba900,0,0xe7f8,1,0xe8919f00,8602,0xee48
,1,0xe891b000,0,0xee49,1,0xe8929e00,9002,0xee4c,1,0xe599b800,7807,0x93d3
,1,0xe78b8400,0,0xe0bd,1,0xe892b400,0,0xee4e,1,0xe795aa00,0,0x94d4
,1,0xe892ba00,0,0xee4f,1,0xe8938000,9924,0xee50,1,0xe8938200,0,0xee51
,1,0xeaadab00,0,0xee52,1,0xe5958500,0,0x9a55,1,0xe4baa100,8401,0x9653
,1,0xe894b200,0,0xee53,1,0xe6b18000,1130,0x92f3,1,0xe6b69300,1975,0x9fb0
,1,0xe2889e00,0,0x8187,1,0xe8949e00,0,0xee54,1,0xe885ba00,2909,0x9142
,1,0xc5b90000,0,0x85a7,1,0xe894a300,9614,0xee55,1,0xe4bb8100,0,0x906d
,1,0xe5a08b00,0,0x9ac2,1,0xe6859500,0,0x95e7,1,0xe894af00,5860,0xee56
,1,0xe6bc8600,0,0x8ebd,1,0xe8959900,0,0xee57,1,0xe293a400,639,0x86d3
,1,0xe7abb600,0,0x8ba3,1,0xe99d8200,0,0xe8cc,1,0xe895a400,0,0xee58
,1,0xe896ad00,8305,0xee5a,1,0xe9acbb00,0,0xe2f6,1,0xe68b8800,0,0x9d5f
,1,0xe5b0a400,0,0x96de,1,0xe895ba00,0,0xee5b,1,0xe8968c00,0,0xee5c
,1,0xe8968f00,0,0xee5d,1,0xe896a200,4823,0xee5e,1,0xe383b400,0,0x8394
,1,0xe896b000,0,0xee5f,1,0xe8978b00,0,0xee60,1,0xe8978e00,8307,0xee61
,1,0xe8989200,0,0xee63,1,0xe6968e00,6044,0x8dd6,1,0xe897bf00,4732,0xee64
,1,0xe6a0b400,0,0x90f1,1,0xe7b28900,0,0x95b2,1,0xe7b79c00,0,0xe363
,1,0xe8989000,0,0xee67,1,0xe8989800,0,0xee69,1,0xe5808600,0,0x98ed
,1,0xc79a0000,0,0x84f9,1,0xe58aac00,0,0x99a0,1,0xe5bc8a00,6554,0x95be
,1,0xe6a19400,7164,0x8b6a,1,0xe6a6a700,0,0x9ed0,1,0xe6abba00,0,0x9f51
,1,0xe898a900,8395,0xee6a,1,0xe8a29900,9699,0xe5d8,1,0xd18f0000,482,0x8491
,1,0xe8999b00,0,0xee6d,1,0xe58b8c00,5181,0x87e2,1,0xe5909f00,0,0x8be1
,1,0xefa4b600,5562,0xee6e,1,0xe6a78700,0,0xeaa0,1,0xe899a200,5192,0xee6f
,1,0xe791a400,6872,0xeaa2,1,0xe899ac00,7585,0xee71,1,0xe8a88c00,0,0xe660
,1,0xe8ad9f00,3165,0xe69f,1,0xe8b2b200,0,0xe6c5,1,0xe899b500,8366,0xee72
,1,0xefbc9000,0,0x824f,1,0xe4bbae00,10058,0x89bc,1,0xe680af00,0,0x8baf
,1,0xe6b28d00,0,0x9f8c,1,0xe89a9800,0,0xee73,1,0xe289ab00,4036,0x81e2
,1,0xe881b400,0,0x92ae,1,0xe89ab800,0,0xee74,1,0xe9989c00,0,0x958c
,1,0xe89bba00,0,0xee75,1,0xe4bc8e00,10941,0x8aea,1,0xc2a30000,499,0x8192
,1,0xe89bbc00,0,0xee76,1,0xe68bb500,0,0x9d6e,1,0xe28a8b00,0,0x81c3
,1,0xe89bbd00,7816,0xee77,1,0xe7a7b000,2470,0x8fcc,1,0xe88cba00,0,0xedee
,1,0xe99e8f00,0,0xe8dd,1,0xcc980000,113,0x8698,1,0xe89db100,0,0xee79
,1,0xe5a78b00,0,0x8e6e,1,0xe89e8700,8419,0xee7a,1,0xe5b1b100,0,0x8e52
,1,0xe89e8800,0,0xee7b,1,0xe8888700,0,0xe46f,1,0xe89eac00,9596,0xee7c
,1,0xe29fb700,2576,0x86b2,1,0xe89ead00,0,0xee7d,1,0xe9a99500,0,0xe981
,1,0xe9aea800,0,0xe9bd,1,0xe89eb500,0,0xee7e,1,0xe5b29100,1460,0x9ba8
,1,0xe6979b00,6071,0x9dd7,1,0xe5bcb700,1559,0x8bad,1,0xe781b800,0,0x8b84
,1,0xe3808e00,518,0x8177,1,0xe89f9600,7935,0xee81,1,0xe89fac00,0,0xee82
,1,0xe8a08a00,0,0xee84,1,0xe9b49b00,5324,0x8995,1,0xe586a600,0,0x996d
,1,0xe58bb900,0,0x99af,1,0xe69d8e00,1275,0x979b,1,0xc4840000,0,0x859f
,1,0xe8a09000,6731,0xee85,1,0xe78cbe00,9004,0xe0cf,1,0xe38b9400,8428,0x86dd
,1,0xe8a09400,4890,0xee86,1,0xe8a09f00,5786,0xee87,1,0xe5878600,0,0x8f79
,1,0xe58c9900,0,0x8dfa,1,0xe8a29800,5498,0xee88,1,0xefbcbd00,5587,0x816e
,1,0xe6a89400,0,0x9ee4,1,0xe6ada700,0,0xebc2,1,0xe6b2ba00,0,0x9f9f
,1,0xe8a38a00,0,0xee8a,1,0xe9899000,0,0xe7ef,1,0x78000000,4980,0x78
,1,0xeaadbc00,0,0xee8c,1,0xe5928c00,0,0x9861,1,0xefbd9d00,1033,0x8170
,1,0xefa9a000,0,0xee8f,1,0xe8a4a700,5314,0xee93,1,0xe8a4b000,4906,0xee94
,1,0xe285a500,1744,0x8759,1,0xe8a4b200,0,0xee95,1,0xe98f8300,0,0xe856
,1,0xe8a4b900,0,0xee96,1,0xe8b9b200,0,0xe74c,1,0xe4b88800,0,0x8fe4
,1,0xe4bd9b00,0,0x98c5,1,0xe8a69400,0,0xee98,1,0xefa9a100,5183,0xee99
,1,0xe6b98d00,0,0x9fd8,1,0xe6bea000,0,0xec52,1,0xe883a100,0,0x8cd3
,1,0xe7a8bd00,8632,0x8c6d,1,0xe8a79400,0,0xee9a,1,0xe8bfa500,3336,0xe78a
,1,0xe8a7a500,0,0xee9b,1,0xe5a38500,0,0x9ad7,1,0xc2aa0000,730,0x8546
,1,0xe5adab00,0,0x91b7,1,0xe6bf8000,1210,0x8c83,1,0xe8a8a200,5479,0xee9f
,1,0xe7a99d00,0,0xed6e,1,0xe88ea700,0,0xedf7,1,0xe380bb00,8863,0x81b4
,1,0xe8a8b700,0,0xeea0,1,0xcc9f0000,0,0x868b,1,0xe9afb500,0,0x88b1
,1,0xe5ae8b00,0,0x9176,1,0xe8a98700,6396,0xeea1,1,0xe698a800,0,0x8df0
,1,0xe69dbb00,0,0xeb72,1,0xe88f8700,0,0xedfb,1,0xe3819b00,45,0x82b9
,1,0xe8a98e00,5312,0xeea2,1,0xe8a9b500,0,0xeea5,1,0xe9b09500,0,0xe9cf
,1,0xe8a9b900,0,0xeea6,1,0xe9babb00,5398,0x9683,1,0xe6998800,0,0xeb57
,1,0xe69e9b00,6159,0xeb76,1,0xe6a3ae00,0,0x9058,1,0xe7b58300,7311,0x8cbc
,1,0x61000000,2350,0x61,1,0xe7bfa900,2769,0xe3c7,1,0xe8abb600,0,0xeeab
,1,0xefa8a200,9912,0xeeac,1,0xe5889300,413,0x87da,1,0xe58da600,0,0x8c54
,1,0xefa9a200,5157,0xeead,1,0xe6a48e00,0,0x92c5,1,0xe6a9a100,1903,0x93c9
,1,0xe6aeb400,4144,0x89a3,1,0xe280bf00,0,0x8677,1,0xe9858a00,0,0xe7c2
,1,0xe8aaa600,0,0xe675,1,0xefa9a300,0,0xeeae,1,0xe8ad8600,0,0xeeaf
,1,0xe8ad9400,0,0xeeb0,1,0xe8ad9900,6308,0xeeb1,1,0xe59dbf00,0,0x9ab2
,1,0xe6af9400,7433,0x94e4,1,0xe7949e00,6913,0xe152,1,0xe799b100,0,0xecdd
,1,0xe8ada900,0,0xeeb2,1,0xe9909000,0,0xe860,1,0xe8ae9d00,9490,0xeeb3
,1,0xe99ab600,0,0xe8ae,1,0xe5998c00,2688,0x9158,1,0xe8b18900,0,0xeeb4
,1,0xe5a3b200,0,0x9484,1,0xefa9a400,0,0xeeb6,1,0xe2879200,0,0x81cb
,1,0xe8b3b400,7745,0xeeb8,1,0xe8b3b800,10359,0xeeb9,1,0xe9968300,0,0x914d
,1,0xe8bb9f00,2761,0x93ee,1,0xe8b3be00,5893,0xeeba,1,0xefa9a500,0,0xeebb
,1,0xe6848900,0,0x96f9,1,0xe8b49200,5849,0xeebc,1,0xe5aeb800,0,0x9b82
,1,0xe8b49b00,6102,0xeebd,1,0xe8858e00,2122,0x9074,1,0xe7aaaa00,4546,0x8c45
,1,0xe88fb400,0,0xe4bd,1,0xe8b6af00,4996,0xeebe,1,0xe8b78e00,0,0xeebf
,1,0xe9abaf00,0,0xe999,1,0xe8b79100,10140,0xeec0,1,0xc2b10000,623,0x817d
,1,0xe694a200,6001,0xeb45,1,0x4a000000,111,0x4a,1,0xe7ab8a00,0,0xe286
,1,0xe8b79700,0,0xeec1,1,0xe382a800,8262,0x8347,1,0xe8b8a000,0,0xeec2
,1,0xe8b8a300,0,0xeec3,1,0xe8b9b000,0,0xeec5,1,0xe583ad00,11003,0x9947
,1,0xe5b58b00,0,0x9bc9,1,0xe8b9bb00,0,0xeec6,1,0xe5bfb100,0,0x9c79
,1,0xeaae8c00,5663,0xeec7,1,0xe3838800,2656,0x8367,1,0xe8bb8000,0,0xeec8
,1,0xe4a18400,0,0xeec9,1,0xe985b700,0,0x8d93,1,0xe8bbba00,6939,0xeeca
,1,0xe8bc9e00,0,0xeecb,1,0xe58eb300,0,0x8cb5,1,0xe8bcad00,6413,0xeecc
,1,0xe7859200,0,0xec75,1,0xe6aaae00,0,0x9e8d,1,0xe78fb800,0,0xe0e7
,1,0xe38e8e00,1721,0x8772,1,0xe9869700,0,0x94ae,1,0xe98baa00,2865,0x95dd
,1,0xeaae8f00,0,0xeecf,1,0xe8beb500,0,0xeed1,1,0xe8bfa400,5058,0xeed2
,1,0xeaacb800,0,0xed48,1,0xe9808800,0,0xeed5,1,0xc4920000,0,0x8598
,1,0xe282ac00,805,0x8540,1,0xe8a78000,0,0xe656,1,0xe980ad00,0,0xeed6
,1,0xefa9a700,5369,0xeed7,1,0xe9828800,6165,0xeed8,1,0xe5958600,0,0x8fa4
,1,0xe59a9900,914,0x8859,1,0xe4bfb500,0,0x9555,1,0xe6b18100,0,0x8f60
,1,0xe7968b00,3995,0x9544,1,0xe2889f00,2044,0x8798,1,0xe880a800,0,0xe3d3
,1,0xe9829500,0,0xeed9,1,0xe9829900,5009,0xeedb,1,0xe489a400,10231,0xed8d
,1,0xe4bb8200,0,0x98bc,1,0xe9829b00,0,0xeedc,1,0x33000000,5813,0x33
,1,0xe982a200,0,0xeedd,1,0xe982b300,9225,0xeede,1,0xe7a19100,0,0xed43
,1,0xe293a500,0,0x86d4,1,0xe982be00,0,0xeedf,1,0xe8bd8c00,0,0xe779
,1,0xe9838500,0,0xeee1,1,0xe9839700,0,0xeee3,1,0xe9acbc00,0,0x8b53
,1,0xe68b8900,0,0x9d66,1,0xe6909c00,0,0x9d72,1,0xe9839d00,0,0xeee4
,1,0xe9839e00,0,0xeee5,1,0xe88c8e00,0,0x8c73,1,0xe891a100,0,0x9592
,1,0xe383b500,2303,0x8395,1,0xe983af00,0,0xeee6,1,0xe983b400,0,0xeee7
,1,0xe9849400,0,0xeee9,1,0xe5b18500,961,0x8b8f,1,0xc2b80000,0,0x854e
,1,0xc9900000,0,0x8652,1,0xe780ac00,0,0x90a3,1,0xe9849500,8836,0xeeea
,1,0xe7b79d00,0,0xe368,1,0xe984a200,7054,0xeeec,1,0xe984a300,0,0xeeed
,1,0xe984a700,6892,0xeeee,1,0xe5859a00,0,0x937d,1,0xe58aad00,11172,0x99a1
,1,0xe5bc8b00,0,0x9c54,1,0xe7818c00,6635,0x9ff3,1,0xe7869f00,0,0x8f6e
,1,0xe6abbb00,0,0x9f4e,1,0xe7bd9000,0,0xe3a3,1,0xe9829100,0,0x9757
,1,0xe984af00,4931,0xeeef,1,0xe984bd00,0,0xeef2,1,0xe58b8d00,3674,0x99a5
,1,0xe590a000,0,0x9669,1,0xe9858800,5371,0xeef3,1,0xe9859b00,0,0xeef4
,1,0xe6ac9b00,7363,0xebba,1,0xe9869e00,0,0xeef6,1,0xe796b800,0,0xe174
,1,0xe986b100,0,0xeef8,1,0xe986bc00,10018,0xeef9,1,0xe8b2b300,6022,0xe6c6
,1,0x1c000000,1000,0x1c,1,0xefbc9100,0,0x8250,1,0xe9879700,10065,0xeefa
,1,0xe987bb00,5734,0xeefb,1,0xe987a400,7475,0xeefc,1,0xc4990000,429,0x85cb
,1,0xe987ad00,0,0xef41,1,0xe8ae8000,8480,0xe6a4,1,0xe8b39300,0,0x956f
,1,0xe9888700,9702,0xef43,1,0xe988b900,0,0xef46,1,0xe4bc8f00,10933,0x959a
,1,0xe6819000,592,0x8bb0,1,0xe5a6ac00,0,0x9369,1,0xe68bb600,0,0x8e41
,1,0xe6bd9400,0,0x8c89,1,0xe9898000,7818,0xef49,1,0xe887a800,0,0x97d5
,1,0xe9898300,0,0xef4a,1,0xe99e9000,0,0xe8de,1,0xe9898f00,9344,0xef4b
,1,0xe989b800,10588,0xef4c,1,0xe6878300,0,0x9ce7,1,0xe98a8800,0,0xef4d
,1,0xe691a900,0,0x9680,1,0xe696bc00,0,0x8997,1,0xe8888800,7907,0x8bbb
,1,0xe98b8200,8044,0xef4e,1,0xe98b8b00,8174,0xef4f,1,0xe9a48300,0,0xe94c
,1,0xe98b9300,0,0xef51,1,0xe98ba000,0,0xef52,1,0xe98c8400,9369,0xef54
,1,0xe98c9f00,0,0xef55,1,0xe5b7a500,0,0x8d48,1,0xe5bcb800,0,0x9c59
,1,0xe98ca500,10517,0xef57,1,0xe3808f00,8201,0x8178,1,0xe98d8800,8982,0xef58
,1,0xe89db400,7252,0xe59a,1,0xe9af8900,4883,0x8cef,1,0xe98d8900,7463,0xef59
,1,0xe98d8a00,5010,0xef5a,1,0xe58bba00,0,0x8ed9,1,0xc2bf0000,29,0x8555
,1,0xc9970000,307,0x8645,1,0xe787ac00,3897,0xe09b,1,0xe8998100,53,0x8888
,1,0xe38b9500,2361,0x86de,1,0x5000000,4891,0x5,1,0xe8a8ba00,0,0x9066
,1,0xe98e9b00,0,0xef5f,1,0xccb40000,984,0x8695,1,0xe98ea300,0,0xef60
,1,0xefbcbe00,0,0x814f,1,0xe6a89500,0,0xeba6,1,0xe98f9e00,0,0xef63
,1,0xe6b2bb00,0,0x8ea1,1,0xe8a48700,0,0x95a1,1,0xe98f9f00,9870,0xef64
,1,0xe9908400,0,0xef65,1,0xe98fbd00,0,0xef66,1,0xe990b300,5500,0xef67
,1,0xe991a300,0,0xef69,1,0xe59cb300,244,0x8864,1,0xe991b100,0,0xef6b
,1,0xe2809300,1714,0x829c,1,0xe285a600,1748,0x875a,1,0xe9969f00,0,0xef6e
,1,0xe996a600,8144,0xef6f,1,0xe996a900,6106,0xef70,1,0xe999aa00,0,0x9486
,1,0xe4b88900,0,0x8e4f,1,0xe4bd9c00,0,0x8dec,1,0xe6829d00,0,0xeacf
,1,0xe996ac00,0,0xef71,1,0xe6b98e00,0,0x9fdc,1,0xe6bea100,0,0xe055
,1,0xe9978b00,10148,0xef74,1,0xe888b500,2142,0x91c7,1,0xe99a8a00,2979,0x91e0
,1,0xe8bfa600,0,0x89de,1,0xe9979000,6196,0xef75,1,0xe4a6b000,0,0xef77
,1,0xc7bd0000,352,0x8664,1,0xce950000,0,0x83a3,1,0xe6bf8100,0,0x91f7
,1,0xe8848200,0,0x8e89,1,0xe9979a00,0,0xef78,1,0xe7aeb100,2558,0x94a0
,1,0xe380bc00,0,0x81b5,1,0xe9a59000,8673,0xe95d,1,0xe9999800,0,0xef7a
,1,0xe99a8400,0,0xef7b,1,0xe5ae8c00,393,0x8aae,1,0xefa79c00,0,0xef7c
,1,0xe99aa400,0,0xef7e,1,0xe69dbc00,0,0x9e60,1,0xe99aa500,6011,0xef80
,1,0xe3819c00,8870,0x82ba,1,0xe99b9200,0,0xef81,1,0xefa9a800,5099,0xef83
,1,0xe99baf00,10808,0xef85,1,0xefa89f00,10961,0xee59,1,0xe9babc00,5401,0x9bf5
,1,0xe6998900,0,0x9de7,1,0xe69e9c00,0,0x89ca,1,0xe6a3af00,0,0x9eaa
,1,0xe7b58400,7315,0x9167,1,0xe99d8e00,0,0xef89,1,0xe38ca200,0,0x8761
,1,0xe99d9a00,0,0xef8b,1,0xe989be00,3526,0x9667,1,0xe5889400,3616,0x9985
,1,0xe99dae00,5453,0xef8c,1,0xe5bf8500,5529,0x954b,1,0xe6a48f00,7240,0x9e93
,1,0xc99e0000,73,0x8651,1,0xe6aeb500,0,0x9269,1,0xe793bf00,9181,0xecc4
,1,0xe9858b00,0,0x8f55,1,0xe8aaa700,0,0xeea7,1,0xe99e9500,8250,0xef8e
,1,0xe99eae00,0,0xef8f,1,0xccbb0000,0,0x869c,1,0xe4b8b600,6310,0x98a4
,1,0xe99eba00,9149,0xef90,1,0xe99f8100,8325,0xef91,1,0xe7949f00,0,0x90b6
,1,0xe799b200,0,0xe19f,1,0xe8ab8700,0,0x926b,1,0xe99f9b00,0,0xef94
,1,0xe8b5ad00,0,0xe6de,1,0xe99ab700,0,0x97ea,1,0xe4b99600,6759,0x98a8
,1,0xe59ea000,0,0x9ab5,1,0xe99fb400,5212,0xef95,1,0xefa9a900,0,0xef96
,1,0xe9a09e00,0,0xef98,1,0xe79fa500,1815,0x926d,1,0xe884af00,0,0xe3fb
,1,0xefa9aa00,0,0xef9b,1,0xefa79000,0,0xefa2,1,0xe9a1a500,0,0xefa3
,1,0xe4bf8900,1097,0x87b7,1,0xe9a1ac00,0,0xefa4,1,0xe5a9a600,0,0x9577
,1,0xe5aeb900,0,0x9765,1,0xe9a2ba00,7768,0xefa5,1,0xc4a70000,7852,0x85fb
,1,0xe9a3a700,0,0xefa7,1,0xe9a88200,8898,0xefaa,1,0xe9a88300,0,0xefab
,1,0xe9a8a400,0,0xefac,1,0xe9a8ad00,0,0xefad,1,0xe9a8ae00,0,0xefae
,1,0xe68f9000,0,0x92f1,1,0xce9c0000,1487,0x83aa,1,0xe699b600,1251,0x8fbb
,1,0xe7ab8b00,0,0x97a7,1,0xe7b09e00,8789,0xed89,1,0xe382a900,519,0x8348
,1,0xe9a98a00,0,0xefb0,1,0xe9a98e00,10659,0xefb1,1,0xe9a99200,5136,0xefb2
,1,0xe583ae00,0,0x9949,1,0xe5b58c00,0,0x9bc6,1,0xe69a9600,0,0x9267
,1,0xe69fa900,0,0x9e6c,1,0xe9ab8100,0,0xefb4,1,0xe3838900,2642,0x8368
,1,0xe89b9b00,0,0xe581,1,0xe9ab8300,9194,0xefb5,1,0xe985b800,0,0x8e5f
,1,0xe9ac8200,11027,0xefb9,1,0xe589a100,0,0x87de,1,0xe9ac8800,5459,0xefba
,1,0xe4b09700,7141,0xefbc,1,0xe6a59c00,7269,0x9eb3,1,0xe9acad00,5974,0xefbd
,1,0xe9adb900,7445,0xefbf,1,0xe38e8f00,441,0x8773,1,0xe8a6a100,0,0xe64d
,1,0xe8abb400,0,0xeeaa,1,0xe9ae8a00,0,0xefc4,1,0xe58f9400,0,0x8f66
,1,0xe9ae9e00,0,0xefc6,1,0xe599ba00,7835,0x94b6,1,0xe78b8600,0,0xe0bc
,1,0xc9a50000,0,0x865b,1,0xe9aea700,7565,0xefc7,1,0xe79abf00,0,0x8e4d
,1,0xe8ac9400,0,0xe682,1,0xe9919e00,0,0xe86c,1,0xe8b6ba00,0,0xe6e5
,1,0xe9af8100,5541,0xefc8,1,0xe9afa500,5649,0xefca,1,0xe59fad00,937,0x8871
,1,0xe6b18200,3758,0x8b81,1,0xe6b69500,7576,0x9fb6,1,0xe288a000,732,0x81da
,1,0xe7a0b200,0,0x9643,1,0xe9afbd00,0,0xefcc,1,0xe8b79a00,0,0xe6e7
,1,0xe99ca400,4848,0xe8c3,1,0xe9b08000,5258,0xefcd,1,0xe9b19000,0,0xefd1
,1,0xe5a5a000,0,0x9af9,1,0xe9b19300,5861,0xefd2,1,0xe9b1a300,0,0xefd3
,1,0xe9b49d00,0,0xefd6,1,0xe293a600,711,0x86d5,1,0xe7abb800,0,0x995e
,1,0xe8bd8d00,3304,0x9351,1,0xe9b49e00,0,0xefd7,1,0xe9b58300,0,0xefd8
,1,0xe9b58700,10684,0xefd9,1,0xe68b8a00,0,0x9d62,1,0xe9b5a300,9754,0xefdb
,1,0xe695b000,0,0x9094,1,0xe9b5b000,0,0xefdc,1,0xe7ac9800,0,0xe298
,1,0xe891a200,0,0xe4e2,1,0xe383b600,26,0x8396,1,0xe9b5bc00,6752,0xefdd
,1,0xe9b68a00,6043,0xefde,1,0xe9b69600,5359,0xefdf,1,0xe584bb00,1194,0x9957
,1,0xe6969000,6045,0x94e3,1,0xcea30000,1262,0x83b0,1,0xe9b6ac00,6161,0xefe1
,1,0xe7b28b00,2577,0x9088,1,0xe7b79e00,2674,0xe36a,1,0xeaaf8800,0,0xefe4
,1,0xe981b200,0,0xe7ad,1,0xe9b7a700,0,0xefe5,1,0xe9b88700,0,0xefe6
,1,0xe9bdb600,0,0xea9b,1,0xe5bc8c00,943,0x989f,1,0xe9b89500,0,0xefe7
,1,0xe9b9bc00,7274,0xefe8,1,0xe9ba9e00,7354,0xefe9,1,0xe7bd9100,0,0xe3a4
,1,0xe9baa400,0,0xefea,1,0xe987a500,0,0xef40,1,0xe9baac00,5330,0xefeb
,1,0xe9baaf00,0,0xefec,1,0xe9bab400,0,0xefed,1,0xe9bab500,0,0xefee
,1,0xe9bb9100,0,0xeff0,1,0xe9bcb900,0,0xeff2,1,0xe9be9000,0,0xeff4
,1,0xe796b900,0,0x905d,1,0xe8a88e00,0,0x93a2,1,0xe9be9400,0,0xeff5
,1,0xe8b2b400,0,0x8b4d,1,0xe5a7b800,0,0xeff8,1,0xefbc9200,0,0x8251
,1,0xe4bbb000,10104,0x8bc2,1,0xe5a0ba00,1295,0x8de4,1,0xe7928600,0,0xecb4
,1,0xc9ac0000,1987,0x85e8,1,0xe5b9b700,7064,0xeffa,1,0xe881b600,9397,0xe3e1
,1,0xe798a600,0,0xeffb,1,0xe7b9ab00,9537,0xeffc,1,0xe99db100,0,0xe8d5
,1,0xe4bc9000,10957,0x94b0,1,0xe5a19a00,0,0x92cb,1,0xe686a400,0,0x95ae
,1,0xe68bb700,0,0x8d89,1,0xe4b88200,0,0xf041,1,0xe8829600,7820,0x8fd1
,1,0xe294b300,0,0x84b1,1,0xe8b98700,8744,0xe740,1,0xe4b88f00,5715,0xf042
,1,0xe4b89200,5735,0xf043,1,0xe9a8b700,0,0xe97a,1,0xe4b8a900,5821,0xf044
,1,0xe4b8ab00,5309,0xf045,1,0xe4b8ae00,6149,0xf046,1,0xe696bd00,0,0x8e7b
,1,0xe8888900,0,0x9da8,1,0xe7ada500,0,0xe2a6,1,0xe892af00,0,0xee4d
,1,0xe4b98000,5777,0xf047,1,0xe9a99700,0,0xe984,1,0xe9aeaa00,0,0x968e
,1,0xe5ad8000,0,0x9b71,1,0xe4b98700,0,0xf048,1,0xe5b7a600,0,0x8db6
,1,0xe4b98800,0,0xf049,1,0xeaaaa300,0,0xf04a,1,0xe3809000,11495,0x8179
,1,0xe7b8ab00,2676,0x9644,1,0xe4b99100,6621,0xf04b,1,0xe9af8a00,0,0xe9c0
,1,0xe5819500,5137,0x98f1,1,0xe586a800,0,0x9579,1,0xe58bbb00,593,0x87e8
,1,0xe69d9000,1276,0x8dde,1,0xe6a2a300,0,0xeb86,1,0xe787ad00,3899,0x9043
,1,0xe7b98b00,0,0x8c71,1,0xe38b9600,2359,0x86df,1,0xe8a3a800,9734,0xe5e9
,1,0xe8a8bb00,0,0x9290,1,0xe3908600,0,0xf04c,1,0xeaaaa400,0,0xf04d
,1,0xe4ba9d00,8269,0xf050,1,0xefbcbf00,6669,0x8151,1,0xe7888d00,0,0xe0a1
,1,0xe6ada900,0,0x95e0,1,0xe6b2bc00,0,0x8fc0,1,0xe390ac00,0,0xf051
,1,0xe8a99b00,0,0xe666,1,0xe390ae00,0,0xf052,1,0xe4babb00,5659,0xf054
,1,0xe5928e00,0,0x99e9,1,0xeaaaa500,0,0xf055,1,0xe59cb400,62,0x8865
,1,0xe6ae8900,0,0x8f7d,1,0xe2809400,6894,0x815c,1,0xe285a700,190,0x875b
,1,0xe79db900,0,0xe1cf,1,0xe4babc00,8993,0xf056,1,0x6d000000,0,0x6d
,1,0xe8b9b400,0,0x8f52,1,0xe4b88a00,5682,0x8fe3,1,0xe4bd9d00,0,0x98c6
,1,0xe6829e00,0,0xead0,1,0xe4bb8300,7986,0xf057,1,0xe7998600,7027,0xe194
,1,0xe28b9a00,11497,0x8496,1,0xe7a3ac00,0,0xe1fc,1,0xe7a8bf00,578,0x8d65
,1,0xe99a8b00,2983,0xe440,1,0xe99f9e00,0,0xef93,1,0xe4bbab00,8241,0xf05a
,1,0xe5a38700,0,0x9264,1,0xe6889100,1151,0x89e4,1,0xe4bb9a00,9627,0xf05b
,1,0xe6bf8200,0,0xe051,1,0xe4bc8000,0,0xf05e,1,0xe296a000,7936,0x81a1
,1,0xe88ea900,0,0xedf8,1,0xe380bd00,2620,0x81b6,1,0xe9a59100,0,0xe95f
,1,0xe4bc9600,9144,0xf05f,1,0xe4bcb700,10806,0xf061,1,0xe5ae8d00,6335,0x8eb3
,1,0xe5b3a000,6397,0x93bb,1,0xe5b8b300,6484,0x92a0,1,0xe2978000,0,0x8243
,1,0xe29c9300,8763,0x849b,1,0xe3819d00,0,0x82bb,1,0xe4bd9400,8441,0xf063
,1,0xe9ab8400,0,0x9091,1,0xe4bd9800,0,0xf064,1,0xeaaaa600,9006,0xf065
,1,0xe587b500,11094,0x9981,1,0xe4bdb700,0,0xf066,1,0xe69e9d00,0,0x8e7d
,1,0xe6a3b000,0,0xeb8d,1,0xe788ba00,2256,0x96ea,1,0xe3879000,8378,0x82f5
,1,0xe38ca300,107,0x876b,1,0xe4bdb800,5686,0xf067,1,0xe5838200,3763,0x9943
,1,0xe5889500,3617,0x87db,1,0xe4bdba00,0,0xf068,1,0xe4bdbd00,5690,0xf069
,1,0xe4be9200,7249,0xf06c,1,0xceb10000,0,0x83bf,1,0xe4be9a00,0,0xf06d
,1,0x56000000,0,0x56,1,0xe9858c00,0,0x8ede,1,0xe8aaa800,9930,0xe671
,1,0xe4bfa600,0,0xf06e,1,0xe4bf8500,0,0xf071,1,0xe4bf8f00,0,0xf073
,1,0xe391aa00,6331,0xf075,1,0xe4bfb200,6069,0xf076,1,0xe78f8d00,3943,0x92bf
,1,0xe6b4a900,7543,0x896b,1,0xe5808000,0,0xf077,1,0xe580a200,0,0xf07c
,1,0xe391a800,7510,0xf07d,1,0xe5818200,0,0xf07e,1,0xe99ab800,0,0xe8af
,1,0xe4b99700,1051,0x8fe6,1,0xc3800000,0,0x8556,1,0xe5818600,5778,0xf080
,1,0xe688be00,7557,0xeae1,1,0xe2879400,824,0x81cc,1,0xe6bfaf00,7750,0x91f3
,1,0xe5818e00,0,0xf081,1,0xe5819300,0,0xf082,1,0xe5819700,10203,0xf083
,1,0xe9a0ab00,0,0xef99,1,0xe4bf8a00,0,0x8f72,1,0xe581a300,5788,0xf084
,1,0xe5a9a700,1393,0x8893,1,0xe581b000,8446,0xf087,1,0xe5828800,7772,0xf089
,1,0xc9ba0000,637,0x8660,1,0xd0920000,4549,0x8442,1,0xe5829200,0,0xf08a
,1,0xe9a18b00,0,0xe8f9,1,0xe9a69e00,0,0xefa9,1,0xe9abb100,0,0xe99d
,1,0xe5829300,0,0xf08b,1,0xe5829500,6666,0xf08c,1,0xe694a400,0,0x9db8
,1,0xe699b700,0,0xeb5f,1,0xe5829600,7976,0xf08d,1,0xe7b09f00,0,0xe2d0
,1,0xe382aa00,521,0x8349,1,0xe5829c00,0,0xf08e,1,0xe582aa00,0,0xf08f
,1,0xeaaaa700,7613,0xf090,1,0xe582ba00,0,0xf092,1,0xe582bb00,6428,0xf093
,1,0xe69a9700,1252,0x88c3,1,0x3f000000,6183,0x3f,1,0xe6a4bd00,0,0x9ebb
,1,0xe3838a00,0,0x8369,1,0xe5838400,0,0xf094,1,0xe5838700,0,0xf095
,1,0xe583b300,9797,0xf096,1,0xe9b79700,0,0xefe3,1,0xeaaaa900,0,0xf097
,1,0xe5838e00,0,0xf098,1,0xe7808100,0,0xe066,1,0xe6a59d00,0,0x9ec0
,1,0xe78aa700,0,0xe0b8,1,0xeaaaa800,5610,0xf099,1,0xe5839400,0,0xf09a
,1,0xcb9b0000,0,0x85ab,1,0xe5839900,10992,0xf09b,1,0xe9bd8a00,5415,0xea8e
,1,0xe583a100,5469,0xf09c,1,0xe583a900,6027,0xf09d,1,0xeaacba00,0,0xed66
,1,0xe3929200,6781,0xf09e,1,0xceb80000,1690,0x83c6,1,0xe795ad00,0,0xe161
,1,0xe5848800,0,0xf140,1,0xeaaaaa00,0,0xf141,1,0xe8b1a800,0,0xeeb5
,1,0xe996b200,0,0x897b,1,0xeaaaab00,0,0xf144,1,0xe4baa400,8431,0x8cf0
,1,0xeaaaac00,0,0xf146,1,0xe392bc00,0,0xf14a,1,0xe5869800,0,0xf14b
,1,0xe586a300,0,0xf14c,1,0xe586ad00,0,0xf14d,1,0xe3938700,0,0xf14e
,1,0xe8b79b00,0,0xe6ea,1,0xe586bc00,0,0xf14f,1,0xe4bb8400,0,0x98ba
,1,0xeaaaad00,0,0xf150,1,0xc3870000,1348,0x855d,1,0xe68aab00,0,0x94e2
,1,0xe6bc8900,0,0x8d97,1,0xe8818a00,0,0xe3d6,1,0xe293a700,131,0x86d6
,1,0xe7abb900,0,0x927c,1,0xe8bd8e00,0,0xe77c,1,0xeaaaae00,0,0xf151
,1,0xe587b300,0,0xf152,1,0xe5a68100,0,0x9b41,1,0x28000000,0,0x28
,1,0xe587b400,6370,0xf153,1,0xe5888200,0,0xf154,1,0xe5889200,0,0xf155
,1,0xe7ac9900,0,0xe299,1,0xd0990000,520,0x844a,1,0xe383b700,734,0x8492
,1,0xe5889600,0,0xf156,1,0xe9ad9e00,0,0xefbe,1,0xeaaaaf00,0,0xf157
,1,0xe584bc00,0,0x9956,1,0xe6969100,0,0x94c1,1,0xe5899500,6259,0xf158
,1,0xe5899c00,0,0xf159,1,0xe589b700,0,0xf15b,1,0xe58a8200,0,0xf15d
,1,0xe89ca900,0,0xe593,1,0xeaaab000,0,0xf15e,1,0xe5808900,0,0x9171
,1,0xe5859c00,0,0x8a95,1,0xe9bdb700,5430,0xea99,1,0xe5bc8d00,0,0x98af
,1,0xe7818e00,6636,0xec61,1,0xe58a9800,0,0xf15f,1,0xeaaab100,6792,0xf160
,1,0xe89d8900,8400,0x90e4,1,0xe58aa400,0,0xf161,1,0xe987a600,0,0x9674
,1,0xe58aa600,0,0xf162,1,0xe9be9700,0,0xeff6,1,0xe58aaf00,0,0xf163
,1,0xe58b8a00,0,0xf166,1,0xe6a78a00,4098,0x9ecc,1,0xe6ac9d00,0,0x8954
,1,0xe6b1b000,0,0x91bf,1,0xe3949f00,9587,0xf167,1,0xeaaab200,0,0xf169
,1,0xe58bb700,0,0xf16a,1,0xe58c8a00,5668,0xf16b,1,0xe5918200,0,0x9843
,1,0xefbc9300,0,0x8252,1,0xe4bbb100,5665,0xf05c,1,0xe58c8b00,0,0xf16c
,1,0xe6b29000,4203,0x9f94,1,0xcebf0000,0,0x83cd,1,0xe58cb500,7084,0xf16e
,1,0xe881b700,0,0x9045,1,0xe58cbe00,6095,0xf16f,1,0xeaaab300,0,0xf171
,1,0x11000000,0,0x11,1,0xe4bc9100,1085,0x8b78,1,0xe6819200,0,0x8d50
,1,0xeaaab400,5818,0xf172,1,0xe58da700,7875,0xf173,1,0xe79d8d00,0,0xecf1
,1,0xe58ea400,0,0xf176,1,0xe294b400,251,0x84a8,1,0xe8b98800,0,0xe744
,1,0xe8be9b00,0,0x9068,1,0xe58eb400,9153,0xf177,1,0xe9a8b800,0,0xefaf
,1,0xeaaab500,0,0xf178,1,0xc38e0000,3769,0x8564,1,0xe58eb700,0,0xf179
,1,0xeaaab600,0,0xf17b,1,0xe8888a00,0,0xe470,1,0xe3959e00,8045,0xf17d
,1,0xe7b2b900,0,0xe2ed,1,0xe9a48500,0,0x96dd,1,0xe58f9500,7743,0xf17e
,1,0xe9aeab00,0,0x8e4c,1,0xe580b600,890,0x8be4,1,0xe5b29400,0,0x9ba9
,1,0xe5b7a700,0,0x8d49,1,0xe69cb100,0,0x8ee9,1,0xe7ae8600,0,0x95cd
,1,0xe3809100,2601,0x817a,1,0xd0a00000,8358,0x8451,1,0xe89db600,0,0x92b1
,1,0xe395a300,9381,0xf181,1,0xe5819600,5141,0x98f5,1,0xe586a900,0,0x996f
,1,0xe58fb400,0,0xf182,1,0xe69d9100,0,0x91ba,1,0xe58fb500,6673,0xf183
,1,0xe787ae00,0,0x99d7,1,0xe590a800,0,0xf186,1,0xe38b9700,0,0x86e0
,1,0xe395ae00,0,0xf187,1,0xe8a8bc00,9843,0x8fd8,1,0xe5878900,3589,0x9979
,1,0xe58c9c00,0,0x87eb,1,0xe591a200,0,0xf189,1,0xe6a38400,0,0x8afc
,1,0xe6a89700,0,0x9294,1,0xe6adaa00,0,0x9863,1,0xe6b2bd00,0,0x9f98
,1,0xe591a600,5921,0xf18a,1,0xe591ac00,0,0xf18b,1,0xe5928a00,0,0xf18c
,1,0xe5928d00,0,0xf18d,1,0xe5928f00,0,0x99e7,1,0xeaaaa100,5602,0xf040
,1,0xe4bcbe00,0,0xf062,1,0xe6ae8a00,6199,0x8eea,1,0xe6b39d00,1947,0x9f9b
,1,0xe285a800,1999,0x875c,1,0xe79dba00,0,0xecf6,1,0xe98f8600,0,0xef62
,1,0xcba90000,0,0x8684,1,0xe999ac00,0,0xe8a3,1,0xefbe8000,61,0xc0
,1,0xe4bd9e00,0,0x9b43,1,0xe6829f00,697,0x8ce5,1,0xe5a7bb00,456,0x88f7
,1,0xe7998700,0,0xe192,1,0xe28b9b00,1811,0x8497,1,0xe883a400,0,0x88fb
,1,0xe888b700,0,0x8cbd,1,0xe592a000,0,0xf18f,1,0xe8bfa800,0,0xeed3
,1,0xe592ad00,10477,0xf191,1,0xe592ae00,0,0xf192,1,0xe6889200,0,0x89fa
,1,0xe68da500,0,0xeaee,1,0xe6bf8300,0,0x945a,1,0xe592b700,0,0xf193
,1,0xe296a100,7938,0x81a0,1,0xe88eaa00,0,0xe4ae,1,0xe592ba00,8867,0xf194
,1,0xe9a59200,0,0xe960,1,0xe592bf00,0,0xf195,1,0xe9afb800,0,0xefcb
,1,0xeaaab700,0,0xf197,1,0xc3950000,134,0x856b,1,0xe698ab00,0,0xeb52
,1,0xe2978100,305,0x8242,1,0xe88f8a00,0,0x8b65,1,0xe3819e00,0,0x82bc
,1,0xe7b9b900,0,0xe388,1,0xe5958100,0,0xf1a0,1,0xe5958700,9714,0xf1a2
,1,0xe582a300,9046,0xf088,1,0xe587b600,1686,0x8ba5,1,0xe6998b00,1247,0x9057
,1,0xe5bea700,0,0xeac0,1,0xe6a3b100,0,0xeb8e,1,0xe788bb00,2265,0xe0a9
,1,0xe3879100,0,0x82f6,1,0xd0a70000,4504,0x8458,1,0xe984ad00,2817,0x9341
,1,0xefa98100,0,0xeb47,1,0xe9bb9e00,5407,0xea79,1,0xe58da900,0,0x99c5
,1,0xe592bc00,6169,0x9a44,1,0xe396a800,7250,0xf1a4,1,0xe7899b00,6758,0x8b8d
,1,0xe6aeb700,0,0x9f75,1,0xe595a000,9183,0xf1a5,1,0xe8a59600,572,0x89a6
,1,0xe595a100,0,0xf1a6,1,0xe595a400,10158,0xf1a7,1,0xeaaab800,0,0xf1a8
,1,0xe5968200,8600,0xf1aa,1,0xe4b8b800,1058,0x8adb,1,0xe6aa8400,0,0x9efa
,1,0xe78f8e00,3945,0xe0dd,1,0xe6b4aa00,0,0x8d5e,1,0xe5968800,0,0xf1ab
,1,0xe5969100,9943,0xf1ac,1,0xe9909300,0,0xe85d,1,0xe3978500,0,0xf1ad
,1,0xe99ab900,0,0xe8b0,1,0xe4b99800,6783,0x98a9,1,0xe59ea200,1143,0x8d43
,1,0xe5979200,0,0xf1ae,1,0xe688bf00,0,0x965b,1,0xe6ba9d00,0,0x8d61
,1,0xe79fa700,4059,0x948a,1,0xe7a4ba00,426,0x8ea6,1,0xeaaaba00,10125,0xf1af
,1,0xcbb00000,494,0x8686,1,0xe9a0ac00,0,0x966a,1,0xe4bf8b00,0,0xf072
,1,0xe5a49500,0,0x975b,1,0xeaaabb00,0,0xf1b0,1,0xe68eb200,0,0x8c66
,1,0xe596bf00,0,0xf1b1,1,0xe7a59a00,2429,0xe24e,1,0xe88aa400,0,0xeddf
,1,0xe88fb700,0,0xe4c6,1,0xe9a18c00,0,0x91e8,1,0xe5978900,0,0xf1b2
,1,0xe5978c00,0,0xf1b3,1,0xe5979100,0,0xf1b4,1,0xe5af9b00,1438,0x8ab0
,1,0xe5979d00,0,0xf1b5,1,0xe699b800,0,0xeb60,1,0xe2988e00,2167,0x83e5
,1,0xe7b0a000,8795,0xed8a,1,0xe382ab00,646,0x834a,1,0xe3979a00,0,0xf1b6
,1,0xe597a200,10660,0xf1b7,1,0xe9b1a500,0,0xefd4,1,0xe9b6b800,0,0xea53
,1,0xe6958500,1218,0x8ccc,1,0xc39c0000,1532,0x8571,1,0xeaaab900,0,0xf1b8
,1,0xe7b18000,0,0xe2d8,1,0xe3838b00,2282,0x836a,1,0xe597a900,7447,0xf1b9
,1,0xe980a700,0,0xe79a,1,0xe598a800,6266,0xf1ba,1,0xeaaabc00,0,0xf1bb
,1,0xe589a300,0,0x8c95,1,0xe58eb600,1253,0x99d1,1,0xe5988700,5716,0xf1bc
,1,0xe6a59e00,1865,0x9ebf,1,0xe5989000,5832,0xf1bd,1,0xe598b000,6338,0xf1be
,1,0xe9818700,0,0x8bf6,1,0xd0ae0000,0,0x845f,1,0xe98bad00,4782,0x8973
,1,0xe58a8300,2131,0x8a63,1,0xe58f9600,0,0x8ee6,1,0xe5998000,0,0xf1c3
,1,0xe5998700,6717,0xf1c4,1,0xe6ab9100,0,0x9f49,1,0xe6b0a400,4168,0x9f85
,1,0xe6b5b700,0,0x8a43,1,0xe599a000,7295,0xf1c6,1,0xe8ac9600,9994,0xe68c
,1,0xe991a000,9420,0xe86a,1,0xe599ad00,7651,0xf1c7,1,0xe3988500,0,0xf1c8
,1,0xe4baa500,8505,0x88e5,1,0xe4bfb800,0,0x95ee,1,0xe59a8800,8153,0xf1c9
,1,0xe7968e00,0,0x9161,1,0xe79ba100,0,0xe1b6,1,0xe7a0b400,0,0x946a
,1,0xe8b28900,0,0xe6b9,1,0xe9979300,0,0xef76,1,0xe8bcaf00,0,0x8f53
,1,0xe59a8c00,8179,0xf1ca,1,0xe59a9500,8286,0xf1cb,1,0xe5a5a200,1349,0x9af8
,1,0xe68aac00,5862,0x9dab,1,0xe79c8100,7070,0x8fc8,1,0xe59a9a00,7473,0xf1cc
,1,0xe293a800,2130,0x86d7,1,0xe88bb100,256,0x8970,1,0xe99d8600,0,0xe8ca
,1,0xe59a9d00,8524,0xf1cd,1,0xe59aad00,8848,0xf1cf,1,0xe5a68200,0,0x9440
,1,0xe68b8c00,0,0x9d61,1,0xe5b0a800,0,0x9b98,1,0xe695b200,0,0x9dc8
,1,0xe59ab200,7698,0xf1d0,1,0xe59b8500,9411,0xf1d1,1,0xe7b1ad00,0,0xed90
,1,0xe383b800,8951,0x8493,1,0xe59b8d00,6685,0xf1d2,1,0xe59b9f00,9978,0xf1d3
,1,0xe59ba800,10108,0xf1d4,1,0xe5b18800,0,0x8bfc,1,0xe59bb600,10403,0xf1d5
,1,0xeaaabd00,0,0xf1d7,1,0xe6a0b800,0,0x8a6a,1,0xe7b28d00,2575,0x96ae
,1,0xe7b7a000,0,0x92f7,1,0xe59ca300,5963,0xf1d9,1,0xe8a1bd00,0,0xe5d1
,1,0xeaaabf00,0,0xf1da,1,0xe59ca900,0,0xf1db,1,0xeaab8100,0,0xf1dc
,1,0xe59d8500,6142,0xf1dd,1,0xc3a30000,0,0x8578,1,0xe59d8600,7278,0xf1de
,1,0xe59d8c00,0,0xf1df,1,0xe59d8d00,0,0xf1e0,1,0xeaab8000,10286,0xf1e1
,1,0xe987a700,0,0x8bfa,1,0xe98cba00,0,0xe844,1,0xe58b9000,0,0x87e3
,1,0xe59daf00,0,0xf1e3,1,0xe59db300,0,0xf1e4,1,0xe6a78b00,0,0x8d5c
,1,0xe6ac9e00,0,0xebbb,1,0xe59db400,0,0xf1e5,1,0xe59db500,0,0xf1e6
,1,0xe8a89000,4936,0xe65f,1,0xd0b50000,10025,0x8475,1,0xe8b2b600,0,0xe6c8
,1,0xe5918300,0,0xf188,1,0xefbc9400,0,0x8253,1,0xe4bbb200,10129,0x9287
,1,0xe680b300,0,0xeacb,1,0xe59dbb00,5886,0xf1e7,1,0xe7979b00,1743,0x92c9
,1,0xeaab8200,0,0xf1e8,1,0xe8ae8300,0,0x8e5d,1,0xeaab8300,7503,0xf1e9
,1,0xe59eac00,0,0xf1ea,1,0xe99db300,0,0xef8d,1,0xe59c8900,5160,0x9a9e
,1,0xe59e9a00,0,0xf1eb,1,0xe59e9d00,0,0xf1ec,1,0xe59e9e00,0,0xf1ed
,1,0xe79d8e00,7091,0xecf2,1,0xc5840000,616,0x85cf,1,0xe59f9700,0,0xf1ef
,1,0xe9998000,2959,0x91c9,1,0xe8be9c00,0,0xe783,1,0xeaab8500,6718,0xf1f0
,1,0xeaab8600,0,0xf1f2,1,0xe6878600,0,0x9ce8,1,0xe68c9900,323,0x8b93
,1,0xeaab8700,0,0xf1f3,1,0xe7a38100,6462,0x8ea5,1,0xe7a89400,0,0x96ab
,1,0xe7ada700,0,0xe2a8,1,0xe59f9e00,0,0xf1f4,1,0xe59fa600,0,0xf1f5
,1,0xe3998a00,0,0xf1f7,1,0x79000000,10723,0x79,1,0xe5a08400,0,0xf1fb
,1,0xe5a09e00,0,0xf1fc,1,0xe5b7a800,0,0x8b90,1,0xe5a0a700,0,0xf241
,1,0xe7ae8700,266,0x89d3,1,0xe3809200,721,0x81a7,1,0xe5a0b200,8983,0xf242
,1,0xe5a0b900,0,0xf243,1,0xeaab8800,0,0xf244,1,0xe9b49f00,5325,0xe9f4
,1,0xe586aa00,0,0x9970,1,0xe5a18900,0,0xf245,1,0xe5a18c00,0,0xf246
,1,0xe6a2a500,0,0xeb87,1,0xe5a28a00,0,0xf248,1,0xe7b98d00,0,0x8f4a
,1,0xe38b9800,0,0x86e1,1,0xe983a100,0,0x8c53,1,0xe988b400,0,0x97e9
,1,0xe5878a00,3586,0x87d4,1,0xe58c9d00,0,0x9178,1,0xe591b000,0,0x99ed
,1,0xe5a28d00,6671,0xf24a,1,0xc3aa0000,0,0x8580,1,0xca820000,0,0x85ef
,1,0xe6b2be00,4230,0x9f9e,1,0xe8a48a00,0,0xe5ed,1,0xe8a99d00,0,0xeea3
,1,0xe98ea700,0,0x8a5a,1,0xe5a29400,6690,0xf24d,1,0xe5929000,0,0x99f3
,1,0xefbda100,10,0xa1,1,0xe5a29d00,0,0xf24e,1,0xe6ae8b00,0,0x8e63
,1,0xe2809600,0,0x8161,1,0xe285a900,7640,0x875d,1,0xeaab8a00,0,0xf251
,1,0xe5a38300,8960,0xf252,1,0xd0bc0000,0,0x847d,1,0xe8b9b600,0,0xe74b
,1,0xefbe8100,5701,0xc1,1,0xe4bd9f00,0,0x87af,1,0xe5a2a900,773,0x887e
,1,0xe5a3a200,6221,0xf254,1,0xe7998800,7031,0xe193,1,0xe6bea400,0,0xe056
,1,0xe883a500,0,0xe3ef,1,0xe888b800,0,0xe476,1,0x62000000,3259,0x62
,1,0xe8bfa900,0,0x93f4,1,0xe5a3b300,10479,0xf255,1,0xe5a3b400,0,0xf256
,1,0xe5a89c00,0,0x9b50,1,0xe5a48500,5896,0xf257,1,0xe692b900,0,0x8a68
,1,0xe8848500,381,0x8bba,1,0xc58b0000,2145,0x85f8,1,0xe88eab00,2199,0x949c
,1,0xe5a48600,0,0xf258,1,0xe5a48b00,0,0xf259,1,0xe5a48d00,0,0xf25a
,1,0xe5a49400,0,0xf25b,1,0xe5ae8f00,0,0x8d47,1,0xe5a4a400,0,0xf25c
,1,0xcf800000,0,0x83ce,1,0xe7aa8100,594,0x93cb,1,0xeaab8b00,8089,0xf25d
,1,0xe3819f00,4616,0x82bd,1,0xe899b100,0,0xe56c,1,0xe39a9100,10016,0xf25e
,1,0xe5a4bd00,10814,0xf25f,1,0xe9b5ac00,4783,0x9651,1,0xe9babf00,0,0x969b
,1,0xe5b99500,3828,0x968b,1,0xe5bea800,0,0x9c72,1,0xe6a3b200,0,0x90b1
,1,0xe788bc00,8242,0xe0aa,1,0xe3879200,8379,0x82f7,1,0xe39a9900,0,0xf260
,1,0xe39a9600,9256,0xf262,1,0xefa98200,5354,0xeb4a,1,0xe5889700,11097,0x97f1
,1,0xeaab8c00,0,0xf263,1,0xe592bd00,0,0x88f4,1,0xe6a49200,1852,0x9ea3
,1,0xe5a59b00,0,0xf264,1,0xe5a59f00,0,0xf265,1,0xeaab8d00,10638,0xf266
,1,0xe9858e00,0,0x9291,1,0xe5a5bc00,7945,0xf269,1,0xe5a69f00,0,0xf26a
,1,0xe5a6ae00,6263,0xf26b,1,0xe5a78800,0,0xf26d,1,0xe4b8b900,5619,0x924f
,1,0xe5a78d00,6028,0xf26e,1,0xc3b10000,611,0x8587,1,0xca890000,1722,0x864c
,1,0x4b000000,884,0x4b,1,0xe5a7a700,0,0xf272,1,0xe9909400,0,0xe85c
,1,0xe8b5b000,0,0x9196,1,0xe5a7ae00,10183,0xf273,1,0xe4b99900,1053,0x89b3
,1,0xe59ea300,0,0x8a5f,1,0xeaab8e00,0,0xf274,1,0xeaab8f00,0,0xf275
,1,0xe39b8f00,7662,0xf276,1,0xe6bfb100,0,0xe05f,1,0xe5a88c00,7848,0xf277
,1,0xe9968700,0,0xe879,1,0xe8bba300,10214,0xe763,1,0xe9a0ad00,6638,0x93aa
,1,0xe59f8300,0,0x9aba,1,0xe5a49600,1339,0x8a4f,1,0xe689a000,0,0x9d4c
,1,0xe5a88d00,0,0xf278,1,0xe5a89700,0,0xf279,1,0xe5a8a700,7856,0xf27a
,1,0xe297af00,80,0x81fc,1,0xe8bc8300,0,0x8a72,1,0xe9a18d00,0,0x8a7a
,1,0xe5a99500,6277,0xf27c,1,0xe5a9a500,0,0xf27d,1,0xe68a8000,0,0x8b5a
,1,0xe5a9ba00,0,0xf27e,1,0xe5aa8b00,0,0xf280,1,0xe5aa9c00,0,0xf281
,1,0xe88b8500,4395,0x8aa1,1,0xc5920000,102,0x864a,1,0xe382ac00,0,0x834b
,1,0xe5aa9f00,0,0xf282,1,0xe5aaa000,0,0xf283,1,0xe5aaa200,6286,0xf284
,1,0xe5aab100,11005,0xf285,1,0xe5aab300,0,0xf286,1,0xe5aab500,6528,0xf287
,1,0xcf870000,551,0x83d4,1,0xe784b600,477,0x9152,1,0xe3838c00,0,0x836b
,1,0xe89b9e00,0,0xe57c,1,0xe8a0b100,0,0xe5c1,1,0xe5aaba00,0,0xf288
,1,0xe9b79900,0,0xea58,1,0xe589a400,3656,0x8ddc,1,0xe5bb8200,5463,0x9bf9
,1,0xe5ab9a00,0,0xf28a,1,0x34000000,1872,0x34,1,0xe5ab9c00,0,0xf28b
,1,0xe5aba000,0,0xf28c,1,0xe5aba500,0,0xf28d,1,0xe5abb000,0,0xf28e
,1,0xe8abb700,0,0xe685,1,0xe58a8400,0,0xf15c,1,0xe58f9700,0,0x8ef3
,1,0xe5abae00,0,0xf28f,1,0xe5abb500,7900,0xf290,1,0xe5ac9700,0,0xf293
,1,0xe5acb400,0,0xf294,1,0xe6b5b800,4247,0x905a,1,0xe5ad8c00,7476,0xf296
,1,0xe8ac9700,0,0xe68e,1,0xe8b1aa00,0,0x8d8b,1,0xe5ad9200,10138,0xf297
,1,0xe5958a00,0,0xf1a3,1,0xe4baa600,5653,0x9692,1,0xe59fb000,0,0xf1f6
,1,0xe5ada800,6243,0xf298,1,0xc3b80000,380,0x858d,1,0xca900000,7067,0x85f0
,1,0xe5adaf00,6019,0xf299,1,0xe885bf00,2912,0x91da,1,0xe8b79d00,3228,0x8b97
,1,0xe99ca700,0,0x96b6,1,0xe4bb8600,0,0x98bb,1,0xe5adbf00,0,0xf29b
,1,0xe6859a00,0,0x9cce,1,0xe5ae8100,0,0xf29c,1,0xe2898300,0,0x81ea
,1,0xe5ae8400,0,0xf29d,1,0xe293a900,0,0x86d8,1,0xeaab9100,0,0xf29e
,1,0xe5aeac00,10246,0xf0a0,1,0xe5af8000,0,0xf0a2,1,0xe9a7ad00,0,0xe96e
,1,0xe5a68300,0,0x94dc,1,0xe68b8d00,1167,0x948f,1,0xe39da200,5923,0xf0a3
,1,0xe5af8e00,0,0xf0a4,1,0xe7a78800,0,0xed64,1,0xe7ac9b00,0,0x934a
,1,0xe5af9600,8145,0xf0a5,1,0xe383b900,0,0x8494,1,0xe39dab00,0,0xf0a7
,1,0xe5afbd00,9309,0xf0a9,1,0xe39db500,9746,0xf0aa,1,0x1d000000,0,0x1d
,1,0xe5b0a900,0,0xf0ac,1,0xe69ba600,0,0x9e46,1,0xe6a0b900,1832,0x8daa
,1,0xe8928500,0,0xee4b,1,0xc5990000,2324,0x85d2,1,0xeaab9200,0,0xf0ae
,1,0xe981b500,2797,0x8f85,1,0xe5808b00,0,0x8cc2,1,0xe5b19f00,0,0xf0af
,1,0xe58ab100,0,0x97e3,1,0xe5bc8f00,0,0x8eae,1,0xc8b60000,1598,0x8668
,1,0xe5b1a300,6374,0xf0b0,1,0xe7b88100,0,0x898f,1,0xe89d8b00,2356,0x9858
,1,0xe8a29e00,4873,0xe5cf,1,0xe5b1a700,9310,0xf0b1,1,0xe98cbb00,0,0xe846
,1,0xe58b9100,0,0xf168,1,0xe590a400,0,0xf185,1,0xe5b1a800,0,0xf0b2
,1,0xe6a78c00,552,0x92c6,1,0xe6ac9f00,1916,0x9f55,1,0xe6b1b200,1939,0x8b82
,1,0xe796bc00,0,0xe175,1,0xe5b1a900,8675,0xf0b3,1,0xe98d9b00,0,0x9262
,1,0xe8b2b700,0,0x9483,1,0xe5b1b000,0,0xf0b4,1,0xefbc9500,0,0x8254
,1,0xeaab9300,10139,0xf0b5,1,0xe5a0bd00,0,0x9acc,1,0xe6b29200,4206,0x9f93
,1,0xeaab9500,7987,0xf0b6,1,0xe6bcb800,0,0x9151,1,0xe881b900,0,0xe3e2
,1,0xeaab9600,0,0xf0b8,1,0xe8b8aa00,3246,0xe748,1,0xe99db400,0,0x8c43
,1,0xe59c8a00,11071,0x8860,1,0xeaab9700,0,0xf0b9,1,0xe686a700,0,0x93b2
,1,0xe6b88500,0,0x90b4,1,0xc3bf0000,1807,0x8594,1,0xe5b28800,8493,0xf0ba
,1,0xe5b28a00,8604,0xf0bb,1,0xe8b98a00,10151,0xe6fc,1,0xe39f8100,0,0xf0bc
,1,0x6000000,5077,0x6,1,0xeaab9900,0,0xf0bd,1,0xe5a79000,210,0x88b7
,1,0xeaab9a00,0,0xf0be,1,0xe5b1b600,3790,0x9ba5,1,0xe5b2a700,0,0xf0c2
,1,0xe8888c00,7909,0x90e3,1,0xeaab9800,0,0xf0c3,1,0xe892b200,0,0x8a97
,1,0xe5b2ad00,6065,0xf0c4,1,0xe9a99a00,0,0x8bc1,1,0xe9aead00,0,0x8df8
,1,0xe5ad8300,0,0x9b6f,1,0xe5b2b500,0,0xf0c5,1,0xe697a000,6072,0x9dd9
,1,0xe5bcbc00,3852,0x954a,1,0xe781bd00,0,0x8dd0,1,0xe3809300,0,0x81ac
,1,0xe7b8ae00,0,0x8f6b,1,0xe89db800,0,0xe597,1,0xeaab9b00,0,0xf0c6
,1,0xeaab9c00,0,0xf0c8,1,0xe586ab00,0,0x9971,1,0xe58bbe00,3816,0x8cf9
,1,0xe69d9300,1266,0x8edb,1,0xe6a2a600,0,0x9aeb,1,0xe6a7b900,0,0x9edd
,1,0xeaab9e00,0,0xf0c9,1,0xc5a00000,4,0x85a4,1,0xe983a200,5241,0xe7b9
,1,0xe5b4ab00,9340,0xf0cc,1,0xe5878b00,872,0x929c,1,0xe5b49d00,6417,0xf0cd
,1,0xe591b100,0,0x99eb,1,0xe6a38600,0,0x9eb0,1,0xe6a89900,1888,0x9557
,1,0xe78da300,6820,0x8f62,1,0xe6b2bf00,55,0x8988,1,0xe9848200,0,0xe7bd
,1,0xe8a99e00,0,0x8e8c,1,0xe5b4a000,8028,0xf0ce,1,0xe5b4a400,0,0xf0cf
,1,0xe5b4a600,0,0xf0d0,1,0xefbda200,1038,0xa2,1,0xe59cb700,0,0x9aa8
,1,0xe5b58200,11110,0xf0d3,1,0xe39fa800,7522,0xf0d4,1,0xe285aa00,7009,0x875e
,1,0xe5b5a100,7717,0xf0d5,1,0xe98f8800,0,0xe859,1,0xe5b5aa00,0,0xf0d6
,1,0xe39fb400,0,0xf0d7,1,0xefbe8200,632,0xc2,1,0xe4bda000,0,0x87ab
,1,0xe5a2aa00,0,0xf24f,1,0xe687b400,0,0x9cf1,1,0xe5b5b000,7656,0xf0d8
,1,0xeaab9f00,7112,0xf0d9,1,0xe7a3af00,8521,0x88e9,1,0xe888b900,0,0x9144
,1,0xe99a8e00,0,0x8a4b,1,0xe8bfaa00,10311,0xe78c,1,0xe39fbd00,0,0xf0da
,1,0xe5a38a00,0,0x89f3,1,0xe6889400,0,0x9cfb,1,0xe68da700,1176,0x95f9
,1,0xe5b68800,0,0xf0db,1,0xe8848600,0,0x90c6,1,0xe7a9a200,2487,0xe271
,1,0xe3a08000,8072,0xf0dc,1,0xe9a08100,2609,0x95c5,1,0xe5b69200,0,0xf0dd
,1,0xe5b69400,9917,0xf0de,1,0xe5b69700,10786,0xf0df,1,0xe5b69900,0,0xf0e0
,1,0xe6939a00,0,0x9d9f,1,0xe698ad00,1243,0x8fba,1,0xe5b6b000,7957,0xf0e1
,1,0xe88f8c00,0,0x8bdb,1,0xe381a000,2614,0x82be,1,0xe7b9bb00,0,0xe38c
,1,0xe5b6b200,0,0xf0e2,1,0xe5b6b400,6088,0xf0e3,1,0xeaaba000,0,0xf0e4
,1,0xe587b800,0,0x93ca,1,0xe5b6b900,6104,0xf0e5,1,0xe5bea900,875,0x959c
,1,0xe5b79700,0,0xf0e7,1,0xe3828000,390,0x82de,1,0xe3879300,2352,0x82f8
,1,0xe38ca600,4505,0x876a,1,0xe8a4b800,4915,0xe5f8,1,0xe5838500,3836,0x8bcd
,1,0xe9bba000,0,0xea7b,1,0xe5b79800,9242,0xf0e8,1,0xe592be00,3729,0x9a43
,1,0xe5b7a000,0,0xf0e9,1,0xe7899d00,1646,0x96c4,1,0xeaaba100,6764,0xf0ea
,1,0xe8a08500,0,0xe5a2,1,0xe5b7a400,0,0xf0eb,1,0xe5b7a900,0,0xf0ec
,1,0xe3a0af00,6943,0xf0ed,1,0xe5b88000,0,0xf0ee,1,0xe3a0b600,0,0xf0ef
,1,0xe598b100,6398,0x8ffa,1,0xe5b89200,0,0xf0f0,1,0xe5b89500,0,0xf0f1
,1,0xe794a300,0,0x8e59,1,0xe6b9bf00,2022,0x8ebc,1,0xe8ab8b00,4954,0x90bf
,1,0xe3a18000,0,0xf0f2,1,0xe8b5b100,0,0xe6df,1,0xe99abb00,11042,0x90c7
,1,0xe4b99a00,6793,0xf04e,1,0xe4bead00,1101,0x9699,1,0xe5a3b700,6228,0x92d9
,1,0xe5b89f00,10815,0xf0f3,1,0xe6ba9f00,2032,0x9ff0,1,0xe79fa900,0,0x8be9
,1,0xe884b300,853,0x945d,1,0xe5b8ae00,0,0xf0f4,1,0xe99b9b00,0,0x9097
,1,0xe5b8be00,0,0xf0f5,1,0xe5b98900,9459,0xf0f6,1,0xe6848e00,0,0x9cbd
,1,0xe5a9aa00,6283,0x9b59,1,0xe68eb400,0,0x92cd,1,0xe8808000,7773,0x9773
,1,0xe8859300,4350,0xe443,1,0xe88aa600,797,0x88b0,1,0xe3a19c00,8129,0xf0f7
,1,0xe9a18e00,0,0x8a7b,1,0xe5b99600,0,0xf0f8,1,0xe9abb400,0,0xe99c
,1,0xe3a1a100,7726,0xf0f9,1,0xe5af9d00,6345,0x9051,1,0xe5b9ab00,0,0xf0fa
,1,0xe699ba00,0,0x9271,1,0xe7ab8f00,4559,0xe288,1,0xe5b9ac00,6502,0xf0fb
,1,0xe382ad00,79,0x834c,1,0xe9a78100,0,0x949d,1,0xe5b9ad00,0,0xf0fc
,1,0xe9b1a700,3925,0xe9e5,1,0xe583b200,3537,0x87c4,1,0xe5b59000,0,0x9792
,1,0xe5b9ae00,0,0xf29f,1,0xeaaba200,7318,0xf2a0,1,0xe5bab900,0,0xf2a4
,1,0xe3838d00,4635,0x836c,1,0xe89b9f00,0,0xe580,1,0xe5babf00,9196,0xf2a5
,1,0xe5bb8600,0,0xf2a6,1,0xe5849200,11029,0x8ef2,1,0xe589a500,0,0x948d
,1,0xe5bb8300,0,0x9470,1,0xe5bb9200,0,0xf2a7,1,0xe6a5a000,0,0x93ed
,1,0xe6aab300,0,0x9f46,1,0xe5bb9900,0,0xf2a8,1,0xe8a19200,3033,0xe5c8
,1,0xe9869c00,0,0x8f58,1,0xe8abb800,0,0x8f94,1,0xeaaba300,0,0xf2a9
,1,0xe5bbbd00,0,0xf2aa,1,0xe594ab00,0,0x8844,1,0xe6a68000,4391,0xeb99
,1,0xe6ab9300,0,0x9845,1,0xe5bc8800,0,0xf2ab,1,0xe795b000,1731,0x88d9
,1,0xe5bc8e00,6556,0xf2ac,1,0xc5ae0000,3140,0x85c3,1,0xcc860000,913,0x8676
,1,0xe8b6be00,0,0xe6e4,1,0xe5bc9c00,0,0xf2ad,1,0xe59a9e00,8558,0x885a
,1,0xe4bfba00,0,0x89b4,1,0xeaaba400,0,0xf2ae,1,0xe6b69900,0,0x97dc
,1,0xe79ba300,1783,0x8ac4,1,0xe5bc9e00,0,0xf2af,1,0xe5bd8700,0,0xf2b0
,1,0xe9979500,0,0xe88d,1,0xe5bdb200,0,0xf2b2,1,0xe4bb8700,0,0x8b77
,1,0xe5bdbe00,0,0xf2b3,1,0xe5bea200,0,0xf2b5,1,0xe5bea400,0,0xf2b6
,1,0xe5beb800,7076,0xf2b7,1,0xe5bf8400,0,0xf2b8,1,0xe886a000,7876,0xe450
,1,0xe88bb300,0,0xe494,1,0xe99d8800,0,0xe8cb,1,0xe3a3ba00,0,0xf2b9
,1,0xe9a7ae00,0,0xe96f,1,0xe5a68400,0,0x96cf,1,0xe5ab9700,0,0x9b61
,1,0xe5bf8700,0,0xf2ba,1,0xe695b400,0,0x90ae,1,0xe7a78900,2451,0xe25c
,1,0xe5bf9300,0,0xf2bd,1,0xe891a600,8147,0x88af,1,0xe383ba00,2672,0x8495
,1,0xe9a88e00,0,0x8b52,1,0xe5bfa200,6611,0xf2bf,1,0xe5bfae00,0,0xf2c0
,1,0xe5b18a00,962,0x93cd,1,0xe5b69d00,0,0x9bd1,1,0xe5bbb000,0,0x9c4c
,1,0xe5bfaf00,0,0xf2c1,1,0xe29f9000,4429,0x84bf,1,0xe5bfb300,8965,0xf2c2
,1,0xd1840000,0,0x8486,1,0xe8a1bf00,3050,0x8bdc,1,0xe5bfbc00,0,0xf2c3
,1,0xe3a49700,0,0xf2c4,1,0xe6809700,0,0xf2c5,1,0xe5bc9000,0,0x93f3
,1,0xe7819100,0,0xe072,1,0xe6a6ad00,0,0xeb9c,1,0xe78bb700,0,0xe0c4
,1,0xe7bd9500,2729,0xe3a5,1,0xe680a200,0,0xf2c6,1,0xe680a400,0,0xf2c7
,1,0xe3a49a00,9505,0xf2c8,1,0xe58b9200,0,0xe8d3,1,0xe6818c00,0,0xf2c9
,1,0xe681bf00,7417,0xf2ca,1,0xe6a78d00,4101,0x9184,1,0xe6aca000,1915,0x8c87
,1,0xe6b1b300,1943,0x9f92,1,0xe796bd00,0,0xe173,1,0xe8a89200,0,0xee9d
,1,0xe98d9c00,0,0xe847,1,0xe8b2b800,0,0x91dd,1,0xe6828a00,0,0xf2cb
,1,0xefbc9600,1017,0x8255,1,0xe6829500,10141,0xf2cc,1,0xe680b500,0,0xeacc
,1,0xe6b29300,0,0x8c42,1,0xe6b7a600,7592,0x9fbe,1,0xe682a800,0,0xf2cd
,1,0xeaaba500,0,0xf2ce,1,0xe682b000,0,0xf2cf,1,0xe682b100,7916,0xf2d0
,1,0xe682be00,6835,0xf2d1,1,0xe59c8b00,11092,0x9aa0,1,0xe5a19e00,300,0x8dc7
,1,0xe6838800,6264,0xf2d2,1,0xe6839900,0,0xf2d3,1,0xe6bd9900,0,0xec49
,1,0xe7a2a300,2407,0xe1f0,1,0xe294b700,281,0x84b8,1,0xe9998200,9555,0xe898
,1,0xe8be9e00,5049,0x8eab,1,0xe6839b00,0,0xf2d4,1,0xe683ae00,0,0xf2d5
,1,0xe5a79100,1376,0x8cc6,1,0xe683b200,0,0xf2d6,1,0xe6849000,0,0xf2d8
,1,0xe6849200,0,0xf2d9,1,0xe8888d00,0,0xe471,1,0xe6849300,8050,0xf2da
,1,0xe7b2bc00,0,0xed95,1,0xe6849900,10434,0xf2db,1,0xe9a99b00,0,0xe983
,1,0xe9aeae00,0,0x914e,1,0xe580b900,0,0x8c90,1,0xe6849e00,0,0xf2dc
,1,0xe697a100,0,0x9dda,1,0xe69cb400,3713,0x9670,1,0xe684ba00,7188,0xf2dd
,1,0xe3809400,2241,0x816b,1,0xe6858600,0,0xf2e0,1,0xe685a000,0,0xf2e1
,1,0xe9af8e00,0,0xefc9,1,0x6e000000,11483,0x6e,1,0xe586ac00,0,0x937e
,1,0xe58bbf00,0,0x96dc,1,0xe5bd9d00,6577,0x9c52,1,0xe6a2a700,0,0x8ce6
,1,0xe685bc00,0,0xf2e2,1,0xeaaba600,0,0xf2e3,1,0xe38b9a00,0,0x86e3
,1,0xd18b0000,0,0x848d,1,0xe9b58100,0,0xe9f8,1,0xe5878c00,0,0x97bd
,1,0xe6869200,0,0xf2e4,1,0xe6869300,0,0xf2e5,1,0xe6869700,0,0xf2e6
,1,0xe6869800,6736,0xf2e7,1,0xe686a500,0,0xf2e8,1,0xe686a800,0,0xf2e9
,1,0xe8a48c00,0,0xe5ec,1,0xeaaba700,9878,0xf2eb,1,0xe6879500,0,0xf2ec
,1,0xe6879d00,0,0xf2ed,1,0xe5929200,0,0x99ee,1,0xefbda300,5235,0xa3
,1,0xe59cb800,0,0x9aa9,1,0xe78e8400,1670,0x8cba,1,0xe2809800,3965,0x8165
,1,0xe285ab00,0,0x8776,1,0xe6879f00,7720,0xf2ee,1,0xeaaba800,0,0xf2f0
,1,0xe6898c00,0,0xf2f5,1,0xe6899100,0,0xf2f6,1,0xefbe8300,21,0xc3
,1,0xe6899200,0,0xf2f7,1,0xe5a2ab00,0,0x9ad1,1,0xe687b500,6907,0xf2ef
,1,0xe689a100,7033,0xf2f8,1,0xe689a400,0,0xf2f9,1,0xe689bb00,0,0xf2fa
,1,0xe68a9900,0,0xf340,1,0xc5bc0000,343,0x85b6,1,0xe8bfab00,0,0x9497
,1,0xe68aa600,8646,0xf341,1,0xeaaba900,0,0xf343,1,0xe6889500,0,0xf2f1
,1,0xe68da800,1197,0x8ecc,1,0xe692bb00,1813,0x9da3,1,0xe8848700,7836,0x9865
,1,0xe7a9a300,2489,0x8ff5,1,0xe68bbd00,0,0xf344,1,0x57000000,464,0x57
,1,0xe9a59500,0,0xe962,1,0xe9aaa800,0,0x8d9c,1,0xe68c8300,0,0xf345
,1,0xe68e8800,0,0x8ef6,1,0xe68c8d00,0,0xf346,1,0xe5b8b700,0,0x9be7
,1,0xe7aa8300,0,0x90de,1,0xe68c9000,8090,0xf347,1,0xe381a100,0,0x82bf
,1,0xe7b9bc00,0,0xe38b,1,0xeaabab00,7191,0xf348,1,0xe9b09b00,0,0xe9da
,1,0xeaabac00,9636,0xf349,1,0xe587b900,0,0x899a,1,0xe5b99700,6490,0x9bec
,1,0xc4800000,291,0x8595,1,0xe68cb200,0,0xf34a,1,0xe3828100,248,0x82df
,1,0xe3879400,2353,0x82f9,1,0xe38ca700,8450,0x8764,1,0xe984b000,4741,0xe7c1
,1,0xefa98400,0,0xeb85,1,0xe68cb500,0,0xf34b,1,0xe58dac00,0,0xf174
,1,0xe69f8100,5531,0x91c6,1,0xe68cbb00,0,0xf34c,1,0xe68cbc00,0,0xf34d
,1,0xe6aeba00,0,0x8e45,1,0xe8a08600,0,0xee83,1,0xe68d8e00,0,0xf350
,1,0xe8aaac00,0,0x90e0,1,0xeaabaa00,0,0xf351,1,0xe68d9900,0,0xf352
,1,0xeaabad00,0,0xf353,1,0xe4b8bb00,1057,0x8ee5,1,0xeaabae00,0,0xf354
,1,0xe68dac00,6836,0xf355,1,0xe68e8400,7544,0xf356,1,0xe68e9900,0,0xf357
,1,0xe8ab8c00,0,0x8ad0,1,0xeaabaf00,10081,0xf358,1,0xe68e9400,0,0xf359
,1,0xe99abc00,0,0x94b9,1,0xe68ebd00,6807,0xf35a,1,0xe4beae00,0,0x958e
,1,0xe68fb700,0,0xf35b,1,0xe7958400,0,0xe156,1,0xe68f9400,0,0xf35c
,1,0x40000000,1803,0x40,1,0xe68f9500,0,0xf35d,1,0xe9968900,0,0x95c2
,1,0xe99b9c00,0,0xe8b6,1,0xe68f9c00,0,0xf35e,1,0xe4bf8e00,0,0x98d7
,1,0xc2a60000,31,0x8544,1,0xe68fa000,0,0xf35f,1,0xe68eb500,0,0x9d7d
,1,0xe8808100,0,0x9856,1,0xe8859400,288,0x8d6f,1,0xe7aab000,8704,0xe27e
,1,0xe8bc8500,0,0xe769,1,0xe9a18f00,0,0xe8f8,1,0xe68fab00,10551,0xf360
,1,0xe3b8bf00,0,0xec86,1,0xe68a8200,5852,0x9d4f,1,0xe5af9e00,6347,0x9b8b
,1,0xe5b4b100,0,0xf0d1,1,0xe68fac00,6523,0xf361,1,0xe68fb200,0,0xf362
,1,0xe7b0a300,0,0xe2cd,1,0xe382ae00,362,0x834d,1,0xe6908900,0,0xf363
,1,0xe6909e00,0,0xf364,1,0xe690af00,0,0xf367,1,0xe9b6bb00,5356,0xea52
,1,0xe6958800,0,0x9dc1,1,0xe6919a00,0,0xf368,1,0xe69fae00,0,0x9e72
,1,0xe7b18300,0,0xe2d5,1,0xe3838e00,0,0x836d,1,0xe6919b00,0,0xf369
,1,0xe6919d00,0,0xf36a,1,0xe691b300,0,0xf36b,1,0xe691bd00,0,0xf36c
,1,0xe6928700,0,0xf36d,1,0xe6929d00,0,0xf36f,1,0xe7808500,0,0xec5b
,1,0xc4870000,1626,0x85c9,1,0xe6929f00,0,0xf370,1,0xe78fbe00,2716,0x8cbb
,1,0xe8a19300,783,0x8f70,1,0xe8a6a600,0,0xe64f,1,0xe6938b00,0,0xf371
,1,0xe9bd8e00,0,0xe6d8,1,0xe58f9900,0,0x8f96,1,0xe6938c00,0,0xf372
,1,0xe6a68100,7967,0x9ec1,1,0x29000000,6779,0x29,1,0xe6939500,0,0xf373
,1,0xe6939700,7572,0xf374,1,0xeaabb000,0,0xf375,1,0xe8ac9900,342,0x8caa
,1,0xe8b1ac00,4990,0xe6b5,1,0xe996b600,0,0xef72,1,0xe5958c00,0,0x9a52
,1,0xe4baa800,8601,0x8b9c,1,0xe693a400,0,0xf376,1,0xe684bc00,0,0x9cc4
,1,0xe7969100,0,0x8b5e,1,0xe288a500,598,0x81d2,1,0xe693bf00,0,0xf378
,1,0xe8b28c00,0,0x9665,1,0xe8b79f00,3230,0xe6ee,1,0xe6948400,6422,0xf379
,1,0xe4bb8800,0,0xf058,1,0xe3a9ae00,0,0xf37a,1,0xe5a5a500,0,0x899c
,1,0xe6948f00,0,0xf37b,1,0xe2898500,2058,0x81eb,1,0xe6949400,0,0xf37c
,1,0xe7a6aa00,0,0xe257,1,0xe88bb400,4400,0xe493,1,0xe99d8900,0,0xe8cd
,1,0xe6949600,0,0xf37d,1,0xe3a9b300,0,0xf37e,1,0xe6949e00,0,0xf380
,1,0xc2ad0000,1400,0x8548,1,0xe5b0ab00,3786,0x987d,1,0xe695b500,0,0x9347
,1,0xe7a78a00,8585,0xed65,1,0xe6958400,0,0xf382,1,0xe695ab00,7151,0xf384
,1,0xe383bb00,2666,0x8145,1,0xe9a88f00,0,0xe975,1,0xe6968100,0,0xf386
,1,0xe6968400,0,0xf387,1,0xe5b18b00,360,0x89ae,1,0xe6968500,0,0xf388
,1,0xe5bbb100,3851,0x9c4a,1,0xe780b200,6625,0xe071,1,0xe29f9100,2220,0x84c0
,1,0xe7b7a300,0,0xedab,1,0xe7bcb600,0,0x8aca,1,0xe981b700,0,0x914a
,1,0xe5808d00,9543,0x947b,1,0xe585a000,0,0xf145,1,0x12000000,10324,0x12
,1,0xe69c8800,1268,0x8c8e,1,0xe6a19b00,0,0xeb82,1,0xe6a6ae00,0,0x9ec4
,1,0xe78bb800,0,0x924b,1,0xe6968a00,0,0xf389,1,0xe9829700,9224,0xeeda
,1,0xe696b200,9318,0xf38a,1,0xe5868000,0,0x9962,1,0xe696b500,0,0xf38b
,1,0xe590a600,0,0x94db,1,0xe696b800,0,0xf38c,1,0xe6a78e00,0,0x9eca
,1,0xc48e0000,246,0x85be,1,0xe6b1b400,4198,0xebd2,1,0xe796be00,9236,0x8ebe
,1,0xe8a89300,9831,0x8c50,1,0xe8ada600,0,0x8c78,1,0xe696bf00,0,0xf38d
,1,0xe5918600,0,0x95f0,1,0xefbc9700,1012,0x8256,1,0xe4bbb500,10142,0xf05d
,1,0xe6978200,0,0xf38e,1,0xe7928b00,6879,0xe0f6,1,0xe7979e00,0,0xe17c
,1,0xe6978900,0,0xf38f,1,0xe6979400,8490,0xf390,1,0xe3ab9600,9471,0xf391
,1,0xe697b200,0,0xf392,1,0xe697b900,0,0xf393,1,0xe697bc00,11096,0xf394
,1,0xe6988800,0,0xf396,1,0xe686a900,1144,0x8c65,1,0xe6b88700,3767,0x8a89
,1,0xe698a100,7095,0xf397,1,0xe8829b00,7813,0xe3e8,1,0xe294b800,0,0x84bd
,1,0xe8b98c00,0,0xe742,1,0xe8be9f00,10270,0xe784,1,0xe9a3a900,0,0xe94a
,1,0xe698aa00,0,0xf398,1,0xe5a79200,1374,0x888f,1,0xe5aca500,0,0x889d
,1,0xe691af00,0,0x9d95,1,0xe6998500,0,0xf399,1,0xe8888e00,613,0x8ec9
,1,0xe6998e00,0,0xf39b,1,0xe7b2bd00,4604,0xe2ee,1,0xe9a48900,0,0xe94d
,1,0xe4969d00,0,0xee70,1,0xe3abaa00,0,0xf39c,1,0xe5ad8500,0,0x9b70
,1,0xc2b40000,0,0x814c,1,0xe697a200,1237,0x8af9,1,0xe5bcbe00,0,0x9265
,1,0xeaabb200,0,0xf39d,1,0xe3809500,4438,0x816c,1,0xe6999b00,8992,0xf39f
,1,0xe699a300,0,0xf3a0,1,0xe9af8f00,0,0xe9c3,1,0xcca90000,944,0x868f
,1,0xe9b9b500,5384,0xea62,1,0xe6988200,1242,0x8d56,1,0xeaabb300,0,0xf3a2
,1,0xe6a2a800,7208,0x979c,1,0xe6a7bb00,0,0x92ce,1,0xe699aa00,0,0xf3a3
,1,0xe38b9b00,2364,0x86e4,1,0xe983a400,5243,0xe7ba,1,0xe988b700,10950,0x8cd8
,1,0xe5878d00,3588,0x9380,1,0xe58ca000,0,0x8fa0,1,0xe591b300,0,0x96a1
,1,0xe6a38800,0,0xeb89,1,0xe6a89b00,0,0x9eda,1,0xe699ab00,0,0xf3a4
,1,0xe699ac00,8437,0xf3a5,1,0xe699ad00,9757,0xf3a6,1,0xe8a9a000,3113,0x8972
,1,0xe699bb00,0,0xf3a7,1,0xe58d8000,0,0x99bd,1,0xe69a9000,8624,0xf3a9
,1,0xefbda400,5246,0xa4,1,0xe69a9200,0,0xf3aa,1,0xe69a9900,7418,0xf3ab
,1,0xe2809900,683,0x8166,1,0xe6b8b400,0,0xebf6,1,0xe28abf00,2077,0x8799
,1,0xe69aad00,0,0xf3ad,1,0xe69ab100,0,0xf3ae,1,0xe999b000,0,0x8941
,1,0xefbe8400,427,0xc4,1,0xe69ab500,0,0xf3af,1,0xe682a300,0,0x8ab3
,1,0xe5a7bf00,1378,0x8e70,1,0xe7998b00,7044,0xecd8,1,0xe79e9e00,4047,0xe1d4
,1,0xe3ac9a00,0,0xf3b0,1,0xe69abf00,0,0xf3b1,1,0xe8ba9900,0,0xe757
,1,0xe3ac9c00,0,0xf3b2,1,0xe4be8200,0,0xf06a,1,0xe5a38c00,0,0x8feb
,1,0xe6889600,1147,0x88bd,1,0xe68da900,0,0x9d80,1,0xe692bc00,4057,0x9d9e
,1,0xe8848800,2111,0x96ac,1,0xe69bac00,0,0xf3b3,1,0xe3ab9700,0,0xf3b4
,1,0xe9a08300,0,0x8da0,1,0xe69c8100,0,0xf3b5,1,0xe497aa00,0,0xee80
,1,0xe69c8500,0,0xf3b6,1,0xe68e8900,0,0x9d7b,1,0xe69c9200,0,0xf3b7
,1,0xe5b8b800,350,0x8fed,1,0xe7aa8400,0,0x8df3,1,0xe88f8e00,4415,0xe4c0
,1,0xe381a200,2607,0x82c0,1,0xe7b9bd00,0,0xe38f,1,0xeaabb500,0,0xf3b8
,1,0xeaabb700,0,0xf3ba,1,0xe9b5af00,0,0xea4a,1,0xe587ba00,0,0x8f6f
,1,0xc2bb0000,1513,0x8551,1,0xc9930000,344,0x8644,1,0xe69d8c00,7212,0xf3bc
,1,0xe3828200,1898,0x82e0,1,0xe69d8d00,0,0xf3bd,1,0xe7bfb000,0,0x8acb
,1,0xe984b100,0,0xeef0,1,0xe9b68f00,3896,0x8c7b,1,0xccb00000,0,0x8693
,1,0xe69d9400,0,0xf3be,1,0xe5bf8b00,0,0xf2bb,1,0xe69d9d00,7251,0xf3bf
,1,0xe7899f00,0,0x96b4,1,0xe6aebb00,852,0x8a6b,1,0xeaabb600,0,0xf3c0
,1,0xeaabbc00,0,0xf3c1,1,0xe8aaad00,0,0x93c7,1,0xe69db400,8752,0xf3c3
,1,0xe69db600,0,0xf3c4,1,0xeaabba00,0,0xf3c5,1,0xe4b8bc00,6499,0x98a5
,1,0xe69e9200,0,0xf3c6,1,0xe6af9b00,0,0x96d1,1,0xe794a500,3977,0x8999
,1,0xe799b800,0,0xe1a1,1,0xe8ab8d00,0,0xe679,1,0xe39da100,10082,0xf0a1
,1,0xe8b5b300,0,0xe6e0,1,0xe5948000,10186,0xf19c,1,0xe5999300,6826,0xeaa5
,1,0xe4beaf00,0,0x8cf2,1,0xe683b000,1121,0x91c4,1,0xe88da300,0,0xf3c8
,1,0xc49c0000,0,0x85d8,1,0xe69eb000,0,0xf3ca,1,0xe7a4be00,8528,0x8ed0
,1,0xe9968a00,0,0xe87a,1,0xe69f8300,10220,0xf3cc,1,0xe9a0b000,0,0xef9a
,1,0xe59f8600,5307,0x9abb,1,0xc7b90000,287,0x84f3,1,0xce910000,1152,0x839f
,1,0xe5aebf00,4027,0x8f68,1,0xe69f9900,8459,0xf3cf,1,0xe7a59e00,886,0x905f
,1,0xe69f9b00,8705,0xf3d0,1,0xe88fbb00,0,0xe4d0,1,0xeaabbe00,0,0xf3d3
,1,0xeaabbf00,10536,0xf3d4,1,0xeaac8000,0,0xf3d5,1,0xe68a8300,0,0x9d56
,1,0xe5af9f00,778,0x8e40,1,0xe694a900,0,0xeb46,1,0xe6a09800,6719,0xf3d6
,1,0xe6a0ad00,7979,0xf3d8,1,0xe6a0b300,0,0xf3da,1,0xe382af00,0,0x834e
,1,0xe6a0bb00,10558,0xf3db,1,0xe6a18400,7346,0xf3dd,1,0xe6a18500,10884,0xf3de
,1,0xe9b6bc00,0,0xefe2,1,0xe5b59200,0,0x9bc7,1,0xe5baa500,0,0xf2a1
,1,0xe5bfb800,1576,0x9c78,1,0xe6a18900,0,0xf3df,1,0xe3838f00,8285,0x836e
,1,0xe6a18c00,7160,0xf3e0,1,0xe6a19500,0,0xf3e1,1,0xe6a19700,0,0xf3e2
,1,0xe5849400,0,0x9951,1,0xe3adb700,11135,0xf3e3,1,0xe6a1ba00,0,0xf3e6
,1,0xe7808600,0,0xec5c,1,0xc99a0000,52,0x8662,1,0xe78aac00,0,0x8ca2
,1,0xe6a1bc00,6888,0xf3e7,1,0xe9818b00,0,0x895e,1,0xe8a6a700,4922,0x9797
,1,0xe8abba00,0,0x8cbf,1,0xe58a8700,3337,0x8c80,1,0xe58f9a00,0,0xf180
,1,0xe6a29000,0,0xf3e9,1,0xe6a68200,7999,0x8a54,1,0xe6a29600,7353,0xf3ea
,1,0xe3adad00,10341,0xf3eb,1,0xe6a29800,7566,0xf3ec,1,0xe6a29900,0,0xf3ed
,1,0xe8ac9a00,9977,0xe68a,1,0xe6a29a00,9427,0xf3ee,1,0xe6a29c00,0,0xf3ef
,1,0xe6a2b400,0,0xf3f2,1,0xe59aa000,8603,0x9a88,1,0xeaac8100,0,0xf3f5
,1,0xe684bd00,0,0x9cc7,1,0xe6b69b00,4248,0x9393,1,0xe288a600,2047,0x81d3
,1,0xeaac8200,0,0xf3f6,1,0xe8b28d00,0,0xe6bc,1,0xefa89300,0,0xf3f7
,1,0xe8bcb300,3289,0xe773,1,0xe59b8000,0,0x9a93,1,0xe6a38300,0,0xf3f8
,1,0xe6859d00,0,0x9cd5,1,0xe6a38500,0,0xf3f9,1,0xe6a39900,0,0xf440
,1,0xe6a3a400,0,0xf441,1,0xe6a3a500,7226,0xf442,1,0xe6a3ac00,0,0xf443
,1,0xe6a48300,0,0xf445,1,0xe6a48700,0,0xf446,1,0xe3ae8700,10567,0xf447
,1,0xe3ae8800,0,0xf448,1,0xe68b9000,0,0x89fb,1,0xce980000,0,0x83a6
,1,0xeaac8400,0,0xf449,1,0xe2948c00,280,0x84a1,1,0xe7ac9e00,0,0xe29a
,1,0xe6a4bb00,0,0xf44a,1,0xe383bc00,0,0x815b,1,0xe3ae8d00,0,0xf44b
,1,0xe6a5a900,10691,0xf44d,1,0xe6a5b200,0,0xf44f,1,0xe6a5ba00,0,0xf450
,1,0xe6a5bf00,0,0xf451,1,0xe69ba900,0,0x9e47,1,0xe6a0bc00,0,0x8a69
,1,0xe29f9200,0,0x84c1,1,0xe7b7a400,0,0xe369,1,0xe3aea400,0,0xf453
,1,0xe981b800,3453,0x9149,1,0xe5808e00,0,0x87bb,1,0xe6a69600,0,0xf454
,1,0xe58ab400,0,0x984a,1,0xe69c8900,0,0x974c,1,0xe6a19c00,0,0x8df7
,1,0xe6a69800,0,0xf455,1,0xe7b88400,1914,0x93ea,1,0xe89d8e00,0,0xe599
,1,0xe6a6a600,7727,0xf456,1,0xe8a7b400,0,0xe65b,1,0xe6a6ba00,0,0xf459
,1,0xe9be9c00,0,0xea9d,1,0xe590a700,0,0x87f4,1,0xe6a78000,9237,0xf45b
,1,0xe6a78f00,0,0xeb9d,1,0xc9a10000,7368,0x85f7,1,0xe6a79100,0,0xf45c
,1,0xe3908200,228,0x87a1,1,0xeaac8500,9330,0xf45e,1,0xeaac8600,7898,0xf45f
,1,0xe6a8b000,0,0xf460,1,0xeaac8700,8417,0xf461,1,0xefbc9800,96,0x8257
,1,0x7a000000,167,0x7a,1,0xeaac8800,0,0xf462,1,0xe6a7a300,6882,0xf463
,1,0xe6b7a800,7597,0x9fc4,1,0xe6a7ae00,0,0xf464,1,0xe6a7af00,0,0xf465
,1,0xe8b39a00,0,0xe6cd,1,0xe6a7b300,7327,0xf466,1,0xe3af8d00,0,0xf467
,1,0xe59c8d00,5275,0x9aa1,1,0xe6a7b400,0,0xf468,1,0xe6a7be00,0,0xf469
,1,0xe6b88800,6296,0x8dcf,1,0xe6bd9b00,0,0xe049,1,0xe6a89100,8495,0xf46a
,1,0xe6a89a00,0,0xf46b,1,0xe9998400,0,0x958d,1,0xeaac8a00,7966,0xf46d
,1,0xe6a8b200,7712,0xf46e,1,0xe6a8b300,0,0xf46f,1,0xe5a79300,1373,0x90a9
,1,0xe6a8b400,7304,0xf470,1,0xe5b1b900,0,0x9ba6,1,0xe7a38500,0,0xe1fa
,1,0xc4aa0000,2137,0x8596,1,0xe88da200,0,0xedeb,1,0xe7b2be00,0,0x90b8
,1,0xe9a48a00,0,0x977b,1,0xe6a8bf00,0,0xf471,1,0xe6a98900,0,0xf473
,1,0xe580bb00,0,0x87c0,1,0xe6a99200,10577,0xf476,1,0xce9f0000,0,0x83ad
,1,0xe69cb600,0,0x9e53,1,0xe7ae8b00,0,0xe2b3,1,0xe3809600,654,0x81d8
,1,0xe7b8b100,0,0xe373,1,0xe6a9a400,0,0xf477,1,0xeaac8c00,0,0xf478
,1,0xe9b4a300,0,0xe9f3,1,0xe6a9be00,0,0xf479,1,0xe5b88c00,1506,0x8af3
,1,0xe69d9600,0,0x8ff1,1,0xe6aa8300,0,0xf47a,1,0xe6aa8b00,0,0xf47b
,1,0xe6aa9f00,0,0xf47e,1,0xe38b9c00,154,0x86e5,1,0xe6aaa100,0,0xf480
,1,0xe988b800,5142,0xef45,1,0x63000000,0,0x63,1,0xe58ca100,0,0x8ba7
,1,0xe591b400,0,0x87f8,1,0xe6a38900,0,0x96c7,1,0xeaac8d00,0,0xf481
,1,0xe6adaf00,4127,0x8e95,1,0xe6aaab00,7517,0xf482,1,0xe6ab8600,9761,0xf484
,1,0xe8a9a100,0,0xeea4,1,0xe6ab9400,0,0xf485,1,0xe58d8100,0,0x8f5c
,1,0xe6ab9000,0,0xf486,1,0xefbda500,5286,0xa5,1,0xeaac8e00,0,0xf489
,1,0xeaac8f00,8807,0xf48a,1,0xc9a80000,567,0x864b,1,0xe6abac00,9275,0xf48b
,1,0xe79dbf00,0,0xe1cd,1,0xe6abb100,0,0xf48c,1,0xe6abb200,0,0xf48d
,1,0xe6abb300,0,0xf48e,1,0xefbe8500,909,0xc5,1,0xe6abbd00,0,0xf48f
,1,0xeaac9000,0,0xf490,1,0xe687b700,0,0x9ce5,1,0xe7998c00,0,0x8ae0
,1,0xe6ac8b00,7729,0xf491,1,0xe7a3b200,0,0xed4b,1,0xe6ac8f00,8633,0xf492
,1,0xe6ac9000,7358,0xf493,1,0xe8bfad00,0,0x9352,1,0xe4be8300,0,0x8aa4
,1,0xe5a38d00,0,0xf253,1,0xe5a8a000,0,0x9050,1,0xe5adb300,0,0x9b79
,1,0xe6ac9100,7116,0xf494,1,0xe7a49200,2418,0xe245,1,0xeaac9100,8662,0xf495
,1,0xe7aeb800,0,0x94a2,1,0xe6ac9700,10316,0xf496,1,0xe9a59700,0,0x8bc0
,1,0xe3b0a600,0,0xf497,1,0xe5a98000,5129,0x9b53,1,0xe5ae9300,0,0x9877
,1,0xe6ad9800,0,0xf49a,1,0xe698b000,0,0xeb53,1,0xe2978600,4387,0x819f
,1,0xe88f8f00,0,0xedfc,1,0xe381a300,242,0x82c1,1,0x4c000000,4671,0x4c
,1,0xe6adac00,0,0xf49b,1,0xe6ae8100,10818,0xf49e,1,0xefa8a600,5147,0xeee8
,1,0xe5b48600,0,0xf0ca,1,0xe6aeae00,7425,0xf4a0,1,0xcea60000,0,0x83b3
,1,0xeaac9200,0,0xf4a1,1,0xe3828300,2257,0x82e1,1,0xe89a9500,0,0x8e5c
,1,0xe6af8700,0,0xf4a4,1,0xe8a4bb00,3074,0xe5f6,1,0xefa98600,0,0xebe7
,1,0xe6af8800,0,0xf4a5,1,0xe58dae00,0,0x99c6,1,0xe5bf8c00,6605,0x8af5
,1,0xe6af8900,0,0xf4a6,1,0xe6af9a00,0,0xf4a7,1,0xe6aebc00,1931,0x9f76
,1,0xe38d8900,0,0x875f,1,0xe9859200,0,0x8ef0,1,0xe6afa700,9347,0xf4a9
,1,0xe6afae00,11122,0xf4aa,1,0xe6afb100,0,0xf4ab,1,0xe593a100,0,0x88f5
,1,0xe598b400,6507,0x9a7b,1,0xe78a8000,3913,0x8dd2,1,0xe6b08200,0,0xf4ac
,1,0xe794a600,6915,0xe153,1,0xe6b08e00,0,0xf4ae,1,0xe6b0b500,0,0xf4af
,1,0xe9909800,5913,0x8fe0,1,0xe8b5b400,0,0x958b,1,0xe5948100,0,0xf19d
,1,0xe4b99d00,6854,0x8be3,1,0xeaac9300,0,0xf4b2,1,0xe5a3ba00,1337,0x9ae2
,1,0xe7958600,0,0xe15b,1,0xc9af0000,38,0x8653,1,0xe6bfb500,7761,0xec5a
,1,0xe8b18100,0,0xe6ad,1,0xe9968b00,0,0x8a4a,1,0xe99b9e00,0,0xef82
,1,0xe6b0bf00,0,0xf4b3,1,0xe59f8700,933,0x886d,1,0xe5a49a00,0,0x91bd
,1,0xe5a9ad00,6284,0x8894,1,0xe6bb8200,0,0x9fef,1,0xe8808300,2091,0x8d6c
,1,0x35000000,2123,0x35,1,0xe88aa900,0,0xede0,1,0xe6b18d00,0,0xf4b4
,1,0xe6b19b00,7465,0xf4b5,1,0xe6b1ad00,0,0xf4b6,1,0xe9abb700,0,0xe99e
,1,0xe68a8400,0,0x8fb4,1,0xe6b28400,7495,0xf4b7,1,0xe694aa00,0,0x9d98
,1,0xe6b28900,0,0xf4b8,1,0xe7ab9200,0,0x9af2,1,0xe6b29400,0,0xf4ba
,1,0xe382b000,8894,0x834f,1,0xe9a78400,0,0x91ca,1,0xe6b29500,0,0xf4bb
,1,0xe6b2ad00,10885,0xf4bd,1,0xe583b500,11012,0x994b,1,0xe5b59300,0,0x9895
,1,0xe5baa600,1533,0x9378,1,0xe69fb000,0,0xf3d1,1,0xe6b39000,8116,0xf4bf
,1,0xe3839000,4636,0x836f,1,0xcb900000,0,0x8674,1,0xe6b39600,0,0xf4c1
,1,0xe6b39a00,0,0xf4c2,1,0xe5849500,0,0x9950,1,0xe6b39c00,0,0xf4c3
,1,0xe58ebb00,6124,0x8b8e,1,0xe6a09000,7144,0xf3c9,1,0xe6a5a300,0,0xeb96
,1,0xe6b3a900,7333,0xf4c4,1,0xe89c8200,0,0x9649,1,0xe6b3ac00,0,0xf4c5
,1,0xe6b3ad00,0,0xf4c6,1,0xe98bb200,0,0x9565,1,0xe58a8800,0,0x999c
,1,0xe58f9b00,0,0x94be,1,0xe594ae00,0,0x9a53,1,0xeaac9500,0,0xf4c7
,1,0xe6b48000,0,0xf4c8,1,0xe6b48a00,0,0xf4c9,1,0xe795b300,0,0x8ff4
,1,0xe6b4a400,0,0xf4ca,1,0xe8ac9b00,0,0x8d75,1,0xe6b4a600,0,0xf4cb
,1,0xe6b4a700,0,0xf4cc,1,0xe6b1a700,0,0xf4cd,1,0xe6b4af00,8607,0xf4ce
,1,0xe59fb400,0,0x8ffb,1,0x1e000000,1341,0x1e,1,0xe6b69c00,0,0x93c0
,1,0xe288a700,1788,0x81c8,1,0xe6b4bc00,8470,0xf4cf,1,0xe8b28e00,0,0xe6bd
,1,0xe8b7a100,526,0x90d5,1,0xe6b5b000,0,0xf4d3,1,0xe4bb8a00,1072,0x8da1
,1,0xe6b68000,0,0xf4d4,1,0xe5a5a700,0,0x9afa,1,0xe68ab100,5868,0x95f8
,1,0xe6bc8f00,0,0x9852,1,0xe6b68100,7167,0xf4d5,1,0xe886a300,0,0xe453
,1,0xe7abbf00,0,0x8ac6,1,0xe8bd9400,0,0xeece,1,0xe6b68a00,7573,0xf4d6
,1,0xe9a7b100,0,0xe970,1,0xe6b68d00,0,0xf4d7,1,0xe68b9100,5872,0x9d59
,1,0xe5b0ad00,0,0x8bc4,1,0xe695b700,0,0x957e,1,0xe6b69100,7886,0xf4d8
,1,0xe299a000,2532,0x83b8,1,0xe891a900,0,0xe4db,1,0xe383bd00,8953,0x8152
,1,0xe6b69800,10056,0xf4d9,1,0xeaac9800,0,0xf4dd,1,0xe6b6ab00,7577,0xf4df
,1,0xe5b18d00,0,0x8e72,1,0xe6969700,907,0x936c,1,0xe5bbb300,0,0x9c4b
,1,0xe6a0bd00,0,0x8dcd,1,0xe29f9300,827,0x84c2,1,0xe8979c00,0,0xe55b
,1,0xe7bcb800,0,0xe39d,1,0xe9ae8400,0,0xefc3,1,0xe5808f00,0,0xe0c5
,1,0xe585a200,0,0x995d,1,0xe58ab500,0,0x99a3,1,0xe5bc9300,0,0x8b7c
,1,0xe6a19d00,7168,0x9691,1,0xe6a6b000,0,0xf457,1,0xe7b88500,7355,0xe36e
,1,0xe7bd9800,0,0xe3a7,1,0xe8a2a200,4874,0xe5d9,1,0xe6b6ae00,9826,0xf4e0
,1,0xe5868200,0,0x9963,1,0xe58b9500,2148,0x93ae,1,0x7000000,5486,0x7
,1,0xe595bb00,0,0x9a60,1,0xe6a79000,0,0x9ec5,1,0xceb40000,145,0x83c2
,1,0xe6b1b600,4200,0xebd3,1,0xe8a38200,0,0x97f4,1,0xe8a89500,0,0xee9e
,1,0xe6b6b400,0,0xf4e1,1,0xe8b2bb00,0,0x94ef,1,0xe5918800,0,0x92e6
,1,0xefbc9900,0,0x8258,1,0xe4bbb700,1079,0x98c1,1,0xe6ad8300,0,0x9f5c
,1,0xe6b29600,0,0x89ab,1,0xe284a100,7003,0x8784,1,0xe6b4b400,7085,0xf4e3
,1,0xe881bd00,0,0xe3e3,1,0xe8b39b00,0,0x8e5e,1,0xe3a5af00,0,0xf2de
,1,0xe6b78800,0,0xf4e4,1,0xe6b78e00,11161,0xf4e5,1,0xc3830000,22,0x8559
,1,0xe686ab00,0,0x9ce0,1,0xe68bbe00,308,0x8f45,1,0xe6bd9c00,0,0x90f6
,1,0xe8829d00,0,0x8acc,1,0xe6b78f00,7588,0xf4e6,1,0xe6b79000,0,0xf4e7
,1,0xe99e9800,0,0x8fe2,1,0xe9a3ab00,0,0xe94b,1,0xe9a8be00,0,0xe980
,1,0xe5a79400,1377,0x88cf,1,0xe6b79f00,0,0xf4e8,1,0xe5b1ba00,0,0x9884
,1,0xe7a38600,0,0xe1f5,1,0xc9bd0000,2138,0x85ee,1,0xd0950000,2194,0x8445
,1,0xe6b7a900,7598,0xf4e9,1,0xe6b8b600,0,0xf4eb,1,0xe6b89e00,10752,0xf4ec
,1,0xe6b8a200,10731,0xf4ed,1,0xe6b8a700,0,0xf4ee,1,0xe6929100,0,0xf36e
,1,0xe6b8b200,0,0xf4f0,1,0xe69cb700,0,0x9e56,1,0xe6b8bc00,7641,0xf4f1
,1,0xe3809700,396,0x81d9,1,0xe7b8b200,4657,0xe380,1,0xe6b98800,9115,0xf4f2
,1,0xe9af9100,0,0xe9c4,1,0xe5819c00,0,0x92e2,1,0xe6b98900,0,0xf4f3
,1,0xe6988400,0,0xf395,1,0xe6b98b00,0,0xf4f4,1,0xe6a2aa00,0,0xf3f0
,1,0xe6a7bd00,0,0x9185,1,0xe7b99200,0,0xedb3,1,0xe38b9d00,0,0x86e6
,1,0xe6b98c00,0,0xf4f5,1,0xe9b58400,4937,0xe9f5,1,0xe9ba9700,0,0x97ed
,1,0xe6b98f00,0,0xf4f6,1,0xe5be8000,1004,0x899d,1,0xe6a38a00,0,0x9e97
,1,0xe6a89d00,0,0xf46c,1,0xe6b99100,0,0xf4f7,1,0xe7bf8500,7519,0xe3c0
,1,0xe6b99300,9758,0xf4f8,1,0xcb9e0000,2853,0x8691,1,0xe98eac00,0,0xe84d
,1,0xe58d8200,0,0xf170,1,0xe5929500,0,0xf18e,1,0xefbda600,5347,0xa6
,1,0xe59cbb00,0,0x9aab,1,0xe78e8700,0,0x97a6,1,0xcebb0000,124,0x83c9
,1,0xe798ad00,7004,0xecd5,1,0xe8aa8200,0,0xe66f,1,0xe6b99400,0,0xf4f9
,1,0xe6ba9300,0,0xf540,1,0xe999b200,4838,0xe8a2,1,0xefbe8600,16,0xc6
,1,0xe4bda400,0,0xf060,1,0xe5a2ae00,0,0x9ad6,1,0xe687b800,0,0x8c9c
,1,0xe6b99600,0,0x8cce,1,0xe79ea000,0,0xe1d3,1,0xe6baa700,0,0xf541
,1,0xe6babf00,0,0xf543,1,0xe6bb8a00,0,0xf545,1,0xe8bfae00,0,0xeed4
,1,0xe6bb9900,0,0xf546,1,0xe5a38e00,324,0x8882,1,0xc38a0000,0,0x8560
,1,0xe68dab00,0,0x9d7e,1,0xe692be00,7117,0xeafb,1,0xe8848a00,0,0x90d2
,1,0xe8899d00,0,0xe47b,1,0xe6bcb500,0,0xf547,1,0xe9a08500,0,0x8d80
,1,0xe9a59800,0,0xefa8,1,0xe6bbab00,0,0xf548,1,0xe5a98100,0,0x984b
,1,0xe6bbbb00,0,0xf54a,1,0xe6bc8a00,0,0xf54b,1,0xe698b100,0,0xeb54
,1,0xe2978700,0,0x819e,1,0xe7af9900,8766,0xed86,1,0xe381a400,11474,0x82c2
,1,0xe7b9bf00,0,0xe391,1,0xe6bc8c00,0,0xf54c,1,0xe6bc9800,0,0xf54d
,1,0xe6bca500,0,0xf54e,1,0xe5b48700,0,0x9092,1,0xe6999100,0,0xf39a
,1,0xe5bead00,0,0x9c73,1,0xe6a3b700,0,0xf444,1,0xe3828400,159,0x82e2
,1,0xe6bcb600,7689,0xf54f,1,0xe6bcbc00,8454,0xf550,1,0xeaac9a00,0,0xf551
,1,0xe5838900,4018,0x9940,1,0xe6bd9200,0,0xf552,1,0xe58daf00,0,0x894b
,1,0xe5bf8d00,661,0x9445,1,0xe6bd9700,0,0xf553,1,0xe789a100,0,0x89b2
,1,0xe6aebd00,0,0xf4a2,1,0xe38d8a00,412,0x876d,1,0xe6bd9a00,10160,0xf554
,1,0xe6bda800,0,0xf556,1,0xe6be9800,0,0xf557,1,0xe6bdbd00,0,0xf558
,1,0xe593a200,0,0x9a4e,1,0xeaabb400,0,0xf3a1,1,0xe78a8100,0,0xe0b3
,1,0xe6be9000,0,0xf559,1,0xe6be9600,0,0xf55a,1,0xe799ba00,0,0x94ad
,1,0xe8ab8f00,9944,0x907a,1,0xcba50000,525,0x8680,1,0xe6bebe00,0,0xf55b
,1,0xe6be9f00,9063,0xf55c,1,0xe4b99e00,3973,0x8cee,1,0xe59ea800,0,0xf1ee
,1,0xe5a3bb00,6230,0x9ae4,1,0xe6bea500,0,0xf55d,1,0xe79a9a00,0,0xe1ab
,1,0xe79fad00,3222,0x925a,1,0xe6beb500,0,0xf560,1,0xe6bf8800,9492,0xf561
,1,0xe6bf8900,0,0xf562,1,0xe6bf9a00,10342,0xf563,1,0xe59f8800,932,0x886e
,1,0xe5a49b00,6232,0x9aea,1,0xe6bf9e00,11126,0xf564,1,0xe6bb8300,0,0xf544
,1,0xe8808400,7776,0xe3cc,1,0xe7a5a000,0,0xe24b,1,0xe7aab300,0,0xed74
,1,0xe88fbd00,9606,0xe4c1,1,0xe9a19200,0,0xef9c,1,0xe9a6a500,0,0xe965
,1,0xe6bfa900,0,0xf565,1,0xe68a8500,0,0xeae6,1,0xc3910000,65,0x8567
,1,0xe694ab00,0,0x9dba,1,0xe7a68000,0,0xe268,1,0xe7ab9300,0,0xe28a
,1,0xe1bdb000,8110,0x8665,1,0xe382b100,171,0x8350,1,0xe9a78500,0,0x8977
,1,0xe9ac9800,0,0xe9a1,1,0xe6bfbc00,10892,0xf567,1,0xe5b08100,0,0x9595
,1,0xe7808000,0,0xf568,1,0xe5baa700,0,0x8dc0,1,0xe69fb100,0,0x928c
,1,0xe7808700,0,0xf569,1,0xe3839100,160,0x8370,1,0xd0a30000,9078,0x8454
,1,0xe8a0b600,4863,0xe5c2,1,0xe7808a00,0,0xf56a,1,0xe5849600,0,0x994f
,1,0xe589a900,0,0x9994,1,0xe780a300,8148,0xf56b,1,0xe780ba00,0,0xf56e
,1,0xe6a5a400,0,0xeb97,1,0xe780bc00,7338,0xf56f,1,0xe89c8300,0,0xe587
,1,0xe9818d00,0,0x95d5,1,0xe8a6a900,0,0xe64e,1,0xe98bb300,9353,0x9292
,1,0xe58a8900,0,0x97ab,1,0xe7818300,0,0xf570,1,0xe594af00,0,0x9742
,1,0xe7818700,0,0xf571,1,0xe78b8e00,0,0xe0be,1,0xe7818b00,0,0xf572
,1,0xe795b400,0,0xe166,1,0xe9878000,0,0xe7d4,1,0xe3b69a00,9370,0xf573
,1,0xe7819400,0,0xf574,1,0xe996b900,0,0xe886,1,0xe5958f00,0,0x96e2
,1,0xe4baab00,560,0x8b9d,1,0xe59fb500,941,0x8872,1,0xe684bf00,1938,0x9cc3
,1,0xe7969400,0,0xe16a,1,0xe288a800,2048,0x81c9,1,0xe7a0ba00,0,0x9376
,1,0xe781a500,0,0xf575,1,0xe781a900,0,0xf576,1,0xe781ac00,9138,0xf577
,1,0xe4bb8b00,1070,0x89ee,1,0xe5a09500,0,0x91c2,1,0xe5a5a800,1362,0x8fa7
,1,0xe781ae00,0,0xf578,1,0xe2898800,2055,0x81ec,1,0xe28e9b00,0,0x829d
,1,0xe886a400,0,0xe452,1,0xe781b600,8725,0xf579,1,0xe781be00,0,0xf57a
,1,0xe7828600,9825,0xf57c,1,0xe9a7b200,0,0xe971,1,0xe7829500,0,0xf57d
,1,0xe68b9200,0,0x8b91,1,0xe690a500,0,0xf365,1,0xe695b800,0,0x9dc9
,1,0xe7829700,7891,0xf57e,1,0xe299a100,86,0x83bb,1,0xe7b1b300,0,0x95c4
,1,0xe383be00,0,0x8153,1,0xe9a89200,0,0x919b,1,0xe782bb00,10695,0xf580
,1,0xeaac9d00,0,0xf581,1,0xe5b18e00,0,0x9b9d,1,0xc3980000,0,0x856d
,1,0xe5bbb400,0,0x9c4d,1,0xe6a0be00,0,0xf3dc,1,0xe29f9400,0,0x84c3
,1,0xe8979d00,8315,0xe559,1,0xe7829f00,0,0xf582,1,0xe981ba00,0,0x88e2
,1,0xe5809000,0,0xf078,1,0xe782b100,0,0xf583,1,0xe783ac00,0,0xf585
,1,0xe5bc9400,682,0x92a2,1,0xe7839100,0,0xf587,1,0xe6a6b100,1876,0x9ee1
,1,0xe78bbb00,0,0xec8b,1,0xe7839300,8474,0xf588,1,0xd0aa0000,0,0x845b
,1,0xe8a7b600,4933,0xee9c,1,0xe5868300,0,0xf148,1,0xe58b9600,0,0x87e4
,1,0xe590a900,0,0x99e4,1,0xe595bc00,0,0x9a65,1,0xe7878800,6714,0x9395
,1,0xe78c9b00,0,0x96d2,1,0xe7839c00,0,0xf589,1,0xe8a38300,0,0xe5df
,1,0xe9888d00,2843,0x93dd,1,0xe98da000,4801,0xe848,1,0xe8b2bc00,6033,0x935c
,1,0xe5918900,0,0x8ce0,1,0xefbc9a00,108,0x8146,1,0xe7848300,10164,0xf58a
,1,0xe7848400,0,0xf58b,1,0xe6b29700,0,0xf4bc,1,0xe6b7aa00,0,0x9fcb
,1,0xe79cb400,0,0xecee,1,0xe881be00,0,0x9857,1,0xe8b39c00,0,0x8e92
,1,0xe7848600,0,0xf58c,1,0xe99db900,0,0xe8d6,1,0xe59c8f00,11163,0x8c97
,1,0x6f000000,1109,0x6f,1,0xe686ac00,0,0x9cdb,1,0xe68bbf00,1735,0x9d5c
,1,0xe28a9500,0,0x81cf,1,0xe7a2a700,0,0x95c9,1,0xe294bb00,0,0x84b3
,1,0xe7848700,0,0xf58d,1,0xe7848800,0,0xf58e,1,0xe7848c00,0,0xf58f
,1,0xe3b78000,8414,0xf590,1,0xe6878c00,0,0x9ce2,1,0xe68c9f00,0,0x8bb2
,1,0xe784af00,0,0xf591,1,0xe784b100,8346,0xf592,1,0xe7a89a00,0,0x9274
,1,0xe7adad00,0,0xed80,1,0xe8bf8200,0,0x8949,1,0xe9a48c00,0,0x8961
,1,0xe9a99f00,0,0xe985,1,0xe7859000,10735,0xf593,1,0xe7858a00,0,0xf594
,1,0xe6929200,0,0x8e54,1,0xe5b7ae00,546,0x8db7,1,0xe69cb800,0,0x9e55
,1,0xe7ae8d00,0,0xe2b0,1,0xe3809800,194,0x81d6,1,0xe7859300,0,0xf595
,1,0xe7869600,0,0xf598,1,0xe9af9200,0,0xe9c5,1,0xe7869b00,0,0xf59a
,1,0xe9b9b800,966,0x8cb2,1,0xe786a000,0,0xf59b,1,0xc39f0000,1278,0x8574
,1,0xe6a2ab00,0,0xf3f1,1,0xe7b48000,1884,0x8b49,1,0xe786a200,0,0xf59c
,1,0xe38b9e00,0,0x86e7,1,0xe786ae00,9245,0xf59d,1,0xe988ba00,0,0xef47
,1,0xe786b300,0,0xf59f,1,0xe58ca300,0,0x99b8,1,0xe5be8100,1005,0x90aa
,1,0xe6a38b00,0,0x8afb,1,0xe6a89e00,4111,0x9ee2,1,0xe78da800,0,0xe0d5
,1,0xe7bf8600,7518,0xe3c1,1,0xe8a49000,0,0x8a8c,1,0xd0b10000,161,0x8471
,1,0xe98ead00,0,0xe84e,1,0x58000000,364,0x58,1,0xe5929600,0,0x87fb
,1,0xefbda700,5363,0xa7,1,0xeaac9f00,0,0xf5a0,1,0xe7879900,0,0xf5a3
,1,0xe2809c00,0,0x8167,1,0xe7879c00,0,0xf5a4,1,0xe7888700,0,0xf5a5
,1,0xe3b88500,0,0xf5a6,1,0xefa98a00,0,0xf5a7,1,0xe999b300,2970,0x92c2
,1,0xefbe8700,64,0xc7,1,0xe788ab00,0,0xf5a8,1,0xe682a600,0,0x8978
,1,0xe6b48400,0,0xebdf,1,0xe6b99700,0,0xf4fa,1,0xe6beaa00,0,0xe059
,1,0xe7a3b400,0,0xe243,1,0xe788b400,0,0xf5a9,1,0xe788b800,0,0xf5aa
,1,0xe8bfaf00,0,0xe78d,1,0xe4be8500,8653,0xf06b,1,0xe6838600,0,0x9cad
,1,0xe788b900,0,0xf5ab,1,0xe5adb500,6334,0x9b7a,1,0xe692bf00,4058,0xeafc
,1,0xe4b8ac00,0,0xf5ac,1,0xe7898200,7942,0xf5ad,1,0xe88eb100,0,0x9789
,1,0xe9a08600,0,0x8f87,1,0xe7899300,0,0xf5ae,1,0xe7899700,0,0xf5af
,1,0xe789af00,0,0xf5b3,1,0xe68e8c00,503,0x8fb6,1,0xe5b3a800,0,0x89e3
,1,0xe789b800,0,0xf5b4,1,0xe789bf00,8934,0xf5b5,1,0xe88f9100,8091,0xee40
,1,0xe381a500,8876,0x82c3,1,0xe3b38300,9014,0xf4b9,1,0xe78a8e00,0,0xf5b6
,1,0xe78aad00,10768,0xf5b8,1,0xe9b5b200,5346,0xea46,1,0xe587bd00,0,0x949f
,1,0xe6999200,0,0x8e4e,1,0xe5beae00,0,0x94f7,1,0xe78aae00,0,0xf5b9
,1,0xe3828500,0,0x82e3,1,0xe78b8100,0,0xf5bc,1,0x41000000,7,0x41
,1,0xe984b400,0,0xeef1,1,0xe5838a00,4033,0x9941,1,0xe5889d00,3619,0x8f89
,1,0xe58db000,0,0x88f3,1,0xe3b9a000,10661,0xf5bd,1,0xc3a60000,4076,0x857b
,1,0xe789a200,3911,0x9853,1,0xe6aebe00,7430,0xf4a3,1,0xe9808100,4711,0x9197
,1,0xe9859400,0,0x908c,1,0xe8aab000,9932,0x924e,1,0xe78b8c00,0,0xf5be
,1,0xe78bb300,0,0xf5c1,1,0xe78bba00,0,0xf5c2,1,0xe4b8bf00,1056,0x98a6
,1,0xe78a8200,6765,0xe0b2,1,0xe78c8700,8368,0xf5c3,1,0xe794a800,3978,0x9770
,1,0xe799bb00,0,0x936f,1,0xe8ab9000,0,0xeea8,1,0xd0b80000,3176,0x8479
,1,0xe78c9200,0,0xf5c4,1,0xe78c9800,0,0xf5c5,1,0xe4b99f00,6906,0x96e7
,1,0xe4beb200,0,0xf06f,1,0xe683b300,1122,0x917a,1,0xe78c9900,7558,0xf5c6
,1,0xe79a9b00,0,0xecdf,1,0xe79fae00,0,0xe1e2,1,0xe3ba8300,0,0xf5c7
,1,0xe78cb900,10127,0xf5c8,1,0xe78cac00,0,0xf5c9,1,0xe78cb100,10343,0xf5ca
,1,0xe4bf9200,0,0xf074,1,0xe5a49c00,6234,0x96e9,1,0xe78cb300,0,0xf5cb
,1,0xe6bb8400,0,0x9fe9,1,0xe8808500,0,0x8ed2,1,0xc5870000,0,0x85c0
,1,0xe88aab00,0,0xe48b,1,0xe8bc8900,275,0x8dda,1,0xe9a19300,0,0xef9d
,1,0xe78cbd00,10539,0xf5cc,1,0xe9abb900,0,0xefb8,1,0xe78d9200,0,0xf5cd
,1,0xe5afa200,0,0x9b8a,1,0xe694ac00,0,0x9dae,1,0xe7a68100,0,0x8bd6
,1,0x2a000000,0,0x2a,1,0xe1bdb100,2206,0x8666,1,0xe382b200,4456,0x8351
,1,0xe9a78600,0,0x8bec,1,0xe3ba9400,0,0xf5ce,1,0xe78dab00,0,0xf5cf
,1,0xe5b08200,10994,0x90ea,1,0xe78dac00,0,0xf5d0,1,0xeaaca300,0,0xf5d1
,1,0xe5bfbb00,0,0x9c76,1,0xe784bc00,0,0x8fc4,1,0xe3839200,406,0x8371
,1,0xe89ba400,0,0x94b8,1,0xe980ae00,7371,0x91df,1,0xe78dae00,0,0xf5d2
,1,0xe5849700,11030,0xf142,1,0xe589aa00,0,0x9992,1,0xe5bb8800,0,0x9bfa
,1,0xe7808900,0,0xe062,1,0xe7859c00,0,0xec76,1,0xe78aaf00,1652,0x94c6
,1,0xe78daf00,0,0xf5d3,1,0xe9818e00,83,0x89df,1,0xe8a6aa00,0,0x9065
,1,0xe78db100,0,0xf5d4,1,0xe9bd9200,0,0xea8f,1,0xe78db700,0,0xf5d5
,1,0xe78e8100,0,0xf5d6,1,0xe78e8500,0,0xf5d7,1,0xc3ad0000,0,0x8583
,1,0xe790a200,0,0x91f4,1,0xe78e9c00,0,0xf5db,1,0xe9878100,0,0xe7d5
,1,0xe8ac9d00,0,0x8ed3,1,0xe78e9e00,0,0xf5dc,1,0xe78ea500,0,0xf5dd
,1,0xe5959000,0,0x8846,1,0xe4baac00,8620,0x8b9e,1,0xe59fb600,0,0x8873
,1,0xe78ea800,0,0xf5de,1,0xe78eb500,0,0xf5df,1,0xe288a900,7679,0x81bf
,1,0xe78f8500,0,0xf5e4,1,0xe78f8b00,0,0xf5e5,1,0xd0bf0000,2954,0x8481
,1,0xe8bcb600,0,0xeecd,1,0xe59b8300,9358,0x9a92,1,0xe6808d00,0,0xeac8
,1,0xe5a5a900,6246,0x9afc,1,0x13000000,1159,0x13,1,0xe6bc9100,2053,0x9ff2
,1,0xe28e9c00,2388,0x829e,1,0xe7a6ae00,0,0xe258,1,0xe78fa700,0,0xf5e7
,1,0xe99d8d00,0,0xef88,1,0xe78fb900,0,0xf5e8,1,0xe7909300,0,0xf5e9
,1,0xe78fba00,0,0xf5ea,1,0xe68b9300,5874,0x91f1,1,0xe690a600,0,0x9d8e
,1,0xe7908100,0,0xf5eb,1,0xe2948f00,0,0x84ac,1,0xe299a200,2546,0x83b9
,1,0xe891ab00,0,0xe4d7,1,0xe790b100,0,0xf5ed,1,0xe9a89300,0,0x8cb1
,1,0xe9ada600,0,0xefc0,1,0xe790b900,8394,0xf5ee,1,0xe5b18f00,0,0x9ba0
,1,0xe6969900,1222,0x97bf,1,0xcf830000,6551,0x83d0,1,0xe7919300,8034,0xf5ef
,1,0xe29f9500,314,0x84c4,1,0xe7918000,0,0xf5f0,1,0xe7bcba00,9102,0xe39e
,1,0xe7919200,0,0xf5f3,1,0xe5809100,0,0x98ec,1,0xe7919d00,0,0xf5f4
,1,0xe791b100,0,0xf5f5,1,0xe5bc9500,0,0x88f8,1,0xe6a19f00,0,0x8e56
,1,0xe6a6b200,0,0x9ec3,1,0xe78bbc00,0,0x9854,1,0xe7928100,8424,0xf5f6
,1,0xe8a2a400,4877,0xe5db,1,0xe7928500,9828,0xf5f7,1,0xe7928800,0,0xf5f8
,1,0xe58b9700,0,0x99a6,1,0xeaaca400,0,0xf5f9,1,0xe595bd00,0,0xf1a9
,1,0xe7878900,0,0xe097,1,0xe78c9c00,0,0xe0c8,1,0xe791af00,0,0xe0e5
,1,0xe8a38400,0,0xe5e0,1,0xe9888e00,421,0x8a62,1,0xe7929200,0,0xf5fa
,1,0xe8b2bd00,0,0xe6c4,1,0xe5918a00,0,0x8d90,1,0xefbc9b00,88,0x8147
,1,0xe59bb000,10167,0x8da2,1,0xe680ba00,0,0x9c8a,1,0xc3b40000,3961,0x858a
,1,0xca8c0000,39,0x8656,1,0xe289b600,1791,0x81ed,1,0xe8ae8a00,2398,0x9dcc
,1,0xe7929900,0,0xf5fc,1,0xe8b8b000,0,0xe6fa,1,0xe99dba00,0,0xe8da
,1,0xe792a000,11178,0xf640,1,0xe6819a00,0,0x9c8b,1,0xe686ad00,0,0xf2ea
,1,0xe6b88b00,4251,0x8f61,1,0xe28a9600,4043,0x81d0,1,0xe792a100,9452,0xf641
,1,0xe294bc00,33,0x84a9,1,0xe8b99000,0,0xe743,1,0xe8bea300,10271,0xe785
,1,0xe9a3ad00,0,0x99aa,1,0xe5a28300,0,0x8bab,1,0xe6878d00,0,0x9cec
,1,0xe792a500,9763,0xf642,1,0xe5b1bc00,6380,0xf0b7,1,0xe792aa00,7822,0xf643
,1,0xe8889200,7914,0x98ae,1,0xe7adae00,0,0xe2ac,1,0xe892b800,0,0x8ff6
,1,0xe792ab00,0,0xf644,1,0xe792b900,0,0xf645,1,0xe792bb00,0,0xf646
,1,0xe792ba00,0,0xf647,1,0xe6929300,0,0x9d9a,1,0xe697a600,0,0x9255
,1,0xe7939600,0,0xf648,1,0xe88e8500,0,0xe4ac,1,0xe3809900,11492,0x81d7
,1,0xe7939800,0,0xf649,1,0xe793af00,0,0xf64b,1,0xeaaca500,0,0xf64d
,1,0xe9b4a600,5022,0xe9f1,1,0xe586b100,3571,0x9973,1,0xe6988600,0,0x8da9
,1,0xe5bda200,1564,0x8c60,1,0xe793ba00,0,0xf64e,1,0xe6a7bf00,0,0x9edb
,1,0xe7b99400,0,0x9044,1,0xe38b9f00,0,0x86e8,1,0xe983a800,9248,0x9594
,1,0xe9b58600,0,0xe9fb,1,0xeaaca600,0,0xf64f,1,0xe58ca400,0,0xf16d
,1,0xe591b700,1280,0x99ec,1,0xe6a38c00,0,0xf3fa,1,0xe6a89f00,0,0x8fbe
,1,0xe794a000,0,0xf650,1,0xe794a400,0,0xf651,1,0xe794aa00,0,0xf652
,1,0xe9899b00,0,0x8994,1,0xe98eae00,0,0x92c1,1,0xe3bd9700,0,0xf653
,1,0xeaaca700,8727,0xf654,1,0xefbda800,5358,0xa8,1,0xe794bd00,0,0xf655
,1,0xe78e8900,0,0x8bca,1,0xe2809d00,46,0x8168,1,0xe285b000,1762,0x86b3
,1,0xe8aa8400,5571,0xe670,1,0xe794be00,0,0xf656,1,0xe7958000,0,0xf657
,1,0xe7958800,0,0xf658,1,0xefbe8800,23,0xc8,1,0xe7959000,0,0xf65a
,1,0xe682a700,0,0x9ca4,1,0xe687ba00,0,0x9cf0,1,0xe2869000,11487,0x81a9
,1,0xe7959200,7730,0xf65b,1,0xe795ac00,0,0xf65c,1,0xe795b200,0,0xf65d
,1,0xe99a9400,0,0x8a75,1,0xe8bfb000,0,0x8f71,1,0xe4be8600,10482,0x98d2
,1,0xe6838700,0,0x93d5,1,0xe5a8a300,356,0x8892,1,0xe795b100,0,0xf65e
,1,0xe7968100,0,0xf662,1,0xeaaca800,0,0xf663,1,0xe8899f00,0,0xe47d
,1,0xe7968c00,0,0xf664,1,0xe796a200,0,0xf666,1,0xe3bdb700,0,0xf667
,1,0xe9aaad00,0,0xe98c,1,0xe796b000,7690,0xf668,1,0xe5ae9600,0,0xf09f
,1,0xe5b3a900,1472,0x9bb4,1,0xe698b300,0,0xeb55,1,0xe2978900,2155,0x825a
,1,0xe796b700,8365,0xf669,1,0xe381a600,2259,0x82c4,1,0xe796bf00,8340,0xf66a
,1,0xe7978600,0,0xf66c,1,0xe7978f00,8008,0xf66d,1,0xe7979300,10972,0xf66e
,1,0xe587be00,1477,0x9982,1,0xe7979d00,0,0xf66f,1,0xe69ea600,3726,0x9e64
,1,0xe6a3b900,0,0x9ea8,1,0xe3828600,2274,0x82e4,1,0xc59c0000,4676,0x85db
,1,0xe7979f00,9178,0xf670,1,0xe797a000,8176,0xf671,1,0xefa98900,0,0xec74
,1,0xe797ac00,0,0xf673,1,0xe58db100,0,0x8aeb,1,0xe69f8600,0,0x9e75
,1,0xc8b90000,345,0x866b,1,0xe789a300,0,0xf5b0,1,0xe6aebf00,0,0x9361
,1,0xe797ae00,9185,0xf674,1,0xe8a59e00,0,0xe5fc,1,0xe797b100,0,0xf675
,1,0xe797b900,8614,0xf676,1,0xe7989800,0,0xf678,1,0xe7988f00,0,0xf67a
,1,0xe598b700,6582,0xf1bf,1,0xe3beae00,0,0xf67b,1,0xe6af9f00,4151,0x9f7a
,1,0xe6b4b200,0,0x8f46,1,0xe799bc00,0,0xe1a2,1,0xeaaca900,0,0xf67c
,1,0xe7989300,0,0xf67d,1,0xe8b5b700,0,0x8b4e,1,0xe5948400,0,0x8953
,1,0xeaac9600,0,0xf4da,1,0xe59eaa00,0,0x9ab8,1,0xe5a3bd00,1333,0x9ae6
,1,0xe7958900,0,0xe159,1,0xe6baa500,7665,0x9fee,1,0xe79faf00,0,0x8bb8
,1,0xe884b900,0,0x92af,1,0xe9968e00,0,0xef6d,1,0xe7989b00,0,0xf67e
,1,0xe9a0b400,114,0x896f,1,0xe7989c00,0,0xf680,1,0xeaacaa00,0,0xf681
,1,0xe798a500,0,0xf682,1,0xe6bb8500,0,0x96c5,1,0xe8808600,0,0xe3cb
,1,0xe7a5a200,0,0x9449,1,0xe88aac00,0,0xe48e,1,0xe8bc8a00,0,0xe768
,1,0xe9a19400,0,0x8ae7,1,0xe798a800,0,0xf683,1,0xe798b300,0,0xf685
,1,0xeaacab00,0,0xf686,1,0xe68f9a00,0,0x9767,1,0xe3bf8900,9796,0xf687
,1,0xe7998100,0,0xf688,1,0xe2989600,2170,0x83e2,1,0xe1bdb200,0,0x866f
,1,0xe382b300,0,0x8352,1,0xe3b49100,8841,0xf4ef,1,0xe9ac9a00,0,0xe9a2
,1,0xeaacac00,10893,0xf689,1,0xe5b08300,0,0xf0ab,1,0xe6958d00,0,0x9dc4
,1,0xe69aa000,0,0xeb63,1,0xe69fb300,6186,0x96f6,1,0xe7998900,0,0xf68a
,1,0xe3839300,2292,0x8372,1,0xe7999500,0,0xf68b,1,0xe799ae00,0,0xf68d
,1,0xe3bf9700,0,0xf68c,1,0xe5849800,11034,0x98d4,1,0xe79a9500,0,0xf68e
,1,0xe5bb8900,0,0x97f5,1,0xe6a09300,7148,0x90f0,1,0xe79a9c00,0,0xf68f
,1,0xe78ab000,0,0xf5ba,1,0xe79aa100,9083,0xf690,1,0xc5a30000,3029,0x85d5
,1,0xe986a200,0,0xe7cd,1,0xe8abbe00,0,0x91f8,1,0xe79aa700,11147,0xf692
,1,0xe79aa800,0,0xf693,1,0xe594b100,0,0x8fa5,1,0xe79aaf00,0,0xf694
,1,0xe78b9000,0,0x8ccf,1,0xeaacad00,0,0xf695,1,0xe795b600,0,0xe163
,1,0xeaacae00,0,0xf697,1,0xe79ba800,9995,0xf698,1,0xeaacaf00,0,0xf69a
,1,0xe996bb00,0,0xe885,1,0xe79c9700,0,0xf69b,1,0xe4baad00,8629,0x92e0
,1,0xe59fb700,0,0x8eb7,1,0xe7918300,0,0xf5f1,1,0xe79c9a00,10170,0xf69c
,1,0xe288aa00,1786,0x81be,1,0xe880b300,0,0x8ea8,1,0xe79cad00,0,0xf69d
,1,0xe79cb500,0,0xf69e,1,0xeaacb000,0,0xf69f,1,0xe4bb8d00,9405,0x98b9
,1,0xe6808e00,0,0x9c83,1,0xe5a5aa00,0,0x9244,1,0xe5aabd00,0,0x9b5f
,1,0xe79c8900,0,0x94fb,1,0xe28e9d00,0,0x8461,1,0xe480b900,7883,0xf6a0
,1,0xe88bb900,0,0xe499,1,0xe8bd9700,5036,0xe77d,1,0xeaacb200,0,0xf6a1
,1,0xeaacb100,0,0xf6a2,1,0xe5a68a00,0,0x9444,1,0xe68b9400,0,0x9d55
,1,0xe5b0b000,0,0xf0ad,1,0xe695ba00,0,0xf385,1,0xe2949000,59,0x84a2
,1,0xe299a300,11500,0x83be,1,0xe891ac00,2219,0x9192,1,0xe79d9800,0,0xf6a3
,1,0xe79daa00,0,0xf6a5,1,0xeaacb400,10419,0xf6a6,1,0xefa5b000,0,0xebc7
,1,0xe5b19000,0,0x9b9f,1,0xe79db200,0,0xf6a7,1,0x7b000000,51,0x7b
,1,0xe79dbc00,8469,0xf6a8,1,0xe29f9600,577,0x84c5,1,0xe7b7a800,8966,0x95d2
,1,0xe79dbd00,8387,0xf6a9,1,0xe981bc00,0,0x97c9,1,0xe5809200,0,0x937c
,1,0xe585a500,0,0x93fc,1,0xeaacb300,0,0xf6aa,1,0xe69c8d00,1269,0x959e
,1,0xe4819800,0,0xf6ab,1,0xe79e9a00,8719,0xf6ac,1,0xe78bbd00,4483,0x9482
,1,0xe79e9f00,0,0xf6ad,1,0xe79ea200,0,0xf6ae,1,0xe8a7b800,0,0xe65c
,1,0xe5868500,0,0x93e0,1,0xe58b9800,3676,0x8aa8,1,0xe590ab00,0,0x8adc
,1,0xe595be00,0,0x9a61,1,0xe6a79300,0,0x9ec8,1,0xe78c9d00,0,0xe0ca
,1,0xe791b000,0,0xe0f1,1,0xe8a38500,5234,0x9195,1,0xc5aa0000,112,0x8597
,1,0xcc820000,2898,0x867e,1,0xe79f9e00,0,0xf6b1,1,0xe79f9f00,9290,0xf6b2
,1,0xefbc9c00,1019,0x8183,1,0xe79fa400,10200,0xf6b3,1,0xe6ad8600,0,0xebbe
,1,0xe6b29900,4221,0x8db9,1,0xe6b7ac00,1988,0x9fc1,1,0xe289b700,1784,0x81ee
,1,0xe79fa600,9398,0xf6b4,1,0xe8b39e00,0,0x8fdc,1,0xe998a800,0,0xe895
,1,0xe79faa00,0,0xf6b5,1,0xe4bc9a00,0,0x89ef,1,0xe5a1a400,0,0x8878
,1,0xe686ae00,0,0x9ce1,1,0xe7988300,0,0xf677,1,0xe28a9700,87,0x81d1
,1,0xe7a2a900,0,0x90d7,1,0xe887b300,0,0x8e8a,1,0xe79fac00,0,0xf6b6
,1,0xe4829300,9628,0xf6b7,1,0xe9a3ae00,0,0x9f5a,1,0xe79fb000,0,0xf6b8
,1,0x64000000,0,0x64,1,0xe5acaa00,0,0x9b6c,1,0xeaacb500,0,0xf6bb
,1,0xe7a08500,0,0xf6bc,1,0xe7a89c00,0,0x97c5,1,0xe7adaf00,0,0xed81
,1,0xe8bf8400,2593,0x9698,1,0xe7a08600,0,0xf6bd,1,0xe7a08900,0,0xf6be
,1,0xe9aeb400,0,0xe9be,1,0xe68d8100,0,0xf34e,1,0xe7a08d00,0,0xf6bf
,1,0xe697a700,0,0x8b8c,1,0xe69cba00,0,0x8af7,1,0xe7ae8f00,4582,0xe2b5
,1,0xe7b3a200,2599,0xe2f5,1,0xd1800000,2328,0x8482,1,0xe89dbf00,0,0x9488
,1,0xe9af9400,0,0xe9c9,1,0xe7a09900,0,0xf6c0,1,0xe586b200,0,0x9974
,1,0xe6988700,0,0x8fb8,1,0xe5bda300,0,0xf2b1,1,0xe6a2ad00,4069,0x9e88
,1,0xe7b48200,0,0xe340,1,0xe7b99500,0,0x9155,1,0xe38ba000,370,0x86e9
,1,0xe8a3b200,0,0xe5ea,1,0xe988bc00,5143,0xef48,1,0xe7a0a100,0,0xf6c1
,1,0xe7a0ac00,0,0xf6c2,1,0xe5be8300,0,0x9c67,1,0xe6a38d00,0,0x9e9e
,1,0xe7a18700,0,0xf6c3,1,0xe6adb300,1925,0x8dce,1,0xe792bd00,0,0x8ea3
,1,0xe8a49200,0,0x964a,1,0xe396a600,0,0xf1a1,1,0xe7a1a400,0,0xf6c4
,1,0xe58d8500,0,0x99bf,1,0xe7a1aa00,8031,0xf6c5,1,0xefbda900,5375,0xa9
,1,0xeaacb700,0,0xf6c6,1,0xe78e8a00,0,0xf5d8,1,0xe7a29400,0,0xf6c8
,1,0xe285b100,2009,0x86b4,1,0xe8aa8500,0,0xe66e,1,0xc5b10000,0,0x85d4
,1,0xe7a2a400,0,0xf6c9,1,0x4d000000,837,0x4d,1,0xefbe8900,90,0xc9
,1,0xe7a29d00,0,0xf6ca,1,0xe5a2b100,6213,0xf250,1,0xe7a29e00,0,0xf6cb
,1,0xe2869100,0,0x81aa,1,0xe7a29f00,0,0xf6cc,1,0xe7a2bb00,0,0xf6cd
,1,0xe7a38800,0,0xf6ce,1,0xe99a9500,9568,0xe8a6,1,0xe7a38c00,9666,0xf6cf
,1,0xe7a39500,10485,0xf6d1,1,0xe5a39100,6220,0x9ad9,1,0xe6889b00,0,0x9cfc
,1,0xe68dae00,0,0x9098,1,0xe7a3a000,0,0xf6d2,1,0xe7a3a100,8517,0xf6d3
,1,0xe7a9a900,4538,0xe272,1,0xe7a3a600,8078,0xf6d4,1,0xe9a08800,0,0x907b
,1,0xe7a3ba00,0,0xf6d6,1,0xeaacb900,0,0xf6d9,1,0xe7a49000,0,0xf6da
,1,0xe5ae9700,1422,0x8f40,1,0xe5b3aa00,1473,0x9bb9,1,0xe5b8bd00,1510,0x9658
,1,0xe7a4b000,8550,0xf6dc,1,0xe88f9300,0,0x89d9,1,0xe381a700,473,0x82c5
,1,0xe899b900,0,0x93f8,1,0xe9ab8e00,9206,0xefb6,1,0xc7920000,3498,0x84f5
,1,0xe582ac00,9709,0x8dc3,1,0xe7a4a500,0,0xf6dd,1,0xe7a4bb00,0,0xf6de
,1,0xe7a58a00,0,0xf6df,1,0xe6a3ba00,1841,0x8abb,1,0xe3828700,2280,0x82e5
,1,0xe7a59800,0,0xf6e0,1,0xd1870000,4710,0x8489,1,0xe9b18100,0,0xefcf
,1,0xe5838c00,7301,0x87c3,1,0xe7a59b00,11101,0xf6e1,1,0xe4848500,0,0xf6e2
,1,0xe7a5b200,8567,0xf6e4,1,0xe6a49a00,0,0x9ead,1,0xe7a69400,0,0xf6e5
,1,0xe78eb700,0,0xf5e0,1,0xe38d8d00,523,0x8762,1,0x36000000,147,0x36
,1,0xe8aab200,0,0x89db,1,0xe9bc8700,0,0xea87,1,0xe7a69500,0,0xf6e6
,1,0xe593a500,0,0x9a46,1,0xe598b800,6565,0x9a7e,1,0xe6aa8d00,0,0x9ef8
,1,0xe7a69600,0,0xf6e7,1,0xe6b4b300,7545,0x9fac,1,0xe799bd00,0,0x9492
,1,0xe8ab9200,0,0x97c8,1,0xe7a69b00,0,0xf6e8,1,0xe7a6a100,8580,0xf6e9
,1,0xe7a6a900,0,0xf6ea,1,0xeaac9700,0,0xf4dc,1,0xe7a78200,0,0xf6ed
,1,0xe683b500,0,0xf2d7,1,0xe7958a00,0,0xe158,1,0xe79a9d00,0,0xece0
,1,0xe6bfb900,7764,0xec58,1,0xe7a78700,9414,0xf6ee,1,0xe9968f00,0,0x895b
,1,0xe99ba200,2996,0x97a3,1,0xe7a78c00,8218,0xf6ef,1,0xe59f8b00,927,0x9684
,1,0xe6849500,0,0x9cb1,1,0xe689a800,0,0x9d4d,1,0xe68ebb00,0,0x917e
,1,0xe7a78d00,0,0xf6f0,1,0xe7a79600,10512,0xf6f1,1,0xe88aad00,2159,0x946d
,1,0xe4858800,0,0xf6f2,1,0xe9a19500,0,0x8cb0,1,0xe9a6a800,0,0x8a5d
,1,0xe9abbb00,0,0xe99f,1,0xeaacbb00,0,0xf6f3,1,0xe68f9b00,1188,0x8ab7
,1,0xeaacbc00,0,0xf6f4,1,0xe4858f00,0,0xf6f5,1,0xe2989700,0,0x83e3
,1,0xe1bdb300,2571,0x8670,1,0xe382b400,0,0x8353,1,0xe9a78800,0,0x8bed
,1,0xe7a88a00,0,0xf6f6,1,0xe7a89100,10894,0xf6f7,1,0xe5b08400,954,0x8ecb
,1,0xe7a89500,0,0xf6f8,1,0xe5baaa00,0,0xf2a2,1,0xe5bfbd00,260,0x8d9a
,1,0x1f000000,0,0x1f,1,0xe3839400,397,0x8373,1,0xe7a89b00,9081,0xf6f9
,1,0xe8a0b900,0,0xe5c3,1,0xe485a300,0,0xf6fb,1,0xe7a8ad00,0,0xf6fc
,1,0xe589ac00,0,0xf15a,1,0xe5bb8a00,1535,0x984c,1,0xe7808b00,0,0xe063
,1,0xe7859e00,0,0xf596,1,0xe78ab100,0,0xf5bb,1,0xe89c8600,0,0xe584
,1,0xe8a19900,3031,0xe5c9,1,0xd18e0000,2829,0x8490,1,0xe7a8b800,0,0xf740
,1,0xe9bd9400,0,0xea90,1,0xe58f9f00,10636,0x99d5,1,0xe7a98c00,0,0xf742
,1,0xe7a99600,0,0xf743,1,0xe6ab9a00,0,0x9f4c,1,0xe790a400,0,0xf5ec
,1,0xe795b700,0,0x93eb,1,0xe7a99900,9818,0xf744,1,0xe7a99c00,9371,0xf745
,1,0xe7a99f00,0,0xf746,1,0xe996bc00,0,0xe884,1,0xe7a9a000,8657,0xf747
,1,0xe4baae00,1065,0x97ba,1,0xe59fb800,0,0xf1f8,1,0xe7918400,6870,0xecac
,1,0xe7a9a700,0,0xf748,1,0xe288ab00,2050,0x81e7,1,0xe7a9b500,0,0xf74a
,1,0xe7a9b800,9443,0xf74b,1,0xe9979c00,0,0xe890,1,0xe8bcb800,0,0x9741
,1,0xe4bb8e00,5660,0x98b8,1,0xc2a20000,1301,0x8191,1,0xe685a200,0,0x969d
,1,0xe68ab500,806,0x92ef,1,0xe6bc9300,4285,0xe041,1,0xe28e9e00,311,0x8462
,1,0xe7a6b000,0,0x9448,1,0xe88bba00,0,0xe495,1,0xe99d8f00,9622,0xef8a
,1,0xe7aa8200,0,0xf74c,1,0xe7aa8a00,0,0xf74d,1,0xe5a68b00,775,0x888b
,1,0xe68b9500,0,0xf342,1,0x8000000,277,0x8,1,0xe7aa9000,0,0xf74e
,1,0xe7aaa300,9210,0xf74f,1,0xe299a400,0,0x83b7,1,0xe891ad00,0,0xe4d1
,1,0xe7aaac00,10387,0xf750,1,0xeaacbe00,0,0xf751,1,0xe486b400,0,0xf752
,1,0xe7aabc00,0,0xf754,1,0xe5b19100,0,0x8bfb,1,0xe5b6a400,1496,0x989a
,1,0xe5bbb700,0,0x92ec,1,0xe486bf00,0,0xf756,1,0xe29f9700,0,0x84c6
,1,0xe7b7a900,0,0x8ac9,1,0xe7ab8c00,8386,0xf757,1,0xe981bd00,0,0xe7af
,1,0xe5809300,0,0xf079,1,0xe7ab9100,9882,0xf758,1,0xe58ab900,0,0x8cf8
,1,0xe5bc9700,0,0x95a4,1,0xc4830000,544,0x85c7,1,0xe6a6b400,0,0x9ed6
,1,0xe7b88900,8973,0xe375,1,0xe89d9300,0,0xe59f,1,0xe7aba700,0,0xf759
,1,0xe7aba800,0,0xf75a,1,0xe5868600,0,0x897e,1,0xe58b9900,0,0x96b1
,1,0xe590ac00,0,0x99df,1,0xe6a28100,0,0x97c0,1,0xe7878b00,0,0xf5a1
,1,0xe6aca700,0,0x89a2,1,0xe6b1ba00,7468,0x8c88,1,0xe7abb400,0,0xf75b
,1,0xe9889000,0,0xef44,1,0xe8adac00,0,0xe6a0,1,0xe8b2bf00,0,0x9666
,1,0xeaacbf00,0,0xf75c,1,0xefbc9d00,1016,0x8181,1,0xe59bb200,548,0x88cd
,1,0xe6ad8700,0,0x9f5b,1,0xe6b29a00,0,0x9f8d,1,0xe797a400,0,0xeccd
,1,0xe79cb700,0,0xe1c5,1,0xe8ae8c00,0,0xe6a5,1,0xeaad8000,0,0xf75d
,1,0xe7ac8700,0,0xf75e,1,0xe99dbc00,0,0xe8d8,1,0xe59c9200,0,0x8980
,1,0xeaad8100,0,0xf75f,1,0xe7acbd00,0,0xf760,1,0xe7aca700,0,0xf761
,1,0xe6bda000,0,0xf555,1,0xe882a100,2102,0x8cd2,1,0xe887b400,6082,0x9276
,1,0xe7acaa00,0,0xf762,1,0xe99e9c00,8265,0xe8df,1,0xe9a3af00,0,0x94d1
,1,0xe5a28500,0,0x9ace,1,0xc2a90000,0,0x8545,1,0xe7acae00,0,0xf763
,1,0xe6be8000,0,0xe047,1,0xe7a38a00,0,0xe1fb,1,0xe7acb100,0,0xf765
,1,0xe7adb000,0,0xe2a9,1,0xe8bf8500,4606,0x9076,1,0xe487a600,9572,0xf766
,1,0xcc9e0000,3405,0x8697,1,0xe487b300,9585,0xf767,1,0xe7adbf00,0,0xf768
,1,0xe6929500,0,0x9d99,1,0xe5b7b100,238,0x8cc8,1,0xe7ad8100,0,0xf769
,1,0xe88e8700,8061,0xe4b4,1,0xe8939a00,0,0xe4e9,1,0xe898ad00,0,0x9796
,1,0xe487ae00,9577,0xf76a,1,0xe7ad9500,0,0xf76b,1,0xe9b4a800,0,0x8a9b
,1,0xe586b300,11069,0x9972,1,0xe5b89100,6475,0x9be4,1,0xe5bda400,0,0xeabb
,1,0xe7ada400,0,0xf76d,1,0xe7ada600,8215,0xf76e,1,0xe7b99600,2687,0xe384
,1,0xe38ba100,118,0x86ea,1,0xe8a3b300,0,0x8fd6,1,0xe5828000,3518,0x98f8
,1,0xe7ada900,11082,0xf76f,1,0xeaad8200,0,0xf771,1,0xe5be8400,0,0x8c61
,1,0xe4888700,0,0xf772,1,0xe6a8a100,0,0x96cd,1,0xe6adb400,7395,0x97f0
,1,0xe89f8000,0,0xe5a9,1,0xe8a49300,0,0xe5ee,1,0xe8a9a600,0,0x8e8e
,1,0xe98eb000,0,0xe84c,1,0xe58d8600,0,0x99be,1,0xe7ae9000,0,0xf773
,1,0xefbdaa00,5378,0xaa,1,0xe7ae9b00,0,0xf775,1,0xe78e8b00,0,0x89a4
,1,0xe7939e00,0,0xf64a,1,0xe285b200,1758,0x86b5,1,0xe4888e00,0,0xf776
,1,0xe98f9000,0,0xe858,1,0xe7aeaf00,9156,0xf777,1,0xe999b600,0,0x93a9
,1,0xefbe8a00,911,0xca,1,0xe7aeb500,0,0xf778,1,0xe682a900,0,0x9459
,1,0xe687bc00,0,0x9cf4,1,0xe2869200,0,0x81a8,1,0xe79ea400,0,0xf6af
,1,0xe7a3b700,0,0xed4d,1,0xe7aebc00,0,0xf779,1,0xe7af8500,0,0xf77a
,1,0xe7af8a00,0,0xf77b,1,0xe4be8800,10487,0x98ca,1,0xe5a39200,0,0x8881
,1,0xe5a8a500,0,0x9b4d,1,0xe5adb800,0,0x9b7b,1,0xe28c8500,0,0x81c6
,1,0xeaad8300,0,0xf77c,1,0xe7a9aa00,0,0xf749,1,0xeaad8400,0,0xf77d
,1,0xe7af9400,0,0xf77e,1,0xe7af9600,10520,0xf780,1,0xe7af9a00,10598,0xf781
,1,0xe7afaa00,0,0xf782,1,0xc2b00000,97,0x818b,1,0xe693a200,0,0x9346
,1,0xe698b500,6094,0x9de1,1,0xe2978b00,7958,0x819b,1,0xe7af9d00,8767,0xe2be
,1,0xe381a800,227,0x82c6,1,0xe7afb000,9022,0xf783,1,0xe9ab8f00,0,0xe990
,1,0xcca50000,10820,0x8687,1,0xe582ad00,0,0x9762,1,0xe5b48b00,0,0x9bba
,1,0xe5b99e00,0,0xeaaa,1,0xe7b08b00,0,0xf785,1,0xe6a3bb00,0,0xf3f4
,1,0xe3828800,165,0x82e6,1,0xe7b08e00,0,0xf786,1,0xe7b08f00,9180,0xf787
,1,0xe7b0a600,9765,0xf788,1,0xe5838d00,4172,0x93ad,1,0xe9bba800,10233,0xea7d
,1,0xe58db300,0,0x91a6,1,0xe69f8800,0,0xf3cd,1,0xe6a49b00,0,0x8a91
,1,0xe7b18500,0,0xf789,1,0xe7b19100,0,0xf78b,1,0xe8a08d00,0,0xe5b6
,1,0xe8a5a000,9782,0xe5fb,1,0xe397b400,0,0xf1c0,1,0xe9bc8800,4988,0xea88
,1,0xe58e9300,0,0x87f0,1,0xe593a600,0,0x9a47,1,0xeaabb800,0,0xf3bb
,1,0xe6aa8e00,0,0x8ce7,1,0xc4910000,0,0x85ce,1,0xe794ab00,6981,0x95e1
,1,0xe799be00,8397,0x9553,1,0xe7b19700,8774,0xf78c,1,0xe7b19e00,0,0xf78d
,1,0xe7b1a100,0,0xf78e,1,0xe5948600,0,0x8db4,1,0xe4b9a200,6944,0x9ba4
,1,0xe4beb500,5704,0x904e,1,0xe683b600,0,0x9cb3,1,0xe7958b00,0,0x9dc0
,1,0xe79a9e00,7053,0xece1,1,0xe6bfba00,0,0xe064,1,0xe8b18600,0,0x93a4
,1,0xe8b69900,0,0xe6e2,1,0xe99ba300,0,0x93ef,1,0xe4ba8200,0,0x98aa
,1,0xe59f8c00,0,0xf1f1,1,0xe7b1a900,0,0xf78f,1,0xe7b1ae00,8827,0xf790
,1,0xe6bb8700,0,0xec40,1,0xe7b1af00,8461,0xf791,1,0xe7b1b000,0,0xf792
,1,0xe88aae00,0,0xede1,1,0xe8bc8c00,0,0xe771,1,0xe9a19600,0,0xef9e
,1,0xeaad8600,0,0xf793,1,0xeaad8700,0,0xf794,1,0xe5aa9200,1395,0x947d
,1,0xe5afa500,6350,0x9b8c,1,0xe694af00,0,0x8e78,1,0xe7a68400,0,0x985c
,1,0xe2989800,0,0x849e,1,0xe7b0aa00,0,0xe2cf,1,0xe382b500,2290,0x8354
,1,0xeaad8900,8569,0xf795,1,0xe3b9a600,10670,0xf5bf,1,0xe7b2a600,0,0xf796
,1,0xe5b08500,0,0x9b91,1,0xc2b70000,631,0x854d,1,0xe5baab00,457,0x8cc9
,1,0xe69fb500,0,0x8df2,1,0xe7b18a00,0,0xf78a,1,0xe3839500,2653,0x8374
,1,0xeaad8800,0,0xf797,1,0xe38dbb00,283,0x877e,1,0xe7b2b600,0,0xf798
,1,0xccac0000,955,0x8688,1,0xe7b2b700,0,0xf799,1,0xe5bb8b00,0,0xeaad
,1,0xe7b2bf00,9633,0xf79a,1,0xe6a5a800,0,0xeb98,1,0xe6aabb00,1908,0x9f42
,1,0xeaad8b00,9085,0xf79b,1,0xe9819100,0,0xe7a1,1,0xe986a400,0,0x8fdd
,1,0xe7b38400,9985,0xf79c,1,0xe58a8d00,11149,0x9998,1,0xeaad8a00,0,0xf79d
,1,0xe594b300,0,0x9a59,1,0xe7b38800,0,0xf79e,1,0xe6ab9b00,1910,0x8bf9
,1,0xe790a500,0,0xe0e6,1,0xe795b800,0,0xe162,1,0xe7b38d00,9820,0xf79f
,1,0xe8aca000,0,0xe68f,1,0xe991aa00,9434,0xe86d,1,0xe996bd00,0,0xef73
,1,0xe5959300,0,0x8c5b,1,0xeaad8c00,8660,0xf7a0,1,0xe59fb900,0,0x947c
,1,0xe6b18e00,0,0x94c4,1,0xc4980000,0,0x85bc,1,0xe288ac00,488,0x81e8
,1,0xe880b500,0,0xedc5,1,0xe7b39700,8843,0xf7a1,1,0xeaad8d00,0,0xf7a2
,1,0xe8bcb900,3290,0xe775,1,0xe4bb8f00,9412,0x95a7,1,0xe5a09900,1293,0x9ac3
,1,0xe685a300,1140,0x8ab5,1,0xe5aabf00,0,0xf289,1,0xe6bc9400,100,0x8989
,1,0xe28e9f00,0,0x8463,1,0xe886a800,4374,0x9663,1,0xe88bbb00,9527,0xe498
,1,0xe7b3ab00,0,0xf7a4,1,0xeaad8e00,0,0xf7a5,1,0xe7b3b500,0,0xf7a6
,1,0xe7b48300,0,0xf7a7,1,0xe68b9600,0,0xeae8,1,0xe690a900,0,0xf366
,1,0xe7b48900,8590,0xf7a8,1,0xe7a79100,2462,0x89c8,1,0xe299a500,8736,0x83bc
,1,0xe891ae00,0,0xe4d9,1,0xe9a38300,0,0xe948,1,0xe48b8600,10581,0xf7a9
,1,0xe7b49200,0,0xf7aa,1,0xe7b49e00,9512,0xf7ab,1,0xeaad8f00,0,0xf7ac
,1,0xe6969c00,0,0x8ece,1,0xe5bbb800,0,0x9c4e,1,0xe780b900,0,0xf56d
,1,0xe29f9800,387,0x84c7,1,0xeaad9000,8968,0xf7ad,1,0xe89cb400,0,0xe58e
,1,0xe7b4bd00,8883,0xf7ae,1,0xe5809400,0,0x98e1,1,0xe7b4be00,0,0xf7af
,1,0xe58aba00,0,0xf164,1,0xc2be0000,185,0x8554,1,0x70000000,11496,0x70
,1,0xe786ac00,0,0xe092,1,0xe7b88a00,0,0xe36f,1,0xe7b58000,10015,0xf7b0
,1,0xe7b58700,0,0xf7b1,1,0xe987b100,0,0xef42,1,0xeaad9100,0,0xf7b2
,1,0xe9bea200,0,0xeff7,1,0xe590ad00,0,0x99e0,1,0xe6a28200,0,0xf3e8
,1,0xeaad9200,0,0xf7b3,1,0xe78c9f00,0,0x97c2,1,0xe48b9600,10069,0xf7b4
,1,0xe7b59900,0,0xf7b5,1,0xe9889100,0,0xe7e5,1,0xe98da400,0,0xef5b
,1,0xe7b59a00,8902,0xf7b6,1,0xe5918d00,0,0x87f5,1,0xefbc9e00,0,0x8184
,1,0xe59bb300,10267,0x907d,1,0xe7b5aa00,8921,0xf7b7,1,0xe6b29b00,7485,0x9f8f
,1,0xe6b7ae00,0,0x9fcc,1,0xe79cb800,0,0xe1c6,1,0xe7b5b000,0,0xf7b8
,1,0xe8b3a000,0,0x9485,1,0xe998aa00,0,0x8de3,1,0xe48b9d00,0,0xf7b9
,1,0xe4bc9c00,1086,0x98e5,1,0xe7b5bf00,0,0xf7ba,1,0xe5a6b900,0,0x9685
,1,0xeaad9300,0,0xf7bb,1,0xe6bda100,7710,0xec4b,1,0xe882a200,0,0x8e88
,1,0xe294bf00,0,0x84b9,1,0xe7b68600,0,0xf7bc,1,0xe8bea600,0,0xeed0
,1,0xe7b68800,10423,0xf7bd,1,0xe7b68c00,0,0xf7be,1,0xe6879000,1146,0x89f9
,1,0xce940000,479,0x83a2,1,0xe691b600,1801,0x9d96,1,0xe7a38b00,0,0xe1f6
,1,0xe7a89e00,0,0xf6fa,1,0xe7adb100,0,0xe2aa,1,0xe892bb00,9635,0xe4e8
,1,0xe9a49000,0,0x8e60,1,0xe7b69700,0,0xf7bf,1,0xeaad9400,0,0xf7c0
,1,0x59000000,3773,0x59,1,0xe5b29f00,0,0x9886,1,0xe697a900,1238,0x9181
,1,0xe7b69d00,0,0xf7c1,1,0xe7ae9100,0,0xf774,1,0xe3809c00,0,0x8160
,1,0xe7b8b700,0,0xe37e,1,0xe7b6a700,0,0xf7c2,1,0xe9af9600,0,0x8e49
,1,0xe7b6aa00,0,0xf7c3,1,0xe586b400,5167,0x8de1,1,0xe6988900,6077,0xeb4c
,1,0xe69d9c00,0,0x936d,1,0xe6a2af00,0,0x92f2,1,0xe7b48400,1885,0x96f1
,1,0xe8998e00,0,0x8cd5,1,0xe38ba200,0,0x86eb,1,0xe8a3b400,9749,0xe5e8
,1,0xe7b6b600,0,0xf7c4,1,0xe7b6b700,0,0xf7c5,1,0xe7b79700,0,0xf7c7
,1,0xe5be8500,0,0x91d2,1,0xe6a38f00,0,0xf3fb,1,0xe6a8a200,0,0x9ef6
,1,0xe6adb500,7397,0xf49c,1,0xe7bf8a00,4231,0xe3c2,1,0xe7b79900,0,0xf7c8
,1,0xe9899e00,5555,0xe7e6,1,0xe7b7a600,0,0xf7c9,1,0xe58d8700,0,0x8fa1
,1,0xccba0000,0,0x869b,1,0xefbdab00,5380,0xab,1,0xe7b7b900,0,0xf7cb
,1,0xe6ae9500,0,0x9f6c,1,0xe280a000,449,0x81f5,1,0xe285b300,0,0x86b6
,1,0xe8aa8700,0,0x8cd6,1,0xe98f9100,0,0x934c,1,0xe48c8200,9648,0xf7cc
,1,0xe999b700,0,0xe89c,1,0xefbe8b00,348,0xcb,1,0xe4bda900,0,0x98ce
,1,0xe682aa00,652,0x88ab,1,0xe687bd00,0,0x9cf3,1,0xe2869300,431,0x81ab
,1,0xe79ea500,0,0x95cb,1,0xe883af00,0,0xe3f6,1,0xeaad9500,0,0xf7cd
,1,0xe99a9700,0,0xe8a7,1,0x42000000,8376,0x42,1,0xe9a4bd00,0,0xe958
,1,0xe5a39300,0,0x9ad8,1,0xe6889d00,0,0xe6c1,1,0xefa99900,0,0xf7ce
,1,0xe28c8600,0,0x81c7,1,0xe7b89000,0,0xf7cf,1,0xe7a9ab00,2490,0x8a6e
,1,0xe88eb500,4842,0xe4b7,1,0xe9a08a00,0,0xef97,1,0xe7b89700,0,0xf7d0
,1,0xe9aab000,0,0xe98d,1,0xe5a98600,0,0x946b,1,0xe5ae9900,0,0x9288
,1,0xce9b0000,1474,0x83a9,1,0xe698b600,0,0x9de2,1,0xe7b89d00,0,0xf7d1
,1,0xe7b8a000,0,0xf7d2,1,0xe381a900,4446,0x82c7,1,0xe899bb00,0,0x88b8
,1,0xe7b8a700,8976,0xf7d3,1,0xe9b0a300,9446,0xefce,1,0xe7b8ac00,10980,0xf7d4
,1,0xe7b98500,0,0xf7d5,1,0xe5b99f00,0,0x9bee,1,0xe69ea900,0,0x9e5f
,1,0xe6a3bc00,0,0xeb8f,1,0xe3828900,2625,0x82e7,1,0xe7b9b300,0,0xf7d6
,1,0xe7b9b500,0,0xf7d7,1,0xe7b9be00,0,0xf7d8,1,0xefa98c00,5153,0xed46
,1,0xe7ba8600,0,0xf7d9,1,0xe58db400,0,0x8b70,1,0xe5bf9200,0,0xf2bc
,1,0xe6a49c00,0,0x8c9f,1,0xe7ba8700,0,0xf7da,1,0xe78eb900,0,0xf5e1
,1,0xe8a08e00,3017,0xe5ba,1,0xe9859800,0,0xe7c4,1,0xe48cab00,9936,0xf7db
,1,0xe7ba9100,0,0xf7dc,1,0xe7ba9800,0,0xf7dd,1,0xe7ba9a00,9029,0xf7de
,1,0xeaabb900,0,0xf3c2,1,0xe48d8300,0,0xf7df,1,0xc9a40000,4152,0x8655
,1,0xe6b4b500,1969,0x9fab,1,0xe8a68100,0,0x9776,1,0x2b000000,4955,0x2b
,1,0xe7bcbc00,0,0xf7e0,1,0xe7bcbb00,0,0xf7e1,1,0xe5948700,0,0x904f
,1,0xeaac9900,0,0xf4de,1,0xe4beb600,0,0x97b5,1,0xe683b700,6846,0x9cb4
,1,0xe7958c00,0,0x8a45,1,0xe7bd8300,0,0xf7e3,1,0xe7bd8400,0,0xf7e4
,1,0xe7bd8f00,9417,0xf7e5,1,0xe9969100,0,0x8ad5,1,0xe3938100,9283,0xf7e6
,1,0xe9a0b700,0,0xe8f5,1,0xeaad9600,0,0xf7e7,1,0xe7bd9200,0,0xf7e8
,1,0xeaad9700,0,0xf7e9,1,0xe2888000,6341,0x81cd,1,0xe7a09200,0,0xe1e5
,1,0xe7a5a500,0,0x8fcb,1,0xe88aaf00,0,0x9063,1,0xe99c8400,0,0xe8ba
,1,0xe9a19700,0,0xef9f,1,0xe7bda100,9111,0xf7ea,1,0xe7bdad00,9112,0xf7ed
,1,0xe68a8a00,0,0x9463,1,0xe5afa600,0,0x9b89,1,0xe5b4b900,0,0xf0d2
,1,0xe7a68500,0,0x9154,1,0xe7bdbd00,0,0xf7ee,1,0xe7b0ab00,2569,0xe2d2
,1,0xe382b600,0,0x8355,1,0xe7bdbe00,0,0xf7ef,1,0xeaad9800,0,0xf7f0
,1,0xe7be9000,10899,0xf7f1,1,0xe5b08600,665,0x8fab,1,0xe585bb00,0,0xf7f2
,1,0xe5baac00,0,0xf2a3,1,0xe5bfbf00,0,0x9c7c,1,0xeaad9900,9109,0xf7f3
,1,0xe3839600,0,0x8375,1,0xe7be9600,0,0xf7f4,1,0xe38dbc00,335,0x878f
,1,0xe7be9c00,9132,0xf7f5,1,0xe5849b00,11038,0xf143,1,0xe7bead00,0,0xf7f6
,1,0xeaad9a00,0,0xf7f7,1,0xe6a09600,0,0x90b2,1,0xe785a000,6688,0xec77
,1,0x14000000,0,0x14,1,0xe89c8800,0,0xe585,1,0xe38e9c00,49,0x876f
,1,0xe7bf8300,0,0xf7f8,1,0xe98bb800,0,0x8b98,1,0xe7bf8f00,0,0xf7f9
,1,0xe58fa100,0,0x8962,1,0xe7bfa300,9011,0xf7fa,1,0xe7868000,0,0xf599
,1,0xe6ab9c00,0,0xf487,1,0xe790a600,0,0xeca4,1,0xe795b900,0,0xecc9
,1,0xe7bfa500,9222,0xf7fb,1,0xe98c9800,803,0x908e,1,0xe991ab00,0,0xef6a
,1,0xe5908100,3218,0x99dc,1,0xe7bfaf00,9167,0xf7fc,1,0xe4bab000,8665,0x98b5
,1,0xe59fba00,0,0x8aee,1,0xe7918600,0,0xecad,1,0xe7bfb200,0,0xf840
,1,0xe79bac00,0,0xf699,1,0xe7a0bf00,754,0x8d7b,1,0xe8b29400,0,0xe6be
,1,0xe9979e00,0,0xef79,1,0xe8808200,0,0xf841,1,0xe4bb9000,9439,0xf059
,1,0xe8808800,0,0xf843,1,0xe5a5ad00,0,0x888a,1,0xe6b78200,0,0xf4e2
,1,0xe79c8c00,394,0x8ca7,1,0xe28ea000,355,0x8464,1,0xe886a900,0,0xe456
,1,0xe8809600,0,0xf846,1,0xe880a400,10356,0xf847,1,0xe880ac00,0,0xf848
,1,0xe880b000,0,0xf849,1,0xe5a68d00,0,0x9b4a,1,0xe68b9700,5875,0x9d58
,1,0xe8818300,0,0xf84a,1,0xe881a600,0,0xf84b,1,0xe2949300,625,0x84ad
,1,0xe299a600,25,0x83ba,1,0xe891af00,0,0xe4de,1,0xe9a38400,0,0xe947
,1,0xe881b100,0,0xf84c,1,0xe881b500,0,0xf84d,1,0xe5ac8000,0,0xf291
,1,0xe5b19300,0,0x9b9e,1,0xe6969d00,0,0xeb49,1,0xe69bb000,3694,0x9e48
,1,0xe7ad8500,7152,0xe2a4,1,0xe29f9900,11494,0x84c8,1,0xcb8c0000,0,0x8673
,1,0xe7bcbe00,0,0xf7e2,1,0xe981bf00,5111,0x94f0,1,0xe881bb00,0,0xf84e
,1,0xe585a800,0,0x9153,1,0xe58abb00,0,0xf165,1,0xe8829900,0,0xf84f
,1,0xcea90000,1599,0x83b6,1,0xe8829c00,0,0xf850,1,0xe7b88b00,2630,0xe376
,1,0xe89d9500,0,0x9049,1,0xe882a400,0,0xf851,1,0xe882a700,0,0xf852
,1,0xe882b800,0,0xf853,1,0xe58b9b00,0,0x87e5,1,0xe590ae00,0,0x99e2
,1,0xe6a28300,0,0x9e8c,1,0xe6a79600,0,0xf45d,1,0xeaad9b00,0,0xf854
,1,0xe791b300,0,0x8dba,1,0xe8838500,0,0xf855,1,0xe8a89b00,0,0xe661
,1,0xe98da500,0,0xef5c,1,0xe8839500,0,0xf856,1,0xe5918e00,0,0x99e6
,1,0xefbc9f00,0,0x8148,1,0xe4bbbd00,10329,0x87a7,1,0xe6ad8900,0,0x9f5d
,1,0xe6b29c00,0,0xebd7,1,0xe284a700,0,0x8280,1,0xe8839800,8525,0xf857
,1,0xe8ae8e00,5833,0xe6a6,1,0xe8b3a100,0,0xeeb7,1,0xe8b8b400,0,0xe6fb
,1,0xe883a600,0,0xf858,1,0xe4bc9d00,0,0x9360,1,0xe5a1a700,0,0xf247
,1,0xeaad9c00,0,0xf859,1,0xe8848d00,0,0xf85a,1,0xc9b20000,4045,0x85f4
,1,0xe883b500,0,0xf85b,1,0xe884b500,0,0xf85e,1,0xe9998b00,9561,0xe89b
,1,0xe8bea700,10272,0x999f,1,0xe8849600,10426,0xf85f,1,0xe8849e00,0,0xf860
,1,0xe5a79a00,0,0x9b4c,1,0xe5acad00,0,0xf295,1,0xe6be8200,0,0xe04c
,1,0xe8838300,7824,0x88dd,1,0xe7a89f00,2478,0xe267,1,0xe7adb200,0,0xed82
,1,0xe892bc00,0,0x9193,1,0xe48fb000,0,0xf861,1,0xe9a9a400,0,0xe988
,1,0xe884a400,0,0xf862,1,0xe68d8400,0,0xf34f,1,0xe5b2a000,0,0xf0bf
,1,0xe5b7b300,0,0x96a4,1,0xe69cbd00,0,0x8b80,1,0xe7ae9200,2555,0xe2b4
,1,0xe3809d00,8837,0x8780,1,0xe898af00,0,0xe1ba,1,0xe884a700,0,0xf863
,1,0xe884ac00,0,0xf864,1,0xe9b4aa00,0,0xe9f0,1,0xe586b500,3572,0x9976
,1,0xe6988a00,0,0x9ddd,1,0xe5bda600,6579,0x9546,1,0xe6a2b000,0,0x8a42
,1,0xe7b48500,0,0x8d67,1,0xeaad9d00,0,0xf865,1,0xe38ba300,2745,0x86ec
,1,0xe8a3b500,0,0xee8d,1,0xe988bf00,10952,0xe7ed,1,0xe9ba9d00,0,0xea6c
,1,0xe884bd00,0,0xf866,1,0xe591bb00,0,0x99ef,1,0xe6a39000,0,0xeb8a
,1,0xe6a8a300,0,0x9ee9,1,0xe885a900,0,0xf868,1,0xe38c8300,0,0x8765
,1,0xe4909700,10559,0xf869,1,0xe8868100,0,0xf86a,1,0xe4909c00,0,0xf86b
,1,0xe58d8800,0,0x8cdf,1,0xe8868400,0,0xf86c,1,0xefbdac00,5389,0xac
,1,0xe8868500,0,0xf86d,1,0xe6ae9600,0,0x9042,1,0xe280a100,1948,0x81f6
,1,0xe285b400,7011,0x86b7,1,0xe490a200,8838,0xf86e,1,0xe8869800,0,0xf86f
,1,0xe886b200,0,0xf870,1,0xe999b800,10938,0x97a4,1,0xefbe8c00,17,0xcc
,1,0xe4bdaa00,1100,0x87b0,1,0xe8878100,0,0xf871,1,0xe687be00,0,0x9cf5
,1,0xe2869400,4023,0x81ef,1,0xe6beaf00,0,0xf55e,1,0xe7a3b900,0,0xf6d5
,1,0xe8878300,0,0xf872,1,0xe8baa100,3261,0xe759,1,0xe8bfb400,0,0xe78f
,1,0xe4be8a00,3369,0x87b3,1,0xe5a39400,3747,0x8883,1,0xe6889e00,5844,0x9d40
,1,0xe5adba00,0,0x9b7d,1,0xe8879600,0,0xf873,1,0xc9b90000,2119,0x85ea
,1,0xd0910000,8664,0x8441,1,0xe8879b00,0,0xf874,1,0xeaad9e00,0,0xf875
,1,0xe887a400,0,0xf876,1,0xeaad9f00,0,0xf877,1,0xe887ac00,0,0xf878
,1,0xe5ae9a00,0,0x92e8,1,0xe5b3ad00,6404,0x9bb7,1,0xeaada000,0,0xf879
,1,0xe887bd00,8685,0xf87a,1,0xe88f9600,0,0x8fd2,1,0xe381aa00,2260,0x82c8
,1,0xe887bf00,0,0xf87b,1,0xe9ab9100,0,0xe991,1,0xe9b0a400,0,0xe9dc
,1,0xeaada100,10087,0xf87c,1,0xe5b48d00,6411,0x9890,1,0xe6999700,0,0xf39e
,1,0xe5beb300,0,0x93bf,1,0xe8888400,0,0xf87d,1,0xe3828a00,316,0x82e8
,1,0xeaada200,9032,0xf87e,1,0xe89faf00,0,0xe5b2,1,0xe8889900,9767,0xf880
,1,0xe5838f00,4336,0x919c,1,0xe888a200,0,0xf882,1,0xe58db500,0,0x9791
,1,0xe69f8a00,6172,0x9541,1,0xe7849400,0,0x898b,1,0xe789a700,0,0x9671
,1,0xeaada300,0,0xf883,1,0xe9808600,2774,0x8b74,1,0xe888b200,9278,0xf884
,1,0xe888b400,0,0xf885,1,0xe888bc00,0,0xf886,1,0xe8898600,0,0xf887
,1,0xe593a800,0,0x8fa3,1,0xe598bb00,6587,0x8851,1,0xe6aa9000,1906,0x9ef7
,1,0xceb70000,0,0x83c5,1,0xe6b4b600,0,0x9fa6,1,0xe8898900,0,0xf888
,1,0xe98b8c00,0,0xef50,1,0xe8898500,0,0xf889,1,0xeaada400,0,0xf88a
,1,0xe8898b00,0,0xf88b,1,0xe5999b00,7086,0x8a9a,1,0xe491b600,0,0xf88c
,1,0xe683b800,0,0xead5,1,0xe7958d00,0,0xe157,1,0xe79aa000,0,0xf691
,1,0xe79fb300,7139,0x90ce,1,0xe8b18800,8563,0xe6af,1,0xe8898f00,0,0xf88d
,1,0xe491ba00,0,0xf88e,1,0xe9a0b800,0,0xe8f2,1,0xe59f8e00,404,0x8fe9
,1,0xc3860000,0,0x855c,1,0xe8899700,0,0xf88f,1,0xe68ebe00,1765,0x9d81
,1,0xe8808a00,0,0xf842,1,0xeaada500,0,0xf890,1,0xe7aab900,0,0xf753
,1,0xe8899c00,0,0xf891,1,0xe9a19800,0,0x8ae8,1,0xe889a300,0,0xf892
,1,0xeaada600,0,0xf893,1,0xeaada700,0,0xf894,1,0xe5afa700,0,0x944a
,1,0xefa99e00,0,0xf895,1,0xe889b900,9491,0xf896,1,0xe7ab9900,0,0xe28b
,1,0xd0980000,0,0x8449,1,0xe382b700,8267,0x8356,1,0xe4929100,0,0xf897
,1,0xe3b9a800,0,0xf5c0,1,0xe889bd00,9326,0xf898,1,0xe5b08700,0,0x9b92
,1,0xe6959100,0,0x8b7e,1,0xe5baad00,0,0x92eb,1,0xe69fb700,0,0xf3d2
,1,0xe7b18c00,0,0xe2d4,1,0xe3839700,2315,0x8376,1,0xe89ba900,0,0xe57d
,1,0xe38dbd00,9205,0x878e,1,0xe889bf00,0,0xf899,1,0xe88a8300,0,0xf89a
,1,0xe589af00,11136,0x959b,1,0xe69b8400,0,0x9e40,1,0xe6a09700,0,0x8c49
,1,0xe6a5aa00,0,0x9ec2,1,0xe6aabd00,0,0xf483,1,0xe89c8900,0,0xe58a
,1,0xe38e9d00,533,0x8770,1,0xe8a6af00,0,0xe651,1,0xe88a8a00,9988,0xf89b
,1,0xe9bd9700,0,0xeff3,1,0xe58fa200,0,0x9170,1,0xe594b500,0,0x8845
,1,0xe6a68a00,0,0x8de5,1,0xe6ab9d00,0,0xf488,1,0xe88a9300,0,0xf89c
,1,0xe795ba00,0,0xf65f,1,0xe9878600,0,0x94ce,1,0xe98c9900,0,0xe840
,1,0xe88aa700,0,0xf89d,1,0xe88aa800,0,0xf89e,1,0xe88ab200,0,0xf89f
,1,0xe59aa800,8666,0xf1ce,1,0xe59fbb00,0,0xf1f9,1,0xe6b19000,4190,0x8eac
,1,0xcebe0000,1976,0x83cc,1,0x7c000000,9,0x7c,1,0xe880b700,0,0xedc6
,1,0xe88abc00,0,0xf8a2,1,0xe8b7a800,0,0x8cd7,1,0xe99cb200,3005,0x9849
,1,0xe59b8800,9473,0x9a94,1,0xe6809200,0,0x937b,1,0xe5a5ae00,1358,0x95b1
,1,0xe88ba200,0,0xf8a3,1,0xe88ba800,0,0xf8a4,1,0xe28ea100,0,0x8465
,1,0xe7a6b300,0,0xe259,1,0xe8b88800,4562,0xe6f1,1,0xe99d9200,0,0x90c2
,1,0xe88bb700,0,0xf8a5,1,0xe9a7b800,0,0xe973,1,0xe88c8700,0,0xf8a6
,1,0xc38d0000,186,0x8563,1,0xe88c8800,0,0xf8a7,1,0xe88c8c00,0,0xf8a8
,1,0xe2949400,4526,0x84a4,1,0xe299a700,636,0x83bd,1,0xe7b1b900,4600,0xed91
,1,0xe3b08f00,0,0xebb8,1,0xe88d9400,0,0xf8a9,1,0xe88c9b00,10711,0xf8aa
,1,0xe88c9d00,0,0xf8ab,1,0xe88cb000,0,0xf8ac,1,0xe5b6a700,0,0x989b
,1,0xe5bbba00,0,0x8c9a,1,0xe7ad8600,0,0x954d,1,0xe29f9a00,282,0x84c9
,1,0xd09f0000,738,0x8450,1,0xe88cbc00,0,0xf8ad,1,0xe88d8400,0,0xf8ae
,1,0xe5809600,0,0x8cf4,1,0xe585a900,0,0x995f,1,0xe58abc00,0,0x99a2
,1,0xe88d9700,0,0xf8af,1,0xe492be00,0,0xf8b0,1,0xe6a6b700,7280,0xf458
,1,0xe88dbf00,11118,0xf8b1,1,0xe7bd9f00,0,0xe3a8,1,0xe4939400,0,0xf8b2
,1,0xe492b300,0,0xf8b3,1,0xe5868900,0,0x9966,1,0xe88e9400,0,0xf8b5
,1,0xe88e9500,9386,0xf8b6,1,0xe88e9b00,0,0xf8b7,1,0x65000000,1881,0x65
,1,0xe88e9d00,0,0xf8b8,1,0xe6b1bd00,0,0x8b44,1,0xe88f8900,0,0xf8b9
,1,0xe3959d00,0,0xf17c,1,0xe8adaf00,0,0xe6a1,1,0xe88f9000,0,0xf8ba
,1,0xe88f9400,0,0xf8bb,1,0xefbca000,0,0x8197,1,0xe88f9d00,10372,0xf8bc
,1,0xe6ad8a00,0,0xf499,1,0xe88fa500,0,0xf8bd,1,0xe797a700,0,0xf672
,1,0xe79cba00,0,0x92ad,1,0xe88fb900,0,0xf8be,1,0xcba80000,172,0x8683
,1,0xe8b8b500,10143,0xe6f9,1,0xefbd8000,0,0x814d,1,0xe59c9500,0,0xf1d8
,1,0xe6819f00,0,0x9c8f,1,0xe686b200,220,0x8c9b,1,0xe7988700,0,0xf679
,1,0xe79d9a00,0,0xe1c8,1,0xe7a2ad00,8498,0xed49,1,0xe8908f00,0,0xf8bf
,1,0xe8b99500,3247,0xe74a,1,0xe8bea800,0,0x999e,1,0xe9a3b200,0,0x88f9
,1,0xe8909100,0,0xf8c0,1,0xe8909500,0,0xf8c1,1,0xeaada900,0,0xf8c2
,1,0xe691b800,0,0x96cc,1,0xe8838400,0,0xe3f2,1,0xe8889700,2140,0x95dc
,1,0xe7adb300,0,0xf770,1,0xe8909700,0,0xf8c3,1,0xe9a49200,4537,0xe94e
,1,0xe9a9a500,0,0xe987,1,0xe890b900,10745,0xf8c4,1,0xe8918f00,0,0xf8c6
,1,0xc3940000,57,0x856a,1,0xe5b7b400,0,0x9462,1,0xe7a98000,0,0x8d92
,1,0xe88e8a00,0,0xe4b5,1,0xe7b3a600,0,0xf7a3,1,0xe7b8b900,2683,0xe37c
,1,0xe8919100,0,0xf8c7,1,0xe8919200,10762,0xf8c8,1,0xe9b4ab00,5328,0x8eb0
,1,0x4e000000,753,0x4e,1,0xe5b89400,0,0xeaa7,1,0xe69d9e00,3719,0x9e58
,1,0xe6a2b100,0,0x8dab,1,0xe787bb00,2246,0xe08e,1,0xe8999000,2335,0x8b73
,1,0xd0a60000,0,0x8457,1,0xe983ad00,0,0x8a73,1,0xe8919900,0,0xf8c9
,1,0xe5879600,5168,0x99c3,1,0xe8919a00,0,0xf8ca,1,0xe591bc00,1286,0x8cc4
,1,0xe7838800,0,0x97f3,1,0xe7889b00,0,0xe0a3,1,0xe6adb700,7404,0xebc3
,1,0xe7bf8c00,0,0x9782,1,0xe8919c00,0,0xf8cb,1,0xe8a9a900,9886,0x8e8d
,1,0xe891a500,0,0xf8cd,1,0xe58d8900,0,0x99c1,1,0xe5929c00,0,0x87f6
,1,0xefbdad00,5397,0xad,1,0xe6a98400,0,0x9eeb,1,0xe891bc00,0,0xf8d0
,1,0xe280a200,1719,0x825f,1,0xe285b500,2008,0x86b8,1,0xe98a8000,792,0x8be2
,1,0xe8928100,0,0xf8d1,1,0xe8939c00,0,0xf8d3,1,0xe8929700,10409,0xf8d4
,1,0xefbe8d00,123,0xcd,1,0xe892a600,0,0xf8d5,1,0xe4948800,0,0xf8d7
,1,0xe2818200,1381,0x86fc,1,0xe8938e00,9441,0xf8d8,1,0xe8938f00,0,0xf8d9
,1,0xe883b100,7834,0xe3f7,1,0xe8939300,0,0xf8da,1,0xcbaf0000,163,0x8685
,1,0xeaadac00,9667,0xf8db,1,0xe59e8200,850,0x9082,1,0xe5a39500,0,0x8d88
,1,0xe6889f00,0,0x8c81,1,0xe68db200,0,0x8c9e,1,0xe79f8700,0,0xe1dc
,1,0xe893aa00,0,0xf8dd,1,0xe889a400,4386,0xe47e,1,0xe893af00,0,0xf8de
,1,0xe9a08c00,0,0xe8f1,1,0x37000000,10526,0x37,1,0xe893b000,0,0xf8df
,1,0xe893b100,0,0xf8e0,1,0xe5ae9b00,571,0x88b6,1,0xe693a500,0,0xf377
,1,0xe893ba00,0,0xf8e1,1,0xe2978e00,0,0x819d,1,0xe7afa000,0,0x8ec2
,1,0xe381ab00,332,0x82c9,1,0xe893bd00,0,0xf8e2,1,0xe8948c00,10629,0xf8e3
,1,0xe9b0a500,0,0xe9db,1,0xe8949b00,0,0xf8e4,1,0xe5b48e00,1486,0x8de8
,1,0xc39b0000,585,0x8570,1,0xe5beb400,0,0x92a5,1,0xe7b08000,0,0xe2c5
,1,0xe3828b00,262,0x82e9,1,0xe894a400,9040,0xf8e5,1,0xe7bfb900,0,0xe3c9
,1,0xe894a500,9774,0xf8e6,1,0xe5839000,4341,0x87c5,1,0xe894ab00,0,0xf8e7
,1,0xe5ba8100,0,0x92a1,1,0xe5bf9400,0,0xf2be,1,0xe894b400,0,0xf8e8
,1,0xe8958f00,0,0xf8e9,1,0xe78ebb00,0,0xe0de,1,0xe38d9100,4717,0x8767
,1,0xd0ad0000,0,0x845e,1,0xe98aad00,0,0x914b,1,0xe5898300,0,0x92e4
,1,0xe58e9600,0,0x99cb,1,0xe593a900,0,0x9689,1,0xeaabbb00,0,0xf3c7
,1,0xe6aa9100,0,0xf47d,1,0xe895af00,0,0xf8ea,1,0xe494a500,0,0xf8eb
,1,0xe8a68300,0,0xe648,1,0xe8ab9600,0,0x985f,1,0xe4958300,9605,0xf8ec
,1,0xe894be00,0,0xf8ed,1,0xe5948900,0,0xf19e,1,0xeaac9b00,0,0xf566
,1,0xe8959100,0,0xf8ee,1,0xe683b900,0,0x8ee4,1,0xe7958e00,0,0xf659
,1,0xe6baaa00,7667,0x9fe2,1,0xe79fb400,0,0xf6b9,1,0x20000000,2428,0x20
,1,0xe9969300,0,0x8ad4,1,0xe8959300,10749,0xf8ef,1,0xe4ba8500,0,0x98ab
,1,0xe59f8f00,929,0x886f,1,0xe5a4a200,6236,0x96b2,1,0xe8959e00,0,0xf8f0
,1,0xe2888200,7055,0x81dd,1,0xe7a09400,2378,0x8ca4,1,0xe7a5a700,0,0xf6e3
,1,0xe7aaba00,68,0x894d,1,0xe99c8600,0,0xe8bb,1,0xe9a19900,0,0xefa0
,1,0xe9a6ac00,0,0x946e,1,0xe895a100,0,0xf8f1,1,0xe895a200,0,0xf8f2
,1,0xe5afa800,0,0x9ecb,1,0xe694b200,0,0xf381,1,0xeaadad00,0,0xf8f3
,1,0xe88b9100,2171,0x8991,1,0xe895bb00,0,0xf8f4,1,0xe382b800,2662,0x8357
,1,0xe4948d00,0,0xf8d2,1,0xe9ac9f00,0,0xe9a3,1,0xefa4a800,10906,0xeaac
,1,0xe5b08800,0,0x9b93,1,0xe895bd00,0,0xf8f5,1,0xe8968100,0,0xf8f7
,1,0xe8968600,0,0xf8f8,1,0xe7b18d00,0,0x90d0,1,0xe3839800,54,0x8377
,1,0xe8969300,0,0xf8f9,1,0xe38dbe00,0,0x878d,1,0xe8969d00,0,0xf8fa
,1,0xe8969f00,0,0xf8fb,1,0xe589b000,0,0x8fe8,1,0xeaadaf00,0,0xf940
,1,0xc3a20000,1817,0x8577,1,0xe6a5ab00,1869,0x9eb5,1,0xe7b78000,0,0xf7c6
,1,0xe89c8a00,0,0xe58d,1,0xe38e9e00,317,0x8771,1,0xeaadb000,9291,0xf941
,1,0xe98bba00,0,0xe7fa,1,0xe896b700,11150,0xf942,1,0xe58fa300,0,0x8cfb
,1,0xe896bc00,0,0xf943,1,0xe8978700,0,0xf944,1,0xe6ab9e00,0,0x9f48
,1,0x9000000,4175,0x9,1,0xe8978a00,0,0xf945,1,0xe9878700,0,0x8dd1
,1,0xd0b40000,3154,0x8474,1,0xe8979800,10072,0xf946,1,0xe5908300,0,0x8b68
,1,0xe5959600,0,0x9a56,1,0xe59aa900,4543,0x885b,1,0xe59fbc00,0,0x8de9
,1,0xe8979900,0,0xf947,1,0xe8979f00,0,0xf948,1,0xe79bae00,1797,0x96da
,1,0xe897a100,9379,0xf949,1,0xe897a600,0,0xf94a,1,0xe897b600,0,0xf94b
,1,0xe99cb300,0,0xef86,1,0xe59b8900,9477,0x885e,1,0xe8988000,0,0xf94c
,1,0xe8989100,0,0xf94d,1,0xe68ab900,4001,0x9695,1,0xe8989e00,10621,0xf94e
,1,0xe28ea200,11490,0x8466,1,0xe7a6b400,0,0xf6eb,1,0xe8b88900,4563,0xe6f2
,1,0xe8bd9c00,0,0xe77e,1,0xe3afb000,0,0xf47c,1,0xe898a100,9385,0xf94f
,1,0xe898a400,0,0xf950,1,0xe68b9900,0,0x90d9,1,0xe690ac00,0,0x94c0
,1,0xe7a28100,0,0x8ce9,1,0xe7a79400,0,0xed67,1,0xe299a800,8013,0x83ea
,1,0xe891b100,0,0x944b,1,0xe9a38600,0,0xe949,1,0xe9a89900,0,0xe978
,1,0xe898a700,10715,0xf951,1,0xeaadb200,0,0xf952,1,0xe5b19500,0,0x9357
,1,0xe6969f00,0,0x9dce,1,0xe5bbbb00,143,0x89f4,1,0xe7ad8700,7154,0xed7c
,1,0xe29f9b00,0,0x84ca,1,0xe897a400,0,0x93a1,1,0xe89cb700,0,0xe590
,1,0xe898bc00,0,0xf954,1,0xeaadb300,0,0xf955,1,0xe585aa00,0,0x9960
,1,0xe8998000,0,0xf956,1,0xe5bc9b00,6558,0x926f,1,0xefa8a000,0,0xf957
,1,0xe786af00,0,0xf59e,1,0xe8988400,0,0xee65,1,0xe7bda000,2734,0xe3a9
,1,0xe8a2aa00,9708,0xee89,1,0xe899af00,9319,0xf95a,1,0xe5868a00,0,0x8dfb
,1,0xe58b9d00,0,0x8f9f,1,0xe899b700,0,0xf95b,1,0xe6a28500,0,0x947e
,1,0xc3a90000,795,0x857e,1,0xca810000,0,0x85fa,1,0xe6b1be00,0,0x9f90
,1,0xe9838100,4729,0x88e8,1,0xe8a89d00,3112,0xe662,1,0xe8adb000,0,0x8b63
,1,0xe899ba00,0,0xf95c,1,0xe89a8700,0,0xf95d,1,0xefbca100,0,0x8260
,1,0xe4bbbf00,5667,0x87a8,1,0xe89a8900,0,0xf95e,1,0xe89a8d00,0,0xf95f
,1,0xe6b7b100,0,0x905b,1,0xe89a9100,0,0xf960,1,0xe8ae9000,0,0x8f51
,1,0xd0bb0000,2923,0x847c,1,0xe89a9c00,9554,0xf961,1,0xefbd8100,0,0x8281
,1,0xe59c9600,0,0x9aa4,1,0xe5a1a900,1292,0x8996,1,0xe5a6bc00,0,0xf26c
,1,0xe7988800,0,0xecd0,1,0xe6bda400,2075,0x8f81,1,0xe882a500,0,0x94ec
,1,0xe89a9d00,0,0xf962,1,0xe9998d00,0,0x8d7e,1,0xe99ea000,6494,0x8b66
,1,0xe89aa800,10428,0xf963,1,0xe5a28900,0,0x887b,1,0xe5a79c00,0,0x9b49
,1,0xefa8a100,10963,0xf964,1,0xe6be8400,3787,0x909f,1,0xe7a38e00,8507,0xf6d0
,1,0xe8889800,0,0x8ada,1,0xe7adb400,0,0xe2a7,1,0xe892be00,0,0xf8d6
,1,0xe9a49300,0,0x89ec,1,0xe89ab100,0,0xf965,1,0xe9aeb900,0,0xe9c1
,1,0xe89ab300,0,0xf966,1,0xe5b2a200,0,0xf0c0,1,0xe697ac00,1240,0x8f7b
,1,0xe69cbf00,0,0x9e52,1,0xe7ae9400,0,0x9493,1,0xe3809f00,833,0x8781
,1,0xe7b8ba00,0,0xe381,1,0xe89b8100,0,0xf967,1,0xe89b9100,0,0xf969
,1,0xe9b4ac00,0,0x89a7,1,0xe9b9bf00,851,0x8ead,1,0xe6988c00,6078,0x8fb9
,1,0xe69d9f00,0,0x91a9,1,0xe6a2b200,0,0xeb88,1,0xe787bc00,4440,0xe09e
,1,0xe7b99a00,0,0xe387,1,0xe38ba500,8430,0x86ef,1,0xe89b9500,0,0xf96a
,1,0xe89b9700,0,0xf96b,1,0xe9ba9f00,0,0x97d9,1,0xe58caa00,0,0x94d9
,1,0xe591bd00,1284,0x96bd,1,0xe6a39200,0,0x965f,1,0xe89ba300,0,0xf96c
,1,0xe6adb800,7411,0x9f64,1,0xe89f8400,0,0xe5ad,1,0xe89ba600,0,0xf96d
,1,0xe496b800,9891,0xf96e,1,0xe89c8500,0,0xf96f,1,0xe58d8a00,0,0x94bc
,1,0xe89c8700,0,0xf970,1,0xefbdae00,5409,0xae,1,0xe6a98500,0,0xebaa
,1,0xc3b00000,1671,0x8586,1,0xca880000,620,0x85eb,1,0xe285b600,1750,0x86b9
,1,0xe89c9900,0,0xf974,1,0xe89c9f00,0,0xf975,1,0xe89ca100,0,0xf976
,1,0xe999ba00,0,0x8caf,1,0xefbe8e00,659,0xce,1,0xe4bdac00,0,0x87b1
,1,0xe89ca300,0,0xf977,1,0xe6b48b00,0,0x976d,1,0xe2869600,4025,0x8246
,1,0xe6beb100,0,0x9362,1,0xe7a3bb00,0,0xf6d7,1,0xe89cb100,0,0xf978
,1,0xe89cbe00,10165,0xf97a,1,0xe99fad00,0,0xe8ea,1,0xe89d8000,0,0xf97b
,1,0xe89d8300,0,0xf97c,1,0xe5a8a900,0,0x95d8,1,0xe5adbc00,0,0xf29a
,1,0xe89d9100,0,0xf97d,1,0xe7a49b00,0,0xf6db,1,0xe89d9800,0,0xf97e
,1,0xe89da400,0,0xf980,1,0xe89da500,10321,0xf981,1,0xe89db200,10527,0xf982
,1,0xe89dbc00,0,0xf983,1,0xe6898000,1154,0x8f8a,1,0xe5ae9c00,0,0x8b58
,1,0xe693a600,679,0x8e43,1,0xeaadb500,0,0xf984,1,0xe2978f00,0,0x819c
,1,0xc5910000,2561,0x85d1,1,0xe381ac00,8880,0x82ca,1,0xeaadb600,0,0xf985
,1,0xe9ab9300,0,0xe992,1,0xe89e8900,0,0xf987,1,0xe582b100,0,0xf091
,1,0xe89e8b00,0,0xf988,1,0xe5b9a200,3830,0x9bef,1,0xcf860000,3728,0x83d3
,1,0xe7b08100,0,0xed87,1,0xe3828c00,2632,0x82ea,1,0xe89e9300,0,0xf989
,1,0xe7bfba00,0,0xedc1,1,0xe9b18600,0,0xe9e1,1,0xe5839100,4345,0x8ba1
,1,0xe588a400,0,0x94bb,1,0xe58db700,0,0x99c9,1,0xe89ea000,10787,0xf98a
,1,0xeaadb400,0,0xf98b,1,0xe789a900,1643,0x95a8,1,0xe78ebc00,0,0xf5e2
,1,0xe8a09100,4859,0xe5bc,1,0xe8a5a400,4920,0xe641,1,0xe497a500,9348,0xf98c
,1,0xe5898400,0,0x998d,1,0xe89ebe00,0,0xf98d,1,0xeaadb700,0,0xf98e
,1,0xe598bd00,6583,0xf1c1,1,0xe89f8100,0,0xf98f,1,0xe89f8e00,0,0xf990
,1,0xe6b4b800,4246,0x9fa9,1,0xe89fb500,0,0xf991,1,0xe89f9f00,9350,0xf992
,1,0xe990a100,5919,0xe864,1,0xeaadb800,0,0xf993,1,0xe89fa300,0,0xf994
,1,0xeaac9c00,0,0xf56c,1,0xe59eb000,0,0x9ab9,1,0xe683ba00,3946,0x9cb7
,1,0xc3b70000,32,0x8180,1,0xe6baab00,0,0xebfa,1,0xe6bfbe00,0,0xe068
,1,0xe8b18a00,3192,0x964c,1,0xe9969400,0,0xe87b,1,0xe89fa500,0,0xf995
,1,0xe4ba8600,0,0x97b9,1,0xe89fa600,0,0xf996,1,0xe6849a00,0,0x8bf0
,1,0xe689ad00,0,0xf2fb,1,0xe2888300,247,0x81ce,1,0xe7a09500,249,0x8dd3
,1,0xe7a5a800,2440,0x955b,1,0xe7aabb00,8706,0xed75,1,0xe99c8700,0,0x906b
,1,0xe9a19a00,6696,0xefa1,1,0xe9a6ad00,0,0xe966,1,0xe89faa00,0,0xf997
,1,0xe89fab00,0,0xf998,1,0xe5afa900,6353,0x9052,1,0xe89fad00,0,0xf999
,1,0xe8a08100,0,0xf99a,1,0xe88b9200,0,0xe492,1,0xe8a08300,0,0xf99b
,1,0xe382b900,0,0x8358,1,0xe8a08b00,9670,0xf99c,1,0xe9aca000,0,0xefbb
,1,0xefa4a900,0,0xeb6d,1,0xe5b08900,0,0x88d1,1,0xe5b59c00,0,0x9bbd
,1,0xe69aa600,0,0x97ef,1,0xe69fb900,0,0xeb78,1,0xe8a0a800,0,0xf99e
,1,0xe3839900,11491,0x8378,1,0xe8a0ae00,0,0xf99f,1,0xe980b500,0,0xe79c
,1,0xe8a0b200,0,0xf9a0,1,0xe5849e00,3555,0x87cb,1,0xe589b100,5177,0x999b
,1,0xe5bb8f00,3845,0x9bfc,1,0xc8b50000,0,0x8667,1,0xe6a5ac00,0,0xf44e
,1,0xe78ab600,0,0x8ff3,1,0xe89c8b00,0,0xee78,1,0xe9819500,2792,0x88e1
,1,0xe8a0bc00,9807,0xf9a1,1,0xe4988f00,9990,0xf9a2,1,0xe58a9100,0,0x999d
,1,0xe58fa400,0,0x8cc3,1,0xe8a18a00,0,0xf9a3,1,0xe8a19800,0,0xf9a4
,1,0xe6ab9f00,0,0x9f4a,1,0xe8a19f00,0,0xf9a5,1,0xeaadb900,0,0xf9a7
,1,0xe9878800,0,0x8edf,1,0xeaadba00,0,0xf9a8,1,0xe8a1a900,0,0xf9a9
,1,0xe5908400,0,0x8a65,1,0xe5959700,0,0x9a57,1,0xe4bab300,8790,0x98b6
,1,0xe59fbd00,0,0xf1fa,1,0xeaadbb00,0,0xf9aa,1,0xe8a1af00,0,0xf9ab
,1,0xe6bbb800,0,0x9ff5,1,0xe8a2a000,0,0xf9ac,1,0xe8a2bc00,0,0xf9ad
,1,0xe8b7aa00,3237,0xe6ec,1,0xe8a2bd00,0,0xf9ae,1,0xe59b8a00,9497,0x885f
,1,0xe5a09d00,3734,0x9ac4,1,0xe685a700,0,0x8c64,1,0xe6b78500,0,0x9fc6
,1,0xc3be0000,7077,0x8593,1,0x71000000,11478,0x71,1,0xe8a2be00,0,0xf9af
,1,0xe8b88a00,0,0x9778,1,0xe8a38000,0,0xf9b0,1,0xeaadbd00,9964,0xf9b2
,1,0xe8a39300,0,0xf9b4,1,0xe6868700,0,0x9cda,1,0xe5aba300,0,0x9b60
,1,0xe690ad00,0,0x938b,1,0xe8a39b00,0,0xf9b5,1,0xe7a79500,0,0xe25d
,1,0xe299a900,2188,0x81fb,1,0xe8a3b000,0,0xf9b6,1,0xe4908800,10388,0xf867
,1,0xe3b5a400,10582,0xf55f,1,0xe8a3b100,0,0xf9b7,1,0xe4998100,0,0xf9b8
,1,0xe8a48100,0,0xf9b9,1,0xeaadbe00,0,0xf9ba,1,0xe69bb300,558,0x8967
,1,0xe7ad8800,0,0x94a4,1,0xe29f9c00,289,0x84cb,1,0xe897a500,0,0xe55a
,1,0xe8a4b700,0,0xf9bb,1,0xeaadbf00,0,0xf9bc,1,0xe5809800,0,0x87bc
,1,0xe585ab00,0,0x94aa,1,0xe58abe00,1230,0x8a4e,1,0xe69c9300,6146,0xeb6c
,1,0xe7819d00,0,0xec5f,1,0xe8a58200,0,0xf9bd,1,0xe8988500,0,0xee66
,1,0xc59f0000,8408,0x85b1,1,0xe8a2ab00,4879,0x94ed,1,0xe987b500,0,0xe7de
,1,0xe5868b00,0,0xf149,1,0xe58b9e00,0,0x99a7,1,0xe8a58500,0,0xf9be
,1,0xe8a58900,0,0xf9bf,1,0xc8bc0000,710,0x866e,1,0xe6acac00,0,0xebbc
,1,0xe791b600,0,0xe0f4,1,0xeaae8000,0,0xf9c0,1,0xe9889500,0,0xe7e4
,1,0xe8adb100,0,0xe3bf,1,0xe8a5a200,9783,0xf9c2,1,0xe5919100,0,0x93db
,1,0xefbca200,0,0x8261,1,0xe59bb700,10442,0xf1d6,1,0x5a000000,66,0x5a
,1,0xe8a68000,0,0xf9c3,1,0xe797a900,0,0x9189,1,0xe79cbc00,0,0x8ae1
,1,0xe8a69000,10042,0xf9c5,1,0xe8b3a400,0,0xe6cb,1,0xe998ae00,0,0xe896
,1,0xefbd8200,1034,0x8282,1,0xe8a69f00,10596,0xf9c6,1,0xe8a6b000,0,0xf9c7
,1,0xe8a79600,0,0xf9c9,1,0xe7988900,0,0xe188,1,0xe79d9c00,0,0xecf3
,1,0xe7a2af00,0,0xe1f3,1,0xe8b48400,0,0xe6d1,1,0xe8a79800,0,0xf9ca
,1,0xe8a7ab00,0,0xf9cb,1,0xe9a3b400,0,0x88b9,1,0xe6828100,5741,0x9c9a
,1,0xe5a79d00,0,0x8890,1,0xe5acb000,1407,0x8964,1,0xe691ba00,0,0x90a0
,1,0xe8838600,0,0x925f,1,0xca9d0000,8622,0x85f5,1,0xe7adb500,0,0xe2a5
,1,0xe892bf00,5054,0xe4e4,1,0xe9a49400,0,0xe94f,1,0xe49aa100,0,0xf9cc
,1,0xe8a7b100,0,0xf9cd,1,0xe5ad9000,0,0x8e71,1,0xe6929a00,902,0x9451
,1,0xe697ad00,0,0x88ae,1,0xe7a98200,0,0x95e4,1,0xe7ae9500,0,0x96a5
,1,0xe380a000,0,0x83e4,1,0xe7b8bb00,0,0xe37a,1,0xe8a7b300,0,0xf9ce
,1,0xe8a7bd00,0,0xf9cf,1,0xe581a500,0,0x8c92,1,0xe8a7bf00,0,0xf9d0
,1,0xe5b89600,0,0x929f,1,0xe5bda900,1563,0x8dca,1,0xe6a2b300,0,0x9e80
,1,0xe7b48800,0,0xed9a,1,0xe8a89100,0,0xf9d2,1,0xe7beae00,0,0xe3bc
,1,0xe8a3b800,4895,0x9787,1,0xe5828500,0,0x98fa,1,0xe8a89400,0,0xf9d3
,1,0x43000000,0,0x43,1,0xe5be8900,0,0xeabe,1,0xe7838a00,0,0xf586
,1,0xeaae8100,0,0xf9d4,1,0xe6adb900,1926,0x9f65,1,0xe7bf8e00,0,0xedbc
,1,0xe8a49800,0,0xee90,1,0xe989a200,878,0x94ab,1,0xe8a8b500,0,0xf9d6
,1,0xe8a8be00,0,0xf9d7,1,0xe8a98d00,0,0xf9d9,1,0xefbdaf00,5417,0xaf
,1,0xe6a98600,0,0xf472,1,0xe8a99800,10410,0xf9da,1,0xe793a300,6900,0xe141
,1,0xe285b700,0,0x86ba,1,0xe8aaae00,0,0xf9db,1,0xe8aab700,11155,0xf9dd
,1,0xe8aabe00,0,0xf9de,1,0xe8ab9700,9946,0xf9df,1,0xefbe8f00,24,0xcf
,1,0xe59da400,0,0x8da3,1,0xe8abbc00,0,0xf9e0,1,0xe6b48c00,1958,0x9fae
,1,0xe2869700,666,0x8244,1,0xe79ea900,0,0xf6b0,1,0xe883b300,0,0xedcb
,1,0xeaae8200,0,0xf9e1,1,0xe99a9b00,0,0x8ddb,1,0xe99fae00,733,0x9442
,1,0xe4be8d00,0,0x8e98,1,0xe5a39700,0,0x9ada,1,0xe688a100,0,0x9d41
,1,0xe5adbd00,0,0x9876,1,0xe8ac8a00,0,0xf9e2,1,0xe7a49c00,0,0xed4e
,1,0xe889a600,0,0x8acd,1,0xe8ac8500,0,0xf9e3,1,0xe8ac8d00,10323,0xf9e4
,1,0xe8ac9c00,9400,0xf9e5,1,0xe8ac9f00,0,0xf9e6,1,0xe6898100,0,0x9d47
,1,0xe5ae9d00,6339,0x95f3,1,0xe5b3b000,1467,0x95f4,1,0xe698ba00,0,0xeb4f
,1,0xe2979000,0,0x84e5,1,0xe8acad00,0,0xf9e7,1,0xe381ad00,2264,0x82cb
,1,0xe8ad8300,0,0xf9e8,1,0x2c000000,3416,0x2c,1,0xe49c8c00,10821,0xf9e9
,1,0xefa8b000,949,0x87b6,1,0xe5b49000,0,0xf0cb,1,0xe5b9a300,3831,0x95bc
,1,0xe8ad9100,10571,0xf9ea,1,0xe8ad9e00,0,0xf9eb,1,0xe3828d00,0,0x82eb
,1,0xe387a000,0,0x8397,1,0xe7bfbb00,2767,0x967c,1,0xe9b18700,0,0xe9df
,1,0xe9b69a00,4348,0xea4c,1,0xe588a500,0,0x95ca,1,0xe58db800,184,0x89b5
,1,0xe5bf9600,0,0x9c75,1,0xe8adbf00,0,0xf9ed,1,0xe6a9b300,0,0xebaf
,1,0xe8ae8100,0,0xf9ee,1,0xe8ae8b00,0,0xf9ef,1,0xe8ae9500,0,0xf9f1
,1,0xe8ae9c00,0,0xf9f2,1,0xe8ae9e00,0,0xf9f3,1,0xe58e9800,0,0x97d0
,1,0xe8b0b900,0,0xf9f4,1,0xeaabbd00,0,0xf3d9,1,0xeaae8300,0,0xf9f5
,1,0xe6afa600,0,0xf4a8,1,0xe794b000,3980,0x9363,1,0xe8b0bd00,10089,0xf9f6
,1,0xc5ad0000,3137,0x85e2,1,0xe39dac00,0,0xf0a6,1,0xeaae8400,0,0xf9f7
,1,0xeaae8500,0,0xf9f8,1,0xe5999e00,7201,0xf1c5,1,0xe8b18500,0,0xf9f9
,1,0xe683bb00,0,0x9cba,1,0xe6b59900,7560,0x9fb4,1,0xe8b18700,0,0xf9fa
,1,0xe8b19400,0,0xf9fc,1,0xe8b19700,0,0xfa40,1,0xe8b1a900,0,0xfa41
,1,0xe99ba800,0,0x894a,1,0xe9a0bb00,0,0x9570,1,0xe4bf9a00,0,0x98db
,1,0xe6849b00,5790,0x88a4,1,0xe5a9b700,653,0x8895,1,0xe79b8300,1790,0x9475
,1,0xe8b1ad00,0,0xfa42,1,0xe885a000,0,0xedce,1,0x15000000,685,0x15
,1,0xe99c8800,0,0xe8bc,1,0xe9a19b00,0,0x935e,1,0xe9a6ae00,0,0xe967
,1,0xe5a58400,0,0x8982,1,0xe8b1b300,0,0xfa43,1,0xe68fa100,0,0x88ac
,1,0xe694b400,0,0x9dbb,1,0xe8868000,0,0xe44e,1,0xe7ab9c00,854,0x97b3
,1,0xeaae8600,0,0xfa44,1,0xe382ba00,0,0x8359,1,0xe8b29300,0,0xfa45
,1,0xc78e0000,0,0x84ee,1,0xe8b29200,0,0xfa46,1,0xe5b08a00,0,0x91b8
,1,0xe6959400,0,0xf383,1,0xe49da400,0,0xfa48,1,0xe8b2a400,0,0xfa4a
,1,0xe7b18f00,2581,0xe2d7,1,0xe3839a00,543,0x8379,1,0xd1830000,2715,0x8485
,1,0xe9ad8100,3027,0x8a40,1,0xe8b39600,0,0xfa4b,1,0xe5849f00,5162,0x8f9e
,1,0xe589b200,11138,0x8a84,1,0xe69b8700,1256,0x93dc,1,0xe7809100,0,0xe065
,1,0xe6a5ad00,573,0x8bc6,1,0xe7b78200,0,0xeda8,1,0xe8b39500,0,0xfa4c
,1,0xe9819600,9212,0xe7a6,1,0xe8a6b200,0,0xe652,1,0xe98bbc00,5163,0x8d7c
,1,0xe58a9200,0,0x999a,1,0xe58fa500,0,0x8be5,1,0xe594b800,0,0x9a58
,1,0xe7868400,0,0xe08f,1,0xe78b9700,0,0x8be7,1,0xe6b0b300,4179,0xebce
,1,0xe795bd00,0,0xf660,1,0xe8a79200,3103,0x8a70,1,0xe8b39900,0,0xfa4d
,1,0xe8b1b800,10117,0xe6b6,1,0xeaae8700,0,0xfa4e,1,0xe8b3b000,0,0xfa4f
,1,0xeaadaa00,0,0xf8cc,1,0xe8b3b100,9689,0xfa50,1,0xeaae8800,0,0xfa51
,1,0xe7969d00,0,0xe16c,1,0xe6bbb900,0,0xf549,1,0xe8b48900,0,0xfa52
,1,0xe8b29800,0,0xe6c0,1,0xcc8c0000,2952,0x867d,1,0xe8bcbe00,0,0xe778
,1,0xe4bb9400,9528,0x8e65,1,0xe6809500,5728,0x9c86,1,0xe685a800,0,0x8a53
,1,0xe68abb00,1740,0x9d5a,1,0xe8b48e00,0,0xfa53,1,0xe28ea400,2391,0x8468
,1,0xe8b5ac00,0,0xfa54,1,0xe8b68400,0,0xfa55,1,0xe8b69500,10126,0xfa56
,1,0xe9a2a800,0,0x9597,1,0xe9a7bb00,0,0xe972,1,0xe8b6a600,0,0xfa57
,1,0xe68b9b00,0,0x8fb5,1,0xeaae8900,0,0xfa58,1,0xe8b78600,0,0xfa59
,1,0xe2949700,2458,0x84af,1,0xe299aa00,0,0x81f4,1,0xe891b300,0,0xee4a
,1,0xe9a38800,0,0xefa6,1,0xe8b78800,0,0xfa5a,1,0xe49aaf00,0,0xf9d1
,1,0xe8b79900,0,0xfa5b,1,0xe6918e00,0,0x9d97,1,0xe696a100,0,0x88b4
,1,0xe69bb400,6136,0x8d58,1,0xe7ad8900,1836,0x9399,1,0xe29f9d00,0,0x84cc
,1,0xe7b7af00,8969,0x88dc,1,0xe49fbd00,0,0xfa5e,1,0xe9ae8e00,0,0x88bc
,1,0xe5809900,0,0x8cf3,1,0xe585ac00,0,0x8cf6,1,0xe6978100,0,0x9dd3
,1,0xe69c9400,3704,0x8df1,1,0xe6a1a700,4066,0x954f,1,0xe786b100,6708,0x944d
,1,0xe8988600,0,0xe562,1,0xe89d9900,0,0xe59e,1,0xd18a0000,541,0x848c
,1,0xe987b600,9321,0xe7df,1,0xe5868c00,0,0x9965,1,0xe58b9f00,0,0x95e5
,1,0xe8b7bd00,0,0xfa5f,1,0xe8b88600,0,0xfa60,1,0xeaae8a00,0,0xfa61
,1,0xe8b89400,0,0xfa62,1,0xe7be8200,0,0xe3ae,1,0xe8b89600,9720,0xfa63
,1,0xe8a89f00,9832,0x8fd7,1,0xe8adb200,0,0x8ff7,1,0xe8b8a100,0,0xfa64
,1,0xe8b8a200,0,0xfa65,1,0xefbca300,0,0x8262,1,0xe8b8a700,10449,0xfa66
,1,0xe78d8400,0,0x8d96,1,0xe7929700,0,0xf5fb,1,0xe284ab00,292,0x81f0
,1,0xeaae8b00,0,0xfa67,1,0xe8ae9200,0,0xe6a7,1,0xe4a09600,0,0xfa68
,1,0xe998af00,0,0xe897,1,0xefbd8300,1030,0x8283,1,0xe59c9800,0,0x9aa3
,1,0xe681a200,490,0x89f8,1,0xe5a6be00,0,0x8fa8,1,0xe6b89300,0,0x8c6b
,1,0xe6bda600,0,0xe052,1,0xe7a2b000,8499,0xed47,1,0xe887ba00,2924,0xe469
,1,0xc5bb0000,3248,0x85a9,1,0xe8b8b600,10273,0xfa69,1,0xe8b98b00,0,0xfa6b
,1,0xe5a28b00,0,0xf249,1,0xe5a79e00,6267,0xf26f,1,0xe68ca800,0,0x88a5
,1,0xe6be8600,0,0xe043,1,0xe7a39000,0,0x94d6,1,0xe8b99400,0,0xfa6c
,1,0xe8b9a200,0,0xfa6d,1,0xe8b9ac00,0,0xfa6e,1,0xe8b9ad00,0,0xfa6f
,1,0xe8b9af00,0,0xfa70,1,0xe8ba9800,0,0xfa71,1,0xe5ad9100,0,0x9b72
,1,0xe8ba9e00,0,0xfa72,1,0xe5b7b700,0,0x8d4a,1,0xe7a98300,0,0xe26d
,1,0xe88e8d00,0,0xf8b4,1,0xe8baae00,0,0xfa73,1,0xe8bab300,0,0xfa74
,1,0xe8bab500,10563,0xfa75,1,0xe9af9b00,0,0x91e2,1,0xe581a600,0,0xf085
,1,0xe8bab600,0,0xfa76,1,0xe6988e00,0,0x96be,1,0xe69da100,619,0x8ff0
,1,0xe782ab00,6658,0xec66,1,0xe3818100,258,0x829f,1,0xe8999300,0,0xf958
,1,0xe7beaf00,0,0xe3b9,1,0xe8a3b900,0,0xe5e5,1,0xe8babb00,0,0xfa77
,1,0xc79c0000,0,0x84fa,1,0xeaae8d00,0,0xfa78,1,0xe591bf00,1282,0x87f9
,1,0xe6a39400,1848,0x9e9f,1,0xe8bb9100,0,0xfa79,1,0xe6adba00,7414,0xf49d
,1,0xe89f8600,8439,0xe5b0,1,0xe8a49900,0,0xee91,1,0xd1910000,2849,0x8476
,1,0xe8bb9400,0,0xfa7a,1,0xe4a18e00,0,0xfa7b,1,0xe8bbb900,0,0xfa7c
,1,0xefbdb000,5424,0xb0,1,0xe6a98700,0,0x9ef0,1,0xeaae8e00,0,0xfa7d
,1,0xe280a500,6901,0x8164,1,0xe285b800,0,0x86bb,1,0xe8aa8c00,285,0x8e8f
,1,0xe98f9600,0,0xe851,1,0xe8bc8000,0,0xfa7e,1,0xe8bc8800,0,0xfa80
,1,0xefbe9000,5907,0xd0,1,0xe8bc9700,0,0xfa81,1,0xe5a2b800,0,0x9ad5
,1,0xe7948400,0,0xe14a,1,0xe2869800,0,0x8245,1,0xe6beb300,4302,0xe053
,1,0xe883b400,2109,0x93b7,1,0xe8bcab00,0,0xfa82,1,0xe99a9c00,0,0x8fe1
,1,0xe8bfb800,0,0xe79e,1,0xe8bd8000,10683,0xfa83,1,0xc2a50000,1117,0x5c
,1,0xe688a200,0,0xeae0,1,0xe6ba8000,0,0x969e,1,0xe8bd8a00,10244,0xfa84
,1,0xe8bd9800,0,0xfa85,1,0xe296b100,2151,0x83eb,1,0xe8bb8500,0,0xe75f
,1,0xe9a08f00,0,0xe8f0,1,0xcc9a0000,0,0x869e,1,0xeaae9000,0,0xfa86
,1,0xe8bea400,0,0xfa87,1,0xe8beb400,0,0xfa88,1,0xefa9a600,0,0xfa89
,1,0xe8beb600,0,0xfa8a,1,0xe2979100,7962,0x84e6,1,0xe8bf8100,0,0xfa8c
,1,0xe381ae00,40,0x82cc,1,0xe8bf8600,0,0xfa8d,1,0xefa8a400,10635,0xfa8e
,1,0xe3bdb200,0,0xeccb,1,0xefa8b100,951,0x87c7,1,0xe5b49100,0,0x9bc0
,1,0xe5b9a400,6501,0x9bf0,1,0xe5beb700,0,0xeac3,1,0xe7b08300,0,0xf784
,1,0xe3828e00,2267,0x82ec,1,0xe387a100,0,0x8398,1,0xe7bfbc00,0,0x9783
,1,0xe9b18800,4916,0x924c,1,0xefa99100,0,0xed58,1,0xe8bf8a00,11102,0xfa8f
,1,0xe5ba8400,0,0x8faf,1,0xe5bf9700,1589,0x8e75,1,0xc4860000,1618,0x85ba
,1,0xe8bf8d00,0,0xfa90,1,0xe8bf9300,0,0xfa91,1,0xe8a09300,0,0xf99d
,1,0xe8a5a600,0,0xe640,1,0xe8aab900,0,0x94ee,1,0xe9bc8e00,0,0x9343
,1,0xe8bf9500,10287,0xfa92,1,0xe593ac00,0,0xf198,1,0xe598bf00,6615,0xf1c2
,1,0xe6aa9400,0,0xebb2,1,0xe78f9e00,6840,0xe0e2,1,0xe794b100,0,0x9752
,1,0xe8a68600,5385,0x95a2,1,0xe8bfa000,9956,0xfa93,1,0xe8bfb100,0,0xfa94
,1,0xe8bfb500,10597,0xfa95,1,0xe8bfbb00,0,0xfa96,1,0xeaac9e00,0,0xf584
,1,0xe9808200,0,0xfa97,1,0xe9808c00,0,0xfa98,1,0xe7959100,1730,0x94a8
,1,0xe980b700,0,0xfa99,1,0xeaae9200,0,0xfa9a,1,0xe39e8d00,2903,0x987e
,1,0xe9969600,0,0xe87c,1,0xe99ba900,0,0xef84,1,0xe4ba8800,807,0x975c
,1,0xe4bf9b00,1105,0x98d9,1,0xe5a4a500,3751,0x9aec,1,0xe689af00,0,0xeae5
,1,0xe2888500,0,0x81c5,1,0xe8808e00,0,0xf844,1,0xe9818300,0,0xfa9b
,1,0xe88ab400,0,0xf8a0,1,0xe8bc9200,0,0xe76b,1,0xe9818400,0,0xfa9c
,1,0xeaae9300,0,0xfa9e,1,0xeaae9400,9712,0xfa9f,1,0xc2ac0000,0,0x81ca
,1,0xe5afab00,0,0x9b8d,1,0xe694b500,0,0x9dbc,1,0xe7a68a00,8570,0xe253
,1,0xe2989e00,411,0x879e,1,0xe9828500,0,0xfaa0,1,0xe382bb00,4634,0x835a
,1,0xe9828c00,0,0xfaa1,1,0xe9aca200,0,0xe9a4,1,0xe9829000,10907,0xfaa2
,1,0xe5b08b00,0,0x9071,1,0xe6959500,0,0x9dc3,1,0xe9989d00,0,0xfaa3
,1,0xe69fbb00,0,0x8db8,1,0xe7b19000,0,0xe2d9,1,0xe3839b00,2297,0x837a
,1,0x7d000000,628,0x7d,1,0xe9ad8200,9673,0x8db0,1,0xe982a100,0,0xfaa4
,1,0xe4a2b500,0,0xfaa5,1,0xe9bcbb00,976,0x9540,1,0xe69b8800,0,0xeb67
,1,0xe982b000,0,0xfaa6,1,0xe6a5ae00,1864,0x9eb8,1,0xe982b600,10812,0xfaa7
,1,0xe89c8d00,0,0xe58b,1,0xe38ea100,0,0x8775,1,0xe8a6b300,3098,0x8acf
,1,0xe5858000,0,0x9959,1,0xe58a9300,0,0x87df,1,0xe9838300,0,0xfaa8
,1,0xe594b900,0,0x9a4f,1,0xe6a68e00,4093,0x897c,1,0xc48d0000,0,0x85ca
,1,0xe6b0b400,4180,0x9085,1,0xe795be00,0,0xf661,1,0xe9838800,0,0xfaa9
,1,0xe8aca600,10000,0xe692,1,0xe8b1b900,3199,0x955e,1,0xeaae9500,0,0xfaaa
,1,0xe9839c00,0,0xfaab,1,0xe59aac00,4609,0x885c,1,0xe59fbf00,0,0x8874
,1,0xe7918b00,0,0xecaf,1,0xe9839f00,0,0xfaac,1,0xe79bb100,0,0xecea
,1,0xe880bb00,4794,0xe3d5,1,0xe8b29900,0,0xfa47,1,0xe8b7ac00,0,0xfa5c
,1,0xe8bcbf00,10242,0x9760,1,0xe4bb9500,9541,0x8e64,1,0xe6809600,0,0x957c
,1,0xeaae9600,0,0xfaad,1,0xe68abc00,1737,0x899f,1,0xe2899200,4032,0x81e0
,1,0xe28ea500,7803,0x8469,1,0xe9848000,10088,0xfab0,1,0xe8b88c00,0,0xfa5d
,1,0xe8bd9f00,763,0x8d8c,1,0xe983ab00,0,0xfab1,1,0xe983be00,10575,0xfab2
,1,0xe5a69200,0,0x888c,1,0xe68b9c00,5876,0x9d60,1,0xe5b0b800,6371,0x9b99
,1,0xe983bf00,0,0xfab3,1,0x66000000,11480,0x66,1,0xe299ab00,8017,0x81f9
,1,0xe9848400,0,0xfab4,1,0xe9848600,0,0xfab5,1,0xe9849800,0,0xfab6
,1,0xe9adaf00,0,0x9844,1,0xe9849c00,0,0xfab7,1,0xc2b30000,0,0x854c
,1,0xe9849e00,0,0xfab8,1,0xe5bbbe00,1549,0x9c4f,1,0xe984b700,0,0xfab9
,1,0xe29f9e00,253,0x84cd,1,0xe984b900,0,0xfaba,1,0xe89cba00,0,0xf979
,1,0xe9ae8f00,0,0xefc5,1,0xe5809a00,0,0x98df,1,0xe585ad00,0,0x985a
,1,0xe5b78b00,6466,0x989d,1,0xe69c9500,6152,0x92bd,1,0xe984ba00,10845,0xfabb
,1,0xe6a6bb00,0,0x9ece,1,0xe8988700,8342,0x9168,1,0xe7bda300,0,0xf7eb
,1,0xe9858600,0,0xfabc,1,0xe9858700,10920,0xfabd,1,0xe5868d00,0,0x8dc4
,1,0xe58ba000,0,0x99ab,1,0xe9859700,0,0xfabe,1,0xe9859900,0,0xfabf
,1,0xe7879200,0,0xe096,1,0xe78ca500,0,0xe0ce,1,0xe7be8300,0,0xe3b0
,1,0xe9838400,0,0xeee0,1,0xe985a100,9840,0xfac0,1,0xe98daa00,0,0xef5d
,1,0xe58c8000,0,0x87e9,1,0xe985a400,0,0xfac1,1,0xefbca400,0,0x8263
,1,0xe59bb900,10456,0x9a9a,1,0xe78d8500,428,0x8e82,1,0xe6b2a100,4209,0x9676
,1,0xe985b400,0,0xfac2,1,0xe985b900,0,0xfac3,1,0xe8ae9300,0,0xe6a8
,1,0xe8b3a600,0,0x958a,1,0xe8b8b900,0,0xfa6a,1,0xefbd8400,0,0x8284
,1,0xe9868500,0,0xfac4,1,0xe681a300,0,0x9c93,1,0x4f000000,58,0x4f
,1,0xe7988b00,0,0xe186,1,0xe9868e00,0,0xfac5,1,0xe986a800,0,0xfac6
,1,0xe887bb00,0,0xe46a,1,0xe9999000,2960,0x8cc0,1,0xe99ea300,0,0xe8e2
,1,0xe986ae00,0,0xfac7,1,0xe6828300,0,0x9c9d,1,0xe986b300,0,0xfac8
,1,0xe5acb200,0,0x9b6a,1,0xe986b600,0,0xfac9,1,0xe7a39100,0,0xe1f4
,1,0xe8889b00,0,0x9143,1,0xe9878300,0,0xfaca,1,0xe99f8300,0,0xe8e5
,1,0xe9878400,10445,0xfacb,1,0xe9a9a900,0,0xe989,1,0xe9879a00,0,0xfacc
,1,0xe68d8900,5889,0x91a8,1,0xeaae9700,0,0xfacd,1,0xeaae9800,0,0xface
,1,0xe987ac00,0,0xfacf,1,0xe7ae9700,2551,0x8e5a,1,0xe987ae00,0,0xfad0
,1,0xe7b8bd00,0,0xe360,1,0xe9888100,0,0xfad1,1,0xe9888a00,0,0xfad2
,1,0xe9889600,0,0xfad3,1,0xe5b38500,0,0x9bb0,1,0xc2ba0000,211,0x8550
,1,0xc9920000,453,0x8659,1,0xe6a2b500,1842,0x9e90,1,0xe3818200,1886,0x82a0
,1,0xe7b99d00,2689,0xe383,1,0xe89ea700,0,0xf986,1,0xe9889700,0,0xfad4
,1,0xeaae9900,0,0xfad5,1,0xccaf0000,0,0x8690,1,0xe988b300,0,0xfad6
,1,0xe5be8b00,0,0x97a5,1,0xe6a39500,0,0x9ea1,1,0xe9898200,0,0xfad7
,1,0xe78db200,239,0x8a6c,1,0xe89f8700,0,0xe5af,1,0xe8a49a00,0,0xee92
,1,0xe8a9ad00,3116,0xe66b,1,0xe9898700,0,0xfad8,1,0xe58d8d00,0,0x99c2
,1,0x38000000,5519,0x38,1,0xefbdb100,5436,0xb1,1,0xe6a98800,0,0x9ef4
,1,0xe6ae9b00,0,0xf49f,1,0xe280a600,3972,0x8163,1,0xe285b900,0,0x86bc
,1,0xe8aa8d00,0,0x9446,1,0xe98f9700,0,0xe852,1,0xe9898a00,0,0xfad9
,1,0xe5938000,804,0x88a3,1,0xefbe9100,5928,0xd1,1,0xe59da600,925,0x9252
,1,0xe5a2b900,6215,0x9acf,1,0xe7948500,3975,0xe14c,1,0xe2869900,11493,0x8247
,1,0xe9898e00,0,0xfada,1,0xe7a3be00,8527,0xf6d8,1,0xe9899100,0,0xfadb
,1,0xe99a9d00,0,0xef7d,1,0xe8bfb900,0,0xe791,1,0xe4be8f00,0,0x98cb
,1,0xc7b80000,1116,0x84f2,1,0xe688a300,0,0xf2f2,1,0xe68db600,5901,0x9d78
,1,0xe6bf9400,0,0xe05d,1,0xe9899600,0,0xfadc,1,0xe296b200,2491,0x81a3
,1,0xe8bb8600,0,0xe75c,1,0xe9a09000,0,0x9761,1,0xe9899900,0,0xfadd
,1,0xe9aab600,0,0xefb3,1,0xe6898300,0,0xeae2,1,0xe5ae9f00,1421,0x8ec0
,1,0xe989a000,0,0xfade,1,0xe698bc00,0,0x928b,1,0xe2979200,0,0x84e7
,1,0xe7afa400,0,0x93c4,1,0xe381af00,0,0x82cd,1,0xe989a100,0,0xfadf
,1,0xe9ab9600,0,0xefb7,1,0xe989a500,10822,0xfae0,1,0xefa8b200,948,0x87ce
,1,0xe989a700,0,0xfae1,1,0xe989a800,0,0xfae2,1,0xe69eaf00,6160,0x8ccd
,1,0xe783b900,0,0x9642,1,0xe3828f00,137,0x82ed,1,0xe387a200,0,0x8399
,1,0xeaae9a00,0,0xfae3,1,0x21000000,10852,0x21,1,0xefa99200,5364,0xed5a
,1,0xe588a700,3622,0x9987,1,0xe58dba00,0,0xf175,1,0xe69f8f00,694,0x9490
,1,0xc9990000,1608,0x864f,1,0xeaae9b00,0,0xfae4,1,0xe78ebf00,9062,0xf5e3
,1,0xe9808b00,4720,0xe799,1,0xe989bc00,0,0xfae5,1,0xe989bd00,0,0xfae6
,1,0xe5898700,0,0x91a5,1,0xe58e9a00,0,0x8cfa,1,0xe593ad00,0,0x9a4c
,1,0xe6a58200,6616,0xeb94,1,0xe989bf00,0,0xfae7,1,0xe98a8d00,0,0xfae9
,1,0xe6b4bb00,191,0x8a88,1,0xe8a68700,0,0x9465,1,0xe8ab9a00,0,0xe67b
,1,0xe98a9700,0,0xfaea,1,0xe995b700,0,0x92b7,1,0xe98a9900,0,0xfaeb
,1,0xe4b9a900,5630,0xf04f,1,0xe59eb300,0,0x9ab6,1,0xe6b08800,0,0x9f81
,1,0xe6b59b00,7564,0xf4d0,1,0xe287a600,0,0x824a,1,0xe98a9f00,0,0xfaec
,1,0xe9918400,0,0xe868,1,0xe98aa700,0,0xfaed,1,0xe99baa00,0,0x90e1
,1,0xe4ba8900,1061,0x9188,1,0xe59f9300,0,0x9abe,1,0xe98aab00,0,0xfaee
,1,0xeaae9c00,0,0xfaef,1,0xe6bb8e00,4274,0xec41,1,0xeaae9d00,0,0xfaf0
,1,0xe98abf00,0,0xfaf2,1,0xe7aabe00,0,0xf755,1,0xe99c8a00,3001,0x97ec
,1,0xe98b8000,0,0xfaf3,1,0xe98b8600,0,0xfaf4,1,0xe5a58600,0,0xf261
,1,0xe98b8e00,10414,0xfaf5,1,0xce970000,892,0x83a5,1,0xe694b600,0,0x9dbe
,1,0xe8868200,0,0xe44f,1,0xe88b9500,0,0xede6,1,0xa000000,4418,0xa
,1,0xe382bc00,8927,0x835b,1,0xe9a79000,0,0x9293,1,0xe9aca300,0,0xe9a5
,1,0xe9b1b600,0,0xe9e6,1,0xe98b9000,0,0xfaf6,1,0xe6959600,0,0x9dc2
,1,0xe98b9700,0,0xfaf7,1,0xe69fbc00,0,0xeb7b,1,0xe8918800,8117,0xee47
,1,0xe3839c00,2654,0x837b,1,0xe89bae00,0,0x94d8,1,0xe980b800,3439,0x88ed
,1,0xe98b9900,0,0xfaf8,1,0xe584a100,0,0x9953,1,0xe589b400,0,0x9993
,1,0xe69b8900,6128,0x9dfa,1,0xe98ba500,0,0xfaf9,1,0xe6a5af00,1868,0x8f7c
,1,0xe78ab900,0,0xe0b9,1,0xe89c8e00,0,0xf971,1,0xe8a1a100,3036,0x8d74
,1,0xe986ab00,0,0xe7ce,1,0xe5858100,0,0x88f2,1,0xe58a9400,0,0x9999
,1,0xe98ba700,0,0xfafa,1,0xe98c9100,0,0xfafb,1,0xeaae9e00,0,0xfafc
,1,0xc9a00000,379,0x8647,1,0xe790ac00,6849,0xeca8,1,0xe795bf00,2796,0x8b45
,1,0xe9878b00,4751,0xe7d7,1,0xeaae9f00,10003,0xfb40,1,0xe8b1ba00,4992,0xe6b7
,1,0xe98bb700,11049,0xfb41,1,0xccbd0000,0,0x868e,1,0xe4bab600,5654,0x98b7
,1,0xe98bb900,0,0xfb42,1,0xe6b19500,0,0x9f88,1,0xe98bbb00,0,0xfb43
,1,0xe79bb200,7069,0x96d3,1,0xe880bc00,0,0xedc7,1,0xe98c8200,0,0xfb44
,1,0xe98c8d00,0,0xfb45,1,0xe98c9500,0,0xfb46,1,0xe4bb9600,4840,0x91bc
,1,0xe5a0a000,6188,0xf240,1,0xe5a5b300,0,0x8f97,1,0xe68abd00,6985,0x928a
,1,0xe98c9d00,0,0xfb47,1,0xe28ea600,0,0x846a,1,0xe7a6b800,0,0xed63
,1,0xe98c9e00,0,0xfb48,1,0xe98ca700,10259,0xfb49,1,0xe9a2aa00,0,0xe943
,1,0xe5a18000,0,0x95bb,1,0xe5a69300,1369,0x8b57,1,0xe68b9d00,1165,0x9471
,1,0xe5b0b900,0,0x9b9a,1,0xe98ca900,0,0xfb4a,1,0xe7a79800,4528,0x94e9
,1,0xe299ac00,4568,0x81fa,1,0xe891b500,750,0x88a8,1,0xeaaea000,0,0xfb4b
,1,0xeaaea100,10476,0xfb4c,1,0xe98d8700,10620,0xfb4d,1,0xe98d9100,0,0xfb4e
,1,0xe98d9700,0,0xfb4f,1,0xce9e0000,1224,0x83ac,1,0xe5bbbf00,0,0x93f9
,1,0xe7ad8b00,0,0x8bd8,1,0xe29f9f00,0,0x84ce,1,0xe7b7b100,0,0xf7ca
,1,0xe89cbb00,0,0xe591,1,0xe98d9a00,0,0xfb50,1,0xe98dab00,0,0xfb51
,1,0xe585ae00,0,0x9961,1,0xe5b78c00,1499,0x8ade,1,0xe5bc9f00,1553,0x92ed
,1,0xe98db100,0,0xfb52,1,0xe6a6bc00,7282,0xf45a,1,0xe7b89100,0,0xedaf
,1,0xe7bda400,0,0xf7ec,1,0xe8a2ae00,0,0xe5d7,1,0xe9b48300,4843,0xe9ee
,1,0xe98db300,0,0xfb53,1,0xe98ea100,0,0xfb54,1,0xeaaea200,0,0xfb55
,1,0xeaaea300,0,0xfb56,1,0xe7879300,0,0xf5a2,1,0xe6acaf00,0,0xf498
,1,0xe98e8800,0,0xfb57,1,0xe8a38e00,4885,0xee8b,1,0xe8a8a100,0,0xf9d5
,1,0xe8adb400,10026,0xe6a2,1,0xe58c8100,0,0x96e6,1,0xe98e8b00,0,0xfb58
,1,0xefbca500,168,0x8264,1,0xe59bba00,10565,0x8cc5,1,0xe98e8f00,0,0xfb59
,1,0xc9a70000,420,0x8661,1,0xe6b7b500,7600,0x95a3,1,0xe8a98100,0,0xe665
,1,0xe8ae9400,10047,0xf9f0,1,0xe98e9e00,0,0xfb5a,1,0xe98fb500,10579,0xfb5b
,1,0xefbd8500,1087,0x8285,1,0xeaaea400,0,0xfb5c,1,0xe681a400,0,0x9c95
,1,0xe6b38200,0,0xf4be,1,0xe6b89500,0,0x9fba,1,0xe79d9f00,7096,0xecf4
,1,0xe882a900,0,0x8ca8,1,0xe887bc00,2927,0x8950,1,0xeaaea500,0,0xfb5d
,1,0xe8bead00,0,0xe786,1,0xe4bd8300,0,0x92cf,1,0xe6828400,5749,0x9c9f
,1,0xe98fb100,0,0xfb5e,1,0xe98f8100,0,0xfb5f,1,0xe6be8800,6381,0xec4e
,1,0xe98f8700,10112,0xfb60,1,0xe8889c00,0,0x8f77,1,0xe98fa200,0,0xfb62
,1,0xe3ac8e00,9652,0xf3ac,1,0xe98fa700,10447,0xfb63,1,0xe9a9aa00,0,0xe98b
,1,0xe9908900,0,0xfb64,1,0xe9908f00,0,0xfb65,1,0xe5b2a600,6383,0xf0c1
,1,0xe9909600,0,0xfb66,1,0xe9909700,0,0xfb67,1,0xe7ae9800,0,0xe2ae
,1,0xcb880000,8212,0x8672,1,0xe7b8be00,0,0x90d1,1,0xe3b79400,0,0xf597
,1,0xe98fbb00,10763,0xfb68,1,0xe990b200,0,0xfb69,1,0xe990b400,0,0xfb6a
,1,0xe5b89900,0,0x9be3,1,0xcea50000,630,0x83b2,1,0xe6a2b600,432,0x8a81
,1,0xe3818300,752,0x82a1,1,0xe8999500,2330,0x997c,1,0xe38ba900,0,0x86ee
,1,0xe983b200,0,0xfaaf,1,0xe9b59000,5331,0xea44,1,0xe5879b00,0,0x997a
,1,0xe990bb00,11146,0xfb6b,1,0xe5be8c00,0,0x8ce3,1,0xe6a39600,0,0xf3fc
,1,0xe6a8a900,0,0x8ca0,1,0xe9918500,0,0xfb6c,1,0xeaaea600,0,0xfb6d
,1,0xe9849200,0,0xe7be,1,0xe8a9ae00,9893,0x9146,1,0xe9bb8300,10221,0xefef
,1,0xeaaea700,0,0xfb6e,1,0xe592a100,0,0x87fc,1,0xefbdb200,5448,0xb2
,1,0xe7898000,6750,0xe0ac,1,0xe991ad00,0,0xfb6f,1,0xe793a600,1717,0x8aa2
,1,0xe285ba00,0,0x86bd,1,0xe98a8500,0,0x93ba,1,0xe98f9800,0,0xe855
,1,0xe991af00,0,0xfb70,1,0xe5938100,7973,0x9569,1,0xefbe9200,1040,0xd2
,1,0xe4bdb000,0,0x98cf,1,0xe5a2ba00,6216,0x9ad2,1,0xe2818700,0,0x84ea
,1,0xc9ae0000,0,0x85e9,1,0xe79eac00,7115,0x8f75,1,0xe9968600,0,0xfb73
,1,0xe9968c00,10256,0xfb74,1,0xe9968d00,0,0xfb75,1,0xe8bfba00,10314,0xe792
,1,0xeaaea800,0,0xfb76,1,0xe5a39a00,874,0x8884,1,0xe5a8ad00,0,0xf27b
,1,0xe68db700,1763,0x8fb7,1,0xe6bf9500,0,0xe05b,1,0xe291a000,8531,0x8740
,1,0xe296b300,0,0x81a2,1,0xe996ab00,0,0xfb77,1,0xe9a09100,0,0x8ae6
,1,0xe996b400,0,0xfb78,1,0xeaaea900,10607,0xfb79,1,0xe9978800,0,0xfb7a
,1,0xeaaeaa00,0,0xfb7b,1,0xeaaeab00,0,0xfb7c,1,0xe998ac00,0,0xfb7d
,1,0xe2979300,445,0x84e8,1,0xe88f9c00,2201,0x8dd8,1,0xe381b000,4623,0x82ce
,1,0xe998b400,0,0xfb80,1,0xeaaeac00,0,0xfb81,1,0xe998bc00,10826,0xfb82
,1,0xefa8b300,218,0x87e1,1,0xe9998100,0,0xfb83,1,0xe6999d00,0,0x9dea
,1,0xe5beb900,6602,0x934f,1,0xe999a100,0,0xfb84,1,0xe3829000,8884,0x82ee
,1,0xe387a300,0,0x839a,1,0xe38cb600,9152,0x8766,1,0xeaaead00,10854,0xfb85
,1,0xe5839500,4362,0x966c,1,0xe99a8200,0,0xfb86,1,0xe58dbb00,0,0x99c8
,1,0xe5bf9900,695,0x965a,1,0xe7849a00,1617,0x95b0,1,0xeaaeae00,0,0xfb87
,1,0xe99a9a00,0,0xfb88,1,0xe8a09500,9671,0xe5be,1,0xeaaeaf00,0,0xfb89
,1,0xe98ab200,0,0xfaf1,1,0xe9bc9000,0,0xeff1,1,0xe4a7a700,0,0xfb8a
,1,0xe593ae00,0,0x9a4b,1,0xe99aa900,0,0xfb8b,1,0xe78a8d00,0,0xec87
,1,0xe78fa000,0,0x8eec,1,0xe794b300,6920,0x905c,1,0xe8a68800,0,0xe649
,1,0xe98b9200,2874,0x964e,1,0xe99aaf00,10289,0xfb8c,1,0xe995b800,0,0xfb71
,1,0xe5948e00,0,0x8843,1,0xeaaca000,0,0xf5b1,1,0xe99ab300,0,0xfb8d
,1,0xe99aba00,0,0xfb8e,1,0xe6b59c00,0,0x956c,1,0xe287a700,1777,0x824b
,1,0xe99abd00,0,0xfb8f,1,0xe8b18e00,10103,0xe6b1,1,0xe9969800,0,0xe87d
,1,0xe99bab00,0,0x8eb4,1,0xe4ba8a00,8007,0x98ad,1,0xe4bf9d00,1103,0x95db
,1,0xe5a4a700,6240,0x91e5,1,0xe689b100,5851,0x88b5,1,0xe2888700,706,0x81de
,1,0xc9b50000,409,0x864e,1,0xe4a7ba00,0,0xfb90,1,0xe7aabf00,0,0xe283
,1,0xe8bc9400,0,0x95e3,1,0xe9a19e00,0,0x97de,1,0xeaaeb000,0,0xfb91
,1,0xe5a58700,0,0x8aef,1,0xe68a9100,1157,0x977d,1,0xe99b9800,0,0xfb92
,1,0xe694b700,0,0x9dbd,1,0xe8868300,0,0xe44b,1,0xe7ab9f00,0,0xe8ed
,1,0xe890a900,0,0x948b,1,0xe382bd00,0,0x835c,1,0xe9a79100,0,0xe96d
,1,0xe499a500,0,0xf9c1,1,0xe9b1b700,0,0xefd5,1,0xe5b08d00,0,0x9b94
,1,0xe6959700,0,0x9473,1,0xe99b9a00,0,0xfb93,1,0xe2998000,0,0x818a
,1,0xe8918900,0,0x9774,1,0xe3839d00,2657,0x837c,1,0xe89baf00,0,0xe582
,1,0xe980b900,3438,0xe79d,1,0xe99b9d00,0,0xfb94,1,0xe4a88400,0,0xfb95
,1,0xe589b500,11141,0x916e,1,0xe5bb9300,0,0x8a66,1,0xe6a09d00,0,0xeb7e
,1,0xe785a700,0,0x8fc6,1,0xe99c9400,0,0xfb96,1,0xc4be0000,0,0x85ad
,1,0x72000000,2794,0x72,1,0xe986ac00,0,0xeef7,1,0xe98bbf00,11047,0xef53
,1,0xe99ca300,11157,0xfb97,1,0xe58fa800,0,0x99d9,1,0xe4a8a900,0,0xfb98
,1,0xe7868700,0,0xec7b,1,0xceb30000,0,0x83c1,1,0xe6b0b600,0,0xf4b0
,1,0xe8a28200,0,0xe5d4,1,0xe9878c00,0,0x97a2,1,0xe8aca800,4960,0xe695
,1,0xe991b200,0,0xef6c,1,0xe5908800,0,0x8d87,1,0xe99cb600,0,0xfb99
,1,0xe59aae00,8925,0x9a8c,1,0xe99d8100,0,0xfb9a,1,0xe7918d00,0,0xf5f2
,1,0xe99d8700,0,0xfb9b,1,0xe288b400,0,0x8188,1,0xe880bd00,0,0x925e
,1,0xe8b29b00,0,0xfa49,1,0xe997a500,0,0xe892,1,0xe99cb800,0,0x9e50
,1,0xe4bb9700,1075,0x98bd,1,0xc3820000,1108,0x8558,1,0xe5a5b400,1351,0x937a
,1,0xe7978000,0,0xf66b,1,0xe99d9500,0,0xfb9c,1,0xe28ea700,0,0x846b
,1,0xe7a6b900,2445,0xe25a,1,0xe99d9700,0,0xfb9d,1,0xe8bda100,0,0x8c44
,1,0xe99d9b00,10364,0xfb9e,1,0xe5a18100,0,0x97db,1,0xe99daa00,0,0xfb9f
,1,0xeaaeb100,0,0xfba0,1,0xe5b0ba00,0,0x8eda,1,0xe7a28600,0,0xe1ec
,1,0xeaaeb200,0,0xfba1,1,0xe299ad00,11472,0x81f3,1,0xe891b600,0,0xf8ce
,1,0xe99e9600,10394,0xfba2,1,0xe99e9a00,0,0xfba3,1,0xe99e9e00,0,0xfba4
,1,0xe99ea200,0,0xfba5,1,0xe6919100,0,0xeaf7,1,0xe696a400,0,0x8bd2
,1,0xe69bb700,0,0x9e4a,1,0x5b000000,2193,0x5b,1,0xe29fa000,27,0x84cf
,1,0xe897a900,2321,0x94cb,1,0xe99eb100,0,0xfba6,1,0xe9ae9100,0,0xe9b8
,1,0xe5809c00,0,0xf07a,1,0xe99eb200,0,0xfba7,1,0xe5b78d00,1500,0x9bd9
,1,0xe69c9700,0,0x984e,1,0xe99ebe00,0,0xfba8,1,0xe99f8c00,0,0xfba9
,1,0xe7b89200,0,0xe372,1,0xe99f9100,0,0xfbaa,1,0xe982a600,0,0x964d
,1,0xe99f9400,0,0xfbab,1,0xe5868f00,0,0x9967,1,0xe58ba200,0,0x90a8
,1,0xe5bd8000,0,0xeab7,1,0xe7828100,0,0xf57b,1,0xe6a79d00,1879,0x9ecd
,1,0xe78ca700,0,0xec8d,1,0xe7be8500,0,0x9785,1,0xe8a38f00,0,0x97a0
,1,0xcb9d0000,4758,0x85b4,1,0xe98dac00,0,0x8c4c,1,0xe58c8200,0,0x93f5
,1,0xe5919500,0,0xf184,1,0xefbca600,1028,0x8265,1,0xe99f9800,10616,0xfbac
,1,0xe6ad9000,0,0x9f5e,1,0xceba0000,0,0x83c8,1,0xe6b7b600,0,0xf4ea
,1,0xe99f9900,0,0xfbad,1,0xe98e8c00,9399,0x8a99,1,0xe99fa100,0,0xfbae
,1,0xe998b200,0,0x9668,1,0xefbd8600,0,0x8286,1,0xe99fb100,0,0xfbaf
,1,0xe681a500,0,0x9270,1,0xe9a08400,0,0xfbb0,1,0xe7988d00,0,0xe187
,1,0xe79da000,0,0xf6a4,1,0xe882aa00,0,0x9662,1,0xe8b48800,8608,0x91a1
,1,0xe9a08d00,0,0xfbb1,1,0xe8beae00,0,0xe390,1,0xe9a08e00,0,0xfbb2
,1,0xe4aabc00,0,0xfbb5,1,0xc3890000,0,0x855f,1,0x44000000,217,0x44
,1,0xe79e8000,0,0xecf7,1,0xe8838a00,0,0xedc8,1,0xeaaeb300,0,0xfbb6
,1,0xe7adb900,0,0xf76c,1,0xe8bf8e00,0,0x8c7d,1,0xe9a49800,0,0xe950
,1,0xe9a9ab00,0,0xe98a,1,0xe5a88100,10747,0x88d0,1,0xe5ad9400,0,0x8d45
,1,0xe6929e00,5976,0x93b3,1,0xe697b100,0,0x9ddb,1,0xe7a98600,0,0x9673
,1,0xe7ae9900,0,0xe2b7,1,0xd09b0000,0,0x844c,1,0xe9a0a300,0,0xfbb7
,1,0xe9a0b200,0,0xfbb8,1,0xe9a0b300,0,0xfbb9,1,0xe9a0a500,0,0xfbba
,1,0xe5b38700,5387,0x9bb2,1,0xe5b89a00,0,0x9be2,1,0xe5bdad00,1567,0x9c64
,1,0xe782ae00,0,0xe07b,1,0xe3818400,0,0x82a2,1,0xe8999600,0,0xf959
,1,0xe7beb200,0,0xe3ba,1,0xe8a3bc00,0,0xe5e7,1,0xe9b59100,0,0xea43
,1,0xe5879c00,5170,0xeaa3,1,0xe58caf00,0,0x99b9,1,0xe9a18700,0,0xfbbb
,1,0xe6a39700,0,0x9ea5,1,0xe6a8aa00,0,0x89a1,1,0xe9a1a600,0,0xfbbc
,1,0xe7bf9200,0,0x8f4b,1,0xe8a49c00,0,0xee8e,1,0xe989a600,0,0x8fde
,1,0xe9bb8400,3168,0x89a9,1,0xe9a2ab00,0,0xfbbd,1,0xe592a200,0,0x99f6
,1,0xefbdb300,5452,0xb3,1,0xe9a2ad00,0,0xfbbe,1,0xe78e9400,0,0xf5d9
,1,0xe6b3b000,1950,0x91d7,1,0xe285bb00,0,0x86be,1,0xe9a2b000,0,0xfbbf
,1,0xeaaeb400,0,0xfbc0,1,0xe9a2b700,0,0xfbc1,1,0xe5938200,0,0x9a41
,1,0x2d000000,11469,0x2d,1,0xe59da800,0,0xf1e2,1,0xe682b200,1114,0x94df
,1,0xe2818800,0,0x84eb,1,0xe6b9a300,0,0xf4fb,1,0xe79ead00,4051,0x97c4
,1,0xe9a2b800,0,0xfbc2,1,0xe9a2bb00,0,0xfbc3,1,0xe9a2bc00,0,0xfbc4
,1,0xe99fb200,9668,0xe8ec,1,0xe4be9100,1098,0x98d0,1,0xe9a2bf00,10855,0xfbc5
,1,0xe9a38200,0,0xfbc6,1,0xe9a38700,0,0xfbc7,1,0xe79f8d00,0,0xe1dd
,1,0xe291a100,0,0x8741,1,0xe889aa00,0,0xe482,1,0xe88ebd00,2992,0xe4cd
,1,0xe9a09200,0,0x94d0,1,0xe9a38b00,0,0xfbc8,1,0xe9aab800,0,0x8a5b
,1,0xe9a3a000,0,0xfbc9,1,0xc3900000,152,0x8566,1,0xe5b3b400,0,0x988d
,1,0xe7a58000,0,0xe24a,1,0xe7aa9300,8694,0x918b,1,0xe7afa600,8769,0xe2c2
,1,0xe381b100,313,0x82cf,1,0xeaaeb500,0,0xfbca,1,0xe9ab9800,0,0x8d82
,1,0xe3bdb500,0,0xf665,1,0xefa8b400,10984,0x87e6,1,0xe5b49400,0,0x9bc1
,1,0xe6999e00,0,0x9de9,1,0xe9a3a100,0,0xfbcb,1,0xe9a3a300,0,0xfbcc
,1,0xe3829100,8251,0x82ef,1,0xe387a400,11477,0x839b,1,0xe89fb600,0,0xe5b8
,1,0xe9a3a500,0,0xfbcd,1,0xe5839600,4367,0x9944,1,0xe588a900,0,0x9798
,1,0xe5ba8700,0,0x94dd,1,0xe69f9100,0,0x8ab9,1,0xe9a3aa00,0,0xfbce
,1,0xe789ae00,0,0xf5b2,1,0xe89b8300,0,0xf968,1,0xe38d9700,2777,0x8768
,1,0xe9a3b000,0,0xfbcf,1,0x16000000,119,0x16,1,0xe5898900,0,0x87dc
,1,0xe9a3b100,0,0xfbd0,1,0xe593af00,0,0xf199,1,0xe9a3b300,0,0xfbd1
,1,0xe6aa9700,7332,0x9f40,1,0xe78fa100,0,0xf5e6,1,0xe6b4bd00,0,0x9fa8
,1,0xe8a68900,0,0xf9c4,1,0xe8ab9c00,4957,0x92b3,1,0xe9a48800,0,0xfbd2
,1,0xe995b900,0,0xfb72,1,0xe5948f00,0,0x9a48,1,0xeaaca100,0,0xf5b7
,1,0xe4bebe00,0,0xf070,1,0xe6b08a00,7450,0xf4ad,1,0xe7959400,0,0x94c8
,1,0xe287a800,7671,0x8249,1,0xeaaeb600,0,0xfbd4,1,0xe8b18f00,0,0xf9fb
,1,0xe9969900,0,0xe87e,1,0xe9a49600,10948,0xfbd5,1,0xe4ba8b00,8011,0x8e96
,1,0xe9a49700,0,0xfbd6,1,0xe6849f00,0,0x8ab4,1,0xeaaeb700,0,0xfbd7
,1,0xe2888800,0,0x81b8,1,0xe8809100,0,0xf845,1,0xe7a5ad00,0,0x8dd5
,1,0xe8b78200,4550,0xe6e3,1,0xe8bc9500,0,0xe76a,1,0xe9a49a00,0,0xfbd8
,1,0xeaaeb800,0,0xfbdb,1,0xe5a58800,0,0x93de,1,0xe5aa9b00,1398,0x9551
,1,0xe5afae00,3774,0x97be,1,0xe694b800,0,0x9dbf,1,0xe7a68d00,8573,0x89d0
,1,0xe7aba000,626,0x8fcd,1,0xe890aa00,4420,0xe4d2,1,0xe382be00,8933,0x835d
,1,0xe9a79200,0,0x8bee,1,0xe9aca500,0,0xe9a6,1,0xe9b1b800,0,0xe9e7
,1,0xe5b08e00,0,0x93b1,1,0xc3970000,1488,0x817e,1,0xe69aab00,0,0x8e62
,1,0xe69fbe00,0,0x968f,1,0xe8918a00,0,0xf8c5,1,0xe3839e00,4465,0x837d
,1,0xe388b100,0,0x878a,1,0xe9ad8500,0,0x96a3,1,0xe9a4b100,0,0xfbdc
,1,0xe9a4b200,0,0xfbdd,1,0xe5b68100,3660,0x9898,1,0xe9a4b300,0,0xfbde
,1,0xe7809500,1592,0x956d,1,0xe785a800,0,0xec78,1,0xe9a4ba00,0,0xfbdf
,1,0xe89c9000,0,0xf972,1,0xd0a90000,35,0x845a,1,0xe9a4bb00,0,0xfbe0
,1,0xe5858300,0,0x8cb3,1,0xe9a4bc00,0,0xfbe1,1,0xe58fa900,0,0x9240
,1,0xe594bc00,0,0xf19f,1,0xe6a69100,1867,0x9ed2,1,0xe78b9b00,3924,0x8d9d
,1,0xe6b0b700,4182,0x9558,1,0xe38f8400,0,0x8774,1,0xe9878d00,9292,0x8f64
,1,0xe98ca000,0,0x8ff9,1,0xe8b1bc00,0,0xe6bf,1,0xe5908900,0,0x8b67
,1,0xe5959c00,0,0x9a54,1,0xeaadae00,0,0xf8fc,1,0xe6ac8400,0,0x9793
,1,0xe6b19700,0,0x8abe,1,0xe6b6aa00,0,0xebec,1,0xe288b500,466,0x81e6
,1,0xe8ad8900,0,0xe69a,1,0xe9a58000,0,0xfbe2,1,0xe8b7af00,0,0x9848
,1,0xe99cb900,0,0xe8c6,1,0xe4bb9800,9600,0x9574,1,0xe6809900,0,0x9c80
,1,0xe5a5b500,0,0xf267,1,0xe9a58100,0,0xfbe3,1,0xe9a58600,0,0xfbe4
,1,0xe28ea800,658,0x846c,1,0xe7a6ba00,0,0xe25b,1,0xe8b88f00,0,0x93a5
,1,0xe99d9900,3010,0x90c3,1,0xe9a58d00,0,0xfbe5,1,0xe9a7bf00,0,0x8f78
,1,0xe9a58e00,0,0xfbe6,1,0xe9a59c00,0,0xfbe7,1,0xe5b0bb00,0,0x904b
,1,0xe7a28700,0,0x92f4,1,0xe2949b00,2459,0x84ae,1,0xe299ae00,4407,0x81f8
,1,0xe891b700,0,0xe4d6,1,0xe9a59f00,0,0xfbe8,1,0xe9a5a000,0,0xfbe9
,1,0xe9adb200,0,0xefc1,1,0xe5ac8800,0,0xf292,1,0xe5b19b00,0,0xeff9
,1,0xe696a500,1226,0x90cb,1,0xe69bb800,0,0x8f91,1,0xe7ad8d00,8745,0xe2a1
,1,0xe29fa100,4435,0x84d0,1,0xe897aa00,0,0xe54d,1,0xe9a6a300,0,0xfbea
,1,0xe9ae9200,0,0x95a9,1,0xe9b3a500,0,0x92b9,1,0xe9a6a600,0,0xfbeb
,1,0xe6978500,0,0x97b7,1,0xc39e0000,0,0x8573,1,0xe6a1ab00,0,0xf3e4
,1,0xe6a6be00,2235,0x9ec9,1,0xe8988a00,0,0xe55d,1,0xe9a6b900,0,0xfbec
,1,0xe8a2b000,0,0xe5dc,1,0xe9a6bd00,0,0xfbed,1,0xe5869000,0,0xe3ec
,1,0xe58ba300,0,0x99a8,1,0xe590b600,1271,0x99e3,1,0xe9a6bf00,0,0xfbee
,1,0xe7879500,1630,0x898d,1,0xe78ca800,0,0xec8e,1,0xe7be8600,0,0xe3af
,1,0xe9838700,0,0xeee2,1,0xd0b00000,166,0x8470,1,0xe8adb600,0,0xf9ec
,1,0xe9a78300,0,0xfbef,1,0xe9a78900,0,0xfbf0,1,0xefbca700,1025,0x8266
,1,0xe9a79400,10732,0xfbf1,1,0xe9a79900,0,0xfbf2,1,0xe9a79e00,0,0xfbf3
,1,0xe6b7b700,7606,0x8dac,1,0xeaaeb900,0,0xfbf4,1,0xe8ae9600,0,0xe6a9
,1,0xe9a7b000,0,0xfbf5,1,0xe998b300,0,0xfb7e,1,0xefbd8700,5592,0x8287
,1,0xe59c9c00,0,0x9aa6,1,0xe9a7b900,10967,0xfbf6,1,0xe6b38400,3762,0x9f95
,1,0xe6b89700,0,0xf4db,1,0xe79da100,0,0x9087,1,0xe9a7bc00,0,0xfbf7
,1,0xe887be00,9480,0xe46b,1,0xe9a88a00,0,0xfbf8,1,0xe8beaf00,3322,0xe787
,1,0xe9a89100,0,0xfbf9,1,0xe5a28f00,0,0xf24b,1,0xe9a89600,0,0xfbfa
,1,0xe9a89a00,0,0xfbfb,1,0xe9a8b100,0,0xfc40,1,0xe7a39400,0,0xe1f7
,1,0xe8889e00,0,0x9591,1,0xe7adba00,0,0xe29f,1,0xe99f8600,0,0xe8e6
,1,0xe9a8b600,0,0xfc41,1,0xe9a98400,0,0xfc42,1,0xe9a98c00,0,0xfc43
,1,0xe68d8c00,1169,0x8e4a,1,0xe5b2a800,6386,0x915a,1,0xe5b7bb00,6472,0x8aaa
,1,0xe7a98700,0,0xf741,1,0xe7ae9a00,0,0xe2b2,1,0xe9a99800,0,0xfc44
,1,0xe4af8200,0,0xfc45,1,0xe9aaaf00,0,0xfc46,1,0xe4af8a00,0,0xfc47
,1,0xe581aa00,10947,0xf086,1,0xe586bd00,0,0x9977,1,0xe5b89b00,0,0x9be5
,1,0xe69da500,0,0x9788,1,0xe782af00,0,0xe076,1,0xe3818500,562,0x82a3
,1,0xe8999700,0,0xee6c,1,0xe9aab700,0,0xfc48,1,0xe8a3bd00,4896,0x90bb
,1,0xe9b59200,0,0xefda,1,0xe5879d00,3592,0x8bc3,1,0xe4af9200,0,0xfc49
,1,0xe69e8500,0,0x9e66,1,0xc3a50000,41,0x857a,1,0xe6a8ab00,0,0x8a7e
,1,0xe78db500,0,0xe0d8,1,0xeaaeba00,0,0xfc4b,1,0xe8a49d00,4897,0xe5fa
,1,0xe8a9b000,9895,0x8b6c,1,0xe98eba00,0,0xef61,1,0xe9ab8600,0,0xfc4c
,1,0xe9ab9000,0,0xfc4d,1,0xefbdb400,5461,0xb4,1,0xe6a98b00,7319,0x8bb4
,1,0xe6ae9e00,4129,0x9f6d,1,0xe6b3b100,0,0x9f96,1,0xe798bb00,0,0xe191
,1,0xe8aa9000,0,0xf9dc,1,0xd0b70000,0,0x8478,1,0xe9ab9200,0,0xfc4e
,1,0xe5938300,0,0xf196,1,0xefbe9400,1043,0xd4,1,0xe59da900,0,0x9aae
,1,0xe682b300,0,0x9c7b,1,0xe2818900,0,0x84ec,1,0xe9ab9500,0,0xfc4f
,1,0xe4afa800,0,0xfc50,1,0xe883b800,0,0x8bb9,1,0xe9aba000,0,0xfc52
,1,0xe99aa000,0,0x8942,1,0xe99fb300,0,0x89b9,1,0xe59e8900,5299,0x9ab3
,1,0xe5a39c00,1327,0x9ade,1,0xe5a8af00,365,0x8ce2,1,0xe9aba500,0,0xfc53
,1,0xe9aba900,0,0xfc54,1,0xe291a200,4512,0x8742,1,0xe7a9b400,2492,0x8c8a
,1,0xe99b8000,0,0x909d,1,0xe9a09300,0,0x93da,1,0xe9ac8300,0,0xfc55
,1,0xe9aab900,0,0xfc4a,1,0xe6898600,0,0xf2f4,1,0xe5aea200,6336,0x8b71
,1,0xe693ac00,0,0x8b5b,1,0xe7a58100,1548,0x8c56,1,0xe88a8b00,0,0x88f0
,1,0xe9ac8c00,0,0xfc56,1,0xe381b200,213,0x82d0,1,0xe9ac9000,0,0xfc57
,1,0xe9ac9200,0,0xfc58,1,0xe9ac9600,0,0xfc59,1,0xefa8b500,268,0x87ec
,1,0xe5b49500,0,0x9bbb,1,0xe6999f00,0,0x9dee,1,0xe69eb200,0,0xf3cb
,1,0xe7b08700,0,0xe2c6,1,0xe3829200,8254,0x82f0,1,0xe387a500,566,0x839c
,1,0xe89fb700,0,0xe5b9,1,0xe9ac9c00,0,0xfc5a,1,0xefa99500,0,0xed6c
,1,0xe588aa00,0,0x9988,1,0xe58dbd00,0,0x87ef,1,0xe69f9200,0,0xf3ce
,1,0xe6a4a500,1855,0x9ea7,1,0xe6a9b800,7324,0x9eef,1,0xe89b8400,0,0xe575
,1,0xe9808e00,0,0xe7a3,1,0xe8a5aa00,9788,0xe643,1,0xe9acab00,0,0xfc5b
,1,0xe5898a00,0,0x8ded,1,0xe58e9d00,0,0x87f1,1,0xe9acb300,0,0xfc5c
,1,0xe9acbd00,0,0xfc5d,1,0xc3ac0000,0,0x8582,1,0xca840000,1681,0x8646
,1,0xe6b4be00,0,0x9468,1,0xe8a68a00,0,0xe64a,1,0xe4b0a000,0,0xfc5e
,1,0xe9ad8b00,0,0xfc5f,1,0xe9ada300,0,0xfc60,1,0xe5949000,0,0x9382
,1,0xcda10000,4119,0x8671,1,0xe4bebf00,0,0x95d6,1,0xe9ada500,0,0xfc61
,1,0xe6b59e00,0,0xf4d1,1,0xe287a900,4026,0x824c,1,0xe79fbb00,0,0xf6ba
,1,0xe8b19000,0,0xe6b2,1,0xd0be0000,270,0x8480,1,0x7e000000,0,0x81b0
,1,0xe4ba8c00,8051,0x93f1,1,0xe4bf9f00,1106,0x98d6,1,0xe5a4a900,0,0x9356
,1,0xe689b300,0,0xf2fc,1,0xe2888900,48,0x81c4,1,0xe8809200,0,0xe3ce
,1,0xe885a500,0,0xe448,1,0xe88ab800,0,0x8c7c,1,0xe99c8d00,0,0xe8b7
,1,0xe9adab00,0,0xfc62,1,0xe9a6b300,0,0x9279,1,0xe5a58900,0,0x95f2
,1,0xe68a9300,5853,0x9d53,1,0xe9adac00,0,0xfc63,1,0xe694b900,0,0x89fc
,1,0xe7a68e00,8575,0x92f5,1,0xc58d0000,2176,0x859e,1,0xe9adb600,0,0xfc65
,1,0xe382bf00,2663,0x835e,1,0xe9adb700,0,0xfc66,1,0xe9aea600,0,0xfc67
,1,0xe9aeac00,0,0xfc68,1,0xe5b08f00,1443,0x8fac,1,0xe6959900,0,0x8bb3
,1,0xcf820000,28,0x83d7,1,0xe2998200,84,0x8189,1,0xe7b19400,0,0xe2d6
,1,0xe3839f00,8294,0x837e,1,0xe388b200,0,0x878b,1,0xe9aeb100,0,0xfc69
,1,0xeaaebb00,0,0xfc6a,1,0xeaaebc00,0,0xfc6b,1,0xe5b68200,5414,0x9bcf
,1,0xe9aeb200,0,0xfc6c,1,0xe6a09f00,0,0xf3d7,1,0xe785a900,6689,0x94cf
,1,0xe7b78700,0,0xe35d,1,0xe89c9100,0,0xe589,1,0xe8a1a400,0,0xf9a6
,1,0xe8a6b700,9811,0xf9c8,1,0xe5858400,0,0x8c5a,1,0xe9bd9f00,0,0xea92
,1,0xe58faa00,0,0x91fc,1,0xe9aeb800,0,0xfc6d,1,0xe6a69200,0,0xf452
,1,0xe9aebe00,0,0xfc6e,1,0xe6b0b800,4189,0x8969,1,0xe9af8700,0,0xfc6f
,1,0x67000000,755,0x67,1,0xe98ca100,0,0xef56,1,0xe9afb300,0,0xfc70
,1,0xe5908a00,0,0x92dd,1,0xe5959d00,0,0x9a5a,1,0xe4bab900,8943,0xf053
,1,0xe6ac8500,0,0x9f4f,1,0xc3b30000,0,0x8589,1,0xca8b0000,6945,0x85e4
,1,0xe6bbbe00,0,0x9ff6,1,0xe880bf00,0,0xe3d4,1,0xe8b29d00,0,0x8a4c
,1,0xe9af9800,0,0xfc71,1,0xe9af9d00,10981,0xfc72,1,0xe4bb9900,4855,0x90e5
,1,0xe9afa700,0,0xfc73,1,0xe5a5b600,0,0xf268,1,0xe2848300,843,0x818e
,1,0xe9afaa00,0,0xfc74,1,0xe28ea900,2392,0x846d,1,0xe7a6bb00,8581,0xf6ec
,1,0xe8b89000,7894,0xe6f6,1,0xe8bda300,5038,0xe781,1,0xe48fae00,10367,0xf85d
,1,0xe9afab00,0,0xfc75,1,0xe5a69600,3760,0x9764,1,0xe68ba000,1166,0x8b92
,1,0xe5b0bc00,0,0x93f2,1,0xe9afaf00,0,0xfc76,1,0xe2949c00,0,0x84a5
,1,0xe299af00,8022,0x81f2,1,0xe891b800,0,0xf8cf,1,0xe9afae00,0,0xfc77
,1,0xe9a8a000,0,0xfbfc,1,0xe9adb300,0,0xfc64,1,0xe5ac8900,0,0x8af0
,1,0xe9afba00,0,0xfc79,1,0xeaaebf00,0,0xfc7a,1,0xe69bb900,0,0x9182
,1,0xe88d8500,4409,0xe4a7,1,0xe29fa200,11489,0x84d1,1,0xe7b7b400,0,0x97fb
,1,0xe9a98000,9103,0xe97d,1,0xe9ae9300,0,0xe9b6,1,0xe5809e00,10911,0xf07b
,1,0xe585b100,0,0x8ba4,1,0xe6978600,0,0x9dd2,1,0xe69c9900,0,0xf3b9
,1,0xcf890000,1837,0x83d6,1,0x50000000,1628,0x50,1,0xe8988b00,0,0xe55f
,1,0xe7bda700,0,0xe3ac,1,0xe8a2b100,3067,0xe5de,1,0xe9b48600,4930,0xe9ef
,1,0xe5869100,0,0x9968,1,0xe58ba400,0,0x8bce,1,0xe9afb700,0,0xfc7b
,1,0xeaaebe00,0,0xfc7c,1,0xe9b09600,0,0xfc7d,1,0xe6acb200,1921,0x977e
,1,0xe7be8700,0,0xe3b2,1,0xe8a39100,9726,0xf9b3,1,0xe9b09800,0,0xfc7e
,1,0xe8adb700,3160,0x8cec,1,0xe9b09900,0,0xfc80,1,0xe9b09a00,0,0xfc81
,1,0xefbca800,389,0x8267,1,0xe59bbd00,10733,0x8d91,1,0xe9b09d00,0,0xfc82
,1,0xe7929c00,0,0xecb8,1,0xe9b0a200,0,0xfc83,1,0xe9b0a700,0,0xfc84
,1,0xe9b0a900,0,0xfc85,1,0xe8b3aa00,0,0x8ebf,1,0xe8b8bd00,10147,0xeec4
,1,0xefbd8800,0,0x8288,1,0xe9b0aa00,0,0xfc86,1,0xe5a1b000,0,0x9ac9
,1,0xe686ba00,1709,0x9ce9,1,0xc3ba0000,7611,0x858f,1,0xca920000,4299,0x85e7
,1,0xe7a2b500,2408,0xe1f1,1,0xe8b48a00,8611,0xe6d3,1,0xeaaf8000,0,0xfc87
,1,0xe8beb000,0,0x9243,1,0xe4bd8600,0,0x9241,1,0xe5a29000,0,0xf24c
,1,0xe5a7a300,0,0xf270,1,0xe5acb600,0,0x9b6d,1,0xe9b0b100,0,0xfc88
,1,0xe8838c00,7825,0x9477,1,0xe8889f00,0,0x8f4d,1,0xe9b0b600,0,0xfc89
,1,0xe9b0b700,0,0xfc8a,1,0xe4919b00,10455,0xedd8,1,0xe9b18500,0,0xfc8b
,1,0xe5a88300,0,0x88a1,1,0xe5ad9600,1414,0x9875,1,0x39000000,105,0x39
,1,0xe9b19c00,0,0xfc8c,1,0xeaaf8200,0,0xfc8d,1,0xe88e9200,8063,0xedf5
,1,0xe9b18900,0,0xfc8e,1,0xe898b800,0,0xee6b,1,0xe9b18a00,0,0xfc8f
,1,0xeaaf8100,0,0xfc90,1,0xe9b19400,0,0xfc91,1,0xe5b38900,0,0xf0c7
,1,0xe6989300,0,0x88d5,1,0xe69da600,0,0xeb6f,1,0xe6a2b900,0,0x9e8e
,1,0xe3818600,0,0x82a4,1,0xc59b0000,4486,0x85ae,1,0xe38bac00,2764,0x86f1
,1,0xe8a3be00,787,0x909e,1,0xe9b19800,0,0xfc92,1,0xe5879e00,581,0x87d5
,1,0xe58cb100,0,0x99ba,1,0xe5be8f00,0,0xf2b4,1,0xc8b80000,6672,0x866a
,1,0xe9b19b00,0,0xfc93,1,0xe6adbf00,0,0x9f66,1,0xe7bf9400,2754,0xe3c4
,1,0xe8a49e00,4902,0xe5f0,1,0xe8a9b100,9896,0x9862,1,0xe9b19d00,0,0xfc94
,1,0xe58d9100,0,0x94da,1,0xe592a400,0,0x9a42,1,0xefbdb500,1041,0xb5
,1,0xe9b19f00,0,0xfc95,1,0xe78e9600,0,0x8be8,1,0xe793a900,0,0xe143
,1,0xe798bc00,0,0xf684,1,0xe8aa9100,4950,0xe673,1,0xe9b1a900,0,0xfc96
,1,0xe9b1aa00,0,0xfc97,1,0xe5938400,0,0x99fa,1,0xefbe9500,302,0xd5
,1,0xe4bdb300,478,0x89c0,1,0xe682b400,0,0x9caa,1,0xe6b49200,0,0x9fad
,1,0xe7999c00,0,0xe195,1,0xe9b1ab00,0,0xfc98,1,0xe9b1ad00,0,0xfc99
,1,0xe9b1ae00,0,0xfc9a,1,0xe8baaa00,0,0xe758,1,0xe8bfbd00,5063,0x92c7
,1,0x22000000,0,0x22,1,0xe6839400,0,0xead2,1,0xe9b1b000,0,0xfc9b
,1,0xe68dba00,0,0x93e6,1,0xe6bf9800,0,0xe05e,1,0xe291a300,0,0x8743
,1,0xe296b600,7944,0x8241,1,0xe99b8100,265,0x8ae5,1,0xe9a09400,0,0xfbb3
,1,0xe9b1b200,0,0xfc9c,1,0xe9b1b500,0,0xfc9d,1,0xe6898700,0,0x90ee
,1,0xe5aea300,0,0x90e9,1,0xe5b3b600,0,0x9387,1,0xe9b3a600,0,0xfc9f
,1,0xe7aa9500,0,0xe279,1,0xe88f9f00,0,0x9370,1,0xe381b300,4612,0x82d1
,1,0xe9b3b200,0,0xfca0,1,0xe9b48b00,0,0xfca1,1,0xe9b0ad00,7578,0x9568
,1,0xefa8b600,1183,0x884b,1,0xe5b49600,0,0x8a52,1,0xe9b48200,0,0xfca2
,1,0xe5bebc00,1584,0x9c74,1,0xe783bd00,0,0xe082,1,0xe3829300,2288,0x82f1
,1,0xe387a600,0,0x839d,1,0xeaaf8300,0,0xfca3,1,0xe9b49100,0,0xfca4
,1,0xefa99600,0,0xed70,1,0xe9b49700,0,0xfca5,1,0xe69a8000,0,0xf3a8
,1,0xe5bf9c00,367,0x899e,1,0xe6a4a600,0,0x9e9b,1,0xe9b49800,0,0xfca6
,1,0xeaaf8500,0,0xfca7,1,0xc5a20000,524,0x85c5,1,0xe985a200,0,0x907c
,1,0xe9b78000,9349,0xefe0,1,0xe9bc9300,975,0x8cdb,1,0xe4b38400,0,0xfca8
,1,0xe593b100,0,0xf19a,1,0xe6a58600,0,0xf44c,1,0xeaaf8400,0,0xfca9
,1,0xe6afac00,4154,0x9f7b,1,0xe6b4bf00,0,0xebe4,1,0xe8a68b00,3095,0x8ca9
,1,0xe8ab9e00,0,0xe686,1,0xb000000,0,0xb,1,0xe9b4b200,0,0xfcaa
,1,0xe4b39100,0,0xfcab,1,0xe599a400,7398,0x9a81,1,0xe9b58200,0,0xfcac
,1,0xe7908300,0,0x8b85,1,0xe9b58a00,0,0xfcad,1,0xe6bab200,0,0x9fea
,1,0xe79fbc00,0,0xe1e3,1,0xe9b59f00,0,0xfcae,1,0xe9b5a200,0,0xfcaf
,1,0xeaaf8600,0,0xfcb0,1,0xe4ba8d00,8086,0x87a5,1,0xe4bfa000,5712,0x87b8
,1,0xe5a4aa00,1336,0x91be,1,0xe9b5a900,0,0xfcb1,1,0xe79b8900,0,0xf696
,1,0xe9b5ab00,0,0xfcb2,1,0xe885a600,0,0xe449,1,0xe88ab900,0,0x8bda
,1,0xe99c8e00,9608,0xe8be,1,0xeaaf8700,0,0xfcb3,1,0xe9a6b400,0,0x93e9
,1,0xe6858100,0,0xf2df,1,0xe68a9400,0,0x9d57,1,0xe5afb000,0,0x9b8e
,1,0xe9b5b300,0,0xfcb4,1,0xe2939000,650,0x86bf,1,0xe7aba200,2531,0xe28f
,1,0xe29db600,2582,0x869f,1,0xe895bf00,0,0xf8f6,1,0xe3b49e00,10561,0xf4fc
,1,0xe9aca700,0,0xe9a7,1,0xe9b1ba00,0,0xfc9e,1,0xe9b5b600,0,0xfcb5
,1,0xe9b5b700,0,0xfcb6,1,0xe5bab600,6532,0x8f8e,1,0xe7ac8200,0,0xe293
,1,0xe9b5be00,0,0xfcb7,1,0xe383a000,8941,0x8380,1,0xe9b68400,0,0xfcb8
,1,0xe980bc00,0,0x954e,1,0xe9b68d00,0,0xfcb9,1,0xe9b7ad00,0,0xea5c
,1,0xe9b69900,0,0xfcba,1,0xe5bb9600,0,0x9c40,1,0xe9b6a100,0,0xfcbb
,1,0xe6a5b300,0,0x9480,1,0xe9b6bf00,0,0xfcbc,1,0xe89c9200,0,0xe583
,1,0xe9819c00,0,0x91bb,1,0xe986af00,0,0xe7cf,1,0xe5858500,11051,0x8f5b
,1,0xe9bda000,5423,0xea93,1,0xe58fab00,0,0x8ba9,1,0xe594be00,1597,0x91c1
,1,0xe7868a00,0,0x8c46,1,0xe9b6b500,0,0xfcbd,1,0xe790b000,0,0xecab
,1,0xe9b6b900,0,0xfcbe,1,0xe9878f00,9297,0x97ca,1,0xcc810000,2886,0x8679
,1,0xe991b500,0,0xe870,1,0xe5908b00,0,0x8944,1,0xe5959e00,0,0x8847
,1,0xe4baba00,8949,0x906c,1,0xe9b6bd00,0,0xfcbf,1,0xe6b19900,0,0xebcf
,1,0xe796a300,4000,0xe16e,1,0xe6bbbf00,0,0x9fde,1,0xe9b78300,0,0xfcc0
,1,0xe8b29e00,0,0x92e5,1,0xe9b78700,0,0xfcc1,1,0xe99cbb00,0,0xef87
,1,0xe59b9100,5268,0x9a96,1,0xe5a0a400,1297,0x92e7,1,0xe685ae00,0,0x97b6
,1,0xe6b78c00,1981,0x9fc3,1,0xe9b78900,0,0xfcc2,1,0xe28eaa00,0,0x846e
,1,0xe886b300,0,0x9156,1,0xe9b79600,0,0xfcc3,1,0xe8bda400,10261,0xe782
,1,0xe9b79a00,0,0xfcc4,1,0xe9b79f00,0,0xfcc5,1,0xe6868e00,0,0x919e
,1,0xe68ba100,0,0x8a67,1,0xe5b0bd00,1441,0x9073,1,0xe9b7a000,0,0xfcc6
,1,0xe2949d00,401,0x84ba,1,0xe7acaf00,0,0xf764,1,0xe891b900,0,0xe4df
,1,0xe9b7a300,0,0xfcc7,1,0xe9b7b400,0,0xfcc8,1,0xe9adb400,0,0xe9b5
,1,0xe68c8100,0,0x8e9d,1,0xe4b48700,0,0xfcc9,1,0xe696a700,6048,0x9580
,1,0xe69bba00,0,0xeb6b,1,0xe7ad8f00,0,0x94b3,1,0xe29fa300,744,0x84d2
,1,0xe9b88a00,0,0xfcca,1,0xe89cbf00,0,0xe58f,1,0xe9b88200,0,0xfccb
,1,0xe5809f00,3502,0x8ed8,1,0xe9b88d00,0,0xfccc,1,0xe9b89900,0,0xfccd
,1,0xe5bca300,0,0xeab3,1,0xe781a400,0,0xec62,1,0xe7b38200,0,0xe2f1
,1,0xe7b89500,0,0xedb0,1,0xe7bda800,2728,0xe3aa,1,0xe9b89c00,0,0xfcce
,1,0xe9b48700,3106,0x93bc,1,0xe5869200,0,0x9660,1,0xe9b89d00,0,0xfccf
,1,0xe590b800,0,0x8b7a,1,0xe6a28d00,0,0x9e94,1,0xe7879700,0,0xe093
,1,0xe78caa00,0,0x9296,1,0xe7be8800,0,0xe3b1,1,0xe8a39200,0,0xf9b1
,1,0xe8a8a500,0,0xe663,1,0xe9b9bb00,0,0xfcd0,1,0xe58c8500,0,0x95ef
,1,0xeaaf8900,0,0xfcd1,1,0xefbca900,120,0x8268,1,0xeaaebd00,0,0xfc78
,1,0xe6ad9300,0,0x8abd,1,0xe9ba8000,0,0xfcd2,1,0xe6b7b900,1992,0x9fb9
,1,0xe8a98500,0,0xf9d8,1,0xc5b00000,10050,0x85c4,1,0xcc880000,0,0x868d
,1,0xe9ba8500,0,0xfcd3,1,0xefbd8900,0,0x8289,1,0xe9ba9b00,0,0xfcd4
,1,0xe681a800,0,0x8da6,1,0xe6b38600,0,0xebd9,1,0xe6b89900,0,0x9fd0
,1,0xe79da300,0,0x93c2,1,0xe882ad00,0,0xe3eb,1,0xe8b48b00,0,0x8ae4
,1,0xe9baa800,0,0xfcd5,1,0xe8beb100,3321,0x904a,1,0xe4bd8700,0,0x98c8
,1,0xeaaf8a00,0,0xfcd6,1,0xe5a7a400,0,0xf271,1,0xe9babd00,0,0xfcd7
,1,0xe6be8c00,0,0xec4f,1,0xeaaf8b00,0,0xfcd8,1,0xe9bb9f00,0,0xfcd9
,1,0xe88db300,0,0xe4b8,1,0xe8bf9100,3328,0x8bdf,1,0xe9a49b00,0,0xfbd9
,1,0xe9bba700,0,0xfcda,1,0xe9bbae00,0,0xfcdb,1,0xe5ad9700,6327,0x8e9a
,1,0xe5b2aa00,0,0x9888,1,0xe5b7bd00,0,0x9246,1,0xe7a98900,2486,0xe26f
,1,0xe7ae9c00,2518,0xe2b1,1,0xe7b3af00,0,0xe2f7,1,0xe898b900,0,0xf953
,1,0xe9bbbf00,0,0xfcdc,1,0xc7910000,3474,0x84f4,1,0xe581ac00,0,0x98f6
,1,0xe6938100,0,0x9769,1,0xe6989400,483,0x90cc,1,0xe5bdb000,0,0x8fb2
,1,0xe6a2ba00,7211,0x9e92,1,0xe3818700,2572,0x82a5,1,0xe9bc8200,0,0xfcdd
,1,0xd1860000,5,0x8488,1,0xe983b600,0,0xfaae,1,0xe4b5b700,0,0xfcde
,1,0xe9bc8300,0,0xfcdf,1,0xe4acbb00,0,0xfbd3,1,0xe5be9000,607,0x8f99
,1,0xe6a39a00,7214,0x9249,1,0xe9bc9700,0,0xfce0,1,0xe7ba8200,7416,0x8e5b
,1,0xe38c8d00,2375,0x8769,1,0xe9849600,0,0xeeeb,1,0xe8a9b200,0,0x8a59
,1,0xe9bc9900,0,0xfce1,1,0xe58d9200,0,0x91b2,1,0xe592a500,0,0x99f8
,1,0xefbdb600,5533,0xb6,1,0xe9bcaf00,0,0xfce2,1,0xe9bcb700,0,0xfce3
,1,0xe6b3b300,0,0x896a,1,0xe9bcba00,0,0xfce4,1,0xe98a8900,0,0xfae8
,1,0xe98f9c00,0,0xfb61,1,0xe9bcbd00,0,0xfce5,1,0xe9bd8100,0,0xfce6
,1,0xefbe9600,44,0xd6,1,0xe9bd8500,0,0xfce7,1,0xe5a2be00,1111,0x8da4
,1,0xe9bd8600,0,0xfce8,1,0xe9bd9300,0,0xfce9,1,0xe6beb900,2081,0xe057
,1,0xe9bd9500,0,0xfcea,1,0xe9bd9800,0,0xfceb,1,0xcc8f0000,334,0x867c
,1,0xe99fb500,0,0xe8ef,1,0xe4be9400,155,0x87b4,1,0xe5a39e00,3748,0x9ad3
,1,0xeaaf8c00,0,0xfcec,1,0xe68dbb00,0,0x9450,1,0xe9bd9d00,0,0xfced
,1,0xe291a400,122,0x8744,1,0xe296b700,125,0x8240,1,0xe8bb8b00,0,0xe761
,1,0xeaaf8d00,0,0xfcee,1,0xe9bda900,0,0xfcef,1,0xeaaf8e00,0,0xfcf0
,1,0xe6898800,0,0xe7bb,1,0xe68e9b00,257,0x8a7c,1,0xe9bdad00,0,0xfcf1
,1,0xe9bdb000,0,0xfcf2,1,0xe7aa9600,2509,0xe27b,1,0xe7afa900,2568,0xe2bf
,1,0xe381b400,0,0x82d2,1,0xe3b39200,0,0xf4c0,1,0xe9bdb500,0,0xfcf3
,1,0xe9b0ae00,0,0xe9d9,1,0xefa8b700,110,0x884e,1,0xe5b49700,0,0x9bbc
,1,0xe699a100,0,0xeb5c,1,0xe5bebd00,1577,0x8b4a,1,0xeaaf8f00,0,0xfcf4
,1,0xe3829400,4455,0x82f2,1,0xe387a700,0,0x839e,1,0xe89fb900,0,0x8a49
,0,0x0,0,0x0,1,0xc7980000,4414,0x84f8,1,0xe9bbb400,0,0xea80
,1,0xe58dbf00,140,0x8ba8,1,0xe69f9400,1290,0x8f5f,1,0xe7849e00,0,0xec6d
,1,0xe6a9ba00,0,0xf474,1,0xe89b8600,0,0xe576,1,0xe9809000,0,0x9280
,1,0xd18d0000,3082,0x848f,1,0xe8aabf00,3536,0x92b2,1,0xe5898c00,0,0x998f
,1,0xe58e9f00,0,0x8cb4,1,0xe593b200,0,0x934e,0,0x0,0,0x0
,0,0x0,0,0x0,0,0x0,0,0x0,1,0xe794b700,0,0x926a
,1,0xe9868300,0,0xeef5,1,0xe8ab9f00,0,0xeea9,0,0x0,0,0x0
,0,0x0,0,0x0,0,0x0,0,0x0,1,0xefbfa300,7478,0x8150
,1,0xe59eb800,0,0x886c,0,0x0,0,0x0,1,0xe6b5a000,0,0xf4d2
,1,0xe79aaa00,0,0xece3,0,0x0,0,0x0,0,0x0,0,0x0
,0,0x0,0,0x0,1,0xe8bbb800,5029,0x8eb2,1,0xe4ba8e00,8088,0x98b0
,1,0xc2a10000,330,0x8542,1,0xe5a4ab00,0,0x9576,1,0xe5a9be00,6925,0x8896
,1,0xe2888b00,50,0x81b9,1,0xe8809400,0,0xedc3,1,0xe885a700,0,0xedcf
,1,0xe88aba00,0,0xf8a1,1,0xc5be0000,3284,0x85b5,1,0x73000000,0,0x73
,0,0x0,0,0x0,1,0xe6858200,0,0x9cc8,1,0xe5aa9e00,505,0x8898
,1,0xe5afb100,0,0xf0a8,1,0xe694bb00,0,0x8d55,1,0xe2939100,0,0x86c0
,1,0xe7aba300,0,0x8f76,1,0xe29db700,4424,0x86a0,0,0x0,0,0x0
,1,0xe9a79500,0,0x89ed,1,0xe9aca800,0,0xe9a8,0,0x0,0,0x0
,1,0xe5b09100,0,0x8fad,0,0x0,0,0x0,1,0xe5bab700,676,0x8d4e
,0,0x0,0,0x0,1,0xe7b19600,0,0xe2dd,1,0xe383a100,0,0x8381
,0,0x0,0,0x0,0,0x0,0,0x0,0,0x0,0,0x0
,0,0x0,0,0x0,1,0xe5b68400,0,0x9bce,0,0x0,0,0x0
,1,0xc4820000,187,0x85b8,1,0xe6a5b400,0,0x9eba,1,0xe78abe00,0,0xec89
,1,0xe89c9300,0,0xf973,1,0xe9819d00,0,0xfa9d,0,0x0,0,0x0
,1,0xe5858600,0,0x929b,1,0xe9bda100,0,0xea94,1,0xe58fac00,0,0x8fa2
,1,0xe6a18100,0,0x8c85,1,0xe6a69400,0,0x9850,1,0xe6aba700,0,0xebb7
,1,0xe6b0ba00,7454,0xf4b1,0,0x0,0,0x0,1,0xe9879000,0,0xe7d8
,1,0xe8acac00,3156,0x9554,0,0x0,0,0x0,1,0xe5908c00,0,0x93af
,0,0x0,0,0x0,1,0xeaadb100,5244,0xee68,0,0x0,0,0x0
,1,0xe6b19a00,0,0x8998,0,0x0,0,0x0,0,0x0,0,0x0
,1,0x5c000000,2889,0x815f,0,0x0,0,0x0,0,0x0,0,0x0
,0,0x0,0,0x0,1,0xeaae9100,0,0xfa8b,1,0xe6809c00,0,0x97e5
,1,0xe5a5b800,1357,0x9b40,0,0x0,0,0x0,1,0xe6bca000,7682,0x9499
,1,0xe28eab00,399,0x846f,1,0xe7a6bd00,6040,0x8bd7,0,0x0,0,0x0
,1,0xe99d9c00,0,0xe8ce,1,0xe9a2af00,8540,0xe944,0,0x0,0,0x0
,1,0xc2a80000,0,0x814e,0,0x0,0,0x0,1,0xe5b0be00,0,0x94f6
,1,0xe7a28a00,0,0xf6c7,0,0x0,0,0x0,0,0x0,0,0x0
,1,0xe891ba00,0,0x9598,0,0x0,0,0x0,1,0xcc9d0000,0,0x8696
,1,0xe9adb500,0,0xefc2,1,0xe5ac8b00,1409,0x9b68,1,0xe5b19e00,0,0x91ae
,0,0x0,0,0x0,1,0xe69bbb00,0,0xeb56,1,0xe7ad9000,4574,0xe29e
,1,0xe29fa400,0,0x84d3,1,0xe897ad00,0,0xee62,1,0xe9a98200,0,0xe97c
,0,0x0,0,0x0,0,0x0,0,0x0,1,0xe585b300,0,0xf147
,1,0xe5b79100,0,0xf0e6,1,0xe69c9b00,0,0x965d,1,0xe6a1ae00,7177,0xf3e5
,0,0x0,0,0x0,0,0x0,0,0x0,1,0xe7bda900,2727,0xe3ab
,1,0xe982aa00,0,0x8ed7,1,0xe5818000,3508,0x87bf,1,0xe5869300,0,0x9969
,1,0xe58ba600,0,0x99a9,1,0xe590b900,0,0x9081,1,0xe7828500,0,0xec64
,1,0xc4890000,0,0x85dd,1,0x45000000,549,0x45,1,0xe791be00,0,0xe0f5
,1,0xe9838a00,9238,0x8d78,0,0x0,0,0x0,1,0xe98db000,0,0xef5e
,1,0xe58c8600,0,0x99b0,0,0x0,0,0x0,1,0xefbcaa00,133,0x8269
,1,0xe59bbf00,10807,0x9a9c,1,0xe6ad9400,0,0x9f60,1,0xe7929e00,0,0xe0f7
,1,0xe6b7ba00,7608,0x9fc7,1,0xe8a98600,0,0xe668,1,0xe8ae9900,0,0xe6aa
,0,0x0,0,0x0,0,0x0,0,0x0,1,0xefbd8a00,5593,0x828a
,1,0xe59c9f00,0,0x9379,1,0xe681a900,1107,0x89b6,1,0xe686bc00,0,0xeadd
,1,0xe6b89a00,0,0x8f8d,1,0xe28aa500,1804,0x81db,0,0x0,0,0x0
,0,0x0,0,0x0,1,0xe8b99f00,0,0x90d6,1,0xe8beb200,0,0x945f
,1,0xe4bd8800,243,0x87ac,1,0xe6828900,0,0x8ebb,1,0xe5a7a500,0,0x8957
,1,0xe68caf00,0,0x9055,1,0xe6be8d00,0,0xec50,1,0xe8838e00,0,0x91d9
,1,0xe888a100,0,0xf881,0,0x0,0,0x0,1,0xe99f8900,0,0xef92
,1,0xe9a49c00,0,0xfbda,0,0x0,0,0x0,0,0x0,0,0x0
,1,0xc2af0000,407,0x854a,1,0xe5b2ab00,0,0x9bab,1,0xe5b7be00,0,0x8bd0
,0,0x0,0,0x0,1,0xe7ae9d00,8757,0xe2ad,1,0xe893a700,0,0xf8dc
,0,0x0,0,0x0,0,0x0,0,0x0,1,0xcca40000,3472,0x8692
,0,0x0,0,0x0,1,0xe5b38b00,1465,0x988a,1,0x2e000000,3818,0x2e
,1,0xe5bdb100,0,0x8965,1,0xe6a2bb00,0,0xf3f3,1,0xe3818800,618,0x82a6
,1,0xe8999a00,0,0x8b95,1,0xe7beb600,4700,0xe3bd,1,0xe983b700,0,0x8bbd
,1,0xe5828d00,0,0x9654,1,0xe587a000,11089,0x997b,1,0xe58cb300,0,0x99bb
,1,0xe5be9100,0,0x9c6c,0,0x0,0,0x0,1,0xe6a8ae00,0,0x9ed1
,1,0xe78db800,2338,0xe0d7,0,0x0,0,0x0,0,0x0,0,0x0
,1,0xe8a9b300,0,0x8fda,1,0xe5888000,0,0x9381,1,0xe58d9300,0,0x91ec
,1,0xe592a600,0,0xf190,1,0xefbdb700,1048,0xb7,1,0xe6a98e00,0,0xf475
,1,0xe78e9800,0,0xf5da,1,0xe793ab00,0,0xf64c,1,0xe8a58000,0,0xee97
,1,0xe8aa9300,0,0x90be,1,0xe98f9d00,0,0xe857,0,0x0,0,0x0
,1,0xe5938600,0,0x8841,1,0xefbe9700,5978,0xd7,1,0xe4bdb500,0,0x95b9
,1,0xe682b600,0,0x96e3,0,0x0,0,0x0,1,0xe6b9a700,0,0x974e
,0,0x0,0,0x0,1,0xe883bb00,0,0xf85c,0,0x0,0,0x0
,1,0xe99aa300,2985,0x97d7,1,0xe4b98200,3340,0x98a7,0,0x0,0,0x0
,1,0xe5a39f00,0,0x9ae0,1,0xe688a900,0,0xf2f3,1,0xe68dbc00,0,0xeaef
,1,0xe28c9200,7118,0x81dc,1,0xe291a500,2420,0x8745,1,0xe889ae00,0,0x8daf
,1,0xe8bb8c00,0,0x8b4f,1,0xe9a09600,0,0xfbb4,0,0x0,0,0x0
,1,0x17000000,3411,0x17,1,0xe6898900,0,0x94e0,1,0xe5aea500,0,0x9747
,1,0xe693af00,0,0x9dad,0,0x0,0,0x0,1,0xe7aa9700,4545,0xe278
,1,0xe88fa100,8092,0xee41,1,0xe381b500,8233,0x82d3,0,0x0,0,0x0
,1,0xe9ab9c00,0,0xfc51,1,0xe9b0af00,0,0x88f1,1,0xefa8b800,538,0x8855
,1,0xc2b60000,1217,0x81f7,1,0xe699a200,6108,0x9def,0,0x0,0,0x0
,1,0xe8908100,0,0xee43,1,0xe3829500,0,0x82f3,1,0xe387a800,0,0x83ec
,1,0xe38cbb00,0,0x876e,1,0xe9b18f00,0,0xefd0,1,0xe5839a00,4473,0x97bb
,0,0x0,0,0x0,0,0x0,0,0x0,1,0xe5bf9e00,0,0xeac5
,1,0xe6a4a800,0,0x9eab,1,0xe789b200,0,0x90b5,1,0xe89b8700,0,0x8ed6
,1,0xe9809100,0,0xe793,1,0xe8a5ad00,9791,0xe642,1,0xe98ab700,3546,0xe7f7
,1,0xe5898d00,3649,0x914f,1,0xe58ea000,0,0x99cc,1,0xe593b300,0,0xf19b
,0,0x0,0,0x0,1,0xe78a9200,0,0xe0b5,1,0xe78fa500,6841,0xe0e0
,1,0xe794b800,0,0x99b2,0,0x0,0,0x0,1,0xe8aba000,9958,0xe683
,0,0x0,0,0x0,1,0xe58f8000,0,0xf17a,0,0x0,0,0x0
,1,0xe599a600,7527,0x8854,0,0x0,0,0x0,1,0xe7908500,6848,0xe0e4
,0,0x0,0,0x0,1,0xe6bab400,0,0xf542,1,0xe8ac8000,0,0x9664
,1,0xe9918a00,0,0xef68,1,0xefbdbb00,1046,0xbb,1,0xefbe9300,322,0xd3
,1,0xe3819900,696,0x82b7,1,0xce960000,1426,0x83a4,1,0xd0940000,415,0x8444
,1,0xd0960000,0,0x8447,1,0xd09c0000,480,0x844d,1,0xd09d0000,2622,0x844e
,1,0xd09e0000,273,0x844f,1,0xd0a20000,2698,0x8453,1,0xe28ea300,0,0x8467
,1,0xd1820000,0,0x8484,1,0xe2949800,0,0x84a3,1,0xe29faa00,0,0x84d9
,1,0xe29fad00,0,0x84dc,1,0xc7960000,0,0x84f7,1,0xc3960000,1484,0x856c
,1,0xc3b50000,1946,0x858b,1,0xc3b60000,0,0x858c,1,0xc3bb0000,309,0x8590
,1,0xcb870000,2224,0x85af,1,0xc5940000,2228,0x85b7,1,0xc5830000,2389,0x85bf
,1,0xc5980000,0,0x85c2,1,0xc5950000,0,0x85c6,1,0xc49b0000,2010,0x85cc
,1,0xc4b40000,554,0x85da,1,0xc4b50000,2238,0x85e0,1,0xc9960000,0,0x85ec
,1,0xc9b30000,0,0x85ed,1,0xc9b00000,0,0x85f9,1,0xca980000,2416,0x8642
,1,0xcaa10000,0,0x865d
};
static struct {
	long hc_flag;
	long hc_hkey;
	long hc_next;
	long hc_datp;
} gucs4_cell_tbl[AKX_SJIS_HASH_MAX]={
 1,0,1507,0x0,1,1,5133,0x1,1,2,0,0x2
,1,3,1467,0x3,1,4,0,0x4,1,5,2006,0x5
,1,6,3061,0x6,1,7,1880,0x7,1,8,0,0x8
,1,9,6051,0x9,1,10,1177,0xa,1,11,5134,0xb
,1,12,0,0xc,1,13,1881,0xd,1,14,375,0xe
,1,15,1136,0xf,1,16,3062,0x10,1,17,0,0x11
,1,18,1444,0x12,1,19,0,0x13,1,20,5135,0x14
,1,21,0,0x15,1,22,1220,0x16,1,23,0,0x17
,1,24,1259,0x18,1,25,8292,0x19,1,26,402,0x1a
,1,27,0,0x1b,1,28,1882,0x1c,1,29,0,0x1d
,1,30,0,0x1e,1,31,0,0x1f,1,32,0,0x20
,1,33,0,0x21,1,34,0,0x22,1,35,1455,0x23
,1,36,0,0x24,1,37,584,0x25,1,38,6053,0x26
,1,39,0,0x27,1,40,728,0x28,1,41,428,0x29
,1,42,3071,0x2a,1,43,3072,0x2b,1,44,8296,0x2c
,1,45,376,0x2d,1,46,0,0x2e,1,47,0,0x2f
,1,48,1884,0x30,1,49,0,0x31,1,50,3064,0x32
,1,51,629,0x33,1,52,611,0x34,1,53,620,0x35
,1,54,0,0x36,1,55,5136,0x37,1,56,0,0x38
,1,57,0,0x39,1,58,401,0x3a,1,59,526,0x3b
,1,60,377,0x3c,1,61,855,0x3d,1,62,408,0x3e
,1,63,0,0x3f,1,64,0,0x40,1,65,0,0x41
,1,66,3066,0x42,1,67,1279,0x43,1,68,1889,0x44
,1,69,0,0x45,1,70,1887,0x46,1,71,384,0x47
,1,72,3069,0x48,1,73,3070,0x49,1,74,3086,0x4a
,1,75,0,0x4b,1,76,0,0x4c,1,77,6058,0x4d
,1,78,3075,0x4e,1,79,415,0x4f,1,80,447,0x50
,1,81,6060,0x51,1,82,1885,0x52,1,83,0,0x53
,1,84,1474,0x54,1,85,1363,0x55,1,86,0,0x56
,1,87,5137,0x57,1,88,6061,0x58,1,89,0,0x59
,1,90,0,0x5a,1,91,0,0x5b,1,92,0,0x815f
,1,93,0,0x5d,1,94,0,0x5e,1,95,0,0x5f
,1,96,399,0x60,1,97,0,0x61,1,98,5717,0x62
,1,99,3077,0x63,1,100,0,0x64,1,101,414,0x65
,1,102,681,0x66,1,103,0,0x67,1,104,3083,0x68
,1,105,631,0x69,1,106,1161,0x6a,1,107,409,0x6b
,1,108,504,0x6c,1,109,397,0x6d,1,110,0,0x6e
,1,111,0,0x6f,1,112,8311,0x70,1,113,0,0x71
,1,114,0,0x72,1,115,1895,0x73,1,116,8314,0x74
,1,117,0,0x75,1,118,6067,0x76,1,119,6070,0x77
,1,120,437,0x78,1,121,0,0x79,1,122,3101,0x7a
,1,123,1893,0x7b,1,124,0,0x7c,1,125,385,0x7d
,1,126,3100,0x81b0,1,127,0,0x7f,1,953,0,0x83c7
,1,954,1528,0x83c8,1,955,1472,0x83c9,1,956,1338,0x83ca
,1,957,1699,0x83cb,1,958,756,0x83cc,1,959,1894,0x83cd
,1,960,8618,0x83ce,1,961,3093,0x83cf,1,963,1891,0x83d0
,1,964,779,0x83d1,1,965,3097,0x83d2,1,966,1371,0x83d3
,1,967,386,0x83d4,1,968,3103,0x83d5,1,969,1439,0x83d6
,1,962,514,0x83d7,1,1040,1638,0x8440,1,1041,436,0x8441
,1,1042,760,0x8442,1,1043,0,0x8443,1,1044,5226,0x8444
,1,1045,0,0x8445,1,1025,1153,0x8446,1,1046,2056,0x8447
,1,1047,535,0x8448,1,1048,2057,0x8449,1,1049,5139,0x844a
,1,1050,5140,0x844b,1,1051,430,0x844c,1,1052,6285,0x844d
,1,1053,1901,0x844e,1,1054,193,0x844f,1,1055,194,0x8450
,1,162,3115,0x8191,1,163,0,0x8192,1,1056,195,0x8451
,1,165,3110,0x5c,1,1057,196,0x8452,1,167,3113,0x8198
,1,168,3123,0x814e,1,1058,197,0x8453,1,1059,198,0x8454
,1,1060,199,0x8455,1,172,3124,0x81ca,1,1061,200,0x8456
,1,1062,201,0x8457,1,1063,202,0x8458,1,176,1291,0x818b
,1,177,3105,0x817d,1,1064,203,0x8459,1,1065,204,0x845a
,1,180,0,0x814c,1,1066,0,0x845b,1,182,0,0x81f7
,1,1067,205,0x845c,1,1072,206,0x8470,1,1073,207,0x8471
,1,1074,208,0x8472,1,1075,209,0x8473,1,12535,210,0x8492
,1,12536,211,0x8493,1,12537,212,0x8494,1,12538,213,0x8495
,1,160,214,0x8541,1,161,215,0x8542,1,164,217,0x8543
,1,166,218,0x8544,1,169,219,0x8545,1,170,220,0x8546
,1,171,221,0x8547,1,173,222,0x8548,1,174,223,0x8549
,1,175,224,0x854a,1,178,225,0x854b,1,179,226,0x854c
,1,183,227,0x854d,1,184,228,0x854e,1,185,229,0x854f
,1,186,230,0x8550,1,187,231,0x8551,1,188,232,0x8552
,1,189,233,0x8553,1,190,234,0x8554,1,191,235,0x8555
,1,192,236,0x8556,1,193,237,0x8557,1,215,388,0x817e
,1,194,238,0x8558,1,195,239,0x8559,1,196,240,0x855a
,1,197,241,0x855b,1,198,242,0x855c,1,199,243,0x855d
,1,200,244,0x855e,1,201,245,0x855f,1,202,246,0x8560
,1,203,247,0x8561,1,204,249,0x8562,1,205,250,0x8563
,1,206,251,0x8564,1,207,252,0x8565,1,208,253,0x8566
,1,209,254,0x8567,1,210,255,0x8568,1,211,256,0x8569
,1,212,257,0x856a,1,213,258,0x856b,1,214,259,0x856c
,1,216,260,0x856d,1,217,261,0x856e,1,218,262,0x856f
,1,219,263,0x8570,1,220,264,0x8571,1,221,265,0x8572
,1,222,266,0x8573,1,223,267,0x8574,1,224,268,0x8575
,1,225,269,0x8576,1,247,8350,0x8180,1,226,270,0x8577
,1,227,271,0x8578,1,228,272,0x8579,1,229,273,0x857a
,1,230,274,0x857b,1,231,275,0x857c,1,232,276,0x857d
,1,233,277,0x857e,1,234,278,0x8580,1,235,280,0x8581
,1,236,284,0x8582,1,237,290,0x8583,1,238,282,0x8584
,1,239,283,0x8585,1,240,285,0x8586,1,241,291,0x8587
,1,242,297,0x8588,1,243,302,0x8589,1,244,776,0x858a
,1,245,1505,0x858b,1,246,286,0x858c,1,248,292,0x858d
,1,249,289,0x858e,1,250,295,0x858f,1,251,859,0x8590
,1,252,296,0x8591,1,253,279,0x8592,1,254,281,0x8593
,1,255,1471,0x8594,1,256,1196,0x8595,1,274,387,0x8598
,1,257,1902,0x859a,1,275,287,0x859d,1,260,293,0x859f
,1,261,288,0x85aa,1,258,294,0x85b8,1,262,298,0x85ba
,1,268,303,0x85bb,1,280,390,0x85bc,1,282,426,0x85bd
,1,270,3407,0x85be,1,259,3146,0x85c7,1,263,1586,0x85c9
,1,269,648,0x85ca,1,281,301,0x85cb,1,283,304,0x85cc
,1,271,1906,0x85cd,1,273,305,0x85ce,1,264,392,0x85d7
,1,284,1897,0x85d8,1,298,8373,0x8596,1,299,423,0x859b
,1,292,391,0x85d9,1,265,532,0x85dd,1,285,389,0x85de
,1,293,1437,0x85df,1,295,393,0x85fb,1,8048,0,0x8665
,1,8049,2617,0x8666,1,8050,672,0x866f,1,308,0,0x85da
,1,309,1910,0x85e0,1,8051,517,0x8670,1,829,0,0x868e
,1,828,1689,0x8694,1,313,0,0x85b9,1,314,1931,0x85c8
,1,826,1688,0x869b,1,827,0,0x869c,1,317,0,0x85a2
,1,318,3168,0x85ad,1,20015,669,0x87a3,1,20016,1919,0x87a4
,1,321,1915,0x85a1,1,322,0,0x85ac,1,323,0,0x85bf
,1,324,6108,0x85cf,1,20223,7776,0x87a8,1,20320,2708,0x87ab
,1,327,0,0x85c0,1,328,0,0x85d0,1,20310,453,0x87ae
,1,20362,3178,0x87b3,1,331,1169,0x85f8,1,332,5150,0x8599
,1,333,1913,0x859e,1,20789,3170,0x87cc,1,20893,6111,0x87d2
,1,336,1916,0x85c1,1,337,0,0x85d1,1,338,3180,0x864a
,1,339,1918,0x8649,1,340,5151,0x85b7,1,341,0,0x85c6
,1,20958,8381,0x87d5,1,20962,8399,0x87d6,1,344,6113,0x85c2
,1,345,1525,0x85d2,1,346,3187,0x85a3,1,347,0,0x85ae
,1,348,3172,0x85db,1,349,8383,0x85e1,1,350,0,0x85a5
,1,351,3175,0x85b1,1,352,8387,0x85a4,1,353,0,0x85b0
,1,354,0,0x85c5,1,355,3179,0x85d5,1,356,1920,0x85a6
,1,357,1922,0x85b2,1,20974,3183,0x87d7,1,21196,5045,0x87e2
,1,21200,0,0x87e3,1,21660,5056,0x87f6,1,362,3171,0x8597
,1,363,0,0x859c,1,364,0,0x85dc,1,365,8389,0x85e2
,1,366,0,0x85c3,1,367,3176,0x85d3,1,368,0,0x85c4
,1,369,3203,0x85d4,1,21665,6116,0x87fc,1,21673,459,0x8840
,1,43716,7843,0x886b,1,43721,1924,0x8880,1,22948,1923,0x888d
,1,22979,0,0x888e,1,22994,8300,0x888f,1,377,0,0x85a7
,1,378,0,0x85b3,1,379,0,0x85a9,1,380,0,0x85b6
,1,381,0,0x85a8,1,382,8393,0x85b5,1,23005,3192,0x8890
,1,23059,1927,0x8891,1,23075,1347,0x8892,1,23143,1265,0x8893
,1,23149,396,0x8894,1,23159,394,0x8895,1,23166,868,0x8896
,1,23172,731,0x8897,1,23198,1909,0x8898,1,23207,1300,0x8899
,1,23236,1899,0x889a,1,43728,1763,0x889b,1,23321,5154,0x889c
,1,23043,438,0x88a1,1,21696,0,0x88a3,1,23030,3188,0x88a6
,1,23333,0,0x889d,1,22992,0,0x88b7,1,34427,3190,0x88b8
,1,34999,1654,0x88bf,1,403,0,0x8648,1,24245,1932,0x88c1
,1,20197,3200,0x88c8,1,20301,1203,0x88ca,1,22996,6119,0x88cf
,1,23041,433,0x88d0,1,23561,3269,0x88d1,1,30959,416,0x88e9
,1,19968,1562,0x88ea,1,21693,1929,0x88f4,1,23035,3080,0x88f7
,1,23013,3081,0x8957,1,23344,472,0x8964,1,24433,1933,0x8965
,1,35424,0,0x8972,1,32257,6047,0x898f,1,23713,0,0x89aa
,1,24681,3166,0x89b6,1,20309,4974,0x89bd,1,23233,0,0x89c5
,1,31185,0,0x89c8,1,33655,1934,0x89d7,1,34662,576,0x89da
,1,35506,549,0x89db,1,34442,1647,0x89e1,1,23784,456,0x89e3
,1,34558,1936,0x89e9,1,20171,3195,0x89ee,1,35299,1937,0x89f0
,1,34809,3205,0x8a49,1,21683,0,0x8a50,1,23830,8418,0x8a52
,1,35442,1706,0x8a59,1,34521,3208,0x8a5e,1,34510,764,0x8a61
,1,30906,6121,0x8a6d,1,35226,0,0x8a6f,1,35282,814,0x8a70
,1,23398,3214,0x8a77,1,23731,632,0x8a78,1,20896,1940,0x8aa5
,1,23506,6122,0x8aa6,1,21002,9940,0x8aa7,1,23014,544,0x8aad
,1,23436,1942,0x8aae,1,23448,1648,0x8aaf,1,31487,660,0x8ac6
,1,450,1125,0x8643,1,31649,1130,0x8ac7,1,31777,1943,0x8ac8
,1,32233,1944,0x8ac9,1,32566,637,0x8aca,1,35251,546,0x8acf
,1,24012,0,0x8ade,1,20238,1475,0x8aea,1,23305,2896,0x8af0
,1,26820,0,0x8afc,1,23395,8423,0x8b47,1,461,461,0x84ed
,1,462,527,0x84ee,1,35352,1945,0x8b4c,1,464,442,0x84ef
,1,465,8425,0x84f4,1,466,6123,0x84f5,1,39138,1946,0x8b51
,1,468,0,0x84f6,1,20605,1127,0x8b55,1,470,0,0x84f7
,1,34811,0,0x8b61,1,472,3217,0x84f8,1,21931,6124,0x8b69
,1,474,1947,0x84f9,1,35440,1993,0x8b6c,1,476,0,0x84fa
,1,20037,1948,0x8b76,1,31496,0,0x8b88,1,23621,513,0x8b8f
,1,35377,0,0x8b96,1,31142,3223,0x8b9a,1,31478,1950,0x8ba3
,1,20982,3222,0x8ba5,1,23777,1951,0x8bac,1,34126,702,0x8bbc
,1,20208,6125,0x8bc2,1,31563,1650,0x8bd8,1,34943,0,0x8bdc
,1,20061,2658,0x8be3,1,21306,680,0x8be6,1,34398,556,0x8bf1
,1,20598,2550,0x8bf4,1,20018,1953,0x8bf8,1,32368,1473,0x8c4a
,1,21234,1148,0x8c4d,1,34219,0,0x8c4f,1,23431,3225,0x8946
,1,35347,687,0x8c50,1,23433,0,0x88c0,1,32676,1469,0x8c51
,1,32619,1110,0x8c72,1,34903,448,0x8a58,1,34509,671,0x8c75
,1,504,3227,0x84f2,1,505,547,0x84f3,1,34907,0,0x8971
,1,35336,0,0x8c76,1,35427,754,0x8c77,1,509,1652,0x8664
,1,35686,8438,0x8c78,1,21127,1226,0x8c80,1,34880,557,0x8c8c
,1,35363,695,0x8c8d,1,34915,449,0x88df,1,20214,1172,0x8c8f
,1,23244,1207,0x8c99,1,23451,0,0x88b6,1,23452,0,0x8b58
,1,32937,1366,0x8ca8,1,35211,8441,0x8ca9,1,36963,676,0x8cad
,1,32067,1756,0x8cbc,1,35328,0,0x8cbe,1,23458,0,0x8b71
,1,22993,869,0x8cc6,1,23396,673,0x8cc7,1,24049,809,0x8cc8
,1,24235,1954,0x8cc9,1,25144,3249,0x8ccb,1,34996,783,0x8cd1
,1,35463,2061,0x8cd6,1,37431,3240,0x8cd8,1,21577,0,0x8ce0
,1,23087,1180,0x8ce2,1,24481,3537,0x8ce4,1,23470,0,0x8b7b
,1,35486,2067,0x8cea,1,35492,585,0x8ceb,1,35703,6357,0x8cec
,1,20062,3241,0x8cee,1,23475,3236,0x8a51,1,23476,489,0x8983
,1,23380,733,0x8d45,1,23478,3235,0x89c6,1,23389,3196,0x8d46
,1,23439,1955,0x8d47,1,24037,1157,0x8d48,1,24039,1938,0x8d49
,1,24247,0,0x8d4e,1,32177,0,0x8d6a,1,34952,0,0x8c55
,1,32771,1816,0x8d6c,1,32943,708,0x8d6d,1,33618,1145,0x8d72
,1,34892,6129,0x8d73,1,34913,684,0x8d74,1,38477,1956,0x8d7e
,1,23492,0,0x8af1,1,38917,1244,0x8d80,1,40251,1464,0x8d83
,1,25335,1957,0x8d89,1,35946,0,0x8d8b,1,21578,1958,0x8d90
,1,564,0,0x8663,1,565,1124,0x8667,1,566,1376,0x8668
,1,567,5164,0x8669,1,568,3257,0x866a,1,569,0,0x866b
,1,570,1960,0x866c,1,571,0,0x866d,1,572,445,0x866e
,1,23507,3237,0x8bf5,1,33136,1833,0x8d98,1,23130,1364,0x8da5
,1,24680,6145,0x8da6,1,32058,3259,0x8dae,1,20304,1653,0x8db2
,1,23919,790,0x8db5,1,35408,0,0x8dbc,1,23515,0,0x8ab0
,1,35039,0,0x8dbe,1,22971,782,0x8dc8,1,23472,1965,0x8dc9
,1,32048,604,0x8dd7,1,35009,1354,0x8dd9,1,23521,0,0x89c7
,1,32618,1964,0x8ddf,1,20916,2783,0x8de1,1,21682,1961,0x8de7
,1,23822,1966,0x8de8,1,592,1963,0x8652,1,593,1266,0x8658
,1,594,2512,0x8659,1,595,531,0x8644,1,596,3243,0x8657
,1,597,1968,0x865e,1,598,403,0x85ec,1,599,0,0x8645
,1,600,1574,0x864d,1,601,0,0x864f,1,602,1969,0x8662
,1,23519,5165,0x8e40,1,604,8453,0x8650,1,23665,1972,0x8e52
,1,606,3265,0x8651,1,607,1189,0x85f3,1,608,587,0x8647
,1,609,1595,0x85f7,1,34453,849,0x8e5c,1,35715,1512,0x8e5d
,1,612,663,0x8655,1,613,0,0x865b,1,614,0,0x8641
,1,615,6150,0x8661,1,616,1138,0x864b,1,20180,693,0x8e65
,1,21496,0,0x8e69,1,22987,1393,0x8e6e,1,620,871,0x85e8
,1,621,5171,0x85f2,1,622,678,0x85e9,1,623,1973,0x8653
,1,624,736,0x85f9,1,625,1976,0x85e3,1,626,1978,0x85f4
,1,627,410,0x85ed,1,22985,1128,0x8e6f,1,629,796,0x864e
,1,23039,0,0x8e70,1,23376,1979,0x8e71,1,24535,1238,0x8e75
,1,633,737,0x85ea,1,634,8463,0x8660,1,635,738,0x85f1
,1,23388,1655,0x8e79,1,637,1980,0x85ee,1,638,583,0x85e5
,1,35222,1687,0x8e8b,1,35422,872,0x8e8c,1,641,0,0x85fa
,1,642,0,0x85ef,1,643,0,0x85e6,1,644,739,0x8646
,1,35433,0,0x8e8d,1,35430,0,0x8e8e,1,35468,1164,0x8e8f
,1,648,1656,0x85eb,1,649,0,0x864c,1,650,1981,0x8654
,1,651,0,0x85e4,1,652,1982,0x8656,1,653,1657,0x865a
,1,654,1479,0x85f6,1,20107,8465,0x8e96,1,656,8466,0x85f0
,1,657,3293,0x865f,1,658,755,0x85e7,1,23383,3209,0x8e9a
,1,660,5173,0x8640,1,661,1658,0x85fc,1,23546,1118,0x8e9b
,1,23597,1526,0x8bc4,1,664,3285,0x8642,1,30913,0,0x8ea5
,1,31034,6155,0x8ea6,1,35672,697,0x8eaf,1,31482,856,0x8eb1
,1,669,848,0x85f5,1,23437,8436,0x8eb3,1,23241,0,0x8eb9
,1,23460,8442,0x8eba,1,673,3286,0x865d,1,674,1985,0x865c
,1,23455,1986,0x8ec0,1,20594,682,0x8ec3,1,23556,778,0x8ecb
,1,31038,1256,0x8ed0,1,32773,802,0x8ed2,1,34503,1358,0x8ed6
,1,23610,1261,0x8eda,1,23616,0,0x8bc7,1,23490,1949,0x8ee2
,1,20027,2653,0x8ee5,1,21462,0,0x8ee6,1,23432,711,0x8ee7
,1,35088,480,0x8a8c,1,27530,1988,0x8eea,1,33131,1438,0x8eee
,1,23624,3299,0x8bfc,1,21463,0,0x8ef3,1,23551,1243,0x8ef5
,1,23627,0,0x89ae,1,23447,3230,0x8f40,1,23629,5177,0x8e72
,1,23601,1989,0x8f41,1,24030,1992,0x8f42,1,32066,1991,0x8f49
,1,23633,5176,0x8bfb,1,32722,3320,0x8f4b,1,34886,1990,0x8f4f
,1,36468,0,0x8f52,1,20160,1215,0x8f59,1,20303,0,0x8f5a
,1,21313,0,0x8f5c,1,27713,0,0x8f60,1,23487,0,0x8f68
,1,31899,1742,0x8f6c,1,20986,5720,0x8f6f,1,34899,5182,0x8f70
,1,711,0,0x85af,1,712,1121,0x8672,1,31459,3304,0x8f76
,1,24490,1221,0x8f7a,1,23649,0,0x8ec6,1,716,1669,0x8673
,1,27529,6158,0x8f7d,1,28516,1111,0x8f81,1,32020,1532,0x8f83
,1,720,0,0x8674,1,721,5186,0x8675,1,24033,865,0x8f84
,1,20966,5172,0x8f88,1,24246,8764,0x8f8e,1,21280,3317,0x8fa0
,1,21809,1994,0x8fa5,1,22974,3318,0x8fa8,1,728,1995,0x85a0
,1,729,1246,0x85d6,1,23100,1504,0x8fa9,1,731,606,0x85ab
,1,23477,5161,0x8faa,1,733,0,0x85b4,1,734,0,0x8691
,1,23558,5194,0x8fab,1,23567,1999,0x8fac,1,23569,1423,0x8fad
,1,23578,3300,0x8fae,1,24432,2000,0x8fb2,1,26157,1676,0x8fba
,1,741,0,0x8680,1,742,6170,0x8681,1,743,0,0x8682
,1,744,8483,0x8683,1,745,0,0x8684,1,27189,0,0x8fbf
,1,30977,3319,0x8fca,1,31505,4174,0x8fce,1,32057,0,0x8fd0
,1,34121,0,0x8fd4,1,751,0,0x8685,1,752,0,0x8686
,1,34909,0,0x8fd5,1,35059,1983,0x8fd6,1,35359,8318,0x8fd7
,1,35388,6178,0x8fd8,1,35158,0,0x89a6,1,35412,2001,0x8fd9
,1,35443,1416,0x8fda,1,19998,0,0x8fe5,1,22732,1678,0x8feb
,1,23696,0,0x8af2,1,23330,2004,0x8fec,1,33976,0,0x8ff6
,1,35698,3325,0x8ff7,1,35167,2005,0x8bdd,1,33394,3322,0x9046
,1,768,0,0x867b,1,769,8497,0x8679,1,770,0,0x867e
,1,771,3327,0x869d,1,772,0,0x867a,1,35302,2034,0x9047
,1,774,0,0x8676,1,34645,2058,0x9049,1,776,0,0x868d
,1,23611,1536,0x904b,1,23072,3381,0x9050,1,779,420,0x8678
,1,780,3328,0x867d,1,23517,1681,0x9051,1,23529,3333,0x9052
,1,783,0,0x867c,1,31206,6192,0x9060,1,35186,6198,0x8f50
,1,32051,975,0x9061,1,23721,3334,0x8ae2,1,35242,1684,0x9065
,1,35386,2007,0x9066,1,36763,1463,0x9068,1,20995,6200,0x906e
,1,792,0,0x8698,1,793,0,0x8699,1,794,0,0x869e
,1,23563,1139,0x9071,1,796,0,0x868a,1,797,443,0x8696
,1,798,857,0x8697,1,799,6202,0x868b,1,800,1518,0x868c
,1,23613,1952,0x9073,1,23736,3338,0x8add,1,35338,0,0x9075
,1,804,1686,0x8692,1,805,1398,0x8687,1,38920,1284,0x907b
,1,36887,3339,0x9080,1,34928,1490,0x908a,1,809,3342,0x868f
,1,810,521,0x869a,1,36930,0,0x908b,1,812,0,0x8688
,1,23815,0,0x9092,1,35215,0,0x8b4b,1,815,2014,0x8690
,1,816,0,0x8693,1,23913,1109,0x9093,1,33733,3343,0x909b
,1,38935,5195,0x909c,1,820,0,0x8695,1,12288,640,0x8140
,1,12289,1278,0x8141,1,12290,3344,0x8142,1,12291,0,0x8156
,1,825,440,0x8689,1,12293,316,0x8158,1,12294,317,0x8159
,1,12295,313,0x815a,1,12296,312,0x8171,1,12297,8521,0x8172
,1,12298,0,0x8173,1,12299,3346,0x8174,1,12300,0,0x8175
,1,12301,0,0x8176,1,12302,2018,0x8177,1,12303,0,0x8178
,1,12304,3348,0x8179,1,12305,1529,0x817a,1,12306,0,0x81a7
,1,12307,3347,0x81ac,1,12308,789,0x816b,1,12309,1240,0x816c
,1,12310,485,0x81d8,1,12311,0,0x81d9,1,12312,0,0x81d6
,1,12313,3352,0x81d7,1,35070,8537,0x909e,1,23544,3267,0x90a1
,1,12316,3354,0x8160,1,12317,429,0x8780,1,21218,2019,0x90a8
,1,12319,2026,0x8781,1,12320,0,0x83e4,1,22995,8541,0x90a9
,1,35069,2023,0x90bb,1,35199,0,0x90bc,1,35488,1431,0x90bd
,1,35475,1430,0x90be,1,38745,0,0x90c3,1,24109,3356,0x90c8
,1,30889,0,0x90d7,1,20999,1691,0x90d8,1,31680,3360,0x90df
,1,35500,1227,0x90e0,1,865,0,0x8671,1,38634,0,0x90e1
,1,34633,1327,0x90e4,1,23459,2022,0x90e9,1,23803,0,0x8f73
,1,23554,0,0x90ea,1,23574,1539,0x90eb,1,12339,0,0x81b1
,1,12340,0,0x81b2,1,12341,0,0x81b3,1,24029,3404,0x90ec
,1,32680,0,0x9141,1,33146,1846,0x9142,1,33307,0,0x9143
,1,35438,0,0x9146,1,12347,6216,0x81b4,1,12348,441,0x81b5
,1,12349,0,0x81b6,1,36983,5198,0x914a,1,37549,6421,0x914b
,1,32341,2028,0x9155,1,12353,0,0x829f,1,12354,1695,0x82a0
,1,12355,592,0x82a1,1,12356,3361,0x82a2,1,12357,6222,0x82a3
,1,12358,2035,0x82a4,1,12359,3365,0x82a5,1,12360,0,0x82a6
,1,12361,2036,0x82a7,1,12362,2030,0x82a8,1,12363,435,0x82a9
,1,12364,2031,0x82aa,1,12365,432,0x82ab,1,12366,2039,0x82ac
,1,12367,2038,0x82ad,1,12368,774,0x82ae,1,12369,3366,0x82af
,1,12370,6230,0x82b0,1,12371,0,0x82b1,1,12372,2033,0x82b2
,1,12373,6237,0x82b3,1,12374,0,0x82b4,1,12375,2037,0x82b5
,1,12376,0,0x82b6,1,12377,6241,0x82b7,1,12378,0,0x82b8
,1,12379,6242,0x82b9,1,12380,11469,0x82ba,1,12381,11470,0x82bb
,1,12382,11471,0x82bc,1,12383,11472,0x82bd,1,12384,11473,0x82be
,1,12385,11474,0x82bf,1,12386,11475,0x82c0,1,12387,11476,0x82c1
,1,12388,11477,0x82c2,1,12389,11478,0x82c3,1,12390,11479,0x82c4
,1,12391,11480,0x82c5,1,12392,11481,0x82c6,1,12393,11482,0x82c7
,1,12394,11483,0x82c8,1,12395,11484,0x82c9,1,12396,11485,0x82ca
,1,12397,3371,0x82cb,1,12398,11486,0x82cc,1,12399,11487,0x82cd
,1,12400,11488,0x82ce,1,12401,11489,0x82cf,1,12402,11490,0x82d0
,1,12403,11491,0x82d1,1,12404,11492,0x82d2,1,12405,0,0x82d3
,1,12406,3376,0x82d4,1,12407,6259,0x82d5,1,12408,1234,0x82d6
,1,12409,0,0x82d7,1,12410,3374,0x82d8,1,12411,1697,0x82d9
,1,12412,11493,0x82da,1,12413,11494,0x82db,1,12414,11495,0x82dc
,1,12415,11496,0x82dd,1,12416,11497,0x82de,1,12417,11498,0x82df
,1,12418,11499,0x82e0,1,12419,11500,0x82e1,1,12420,129,0x82e2
,1,12421,130,0x82e3,1,12422,131,0x82e4,1,12423,132,0x82e5
,1,12424,133,0x82e6,1,12425,134,0x82e7,1,12426,135,0x82e8
,1,12427,136,0x82e9,1,12428,137,0x82ea,1,12429,145,0x82eb
,1,12430,138,0x82ec,1,12431,139,0x82ed,1,12432,140,0x82ee
,1,12433,141,0x82ef,1,12434,142,0x82f0,1,12435,143,0x82f1
,1,12436,144,0x82f2,1,12437,0,0x82f3,1,12438,6262,0x82f4
,1,35373,0,0x90dd,1,33203,0,0x9156,1,23720,0,0x915a
,1,25514,0,0x915b,1,12443,481,0x814a,1,12444,0,0x814b
,1,12445,1484,0x8154,1,12446,818,0x8155,1,30990,6263,0x9162
,1,31199,3384,0x9164,1,12449,2048,0x8340,1,12450,1701,0x8341
,1,12451,0,0x8342,1,12452,580,0x8343,1,12453,1181,0x8344
,1,12454,757,0x8345,1,12455,0,0x8346,1,12456,2049,0x8347
,1,12457,0,0x8348,1,12458,0,0x8349,1,12459,2050,0x834a
,1,12460,0,0x834b,1,12461,0,0x834c,1,12462,0,0x834d
,1,12463,8622,0x834e,1,12464,3389,0x834f,1,12465,5217,0x8350
,1,12466,3390,0x8351,1,12467,0,0x8352,1,12468,0,0x8353
,1,12469,0,0x8354,1,12470,1702,0x8355,1,12471,2053,0x8356
,1,12472,5219,0x8357,1,12473,2052,0x8358,1,12474,581,0x8359
,1,12475,1134,0x835a,1,12476,2051,0x835b,1,12477,6275,0x835c
,1,12478,759,0x835d,1,12479,1367,0x835e,1,12480,1228,0x835f
,1,12481,2024,0x8360,1,12482,8629,0x8361,1,12483,0,0x8362
,1,12484,0,0x8363,1,12485,2059,0x8364,1,12486,0,0x8365
,1,12487,5224,0x8366,1,12488,641,0x8367,1,12489,6281,0x8368
,1,12490,418,0x8369,1,12491,5225,0x836a,1,12492,152,0x836b
,1,12493,509,0x836c,1,12494,6283,0x836d,1,12495,0,0x836e
,1,12496,647,0x836f,1,12497,0,0x8370,1,12498,2055,0x8371
,1,12499,646,0x8372,1,12500,0,0x8373,1,12501,1645,0x8374
,1,12502,3394,0x8375,1,12503,2054,0x8376,1,12504,880,0x8377
,1,12505,1705,0x8378,1,12506,476,0x8379,1,12507,146,0x837a
,1,12508,147,0x837b,1,12509,148,0x837c,1,12510,149,0x837d
,1,12511,150,0x837e,1,12512,151,0x8380,1,12513,153,0x8381
,1,12514,154,0x8382,1,12515,155,0x8383,1,12516,156,0x8384
,1,12517,157,0x8385,1,12518,158,0x8386,1,12519,159,0x8387
,1,12520,160,0x8388,1,12521,161,0x8389,1,12522,162,0x838a
,1,12523,165,0x838b,1,12524,167,0x838c,1,12525,170,0x838d
,1,12526,171,0x838e,1,12527,172,0x838f,1,12528,174,0x8390
,1,12529,175,0x8391,1,12530,176,0x8392,1,12531,179,0x8393
,1,12532,180,0x8394,1,12533,182,0x8395,1,12534,184,0x8396
,1,1068,189,0x845d,1,1069,190,0x845e,1,1070,191,0x845f
,1,1071,192,0x8460,1,12539,185,0x8145,1,12540,186,0x815b
,1,12541,187,0x8152,1,12542,188,0x8153,1,1076,1162,0x8474
,1,1077,1709,0x8475,1,1078,457,0x8477,1,1079,1503,0x8478
,1,1080,0,0x8479,1,1081,3411,0x847a,1,1082,0,0x847b
,1,1083,6290,0x847c,1,1084,2069,0x847d,1,1085,538,0x847e
,1,1086,0,0x8480,1,1087,858,0x8481,1,1088,2070,0x8482
,1,1089,6298,0x8483,1,1090,3413,0x8484,1,1091,539,0x8485
,1,1092,3408,0x8486,1,1093,2071,0x8487,1,1094,5228,0x8488
,1,1095,876,0x8489,1,1096,698,0x848a,1,1097,0,0x848b
,1,1098,6303,0x848c,1,1099,723,0x848d,1,1100,1242,0x848e
,1,1101,1114,0x848f,1,1102,6307,0x8490,1,1103,548,0x8491
,1,24038,0,0x8db6,1,1105,427,0x8476,1,24040,0,0x8b90
,1,32068,6308,0x9167,1,35380,2016,0x9169,1,23435,2072,0x9176
,1,23652,8652,0x9177,1,25475,0,0x917c,1,24046,1356,0x8db7
,1,24035,8646,0x9183,1,32207,0,0x918d,1,35516,528,0x8b62
,1,32156,2073,0x918e,1,35013,1460,0x9195,1,20596,1197,0x91a4
,1,36275,2074,0x91ab,1,23646,0,0x91ae,1,24055,3416,0x8d4a
,1,32154,0,0x91b1,1,34966,3414,0x91b3,1,23384,3085,0x91b6
,1,24059,0,0x8aaa,1,23403,1163,0x91b7,1,23562,1155,0x91b8
,1,24062,0,0x8bd0,1,20182,0,0x91bc,1,35531,6317,0x90bf
,1,35532,0,0x8ad0,1,24066,3415,0x8e73,1,35409,1380,0x91c0
,1,35535,0,0x907a,1,22949,5229,0x91c3,1,38464,1340,0x91c9
,1,23550,1575,0x91ce,1,23729,0,0x91d0,1,24111,0,0x91d1
,1,32974,0,0x91d9,1,33151,1636,0x91da,1,24076,8683,0x8af3
,1,33492,0,0x91db,1,34955,6139,0x91dc,1,36024,3417,0x91dd
,1,31532,3445,0x91e6,1,23429,1198,0x91ee,1,25299,6324,0x91f1
,1,35351,2042,0x91f5,1,37944,4415,0x91f6,1,20967,3437,0x91fa
,1,34552,1190,0x91fb,1,21481,3439,0x9240,1,24061,1497,0x9246
,1,31466,2078,0x9247,1,35504,1509,0x924e,1,22022,1214,0x9251
,1,22374,3425,0x9252,1,25506,1208,0x9254,1,34507,0,0x9260
,1,35477,6330,0x9261,1,35527,3215,0x926b,1,34584,1903,0x9277
,1,31481,1486,0x927c,1,35566,0,0x8e90,1,31209,0,0x9281
,1,24101,0,0x9083,1,23265,3430,0x9284,1,20013,1767,0x9286
,1,20210,3420,0x9287,1,23449,2684,0x9288,1,24544,0,0x9289
,1,24107,5241,0x8e74,1,27880,3440,0x928d,1,35576,861,0x8f94
,1,34411,6331,0x928e,1,35578,1140,0x8cbf,1,24112,0,0x8b41
,1,34935,5138,0x928f,1,35387,6271,0x9290,1,35582,1191,0x91f8
,1,37198,0,0x9291,1,37619,1448,0x9292,1,35585,2082,0x8979
,1,35586,2083,0x88e0,1,24120,0,0x8fed,1,20939,1235,0x929c
,1,23541,8455,0x929e,1,24086,3395,0x929f,1,24115,3449,0x92a0
,1,24500,1445,0x92a5,1,28526,6332,0x92aa,1,32884,8686,0x92ae
,1,33144,2086,0x92b0,1,34678,1621,0x92b1,1,35519,1281,0x92b2
,1,35548,1249,0x92b3,1,37530,2084,0x92b6,1,25431,1395,0x92bb
,1,27131,0,0x92ce,1,28528,2095,0x92d7,1,23340,3201,0x92da
,1,20302,3436,0x92e1,1,20597,7821,0x92e3,1,21576,3453,0x92e6
,1,23450,1452,0x92e8,1,24093,0,0x92e9,1,35609,2087,0x8caa
,1,24237,3451,0x92eb,1,35611,0,0x8d75,1,35330,8721,0x92f9
,1,35613,0,0x8ed3,1,35558,2080,0x92fa,1,23637,2090,0x9357
,1,32399,1457,0x935a,1,36028,1521,0x935c,1,20253,2088,0x9360
,1,22581,4418,0x9367,1,22956,8291,0x9369,1,23648,3305,0x936a
,1,36077,3463,0x9371,1,36884,0,0x9372,1,20908,3469,0x937e
,1,20941,4420,0x9380,1,23445,2091,0x9386,1,23798,3467,0x9387
,1,23947,1333,0x9388,1,31561,2092,0x9399,1,31572,1404,0x939a
,1,31958,2094,0x939c,1,34153,0,0x93a0,1,34276,0,0x93a1
,1,35342,0,0x93a2,1,35588,2790,0x93a3,1,38957,0,0x93aa
,1,38360,0,0x93ac,1,23566,3278,0x93b1,1,36947,6345,0x93b9
,1,23776,6347,0x93bb,1,35641,6350,0x8bde,1,35501,4413,0x93c7
,1,23626,2672,0x93cd,1,23493,0,0x93d0,1,24178,0,0x8ab1
,1,23663,1405,0x93d4,1,27788,1267,0x93d7,1,34201,2096,0x93e3
,1,35598,2079,0x93e4,1,25466,5725,0x93e6,1,24184,8733,0x8d4b
,1,24185,0,0x8ab2,1,32260,2099,0x93ea,1,24187,5242,0x8cb6
,1,27741,1506,0x93f0,1,23612,1397,0x93f2,1,24190,3482,0x8af4
,1,36049,2100,0x93f6,1,34425,6052,0x93f8,1,24193,3471,0x92a1
,1,23615,3298,0x9441,1,24195,3476,0x8d4c,1,24196,3472,0x8faf
,1,38894,0,0x9442,1,35469,8400,0x9446,1,23527,1349,0x944a
,1,24180,0,0x944e,1,24565,5243,0x944f,1,24202,0,0x8fb0
,1,12736,3483,0x81b7,1,12737,0,0x825b,1,12738,668,0x825c
,1,12739,5244,0x825d,1,24207,3486,0x8f98,1,20035,3485,0x9454
,1,24316,1389,0x9455,1,33075,0,0x945d,1,35223,8750,0x9460
,1,34468,1888,0x9461,1,24213,0,0x92ea,1,24052,1433,0x9462
,1,24215,0,0x9358,1,25773,1694,0x9464,1,35207,2012,0x9465
,1,24218,0,0x8d4d,1,12752,511,0x82f5,1,12753,1385,0x82f6
,1,12754,0,0x82f7,1,12755,5245,0x82f8,1,12756,0,0x82f9
,1,23110,2101,0x946b,1,32629,3494,0x946c,1,20467,0,0x946f
,1,24259,3503,0x9470,1,32954,3495,0x9478,1,35696,6352,0x8b63
,1,24230,2823,0x9378,1,24231,766,0x8dc0,1,22521,0,0x947c
,1,23186,1375,0x947d,1,36064,2196,0x9485,1,12768,529,0x8397
,1,12769,540,0x8398,1,12770,1210,0x8399,1,12771,0,0x839a
,1,12772,0,0x839b,1,12773,0,0x839c,1,12774,0,0x839d
,1,12775,3499,0x839e,1,12776,0,0x83ec,1,12777,8763,0x83ed
,1,12778,405,0x83ee,1,12779,725,0x83ef,1,12780,550,0x83f0
,1,12781,612,0x83f1,1,12782,6358,0x83f2,1,12783,0,0x83f3
,1,12784,0,0x83f4,1,12785,0,0x83f5,1,12786,0,0x83f6
,1,12787,4423,0x83f7,1,12788,2476,0x83f8,1,12789,8765,0x83f9
,1,12790,2102,0x83fa,1,12791,2103,0x83fb,1,12792,1294,0x83fc
,1,34687,3143,0x9488,1,35728,0,0x8f51,1,31204,6360,0x9489
,1,31893,3504,0x9494,1,32283,2104,0x949b,1,33707,1596,0x949c
,1,24161,1365,0x94a6,1,37474,3507,0x94ab,1,20240,0,0x94b0
,1,38309,4774,0x94b4,1,40169,3509,0x94b5,1,34532,2044,0x94b8
,1,21322,2105,0x94bc,1,24070,8782,0x94bf,1,25644,5249,0x94c0
,1,24275,8790,0x8a66,1,29256,0,0x94c5,1,32321,6947,0x94c9
,1,31684,2108,0x94cd,1,30928,0,0x94d6,1,34542,6117,0x94d8
,1,22915,6364,0x94dc,1,24199,2111,0x94dd,1,25209,2112,0x94e1
,1,26000,0,0x94e3,1,31192,2110,0x94e9,1,32203,0,0x94ea
,1,34987,1369,0x94ed,1,24288,0,0x8fb1,1,35513,2114,0x94ee
,1,36991,2113,0x94f0,1,23614,2109,0x94f6,1,24494,1368,0x94f7
,1,21305,2842,0x9543,1,32920,0,0x9549,1,24517,0,0x954b
,1,23019,2115,0x9550,1,23195,1708,0x9551,1,35628,1613,0x9554
,1,34920,1429,0x955c,1,35413,2117,0x955d,1,35961,0,0x955e
,1,24287,1970,0x955f,1,31186,11027,0x9562,1,34541,2032,0x9567
,1,21697,2118,0x9569,1,24428,0,0x956a,1,36007,2119,0x956e
,1,23142,1465,0x9577,1,23500,3251,0x9578,1,24310,0,0x8984
,1,24311,0,0x92ec,1,20904,2122,0x9579,1,24067,4430,0x957a
,1,24314,0,0x8c9a,1,24315,0,0x89f4,1,12849,1276,0x878a
,1,12850,6369,0x878b,1,24220,2124,0x957b,1,24319,0,0x93f9
,1,31526,4431,0x9584,1,33433,1417,0x9587,1,35676,4432,0x9588
,1,36212,2125,0x958b,1,12857,1617,0x878c,1,33310,0,0x9591
,1,23553,0,0x9595,1,24489,4433,0x959c,1,24133,3454,0x959d
,1,33145,2126,0x95a0,1,35079,1409,0x95a1,1,35206,2008,0x95a2
,1,20998,1710,0x95aa,1,21563,1724,0x95ab,1,32862,6371,0x95b7
,1,24335,0,0x8eae,1,24336,0,0x93f3,1,24163,2131,0x95bc
,1,24179,3468,0x95bd,1,24339,0,0x8b7c,1,24340,0,0x92a2
,1,24341,0,0x88f8,1,24330,1491,0x95be,1,24343,0,0x95a4
,1,24344,0,0x8d4f,1,30887,0,0x95c9,1,20559,0,0x95ce
,1,24347,0,0x926f,1,36820,6373,0x95d4,1,23081,4434,0x95d8
,1,24321,6375,0x95d9,1,24351,0,0x92ed,1,20445,2722,0x95db
,1,37610,0,0x95dd,1,25429,0,0x95df,1,27497,4438,0x95e0
,1,35036,8479,0x95e2,1,36628,1482,0x95e3,1,24358,0,0x8cb7
,1,24359,0,0x8cca,1,31298,11096,0x95e4,1,21574,2133,0x95f0
,1,23453,0,0x95f3,1,23792,5144,0x95f4,1,23791,3355,0x95f5
,1,23849,2134,0x95f6,1,24214,2740,0x95f7,1,25447,2140,0x95f9
,1,27873,0,0x9641,1,24369,0,0x8ee3,1,34562,1898,0x9649
,1,35090,5926,0x964a,1,35370,1890,0x964b,1,24373,0,0x92a3
,1,37030,4440,0x964d,1,24375,0,0x8bad,1,37586,2135,0x964e
,1,22952,0,0x9657,1,24125,0,0x9658,1,24536,0,0x9659
,1,24380,0,0x954a,1,35584,4441,0x9664,1,24382,0,0x9265
,1,21187,0,0x9675,1,22528,4442,0x9678,1,24140,2136,0x9679
,1,32763,0,0x967c,1,30952,0,0x9681,1,22969,0,0x9685
,1,26151,4443,0x9686,1,24149,3460,0x968b,1,20451,2723,0x9693
,1,32365,1773,0x969a,1,24051,8653,0x96a4,1,24394,0,0x8bae
,1,31637,0,0x96a5,1,23724,2138,0x96a6,1,23494,0,0x96a7
,1,34588,1912,0x96a8,1,34001,0,0x96aa,1,22937,0,0x96ad
,1,31885,2142,0x96ae,1,21209,0,0x96b1,1,24403,0,0x9396
,1,23167,3131,0x96b9,1,23064,8317,0x96ba,1,20901,2143,0x96bb
,1,23018,2144,0x96c3,1,23391,4445,0x96d0,1,32178,2145,0x96d4
,1,33945,0,0x96d6,1,20786,6014,0x96d7,1,23588,2129,0x96de
,1,32011,2128,0x96e4,1,20063,7729,0x96e7,1,24357,0,0x96ed
,1,32004,0,0x96f1,1,35379,2146,0x96f3,1,24418,0,0x8c60
,1,35565,5738,0x9740,1,20305,4446,0x9743,1,21451,0,0x9746
,1,24422,0,0x9546,1,23461,2015,0x9747,1,24189,2132,0x9748
,1,24425,0,0x8dca,1,24426,0,0x9556,1,24427,0,0x92a4
,1,35895,1373,0x924a,1,24736,2148,0x9749,1,35029,4417,0x9754
,1,12964,3511,0x8785,1,12965,740,0x8786,1,12966,417,0x8787
,1,12967,6387,0x8788,1,12968,2149,0x8789,1,35480,2064,0x9755
,1,34701,1905,0x975a,1,35465,3517,0x975f,1,24188,2150,0x9763
,1,22934,0,0x9764,1,24441,0,0x96f0,1,23481,2076,0x9765
,1,35910,0,0x93a4,1,24444,8800,0x94de,1,24248,3268,0x9766
,1,26332,5255,0x976a,1,35914,2153,0x964c,1,24448,0,0x899d
,1,24449,3522,0x90aa,1,32768,2152,0x9773,1,35201,2010,0x9776
,1,24452,8804,0x8c61,1,24453,3525,0x91d2,1,35617,2182,0x9777
,1,32716,2160,0x9782,1,32764,1811,0x9783,1,32645,4449,0x9785
,1,34746,2155,0x9786,1,35064,1541,0x9787,1,24460,8815,0x8ce3
,1,23888,3377,0x9792,1,35239,0,0x9797,1,35930,6390,0x93d8
,1,24464,0,0x8f99,1,23653,2159,0x979a,1,24466,0,0x936b
,1,24467,0,0x8f5d,1,26446,0,0x979b,1,35023,1679,0x97a0
,1,35937,0,0x8fdb,1,24471,3528,0x93be,1,35041,2021,0x97a1
,1,38626,2163,0x97a3,1,24459,0,0x97a5,1,13008,0,0x86d9
,1,13009,4450,0x86da,1,13010,5257,0x86db,1,13011,2161,0x86dc
,1,13012,563,0x86dd,1,13013,1722,0x86de,1,13014,536,0x86df
,1,13015,6393,0x86e0,1,13016,0,0x86e1,1,13017,6394,0x86e2
,1,13018,0,0x86e3,1,13019,0,0x86e4,1,13020,4452,0x86e5
,1,13021,2167,0x86e6,1,13022,1394,0x86e7,1,13023,715,0x86e8
,1,13024,0,0x86e9,1,13025,0,0x86ea,1,13026,2168,0x86eb
,1,13027,1359,0x86ec,1,31435,3539,0x97a7,1,13029,0,0x86ef
,1,31452,3556,0x97b3,1,40845,0,0x97b4,1,24499,0,0x93bf
,1,13033,1192,0x86ee,1,20406,4457,0x97b5,1,20102,0,0x97b9
,1,13036,3540,0x86f1,1,13037,6395,0x86f0,1,24505,0,0x934f
,1,20940,3543,0x97bd,1,23534,0,0x97be,1,35538,2170,0x97c8
,1,24509,0,0x8b4a,1,21147,3542,0x97cd,1,32209,3545,0x97ce
,1,38563,0,0x97d7,1,35980,0,0x9665,1,20278,3551,0x97e0
,1,24515,3553,0x9053,1,20919,6396,0x97e2,1,13050,1362,0x86ed
,1,21169,0,0x97e3,1,23994,1606,0x97e4,1,24604,8833,0x97e5
,1,31036,3555,0x97e7,1,33491,0,0x97e8,1,38646,6405,0x97eb
,1,24524,0,0x8af5,1,24525,3557,0x9445,1,13059,8859,0x8765
,1,21155,0,0x97f2,1,35010,8868,0x97f4,1,24265,8778,0x97f5
,1,35997,6408,0x8a4c,1,35998,6409,0x92e5,1,31806,6414,0x97fa
,1,36000,0,0x9589,1,36001,2174,0x8de0,1,36002,633,0x8d76
,1,13069,1446,0x8769,1,24537,8871,0x965a,1,21570,7923,0x9843
,1,23105,3157,0x984b,1,24540,1374,0x899e,1,36008,2183,0x89dd
,1,36009,4461,0x94cc,1,13076,3563,0x8760,1,36011,1173,0x8ad1
,1,36012,4462,0x90d3,1,24266,3560,0x984c,1,13080,3567,0x8763
,1,36015,2179,0x9299,1,36016,0,0x96e1,1,24324,0,0x984d
,1,32769,3565,0x9856,1,32894,3566,0x9857,1,36020,4465,0x8b4d
,1,34635,1904,0x9858,1,24555,3573,0x89f5,1,36023,0,0x9483
,1,13090,1146,0x8761,1,13091,6416,0x876b,1,40595,6417,0x985b
,1,36027,2252,0x94ef,1,13094,1217,0x876a,1,13095,3564,0x8764
,1,31108,6419,0x985c,1,36031,0,0x9666,1,36032,1268,0x89ea
,1,13099,3577,0x876c,1,36034,0,0x9847,1,36035,2180,0x92c0
,1,35542,1641,0x985f,1,20525,0,0x9860,1,35441,2176,0x9862
,1,36039,6420,0x8e91,1,24573,3576,0x8d9a,1,36036,2075,0x9864
,1,36042,2185,0x91af,1,39952,0,0x986b,1,13110,0,0x8766
,1,35435,1703,0x986c,1,36046,0,0x9147,1,23361,0,0x9874
,1,23382,3212,0x9875,1,13115,1258,0x876e,1,23421,0,0x9876
,1,36051,0,0x956f,1,23443,3229,0x9877,1,23512,8886,0x9878
,1,23532,8879,0x9879,1,23570,3220,0x987a,1,23582,4466,0x987b
,1,23587,2192,0x987c,1,23595,2199,0x987d,1,36059,2190,0x8e5e
,1,36060,0,0x8e92,1,24594,0,0x937b,1,36062,0,0x8fdc
,1,13129,4468,0x875f,1,13130,1301,0x876d,1,24598,5267,0x957c
,1,36066,6424,0x8cab,1,13133,3580,0x8762,1,14221,2189,0x987e
,1,23650,0,0x9880,1,36070,2195,0x958a,1,13137,1587,0x8767
,1,24605,0,0x8e76,1,43732,2760,0x9883,1,36074,0,0x8ebf
,1,24608,0,0x91d3,1,23674,2187,0x9884,1,13143,1222,0x8768
,1,23695,0,0x9885,1,23711,3332,0x9886,1,24613,8914,0x8b7d
,1,23715,2198,0x9887,1,24615,0,0x90ab,1,24616,0,0x8985
,1,23722,2191,0x9888,1,24618,0,0x89f6,1,23738,2197,0x9889
,1,23755,0,0x988a,1,23760,5276,0x988b,1,23762,0,0x988c
,1,24623,3583,0x8baf,1,23796,3357,0x988d,1,36092,2194,0x8d77
,1,43741,2758,0x988e,1,14306,4471,0x988f,1,23821,8575,0x9890
,1,23847,4472,0x9891,1,23878,8610,0x9893,1,23879,5210,0x9894
,1,23891,1896,0x9895,1,23882,3587,0x9896,1,23917,2201,0x9897
,1,23937,0,0x9898,1,23968,3593,0x9899,1,36104,0,0x91a1
,1,23972,8925,0x989a,1,23975,3590,0x989b,1,36107,0,0x8ae4
,1,23992,2116,0x989c,1,24011,2221,0x989d,1,24332,2216,0x989f
,1,19989,3594,0x98a1,1,20017,3599,0x98a3,1,13179,2212,0x877e
,1,13180,4473,0x878f,1,13181,3600,0x878e,1,13182,3603,0x878d
,1,20022,2210,0x98a4,1,24651,3605,0x97f6,1,20028,6430,0x98a5
,1,20031,2213,0x98a6,1,20034,0,0x98a7,1,35947,2166,0x98ac
,1,24656,5280,0x8bb0,1,24333,0,0x98af,1,24658,0,0x8d50
,1,20127,0,0x98b2,1,20147,0,0x98b6,1,24661,0,0x8f9a
,1,20174,2681,0x98b8,1,20173,2671,0x98b9,1,20190,0,0x98be
,1,13198,2224,0x8772,1,13199,2203,0x8773,1,20205,0,0x98bf
,1,20191,0,0x98c0,1,20215,0,0x98c1,1,20336,0,0x98cf
,1,20436,2209,0x98d5,1,20433,2188,0x98da,1,20442,0,0x98db
,1,24674,0,0x89f8,1,20432,2215,0x98dc,1,20500,2220,0x98e1
,1,24677,0,0x9270,1,20565,0,0x98f1,1,13212,0,0x876f
,1,13213,577,0x8770,1,13214,421,0x8771,1,20560,2206,0x98f2
,1,20566,2223,0x98f5,1,13217,2222,0x8775,1,24685,0,0x8bb1
,1,20588,0,0x98f6,1,24687,0,0x91a7,1,24688,0,0x8a86
,1,20600,3234,0x98f7,1,20608,2738,0x98f8,1,20613,0,0x98fa
,1,20718,0,0x9949,1,24693,0,0x8c62,1,20757,5842,0x9950
,1,20769,2208,0x9953,1,20794,2756,0x9954,1,20791,6048,0x9955
,1,20796,6156,0x9956,1,20795,6134,0x9957,1,31480,0,0x995e
,1,20864,7107,0x9962,1,20866,7114,0x9963,1,20873,6431,0x9966
,1,20900,4945,0x996c,1,20902,2225,0x996d,1,20898,2137,0x996e
,1,20905,2231,0x996f,1,20906,2234,0x9970,1,20907,7727,0x9971
,1,20915,7798,0x9972,1,20913,2779,0x9973,1,20914,2781,0x9974
,1,24713,0,0x8ebb,1,20912,2780,0x9975,1,20955,2244,0x997a
,1,24716,0,0x92ee,1,34389,2227,0x997c,1,20976,8832,0x9980
,1,13252,0,0x8774,1,20990,9553,0x9982,1,21117,0,0x9997
,1,21140,2241,0x9999,1,21138,0,0x999a,1,24724,0,0x89f7
,1,21128,6435,0x999c,1,13259,2236,0x827d,1,21137,2239,0x999d
,1,13261,0,0x8783,1,36196,0,0x90d4,1,36776,2233,0x999e
,1,36198,2235,0x8ecd,1,21164,3606,0x99a0,1,21197,4474,0x99a5
,1,21214,4475,0x99a7,1,24735,0,0x8ce5,1,36203,1496,0x8a71
,1,21219,2822,0x99a8,1,21261,3607,0x99b3,1,24739,0,0x8ab3
,1,21269,0,0x99b6,1,36208,0,0x9196,1,24742,3608,0x8978
,1,21295,2242,0x99b9,1,21297,2158,0x99ba,1,24745,1390,0x9459
,1,24746,0,0x88ab,1,21299,0,0x99bb,1,36215,0,0x8b4e
,1,21304,0,0x99bc,1,21317,0,0x99bf,1,21398,0,0x99cb
,1,31762,6438,0x99d3,1,21471,6440,0x99d5,1,24754,0,0x94df
,1,29166,2184,0x99d7,1,21505,2251,0x99dc,1,21565,2256,0x99dd
,1,24758,3611,0x96e3,1,21549,0,0x99e0,1,21564,2246,0x99e1
,1,21545,8934,0x99e4,1,36229,0,0x92b4,1,21666,5058,0x99f6
,1,24764,0,0x9389,1,21688,2253,0x99f7,1,21669,1959,0x99f8
,1,36234,0,0x897a,1,21676,0,0x99f9,1,21700,0,0x99fa
,1,21672,0,0x99fc,1,21675,0,0x9a40,1,21698,0,0x9a41
,1,24773,0,0x8fee,1,21668,2254,0x9a42,1,24775,0,0x93d5
,1,21694,6444,0x9a43,1,21692,0,0x9a44,1,21836,8935,0x9a52
,1,21679,4478,0x9a5d,1,21884,0,0x9a65,1,13314,0,0x87a1
,1,21978,3612,0x9a6a,1,21988,0,0x9a6f,1,22043,0,0x9a76
,1,24785,5743,0x9866,1,22123,0,0x9a80,1,22227,2248,0x9a97
,1,22271,4479,0x9a9c,1,22280,4480,0x9a9f,1,22283,0,0x9aa0
,1,22310,0,0x9aa7,1,36259,2257,0x8eef,1,22464,6454,0x9aaf
,1,24794,0,0x8d9b,1,22442,6456,0x9ab8,1,24796,0,0x90c9
,1,36264,4482,0x9096,1,22642,0,0x9ac5,1,24799,0,0x88d2
,1,22561,2247,0x9ac6,1,22626,2245,0x9ac7,1,22713,0,0x9acf
,1,24803,0,0x9179,1,22687,5281,0x9ad0,1,22750,0,0x9ad3
,1,22756,0,0x9adf,1,22751,2229,0x9ae0,1,24808,1120,0x8e53
,1,22786,0,0x9ae7,1,22828,0,0x9aed,1,31442,0,0x9af2
,1,22882,0,0x9af8,1,22941,0,0x9b42,1,22947,6458,0x9b45
,1,22962,3621,0x9b46,1,24816,0,0x91c4,1,22982,2274,0x9b47
,1,23016,6462,0x9b48,1,24819,3616,0x917a,1,23004,2266,0x9b49
,1,23001,3492,0x9b4b,1,23002,2261,0x9b4c,1,23077,2262,0x9b4d
,1,23071,4483,0x9b4e,1,24825,5751,0x8ee4,1,23057,2267,0x9b4f
,1,23068,2272,0x9b50,1,23049,3091,0x9b51,1,23066,8945,0x9b52
,1,23104,3402,0x9b53,1,23148,3134,0x9b54,1,23113,2263,0x9b55
,1,24833,3625,0x8f44,1,23093,3114,0x9b56,1,23094,2269,0x9b57
,1,23138,3158,0x9b58,1,23146,3447,0x9b59,1,23194,1911,0x9b5a
,1,23228,3139,0x9b5b,1,24840,0,0x96fa,1,24841,0,0x96f9
,1,23230,3162,0x9b5c,1,23243,3624,0x9b5d,1,23234,2060,0x9b5e
,1,23229,2062,0x9b5f,1,23267,2277,0x9b60,1,24847,3623,0x88d3
,1,23255,3167,0x9b61,1,23270,0,0x9b62,1,36317,6467,0x8b97
,1,23273,6469,0x9b63,1,23254,3632,0x9b64,1,23290,2259,0x9b65
,1,36321,0,0x90d5,1,23291,3184,0x9b66,1,23308,3634,0x9b67
,1,23307,5153,0x9b68,1,24858,0,0x8bf0,1,24859,0,0x88a4
,1,23318,3092,0x9b69,1,36328,0,0x8cd7,1,23346,2900,0x9b6a
,1,24863,3629,0x8ab4,1,23248,3630,0x9b6b,1,23338,2270,0x9b6c
,1,23350,8412,0x9b6d,1,23358,0,0x9b6e,1,36335,0,0x9848
,1,23363,3202,0x9b6f,1,23365,2040,0x9b70,1,23360,2281,0x9b71
,1,36339,2280,0x92b5,1,23377,4897,0x9b72,1,36341,0,0x9148
,1,23381,4485,0x9b73,1,23386,2289,0x9b74,1,23387,5894,0x9b75
,1,23397,0,0x9b76,1,23401,0,0x9b77,1,23408,3221,0x9b78
,1,23411,3635,0x9b79,1,23413,1987,0x9b7a,1,23416,5736,0x9b7b
,1,23418,2290,0x9b7d,1,23424,2013,0x9b7e,1,23427,4960,0x9b80
,1,23462,4487,0x9b81,1,23480,0,0x9b82,1,23491,0,0x9b83
,1,23495,3253,0x9b84,1,23497,8969,0x9b85,1,23508,2288,0x9b86
,1,23504,2291,0x9b87,1,23524,2279,0x9b88,1,36362,2286,0x9778
,1,23526,3266,0x9b89,1,23522,6478,0x9b8a,1,23518,2292,0x9b8b
,1,23525,3263,0x9b8c,1,36367,2294,0x93a5,1,23531,3648,0x9b8d
,1,23536,6479,0x9b8e,1,23542,2278,0x9b8f,1,24904,0,0x8e9c
,1,23539,8454,0x9b90,1,23557,2284,0x9b91,1,24907,8992,0x91d4
,1,24908,0,0x8d51,1,23559,2258,0x9b92,1,24910,0,0x9054
,1,23560,8461,0x9b93,1,23565,3271,0x9b94,1,23571,0,0x9b95
,1,23584,3640,0x9b96,1,23586,2315,0x9b97,1,23592,3649,0x9b98
,1,24917,5288,0x95e7,1,23608,8995,0x9b99,1,23609,3294,0x9b9a
,1,23617,2297,0x9b9b,1,23622,2298,0x9b9c,1,23630,2299,0x9b9d
,1,23635,3302,0x9b9e,1,23632,9009,0x9b9f,1,23631,2314,0x9ba0
,1,23409,8427,0x9ba1,1,23660,2312,0x9ba2,1,23662,5191,0x9ba3
,1,20066,4962,0x9ba4,1,24930,0,0x969d,1,24931,0,0x8ab5
,1,23670,3306,0x9ba5,1,23673,2309,0x9ba6,1,23692,0,0x9ba7
,1,24935,0,0x8c64,1,24936,0,0x8a53,1,23697,3206,0x9ba8
,1,23700,3652,0x9ba9,1,22939,2301,0x9baa,1,23723,3262,0x9bab
,1,23739,2130,0x9bac,1,24942,9043,0x97b6,1,23734,2151,0x9bad
,1,24944,0,0x88d4,1,23740,2310,0x9bae,1,23735,5291,0x9baf
,1,23749,2295,0x9bb0,1,23742,2302,0x9bb1,1,23751,2316,0x9bb2
,1,24950,0,0x8c63,1,23769,2296,0x9bb3,1,23785,7889,0x9bb4
,1,36420,0,0x92fb,1,23805,2164,0x9bb5,1,23802,8298,0x9bb6
,1,23789,3659,0x9bb7,1,23948,3665,0x9bb8,1,24958,3660,0x977c
,1,23786,3658,0x9bb9,1,13493,9063,0x87d1,1,23819,3662,0x9bba
,1,24962,0,0x974a,1,23829,0,0x9bbb,1,23831,0,0x9bbc
,1,23900,2120,0x9bbd,1,23839,0,0x9bbe,1,23835,2319,0x9bbf
,1,23825,3362,0x9bc0,1,23828,9064,0x9bc1,1,23842,2323,0x9bc2
,1,23834,5202,0x9bc3,1,23833,7875,0x9bc4,1,23832,4489,0x9bc5
,1,24974,3667,0x919e,1,23884,3442,0x9bc6,1,24976,0,0x97f7
,1,23890,2106,0x9bc7,1,23886,5853,0x9bc8,1,23883,3373,0x9bc9
,1,36447,2321,0x90d6,1,23916,3670,0x9bca,1,23923,2318,0x9bcb
,1,23926,3385,0x9bcc,1,23943,3388,0x9bcd,1,23940,2317,0x9bce
,1,23938,2322,0x9bcf,1,23970,3392,0x9bd0,1,23965,0,0x9bd1
,1,23980,3125,0x9bd2,1,23982,3117,0x9bd3,1,23997,6092,0x9bd4
,1,23952,3387,0x9bd5,1,23991,2260,0x9bd6,1,23996,3132,0x9bd7
,1,24009,2121,0x9bd8,1,24996,9098,0x95ae,1,24013,3519,0x9bd9
,1,13531,0,0x87d9,1,24999,3682,0x93b2,1,24019,6491,0x9bda
,1,25001,703,0x8c65,1,24018,0,0x9bdb,1,24022,2325,0x9bdc
,1,24027,2320,0x9bdd,1,24043,6492,0x9bde,1,24050,2327,0x9bdf
,1,24053,0,0x9be0,1,24075,5293,0x9be1,1,24090,3683,0x9be2
,1,25010,0,0x8c9b,1,24089,3429,0x9be3,1,24081,2085,0x9be4
,1,24091,0,0x9be5,1,25014,3684,0x89af,1,24118,0,0x9be6
,1,24119,0,0x9be7,1,24132,3688,0x9be8,1,24131,2097,0x9be9
,1,24128,5931,0x9bea,1,24142,3685,0x9beb,1,24151,0,0x9bec
,1,25022,0,0x8ab6,1,24148,2676,0x9bed,1,24159,3691,0x9bee
,1,24162,8726,0x9bef,1,36493,0,0x96f4,1,24164,2334,0x9bf0
,1,24135,0,0x9bf1,1,24181,0,0x9bf2,1,24182,2335,0x9bf3
,1,25031,0,0x8da7,1,24186,2333,0x9bf4,1,24191,2330,0x9bf6
,1,24224,2329,0x9bf7,1,24257,2338,0x9bf8,1,24258,2328,0x9bf9
,1,24264,2339,0x9bfa,1,24272,3699,0x9bfb,1,24271,2324,0x9bfc
,1,25040,0,0x89f9,1,24278,0,0x9c40,1,24291,0,0x9c41
,1,24285,9105,0x9c42,1,24282,0,0x9c43,1,24283,6493,0x9c44
,1,24290,3701,0x9c45,1,24289,0,0x9c46,1,24296,3073,0x9c47
,1,24297,2204,0x9c48,1,24300,0,0x9c49,1,24305,0,0x9c4a
,1,24307,0,0x9c4b,1,24304,3664,0x9c4c,1,24308,2139,0x9c4d
,1,24312,3700,0x9c4e,1,36523,0,0x9067,1,24318,3702,0x9c4f
,1,24323,0,0x9c50,1,24329,2342,0x9c51,1,36527,0,0x8beb
,1,24413,3703,0x9c52,1,24412,2341,0x9c53,1,24331,0,0x9c54
,1,24337,9108,0x9c55,1,24342,0,0x9c56,1,24361,5880,0x9c57
,1,24365,3368,0x9c58,1,24376,0,0x9c59,1,24385,3459,0x9c5a
,1,24392,4907,0x9c5b,1,24396,3335,0x9c5c,1,24398,2275,0x9c5d
,1,24367,0,0x9c5e,1,25074,0,0x92a6,1,24401,3706,0x9c5f
,1,24406,2347,0x9c60,1,24407,5297,0x9c61,1,24409,2345,0x9c62
,1,24417,2331,0x9c63,1,25080,0,0x8c9c,1,24429,0,0x9c64
,1,24435,2346,0x9c65,1,24439,3704,0x9c66,1,24451,2308,0x9c67
,1,24450,2349,0x9c68,1,24447,2351,0x9c69,1,36554,2348,0x8ed4
,1,24458,2353,0x9c6a,1,36556,0,0x8b4f,1,36557,0,0x8c52
,1,24456,3609,0x9c6b,1,24465,5183,0x9c6c,1,24455,0,0x9c6d
,1,24478,8817,0x9c6e,1,36562,0,0x8cac,1,24473,2354,0x9c6f
,1,24472,2355,0x9c70,1,25098,0,0x95e8,1,24480,0,0x9c71
,1,24488,2357,0x9c72,1,24493,2356,0x9c73,1,25102,0,0x8f5e
,1,24508,0,0x9c74,1,25104,3711,0x90ac,1,25105,0,0x89e4
,1,25106,0,0x89fa,1,24534,0,0x9c75,1,36575,2358,0x93ee
,1,24571,6079,0x9c76,1,25110,0,0x88bd,1,36578,0,0x935d
,1,24548,3712,0x9c77,1,24568,0,0x9c78,1,25114,0,0x90ca
,1,24561,2359,0x9c79,1,24541,0,0x9c7a,1,24755,3558,0x9c7b
,1,24575,2361,0x9c7c,1,25119,0,0x8c81,1,24609,3717,0x9c7d
,1,24672,2362,0x9c7e,1,24601,3578,0x9c80,1,24592,6505,0x9c81
,1,24617,5269,0x9c82,1,24590,8893,0x9c83,1,25126,0,0x90ed
,1,24625,0,0x9c84,1,24603,5901,0x9c85,1,24597,5902,0x9c86
,1,24619,2364,0x9c87,1,24614,8917,0x9c88,1,24591,3581,0x9c89
,1,36600,0,0x8eb2,1,24634,2367,0x9c8a,1,25135,5298,0x8b59
,1,24666,2368,0x9c8b,1,24641,3129,0x9c8c,1,36605,2369,0x8c79
,1,24682,2370,0x9c8d,1,25140,0,0x91d5,1,24695,5937,0x9c8e
,1,24671,0,0x9c8f,1,24650,0,0x9c90,1,36611,530,0x8a72
,1,24646,5915,0x9c91,1,24653,2652,0x9c92,1,25147,0,0x96df
,1,24675,3303,0x9c93,1,24643,8927,0x9c94,1,36617,4503,0x8dda
,1,25151,3719,0x965b,1,25152,0,0x8f8a,1,24676,2373,0x9c95
,1,24642,3598,0x9c96,1,24684,4504,0x9c97,1,24683,0,0x9c98
,1,24665,0,0x9c99,1,24705,6507,0x9c9a,1,25159,3723,0x90ee
,1,24717,3727,0x9c9b,1,25161,1424,0x94e0,1,24807,3722,0x9c9c
,1,25163,0,0x8ee8,1,24707,6508,0x9c9d,1,25165,0,0x8dcb
,1,24730,2377,0x9c9e,1,24708,7725,0x9c9f,1,24731,3730,0x9ca0
,1,24726,3728,0x9ca1,1,36637,6510,0x8b50,1,25171,5299,0x91c5
,1,24727,3729,0x9ca2,1,25173,0,0x95a5,1,24722,2806,0x9ca3
,1,24743,5050,0x9ca4,1,25176,0,0x91ef,1,24715,5123,0x9ca5
,1,24801,4506,0x9ca6,1,24760,2382,0x9ca7,1,24800,0,0x9ca8
,1,24787,0,0x9ca9,1,36649,2378,0x9479,1,36650,0,0x97d6
,1,24756,2383,0x9caa,1,24560,6511,0x9cab,1,24765,2899,0x9cac
,1,24774,2381,0x9cad,1,36655,6512,0x8f53,1,24757,0,0x9cae
,1,24792,0,0x9caf,1,24909,8456,0x9cb0,1,24853,2387,0x9cb1
,1,24838,3313,0x9cb2,1,24822,0,0x9cb3,1,24823,3613,0x9cb4
,1,24832,3151,0x9cb5,1,36664,6514,0x9741,1,25198,3738,0x95af
,1,24820,4508,0x9cb6,1,24826,3737,0x9cb7,1,25201,5944,0x88b5
,1,24835,3144,0x9cb8,1,24865,3501,0x9cb9,1,36671,0,0x9760
,1,24827,8944,0x9cba,1,25206,0,0x957d,1,24817,3740,0x9cbb
,1,24845,3497,0x9cbc,1,36676,1350,0x8a8d,1,24846,3739,0x9cbd
,1,24903,3746,0x9cbe,1,24894,2388,0x9cbf,1,24872,0,0x9cc0
,1,24871,2393,0x9cc1,1,25215,9190,0x8fb3,1,25216,0,0x8b5a
,1,24906,3744,0x9cc2,1,36685,2389,0x9351,1,24895,2407,0x9cc3
,1,25220,0,0x8fb4,1,24892,4510,0x9cc4,1,24876,0,0x9cc5
,1,24884,3351,0x9cc6,1,24893,3636,0x9cc7,1,24898,2390,0x9cc8
,1,25226,0,0x9463,1,24900,0,0x9cc9,1,24947,3750,0x9cca
,1,24951,3650,0x9ccb,1,24920,3260,0x9ccc,1,24921,0,0x9ccd
,1,24922,3287,0x9cce,1,25233,3753,0x977d,1,24939,2397,0x9ccf
,1,24948,2400,0x9cd0,1,36703,2408,0x8d8c,1,25237,0,0x938a
,1,36705,2403,0x8c44,1,25239,3754,0x8d52,1,25240,3759,0x90dc
,1,24943,2350,0x9cd1,1,24933,3312,0x9cd2,1,24945,2427,0x9cd3
,1,25244,0,0x94b2,1,24927,3672,0x9cd4,1,25246,0,0x91f0
,1,24925,5179,0x9cd5,1,24915,3645,0x9cd6,1,24949,0,0x9cd7
,1,24985,3669,0x9cd8,1,24982,8621,0x9cd9,1,24967,3326,0x9cda
,1,25004,3410,0x9cdb,1,24980,6518,0x9cdc,1,24986,8623,0x9cdd
,1,24970,3677,0x9cde,1,24977,3379,0x9cdf,1,25003,3405,0x9ce0
,1,25259,0,0x94e2,1,25006,2466,0x9ce1,1,25036,0,0x9ce2
,1,25034,3122,0x9ce3,1,25033,3698,0x9ce4,1,25079,0,0x9ce5
,1,25265,0,0x95f8,1,25032,3696,0x9ce6,1,25027,0,0x9ce7
,1,25030,2337,0x9ce8,1,25269,0,0x92ef,1,25018,3470,0x9ce9
,1,25035,3502,0x9cea,1,25037,3506,0x9cec,1,25273,0,0x9695
,1,25062,3705,0x9ced,1,25059,2413,0x9cee,1,25276,0,0x899f
,1,25277,0,0x928a,1,25078,0,0x9cef,1,25082,3707,0x9cf0
,1,25076,6132,0x9cf1,1,25087,0,0x9cf2,1,25085,2425,0x9cf3
,1,25084,3523,0x9cf4,1,25086,4512,0x9cf5,1,25285,0,0x9253
,1,25088,2419,0x9cf6,1,25096,2426,0x9cf7,1,25097,2421,0x9cf8
,1,25101,2429,0x9cf9,1,25100,2424,0x9cfa,1,25108,0,0x9cfb
,1,25115,2423,0x9cfc,1,25293,0,0x948f,1,25118,0,0x9d40
,1,25121,2415,0x9d41,1,25296,791,0x89fb,1,25130,2411,0x9d42
,1,25298,0,0x8b91,1,36766,1149,0x8eab,1,25134,2406,0x9d43
,1,25136,3716,0x9d44,1,25138,4519,0x9d45,1,25139,2410,0x9d46
,1,25304,3765,0x8d53,1,25305,9202,0x90d9,1,25153,3731,0x9d47
,1,25307,5308,0x8fb5,1,36775,2422,0x999f,1,25309,1797,0x9471
,1,25166,3726,0x9d48,1,25182,0,0x9d49,1,25312,0,0x8b92
,1,25313,0,0x8a67,1,25187,3766,0x9d4a,1,25179,2782,0x9d4b
,1,25184,3767,0x9d4c,1,36784,0,0x9243,1,36785,0,0x904a
,1,36786,0,0x945f,1,25192,3736,0x9d4d,1,25212,3638,0x9d4e
,1,25218,5309,0x9d4f,1,25225,5303,0x9d50,1,25324,3770,0x8a87
,1,25325,0,0x9040,1,25214,2431,0x9d51,1,36794,2435,0x95d3
,1,36795,0,0x92d2,1,36796,2432,0x8d9e,1,25234,9022,0x9d52
,1,25331,0,0x8c9d,1,36799,0,0x9248,1,25235,2437,0x9d53
,1,25334,9225,0x8e41,1,36802,562,0x8949,1,25238,0,0x9d54
,1,36804,0,0x9698,1,36805,0,0x9076,1,25300,3588,0x9d55
,1,25219,3749,0x9d56,1,25236,6521,0x9d57,1,25342,0,0x8f45
,1,25303,2416,0x9d58,1,25297,3762,0x9d59,1,25345,0,0x8e9d
,1,25275,2433,0x9d5a,1,36814,6523,0x8c7d,1,25295,0,0x9d5b
,1,25343,3718,0x9d5c,1,36817,0,0x8bdf,1,25351,0,0x8e77
,1,25286,2434,0x9d5d,1,25353,1415,0x88c2,1,25288,4523,0x9d5f
,1,25308,0,0x9d60,1,25292,2430,0x9d61,1,25290,6524,0x9d62
,1,25282,3708,0x9d63,1,25287,3773,0x9d64,1,25243,6525,0x9d65
,1,25361,0,0x92a7,1,25289,3538,0x9d66,1,25356,6338,0x9d67
,1,25326,3745,0x9d68,1,25329,6012,0x9d69,1,25346,5932,0x9d6b
,1,25352,3526,0x9d6c,1,25327,4525,0x9d6d,1,25369,5310,0x8b93
,1,25333,3776,0x9d6e,1,36838,0,0x89de,1,25423,0,0x9d73
,1,25494,5312,0x9d74,1,36841,0,0x93f4,1,25375,3778,0x8bb2
,1,36843,0,0x9497,1,25472,0,0x9d76,1,36845,0,0x9352
,1,25462,5316,0x9d78,1,25507,3779,0x9d79,1,36848,0,0x8f71
,1,25481,9268,0x9d7b,1,25383,0,0x9d6a,1,25384,0,0x88a5
,1,25542,3782,0x9d84,1,25590,9275,0x9d88,1,25387,0,0x8dc1
,1,36855,0,0x96c0,1,25622,3802,0x9d8a,1,25788,3787,0x9d9e
,1,25391,3788,0x9055,1,25797,9522,0x9da1,1,25799,0,0x9da2
,1,36861,6528,0x92c7,1,25794,0,0x9da5,1,33289,0,0x9da8
,1,36864,6531,0x91de,1,36865,0,0x9197,1,25260,9531,0x9dab
,1,36867,0,0x93a6,1,25853,4533,0x9db4,1,25402,3785,0x92f0
,1,36870,6532,0x8b74,1,25880,6533,0x9db5,1,25405,5323,0x94d2
,1,25406,0,0x9d70,1,25407,0,0x917d,1,25884,3797,0x9db6
,1,35722,6359,0x9dcc,1,26011,3793,0x9dcd,1,26054,3808,0x9dd2
,1,36879,6539,0x93a7,1,36880,0,0x9280,1,26205,3789,0x9dea
,1,26212,0,0x9deb,1,36883,0,0x92fc,1,25417,1223,0x91a8
,1,26215,3791,0x9dec,1,26249,3796,0x9df4,1,25420,808,0x8e4a
,1,25421,0,0x9d71,1,36889,6540,0x9487,1,36890,2439,0x92ca
,1,25424,0,0x9d6f,1,26305,0,0x9df8,1,36893,0,0x90c0
,1,26390,3795,0x9e4c,1,36895,0,0x91ac,1,36896,1421,0x91a2
,1,26805,3792,0x9e90,1,36898,1200,0x88a7,1,36899,0,0x9841
,1,26840,6542,0x9e99,1,26855,0,0x9ea0,1,26839,0,0x9ea5
,1,25436,3798,0x917b,1,26822,0,0x9eb0,1,27001,0,0x9eb1
,1,26996,0,0x9eba,1,27058,0,0x9ec3,1,27088,6046,0x9ec5
,1,27071,5324,0x9ec6,1,36910,0,0x91df,1,27086,6044,0x9eca
,1,23528,4540,0x9ecb,1,36913,0,0x8f54,1,36914,1434,0x9069
,1,25448,0,0x8ecc,1,25449,0,0x9d80,1,27067,3800,0x9ece
,1,25451,3799,0x9d7e,1,27075,6551,0x9ecf,1,36920,0,0x88ed
,1,25454,3801,0x9098,1,27129,0,0x9edd,1,27298,0,0x9efb
,1,36924,0,0x954e,1,25458,0,0x8c9e,1,27287,3812,0x9f40
,1,27330,0,0x9f44,1,27448,0,0x9f56,1,36929,2446,0x93d9
,1,25463,812,0x8fb7,1,27447,9542,0x9f57,1,27463,9550,0x9f5b
,1,36933,1250,0x9278,1,25467,0,0x9450,1,36935,4541,0x8bf6
,1,27489,0,0x9f63,1,27523,3811,0x9f69,1,36938,0,0x9756
,1,36939,2444,0x895e,1,27533,0,0x9f6a,1,36941,0,0x95d5
,1,36942,1112,0x89df,1,27668,3803,0x9f83,1,27684,3804,0x9f85
,1,27683,3805,0x9f86,1,27869,3806,0x9f9b,1,25480,1239,0x8ef6
,1,36948,2449,0x9242,1,36949,0,0x88e1,1,34893,3813,0x9fa5
,1,25484,0,0x8fb6,1,28153,3814,0x9fb9,1,25486,4411,0x9d75
,1,25487,0,0x9d7a,1,28278,0,0x9fd7,1,36956,0,0x91bb
,1,25490,9551,0x9472,1,28191,3815,0x9fd9,1,28303,6556,0x9fed
,1,36960,0,0x8993,1,36961,2440,0x916b,1,28414,0,0x9ff6
,1,25496,522,0x8c40,1,28698,6555,0xe06a,1,36965,6064,0x9779
,1,25499,0,0x8a7c,1,29140,0,0xe098,1,29177,3816,0xe09f
,1,36969,0,0x934b,1,25503,0,0x9d7c,1,25504,0,0x97a9
,1,25505,0,0x8dcc,1,36973,1160,0x9198,1,36974,2447,0x8ed5
,1,29200,3819,0xe0a2,1,25509,0,0x90da,1,40664,0,0xe0d2
,1,25511,3825,0x8d54,1,25512,0,0x9084,1,25513,0,0x8986
,1,36981,976,0x8f85,1,25515,3821,0x9d77,1,25516,884,0x8b64
,1,36984,0,0x9149,1,31014,0,0xe1e6,1,36986,0,0x88e2
,1,31018,0,0xe1e8,1,36988,0,0x97c9,1,25522,3827,0x8c66
,1,30844,0,0xe1ed,1,25524,1357,0x92cd,1,25525,3829,0x9d7d
,1,30883,3828,0xe1f0,1,30901,3826,0xe1f1,1,30890,2642,0xe1f2
,1,36996,0,0x8ad2,1,30895,9554,0xe1f3,1,25531,0,0x917e
,1,30929,3781,0xe1f4,1,30918,5328,0xe1f5,1,25534,3831,0x9d81
,1,30923,3830,0xe1f6,1,25536,0,0x9d83,1,30932,9556,0xe1f7
,1,30910,0,0xe1f8,1,25539,0,0x91b5,1,25540,3833,0x9d89
,1,30908,9557,0xe1f9,1,37009,2452,0x9757,1,30917,0,0xe1fa
,1,30922,0,0xe1fb,1,25545,0,0x9d86,1,30956,5331,0xe1fc
,1,30951,0,0xe240,1,30938,5332,0xe241,1,30964,0,0xe243
,1,30983,5333,0xe244,1,25551,0,0x9560,1,25552,5335,0x92f1
,1,30994,0,0xe245,1,25554,0,0x9d87,1,30993,0,0xe246
,1,31001,6560,0xe247,1,31020,6563,0xe248,1,25558,9559,0x974b
,1,31019,5336,0xe249,1,37027,0,0x93df,1,40779,0,0xe256
,1,25562,0,0x9767,1,25563,1441,0x8ab7,1,31146,6564,0xe257
,1,31150,3835,0xe258,1,31155,0,0xe259,1,37034,0,0x8ed7
,1,31177,6565,0xe25c,1,25569,0,0x88ac,1,31189,0,0xe25d
,1,25571,0,0x9d85,1,31207,3836,0xe25e,1,40861,4544,0xe273
,1,31401,3837,0xe27c,1,36995,0,0xe284,1,31445,5342,0xe289
,1,25577,0,0x9d82,1,31443,3840,0xe28a,1,31449,6566,0xe28b
,1,31457,6567,0xe28e,1,37048,4546,0x9340,1,25582,0,0x8af6
,1,31458,0,0xe28f,1,31503,0,0xe294,1,31498,3720,0xe295
,1,31494,6568,0xe296,1,31512,5343,0xe298,1,25588,0,0x8987
,1,31513,4547,0xe299,1,37057,2453,0x88e8,1,31528,6126,0xe29c
,1,31568,9566,0xe29e,1,31492,5344,0xe2a0,1,25594,5350,0x9768
,1,31565,0,0xe2a1,1,31499,5351,0xe2a2,1,31591,9569,0xe2a8
,1,31600,0,0xe2a9,1,37066,0,0x8d78,1,31601,0,0xe2aa
,1,31596,0,0xe2ab,1,31645,0,0xe2ad,1,37070,0,0x9859
,1,31640,0,0xe2ae,1,31647,5166,0xe2af,1,25606,0,0x9d8c
,1,31629,0,0xe2b0,1,31642,0,0xe2b2,1,31627,3692,0xe2b3
,1,31634,0,0xe2b4,1,31631,0,0xe2b5,1,31581,5352,0xe2b6
,1,25613,0,0x91b9,1,31641,0,0xe2b7,1,25615,0,0x9d93
,1,31691,3845,0xe2b8,1,31681,5239,0xe2b9,1,31692,5356,0xe2ba
,1,25619,5360,0x9d8d,1,31668,4551,0xe2bc,1,31686,0,0xe2bd
,1,37089,2456,0x8c53,1,25623,3841,0x9d91,1,31761,0,0xe2c0
,1,31764,3843,0xe2c1,1,31717,0,0xe2c3,1,31744,0,0xe2c5
,1,25628,0,0x9d72,1,37096,0,0x9594,1,31751,6572,0xe2c6
,1,31763,0,0xe2c7,1,31735,9587,0xe2c9,1,31757,0,0xe2cb
,1,37101,4552,0x8a73,1,31734,0,0xe2cc,1,31783,5362,0xe2ce
,1,31786,6573,0xe2cf,1,25638,0,0x9d8e,1,31799,9579,0xe2d1
,1,25640,0,0x9d92,1,31787,5363,0xe2d2,1,37109,0,0x9758
,1,31808,9578,0xe2d8,1,37111,1341,0x8bbd,1,25645,6140,0x938b
,1,31832,0,0xe2da,1,31839,6577,0xe2db,1,31830,0,0xe2dd
,1,31852,0,0xe2df,1,37117,0,0x9373,1,31906,9588,0xe2e5
,1,25652,9589,0x9d8b,1,31915,9586,0xe2e6,1,25654,6147,0x9d8f
,1,31912,3847,0xe2e8,1,31918,6384,0xe2ec,1,31929,9590,0xe2ed
,1,25658,0,0x8c67,1,31936,9591,0xe2ef,1,31941,0,0xe2f0
,1,31960,0,0xe2f2,1,25662,0,0x8def,1,31964,0,0xe2f4
,1,31983,6151,0xe2f7,1,31986,0,0xe2f8,1,25666,0,0x90db
,1,31988,0,0xe2f9,1,31990,0,0xe2fa,1,32006,0,0xe2fc
,1,32069,0,0xe344,1,32075,3849,0xe345,1,32050,0,0xe347
,1,32063,5365,0xe348,1,32053,5366,0xe349,1,32070,5369,0xe34a
,1,32099,0,0xe352,1,32137,9592,0xe354,1,25678,3852,0x9d97
,1,32125,0,0xe357,1,32155,0,0xe358,1,32174,4555,0xe35a
,1,32189,0,0xe35e,1,32184,9596,0xe364,1,32176,0,0xe366
,1,32221,5828,0xe368,1,32251,6006,0xe36b,1,32225,5372,0xe36d
,1,25688,1668,0x9345,1,32261,8409,0xe36e,1,32266,6166,0xe36f
,1,32274,3582,0xe372,1,32287,5374,0xe374,1,32265,6580,0xe375
,1,32267,6171,0xe376,1,32290,6543,0xe377,1,32358,4560,0xe379
,1,32309,6791,0xe37b,1,37165,0,0x9341,1,32313,6882,0xe37c
,1,32323,5375,0xe37d,1,32311,3855,0xe37e,1,32359,5376,0xe382
,1,25703,3854,0x9d94,1,32349,7565,0xe383,1,25705,5378,0x9680
,1,32350,7603,0xe385,1,32346,7530,0xe387,1,32377,7778,0xe388
,1,32362,4562,0xe389,1,32361,7660,0xe38a,1,25711,9606,0x9d95
,1,32380,7797,0xe38b,1,32379,6434,0xe38c,1,32381,5380,0xe38f
,1,36782,3735,0xe390,1,32383,3255,0xe391,1,32392,7830,0xe392
,1,25718,0,0x9d96,1,32400,7914,0xe396,1,25720,0,0x96cc
,1,32404,4553,0xe398,1,25722,0,0x90a0,1,32406,3461,0xe399
,1,32398,7885,0xe39a,1,32570,5381,0xe39e,1,37193,0,0x93d1
,1,32588,3856,0xe3a0,1,37195,0,0x8f55,1,37196,0,0x8ede
,1,37197,0,0x947a,1,25731,1183,0x8c82,1,32589,5870,0xe3a1
,1,32590,0,0xe3a2,1,32596,0,0xe3a6,1,37202,6583,0x8ef0
,1,25736,0,0x9d9d,1,37204,0,0x908c,1,32600,0,0xe3a7
,1,32607,3860,0xe3a8,1,32616,6179,0xe3aa,1,32617,3861,0xe3ab
,1,32615,9614,0xe3ac,1,32632,0,0xe3ad,1,32647,5382,0xe3b2
,1,32652,6584,0xe3b3,1,25746,0,0x8e54,1,25747,0,0x9d9a
,1,32660,0,0xe3b4,1,25749,0,0x9d99,1,32675,9615,0xe3b8
,1,37218,0,0x907c,1,32687,3862,0xe3b9,1,32690,9616,0xe3ba
,1,25754,3865,0x9451,1,32686,8930,0xe3bc,1,35697,0,0xe3bf
,1,32709,6588,0xe3c0,1,25758,3869,0x93b3,1,37226,6589,0x978f
,1,32710,0,0xe3c1,1,37228,0,0x8f56,1,32714,0,0xe3c2
,1,32737,0,0xe3c5,1,25764,0,0x9350,1,25765,0,0x9d9b
,1,32761,0,0xe3c9,1,39132,3874,0xe3ca,1,32774,3871,0xe3cb
,1,25769,9621,0x9d9c,1,37237,0,0x8d79,1,25771,0,0x958f
,1,37239,0,0x8d93,1,37240,1283,0x8e5f,1,25774,9622,0x8e42
,1,32772,0,0xe3cc,1,25776,0,0x90ef,1,32779,0,0xe3cd
,1,25778,0,0x966f,1,32786,6261,0xe3ce,1,32792,0,0xe3cf
,1,32796,0,0xe3d1,1,32831,0,0xe3d4,1,32827,3882,0xe3d5
,1,32842,5385,0xe3d6,1,25785,0,0x8a68,1,32838,9624,0xe3d7
,1,25787,0,0x9da3,1,37255,2457,0x8f86,1,32872,0,0xe3dd
,1,32883,3880,0xe3de,1,32880,4569,0xe3e0,1,32889,3878,0xe3e2
,1,25793,0,0x9769,1,37261,2462,0x91e7,1,32893,9628,0xe3e3
,1,32902,4576,0xe3e6,1,37264,2459,0x8ced,1,32915,0,0xe3e9
,1,37266,2460,0x90c1,1,32941,6274,0xe3eb,1,32940,0,0xe3ed
,1,32987,4577,0xe3ee,1,32985,6276,0xe3f0,1,37271,6591,0x94ae
,1,25805,0,0x9180,1,32989,4563,0xe3f1,1,32986,0,0xe3f3
,1,32982,4578,0xe3f4,1,37276,0,0x8f58,1,25810,0,0x9da0
,1,33033,5387,0xe3f5,1,25812,0,0x9d5e,1,33009,6592,0xe3f7
,1,33099,0,0xe3fc,1,33094,3884,0xe441,1,25816,0,0x9da4
,1,37284,0,0x8fdd,1,25818,0,0x9d9f,1,33086,0,0xe442
,1,33107,0,0xe443,1,33105,9629,0xe444,1,33137,5928,0xe446
,1,33134,3888,0xe447,1,25824,3885,0x9da9,1,25825,5388,0x9daa
,1,25826,0,0x9346,1,25827,9630,0x9dac,1,33126,3886,0xe449
,1,33140,6597,0xe44a,1,25830,5390,0x8e43,1,25831,0,0x9da7
,1,33155,5300,0xe44b,1,33160,3895,0xe44c,1,33162,3890,0xe44d
,1,33152,9644,0xe44e,1,25836,0,0x8b5b,1,37304,0,0x8ff8
,1,33154,5933,0xe44f,1,25839,3897,0x9dad,1,33187,0,0xe453
,1,25841,5393,0x9da6,1,25842,0,0x9db1,1,33171,0,0xe455
,1,25844,0,0x9db0,1,33200,3898,0xe457,1,25846,3899,0x9daf
,1,33208,4581,0xe45a,1,33213,9649,0xe45b,1,33233,9654,0xe461
,1,25850,6316,0x9db2,1,37318,4583,0x94ce,1,37319,0,0x8dd1
,1,37320,2468,0x8edf,1,25854,3900,0x8fef,1,33241,6598,0xe462
,1,25856,3902,0x9db3,1,37324,6319,0x97a2,1,37325,0,0x8f64
,1,37326,0,0x96ec,1,37327,6599,0x97ca,1,25861,3905,0x9db7
,1,37329,0,0x8be0,1,33224,0,0xe464,1,33242,0,0xe465
,1,33247,0,0xe466,1,33274,0,0xe469,1,33278,3910,0xe46b
,1,33296,5394,0xe472,1,37336,0,0x9342,1,33321,0,0xe474
,1,33369,6601,0xe479,1,33380,3913,0xe47e,1,37340,0,0x8a98
,1,37341,0,0x906a,1,33384,0,0xe481,1,33386,3911,0xe482
,1,33326,0,0xe484,1,33399,3912,0xe486,1,33400,0,0xe487
,1,37347,2471,0x92de,1,33426,5399,0xe48a,1,33439,5400,0xe48c
,1,37350,0,0x9674,1,37351,2475,0x8bfa,1,25885,0,0x9d90
,1,33505,6605,0xe48f,1,33507,0,0xe490,1,33503,0,0xe491
,1,33524,9677,0xe493,1,33530,4584,0xe495,1,25891,9687,0x9db9
,1,25892,0,0x9db8,1,33683,0,0xe496,1,33539,5403,0xe497
,1,33542,6335,0xe49b,1,33497,0,0xe49e,1,33589,4585,0xe49f
,1,25898,3919,0x9d98,1,25899,3920,0x9dba,1,25900,6341,0x9dae
,1,33585,0,0xe4a3,1,33605,0,0xe4a7,1,25903,0,0x8e78
,1,33583,5395,0xe4a8,1,33559,3917,0xe4aa,1,33698,6607,0xe4b0
,1,33678,0,0xe4b3,1,25908,3923,0x9dbb,1,25909,0,0x9dbc
,1,25910,9694,0x9dbe,1,25911,0,0x9dbd,1,25912,0,0x9dbf
,1,25913,0,0x89fc,1,33671,0,0xe4b4,1,25915,0,0x8d55
,1,33742,5406,0xe4c0,1,33795,0,0xe4c2,1,25918,0,0x95fa
,1,25919,9702,0x90ad,1,33752,0,0xe4c3,1,33803,0,0xe4c4
,1,37389,0,0x93dd,1,37390,0,0x8a62,1,33778,6608,0xe4c9
,1,25925,5407,0x8ccc,1,33805,3927,0xe4ca,1,33725,0,0xe4cd
,1,25928,0,0x9dc1,1,33787,3924,0xe4d0,1,33901,3926,0xe4d1
,1,33899,9708,0xe4d7,1,33902,9718,0xe4d9,1,25933,0,0x9dc4
,1,33836,0,0xe4dd,1,25935,0,0x9571,1,33903,0,0xe4de
,1,25937,0,0x8b7e,1,33845,0,0xe4e0,1,33994,3921,0xe4e1
,1,33983,6610,0xe4e4,1,25941,0,0x9dc3,1,25942,0,0x9dc2
,1,25943,0,0x9473,1,25944,0,0x9dc5,1,25945,0,0x8bb3
,1,33951,0,0xe4e5,1,33997,0,0xe4e7,1,34000,0,0xe4ea
,1,25949,0,0x9dc7,1,25950,4038,0x9dc6,1,33985,0,0xe4eb
,1,34006,0,0xe4ed,1,33953,3925,0xe4ee,1,25954,0,0x8ab8
,1,25955,0,0x8e55,1,34047,0,0xe4f0,1,34092,0,0xe4f4
,1,25958,0,0x93d6,1,34079,4594,0xe4f5,1,34069,9722,0xe4f6
,1,37428,0,0x97e9,1,34044,0,0xe4f8,1,34112,6612,0xe4f9
,1,25964,533,0x8c68,1,34306,5408,0xe541,1,34212,5412,0xe545
,1,34204,5414,0xe54f,1,25968,0,0x9094,1,34174,5415,0xe551
,1,25970,0,0x9dc8,1,34192,0,0xe552,1,25972,3938,0x90ae
,1,25973,5417,0x9347,1,34234,0,0xe554,1,25975,9738,0x957e
,1,25976,5418,0x9dc9,1,37444,0,0x9353,1,34233,3932,0xe556
,1,34261,0,0xe558,1,34277,9744,0xe55a,1,34297,3936,0xe55c
,1,34323,3934,0xe55e,1,34338,9746,0xe563,1,34330,3939,0xe564
,1,34352,6615,0xe565,1,25986,0,0x9dca,1,25987,9755,0x9dcb
,1,34367,5419,0xe566,1,34381,3942,0xe567,1,20053,9758,0xe568
,1,25991,0,0x95b6,1,25992,0,0x9b7c,1,25993,0,0x90c4
,1,34407,6620,0xe56b,1,34417,9760,0xe56c,1,25996,3931,0x956b
,1,34451,6054,0xe56d,1,25998,9762,0x8dd6,1,34467,3950,0xe56e
,1,37467,1351,0x8994,1,26001,0,0x94c1,1,34473,6057,0xe56f
,1,34474,3928,0xe570,1,34443,0,0xe571,1,34444,9765,0xe572
,1,34486,9767,0xe573,1,26007,1334,0x936c,1,34479,6059,0xe574
,1,26009,3935,0x97bf,1,34500,9778,0xe575,1,37478,2477,0x8fde
,1,26012,9779,0x8ece,1,34502,3197,0xe576,1,34480,6118,0xe577
,1,26015,0,0x9dce,1,34505,0,0xe578,1,26017,0,0x88b4
,1,34851,0,0xe579,1,34475,0,0xe57a,1,26020,0,0x8bd2
,1,26021,0,0x90cb,1,37489,0,0x8d7a,1,26023,0,0x9580
,1,34516,0,0xe57b,1,34526,4484,0xe57c,1,34537,5213,0xe57d
,1,26027,0,0x9dcf,1,26028,0,0x8e61,1,26029,5429,0x9266
,1,34540,0,0xe57e,1,26031,0,0x8e7a,1,26032,0,0x9056
,1,34527,9800,0xe580,1,34523,6622,0xe581,1,37502,0,0x9667
,1,34543,9813,0xe582,1,37504,6623,0x8be2,1,34578,0,0xe583
,1,26039,0,0x9dd0,1,37507,6625,0x8f65,1,26041,0,0x95fb
,1,37509,0,0x93ba,1,34566,0,0xe584,1,26044,0,0x8997
,1,26045,5432,0x8e7b,1,34568,9816,0xe585,1,34560,6626,0xe586
,1,34563,0,0xe587,1,26049,0,0x9dd3,1,34555,5227,0xe588
,1,26051,0,0x9dd1,1,26052,0,0x9dd4,1,26053,6398,0x97b7
,1,37521,2478,0x914c,1,34577,3488,0xe589,1,34569,3946,0xe58a
,1,34573,6632,0xe58b,1,34553,3937,0xe58c,1,26059,3945,0x90f9
,1,26060,9823,0x9dd5,1,37528,0,0x96c1,1,34570,3135,0xe58d
,1,26063,1199,0x91b0,1,34612,3141,0xe58e,1,34623,3137,0xe58f
,1,26066,0,0x9dd6,1,34615,3163,0xe590,1,34619,3136,0xe591
,1,34597,3715,0xe592,1,34601,3165,0xe593,1,26071,0,0x8af8
,1,34586,5145,0xe594,1,26073,0,0x9dd8,1,34656,3949,0xe595
,1,26075,0,0x9dd7,1,34655,3150,0xe596,1,34680,3145,0xe597
,1,34636,5282,0xe598,1,34638,3169,0xe599,1,26080,9862,0x9dd9
,1,26081,0,0x9dda,1,26082,885,0x8af9,1,34676,8403,0xe59a
,1,34647,3627,0xe59b,1,26085,6428,0x93fa,1,26086,0,0x9255
,1,26087,9880,0x8b8c,1,26088,0,0x8e7c,1,26089,0,0x9181
,1,34664,3398,0xe59c,1,34670,3161,0xe59d,1,26092,3956,0x8f7b
,1,26093,0,0x88ae,1,34649,3292,0xe59e,1,34643,8333,0xe59f
,1,34659,6101,0xe5a0,1,26097,0,0x9ddb,1,34666,3944,0xe5a1
,1,34821,0,0xe5a2,1,34722,3626,0xe5a3,1,34719,9891,0xe5a4
,1,34690,8328,0xe5a5,1,34735,5434,0xe5a6,1,34763,6115,0xe5a7
,1,34749,6636,0xe5a8,1,26106,0,0x89a0,1,26107,9900,0x9ddf
,1,34752,6114,0xe5a9,1,34768,0,0xe5aa,1,38614,0,0xe5ab
,1,34731,6110,0xe5ac,1,34756,4599,0xe5ad,1,34739,5437,0xe5ae
,1,26114,0,0x8d56,1,26115,9903,0x9dde,1,34759,3953,0xe5af
,1,34758,9906,0xe5b0,1,26118,0,0x8da9,1,26119,1443,0x8fb8
,1,34747,5441,0xe5b1,1,34799,4601,0xe5b2,1,26122,0,0x9ddd
,1,34802,0,0xe5b3,1,26124,9908,0x8fb9,1,34784,8301,0xe5b4
,1,26126,9910,0x96be,1,26127,0,0x8da8,1,34831,0,0xe5b5
,1,34829,3250,0xe5b6,1,34814,4595,0xe5b7,1,26131,0,0x88d5
,1,26132,0,0x90cc,1,34806,4602,0xe5b8,1,34807,0,0xe5b9
,1,34830,3391,0xe5ba,1,34770,0,0xe5bb,1,37604,0,0x8f9b
,1,34833,9911,0xe5bc,1,34838,3655,0xe5bd,1,26140,9913,0x9de4
,1,34837,0,0xe5be,1,34850,3288,0xe5bf,1,26143,1420,0x90af
,1,26144,0,0x8966,1,34849,6641,0xe5c0,1,37613,4607,0x8973
,1,34865,0,0xe5c1,1,34870,3418,0xe5c2,1,26149,0,0x8f74
,1,34873,0,0xe5c3,1,37618,1456,0x9565,1,26152,1184,0x8df0
,1,34855,3280,0xe5c4,1,34875,4486,0xe5c5,1,34884,4612,0xe5c6
,1,34882,9919,0xe5c7,1,37624,741,0x8b98,1,34898,9920,0xe5c8
,1,26159,3964,0x90a5,1,34905,4613,0xe5c9,1,37628,4614,0x8d7c
,1,34910,0,0xe5ca,1,34914,4616,0xe5cb,1,26164,5443,0x9de3
,1,26165,0,0x9de1,1,26166,0,0x9de2,1,34923,9924,0xe5cc
,1,34945,0,0xe5cd,1,34942,5444,0xe5ce,1,34974,4605,0xe5cf
,1,37638,0,0x8e4b,1,26172,0,0x928b,1,34933,0,0xe5d0
,1,34941,0,0xe5d1,1,26175,0,0x9e45,1,34997,0,0xe5d2
,1,26177,0,0x9de8,1,26178,9926,0x8e9e,1,26179,0,0x8d57
,1,26180,3963,0x9de6,1,37648,6645,0x908d,1,34930,9915,0xe5d3
,1,34946,8422,0xe5d4,1,34967,4620,0xe5d5,1,26185,0,0x9de7
,1,34962,6473,0xe5d6,1,26187,0,0x9057,1,34990,4621,0xe5d7
,1,37656,0,0x908e,1,34969,3968,0xe5d8,1,26191,3970,0x9de5
,1,34978,0,0xe5d9,1,34957,3296,0xe5da,1,26194,9928,0x8e4e
,1,34980,9931,0xe5db,1,34992,5445,0xe5dc,1,37664,0,0x8ff9
,1,35007,5446,0xe5dd,1,34993,3969,0xe5de,1,35011,3971,0xe5df
,1,35012,3520,0xe5e0,1,35028,4631,0xe5e1,1,37670,6652,0x8bd1
,1,35032,3337,0xe5e2,1,37672,2481,0x9564,1,26206,9938,0x9de9
,1,26207,0,0x9dee,1,37675,0,0x8ee0,1,37676,4632,0x9842
,1,26210,0,0x9def,1,35033,3967,0xe5e3,1,37679,2482,0x8df6
,1,35037,4633,0xe5e4,1,26214,0,0x8a41,1,37682,2485,0x985e
,1,26216,0,0x9ded,1,26217,0,0x94d3,1,35065,3977,0xe5e5
,1,35074,6658,0xe5e6,1,35068,5178,0xe5e7,1,35060,3957,0xe5e8
,1,26222,0,0x9581,1,26223,3976,0x8c69,1,26224,3978,0x9df0
,1,35048,3622,0xe5e9,1,35058,0,0xe5ea,1,35076,4635,0xe5eb
,1,26228,0,0x90b0,1,35084,5290,0xe5ec,1,26230,0,0x8fbb
,1,35082,4641,0xe5ed,1,35091,4643,0xe5ee,1,35139,3966,0xe5ef
,1,26234,0,0x9271,1,35102,6661,0xe5f0,1,35109,9944,0xe5f1
,1,35114,5452,0xe5f2,1,35115,5455,0xe5f3,1,35137,5456,0xe5f4
,1,37707,6662,0x93e7,1,26241,0,0x8bc5,1,37709,0,0x9374
,1,26243,0,0x9df1,1,26244,0,0x9df5,1,35140,3654,0xe5f5
,1,35131,3401,0xe5f6,1,26247,0,0x89c9,1,26248,0,0x9df2
,1,37716,2486,0x92d5,1,35126,0,0xe5f7,1,35128,3990,0xe5f8
,1,35148,9957,0xe5f9,1,35101,4644,0xe5fa,1,26254,0,0x9df3
,1,35168,9959,0xe5fb,1,37723,6663,0x9262,1,26257,3979,0x8f8b
,1,35166,6664,0xe5fc,1,35174,5848,0xe640,1,35172,0,0xe641
,1,35181,3983,0xe642,1,26262,0,0x9267,1,26263,0,0x88c3
,1,26264,0,0x9df6,1,35178,5457,0xe643,1,35183,5458,0xe644
,1,35188,0,0xe645,1,35191,5974,0xe646,1,26269,0,0x9df7
,1,35198,9935,0xe647,1,35203,5459,0xe648,1,35208,4648,0xe649
,1,37740,0,0x8c4c,1,26274,0,0x92a8,1,35210,3988,0xe64a
,1,35219,0,0xe64b,1,35224,5461,0xe64c,1,26278,9966,0x97ef
,1,35233,0,0xe64d,1,35241,9971,0xe64e,1,35238,0,0xe64f
,1,37749,0,0x8cae,1,26283,0,0x8e62,1,35244,3639,0xe650
,1,35247,6670,0xe651,1,26286,0,0x95e9,1,35250,0,0xe652
,1,35258,0,0xe653,1,35261,3986,0xe654,1,35263,3585,0xe655
,1,37758,0,0x8fdf,1,26292,0,0x965c,1,35264,5761,0xe656
,1,35290,0,0xe657,1,35292,0,0xe658,1,26296,0,0x9e41
,1,26297,0,0x9df9,1,35293,0,0xe659,1,35303,4652,0xe65a
,1,26300,0,0x9dfc,1,35316,9112,0xe65b,1,26302,0,0x9dfb
,1,35320,6675,0xe65c,1,35331,9981,0xe65d,1,37772,2492,0x8a99
,1,35350,6485,0xe65e,1,35344,0,0xe65f,1,26308,9993,0x9e40
,1,35340,0,0xe660,1,35355,4653,0xe661,1,26311,0,0x93dc
,1,35357,3510,0xe662,1,26313,3994,0x9dfa,1,35365,6186,0xe663
,1,37782,0,0x8dbd,1,37783,0,0x9199,1,35382,0,0xe664
,1,35393,6486,0xe665,1,37786,0,0x92c8,1,35419,5462,0xe666
,1,35410,3675,0xe667,1,35398,0,0xe668,1,35400,9998,0xe669
,1,35452,0,0xe66a,1,35437,0,0xe66b,1,26326,9973,0x9e42
,1,35436,0,0xe66c,1,35426,3452,0xe66d,1,26329,0,0x8f8c
,1,26330,0,0x9e43,1,35461,4658,0xe66e,1,37799,1513,0x8a5a
,1,26333,0,0x9498,1,35458,5284,0xe66f,1,35460,5142,0xe670
,1,26336,0,0x9e44,1,35496,3992,0xe671,1,35489,3681,0xe672
,1,37806,0,0x92c1,1,35473,8336,0xe673,1,35493,3991,0xe674
,1,26342,0,0x9e46,1,35494,5954,0xe675,1,35482,4659,0xe676
,1,26345,0,0x9e47,1,35491,6302,0xe677,1,35524,0,0xe678
,1,35533,6679,0xe679,1,35522,0,0xe67a,1,35546,3996,0xe67b
,1,35563,4610,0xe67c,1,26352,0,0x9e48,1,35571,0,0xe67d
,1,26354,0,0x8bc8,1,26355,0,0x8967,1,26356,0,0x8d58
,1,26357,0,0x9e49,1,35559,10036,0xe67e,1,26359,0,0x9e4a
,1,26360,4005,0x8f91,1,26361,0,0x9182,1,35556,4660,0xe680
,1,35569,4618,0xe681,1,26364,10038,0x99d6,1,26365,4009,0x915d
,1,26366,0,0x915c,1,26367,0,0x91d6,1,26368,0,0x8dc5
,1,35604,6334,0xe682,1,35552,6328,0xe683,1,26371,0,0x98f0
,1,35554,0,0xe684,1,35575,4007,0xe685,1,37841,0,0x934c
,1,35550,0,0xe686,1,26376,0,0x8c8e,1,26377,0,0x974c
,1,35547,0,0xe687,1,26379,3997,0x95fc,1,35596,3998,0xe688
,1,26381,4003,0x959e,1,35591,0,0xe689,1,26383,0,0x9e4b
,1,35610,0,0xe68a,1,35553,5464,0xe68b,1,35606,4006,0xe68c
,1,35600,4661,0xe68d,1,26388,5473,0x8df1,1,26389,0,0x92bd
,1,37857,2494,0x8bbe,1,26391,10043,0x984e,1,35607,0,0xe68e
,1,35616,4011,0xe68f,1,35635,4002,0xe690,1,26395,0,0x965d
,1,35622,6425,0xe692,1,26397,4000,0x92a9,1,26398,0,0x9e4d
,1,26399,0,0x8afa,1,35627,6160,0xe693,1,35646,0,0xe694
,1,35624,7747,0xe695,1,35649,3687,0xe696,1,35660,0,0xe697
,1,35663,0,0xe698,1,26406,10035,0x9e4e,1,26407,0,0x9e4f
,1,26408,0,0x96d8,1,35662,0,0xe699,1,26410,10003,0x96a2
,1,26411,0,0x9696,1,26412,0,0x967b,1,26413,0,0x8e44
,1,26414,0,0x9e51,1,35657,0,0xe69a,1,35670,10066,0xe69b
,1,26417,0,0x8ee9,1,35675,5477,0xe69c,1,35674,4663,0xe69d
,1,26420,0,0x9670,1,35691,0,0xe69e,1,26422,0,0x9e53
,1,26423,0,0x9e56,1,26424,4015,0x9e55,1,35679,5476,0xe69f
,1,26426,0,0x8af7,1,35692,0,0xe6a0,1,35695,4016,0xe6a1
,1,26429,0,0x8b80,1,35700,5147,0xe6a2,1,26431,0,0x9e52
,1,35709,0,0xe6a3,1,26433,0,0x9e54,1,35712,3742,0xe6a4
,1,35724,0,0xe6a5,1,35726,10058,0xe6a6,1,35730,4017,0xe6a7
,1,26438,0,0x9e57,1,35731,4667,0xe6a8,1,35734,4014,0xe6a9
,1,26441,4013,0x9099,1,35737,0,0xe6aa,1,35738,4654,0xe6ab
,1,35898,6698,0xe6ac,1,37912,6702,0x8fe0,1,37913,1535,0x93a8
,1,26447,4012,0x88c7,1,26448,0,0x8dde,1,26449,0,0x91ba
,1,35905,6088,0xe6ad,1,26451,0,0x8edb,1,35903,5292,0xe6ae
,1,35912,4448,0xe6af,1,26454,4022,0x8ff1,1,35916,0,0xe6b0
,1,35918,3760,0xe6b1,1,26457,0,0x9e5a,1,35920,0,0xe6b2
,1,35925,3709,0xe6b3,1,26460,0,0x936d,1,35938,6705,0xe6b4
,1,26462,0,0x9e58,1,26463,0,0x91a9,1,26464,4019,0x9e59
,1,26465,0,0x8ff0,1,26466,0,0x96db,1,26467,0,0x9e5b
,1,26468,0,0x9e5c,1,26469,0,0x9788,1,35948,4666,0xe6b5
,1,35960,10072,0xe6b6,1,35962,4455,0xe6b7,1,35970,4458,0xe6b8
,1,26474,4020,0x9e61,1,35977,4018,0xe6b9,1,35973,8120,0xe6ba
,1,26477,1151,0x8d59,1,35978,0,0xe6bb,1,26479,4023,0x9474
,1,26480,10075,0x9e5e,1,26481,0,0x938c,1,26482,0,0x9ddc
,1,26483,0,0x9de0,1,35981,4972,0xe6bc,1,26485,0,0x8b6e
,1,35982,4024,0xe6bd,1,26487,0,0x9466,1,35988,4460,0xe6be
,1,35964,4026,0xe6bf,1,35992,10085,0xe6c0,1,25117,4672,0xe6c1
,1,26492,0,0x9e60,1,36013,6415,0xe6c2,1,26494,0,0x8fbc
,1,26495,5480,0x94c2,1,36010,0,0xe6c3,1,36029,4673,0xe6c4
,1,36018,0,0xe6c5,1,36019,0,0xe6c6,1,36014,0,0xe6c7
,1,26501,0,0x9e66,1,37969,0,0x8ad3,1,26503,4025,0x94f8
,1,37971,0,0x96f8,1,26505,0,0x9e5d,1,36022,0,0xe6c8
,1,26507,0,0x9e63,1,26508,0,0x9e62,1,36040,0,0xe6c9
,1,36033,0,0xe6ca,1,36068,4051,0xe6cb,1,26512,4027,0x90cd
,1,36067,4470,0xe6cc,1,36058,6775,0xe6cd,1,36093,4034,0xe6ce
,1,36090,0,0xe6cf,1,26517,4029,0x968d,1,36091,4649,0xe6d0
,1,26519,4033,0x97d1,1,36100,4676,0xe6d1,1,36101,3764,0xe6d2
,1,26522,0,0x9687,1,36106,4677,0xe6d3,1,26524,0,0x89ca
,1,26525,0,0x8e7d,1,36103,3602,0xe6d4,1,36111,4035,0xe6d5
,1,26528,5486,0x9867,1,26529,6634,0x9e65,1,26530,6680,0x9095
,1,36109,0,0xe6d6,1,36112,10114,0xe6d7,1,36115,4046,0xe6d9
,1,26534,5488,0x9e64,1,36045,5491,0xe6da,1,36116,0,0xe6db
,1,26537,0,0x9e5f,1,36118,4047,0xe6dc,1,36199,0,0xe6dd
,1,36205,4048,0xe6de,1,36209,0,0xe6df,1,36211,6437,0xe6e0
,1,26543,0,0x8ccd,1,36225,6778,0xe6e1,1,36249,4052,0xe6e2
,1,36290,4049,0xe6e3,1,26547,4055,0x9e6b,1,26548,4061,0x9e69
,1,36286,6649,0xe6e4,1,26550,4060,0x89cb,1,26551,0,0x9e67
,1,26552,0,0x9e6d,1,26553,0,0x9e73,1,36282,0,0xe6e5
,1,36303,3954,0xe6e6,1,36314,0,0xe6e7,1,36310,0,0xe6e8
,1,36300,0,0xe6e9,1,36315,6465,0xe6ea,1,36299,4683,0xe6eb
,1,26561,0,0x91c6,1,36330,0,0xe6ec,1,36331,6780,0xe6ed
,1,26564,0,0x95bf,1,36319,0,0xe6ee,1,26566,0,0x9e75
,1,36323,8290,0xe6ef,1,36348,6781,0xe6f0,1,36360,0,0xe6f1
,1,26570,0,0x9541,1,36361,9205,0xe6f2,1,36351,0,0xe6f3
,1,36381,0,0xe6f4,1,26574,0,0x9e74,1,26575,0,0x9490
,1,26576,0,0x965e,1,26577,0,0x8ab9,1,36382,6782,0xe6f5
,1,26579,0,0x90f5,1,26580,0,0x8f5f,1,36368,0,0xe6f6
,1,36383,6176,0xe6f7,1,36418,4684,0xe6f8,1,26584,0,0x92d1
,1,36405,6668,0xe6f9,1,26586,0,0x974d,1,36400,6784,0xe6fa
,1,36404,8410,0xe6fb,1,26589,0,0x9e70,1,26590,0,0x9e6f
,1,36426,0,0xe6fc,1,36423,6481,0xe740,1,36425,0,0xe741
,1,26594,0,0x9e71,1,36428,7356,0xe742,1,26596,0,0x9e6e
,1,36432,6494,0xe743,1,36424,0,0xe744,1,26599,6677,0x9e76
,1,36441,0,0xe745,1,26601,0,0x9e6c,1,36452,0,0xe746
,1,36448,5215,0xe747,1,26604,0,0x9e6a,1,36394,4655,0xe748
,1,26606,0,0x9e72,1,26607,0,0x9e68,1,36451,6489,0xe749
,1,26609,0,0x928c,1,36437,0,0xe74a,1,26611,0,0x96f6
,1,26612,0,0x8ec4,1,26613,0,0x8df2,1,36470,3993,0xe74b
,1,36466,6786,0xe74c,1,36476,8799,0xe74d,1,36481,4681,0xe74e
,1,36487,4496,0xe74f,1,26619,0,0x8db8,1,36485,3697,0xe750
,1,36484,4497,0xe751,1,26622,0,0x968f,1,26623,0,0x8a60
,1,36491,7876,0xe752,1,36490,4686,0xe753,1,26626,0,0x92cc
,1,26627,0,0x93c8,1,26628,0,0x8968,1,36499,6063,0xe754
,1,36497,4685,0xe755,1,36500,0,0xe756,1,36505,0,0xe757
,1,36522,6495,0xe758,1,36513,0,0xe759,1,36524,0,0xe75a
,1,36528,0,0xe75b,1,36550,0,0xe75c,1,36529,0,0xe75d
,1,36542,0,0xe75e,1,36549,5969,0xe75f,1,36552,5890,0xe760
,1,36555,3777,0xe761,1,26643,0,0x90f0,1,36571,0,0xe762
,1,36579,0,0xe763,1,26646,0,0x90b2,1,26647,0,0x8c49
,1,36604,5157,0xe764,1,36603,6519,0xe765,1,36587,8480,0xe766
,1,36606,9255,0xe767,1,36618,0,0xe768,1,36613,4690,0xe769
,1,26654,0,0x9e78,1,36629,6793,0xe76a,1,36626,0,0xe76b
,1,26657,0,0x8d5a,1,26658,0,0x8a9c,1,36633,0,0xe76c
,1,36627,3844,0xe76d,1,36636,6509,0xe76e,1,36639,0,0xe76f
,1,36635,0,0xe770,1,36620,0,0xe771,1,26665,0,0x9e7a
,1,26666,0,0x8a94,1,26667,0,0x9e81,1,36646,4691,0xe772
,1,36659,6794,0xe773,1,36667,0,0xe774,1,36665,0,0xe775
,1,36677,8946,0xe776,1,36674,4692,0xe777,1,26674,0,0x9e7d
,1,36670,5947,0xe778,1,26676,0,0x90f1,1,36684,0,0xe779
,1,36681,8414,0xe77a,1,36678,6513,0xe77b,1,26680,6832,0x8a6a
,1,26681,0,0x8daa,1,36686,4522,0xe77c,1,36695,6798,0xe77d
,1,26684,0,0x8a69,1,26685,0,0x8dcd,1,36700,6799,0xe77e
,1,36706,0,0xe780,1,26688,0,0x9e7b,1,26689,0,0x8c85
,1,26690,6820,0x8c6a,1,26691,0,0x938d,1,36707,6802,0xe781
,1,36708,6516,0xe782,1,26694,0,0x9e79,1,36764,0,0xe783
,1,26696,0,0x88c4,1,36767,5907,0xe784,1,36771,0,0xe785
,1,36781,0,0xe786,1,36783,6812,0xe787,1,26701,0,0x9e7c
,1,26702,0,0x9e7e,1,36791,0,0xe788,1,26704,0,0x8bcb
,1,26705,0,0x8c4b,1,36826,4688,0xe789,1,26707,6880,0x8aba
,1,26708,0,0x8b6a,1,36837,4447,0xe78a,1,36834,9262,0xe78b
,1,36842,6819,0xe78c,1,36847,6884,0xe78d,1,26713,0,0x9e82
,1,36999,0,0xe78e,1,36852,4693,0xe78f,1,26716,0,0x8df7
,1,26717,0,0x9691,1,36869,0,0xe790,1,26719,0,0x8e56
,1,36857,0,0xe791,1,36858,0,0xe792,1,36881,0,0xe793
,1,26723,0,0x9e83,1,36885,0,0xe794,1,36897,0,0xe795
,1,36877,0,0xe796,1,26727,0,0x954f,1,36894,0,0xe797
,1,36886,0,0xe798,1,36875,0,0xe799,1,36903,6039,0xe79a
,1,36918,0,0xe79b,1,36917,0,0xe79c,1,36921,6823,0xe79d
,1,36856,6893,0xe79e,1,36943,6552,0xe79f,1,36944,0,0xe7a0
,1,36945,4696,0xe7a1,1,36946,0,0xe7a2,1,26740,0,0x9e8f
,1,36878,4534,0xe7a3,1,26742,0,0x89b1,1,26743,0,0x9e84
,1,36937,0,0xe7a4,1,36926,0,0xe7a5,1,36950,5000,0xe7a6
,1,36952,0,0xe7a7,1,36958,6825,0xe7a8,1,36968,0,0xe7a9
,1,26750,0,0x9e95,1,26751,0,0x9e85,1,36975,0,0xe7aa
,1,26753,4728,0x97c0,1,36982,6827,0xe7ab,1,26755,0,0x9e8c
,1,38568,0,0xe7ac,1,26757,0,0x947e,1,36978,0,0xe7ad
,1,36994,0,0xe7ae,1,36989,0,0xe7af,1,36993,0,0xe7b0
,1,36992,0,0xe7b1,1,37002,0,0xe7b2,1,37001,0,0xe7b3
,1,26765,0,0x9e94,1,37007,4729,0xe7b4,1,26767,0,0x9e87
,1,37032,6829,0xe7b5,1,37039,0,0xe7b6,1,37041,0,0xe7b7
,1,26771,0,0x88b2,1,26772,0,0x9e89,1,37045,0,0xe7b8
,1,37090,6830,0xe7b9,1,26775,0,0x8d5b,1,37092,6835,0xe7ba
,1,25160,6836,0xe7bb,1,37083,4549,0xe7bc,1,26779,0,0x9e8b
,1,37122,6146,0xe7bd,1,26781,0,0x9e8a,1,37138,0,0xe7be
,1,26783,0,0x9e86,1,26784,0,0x9e91,1,37145,0,0xe7bf
,1,26786,0,0x8fbd,1,37170,4699,0xe7c0,1,37168,6846,0xe7c1
,1,37194,4700,0xe7c2,1,26790,0,0x9aeb,1,26791,0,0x8ce6
,1,26792,0,0x979c,1,37206,4505,0xe7c3,1,37208,6840,0xe7c4
,1,37219,6841,0xe7c5,1,38263,0,0x92b7,1,26797,10120,0x9e88
,1,37221,10122,0xe7c6,1,26799,0,0x92f2,1,26800,0,0x8a42
,1,26801,0,0x8dab,1,37225,4701,0xe7c7,1,26803,0,0x9e80
,1,37235,6842,0xe7c8,1,38272,2497,0x96e5,1,26806,0,0x8a81
,1,37234,4065,0xe7c9,1,38275,0,0x914d,1,26809,0,0x9e8e
,1,26810,0,0x9e92,1,37259,6843,0xe7ca,1,26812,4069,0x938e
,1,37257,4566,0xe7cb,1,38281,0,0x95c2,1,37250,4070,0xe7cc
,1,38283,0,0x8a4a,1,37282,6593,0xe7cd,1,37291,10128,0xe7ce
,1,37295,5493,0xe7cf,1,38287,460,0x895b,1,37290,6851,0xe7d0
,1,38289,2504,0x8ad5,1,37301,5936,0xe7d1,1,38291,4703,0x8ad4
,1,26825,4074,0x96c7,1,26826,0,0x9e97,1,26827,4075,0x8afb
,1,37300,6472,0xe7d2,1,26829,4078,0x9e9e,1,37306,4079,0xe7d3
,1,37312,6860,0xe7d4,1,37313,4704,0xe7d5,1,37321,0,0xe7d6
,1,26834,0,0x965f,1,37323,0,0xe7d7,1,26836,5495,0x9e9f
,1,26837,4081,0x9ea1,1,37328,6864,0xe7d8,1,38306,2502,0x8ad6
,1,38307,2500,0x8a74,1,38308,6865,0x8d7d,1,26842,1336,0x9249
,1,37334,5496,0xe7d9,1,37343,4083,0xe7da,1,37345,4082,0xe7db
,1,37339,5497,0xe7dc,1,26847,0,0x938f,1,26848,10131,0x9ea9
,1,26849,5501,0x9e9c,1,37372,4084,0xe7dd,1,26851,0,0x9ea6
,1,37365,6866,0xe7de,1,37366,6867,0xe7df,1,37406,0,0xe7e0
,1,38322,2501,0x897b,1,37375,4705,0xe7e1,1,37396,10134,0xe7e2
,1,37420,0,0xe7e3,1,37397,5510,0xe7e4,1,37393,6871,0xe7e5
,1,37470,4706,0xe7e6,1,26862,4088,0x9058,1,26863,0,0x9eaa
,1,37463,4087,0xe7e7,1,37445,4086,0xe7e8,1,26866,5511,0x90b1
,1,37449,4090,0xe7e9,1,37476,0,0xe7ea,1,37448,0,0xe7eb
,1,37525,0,0xe7ec,1,37439,6873,0xe7ed,1,37451,4094,0xe7ee
,1,26873,0,0x9ea8,1,26874,0,0x8abb,1,37456,6617,0xe7ef
,1,38343,4712,0x88c5,1,37532,6090,0xe7f0,1,37526,0,0xe7f1
,1,37523,4093,0xe7f2,1,26880,5513,0x986f,1,26881,4104,0x9e96
,1,37531,4097,0xe7f3,1,37466,6876,0xe7f4,1,26884,0,0x9ea4
,1,26885,5514,0x88d6,1,37583,6638,0xe7f5,1,37561,4679,0xe7f6
,1,26888,5516,0x9e98,1,37559,4108,0xe7f7,1,37609,4105,0xe7f8
,1,26891,4109,0x96b8,1,26892,0,0x9e9d,1,26893,1237,0x9041
,1,26894,0,0x92c5,1,26895,5523,0x9e93,1,37647,0,0xe7f9
,1,37626,4110,0xe7fa,1,26898,0,0x9ea3,1,37700,5524,0xe7fb
,1,37678,6343,0xe7fc,1,37657,6142,0xe840,1,37666,4111,0xe841
,1,37658,4116,0xe842,1,37667,0,0xe843,1,26905,0,0x909a
,1,26906,4113,0x9ead,1,26907,0,0x8a91,1,26908,0,0x8c9f
,1,37690,0,0xe844,1,37685,6655,0xe845,1,37691,0,0xe846
,1,37724,0,0xe847,1,26913,0,0x9eaf,1,26914,0,0x9e9a
,1,26915,0,0x9eae,1,37728,6189,0xe848,1,26917,0,0x9ea7
,1,26918,0,0x9e9b,1,37756,6671,0xe849,1,26920,0,0x9eab
,1,37742,0,0xe84a,1,26922,0,0x9eac,1,37718,8295,0xe84b
,1,37808,0,0xe84c,1,37804,0,0xe84d,1,37805,0,0xe84e
,1,37780,0,0xe84f,1,26928,0,0x9ebd,1,37817,0,0xe850
,1,37846,0,0xe851,1,37847,6083,0xe852,1,26932,0,0x93cc
,1,37864,4716,0xe853,1,26934,0,0x9ea2,1,37861,8118,0xe854
,1,37848,0,0xe855,1,26937,0,0x9eb9,1,37827,0,0xe856
,1,37853,6690,0xe857,1,37840,6689,0xe858,1,26941,0,0x9ebb
,1,37832,0,0xe859,1,26943,0,0x92d6,1,37860,6695,0xe85a
,1,37914,0,0xe85b,1,37908,4717,0xe85c,1,37907,4428,0xe85d
,1,37891,0,0xe85e,1,37895,0,0xe85f,1,37904,6885,0xe860
,1,37942,0,0xe861,1,37931,0,0xe862,1,37941,4714,0xe863
,1,26954,0,0x976b,1,37921,0,0xe864,1,37946,0,0xe865
,1,37953,6774,0xe866,1,37970,0,0xe867,1,37956,0,0xe868
,1,37979,0,0xe869,1,38428,0,0x958c,1,37984,9558,0xe86a
,1,26963,0,0x9596,1,26964,0,0x9eb6,1,26965,0,0x91c8
,1,37986,4119,0xe86b,1,37982,4674,0xe86c,1,37994,0,0xe86d
,1,26969,0,0x9ebc,1,26970,0,0x915e,1,37417,0,0xe86e
,1,26972,0,0x9eb3,1,26973,4123,0x9ec0,1,26974,0,0x9ebf
,1,38442,0,0x8de3,1,26976,0,0x93ed,1,26977,10147,0x9ebe
,1,26978,0,0x93e8,1,38000,4126,0xe86f,1,38005,4128,0xe870
,1,38007,0,0xe871,1,38013,4439,0xe872,1,38450,0,0x9668
,1,37978,4500,0xe873,1,38012,6886,0xe874,1,26986,0,0x9ec2
,1,26987,0,0x9eb5,1,38014,6887,0xe875,1,26989,0,0x8bc6
,1,26990,0,0x9eb8,1,26991,0,0x8f7c,1,38459,0,0x916a
,1,38017,10160,0xe876,1,38015,6888,0xe877,1,26995,0,0x9480
,1,38463,2506,0x88a2,1,26997,1137,0x8bc9,1,38274,10164,0xe878
,1,26999,4129,0x9eb2,1,27000,0,0x9eb4,1,38468,2505,0x958d
,1,38279,6890,0xe879,1,38282,0,0xe87a,1,27004,0,0x984f
,1,27005,0,0x8a79,1,27006,0,0x9eb7,1,38292,6891,0xe87b
,1,38294,4132,0xe87c,1,27009,4130,0x9ec1,1,27010,558,0x8a54
,1,38296,0,0xe87d,1,38297,4131,0xe87e,1,38480,0,0x8cc0
,1,38304,0,0xe880,1,38312,6603,0xe881,1,38311,6602,0xe882
,1,38317,0,0xe883,1,27018,0,0x8de5,1,38332,4711,0xe884
,1,38331,4709,0xe885,1,38329,5528,0xe886,1,27022,7044,0x897c
,1,38334,0,0xe887,1,38491,0,0x95c3,1,27025,4141,0x9ed2
,1,38346,4145,0xe888,1,38339,4144,0xe88a,1,27028,4146,0x9850
,1,27029,0,0x9ed5,1,38349,4444,0xe88b,1,38498,0,0x8940
,1,38499,6896,0x9077,1,38500,0,0x8f9c,1,38501,0,0x8ad7
,1,27035,4147,0x9059,1,27036,0,0x9ed4,1,38348,7058,0xe88c
,1,38357,4604,0xe88d,1,38506,0,0x9486,1,27040,0,0x9ed3
,1,38356,4157,0xe88e,1,38358,0,0xe88f,1,38364,0,0xe890
,1,38369,6651,0xe891,1,38512,4725,0x8941,1,38373,6899,0xe892
,1,27047,4154,0x9ed0,1,38515,0,0x92c2,1,38370,0,0xe893
,1,38517,0,0x97cb,1,38518,0,0x93a9,1,38433,4135,0xe894
,1,38520,4726,0x97a4,1,27054,0,0x9ec4,1,38522,0,0x8caf
,1,38440,6901,0xe895,1,27057,0,0x9ee1,1,38525,2507,0x977a
,1,38446,4719,0xe896,1,27060,0,0x9ed6,1,38447,4720,0xe897
,1,38466,0,0xe898,1,38476,6903,0xe899,1,38479,0,0xe89a
,1,38475,4722,0xe89b,1,38533,6904,0x8bf7,1,38534,2517,0x97b2
,1,38519,6905,0xe89c,1,38536,0,0x8c47,1,27070,0,0x9ec9
,1,38538,2509,0x91e0,1,38539,6906,0xe440,1,27073,0,0x9ec7
,1,38492,4159,0xe89d,1,38542,2519,0x8a4b,1,38543,0,0x908f
,1,38494,0,0xe89e,1,38493,6892,0xe89f,1,38495,4410,0xe8a0
,1,38502,0,0xe8a1,1,38548,0,0x8a75,1,27082,4162,0x9ecc
,1,27083,0,0x8d5c,1,27084,4165,0x92c6,1,27085,4161,0x9184
,1,38553,2511,0x8c84,1,38514,4727,0xe8a2,1,38555,2508,0x8ddb
,1,38556,6908,0x8fe1,1,38508,5535,0xe8a3,1,27091,0,0x9ec8
,1,38541,0,0xe8a4,1,38560,0,0x8942,1,38552,6909,0xe8a5
,1,38549,0,0xe8a6,1,27096,1579,0x976c,1,27097,5536,0x968a
,1,38551,5537,0xe8a7,1,38578,0,0xe8ab,1,38567,7152,0xe8a9
,1,27101,3823,0x9ecd,1,27102,10191,0x9ed7,1,38570,0,0xe8a8
,1,38582,0,0xe8ae,1,38584,0,0xe8af,1,38605,4733,0xe8b4
,1,35149,6916,0xe8b5,1,38613,10235,0xe8b8,1,38576,0,0xe8ac
,1,38577,0,0xe8aa,1,27111,4166,0x9edf,1,27112,10241,0x9ed8
,1,38580,4736,0xe8ad,1,38649,0,0xe8b9,1,27115,4171,0x9ee5
,1,38583,0,0x97ea,1,27117,4172,0x9ee3,1,38585,6917,0xe8b0
,1,38660,6918,0xe8ba,1,38587,0,0x90c7,1,38588,0,0x94b9
,1,27122,10244,0x9ede,1,38664,6920,0xe8bc,1,38675,6924,0xe8bd
,1,38592,4737,0x909d,1,38593,4739,0x8ae5,1,38671,0,0xe8c0
,1,38678,0,0xe8c1,1,38596,2522,0x9759,1,38597,0,0x89eb
,1,38598,1201,0x8f57,1,38599,0,0x8cd9,1,27133,0,0x9185
,1,38601,6927,0xe8b3,1,27135,0,0x9edb,1,38603,0,0xe8b2
,1,38604,0,0x8e93,1,27138,4173,0x9ed9,1,38606,0,0xe8b1
,1,38692,0,0xe8c3,1,27141,0,0x9ee0,1,38609,0,0x8e47
,1,38698,5538,0xe8c4,1,38717,0,0xe8c7,1,21202,0,0xe8d3
,1,27146,4175,0x9ee6,1,27147,3177,0x94f3,1,27148,0,0x9eec
,1,38763,10250,0xe8d4,1,38617,0,0x99d4,1,38777,4740,0xe8d6
,1,38619,0,0x9097,1,38620,6928,0xe8b6,1,27154,10253,0x9ee7
,1,27155,5541,0x9eea,1,27156,0,0x9ee4,1,38790,4741,0xe8db
,1,38812,0,0xe8df,1,27159,1540,0x9294,1,38627,0,0x93ef
,1,27161,0,0x9557,1,38822,6933,0xe8e1,1,27163,0,0x9eda
,1,38835,0,0xe8e3,1,38632,6934,0x894a,1,27166,5547,0x9ee2
,1,27167,867,0x8fbe,1,38635,0,0x8eb4,1,27169,0,0x96cd
,1,27170,0,0x9ef6,1,27171,0,0x9ee9,1,38854,5550,0xe8e6
,1,38640,0,0x95b5,1,38898,0,0xe8ec,1,38642,0,0x895f
,1,31455,0,0xe8ed,1,27177,0,0x8ca0,1,27178,0,0x89a1
,1,27179,1590,0x8a7e,1,38647,0,0x978b,1,38948,0,0xe8f3
,1,27182,4181,0x9ed1,1,38982,7227,0xe8f7,1,38651,6912,0x9364
,1,38991,0,0xe8f8,1,38987,6935,0xe8f9,1,39023,6936,0xe8fb
,1,39082,6937,0xe943,1,38656,747,0x8ef9,1,27190,0,0x9eee
,1,39094,0,0xe946,1,27192,0,0x9ef5,1,27193,4186,0x8ef7
,1,27194,0,0x8a92,1,38662,4744,0xe8bb,1,38663,0,0x906b
,1,27197,4190,0x924d,1,39171,4746,0xe94c,1,38666,6938,0x97ec
,1,39177,0,0xe94d,1,39200,0,0xe955,1,38669,0,0xe8b7
,1,38670,0,0xe8be,1,27204,4194,0x9eeb,1,39229,4748,0xe958
,1,38673,6940,0xe8bf,1,27207,0,0x9ef0,1,27208,4191,0x9ef4
,1,39230,6941,0xe959,1,39241,0,0xe95b,1,27211,4195,0x8bb4
,1,39237,0,0xe95c,1,39244,0,0xe961,1,38681,6943,0xe8c2
,1,39319,0,0xe963,1,39341,4752,0xe966,1,38684,0,0x919a
,1,39342,6945,0xe967,1,38686,0,0x89e0,1,39356,0,0xe968
,1,39389,0,0xe96b,1,39425,4753,0xe974,1,39439,0,0xe975
,1,27224,10258,0x8b6b,1,27225,4207,0x9ef2,1,39394,0,0xe977
,1,39479,4755,0xe97a,1,38695,0,0x96b6,1,39490,0,0xe97c
,1,39491,0,0xe97e,1,27231,4210,0x8b40,1,39511,0,0xe984
,1,27233,0,0x93c9,1,27234,0,0x9ef1,1,39524,0,0xe988
,1,39612,6946,0xe98e,1,38704,7258,0xe8c5,1,27238,0,0x9ef3
,1,38706,0,0x9849,1,39616,5551,0xe98f,1,39635,0,0xe992
,1,39636,0,0xe993,1,39646,4757,0xe994,1,39654,0,0xe998
,1,38712,0,0x9e50,1,38713,0,0xe8c6,1,39663,0,0xe999
,1,39659,5553,0xe99a,1,39662,0,0xe99b,1,27250,4211,0x9eed
,1,38718,4758,0xe8c8,1,39668,0,0xe99c,1,39665,0,0xe99d
,1,39671,10265,0xe99e,1,38722,0,0xe8cc,1,27256,0,0x9eef
,1,38724,0,0xe8c9,1,39704,6942,0xe9a1,1,38726,0,0xe8ca
,1,39715,10266,0xe9a5,1,38728,0,0xe8cb,1,38729,6948,0xe8cd
,1,27263,0,0x8a80,1,27264,0,0x9268,1,39722,0,0xe9aa
,1,39730,5558,0xe9ad,1,39757,5559,0xe9b1,1,27268,5560,0x9efa
,1,39830,0,0xe9b9,1,39831,0,0xe9ba,1,38738,0,0x90c2
,1,39839,0,0xe9bb,1,39840,4760,0xe9bc,1,39848,10273,0xe9bd
,1,38742,6950,0x96f5,1,39882,10287,0xe9c0,1,27277,0,0x9ef8
,1,27278,860,0x8ce7,1,39889,5561,0xe9c4,1,27280,10290,0x9ef7
,1,38748,6953,0xe8ce,1,39892,0,0xe9c9,1,38750,0,0x94f1
,1,39994,4761,0xe9cb,1,38752,0,0xe8cf,1,39945,4392,0xe9d1
,1,38754,2526,0x96ca,1,39948,0,0xe9d3,1,38756,0,0xe8d0
,1,39946,0,0xe9d7,1,38758,0,0xe8d1,1,27292,0,0x9e77
,1,38760,4762,0xe8d2,1,38761,4763,0x8a76,1,40007,6955,0xe9df
,1,27296,4216,0x9ef9,1,40201,6956,0xe9eb,1,38765,2523,0x9078
,1,27299,5562,0x9efc,1,40227,0,0xe9f3,1,40264,4768,0xe9fc
,1,38769,0,0xe8d5,1,40285,0,0xea40,1,40292,5565,0xea42
,1,38772,0,0x8c43,1,27306,0,0x9f4b,1,40281,6958,0xea45
,1,27308,0,0x9f47,1,40314,0,0xea4b,1,27310,4218,0x9e8d
,1,38778,0,0xe8da,1,40346,0,0xea4c,1,38780,0,0xe8d8
,1,40356,0,0xea4d,1,27315,0,0x9f46,1,40388,0,0xea50
,1,40605,0,0xea6c,1,38785,0,0xe8d9,1,40632,0,0xea6f
,1,27320,0,0x9f45,1,38788,0,0x8a93,1,38789,0,0xe8d7
,1,27323,4224,0x9f42,1,40618,0,0xea70,1,38753,6959,0xea72
,1,40695,0,0xea82,1,40699,0,0xea84,1,38795,0,0xe8dc
,1,27329,0,0x9ee8,1,38797,2527,0x88c6,1,27331,0,0x9f43
,1,38799,0,0xe8dd,1,38800,0,0xe8de,1,40701,6960,0xea86
,1,40711,0,0xea87,1,40712,0,0xea88,1,40748,0,0xea8c
,1,40786,5566,0xea8f,1,40801,10311,0xea94,1,40823,0,0xea99
,1,38808,0,0x8fe2,1,40818,0,0xea9a,1,27079,10312,0xeaa0
,1,36953,6962,0xeaa1,1,27345,4225,0x9f49,1,24034,0,0xeaa6
,1,27347,10313,0x9845,1,24084,5230,0xeaa7,1,38816,0,0x8b66
,1,24088,8791,0xeaa8,1,24152,10317,0xeaa9,1,38819,0,0xe8e2
,1,24158,7869,0xeaaa,1,27354,0,0x9f4c,1,27355,4229,0x8bf9
,1,24254,6966,0xeaab,1,38824,6967,0xe8e0,1,27358,0,0x9f48
,1,27359,0,0x9f4a,1,38827,0,0xe691,1,24267,4668,0xeaad
,1,38829,0,0x95da,1,24313,5568,0xeaae,1,24320,4769,0xeaaf
,1,24322,0,0xeab0,1,24327,10321,0xeab1,1,24349,4772,0xeab2
,1,27368,4231,0x94a5,1,38836,0,0xe8e4,1,27370,0,0x9f4d
,1,24355,0,0xeab3,1,24372,5302,0xeab4,1,24374,0,0xeab5
,1,24381,0,0xeab6,1,24384,5570,0xeab7,1,24389,0,0xeab8
,1,24404,6895,0xeab9,1,24408,4971,0xeaba,1,24420,6973,0xeabb
,1,24423,6818,0xeabc,1,24445,4626,0xeabd,1,24457,5576,0xeabe
,1,24476,0,0xeabf,1,38851,0,0xe8e5,1,24487,0,0xeac0
,1,27386,0,0x9f51,1,27387,4239,0x9f4e,1,24495,5479,0xeac1
,1,38856,6974,0xe8e7,1,24501,5579,0xeac2,1,24503,10073,0xeac3
,1,38859,0,0xe8e8,1,24521,10329,0xeac4,1,24542,0,0xeac5
,1,24545,0,0xeac6,1,27396,0,0x9793,1,27397,0,0x9f4f
,1,24553,10330,0xeac7,1,24589,0,0xeac8,1,38867,0,0x8ad8
,1,24596,10331,0xeac9,1,27402,4883,0x9edc,1,24600,6976,0xeaca
,1,24627,5277,0xeacb,1,24629,10332,0xeacc,1,24647,5278,0xeacd
,1,24733,6977,0xeacf,1,24734,5588,0xead0,1,38876,6979,0xe8e9
,1,27410,0,0x9f52,1,24779,5587,0xead1,1,24788,0,0xead2
,1,24789,0,0xead3,1,27414,10336,0x9f53,1,24797,6982,0xead4
,1,24824,0,0xead5,1,24860,0,0xead6,1,24875,0,0xead7
,1,24880,4775,0xead8,1,24887,8760,0xead9,1,27421,0,0x8954
,1,24973,4776,0xeadb,1,27423,0,0x9f55,1,27424,0,0x8c87
,1,27425,0,0x8e9f,1,38893,0,0xe8ea,1,27427,1264,0x8bd3
,1,64063,0,0xeadc,1,25020,0,0xeadd,1,25017,10338,0xeade
,1,27431,4241,0x89a2,1,38899,0,0x89b9,1,25122,4721,0xeae0
,1,38901,0,0xe8ef,1,38902,7374,0xe8ee,1,25150,4782,0xeae1
,1,25155,0,0xeae2,1,25174,0,0xeae3,1,25178,6985,0xeae4
,1,38907,0,0x8943,1,25199,7382,0xeae5,1,27442,0,0x977e
,1,25221,6442,0xeae6,1,38911,0,0x8bbf,1,25284,4783,0xeae7
,1,38913,0,0x95c5,1,38914,2531,0x92b8,1,38915,2528,0x8da0
,1,27449,10339,0x9f59,1,27450,560,0x8b5c,1,38918,0,0x8f87
,1,25302,0,0xeae8,1,27453,807,0x8bd4,1,27454,0,0x8abc
,1,25340,5590,0xeae9,1,25354,9151,0xeaea,1,38924,0,0xe8f1
,1,25368,10353,0xeaeb,1,27459,10366,0x9f5c,1,38927,0,0xe8f0
,1,38928,0,0x9761,1,38929,4785,0x8ae6,1,38930,2532,0x94d0
,1,38931,0,0x93da,1,27465,10367,0x9f5d,1,25401,6986,0xeaec
,1,25411,4697,0xeaed,1,27468,820,0x89cc,1,38936,0,0x97cc
,1,27470,7405,0x9256,1,38938,0,0x8c7a,1,27472,0,0x9f5e
,1,25445,0,0xeaee,1,25468,0,0xeaef,1,27475,5591,0x8abd
,1,27476,0,0x9f60,1,25573,9561,0xeaf0,1,38945,4789,0xe8f4
,1,25581,0,0xeaf1,1,25589,6988,0xeaf2,1,27481,4248,0x9f5f
,1,25616,5762,0xeaf3,1,27483,0,0x9f61,1,25620,9573,0xeaf4
,1,25634,0,0xeaf5,1,25721,8110,0xeaf6,1,27487,7420,0x9f62
,1,25681,4790,0xeaf7,1,38956,2536,0x966a,1,27490,1236,0x8e7e
,1,27491,0,0x90b3,1,27492,0,0x8d9f,1,25696,4791,0xeaf8
,1,27494,0,0x9590,1,25709,4794,0xeaf9,1,25806,10385,0xeafa
,1,38964,1422,0x896f,1,27498,0,0x9863,1,25790,0,0xeafb
,1,38967,6991,0xe8f5,1,38968,0,0xe8f2,1,25791,6266,0xeafc
,1,27503,0,0x8e95,1,38971,0,0x9570,1,38972,0,0x978a
,1,38973,0,0xe8f6,1,27507,0,0x8dce,1,27508,0,0x97f0
,1,25796,6992,0xeb40,1,25802,0,0xeb41,1,25808,4796,0xeb42
,1,27512,0,0x9f64,1,27513,0,0x9f65,1,25847,6995,0xeb43
,1,27515,4250,0x8e80,1,25851,10395,0xeb44,1,25890,5401,0xeb45
,1,25897,0,0xeb46,1,27519,0,0x9f66,1,27520,4253,0x9f67
,1,38988,6997,0x91e8,1,38989,4799,0x8a7a,1,38990,2537,0x8a7b
,1,27524,4252,0x9f68,1,64065,0,0xeb47,1,27526,0,0x9677
,1,25959,5601,0xeb48,1,26013,5602,0xeb49,1,38996,718,0x8ae7
,1,38997,689,0x8cb0,1,27531,5604,0x8e63,1,26112,5435,0xeb4b
,1,39000,2540,0x8ae8,1,26121,5614,0xeb4c,1,26133,5442,0xeb4d
,1,39003,0,0x935e,1,26142,4713,0xeb4e,1,26170,6474,0xeb4f
,1,39006,0,0x97de,1,26146,0,0xeb50,1,27541,0,0x9f6c
,1,27542,0,0x9042,1,26148,5003,0xeb51,1,27544,0,0x9f6b
,1,26155,5296,0xeb52,1,26160,5620,0xeb53,1,26161,6998,0xeb54
,1,39015,0,0x8cda,1,26163,0,0xeb55,1,27550,0,0x9f6d
,1,26363,5148,0xeb56,1,39019,0,0xe8fa,1,26184,5621,0xeb57
,1,26188,7804,0xeb58,1,43761,0,0xeb59,1,27556,4254,0x9f6e
,1,39024,0,0xe8fc,1,39025,0,0xe940,1,26201,0,0xeb5a
,1,39027,0,0xe942,1,39028,4803,0xe941,1,27562,0,0x9f6f
,1,27563,0,0x9f70,1,26202,5447,0xeb5b,1,26209,4806,0xeb5c
,1,26213,6999,0xeb5d,1,27567,0,0x9f71,1,26227,8998,0xeb5e
,1,27569,0,0x9f73,1,27570,0,0x9f72,1,27571,0,0x9f74
,1,27572,0,0x89a3,1,27573,0,0x9269,1,26231,7817,0xeb5f
,1,27575,0,0x9f75,1,26232,0,0xeb60,1,26253,0,0xeb61
,1,27578,0,0x8e45,1,27579,0,0x8a6b,1,27580,0,0x9f76
,1,26272,7002,0xeb63,1,26290,7006,0xeb64,1,27583,0,0x9361
,1,27584,0,0x9aca,1,26299,0,0xeb65,1,26310,6072,0xeb66
,1,26312,10061,0xeb67,1,15138,8476,0xeb68,1,27589,0,0x8b42
,1,27590,0,0x9f77,1,26331,6404,0xeb69,1,26344,7008,0xeb6a
,1,26362,7012,0xeb6b,1,26387,5470,0xeb6c,1,27595,0,0x9f78
,1,26419,0,0xeb6e,1,27597,0,0x95ea,1,27598,0,0x9688
,1,26470,4807,0xeb6f,1,26439,6482,0xeb70,1,26440,0,0xeb71
,1,27602,0,0x93c5,1,27603,0,0x9f79,1,27604,0,0x94e4
,1,26491,0,0xeb72,1,26497,4815,0xeb73,1,26515,4718,0xeb74
,1,27608,0,0x94f9,1,26520,5211,0xeb75,1,26523,5160,0xeb76
,1,27611,0,0x96d1,1,26555,6877,0xeb77,1,39080,0,0x9597
,1,26617,0,0xeb78,1,27615,4255,0x9f7a,1,26560,6094,0xeb79
,1,26583,9196,0xeb7a,1,26620,10406,0xeb7b,1,26625,6571,0xeb7c
,1,39087,0,0xe944,1,26706,6879,0xeb7d,1,39089,7014,0xe945
,1,26653,7015,0xeb7e,1,26668,6581,0xeb80,1,26673,8297,0xeb81
,1,26715,0,0xeb82,1,27627,4258,0x9f7c,1,27628,10410,0x9f7b
,1,26738,10412,0xeb83,1,26741,7016,0xeb84,1,27631,5622,0x9f7e
,1,26787,10413,0xeb86,1,26789,7017,0xeb87,1,26802,0,0xeb88
,1,27635,0,0x9f7d,1,26824,10415,0xeb89,1,26832,0,0xeb8a
,1,26856,0,0xeb8b,1,26861,10421,0xeb8c,1,39107,0,0xe948
,1,39108,0,0xe947,1,26864,0,0xeb8d,1,39110,0,0xe949
,1,26865,6614,0xeb8e,1,26876,5627,0xeb8f,1,26890,6153,0xeb90
,1,26953,4817,0xeb91,1,43779,10438,0xeb92,1,26933,0,0xeb93
,1,26946,7019,0xeb94,1,26967,4816,0xeb95,1,26979,0,0xeb96
,1,26980,4818,0xeb97,1,26984,5373,0xeb98,1,27008,5529,0xeb99
,1,27656,0,0x9f81,1,64020,0,0xeb9a,1,27045,7021,0xeb9b
,1,27053,0,0xeb9c,1,27087,10185,0xeb9d,1,15286,0,0xeb9e
,1,15299,7025,0xeb9f,1,27663,0,0x8e81,1,39131,4820,0x94f2
,1,27665,2834,0x96af,1,27106,0,0xeba0,1,27667,0,0x9f82
,1,39135,2543,0x9048,1,27113,7177,0xeba1,1,27125,10455,0xeba3
,1,27671,468,0x8b43,1,27126,10456,0xeba4,1,27151,10252,0xeba5
,1,27157,10459,0xeba6,1,27675,0,0x9f84,1,43785,5629,0xeba7
,1,27195,0,0xeba8,1,39145,0,0xe94a,1,27198,10469,0xeba9
,1,39147,0,0xe94b,1,27205,0,0xebaa,1,39149,7544,0x99aa
,1,39150,2545,0x9f5a,1,39151,2544,0x94d1,1,27216,10471,0xebab
,1,27222,10475,0xebac,1,39154,0,0x88f9,1,27227,10477,0xebad
,1,39156,0,0x88b9,1,27243,7705,0xebae,1,27251,0,0xebaf
,1,43787,0,0xebb0,1,27273,0,0xebb1,1,27284,0,0xebb2
,1,27293,0,0xebb3,1,27294,0,0xebb4,1,39164,0,0x8e94
,1,39165,0,0x964f,1,39166,4822,0x8ffc,1,27700,0,0x9085
,1,27301,7026,0xebb5,1,27364,7027,0xebb6,1,27703,0,0x9558
,1,27704,4265,0x8969,1,27367,10324,0xebb7,1,39173,7028,0x96dd
,1,15375,0,0xebb8,1,27419,6643,0xebba,1,27422,4896,0xebbb
,1,27710,4267,0x94c3,1,39178,7029,0x977b,1,27712,7570,0x92f3
,1,39180,707,0x8961,1,27714,7573,0x8b81,1,27436,0,0xebbc
,1,27445,0,0xebbd,1,39184,0,0x8e60,1,27462,0,0xebbe
,1,39186,0,0xe94e,1,39187,0,0x89ec,1,39188,0,0xe94f
,1,27478,0,0xebbf,1,27488,5596,0xebc0,1,27493,5597,0xebc1
,1,39192,7030,0xe950,1,27726,0,0x94c4,1,27495,7667,0xebc2
,1,27728,10492,0x8eac,1,27511,10496,0xebc3,1,39197,0,0xe952
,1,39198,0,0xe953,1,27522,0,0xebc4,1,27733,4268,0x9f88
,1,39201,0,0xe951,1,27735,0,0x8abe,1,27561,0,0xebc5
,1,39204,4823,0xe954,1,27738,0,0x8998,1,27565,7035,0xebc6
,1,27599,4824,0xebc8,1,39208,1255,0x8ad9,1,27742,0,0x9f87
,1,27743,0,0x8d5d,1,27744,0,0x9272,1,39212,0,0xe956
,1,27746,0,0x9f89,1,39214,0,0xe957,1,27606,0,0xebc9
,1,27607,6579,0xebca,1,27647,10499,0xebcb,1,27653,7084,0xebcc
,1,27752,10502,0x9f91,1,27664,0,0xebcd,1,27754,0,0x9f8a
,1,27699,0,0xebce,1,27737,0,0xebcf,1,27740,7037,0xebd0
,1,27781,0,0xebd4,1,27782,10505,0xebd5,1,27760,10507,0x91bf
,1,27899,10509,0xebd8,1,27762,4272,0x8b82,1,27763,4276,0x9f92
,1,27764,0,0xebd2,1,28005,10523,0xebe6,1,27766,10528,0xebd3
,1,39234,0,0xe95a,1,64070,0,0xebe7,1,28034,7600,0xebe8
,1,27770,4279,0x8c88,1,28052,10529,0xebeb,1,28125,0,0xebf3
,1,27773,0,0x8b44,1,27774,4277,0x9f90,1,28252,0,0xebf8
,1,39243,0,0xe95e,1,27777,4280,0x9f8e,1,27778,10539,0x9f8b
,1,27779,10540,0x9780,1,28254,7039,0xebf9,1,39248,4825,0xe95d
,1,39249,4826,0xe95f,1,39250,0,0xe960,1,27784,0,0x92be
,1,28359,7041,0xec40,1,39253,0,0xe962,1,28432,0,0xec42
,1,39255,1247,0x8bc0,1,27789,5630,0x9f8c,1,28458,0,0xec45
,1,28505,0,0xec49,1,27792,0,0x9f94,1,28514,10542,0xec4c
,1,27794,0,0x9f93,1,27795,0,0x8c42,1,28556,7046,0xec4f
,1,28712,7048,0xec5d,1,27798,0,0x89ab,1,28746,7051,0xec5e
,1,27800,0,0xebd6,1,27801,0,0x8db9,1,27802,0,0x9f8d
,1,27803,0,0x9f8f,1,27804,0,0xebd7,1,28765,0,0xec5f
,1,28805,0,0xec64,1,28836,0,0xec65,1,28900,0,0xec6a
,1,27809,0,0x9676,1,27810,7619,0x91f2,1,28974,0,0xec6f
,1,29121,0,0xec7d,1,29205,0,0xec83,1,29269,0,0xec84
,1,29270,0,0xec85,1,15935,0,0xec86,1,29339,0,0xec88
,1,27818,0,0xebd1,1,27819,0,0x9697,1,43810,0,0xec8c
,1,29647,7052,0xec9a,1,27822,0,0x9f9c,1,29654,0,0xec9b
,1,29657,0,0xec9c,1,27825,0,0x9f9d,1,29667,0,0xec9d
,1,27827,0,0x89cd,1,64075,0,0xeca3,1,16242,5158,0xeccb
,1,43830,5727,0xecfc,1,30896,0,0xed47,1,27832,0,0x95a6
,1,27833,0,0x96fb,1,27834,0,0x9f9f,1,27835,0,0x8ea1
,1,27836,0,0x8fc0,1,27837,0,0x9f98,1,27838,0,0x9f9e
,1,27839,0,0x8988,1,43832,0,0xed48,1,27841,0,0x8bb5
,1,30893,7053,0xed49,1,30948,0,0xed4a,1,27844,0,0x9f95
,1,27845,0,0x9f9a,1,27846,0,0xebd9,1,30962,0,0xed4b
,1,30976,0,0xed4c,1,27849,0,0x90f2,1,27850,0,0x9491
,1,39318,0,0x8ef1,1,27852,4282,0x94e5,1,39320,0,0xe964
,1,39321,0,0x8d81,1,30967,0,0xed4d,1,31004,7055,0xed4e
,1,31022,0,0xed4f,1,31025,0,0xed50,1,27859,5633,0x9f97
,1,27860,0,0xebda,1,27861,0,0x9640,1,31028,7060,0xed51
,1,27863,0,0x9f99,1,64078,10550,0xed53,1,27865,0,0x9fa2
,1,39333,7061,0xe965,1,27867,10551,0x9fa0,1,64079,7064,0xed56
,1,39336,2546,0x8a5d,1,64082,0,0xed5a,1,31160,0,0xed63
,1,27872,0,0xebdb,1,39340,1435,0x946e,1,27874,4283,0x9467
,1,27875,4285,0x8b83,1,31176,0,0xed64,1,27877,0,0x9344
,1,43834,0,0xed66,1,31188,0,0xed67,1,39347,1175,0x9279
,1,39348,7065,0x93e9,1,27882,0,0x9fa3,1,27883,0,0xebdc
,1,31198,7071,0xed68,1,43837,7072,0xed73,1,27886,10552,0xebdd
,1,27887,0,0x9fa1,1,27888,0,0x91d7,1,27889,4287,0x9f96
,1,31438,10553,0xed76,1,27891,0,0x896a,1,31467,10554,0xed77
,1,31485,0,0xed78,1,39361,0,0x949d,1,31533,0,0xed7a
,1,31584,10555,0xed7e,1,39364,0,0x91ca,1,39365,0,0x8977
,1,39366,4828,0x8bec,1,31597,0,0xed80,1,39368,0,0x8bed
,1,31599,10561,0xed81,1,31646,0,0xed83,1,31705,6972,0xed86
,1,31745,6760,0xed87,1,31774,0,0xed89,1,31776,5156,0xed8a
,1,27908,7649,0xebdf,1,39376,0,0x9293,1,39377,0,0xe96d
,1,39378,0,0x8bee,1,16996,10779,0xed8d,1,43845,10563,0xed8e
,1,39381,7080,0x89ed,1,27915,0,0x976d,1,27916,0,0x9fae
,1,39384,0,0xe96c,1,27918,10565,0xebe0,1,31865,0,0xed91
,1,39387,0,0xe96a,1,31887,0,0xed92,1,27922,4288,0x9fad
,1,31892,10568,0xed93,1,39391,0,0xe969,1,31932,0,0xed95
,1,31961,0,0xed97,1,27927,4293,0x90f4,1,31965,0,0xed98
,1,27929,0,0x9faa,1,32019,0,0xed9b,1,27931,0,0x978c
,1,32049,5804,0xed9e,1,32065,6824,0xed9f,1,27934,0,0x93b4
,1,27935,0,0x9fa4,1,32072,5467,0xeda0,1,32092,0,0xeda2
,1,39405,0,0xe96e,1,39406,7665,0xe96f,1,32160,7081,0xeda6
,1,27941,10575,0x92c3,1,39409,7082,0xe970,1,39410,7083,0xe971
,1,32204,0,0xeda9,1,27945,0,0x896b,1,27946,0,0x8d5e
,1,27947,0,0x9fa7,1,32214,5730,0xedaa,1,39416,0,0xe973
,1,27950,10580,0xebe1,1,32227,5861,0xedab,1,39419,0,0xe972
,1,27953,10581,0xebe2,1,27954,0,0x8f46,1,27955,0,0x9fac
,1,39423,7132,0x8f78,1,27957,0,0x9fab,1,27958,4289,0x9fa6
,1,32264,5635,0xedae,1,27960,5636,0x9fa9,1,27961,0,0xebe3
,1,39429,0,0xe976,1,27963,0,0x8a88,1,32273,6211,0xedaf
,1,27965,0,0x9fa8,1,27966,0,0x9468,1,27967,10583,0xebe4
,1,32277,6288,0xedb0,1,27969,0,0x97ac,1,32327,7000,0xedb2
,1,39438,0,0x8b52,1,27972,4290,0x8ff2,1,27973,0,0x90f3
,1,32353,7616,0xedb4,1,39442,0,0x919b,1,39443,0,0x8cb1
,1,32583,0,0xedb7,1,32663,0,0xedba,1,32731,10585,0xedbd
,1,32748,0,0xedbf,1,32750,0,0xedc0,1,39449,0,0xe978
,1,32762,10586,0xedc1,1,32788,0,0xedc3,1,32828,0,0xedc7
,1,32983,0,0xedc9,1,33048,0,0xedcc,1,33098,0,0xedcd
,1,33127,10587,0xedcf,1,33128,0,0xedd0,1,33133,0,0xedd1
,1,27992,0,0xebe5,1,27993,5637,0x9fb4,1,27994,0,0x9fb2
,1,33226,7088,0xedd3,1,27996,0,0x956c,1,39464,0,0x91cb
,1,33231,7090,0xedd4,1,17491,0,0xedd7,1,39467,7091,0xe979
,1,17499,0,0xedd8,1,33396,5638,0xedda,1,28003,5640,0x9faf
,1,28004,0,0x9fb1,1,39472,4832,0x93ab,1,28006,10589,0x8959
,1,33454,0,0xede1,1,33533,0,0xede7,1,28009,0,0x8d5f
,1,28010,0,0x9851,1,33537,10592,0xedea,1,28012,4294,0x8a5c
,1,33594,5641,0xedee,1,28014,0,0x9582,1,33603,0,0xedef
,1,33617,7097,0xedf1,1,33621,0,0xedf2,1,33705,0,0xedf8
,1,39486,0,0xe980,1,28020,0,0x9781,1,39488,0,0xe97d
,1,33743,0,0xedfc,1,28023,4296,0x8a43,1,28024,4297,0x905a
,1,28025,10593,0x9fb3,1,39493,0,0xe97b,1,33761,0,0xee41
,1,33770,0,0xee42,1,33798,0,0xee44,1,64095,0,0xee46
,1,33950,5642,0xee4c,1,33967,7099,0xee4d,1,33984,7101,0xee50
,1,39501,4836,0xe982,1,33986,5645,0xee51,1,34083,0,0xee55
,1,28037,0,0x9fb8,1,34095,0,0xee56,1,28039,5648,0xebe9
,1,28040,0,0x8fc1,1,28041,0,0xebea,1,39509,7102,0xe981
,1,34188,0,0xee5c,1,28044,4299,0x974f,1,34210,7104,0xee5e
,1,28046,0,0x9fb5,1,39514,0,0x8bc1,1,39515,0,0xe983
,1,34224,7105,0xee5f,1,34285,0,0xee62,1,28051,0,0x9fb0
,1,39519,4838,0xe985,1,28053,0,0x9fb6,1,34322,0,0xee63
,1,39522,0,0xe986,1,43889,7106,0xee68,1,28057,4302,0x97dc
,1,39525,0,0xe987,1,28059,0,0x9393,1,28060,0,0x93c0
,1,34360,0,0xee6b,1,39529,0,0xe989,1,39530,0,0xe98b
,1,39531,0,0xe98a,1,63798,0,0xee6e,1,34402,0,0xee6f
,1,34412,0,0xee71,1,34421,0,0xee72,1,34456,6056,0xee73
,1,34488,0,0xee74,1,34554,6136,0xee75,1,34556,0,0xee76
,1,34557,6284,0xee77,1,28074,0,0xebec,1,34571,6089,0xee78
,1,28076,0,0xebed,1,34673,6085,0xee79,1,34695,6086,0xee7a
,1,28079,0,0x8a55,1,34696,6604,0xee7b,1,34732,5468,0xee7c
,1,28082,0,0x8974,1,34733,0,0xee7d,1,34741,7127,0xee7e
,1,28085,0,0x9fbc,1,34774,6471,0xee81,1,34796,0,0xee82
,1,28088,0,0x9fbf,1,34822,0,0xee83,1,34826,6104,0xee84
,1,34832,10479,0xee85,1,28092,0,0x97c1,1,34836,7013,0xee86
,1,34847,8304,0xee87,1,28095,0,0xebee,1,28096,0,0x9784
,1,34968,0,0xee88,1,34986,6149,0xee89,1,35018,0,0xee8a
,1,28100,0,0xebef,1,28101,0,0x9fc6,1,28102,0,0x9fc0
,1,28103,0,0x9fbd,1,35022,6152,0xee8b,1,43900,6161,0xee8c
,1,35061,0,0xee8d,1,28107,0,0x97d2,1,28108,0,0x9fc3
,1,35100,0,0xee8e,1,35096,7136,0xee90,1,35097,5950,0xee91
,1,35098,5840,0xee92,1,28113,0,0x8f69,1,28114,0,0x9fc5
,1,35111,0,0xee93,1,35120,0,0xee94,1,28117,0,0x9fca
,1,28118,0,0xebf0,1,35122,6159,0xee95,1,28120,0,0x9391
,1,28121,0,0x9fc8,1,28122,0,0xebf1,1,28123,0,0xebf2
,1,35129,6480,0xee96,1,39592,4839,0x8d9c,1,28126,0,0x9fc2
,1,35136,7143,0xee97,1,35220,6209,0xee98,1,28129,0,0x9257
,1,39597,0,0xe98c,1,35284,0,0xee9a,1,28132,10620,0x9fc9
,1,39600,0,0xe98d,1,28134,0,0x9fbe,1,35301,0,0xee9b
,1,28136,0,0x9fc4,1,35318,6227,0xee9c,1,28138,0,0x9fcb
,1,28139,5649,0x88fa,1,28140,10624,0x9fc1,1,39608,0,0x8a5b
,1,28142,10634,0x9fcc,1,35346,0,0xee9d,1,35349,5483,0xee9e
,1,28145,4303,0x905b,1,35362,0,0xee9f,1,28147,0,0x8f7e
,1,35383,0,0xeea0,1,28149,4307,0x95a3,1,35399,5651,0xeea1
,1,28151,0,0x8dac,1,35406,5652,0xeea2,1,39620,2552,0x9091
,1,28154,0,0x9fc7,1,28155,10636,0x9359,1,28156,0,0xebf4
,1,35421,6279,0xeea3,1,35425,0,0xeea4,1,35445,0,0xeea5
,1,35449,6628,0xeea6,1,35495,0,0xeea7,1,35536,0,0xeea8
,1,35551,5655,0xeea9,1,39631,0,0xe990,1,28165,10639,0x90b4
,1,39633,0,0xe991,1,28167,10640,0x8a89,1,28168,4308,0x8dcf
,1,28169,4309,0x8fc2,1,28170,10642,0x9fbb,1,28171,5657,0x8f61
,1,35572,8439,0xeeaa,1,39640,0,0x8d82,1,35574,0,0xeeab
,1,35654,0,0xeeaf,1,35668,0,0xeeb0,1,35673,10647,0xeeb1
,1,35689,0,0xeeb2,1,28179,4310,0x8c6b,1,39647,0,0xe995
,1,28181,10649,0x9fba,1,35741,6137,0xeeb3,1,39650,7113,0xe996
,1,39651,0,0xe997,1,28185,0,0x9fd0,1,28186,7709,0x8f8d
,1,28187,4311,0x8cb8,1,35913,6389,0xeeb4,1,28189,0,0x9fdf
,1,35944,7148,0xeeb5,1,39658,2558,0x94af,1,28192,4315,0x8b94
,1,28193,0,0x936e,1,39661,7150,0x9545,1,28195,4316,0x9fd4
,1,28196,4314,0x9fdd,1,28197,0,0x88ad,1,28198,4320,0x8951
,1,64100,7151,0xeeb6,1,36065,0,0xeeb7,1,28201,4319,0x89b7
,1,36084,0,0xeeb8,1,28203,0,0x9fd6,1,28204,4321,0x91aa
,1,28205,0,0x9fcd,1,28206,5658,0x9fcf,1,28207,0,0x8d60
,1,39675,0,0xe99f,1,36088,0,0xeeb9,1,36094,7155,0xeeba
,1,36114,10333,0xeebc,1,28212,0,0xebf6,1,36123,0,0xeebd
,1,36271,7147,0xeebe,1,36302,5659,0xeebf,1,28216,10669,0x9fe0
,1,36305,5287,0xeec0,1,28218,0,0x9fdb,1,39686,0,0xe9a0
,1,36311,6106,0xeec1,1,36384,5661,0xeec2,1,28222,0,0x9fd3
,1,36387,6135,0xeec3,1,36413,5790,0xeec4,1,36464,6490,0xeec5
,1,36475,0,0xeec6,1,28227,0,0x9fda,1,28228,0,0xebf7
,1,43916,10675,0xeec7,1,36544,6496,0xeec8,1,36602,10679,0xeeca
,1,36638,7160,0xeecb,1,36653,7163,0xeecc,1,28234,0,0x96a9
,1,36662,5749,0xeecd,1,36692,7168,0xeece,1,28237,4325,0x9fd8
,1,28238,0,0x9fdc,1,39706,7169,0xe9a2,1,43919,0,0xeecf
,1,36774,7170,0xeed0,1,36789,0,0xeed1,1,36836,7171,0xeed2
,1,39711,7172,0xe9a3,1,36840,5663,0xeed3,1,28246,0,0x8cce
,1,39714,7174,0xe9a4,1,28248,4327,0x8fc3,1,36846,0,0xeed4
,1,39717,0,0xe9a6,1,28251,0,0x9258,1,39719,4842,0xe9a7
,1,39720,0,0xe9a8,1,39721,4847,0xe9a9,1,28255,4332,0x9fd2
,1,36872,10689,0xeed5,1,36909,0,0xeed6,1,64103,5666,0xeed7
,1,39726,7175,0xe9ab,1,39727,0,0xe9ac,1,37000,6557,0xeed8
,1,39729,0,0x9f54,1,28263,4333,0x974e,1,37013,10692,0xeed9
,1,37015,0,0xeeda,1,37017,8374,0xeedb,1,28267,0,0x9fd5
,1,37019,0,0xeedc,1,37026,0,0xeedd,1,28270,0,0x9fce
,1,28271,0,0x9392,1,39739,0,0xe2f6,1,39740,0,0x8b53
,1,28274,10694,0x9fd1,1,37043,0,0xeede,1,37054,0,0xeedf
,1,37060,0,0xeee0,1,39745,2555,0x8a40,1,39746,0,0x8db0
,1,39747,0,0xe9af,1,39748,0,0xe9ae,1,39749,0,0x96a3
,1,37061,0,0xeee1,1,37063,5739,0xeee2,1,37079,0,0xeee3
,1,28286,0,0x9870,1,28287,7715,0x8ebc,1,28288,10696,0x969e
,1,37085,0,0xeee4,1,28290,4334,0x9fe1,1,39758,0,0xe9b2
,1,39759,0,0xe9b0,1,37086,0,0xeee5,1,39761,0,0xe9b3
,1,37103,0,0xeee6,1,37108,6109,0xeee7,1,39764,0,0x9682
,1,37140,7717,0xeee9,1,37141,5805,0xeeea,1,28300,0,0x94ac
,1,39768,0,0xe9b4,1,37142,0,0xeeeb,1,39770,2559,0x8b9b
,1,28304,0,0x8cb9,1,37154,5844,0xeeec,1,37155,5589,0xeeed
,1,37159,5667,0xeeee,1,37167,6968,0xeeef,1,37169,7650,0xeef0
,1,28310,0,0x8f80,1,37172,0,0xeef1,1,28312,10697,0x9fe3
,1,37181,7802,0xeef2,1,37192,6582,0xeef3,1,37211,5670,0xeef4
,1,28316,0,0x97ad,1,28317,0,0x8d61,1,37251,0,0xeef5
,1,28319,0,0x9ff0,1,37278,10705,0xeef6,1,37292,10706,0xeef7
,1,28322,0,0x88ec,1,37297,0,0xeef8,1,39791,0,0x9844
,1,28325,0,0x9fee,1,37308,0,0xeef9,1,37335,5672,0xeefa
,1,37371,10709,0xeefb,1,39796,0,0xe9b5,1,28330,5673,0x9fe2
,1,28331,10711,0xebfa,1,37348,10713,0xeefc,1,37349,0,0xef40
,1,37357,5669,0xef41,1,28335,0,0x9fe8,1,37361,0,0xef42
,1,28337,7733,0xebfb,1,28338,0,0x9fea,1,37383,0,0xef43
,1,37392,7180,0xef44,1,37432,0,0xef45,1,28342,0,0x976e
,1,28343,0,0x9fe5,1,39811,0,0xe9b7,1,37433,5674,0xef46
,1,28346,0,0x934d,1,37434,0,0xef47,1,37436,0,0xef48
,1,28349,0,0x9fe7,1,37440,7734,0xef49,1,37443,5679,0xef4a
,1,37455,6616,0xef4b,1,28353,0,0xebfc,1,28354,0,0x9fef
,1,39822,7183,0x88bc,1,28356,5680,0x9fe9,1,28357,0,0x96c5
,1,39825,0,0xe9b8,1,39826,4852,0x95a9,1,39827,0,0xe9b6
,1,28361,0,0x9fe4,1,37496,7185,0xef4c,1,28363,4336,0x8ea0
,1,28364,4337,0x9ffc,1,37512,0,0xef4d,1,28366,0,0xec41
,1,37570,6037,0xef4e,1,37579,5610,0xef4f,1,28369,0,0x8a8a
,1,37580,7748,0xef50,1,28371,5683,0x9fe6,1,28372,4339,0x9feb
,1,28373,4340,0x9fec,1,37587,6112,0xef51,1,37600,5616,0xef52
,1,37631,0,0xef53,1,37636,7186,0xef54,1,37663,0,0xef55
,1,37665,10714,0xef56,1,37669,5684,0xef57,1,28381,4341,0x91ea
,1,28382,0,0x91d8,1,39850,0,0x968e,1,39851,0,0x8e4c
,1,37704,6157,0xef58,1,39853,0,0x8df8,1,39854,0,0x914e
,1,37705,0,0xef59,1,37706,0,0xef5a,1,37732,6427,0xef5b
,1,37733,10731,0xef5c,1,37738,0,0xef5d,1,39860,0,0xe9be
,1,37744,0,0xef5e,1,37787,7188,0xef5f,1,28396,0,0x9ff4
,1,37795,6082,0xef60,1,39865,0,0xe9c1,1,28399,0,0x9ffa
,1,37818,0,0xef61,1,37830,0,0xef62,1,28402,0,0x9ff8
,1,37854,6370,0xef63,1,28404,0,0x9348,1,39872,0,0xe9bf
,1,37855,5686,0xef64,1,28407,0,0xe042,1,28408,0,0x9ff5
,1,37892,6463,0xef65,1,37885,0,0xef66,1,39878,7191,0xe9c2
,1,37939,10735,0xef67,1,37962,0,0xef68,1,39881,2562,0x8cef
,1,28415,4343,0x9fde,1,37987,0,0xef69,1,28417,0,0x8b99
,1,28418,0,0x9559,1,37995,5687,0xef6a,1,39887,0,0xe9c3
,1,38001,0,0xef6b,1,28422,4346,0x8ebd,1,39890,0,0xe9c5
,1,38002,0,0xef6c,1,28425,4349,0x8d97,1,38286,6596,0xef6d
,1,39894,0,0x8e49,1,38303,7193,0xef6e,1,38310,6329,0xef6f
,1,38313,0,0xef70,1,28431,0,0x9852,1,39899,4854,0x91e2
,1,28433,0,0x9ff2,1,38316,10748,0xef71,1,28435,0,0xe041
,1,28436,0,0x8989,1,28437,0,0x9186,1,39905,0,0xe9ca
,1,39906,0,0xe9c7,1,39907,7194,0xe9c6,1,39908,0,0xe9c8
,1,28442,5688,0xec43,1,38326,0,0xef72,1,38333,10754,0xef73
,1,39912,0,0x8c7e,1,38347,0,0xef74,1,38352,10764,0xef75
,1,28448,10767,0x9499,1,38355,0,0xef76,1,28450,0,0x8abf
,1,28451,10776,0x97f8,1,18864,10775,0xef77,1,39920,7195,0xe9ce
,1,39921,0,0xe9cd,1,39922,0,0xe9cc,1,38362,10739,0xef78
,1,38366,0,0xef79,1,39925,4857,0x88b1,1,28459,0,0x969f
,1,28460,10783,0x92d0,1,38488,5689,0xef7a,1,38532,0,0xef7b
,1,28463,4979,0xec46,1,63964,0,0xef7c,1,28465,0,0x9ff9
,1,28466,5690,0x9ffb,1,28467,0,0xec47,1,38557,0,0xef7d
,1,38564,5691,0xef7e,1,38565,7199,0xef80,1,38610,0,0xef81
,1,28472,0,0x9151,1,39940,0,0xe9d8,1,38622,0,0xef82
,1,39942,0,0xe9d4,1,64104,7201,0xef83,1,39944,0,0xe9d5
,1,28478,4353,0xe040,1,28479,4357,0x9ff7,1,38633,0,0xef84
,1,28481,4355,0x9ff1,1,39949,0,0x8a82,1,38639,0,0xef85
,1,38707,0,0xef86,1,28485,1643,0x8ac1,1,38715,0,0xef87
,1,39954,0,0xe9d6,1,39955,0,0xe9d2,1,39956,0,0xe9d0
,1,39957,0,0xe9cf,1,38733,10786,0xef88,1,38734,6949,0xef89
,1,38735,10787,0xef8a,1,38746,10789,0xef8b,1,38766,10791,0xef8c
,1,39963,0,0xe9da,1,28497,0,0xec48,1,38771,7202,0xef8d
,1,38805,0,0xef8e,1,28500,0,0x8c89,1,38830,0,0xef8f
,1,39969,0,0xe9dd,1,38842,7205,0xef90,1,28504,5692,0xe04e
,1,39972,4858,0xe9dc,1,39973,7206,0xe9db,1,28507,0,0xe049
,1,28508,10808,0x90f6,1,38849,5878,0xef91,1,28510,10813,0xec4a
,1,28511,10818,0x8a83,1,38857,7210,0xef92,1,28513,0,0xec4b
,1,39981,4860,0x9568,1,39982,0,0xe9d9,1,39983,719,0x88f1
,1,39984,0,0xe9de,1,28518,10822,0xe052,1,39986,0,0xe9e0
,1,38878,7211,0xef93,1,38875,6978,0xef94,1,38900,10149,0xef95
,1,38922,10829,0xef97,1,38942,10831,0xef98,1,28525,0,0xe04b
,1,39993,1193,0x8a8f,1,28527,4351,0xe048,1,39995,1202,0x8956
,1,38955,9595,0xef99,1,38960,7651,0xef9a,1,39998,0,0xe9e2
,1,28532,0,0xe06b,1,64106,0,0xef9b,1,38994,5693,0xef9c
,1,38995,9794,0xef9d,1,28536,0,0xe045,1,38998,0,0xef9e
,1,28538,10833,0xe044,1,40006,0,0xe9e1,1,28540,4362,0xe04d
,1,40008,7215,0x924c,1,28542,10848,0xec4d,1,38999,10849,0xef9f
,1,28544,0,0xe047,1,28545,0,0xe046,1,28546,0,0xe04c
,1,39001,8306,0xefa0,1,28548,5694,0x909f,1,39002,5697,0xefa1
,1,28550,0,0xe043,1,40018,0,0x9690,1,28552,5698,0xec4e
,1,39013,9922,0xefa3,1,39020,0,0xefa4,1,39098,0,0xefa5
,1,40023,4863,0x97d8,1,28557,10864,0xec50,1,28558,0,0xe04f
,1,40026,0,0xe9e3,1,39112,7217,0xefa6,1,28561,7814,0xe050
,1,39143,5845,0xefa7,1,39256,0,0xefa8,1,28564,10879,0xec51
,1,40032,0,0xe9e4,1,39326,7219,0xefa9,1,28567,0,0x8ac0
,1,39426,5700,0xefaa,1,39427,0,0xefab,1,39460,5701,0xefac
,1,39469,0,0xefad,1,40039,0,0xe9e5,1,39470,0,0xefae
,1,39480,10881,0xefaf,1,39498,7222,0xefb0,1,28576,10889,0xec52
,1,28577,7827,0xe055,1,39502,10890,0xefb1,1,28579,10892,0xe054
,1,28580,0,0xe056,1,39506,7223,0xefb2,1,39606,0,0xefb3
,1,28583,10895,0xec53,1,39617,7145,0xefb4,1,39619,0,0xefb5
,1,28586,10896,0xe059,1,40054,0,0xe9e6,1,39630,5703,0xefb6
,1,40056,0,0xe9e7,1,39638,0,0xefb7,1,39673,7226,0xefb8
,1,39682,6286,0xefb9,1,28593,0,0x9362,1,39688,0,0xefba
,1,28595,0,0xe053,1,39712,0,0xefbb,1,19479,7228,0xefbc
,1,28598,0,0xec54,1,39725,0,0xefbd,1,39774,6541,0xefbe
,1,28601,0,0xe057,1,39801,0,0xefbf,1,39782,0,0xefc0
,1,28604,0,0xec55,1,39794,7179,0xefc1,1,39797,7220,0xefc2
,1,39812,7737,0xefc3,1,28608,0,0x8c83,1,28609,0,0x91f7
,1,28610,0,0xe051,1,28611,0,0x945a,1,39818,7181,0xefc4
,1,39823,0,0xefc5,1,28614,0,0xe058,1,28615,0,0xec56
,1,39838,7230,0xefc6,1,39847,7233,0xefc7,1,28618,0,0xec57
,1,39873,0,0xefc8,1,39886,0,0xefc9,1,39909,7844,0xefca
,1,39928,0,0xefcb,1,39933,0,0xefcc,1,39936,7187,0xefcd
,1,39971,0,0xefce,1,40001,0,0xefcf,1,40015,0,0xefd0
,1,28628,0,0xe05d,1,28629,0,0xe05b,1,40016,0,0xefd1
,1,40019,0,0xefd2,1,28632,0,0xe05e,1,40035,6164,0xefd3
,1,40037,7234,0xefd4,1,28635,0,0xe061,1,40055,6961,0xefd5
,1,40259,0,0xefd8,1,40263,7237,0xefd9,1,28639,0,0xe05a
,1,28640,0,0x8d8a,1,28641,0,0x9447,1,40342,0,0xefdf
,1,40380,0,0xefe2,1,28644,0,0x9fb7,1,40423,0,0xefe5
,1,40455,0,0xefe6,1,40572,0,0xefe8,1,40657,0,0xeff0
,1,40866,7239,0xeff7,1,23032,8308,0xeff8,1,28651,0,0x9794
,1,28652,0,0xe05c,1,23643,9199,0xeff9,1,28654,0,0xe060
,1,28655,0,0x91f3,1,28656,0,0xec59,1,28657,0,0xe05f
,1,24183,0,0xeffa,1,28659,0,0xe04a,1,32363,0,0xeffc
,1,28661,0,0xec5a,1,28662,7864,0xe889,1,43681,0,0xf040
,1,19970,0,0xf041,1,28665,0,0xec58,1,28666,0,0xe064
,1,19983,0,0xf042,1,19986,7242,0xf043,1,20011,5794,0xf045
,1,28670,0,0xe068,1,20014,6385,0xf046,1,20032,7243,0xf047
,1,28673,0,0xe066,1,20039,0,0xf048,1,43683,0,0xf04a
,1,13318,0,0xf04c,1,28677,0,0xec5b,1,28678,0,0xec5c
,1,20058,7244,0xf04e,1,20125,0,0xf050,1,28681,0,0xe062
,1,13356,6779,0xf051,1,28683,0,0xe063,1,13358,0,0xf052
,1,20155,0,0xf054,1,43685,7870,0xf055,1,28687,0,0xe067
,1,20156,0,0xf056,1,28689,0,0xe065,1,20176,7756,0xf059
,1,20203,0,0xf05a,1,20209,7773,0xf05c,1,28693,0,0x956d
,1,20213,6672,0xf05d,1,20224,6837,0xf05e,1,28696,0,0xe06d
,1,20308,0,0xf063,1,40165,2564,0x92b9,1,28699,10898,0xe069
,1,40167,0,0xe9e8,1,28701,0,0xe06c,1,28702,1337,0x93d2
,1,28703,0,0xe06e,1,40171,0,0xe9ed,1,40172,0,0xe9e9
,1,20312,0,0xf064,1,20344,7249,0xf067,1,20354,0,0xf06a
,1,40176,0,0xe9ea,1,28710,0,0x9295,1,28711,10899,0x91eb
,1,40179,4864,0x9650,1,40180,0,0x96c2,1,20357,0,0xf06b
,1,40182,0,0x93ce,1,28716,0,0x90a3,1,20378,0,0xf06d
,1,20454,0,0xf06e,1,20414,0,0xf070,1,28720,0,0xe06f
,1,20421,0,0xf071,1,28722,0,0xe071,1,13418,0,0xf075
,1,20466,0,0xf076,1,20499,0,0xf079,1,20508,0,0xf07a
,1,13416,10904,0xf07d,1,40195,0,0xe9ee,1,20546,7251,0xf07e
,1,20558,7253,0xf081,1,40198,0,0xe9ef,1,40199,7254,0x93bc
,1,40200,0,0xe9ec,1,28734,4364,0xe070,1,20567,0,0xf083
,1,20579,10902,0xf084,1,20582,0,0xf085,1,20586,0,0xf086
,1,40206,7255,0x89a8,1,20592,0,0xf087,1,20643,0,0xf088
,1,20616,10907,0xf089,1,40210,7256,0xe9f7,1,20627,0,0xf08b
,1,20630,0,0xf08d,1,40213,4866,0xe9f6,1,20650,7257,0xf08f
,1,28748,10909,0x9ff3,1,43687,10913,0xf090,1,28750,0,0xec61
,1,20676,0,0xf094,1,40219,0,0x8995,1,28753,0,0xe072
,1,40221,0,0xefd6,1,40222,0,0xefd7,1,40223,7260,0xe9f4
,1,20723,0,0xf096,1,43689,0,0xf097,1,20686,0,0xf098
,1,28760,4367,0x93e5,1,43688,0,0xf099,1,20692,0,0xf09a
,1,40230,0,0xe9f1,1,20705,0,0xf09c,1,40232,4872,0x8a9b
,1,28766,0,0xec60,1,40234,0,0xe9f0,1,40235,0,0x8eb0
,1,40236,0,0x89a7,1,20713,0,0xf09d,1,28771,0,0xe073
,1,28772,0,0xec62,1,13458,7142,0xf09e,1,43690,0,0xf141
,1,43691,10927,0xf144,1,20832,6706,0xf145,1,43692,7265,0xf146
,1,20851,10868,0xf147,1,28779,0,0x89ce,1,20875,7266,0xf149
,1,13500,0,0xf14a,1,20899,7267,0xf14c,1,28783,0,0x9394
,1,28784,561,0x8a44,1,20909,7777,0xf14d,1,13511,6483,0xf14e
,1,40254,0,0xe9fa,1,40255,0,0xe9f9,1,28789,0,0xec63
,1,40257,7268,0xe9f8,1,43693,10934,0xf150,1,28792,5704,0x8b84
,1,40260,0,0xe9f5,1,43694,7085,0xf151,1,40262,0,0xe9fb
,1,28796,5705,0x8edc,1,28797,4368,0x8dd0,1,20979,7270,0xf152
,1,20994,9670,0xf154,1,21014,10159,0xf156,1,43695,7272,0xf157
,1,21100,0,0xf15a,1,21122,0,0xf15d,1,43696,0,0xf15e
,1,40272,4873,0xea44,1,40273,7278,0xea43,1,40274,0,0xefda
,1,43697,7841,0xf160,1,28809,0,0x9846,1,28810,0,0x9086
,1,21178,8809,0xf164,1,21179,0,0xf165,1,13599,9109,0xf167
,1,28814,4374,0x898a,1,21201,7888,0xf168,1,21258,0,0xf16b
,1,40284,0,0x894c,1,28818,4370,0xe075,1,40286,0,0xea41
,1,21259,10939,0xf16c,1,40288,7279,0x8d94,1,40289,0,0x96b7
,1,21301,7280,0xf16e,1,40291,0,0xefdb,1,28825,4371,0xe074
,1,43699,0,0xf171,1,43700,0,0xf172,1,21356,0,0xf174
,1,21412,0,0xf176,1,21428,10952,0xf177,1,43701,7285,0xf178
,1,21440,10955,0xf17a,1,40300,0,0x9651,1,13661,0,0xf17c
,1,13662,6353,0xf17d,1,40303,4874,0xea4a,1,40304,0,0xefdc
,1,21466,0,0xf180,1,40306,0,0xea46,1,13667,6808,0xf181
,1,21540,0,0xf185,1,21544,0,0xf186,1,28843,10973,0xec66
,1,28844,10975,0xe078,1,28845,0,0x9259,1,28846,0,0xe07b
,1,28847,4376,0xe076,1,13678,0,0xf187,1,40316,7287,0xefdd
,1,21571,0,0xf188,1,28851,10984,0xe07a,1,21606,0,0xf18a
,1,21642,0,0xf18c,1,21653,0,0xf18e,1,28855,0,0xec67
,1,28856,0,0xe079,1,28857,7924,0x935f,1,28858,0,0x88d7
,1,21664,7281,0xf18f,1,40327,0,0xea48,1,21670,0,0xf190
,1,40329,0,0xea47,1,40330,0,0xefde,1,21677,0,0xf191
,1,21678,9107,0xf192,1,21687,10987,0xf193,1,21690,8006,0xf194
,1,40335,0,0x8c7b,1,21695,0,0xf195,1,43703,9174,0xf197
,1,21743,0,0xf199,1,28872,0,0x97f3,1,21747,0,0xf19b
,1,21820,7290,0xf19f,1,28875,5709,0xe07d,1,21825,0,0xf1a0
,1,13734,0,0xf1a1,1,21831,10993,0xf1a2,1,28879,4379,0x8947
,1,13736,6515,0xf1a4,1,21856,7291,0xf1a5,1,21857,0,0xf1a6
,1,13765,6659,0xf1ad,1,28884,0,0xec68,1,21970,0,0xf1ae
,1,43706,10994,0xf1af,1,13786,0,0xf1b6,1,28888,0,0xec69
,1,28889,4381,0xe080,1,21986,0,0xf1b7,1,22056,0,0xf1ba
,1,22023,7293,0xf1bc,1,28893,0,0xe07e,1,40361,0,0xea4e
,1,28895,0,0xe07c,1,40363,0,0xea49,1,40364,0,0xefe1
,1,22032,7939,0xf1bd,1,22064,0,0xf1be,1,40367,4875,0xe9f2
,1,22071,0,0xf1bf,1,13812,6777,0xf1c0,1,40370,7942,0xea4f
,1,22112,0,0xf1c6,1,40372,0,0x92df,1,22125,10996,0xf1c7
,1,13829,0,0xf1c8,1,22184,7289,0xf1ce,1,40376,8042,0xea53
,1,22194,10997,0xf1d0,1,40378,0,0xea54,1,40379,0,0xea52
,1,28913,5710,0xe077,1,22221,10999,0xf1d2,1,22239,0,0xf1d3
,1,22262,10995,0xf1d5,1,40384,0,0xefe0,1,40385,0,0xea51
,1,40386,0,0xea57,1,43709,11000,0xf1d7,1,28921,4383,0x9642
,1,22293,0,0xf1d8,1,40390,0,0xea55,1,43711,11002,0xf1da
,1,28925,0,0xe082,1,22313,11008,0xf1db,1,43713,0,0xf1dc
,1,43712,0,0xf1e1,1,22395,0,0xf1e7,1,43714,0,0xf1e8
,1,43715,7294,0xf1e9,1,40399,7295,0xea56,1,22444,0,0xf1ea
,1,22430,7298,0xf1ed,1,22440,7299,0xf1ee,1,40403,7301,0xea59
,1,28937,0,0xe081,1,43717,0,0xf1f0,1,43718,11011,0xf1f2
,1,40407,7302,0xefe3,1,43719,0,0xf1f3,1,40409,0,0xea58
,1,28943,11014,0xec6b,1,22512,0,0xf1f6,1,13898,9267,0xf1f7
,1,22523,0,0xf1f9,1,43720,0,0xf244,1,28948,11019,0x898b
,1,22601,11022,0xf245,1,22604,0,0xf246,1,22631,0,0xf247
,1,22666,11023,0xf248,1,28953,0,0xe084,1,28954,0,0x95b0
,1,40422,0,0xea5b,1,28956,5712,0xe083,1,22669,0,0xf24a
,1,28958,5066,0xec6d,1,22671,0,0xf24b,1,28960,0,0xec6e
,1,28961,0,0x96b3,1,40429,0,0xea5c,1,22685,0,0xf24e
,1,40431,0,0xea5d,1,43722,0,0xf251,1,28966,5068,0x8fc5
,1,40434,0,0x9868,1,22754,0,0xf254,1,22789,11024,0xf257
,1,43723,0,0xf25d,1,28971,0,0xec6c,1,13969,6822,0xf25e
,1,40440,0,0xea5a,1,40441,4878,0x91e9,1,40442,7304,0x8deb
,1,28976,0,0xec70,1,13977,7310,0xf260,1,40445,0,0xea5e
,1,13974,0,0xf262,1,43724,8413,0xf263,1,43725,0,0xf266
,1,28982,0,0x9152,1,22902,11035,0xf268,1,22943,0,0xf26a
,1,22958,0,0xf26b,1,22972,8293,0xf26c,1,22984,0,0xf26d
,1,28988,5713,0x8fc4,1,22989,0,0xf26e,1,23006,0,0xf26f
,1,23011,11031,0xf270,1,23012,0,0xf271,1,23015,8142,0xf272
,1,23022,11038,0xf273,1,43726,0,0xf274,1,43727,0,0xf275
,1,14031,8145,0xf276,1,28998,0,0xec71,1,28999,0,0xec72
,1,23052,8315,0xf277,1,29001,0,0x97f9,1,40469,7315,0xefe7
,1,23053,0,0xf278,1,29004,0,0xe08a,1,23063,0,0xf279
,1,29006,11040,0x90f7,1,40474,0,0xea5f,1,40475,7312,0xea60
,1,29009,11041,0xec73,1,29010,11042,0xec75,1,40478,7316,0xea61
,1,23079,6501,0xf27a,1,29013,0,0xe086,1,29014,0,0xe08b
,1,23085,6282,0xf27b,1,23125,8649,0xf27c,1,29017,0,0x898c
,1,23141,0,0xf27d,1,23162,7123,0xf27e,1,29020,8156,0xec76
,1,23179,8335,0xf280,1,23196,6107,0xf281,1,23199,6635,0xf282
,1,29024,0,0xec77,1,23200,8334,0xf283,1,29026,0,0xe089
,1,23202,6102,0xf284,1,29028,0,0x9481,1,29029,0,0xe085
,1,29030,0,0xe088,1,29031,0,0x8fc6,1,29032,0,0xec78
,1,29033,0,0x94cf,1,23217,8357,0xf285,1,23219,8401,0xf286
,1,29036,0,0xe08c,1,23221,6633,0xf287,1,29038,0,0x8ecf
,1,23226,6162,0xf288,1,23231,6289,0xf289,1,23258,0,0xf28a
,1,23260,6574,0xf28b,1,23264,0,0xf28c,1,23269,7664,0xf28d
,1,23280,6640,0xf28e,1,23278,0,0xf28f,1,23285,6637,0xf290
,1,23296,0,0xf291,1,23304,0,0xf292,1,23319,0,0xf293
,1,23348,0,0xf294,1,23341,0,0xf295,1,29053,0,0x90f8
,1,23372,8179,0xf296,1,23378,0,0xf297,1,23400,7321,0xf298
,1,23407,0,0xf299,1,23420,0,0xf29a,1,23423,6558,0xf29b
,1,29060,0,0xe08f,1,29061,0,0xec7a,1,23425,0,0xf29c
,1,29063,0,0xec7b,1,29064,0,0xe087,1,23428,7706,0xf29d
,1,29066,0,0x8c46,1,43729,0,0xf29e,1,23446,6647,0xf09f
,1,23468,8685,0xf0a0,1,14177,6362,0xf0a1,1,29071,0,0xe08d
,1,23488,0,0xf0a2,1,14178,0,0xf0a3,1,29074,0,0xec7c
,1,23502,6648,0xf0a4,1,29076,0,0x976f,1,29077,0,0xe090
,1,23510,7319,0xf0a5,1,14188,6839,0xf0a6,1,14187,0,0xf0a7
,1,29081,0,0xeaa4,1,23537,7129,0xf0a8,1,23549,7322,0xf0a9
,1,14197,0,0xf0aa,1,23555,7133,0xf0ab,1,23593,6660,0xf0ac
,1,29087,8178,0x8f6e,1,23600,7324,0xf0ad,1,43730,0,0xf0ae
,1,23647,7325,0xf0af,1,23651,0,0xf0b0,1,23655,0,0xf0b1
,1,23656,0,0xf0b2,1,23657,0,0xf0b3,1,23664,7159,0xf0b4
,1,29096,0,0xe091,1,43731,7212,0xf0b5,1,40565,0,0xea62
,1,43733,6578,0xf0b6,1,29100,0,0xe092,1,40568,0,0x8cb2
,1,40569,7326,0xea63,1,23676,7329,0xf0b7,1,43734,11048,0xf0b8
,1,29105,5714,0x944d,1,40573,0,0xea64,1,43735,7330,0xf0b9
,1,40575,0,0x8ead,1,23688,11069,0xf0ba,1,40577,0,0xea65
,1,23690,0,0xf0bb,1,14273,9609,0xf0bc,1,29113,0,0xe094
,1,29114,11074,0xec7e,1,43737,0,0xf0bd,1,43738,0,0xf0be
,1,40584,0,0xea66,1,29118,0,0xe095,1,23712,0,0xf0bf
,1,40587,0,0xea67,1,40588,4879,0xea68,1,23714,0,0xf0c0
,1,29123,0,0x9452,1,29124,0,0xec80,1,23718,0,0xf0c1
,1,40593,0,0xea6b,1,40594,0,0xea69,1,29128,1626,0x9395
,1,29129,0,0xe097,1,40597,0,0xea6a,1,23719,7332,0xf0c2
,1,40599,0,0x97ed,1,23725,9720,0xf0c4,1,29134,0,0xe099
,1,23733,8500,0xf0c5,1,29136,11075,0x97d3,1,43739,7848,0xf0c6
,1,29138,4384,0xe096,1,40606,7333,0xefe9,1,40607,2567,0x97d9
,1,29141,0,0x898d,1,23753,0,0xf0c7,1,29143,0,0xe093
,1,43740,7086,0xf0c8,1,40612,7335,0xefea,1,40613,0,0xea6d
,1,40614,0,0x949e,1,43742,7336,0xf0c9,1,23814,11078,0xf0ca
,1,40617,0,0xea6e,1,29151,4391,0x9a7a,1,29152,0,0xe09a
,1,40620,0,0xefeb,1,40621,0,0xea71,1,23824,0,0xf0cb
,1,40623,0,0xefec,1,29157,0,0x9187,1,29158,0,0x8e57
,1,29159,0,0xe09c,1,23851,7144,0xf0cc,1,40628,0,0xefed
,1,40629,0,0xefee,1,23837,0,0xf0cd,1,29164,0,0xe09b
,1,29165,4386,0x9043,1,40633,1822,0x8d8d,1,40634,0,0x96cb
,1,40635,8238,0x9683,1,40636,0,0x9bf5,1,23840,8576,0xf0ce
,1,40638,0,0x9f80,1,40639,0,0x969b,1,29173,0,0xe09d
,1,23844,0,0xf0cf,1,23846,8584,0xf0d0,1,40643,0,0xefef
,1,40644,2568,0x89a9,1,23857,0,0xf0d1,1,29179,0,0xe08e
,1,29180,0,0xe09e,1,23865,0,0xf0d2,1,29182,0,0xec81
,1,29183,0,0xe0a0,1,29184,0,0xec82,1,40652,0,0xea73
,1,40653,0,0x8b6f,1,40654,0,0xea74,1,40655,0,0xea75
,1,40656,0,0xea76,1,29190,5715,0x949a,1,40658,7337,0x8d95
,1,23874,0,0xf0d3,1,40660,0,0xea77,1,14312,0,0xf0d4
,1,23905,0,0xf0d5,1,23914,8619,0xf0d6,1,29197,2577,0xe0a1
,1,40665,8239,0x96d9,1,14324,0,0xf0d7,1,40667,2575,0x91e1
,1,40668,0,0xea78,1,40669,0,0xea7a,1,40670,0,0xea79
,1,23920,11084,0xf0d8,1,40672,4880,0xea7b,1,43743,0,0xf0d9
,1,14333,0,0xf0da,1,23944,0,0xf0db,1,14336,6590,0xf0dc
,1,40677,0,0xea7c,1,29211,0,0xe0a3,1,23954,11085,0xf0dd
,1,40680,0,0xea7d,1,23956,0,0xf0de,1,23959,10040,0xf0df
,1,23961,0,0xf0e0,1,23984,0,0xf0e1,1,23986,0,0xf0e2
,1,23988,8338,0xf0e3,1,40687,0,0xea7e,1,43744,0,0xf0e4
,1,23993,7118,0xf0e5,1,24017,0,0xf0e6,1,29224,0,0xe0a4
,1,40692,0,0xea80,1,29226,0,0x92dc,1,40694,7340,0xea81
,1,29228,4393,0xe0a6,1,29229,0,0xe0a5,1,40697,0,0xea83
,1,24023,0,0xf0e7,1,29232,4394,0xe0a7,1,40700,0,0xea85
,1,29234,4401,0xe0a8,1,24024,0,0xf0e8,1,24032,7343,0xf0e9
,1,29237,0,0x8edd,1,29238,0,0x9583,1,43745,11089,0xf0ea
,1,24036,7344,0xf0eb,1,24041,7348,0xf0ec,1,29242,0,0x96ea
,1,29243,0,0xe0a9,1,29244,4402,0xe0aa,1,29245,4403,0x9175
,1,29246,0,0x8ea2,1,29247,0,0xe0ab,1,29248,0,0xe0ac
,1,14383,0,0xf0ed,1,24064,7352,0xf0ee,1,40718,0,0x9343
,1,14390,0,0xf0ef,1,40720,0,0xeff1,1,29254,0,0xe0ad
,1,29255,0,0x95d0,1,40723,1343,0x8cdb,1,24082,0,0xf0f0
,1,40725,0,0xea8a,1,29259,0,0xe0ae,1,29260,11105,0x9476
,1,24085,8457,0xf0f1,1,14400,10746,0xf0f2,1,24095,0,0xf0f3
,1,24110,0,0xf0f4,1,24126,0,0xf0f5,1,29266,0,0x92ab
,1,24137,6684,0xf0f6,1,14428,0,0xf0f7,1,40736,4881,0x916c
,1,40737,4882,0xea8b,1,24150,7354,0xf0f8,1,29272,0,0xe0af
,1,29273,0,0x89e5,1,14433,0,0xf0f9,1,29275,0,0x8b8d
,1,24171,6653,0xf0fa,1,29277,0,0x96c4,1,24172,0,0xf0fb
,1,29279,0,0x96b4,1,24173,0,0xf0fc,1,29281,4404,0x89b2
,1,29282,0,0x9853,1,24174,7355,0xf29f,1,43746,7849,0xf2a0
,1,24229,0,0xf2a1,1,24234,6506,0xf2a2,1,29287,0,0x9671
,1,17821,0,0xee70,1,29289,0,0x95a8,1,24236,0,0xf2a3
,1,24249,0,0xf2a4,1,24255,6537,0xf2a5,1,24262,0,0xf2a6
,1,40761,7358,0xeff2,1,24274,7359,0xf2a7,1,40763,0,0x9540
,1,24281,0,0xf2a8,1,29298,11116,0x90b5,1,40766,0,0xea8d
,1,29300,0,0xe0b0,1,43747,0,0xf2a9,1,24317,11117,0xf2aa
,1,24328,10734,0xf2ab,1,24334,7360,0xf2ac,1,29305,0,0x93c1
,1,24348,11118,0xf2ad,1,43748,11122,0xf2ae,1,24350,0,0xf2af
,1,29309,0,0x8ca1,1,29310,0,0xe0b1,1,40778,7362,0xea8e
,1,29312,2628,0x8dd2,1,29313,0,0xe0b3,1,29314,0,0xe0b2
,1,40782,8321,0xe6d8,1,40783,0,0xe8eb,1,24391,0,0xf2b0
,1,24419,0,0xf2b1,1,29319,4405,0xe0b4,1,24434,8794,0xf2b2
,1,40788,0,0xea90,1,24446,11125,0xf2b3,1,24463,0,0xf2b4
,1,40791,0,0xeff3,1,29325,11128,0xec87,1,24482,7364,0xf2b5
,1,24484,0,0xf2b6,1,24504,0,0xf2b7,1,24516,7810,0xf2b8
,1,29330,11130,0xe0b5,1,14586,0,0xf2b9,1,40799,0,0xea92
,1,40800,0,0xea93,1,29334,4406,0xe0b6,1,40802,0,0x97ee
,1,40803,0,0xea91,1,24519,7007,0xf2ba,1,24523,0,0xf2bb
,1,40806,4884,0xea95,1,40807,0,0xea96,1,24530,0,0xf2bc
,1,24531,11133,0xf2bd,1,40810,0,0xea98,1,29344,0,0x8b5d
,1,40812,0,0xea97,1,29346,11147,0xe0b7,1,24532,0,0xf2be
,1,24546,8781,0xf2bf,1,24558,11154,0xf2c0,1,24559,0,0xf2c1
,1,29351,4409,0xe0b8,1,24563,0,0xf2c2,1,24572,0,0xf2c3
,1,14615,11155,0xf2c4,1,40822,0,0xea9b,1,29356,4407,0x8ca2
,1,24599,7366,0xf2c5,1,24610,7368,0xf2c6,1,29359,0,0x94c6
,1,24612,6665,0xf2c7,1,14618,7370,0xf2c8,1,29362,0,0xe0ba
,1,24652,7726,0xf2c9,1,24703,7331,0xf2ca,1,17898,0,0xee80
,1,29366,0,0x8ff3,1,24714,7795,0xf2cb,1,24725,0,0xf2cc
,1,29369,0,0xe0b9,1,24744,0,0xf2cd,1,24752,0,0xf2cf
,1,63773,0,0xebb9,1,24753,0,0xf2d0,1,29374,0,0xec89
,1,24766,7385,0xf2d1,1,29376,0,0xec8a,1,24776,7371,0xf2d2
,1,29378,1565,0x8bb6,1,29379,0,0xe0bb,1,29380,0,0xe0bd
,1,40848,0,0xeff4,1,29382,0,0xe0bc,1,63784,0,0xeaac
,1,63785,0,0xeb6d,1,40852,0,0xeff5,1,40853,0,0xea9c
,1,24793,0,0xf2d3,1,40855,7377,0xeff6,1,24795,8938,0xf2d4
,1,29390,0,0xe0be,1,24814,0,0xf2d5,1,29392,0,0x8ccf
,1,40860,0,0xea9d,1,29394,2640,0xe0bf,1,24818,8305,0xf2d6
,1,24821,6600,0xf2d7,1,40864,5132,0xea9e,1,24848,9882,0xf2d8
,1,29399,5716,0x8be7,1,24850,0,0xf2d9,1,29401,0,0x915f
,1,24851,8433,0xf2da,1,29403,0,0x8d9d,1,24857,0,0xf2db
,1,24862,6856,0xf2dc,1,24890,9927,0xf2dd,1,14703,0,0xf2de
,1,29408,0,0xe0c1,1,29409,0,0xe0c2,1,29410,0,0xe0c0
,1,24897,8951,0xf2df,1,24902,0,0xf2e0,1,24928,0,0xf2e1
,1,24956,8380,0xf2e2,1,43750,9103,0xf2e3,1,24978,0,0xf2e4
,1,29417,0,0x8eeb,1,24979,0,0xf2e5,1,24983,9066,0xf2e6
,1,29420,0,0x93c6,1,29421,0,0x8bb7,1,24984,6785,0xf2e7
,1,24997,6703,0xf2e8,1,25000,0,0xf2e9,1,25005,0,0xf2ea
,1,25045,0,0xf2ec,1,25053,6674,0xf2ed,1,25055,7714,0xf2ee
,1,25077,0,0xf2ef,1,43752,7851,0xf2f0,1,29431,0,0xe0c4
,1,29432,0,0x924b,1,29433,0,0xe0c3,1,25109,7391,0xf2f1
,1,29435,0,0xec8b,1,29436,0,0x9854,1,29437,0,0x9482
,1,25123,0,0xf2f2,1,25129,0,0xf2f3,1,25158,7708,0xf2f4
,1,25164,7724,0xf2f5,1,25169,0,0xf2f6,1,25170,0,0xf2f7
,1,25185,0,0xf2f8,1,25188,0,0xf2f9,1,25211,0,0xf2fa
,1,25197,7392,0xf2fb,1,25203,6796,0xf2fc,1,25241,6804,0xf340
,1,29450,0,0xe0c7,1,25254,0,0xf341,1,25301,8467,0xf342
,1,43753,0,0xf343,1,25341,0,0xf344,1,63856,0,0xebc7
,1,25347,0,0xf345,1,25357,8827,0xf346,1,25360,7394,0xf347
,1,43755,7852,0xf348,1,43756,0,0xf349,1,25394,0,0xf34a
,1,29462,0,0xe0c9,1,29463,0,0xe0c6,1,25397,7396,0xf34b
,1,25403,7402,0xf34c,1,25404,0,0xf34d,1,29467,0,0x96d2
,1,29468,0,0xe0c8,1,29469,0,0xe0ca,1,25409,9535,0xf34e
,1,29471,0,0x97c2,1,25412,0,0xf34f,1,25422,0,0xf350
,1,43754,7178,0xf351,1,25433,0,0xf352,1,43757,0,0xf353
,1,29477,0,0xe0ce,1,43758,0,0xf354,1,29479,0,0xec8d
,1,29480,0,0xec8e,1,29481,0,0xe0cd,1,29482,0,0x9296
,1,29483,0,0x944c,1,25452,7412,0xf355,1,25476,0,0xf356
,1,29486,0,0x8ca3,1,29487,0,0xe0cc,1,25497,8493,0xf357
,1,25492,7413,0xf359,1,25533,0,0xf35a,1,25591,7415,0xf35b
,1,29492,0,0xe0cb,1,25556,0,0xf35c,1,29494,0,0x9750
,1,29495,0,0x9751,1,25557,0,0xf35d,1,25564,7408,0xf35e
,1,25568,0,0xf35f,1,25579,9564,0xf360,1,25580,0,0xf361
,1,25586,7416,0xf362,1,29502,0,0xe0cf,1,29503,0,0x898e
,1,25609,8722,0xf363,1,25630,0,0xf364,1,25637,0,0xf365
,1,25641,8368,0xf366,1,29508,0,0x8d96,1,29509,0,0x8e82
,1,25647,0,0xf367,1,25690,0,0xf368,1,25691,0,0xf369
,1,25693,0,0xf36a,1,25715,0,0xf36b,1,25725,10700,0xf36c
,1,25735,0,0xf36d,1,25745,0,0xf36e,1,29518,0,0xe0d0
,1,29519,0,0xe0d1,1,29520,0,0xec90,1,25757,0,0xf36f
,1,25759,7418,0xf370,1,25803,0,0xf371,1,25804,0,0xf372
,1,25813,0,0xf373,1,25815,10125,0xf374,1,29527,0,0xe0d3
,1,43760,0,0xf375,1,25828,6849,0xf376,1,25829,0,0xf377
,1,25855,0,0xf378,1,25860,0,0xf379,1,14958,7189,0xf37a
,1,25871,9668,0xf37b,1,25876,0,0xf37c,1,25878,0,0xf37d
,1,14963,8340,0xf37e,1,25886,0,0xf380,1,29539,0,0x8f62
,1,25906,0,0xf381,1,25924,0,0xf382,1,29542,0,0xec91
,1,25940,0,0xf383,1,29544,0,0xe0d5,1,25963,0,0xf384
,1,29546,0,0xe0d4,1,25978,6984,0xf385,1,25985,7423,0xf386
,1,25988,0,0xf387,1,25989,6844,0xf388,1,63952,7427,0xefa2
,1,29552,0,0xe0d6,1,25994,7431,0xf389,1,29554,0,0x8a6c
,1,26034,9801,0xf38a,1,26037,0,0xf38b,1,29557,0,0xe0d8
,1,26040,0,0xf38c,1,26047,7434,0xf38d,1,29560,0,0xe0d7
,1,26050,9820,0xf38e,1,29562,8605,0xe0da,1,29563,5531,0xe0d9
,1,29564,0,0xec92,1,26057,0,0xf38f,1,26068,6792,0xf390
,1,15062,0,0xf391,1,26098,9089,0xf392,1,26105,7435,0xf393
,1,26108,0,0xf394,1,26116,8796,0xf395,1,29572,0,0x8cba
,1,26120,7436,0xf396,1,26145,0,0xf397,1,29575,0,0x97a6
,1,26154,0,0xf398,1,29577,0,0x8bca,1,26181,7437,0xf399
,1,29579,0,0x89a4,1,26193,8958,0xf39a,1,26190,0,0xf39b
,1,15082,0,0xf39c,1,43762,7853,0xf39d,1,26199,0,0xf39e
,1,26203,0,0xf39f,1,26211,0,0xf3a0,1,43763,0,0xf3a2
,1,26218,7438,0xf3a3,1,29589,0,0xec93,1,29590,0,0x8be8
,1,26219,0,0xf3a4,1,26220,7141,0xf3a5,1,26221,0,0xf3a6
,1,26235,0,0xf3a7,1,26240,0,0xf3a8,1,26256,7440,0xf3a9
,1,26258,0,0xf3aa,1,26265,7369,0xf3ab,1,29599,0,0xec94
,1,29600,0,0xec95,1,15118,6783,0xf3ac,1,29602,0,0xec96
,1,26285,0,0xf3ad,1,26289,0,0xf3ae,1,26293,7443,0xf3af
,1,29606,0,0xec97,1,15130,7390,0xf3b0,1,26303,7444,0xf3b1
,1,29609,0,0x8adf,1,15132,0,0xf3b2,1,29611,0,0xec98
,1,26348,0,0xf3b3,1,15063,10091,0xf3b4,1,64015,0,0x886a
,1,64016,0,0x8876,1,64017,0,0x9892,1,26369,7353,0xf3b5
,1,29618,6847,0x97e6,1,29619,4724,0xe0dc,1,64021,0,0xec79
,1,64022,7446,0xec8f,1,26373,0,0xf3b6,1,26386,6883,0xf3b7
,1,64025,0,0xed5b,1,64026,7448,0xed5c,1,64027,0,0xed60
,1,29627,0,0xe0de,1,26393,7450,0xf3b9,1,43768,0,0xf3bb
,1,64031,0,0xee59,1,26444,7452,0xf3bc,1,29632,8303,0xe0df
,1,64034,0,0xeeac,1,29634,0,0x89cf,1,26445,9246,0xf3bd
,1,26452,10673,0xf3be,1,64038,7453,0xeee8,1,26461,0,0xf3bf
,1,43766,0,0xf3c0,1,29640,0,0xe0db,1,29641,0,0xec99
,1,29642,0,0x8e58,1,43772,7454,0xf3c1,1,43769,0,0xf3c2
,1,29645,0,0x92bf,1,29646,0,0xe0dd,1,64048,4888,0x87b6
,1,64049,0,0x87c7,1,64050,0,0x87ce,1,64051,0,0x87e1
,1,64052,0,0x87e6,1,64053,0,0x87ec,1,64054,0,0x884b
,1,64055,4890,0x884e,1,64056,8738,0x8855,1,64057,0,0x8879
,1,64058,4891,0x887d,1,64059,0,0x9881,1,64060,0,0x9882
,1,64061,0,0xeace,1,64062,0,0xeada,1,29662,4495,0xe0e2
,1,64064,0,0xeadf,1,29664,4592,0x8eec,1,64066,7456,0xeb4a
,1,64067,0,0xeb62,1,64068,4893,0xeb85,1,64069,0,0xeba2
,1,29669,4835,0xe0e0,1,64071,0,0xebf5,1,64072,7462,0xec44
,1,64073,0,0xec74,1,29673,7339,0xec9e,1,29674,4895,0x8c5d
,1,64076,0,0xed46,1,64077,0,0xed52,1,29677,4931,0x94c7
,1,29678,4935,0xe0e1,1,64080,0,0xed57,1,64081,0,0xed58
,1,29681,4937,0xe0fc,1,64083,0,0xed5e,1,64084,0,0xed5f
,1,64085,0,0xed6c,1,64086,0,0xed70,1,64087,0,0xed84
,1,64088,0,0xedac,1,29688,7847,0xe0e7,1,64090,7463,0xedb1
,1,64091,7465,0xedb8,1,64092,0,0xedc2,1,64093,0,0xedd6
,1,26484,8154,0xf3c3,1,29694,5097,0x8cbb,1,64096,0,0xee8f
,1,64097,0,0xee99,1,64098,7467,0xeead,1,64099,0,0xeeae
,1,29699,5266,0x8b85,1,64101,0,0xeebb,1,29701,9209,0xe0e4
,1,29702,5325,0x979d,1,29703,5543,0xec9f,1,64105,0,0xef96
,1,29705,5600,0x97ae,1,29706,0,0xeca0,1,26486,0,0xf3c4
,1,26514,0,0xf3c6,1,43771,0,0xf3c7,1,26640,0,0xf3c9
,1,26544,0,0xf3ca,1,26546,6969,0xf3cb,1,26563,0,0xf3cc
,1,26568,0,0xf3cd,1,26578,7464,0xf3ce,1,26585,0,0xf3cf
,1,26587,0,0xf3d0,1,26608,0,0xf3d1,1,26615,0,0xf3d2
,1,43774,0,0xf3d3,1,43775,0,0xf3d4,1,29722,0,0xeca1
,1,29723,0,0xeca2,1,43776,0,0xf3d5,1,26648,8444,0xf3d6
,1,26655,0,0xf3d7,1,26669,0,0xf3d8,1,43773,0,0xf3d9
,1,26675,8390,0xf3da,1,29730,0,0x91f4,1,26683,8514,0xf3db
,1,26686,7468,0xf3dc,1,29733,0,0xe0e6,1,29734,0,0xeca4
,1,26692,0,0xf3dd,1,29736,0,0xeca5,1,26693,0,0xf3de
,1,29738,0,0xeca6,1,29739,0,0xeca7,1,29740,0,0xeca8
,1,26697,9154,0xf3df,1,29742,0,0xeca9,1,29743,8861,0xecaa
,1,29744,0,0xecab,1,26700,7471,0xf3e0,1,29746,0,0xe0e8
,1,29747,0,0x97d4,1,29748,0,0x8bd5,1,29749,0,0x94fa
,1,29750,0,0x9469,1,26709,8507,0xf3e1,1,26711,0,0xf3e2
,1,15223,7474,0xf3e3,1,29754,0,0xe0e9,1,26731,0,0xf3e4
,1,26734,0,0xf3e5,1,26746,0,0xf3e6,1,26748,0,0xf3e7
,1,29759,0,0xe0eb,1,26754,7478,0xf3e8,1,29761,0,0xe0ee
,1,26768,0,0xf3e9,1,26774,7482,0xf3ea,1,29764,0,0xecac
,1,15213,0,0xf3eb,1,29766,0,0xecad,1,29767,0,0xecae
,1,26776,6923,0xf3ec,1,26777,0,0xf3ed,1,26778,10392,0xf3ee
,1,29771,0,0xecaf,1,26780,0,0xf3ef,1,26794,7483,0xf3f0
,1,26795,0,0xf3f1,1,26804,0,0xf3f2,1,26811,6951,0xf3f3
,1,26875,7426,0xf3f4,1,43777,7484,0xf3f5,1,43778,7476,0xf3f6
,1,64019,0,0xf3f7,1,29781,0,0xe0ea,1,26819,7192,0xf3f8
,1,29783,0,0xecb0,1,26821,0,0xf3f9,1,29785,0,0xe0ed
,1,29786,0,0x8ce8,1,29787,0,0x896c,1,29788,0,0xe0ef
,1,26828,7486,0xf3fa,1,29790,0,0x9090,1,29791,0,0xe0ec
,1,29792,0,0x97da,1,26831,0,0xf3fb,1,29794,0,0xecb1
,1,29795,0,0xe0f2,1,29796,0,0xeaa2,1,26838,0,0xf3fc
,1,26841,0,0xf440,1,26852,0,0xf441,1,26853,0,0xf442
,1,29801,0,0xe0f0,1,29802,0,0xe0f3,1,29803,0,0xecb2
,1,26860,0,0xf443,1,29805,0,0xecb3,1,26871,0,0xf444
,1,29807,0,0xe0e5,1,29808,0,0xe0f1,1,26883,7488,0xf445
,1,26887,8346,0xf446,1,29811,0,0x8dba,1,15239,10408,0xf447
,1,15240,0,0xf448,1,29814,0,0xe0f4,1,43780,0,0xf449
,1,26939,7447,0xf44a,1,15245,0,0xf44b,1,26950,0,0xf44c
,1,26985,10158,0xf44d,1,26988,0,0xf44e,1,26994,0,0xf44f
,1,29822,0,0xe0f5,1,27002,0,0xf450,1,27007,0,0xf451
,1,27026,7491,0xf452,1,15268,0,0xf453,1,29827,0,0x979e
,1,27030,6971,0xf454,1,27032,7492,0xf455,1,29830,0,0xecb4
,1,29831,0,0xecb5,1,27046,7493,0xf456,1,29833,0,0xecb6
,1,27056,0,0xf457,1,29835,0,0xe0f6,1,27063,10173,0xf458
,1,27066,0,0xf459,1,27068,0,0xf45a,1,27072,0,0xf45b
,1,29840,0,0xecbb,1,27089,0,0xf45c,1,27094,7499,0xf45d
,1,43781,0,0xf45e,1,43782,0,0xf45f,1,27184,0,0xf460
,1,43783,0,0xf461,1,43784,7501,0xf462,1,29848,0,0xecb7
,1,27107,7507,0xf463,1,27118,0,0xf464,1,27119,10242,0xf465
,1,29852,0,0xecb8,1,27123,0,0xf466,1,29854,0,0xe0f7
,1,29855,0,0xecb9,1,15309,7508,0xf467,1,27124,7510,0xf468
,1,29858,0,0xe0e3,1,29859,0,0xecba,1,27134,0,0xf469
,1,27153,7511,0xf46a,1,29862,0,0xecbc,1,29863,0,0xe0f8
,1,29864,0,0xecbd,1,29865,0,0xecbe,1,27162,7512,0xf46b
,1,27165,7514,0xf46c,1,27186,0,0xf46e,1,27187,0,0xf46f
,1,27188,0,0xf470,1,27199,0,0xf471,1,29872,0,0x8ac2
,1,27206,0,0xf472,1,27209,10257,0xf473,1,27258,0,0xf474
,1,27214,0,0xf475,1,29877,0,0xecbf,1,27218,0,0xf476
,1,27236,0,0xf477,1,43788,0,0xf478,1,27262,7515,0xf479
,1,27267,7520,0xf47a,1,27275,7303,0xf47b,1,15344,10124,0xf47c
,1,29885,0,0x8ea3,1,27281,0,0xf47d,1,29887,0,0xecc0
,1,27295,7318,0xf47e,1,27297,0,0xf480,1,43789,0,0xf481
,1,27307,0,0xf482,1,27325,0,0xf483,1,27334,0,0xf484
,1,27348,0,0xf485,1,27344,7338,0xf486,1,29896,8956,0xecc1
,1,29897,0,0xecc2,1,29898,0,0xe0f9,1,27356,0,0xf487
,1,27357,0,0xf488,1,43790,0,0xf489,1,27372,7166,0xf48b
,1,29903,0,0xe0fa,1,27377,8337,0xf48c,1,27378,7076,0xf48d
,1,27379,10326,0xf48e,1,27389,0,0xf48f,1,29908,0,0xe0fb
,1,27403,0,0xf491,1,27407,7521,0xf492,1,27408,0,0xf493
,1,27409,7522,0xf494,1,43793,7884,0xf495,1,29914,0,0xecc3
,1,27415,0,0xf496,1,29916,0,0x895a,1,15398,7421,0xf497
,1,27439,7524,0xf498,1,27466,0,0xf499,1,29920,0,0xe140
,1,27480,9018,0xf49a,1,29922,0,0x955a,1,29923,0,0xe141
,1,27500,0,0xf49b,1,27509,0,0xf49c,1,29926,0,0x8aa2
,1,29927,0,0xe142,1,27514,0,0xf49d,1,29929,0,0xe143
,1,27521,0,0xf49e,1,27547,7528,0xf49f,1,27566,0,0xf4a0
,1,43794,0,0xf4a1,1,29934,0,0xe144,1,27581,7525,0xf4a2
,1,29936,0,0xe146,1,29937,0,0xe147,1,29938,0,0xe145
,1,27582,9877,0xf4a3,1,27591,8857,0xf4a4,1,27592,0,0xf4a5
,1,29942,0,0x9572,1,29943,0,0xe149,1,29944,0,0xe148
,1,27593,0,0xf4a6,1,27610,7531,0xf4a7,1,27622,0,0xf4a8
,1,27623,0,0xf4a9,1,27630,10376,0xf4aa,1,27633,10414,0xf4ab
,1,29951,0,0xecc4,1,27650,0,0xf4ac,1,29953,0,0xecc5
,1,27658,0,0xf4ad,1,29955,0,0xe14b,1,29956,0,0xe14a
,1,29957,0,0xe14c,1,27662,0,0xf4ae,1,27701,0,0xf4af
,1,27702,0,0xf4b0,1,27706,0,0xf4b1,1,27711,0,0xf4b3
,1,27725,0,0xf4b4,1,29964,0,0xe14d,1,29965,0,0xe14f
,1,29966,0,0xe14e,1,18500,0,0xeec9,1,27739,0,0xf4b5
,1,29969,0,0x8d99,1,27757,0,0xf4b6,1,29971,0,0xe151
,1,27780,0,0xf4b7,1,29973,0,0xe150,1,27785,0,0xf4b8
,1,29975,0,0xecc6,1,29976,0,0x8ac3,1,15555,9152,0xf4b9
,1,29978,0,0x9072,1,27796,7614,0xf4ba,1,29980,0,0x935b
,1,27797,10549,0xf4bb,1,29982,0,0xe152,1,29983,0,0x90b6
,1,27799,7534,0xf4bc,1,27821,0,0xf4bd,1,27842,0,0xf4be
,1,29987,0,0x8e59,1,27856,7536,0xf4bf,1,29989,0,0x8999
,1,29990,0,0xe153,1,15570,0,0xf4c0,1,29992,0,0x9770
,1,27862,0,0xf4c1,1,27866,7537,0xf4c2,1,29995,0,0x95e1
,1,29996,0,0xe154,1,27868,0,0xf4c3,1,27881,0,0xf4c4
,1,29999,0,0xecc7,1,30000,0,0x9363,1,30001,0,0x9752
,1,30002,0,0x8d62,1,30003,0,0x905c,1,27884,0,0xf4c5
,1,27885,0,0xf4c6,1,43797,0,0xf4c7,1,30007,0,0x926a
,1,30008,0,0x99b2,1,27904,0,0xf4c8,1,30010,0,0x92ac
,1,30011,0,0x89e6,1,30012,0,0xe155,1,27914,7546,0xf4c9
,1,27940,7547,0xf4ca,1,27942,0,0xf4cb,1,27943,7548,0xf4cc
,1,27751,10501,0xf4cd,1,27951,0,0xf4ce,1,27964,8199,0xf4cf
,1,30020,0,0xe156,1,27995,0,0xf4d0,1,30022,0,0xe15b
,1,27998,7666,0xf4d1,1,28000,7549,0xf4d2,1,30025,0,0xe159
,1,30026,0,0xe158,1,30027,0,0x9dc0,1,30028,0,0x8a45
,1,30029,0,0xe157,1,28016,7550,0xf4d3,1,30031,0,0x88d8
,1,28032,7551,0xf4d4,1,30033,0,0x94a8,1,28033,7552,0xf4d5
,1,28042,0,0xf4d6,1,30036,0,0x94c8,1,28045,10618,0xf4d7
,1,28049,0,0xf4d8,1,28056,0,0xf4d9,1,43798,0,0xf4da
,1,30041,0,0x97af,1,30042,0,0xe15c,1,30043,0,0xe15a
,1,30044,0,0x927b,1,30045,0,0x90a4,1,28183,0,0xf4db
,1,43800,0,0xf4dd,1,30048,0,0x94a9,1,43801,0,0xf4de
,1,30050,0,0x954c,1,28075,7317,0xf4df,1,30052,0,0xe15e
,1,30053,0,0x97aa,1,30054,0,0x8c6c,1,30055,0,0xe15f
,1,28078,0,0xf4e0,1,30057,0,0xe15d,1,30058,0,0x94d4
,1,30059,0,0xe160,1,28084,7553,0xf4e1,1,30061,0,0xe161
,1,28098,8447,0xf4e2,1,30063,0,0xecc8,1,30064,0,0x88d9
,1,27956,7555,0xf4e3,1,28104,7554,0xf4e4,1,30067,0,0x8ff4
,1,30068,0,0xe166,1,28110,0,0xf4e5,1,30070,0,0xe163
,1,30071,0,0x93eb,1,30072,0,0xe162,1,30073,0,0xecc9
,1,28111,7292,0xf4e6,1,28112,7264,0xf4e7,1,28127,8460,0xf4e8
,1,28137,7559,0xf4e9,1,28150,7560,0xf4ea,1,30079,0,0x8b45
,1,28214,9560,0xf4eb,1,28190,7564,0xf4ec,1,30082,0,0xe169
,1,28194,0,0xf4ed,1,28199,0,0xf4ee,1,15633,0,0xf4ef
,1,30086,0,0xe164,1,30087,0,0xe165,1,28210,0,0xf4f0
,1,30089,0,0xe168,1,30090,0,0xe167,1,30091,0,0x9544
,1,28220,7566,0xf4f1,1,28232,0,0xf4f2,1,30094,0,0x9161
,1,30095,0,0x9160,1,28233,9636,0xf4f3,1,30097,0,0x8b5e
,1,30098,0,0xecca,1,28235,7248,0xf4f4,1,30100,0,0xe16a
,1,28236,9212,0xf4f5,1,28239,0,0xf4f6,1,28241,10681,0xf4f7
,1,28243,0,0xf4f8,1,28244,0,0xf4f9,1,30106,0,0xe16b
,1,28247,0,0xf4fa,1,28259,0,0xf4fb,1,30109,0,0xe16c
,1,15646,10441,0xf4fc,1,28307,0,0xf540,1,28327,0,0xf541
,1,28340,0,0xf542,1,28351,7571,0xf543,1,30115,0,0xe16e
,1,28355,0,0xf544,1,30117,0,0xe16d,1,28362,0,0xf545
,1,28377,7763,0xf546,1,28469,0,0xf547,1,28395,0,0xf548
,1,28409,0,0xf549,1,30123,0,0x8975,1,28411,0,0xf54a
,1,28426,0,0xf54b,1,28428,0,0xf54c,1,28440,0,0xf54d
,1,28453,7574,0xf54e,1,30129,0,0xe176,1,30130,0,0x94e6
,1,30131,0,0xe170,1,28470,0,0xf54f,1,30133,0,0xe172
,1,28476,0,0xf550,1,28498,7575,0xf552,1,30136,0,0xe174
,1,30137,0,0x905d,1,28503,10801,0xf553,1,28506,0,0xf554
,1,30140,0,0xe175,1,30141,0,0xe173,1,30142,0,0x8ebe
,1,28512,7576,0xf555,1,28520,7577,0xf556,1,28568,0,0xf557
,1,30146,0,0xe16f,1,30147,0,0xe171,1,28541,0,0xf558
,1,30149,0,0x9561,1,28560,7578,0xf559,1,30151,0,0x8fc7
,1,28566,0,0xf55a,1,28606,0,0xf55b,1,30154,0,0xe178
,1,28575,10888,0xf55c,1,28581,10894,0xf55d,1,30157,0,0xe177
,1,30158,0,0xeccc,1,28591,7579,0xf55e,1,15716,0,0xf55f
,1,28597,0,0xf560,1,30162,0,0xe179,1,28616,7580,0xf561
,1,30164,0,0x8ea4,1,30165,0,0x8dad,1,28617,0,0xf562
,1,28634,0,0xf563,1,30168,0,0x9397,1,30169,0,0xe17a
,1,28638,0,0xf564,1,30171,0,0x92c9,1,28649,0,0xf565
,1,43803,7581,0xf566,1,30174,0,0xe17c,1,28668,7582,0xf567
,1,28672,7584,0xf568,1,28679,0,0xf569,1,30178,0,0x979f
,1,30179,0,0xe17b,1,30180,0,0xeccd,1,28682,10680,0xf56a
,1,28707,0,0xf56b,1,43804,7588,0xf56c,1,28729,0,0xf56d
,1,30185,0,0x9189,1,28730,0,0xf56e,1,28732,0,0xf56f
,1,28739,7590,0xf570,1,28743,0,0xf571,1,28747,7592,0xf572
,1,15770,0,0xf573,1,30192,0,0xe182,1,28756,7593,0xf574
,1,30194,0,0xe184,1,30195,0,0xe185,1,30196,0,0x9273
,1,28773,7698,0xf575,1,28777,0,0xf576,1,28780,7365,0xf577
,1,28782,7686,0xf578,1,28790,7594,0xf579,1,30202,0,0xe183
,1,28798,8943,0xf57a,1,30204,0,0xe180,1,28801,0,0xf57b
,1,30206,0,0xe17d,1,30207,0,0xe17e,1,30208,0,0xecce
,1,30209,0,0xe181,1,30210,0,0xeccf,1,28806,7596,0xf57c
,1,28821,0,0xf57d,1,28823,10944,0xf57e,1,28859,7930,0xf580
,1,43805,7598,0xf581,1,30216,0,0xecd0,1,30217,0,0xe188
,1,28831,0,0xf582,1,30219,0,0xe186,1,28849,0,0xf583
,1,30221,0,0xe187,1,28908,0,0xf585,1,28874,7599,0xf586
,1,28881,0,0xf587,1,28883,7439,0xf588,1,28892,0,0xf589
,1,28931,7604,0xf58a,1,28932,0,0xf58b,1,30229,0,0xecd1
,1,30230,0,0xecd2,1,28934,0,0xf58c,1,28935,7597,0xf58d
,1,30233,0,0xecd3,1,28936,0,0xf58e,1,28940,7605,0xf58f
,1,15808,7518,0xf590,1,28975,0,0xf591,1,30238,0,0xecd4
,1,30239,0,0xe189,1,30240,0,0xe18b,1,30241,0,0xe18c
,1,30242,0,0xe18d,1,28977,0,0xf592,1,30244,0,0xe18e
,1,29008,7607,0xf593,1,30246,0,0xeffb,1,30247,0,0xe18a
,1,29002,7610,0xf594,1,29011,0,0xf595,1,29022,8347,0xf596
,1,15828,10297,0xf597,1,29078,0,0xf598,1,30253,0,0xecd5
,1,29056,0,0xf599,1,29083,0,0xf59a,1,30256,0,0xe190
,1,29088,0,0xf59b,1,29090,10716,0xf59c,1,29102,7612,0xf59d
,1,30260,0,0xe18f,1,30261,0,0xecd6,1,29103,0,0xf59e
,1,29107,0,0xf59f,1,43807,0,0xf5a0,1,29131,0,0xf5a1
,1,29139,8202,0xf5a2,1,30267,0,0xe191,1,29145,7611,0xf5a3
,1,29148,0,0xf5a4,1,29191,0,0xf5a5,1,15877,0,0xf5a6
,1,64074,0,0xf5a7,1,29227,7615,0xf5a8,1,30274,0,0x97c3
,1,30275,0,0xecd7,1,29236,11088,0xf5a9,1,29240,11102,0xf5aa
,1,30278,0,0xe194,1,30279,0,0xe192,1,30280,0,0xe193
,1,29241,7617,0xf5ab,1,20012,0,0xf5ac,1,30283,0,0xecd8
,1,30284,0,0x8ae0,1,29250,0,0xf5ad,1,29267,0,0xf5ae
,1,29271,0,0xf5af,1,29283,0,0xf5b0,1,43808,8560,0xf5b1
,1,30290,0,0x96fc,1,29294,0,0xf5b2,1,29295,11113,0xf5b3
,1,29304,7618,0xf5b4,1,30294,0,0x95c8,1,29311,0,0xf5b5
,1,30296,0,0xe196,1,29326,8821,0xf5b6,1,43809,0,0xf5b7
,1,29357,0,0xf5b8,1,30300,0,0xe195,1,29358,10044,0xf5b9
,1,29360,7442,0xf5ba,1,29361,0,0xf5bb,1,29377,0,0xf5bc
,1,30305,0,0xe197,1,30306,0,0xe198,1,15968,0,0xf5bd
,1,30308,0,0xecd9,1,30309,0,0xecda,1,29388,0,0xf5be
,1,30311,0,0xe19c,1,30312,0,0xe199,1,30313,0,0xe19a
,1,30314,0,0xe19b,1,15974,0,0xf5bf,1,30316,0,0xe19d
,1,30317,0,0xecdb,1,15976,7620,0xf5c0,1,30319,0,0xecdc
,1,30320,0,0xe19e,1,30321,0,0xecdd,1,30322,0,0xe19f
,1,29427,0,0xf5c1,1,29434,0,0xf5c2,1,29447,0,0xf5c3
,1,30326,0,0xe1a0,1,29458,0,0xf5c4,1,30328,0,0xe1a1
,1,29464,0,0xf5c5,1,30330,0,0x94ad,1,30331,5519,0x936f
,1,30332,0,0xe1a2,1,30333,0,0x9492,1,30334,0,0x9553
,1,29465,0,0xf5c6,1,30336,0,0xe1a3,1,30337,0,0xecde
,1,16003,0,0xf5c7,1,30339,0,0xe1a4,1,30340,0,0x9349
,1,29497,0,0xf5c8,1,30342,0,0x8a46,1,30343,0,0x8d63
,1,30344,0,0xe1a5,1,29484,9538,0xf5c9,1,29489,0,0xf5ca
,1,30347,0,0xe1a6,1,29491,8434,0xf5cb,1,29501,0,0xf5cc
,1,30350,0,0xe1a7,1,29522,0,0xf5cd,1,30352,0,0x8e48
,1,16020,0,0xf5ce,1,29547,10427,0xf5cf,1,30355,0,0xe1a9
,1,29548,8578,0xf5d0,1,43811,7624,0xf5d1,1,30358,0,0xe1a8
,1,29550,0,0xf5d2,1,29551,0,0xf5d3,1,30361,0,0xe1aa
,1,30362,0,0xe1ab,1,30363,0,0xecdf,1,29553,7625,0xf5d4
,1,30365,0,0xece0,1,30366,0,0xece1,1,29559,0,0xf5d5
,1,29569,7631,0xf5d6,1,29573,7627,0xf5d7,1,29578,0,0xf5d8
,1,29588,0,0xf5d9,1,29592,7557,0xf5da,1,29596,0,0xf5db
,1,30374,0,0xece2,1,29598,7634,0xf5dc,1,29605,7636,0xf5dd
,1,29608,0,0xf5de,1,30378,0,0xece3,1,29621,0,0xf5df
,1,29623,0,0xf5e0,1,29625,0,0xf5e1,1,30382,0,0x94e7
,1,29628,7637,0xf5e2,1,30384,0,0xe1ac,1,29631,8288,0xf5e3
,1,29637,10190,0xf5e4,1,29643,0,0xf5e5,1,30388,0,0xe1ad
,1,29665,0,0xf5e6,1,30390,0,0xece4,1,30391,0,0xea89
,1,30392,0,0xe1ae,1,30393,0,0xe1af,1,30394,0,0xe1b0
,1,29671,0,0xf5e7,1,29689,0,0xf5e8,1,29715,0,0xf5e9
,1,29690,0,0xf5ea,1,30399,0,0x8e4d,1,29697,0,0xf5eb
,1,29732,0,0xf5ec,1,30402,0,0xe1b1,1,30403,0,0x9475
,1,29745,9178,0xf5ed,1,30405,10246,0xece5,1,30406,0,0x967e
,1,29753,0,0xf5ee,1,30408,0,0x896d,1,29779,7640,0xf5ef
,1,30410,0,0x8976,1,29760,0,0xf5f0,1,30412,0,0xece6
,1,30413,0,0xe1b2,1,30414,0,0xece7,1,29763,10256,0xf5f1
,1,29773,0,0xf5f2,1,29778,0,0xf5f3,1,30418,0,0xe1b4
,1,29789,0,0xf5f4,1,30420,0,0xece8,1,29809,0,0xf5f5
,1,30422,0,0xe1b3,1,30423,0,0x9390,1,29825,0,0xf5f6
,1,29829,0,0xf5f7,1,29832,0,0xf5f8,1,30427,0,0x90b7
,1,30428,0,0x9f58,1,43812,0,0xf5f9,1,30430,0,0xe1b5
,1,30431,0,0x96bf,1,29842,0,0xf5fa,1,30433,0,0xe1b6
,1,29847,0,0xf5fb,1,30435,0,0x8ac4,1,30436,0,0x94d5
,1,30437,0,0xe1b7,1,30438,0,0xece9,1,30439,0,0xe1b8
,1,29849,7642,0xf5fc,1,29856,0,0xf640,1,30442,0,0xe1b9
,1,29857,0,0xf641,1,29861,7643,0xf642,1,29866,0,0xf643
,1,30446,0,0x96da,1,29867,0,0xf644,1,29881,0,0xf645
,1,30449,0,0xecea,1,30450,0,0x96d3,1,29883,7606,0xf646
,1,30452,10261,0x92bc,1,29882,0,0xf647,1,29910,0,0xf648
,1,29912,0,0xf649,1,30456,0,0x918a,1,29918,0,0xf64a
,1,29935,9963,0xf64b,1,30459,0,0xe1bb,1,30460,0,0xeceb
,1,29931,10396,0xf64c,1,30462,0,0x8f82,1,43813,0,0xf64d
,1,29946,0,0xf64e,1,30465,0,0x8fc8,1,43814,0,0xf64f
,1,29984,0,0xf650,1,30468,0,0xe1be,1,29988,0,0xf651
,1,29994,0,0xf652,1,30471,0,0xe1bd,1,30472,0,0xe1bc
,1,30473,0,0x94fb,1,30474,0,0xecec,1,30475,0,0x8ac5
,1,30476,0,0x8ca7,1,16215,0,0xf653,1,43815,0,0xf654
,1,30013,0,0xf655,1,30014,0,0xf656,1,30016,0,0xf657
,1,30024,0,0xf658,1,30030,0,0xf659,1,30032,0,0xf65a
,1,30034,10595,0xf65b,1,30060,0,0xf65c,1,30066,7645,0xf65d
,1,30065,0,0xf65e,1,30489,0,0xeced,1,30074,7646,0xf65f
,1,30491,0,0xe1c4,1,30077,0,0xf660,1,30078,7828,0xf661
,1,30494,0,0xe1c1,1,30495,0,0x905e,1,30496,0,0x96b0
,1,30081,10665,0xf662,1,43816,0,0xf663,1,30092,8332,0xf664
,1,30500,0,0xe1c0,1,30501,0,0xe1c2,1,30502,0,0xe1c3
,1,16245,0,0xf665,1,30114,0,0xf666,1,30505,0,0xe1bf
,1,16247,0,0xf667,1,30128,0,0xf668,1,30135,10798,0xf669
,1,30143,7647,0xf66a,1,30144,0,0xf66b,1,30150,10867,0xf66c
,1,30159,7711,0xf66d,1,30163,0,0xf66e,1,30173,0,0xf66f
,1,30175,0,0xf670,1,30516,0,0xecee,1,30176,7648,0xf671
,1,30518,0,0xecef,1,30519,0,0xe1c5,1,30520,0,0xe1c6
,1,30183,0,0xf672,1,30522,0,0x92ad,1,30188,0,0xf673
,1,30524,0,0x8ae1,1,30190,0,0xf674,1,30193,0,0xf675
,1,30201,0,0xf676,1,30528,0,0x9285,1,30211,0,0xf677
,1,30232,0,0xf678,1,30215,0,0xf679,1,30223,0,0xf67a
,1,16302,0,0xf67b,1,30534,0,0xecf0,1,30535,0,0xe1c7
,1,43817,0,0xf67c,1,30227,0,0xf67d,1,30235,0,0xf67e
,1,30236,0,0xf680,1,30245,0,0xf682,1,30541,0,0xecf1
,1,30542,0,0xecf2,1,30248,0,0xf683,1,30268,0,0xf684
,1,30259,0,0xf685,1,43819,0,0xf686,1,16329,10546,0xf687
,1,30273,0,0xf688,1,43820,10584,0xf689,1,30281,0,0xf68a
,1,30293,0,0xf68b,1,16343,7654,0xf68c,1,30318,0,0xf68d
,1,30554,0,0xe1c8,1,30555,0,0xe1cb,1,30556,0,0xecf3
,1,30357,0,0xf68e,1,30364,0,0xf68f,1,30559,0,0xecf4
,1,30369,7655,0xf690,1,30561,0,0x9087,1,30562,0,0xecf5
,1,30563,0,0x93c2,1,30368,0,0xf691,1,30565,0,0xe1cc
,1,30566,0,0x9672,1,30375,8382,0xf692,1,30568,0,0xe1c9
,1,30376,0,0xf693,1,30383,7656,0xf694,1,30571,0,0xe1ca
,1,43821,0,0xf695,1,30409,0,0xf696,1,43822,0,0xf697
,1,30440,0,0xf698,1,30444,0,0xf699,1,43823,0,0xf69a
,1,30487,7661,0xf69b,1,30490,7933,0xf69c,1,30509,0,0xf69d
,1,30517,0,0xf69e,1,16441,0,0xf6a0,1,43826,0,0xf6a1
,1,43825,10483,0xf6a2,1,30585,0,0xe1cf,1,30586,0,0xecf6
,1,30552,0,0xf6a3,1,30560,7662,0xf6a4,1,30570,7663,0xf6a5
,1,30590,0,0xe1ce,1,30591,0,0xe1cd,1,30592,0,0xecf7
,1,43828,9604,0xf6a6,1,30578,0,0xf6a7,1,30588,0,0xf6a8
,1,30589,7890,0xf6a9,1,43827,0,0xf6aa,1,16472,0,0xf6ab
,1,16531,10379,0xf6b7,1,43829,10384,0xf6bb,1,43831,0,0xf6c6
,1,30884,0,0xf6c9,1,30603,0,0xe1d1,1,30907,0,0xf6cd
,1,30920,0,0xf6ce,1,30606,0,0xe1d0,1,30924,0,0xf6cf
,1,30926,0,0xf6d0,1,30609,0,0xe1d2,1,30933,0,0xf6d1
,1,30944,0,0xf6d2,1,30612,0,0xecf8,1,30945,0,0xf6d3
,1,30950,0,0xf6d4,1,30969,0,0xf6d5,1,30970,0,0xf6d6
,1,30971,0,0xf6d7,1,30618,0,0xf6ac,1,43833,0,0xf6d9
,1,30992,0,0xf6da,1,31003,0,0xf6db,1,30622,0,0xe1d4
,1,30623,0,0xf6ad,1,30624,0,0xe1d3,1,31024,0,0xf6dc
,1,30626,0,0xf6ae,1,31013,0,0xf6dd,1,30628,0,0xf6af
,1,30629,0,0x95cb,1,31035,0,0xf6de,1,16645,0,0xf6e2
,1,31145,0,0xf6ea,1,30633,0,0xf6b0,1,30634,0,0xecf9
,1,31163,0,0xf6ec,1,30636,0,0x8f75,1,30637,0,0x97c4
,1,16712,10262,0xf6f2,1,43835,8174,0xf6f3,1,30640,0,0xe1d5
,1,43836,0,0xf6f4,1,16719,10662,0xf6f5,1,30643,0,0x93b5
,1,16739,10897,0xf6fb,1,31404,0,0xf750,1,30646,0,0xe1d6
,1,43838,7859,0xf751,1,16820,0,0xf752,1,30649,0,0xe1d7
,1,16831,8692,0xf756,1,30651,0,0xe1db,1,30652,0,0xe1d9
,1,30653,0,0xe1da,1,31436,0,0xf757,1,30655,0,0xe1d8
,1,31476,0,0xf75b,1,43839,9176,0xf75c,1,43840,7860,0xf75d
,1,31495,0,0xf75e,1,43841,0,0xf75f,1,31527,0,0xf761
,1,31530,0,0xf762,1,30663,0,0xe1dc,1,31534,0,0xf763
,1,31535,0,0xf764,1,16870,0,0xf766,1,16883,0,0xf767
,1,31615,0,0xf768,1,30669,0,0xe1dd,1,16878,0,0xf76a
,1,31573,0,0xf76b,1,31609,0,0xf76c,1,31588,0,0xf76d
,1,31590,0,0xf76e,1,31593,0,0xf76f,1,7742,0,0x84f0
,1,7743,0,0x84f1,1,31603,0,0xf770,1,30679,0,0xe1de
,1,43842,0,0xf771,1,16903,0,0xf772,1,30682,0,0xe1df
,1,30683,0,0x96b5,1,30684,0,0xe1e0,1,31633,0,0xf774
,1,30686,0,0xf6b1,1,30687,0,0xf6b2,1,30688,0,0xecfa
,1,31643,0,0xf775,1,30690,0,0x96ee,1,30691,0,0xe1e1
,1,30692,0,0xf6b3,1,30693,0,0x926d,1,30694,0,0xf6b4
,1,30695,0,0x948a,1,16910,0,0xf776,1,30697,0,0x8be9
,1,30698,0,0xf6b5,1,31663,0,0xf777,1,30700,0,0xf6b6
,1,30701,0,0x925a,1,30702,0,0xe1e2,1,30703,0,0x8bb8
,1,30704,0,0xf6b8,1,31669,0,0xf778,1,31676,0,0xf779
,1,30707,0,0x90ce,1,30708,0,0xf6b9,1,31690,0,0xf77b
,1,43843,0,0xf77c,1,43844,9598,0xf77d,1,31700,0,0xf77e
,1,31702,0,0xf780,1,31706,0,0xf781,1,30715,0,0xf6ba
,1,30716,0,0xe1e3,1,31747,0,0xf784,1,31758,0,0xf786
,1,31782,0,0xf788,1,31813,0,0xf789,1,31818,0,0xf78a
,1,30722,0,0x8dbb,1,31825,0,0xf78b,1,31841,0,0xf78e
,1,30725,0,0xf6bc,1,30726,10480,0xf6bd,1,31856,0,0xf792
,1,43846,0,0xf793,1,30729,0,0xf6be,1,43847,0,0xf794
,1,43849,7861,0xf795,1,30732,0,0xe1e4,1,30733,0,0xf6bf
,1,31910,0,0xf796,1,43848,0,0xf797,1,43851,0,0xf79b
,1,43850,0,0xf79d,1,30738,0,0xe1e5,1,31949,0,0xf79f
,1,30740,0,0x8ca4,1,30741,0,0x8dd3,1,43852,0,0xf7a0
,1,43853,0,0xf7a2,1,43854,0,0xf7a5,1,30745,0,0xf6c0
,1,31989,0,0xf7a6,1,17094,10835,0xf7a9,1,32018,0,0xf7aa
,1,32030,0,0xf7ab,1,32061,0,0xf7ae,1,32062,0,0xf7af
,1,30752,0,0xe1e7,1,30753,0,0xf6c1,1,32064,0,0xf7b0
,1,32071,0,0xf7b1,1,43858,7863,0xf7b3,1,30757,0,0x9375
,1,30758,0,0x8dd4,1,30759,0,0x8b6d,1,17110,0,0xf7b4
,1,17117,0,0xf7b9,1,32127,0,0xf7ba,1,43859,0,0xf7bb
,1,30764,0,0xf6c2,1,30765,0,0xecfb,1,32134,0,0xf7bc
,1,32151,0,0xf7bf,1,43860,0,0xf7c0,1,32192,0,0xf7c6
,1,30770,0,0x9643,1,32215,0,0xf7c7,1,30772,0,0x946a
,1,32217,0,0xf7c8,1,32230,0,0xf7c9,1,32241,0,0xf7ca
,1,32249,0,0xf7cb,1,17154,0,0xf7cc,1,30778,0,0x9376
,1,43861,0,0xf7cd,1,64089,0,0xf7ce,1,32272,0,0xf7cf
,1,32279,11107,0xf7d0,1,30783,0,0x8d7b,1,32285,0,0xf7d1
,1,32288,0,0xf7d2,1,32295,0,0xf7d3,1,30787,0,0xed40
,1,32300,0,0xf7d4,1,30789,0,0xe1e9,1,32325,0,0xf7d5
,1,30791,0,0xf6c3,1,32371,0,0xf7d6,1,32373,0,0xf7d7
,1,32382,0,0xf7d8,1,32390,0,0xf7d9,1,32391,0,0xf7da
,1,17195,0,0xf7db,1,30798,0,0xed41,1,30799,0,0xed42
,1,32401,7920,0xf7dc,1,30801,0,0xed43,1,32408,8124,0xf7dd
,1,17219,0,0xf7df,1,32572,0,0xf7e0,1,32571,0,0xf7e1
,1,32574,0,0xf7e2,1,32591,0,0xf7e5,1,13505,0,0xf7e6
,1,32594,0,0xf7e8,1,43863,0,0xf7e9,1,32609,0,0xf7ea
,1,32611,0,0xf7eb,1,30813,0,0x8fc9,1,32612,0,0xf7ec
,1,43864,0,0xf7f0,1,32656,0,0xf7f1,1,20859,0,0xf7f2
,1,43865,9608,0xf7f3,1,32662,0,0xf7f4,1,30820,0,0xf6c4
,1,32668,0,0xf7f5,1,32685,0,0xf7f6,1,43866,0,0xf7f7
,1,30824,0,0xed44,1,32707,0,0xf7f8,1,30826,0,0xf6c5
,1,30827,0,0x97b0,1,30828,0,0x8d64,1,32741,0,0xf7fb
,1,30830,0,0xed45,1,30831,0,0x8ca5,1,32751,0,0xf7fc
,1,32770,0,0xf841,1,30834,0,0x94a1,1,32778,0,0xf842
,1,30836,0,0xe1eb,1,32782,0,0xf844,1,32785,0,0xf845
,1,32812,0,0xf848,1,32835,0,0xf84a,1,32881,0,0xf84c
,1,32891,0,0xf84e,1,32921,0,0xf84f,1,134047,2590,0x9873
,1,32932,0,0xf851,1,32952,0,0xf853,1,43867,0,0xf854
,1,32965,0,0xf855,1,30849,0,0x8ce9,1,32981,0,0xf856
,1,32984,0,0xf857,1,32998,0,0xf858,1,43868,0,0xf859
,1,30854,0,0xe1ec,1,30855,0,0x92f4,1,33037,0,0xf85a
,1,17390,10986,0xf85d,1,30858,0,0xf6c7,1,33077,0,0xf85e
,1,30860,0,0xe1ef,1,30861,10619,0x8a56,1,30862,0,0xe1ea
,1,17392,0,0xf861,1,43869,0,0xf865,1,30865,0,0x94e8
,1,17416,0,0xf867,1,30867,0,0x894f,1,30868,0,0xf6c8
,1,30869,10622,0x8dea,1,33129,0,0xf868,1,30871,0,0x9871
,1,17431,0,0xf869,1,33153,0,0xf86a,1,30874,0,0xe1ee
,1,17436,0,0xf86b,1,33156,0,0xf86c,1,30877,10629,0xf6ca
,1,30878,0,0xf6cb,1,30879,0,0xf6cc,1,65281,0,0x8149
,1,65282,0,0x81ae,1,65283,0,0x8194,1,65284,2593,0x8190
,1,65285,7669,0x8193,1,65286,0,0x8195,1,65287,0,0x81ad
,1,65288,1412,0x8169,1,65289,0,0x816a,1,65290,862,0x8196
,1,65291,2595,0x817b,1,65292,0,0x8143,1,65293,0,0x81af
,1,65294,4909,0x8144,1,65295,0,0x815e,1,65296,2597,0x824f
,1,65297,4898,0x8250,1,65298,0,0x8251,1,65299,0,0x8252
,1,65300,10645,0x8253,1,65301,0,0x8254,1,65302,2594,0x8255
,1,65303,0,0x8256,1,65304,0,0x8257,1,65305,0,0x8258
,1,65306,0,0x8146,1,65307,439,0x8147,1,65308,7671,0x8183
,1,65309,2608,0x8181,1,65310,0,0x8184,1,65311,2605,0x8148
,1,65312,0,0x8197,1,65313,0,0x8260,1,65314,666,0x8261
,1,65315,0,0x8262,1,65316,0,0x8263,1,65317,0,0x8264
,1,65318,2610,0x8265,1,65319,2600,0x8266,1,65320,0,0x8267
,1,65321,7672,0x8268,1,65322,0,0x8269,1,65323,2611,0x826a
,1,65324,2602,0x826b,1,65325,7674,0x826c,1,65326,0,0x826d
,1,65327,7675,0x826e,1,65328,0,0x826f,1,65329,1346,0x8270
,1,65330,2599,0x8271,1,65331,0,0x8272,1,65332,0,0x8273
,1,65333,2604,0x8274,1,65334,7677,0x8275,1,65335,0,0x8276
,1,65336,0,0x8277,1,65337,0,0x8278,1,65338,0,0x8279
,1,65339,2615,0x816d,1,33157,0,0xf86d,1,65341,0,0x816e
,1,65342,0,0x814f,1,65343,0,0x8151,1,65344,0,0x814d
,1,65345,7678,0x8281,1,65346,7680,0x8282,1,65347,5664,0x8283
,1,65348,0,0x8284,1,65349,4910,0x8285,1,65350,0,0x8286
,1,65351,7681,0x8287,1,65352,2614,0x8288,1,65353,1454,0x8289
,1,65354,0,0x828a,1,65355,0,0x828b,1,65356,10695,0x828c
,1,65357,2613,0x828d,1,65358,0,0x828e,1,65359,0,0x828f
,1,65360,411,0x8290,1,65361,0,0x8291,1,65362,0,0x8292
,1,65363,4914,0x8293,1,65364,0,0x8294,1,65365,2616,0x8295
,1,65366,0,0x8296,1,65367,0,0x8297,1,65368,4922,0x8298
,1,65369,0,0x8299,1,65370,7682,0x829a,1,65371,7683,0x816f
,1,65372,7684,0x8162,1,65373,0,0x8170,1,30973,0,0xe242
,1,30974,0,0xf6d8,1,17442,0,0xf86e,1,65377,4915,0xa1
,1,65378,748,0xa2,1,65379,0,0xa3,1,65380,0,0xa4
,1,65381,0,0xa5,1,65382,0,0xa6,1,65383,306,0xa7
,1,65384,307,0xa8,1,65385,308,0xa9,1,65386,311,0xaa
,1,65387,0,0xab,1,65388,0,0xac,1,65389,0,0xad
,1,65390,0,0xae,1,65391,981,0xaf,1,65392,0,0xb0
,1,65393,7687,0xb1,1,65394,2622,0xb2,1,65395,2620,0xb3
,1,65396,0,0xb4,1,65397,0,0xb5,1,65398,0,0xb6
,1,65399,0,0xb7,1,65400,0,0xb8,1,65401,0,0xb9
,1,65402,2623,0xba,1,65403,0,0xbb,1,65404,7688,0xbc
,1,65405,4923,0xbd,1,65406,0,0xbe,1,65407,0,0xbf
,1,65408,0,0xc0,1,65409,0,0xc1,1,65410,0,0xc2
,1,65411,0,0xc3,1,65412,0,0xc4,1,65413,0,0xc5
,1,65414,7694,0xc6,1,65415,2585,0xc7,1,65416,0,0xc8
,1,65417,0,0xc9,1,65418,0,0xca,1,65419,2587,0xcb
,1,65420,2626,0xcc,1,65421,2624,0xcd,1,65422,0,0xce
,1,65423,4924,0xcf,1,65424,0,0xd0,1,65425,7692,0xd1
,1,65426,4925,0xd2,1,65427,0,0xd3,1,65428,0,0xd4
,1,65429,4929,0xd5,1,65430,0,0xd6,1,65431,0,0xd7
,1,65432,0,0xd8,1,65433,0,0xd9,1,65434,0,0xda
,1,65435,667,0xdb,1,65436,7697,0xdc,1,65437,1588,0xdd
,1,65438,0,0xde,1,65439,679,0xdf,1,33243,0,0xf874
,1,31040,0,0xe24a,1,31041,0,0x8c56,1,43870,0,0xf875
,1,43871,0,0xf877,1,43872,0,0xf879,1,31045,0,0xed54
,1,31046,0,0xed55,1,31047,0,0x8b5f,1,31048,0,0x8b46
,1,31049,0,0x8e83,1,31050,0,0xf6df,1,43873,9607,0xf87c
,1,33284,0,0xf87d,1,43874,0,0xf87e,1,33313,0,0xf881
,1,33314,0,0xf882,1,31056,0,0x9753,1,43875,0,0xf883
,1,33350,0,0xf887,1,31059,0,0xe250,1,33353,0,0xf888
,1,31061,0,0xe24f,1,31062,0,0x9163,1,31063,0,0xe24c
,1,31064,0,0xf6e0,1,43876,0,0xf88a,1,31066,0,0xe24e
,1,31067,0,0xf6e1,1,31068,0,0xed59,1,31069,0,0x8f6a
,1,31070,0,0x905f,1,31071,0,0xe24d,1,31072,0,0xe24b
,1,33355,0,0xf88b,1,31074,0,0x9449,1,17526,0,0xf88c
,1,33359,0,0xf88d,1,31077,0,0x8fcb,1,17530,0,0xf88e
,1,31079,0,0xf6e3,1,31080,0,0x955b,1,43877,0,0xf890
,1,33372,0,0xf891,1,33379,0,0xf892,1,43878,0,0xf893
,1,31085,0,0x8dd5,1,43879,0,0xf894,1,64094,0,0xf895
,1,33401,0,0xf896,1,17553,0,0xf897,1,31090,0,0xf6e4
,1,33418,0,0xf89b,1,33447,0,0xf89d,1,33448,0,0xf89e
,1,33460,0,0xf8a0,1,31095,0,0x9398,1,33506,0,0xf8a3
,1,31097,0,0xed5d,1,31098,0,0xe251,1,33544,0,0xf8a7
,1,33563,0,0xf8aa,1,33596,0,0xf8ad,1,33623,0,0xf8af
,1,31103,0,0xe252,1,31104,0,0xe268,1,31105,0,0x8bd6
,1,65507,0,0x8150,1,17598,0,0xf8b0,1,65509,1630,0x818f
,1,31109,0,0x9154,1,33663,0,0xf8b1,1,17620,0,0xf8b2
,1,17587,0,0xf8b3,1,33677,0,0xf8b4,1,31114,0,0xe253
,1,33684,0,0xf8b5,1,33685,0,0xf8b6,1,31117,0,0x89d0
,1,31118,0,0x92f5,1,31119,10924,0x959f,1,33748,0,0xf8bb
,1,33757,0,0xf8bc,1,33809,0,0xf8c0,1,33813,0,0xf8c1
,1,31124,0,0xf6e5,1,31125,0,0xf6e6,1,31126,0,0xf6e7
,1,33815,0,0xf8c3,1,31128,0,0xed61,1,33866,0,0xf8c5
,1,33893,0,0xf8cd,1,31131,0,0xf6e8,1,17677,10928,0xf8d2
,1,31133,0,0xe254,1,33943,0,0xf8d4,1,17672,0,0xf8d7
,1,33999,0,0xf8d9,1,31137,0,0xf6e9,1,43884,0,0xf8db
,1,34031,0,0xf8de,1,34042,0,0xf8e1,1,34045,0,0xf8e2
,1,8208,482,0x815d,1,31143,0,0xe255,1,34060,0,0xf8e3
,1,8211,7699,0x829c,1,8212,2631,0x815c,1,34084,0,0xf8e5
,1,8214,0,0x8161,1,34085,0,0xf8e6,1,8216,2632,0x8165
,1,8217,0,0x8166,1,31152,0,0x9448,1,31153,0,0xed62
,1,8220,0,0x8167,1,8221,2633,0x8168,1,31156,0,0xf6eb
,1,34100,0,0xf8e8,1,8224,0,0x81f5,1,8225,0,0x81f6
,1,8226,4938,0x825f,1,31161,0,0xe25a,1,31162,0,0xe25b
,1,8229,7702,0x8164,1,8230,0,0x8163,1,31165,0,0x8bd7
,1,31166,0,0x89d1,1,31167,0,0x93c3,1,31168,0,0x8f47
,1,31169,0,0x8e84,1,31170,0,0xf6ed,1,17701,0,0xf8eb
,1,17731,0,0xf8ec,1,34142,0,0xf8f0,1,8240,0,0x81f1
,1,31175,0,0xf6ee,1,8242,4943,0x818c,1,8243,2635,0x818d
,1,31178,0,0xed65,1,31179,0,0x8f48,1,31180,0,0xf6ef
,1,31181,0,0xf6f0,1,34146,0,0xf8f2,1,43885,0,0xf8f3
,1,34182,0,0xf8f8,1,8251,424,0x81a6,1,8252,1370,0x84e9
,1,34205,0,0xf8fa,1,8254,4946,0x7e,1,8255,2637,0x8677
,1,31190,0,0xf6f1,1,34207,0,0xf8fb,1,8258,1352,0x86fc
,1,43888,0,0xf941,1,34231,0,0xf942,1,34236,0,0xf943
,1,34247,0,0xf944,1,8263,0,0x84ea,1,8264,4951,0x84eb
,1,8265,982,0x84ec,1,34264,0,0xf946,1,31201,0,0xe260
,1,34271,0,0xf948,1,31203,0,0xe261,1,8270,1329,0x829b
,1,34273,0,0xf949,1,8272,785,0x81d4,1,8273,2639,0x81d5
,1,8274,0,0x8259,1,8275,1167,0x86fb,1,34278,0,0xf94a
,1,31211,0,0xed69,1,31212,0,0xe25f,1,31213,0,0xed6a
,1,34294,0,0xf94b,1,34321,0,0xf94d,1,31216,0,0x8fcc
,1,43890,0,0xf952,1,34361,0,0xf953,1,43891,0,0xf955
,1,34368,0,0xf956,1,64032,10062,0xf957,1,34390,0,0xf959
,1,34415,0,0xf95a,1,34423,0,0xf95b,1,34426,0,0xf95c
,1,34439,0,0xf95d,1,31227,0,0x88da,1,34441,0,0xf95e
,1,34445,0,0xf95f,1,34449,0,0xf960,1,34460,0,0xf961
,1,31232,0,0x8b48,1,34461,0,0xf962,1,34472,0,0xf963
,1,31235,0,0xed6b,1,64033,0,0xf964,1,34481,0,0xf965
,1,34483,0,0xf966,1,34497,0,0xf967,1,31240,0,0xe262
,1,34499,0,0xf968,1,31242,0,0xf6f6,1,31243,0,0x92f6
,1,34513,0,0xf969,1,31245,0,0xe263,1,31246,0,0x90c5
,1,34517,0,0xf96a,1,34519,0,0xf96b,1,31249,0,0xf6f7
,1,34531,0,0xf96c,1,34534,0,0xf96d,1,31252,0,0x96ab
,1,31253,0,0xf6f8,1,17848,0,0xf96e,1,31255,0,0x9542
,1,31256,0,0xe264,1,31257,0,0xe265,1,31258,0,0x9274
,1,31259,0,0xf6f9,1,31260,0,0x97c5,1,34565,8339,0xf96f
,1,31262,0,0xf6fa,1,31263,0,0xe267,1,31264,0,0xe266
,1,34567,8341,0xf970,1,34574,8342,0xf971,1,34576,8343,0xf972
,1,34579,0,0xf973,1,34585,0,0xf974,1,34591,10167,0xf975
,1,34593,11086,0xf976,1,34595,8351,0xf977,1,34609,8353,0xf978
,1,34618,8352,0xf979,1,34622,8371,0xf97a,1,34624,0,0xf97b
,1,31277,0,0xf6fc,1,31278,0,0x8eed,1,34627,0,0xf97c
,1,34641,0,0xf97d,1,31281,0,0xe269,1,31282,0,0x88ee
,1,34648,0,0xf97e,1,34660,0,0xf980,1,34661,9082,0xf981
,1,34674,9144,0xf982,1,31287,0,0xe26c,1,31288,0,0xf740
,1,31289,0,0xed6d,1,34684,0,0xf983,1,31291,0,0xe26a
,1,31292,0,0x89d2,1,31293,0,0x8c6d,1,31294,0,0xe26b
,1,31295,0,0x8d65,1,31296,0,0x8d92,1,43893,0,0xf984
,1,8364,1427,0x8540,1,31299,0,0xe26d,1,43894,0,0xf985
,1,34727,0,0xf986,1,31302,0,0x9673,1,31303,0,0xf741
,1,34697,0,0xf987,1,31305,0,0xe26f,1,34699,0,0xf988
,1,34707,0,0xf989,1,31308,0,0xf742,1,31309,0,0x90cf
,1,31310,0,0x896e,1,31311,0,0x89b8,1,31312,0,0x88aa
,1,34720,0,0xf98a,1,43892,0,0xf98b,1,17893,10717,0xf98c
,1,34750,0,0xf98d,1,43895,0,0xf98e,1,31318,0,0xf743
,1,31319,0,0xe26e,1,34753,0,0xf98f,1,31321,0,0xf744
,1,34766,0,0xf990,1,34805,8947,0xf991,1,31324,0,0xf745
,1,31325,0,0xed6e,1,34783,0,0xf992,1,31327,0,0xf746
,1,31328,0,0xf747,1,31329,0,0xe270,1,31330,0,0xe271
,1,31331,0,0x8ff5,1,43896,0,0xf993,1,34787,8741,0xf994
,1,34789,8407,0xf995,1,31335,0,0xf748,1,34790,9861,0xf996
,1,31337,0,0xe272,1,31338,0,0xf749,1,31339,0,0x8a6e
,1,34794,10405,0xf997,1,31341,0,0xed6f,1,34795,0,0xf998
,1,34797,0,0xf999,1,31344,0,0xe274,1,34817,0,0xf99a
,1,34819,0,0xf99b,1,34827,0,0xf99c,1,31348,0,0x8c8a
,1,31349,0,0xf74a,1,31350,0,0x8b86,1,34835,0,0xf99d
,1,31352,0,0xf74b,1,31353,0,0xe275,1,31354,0,0x8bf3
,1,34856,0,0xf99e,1,34862,0,0xf99f,1,31357,0,0xe276
,1,34866,0,0xf9a0,1,31359,0,0x90fa,1,34876,0,0xf9a1
,1,31361,0,0x93cb,1,31362,0,0xf74c,1,31363,0,0x90de
,1,31364,0,0x8df3,1,31365,0,0xed71,1,17935,0,0xf9a2
,1,34890,0,0xf9a3,1,31368,0,0xe277,1,34904,0,0xf9a4
,1,31370,0,0xf74d,1,34911,0,0xf9a5,1,34916,9571,0xf9a6
,1,43897,0,0xf9a7,1,34921,0,0xf9a9,1,34927,8470,0xf9ab
,1,31376,0,0xf74e,1,34976,9842,0xf9ac,1,31378,0,0x9282
,1,31379,0,0x918b,1,35004,0,0xf9ad,1,31381,0,0xe279
,1,31382,0,0xe27b,1,31383,0,0xe278,1,31384,0,0xe27a
,1,8451,0,0x818e,1,35005,0,0xf9ae,1,35006,8991,0xf9af
,1,35008,8691,0xf9b0,1,35026,0,0xf9b1,1,43901,0,0xf9b2
,1,31391,0,0x8c41,1,31392,0,0xed72,1,35025,0,0xf9b3
,1,35027,0,0xf9b4,1,31395,0,0xf74f,1,35035,0,0xf9b5
,1,8463,0,0x827c,1,35056,0,0xf9b6,1,35057,0,0xf9b7
,1,17985,0,0xf9b8,1,8467,2641,0x827e,1,31402,0,0x8c45
,1,35073,0,0xf9b9,1,8470,7712,0x8782,1,43902,0,0xf9ba
,1,31406,0,0x8b87,1,31407,0,0x9771,1,31408,0,0xe27e
,1,35127,0,0xf9bb,1,43903,0,0xf9bc,1,31411,0,0xed74
,1,35138,0,0xf9bd,1,35141,0,0xf9be,1,31414,0,0xe280
,1,8481,0,0x8784,1,35145,0,0xf9bf,1,31417,0,0xf753
,1,31418,0,0x894d,1,31419,0,0xed75,1,31420,0,0xf754
,1,8487,0,0x8280,1,31422,0,0xf755,1,31423,0,0xe283
,1,43904,0,0xf9c0,1,8491,0,0x81f0,1,18021,0,0xf9c1
,1,31427,0,0x8a96,1,31428,0,0xe282,1,31429,0,0xe281
,1,35170,0,0xf9c2,1,31431,0,0xe285,1,31432,0,0xe27d
,1,35200,0,0xf9c3,1,31434,0,0xe286,1,8501,412,0x827b
,1,19969,7721,0x929a,1,31437,5731,0xe287,1,19971,4957,0x8eb5
,1,31439,0,0xe288,1,35209,0,0xf9c4,1,31441,0,0xf758
,1,19975,1878,0x969c,1,19976,2645,0x8fe4,1,19977,0,0x8e4f
,1,19978,2643,0x8fe3,1,19979,0,0x89ba,1,35216,0,0xf9c5
,1,19981,0,0x9573,1,19982,2646,0x975e,1,31450,5734,0xe28c
,1,19984,0,0x98a0,1,19985,1564,0x894e,1,31453,5735,0xe28d
,1,35231,0,0xf9c6,1,19988,4243,0x8a8e,1,31456,1711,0x8fcd
,1,19990,2647,0x90a2,1,19991,2650,0x99c0,1,19992,714,0x8b75
,1,19993,0,0x95b8,1,31461,0,0x93b6,1,31462,0,0xe290
,1,31463,0,0xf759,1,31464,0,0xf75a,1,8531,761,0x8498
,1,8532,1156,0x8499,1,8533,4959,0x849a,1,20001,0,0x97bc
,1,31469,0,0xe291,1,35248,0,0xf9c7,1,31471,0,0x925b
,1,31472,0,0xe292,1,20006,0,0x95c0,1,35255,0,0xf9c8
,1,20008,0,0x87a2,1,20009,7723,0xf044,1,20010,0,0x98a2
,1,8544,483,0x8754,1,8545,7349,0x8755,1,8546,1170,0x8756
,1,8547,1165,0x8757,1,8548,320,0x8758,1,8549,321,0x8759
,1,8550,1712,0x875a,1,8551,494,0x875b,1,8552,0,0x875c
,1,8553,450,0x875d,1,8554,0,0x875e,1,8555,1717,0x8776
,1,31490,0,0xe293,1,20024,0,0x8adb,1,20025,2660,0x924f
,1,35286,0,0xf9c9,1,8560,685,0x86b3,1,8561,1719,0x86b4
,1,8562,479,0x86b5,1,8563,0,0x86b6,1,8564,1720,0x86b7
,1,8565,2663,0x86b8,1,8566,0,0x86b9,1,8567,1721,0x86ba
,1,8568,1275,0x86bb,1,8569,2651,0x86bc,1,8570,478,0x86bd
,1,8571,749,0x86be,1,31506,5741,0xed79,1,20040,0,0xf049
,1,35288,0,0xf9ca,1,35307,11081,0xf9cb,1,20043,0,0x9456
,1,18081,0,0xf9cc,1,20045,2654,0x93e1,1,20046,2656,0x8cc1
,1,20047,0,0x9652,1,31515,0,0x934a,1,20049,0,0xf04b
,1,35313,0,0xf9cd,1,31518,0,0xe29a,1,35315,0,0xf9ce
,1,31520,3057,0x8a7d,1,20054,0,0x98a8,1,20055,0,0x8fe6
,1,20056,0,0x98a9,1,20057,0,0x89b3,1,31525,5746,0x9079
,1,8592,1387,0x81a9,1,8593,7728,0x81aa,1,8594,490,0x81a8
,1,8595,541,0x81ab,1,8596,1481,0x81ef,1,35325,0,0xf9cf
,1,8598,1147,0x8246,1,8599,1996,0x8244,1,8600,7731,0x8245
,1,8601,7732,0x8247,1,35327,0,0xf9d0,1,31537,0,0xf765
,1,18095,0,0xf9d1,1,31539,0,0xe297,1,20073,0,0xf04f
,1,31541,0,0xe29b,1,31542,0,0xe29d,1,35345,0,0xf9d2
,1,35348,0,0xf9d3,1,31545,0,0x8df9,1,43905,0,0xf9d4
,1,31547,0,0xed7b,1,20081,0,0x9790,1,31549,0,0xf760
,1,20083,0,0x93fb,1,35361,0,0xf9d5,1,35381,0,0xf9d6
,1,31553,0,0xf769,1,35390,0,0xf9d7,1,35397,0,0xf9d8
,1,35405,0,0xf9d9,1,31557,0,0xe2a4,1,31558,0,0x954d
,1,31559,0,0xed7c,1,31560,0,0x94a4,1,20094,1229,0x8aa3
,1,35416,0,0xf9da,1,20096,488,0x8b54,1,31564,0,0xe2a3
,1,20098,2662,0x98aa,1,31566,0,0xed7d,1,31567,0,0x94b3
,1,20101,2659,0x98ab,1,31569,1569,0x927d,1,31570,0,0x939b
,1,20104,0,0x975c,1,20105,1230,0x9188,1,20106,7738,0x98ad
,1,31574,656,0x8df4,1,20108,0,0x93f1,1,20109,0,0x87a5
,1,20110,0,0x98b0,1,8644,0,0x8248,1,35502,0,0xf9db
,1,20113,0,0x895d,1,20114,2679,0x8cdd,1,35472,0,0xf9dc
,1,20116,0,0x8cdc,1,20117,4963,0x88e4,1,35511,0,0xf9dd
,1,35518,0,0xf9de,1,20120,0,0x986a,1,20121,7740,0x9869
,1,31589,0,0xe2a6,1,20123,7741,0x8db1,1,20124,2664,0x889f
,1,8658,5747,0x81cb,1,20126,7742,0x98b1,1,8660,1726,0x81cc
,1,20128,0,0x98b3,1,20129,2668,0x9653,1,20130,4967,0x98b4
,1,31598,0,0xe2ac,1,20132,4969,0x8cf0,1,20133,2665,0x88e5
,1,20134,2667,0x9692,1,31602,0,0xed82,1,20136,7745,0x8b9c
,1,31604,0,0xe2a7,1,31605,0,0xe2a5,1,20139,0,0x8b9d
,1,20140,0,0x8b9e,1,20141,0,0x92e0,1,20142,7739,0x97ba
,1,31610,0,0xe29f,1,20144,0,0x98b5,1,8678,0,0x824a
,1,8679,0,0x824b,1,8680,1727,0x8249,1,8681,7735,0x824c
,1,35543,0,0xf9df,1,20150,0,0x98b7,1,35580,0,0xf9e0
,1,35594,0,0xf9e2,1,20153,0,0xf053,1,20154,0,0x906c
,1,31622,5752,0x95cd,1,31623,5755,0x89d3,1,35589,0,0xf9e3
,1,35597,0,0xf9e4,1,8692,0,0x824d,1,8693,704,0x824e
,1,20161,0,0x906d,1,20162,2674,0x98bc,1,20163,0,0xf057
,1,20164,2678,0x98ba,1,31632,0,0xf773,1,20166,7752,0x98bb
,1,20167,2677,0x8b77,1,20168,0,0xf058,1,31636,0,0x9493
,1,20170,1462,0x8da1,1,8704,431,0x81cd,1,31639,0,0x8e5a
,1,8706,1730,0x81dd,1,8707,1729,0x81ce,1,20175,2675,0x95a7
,1,8709,5757,0x81c5,1,31644,0,0xe2b1,1,8711,2669,0x81de
,1,8712,4970,0x81b8,1,8713,618,0x81c4,1,20181,0,0x8e64
,1,8715,452,0x81b9,1,20183,0,0x98bd,1,20184,0,0x9574
,1,20185,0,0x90e5,1,20186,0,0xf05b,1,35612,0,0xf9e5
,1,35615,0,0xf9e6,1,20189,11468,0x8157,1,8723,1731,0x827a
,1,31658,1735,0x925c,1,35629,0,0xf9e7,1,20193,0,0x87a6
,1,31661,0,0x90fb,1,20195,0,0x91e3,1,20196,7766,0x97df
,1,8730,406,0x81e3,1,31665,0,0x94a0,1,35651,0,0xf9e8
,1,8733,0,0x81e5,1,8734,2687,0x8187,1,8735,7772,0x8798
,1,8736,5758,0x81da,1,18188,0,0xf9e9,1,31672,1734,0x94a2
,1,20206,0,0x89bc,1,35665,0,0xf9ea,1,8741,487,0x81d2
,1,8742,5759,0x81d3,1,8743,1171,0x81c8,1,8744,0,0x81c9
,1,8745,0,0x81bf,1,8746,864,0x81be,1,8747,516,0x81e7
,1,8748,1736,0x81e8,1,35678,0,0xf9eb,1,8750,1345,0x8793
,1,31685,0,0xf77a,1,20219,2688,0x9443,1,31687,0,0x95d1
,1,20221,0,0x87a7,1,31689,0,0x927a,1,8756,326,0x8188
,1,8757,2683,0x81e6,1,20225,2685,0x8ae9,1,35702,0,0xf9ec
,1,20227,0,0x87a9,1,31695,0,0xe2bb,1,35711,0,0xf9ed
,1,35713,0,0xf9ee,1,35723,0,0xf9ef,1,8765,0,0x81e4
,1,20233,7779,0x98c2,1,20234,0,0x88c9,1,20235,7780,0x87aa
,1,31703,0,0xed85,1,20237,0,0x8cde,1,8771,458,0x81ea
,1,20239,7781,0x959a,1,8773,1335,0x81eb,1,20241,0,0x8b78
,1,31709,0,0xe2be,1,8776,0,0x81ec,1,35732,0,0xf9f0
,1,31712,0,0x8ec2,1,20246,0,0xf05f,1,35733,0,0xf9f1
,1,35740,0,0xf9f2,1,31716,0,0x93c4,1,20250,2693,0x89ef
,1,31718,0,0xe2c2,1,20252,0,0x98e5,1,8786,1218,0x81e0
,1,31721,0,0xe2bf,1,31722,0,0xf782,1,35742,0,0xf9f3
,1,35897,0,0xf9f4,1,31725,0,0x9855,1,43907,0,0xf9f5
,1,35901,11124,0xf9f6,1,31728,0,0xf783,1,43908,0,0xf9f7
,1,43909,0,0xf9f8,1,31731,0,0xe2c8,1,35909,0,0xf9f9
,1,35911,0,0xf9fa,1,8800,2702,0x8182,1,8801,2699,0x81df
,1,8802,0,0x81e9,1,35919,0,0xf9fb,1,20271,0,0x948c
,1,20272,0,0x98c4,1,8806,0,0x8185,1,8807,0,0x8186
,1,35924,0,0xf9fc,1,20276,0,0x94ba,1,8810,2694,0x81e1
,1,8811,1581,0x81e2,1,20279,0,0xf061,1,20280,7784,0x904c
,1,35927,0,0xfa40,1,20282,0,0x8e66,1,35945,9149,0xfa41
,1,20284,2697,0x8e97,1,20285,0,0x89be,1,20286,0,0xf062
,1,35949,0,0xfa42,1,31755,0,0xf785,1,8822,0,0x81ed
,1,8823,2700,0x81ee,1,20291,7785,0x92cf,1,31759,0,0xf787
,1,35955,0,0xfa43,1,20294,2691,0x9241,1,20295,1819,0x98c8
,1,20296,2698,0x87ac,1,20297,2692,0x87ad,1,43910,0,0xfa44
,1,35987,0,0xfa45,1,31767,0,0xe2ca,1,8834,407,0x81bc
,1,8835,1204,0x81bd,1,8836,705,0x81c0,1,8837,579,0x81c1
,1,8838,1487,0x81ba,1,8839,0,0x81bb,1,20307,4973,0x91cc
,1,31775,5764,0xe2d0,1,8842,422,0x81c2,1,8843,330,0x81c3
,1,20311,0,0x98c7,1,31779,5773,0xe2cd,1,20313,0,0x975d
,1,20314,0,0x98c3,1,20315,7786,0x98c5,1,20316,2703,0x8dec
,1,20317,0,0x98c6,1,20318,0,0x9b43,1,20319,2704,0x87af
,1,8853,327,0x81cf,1,8854,0,0x81d0,1,8855,0,0x81d1
,1,35986,0,0xfa46,1,20324,0,0xf060,1,35993,0,0xfa47
,1,31793,0,0xed88,1,18276,0,0xfa48,1,31795,0,0xed8b
,1,20329,0,0x98ce,1,20330,0,0x87b0,1,31798,0,0xed8c
,1,20332,2706,0x87b1,1,31800,0,0x94f4,1,35995,0,0xfa49
,1,20335,0,0x98d1,1,8869,1737,0x81db,1,36004,0,0xfa4a
,1,31805,0,0xe2d3,1,20339,1599,0x89c0,1,31807,0,0x95eb
,1,20341,2710,0x95b9,1,20342,0,0x98c9,1,20343,0,0xf066
,1,31811,5774,0xe2d5,1,36054,0,0xfa4b,1,20346,7787,0xf068
,1,20347,0,0x98cd,1,20348,0,0x8cf1,1,20349,0,0xf069
,1,20350,0,0x87b2,1,20351,7788,0x8e67,1,36053,0,0xfa4c
,1,31820,0,0xe2d4,1,31821,5775,0x90d0,1,20355,0,0x8aa4
,1,31823,0,0xe2d7,1,31824,5781,0xe2d9,1,20358,7790,0x98d2
,1,36057,0,0xfa4d,1,20360,0,0x98ca,1,31828,0,0xe2d6
,1,8895,331,0x8799,1,20363,2715,0x97e1,1,31831,0,0xf78c
,1,20365,2713,0x8e98,1,31833,0,0xed8f,1,20367,0,0x98cb
,1,43911,0,0xfa4e,1,20369,0,0x98d0,1,20370,0,0xf06c
,1,31838,0,0xf78d,1,20372,2714,0x87b4,1,31840,0,0xe2c4
,1,20374,7791,0x98d3,1,20375,0,0x87b5,1,20376,0,0x98cc
,1,31844,0,0xe2dc,1,31845,5784,0xe2de,1,20379,0,0x8b9f
,1,36080,0,0xfa4f,1,20381,0,0x88cb,1,31849,0,0xf78f
,1,36081,0,0xfa50,1,20384,0,0x8ba0,1,20385,2716,0x89bf
,1,31853,0,0xed90,1,31854,0,0xf790,1,31855,0,0xf791
,1,8922,7794,0x8496,1,8923,0,0x8497,1,36105,0,0xfa52
,1,31859,0,0x95c4,1,36110,9175,0xfa53,1,31861,0,0xe2e0
,1,20395,0,0x9b44,1,36204,0,0xfa54,1,20397,0,0x9699
,1,20398,4986,0x958e,1,20399,0,0x8cf2,1,36228,0,0xfa55
,1,36245,0,0xfa56,1,20402,0,0xf06f,1,31870,0,0x96e0
,1,36262,0,0xfa57,1,20405,0,0x904e,1,31873,1568,0x8bcc
,1,31874,0,0x8c48,1,31875,0,0xe2e1,1,43913,0,0xfa58
,1,36294,9183,0xfa59,1,36296,0,0xfa5a,1,36313,0,0xfa5b
,1,36332,0,0xfa5c,1,31881,5786,0x95b2,1,20415,0,0x95d6
,1,31883,0,0x9088,1,36364,0,0xfa5d,1,20418,1468,0x8c57
,1,20419,0,0x91a3,1,20420,4988,0x89e2,1,31888,5788,0xe2e2
,1,18429,0,0xfa5e,1,31890,0,0x97b1,1,36349,0,0xfa5f
,1,20425,4990,0x87b7,1,20426,1330,0x8f72,1,20427,0,0xf072
,1,31895,0,0x9165,1,31896,0,0x9453,1,20430,0,0x98d7
,1,20431,0,0xf073,1,8965,709,0x81c6,1,8966,1739,0x81c7
,1,20434,0,0xf074,1,36358,0,0xfa60,1,31903,1738,0x88be
,1,31904,0,0xed94,1,31905,0,0xe2e7,1,20439,2718,0x91ad
,1,20440,0,0x98d8,1,31908,0,0xe2e3,1,31909,1740,0x8a9f
,1,20443,7801,0x98d9,1,31911,0,0x8fcf,1,8978,1419,0x81dc
,1,43914,0,0xfa61,1,20447,0,0x98d6,1,20448,2720,0x87b8
,1,20449,0,0x904d,1,31917,0,0xe2e4,1,8984,1458,0x849c
,1,20452,0,0x98dd,1,20453,0,0x98de,1,31921,5785,0xe2eb
,1,31922,0,0xe2ea,1,31923,0,0xe2e9,1,36372,0,0xfa62
,1,36374,0,0xfa63,1,31926,0,0xf798,1,31927,0,0xf799
,1,36385,0,0xfa64,1,20462,2724,0x8f43,1,20463,0,0x98eb
,1,36386,0,0xfa65,1,20465,4992,0x879f,1,31933,5791,0xe2ee
,1,31934,1293,0x90b8,1,31935,0,0xf79a,1,20469,2726,0x9555
,1,20470,0,0x98e6,1,31938,0,0xe2f1,1,20472,0,0x95ee
,1,31940,0,0xf79c,1,20474,2727,0x89b4,1,36391,0,0xfa66
,1,43915,0,0xfa67,1,31944,0,0xf79e,1,20478,0,0x98ea
,1,31946,0,0x8cd0,1,20480,0,0xf077,1,20481,0,0x87b9
,1,20482,7806,0x87ba,1,31950,0,0x9157,1,18454,10383,0xfa68
,1,20485,0,0x98e4,1,20486,0,0x98ed,1,31954,0,0xe2f3
,1,36406,0,0xfa69,1,20489,0,0x9171,1,31957,0,0xed96
,1,20491,1231,0x8cc2,1,31959,0,0xf7a1,1,20493,2728,0x947b
,1,20494,4993,0x87bb,1,20495,0,0xe0c5,1,20496,0,0xf078
,1,20497,2730,0x98ec,1,20498,4995,0x937c,1,31966,5792,0x95b3
,1,31967,1743,0x918c,1,31968,0,0x8d66,1,20502,0,0x8cf4
,1,31970,0,0xe2f5,1,20504,0,0x87bc,1,20505,0,0x8cf3
,1,20506,0,0x98df,1,31974,0,0xf7a3,1,31975,5793,0x97c6
,1,36409,0,0xfa6a,1,20510,0,0xf07b,1,20511,0,0x8ed8
,1,31979,0,0xf7a4,1,20513,0,0x98e7,1,20514,0,0xf07c
,1,20515,0,0x95ed,1,20516,2731,0x926c,1,20517,0,0x98e3
,1,20518,0,0x8c91,1,20519,2732,0x87bd,1,20520,0,0x98e0
,1,20521,2734,0x98e8,1,20522,7813,0x98e2,1,20523,2735,0x97cf
,1,20524,0,0x98e9,1,31992,1637,0x8e85,1,20526,0,0x87be
,1,31994,0,0xe2fb,1,31995,0,0x8c6e,1,36427,0,0xfa6b
,1,36436,0,0xfa6c,1,31998,0,0x8b8a,1,36450,0,0xfa6d
,1,32000,0,0x8b49,1,20534,0,0x8be4,1,32002,0,0xe340
,1,32003,0,0xf7a7,1,20537,1483,0x8c90,1,32005,0,0x8d67
,1,20539,2736,0x87c0,1,32007,0,0xed99,1,32008,0,0xed9a
,1,32009,0,0xf7a8,1,32010,0,0xe343,1,20544,1480,0x87bf
,1,20545,0,0x87c1,1,32013,5796,0x945b,1,20547,0,0x98ee
,1,36460,9945,0xfa6e,1,32016,0,0x9552,1,20550,0,0xf080
,1,20551,7815,0x98ef,1,20552,4997,0x98f3,1,20553,720,0x88cc
,1,32021,0,0xe342,1,36461,10139,0xfa6f,1,32023,0,0x8ed1
,1,32024,0,0x8d68,1,32025,5797,0x8e86,1,32026,1413,0x8b89
,1,32027,1749,0x95b4,1,32028,0,0xe341,1,32029,0,0xed9c
,1,20563,7816,0xf082,1,36463,0,0xfa70,1,32032,1745,0x9166
,1,32033,1750,0x9661,1,32034,5802,0x8df5,1,32035,0,0xed9d
,1,36504,0,0xfa71,1,20570,0,0x98f4,1,36510,0,0xfa72
,1,20572,0,0x92e2,1,36526,9260,0xfa73,1,36531,0,0xfa74
,1,36533,0,0xfa75,1,32043,0,0x8e87,1,32044,0,0x92db
,1,36534,9980,0xfa76,1,32046,5803,0xe346,1,32047,0,0x97dd
,1,20581,586,0x8c92,1,9115,4999,0x829d,1,9116,2739,0x829e
,1,9117,787,0x8461,1,9118,0,0x8462,1,9119,2741,0x8463
,1,9120,0,0x8464,1,9121,1753,0x8465,1,9122,0,0x8466
,1,9123,750,0x8467,1,9124,578,0x8468,1,9125,5807,0x8469
,1,9126,0,0x846a,1,9127,677,0x846b,1,9128,7818,0x846c
,1,9129,1119,0x846d,1,9130,1205,0x846e,1,9131,493,0x846f
,1,9132,699,0x83d8,1,9133,523,0x83d9,1,9134,1108,0x83da
,1,9135,2737,0x83db,1,9136,2742,0x83dc,1,9137,7822,0x83dd
,1,9138,470,0x83de,1,9139,0,0x83df,1,9140,0,0x83e0
,1,9141,1757,0x83e1,1,32076,0,0x8c6f,1,36539,10045,0xfa77
,1,32078,0,0xe34d,1,32079,0,0xe351,1,32080,1758,0x8c8b
,1,43917,0,0xfa78,1,36561,0,0xfa79,1,32083,5809,0xeda1
,1,36564,9264,0xfa7a,1,18510,0,0xfa7b,1,32086,0,0xe34c
,1,36601,9235,0xfa7c,1,20621,0,0x9654,1,32089,0,0xf7b5
,1,32090,0,0xf7b6,1,32091,0,0xe355,1,20625,5004,0x8c86
,1,20626,0,0xf08a,1,32094,5811,0x8d69,1,20628,0,0x87c2
,1,20629,0,0xf08c,1,32097,5812,0x978d,1,32098,0,0x88ba
,1,20632,2743,0x8e50,1,20633,0,0x94f5,1,20634,0,0x98f9
,1,32102,0,0x8b8b,1,20636,0,0xf08e,1,32104,0,0xe34f
,1,43918,0,0xfa7d,1,32106,0,0xf7b7,1,36608,0,0xfa7e
,1,36616,9867,0xfa80,1,36631,0,0xfa81,1,32110,5808,0xe350
,1,36651,0,0xfa82,1,32112,0,0xf7b8,1,32113,0,0x939d
,1,32114,0,0xe34e,1,32115,0,0xe34b,1,36672,0,0xfa83
,1,32117,5814,0x8a47,1,32118,0,0x90e2,1,20652,0,0x8dc3
,1,20653,0,0x9762,1,32121,0,0x8ca6,1,32122,0,0xeda3
,1,36682,0,0xfa84,1,20657,0,0xf091,1,20658,2746,0x98fc
,1,20659,0,0x9942,1,20660,7829,0x98fb,1,20661,0,0x8dc2
,1,36696,0,0xfa85,1,20663,0,0x8f9d,1,32131,0,0xeda4
,1,43920,0,0xfa86,1,20666,0,0xf092,1,20667,7833,0xf093
,1,36772,0,0xfa87,1,32136,0,0xf7bd,1,20670,2744,0x8c58
,1,36788,0,0xfa88,1,32139,0,0xeda5,1,32140,0,0xf7be
,1,20674,0,0x9943,1,64102,0,0xfa89,1,32143,5818,0xe356
,1,20677,0,0x8bcd,1,36790,0,0xfa8a,1,20679,0,0xf095
,1,32147,0,0xe353,1,20681,0,0x9940,1,20682,0,0x9941
,1,43921,0,0xfa8b,1,20684,7834,0x87c3,1,20685,0,0x93ad
,1,32153,5826,0x8c70,1,20687,1123,0x919c,1,20688,2747,0x87c5
,1,20689,1117,0x8ba1,1,32157,0,0xf7c1,1,36801,0,0xfa8c
,1,32159,5829,0xe365,1,20693,5007,0x966c,1,20694,0,0x9944
,1,32162,0,0xe361,1,32163,0,0xe35b,1,20697,0,0xf09b
,1,20698,0,0x97bb,1,32166,0,0xeda7,1,32167,0,0xf7c2
,1,36806,10965,0xfa8d,1,20702,0,0x9945,1,32170,0,0xf7c3
,1,32171,0,0xe35f,1,32172,5831,0x8ef8,1,32173,0,0x88db
,1,20707,2748,0x9948,1,32175,0,0xe362,1,20709,2751,0x9946
,1,20710,551,0x87c6,1,20711,1476,0x916d,1,64036,0,0xfa8e
,1,32180,5837,0x92d4,1,32181,0,0xe35c,1,32182,0,0xf7c4
,1,32183,0,0xf7c5,1,20717,2750,0x9947,1,9251,1759,0x849d
,1,32186,0,0xe359,1,32187,0,0x925d,1,36810,0,0xfa8f
,1,20722,2749,0x87c4,1,32190,5824,0x88bb,1,32191,0,0x96c8
,1,20725,7836,0x994b,1,36813,0,0xfa90,1,32194,0,0xeda8
,1,36819,0,0xfa91,1,20729,0,0x994a,1,36821,0,0xfa92
,1,20731,0,0x95c6,1,32199,0,0xe35d,1,36832,0,0xfa93
,1,36849,0,0xfa94,1,32202,0,0x8bd9,1,20736,1353,0x8b56
,1,20737,5011,0x994d,1,20738,0,0x994e,1,20739,0,0x87c9
,1,20740,1115,0x89ad,1,36853,0,0xfa95,1,20742,1578,0x87c8
,1,32210,0,0x8f8f,1,20744,0,0xf140,1,20745,0,0x994c
,1,32213,0,0xe38e,1,20747,5015,0x87ca,1,43682,7838,0x87a0
,1,32216,5742,0xe367,1,43684,7840,0xf04d,1,32218,5753,0x90fc
,1,43686,0,0xf065,1,32220,5816,0xe363,1,20754,2752,0x8ef2
,1,32222,5825,0xe36a,1,20756,5841,0x9951,1,32224,1761,0x92f7
,1,20758,2754,0x994f,1,20759,5858,0xf142,1,20760,5018,0x98d4
,1,32228,5868,0xe369,1,20762,5871,0x9952,1,20763,5875,0xf143
,1,43698,0,0xf169,1,32232,5893,0x95d2,1,20766,454,0x87cb
,1,20767,5898,0x8f9e,1,43702,0,0xf17b,1,32236,1762,0x96c9
,1,43704,0,0xf1a8,1,43705,0,0xf1b8,1,32239,5953,0x88dc
,1,43707,0,0xf1b0,1,43708,7842,0xf1bb,1,32242,5987,0xe36c
,1,43710,0,0x8861,1,32244,5991,0x97fb,1,20778,5995,0x9744
,1,9312,5994,0x8740,1,9313,5997,0x8741,1,9314,5998,0x8742
,1,9315,373,0x8743,1,9316,6005,0x8744,1,9317,2753,0x8745
,1,9318,6008,0x8746,1,9319,1478,0x8747,1,9320,374,0x8748
,1,9321,6032,0x8749,1,9322,335,0x874a,1,9323,419,0x874b
,1,9324,1764,0x874c,1,9325,6062,0x874d,1,9326,1253,0x874e
,1,9327,395,0x874f,1,9328,1766,0x8750,1,9329,1765,0x8751
,1,9330,5026,0x8752,1,9331,1673,0x8753,1,20799,2757,0x9958
,1,20800,2761,0x9959,1,20801,6174,0x88f2,1,43736,0,0xf0c3
,1,20803,6182,0x8cb3,1,20804,6183,0x8c5a,1,20805,6204,0x8f5b
,1,20806,5031,0x929b,1,20807,1693,0x8ba2,1,20808,6215,0x90e6
,1,20809,6273,0x8cf5,1,20810,5035,0x87cd,1,20811,6306,0x8d8e
,1,20812,6351,0x995b,1,20813,6368,0x96c6,1,20814,6374,0x9365
,1,43749,0,0xf2ce,1,20816,1331,0x8e99,1,43751,0,0xf2eb
,1,20818,6497,0x995a,1,32286,6520,0x8ec8,1,20820,2759,0x995c
,1,20821,6526,0x87cf,1,32289,6527,0xe371,1,20823,2762,0x87d0
,1,32291,6545,0xe370,1,43759,0,0xf358,1,20826,6595,0x937d
,1,32294,4622,0x8f63,1,20828,6650,0x8a95,1,32296,6654,0xedad
,1,43764,0,0xf3a1,1,43765,0,0xf3b8,1,32299,5843,0x9644
,1,43767,7855,0xf3ba,1,20834,6696,0x995d,1,32302,6711,0x8f6b
,1,43770,0,0xf3c5,1,20837,6776,0x93fc,1,32305,6710,0xe373
,1,32306,6795,0xe380,1,20840,6787,0x9153,1,20841,6788,0x995f
,1,20842,2764,0x9960,1,20843,6845,0x94aa,1,20844,2768,0x8cf6
,1,20845,4715,0x985a,1,20846,2766,0x9961,1,32314,6910,0xe381
,1,32315,6911,0xe37a,1,20849,6913,0x8ba4,1,32317,6914,0xe360
,1,32318,4743,0x90d1,1,43786,0,0xf46d,1,20853,4759,0x95ba
,1,20854,1344,0x91b4,1,20855,6957,0x8bef,1,20856,2767,0x9354
,1,43791,0,0xf48a,1,43792,7857,0xf490,1,32326,6980,0xe378
,1,20860,5037,0x8c93,1,43795,0,0xf4b2,1,43796,0,0xebde
,1,32330,7073,0x9140,1,32331,1768,0x8c71,1,43799,0,0xf4dc
,1,32333,1769,0x8f4a,1,20867,7116,0xf148,1,43802,0,0xf551
,1,20869,7240,0x93e0,1,20870,7250,0x897e,1,32338,7282,0xedb3
,1,43806,0,0xf584,1,32340,1770,0x9044,1,20874,886,0x8dfb
,1,32342,5847,0xe384,1,20876,4887,0x9965,1,20877,7424,0x8dc4
,1,32345,7496,0xe386,1,20879,2774,0x9967,1,20880,7533,0xe3ec
,1,20881,7545,0x9968,1,20882,2771,0x9660,1,20883,2773,0x9969
,1,43818,0,0xf681,1,20885,7613,0x996a,1,20886,5041,0x996b
,1,20887,7639,0x8fe7,1,20888,7641,0xf14b,1,20889,7644,0x8eca
,1,43824,0,0xf69f,1,9424,2763,0x86bf,1,9425,2769,0x86c0
,1,9426,336,0x86c1,1,9427,2777,0x86c2,1,9428,2776,0x86c3
,1,9429,444,0x86c4,1,9430,7668,0x86c5,1,9431,1459,0x86c6
,1,9432,5849,0x86c7,1,9433,1771,0x86c8,1,9434,495,0x86c9
,1,9435,1772,0x86ca,1,9436,4952,0x86cb,1,9437,1379,0x86cc
,1,9438,1774,0x86cd,1,9439,1775,0x86ce,1,9440,1776,0x86cf
,1,9441,1224,0x86d0,1,9442,5852,0x86d1,1,9443,2775,0x86d2
,1,9444,4980,0x86d3,1,9445,1781,0x86d4,1,9446,1778,0x86d5
,1,9447,1779,0x86d6,1,9448,1777,0x86d7,1,9449,590,0x86d8
,1,20917,7803,0x9976,1,20918,7809,0x96e8,1,32386,1583,0x8e5b
,1,32387,7811,0xe38d,1,43855,0,0xf7ac,1,43856,0,0xf7ad
,1,43857,7862,0xf7b2,1,20924,7823,0xf14f,1,20925,2784,0x9977
,1,32393,7835,0xe393,1,32394,7846,0xedb5,1,43862,0,0xf7e7
,1,32396,7877,0xe394,1,32397,7882,0xedb6,1,20931,2791,0x87d3
,1,20932,1216,0x90a6,1,20933,2786,0x9978,1,20934,7867,0x8f79
,1,32402,7931,0xe395,1,32403,8109,0xe397,1,20937,2788,0x9979
,1,20938,8111,0x87d4,1,9472,1188,0x849f,1,9473,1573,0x84aa
,1,9474,1225,0x84a0,1,9475,8132,0x84ab,1,32410,8148,0xf7de
,1,32411,8151,0xe39b,1,32412,8153,0xe39c,1,43880,0,0xeddb
,1,43881,0,0xf8c2,1,43882,0,0xf8cc,1,43883,0,0xee52
,1,20950,8205,0x99c3,1,9484,8250,0x84a1,1,43886,0,0xf8fc
,1,43887,0,0xf940,1,9487,8260,0x84ac,1,9488,1782,0x84a2
,1,20956,8284,0xeaa3,1,20957,8286,0x8bc3,1,9491,343,0x84ad
,1,9492,8364,0x84a4,1,20960,8367,0x997b,1,20961,8384,0x967d
,1,9495,344,0x84af,1,9496,8440,0x84a3,1,43898,0,0xf9a8
,1,43899,0,0xf9aa,1,9499,724,0x84ae,1,9500,1152,0x84a5
,1,9501,8472,0x84ba,1,20969,8477,0x997d,1,20970,8491,0x93e2
,1,9504,8613,0x84b5,1,43906,0,0xf9e1,1,20973,8793,0x997e
,1,9507,359,0x84b0,1,9508,8797,0x84a7,1,9509,1785,0x84bc
,1,20977,8902,0x8a4d,1,43912,0,0xfa51,1,9512,5865,0x84b7
,1,20980,8980,0xf153,1,20981,9010,0x9981,1,9515,484,0x84b2
,1,9516,9148,0x84a6,1,20984,9172,0x93ca,1,20985,5307,0x899a
,1,9519,710,0x84b6,1,9520,9217,0x84bb,1,36859,9541,0xfa96
,1,20989,9552,0x949f,1,9523,1787,0x84b1,1,9524,9570,0x84a8
,1,20992,9575,0x9381,1,20993,9669,0x87d8,1,9527,5866,0x84b8
,1,9528,792,0x84bd,1,20996,9796,0x9983,1,36866,9799,0xfa97
,1,9531,1399,0x84b3,1,9532,863,0x84a9,1,21000,9916,0x8aa0
,1,36876,9918,0xfa98,1,9535,446,0x84b9,1,21003,9943,0x9984
,1,36919,9978,0xfa99,1,9538,9979,0x84be,1,21006,10006,0x9986
,1,43922,10034,0xfa9a,1,36931,10089,0xfa9b,1,21009,10090,0x8c59
,1,21010,10129,0xf155,1,21011,10137,0x87da,1,21012,10141,0x9985
,1,21013,10143,0x87db,1,9547,5867,0x84b4,1,21015,10168,0x97f1
,1,36932,10183,0xfa9c,1,36957,10189,0xfa9d,1,43923,10249,0xfa9e
,1,43924,10299,0xfa9f,1,36997,10304,0xfaa0,1,21021,10382,0x8f89
,1,37004,10409,0xfaa1,1,37008,10444,0xfaa2,1,38429,10481,0xfaa3
,1,37025,10484,0xfaa4,1,18613,10497,0xfaa5,1,37040,10571,0xfaa6
,1,21028,10635,0x94bb,1,21029,10721,0x95ca,1,37046,10725,0xfaa7
,1,21031,10778,0x9987,1,37059,10785,0xfaa8,1,21033,10781,0x9798
,1,21034,10820,0x9988,1,37064,10855,0xfaa9,1,43925,10840,0xfaaa
,1,37084,10905,0xfaab,1,21038,10926,0x9989,1,37087,10914,0xfaac
,1,21040,10947,0x939e,1,43926,10959,0xfaad,1,43976,0,0xefe4
,1,21043,11063,0x998a,1,37110,11080,0xfaae,1,37106,11082,0xfaaf
,1,21046,11129,0x90a7,1,21047,11131,0x8dfc,1,21048,11140,0x8c94
,1,21049,11157,0x998b,1,21050,0,0x8e68,1,21051,0,0x8d8f
,1,37120,0,0xfab0,1,37099,0,0xfab1,1,37118,0,0xfab2
,1,37119,0,0xfab3,1,37124,0,0xfab4,1,37126,0,0xfab5
,1,37144,0,0xfab6,1,21059,0,0x92e4,1,21060,0,0x998d
,1,37148,10482,0xfab7,1,37150,0,0xfab8,1,21063,0,0x91a5
,1,37175,0,0xfab9,1,21065,0,0x87dc,1,21066,0,0x8ded
,1,21067,0,0x998e,1,21068,0,0x998f,1,21069,0,0x914f
,1,37177,0,0xfaba,1,21071,0,0x998c,1,37178,0,0xfabb
,1,37190,0,0xfabc,1,37191,0,0xfabd,1,37207,0,0xfabe
,1,21076,0,0x9991,1,21077,0,0xf158,1,21078,0,0x9655
,1,21079,0,0x87dd,1,37209,0,0xfabf,1,37217,0,0xfac0
,1,37220,0,0xfac1,1,21083,0,0x8d84,1,21084,0,0xf159
,1,21085,0,0x889e,1,21086,0,0x9990,1,37236,0,0xfac2
,1,37241,0,0xfac3,1,21089,0,0x87de,1,37253,0,0xfac4
,1,21091,0,0x8c95,1,21092,0,0x8ddc,1,21093,0,0x948d
,1,37262,0,0xfac5,1,37288,0,0xfac6,1,37294,0,0xfac7
,1,21097,0,0x9994,1,21098,0,0x9992,1,9632,455,0x81a1
,1,9633,5869,0x81a0,1,32568,0,0xe39d,1,37299,0,0xfac8
,1,21103,2792,0x959b,1,21104,7872,0x8fe8,1,21105,7871,0x999b
,1,21106,0,0x8a84,1,21107,7873,0x9995,1,21108,0,0x9993
,1,21109,0,0x916e,1,37302,0,0xfac9,1,21111,0,0xf15b
,1,32579,0,0xf7e3,1,32580,0,0xf7e4,1,32581,0,0xe39f
,1,37315,0,0xfaca,1,9649,5044,0x83eb,1,9650,1788,0x81a3
,1,9651,0,0x81a2,1,21119,0,0x9996,1,37316,0,0xfacb
,1,9654,2794,0x8241,1,9655,2799,0x8240,1,21123,2800,0x8a63
,1,21124,7874,0xf15c,1,32592,0,0xe3a3,1,32593,0,0xe3a4
,1,9660,512,0x81a5,1,9661,1792,0x81a4,1,21129,2801,0x97ab
,1,32597,0,0xe3a5,1,9664,0,0x8243,1,9665,0,0x8242
,1,21133,2805,0x9998,1,37338,0,0xfacc,1,43927,0,0xfacd
,1,43928,10935,0xface,1,9670,1794,0x819f,1,9671,1790,0x819e
,1,21139,0,0x87df,1,9673,1789,0x825a,1,32608,0,0xe3a9
,1,9675,7878,0x819b,1,37356,0,0xfacf,1,21144,7879,0xf15f
,1,9678,7881,0x819d,1,9679,0,0x819c,1,9680,1577,0x84e5
,1,9681,2809,0x84e6,1,9682,2807,0x84e7,1,9683,2808,0x84e8
,1,21151,589,0x8cf7,1,21152,502,0x89c1,1,37358,0,0xfad0
,1,32621,0,0xf7ed,1,32622,1594,0x9275,1,21156,0,0xf161
,1,32624,0,0x94b1,1,21158,0,0xf162,1,32626,0,0x8f90
,1,37377,0,0xfad1,1,21161,0,0x8f95,1,21162,1292,0x9377
,1,21163,0,0x8d85,1,32631,1799,0x94eb,1,21165,2810,0x99a1
,1,32633,0,0x9ceb,1,21167,0,0xf163,1,37386,0,0xfad2
,1,9702,1585,0x825e,1,32637,0,0xf7ee,1,32638,0,0xf7ef
,1,21172,0,0x984a,1,21173,0,0x99a3,1,37398,0,0xfad3
,1,32642,0,0xe3ae,1,32643,0,0xe3b0,1,21177,0,0x8cf8
,1,9711,1524,0x81fc,1,32646,5879,0xe3af,1,21180,2811,0x99a2
,1,32648,0,0xe3b1,1,21182,0,0x8a4e,1,32650,0,0x9772
,1,37399,0,0xfad4,1,21185,2812,0x99a4,1,43929,0,0xfad5
,1,32654,1450,0x94fc,1,37427,0,0xfad6,1,21189,7883,0x92ba
,1,32657,0,0xedb9,1,21191,0,0x9745,1,21192,0,0x87e0
,1,21193,2815,0x95d7,1,21194,0,0xf166,1,9728,7886,0x83e6
,1,9729,360,0x83e7,1,9730,1800,0x83e8,1,9731,0,0x83e9
,1,32666,0,0xe3b7,1,9733,361,0x819a,1,9734,5882,0x8199
,1,32669,4212,0xe3b6,1,32670,0,0xe3b5,1,37442,0,0xfad7
,1,21205,0,0x93ae,1,21206,0,0x87e4,1,21207,0,0x99a6
,1,21208,2817,0x8aa8,1,9742,501,0x83e5,1,37447,0,0xfad8
,1,21211,0,0x87e5,1,37450,0,0xfad9,1,21213,877,0x8f9f
,1,32681,1801,0x8b60,1,21215,0,0x95e5,1,21216,0,0x99ab
,1,9750,0,0x83e2,1,9751,852,0x83e3,1,9752,1804,0x849e
,1,21220,2819,0x8bce,1,37454,0,0xfada,1,21222,0,0x99a9
,1,21223,2820,0x8aa9,1,37457,0,0xfadb,1,9758,0,0x879e
,1,37462,0,0xfadc,1,32694,0,0xe3bd,1,37465,0,0xfadd
,1,32696,0,0xe3be,1,32697,0,0xe3bb,1,37472,0,0xfade
,1,21232,0,0x87e7,1,37473,0,0xfadf,1,32701,496,0x8948
,1,21235,0,0x99ac,1,32703,0,0xedbb,1,21237,0,0x99ad
,1,32705,0,0x89a5,1,21239,0,0xf16a,1,21240,7892,0x99ae
,1,21241,0,0x99af,1,21242,2824,0x8ed9,1,21243,2827,0x87e8
,1,37477,0,0xfae0,1,37479,0,0xfae1,1,21246,0,0x8cf9
,1,21247,2829,0x96dc,1,21248,0,0x87e9,1,21249,1522,0x96e6
,1,21250,0,0x93f5,1,32718,0,0xedbc,1,32719,0,0xf7f9
,1,21253,0,0x95ef,1,21254,0,0x99b0,1,21255,701,0x87ea
,1,21256,0,0x99b1,1,32724,0,0xe3c4,1,32725,5883,0xe3c3
,1,9792,5887,0x818a,1,37480,0,0xfae2,1,9794,1805,0x8189
,1,43930,0,0xfae3,1,21263,0,0x99b5,1,21264,5046,0x99b4
,1,43931,0,0xfae4,1,37500,0,0xfae5,1,37501,0,0xfae6
,1,32735,0,0xedbe,1,32736,1807,0x9089,1,21270,2830,0x89bb
,1,21271,0,0x966b,1,32739,0,0xf7fa,1,21273,0,0x8dfa
,1,21274,7896,0x99b7,1,32742,0,0xe3c6,1,21276,0,0x87eb
,1,21277,0,0x9178,1,32745,0,0xe3c7,1,37503,0,0xfae7
,1,32747,726,0x8ae3,1,21281,5047,0x8ba7,1,37513,0,0xfae8
,1,21283,5048,0x99b8,1,21284,7899,0xf16d,1,32752,0,0x8acb
,1,37517,0,0xfae9,1,32754,0,0xf840,1,32755,0,0xe3c8
,1,37527,0,0xfaea,1,21290,0,0x94d9,1,9824,0,0x83b8
,1,9825,0,0x83bb,1,9826,0,0x83b9,1,9827,2833,0x83be
,1,9828,1810,0x83b7,1,9829,1453,0x83bc,1,9830,1523,0x83ba
,1,9831,0,0x83bd,1,9832,1814,0x83ea,1,9833,0,0x81fb
,1,9834,1517,0x81f4,1,9835,1618,0x81f9,1,9836,7900,0x81fa
,1,9837,553,0x81f3,1,9838,1360,0x81f8,1,9839,491,0x81f2
,1,21307,2835,0x88e3,1,37529,9850,0xfaeb,1,32776,0,0xf843
,1,21310,0,0xf16f,1,21311,7902,0x93bd,1,21312,2844,0x99bd
,1,32780,706,0x8ea7,1,21314,0,0xf170,1,21315,7904,0x90e7
,1,37535,0,0xfaec,1,32784,1817,0x91cf,1,21318,7905,0x99be
,1,21319,2846,0x8fa1,1,21320,0,0x8cdf,1,21321,5051,0x99c1
,1,32789,1339,0x8d6b,1,32790,0,0xf846,1,32791,0,0x96d5
,1,21325,2847,0x99c2,1,32793,0,0xe3d0,1,37543,0,0xfaed
,1,37547,0,0xfaee,1,21329,2848,0x94da,1,21330,0,0x91b2
,1,21331,0,0x91ec,1,21332,0,0x8ba6,1,43932,0,0xfaef
,1,32801,0,0xe3d2,1,21335,0,0x93ec,1,21336,0,0x9250
,1,32804,0,0xf847,1,21338,0,0x948e,1,32806,0,0xedc4
,1,21340,0,0x966d,1,32808,0,0xe3d3,1,21342,0,0x99c4
,1,43933,0,0xfaf0,1,21344,0,0x90e8,1,21345,7906,0x87ed
,1,37554,0,0xfaf1,1,21347,0,0x87ee,1,37567,0,0xfaf2
,1,32816,0,0xf849,1,21350,0,0x8c54,1,21351,0,0xf173
,1,32819,0,0x8ea8,1,21353,0,0x99c5,1,32821,0,0xedc5
,1,32822,0,0x96eb,1,32823,5895,0xedc6,1,37568,0,0xfaf3
,1,21358,0,0x99c6,1,21359,0,0x894b,1,21360,2850,0x88f3
,1,21361,5052,0x8aeb,1,32829,0,0x925e,1,21363,0,0x91a6
,1,21364,2849,0x8b70,1,21365,0,0x9791,1,37574,0,0xfaf4
,1,21367,0,0x99c9,1,21368,7907,0x89b5,1,37582,0,0xfaf5
,1,21370,0,0xf175,1,21371,2853,0x99c8,1,37584,0,0xfaf6
,1,21373,0,0x87ef,1,37591,0,0xfaf7,1,21375,2851,0x8ba8
,1,37593,0,0xfaf8,1,37605,0,0xfaf9,1,21378,0,0x99ca
,1,37607,0,0xfafa,1,21380,0,0x96ef,1,37649,0,0xfafb
,1,43934,0,0xfafc,1,32850,0,0xe3d8,1,43935,11108,0xfb40
,1,37623,0,0xfb41,1,37625,0,0xfb42,1,32854,0,0x90b9
,1,37627,10859,0xfb43,1,32856,0,0xe3d9,1,37634,0,0xfb44
,1,32858,0,0xe3da,1,37645,0,0xfb45,1,37653,0,0xfb46
,1,37661,0,0xfb47,1,21395,1401,0x87f0,1,32863,0,0xe3db
,1,37662,0,0xfb48,1,32865,1818,0x918f,1,32866,0,0xe3dc
,1,21400,0,0x97d0,1,37671,0,0xfb49,1,21402,0,0x8cfa
,1,32870,0,0xf84b,1,37673,0,0xfb4a,1,21405,2856,0x87f1
,1,43936,0,0xfb4b,1,21407,0,0x8cb4,1,21408,0,0x99cc
,1,43937,0,0xfb4c,1,37703,0,0xfb4d,1,37713,0,0xfb4e
,1,32879,5896,0x97fc,1,21413,2858,0x99ce,1,21414,7908,0x99cd
,1,32882,0,0xe3df,1,21416,2857,0x907e,1,21417,1194,0x8958
,1,32885,0,0xf84d,1,32886,0,0xe3e1,1,32887,0,0x9045
,1,21421,0,0x897d,1,21422,2859,0x99cf,1,37719,0,0xfb4f
,1,21424,7909,0x99d0,1,37722,0,0xfb50,1,21426,2862,0x87f2
,1,21427,1619,0x8cb5,1,32895,5897,0xe3e4,1,37739,0,0xfb51
,1,21430,0,0x99d1,1,21431,0,0xf179,1,37745,0,0xfb52
,1,32900,0,0xe3e5,1,32901,0,0xe3e7,1,21435,2863,0x8b8e
,1,32903,0,0x94a3,1,37747,0,0xfb53,1,32905,0,0x93f7
,1,37793,0,0xfb54,1,32907,5899,0x985d,1,32908,0,0x94a7
,1,21442,0,0x8e51,1,21443,0,0x99d2,1,43938,0,0xfb55
,1,43939,0,0xfb56,1,37768,0,0xfb57,1,37771,0,0xfb58
,1,21448,2865,0x9694,1,21449,0,0x8db3,1,21450,0,0x8b79
,1,32918,1488,0x8fd1,1,21452,0,0x916f,1,21453,1361,0x94bd
,1,21454,7910,0x8efb,1,32922,0,0xe3ea,1,32923,0,0xe3e8
,1,32924,0,0xf850,1,32925,0,0x8acc,1,37775,0,0xfb59
,1,21460,0,0x8f66,1,21461,0,0xf17e,1,32929,686,0x8cd2
,1,32930,692,0x8e88,1,37790,0,0xfb5a,1,21465,7912,0x8f96
,1,32933,5905,0x94ec,1,21467,0,0x94be,1,32935,0,0xf852
,1,37877,0,0xfb5b,1,10003,520,0x849b,1,32938,1820,0x9662
,1,43940,0,0xfb5c,1,21473,2868,0x8962,1,21474,2867,0x9170
,1,21475,0,0x8cfb,1,21476,554,0x8cc3,1,21477,0,0x8be5
,1,32945,0,0x8d6e,1,32946,0,0x88e7,1,21480,0,0x99d9
,1,32948,1154,0x8de6,1,21482,0,0x91fc,1,21483,0,0x8ba9
,1,21484,0,0x8fa2,1,21485,7913,0x99da,1,21486,0,0x99d8
,1,21487,1295,0x89c2,1,21488,0,0x91e4,1,21489,0,0x8eb6
,1,21490,0,0x8e6a,1,21491,0,0x8945,1,21492,0,0xf182
,1,21493,0,0xf183,1,21494,0,0x8a90,1,21495,0,0x8d86
,1,32963,619,0x88dd,1,32964,0,0xe3f2,1,21498,7915,0x99db
,1,32966,0,0x925f,1,43941,0,0xfb5d,1,37873,0,0xfb5e
,1,37825,0,0xfb5f,1,32970,0,0xedc8,1,37831,0,0xfb60
,1,32972,1823,0x9477,1,37852,10733,0xfb61,1,21507,1141,0x8b68
,1,21508,0,0x8a65,1,37858,0,0xfb62,1,37863,0,0xfb63
,1,37897,0,0xfb64,1,21512,0,0x8d87,1,21513,0,0x8b67
,1,21514,7917,0x92dd,1,21515,2875,0x8944,1,21516,5053,0x93af
,1,21517,7918,0x96bc,1,21518,2870,0x8d40,1,21519,2874,0x9799
,1,21520,2869,0x9366,1,21521,0,0x8cfc,1,21522,2873,0x87f3
,1,32990,0,0x9645,1,37903,0,0xfb65,1,32992,0,0xedca
,1,32993,0,0x8cd3,1,37910,0,0xfb66,1,37911,0,0xfb67
,1,32996,0,0x88fb,1,32997,0,0xe3ef,1,21531,7919,0x8c4e
,1,37883,0,0xfb68,1,21533,0,0x99e5,1,21534,0,0x989e
,1,21535,0,0x8be1,1,21536,0,0x9669,1,10070,0,0x879d
,1,37938,0,0xfb69,1,37940,0,0xfb6a,1,33007,5908,0xe3f6
,1,37947,0,0xfb6b,1,21542,2880,0x94db,1,21543,0,0x87f4
,1,33011,5909,0xedcb,1,33012,1828,0x93b7,1,33013,0,0xf85b
,1,21547,0,0x8adc,1,21548,0,0x99df,1,33016,1826,0x8bb9
,1,21550,0,0x99e2,1,37957,0,0xfb6c,1,33019,0,0xf85c
,1,33020,0,0xe445,1,33021,0,0x945c,1,43942,0,0xfb6d
,1,43943,0,0xfb6e,1,37997,0,0xfb6f,1,21558,0,0x99e3
,1,33026,0,0x8e89,1,21560,0,0x8b7a,1,21561,0,0x9081
,1,33029,0,0x8bba,1,33030,1400,0x90c6,1,33031,1827,0x9865
,1,33032,1824,0x96ac,1,21566,2878,0x8ce1,1,33034,0,0x90d2
,1,21568,0,0x99de,1,10102,0,0x869f,1,10103,1605,0x86a0
,1,10104,5917,0x86a1,1,10105,0,0x86a2,1,10106,0,0x86a3
,1,10107,1428,0x86a4,1,10108,0,0x86a5,1,10109,1206,0x86a6
,1,10110,534,0x86a7,1,10111,564,0x86a8,1,33046,0,0xf85f
,1,37999,0,0xfb70,1,21581,5054,0x87f5,1,21582,0,0x99e6
,1,33050,0,0x8b72,1,33051,0,0xe3f8,1,21585,0,0x93db
,1,38264,0,0xfb71,1,33054,0,0xf860,1,38265,0,0xfb72
,1,21589,0,0xf184,1,38278,0,0xfb73,1,38284,0,0xfb74
,1,33059,0,0xe3fa,1,33060,0,0xf862,1,38285,0,0xfb75
,1,43944,0,0xfb76,1,33063,0,0xf863,1,38315,0,0xfb77
,1,33065,0,0xe3f9,1,21599,0,0x99ea,1,38324,0,0xfb78
,1,33068,0,0xf864,1,21602,0,0xf189,1,43945,0,0xfb79
,1,33071,0,0xe3fb,1,38344,0,0xfb7a,1,33073,5919,0x9245
,1,43946,0,0xfb7b,1,21608,1277,0x8efc,1,43947,0,0xfb7c
,1,21610,7926,0x8ef4,1,21611,0,0x87f7,1,21612,0,0xf18b
,1,38444,0,0xfb7d,1,33081,0,0x92af,1,38451,0,0xfb7e
,1,21616,0,0x99ed,1,21617,0,0x99eb,1,33085,0,0xf866
,1,21619,2886,0x96a1,1,21620,0,0x87f8,1,21621,0,0x99e8
,1,21622,0,0x99f1,1,21623,0,0x99ec,1,38452,0,0xfb80
,1,43948,0,0xfb81,1,38460,0,0xfb82,1,21627,2882,0x99ef
,1,21628,0,0x8cc4,1,21629,0,0x96bd,1,38465,0,0xfb83
,1,21631,5055,0x87f9,1,21632,2881,0x99f0,1,38497,0,0xfb84
,1,43949,0,0xfb85,1,33102,0,0x9074,1,21636,0,0x99f2
,1,33104,0,0x9585,1,21638,2888,0x99f4,1,38530,0,0xfb86
,1,21640,2887,0x87fa,1,33108,0,0x8d6f,1,33109,5920,0x9872
,1,21643,0,0x8dee,1,21644,0,0x9861,1,21645,0,0xf18d
,1,21646,0,0x99e9,1,21647,0,0x99e7,1,21648,0,0x99f3
,1,43950,0,0xfb87,1,21650,0,0x99ee,1,38554,0,0xfb88
,1,33119,0,0xe454,1,33120,5921,0xedce,1,21654,0,0x87fb
,1,43951,0,0xfb89,1,18919,0,0xfb8a,1,38569,0,0xfb8b
,1,33125,0,0xe448,1,10192,2895,0x84bf,1,10193,362,0x84c0
,1,10194,5057,0x84c1,1,10195,7937,0x84c2,1,10196,0,0x84c3
,1,10197,690,0x84c4,1,10198,371,0x84c5,1,10199,1830,0x84c6
,1,10200,2890,0x84c7,1,10201,1841,0x84c8,1,10202,575,0x84c9
,1,10203,2889,0x84ca,1,10204,0,0x84cb,1,10205,1837,0x84cc
,1,10206,372,0x84cd,1,10207,0,0x84ce,1,10208,1838,0x84cf
,1,10209,1835,0x84d0,1,10210,1195,0x84d1,1,10211,1396,0x84d2
,1,10212,878,0x84d3,1,10213,0,0x84d4,1,10214,0,0x84d5
,1,10215,591,0x84d6,1,10216,434,0x84d7,1,10217,1142,0x84d8
,1,10218,2902,0x84d9,1,10219,7940,0x84da,1,10220,2905,0x84db
,1,10221,1832,0x84dc,1,10222,7943,0x86a9,1,10223,5934,0x86aa
,1,10224,0,0x86ab,1,10225,1844,0x86ac,1,10226,413,0x86ad
,1,10227,1843,0x86ae,1,10228,2901,0x86af,1,10229,398,0x86b0
,1,10230,1372,0x86b1,1,10231,1839,0x86b2,1,21699,0,0xf196
,1,33167,1836,0x8d70,1,38575,0,0xfb8c,1,21702,0,0x8841
,1,21703,0,0x99f5,1,21704,2910,0x99fb,1,21705,0,0x8dc6
,1,33173,0,0xe451,1,38579,0,0xfb8d,1,38586,0,0xfb8e
,1,33176,0,0xf86f,1,38589,0,0xfb8f,1,33178,0,0x9586
,1,18938,0,0xfb90,1,33180,0,0x968c,1,33181,0,0x9547
,1,43952,0,0xfb91,1,38616,0,0xfb92,1,33184,0,0xe450
,1,38618,0,0xfb93,1,38621,0,0xfb94,1,21720,2907,0x9a45
,1,33188,0,0xe452,1,18948,0,0xfb95,1,38676,0,0xfb96
,1,38691,0,0xfb97,1,33192,0,0x9663,1,33193,0,0xe456
,1,18985,0,0xfb98,1,38710,0,0xfb99,1,21729,0,0x88f5
,1,21730,0,0x9a4e,1,38721,0,0xfb9a,1,38727,0,0xfb9b
,1,21733,2912,0x9a46,1,21734,0,0x9a47,1,33202,0,0xf870
,1,21736,974,0x8fa3,1,21737,0,0x9689,1,33205,0,0xe458
,1,38741,0,0xfb9c,1,21740,0,0xf198,1,21741,2914,0x9a4c
,1,21742,0,0x9a4b,1,33210,5938,0xe45e,1,33211,0,0xedd2
,1,21745,0,0xf19a,1,21746,2915,0x934e,1,33214,5940,0xe459
,1,33215,0,0x945e,1,33216,0,0xe45c,1,33217,0,0xf871
,1,33218,0,0xe45d,1,33219,0,0xf872,1,38743,0,0xfb9d
,1,21754,0,0x9a4d,1,33222,0,0x89b0,1,38747,0,0xfb9e
,1,21757,2930,0x9a4a,1,33225,0,0xe45f,1,21759,5062,0x8842
,1,21760,0,0xf19c,1,21761,0,0xf19d,1,33229,0,0xe460
,1,38762,0,0xfb9f,1,21764,5065,0x8953,1,43953,0,0xfba0
,1,21766,2916,0x8db4,1,21767,0,0x904f,1,33235,0,0x919f
,1,21769,0,0xf19e,1,43954,0,0xfba1,1,33238,0,0xf873
,1,33239,0,0xedd5,1,33240,0,0xe463,1,21774,2922,0x8843
,1,21775,2931,0x9a48,1,21776,8106,0x9382,1,38806,0,0xfba2
,1,38810,0,0xfba3,1,38814,0,0xfba4,1,21780,2932,0x9a49
,1,33248,0,0xe467,1,21782,0,0x88a0,1,38818,0,0xfba5
,1,33251,0,0x9062,1,33252,0,0xf876,1,33253,0,0x89e7
,1,38833,0,0xfba6,1,33255,0,0xe468,1,33256,0,0x97d5
,1,38834,0,0xfba7,1,33258,0,0x8ea9,1,38846,0,0xfba8
,1,33260,0,0xf878,1,33261,0,0x8f4c,1,38860,0,0xfba9
,1,38865,0,0xfbaa,1,38868,0,0xfbab,1,38872,0,0xfbac
,1,38873,0,0xfbad,1,33267,0,0x8e8a,1,33268,0,0x9276
,1,38881,0,0xfbae,1,21803,0,0x8844,1,38897,0,0xfbaf
,1,38916,0,0xfbb0,1,21806,0,0x9a53,1,21807,2933,0x9742
,1,33275,0,0xe46a,1,33276,727,0x8950,1,33277,0,0xf87a
,1,21811,2934,0x9a59,1,33279,0,0xf87b,1,21813,0,0x8845
,1,33281,0,0xe46c,1,33282,0,0xe46d,1,21816,0,0x9a58
,1,21817,8119,0x9a4f,1,33285,0,0xe46e,1,38925,0,0xfbb1
,1,33287,5941,0xe46f,1,33288,0,0x8bbb,1,21822,2463,0x91c1
,1,33290,0,0xe470,1,21824,0,0x9a50,1,33292,5943,0x90e3
,1,33293,0,0xe471,1,33294,0,0x8ec9,1,21828,0,0x91ed
,1,21829,2935,0x9a55,1,21830,0,0x8fa4,1,33298,5945,0x98ae
,1,38926,0,0xfbb2,1,38932,0,0xfbb3,1,21834,0,0xf1a3
,1,33302,0,0xe473,1,33303,1845,0x95dc,1,33304,0,0x8ada
,1,33305,0,0xf880,1,21839,0,0x96e2,1,21840,879,0x8846
,1,33308,0,0x8f77,1,38934,0,0xfbb4,1,21843,1392,0x8c5b
,1,33311,0,0x8f4d,1,19132,0,0xfbb5,1,21846,8121,0x9a56
,1,21847,8122,0x9a57,1,43955,0,0xfbb6,1,38947,0,0xfbb7
,1,38962,10487,0xfbb8,1,38963,0,0xfbb9,1,21852,0,0x9a54
,1,21853,0,0x9a5a,1,21854,2937,0x8847,1,33322,0,0x8d71
,1,33323,5948,0xe475,1,33324,5949,0x94ca,1,38949,0,0xfbba
,1,21859,2944,0x9a51,1,21860,0,0xf1a7,1,38983,0,0xfbbb
,1,39014,0,0xfbbc,1,33330,0,0xf884,1,33331,0,0xe477
,1,33332,0,0xf885,1,33333,0,0x91c7,1,33334,0,0x9495
,1,33335,0,0x8cbd,1,33336,0,0xe476,1,33337,0,0x9144
,1,39083,0,0xfbbd,1,39085,0,0xfbbe,1,33340,0,0xf886
,1,39088,0,0xfbbf,1,43956,0,0xfbc0,1,39095,0,0xfbc1
,1,33344,0,0xe478,1,39096,0,0xfbc2,1,39099,0,0xfbc3
,1,39100,0,0xfbc4,1,39103,0,0xfbc5,1,33349,0,0xf889
,1,21883,8125,0x9a60,1,33351,1847,0x92f8,1,21885,0,0xf1a9
,1,21886,8127,0x9a61,1,39106,0,0xfbc6,1,21888,8140,0x9a5c
,1,21889,0,0x8848,1,21890,0,0xf1aa,1,21891,0,0x9a66
,1,21892,8143,0x9150,1,39111,0,0xfbc7,1,21894,0,0x8849
,1,21895,0,0x9a68,1,21896,0,0xf1ab,1,21897,0,0x8d41
,1,21898,0,0x9a5e,1,21899,0,0x929d,1,33367,0,0xf88f
,1,33368,0,0xe47a,1,21902,2938,0x884a,1,33370,0,0xe47c
,1,39115,0,0xfbc8,1,21905,8149,0xf1ac,1,33373,0,0xe47b
,1,39136,0,0xfbc9,1,33375,0,0xe47d,1,33376,0,0xedd9
,1,43957,0,0xfbca,1,33378,0,0xe480,1,21912,8150,0x9a62
,1,21913,2939,0x9a5b,1,21914,0,0x8aab,1,33382,0,0x8acd
,1,21916,0,0x8aec,1,21917,2942,0x8a85,1,21918,0,0x9a63
,1,21919,2943,0x9a5f,1,33387,0,0xe483,1,39137,0,0xfbcb
,1,39139,0,0xfbcc,1,33390,0,0x8daf,1,33391,0,0x97c7
,1,39141,0,0xfbcd,1,33393,0,0xe485,1,21927,768,0x8c96
,1,21928,0,0x9a69,1,21929,5069,0x9a67,1,21930,0,0x9172
,1,33398,474,0x8990,1,21932,2945,0x8baa,1,21933,2946,0x884c
,1,21934,8155,0x9a64,1,39146,0,0xfbce,1,21936,0,0x8bf2
,1,39152,0,0xfbcf,1,33405,0,0xf898,1,33406,0,0xe488
,1,33407,0,0xf899,1,39153,0,0xfbd0,1,21942,0,0x8963
,1,39155,0,0xfbd1,1,33411,0,0xf89a,1,39176,0,0xfbd2
,1,19259,0,0xfbd3,1,43958,0,0xfbd4,1,39190,0,0xfbd5
,1,39191,0,0xfbd6,1,43959,0,0xfbd7,1,21951,8158,0xf1b1
,1,33419,0,0x88f0,1,39194,0,0xfbd8,1,33421,0,0xe489
,1,33422,0,0xeddc,1,21956,0,0x9a6d,1,21957,0,0x9a6b
,1,39195,0,0xfbd9,1,21959,2948,0x9aa5,1,33427,0,0xf89c
,1,21961,0,0xf1b2,1,39196,0,0xfbda,1,43960,0,0xfbdb
,1,21964,0,0xf1b3,1,39217,0,0xfbdc,1,21966,1388,0x884d
,1,39218,0,0xfbdd,1,39219,0,0xfbde,1,21969,0,0xf1b4
,1,33437,5952,0x8ec5,1,39226,0,0xfbdf,1,21972,2949,0x9a70
,1,39227,0,0xfbe0,1,33441,0,0xeddd,1,39228,0,0xfbe1
,1,33443,0,0xedde,1,33444,0,0xeddf,1,33445,1849,0x8a48
,1,33446,0,0x88b0,1,21980,8159,0x9a6e,1,21981,8160,0xf1b5
,1,33449,0,0xede0,1,21983,0,0x9a6c,1,33451,0,0xe48b
,1,33452,0,0xe48e,1,33453,5957,0x946d,1,21987,5074,0x8e6b
,1,33455,1850,0x9063,1,39232,0,0xfbe2,1,33457,0,0x89d4
,1,33458,0,0xf89f,1,33459,0,0x9646,1,21993,8161,0xf1b9
,1,39233,0,0xfbe3,1,39238,0,0xfbe4,1,33463,0,0xede2
,1,33464,0,0x8c7c,1,33465,0,0x8bda,1,33466,0,0xf8a1
,1,33467,0,0xe48d,1,33468,0,0xf8a2,1,33469,0,0x89e8
,1,33470,0,0xede3,1,33471,0,0xede4,1,39245,0,0xfbe5
,1,39246,0,0xfbe6,1,22007,0,0x9a72,1,39260,0,0xfbe7
,1,22009,0,0x9a77,1,33477,0,0x8aa1,1,33478,0,0xede5
,1,39263,0,0xfbe8,1,22013,0,0x9a75,1,22014,0,0x9a74
,1,39264,0,0xfbe9,1,39331,0,0xfbea,1,39334,0,0xfbeb
,1,39353,0,0xfbec,1,39357,0,0xfbed,1,39359,0,0xfbee
,1,39363,0,0xfbef,1,33489,1158,0x8991,1,33490,5959,0xe492
,1,22024,1589,0x884f,1,22025,1144,0x89c3,1,33493,0,0xede6
,1,39369,0,0xfbf0,1,33495,0,0x9563,1,39380,0,0xfbf1
,1,22030,2963,0x8850,1,39385,0,0xfbf2,1,33499,5965,0x89d5
,1,33500,0,0xe49c,1,39390,0,0xfbf3,1,33502,0,0xe49a
,1,22036,2955,0x9a71,1,43961,0,0xfbf4,1,22038,2953,0x9a73
,1,22039,8163,0x8fa6,1,22040,2954,0x8952,1,39408,0,0xfbf5
,1,33509,0,0x8ee1,1,33510,1851,0x8bea,1,33511,0,0x9297
,1,33512,0,0xf8a4,1,39417,0,0xfbf6,1,39420,0,0xfbf7
,1,33515,0,0x93cf,1,39434,0,0xfbf8,1,39441,0,0xfbf9
,1,39446,0,0xfbfa,1,39450,0,0xfbfb,1,39456,0,0xfbfc
,1,33521,0,0x8970,1,39473,0,0xfc40,1,33523,5958,0xe494
,1,22057,2956,0x89dc,1,39478,0,0xfc41,1,39492,0,0xfc42
,1,33527,0,0xf8a5,1,39500,0,0xfc43,1,33529,0,0xe499
,1,22063,2957,0x9a82,1,33531,5966,0xe498,1,22065,0,0x8ffa
,1,22066,5075,0x9a7d,1,33534,0,0xede8,1,22068,0,0x9a7b
,1,33536,0,0xede9,1,22070,5078,0x9a7c,1,33538,5968,0x96ce
,1,22072,2961,0x9a7e,1,33540,0,0x89d6,1,33541,0,0x8a9d
,1,22075,2962,0x8851,1,33543,0,0xf8a6,1,22077,8166,0xf1c1
,1,33545,0,0xe49d,1,22079,0,0xf1c2,1,22080,0,0xf1c3
,1,33548,0,0xf8a8,1,22082,0,0x895c,1,33550,0,0x8c73
,1,39512,0,0xfc44,1,19394,0,0xfc45,1,39599,0,0xfc46
,1,22087,0,0xf1c4,1,19402,0,0xfc47,1,22089,0,0x8852
,1,39607,0,0xfc48,1,33558,0,0xe4a1,1,22092,2972,0x9158
,1,33560,0,0xe4ab,1,22094,0,0x9a78,1,19410,0,0xfc49
,1,22096,8167,0x9a79,1,33564,0,0x88a9,1,33565,0,0xf8ab
,1,22099,0,0xeaa5,1,39609,0,0xfc4a,1,43962,0,0xfc4b
,1,39622,0,0xfc4c,1,33570,0,0xedec,1,33571,0,0xe4b2
,1,39632,0,0xfc4d,1,39634,0,0xfc4e,1,22107,0,0x8a9a
,1,39637,0,0xfc4f,1,33576,0,0x88ef,1,22110,0,0xf1c5
,1,19432,0,0xfc50,1,33579,5971,0xe4a9,1,39644,0,0xfc51
,1,33581,0,0xeded,1,39648,0,0xfc52,1,22116,2971,0x9a81
,1,33584,0,0xf8ac,1,22118,2968,0x8854,1,33586,0,0xe4a2
,1,22120,0,0x8aed,1,33588,0,0xe4a0,1,22122,2964,0x9a84
,1,33590,1853,0x9283,1,22124,0,0x9a83,1,33592,5973,0x91f9
,1,33593,0,0xe4a5,1,22127,5080,0x8856,1,39653,0,0xfc53
,1,22129,8168,0x8857,1,22130,0,0x8858,1,39657,0,0xfc54
,1,22132,0,0x95ac,1,33600,0,0xe4a4,1,22134,0,0x8853
,1,39683,0,0xfc55,1,22136,5082,0x93d3,1,33604,0,0xf8ae
,1,22138,2969,0x94b6,1,39692,0,0xfc56,1,33607,0,0xedf0
,1,39696,0,0xfc57,1,33609,0,0x9190,1,33610,0,0x8c74
,1,22144,0,0x9a86,1,39698,0,0xfc58,1,39702,0,0xfc59
,1,39708,0,0xfc5a,1,33615,0,0x8960,1,33616,0,0xe4a6
,1,22150,5083,0x9a85,1,22151,555,0x8a64,1,22152,0,0xf1c9
,1,33620,0,0xf8a9,1,22154,5084,0x9a87,1,39723,0,0xfc5b
,1,22156,8169,0xf1ca,1,33624,0,0x9191,1,39731,0,0xfc5c
,1,22159,0,0x9a8a,1,39741,0,0xfc5d,1,19488,0,0xfc5e
,1,39755,0,0xfc5f,1,39779,0,0xfc60,1,22164,0,0x9a89
,1,22165,0,0xf1cb,1,39781,0,0xfc61,1,33634,0,0xedeb
,1,33635,0,0xf3c8,1,22169,0,0x8859,1,22170,0,0xf1cc
,1,39787,0,0xfc62,1,39788,0,0xfc63,1,22173,0,0xf1cd
,1,22174,0,0x885a,1,39795,0,0xfc64,1,22176,0,0x9a88
,1,39798,0,0xfc65,1,22178,0,0x9458,1,39799,0,0xfc66
,1,39846,0,0xfc67,1,22181,0,0x9a8b,1,39852,0,0xfc68
,1,39857,0,0xfc69,1,33651,5975,0xe4b8,1,22185,0,0x885b
,1,33653,0,0xe4b9,1,43963,0,0xfc6a,1,22188,425,0x885c
,1,22189,0,0xf1cf,1,22190,0,0x9a8c,1,43964,0,0xfc6b
,1,33659,0,0x89ac,1,33660,0,0xe4b6,1,33661,5977,0xedf3
,1,22195,0,0x885d,1,22196,8177,0x9a8e,1,39858,0,0xfc6c
,1,22198,0,0x9a8d,1,39864,0,0xfc6d,1,39870,0,0xfc6e
,1,39879,0,0xfc6f,1,33669,0,0xe4ac,1,33670,0,0xedf4
,1,22204,2981,0x9a90,1,39923,0,0xfc70,1,33673,0,0xe4bb
,1,33674,0,0xe4b5,1,22208,0,0x9a93,1,22209,0,0x9a91
,1,22210,8180,0x9a8f,1,22211,2974,0x9a92,1,39896,11106,0xfc71
,1,22213,0,0xf1d1,1,39901,0,0xfc72,1,33682,0,0xedf5
,1,22216,2960,0x9a94,1,22217,8182,0x885e,1,22218,8183,0x885f
,1,33686,0,0xe4b1,1,39911,0,0xfc73,1,33688,5981,0xedf6
,1,22222,0,0x9a95,1,33690,0,0xe4ad,1,33691,0,0xf8b7
,1,22225,0,0x9a96,1,33693,0,0xf8b8,1,33694,1854,0x8ace
,1,33695,0,0xe4af,1,33696,0,0xe4ba,1,39914,0,0xfc74
,1,22231,2973,0x9a98,1,22232,0,0x9964,1,39915,0,0xfc75
,1,22234,0,0x8efa,1,22235,0,0x8e6c,1,33703,0,0xedf7
,1,33704,0,0xe4bc,1,22238,5085,0x89f1,1,33706,5982,0xe4ae
,1,22240,1332,0x88f6,1,39919,0,0xfc76,1,39918,0,0xfc77
,1,22243,0,0x9263,1,43965,0,0xfc78,1,39930,0,0xfc79
,1,33713,0,0x9789,1,43967,0,0xfc7a,1,22248,0,0xf1d4
,1,39927,0,0xfc7b,1,33717,0,0xe4b7,1,43966,0,0xfc7c
,1,39958,0,0xfc7d,1,39960,0,0xfc7e,1,22254,0,0x9a99
,1,39961,0,0xfc80,1,22256,0,0x8da2,1,39962,0,0xfc81
,1,22258,2994,0x88cd,1,22259,0,0x907d,1,33727,0,0xedf9
,1,33728,0,0xedfa,1,33729,5983,0xe4c5,1,22263,0,0xf1d6
,1,39965,0,0xfc82,1,22265,0,0x9a9a,1,22266,819,0x8cc5
,1,39970,0,0xfc83,1,33735,0,0xedfb,1,22269,0,0x8d91
,1,33737,0,0xf8b9,1,33738,1855,0x8b65,1,22272,0,0x9a9b
,1,33740,0,0x8bdb,1,39975,0,0xfc84,1,22275,2983,0x95de
,1,22276,5089,0x9a9d,1,33744,0,0xf8ba,1,33745,0,0xee40
,1,39977,0,0xfc85,1,33747,1856,0x89d9,1,22281,8187,0x9a9e
,1,22282,0,0x8860,1,33750,1857,0x8fd2,1,39978,0,0xfc86
,1,22285,2987,0x9aa1,1,43968,0,0xfc87,1,22287,0,0x8c97
,1,39985,0,0xfc88,1,33756,0,0x8dd8,1,22290,8188,0x8980
,1,22291,0,0x9aa2,1,33759,0,0x9370,1,33760,5989,0xe4c8
,1,22294,5094,0x9aa4,1,39990,0,0xfc89,1,22296,0,0x9aa3
,1,39991,0,0xfc8a,1,33765,0,0xf8bd,1,40005,0,0xfc8b
,1,22300,0,0x9aa6,1,40028,0,0xfc8c,1,33769,0,0x95ec
,1,22303,5095,0x9379,1,33771,0,0xe4bf,1,22305,0,0x8862
,1,43970,0,0xfc8d,1,22307,0,0xf1d9,1,33775,0,0x89d8
,1,33776,0,0x8cd4,1,33777,1858,0x9548,1,22311,2991,0x88b3
,1,22312,0,0x8ddd,1,33780,5993,0xe4bd,1,40009,0,0xfc8e
,1,40010,0,0xfc8f,1,33783,0,0xe4c6,1,22317,0,0x8c5c
,1,33785,0,0xf8be,1,22319,0,0x8863,1,22320,2996,0x926e
,1,43969,0,0xfc90,1,33789,0,0xe4c1,1,22323,0,0x8864
,1,22324,0,0x8865,1,40020,0,0xfc91,1,33793,0,0xee43
,1,22327,0,0x9aa8,1,22328,2984,0x9aa9,1,33796,0,0x93b8
,1,40024,0,0xfc92,1,22331,5096,0x9aab,1,33799,0,0xe4c7
,1,40027,0,0xfc93,1,40029,0,0xfc94,1,33802,0,0xee45
,1,22336,2988,0x9aac,1,33804,0,0x9647,1,22338,2993,0x8de2
,1,33806,0,0x88de,1,33807,0,0xf8bf,1,22341,0,0xf1dd
,1,22342,8189,0xf1de,1,22343,0,0x8bcf,1,33811,0,0xe4be
,1,40031,0,0xfc95,1,22346,8190,0x9656,1,40041,0,0xfc96
,1,22348,8194,0xf1df,1,22349,0,0xf1e0,1,22350,0,0x9aaa
,1,22351,0,0x9aad,1,22352,0,0x8dbf,1,22353,0,0x8d42
,1,40042,0,0xfc97,1,40043,0,0xfc98,1,40045,0,0xfc99
,1,33824,0,0xe4cc,1,40046,0,0xfc9a,1,33826,0,0xe4cb
,1,40048,0,0xfc9b,1,40050,0,0xfc9c,1,40053,0,0xfc9d
,1,40058,0,0xfc9e,1,40166,0,0xfc9f,1,40178,0,0xfca0
,1,33833,0,0x948b,1,33834,0,0xe4d2,1,40203,0,0xfca1
,1,22369,3001,0x9ab1,1,40194,0,0xfca2,1,43971,0,0xfca3
,1,22372,0,0x8da3,1,40209,0,0xfca4,1,33841,1159,0x8a9e
,1,40215,0,0xfca5,1,22376,0,0xf1e2,1,22377,0,0x9aae
,1,22378,3005,0x92d8,1,40216,0,0xfca6,1,43973,0,0xfca7
,1,33848,0,0xe4ce,1,33849,0,0xf8c4,1,22383,0,0xf1e3
,1,22384,0,0x8866,1,33852,0,0xe4d3,1,33853,0,0x978e
,1,22387,0,0xf1e4,1,22388,0,0xf1e5,1,22389,0,0xf1e6
,1,19652,0,0xfca8,1,22391,0,0x8867,1,43972,0,0xfca9
,1,40242,0,0xfcaa,1,19665,0,0xfcab,1,33862,5996,0xe4dc
,1,22396,0,0x8868,1,33864,0,0xee47,1,33865,0,0x9774
,1,22399,8196,0x9ab2,1,40258,0,0xfcac,1,40266,0,0xfcad
,1,22402,0,0x9082,1,33870,0,0x97a8,1,33871,0,0xf8c6
,1,40287,0,0xfcae,1,33873,0,0xf8c7,1,33874,0,0xf8c8
,1,22408,0,0x9ab0,1,22409,0,0x9ab3,1,40290,0,0xfcaf
,1,22411,0,0x8c5e,1,33879,0,0x9298,1,43974,0,0xfcb0
,1,33881,0,0xf8c9,1,33882,0,0xf8ca,1,33883,0,0x8a8b
,1,33884,0,0xf8cb,1,40297,0,0xfcb1,1,22419,0,0x9ab4
,1,33887,0,0xee48,1,40299,0,0xfcb2,1,33889,0,0x9592
,1,33890,0,0xe4e2,1,33891,0,0x939f,1,43975,0,0xfcb3
,1,22426,8197,0xf1eb,1,33894,0,0x88af,1,22428,0,0x8869
,1,22429,0,0xf1ec,1,33897,6001,0xe4db,1,40307,0,0xfcb4
,1,22432,2998,0x9ab5,1,33900,0,0x9192,1,22434,2997,0x8d43
,1,22435,2999,0x8a5f,1,22436,3003,0x9ab7,1,33904,0,0xee49
,1,33905,0,0x944b,1,40310,0,0xfcb5,1,33907,6002,0xee4a
,1,40311,0,0xfcb6,1,33909,1862,0x88a8,1,33910,0,0xf8ce
,1,33911,6000,0xe4d6,1,33912,0,0xf8cf,1,33913,0,0xe4df
,1,33914,0,0x9598,1,22448,0,0x9ab9,1,33916,0,0xf8d0
,1,40318,0,0xfcb7,1,22451,0,0x9ab6,1,40324,0,0xfcb8
,1,40333,0,0xfcb9,1,33921,0,0xf8d1,1,33922,0,0xe4da
,1,22456,0,0x886c,1,33924,0,0xe4d5,1,33925,0,0xee4b
,1,40345,0,0xfcba,1,40353,0,0xfcbb,1,40383,0,0xfcbc
,1,40373,0,0xfcbd,1,40377,0,0xfcbe,1,33931,1860,0x8fd3
,1,40381,0,0xfcbf,1,40387,0,0xfcc0,1,22467,0,0x9aba
,1,40391,0,0xfcc1,1,33936,0,0x8f4e,1,22470,0,0x9abb
,1,22471,0,0x886d,1,22472,0,0x886e,1,33940,0,0x8eaa
,1,40393,0,0xfcc2,1,22475,0,0x9684,1,22476,8201,0xf1f1
,1,40406,0,0xfcc3,1,22478,1477,0x8fe9,1,22479,0,0x886f
,1,40410,0,0xfcc4,1,33948,0,0x9566,1,22482,0,0x9abd
,1,22483,5098,0x9abe,1,22484,3013,0x9abc,1,40415,0,0xfcc5
,1,22486,3020,0x9ac0,1,22487,0,0xf1ef,1,40416,0,0xfcc6
,1,40419,0,0xfcc7,1,40436,0,0xfcc8,1,33958,0,0xf8d5
,1,22492,0,0x9457,1,19719,0,0xfcc9,1,22494,0,0xf1f4
,1,22495,0,0x88e6,1,22496,0,0x9575,1,40458,0,0xfcca
,1,33965,0,0xe4d8,1,22499,0,0x9ac1,1,22500,5099,0x8870
,1,40450,0,0xfccb,1,22502,0,0xf1f5,1,33970,0,0x8a97
,1,40461,0,0xfccc,1,33972,0,0xee4e,1,40473,0,0xfccd
,1,40476,0,0xfcce,1,40477,0,0xfccf,1,22509,765,0x8871
,1,33977,0,0xe4e3,1,33978,0,0xee4f,1,33979,6011,0xe4e8
,1,33980,0,0x9193,1,40571,0,0xfcd0,1,33982,0,0xf8d6
,1,22516,3007,0x8ffb,1,22517,5100,0x8872,1,22518,3018,0x8873
,1,22519,5102,0x8eb7,1,22520,0,0xf1f8,1,33988,1299,0x927e
,1,22522,0,0x8aee,1,33990,6013,0xe4ec,1,22524,0,0x8de9
,1,22525,0,0xf1fa,1,33993,0,0x9775,1,22527,3006,0x8874
,1,33995,1451,0x8a57,1,43977,0,0xfcd1,1,22530,3014,0x93b0
,1,33998,0,0xf8d8,1,22532,8203,0xf1fb,1,22533,3015,0x8c98
,1,22534,1466,0x91cd,1,40576,0,0xfcd2,1,34003,0,0xf8da
,1,22537,0,0x8875,1,22538,0,0x9abf,1,22539,3019,0x9ac2
,1,40581,0,0xfcd3,1,40603,0,0xfcd4,1,34009,0,0xe4e6
,1,34010,0,0xe4e9,1,40616,0,0xfcd5,1,34012,0,0xf8d3
,1,43978,0,0xfcd6,1,40637,0,0xfcd7,1,43979,0,0xfcd8
,1,22549,0,0x91c2,1,40671,0,0xfcd9,1,40679,0,0xfcda
,1,40686,0,0xfcdb,1,22553,0,0x9ac3,1,40703,0,0xfcdc
,1,40706,0,0xfcdd,1,34023,0,0xf8dc,1,22557,0,0x9ac4
,1,22558,0,0xf1fc,1,34026,0,0xf8dd,1,22560,0,0xf240
,1,34028,1867,0x9648,1,19831,0,0xfcde,1,34030,0,0x9840
,1,22564,8206,0x92e7,1,34032,0,0xf8df,1,34033,0,0xf8e0
,1,22567,0,0xf241,1,40707,0,0xfcdf,1,34036,0,0xe4f1
,1,22570,0,0x8aac,1,40727,0,0xfce0,1,40729,0,0xfce1
,1,40751,0,0xfce2,1,40759,0,0xfce3,1,22575,8207,0xea9f
,1,22576,0,0x8981,1,22577,3029,0x95f1,1,22578,8208,0xf242
,1,40762,0,0xfce4,1,22580,3023,0x8fea,1,34048,1219,0x8ec1
,1,40765,0,0xfce5,1,40769,0,0xfce6,1,40773,0,0xfce7
,1,22585,0,0xf243,1,22586,0,0x8de4,1,34054,0,0xe4cf
,1,40774,0,0xfce8,1,22589,0,0x9acc,1,40787,0,0xfce9
,1,40789,0,0xfcea,1,22592,0,0x95bb,1,22593,8211,0x97db
,1,40792,0,0xfceb,1,43980,0,0xfcec,1,40797,0,0xfced
,1,43981,0,0xfcee,1,34065,0,0x95cc,1,40809,0,0xfcef
,1,34067,0,0x96a0,1,34068,6016,0xe4f7,1,22602,3027,0x89f2
,1,22603,0,0x9ac8,1,34071,6017,0xe4f2,1,34072,0,0xe4f3
,1,43982,0,0xfcf0,1,34074,0,0x8955,1,34075,0,0xf8e4
,1,22609,0,0x9159,1,22610,0,0x9acb,1,34078,0,0xee54
,1,22612,3026,0x9383,1,40813,0,0xfcf1,1,34081,0,0xe4ef
,1,22615,0,0x9368,1,22616,5103,0x9384,1,22617,8214,0x94b7
,1,22618,8216,0x92cb,1,34086,0,0x92d3,1,40816,0,0xfcf2
,1,40821,0,0xfcf3,1,22622,0,0x8dc7,1,43983,0,0xfcf4
,1,34091,0,0xf8e7,1,22625,3024,0x8877,1,34093,1868,0x88fc
,0,0,0,0x0,1,22628,5105,0x8878,0,0,0,0x0
,0,0,0,0x0,1,34098,6018,0xee53,0,0,0,0x0
,1,22633,8224,0x8996,1,34101,0,0x91a0,1,22635,0,0x9355
,0,0,0,0x0,0,0,0,0x0,0,0,0,0x0
,0,0,0,0x0,1,22640,0,0x9ac9,0,0,0,0x0
,1,34109,1865,0x95c1,1,34110,0,0xf8ed,0,0,0,0x0
,1,22645,3030,0x906f,1,34113,0,0xe540,0,0,0,0x0
,1,34115,0,0x94d7,1,22649,0,0x9acd,0,0,0,0x0
,0,0,0,0x0,1,22652,0,0x887a,1,34120,0,0xe4fc
,1,22654,751,0x8f6d,1,34122,0,0x8ec7,1,34123,0,0xe542
,0,0,0,0x0,0,0,0,0x0,1,22659,486,0x8bab
,1,34127,0,0xf8e9,1,22661,0,0x9ace,1,34129,0,0xf8ee
,0,0,0,0x0,1,34131,0,0xf8ef,1,22665,0,0x887b
,1,34133,6019,0xe543,1,22667,0,0xf249,1,34135,0,0x9599
,1,34136,6024,0xe4fb,1,34137,0,0xee57,1,34138,6026,0xe4d4
,1,22672,0,0xf24c,0,0,0,0x0,0,0,0,0x0
,1,22675,8240,0x95e6,1,22676,0,0xf24d,0,0,0,0x0
,1,34145,0,0xf8f1,1,22679,8249,0x919d,1,34147,0,0xe4fa
,1,34148,0,0xee58,0,0,0,0x0,0,0,0,0x0
,1,22684,0,0x92c4,1,34152,6030,0x986e,1,22686,1232,0x887c
,1,34154,1871,0x9593,0,0,0,0x0,0,0,0,0x0
,1,34157,0,0xe54a,0,0,0,0x0,1,34159,0,0xf8ea
,0,0,0,0x0,0,0,0,0x0,0,0,0,0x0
,1,22696,0,0x966e,1,22697,0,0x887e,1,22698,0,0xf24f
,1,22699,0,0x9ad1,1,34167,0,0xe550,0,0,0,0x0
,1,22702,0,0x9ad6,1,34170,0,0xee5b,1,34171,0,0xf8f4
,1,22705,0,0xf250,1,34173,0,0xf8f5,1,22707,3036,0x95ad
,1,34175,0,0xf8f6,1,34176,0,0xe544,1,34177,0,0xf8f7
,0,0,0,0x0,1,22712,0,0x9ad5,1,34180,1869,0x9496
,1,22714,0,0x9ad2,1,22715,8251,0x9ad4,1,34183,0,0xe54e
,1,34184,0,0xe546,1,22718,0,0x8da4,1,34186,0,0xe548
,0,0,0,0x0,1,22721,5110,0x95c7,0,0,0,0x0
,1,22723,0,0xf252,1,34191,0,0xee5d,1,22725,3038,0x9ad7
,1,34193,0,0xe547,1,22727,0,0x9264,1,34195,0,0xf8f9
,1,34196,0,0xe54b,1,22730,0,0x89f3,0,0,0,0x0
,1,34199,762,0x8992,1,22733,0,0xf253,1,22734,1248,0x8882
,0,0,0,0x0,1,34203,0,0xe54c,1,22737,3034,0x9ad9
,1,22738,8254,0x8881,1,22739,0,0x9ad8,1,22740,8258,0x8883
,1,22741,0,0x8d88,0,0,0,0x0,1,22743,5112,0x9ada
,1,22744,0,0x9adc,1,22745,3033,0x9adb,1,22746,0,0x8884
,1,34214,0,0x9145,1,22748,0,0x9ade,1,34216,0,0xe549
,1,34217,1872,0x8e46,1,34218,1874,0x9064,1,22752,497,0x8885
,1,34220,0,0x96f2,1,34221,6035,0xee5a,1,34222,0,0x96f7
,1,34223,1873,0x8f92,1,22757,5116,0x9add,0,0,0,0x0
,0,0,0,0x0,0,0,0,0x0,1,22761,0,0x8886
,0,0,0,0x0,1,22763,0,0x8e6d,1,22764,8261,0x9070
,0,0,0,0x0,1,22766,3045,0x9173,1,22767,3041,0x9ae1
,1,22768,0,0x90ba,1,22769,8262,0x88eb,1,22770,0,0x9484
,1,22771,0,0xf255,1,22772,0,0xf256,0,0,0,0x0
,1,34241,0,0x986d,1,22775,0,0x92d9,0,0,0,0x0
,1,22777,0,0x9ae3,1,22778,0,0x9ae2,1,22779,0,0x9ae4
,1,22780,8263,0x9ae5,1,22781,0,0x9ae6,1,34249,0,0xe553
,1,34250,0,0xf945,1,34251,0,0xee60,0,0,0,0x0
,1,34253,1876,0x9795,1,34254,0,0xee61,1,34255,0,0xe555
,1,34256,6036,0xe557,1,22790,0,0xf258,0,0,0,0x0
,0,0,0,0x0,1,22793,0,0x95cf,1,22794,3046,0x9ae8
,1,22795,0,0xf259,1,22796,0,0x8887,1,22797,8267,0xf25a
,1,34265,0,0xf947,1,22799,0,0x89c4,1,22800,0,0x9ae9
,1,34268,0,0xe55b,1,34269,0,0xe559,0,0,0,0x0
,1,22804,8269,0xf25b,1,22805,0,0x975b,1,22806,8272,0x8a4f
,0,0,0,0x0,1,22808,0,0x99c7,1,22809,1233,0x8f67
,1,22810,3047,0x91bd,1,22811,8277,0x9aea,1,22812,0,0x96e9
,0,0,0,0x0,1,34281,0,0x94cb,1,34282,0,0xe54d
,0,0,0,0x0,0,0,0,0x0,1,22818,5117,0x96b2
,0,0,0,0x0,1,22820,0,0xf25c,1,22821,0,0x9aec
,0,0,0,0x0,1,22823,0,0x91e5,0,0,0,0x0
,1,22825,0,0x9356,1,22826,0,0x91be,1,22827,8281,0x9576
,1,34295,1877,0x8f93,1,22829,0,0x9aee,1,22830,3048,0x899b
,1,34298,0,0xe561,1,34299,0,0x9194,1,22833,0,0x8eb8
,1,22834,0,0x9aef,1,34302,0,0xe560,1,34303,0,0xee64
,1,34304,0,0xf94c,0,0,0,0x0,1,22839,3032,0x88ce
,1,22840,0,0x9af0,1,34308,0,0xee65,1,34309,0,0xee66
,1,34310,0,0xe562,1,34311,0,0x9168,1,22845,0,0xf25f
,1,22846,0,0x9af1,1,34314,0,0xe55d,1,34315,0,0xe55f
,0,0,0,0x0,0,0,0,0x0,0,0,0,0x0
,1,22852,0,0x8982,1,34320,0,0xee67,1,22854,8282,0xf261
,1,22855,5121,0x8aef,1,22856,3049,0x93de,1,22857,0,0x95f2
,0,0,0,0x0,1,34326,0,0x9f50,1,34327,0,0x9f41
,1,34328,0,0xee69,1,22862,0,0x9af5,1,22863,3051,0x9174
,1,22864,0,0x9af4,1,22865,0,0x8c5f,0,0,0,0x0
,1,34334,0,0xf94e,1,22868,0,0x967a,1,22869,0,0x9af3
,1,34337,0,0xf94f,1,22871,3050,0x9385,1,22872,0,0x9af7
,1,34340,0,0xf950,1,22874,0,0x9af6,1,22875,0,0xf264
,1,34343,0,0xf951,1,22877,0,0x8889,1,34345,0,0xee6a
,1,22879,0,0xf265,1,22880,0,0x9af9,0,0,0,0x0
,1,34349,1879,0x9796,0,0,0,0x0,1,34351,0,0xe1ba
,1,22885,3052,0x899c,0,0,0,0x0,1,22887,0,0x9afa
,1,22888,0,0x8fa7,1,22889,0,0x9afc,1,22890,0,0x9244
,0,0,0,0x0,1,22892,0,0x9afb,1,22893,5128,0x888a
,1,22894,8285,0x95b1,0,0,0,0x0,0,0,0,0x0
,1,34364,0,0xf954,0,0,0,0x0,1,22899,0,0x8f97
,1,22900,3055,0x937a,1,22901,8287,0xf267,1,34369,6050,0x8888
,0,0,0,0x0,1,22904,0,0x9b40,0,0,0,0x0
,0,0,0,0x0,0,0,0,0x0,1,22908,0,0xf269
,1,22909,0,0x8d44,0,0,0,0x0,0,0,0,0x0
,0,0,0,0x0,1,22913,0,0x9b41,1,22914,3056,0x9440
,1,34382,1348,0x8cd5,1,22916,0,0x96cf,1,34384,0,0x8b73
,0,0,0,0x0,0,0,0,0x0,1,34387,0,0xf958
,1,34388,0,0xe569,1,22922,1784,0x9444,1,22923,8289,0x888b
,1,34391,0,0xee6c,1,22925,0,0x9b4a,0,0,0,0x0
,1,34394,0,0x8b95,1,34395,0,0xee6d,1,34396,0,0x97b8
,1,22930,0,0x888c,1,22931,492,0x8b57,1,34399,0,0xe56a
,0,0,0,0x0,1,8722,0,0x817c,1,913,1696,0x839f
,1,914,8586,0x83a0,1,915,1432,0x83a1,1,916,0,0x83a2
,1,917,5204,0x83a3,1,918,0,0x83a4,1,919,3370,0x83a5
,1,920,0,0x83a6,1,921,0,0x83a7,1,922,0,0x83a8
,1,923,6245,0x83a9,1,924,8598,0x83aa,1,925,0,0x83ab
,1,926,8603,0x83ac,1,927,524,0x83ad,1,928,0,0x83ae
,1,929,1212,0x83af,1,931,6248,0x83b0,1,932,0,0x83b1
,1,933,0,0x83b2,1,934,0,0x83b3,1,935,508,0x83b4
,1,936,0,0x83b5,1,937,804,0x83b6,1,945,1698,0x83bf
,1,946,499,0x83c0,1,947,8611,0x83c1,1,948,1700,0x83c2
,1,949,2046,0x83c3,1,950,1150,0x83c4,1,951,464,0x83c5
,1,952,2045,0x83c6
};
static HASHB gha;
static HASHB gha4;
static int _init_sj_utf8()
{
	gutf8 = gutf8_tbl;
	gucs4 = gucs4_tbl;

	ha = &gha;
	memset(ha,0,sizeof(HASHB));
	ha->ha_id[0] = 'H';
	ha->ha_id[1] = 'L';
	ha->ha_keylen = 4;
	ha->ha_maxreg = AKX_SJIS_HASH_MAX;
	ha->ha_prereg = AKX_SJIS_HASH_PRE;
	ha->ha_reg    = (char *)gutf8_cell_tbl;

	ha4 = &gha4;
	memset(ha4,0,sizeof(HASHB));
	ha4->ha_id[0] = 'H';
	ha4->ha_id[1] = 'L';
	ha4->ha_keylen = 4;
	ha4->ha_maxreg = AKX_SJIS_HASH_MAX;
	ha4->ha_prereg = AKX_SJIS_HASH_PRE;
	ha4->ha_reg    = (char *)gucs4_cell_tbl;

	gexe_set_utf8 = 1;

	return 0;
}
#endif
