package jp.nanah.bastub.controller;

import static org.springframework.web.bind.annotation.RequestMethod.*;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import jp.nanah.bastub.BastubConsts;
import jp.nanah.bastub.data.FilterParam;
import jp.nanah.bastub.data.JsonInfo;
import jp.nanah.bastub.data.KvData;
import jp.nanah.bastub.service.JsonService;
import jp.nanah.bastub.service.PageDataService;
import jp.nanah.bastub.service.UsingInfo;
import jp.nanah.bastub.util.BastubUtils;

@RestController
public class AnyRestController {
	/** 動作ログ */
	private static final Logger logger = LoggerFactory.getLogger(AnyRestController.class);
	/** リクエストのみ */
	private static final Logger reqlog = LoggerFactory.getLogger("REQLOG");

	@Autowired
	private JsonService jsonService;

	/**
	 * アプリケーション識別値。ログを区別するため。
	 */
	private static String APP_SNO = Long.toString(System.currentTimeMillis()/1000, 36).toUpperCase();


	/**
	 * リクエスト通番。起動するたびに0から始まる。
	 */
	private static long reqNo = 0;

	//*********************************
	// 主処理
	//*********************************

	@Value("${path_last_slash_valid:1}")
	private int pathLastSlashValid;

	/**
	 * パスの深さは最大10まで(可変にできる書き方を知らないので)。
	 * HTTPのメソッドは、GET/POST/DELETE/PUTのみ対応。(これ以外は使わないと思うので)
	 *
	 * @param body
	 * @param model
	 * @param req
	 * @param res
	 * @return
	 */
	@RequestMapping(path="/**"
					, method={GET, POST, PUT, DELETE})
	public String path1(@RequestBody(required=false) ModelMap body, ModelMap model, HttpServletRequest req, HttpServletResponse res) {
		logger.debug("★１；" + req.getRequestURI());
		return path_common(body, model, req, res);
	}

	/**
	 * Context-typeが"multipart/form-data"のときはSpringがBody部分をModelMapに変換できないので@RequestParamで受け取らないとダメ。
	 *
	 * @param p1～p10
	 * @param paramMap
	 * @param model
	 * @param req
	 * @param res
	 * @return
	 */
	@RequestMapping(path= "/**"
					, method={GET, POST, PUT, DELETE}
					, consumes = {MediaType.APPLICATION_FORM_URLENCODED_VALUE}
	)
	public String path1_multipart( @RequestParam(required=false) Map<String, String> paramMap, ModelMap model, HttpServletRequest req, HttpServletResponse res) {
		return path_common(paramMap, model, req, res);
	}

	/**
	 * リクエストの共通処理。
	 */
	private String path_common( Map paramMap, ModelMap model, HttpServletRequest req, HttpServletResponse res){

		synchronized (APP_SNO){
			Thread.currentThread().setName(APP_SNO + "_" + reqNo);
			reqNo++;
		}

		//リクエストを出力
		String method = req.getMethod();
		String reqPath = req.getRequestURL().toString();
		String queryStr = (req.getQueryString() == null) ? "" : req.getQueryString();
		Object bodyData = (paramMap == null) ? "" : paramMap;
		reqlog.info("[{}] {}{} {}", method, reqPath, queryStr, bodyData);

		logger.info("== ▼ == [{}] {}", method, reqPath);
		res.setContentType("application/json;charset=UTF-8");
		String[] pathArray = req.getRequestURI().split("/");
		List<String> pathList = Arrays.asList(pathArray); //BastubUtils.toValidList(p1, p2, p3, p4, p5, p6, p7, p8, p9, p10);

		if (pathLastSlashValid == 1){
			//URLがスラッシュ終わりなら、ファイル名が""の要求が来たものとする
			if (reqPath.endsWith("/")){
				pathList.add("");
			}
		}
		String result = getAny(pathList, paramMap, model, req, res);
		logger.info("== △ ==");

		return result;
	}


	//=================================
	// 共通処理
	//=================================

	@Autowired
	private PageDataService pgService;

	/**
	 * 【共通処理】ここが主処理。
	 *
	 * @param pathList
	 * @param body nullの場合あり
	 * @param model 未使用。将来削除するかも。
	 * @param req HTTPリクエスト
	 * @param res HTTPレスポンス
	 * @return
	 */
	public String getAny(List<String> pathList, Map body, ModelMap model, HttpServletRequest req, HttpServletResponse res){
		try {
			File pageDir = pgService.getPageDir();

			if (pageDir.exists() == false){
				String errormsg = "Bastubの設定フォルダ[" + pageDir.getAbsolutePath() + "]が見つかりません。application.propertiesのpagedata.rootを見直してください。";
				logger.info(errormsg);
				return errormsg;
			}

			UsingInfo ui = UsingInfo.getInitInstance(pathList, req);

			//パスを生成
			String path = StringUtils.join(pathList, "/");
			File dataFile = BastubUtils.getPagedataPath(pageDir, path, req.getMethod(), ".xlsx,.xls");
			File jsonFile = BastubUtils.getPagedataPath(pageDir, path, req.getMethod(), ".json");

			//データファイルを読み込む
			//-----------------------------------------------------------------------------------------------------
			//データファイルは、パス名＋"_"＋HTTPメソッド名のファイルがあればそれを優先する。ない場合はパス名のみ。
			//データファイルはファイルなし、あるいはファイル内のdataシートやfilterシートが無くてもOK。
			//==>データファイルがなければデータ埋め込みされないのでJSONは固定値で返る。
			//-----------------------------------------------------------------------------------------------------
			Sheet paramSheet = null;
			Sheet dataSheet = null;
			Workbook wb = readWorkbook(dataFile);
			if (wb == null) {
				logger.warn("Excelファイル無しのためJSONを固定で応答 (ExcelFile=[{}])", dataFile.getAbsolutePath());
			} else {
				paramSheet = wb.getSheet("filter");
				dataSheet = wb.getSheet("data");
				wb.close();

				if (paramSheet == null) {
					logger.debug("定義ファイル[{}]に[filter]シートが見つかりません", dataFile.getAbsolutePath());
				}
				if (dataSheet == null) {
					logger.debug("定義ファイル[{}]に[data]シートが見つかりません", dataFile.getAbsolutePath());
				}
			}
			//データ範囲をトリミング(エクセルはデータがなくても編集範囲だったりするので)
			trimDataSheet(dataSheet);

			//=================================
			// [1] リクエスト値をKeyValueリスト化
			//=================================
			List<KvData> requestData = getRequestData(req.getParameterMap(), body);

			//=================================
			// [2] 絞り込み条件を取得
			//=================================
			List<FilterParam> allFilter = getFilterParamList(paramSheet);
			List<FilterParam> validFilter = null;
			if (allFilter != null){
				validFilter = getValidFilter(allFilter, requestData, pathList);
			}

			//=================================
			// [3] エクセルデータから応答に返すレコードを絞り込む
			//=================================
			List<Row> resultTarget = pickupSheetData(dataSheet, validFilter);

			//=================================
			// [4] 応答用の入れ物を読み込む
			//=================================
			JsonInfo jsonInfo = jsonService.readJsonFile(jsonFile);
			if (jsonInfo.getJsonObject() == null) {
				//ファイルがない場合は自動で作られるのでここを通過することは殆どない
				res.setStatus(HttpServletResponse.SC_NOT_FOUND);
				String jsonPath = jsonFile.getAbsolutePath();
				logger.warn("## ◆ 応答JSON異常: 理由=[{}] パス=[{}]", jsonInfo.getErrorMessage(), jsonPath);
				//String errorJson = "{ \"no_file\": \"" + jsonPath.replaceAll("\\\\", "/") + "\"}";
				String errorJson = "no_file - " + jsonPath.replaceAll("\\\\", "/") + "]";
				return errorJson;
			}

			//=================================
			// [5] 応答データを生成
			//=================================
			jsonService.setDataToJsonObject(null, jsonInfo.getJsonObject(), resultTarget, ui);

			//応答
			String jsonText = jsonInfo.getJsonObject().toString(4);
			if (jsonInfo.isTopArray()){
				//{"dummy":xxxx  … } のカッコを除去する
				int n1 = jsonText.indexOf(":");
				int n2 = jsonText.lastIndexOf("}");
				jsonText = jsonText.substring(n1+1, n2);
			}

			//Edgeから直接リクエストするとエラーになることへの対応
			res.addHeader("Access-Control-Allow-Origin", "*");

			return jsonText;

		} catch (Throwable th) {
			logger.info("処理異常: {}", th.toString());
			return th.toString();
		}

	}

	//-------------------
	// Excelデータの処理
	//-------------------

	/**
	 * エクセルファイルごと保存。
	 */
	private Map<String, Object[]> workbookCache = new TreeMap<String, Object[]>();

	/**
	 * Workbookを読み込む。
	 * キャッシュがあればそれを返す。
	 *
	 * @param file
	 * @return
	 * @throws IOException
	 */
	protected synchronized Workbook readWorkbook(File file) throws IOException {
		if (file.exists() == false) {
			return null;
		}

		String key = file.getAbsolutePath();
		Object[] cache = workbookCache.get(key);
		if (cache != null){
			Long filetime = (Long)cache[0];
			if (filetime.longValue() == file.lastModified()){
				return (Workbook)cache[1];
			}
		}

		Workbook wb =  WorkbookFactory.create(file, null, true);//new HSSFWorkbook(poiFileSystem);
		workbookCache.put(key, new Object[]{file.lastModified(), wb});
		return wb;
	}

	protected void trimDataSheet(Sheet sheet){
		if (sheet == null || sheet.getLastRowNum() == 0){
			return;
		}

		//１行目(ヘッダ行)から、列数を確認
		Row topRow = sheet.getRow(0);
		int lastColNum = topRow.getLastCellNum() - 1;
		while (lastColNum >= 0){
			Cell cell = topRow.getCell(lastColNum);
			String s = BastubUtils.getCellText(cell);
			if (StringUtils.isNotBlank(s)){
				break;
			}
			lastColNum--;
		}

		int rowNum = sheet.getLastRowNum();
		while (rowNum >= 0) {
			Row row = sheet.getRow(rowNum);
			if (row != null){
				if (BastubUtils.isBlankRow(row)){
					sheet.removeRow(row);
				} else {
					for (int i=row.getLastCellNum()-1; i>lastColNum; i--){
						Cell cell = row.getCell(i);
						if (cell != null){
							row.removeCell(cell);
						}
					}
				}
			}
			rowNum--;
		}
	}

	/**
	 * [1]
	 * @param paramMap
	 * @param body
	 * @return
	 */
	public List<KvData> getRequestData(Map<String, String[]> paramMap, Map body){
		List<KvData> paramList = new ArrayList<KvData>();

		//URLの後ろに渡されるリクエストパラメータをKey/Value形式にする
		if (paramMap != null) {
			int cnt = 0;
			for (Map.Entry<String, String[]> ent : paramMap.entrySet()) {
				//BODYで渡していてもParamMapに入っているので除外する。
				if (BastubUtils.isJsonEntry(ent)){
					continue;
				}
				List<String> key = Arrays.asList(new String[] {ent.getKey()});
				for (String val : ent.getValue()) {
					KvData kv = appendKvParam(new KvData(key, val), paramList);
					cnt++;
				}
			}
		}

		//BodyリクエストをKey/Value形式にする
		if (body != null) {
			int startSize = paramList.size();
			appendEntryToParamList(body.entrySet(), null, paramList);
			for (int i=startSize; i<paramList.size(); i++){
				KvData kv = paramList.get(i);
			}
		}

		return paramList;
	}

	//■TODO 見直し候補
	//ここの考え方は見直したほうがよい。
	//リクエスト(QueryStringやBODY)の取得時点ではキー値は重複していてよいし、
	//key1/key2で組み合わせのときに一致する考え方でないとうまくいかなくなる。
	//クラス化してあるべきデータ保持形式にすべき。

	/**
	 * キーが一意になるようにKvDataを追加する。
	 * リクエストやJSONで同一キーがある場合、どれかが一致するような判断となるようにする。
	 *
	 * @param src
	 * @param dstList
	 */
	private KvData appendKvParam(KvData src, List<KvData> dstList){
		KvData nowval = null;
		for (int i=0; i<dstList.size(); i++){
			if (BastubUtils.equals(dstList.get(i).getKey(), src.getKey())){
				nowval = dstList.get(i);
				break;
			}
		}

		if (nowval == null){
			dstList.add(src);
			nowval = src;

		} else {
			nowval.addValue(src.getValueAsOne());
		}
		return nowval;
	}

	private void appendEntryToParamList(Set<Map.Entry<String, Object>> set, List<String> parentKeys, List<KvData> paramList){
		int count = 0;
		for (Map.Entry<String, Object> ent : set) {
			String key = ent.getKey();
			Object val = (ent.getValue() == null) ? "" : ent.getValue();

			List<String> thisKeys = new ArrayList<String>();
			if (parentKeys != null){
				thisKeys.addAll(parentKeys);
			}
			thisKeys.add(key);

			if (val instanceof Map){

				Map<String, Object> vmap = (Map<String, Object>)val;
				appendEntryToParamList(vmap.entrySet(), thisKeys, paramList);

			} else if (val instanceof List){

				List<Object> vlist = (List<Object>)val;
				appendListToParamList(vlist, thisKeys, paramList);

			} else {
				appendToParamList(val, thisKeys, paramList);
			}
			count++;
		}
	}

	private void appendListToParamList(List<Object> list, List<String> parentKeys, List<KvData> paramList){
		for (Object o : list){
			if (o instanceof Map){
				Map<String, Object> map = (Map<String, Object>)o;
				appendEntryToParamList(map.entrySet(), parentKeys, paramList);
			} else if (o instanceof List){
				List<Object> vlist = (List<Object>)o;
				appendListToParamList(vlist, parentKeys, paramList);
			} else {
				appendToParamList(o, parentKeys, paramList);
			}
		}
	}

	private void appendToParamList(Object obj, List<String> parentKeys, List<KvData> paramList){
		String valstr = obj.toString();
		KvData kv = new KvData(parentKeys, valstr);
		appendKvParam(kv, paramList);
	}

	/**
	 * [2]
	 * エクセルデータをフィルタリングするときの定義パラメータを取得する。
	 * ここでは置き換え文字や接尾辞を加工できないので設定値をそのまま取り出す(置き換えない)
	 *
	 * @param sheet
	 * @return Map<Cellの列名, フィルタリングデータのリスト>
	 */
	private List<FilterParam> getFilterParamList(Sheet sheet){
		List<FilterParam> filterList = null;
		if (sheet == null) {
			return filterList;
		}

		for (int i=0; i<=sheet.getLastRowNum(); i++) {
			Row row = sheet.getRow(i);
			List<String> cellList = BastubUtils.getRowValueList(row, false);

			//String prevName = "";
			if (cellList.size() >= 3) {
				// １行でもフィルタリングデータがあるときだけフィルタリングを適用する
				if (filterList == null){
					filterList =new ArrayList<FilterParam>();
				}
				String cellName = cellList.get(0);
				String compType = cellList.get(1);
				List<String> paramKey = cellList.subList(2, cellList.size());
				FilterParam fp = new FilterParam(cellName, compType, paramKey);

				filterList.add(fp);
			}
		}
		return filterList;
	}

	/**
	 * [2]-2
	 * フィルタリング時に判定する、データシートの列値と比較する判定値を設定する。
	 * フィルタリングで使用するキー値がrequestDataにあるときだけ、requestDataの値をフィルタリングに採用する。
	 * 判定値がないときはnullのまま。
	 *
	 * @param requestData
	 * @param filterParamList
	 */
	public List<FilterParam> getValidFilter(List<FilterParam> filterParamList, List<KvData> requestData, List<String> pathList) {
		List<FilterParam> validList = new ArrayList<FilterParam>();
		for (FilterParam fp : filterParamList) {
			//パスインデックスのときは、URLのパス値に置き換える
			if (fp.getRequestKeys().size() > 0) {
				String key0 = fp.getRequestKeys().get(0);
				if (key0.startsWith(BastubConsts.PATH_REPLACE_HEAD)) {
					int pathNumber = NumberUtils.toInt(key0.substring(1));
					if (pathNumber > 0 && pathNumber <= pathList.size()) {
						fp.setOneValue(pathList.get(pathNumber - 1));
						validList.add(fp);
						continue;
					}
				}
			}

			//要求キーのときは要求パラメータから値を探す
			List<String> reqKey = fp.getRequestKeys(); //こっちは定義

			//実際のリクエストのキー値と比較
			boolean isFound = false;
			String delimText = getDelimText(fp.getCompareType());
			for (KvData kv : requestData) {
				if (BastubUtils.equalsTail(kv.getKey(), reqKey)) {
					List<String> fpValues = toFilterValue(delimText, kv.getValues());
					fp.setValues(fpValues);
					validList.add(fp);
					isFound = true;
					//break;
				}
			}
			if (isFound == false){
				logger.warn("HTTPリクエスト内に、項目値[{}]が見つかりません。", reqKey);
			}
			if (delimText != null){
				fp.setCompareType("=");
			}
		}
		return validList;
	}

	private String getDelimText(String compareType){
		int n1 = compareType.indexOf("[");
		int n2 = compareType.indexOf("]");
		if (n1 <0 || n2 < 0){
			return null;
		}
		String delim = compareType.substring(n1+1, n2);
		return delim;
	}

	// 記号が=[x]のとき、値を分割する
	private List<String> toFilterValue(String delimText, List<String> org){
		if (delimText == null){
			return org;
		}

		List<String> dst = new ArrayList<String>();
		for (String s: org){
			dst.addAll(Arrays.asList((String[]) s.split(delimText, 0)));
		}
		return dst;
	}

	/**
	 * [3]
	 */
	public List<Row> pickupSheetData(Sheet sheet, List<FilterParam> validFilter) {
		List<Row> noDatasheet = new ArrayList<Row>();
		if (sheet == null) {
			return noDatasheet;
		}

		//まずは全データを対象にする
		List<Row> sheetData = new ArrayList<Row>();
		for (int i=sheet.getFirstRowNum(); i<=sheet.getLastRowNum(); i++) {
			Row row = sheet.getRow(i);
			if (row != null){
				sheetData.add(sheet.getRow(i));
			}
		}

		// フィルターがないときは全データ対象
		if (validFilter == null || validFilter.isEmpty()){
			return sheetData;
		}

		//列名
		String[] columnNames = getColumnNames(sheet);

		//フィルターがあるときは適合しないものを除去する
		for (FilterParam fp : validFilter) {
			int columnIndex = ArrayUtils.indexOf(columnNames, fp.getColumnName());
			if (columnIndex < 0) {
				logger.warn("***** dataシートに、列名[{}]がありません。", fp.getColumnName());
				continue;
			}

			//↓先頭行は列名なので除外する
			for (int i=1; i<sheetData.size(); i++) {
				Row row = sheetData.get(i);
				if (row == null) {
					continue;
				}

				Cell cell = row.getCell(columnIndex);
				String v = BastubUtils.getCellText(cell);
				if (v == null){
					continue;
				}

				boolean isMatch = false;
				String ct = fp.getCompareType();
				for (String fv : fp.getValues()){

					int n;
					if (StringUtils.isNumericSpace(v) && StringUtils.isNumericSpace(fv)){
						Integer v1 = Integer.parseInt(v);
						Integer v2 = Integer.parseInt(fv);
						n = v1.compareTo(v2);
					} else {
						n = v.compareTo(fv);
					}

					if (ct.equals("=")) {
						isMatch = (n == 0);
					} else if (ct.equals("<")) {
						isMatch = n < 0;
					} else if (ct.equals("<=")) {
						isMatch = n <= 0;
					} else if (ct.equals(">")) {
						isMatch = n > 0;
					} else if (ct.equals(">=")) {
						isMatch = n >= 0;
					} else if (ct.equals("!=") || ct.equals("<>")) {
						isMatch = n != 0;
					}
					if (isMatch){
						break;
					}
				}

				if (isMatch == false) {
					sheetData.set(i, null);
				}
			}
		}

		sheetData.removeAll(Collections.singleton(null));

		//debug
		for (Row row : sheetData) {
			StringBuilder sb = new StringBuilder();
			for (int i=row.getFirstCellNum(); i<row.getLastCellNum(); i++) {
				if (sb.length() > 0) {
					sb.append(",");
				}
				sb.append(BastubUtils.getCellText(row.getCell(i)));
			}
		}

		return sheetData;
	}

	private String[] getColumnNames(Sheet sheet) {
		Row row = sheet.getRow(sheet.getFirstRowNum());
		String[] columns = new String[row.getLastCellNum()];
		for (int i=row.getFirstCellNum(); i<row.getLastCellNum(); i++) {
			columns[i] = BastubUtils.getCellText(row.getCell(i));
		}
		return columns;
	}


}
