/**************************************************************************
 FolderMenu - easy access to project folders from menu.
 
 Copyright (C) 2013 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/pf/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.plugin.foldermenu;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import javax.swing.JMenu;
import static org.omegat.plugin.foldermenu.MenuHelper.getComparator;
import static org.omegat.plugin.foldermenu.MenuHelper.getFilteredListFiles;
import static org.omegat.plugin.foldermenu.MenuHelper.getIcon;
import static org.omegat.plugin.foldermenu.MenuHelper.getMenuListener;
import static org.omegat.plugin.foldermenu.MenuHelper.getMouseListener;
import org.omegat.util.Log;
import org.openide.awt.Mnemonics;

/**
 *
 * @author Yu-Tang
 */
public class ShellLinkMenu extends ShellLinkMenuItem {

    public ShellLinkMenu(JMenu m) {
        super(m);
    }

    public ShellLinkMenu(File folder) throws IOException {
        this(folder, null);
    }

    public ShellLinkMenu(String label) throws IOException {
        this(null, label);
    }

    public ShellLinkMenu(File folder, String label) throws IOException {
        JMenu m;
        if (label == null) {
            m = new JMenu(folder.getName());
        } else {
            m = new JMenu();
            Mnemonics.setLocalizedText(m, label);
        }
        
        m.addMenuListener(getMenuListener());
        m.addMouseListener(getMouseListener());
        
        if (folder != null) {
            m.setIcon(getIcon(folder));
            m.setActionCommand(folder.getCanonicalPath());
        } else {
            m.setEnabled(false);     
        }

        menuItem = m;
    }

    public void createChildren() {
        JMenu menu = (JMenu) menuItem;
        if (menu.isEnabled() && menu.getItemCount() == 0) {
            File folder = new File(menu.getActionCommand());
            File[] filteredListFiles = getFilteredListFiles(folder);
            Arrays.sort(filteredListFiles, getComparator());

            for (File file : filteredListFiles) {
                try {
                    if (file.isDirectory() && hasChildren(file)) {
                        menu.add(new ShellLinkMenu(file).getMenuItem());
                    } else {
                        menu.add(new ShellLinkMenuItem(file).getMenuItem());
                    }
                } catch (IOException ex) {
                    Log.log(ex);
                }
            }
        }
    }
    
    public JMenu getMenu() {
        return (JMenu) menuItem;
    }

    public void link(File folder) throws IOException {
        JMenu menu = (JMenu) menuItem;
        menu.setActionCommand(folder.getCanonicalPath());
        menu.setIcon(getIcon(folder));
        menu.setEnabled(true);
    }

    public void unlink() {
        JMenu menu = (JMenu) menuItem;
        menu.setEnabled(false);
        menu.removeAll();
        menu.setActionCommand("");
        menu.setIcon(null);
    }

    private boolean hasChildren(File folder) {
        return getFilteredListFiles(folder).length > 0;
    }
}
