/*
 *  The MIT License
 * 
 *  Copyright 2011 Masahiko, SAWAI <say@users.sourceforge.jp>.
 * 
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 * 
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 * 
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.android_r;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.net.Uri;
import android.os.Bundle;
import android.text.format.DateFormat;
import android.view.View;
import android.widget.Button;
import android.widget.TextView;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Date;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.routine_work.android_r.utils.Log;

/**
 *
 * @author Masahiko, SAWAI <say@users.sourceforge.jp>
 */
public class AboutAppActivity extends Activity
	implements View.OnClickListener
{
	private static final String LOG_TAG = "android.R";
	private Intent[] applicationManagementIntents;

	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		setContentView(R.layout.about_app_activity);

		// init buttons
		Button openAndroidMarketButton = (Button) findViewById(R.id.open_android_market_button);
		openAndroidMarketButton.setOnClickListener(this);
		Button openApplicationManagementButton = (Button) findViewById(R.id.open_application_management_button);
		openApplicationManagementButton.setOnClickListener(this);


		// init package build time
		Date buildTime = getPackageBuildTime();
		if (buildTime != null)
		{
			String buildTimeformat = getString(R.string.build_time_format);
			CharSequence buildTimeText = DateFormat.format(buildTimeformat, buildTime);

			TextView buildTimeTextView = (TextView) findViewById(R.id.build_time_textview);
			buildTimeTextView.setText(buildTimeText);
		}

		// init package version
		PackageManager packageManager = getPackageManager();
		try
		{
			PackageInfo packageInfo = packageManager.getPackageInfo(getPackageName(), 0);

			TextView versionTextView = (TextView) findViewById(R.id.version_textview);
			versionTextView.setText(packageInfo.versionName);

		}
		catch (NameNotFoundException ex)
		{
			Log.e(LOG_TAG, "Get package infomation failed.", ex);
		}

	}

	public void onClick(View view)
	{
		int id = view.getId();
		switch (id)
		{
			case R.id.open_android_market_button:
				openAndroidMarket(view);
				break;
			case R.id.open_application_management_button:
				openApplicationManagement(view);
				break;
			default:
				throw new AssertionError();
		}
	}

	public void openAndroidMarket(View view)
	{
		String marketUri = String.format("http://market.android.com/details?id=%s", getPackageName());
		Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(marketUri));
		intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
		startActivity(intent);
	}

	public void openApplicationManagement(View view)
	{
		Intent[] intents = getApplicationManagementIntents();
		int index = 0;
		boolean done = false;
		do
		{
			try
			{
				Log.d(LOG_TAG, "open application management : index => " + index + ", intent => " + intents[index]);
				startActivity(intents[index]);
				done = true;
			}
			catch (ActivityNotFoundException e)
			{
				index++;
				if (index == intents.length)
					done = true;
			}
		}
		while (done == false);
	}

	private Intent[] getApplicationManagementIntents()
	{
		if (applicationManagementIntents == null)
		{
			// API Level 9 or later
			Intent intent0 = new Intent(Intent.ACTION_VIEW);
			intent0.setClassName("com.android.settings",
				"com.android.settings.applications.InstalledAppDetails");
			intent0.setData(Uri.fromParts("package", getPackageName(), null));
			intent0.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

			// API Level 8 or earlier
			Intent intent1 = new Intent(Intent.ACTION_VIEW);
			intent1.setClassName("com.android.settings",
				"com.android.settings.InstalledAppDetails");
			intent1.putExtra("pkg", getPackageName());
			intent1.putExtra("com.android.settings.ApplicationPkgName",
				getPackageName());
			intent1.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

			// API Level 3(not in use?)
			Intent intent2 = new Intent(android.provider.Settings.ACTION_MANAGE_APPLICATIONS_SETTINGS);
			intent2.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

			applicationManagementIntents = new Intent[]
			{
				intent0, intent1, intent2,
			};
		}
		return applicationManagementIntents;
	}

	private Date getPackageBuildTime()
	{
		final String targetFileName = "classes.dex";
		Date packageBuildTime = null;
		try
		{
			FileInputStream fis = new FileInputStream(getPackageCodePath());
			try
			{
				ZipInputStream zipInputStream = new ZipInputStream(fis);
				try
				{
					ZipEntry zipEntry = null;
					while ((zipEntry = zipInputStream.getNextEntry()) != null)
					{
						try
						{
							String entryName = zipEntry.getName();
							Date entryTime = new Date(zipEntry.getTime());
							if (targetFileName.equals(entryName))
							{
								packageBuildTime = entryTime;
							}
						}
						finally
						{
							zipInputStream.closeEntry();
						}

					}
				}
				finally
				{
					zipInputStream.close();
				}
			}
			finally
			{
				fis.close();
			}
		}
		catch (IOException ex)
		{
			Log.e(LOG_TAG, "Getting package build time failed.", ex);
		}

		return packageBuildTime;
	}
}
