/* cdedit3 -- class diagram creation/manipulation program
 * Copyright (C) 2001 Touge Kamisimo
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "math.h"
#include "cd_util.h"
#include "cd_arc.h"

extern GdkFont *cd_font;
extern GdkGC *cd_black_gc;
extern GdkGC *cd_white_gc;
extern GdkGC *cd_select_gc;

Arc cd_arc;
ArcItem cd_arc_item_last_deleted;

/*
 * cd_arc_iterm_term_init
 */
void cd_arc_item_term_init(ArcTerm *at_ptr)
{
  at_ptr->point.x          = 0;
  at_ptr->point.y          = 0;
  at_ptr->theta            = 0.0;
  at_ptr->face             = face_none;
  at_ptr->node_id          = 0;
  at_ptr->type             = association;
  at_ptr->mult             = m_none;
  at_ptr->mult_rect.x      = 0;
  at_ptr->mult_rect.y      = 0;
  at_ptr->mult_rect.width  = 0;
  at_ptr->mult_rect.height = 0;
}

/*
 * cd_arc_item_init
 */
void cd_arc_item_init(int i)
{
  cd_arc.item[i].id = 0;

  sprintf(cd_arc.item[i].name, "");
  
  cd_arc.item[i].name_rect.x      = 0;
  cd_arc.item[i].name_rect.y      = 0;
  cd_arc.item[i].name_rect.width  = 0;
  cd_arc.item[i].name_rect.height = 0;

  cd_arc_item_term_init(&cd_arc.item[i].start);
  cd_arc_item_term_init(&cd_arc.item[i].end);

  cd_arc.item[i].select           = FALSE;
}

/*
 * cd_arc_item_new_start
 */
void cd_arc_item_new_start(int x, int y, int start_node_index, CdToolType type)
{
  int i = cd_arc.count;

  cd_arc_item_init(i);

  if(type == generalization)
    cd_arc.item[i].start.type = generalization;

  cd_node_item_calc_center(start_node_index, &cd_arc.item[i].start.point);
  cd_arc.item[i].start.node_id = cd_node_item_get_id(start_node_index);
}

/*
 * cd_arc_item_new_end
 */
void cd_arc_item_new_end(int x, int y, int end_node_index, CdToolType type)
{
  int i = cd_arc.count;

  cd_node_item_calc_center(end_node_index, &cd_arc.item[i].end.point);
  cd_arc.item[i].end.node_id = cd_node_item_get_id(end_node_index);

  if(strlen(cd_arc.item[i].name) > 0)
    cd_arc_item_calc_name_rect(i);

  switch(type){
  case association2:
    cd_arc.item[i].end.type = association2;
    break;
  case aggregation:
    cd_arc.item[i].end.type = aggregation;
    break;
  case aggregation2:
    cd_arc.item[i].end.type = aggregation2;
    break;
  case implements:
    cd_arc.item[i].end.type = implements;
    break;
  }

  cd_node_item_calc_arc_points(cd_arc.item[i].start.node_id,
			       cd_arc.item[i].end.node_id,
			       &cd_arc.item[i]);

  cd_arc.item[i].id = cd_arc.seq;

  cd_arc.count++;
  cd_arc.seq++;
}

/*
 * cd_arc_item_is_exist
 */
gboolean cd_arc_item_is_exist(int start_node_id, int end_node_id)
{
  int i;

  for(i=0; i<cd_arc.count; i++){
    if(cd_arc.item[i].start.node_id == start_node_id &&
       cd_arc.item[i].end.node_id == end_node_id)
      return TRUE;
    if(cd_arc.item[i].start.node_id == end_node_id &&
       cd_arc.item[i].end.node_id == start_node_id)
      return TRUE;
  }

  return FALSE;
}

void cd_arc_item_term_copy(ArcTerm *dst, ArcTerm *src)
{
  memcpy(dst, src, sizeof(ArcTerm));
}

/*
 * cd_arc_item_copy
 */
void cd_arc_item_copy(int dst, int src)
{
  cd_arc.item[dst].id               = cd_arc.item[src].id;

  sprintf(cd_arc.item[dst].name, "%s", cd_arc.item[src].name);
  
  cd_arc.item[dst].name_rect.x      = cd_arc.item[src].name_rect.x;
  cd_arc.item[dst].name_rect.y      = cd_arc.item[src].name_rect.y;
  cd_arc.item[dst].name_rect.width  = cd_arc.item[src].name_rect.width;
  cd_arc.item[dst].name_rect.height = cd_arc.item[src].name_rect.height;

  cd_arc_item_term_copy(&cd_arc.item[dst].start, &cd_arc.item[src].start);
  cd_arc_item_term_copy(&cd_arc.item[dst].end, &cd_arc.item[src].end);

  cd_arc.item[dst].select           = cd_arc.item[src].select;
}

/*
 * cd_arc_item_calc_name_rect
 */
void cd_arc_item_calc_name_rect(int i)
{
  int sw, sh;
  int mx, my;

  sw = gdk_string_width(cd_font, cd_arc.item[i].name);
  sh = gdk_string_height(cd_font, cd_arc.item[i].name);

  mx = (cd_arc.item[i].start.point.x + cd_arc.item[i].end.point.x) / 2;
  my = (cd_arc.item[i].start.point.y + cd_arc.item[i].end.point.y) / 2;

  cd_arc.item[i].name_rect.x      = mx - sw / 2;
  cd_arc.item[i].name_rect.y      = my - sh;
  cd_arc.item[i].name_rect.width  = sw;
  cd_arc.item[i].name_rect.height = sh;
}  

/*
 * cd_arc_item_delete
 */
gboolean cd_arc_item_delete()
{
  int i, j;

  if(cd_arc.count <= 0)
    return FALSE;

  j = cd_arc.count-1;

  if(cd_arc.item[j].select){
    cd_arc_item_escape(j);
    cd_arc_item_init(j);
    cd_arc.count--;
    cd_arc.selecting = CD_SELECTING_CLEAR;
    cd_arc.selected = CD_SELECTING_CLEAR;
    return TRUE;
  }

  for(i=cd_arc.count-2; i>=0; i--){
    if(cd_arc.item[i].select){
      cd_arc_item_escape(i);
      cd_arc_item_copy(i, j);
      cd_arc_item_init(j);
      cd_arc.count--;
      cd_arc.selecting = CD_SELECTING_CLEAR;
      cd_arc.selected = CD_SELECTING_CLEAR;
      return TRUE;
    }
  }

  return FALSE;
}
    
/*
 * cd_arc_item_escape
 */
void cd_arc_item_escape(int i)
{
  cd_arc_item_last_deleted.id               = cd_arc.item[i].id;

  sprintf(cd_arc_item_last_deleted.name, "%s", cd_arc.item[i].name);
  
  cd_arc_item_last_deleted.name_rect.x      = cd_arc.item[i].name_rect.x;
  cd_arc_item_last_deleted.name_rect.y      = cd_arc.item[i].name_rect.y;
  cd_arc_item_last_deleted.name_rect.width  = cd_arc.item[i].name_rect.width;
  cd_arc_item_last_deleted.name_rect.height = cd_arc.item[i].name_rect.height;

  cd_arc_item_term_copy(&cd_arc_item_last_deleted.start,
			&cd_arc.item[i].start);
  cd_arc_item_term_copy(&cd_arc_item_last_deleted.end, &cd_arc.item[i].end);

  cd_arc_item_last_deleted.select           = cd_arc.item[i].select;
}

/*
 * cd_arc_init
 */
void cd_arc_init()
{
  int i;

  cd_arc.seq       = 0;
  cd_arc.count     = 0;
  cd_arc.selecting = CD_SELECTING_CLEAR;
  cd_arc.selected  = CD_SELECTING_CLEAR;
 
  for(i=0; i<CD_ARC_ITEM_MAX; i++)
    cd_arc_item_init(i);
}

/*
 * cd_arc_term_load
 */
void cd_arc_term_load(FILE *fp, ArcTerm *at_ptr)
{
  char line[BUFSIZ];
  char str[6];

  fgets(line, BUFSIZ, fp);
  sscanf(line, "%s %d %d %f %d %d %d %d %d %d %d %d",
	 str,
	 &at_ptr->point.x,
	 &at_ptr->point.y,
	 &at_ptr->theta,
	 &at_ptr->face,
	 &at_ptr->node_id,
	 &at_ptr->type,
	 &at_ptr->mult,
	 &at_ptr->mult_rect.x,
	 &at_ptr->mult_rect.y,
	 &at_ptr->mult_rect.width,
	 &at_ptr->mult_rect.height);
}

/*
 * cd_arc_item_load
 */
void cd_arc_item_load(FILE *fp, int i)
{
  char line[BUFSIZ];
  char str[5];

  fgets(line, BUFSIZ, fp);
  sscanf(line, "%s %d %s %d %d %d %d",
	 str,
	 &cd_arc.item[i].id,
	 cd_arc.item[i].name,
	 &cd_arc.item[i].name_rect.x,
	 &cd_arc.item[i].name_rect.y,
	 &cd_arc.item[i].name_rect.width,
	 &cd_arc.item[i].name_rect.height);

  if(!strcmp(cd_arc.item[i].name, "None"))
    sprintf(cd_arc.item[i].name, "");

  cd_arc_term_load(fp, &cd_arc.item[i].start);
  cd_arc_term_load(fp, &cd_arc.item[i].end);

  cd_node_item_calc_arc_points(cd_arc.item[i].start.node_id,
			       cd_arc.item[i].end.node_id,
			       &cd_arc.item[i]);
}

/*
 * cd_arc_load
 */
void cd_arc_load(FILE *fp)
{
  char line[BUFSIZ];
  int count, seq;
  char str[5];

  int i;

  fgets(line, BUFSIZ, fp);
  sscanf(line, "%s %d %d", str, &count, &seq);

  cd_arc.count = count;
  cd_arc.seq = seq;

  for(i=0; i<cd_arc.count; i++)
    cd_arc_item_load(fp, i);
}

/*
 * cd_arc_term_save
 */
void cd_arc_term_save(FILE *fp, ArcTerm *at_ptr)
{
  fprintf(fp, " term: %d %d %f %d %d %d %d %d %d %d %d\n",
	  at_ptr->point.x,
	  at_ptr->point.y,
	  at_ptr->theta,
	  at_ptr->face,
	  at_ptr->node_id,
	  at_ptr->type,
	  at_ptr->mult,
	  at_ptr->mult_rect.x,
	  at_ptr->mult_rect.y,
	  at_ptr->mult_rect.width,
	  at_ptr->mult_rect.height
	  );
}

/*
 * cd_arc_save
 */
void cd_arc_save(FILE *fp)
{
  int i;

  fprintf(fp, "ARC! %d %d\n", cd_arc.count, cd_arc.seq);

  for(i=0; i<cd_arc.count; i++){
    fprintf(fp, "arc: %d", cd_arc.item[i].id);

    if(strlen(cd_arc.item[i].name) > 0)
      fprintf(fp, " %s", cd_arc.item[i].name);
    else
      fprintf(fp, " None");

    fprintf(fp, " %d %d %d %d",
	    cd_arc.item[i].name_rect.x,
	    cd_arc.item[i].name_rect.y,
	    cd_arc.item[i].name_rect.width,
	    cd_arc.item[i].name_rect.height);
    fprintf(fp, "\n");

    cd_arc_term_save(fp, &cd_arc.item[i].start);
    cd_arc_term_save(fp, &cd_arc.item[i].end);
  }
}

/*
 * cd_arc_item_draw_last
 */
void cd_arc_item_draw_last(GdkPixmap *pixmap, GdkGC *gc)
{
  int i = cd_arc.count-1;

  cd_arc_item_draw(pixmap, gc, i);
}

/*
 * cd_arc_item_draw
 */
void cd_arc_item_draw(GdkPixmap *pixmap, GdkGC *gc, int i)
{
  cd_arc_item_draw2(pixmap, gc, &cd_arc.item[i]);
}

/*
 * cd_arc_item_draw_mult
 */
void cd_arc_item_draw_mult(GdkPixmap *pixmap, GdkGC *gc, ArcTerm *term)
{
  gchar str[5];

  if(term->mult == m_none)
    return;

  switch(term->mult){
  case m0:
    sprintf(str, CD_MULT_ZERO);
    break;
  case m0_1:
    sprintf(str, CD_MULT_ZERO_ONE);
    break;
  case m1:
    sprintf(str, CD_MULT_ONE);
    break;
  case m0_n:
    sprintf(str, CD_MULT_ZERO_N);
    break;
  case m1_n:
    sprintf(str, CD_MULT_ONE_N);
    break;
  }

  gdk_draw_rectangle(pixmap, cd_white_gc, TRUE,
		     term->mult_rect.x,
		     term->mult_rect.y - term->mult_rect.height,
		     term->mult_rect.width,
		     term->mult_rect.height);

  gdk_draw_string(pixmap, cd_font, gc,
		  term->mult_rect.x,
		  term->mult_rect.y,
		  str);
}

/*
 * cd_arc_item_draw_term
 */
void cd_arc_item_draw_term(GdkPixmap *pixmap, GdkGC *gc, ArcTerm *term)
{
  switch(term->type){
  case association2:
    cd_arc_item_draw_arrow(pixmap, gc,
			   term->point.x, term->point.y,
			   term->theta, term->face);
    break;
  case aggregation:
    cd_arc_item_draw_diamond(pixmap, gc,
			     term->point.x, term->point.y,
			     term->theta, term->face, FALSE);
    break;
  case aggregation2:
    cd_arc_item_draw_diamond(pixmap, gc,
			     term->point.x, term->point.y,
			     term->theta, term->face, TRUE);
    break;
  case generalization:
    cd_arc_item_draw_triangle(pixmap, gc,
			      term->point.x, term->point.y,
			      term->theta, term->face);
    break;
  }

  cd_arc_item_draw_mult(pixmap, gc, term);
}

/*
 * cd_arc_item_draw_string
 */
void cd_arc_item_draw_string(GdkPixmap *pixmap, GdkGC *gc, ArcItem *a_ptr)
{
  if(strlen(a_ptr->name) > 0) {

    gdk_draw_rectangle(pixmap, cd_white_gc, TRUE,
		       a_ptr->name_rect.x,
		       a_ptr->name_rect.y - a_ptr->name_rect.height,
		       a_ptr->name_rect.width,
		       a_ptr->name_rect.height);

    gdk_draw_string(pixmap, cd_font, gc,
		    a_ptr->name_rect.x,
		    a_ptr->name_rect.y,
		    a_ptr->name);
  }
}

/*
 * cd_arc_item_draw2
 */
void cd_arc_item_draw2(GdkPixmap *pixmap, GdkGC *gc, ArcItem *a_ptr)
{
  gdk_draw_line(pixmap, gc,
		a_ptr->start.point.x, a_ptr->start.point.y,
		a_ptr->end.point.x, a_ptr->end.point.y);

  cd_arc_item_draw_string(pixmap, gc, a_ptr);

  cd_arc_item_draw_term(pixmap, gc, &a_ptr->start);
  cd_arc_item_draw_term(pixmap, gc, &a_ptr->end);
}

/*
 * cd_arc_item_draw_selecting
 */
void cd_arc_item_draw_selecting(GdkPixmap *pixmap, GdkGC *gc)
{
  int i = cd_arc.selecting;

  if(i < cd_arc.count)
    cd_arc_item_draw(pixmap, gc, i);
}

/*
 * cd_arc_item_draw_selected
 */
void cd_arc_item_draw_selected(GdkPixmap *pixmap, GdkGC *gc)
{
  int i = cd_arc.selected;

  if(i < cd_arc.count)
    cd_arc_item_draw(pixmap, gc, i);
}

/*
 * cd_arc_item_erase_deleted
 */
void cd_arc_item_erase_deleted(GdkPixmap *pixmap, GdkGC *gc)
{
  cd_arc_item_draw2(pixmap, gc, &cd_arc_item_last_deleted);
}

/*
 * cd_arc_item_start_move
 */
void cd_arc_item_start_move(int dx, int dy, int i)
{
  cd_arc.item[i].start.point.x += dx;
  cd_arc.item[i].start.point.y += dy;
  cd_arc_item_calc_name_rect(i);

  cd_node_item_calc_arc_points(cd_arc.item[i].start.node_id,
			       cd_arc.item[i].end.node_id,
			       &cd_arc.item[i]);
}

/*
 * cd_arc_item_end_move
 */
void cd_arc_item_end_move(int dx, int dy, int i)
{
  cd_arc.item[i].end.point.x += dx;
  cd_arc.item[i].end.point.y += dy;
  cd_arc_item_calc_name_rect(i);

  cd_node_item_calc_arc_points(cd_arc.item[i].start.node_id,
			       cd_arc.item[i].end.node_id,
			       &cd_arc.item[i]);
}

/*
 * cd_arc_item_selection
 */
int cd_arc_item_selection(int x, int y)
{
  int i;

  for(i=0; i<cd_arc.count; i++){
    if(cd_util_select_line
       (&cd_arc.item[i].start.point, &cd_arc.item[i].end.point, x, y)){
      cd_arc.item[i].select = TRUE;
      cd_arc.selecting = i;
      return i;
    }
  }

  return CD_SELECTING_CLEAR;
}

/*
 * void cd_arc_item_selection_clear_all
 */
void cd_arc_item_selection_clear_all()
{
  int i;

  for(i=0; i<cd_arc.count; i++)
    cd_arc.item[i].select = FALSE;
}

/*
 * cd_arc_selecting_end
 */
void cd_arc_selecting_end()
{
  cd_arc.selected = cd_arc.selecting;
  cd_arc.selecting = CD_SELECTING_CLEAR;
}

/*
 * cd_arc_selection_clear
 */
void cd_arc_selection_clear()
{
  cd_arc.selected = CD_SELECTING_CLEAR;
  cd_arc.selecting = CD_SELECTING_CLEAR;
}

/*
 * cd_arc_item_draw_arrow
 */
void cd_arc_item_draw_arrow(GdkPixmap *pixmap, GdkGC *gc,
			    int x, int y, double theta, CdNodeFace face)
{
  Matrix mx1 = { {0.0,0.0}, {0.0,0.0}, {0.0,0.0} };
  Matrix mx2 = { {0.0,0.0}, {0.0,0.0}, {0.0,0.0} };

  double x1, y1, x2, y2;
  double theta_base;

  if(face == face_first || face == face_third){
    theta_base = CD_FT_BASE;

    mx1[0][0] = x;
    mx1[0][1] = y;
    mx1[1][0] = x-12;
    mx1[1][1] = y+4;

    mx2[0][0] = x;
    mx2[0][1] = y;
    mx2[1][0] = x-12;
    mx2[1][1] = y-4;

  }else if(face == face_second || face == face_fourth){
    theta_base = CD_SF_BASE;

    mx1[0][0] = x;
    mx1[0][1] = y;
    mx1[1][0] = x+4;
    mx1[1][1] = y-12;

    mx2[0][0] = x;
    mx2[0][1] = y;
    mx2[1][0] = x-4;
    mx2[1][1] = y-12;
  }

  cd_util_rotate(mx1, theta_base - theta);
  cd_util_rotate(mx2, theta_base - theta);

  x1 = mx1[2][0];
  y1 = mx1[2][1];

  x2 = mx2[2][0];
  y2 = mx2[2][1];

  gdk_draw_line(pixmap, gc, x1, y1, x, y);
  gdk_draw_line(pixmap, gc, x2, y2, x, y);
}

/*
 * cd_arc_item_draw_triagnle
 */
void cd_arc_item_draw_triangle(GdkPixmap *pixmap, GdkGC *gc,
			       int x, int y, double theta, CdNodeFace face)
{
  Matrix mx1 = { {0.0,0.0}, {0.0,0.0}, {0.0,0.0} };
  Matrix mx2 = { {0.0,0.0}, {0.0,0.0}, {0.0,0.0} };

  double theta_base;

  GdkPoint poli[3];

  if(face == face_first || face == face_third){
    theta_base = CD_FT_BASE;

    mx1[0][0] = x;
    mx1[0][1] = y;
    mx1[1][0] = x-10;
    mx1[1][1] = y+6;

    mx2[0][0] = x;
    mx2[0][1] = y;
    mx2[1][0] = x-10;
    mx2[1][1] = y-6;

  }else if(face == face_second || face == face_fourth){
    theta_base = CD_SF_BASE;

    mx1[0][0] = x;
    mx1[0][1] = y;
    mx1[1][0] = x+6;
    mx1[1][1] = y-10;

    mx2[0][0] = x;
    mx2[0][1] = y;
    mx2[1][0] = x-6;
    mx2[1][1] = y-10;
  }

  cd_util_rotate(mx1, theta_base - theta);
  cd_util_rotate(mx2, theta_base - theta);

  poli[0].x = x;
  poli[0].y = y;
  poli[1].x = mx1[2][0];
  poli[1].y = mx1[2][1];
  poli[2].x = mx2[2][0];
  poli[2].y = mx2[2][1];

  gdk_draw_polygon(pixmap, cd_white_gc, TRUE, poli, 3);
  gdk_draw_polygon(pixmap, gc, FALSE, poli, 3);
}

/*
 * cd_arc_item_draw_diamond
 */
void cd_arc_item_draw_diamond(GdkPixmap *pixmap, GdkGC *gc,
			       int x, int y, double theta,
			      CdNodeFace face, gboolean flg)
{
  Matrix mx1 = { {0.0,0.0}, {0.0,0.0}, {0.0,0.0} };
  Matrix mx2 = { {0.0,0.0}, {0.0,0.0}, {0.0,0.0} };
  Matrix mx3 = { {0.0,0.0}, {0.0,0.0}, {0.0,0.0} };

  double theta_base;

  GdkPoint poli[4];

  if(face == face_first || face == face_third){
    theta_base = CD_FT_BASE;

    mx1[0][0] = x;
    mx1[0][1] = y;
    mx1[1][0] = x-10;
    mx1[1][1] = y+4;

    mx2[0][0] = x;
    mx2[0][1] = y;
    mx2[1][0] = x-10;
    mx2[1][1] = y-4;

    mx3[0][0] = x;
    mx3[0][1] = y;
    mx3[1][0] = x-20;
    mx3[1][1] = y;

  }else if(face == face_second || face == face_fourth){
    theta_base = CD_SF_BASE;

    mx1[0][0] = x;
    mx1[0][1] = y;
    mx1[1][0] = x+4;
    mx1[1][1] = y-10;

    mx2[0][0] = x;
    mx2[0][1] = y;
    mx2[1][0] = x-4;
    mx2[1][1] = y-10;

    mx3[0][0] = x;
    mx3[0][1] = y;
    mx3[1][0] = x;
    mx3[1][1] = y-20;
  }

  cd_util_rotate(mx1, theta_base - theta);
  cd_util_rotate(mx2, theta_base - theta);
  cd_util_rotate(mx3, theta_base - theta);

  poli[0].x = x;
  poli[0].y = y;
  poli[1].x = mx1[2][0];
  poli[1].y = mx1[2][1];
  poli[2].x = mx3[2][0];
  poli[2].y = mx3[2][1];
  poli[3].x = mx2[2][0];
  poli[3].y = mx2[2][1];

  gdk_draw_polygon(pixmap, cd_white_gc, TRUE, poli, 4);
  gdk_draw_polygon(pixmap, gc, flg, poli, 4);
}
