/* cdedit3 -- class diagram creation/manipulation program
 * Copyright (C) 2001 Touge Kamisimo
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "cd_define.h"
#include "cd_canvas.h"
#include "cd_menu.h"

#include "pixmaps/arrow.xpm"
#include "pixmaps/umlclass.xpm"
#include "pixmaps/association.xpm"
#include "pixmaps/association2.xpm"
#include "pixmaps/aggregation.xpm"
#include "pixmaps/aggregation2.xpm"
#include "pixmaps/generalization.xpm"
#include "pixmaps/pgclass.xpm"

CdToolbarItem cd_toolbar_item[] =
{
  { "Select",            arrow_xpm,          cd_canvas_mode_select },
  { "Uml Class",         umlclass_xpm,       cd_canvas_mode_umlclass },
  { "Association",       association_xpm,    cd_canvas_mode_association },
  { "Association2",      association2_xpm,   cd_canvas_mode_association2 },
  { "Aggregation",       aggregation_xpm,    cd_canvas_mode_aggregation },
  { "Aggregation2",      aggregation2_xpm,   cd_canvas_mode_aggregation2 },
  { "Generalization",    generalization_xpm, cd_canvas_mode_generalization },
  { "PostgresSQL Class", pgclass_xpm,        cd_canvas_mode_pgclass },
  { NULL,                NULL,               NULL }
};

GtkWidget *cd_drawing_area;
int cd_drawing_area_width;
int cd_drawing_area_height;

GdkPixmap *cd_pixmap;
GdkGC *cd_black_gc;
GdkGC *cd_white_gc;
GdkGC *cd_select_gc;
GdkGC *cd_blue_gc;
GdkFont *cd_font;
CdToolType cd_tool_type;

int lastX, lastY;

/*
 * cd_canvas_configure
 */
static gint cd_canvas_configure(GtkWidget *widget, GdkEventConfigure *event)
{
  if (cd_pixmap)
    gdk_pixmap_unref(cd_pixmap);

  cd_pixmap = gdk_pixmap_new(widget->window,
			     widget->allocation.width,
			     widget->allocation.height,
			     -1);

  gdk_draw_rectangle (cd_pixmap,
		      widget->style->white_gc,
		      TRUE,
		      0, 0,
		      widget->allocation.width,
		      widget->allocation.height);

  cd_control_draw_all();
  
  return TRUE;
}

/*
 * cd_canvas_expose
 */
static gint cd_canvas_expose(GtkWidget *widget, GdkEventExpose *event)
{
  cd_control_draw_all(); // ???

  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
		  cd_pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);

  return TRUE;
}

/*
 * cd_canvas_motion_nofity
 */
gint cd_canvas_motion_notify(GtkWidget *widget, GdkEventMotion *event)
{
  GdkModifierType state;
  int x, y;

  if(!cd_control_selecting()){
    return;
  }

  x = event->x;
  y = event->y;

  if (event->is_hint)
    gdk_window_get_pointer(event->window, &x, &y, &state);
  else {
    x = event->x;
    y = event->y;
    state = event->state;
  }

  switch(cd_tool_type){
  case cdselect:
    cd_control_motion(x, y);
    break;
  }

  cd_canvas_update_all();

  return TRUE;
}

/*
 * cd_canvas_button_press
 */
gint cd_canvas_button_press(GtkWidget *widget, GdkEventButton *event)
{
  int x, y, event_type;

  cd_control_selecting_clear();

  lastX = x = event->x;
  lastY = y = event->y;
  event_type = event->type;

  if(event->button == 1){
    switch(cd_tool_type){
    case cdselect:
      cd_control_selection(x, y, event_type);
      break;
    case umlclass:
    case pgclass:
      cd_control_node_add(x, y);
      break;
    case association:
    case association2:
    case aggregation:
    case aggregation2:
    case generalization:
      cd_control_arc_add_start(x, y, cd_tool_type);
      break;
    }
  }

  cd_canvas_update_all();
}

/*
 * cd_canvas_button_release
 */
gint cd_canvas_button_release(GtkWidget *widget, GdkEventButton *event)
{
  int x, y;

  x = event->x;
  y = event->y;

  if(event->button == 1){
    switch(cd_tool_type){
    case association:
    case association2:
    case aggregation:
    case aggregation2:
    case generalization:
      cd_control_arc_add_end(x, y, cd_tool_type);
      break;
    }
  }

  cd_control_selection_end();

  cd_canvas_update_all();
}

/*
 * cd_canvas_size_allocate
 */
void cd_canvas_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
  cd_control_draw_all();
  cd_canvas_update_all();
}

/*
 * cd_canvas_set_events
 */
void cd_canvas_set_events(GtkWidget *drawing_area)
{
  gtk_signal_connect(GTK_OBJECT(drawing_area), "configure_event",
		     (GtkSignalFunc)cd_canvas_configure, NULL);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "expose_event",
		     (GtkSignalFunc)cd_canvas_expose, NULL);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "motion_notify_event",
		     (GtkSignalFunc)cd_canvas_motion_notify, NULL);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "button_press_event",
		     (GtkSignalFunc)cd_canvas_button_press, NULL);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "button_release_event",
		     (GtkSignalFunc)cd_canvas_button_release, NULL);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "size_allocate",
		     (GtkSignalFunc)cd_canvas_size_allocate, NULL);

  gtk_widget_set_events(drawing_area, GDK_EXPOSURE_MASK
			| GDK_LEAVE_NOTIFY_MASK
			| GDK_BUTTON_PRESS_MASK
			| GDK_BUTTON_RELEASE_MASK
			| GDK_POINTER_MOTION_MASK
			| GDK_POINTER_MOTION_HINT_MASK
			| GDK_KEY_PRESS_MASK
			);
}  

/*
 * cd_scrolled_window_new
 */
void cd_scrolled_window_new(GtkWidget *drawing_area, GtkWidget *vbox)
{
  GtkWidget *scrolled_window;

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolled_window);
  gtk_container_set_border_width (GTK_CONTAINER (scrolled_window), 10);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

  gtk_scrolled_window_add_with_viewport
    (GTK_SCROLLED_WINDOW (scrolled_window), drawing_area);

  gtk_box_pack_start (GTK_BOX(vbox), scrolled_window, TRUE, TRUE, 0);
};

/*
 * cd_canvas_gc_init
 */
void cd_canvas_gc_init(GtkWidget *window)
{
  GdkColor color;

  cd_font = gdk_fontset_load(CD_FONT);

  cd_white_gc = window->style->white_gc;
  cd_black_gc = window->style->black_gc;

  cd_select_gc = gdk_gc_new(window->window);
  gdk_gc_set_line_attributes
    (cd_select_gc, 1, GDK_LINE_ON_OFF_DASH, GDK_CAP_ROUND, GDK_JOIN_ROUND);

  color.red   = 0x0000;
  color.green = 0x0000;
  color.blue  = 0xffff;
  gdk_color_alloc(gdk_colormap_get_system(),&color);
  cd_blue_gc = gdk_gc_new(window->window);
  gdk_gc_set_foreground(cd_blue_gc, &color);
}  

/*
 * cd_canvas_radio_buttons_new (local)
 */
void cd_canvas_radio_buttons_new(GtkWidget *window, GtkWidget *toolbar)
{
  GSList *group = NULL;
  GtkWidget *button;
  GtkWidget *gtk_pixmap;
  GdkPixmap *gdk_pixmap;
  GdkBitmap *gtk_mask;

  int i = 0;

  while(cd_toolbar_item[i].tooltips != NULL){
    if(strcmp(cd_toolbar_item[i].tooltips, "_SPACE_") == 0){
      gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
      i++;
      continue;
    }

    gdk_pixmap =
      gdk_pixmap_create_from_xpm_d(window->window,
				   &gtk_mask,
				   &window->style->bg[GTK_STATE_NORMAL],
				   cd_toolbar_item[i].xpm);

    gtk_pixmap = gtk_pixmap_new(gdk_pixmap, gtk_mask);

    button = gtk_radio_button_new(group);
    group = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
    gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button), FALSE);

    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(cd_toolbar_item[i].func),
		       NULL);

    gtk_container_add(GTK_CONTAINER(button), gtk_pixmap);

    gtk_toolbar_append_widget(GTK_TOOLBAR(toolbar),
			      GTK_WIDGET(button),
			      cd_toolbar_item[i].tooltips,
			      NULL);

    gtk_widget_show(GTK_WIDGET(button));
    gtk_widget_show(GTK_WIDGET(gtk_pixmap));

    i++;
  }
}

/*
 * cd_canvas_toolbar_new (global)
 */
void cd_canvas_toolbar_new(GtkWidget *window, GtkWidget *vbox)
{
  GtkWidget *toolbar;
  GtkWidget *handle;

  handle = gtk_handle_box_new();

  toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);

  gtk_widget_show(GTK_WIDGET(toolbar));

  cd_canvas_radio_buttons_new(window, toolbar);

  gtk_container_add(GTK_CONTAINER(handle), GTK_WIDGET(toolbar));
  gtk_box_pack_start(GTK_BOX(vbox), handle, FALSE, FALSE, 0);
  gtk_widget_show(handle);
}

/*
 * cd_canvas_new
 */
void cd_canvas_new(GtkWidget *window, GtkWidget *vbox)
{
  cd_canvas_gc_init(window);

  cd_drawing_area = gtk_drawing_area_new();

  cd_canvas_set_size(CD_CANVAS_WIDTH_DEFAULT, CD_CANVAS_HEIGHT_DEFAULT);

  cd_canvas_set_events(cd_drawing_area);

  gtk_widget_show(cd_drawing_area);

  cd_scrolled_window_new(cd_drawing_area, vbox);
}

/*
 * cd_canvas_update_all
 */
void cd_canvas_update_all()
{
  GdkRectangle update_rect;

  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = cd_drawing_area_width;
  update_rect.height = cd_drawing_area_height;

  gtk_widget_draw(cd_drawing_area, &update_rect);
}

/*
 * cd_canvas_white
 */
void cd_canvas_white()
{
  GdkRectangle update_rect;

  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = cd_drawing_area_width;
  update_rect.height = cd_drawing_area_height;

  gdk_draw_rectangle(cd_pixmap, cd_white_gc, TRUE,
		     update_rect.x, update_rect.y,
		     update_rect.width, update_rect.height);
}

/*
 * cd_canvas_mode_select
 */
void cd_canvas_mode_select()
{
  cd_tool_type = cdselect;
}

/*
 * cd_canvas_mode_umlclass
 */
void cd_canvas_mode_umlclass()
{
  if(cd_tool_type == cdselect){
    cd_control_mode_change_from_select();
    cd_canvas_update_all();
  }

  cd_tool_type = umlclass;
}

/*
 * cd_canvas_mode_pgclass
 */
void cd_canvas_mode_pgclass()
{
  if(cd_tool_type == cdselect){
    cd_control_mode_change_from_select();
    cd_canvas_update_all();
  }

  cd_tool_type = pgclass;
}

/*
 * cd_canvas_mode_association
 */
void cd_canvas_mode_association()
{
  if(cd_tool_type == cdselect){
    cd_control_mode_change_from_select();
    cd_canvas_update_all();
  }

  cd_tool_type = association;
}

/*
 * cd_canvas_mode_association2
 */
void cd_canvas_mode_association2()
{
  if(cd_tool_type == cdselect){
    cd_control_mode_change_from_select();
    cd_canvas_update_all();
  }

  cd_tool_type = association2;
}

/*
 * cd_canvas_mode_aggregation
 */
void cd_canvas_mode_aggregation()
{
  if(cd_tool_type == cdselect){
    cd_control_mode_change_from_select();
    cd_canvas_update_all();
  }

  cd_tool_type = aggregation;
}

/*
 * cd_canvas_mode_aggregation2
 */
void cd_canvas_mode_aggregation2()
{
  if(cd_tool_type == cdselect){
    cd_control_mode_change_from_select();
    cd_canvas_update_all();
  }

  cd_tool_type = aggregation2;
}

/*
 * cd_canvas_mode_generalization
 */
void cd_canvas_mode_generalization()
{
  if(cd_tool_type == cdselect){
    cd_control_mode_change_from_select();
    cd_canvas_update_all();
  }

  cd_tool_type = generalization;
}

CdToolType cd_canvas_mode_get_tool_type()
{
  return cd_tool_type;
}

/*
 * cd_canvas_set_size
 */
void cd_canvas_set_size(int width, int height)
{
  cd_drawing_area_width = width;
  cd_drawing_area_height = height;

  gtk_drawing_area_size(GTK_DRAWING_AREA(cd_drawing_area),
			cd_drawing_area_width,
			cd_drawing_area_height);
}

/*
 * cd_canvas_get_size
 */
void cd_canvas_get_size(int *width, int *height)
{
  *width = cd_drawing_area_width;
  *height = cd_drawing_area_height;
}

/*
 * cd_canvas_size_update
 */
void cd_canvas_size_update()
{
  gtk_drawing_area_size(GTK_DRAWING_AREA(cd_drawing_area),
			cd_drawing_area_width,
			cd_drawing_area_height);
}

/*
 * cd_canvas_save
 */
void cd_canvas_save(FILE *fp)
{
  fprintf(fp, "Canvas!: %d %d\n",
	  cd_drawing_area_width, cd_drawing_area_height);
}

/*
 * cd_cavas_load
 */
void cd_canvas_load(FILE *fp)
{
  char line[BUFSIZ];
  int width, height;
  char str[9];
  
  fgets(line, BUFSIZ, fp);
  sscanf(line, "%s %d %d", str, &width, &height);

  cd_drawing_area_width = width;
  cd_drawing_area_height = height;
}
