/*
 * @file  protomod_cinsert.c
 * @brief protocol module of HTTP Cookie.
 * @brief this module provide session persistence by Original Cookie.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#define __STDC_LIMIT_MACROS
#include <stdlib.h>
#include <time.h>
#include <getopt.h>
#include <arpa/inet.h>
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_dest.h"
#include "l7vs_module.h"
#include "module_http.h"

#define SERVICE_ARG_MAXSIZE    (512)
#define CINSERT_SERVICE_NUMBER (128)
#define COOKIE_NAME_MAXSIZE    (128)
#define X_FORWARDED_FOR_LENGTH (48)
#define DEST_ENCODE_LENGTH     (16)

struct l7vs_cinsert_service {
	handle_t service_handle;
	char cookie_name[COOKIE_NAME_MAXSIZE];
	int cookie_expire;
	int forwarded_for;
	int reschedule;
};

struct  l7vs_cinsert_service_arg {
	char cookie_name[COOKIE_NAME_MAXSIZE];
	int cookie_expire;
	int forwarded_for;
	int reschedule;
};

static void  fini(void);
static int   create(void*, handle_t);
static void* create_sa(struct l7vs_service_arg*);
static int   compare(handle_t, handle_t);
static int   match_cldata(struct l7vs_service*, struct l7vs_conn*,
                        char*, size_t*, struct l7vs_dest**, int*);
static int   analyze_rsdata(struct l7vs_service*, struct l7vs_conn*,
                        char*, size_t*);
static int   destroy(handle_t);
static void  destroy_sa(void**);
static int   service_arg(struct l7vs_service_arg_multi*, handle_t);
static int   parse(void*, int, char**);

static struct l7vs_cinsert_service* l7vs_protomod_cinsert_search_service(handle_t);
static struct l7vs_cinsert_service* l7vs_protomod_cinsert_create_service();
static struct l7vs_cinsert_service* l7vs_protomod_cinsert_create_temp_service();

static time_t l7vs_protomod_cinsert_timeout(int);
//static int    l7vs_protomod_insert_field(char*, int, char*, int);

static void l7vs_cinsert_service_c_str(char*, struct l7vs_cinsert_service*);
static void l7vs_cinsert_service_arg_c_str(char*, struct l7vs_cinsert_service_arg*);

struct l7vs_cinsert_service* cinsert_service_list[CINSERT_SERVICE_NUMBER];

static struct l7vs_protomod cinsert_protomod = {
	NULL,           /* handle */
	"cinsert",      /* modname */
	0,              /* refcnt */
	create,         /* create function */
	compare,        /* compare function */
	match_cldata,   /* match_cldata function */
	analyze_rsdata, /* analyze_rsdata function */
	destroy,        /* destroy function */
	fini,           /* fini function */
	create_sa,      /* create_sa function */
	service_arg,    /* service_arg function */
	parse,          /* parse function */
	destroy_sa,     /* destroy_sa function */
	NULL,           /* initialize function */
	NULL,           /* finalize function */
	NULL,           /* get_log_level function */
	NULL,           /* put_log_debug function */
	NULL,           /* put_log_info function */
	NULL,           /* put_log_warn function */
	NULL,           /* put_log_error function */
	NULL            /* put_log_fatal function */
};

/*!
 * Protocol module initialize function. This function run when dlopen and dlsym at first time.
 * @param[in] handle dlopen's handle
 * @return l7vs_protomod struct
 */
extern "C" struct l7vs_protomod *
init(void *handle)
{
	struct l7vs_protomod* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,17,
	            "in_function: struct l7vs_protomod* init(void* handle): handle=%p", handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (handle == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,14, "Arg(handle) is NULL pointer.");
		goto init_out;
	}

	/* initialize cinsert service list */
	memset(cinsert_service_list, 0, sizeof(struct l7vs_cinsert_service *) * CINSERT_SERVICE_NUMBER);
	/* set dlopen's handle */
	cinsert_protomod.handle = handle;

	return_value = &cinsert_protomod;

init_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char protomod_str[DEBUG_STR_LEN];
		l7vs_protomod_c_str(protomod_str, &cinsert_protomod);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,18,
		    "out_function: struct l7vs_protomod* init(void* handle): return=&(%s)", protomod_str);
	}
	/*------ DEBUG LOG END ------*/
	return return_value;
}

/*!
 * Protocol module finalize function. free all cinsert service list just in case.
 * @param   void
 * @return  void
 */
static void
fini(void)
{
	/* cinsert service list counter */
	int service_number = 0;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,19, "in_function: void fini(void)");
	}
	/*------ DEBUG LOG END ------*/

	/* check all cinsert service list */
	for (service_number = 0; service_number < CINSERT_SERVICE_NUMBER; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (cinsert_service_list[service_number] != NULL) {

			/*-------- DEBUG LOG --------*/
			if (cinsert_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,14, "free: %p",
				    cinsert_service_list[service_number]);
			}
			/*------ DEBUG LOG END ------*/

			/* free and points NULL */
			free(cinsert_service_list[service_number]);
			cinsert_service_list[service_number] = NULL;
		}
	}
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,20, "out_function: void fini(void)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Create cinsert service struct.
 * @param[in] cinsert_arg    cinsert service argument struct
 * @param[in] service_handle a unique service ID
 * @retval 0  successfully create cinsert service.
 * @retval -1 some errors occur.
 */
static int
create(void *cinsert_arg, handle_t service_handle)
{
	struct l7vs_cinsert_service *cinsert_service;
	struct l7vs_cinsert_service_arg *cinsert_service_arg;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cinsert_arg_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_arg_c_str(cinsert_arg_str, (struct l7vs_cinsert_service_arg*) cinsert_arg);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,21,
		    "in_function: int create(void* cinsert_arg, handle_t service_handle):cinsert_arg=&(%s), "
		    "service_handle=%u", cinsert_arg_str, service_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (cinsert_arg == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,15, "Arg(cinsert_arg) is NULL pointer.");
		return_value = -1;
		goto create_out;
	}

	if (service_handle != TEMP_SERVICEHANDLE) {
		/* search empty cinsert service list and create cinsert service */
		cinsert_service = l7vs_protomod_cinsert_create_service();
	} else {
		/* create temporary cinsert service */
		cinsert_service = l7vs_protomod_cinsert_create_temp_service();
	}

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cinsert_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_c_str(cinsert_str, cinsert_service);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,22, "pointer assign: cinsert_service=&(%s)",
		    cinsert_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cinsert_service == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,16, "Could not make cinsert service.");
		return_value = -1;
		goto create_out;
	}

	cinsert_service_arg = (struct l7vs_cinsert_service_arg *) cinsert_arg;

	/* set service handle, option value */
	cinsert_service->service_handle = service_handle;
	strncpy(cinsert_service->cookie_name, cinsert_service_arg->cookie_name, COOKIE_NAME_MAXSIZE);
	cinsert_service->cookie_expire = cinsert_service_arg->cookie_expire;
	cinsert_service->forwarded_for = cinsert_service_arg->forwarded_for;
	cinsert_service->reschedule = cinsert_service_arg->reschedule;

create_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,23,
		    "out_function: int create(void* cinsert_arg, handle_t service_handle):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create cinsert service argument struct.
 * @param[out] srv_arg service argument struct
 * @return cinsert service argument struct
 */
static void *
create_sa(struct l7vs_service_arg *srv_arg)
{
	struct l7vs_cinsert_service_arg *cinsert_service_arg;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char service_arg_str[DEBUG_STR_LEN];
		l7vs_service_arg_c_str(service_arg_str, srv_arg);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,24,
		    "in_function: void* create_sa(struct l7vs_service_arg* srv_arg):srv_arg=&(%s)",
		    service_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv_arg == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,17, "Arg(srv_arg) is NULL pointer.");
		cinsert_service_arg = NULL;
		goto create_sa_out;
	}

	/* create cinsert service argument struct */
	cinsert_service_arg = (struct l7vs_cinsert_service_arg *) calloc(1, sizeof(struct l7vs_cinsert_service_arg));

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,15, "calloc: addr=%p, size=%ld",
		    cinsert_service_arg, (unsigned long int) sizeof(struct l7vs_cinsert_service_arg));
	}
	/*------ DEBUG LOG END ------*/

	if (cinsert_service_arg == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,16, "Could not allocate memory.");
		goto create_sa_out;
	}

	/* set cinsert service argument size and protomod name "cinsert" */
	srv_arg->len = sizeof(struct l7vs_cinsert_service_arg);
	strcpy(srv_arg->protomod, cinsert_protomod.modname);

create_sa_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cinsert_service_arg_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_arg_c_str(cinsert_service_arg_str, cinsert_service_arg);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,25,
		    "out_function: void* create_sa(struct l7vs_service_arg* srv_arg):return_value=&(%s)",
		    cinsert_service_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	return (void*) cinsert_service_arg;
}

/*!
 * Compare two service.
 * @param[in] srv_handle1 one of a unique service ID
 * @param[in] srv_handle2 one of a unique service ID
 * @retval 0  they matched perfectly.
 * @retval -1 they are different.
 */
static int
compare(handle_t srv_handle1, handle_t srv_handle2)
{
	struct l7vs_cinsert_service *cinsert_srv1, *cinsert_srv2;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,26,
		    "in_function: int compare(handle_t srv_handle1, handle_t srv_handle2):"
		    "srv_handle1=%u, srv_handle2=%u", srv_handle1, srv_handle2);
	}
	/*------ DEBUG LOG END ------*/

	/* search service that has such a service ID(1) */
	cinsert_srv1 = l7vs_protomod_cinsert_search_service(srv_handle1);

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cinsert_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_c_str(cinsert_str, cinsert_srv1);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,27, "pointer assign: cinsert_srv1=&(%s)",
		    cinsert_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cinsert_srv1 == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,18,
		    "Could not find such service handle's cinsert service.");
		return_value = -1;
		goto compare_out;
	}

	/* search service that has such a service ID(2) */
	cinsert_srv2 = l7vs_protomod_cinsert_search_service(srv_handle2);

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cinsert_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_c_str(cinsert_str, cinsert_srv2);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,28, "pointer assign: cinsert_srv2=&(%s)",
		    cinsert_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cinsert_srv2 == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,19,
		    "Could not find such service handle's cinsert service.");
		return_value = -1;
		goto compare_out;
	}

	/* compare two cookie name */
	if (strcmp(cinsert_srv1->cookie_name, cinsert_srv2->cookie_name) != 0) {
		return_value = -1;
		goto compare_out;
	}

compare_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,29,
		    "out_function: int compare(handle_t srv_handle1, handle_t srv_handle2):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Check the client packet and determine a real server.
 * @param[in]  srv service struct include service handle, protocol module and schedule module.
 * @param[in]  conn connection data.
 * @param[in]  request packet data from client
 * @param[in]  len length of packet data
 * @param[out] dest destination (real server) list
 * @param[out] tcps TCP Splicer flag
 * @retval 0  successfully check packet data
 * @retval -1 some errors occur.
 */
static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *request, size_t *len, struct l7vs_dest **dest, int *tcps)
{
	struct l7vs_cinsert_service *cinsert_service;
	struct l7vs_dest destination;
	int ret;
	int offset_length;
	char *cookie_value;
	char *x_forwarded_value;
	char *next_line = NULL;
	char encoded_address[DEST_ENCODE_LENGTH];
	char x_forwarded_for_header[X_FORWARDED_FOR_LENGTH];
	size_t uri_len = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_str[DEBUG_STR_LEN];
		char conn_str[DEBUG_STR_LEN];
		char dest_str[DEBUG_STR_LEN];
		char len_str[DEBUG_STR_LEN];
		l7vs_service_c_str(srv_str, srv);
		l7vs_conn_c_str(conn_str, conn);
		if (dest != NULL) {
			l7vs_dest_c_str(dest_str, *dest);
		}
		else {
			strncpy(dest_str, "NULL", DEBUG_STR_LEN);
		}
		if (len != NULL) {
			snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
		}
		else {
			strncpy(len_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,30,
		    "in_function: int match_cldata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* request, size_t* len, struct l7vs_dest** dest, int* tcps):srv=&(%s), conn=&(%s), "
		    "request=\"%s\", len=&(%s), dest=&(&(%s)), tcps=&(%d)",
		    srv_str, conn_str, request, len_str, dest_str, *tcps);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,20, "Arg(srv) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (srv->pm == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,21, "Arg(srv->pm) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (request == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,22, "Arg(request) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (len == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,23, "Arg(len) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (dest == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,24, "Arg(dest) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (tcps == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,25, "Arg(tcps) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}

	/* search service that has such a service ID */
	cinsert_service = l7vs_protomod_cinsert_search_service(srv->handle);

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cinsert_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_c_str(cinsert_str, cinsert_service);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,31, "pointer assign: cinsert_service=&(%s)",
		    cinsert_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cinsert_service == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,26, "Could not find such service handle's cinsert service.");
		return_value = -1;
		goto match_cldata_out;
	}

	/* initialize protocol module ... clear destination list */
	ret = srv->pm->initialize(srv, conn, request, *len, dest);
	if (ret != 0){
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,27, "Could not initialize protomod.");
		return_value = -1;
		goto match_cldata_out;
	}

	/* check the HTTP method in HTTP request header */
	uri_len = *len;
	if (http_check_request_method(request, &uri_len) == NULL) {
		PUT_LOG_INFO(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,4, "Client message is not HTTP request.");
		goto match_cldata_finalize;
	}

	/* add X-Forwarded-For field */
	if (cinsert_service->forwarded_for) {
		x_forwarded_value = http_search_header_field(request, "X-Forwarded-For");

		/* already exists X-Forwarded-For field */
		if (x_forwarded_value) {
			next_line = http_skip_header_line(x_forwarded_value);
			/* backtrack to look up insert point */
			if (next_line) {
				next_line--; // *next_line == '\n'
				if (*(next_line - 1) == '\r') {
					next_line--;
				}
				/* append client IP address */
				snprintf(x_forwarded_for_header, X_FORWARDED_FOR_LENGTH, ", %s", inet_ntoa(conn->caddr.sin_addr));
			}
		}

		/* not exists X-Forwarded-For field */
		if (!next_line) {
			/* construct new X-Forwarded-For header item */
			snprintf(x_forwarded_for_header, X_FORWARDED_FOR_LENGTH, "X-Forwarded-For: %s\r\n", inet_ntoa(conn->caddr.sin_addr));
	
			next_line = http_skip_header_line(request);
		}

		/* when insert point exist */
		if (next_line != NULL) {
			offset_length = (int) (next_line - request);
	
			/* insert X-Forwarded-For header field */
			http_insert_field(request, offset_length, x_forwarded_for_header, *len);
		
			/* add header length */
			*len += strlen(x_forwarded_for_header);
		}
	}

	/* search Cookie field in HTTP request header */
	cookie_value = http_search_header_cookie_value(request, cinsert_service->cookie_name);
	if (cookie_value == NULL) {
		/*-------- DEBUG LOG --------*/
		if (cinsert_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
			PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,32, "Cookie not found.");
		}
		/*------ DEBUG LOG END ------*/

		goto match_cldata_finalize;
	}

	/* memory copy encoded strings(16 byte) of IP address and port number */
	memcpy(encoded_address, cookie_value, DEST_ENCODE_LENGTH);
	encoded_address[DEST_ENCODE_LENGTH] = '\0';

	/* decode IP address and port number and set to destination struct */
	http_decode_address(encoded_address, (u_long *) &destination.addr.sin_addr.s_addr, &destination.addr.sin_port);

	/* set return value */
	*dest = &destination;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char dest_str[DEBUG_STR_LEN];
		l7vs_dest_c_str(dest_str, *dest);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,33, "pointer assign: dest=&(&(%s))",
		    dest_str);
	}
	/*------ DEBUG LOG END ------*/

match_cldata_finalize:
	*tcps = 0;

	/* finalize */
	ret = srv->pm->finalize(srv, conn, request, *len, dest, cinsert_service->reschedule);

	if (ret != 0){
		PUT_LOG_INFO(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,9, "Could not finalize protomod. (Realserver decision failure)");
		return_value = -1;
		goto match_cldata_out;
	}

match_cldata_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,34,
		    "out_function: int match_cldata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* request, size_t* len, struct l7vs_dest** dest, int* tcps):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Check the real server packet and insert a Set-Cookie field.
 * @param[in] srv service struct include service handle, protocol module and schedule module.
 * @param[in] conn connection data.
 * @param[in] response packet data from real server
 * @param[in] len length of packet data. it will be lengthened.
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn,
	char* response, size_t* len)
{
	struct l7vs_cinsert_service *cinsert_service;
	char *next_line;
	char expire_strings[EXPIRE_MAXSIZE];
	char encoded_address[DEST_ENCODE_LENGTH];
	char set_cookie_field[L7VS_PROTOMOD_MAX_ADD_BUFSIZE];
	int  offset_length;
	int  ret;
	time_t expire_time;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_str[DEBUG_STR_LEN];
		char conn_str[DEBUG_STR_LEN];
		char len_str[DEBUG_STR_LEN];
		l7vs_service_c_str(srv_str, srv);
		l7vs_conn_c_str(conn_str, conn);
		if (len != NULL) {
			snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
		}
		else {
			strncpy(len_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,35,
		    "in_function: int analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* response, size_t* len):srv=&(%s), conn=&(%s), response=\"%s\", len=&(%s)",
		    srv_str, conn_str, response, len_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,29, "Arg(srv) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (conn == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,30, "Arg(conn) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (conn->dest == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,31, "Arg(conn->dest) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (response == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,32, "Arg(response) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (len == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,33, "Arg(len) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}

	/* sorry flag check */
	if (conn->sorry_conn_flag == 1) {
		/*-------- DEBUG LOG --------*/
		if (cinsert_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
			PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,36, "Response from sorry server.");
		}
		/*------ DEBUG LOG END ------*/
		goto analyze_rsdata_out;
	}

	/* search service that has such a service ID */
	cinsert_service = l7vs_protomod_cinsert_search_service(srv->handle);

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cinsert_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_c_str(cinsert_str, cinsert_service);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,37, "pointer assign: cinsert_service=&(%s)",
		    cinsert_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cinsert_service == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,34,
		    "Could not find such service handle's cinsert service.");
		return_value = -1;
		goto analyze_rsdata_out;
	}

	/* check the HTTP status in HTTP response header */
	ret = http_check_response_status(response);
	/* when status is 4xx or 5xx, do not insert Set-Cookie field */
	if (ret != 0)
		goto analyze_rsdata_out;

	/* get top of HTTP header field line */
	next_line = http_skip_header_line(response);
	if (next_line == NULL)
		goto analyze_rsdata_out;

	/* create encode strings of IP address and port number */
	http_encode_address(encoded_address, conn->dest->addr.sin_addr.s_addr, conn->dest->addr.sin_port);

	/* calculate expire time */
	expire_time = l7vs_protomod_cinsert_timeout(cinsert_service->cookie_expire);

	/* create expire strings */
	http_cookie_expire(expire_time, expire_strings);

	/* create Set-Cookie header field */
	snprintf(set_cookie_field, L7VS_PROTOMOD_MAX_ADD_BUFSIZE, "Set-Cookie: %s=%s; expires=%s; path=/\r\n",
	        cinsert_service->cookie_name, encoded_address, expire_strings);

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,38, "Add field: \"%s\"", set_cookie_field);
	}
	/*------ DEBUG LOG END ------*/

	/* offset (HTTP status line length) */
	offset_length = (int) (next_line - response);

	/* insert Set-Cookie header field */
	http_insert_field(response, offset_length, set_cookie_field, *len);

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,39, "Add field: \"%s\"", set_cookie_field);
	}
	/*------ DEBUG LOG END ------*/

	/* add response length */
	*len += strlen(set_cookie_field);

analyze_rsdata_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,40,
		    "out_function: int analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* response, size_t* len):return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Destroy cinsert service
 * @param[in] srv_handle a unique service ID
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
destroy(handle_t srv_handle)
{
	/* cinsert service list counter */
	int service_number = 0;
	int free_flag = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,41,
		    "in_function: int destroy(handle_t srv_handle):srv_handle=%u",
		    srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check all cinsert service list */
	for (service_number = 0; service_number < CINSERT_SERVICE_NUMBER; ++service_number) {
		/* found cinsert service that has srv_handle */
		if (cinsert_service_list[service_number] != NULL && 
		    cinsert_service_list[service_number]->service_handle == srv_handle) {

			/*-------- DEBUG LOG --------*/
			if (cinsert_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,16, "free: %p",
				    cinsert_service_list[service_number]);
			}
			/*------ DEBUG LOG END ------*/

			/* free and NULL */
			free(cinsert_service_list[service_number]);
			cinsert_service_list[service_number] = NULL;

			free_flag = 1;
			break;
		}
	}
	
	/* cinsert service was not found */
	if (free_flag == 0) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,35, "Could not find such service handle's cinsert service.");
		return_value = -1;
		goto destroy_out;
	}

destroy_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,42,
		    "out_function: int destroy(handle_t srv_handle):return_value=%d",
		    srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Destroy cinsert service argument
 * @param[in] cinsert_arg cinsert service argument
 * @return void
 */
static void
destroy_sa(void **cinsert_arg)
{
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cinsert_arg_str[DEBUG_STR_LEN];
		if (cinsert_arg != NULL) {
			l7vs_cinsert_service_arg_c_str(cinsert_arg_str, (struct l7vs_cinsert_service_arg*) *cinsert_arg);
		}
		else {
			strncpy(cinsert_arg_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,43,
		    "in_function: void destroy_sa(void** cinsert_arg):cinsert_arg=&(&(%s))",
		    cinsert_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (cinsert_arg == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,36, "Arg(cinsert_arg) is NULL pointer.");
	}
	else if (*cinsert_arg == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,37, "Arg(*cinsert_arg) is NULL pointer.");
	}
	else {
		/*-------- DEBUG LOG --------*/
		if (cinsert_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
			PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,17, "free: %p",
			    *cinsert_arg);
		}
		/*------ DEBUG LOG END ------*/

		/* free and NULL */
		free((struct l7vs_cinsert_service_arg*) *cinsert_arg);
		*cinsert_arg = NULL;
	}

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,44,
		    "out_function: void destroy_sa(void** cinsert_arg)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Create strings for service list of l7vsadm
 * @param[out] srv_arg service argument struct
 * @param[in]  srv_handle a unique service ID
 * @retval 0  successfully create strings
 * @retval -1 some errors occur.
 */
static int
service_arg(struct l7vs_service_arg_multi *srv_arg_mt, handle_t srv_handle)
{
	struct l7vs_cinsert_service *cinsert_service;
	struct l7vs_cinsert_service_arg c_sarg;
	char cinsert_argument[SERVICE_ARG_MAXSIZE];
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_arg_mt_str[DEBUG_STR_LEN];
		l7vs_service_arg_multi_c_str(srv_arg_mt_str, srv_arg_mt);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,45,
		    "in_function: int service_arg(struct l7vs_service_arg_multi* srv_arg_mt, "
		    "handle_t srv_handle):srv_arg_mt=&(%s), srv_handle=%u",
		    srv_arg_mt_str, srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv_arg_mt == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,38, "Arg(srv_arg_mt) is NULL pointer.");
		return_value = -1;
		goto service_arg_out;
	}

	/* search service that has such a service ID */
	cinsert_service = l7vs_protomod_cinsert_search_service(srv_handle);

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cinsert_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_c_str(cinsert_str, cinsert_service);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,46, "pointer assign: cinsert_service=&(%s)",
		    cinsert_str);
	}
	/*------ DEBUG LOG END ------*/

	if (cinsert_service == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,39, "Could not find such service handle's cinsert service.");
		return_value = -1;
		goto service_arg_out;
	}

	/* initialize argument strings */
	memset(cinsert_argument, 0, SERVICE_ARG_MAXSIZE);

	/* set cinsert args to service argument struct */
	srv_arg_mt->srv_arg.reschedule = cinsert_service->reschedule;

	/* create long argument (l7vsadm option -L/-l) */
	snprintf(cinsert_argument, SERVICE_ARG_MAXSIZE, "--cookie-name %s", cinsert_service->cookie_name);
	strncpy(srv_arg_mt->srv_arg.protomod_key_string, cinsert_argument, 256);

	/* create verbose argument (l7vsadm option -V/-v) */
	snprintf(cinsert_argument + strlen(cinsert_argument), SERVICE_ARG_MAXSIZE - strlen(cinsert_argument),
	    " --cookie-expire %d", cinsert_service->cookie_expire);
	if (cinsert_service->forwarded_for) {
		snprintf(cinsert_argument + strlen(cinsert_argument), SERVICE_ARG_MAXSIZE - strlen(cinsert_argument),
		    " --forwarded-for");
	}
	strncpy(srv_arg_mt->srv_arg.protomod_opt_string, cinsert_argument, 512);

	strncpy(c_sarg.cookie_name, cinsert_service->cookie_name, COOKIE_NAME_MAXSIZE);
	c_sarg.cookie_expire = cinsert_service->cookie_expire;
	c_sarg.reschedule = cinsert_service->reschedule;

	memcpy(srv_arg_mt->protomod_arg, &c_sarg, sizeof(struct l7vs_cinsert_service_arg));

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char cinsert_arg_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_arg_c_str(cinsert_arg_str, &c_sarg);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,47,
		    "pointer assign: srv_arg_mt->protomod_arg=&(%s)", cinsert_arg_str);
	}
	/*------ DEBUG LOG END ------*/

service_arg_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,48,
		    "out_function: int service_arg(struct l7vs_service_arg_multi* srv_arg_mt, "
		    "handle_t srv_handle):return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Parse l7vsadm options to cinsert argument
 * @param[out] cinsert_arg cinsert service argument struct
 * @param[in]  argc number of l7vsadm argument
 * @param[in]  argv l7vsadm argument list
 * @retval 0  successfully parse argument
 * @retval -1 some errors occur.
 */
static int
parse(void *cinsert_arg, int argc, char *argv[])
{
	struct l7vs_cinsert_service_arg *cinsert_service_arg;
	static struct option opt[] = {
		{"cookie-name",   required_argument, NULL, 'C'},
		{"cookie-expire", required_argument, NULL, 'E'},
		{"forwarded-for", no_argument,       NULL, 'F'},
		{"reschedule",    no_argument,       NULL, 'R'},
		{"no-reschedule", no_argument,       NULL, 'N'},
		{NULL,            0,                 NULL, 0  }
	};
	int c;
	unsigned long buffer;
	int cookie_name_flag = 0;
	int cookie_expire_flag = 0;
	int forwarded_for_flag = 0;
	int reschedule_flag = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		int i;
		char argv_str[DEBUG_STR_LEN];
		char cinsert_arg_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_arg_c_str(cinsert_arg_str, (struct l7vs_cinsert_service_arg*) cinsert_arg);
		argv_str[0] = '\0';
		if (argv == NULL)
			snprintf(argv_str, DEBUG_STR_LEN, "NULL");
		else {
			for (i = 0; i < argc; i++) {
				snprintf(argv_str, DEBUG_STR_LEN, "%sargv[%d]=\"%s\", ", argv_str, i, argv[i]);
			}
			i = strnlen(argv_str, DEBUG_STR_LEN);
			if (i > 1)
				argv_str[i - 2] = '\0';
		}
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,49,
		    "in_function: int parse(void* cinsert_arg, int argc, char* argv[]):cinsert_arg=&(%s), "
		    "argc=%d, %s", cinsert_arg_str, argc, argv_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (cinsert_arg == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,40, "Arg(cinsert_arg) is NULL pointer.");
		return_value = -1;
		goto parse_out;
	}
	if (argv == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,41, "Arg(argv) is NULL pointer.");
		return_value = -1;
		goto parse_out;
	}

	cinsert_service_arg = (struct l7vs_cinsert_service_arg *) cinsert_arg;
	optind = 0;

	/* check all argument */
	while ((c = getopt_long(argc, argv, "C:E:FRN", opt, NULL)) != -1) {
		switch (c) {
		/* --cookie-name / -C */
		case 'C':
			/* check maximum length */
			if (strlen(optarg) >= sizeof(cinsert_service_arg->cookie_name)) {
				PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,42,
				    "-C/--cookie-name option value '%s' path too long", optarg);
				return_value = -1;
				goto parse_out;
			}
			strncpy(cinsert_service_arg->cookie_name, optarg, COOKIE_NAME_MAXSIZE);
			cookie_name_flag++;
			break;

		/* --cookie-expire / -E */
		case 'E':
			if (sscanf(optarg, "%lu", &buffer) == 0) {
				PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,43,
				    "-E/--cookie-expire option value '%s' is invalid.", optarg);
				return_value = -1;
				goto parse_out;
			}
			if (buffer > INT32_MAX) {
				PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,44,
				    "-E/--cookie-expire option value '%s' is too large.", optarg);
				return_value = -1;
				goto parse_out;
			}
			else
				cinsert_service_arg->cookie_expire = buffer;
			cookie_expire_flag++;
			break;

		/* --forwarded-for / -F */
		case 'F':
			/* x-forwarded-for on */
			cinsert_service_arg->forwarded_for = 1;
			forwarded_for_flag++;
			break;

		/* --reschedule / -R */
		case 'R':
			/* reschedule on */
			cinsert_service_arg->reschedule = 1;
			reschedule_flag++;
			break;

		/* --no-reschedule / -N */
		case 'N':
			/* reschedule off */
			cinsert_service_arg->reschedule = 0;
			reschedule_flag++;
			break;

		/* else error */
		default:
			PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,45, "Option error.");
			return_value = -1;
			goto parse_out;
		}
	}

	/* when set both -R and -N at the same time */
	if (reschedule_flag > 1) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,46,
		    "You have to choose either of reschdule or no-reschedule.");
		return_value = -1;
		goto parse_out;
	}
	/* same option */
	if (cookie_name_flag > 1) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,47,
		    "Cannot set multiple option '--cookie-name/-C'.");
		return_value = -1;
		goto parse_out;
	}
	if (cookie_expire_flag > 1) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,48,
		    "Cannot set multiple option '--cookie-expire/-E'.");
		return_value = -1;
		goto parse_out;
	}


	/* set default no forwarded_for */
	if (forwarded_for_flag == 0) {
		cinsert_service_arg->forwarded_for = 0;
	}

	/* set default no reschedule */
	if (reschedule_flag == 0) {
		cinsert_service_arg->reschedule = 0;
	}

	/* set default cookie name */
	if (cookie_name_flag == 0) {
		strcpy(cinsert_service_arg->cookie_name, "CookieName");
	}

	/* set default cookie expire (1 day) */
	if (cookie_expire_flag == 0) {
		cinsert_service_arg->cookie_expire = 86400;
	}

parse_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,50,
		    "out_function: int parse(void* cinsert_arg, int argc, char* argv[]):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Calculate future time
 * @param[in] timeout plus second
 * @return time_t future time
 */
static time_t
l7vs_protomod_cinsert_timeout(int timeout)
{
	time_t timer;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,51,
		    "in_function: time_t l7vs_protomod_cinsert_timeout(int timeout):"
		    "timeout=%d", timeout);
	}
	/*------ DEBUG LOG END ------*/

	/* now */
	timer = time((time_t *) 0);

	/* future */
	timer += timeout;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,52,
		    "out_function: time_t l7vs_protomod_cinsert_timeout(int timeout):"
		    "return_value=%d", (int) timer);
	}
	/*------ DEBUG LOG END ------*/

	return timer;
}

/*!
 * Insert any header field to HTTP header
 * @param[in,out] header HTTP header strings
 * @param[in] offset_length offset length
 * @param[in] insert_field insert header field strings
 * @param[in,out] header_length all of HTTP header length
 * @retval 0  successfully insert header field
 * @retval -1 some errors occur.
 */
//static int
//l7vs_protomod_insert_field(char *header, int offset_length, char *insert_field, int header_length)
//{
//	char *copy_buffer;
//	int   return_value = 0;
//
//	/*-------- DEBUG LOG --------*/
//	if (cinsert_protomod.get_log_level != NULL &&
//	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
//		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL, 0,
//		    "in_function: int l7vs_protomod_insert_field(char* header, int offset_length, "
//		    "char* insert_field, int header_length):header=\"%s\", "
//		    "offset_length=%d, insert_field=\"%s\", header_length=%d",
//		    header, offset_length, insert_field, header_length);
//	}
//	/*------ DEBUG LOG END ------*/
//
//	/* check args */
//	if (header == NULL) {
//		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL, 0, "Arg(header) is NULL pointer.");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//	if (insert_field == NULL) {
//		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL, 0, "Arg(insert_field) is NULL pointer.");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//	if (offset_length < 0) {
//		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL, 0, "Arg(offset_length) must be positive number.");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//	if (header_length < 0) {
//		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL, 0, "Arg(header_length) must be positive number.");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//	if (header_length < offset_length) {
//		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL, 0, "Arg(header_length) must be "
//		    "longer than arg(offset_length)");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//
//	/* calloc buffer */
//	copy_buffer = (char *) calloc(1, header_length - offset_length);
//
//	/*-------- DEBUG LOG --------*/
//	if (cinsert_protomod.get_log_level != NULL &&
//	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
//		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY, 0, "calloc: addr=%p, size=%d",
//		    copy_buffer, header_length - offset_length);
//	}
//	/*------ DEBUG LOG END ------*/
//
//	if (copy_buffer == NULL) {
//		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY, 0, "Could not allocate memory.");
//		return_value = -1;
//		goto set_cookie_out;
//	}
//
//	/* backup strings */
//	memcpy(copy_buffer, header + offset_length, header_length - offset_length);
//
//	/* insert Set-Cookie field */
//	memcpy(header + offset_length, insert_field, strlen(insert_field));
//
//	/* append backup strings and terminate null*/
//	memcpy(header + offset_length + strlen(insert_field), copy_buffer, header_length - offset_length);
//	header[offset_length + strlen(insert_field) + header_length - offset_length] = '\0';
//
//	/*-------- DEBUG LOG --------*/
//	if (cinsert_protomod.get_log_level != NULL &&
//	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
//		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY, 0, "free: %p",
//		    copy_buffer);
//	}
//	/*------ DEBUG LOG END ------*/
//
//	/* free */
//	free(copy_buffer);
//	copy_buffer = NULL;
//
//set_cookie_out:
//	/*-------- DEBUG LOG --------*/
//	if (cinsert_protomod.get_log_level != NULL &&
//	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
//		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL, 0,
//		    "out_function: int l7vs_protomod_insert_field(char* header, int offset_length, "
//		    "char* insert_field, int header_length):return_value=%d",
//		    return_value);
//	}
//	/*------ DEBUG LOG END ------*/
//
//	return return_value;
//}

/*!
 * Search cinsert service from cinsert service list using service handle
 * @param[in] service_handle a unique service ID
 * @return cinsert service struct when service was found. NULL when service was not found.
 */
static struct l7vs_cinsert_service *
l7vs_protomod_cinsert_search_service(handle_t service_handle)
{
	/* cinsert service list counter */
	int service_number = 0;
	struct l7vs_cinsert_service* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,53,
		    "in_function: struct l7vs_cinsert_service* l7vs_protomod_cinsert_search_service(handle_t service_handle):"
		    "service_handle=%d", service_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check all cinsert service list */
	for (service_number = 0; service_number < CINSERT_SERVICE_NUMBER; ++service_number) {
		/* found the service has same service handle */
		if (cinsert_service_list[service_number] != NULL && 
		    cinsert_service_list[service_number]->service_handle == service_handle) {
			return_value = cinsert_service_list[service_number];
			break;
		}
	}
	
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,54,
		    "out_function: struct l7vs_cinsert_service* l7vs_protomod_cinsert_search_service(handle_t service_handle):"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create cinsert service.
 * @param  void
 * @return cinsert service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_cinsert_service *
l7vs_protomod_cinsert_create_service()
{
	struct l7vs_cinsert_service* return_value = NULL;
	int service_number = 0;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,55,
		    "in_function: struct l7vs_cinsert_service* l7vs_protomod_cinsert_create_service()");
	}
	/*------ DEBUG LOG END ------*/

	/* check all cinsert service list */
	for (service_number = 0; service_number < CINSERT_SERVICE_NUMBER - 1; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (cinsert_service_list[service_number] == NULL) {
			/* create a service at empty pointer */
			cinsert_service_list[service_number] = (struct l7vs_cinsert_service *) calloc(1, sizeof(struct l7vs_cinsert_service));

			/*-------- DEBUG LOG --------*/
			if (cinsert_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,18, "calloc: addr=%p, size=%ld",
				    cinsert_service_list[service_number], (unsigned long int) sizeof(struct l7vs_cinsert_service));
			}
			/*------ DEBUG LOG END ------*/

			if (cinsert_service_list[service_number] == NULL) {
				PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,17, "Could not allocate memory.");
				goto create_service_out;
			}
			return_value = cinsert_service_list[service_number];
			goto create_service_out;
		}
	}
	
	/* all service list is full */
	PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,49, "cinsert service list is full.");

create_service_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,56,
		    "out_function: struct l7vs_cinsert_service* l7vs_protomod_cinsert_create_service():"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create temporary cinsert service.
 * @param  void
 * @return cinsert service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_cinsert_service *
l7vs_protomod_cinsert_create_temp_service()
{
	struct l7vs_cinsert_service* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,57,
		    "in_function: struct l7vs_cinsert_service* l7vs_protomod_cinsert_create_temp_service()");
	}
	/*------ DEBUG LOG END ------*/

	/* if pointer that does not point NULL exists ... */
	if (cinsert_service_list[CINSERT_SERVICE_NUMBER - 1] != NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,50, "Temporary cinsert service is being used by other process.");
		goto create_temp_service_out;
	}

	/* create temp service */
	cinsert_service_list[CINSERT_SERVICE_NUMBER - 1] = (struct l7vs_cinsert_service *) calloc(1, sizeof(struct l7vs_cinsert_service));

	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,19, "calloc: addr=%p, size=%ld",
		    cinsert_service_list[CINSERT_SERVICE_NUMBER - 1], (unsigned long int) sizeof(struct l7vs_cinsert_service));
	}
	/*------ DEBUG LOG END ------*/

	if (cinsert_service_list[CINSERT_SERVICE_NUMBER - 1] == NULL) {
		PUT_LOG_ERROR(cinsert_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,18, "Could not allocate memory");
		goto create_temp_service_out;
	}

	return_value = cinsert_service_list[CINSERT_SERVICE_NUMBER - 1];

create_temp_service_out:
	/*-------- DEBUG LOG --------*/
	if (cinsert_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == cinsert_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN];
		l7vs_cinsert_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(cinsert_protomod, LOG_CAT_L7VSD_PROTOCOL,58,
		    "out_function: struct l7vs_cinsert_service* l7vs_protomod_cinsert_create_service():"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Serialize struct l7vs_cinsert_service for debug log.
 * @param[out] buf   serialized string
 * @param[in]  cinsert l7vs_cinsert_service struct
 */
static void l7vs_cinsert_service_c_str(char* buf, struct l7vs_cinsert_service* cinsert) {
	if (cinsert == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		snprintf(buf, DEBUG_STR_LEN, "service_handle=%d, cookie_name=\"%s\", cookie_expire=%d, "
		    "forwarded_for=%d, reschedule=%d", cinsert->service_handle, cinsert->cookie_name,
		    cinsert->cookie_expire, cinsert->forwarded_for, cinsert->reschedule);
	}
}

/*!
 * Serialize struct l7vs_cinsert_service_arg for debug log.
 * @param[out] buf       serialized string
 * @param[in]  cinsert_arg l7vs_cinsert_service_arg struct
 */
void l7vs_cinsert_service_arg_c_str(char* buf, struct l7vs_cinsert_service_arg* cinsert_arg) {
	if (cinsert_arg == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		snprintf(buf, DEBUG_STR_LEN, "cookie_name=\"%s\", cookie_expire=%d, forwarded_for=%d, "
		    "reschedule=%d", cinsert_arg->cookie_name, cinsert_arg->cookie_expire,
		    cinsert_arg->forwarded_for, cinsert_arg->reschedule);
	}
}
