/*
 * @file	l7vs_conn.h
 * @bref	Connection is managed. 
 * @bref	The previous substance becomes one conn though two iom is registered
 * @bref	in sets of imoux so that conn may have sockaddr for a real server 
 * @bref	and sockaddr for the client internally. 
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/
#ifndef	L7VS_CONN_H
#define	L7VS_CONN_H

#include <netinet/in.h>
#include <arpa/inet.h>
#include "l7vs_iom.h"
#include "l7vs_iomuxlist.h"
#include "l7vs_dest.h"
#include "logger_wrapper.h"

#define	FDSET_SIZE	(8192)
struct	my_fd_set{
	char	x[FDSET_SIZE];
};
#undef	FD_ZERO
#define	FD_ZERO(ptr) memset((ptr), 0, FDSET_SIZE)

// connection management datatype
#define	L7VS_CLDATA_CHUNKSIZE	(512)
#define	L7VS_CLDATA_BUFSIZE(L)	(((L)->cldata_len + L7VS_CLDATA_CHUNKSIZE - 1) & ~(L7VS_CLDATA_CHUNKSIZE - 1))
#define	L7VS_CLDATA_REST(L)	(L7VS_CLDATA_BUFSIZE(L) - (L)->cldata_len)

#define	L7VS_CONN_S_CL_CONNECTED	0
#define	L7VS_CONN_S_RS_CONNECTING	1
#define	L7VS_CONN_S_RS_CONNECTED	2

struct	l7vs_conn{
	struct	l7vs_lsock*	lsock;	//! l7vs_lsock pointer
	struct	l7vs_service*	srv;	//! l7vs_service pointer
	struct	l7vs_dest*	dest;	//! real server pointer
	int			splice;	//! tcp splice handle (don't use)
	struct	sockaddr_in	caddr;	//! client sockaddr struct
	struct	sockaddr_in	raddr;	//! real server sockaddr struct
	struct	sockaddr_in	vaddr;	//! 
	struct	sockaddr_in	laddr;  //! 
	struct	l7vs_iomux*	ciom;	//!	client-side socket
	struct	l7vs_iomux*	riom;	//!	realserver-side socket	
	uint8_t			proto;	//!
	uint32_t		state;	//! connn status
	char*			cldata; //! client data buffer
	size_t			cldata_len; //! client data buffer length
	size_t			cldata_bufsize; //! client buffer size
	int			cmss;	//! cmess??
	struct	timeval		recvtime;//! client data recv time
	size_t			recvsize;//! client data recv size
	int			sorry_conn_flag; //! sorry connection flag
	struct  l7vs_dest*	old_dest; //! old destination
};

//
// functions.
//
extern	struct	l7vs_conn*	l7vs_conn_create( int, struct l7vs_lsock* );
extern	void			l7vs_conn_destroy( struct l7vs_conn* );
//extern	int			l7vs_conn_recv_client( struct l7vs_conn* );
extern	int			l7vs_conn_connect_rs( struct l7vs_conn*, struct l7vs_dest* );
//extern	int			l7vs_conn_splice( struct l7vs_conn* );
extern	void			l7vs_conn_close_csock( struct l7vs_conn* );
extern	void			l7vs_conn_close_rsock( struct l7vs_conn* );
extern	int			l7vs_conn_closed( struct l7vs_conn* );
/*!
 * serialize struct sockaddr_in
 *
 * @param[out]	ret	serialized string
 * @param[in]	conn	source sockaddr_in
 * @return	void
 */
inline	void	l7vs_conn_sockaddr_in_c_str(char* ret, const struct sockaddr_in* addr)
{
	if (!ret) {
		return;
	}

	if (!addr) {
		snprintf(ret, DEBUG_STR_LEN, "(nil)");
		return;
	}

	snprintf(ret, DEBUG_STR_LEN,
		"sockaddr_in="
		"{sin_addr=%s: "
		"sin_port=%d}"
		, inet_ntoa(addr->sin_addr)
		, ntohs(addr->sin_port));
}

/*!
 * serialize struct l7vs_conn
 *
 * @param[out]	ret	serialized string
 * @param[in]	conn	source l7vs_conn
 * @return	void
 */
inline	void	l7vs_conn_c_str(char* ret, const struct l7vs_conn* conn)
{
	if (!ret) {
		return;
	}

	char dest_str[DEBUG_STR_LEN];
	char caddr_str[DEBUG_STR_LEN];
	char raddr_str[DEBUG_STR_LEN];
	char ciom_str[DEBUG_STR_LEN];
	char riom_str[DEBUG_STR_LEN];
	char recvtime_str[DEBUG_STR_LEN];
	char old_dest_str[DEBUG_STR_LEN];

	if (!conn) {
		snprintf(ret, DEBUG_STR_LEN, "(nil)");
		return;
	}

	l7vs_conn_sockaddr_in_c_str(caddr_str, &conn->caddr);
	l7vs_conn_sockaddr_in_c_str(raddr_str, &conn->raddr);
	l7vs_dest_c_str(dest_str, conn->dest);
	l7vs_iomux_c_str(ciom_str, conn->ciom);
	l7vs_iomux_c_str(riom_str, conn->riom);
	l7vs_dest_c_str(old_dest_str, conn->old_dest);
	
	snprintf(recvtime_str, DEBUG_STR_LEN,
		"timeval="
		"{tv_sec=%ld: "
		"tv_usec=%ld}"
		, conn->recvtime.tv_sec
		, conn->recvtime.tv_usec);

	snprintf(ret, DEBUG_STR_LEN, 
		"l7vs_conn="
		"{lsock=%p: "
		"srv=%p: "
		"dest=%s: "
		"caddr=%s: "
		"raddr=%s: "
		"ciom=%s: "
		"riom=%s: "
		"proto=%x: "
		"cldata=%s: "
		"cldata_len=%zu: "
		"cldata_bufsize=%zu: "
		"cmss=%d: "
		"recvtime=%s: "
		"recvsize=%zu: "
		"sorry_conn_flag=%d: "
		"old_dest=%s}"
		, conn->lsock
		, conn->srv
		, dest_str
		, caddr_str
		, raddr_str
		, ciom_str
		, riom_str
		, conn->proto
		, conn->cldata
		, conn->cldata_len
		, conn->cldata_bufsize
		, conn->cmss
		, recvtime_str
		, conn->recvsize
		, conn->sorry_conn_flag
		, old_dest_str);
}

#endif	//L7VS_CONN_H
