/**********************************************************************
 * protomod_url.c                                           August 2005
 *
 * URL Persistence Module for L7vsd
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#define _GNU_SOURCE
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fnmatch.h>
#include <getopt.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"

struct l7vs_url_service {
        struct l7vs_service srv;
        char pattern_match[129];
	int reschedule;
};

struct  l7vs_url_service_arg {
        struct l7vs_service_arg arg;
        char pattern_match[129];
	int reschedule;
};

static void fini(void);
static struct l7vs_service *create(struct l7vs_service_arg *arg);
static struct l7vs_service_arg *create_sa(void);
static int compare(struct l7vs_service *s1, struct l7vs_service *s2);
static int match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn, char *buf, size_t *len, struct l7vs_dest **dest, int *tcps);
static int analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn, char *buf, size_t *len);
static void destroy(struct l7vs_service *srv);
static struct l7vs_service_arg *service_arg(struct l7vs_service *srv);
static int parse(struct l7vs_service_arg *arg, int argc, char *argv[]);

static struct l7vs_protomod url_protomod = {
        NULL,                   /* handle */
        "url",                  /* modname */
        0,                      /* refcnt */
        create,                 /* create function */
        compare,                /* compare function */
        match_cldata,         	/* match_cldata function */
	analyze_rsdata,		/* analyze_rsdata function */
        destroy,                /* destroy function */
        fini,                   /* fini function */
        create_sa,              /* create_sa function */
        service_arg,            /* service_arg function */
        parse,                  /* parse function */
};

struct l7vs_protomod *
init(void *handle)
{
        url_protomod.handle = handle;
        return &url_protomod;
}

static void 
fini(void)
{
        /* XXX  maybe some work needed */
}

static struct l7vs_service *
create(struct l7vs_service_arg *arg)
{
        struct l7vs_url_service *hs;
        struct l7vs_url_service_arg *ha;

        hs = (struct l7vs_url_service *)calloc(1, sizeof(*hs));
        if (hs == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service *)hs;
        }

        ha = (struct l7vs_url_service_arg *)arg;
        strcpy(hs->pattern_match, ha->pattern_match);
        hs->reschedule = ha->reschedule;
        return (struct l7vs_service *)hs;
}

static struct l7vs_service_arg *
create_sa(void)
{
        struct l7vs_url_service_arg *ha;

        ha = (struct l7vs_url_service_arg *)calloc(1, sizeof(*ha));
        if (ha == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service_arg *)ha;
        }

        ha->arg.len = sizeof(*ha);
        strcpy(ha->arg.protomod, url_protomod.modname);

        return (struct l7vs_service_arg *)ha;
}

static int
compare(struct l7vs_service *s1, struct l7vs_service *s2)
{
        struct l7vs_url_service *h1, *h2;

        h1 = (struct l7vs_url_service *)s1;
        h2 = (struct l7vs_url_service *)s2;

        return strcmp(h1->pattern_match, h2->pattern_match);

}

static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *buf, size_t *len, struct l7vs_dest **dest, int *tcps)
{
        struct l7vs_url_service *hs;
        char *s1, *s2, tmp;
        int ret;

        hs = (struct l7vs_url_service *)srv;

	ret = srv->pm->initialize(srv, conn, buf, *len, dest);
	if (ret != 0){
		VANESSA_LOGGER_ERR("Could not initialize protomod");
		return -1;
	}

	if (hs->pattern_match[0] == '\0'){
		ret = srv->pm->finalize(srv, conn, buf, *len, dest, hs->reschedule);
		if (ret != 0){
			VANESSA_LOGGER_ERR("Could not finalize protomod");
			return -1;
		}
		*tcps = 1;
		return 0;
	}

	if ((s1 = strchr(buf, ' ')) == NULL) {
		tmp = buf[*len - 1];
		buf[*len - 1] = '\0';
		ret = fnmatch(hs->pattern_match, buf, 0);
		buf[*len - 1] = tmp;
	} else {
		s1++;
		if((s2 = strchr(s1, ' ')) == NULL){
			tmp = buf[*len - 1];
			buf[*len - 1] = '\0';
			ret = fnmatch(hs->pattern_match, s1, 0);
			buf[*len - 1] = tmp;
		} else {
			*s2 = '\0';
			ret = fnmatch(hs->pattern_match, s1, 0);
			*s2 = ' ';
		}
	}

        if (ret == 0) {
		*tcps = 1;       // possible to omit
		ret = srv->pm->finalize(srv, conn, buf, *len, dest, hs->reschedule);
		if (ret != 0){
			VANESSA_LOGGER_ERR("Could not finalize protomod");
			return -1;
		}
		return 0;
	}

        return 1;
}

static int
analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *buf, size_t *len)
{
	return 0;
}

static void
destroy(struct l7vs_service *srv)
{
	free(srv);
}

static struct l7vs_service_arg *
service_arg(struct l7vs_service *srv)
{
        struct l7vs_url_service *hs;
        struct l7vs_url_service_arg *ha;
	char temp[512];
	char *ptr;

        hs = (struct l7vs_url_service *)srv;

        ha = (struct l7vs_url_service_arg *)malloc(sizeof(*ha));
        if (ha == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service_arg *)ha;
        }

        ha->arg.len = sizeof(*ha);
        ha->reschedule = hs->reschedule;
        strcpy(ha->pattern_match, hs->pattern_match);

	ptr = temp;
	ha->arg.reschedule = hs->reschedule;
	sprintf(ptr, "--pattern-match %s\0", hs->pattern_match);
	strcpy(ha->arg.protomod_key_string, ptr);
	sprintf(ptr + strlen(ptr), "\0");
	strcpy(ha->arg.protomod_opt_string, ptr);

        return (struct l7vs_service_arg *)ha;
}

static int
parse(struct l7vs_service_arg *arg, int argc, char *argv[])
{
        static struct option opt[] = {
                {"pattern-match",	required_argument,	NULL, 'P'},
                {"reschedule",		no_argument,		NULL, 'F'},
                {"no-reschedule",	no_argument,		NULL, 'N'},
                {NULL,          0,                      NULL, 0}
        };
        struct l7vs_url_service_arg *ha = (struct l7vs_url_service_arg *)arg;
        int c;
	int c1 = 0;

        optind = 0;
        while ((c = getopt_long(argc, argv, "P:", opt, NULL)) != -1) {
                switch (c) {
                case 'P':
                        if (strlen(optarg) >= sizeof(ha->pattern_match)) {
                                VANESSA_LOGGER_ERR_UNSAFE(
                                                "%s: pattern too long",
                                                optarg);
                                return -1;
                        }
                        strcpy(ha->pattern_match, optarg);
                        break;
                case 'F':
                        ha->reschedule = 1;
			c1++;
                        break;
		case 'N':
			ha->reschedule = 0;
			c1++;
			break;
                default:
                        return -1;
                }
        }

	if (c1 == 2) {
		VANESSA_LOGGER_ERR("You should choose either of reschdule or no-reschedule");
		return -1;
	}
	if (c1 == 0) {
		ha->reschedule = 0;
	}

        return 0;
}
