/**********************************************************************
 * record.h                                                 August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifndef _KSSL_CONN_RECORD_H
#define _KSSL_CONN_RECORD_H

#include <linux/net.h>
#include <linux/list.h>
#include <asm/scatterlist.h>

#include "conn.h"
#include "record_list.h"
#include "message_mask.h"
#include "log.h"

#include "types/record_t.h"
#include "types/ssl2_record_t.h"

#define KSSL_CONN_RECORD_IOV_BLOCK 4

#define KSSL_CONN_RECORD_RCV_STARTED  0x00000001
#define KSSL_CONN_RECORD_RCV_COMPLETE 0x00000002

#define KSSL_CONN_RECORD_IN_STARTED   0x00000010
#define KSSL_CONN_RECORD_IN_UNCMP     0x00000020
#define KSSL_CONN_RECORD_IN_UNENC     0x00000030

#define KSSL_CONN_RECORD_OUT_STARTED  0x00000100
#define KSSL_CONN_RECORD_OUT_CMP      0x00000200
#define KSSL_CONN_RECORD_OUT_DIGEST   0x00000400
#define KSSL_CONN_RECORD_OUT_ENC      0x00000800

#define KSSL_CONN_RECORD_SND_STARTED  0x00001000
#define KSSL_CONN_RECORD_SND_COMPLETE 0x00002000

#define KSSL_CONN_RECORD_NLEN(cr) ((cr)->record.head.type == ct_ssl2 ? \
	SSL2_RECORD_NLEN(&((cr)->ssl2_head)) : RECORD_NLEN(&((cr)->record)))

#define KSSL_CONN_HEAD_NLEN(cr) ((cr)->record.head.type == ct_ssl2 ? \
	SSL2_HEAD_NLEN(&((cr)->ssl2_head)) : TLS_HEAD_NLEN)


static inline void kssl_record_clear(kssl_record_t *cr)
{
	kssl_conn_t *conn;

	conn = cr->conn;
	memset(cr, 0, sizeof(kssl_record_t));
	cr->conn = conn;
	INIT_LIST_HEAD(&(cr->list));
}

kssl_record_t *__kssl_record_create(kssl_conn_t *conn, u32 conn_flag,
		const char *function, const char *file, size_t line);
#define kssl_record_create(conn, conn_flag) 				\
	__kssl_record_create(conn, conn_flag, __FUNCTION__, __FILE__, __LINE__)

void __kssl_record_destroy(kssl_record_t *cr,
		const char *function, const char *file, size_t line);
#define kssl_record_destroy(cr) 					\
	__kssl_record_destroy(cr, __FUNCTION__, __FILE__, __LINE__)

int kssl_record_vec_split(kssl_record_t *head,
		kssl_record_t **tail, size_t record_len);

int kssl_record_vec_cpy(kssl_record_t *cr,
		u8 *buf, size_t buf_len, size_t offset);

int kssl_record_vec_cmp(kssl_record_t *cr,
		u8 *buf, size_t buf_len, size_t offset);

int kssl_record_vec_seek(kssl_record_t *cr,
		size_t offset, size_t *vecno,
		size_t *vecoffset);

int kssl_record_vec_cpy_ptr(kssl_record_t *cr,
		struct iovec *iov, struct sk_buff **skbv,
		size_t len, size_t offset);

int kssl_record_update(kssl_record_t **cr);

int kssl_record_process_in(kssl_record_t *cr);

int kssl_record_process_out(kssl_record_t *cr);

int kssl_record_add_skb(kssl_record_t *cr,
			struct sk_buff *skb, size_t offset, size_t len);

int kssl_version_verify(protocol_version_t *version);

size_t kssl_record_tail_len(kssl_record_t *cr, size_t content_len);

int kssl_record_send(kssl_record_t *cr);

void kssl_record_head_set(kssl_record_t *cr, content_type_t type, 
		size_t len);

typedef int(*record_write_func_t)(kssl_record_t *cr, void *data);

int kssl_record_build_send(kssl_record_t *cr, record_write_func_t write_func, 
		size_t record_len, void *data, int reuse);

int kssl_record_pt_send(kssl_record_t *cr, size_t len, size_t offset);

int kssl_record_to_sg(kssl_record_t *cr, u32 offset, u32 size,
                struct scatterlist **sg_r, u32 *nvec_r, u32 *len_r);


#endif /* _KSSL_CONN_RECORD_H */

