/**********************************************************************
 * kssl_proc.c                                              August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include "kssl.h"
#include "kssl_proc.h"
#include "conn_list.h"
#include "conn.h"

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/proc_fs.h>
#include <linux/sysctl.h>


static int 
kssld_get_version(char *buf, char **start, off_t offset, int length)
{
	int pos = 0;
	int len;
	char tmp[32];

	if (length == 0)
		return 0;

	len = 32;
	if (offset < len) {
		memset(tmp, ' ', 32);
		sprintf(tmp, "KSSLD version " VERSION);
		len = len > length ? length : len;
		memcpy(buf, tmp + offset, len);
		pos += len;
		offset = 0;
	}
	else {
		offset -= len;
	}

	*(buf + pos - 2) = '\n';
	*(buf + pos - 1) = '\0';
	return pos;
}


static int 
kssld_get_conn(char *buf, char **start, off_t offset, int length)
{
	int pos = 0;
	int len;
	char tmp[128];
	const char *fmt = "%08x:%04x->%08x:%04x in=%08x out=%08x (%4s)";
	kssl_conn_t *conn;
	struct list_head *c_list;
	struct list_head *t_list;

	*start = buf;

	if (length == 0)
		return 0;

	list_for_each_safe(c_list, t_list, &kssl_conn_list) {
		if (pos >= length) 
			break;

		if (offset > 128) {
			offset -= 128;
			continue;
		}

		conn = list_entry(c_list, kssl_conn_t, list);

		spin_lock(&(conn->sock_lock));

		if(conn->ssl_sock) {
			len = sprintf(tmp, fmt, 
					ntohl(conn->ssl_sock->sk->daddr),
					ntohs(conn->ssl_sock->sk->dport),
					ntohl(conn->ssl_sock->sk->saddr),
					ntohs(conn->ssl_sock->sk->sport),
					conn->ssl_in_bytes,
					conn->ssl_out_bytes, "ssl");
		}
		else {
			len = sprintf(tmp, fmt, 0, 0, 0, 0, 0, 0 , "none");
		}
		len += sprintf(tmp + len, " | ");
		if(conn->pt_sock) {
			len += sprintf(tmp + len, fmt, 
					ntohl(conn->pt_sock->sk->daddr),
					ntohs(conn->pt_sock->sk->dport),
					ntohl(conn->pt_sock->sk->saddr),
					ntohs(conn->pt_sock->sk->sport),
					conn->pt_in_bytes,
					conn->pt_out_bytes, "pt");
		}
		else {
			len += sprintf(tmp + len, fmt, 0, 0, 0, 0, 0, 0 , 
					"none");
		}

		spin_unlock(&(conn->sock_lock));

		memset(tmp + len, ' ', 128 - len);
		*(tmp + 128 - 1) = '\n';

		len = 128 > length - pos ? length - pos : 128;
		memcpy(buf + pos, tmp + offset, len - offset);
		pos += len - offset;
		offset = 0;
	}

	return pos;
}


/* Sysctl Table  under /proc/sys/net/ipv4/kssl/ */

#ifdef CONFIG_KSSL_DEBUG
//static int sysctl_kssl_debug_level = 0;
static int sysctl_kssl_debug_level = 3;

int kssl_get_debug_level(void)
{
        return sysctl_kssl_debug_level;
}
#endif /* CONFIG_KSSL_DEBUG */

#define NET_IPV4_KSSL              22

enum {
	NET_IPV4_KSSL_DEBUG_LEVEL=1,
	NET_IPV4_KSSL_LAST
};

struct kssl_sysctl_table {
	struct ctl_table_header *sysctl_header;
	ctl_table kssl_vars[NET_IPV4_KSSL_LAST];
	ctl_table kssl_dir[2];
	ctl_table ipv4_dir[2];
	ctl_table root_dir[2];
};

static struct kssl_sysctl_table ipv4_kssl_table = {
	NULL,
#ifdef CONFIG_KSSL_DEBUG
	{
		{
			NET_IPV4_KSSL_DEBUG_LEVEL, "debug_level", 
			&sysctl_kssl_debug_level, sizeof(int), 0644, NULL,
			&proc_dointvec
		},
		{ 0 }
	},
#endif /* CONFIG_KSSL_DEBUG */
	{
		{ 
			NET_IPV4_KSSL, "kssld", NULL, 0, 0555, 
			ipv4_kssl_table.kssl_vars 
		},
		{ 0 }
	},
	{
		{ 
			NET_IPV4, "ipv4", NULL, 0, 0555, 
			ipv4_kssl_table.kssl_dir 
		},
		{ 0 }
	},
	{
		{ 
			CTL_NET, "net", NULL, 0, 0555, 
			ipv4_kssl_table.ipv4_dir 
		},
		{ 0 }
	}
};


void 
kssl_proc_destroy(void)
{
	proc_net_remove("kssld");
	proc_net_remove("kssld_conn");
	if (ipv4_kssl_table.sysctl_header)
		unregister_sysctl_table(ipv4_kssl_table.sysctl_header);
}


int 
kssl_proc_init(void)
{
	ipv4_kssl_table.sysctl_header = register_sysctl_table(
			ipv4_kssl_table.root_dir, 0);
	if (!ipv4_kssl_table.sysctl_header)
		return -ENOMEM;

	if (! proc_net_create("kssld", 0, kssld_get_version) ) {
		unregister_sysctl_table(ipv4_kssl_table.sysctl_header);
		return -EINVAL;
	}

	if (! proc_net_create("kssld_conn", 0, kssld_get_conn) ) {
		unregister_sysctl_table(ipv4_kssl_table.sysctl_header);
		proc_net_remove("kssld");
		return -EINVAL;
	}

	return 0;
}

