/**********************************************************************
 * kssl_asym.c                                              August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include "aep.h"

#include "types/handshake_t.h"
#include "daemon.h"
#include "kssl_alloc.h"
#include "pk.h"
#include "aep.h"
#include "record.h"
#include "log.h"

#include <linux/kernel.h>

static int 
kssl_asym_decrypt_aep_kernel(client_key_exchange_t *cke, kssl_key_t *key, 
		u8 *out_buf, size_t out_buf_len) 
{
	/* Not implemented */
	return -EINVAL;
}


static int 
kssl_asym_decrypt_aep_user(client_key_exchange_t *cke, kssl_key_t *key,
		u8 *out_buf, size_t out_buf_len) 
{
	int status = -EINVAL;
	opaque_t aep_buf[1096];
	size_t aep_buf_len;

	aep_buf_len = sizeof(aep_buf);

	status = kssl_aep_msg_make_req_crt(0, 0, 
			cke->exchange_keys.rsa.encrypted_data +
				cke->exchange_keys.rsa.offset,
			cke->exchange_keys.rsa.len, 
			key->key.rsa.p.iov_base, key->key.rsa.p.iov_len,
			key->key.rsa.q.iov_base, key->key.rsa.q.iov_len,
			key->key.rsa.dmp1.iov_base, key->key.rsa.dmp1.iov_len,
			key->key.rsa.dmq1.iov_base, key->key.rsa.dmq1.iov_len,
			key->key.rsa.iqmp.iov_base, key->key.rsa.iqmp.iov_len,
			aep_buf, &aep_buf_len, KSSL_AEP_REVERSE);
	if(status < 0) {
		KSSL_DEBUG(6, "kssl_asym_decrypt_aep_user: "
				"message_make_req_crt\n"); 
		return status;
	}

	status = kssl_aep_write(aep_buf, aep_buf_len);
	if (status < 0) {
		KSSL_DEBUG(9, "kssl_asym_decrypt_aep_user: "
				"kssl_aep_write: %d\n", status);
		return status;
	}

	aep_buf_len = sizeof(aep_buf);
	status = kssl_aep_read(aep_buf, aep_buf_len);
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_asym_decrypt_aep_user: "
				"kssl_aep_read: %d\n", status);
		return status;
	}

	if (status < 32 + 4) {
		KSSL_DEBUG(6, "kssl_asym_decrypt_aep_user: "
				"kssl_aep_recv: read too short %d < 36\n",
				status);
		return -EINVAL;
	}

	memcpy(&aep_buf_len, aep_buf + 32, 4);
	if(aep_buf_len > status - 32) {
		KSSL_DEBUG(6, "kssl_asym_decrypt_aep_user: "
				"kssl_aep_recv: data too short %u > %d\n",
				aep_buf_len, status - 32);
		return -EINVAL;
	}

	if(aep_buf_len < out_buf_len) {
		KSSL_DEBUG(6, "kssl_asym_decrypt_aep_user: "
				"kssl_aep_recv: not enough data %u < %u\n",
				aep_buf_len, out_buf_len);
		return -EINVAL;
	}

	kssl_aep_reverse_8(out_buf, aep_buf+32+4, out_buf_len);

	 KSSL_NOTICE(3, "ISSL012: Info(ssl): Decrypted successfully\n");

	return 0;
}


static int 
kssl_asym_decrypt_sofware(client_key_exchange_t *cke, kssl_key_t *key, 
		u8 *out_buf, size_t out_buf_len) 
{
	int status = -EINVAL;
	u8 *pri_buf;

	pri_buf = kssl_key_get_private_buf(key);

	status = pk_init(cke->exchange_keys.rsa.len << 3, 0);
	if (status) {
		KSSL_DEBUG(6, "kssl_asym_decrypt_sofware: "
				"pk_init(%d, %d): %d\n", 
				cke->exchange_keys.rsa.len << 3, 0,
				status);
		if (status > 0)
			status = -EINVAL;
		kssl_kfree (pri_buf);
		return status;
	}

	status = pk_decrypt(out_buf, &out_buf_len, 
			cke->exchange_keys.rsa.encrypted_data +
				cke->exchange_keys.rsa.offset, 
			cke->exchange_keys.rsa.len, pri_buf, 
			(cke->exchange_keys.rsa.len << 1) + 2,
			PK_ALGO_RSA|PK_PKCS1_ENCRYPT);
	if (status) {
		KSSL_DEBUG(6, "kssl_asym_decrypt_sofware: "
				"decrypt failed: %d\n", status);
		if (status > 0)
			status = -EINVAL;
		kssl_kfree (pri_buf);
		return status;
	}

	kssl_kfree (pri_buf);
	return 0;
}


static inline int 
__kssl_asym_decrypt(client_key_exchange_t *cke, kssl_key_t *key, 
		u8 *out_buf, size_t out_buf_len, kssl_asym_method_t am) 
{
	switch (am) {
		case kssl_asym_method_software:
			return kssl_asym_decrypt_sofware(cke, key, out_buf, 
					out_buf_len);
		case kssl_asym_method_aep_user:
			return kssl_asym_decrypt_aep_user(cke, key, out_buf, 
					out_buf_len);
		case kssl_asym_method_aep_kernel:
			return kssl_asym_decrypt_aep_kernel(cke, key, out_buf, 
					out_buf_len);
		default:
			break;
	}

	return -EINVAL;
}


int 
kssl_asym_decrypt(kssl_record_t *cr, u8 *out_buf, size_t out_buf_len) 
{
	int status = -EINVAL;
	kssl_key_t *key;
	kssl_asym_method_t *am_list;
	client_key_exchange_t *cke;

	/* Decrypt Pre Master Secret */
	cke = &(cr->msg->data.handshake.body.client_key_exchange);

	kssl_daemon_get_read(cr->conn->daemon);
	key = &(cr->conn->daemon->key);
	am_list = cr->conn->daemon->am;
	kssl_daemon_put_read(cr->conn->daemon);

	if (cr->conn->daemon->key.type != kssl_key_type_rsa) {
		KSSL_DEBUG(3, "kssl_asym_decrypt: "
				"no certificate / non RSA key\n");
		return -EEXIST;
	}

	/* Sometimes, for instance if you follow another link
	 * while images are still loading, MSIE sends am empty
	 * pre master secret. It then gets really confused when
	 * an error is picked up when the asym routines try
	 * to decrypt a zero length buffer and the connection
	 * is subsequently removed.
	 *
	 * I consider this a bug in MSIE untill otherwise is
	 * explained to me. So just check the length and bail-out
	 * if it is zero.
	 *
	 * Tested against MSIE 5.2 for Mac 5.2.1 and 5.2.2
	 *
	 * NTT COMWARE Corporation - 2005 */
	if (!cke->exchange_keys.rsa.len) {
		KSSL_DEBUG(3, "kssl_asym_decrypt: premaster secret is "
				"zero bytes long\n");
		return -EEXIST;
	}

	while (*am_list != kssl_asym_method_none) {
		status = __kssl_asym_decrypt(cke, key, out_buf, out_buf_len,
				*am_list);
		if (status >= 0)
			break;
		am_list++;
	}

	return status;
}


