/**********************************************************************
 * daemon.h                                                 August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifndef KSSLD_DAEMON_H
#define KSSLD_DAEMON_H

#include <asm/uaccess.h>

#include "types/cipher_suite_t.h"

#include "kssl.h"

#define KSSL_DAEMON_RUNNING   0x00000001
#define KSSL_DAEMON_STOPPED   0x00000002
#define KSSL_DAEMON_QUIESCENT 0x00000004

typedef struct {
	struct iovec cert;
} kssl_cert_t;

typedef struct {
	struct iovec n;
	struct iovec e;
	struct iovec d;
	struct iovec p;
	struct iovec q;
	struct iovec dmp1;
	struct iovec dmq1;
	struct iovec iqmp;
} kssl_key_rsa_t;

typedef struct {
	struct iovec p;
	struct iovec q;
	struct iovec g;
	struct iovec priv_key;
	struct iovec pub_key;
} kssl_key_dsa_t;

typedef struct {
	struct iovec p;
	struct iovec g;
} kssl_key_dh_t;

typedef struct {
	kssl_key_type_t type;
	union {
		kssl_key_rsa_t rsa;
		kssl_key_dsa_t dsa;
		kssl_key_dh_t  dh;
	} key;
} kssl_key_t;

typedef struct {
	u32 vaddr;
	u16 vport;
	u32 raddr;
	u16 rport;
	kssl_cert_t cert;
	u32 state;
	kssl_daemon_mode_t mode;
	struct socket *sock;
	kssl_key_t key;
	rwlock_t lock;
	atomic_t users;
	cipher_suite_t *cs;
	kssl_asym_method_t *am;
	struct list_head list;
} kssl_daemon_t;

kssl_daemon_t *
kssl_daemon_new(const u32 vaddr, const u16 vport);

void kssl_daemon_destroy(kssl_daemon_t *daemon);

kssl_daemon_t * kssl_daemon_create(void);  

int
kssl_daemon_foreach(int (*action)(kssl_daemon_t *daemon, void *data),
                void *data);

kssl_daemon_t * kssl_daemon_find(u32 vaddr, u16 vport);

#define kssl_daemon_get_read(daemon) read_lock(&(daemon)->lock)
#define kssl_daemon_put_read(daemon) read_unlock(&(daemon)->lock)

#define kssl_daemon_get_write(daemon) write_lock(&(daemon)->lock)
#define kssl_daemon_put_write(daemon) write_unlock(&(daemon)->lock)

#define kssl_daemon_get(daemon) atomic_inc(&(daemon)->users)
#define kssl_daemon_put(daemon) atomic_dec(&(daemon)->users)

void kssl_daemon_cpy_cert(kssl_daemon_t *dest, kssl_daemon_t *src);

void
kssl_cert_destroy_data(kssl_cert_t *cert);

void
kssl_key_destroy_data(kssl_key_t *key);

u8 * kssl_key_get_private_buf(kssl_key_t *key);

static inline kssl_daemon_mode_t
kssl_daemon_get_mode(kssl_daemon_t *daemon)
{
	kssl_daemon_mode_t mode;

	kssl_daemon_get_read(daemon);
	mode = daemon->mode;
	kssl_daemon_put_read(daemon);

	return mode;
}

extern struct list_head kssl_daemon_list;

int kssl_daemon_cipher_suite_set_cpy(kssl_daemon_t *daemon, 
		const cipher_suite_t *cs_list, size_t bytes);

cipher_suite_t * kssl_daemon_cipher_suite_find_list(kssl_daemon_t *daemon,
		cipher_suite_t *cs_list, size_t bytes);

cipher_suite_t *
kssl_daemon_cipher_suite_find(kssl_daemon_t *daemon, cipher_suite_t *cs);

kssl_daemon_t *
kssl_daemon_find_or_create_and_add(u32 vaddr, u16 vport);

size_t kssl_daemon_asym_method_list_count(const kssl_asym_method_t *am_list);

size_t kssl_daemon_asym_method_list_ncount(const kssl_asym_method_t *am_list,
		size_t bytes);

int kssl_daemon_asym_method_set_cpy(kssl_daemon_t *daemon, 
		const kssl_asym_method_t *am_list, size_t noam);

void kssl_daemon_list_add(kssl_daemon_t *daemon);

int kssl_daemon_list_for_each(int (*func)(kssl_daemon_t *daemon, void *data), 
		void *data);

int __init kssl_daemon_init(void);

void __exit kssl_daemon_cleanup(void);

#endif /* KSSLD_DAEMON_H */
