/**********************************************************************
 * to_kernel.c                                              August 2005
 *
 * KSSLD(key_tool): An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <openssl/ssl.h>
#include <vanessa_logger.h>
#include <vanessa_socket.h>

#include "types/cipher_suite_t.h"

#include "log.h"
#include "kssl.h"
#include "key_tool.h"
#include "cipher.h"
#include "asym_method.h"

static const char *evp_pkey_type_str(int type);

#ifdef WITH_DH_DSA_SUPPORT
static int
write_dh_param(BIO *bio_out, DH *dh_param);
#endif

static int write_key(BIO *bio_out, EVP_PKEY *key, int cmd);

static int write_cert(BIO *bio_out, FILE *cert_fp);

static int 
key_tool_to_kernel_key(int cmd, const char *vserver, const char *key_filename, 
		const char *cert_filename);

static int 
key_tool_to_kernel_real(const char *vserver, const char *rserver);

static int 
key_tool_to_kernel_ciphers(const char *vserver, const char *cipher_str);

static int 
key_tool_to_kernel_mode(const char *vserver, const char *mode_str);

static int 
key_tool_to_kernel_asym_methods(const char *vserver, 
		const char *asym_methods_str);

static int 
key_tool_to_kernel_add_del_daemon(const char *vserver, int opname);


int 
key_tool_to_kernel(int cmd, const char *arg_1, const char *arg_2,
		const char *arg_3)
{
	switch(cmd) {
		case CMD_RSA:
		case CMD_DSA:
		case CMD_DH:
			return key_tool_to_kernel_key(cmd, arg_1, arg_2,
					arg_3);
		case CMD_REAL:
			return key_tool_to_kernel_real(arg_1, arg_2);
		case CMD_CIPHERS:
			return key_tool_to_kernel_ciphers(arg_1, arg_2);
		case CMD_MODE:
			return key_tool_to_kernel_mode(arg_1, arg_2);
		case CMD_ASYM_METHODS:
			return key_tool_to_kernel_asym_methods(arg_1, arg_2);
		case CMD_ADD_DAEMON:
			return key_tool_to_kernel_add_del_daemon(arg_1,
					KSSL_CTL_DAEMON_ADD);
		case CMD_DEL_DAEMON:
			return key_tool_to_kernel_add_del_daemon(arg_1,
					KSSL_CTL_DAEMON_DEL);
		case CMD_FLUSH:
			return key_tool_to_kernel_flush();
	}

	VANESSA_LOGGER_DEBUG("Unknown command\n");
	return -1;
}


static int 
key_tool_to_kernel_key(int cmd, const char *vserver, const char *key_filename, 
		const char *cert_filename)
{
	FILE *cert_fp = NULL;
	FILE *key_fp = NULL;
	X509 *cert = NULL;
	EVP_PKEY *key = NULL;
	DH *dh_param = NULL;
	int status = -1;
	int tmp_status = -1;
	kssl_ctl_t ctl;
	int sock_fd = -1;
	BIO *bio_out = NULL;
	long len_out;
	char *buf_out;

	/* Setup bio for output */
	bio_out = BIO_new(BIO_s_mem());
        if (!bio_out)
                goto leave;

	/* Write header */
	if (key_tool_write_head(&ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_to_kernel_key: "
				"key_tool_write_head");
		return -1;
	}

	if(BIO_write(bio_out, &ctl, sizeof(kssl_ctl_t)) != 
			sizeof(kssl_ctl_t)) {
		VANESSA_LOGGER_DEBUG("BIO_write");
		VANESSA_LOGGER_ERR("Could not write header");
		goto leave;
	}

	/* Read Private Key */
	key_fp = fopen(key_filename, "r");
	if (!key_fp) {
		VANESSA_LOGGER_DEBUG_ERRNO("fopen");
		VANESSA_LOGGER_ERR_UNSAFE("Error opening key file: "
				"\"%s\"", key_filename);
		goto leave;
	}
 
	if (cmd != CMD_DH) {
		key = PEM_read_PrivateKey(key_fp, NULL, NULL, NULL);
		if (!key) {
			KSSLD_KEY_TOOL_DEBUG_SSL_ERR("PEM_read_PrivateKey");
			VANESSA_LOGGER_ERR("Error loading key file");
			VANESSA_LOGGER_ERR("If it is a DH parameters file "
				"a certificate file must not be specified");
			goto leave;
		}
	}
#ifdef WITH_DH_DSA_SUPPORT
	else {
		dh_param = PEM_read_DHparams(key_fp, NULL, NULL, NULL);
		if (!dh_param) {
			KSSLD_KEY_TOOL_DEBUG_SSL_ERR("PEM_read_DH");
			VANESSA_LOGGER_ERR("Error loading parameters file");
			VANESSA_LOGGER_ERR("If it is an RSA or DSA key file "
				"a certificate file must be specified");
			goto leave;
		}
	}
#endif /* WITH_DH_DSA_SUPPORT */

	fclose(key_fp);
	key_fp = NULL;

	/* Open, read and verify certificate syntax */
	if (cmd != CMD_DH) {
		cert_fp = fopen(cert_filename, "r");
		if (!cert_fp) {
			VANESSA_LOGGER_DEBUG_ERRNO("fopen");
			VANESSA_LOGGER_ERR_UNSAFE("Error opening cert file: "
					"\"%s\"", cert_filename);
			goto leave;
		}

		cert = PEM_read_X509(cert_fp, NULL, NULL, NULL);
		if (!cert) {
			KSSLD_KEY_TOOL_DEBUG_SSL_ERR("PEM_read_PrivateKey");
			VANESSA_LOGGER_ERR("Error loading cert file");
			goto leave;
		}

		X509_free(cert);
		cert = NULL;
	}

	/* Write Key */
	if (key) {
		if(write_key(bio_out, key, cmd) < 0) {
			VANESSA_LOGGER_DEBUG("write_key");
			VANESSA_LOGGER_ERR("Error processing key");
			goto leave;
		}
	}
#ifdef WITH_DH_DSA_SUPPORT
	else {
		if(write_dh_param(bio_out, dh_param) < 0) {
			VANESSA_LOGGER_DEBUG("write_key_dh");
			VANESSA_LOGGER_ERR("Error processing key");
			goto leave;
		}
	}
#endif /* WITH_DH_DSA_SUPPORT */

	/* Write Certificate */
	if(cmd != CMD_DH) {
		rewind(cert_fp);
		if (write_cert(bio_out, cert_fp) < 0) {
			VANESSA_LOGGER_DEBUG("write_cert");
			VANESSA_LOGGER_ERR("Could not write certificate");
			goto leave;	
		}

		tmp_status = fclose(cert_fp);
		cert_fp = NULL;
	}
 
	/* Send Key and Certificate to Kernel */
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		goto leave;
	}

	len_out = BIO_get_mem_data(bio_out, &buf_out);
	if (!len_out) {
		VANESSA_LOGGER_ERR("No data");
		goto leave;
	}
	if (setsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_CERT_KEY, buf_out, 
				len_out) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("setsockopt");
		VANESSA_LOGGER_ERR("Error sending data to kernel");
		goto leave;
	}

	status = 0;
leave:
	if (bio_out)
		BIO_free(bio_out);
	if (sock_fd >= 0)
		tmp_status = close(sock_fd);
	if (key_fp)
		tmp_status = fclose(key_fp);
	if (cert_fp)
		tmp_status = fclose(cert_fp);
	if (cert)
		X509_free(cert);
	if (key)
		EVP_PKEY_free(key);
	if (dh_param)
		DH_free(dh_param);
	
	return status;
}


static int 
key_tool_to_kernel_real(const char *vserver, const char *rserver)
{
	kssl_ctl_t ctl;
	int sock_fd;
	struct sockaddr_in raddr;
	char buf[sizeof(kssl_ctl_t) + sizeof(uint32_t) + sizeof(uint16_t)];

	/* Write header */
	if (key_tool_write_head(&ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_to_kernel_real: "
				"key_tool_write_head");
		return -1;
	}

	/* Write body */
	if (key_tool_parse_server(rserver, &raddr) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_to_kernel_real: "
				"key_tool_parse_server: real server");
		return -1;
	}
	memcpy(buf, &ctl, sizeof(kssl_ctl_t));
	memcpy(buf + sizeof(kssl_ctl_t), &(raddr.sin_addr.s_addr), 
			sizeof(uint32_t));
	memcpy(buf + sizeof(kssl_ctl_t) + sizeof(uint32_t), &(raddr.sin_port), 
			sizeof(uint16_t));
 
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		return -1;
	}

	if (setsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_R_IP_PORT, buf, 
				sizeof(kssl_ctl_t) + sizeof(uint32_t) + 
				sizeof(uint16_t)) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("setsockopt");
		VANESSA_LOGGER_ERR("Error sending data to kernel");
		close(sock_fd);
		return -1;
	}

	close(sock_fd);
	return 0;
}


static int 
key_tool_to_kernel_ciphers(const char *vserver, const char *cipher_str)
{
	int status = -1;
	int tmp_status = -1;
	kssl_ctl_t ctl;
	int sock_fd = -1;
	BIO *bio_out = NULL;
	long len_out;
	char *buf_out;
	cipher_suite_t *cs_list;
	size_t nocs;

	/* Setup bio for output */
	bio_out = BIO_new(BIO_s_mem());
        if (!bio_out)
                goto leave;

	/* Write header */
	if (key_tool_write_head(&ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_to_kernel_ciphers: "
				"key_tool_write_head");
		return -1;
	}

	if (BIO_write(bio_out, &ctl, sizeof(kssl_ctl_t)) != 
			sizeof(kssl_ctl_t)) {
		VANESSA_LOGGER_DEBUG("BIO_write 1");
		VANESSA_LOGGER_ERR("Could not write header");
		goto leave;
	}

	if (cipher_byname_list_alloc(cipher_str, ',', &cs_list, &nocs) < 0) {
		VANESSA_LOGGER_DEBUG("cipher_byname_list_alloc");
		VANESSA_LOGGER_ERR("could not parse ciphers");
		goto leave;
	}

	if(BIO_write(bio_out, cs_list, (nocs +1) * 
				sizeof(cipher_suite_t)) != 
			(nocs + 1) * sizeof(cipher_suite_t)) {
		VANESSA_LOGGER_DEBUG("BIO_write 2");
		VANESSA_LOGGER_ERR("Could not write ciphers");
		goto leave;
	}
 
	/* Send Key and Certificate to Kernel */
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		goto leave;
	}

	len_out = BIO_get_mem_data(bio_out, &buf_out);
	if (!len_out) {
		VANESSA_LOGGER_ERR("No data");
		goto leave;
	}
	if (setsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_CIPHERS, buf_out, 
				len_out) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("setsockopt");
		VANESSA_LOGGER_ERR("Error sending data to kernel");
		goto leave;
	}

	status = 0;
leave:
	if (bio_out)
		BIO_free(bio_out);
	if (sock_fd >= 0)
		tmp_status = close(sock_fd);
	
	return status;
}


static int 
key_tool_to_kernel_mode(const char *vserver, const char *mode_str)
{
	kssl_ctl_t ctl;
	int sock_fd;
	kssl_daemon_mode_t mode;
	char buf[sizeof(kssl_ctl_t) + sizeof(uint32_t)];

	if (!strcasecmp(mode_str, "start")) {
		mode = htonl(kssl_daemon_mode_running);
	}
	else if (!strcasecmp(mode_str, "stop")) {
		mode = htonl(kssl_daemon_mode_stopped);
	}
	else if (!strcasecmp(mode_str, "quiescent")) {
		mode = htonl(kssl_daemon_mode_quiescent);
	}
	else {
		VANESSA_LOGGER_ERR_UNSAFE("Unknown mode: \"%s\"", mode_str);
		return -1;
	}

	/* Write header */
	if (key_tool_write_head(&ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_to_kernel_mode: "
				"key_tool_write_head");
		return -1;
	}

	/* Write body */
	memcpy(buf, &ctl, sizeof(kssl_ctl_t));
	memcpy(buf + sizeof(kssl_ctl_t), &mode, sizeof(uint32_t));
 
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		return -1;
	}

	if (setsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_DAEMON_MODE, buf, 
				sizeof(kssl_ctl_t) + sizeof(uint32_t)) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("setsockopt");
		VANESSA_LOGGER_ERR("Error sending data to kernel");
		close(sock_fd);
		return -1;
	}

	close(sock_fd);
	return 0;
}


static int 
key_tool_to_kernel_asym_methods(const char *vserver, 
		const char *asym_method_str)
{
	int status = -1;
	int tmp_status = -1;
	kssl_ctl_t ctl;
	int sock_fd = -1;
	BIO *bio_out = NULL;
	long len_out;
	char *buf_out;
	kssl_asym_method_t *am_list;
	size_t noam;

	/* Setup bio for output */
	bio_out = BIO_new(BIO_s_mem());
        if (!bio_out)
                goto leave;

	/* Write header */
	if (key_tool_write_head(&ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_to_kernel_asym_methods: "
				"key_tool_write_head");
		return -1;
	}

	if (BIO_write(bio_out, &ctl, sizeof(kssl_ctl_t)) != 
			sizeof(kssl_ctl_t)) {
		VANESSA_LOGGER_DEBUG("BIO_write 1");
		VANESSA_LOGGER_ERR("Could not write header");
		goto leave;
	}

	if (asym_method_byname_list_alloc(asym_method_str, ',', 
				&am_list, &noam) < 0) {
		VANESSA_LOGGER_DEBUG("asym_method_byname_list_alloc");
		VANESSA_LOGGER_ERR("could not parse asym methods");
		goto leave;
	}

	if(BIO_write(bio_out, am_list, (noam +1) * 
				sizeof(kssl_asym_method_t)) != 
			(noam + 1) * sizeof(kssl_asym_method_t)) {
		VANESSA_LOGGER_DEBUG("BIO_write 2");
		VANESSA_LOGGER_ERR("Could not write ciphers");
		goto leave;
	}
 
	/* Send Key and Certificate to Kernel */
	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		goto leave;
	}

	len_out = BIO_get_mem_data(bio_out, &buf_out);
	if (!len_out) {
		VANESSA_LOGGER_ERR("No data");
		goto leave;
	}
	if (setsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_ASYM_METHODS, buf_out, 
				len_out) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("setsockopt");
		VANESSA_LOGGER_ERR("Error sending data to kernel");
		goto leave;
	}

	status = 0;
leave:
	if (bio_out)
		BIO_free(bio_out);
	if (sock_fd >= 0)
		tmp_status = close(sock_fd);
	
	return status;
}


static int 
key_tool_to_kernel_add_del_daemon(const char *vserver, int opname)
{
	kssl_ctl_t ctl;
	int sock_fd;
	kssl_daemon_mode_t mode;

	/* Write header */
	if (key_tool_write_head(&ctl, vserver) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_to_kernel_mode: "
				"key_tool_write_head");
		return -1;
	}

	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		return -1;
	}

	if (setsockopt(sock_fd, IPPROTO_IP, opname, &ctl, 
				sizeof(kssl_ctl_t)) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("setsockopt");
		VANESSA_LOGGER_ERR("Error sending data to kernel");
		close(sock_fd);
		return -1;
	}

	close(sock_fd);
	return 0;
}




static int 
key_tool_to_kernel_flush(void)
{
	kssl_ctl_t ctl;
	int sock_fd;
	kssl_daemon_mode_t mode;

	/* Write header */
	if (key_tool_write_head(&ctl, NULL) < 0) {
		VANESSA_LOGGER_DEBUG("key_tool_to_kernel_mode: "
				"key_tool_write_head");
		return -1;
	}

	sock_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock_fd == -1) {
		VANESSA_LOGGER_DEBUG_ERRNO("socket");
		VANESSA_LOGGER_ERR("Error opening socket");
		return -1;
	}

	if (setsockopt(sock_fd, IPPROTO_IP, KSSL_CTL_FLUSH, &ctl, 
				sizeof(kssl_ctl_t)) < 0) {
		VANESSA_LOGGER_DEBUG_ERRNO("setsockopt");
		VANESSA_LOGGER_ERR("Error sending data to kernel");
		close(sock_fd);
		return -1;
	}

	close(sock_fd);
	return 0;
}


static const char 
*evp_pkey_type_str(int type)
{
	switch (type) {
		case EVP_PKEY_RSA:
			return KSSL_KEY_TYPE_RSA;
#ifdef WITH_DH_DSA_SUPPORT
		case EVP_PKEY_DSA:
			return KSSL_KEY_TYPE_DSA;
		case EVP_PKEY_DH:
			return KSSL_KEY_TYPE_DH;
#endif /* WITH_DH_DSA_SUPPORT */
#ifdef EVP_PKEY_EC
		case EVP_PKEY_EC:
			return "EC";
#endif
		case NID_undef:
			return "undefined";
	}

	return "unknown";
}


#define WRITE_BUF(bio, buf, size)                                             \
do {                                                                          \
	size_t tmp_size = size;                                               \
	size_t tmp_size_n = htonl(tmp_size);                                  \
	if (BIO_write(bio, &tmp_size_n, sizeof(uint32_t))                     \
			!= sizeof(uint32_t)) {                                \
		VANESSA_LOGGER_DEBUG("BIO_write");                            \
		goto leave;                                                   \
	}                                                                     \
	if (BIO_write(bio, buf, tmp_size) != tmp_size) {                      \
		VANESSA_LOGGER_DEBUG("BN_write");                             \
		goto leave;                                                   \
	}                                                                     \
} while(0)

#define WRITE_BN(bio, bn)                                                     \
do {                                                                          \
	int bytes;                                                            \
	uint32_t size;                                                        \
	bytes = BN_num_bytes(bn);                                             \
	if (bytes < 0 || bytes > (uint32_t)(~0U>>1)) {                        \
		VANESSA_LOGGER_DEBUG("Invalid number of bytes");              \
		goto leave;                                                   \
	}                                                                     \
	size = htonl(bytes);                                                  \
	if (BIO_write(bio, &size, sizeof(uint32_t)) != sizeof(uint32_t)) {    \
		VANESSA_LOGGER_DEBUG("BIO_write");                            \
		goto leave;                                                   \
	}                                                                     \
	if (bytes > buf_len) {                                                \
		buf = realloc(buf, bytes);                                    \
		if (!buf) {                                                   \
			VANESSA_LOGGER_DEBUG_ERRNO("realloc");                \
			goto leave;                                           \
		}                                                             \
		buf_len = bytes;                                              \
	}                                                                     \
	BN_bn2bin(bn, buf);                                                   \
	if(BIO_write(bio, buf, bytes) != bytes) {                             \
		VANESSA_LOGGER_DEBUG("BIO_write");                            \
		goto leave;                                                   \
	}                                                                     \
} while(0)

static int 
write_rsa_key(BIO *out_bio, EVP_PKEY *key)
{
	RSA *rsa_key = NULL;
	int status = -1;
	unsigned char *buf = NULL;
	size_t buf_len = 0;

	rsa_key = EVP_PKEY_get1_RSA(key);
	if (!rsa_key) {
		VANESSA_LOGGER_DEBUG("EVP_PKEY_get1_RSA");
		goto leave;
	}

	if (!rsa_key->n || !rsa_key->e || !rsa_key->d || !rsa_key->p ||
			!rsa_key->q || !rsa_key->dmp1 || !rsa_key->dmq1 ||
			!rsa_key->iqmp) {
		VANESSA_LOGGER_DEBUG("Values missing in RSA key\n");
		goto leave;
	}

	WRITE_BUF(out_bio, KSSL_KEY_TYPE_RSA, strlen(KSSL_KEY_TYPE_RSA));

	WRITE_BN(out_bio, rsa_key->n);
	WRITE_BN(out_bio, rsa_key->e);
	WRITE_BN(out_bio, rsa_key->d);
	WRITE_BN(out_bio, rsa_key->p);
	WRITE_BN(out_bio, rsa_key->q);
	WRITE_BN(out_bio, rsa_key->dmp1);
	WRITE_BN(out_bio, rsa_key->dmq1);
	WRITE_BN(out_bio, rsa_key->iqmp);

	status = 0;
leave:
	if (buf)
		free(buf);
	if (rsa_key)
		RSA_free(rsa_key);
	return status;
}


#ifdef WITH_DH_DSA_SUPPORT
static int 
write_dsa_key(BIO *out_bio, EVP_PKEY *key)
{
	DSA *dsa_key;
	int status = -1;
	unsigned char *buf = NULL;
	size_t buf_len = 0;

	dsa_key = EVP_PKEY_get1_DSA(key);
	if (!dsa_key) {
		VANESSA_LOGGER_DEBUG("EVP_PKEY_get1_DSA");
		return -1;
	}

	if (!dsa_key->p || !dsa_key->q || !dsa_key->g || 
			!dsa_key->priv_key || !dsa_key->pub_key) {
		VANESSA_LOGGER_DEBUG("Values missing in DSA key\n");
		return -1;
	}

	WRITE_BUF(out_bio, KSSL_KEY_TYPE_DSA, strlen(KSSL_KEY_TYPE_DSA));

	WRITE_BN(out_bio, dsa_key->p);
	WRITE_BN(out_bio, dsa_key->q);
	WRITE_BN(out_bio, dsa_key->g);
	WRITE_BN(out_bio, dsa_key->priv_key);
	WRITE_BN(out_bio, dsa_key->pub_key);

	status = 0;
leave:
	if (buf)
		free(buf);
	if (dsa_key)
		DSA_free(dsa_key);
	return status;
}


static int 
write_dh_param(BIO *out_bio, DH *dh_param)
{
	int status = -1;
	unsigned char *buf = NULL;
	size_t buf_len = 0;

	if (!dh_param->p || !dh_param->g) {
		VANESSA_LOGGER_DEBUG("Values missing in DH parameters\n");
		return -1;
	}

	WRITE_BUF(out_bio, KSSL_KEY_TYPE_DH, strlen(KSSL_KEY_TYPE_DH));

	WRITE_BN(out_bio, dh_param->p);
	WRITE_BN(out_bio, dh_param->g);

	status = 0;
leave:
	if (buf)
		free(buf);
	return status;
}
#endif /* WITH_DH_DSA_SUPPORT */


static int 
write_key(BIO *out_bio, EVP_PKEY *key, int cmd)
{
	int type;

	type = EVP_PKEY_type(key->type);

	if (cmd == CMD_RSA) {
		if (type != EVP_PKEY_RSA) {
			VANESSA_LOGGER_DEBUG_UNSAFE("RSA key requested, "
					"but kernel returned a %s key\n",
					evp_pkey_type_str(type));
			return -1;
		}
		return(write_rsa_key(out_bio, key));
	}
#ifdef WITH_DH_DSA_SUPPORT
	else {
		if (type != EVP_PKEY_DSA) {
			VANESSA_LOGGER_DEBUG_UNSAFE("DSA key requested, "
					"but %s key supplied\n",
					evp_pkey_type_str(type));
			return -1;
		}
		return(write_dsa_key(out_bio, key));
	}
#endif

	return -1;
}

#define WRITE_CERT_BLOCK 4096

static int
write_cert(BIO *out_bio, FILE *cert_fp)
{
	int bytes;
	int status = -1;
	BIO *in = NULL;
	BIO *b64 = NULL;
	BIO *in_b64 = NULL;
	char *in_buf = NULL;
	size_t in_len = 0;
	size_t out_len = 0;

	b64 = BIO_new(BIO_f_base64());
	if (!b64) {
		KSSLD_KEY_TOOL_DEBUG_SSL_ERR("BIO_f_base64");
		goto leave;
	}

	in = BIO_new_fp(cert_fp, BIO_NOCLOSE);
	if (!in) {
		VANESSA_LOGGER_DEBUG("BIO_new_fp");
		goto leave;
	}

	in_b64 = BIO_push(b64, in);

	while (1) {
		if(in_len == out_len) {
			in_len += WRITE_CERT_BLOCK;
			in_buf = realloc(in_buf, in_len);
			if (!in_buf)
				goto leave;
		}

		bytes = BIO_read(in_b64, in_buf+out_len, in_len-out_len);
		if(bytes == 0) {
			break;
		}
		else if(bytes < 0) {
			VANESSA_LOGGER_DEBUG("BIO_read");
			goto leave;
		}
		out_len += bytes;
	}

	WRITE_BUF(out_bio, in_buf, out_len);

	status = 0;
leave:
	if (in_buf)
		free(in_buf);
	if (in) {
		if (!BIO_free(in))
			VANESSA_LOGGER_DEBUG("BIO_free: in");
	}
	if (b64) {
		if (!BIO_free(b64))
			VANESSA_LOGGER_DEBUG("BIO_free: b64");
	}

	return status;
}
