/**********************************************************************
 * tssl3mac.c                                               August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based on tcrypt.c from Linux 2.4.73
 * Copyright (c) 2002 James Morris <jmorris@intercode.com.au>
 * Copyright (c) 2002 Jean-Francois Dive <jef@linuxbe.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include <linux/init.h>
#include <linux/module.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <asm/scatterlist.h>
#include <linux/string.h>
#include <linux/crypto.h>
#include <linux/highmem.h>
#include "kssl/vectors/ssl3mac.h"


/*
 * Need to kmalloc() memory for testing kmap().
 */
#define TVMEMSIZE	4096
#define XBUFSIZE	32768

/*
 * Indexes into the xbuf to simulate cross-page access.
 */
#define IDX1		37
#define IDX2		32400

static int mode = 0;
static char *xbuf;
static char *tvmem;

static void
hexdump(unsigned char *buf, unsigned int len)
{
	while (len--)
		printk("%02x", *buf++);

	printk("\n");
}

static void
__test_ssl3mac_md5(struct crypto_tfm *tfm, const char *tag, 
		struct ssl3mac_md5_testvec *tv, unsigned int tsize,
		unsigned int count)
{
	char *p;
	unsigned int i, klen;
	struct scatterlist sg[2];
	char result[128];
	struct ssl3mac_md5_testvec *tv_cpy;

	if (tsize > TVMEMSIZE) {
		printk("template (%u) too big for tvmem (%u)\n", tsize,
		       TVMEMSIZE);
		return;
	}

	memcpy(tvmem, tv, tsize);
	tv_cpy = (void *) tvmem;

	for (i = 0; i < count; i++) {
		printk("test %s%u:\n", tag, i + 1);
		memset(result, 0, sizeof (result));

		p = tv_cpy[i].plaintext;
		sg[0].page = virt_to_page(p);
		sg[0].offset = ((long) p & ~PAGE_MASK);
		sg[0].length = tv_cpy[i].plaintext_len;

		klen = tv_cpy[i].key_len;
		crypto_ssl3mac(tfm, tv_cpy[i].key, &klen, sg, 1, result);

		hexdump(result, crypto_tfm_alg_digestsize(tfm));
		printk("%s\n", memcmp(result, tv_cpy[i].digest,
			      crypto_tfm_alg_digestsize(tfm)) ? "fail" :
		       "pass");
	}
}

static void
__test_ssl3mac_md5_cross_page(struct crypto_tfm *tfm, const char *tag, 
		struct ssl3mac_md5_testvec *tv, unsigned int tsize,
		unsigned int count)
{
	char *p;
	unsigned int klen;
	struct scatterlist sg[2];
	char result[128];
	struct ssl3mac_md5_testvec *tv_cpy;

	if (tsize > TVMEMSIZE) {
		printk("template (%u) too big for tvmem (%u)\n", tsize,
		       TVMEMSIZE);
		return;
	}

	memcpy(tvmem, tv, tsize);
	tv_cpy = (void *) tvmem;

	memset(xbuf, 0, XBUFSIZE);

	memcpy(&xbuf[IDX1], "what do ya want ", 16);
	memcpy(&xbuf[IDX2], "for nothing?", 12);

	p = &xbuf[IDX1];
	sg[0].page = virt_to_page(p);
	sg[0].offset = ((long) p & ~PAGE_MASK);
	sg[0].length = 16;

	p = &xbuf[IDX2];
	sg[1].page = virt_to_page(p);
	sg[1].offset = ((long) p & ~PAGE_MASK);
	sg[1].length = 12;

	memset(result, 0, sizeof (result));
	klen = tv_cpy[count].key_len;
	crypto_ssl3mac(tfm, tv_cpy[count].key, &klen, sg, 2, result);
	hexdump(result, crypto_tfm_alg_digestsize(tfm));

	printk("%s\n", memcmp(result, tv_cpy[count].digest,
		crypto_tfm_alg_digestsize(tfm)) ? "fail" : "pass");

}

static void
test_ssl3mac_md5(void)
{
	struct crypto_tfm *tfm;

	tfm = crypto_alloc_tfm("md5", 0);
	if (tfm == NULL) {
		printk("failed to load transform for md5\n");
		return;
	}

	printk("\ntesting ssl3mac_md5\n");


	__test_ssl3mac_md5(tfm, "1.", ssl3mac_md5_tv_template1,
			sizeof(ssl3mac_md5_tv_template1),
			SSL3MAC_MD5_TEST_VECTORS1);
	__test_ssl3mac_md5(tfm, "2.", ssl3mac_md5_tv_template2,
			sizeof(ssl3mac_md5_tv_template2),
			SSL3MAC_MD5_TEST_VECTORS2);
	__test_ssl3mac_md5(tfm, "3.", ssl3mac_md5_tv_template3,
			sizeof(ssl3mac_md5_tv_template3),
			SSL3MAC_MD5_TEST_VECTORS3);
	
	printk("\ntesting ssl3mac_md5 across pages\n");

	__test_ssl3mac_md5_cross_page(tfm, "3.", ssl3mac_md5_tv_template1,
			sizeof(ssl3mac_md5_tv_template1),
			SSL3MAC_MD5_TEST_VECTORS1);

	crypto_free_tfm(tfm);
}


static void
__test_ssl3mac_sha1(struct crypto_tfm *tfm, const char *tag, 
		struct ssl3mac_sha1_testvec *tv, unsigned int tsize,
		unsigned int count)
{
	char *p;
	unsigned int i, klen;
	struct scatterlist sg[2];
	char result[128];
	struct ssl3mac_sha1_testvec *tv_cpy;

	if (tsize > TVMEMSIZE) {
		printk("template (%u) too big for tvmem (%u)\n", tsize,
		       TVMEMSIZE);
		return;
	}

	memcpy(tvmem, tv, tsize);
	tv_cpy = (void *) tvmem;

	for (i = 0; i < count; i++) {
		printk("test %s%u:\n", tag, i + 1);
		memset(result, 0, sizeof (result));

		p = tv_cpy[i].plaintext;
		sg[0].page = virt_to_page(p);
		sg[0].offset = ((long) p & ~PAGE_MASK);
		sg[0].length = tv_cpy[i].plaintext_len;

		klen = tv_cpy[i].key_len;
		crypto_ssl3mac(tfm, tv_cpy[i].key, &klen, sg, 1, 
				result);

		hexdump(result, crypto_tfm_alg_digestsize(tfm));
		printk("%s\n", memcmp(result, tv_cpy[i].digest,
			      crypto_tfm_alg_digestsize(tfm)) ? "fail" :
		       "pass");
	}
}

static void
__test_ssl3mac_sha1_cross_page(struct crypto_tfm *tfm, const char *tag, 
		struct ssl3mac_sha1_testvec *tv, unsigned int tsize,
		unsigned int count)
{
	char *p;
	unsigned int klen;
	struct scatterlist sg[2];
	char result[128];
	struct ssl3mac_sha1_testvec *tv_cpy;

	if (tsize > TVMEMSIZE) {
		printk("template (%u) too big for tvmem (%u)\n", tsize,
		       TVMEMSIZE);
		return;
	}

	memcpy(tvmem, tv, tsize);
	tv_cpy = (void *) tvmem;

	memset(xbuf, 0, XBUFSIZE);

	memcpy(&xbuf[IDX1], "what do ya want ", 16);
	memcpy(&xbuf[IDX2], "for nothing?", 12);

	p = &xbuf[IDX1];
	sg[0].page = virt_to_page(p);
	sg[0].offset = ((long) p & ~PAGE_MASK);
	sg[0].length = 16;

	p = &xbuf[IDX2];
	sg[1].page = virt_to_page(p);
	sg[1].offset = ((long) p & ~PAGE_MASK);
	sg[1].length = 12;

	memset(result, 0, sizeof (result));
	klen = tv_cpy[count].key_len;
	crypto_ssl3mac(tfm, tv_cpy[count].key, &klen, sg, 2, result);
	hexdump(result, crypto_tfm_alg_digestsize(tfm));

	printk("%s\n", memcmp(result, tv_cpy[count].digest,
		crypto_tfm_alg_digestsize(tfm)) ? "fail" : "pass");

}

static void
test_ssl3mac_sha1(void)
{
	struct crypto_tfm *tfm;

	tfm = crypto_alloc_tfm("sha1", 0);
	if (tfm == NULL) {
		printk("failed to load transform for sha1\n");
		return;
	}

	printk("\ntesting ssl3mac_sha1\n");


	__test_ssl3mac_sha1(tfm, "1.", ssl3mac_sha1_tv_template1,
			sizeof(ssl3mac_sha1_tv_template1),
			SSL3MAC_SHA1_TEST_VECTORS1);
	__test_ssl3mac_sha1(tfm, "2.", ssl3mac_sha1_tv_template2,
			sizeof(ssl3mac_sha1_tv_template2),
			SSL3MAC_SHA1_TEST_VECTORS2);
	
	printk("\ntesting ssl3mac_sha1 across pages\n");

	__test_ssl3mac_sha1_cross_page(tfm, "3.", ssl3mac_sha1_tv_template1,
			sizeof(ssl3mac_sha1_tv_template1),
			SSL3MAC_SHA1_TEST_VECTORS1);

	crypto_free_tfm(tfm);
}

static void
do_test(void)
{
	switch (mode) {

	case 0:
		test_ssl3mac_md5();
		test_ssl3mac_sha1();
		break;

	case 200:
		test_ssl3mac_md5();
		break;
		
	case 201:
		test_ssl3mac_sha1();
		break;

	default:
		/* useful for debugging */
		printk("not testing anything\n");
		break;
	}
}

static int __init
init(void)
{
	tvmem = kmalloc(TVMEMSIZE, GFP_KERNEL);
	if (tvmem == NULL)
		return -ENOMEM;

	xbuf = kmalloc(XBUFSIZE, GFP_KERNEL);
	if (xbuf == NULL) {
		kfree(tvmem);
		return -ENOMEM;
	}

	do_test();

	kfree(xbuf);
	kfree(tvmem);
	return 0;
}

module_init(init);

MODULE_PARM(mode, "i");

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Test for kernel crypto ssl3mac");
MODULE_AUTHOR("NTT COMWARE Corporation");

