/**********************************************************************
 * unsx.h                                                   August 2005
 *
 * ASYM: An implementation of Asymetric Cryptography in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifndef UNSX_H
#define UNSX_H

#ifdef __KERNEL__
#include <linux/slab.h>
#include <linux/string.h>
#else
#include "compat.h"
#endif


/* Jean-Luc Cooke <jlcooke@certainkey.com> */

/* make this non 32bit specific  :P */
typedef unsigned char uns8;
typedef unsigned short uns16;
typedef unsigned int uns32;
typedef unsigned long long uns64;
typedef uns32 unsx;
typedef uns64 uns2x;

#define uns8_Max\
  0xff
#define uns16_Max\
  0xffff
#define uns32_Max\
  0xffffffff
#define uns64_Max\
  0xffffffffffffffff
#define unsx_Max\
  uns32_Max
#define uns2x_Max\
  uns64_Max

#define uns8_log\
  3
#define uns16_log\
  4
#define uns32_log\
  5
#define uns64_log\
  6
#define unsx_log\
  uns32_log
#define uns2x_log\
  uns64_log

#define uns32_Bytes_Per\
  sizeof(unsx)
#define unsx_Bit_Length\
  (8 * uns32_Bytes_Per)

#define MAX(X,Y)\
  ( ((X)<(Y)) ? (Y) : (X) )
 
/*! pointer to a top-level stack item to measure how deep our rabit hole goes */
extern uns32 unsx_stack;
/*! the maximum discoverd depth of our rabit hole */
extern uns32 unsx_stackMaxDepth;
 
/*****************************************************************
**  NOTICE!!! YOU MUST DO THIS OR THINGS WILL GO REALLY WRONG!  **
** Before any UNSX fcns are called, "unsxInit();" must be run. **
*****************************************************************/
extern int unsx_init(int bitLen);
extern void unsx_deinit(void);

#define UNSX_SAFE_FREE(ptr) \
if (ptr) {\
  kfree(ptr);\
  (ptr)=NULL;\
}


/*! Zero-izes any potentially sencitive data (SLIGHTLY Slower) */
#ifndef UNSX_SECURE
  #define UNSX_SECURE
#endif

/*! Use ANSI-C malloc() instead of stack space
 * saves like 70% of stack space...handy for kernel level crypto
 */
#ifndef UNSX_MALLOC
  #define UNSX_MALLOC
#endif

#ifdef UNSX_MALLOC
  #define UNSX_NEW(n,s)\
    unsx *n = (unsx*) kmalloc((s) * sizeof(unsx), GFP_KERNEL)
  #ifdef UNSX_SECURE
    #define UNSX_FREE(n,s) do {\
      memset(n, 0, (s) * sizeof(unsx));\
      kfree(n); n = NULL;\
    } while(0)
  #else
    #define UNSX_FREE(n,s) do{\
      free(n); n = NULL;\
    }while(0)
  #endif
#else
  #define UNSX_NEW(n,s)\
    unsx n[s]
  #ifdef UNSX_SECURE
    #define UNSX_FREE(n,s)\
      memset(n, 0, (s) * sizeof(unsx))
  #else
    #define UNSX_FREE(n,s)\
      /**/
  #endif
#endif

/*********************************************
**      CONFIGS SHOULD BE Dunsx_ONE BY NOW       **
**    VENTURE FORWARD AT YOUR OWN RISK      **
*********************************************/

#ifndef unsx_Trial_Prime_size
 /*! entries in unsx_Trial_Prime table used in unsx_nextPrime() */
 #define unsx_Trial_Prime_size\
   2*3*5*7*11
#endif

#ifndef unsx_Trial_Prime_Sets
 #define unsx_Trial_Prime_Sets\
   80
 /*! unsx_Trial_Prime array starts holding primes greater than this. */
 #define unsx_Trial_Prime_Sets_Start\
   11
#endif

/*! number of elements in unsx arrays
 * (maxBitLength = UNSX_LENGTH * sizeof(unsx) * 8
 */
extern int UNSX_LENGTH;
/*! unsx type of ZERO value */
extern unsx *unsx_ZERO;
/*! unsx type of ONE value */
extern unsx *unsx_ONE;
/*! unsx type of TWO value */
extern unsx *unsx_TWO;
/*! unsx type of THREE value */
extern unsx *unsx_THREE;
/*! unsx type of FIVE value */
extern unsx *unsx_FIVE;
/*! unsx type of SEVEN value */
extern unsx *unsx_SEVEN;
/*! unsx type of ELEVEN value */
extern unsx *unsx_ELEVEN;
/*! unsx type of THIRTEEN value */
extern unsx *unsx_THIRTEEN;
/*! unsx type of TWENTYTHREE value */
extern unsx *unsx_TWENTYTHREE;
/*! unsx type of 1662803 value */
extern unsx *unsx_1662803;

/*! unsx a[len], int i */
#define UNSX_TESTBIT(a,i)\
  ( ( (a)[(i)>>unsx_log]  >> ( (i) & (unsx_Bit_Length-1) ) ) & 1 )
/*! unsx a[len], int i */
#define UNSX_SETBIT(a,i)\
  ( (a)[(i)>>unsx_log] |= 1 << ( (i) & (unsx_Bit_Length-1) ) )
/*! unsx a[len], int i */
#define UNSX_CLEARBIT(a,i)\
  ( (a)[(i)>>unsx_log] &= ~( 1 << ( (i) & (unsx_Bit_Length-1) ) ) )

/*! 64bit division. unsx a,bh,hl,c;  a = bh:bl / c; */
#define UNSX_DOUBLEDIVI(a,bh,bl,c) {       \
  register uns2x tmp;                      \
  tmp  = (((uns2x)bh) << 32) | ((uns2x)bl);\
  a = (unsx) (tmp / ((uns2x)c));           \
}

/*! 64bit multiplication. unsx ah,al,b,c;  ah:al = b * c; */
#define UNSX_DOUBLEMULT(ah,al,b,c) { \
  register uns2x tmp;                \
  tmp  = ((uns2x)b) * ((uns2x)c);    \
  ah   = (unsx)(tmp >> 32);          \
  al   = (unsx)(tmp & unsx_Max);     \
}

/*! unsx a[len]; true is a is an odd number */
#define UNSX_ISODD(a,len)\
  UNSX_TESTBIT(a,0)

/*! unsx a[len]; true is a is an even number */
#define UNSX_ISEVEN(a,len)\
  (!UNSX_ISODD(a,len))

/*! unsx a[len],b[len],c,d[len]; a = b + c*d; */
extern unsx unsx_addMultSingle(unsx *a, const unsx *b, unsx c, const unsx *d, int len);

/*! unsx a[len],b[len],c,d[len]; a = b - c*d; */
extern unsx unsx_subMultSingle(unsx *a, const unsx *b, unsx c, const unsx *d, int len);

/*! unsx a[len],b; a[len-2] = b; a[i] = 0; where 0<=i; */
#define unsx_setHigh(A,VAL,LEN)\
  do { unsx_setZero(A, LEN); (A)[LEN-1] = VAL; } while(0)

/*! unsx a[len],b; a[0] = b; a[i] = 0; where 0<i; */
#define unsx_setLow(A,VAL,LEN)\
  do { unsx_setZero(A, LEN); (A)[0] = VAL; } while(0)

/*! unsx a[len],b[len],c; a[0]=b[0]=c; a[i]=b[i]=0; where 0<i; */
#define unsx_setLowDual(A,B,VAL,LEN)\
  do { unsx_setZeroDual(A, B, LEN); (A)[0] = (B)[0] = VAL; } while(0)

/*! unsx a[len],b[len]; a = b; */
#define unsx_set(A,B,LEN)\
  memcpy(A, B, (LEN)*sizeof(unsx))

/*! unsx a[len],b[len],c[len]; a=c; b=c; */
#define unsx_setDual(A,B,C,LEN)\
  do { memcpy(A, C, (LEN)*sizeof(unsx)); \
	  memcpy(B, C, (LEN)*sizeof(unsx)); } while(0)

/*! unsx a[len]; a=0; probably optimized for us. */
#define unsx_setZero(A,LEN)\
  memset(A, 0, (LEN)*sizeof(unsx))

/*! unsx a[len],b[len]; a=0; b=0; */
#define unsx_setZeroDual(A,B,LEN)\
  do { unsx_setZero(A,LEN); unsx_setZero(B,LEN); } while(0)

/*! unsx a[len]; a=1; probably optimized for us. */
#define unsx_setOne(A,LEN)\
  unsx_setLow(A, 1, LEN)

/*! unsx a[len]; a=1; probably optimized for us. */
#define unsx_setOneDual(A,B,LEN)\
  unsx_setLowDual(A, B, 1, LEN)

/*! unsx a[len]; int b; a = 2^b; */
extern void unsx_set2Exp(unsx *a, int b, int len);

/*! unsx a[len],b[len]; a = b++; */
extern unsx unsx_inc(unsx *a, const unsx *b, int len);

/*! unsx a[len],b[len],c[len]; a = b+c; */
extern unsx unsx_add(unsx *a, const unsx *b, const unsx *c, int len);

/*! unsx a[len],b[len]; a = b--; */
extern unsx unsx_dec(unsx *a, const unsx *b, int len);

/*! unsx a[len],b[len],c[len]; a = b-c; */
extern unsx unsx_sub(unsx *a, const unsx *b, const unsx *c, int len);

/*! unsx a[len],b[len],c[len]; a = b*c; */
extern int unsx_mult(unsx *a, const unsx *b, const unsx *c, int len);

/*! unsx a[cLen],b[dLen],c[cLen],d[dLen]; a = c/d; b = c%d; */
extern int unsx_divi(unsx *a, unsx *b,
                      const unsx *c, int cLen,
                      const unsx *d, int dLen);

/*! unsx a[cLen],b[bLen],c[clen]; a = b%c; */
#define unsx_mod(A, B, BLEN, C, CLEN)\
  unsx_divi(NULL, A,\
            B, BLEN,\
            C, CLEN)

/*! unsx a[len],b[len]; int c; a = b<<c; */
extern unsx unsx_shl(unsx *a, const unsx *b, int c, int len);

/*! unsx a[len],b[len]; int c; a = b>>c; */
extern unsx unsx_shr(unsx *a, const unsx *b, int c, int len);

#if 0
 /* 
  *unsx a[l],b[l],c[len>l],d[l];  a = (b^c)%d;  uses RSAREF
  */
 #define unsx_modPow(a,b,c,d,l)\
   unsx_modPowRSAREF(a,b,c,unsx_countArr(c,l),d,l)
#else
 /* 
  * unsx a[l],b[l],c[len>l],d[l];  a = (b^c)%d;  uses a basic modexp
  */
 #define unsx_modPow(a,b,c,d,l)\
   unsx_modPowJL(a,b,c,d,l)
#endif

/*! unsx a[len],b[len],c[len],d[len]; a = (b*c)%d; */
extern int unsx_modMult(unsx *a,
                         const unsx *b,
                         const unsx *c,
                         const unsx *d, int len);


/*! unsx a[len],b[len],c[len],d[len]; a = (b^c)%d; */
extern int unsx_modPowJL(unsx *a,
                          const unsx *b,
                          const unsx *c,
                          const unsx *d, int len);

/*! unsx a[len],b[len],c[len]; Finds a such that (a*b)%c = 1  or  a = (1/b) % c */
extern int unsx_modInv(unsx *a,
                        const unsx *b,
                        const unsx *c, int len);

/*! unsx a[len],b[len],c[len]; a = GreatestCommonDenominatorOf(b,c); */
extern int unsx_gcd(unsx *a,
                     const unsx *b,
                     const unsx *c, int len);

/*! unsx a[len],b[len];  1 if (a>b), 0 if (a==b), -1 if (a<b) */
extern int __inline unsx_cmp(const unsx *a, const unsx *b, int len);

/*! unsx a[len],b[len];  TRUE if (a==b) */
#define unsx_isEqual(A,B,LEN)\
 (memcmp(A,B,LEN*sizeof(unsx)) == 0)

/*! unsx a[len];         TRUE if (a==val) */
#define unsx_isLow(A,VAL,LEN)\
  ((A)[0] == VAL  &&  memcmp(&((A)[1]), unsx_ZERO, (LEN-1)*sizeof(unsx)) == 0)

/*! true is a equals unsx_ZERO */
#define unsx_isZero(A,LEN)\
  (memcmp(A, unsx_ZERO, LEN*sizeof(unsx)) == 0)

/*! true is a equals unsx_ONE */
#define unsx_isOne(a,len)\
  unsx_isLow(a,1,len)

/*! unsx a[len];         TRUE if (a==val) */
#define unsx_isLowDual(A,B,VAL,LEN)\
  ((A)[0] == VAL  &&  memcmp(&((A)[1]), unsx_ZERO, (LEN-1)*sizeof(unsx)) == 0\
 &&(B)[0] == VAL  &&  memcmp(&((B)[1]), unsx_ZERO, (LEN-1)*sizeof(unsx)) == 0)
/*
extern int unsx_isLowDual(const unsx *a, const unsx *b, unsx val, int len);
*/

/*! true is a and b equal unsx_ZERO */
#define unsx_isZeroDual(a,b,len)\
  (memcmp(A, unsx_ZERO, LEN*sizeof(unsx)) == 0\
 &&memcmp(B, unsx_ZERO, LEN*sizeof(unsx)) == 0)

/*! true is a and b equal unsx_ONE */
#define unsx_isOneDual(a,b,len)\
  unsx_isLowDual(a,b,1,len)

/*! unsx a[???]; returns index of last unsx_set bit */
extern int __inline unsx_countBits(unsx a);
/*! unsx a[len];  returns index of last non-zero UNSX-word */
extern int __inline unsx_countArr(const unsx *a, int len);
/*! unsx a[len];  returns index of last unsx_set bit */
extern int __inline unsx_countBitsArr(const unsx *a, int len);
/*! unsx a[len];  returns index of first unsx_set bit */
extern int __inline unsx_lowestSetBit(const unsx *a, int len);

/*!
** Add unsxPrim_unsx_add[N MOD 210] to send N to the next number
**   that is not unsx_divisable by 2,3,5 or 7
*/
extern unsx **unsx_Trial_Prime;
extern const uns8 unsx_Prime_unsx_add[unsx_Trial_Prime_size];

/*! unsx a[len];  TRUE if a passes MANY primality tests. */
extern int unsx_isPrimeFull(const unsx *a, int len);

/*! unsx a[len];  TRUE if a is not unsx_divisable by small primes */
#define unsx_isPrimeTrial(a,len)\
  unsx_isPrimeTrial_2to29(a,len)
/*! unsx a[len];  TRUE if a is not unsx_divisable by primes from 2 to 29 */
extern int unsx_isPrimeTrial_2to29(const unsx *a, int len);
/*! unsx a[len];  TRUE is a is not unsx_divisable by primes 11 to 31 */
extern int unsx_isPrimeTrial_11to31(const unsx *a, int len);
/*! unsx a[len];  TRUE is a is not unsx_divisable by MANY small primes */
extern int unsx_isPrimeTrialLarge(const unsx *a, int len);

/* unsx a[len]; int certanty;  FALSE if a is not prime.  TRUE if 0.5^certanty chance of not being prime */
extern int unsx_isPrimeProb(const unsx *a, int certanty, int len);

/*! unsx a[len],b[len];  TRUE if a is not pseudoprime-b */
extern int unsx_isPrimePRPn(const unsx *a, const unsx *b, int len);
/*! unsx a[len];  TRUE if a is not pesudoprime-{few numbers} */
extern int unsx_isPrimePRP(const unsx *a, int len);
/*! unsx a[len],b[len];  TRUE if a is not strong pseudoprime-b */
extern int unsx_isPrimeSPRPn(const unsx *a, const unsx *b, int len);
/*! unsx a[len];  TRUE if a is not stong pseudoprime-{few numbers} */
extern int unsx_isPrimeSPRP(const unsx *a, int len);

/*! unsx a[len];  a = next larger prime */
extern int unsx_nextPrime(unsx *a, int len);
/*! unsx a[l];  a = number at least bl bits long and prime using "unsx_isPrime(a,l)" */
extern void unsx_newPrime(unsx *a, int l, int bl);

#endif /* UNSX_H */
