/**********************************************************************
 * pk_test.c                                                August 2005
 *
 * ASYM: An implementation of Asymetric Cryptography in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifdef __KERNEL__
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/random.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/string.h>
#else
#include "compat.h"
#endif

#include "pk.h"
#include "test_util.h"

static int __pk_test(u8 *input, size_t inputLen, u8 *Public, 
		size_t PublicLen, u8 *Private, size_t PrivateLen,
		size_t bits, int algo);

int pk_test(u8 *input, size_t inputLen, size_t bits, int algo)
{
	int ret = 0;
	u8 *Public = NULL;
	size_t PublicLen;
	u8 *Private = NULL;
	size_t PrivateLen;

	u8 id[64 / 8];

	unsx_stack = (uns32) (&(ret));

	ret = pk_keygen(&Public, &PublicLen, &Private, &PrivateLen,
			     bits, algo);
	if (ret) {
		printk(KERN_DEBUG "pk_test: pk_keygen: failed %d/%s\n",
		       ret, pk_errstr(ret));
		goto leave;
	}
	printk(KERN_DEBUG "pk_test: pk_keygen: ok\n");
	yield();

	ret = pk_getKeyID(id, Public, PublicLen, algo);
	if (ret) {
		printk(KERN_DEBUG "pk_test: pk_getKeyID: failed %d/%s\n",
		       ret, pk_errstr(ret));
		goto leave;
	}
	printk(KERN_DEBUG "pk_test: pk_getKeyID: ok\n");

	asym_print_char(KERN_DEBUG "ID", id, PK_IDLEN / 8);
	asym_print_char(KERN_DEBUG "Public", Public, PublicLen);
	asym_print_char(KERN_DEBUG "Private", Private, PrivateLen);
	asym_print_char(KERN_DEBUG "input", input, inputLen);
	yield();

	ret = __pk_test(input, inputLen, Public, PublicLen, 
			Private, PrivateLen, bits, algo);
	if (ret) {
		goto leave;
	}

	ret = 0;
leave:
	/*printk(KERN_DEBUG "unsx_stackMaxDepth = %d\n", unsx_stackMaxDepth);*/
	/*
	if (Public)
		kfree(Public);
	if (Private)
		kfree(Private);
		*/

	return (ret);
}

static int __pk_test(u8 *input, size_t inputLen, u8 *Public, 
		size_t PublicLen, u8 *Private, size_t PrivateLen,
		size_t bits, int algo)
{
	int ret;
	u8 *enc = NULL;
	size_t encLen;
	u8 *dec = NULL;
	size_t decLen;

	printk(KERN_DEBUG "ENC_LENGTH: %d\n", PK_ENC_LENGTH(bits, algo));
	encLen = PK_ENC_LENGTH(bits, algo);
	enc = (u8 *) kmalloc(encLen, GFP_KERNEL);
	decLen = PK_ENC_LENGTH(bits, algo);
	dec = (u8 *) kmalloc(decLen, GFP_KERNEL);
	if (!enc || !dec) {
		ret = -ENOMEM;
		goto leave;
	}

	ret = pk_encrypt(enc, &encLen, input, inputLen, Public, PublicLen,
			algo | PK_PKCS1_ENCRYPT);
	if (ret) {
		printk(KERN_DEBUG "pk_test: encrypt failed!  %d/%s\n", ret,
		       pk_errstr(ret));
		goto leave;
	}
	printk(KERN_DEBUG "pk_test: encrypt: ok\n");
	asym_print_char(KERN_DEBUG "enc", enc, encLen);
	yield();

	ret = pk_decrypt(dec, &decLen, enc, encLen, Private, PrivateLen,
			      algo | PK_PKCS1_ENCRYPT);
	if (ret) {
		printk(KERN_DEBUG "pk_test: decrypt failed!  %d/%s\n", ret,
		       pk_errstr(ret));
		goto leave;
	}
	printk(KERN_DEBUG "pk_test: decrypt: ok\n");
	asym_print_char(KERN_DEBUG "dec", dec, decLen);
	yield();

	ret = pk_sign(enc, &encLen, input, inputLen, Private, PrivateLen,
			   algo | PK_PKCS1_SIGN);
	if (ret) {
		printk(KERN_DEBUG "pk_test: sign failed!  %d/%s\n", ret,
		       pk_errstr(ret));
		goto leave;
	}
	printk(KERN_DEBUG "pk_test: sign: ok\n");
	asym_print_char(KERN_DEBUG "sig", enc, encLen);
	yield();

	ret = pk_verify(input, inputLen, enc, encLen, Public, PublicLen,
			     algo | PK_PKCS1_SIGN);
	if (ret) {
		printk(KERN_DEBUG "pk_test: verify failed!  %d/%s\n", ret,
		       pk_errstr(ret));
		goto leave;
	}
	printk(KERN_DEBUG "pk_test: verify: ok\n");

	ret = 0;
leave:
	printk(KERN_DEBUG "unsx_stackMaxDepth = %d\n", unsx_stackMaxDepth);
	if (enc)
		kfree(enc);
	if (dec)
		kfree(dec);

	return (ret);
}
