#!/usr/bin/env python
# -*- coding: utf-8 -*-

import pygtk
pygtk.require("2.0")
import gtk
import gtkmozembed
import webbrowser

import urlexpander
import htmlparser

APP_NAME = "TubeBrowser"
APP_VERSION = "0.2.1"
PROJECT_URL = "http://tubebrowser.sourceforge.jp/"
PROJECT_EMAIL = "tubebrowser-dev@lists.sourceforge.jp"


class TubeBrowser:

  def __init__(self):
    self.url = "about:"
    self.default_size = (460, 420) # width, height

    # mozembed renderer
    self.renderer = gtkmozembed.MozEmbed()
    self.renderer.set_flags(self.renderer.flags() | gtk.CAN_FOCUS)
    self.renderer.connect("location", self.url_changed)

    # address bar
    self.addressbar = gtk.Entry()
    self.addressbar.set_text(self.url)
    self.addressbar.connect("activate", self.open_addressbar, self.addressbar)
    self.addressbar.connect("drag_data_received", self.drag_data_received)
    self.addressbar.drag_dest_set(
      gtk.DEST_DEFAULT_MOTION, [('text/uri-list', 0, 0)], gtk.gdk.ACTION_COPY)

    # status message
    self.statusbar = gtk.Statusbar()
    self.statusbar_id = self.statusbar.get_context_id("main")

    self.window = gtk.Window()
    #self.window.set_size_request(self.default_size[0], self.default_size[1])
    self.window.set_default_size(self.default_size[0], self.default_size[1])
    self.window.connect("delete_event", gtk.main_quit)

    # window title
    self.window.set_title(APP_NAME)

    # window icon
    pixbuf = self.get_icon()
    if (pixbuf):
      self.window.set_icon(pixbuf)

    menubar = self.get_menubar(self.window)
    #message_id = self.statusbar.push(self.statusbar_id, "Ready")

    vbox = gtk.VBox()
    vbox.pack_start(menubar, False, True, 0)
    vbox.pack_start(self.addressbar, False, True, 0)
    vbox.pack_start(self.renderer, True, True, 0)
    vbox.pack_start(self.statusbar, False, True, 0)
    self.window.add(vbox)
    self.window.show_all()

    # default start page
    self.show_start_page()

  def url_changed(self, data=None):
    url = self.renderer.get_location()
    if (url != "about:" and url != self.url):
      self.addressbar.set_text(url)
      self.open_location(url)

  def get_menubar(self, window):
    "Create a main menu."
    # TODO: add 'Bookmarks' menu
    main_menu = """
<ui>
  <menubar name="MenuBar">
    <menu action="File">
      <menuitem action="Open" />
      <menuitem action="Quit" />
    </menu>
    <menu action="View">
      <menuitem action="Reload" />
      <menuitem action="Stop" />
      <menuitem action="Reset Size" />
    </menu>
    <menu action="Help">
      <menuitem action="Home Page" />
      <placeholder action="Sites holder">
        <menu action="Video Sites">
          <menuitem action="YouTube" />
          <menuitem action="Tube8" />
          <menuitem action="FC2 Video" />
          <menuitem action="YourFileHost" />
        </menu>
      </placeholder>
      <menuitem action="About" />
    </menu>
  </menubar>
</ui>
"""
    ui_manager = gtk.UIManager()
    accel_group = ui_manager.get_accel_group()
    window.add_accel_group(accel_group)

    # action group
    action_group = gtk.ActionGroup("MenuActionBase")
    action_group.add_actions(
      [("File", None, "_File"),
       ("View", None, "_View"),
       ("Help", None, "_Help")])
    ui_manager.insert_action_group(action_group, 0)

    action_group.add_actions(
      # name, stock id, label, accelerator, tooltip, callback
      [("Open", None, "_Open", "<control>L", None, self.focus_addressbar),
       ("Quit", None, "_Quit", None, None, gtk.main_quit),
       ("Reload", None, "_Reload", "F5", None, self.reload),
       ("Stop", None, "_Stop", None, None, self.stop),
       ("Reset Size", None, "Reset Size", None, None, self.reset_size),
       ("Home Page", None, "Home Page", None, None, self.open_homepage),
       ("Video Sites", None, "Video Sites", None, None, None),
       ("YouTube", None, "YouTube", None, None, self.open_sites),
       ("Tube8", None, "Tube8", None, None, self.open_sites),
       ("FC2 Video", None, "FC2 Video", None, None, self.open_sites),
       ("YourFileHost", None, "YourFileHost", None, None, self.open_sites),
       ("About", None, "_About", None, None, self.show_about_dialog)])

    ui_manager.add_ui_from_string(main_menu)
    return ui_manager.get_widget("/MenuBar")

  def get_icon(self):
    pixbuf = None
    icon_theme = gtk.icon_theme_get_default()
    icon_theme.append_search_path(".")
    #pixbuf = gtk.gdk.pixbuf_new_from_file("tubebrowser.svg")
    try:
      pixbuf = icon_theme.load_icon("tubebrowser", 48, 0)
    except:
      pass
    return pixbuf

  def drag_data_received(self, widget, context, x, y, selection, info, time):
    self.addressbar.set_text(selection.data)
    self.open_location(selection.data)

  def focus_addressbar(self, action, data=None):
    self.addressbar.grab_focus()

  def open_addressbar(self, widget, addressbar):
    self.open_location(addressbar.get_text())

  def open_location(self, location):
    self.url = location
    if (self.url == "about:"):
      self.show_start_page()
      return

    # Check redirection URLs.
    expander = urlexpander.URLExpander()
    expander.set_useragent("Mozilla/5.0")
    dest_url = expander.expand(self.url)
    if (dest_url):
      self.url = dest_url

    parser = htmlparser.HTMLParser(self.url)
    if (parser.is_supported()):
      content = parser.get_content()
      title = parser.get_title()
      self.set_title(title)
      self.renderer.render_data(
        content, long(len(content)), self.url, "text/html")
      self.renderer.grab_focus()

    else:
      self.show_start_page()
      dialog = gtk.MessageDialog(
        None, gtk.DIALOG_MODAL, gtk.MESSAGE_WARNING, gtk.BUTTONS_OK,
        "The URL you entered is not supported.")
      dialog.set_title("Warning")
      dialog.run()
      dialog.destroy()

  def show_start_page(self):
    "Show the default start page."
    html = '<html><body bgcolor="#000000"></body></html>'
    self.renderer.render_data(html, long(len(html)), "about:", "text/html")
    self.set_title()
    self.renderer.grab_focus()

  def stop(self, action, data=None):
    self.addressbar.set_text("about:")
    self.open_location("about:")

  def reload(self, action, data=None):
    self.addressbar.set_text(self.url)
    self.open_location(self.url)

  def reset_size(self, action, data=None):
    "Reset the window size to default."
    self.window.resize(self.default_size[0], self.default_size[1])

  def open_homepage(self, action, data=None):
    webbrowser.open("http://tubebrowser.sourceforge.jp/")

  def open_sites(self, action, data=None):
    sites = {"YouTube": "http://www.youtube.com/",
             "Tube8": "http://www.tube8.com/",
             "FC2 Video": "http://video.fc2.com/",
             "YourFileHost": "http://www.yourfilehost.com/"}
    webbrowser.open(sites[action.get_name()])

  def show_about_dialog(self, action, data=None):
    "Show the 'About' dialog."
    dialog = gtk.AboutDialog()
    dialog.set_name(APP_NAME)
    dialog.set_version(APP_VERSION)
    dialog.set_comments("Web browser for video hosting sites.")
    dialog.set_website(PROJECT_URL)
    pixbuf = self.get_icon()
    if (pixbuf):
      dialog.set_logo(pixbuf)
    response = dialog.run()
    #dialog.hide()
    dialog.destroy()

  def set_title(self, title=None):
    if (title):
      self.window.set_title("%s - %s" % (title, APP_NAME))
    else:
      self.window.set_title(APP_NAME)

  def main(self):
    gtk.main()
