/*
 * ccstools.c
 *
 * A busybox version of TOMOYO Linux's utilities.
 *
 * Copyright (C) 2005-2006  NTT DATA CORPORATION
 *
 * Version: 1.3.1     2006/12/08
 *
 *   gcc -Wall -O3 -o ccstools ccstools.c -lncurses
 *
 * Be sure to make symlinks or links to this program.
 *  
 */

/* CONFIGURATION: Comment out what you don't want to compile. */

#define NEED_SORTPOLICY
#define NEED_SETPROFILE
#define NEED_SETLEVEL
#define NEED_SAVEPOLICY
#define NEED_PATHMATCH
#define NEED_LOADPOLICY
#define NEED_LDWATCH
#define NEED_FINDTEMP
#define NEED_EDITPOLICY
#define NEED_CHECKPOLICY
#define NEED_CCSTREE
#define NEED_CCSQUERYD
#define NEED_CCSAUDITD
#define NEED_PATTERNIZE

/* CODE */

#define _FILE_OFFSET_BITS 64
#define _LARGEFILE_SOURCE
#define _LARGEFILE64_SOURCE
#define u8 __u8
#define u16 __u16
#define u32 __u32
#define _GNU_SOURCE
#include <arpa/inet.h>
#include <asm/types.h>
#include <curses.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <sys/mount.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>
#include <syslog.h>
#include <termios.h>
#include <time.h>
#include <unistd.h>

#define SYSTEM_POLICY_FILE    "system_policy"
#define EXCEPTION_POLICY_FILE "exception_policy"
#define DOMAIN_POLICY_FILE    "domain_policy"

#define SCREEN_SYSTEM_LIST    0
#define SCREEN_EXCEPTION_LIST 1
#define SCREEN_DOMAIN_LIST    2
#define SCREEN_ACL_LIST       3
#define MAXSCREEN             4

#define POLICY_TYPE_UNKNOWN          0
#define POLICY_TYPE_DOMAIN_POLICY    1
#define POLICY_TYPE_EXCEPTION_POLICY 2
#define POLICY_TYPE_SYSTEM_POLICY    3

#define VALUE_TYPE_DECIMAL     1
#define VALUE_TYPE_OCTAL       2
#define VALUE_TYPE_HEXADECIMAL 3

#define NETWORK_ACL_UDP_BIND    0
#define NETWORK_ACL_UDP_CONNECT 1
#define NETWORK_ACL_TCP_BIND    2
#define NETWORK_ACL_TCP_LISTEN  3
#define NETWORK_ACL_TCP_CONNECT 4
#define NETWORK_ACL_TCP_ACCEPT  5
#define NETWORK_ACL_RAW_BIND    6
#define NETWORK_ACL_RAW_CONNECT 7

#define KEYWORD_AGGREGATOR               "aggregator "
#define KEYWORD_AGGREGATOR_LEN         (sizeof(KEYWORD_AGGREGATOR) - 1)
#define KEYWORD_ALIAS                    "alias "
#define KEYWORD_ALIAS_LEN              (sizeof(KEYWORD_ALIAS) - 1)
#define KEYWORD_ALLOW_ARGV0              "allow_argv0 "
#define KEYWORD_ALLOW_ARGV0_LEN        (sizeof(KEYWORD_ALLOW_ARGV0) - 1)
#define KEYWORD_ALLOW_BIND               "allow_bind "
#define KEYWORD_ALLOW_BIND_LEN         (sizeof(KEYWORD_ALLOW_BIND) - 1)
#define KEYWORD_ALLOW_CAPABILITY         "allow_capability "
#define KEYWORD_ALLOW_CAPABILITY_LEN   (sizeof(KEYWORD_ALLOW_CAPABILITY) - 1)
#define KEYWORD_ALLOW_CHROOT             "allow_chroot "
#define KEYWORD_ALLOW_CHROOT_LEN       (sizeof(KEYWORD_ALLOW_CHROOT) - 1)
#define KEYWORD_ALLOW_CONNECT            "allow_connect "
#define KEYWORD_ALLOW_CONNECT_LEN      (sizeof(KEYWORD_ALLOW_CONNECT) - 1)
#define KEYWORD_ALLOW_MOUNT              "allow_mount "
#define KEYWORD_ALLOW_MOUNT_LEN        (sizeof(KEYWORD_ALLOW_MOUNT) - 1)
#define KEYWORD_ALLOW_NETWORK            "allow_network "
#define KEYWORD_ALLOW_NETWORK_LEN      (sizeof(KEYWORD_ALLOW_NETWORK) - 1)
#define KEYWORD_ALLOW_READ               "allow_read "
#define KEYWORD_ALLOW_READ_LEN         (sizeof(KEYWORD_ALLOW_READ) - 1)
#define KEYWORD_ALLOW_SIGNAL             "allow_signal "
#define KEYWORD_ALLOW_SIGNAL_LEN       (sizeof(KEYWORD_ALLOW_SIGNAL) - 1)
#define KEYWORD_DELETE                   "delete "
#define KEYWORD_DELETE_LEN             (sizeof(KEYWORD_DELETE) - 1)
#define KEYWORD_DENY_AUTOBIND            "deny_autobind "
#define KEYWORD_DENY_AUTOBIND_LEN      (sizeof(KEYWORD_DENY_AUTOBIND) - 1)
#define KEYWORD_DENY_REWRITE             "deny_rewrite "
#define KEYWORD_DENY_REWRITE_LEN       (sizeof(KEYWORD_DENY_REWRITE) - 1)
#define KEYWORD_DENY_UNMOUNT             "deny_unmount "
#define KEYWORD_DENY_UNMOUNT_LEN       (sizeof(KEYWORD_DENY_UNMOUNT) - 1)
#define KEYWORD_FILE_PATTERN             "file_pattern "
#define KEYWORD_FILE_PATTERN_LEN       (sizeof(KEYWORD_FILE_PATTERN) - 1)
#define KEYWORD_INITIALIZER              "initializer "
#define KEYWORD_INITIALIZER_LEN        (sizeof(KEYWORD_INITIALIZER) - 1)
#define KEYWORD_MAC_FOR_CAPABILITY      "MAC_FOR_CAPABILITY::"
#define KEYWORD_MAC_FOR_CAPABILITY_LEN  (sizeof(KEYWORD_MAC_FOR_CAPABILITY) - 1)
#define KEYWORD_SELECT                   "select "
#define KEYWORD_SELECT_LEN             (sizeof(KEYWORD_SELECT) - 1)
#define KEYWORD_DOMAIN_KEEPER            "keep_domain "
#define KEYWORD_DOMAIN_KEEPER_LEN      (sizeof(KEYWORD_DOMAIN_KEEPER) - 1)
#define KEYWORD_USE_PROFILE              "use_profile "
#define KEYWORD_USE_PROFILE_LEN        (sizeof(KEYWORD_USE_PROFILE) - 1)
#define MAXBUFSIZE 8192
#define MAX_FILES 2
#define MAX_HASH 256
#define PAGE_SIZE 4096
#define ROOT_NAME "<kernel>"
#define ROOT_NAME_LEN (sizeof(ROOT_NAME) - 1)

typedef struct name_entry {
	struct name_entry *next; /* Pointer to next record. NULL if none.             */
	unsigned int hash;       /* hash and length                                   */
	const char *name;        /* Text form of filename and domainname. Never NULL. */
} NAME_ENTRY;

typedef struct free_memory_block_list {
	struct free_memory_block_list *next; /* Pointer to next record. NULL if none. */
	char *ptr;                           /* Pointer to a free area.               */
	int len;                             /* Length of the area.                   */
} FREE_MEMORY_BLOCK_LIST;

typedef struct {
	char *pathname;
	char *real_pathname;
} PATHNAME_ENTRY;

typedef struct domain_info {
	const char *domainname;
	const char **string_ptr;
	int string_count;
	u8 is_virtual;
	u8 profile;
} DOMAIN_INFO;

typedef struct {
	pid_t pid;
	pid_t ppid;
	u8 done;
} TASK_ENTRY;

static void OutOfMemory(void) {
	fprintf(stderr, "Out of memory. Aborted.\n");
	exit(1);
}

/* Copied from kernel source. */
static inline unsigned long partial_name_hash(unsigned long c, unsigned long prevhash) {
	return (prevhash + (c << 4) + (c >> 4)) * 11;
}

/* Copied from kernel source. */
static inline unsigned int full_name_hash(const unsigned char *name, unsigned int len) {
	unsigned long hash = 0;
	while (len--) hash = partial_name_hash(*name++, hash);
	return (unsigned int) hash;
}

static char *alloc_element(const unsigned int size) {
	static char *buf = NULL;
	static unsigned int buf_used_len = PAGE_SIZE;
	char *ptr = NULL;
	if (size > PAGE_SIZE) return NULL;
	if (buf_used_len + size > PAGE_SIZE) {
		if ((ptr = malloc(PAGE_SIZE)) == NULL) OutOfMemory();
		buf = ptr;
		memset(buf, 0, PAGE_SIZE);
		buf_used_len = size;
		ptr = buf;
	} else if (size) {
		int i;
		ptr = buf + buf_used_len;
		buf_used_len += size;
		for (i = 0; i < size; i++) if (ptr[i]) OutOfMemory();
	}
	return ptr;
}

static const char *SaveName(const char *name) {
	static FREE_MEMORY_BLOCK_LIST fmb_list = { NULL, NULL, 0 };
	static NAME_ENTRY name_list[MAX_HASH]; /* The list of names. */
	NAME_ENTRY *ptr, *prev = NULL;
	unsigned int hash;
	FREE_MEMORY_BLOCK_LIST *fmb = &fmb_list;
	int len;
	static int first_call = 1;
	if (!name) return NULL;
	len = strlen(name) + 1;
	if (len > PAGE_SIZE) {
		fprintf(stderr, "ERROR: Name too long for SaveName().\n");
		return NULL;
	}
	hash = full_name_hash((const unsigned char *) name, len - 1);
	if (first_call) {
		int i;
		first_call = 0;
		memset(&name_list, 0, sizeof(name_list));
		for (i = 0; i < MAX_HASH; i++) name_list[i].name = "/";
	}
	ptr = &name_list[hash % MAX_HASH];
	hash ^= len; /* The hash % MAX_HASH are always same for ptr->hash, so embed length into the hash value. */
	while (ptr) {
		if (hash == ptr->hash && strcmp(name, ptr->name) == 0) goto out;
		prev = ptr; ptr = ptr->next;
	}
	while (len > fmb->len) {
		if (fmb->next) {
			fmb = fmb->next;
		} else {
			char *cp;
			if ((cp = (char *) malloc(PAGE_SIZE)) == NULL || (fmb->next = (FREE_MEMORY_BLOCK_LIST *) alloc_element(sizeof(FREE_MEMORY_BLOCK_LIST))) == NULL) OutOfMemory();
			memset(cp, 0, PAGE_SIZE);
			fmb = fmb->next;
			fmb->ptr = cp;
			fmb->len = PAGE_SIZE;
		}
	}
	if ((ptr = (NAME_ENTRY *) alloc_element(sizeof(NAME_ENTRY))) == NULL) OutOfMemory();
	memset(ptr, 0, sizeof(NAME_ENTRY));
	ptr->hash = hash;
	ptr->name = fmb->ptr;
	memmove(fmb->ptr, name, len);
	fmb->ptr += len;
	fmb->len -= len;
	prev->next = ptr; /* prev != NULL because name_list is not empty. */
	if (fmb->len == 0) {
		FREE_MEMORY_BLOCK_LIST *ptr = &fmb_list;
		while (ptr->next != fmb) ptr = ptr->next; ptr->next = fmb->next;
	}
 out:
	return ptr ? (const char *) ptr->name : NULL;
}

static int IsCorrectDomain(const unsigned char *domainname) {
	unsigned char c, d, e;
	if (!domainname || strncmp(domainname, ROOT_NAME, ROOT_NAME_LEN)) goto out;
	domainname += ROOT_NAME_LEN;
	if (!*domainname) return 1;
	do {
		if (*domainname++ != ' ') goto out;
		if (*domainname++ != '/') goto out;
		while ((c = *domainname) != '\0' && c != ' ') {
			domainname++;
			if (c == '\\') {
				switch ((c = *domainname++)) {
				case '\\':  /* "\\" */
					continue;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *domainname++) >= '0' && d <= '7' && (e = *domainname++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				goto out;
			} else if (c < ' ' || c >= 127) {
				goto out;
			}
		}
	} while (*domainname);
	return 1;
 out:
	return 0;
}

static DOMAIN_INFO *domain_list[2] = { NULL, NULL };
static int domain_list_count[2] = { 0, 0 };

static int AddStringEntry(const char *entry, const int index, const int type) {
	const char **acl_ptr;
	int acl_count;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count[type]) {
		fprintf(stderr, "AddStringEntry: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	if ((cp = SaveName(entry)) == NULL) OutOfMemory();

	acl_ptr = domain_list[type][index].string_ptr;
	acl_count = domain_list[type][index].string_count;

	// Check for the same entry.
	for (i = 0; i < acl_count; i++) {
		// Faster comparison, for they are SaveName'd.
		if (cp == acl_ptr[i]) return 0;
	}

	if ((acl_ptr = (const char **) realloc(acl_ptr, (acl_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
	acl_ptr[acl_count++] = cp;
	domain_list[type][index].string_ptr = acl_ptr;
	domain_list[type][index].string_count = acl_count;
	return 0;
}

static int DelStringEntry(const char *entry, const int index, const int type) {
	const char **acl_ptr;
	int acl_count;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count[type]) {
		fprintf(stderr, "DelStringEntry: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	if ((cp = SaveName(entry)) == NULL) OutOfMemory();

	acl_ptr = domain_list[type][index].string_ptr;
	acl_count = domain_list[type][index].string_count;

	for (i = 0; i < acl_count; i++) {
		// Faster comparison, for they are SaveName'd.
		if (cp != acl_ptr[i]) continue;
		domain_list[type][index].string_count--;
		for (; i < acl_count - 1; i++) acl_ptr[i] = acl_ptr[i + 1];
		return 0;
	}
	return -ENOENT;
}

static void ClearDomainPolicy(const int type) {
	int index;
	for (index = 0; index < domain_list_count[type]; index++) {
		free(domain_list[type][index].string_ptr);
		domain_list[type][index].string_ptr = NULL;
		domain_list[type][index].string_count = 0;
	}
	free(domain_list[type]);
	domain_list[type] = NULL;
	domain_list_count[type]= 0;
}

static int FindDomain(const char *domainname, const int type) {
	int i;
	for (i = 0; i < domain_list_count[type]; i++) {
		if (strcmp(domainname, domain_list[type][i].domainname) == 0) {
			return i;
		}
	}
	return EOF;
}

static int FindOrAssignNewDomain(const char *domainname, const int type) {
	const char *saved_domainname;
	int index;
	if ((index = FindDomain(domainname, type)) == EOF) {
		if (IsCorrectDomain(domainname)) {
			if ((domain_list[type] = (DOMAIN_INFO *) realloc(domain_list[type], (domain_list_count[type] + 1) * sizeof(DOMAIN_INFO))) == NULL) OutOfMemory();
			memset(&domain_list[type][domain_list_count[type]], 0, sizeof(DOMAIN_INFO));
			if ((saved_domainname = SaveName(domainname)) == NULL) OutOfMemory();
			domain_list[type][domain_list_count[type]].domainname = saved_domainname;
			index = domain_list_count[type]++;
		} else {
			fprintf(stderr, "FindOrAssignNewDomain: Invalid domainname '%s'\n", domainname);
		}
	}
	return index;
}

static void DeleteDomain(const int index, const int type) {
	if (index > 0 && index < domain_list_count[type]) {
		int i;
		free(domain_list[type][index].string_ptr);
		for (i = index; i < domain_list_count[type] - 1; i++) domain_list[type][i] = domain_list[type][i + 1];
		domain_list_count[type]--;
	}
}


static int IsDomainDef(const unsigned char *domainname) {
	return strncmp(domainname, ROOT_NAME, ROOT_NAME_LEN) == 0 && (domainname[ROOT_NAME_LEN] == '\0' || domainname[ROOT_NAME_LEN] == ' ');
}

static void NormalizeLine(unsigned char *line) {
	unsigned char *sp = line, *dp = line;
	int first = 1;
	while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	while (*sp) {
		if (!first) *dp++ = ' ';
		first = 0;
		while (32 < *sp && *sp < 127) *dp++ = *sp++;
		while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	}
	*dp = '\0';
}

static void SortPolicy(const int type) {
	int i, j, k;
	for (i = 0; i < domain_list_count[type]; i++) {
		for (j = i + 1; j < domain_list_count[type]; j++) {
			if (strcmp(domain_list[type][i].domainname, domain_list[type][j].domainname) > 0) {
				DOMAIN_INFO tmp = domain_list[type][i]; domain_list[type][i] = domain_list[type][j]; domain_list[type][j] = tmp;
			}
		}
	}
	for (i = 0; i < domain_list_count[type]; i++) {
		const char **string_ptr = domain_list[type][i].string_ptr;
		const int string_count = domain_list[type][i].string_count;
		for (j = 0; j < string_count; j++) {
			for (k = j + 1; k < string_count; k++) {
				const char *a = string_ptr[j];
				const char *b = string_ptr[k];
				if (*a && *b && strcmp(a + 1, b + 1) > 0) {
					string_ptr[j] = b; string_ptr[k] = a;
				}
			}
		}
	}
}

static void fprintf_encoded(FILE *fp, const char *pathname) {
	unsigned char c;
	while ((c = * (const unsigned char *) pathname++) != 0) {
		if (c == '\\') {
			fputc('\\', fp);
			fputc('\\', fp);
		} else if (c > 32 && c < 127) {
			fputc(c, fp);
		} else {
			fprintf(fp, "\\%c%c%c", (c >> 6) + '0', ((c >> 3) & 7) + '0', (c & 7) + '0'); 
		}
	}
}

static void RemoveHeader(char *line, const int len) {
	memmove(line, line + len, strlen(line + len) + 1); 
}

static const int shared_buffer_len = MAXBUFSIZE;
static char *shared_buffer = NULL;
static int buffer_lock = 0;
static void get(void) {
	if (buffer_lock) OutOfMemory();
	if (!shared_buffer && (shared_buffer = malloc(shared_buffer_len)) == NULL) OutOfMemory();
	buffer_lock++;
}
static void put(void) {
	if (buffer_lock != 1) OutOfMemory();
	buffer_lock--;
	//free(shared_buffer); shared_buffer = NULL;
}

static int freadline(FILE *fp) {
	char *cp;
	memset(shared_buffer, 0, shared_buffer_len);
	if (fgets(shared_buffer, shared_buffer_len - 1, fp) == NULL ||
		(cp = strchr(shared_buffer, '\n')) == NULL) return 0;
	*cp = '\0';
	NormalizeLine(shared_buffer);
	return 1;
}

static void ReadDomainPolicy(const char *filename, const int type) {
	FILE *fp = stdin;
	int index;
	if (filename) {
		if ((fp = fopen(filename, "r")) == NULL) {
			fprintf(stderr, "Can't open %s\n", filename);
			return;
		}
	}
	index = EOF;
	get();
	while (freadline(fp)) {
		if (IsDomainDef(shared_buffer)) {
			index = FindOrAssignNewDomain(shared_buffer, type);
		} else if (index >= 0 && shared_buffer[0]) {
			AddStringEntry(shared_buffer, index, type);
		}
	}
	put();
	if (fp != stdin) fclose(fp);
	SortPolicy(type);
}

static int WriteDomainPolicy(const int fd, const int type) {
	int i, j;
	for (i = 0; i < domain_list_count[type]; i++) {
		const char **string_ptr = domain_list[type][i].string_ptr;
		const int string_count = domain_list[type][i].string_count;
		int len = strlen(domain_list[0][i].domainname);
		write(fd, domain_list[type][i].domainname, len);
		write(fd, "\n\n", 2);
		for (j = 0; j < string_count; j++) {
			len = strlen(string_ptr[j]);
			write(fd, string_ptr[j], len);
			write(fd, "\n", 1);
		}
		write(fd, "\n", 1);
	}
	return 0;
}

static int strendswith(const char *name, const char *tail) {
	int len;
	if (!name || !tail) return 0;
	len = strlen(name) - strlen(tail);
	return len >= 0 && strcmp(name + len, tail) == 0;
}

static int IsCorrectPath(const char *filename, const int start_type, const int pattern_type, const int end_type) {
	int contains_pattern = 0;
	char c, d, e;
	if (!filename) goto out;
	c = *filename;
	if (start_type == 1) { /* Must start with '/' */
		if (c != '/') goto out;
	} else if (start_type == -1) { /* Must not start with '/' */
		if (c == '/') goto out;
	}
	if (c) c = * (strchr(filename, '\0') - 1);
	if (end_type == 1) { /* Must end with '/' */
		if (c != '/') goto out;
	} else if (end_type == -1) { /* Must not end with '/' */
		if (c == '/') goto out;
	}
	while ((c = *filename++) != '\0') {
		if (c == '\\') {
			switch ((c = *filename++)) {
			case '\\':  /* "\\" */
				continue;
			case '$':   /* "\$" */
			case '+':   /* "\+" */
			case '?':   /* "\?" */
			case '*':   /* "\*" */
			case '@':   /* "\@" */
			case 'x':   /* "\x" */
			case 'X':   /* "\X" */
			case 'a':   /* "\a" */
			case 'A':   /* "\A" */
				if (pattern_type == -1) break; /* Must not contain pattern */
				contains_pattern = 1;
				continue;
			case '0':   /* "\ooo" */
			case '1':
			case '2':
			case '3':
				if ((d = *filename++) >= '0' && d <= '7' && (e = *filename++) >= '0' && e <= '7') {
					const unsigned char f =
						(((unsigned char) (c - '0')) << 6) +
						(((unsigned char) (d - '0')) << 3) +
						(((unsigned char) (e - '0')));
					if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
				}
			}
			goto out;
		} else if (c <= ' ' || c >= 127) {
			goto out;
		}
	}
	if (pattern_type == 1) { /* Must contain pattern */
		if (!contains_pattern) goto out;
	}
	return 1;
 out:
	return 0;
}

static int FileMatchesToPattern(const char *filename, const char *filename_end, const char *pattern, const char *pattern_end) {
	while (filename < filename_end && pattern < pattern_end) {
		if (*pattern != '\\') {
			if (*filename++ != *pattern++) return 0;
		} else {
			char c = *filename;
			pattern++;
			switch (*pattern) {
			case '?':
				if (c == '/') {
					return 0;
				} else if (c == '\\') {
					if ((c = filename[1]) == '\\') {
						filename++; /* safe because filename is \\ */
					} else if (c >= '0' && c <= '3' && (c = filename[2]) >= '0' && c <= '7' && (c = filename[3]) >= '0' && c <= '7') {
						filename += 3; /* safe because filename is \ooo */
					} else {
						return 0;
					}
				}
				break;
			case '\\':
				if (c != '\\') return 0;
				if (*++filename != '\\') return 0; /* safe because *filename != '\0' */
				break;
			case '+':
				if (c < '0' || c > '9') return 0;
				break;
			case 'x':
				if (!((c >= '0' && c <= '9') || (c >= 'A' && c <= 'F') || (c >= 'a' && c <= 'f'))) return 0;
				break;
			case 'a':
				if (!((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z'))) return 0;
				break;
			case '0':
			case '1':
			case '2':
			case '3':
				if (c == '\\' && (c = filename[1]) >= '0' && c <= '3' && c == *pattern
					&& (c = filename[2]) >= '0' && c <= '7' && c == pattern[1]
					&& (c = filename[3]) >= '0' && c <= '7' && c == pattern[2]) {
					filename += 3; /* safe because filename is \ooo */
					pattern += 2; /* safe because pattern is \ooo  */
					break;
				}
				return 0; /* Not matched. */
			case '*':
			case '@':
				{
					int i;
					for (i = 0; i <= filename_end - filename; i++) {
						if (FileMatchesToPattern(filename + i, filename_end, pattern + 1, pattern_end)) return 1;
						if ((c = filename[i]) == '.' && *pattern == '@') break;
						if (c == '\\') {
							if ((c = filename[i + 1]) == '\\') {
								i++; /* safe because filename is \\ */
							} else if (c >= '0' && c <= '3' && (c = filename[i + 2]) >= '0' && c <= '7' && (c = filename[i + 3]) >= '0' && c <= '7') {
								i += 3; /* safe because filename is \ooo */
							} else {
								break; /* Bad pattern. */
							}
						}
					}
					return 0; /* Not matched. */
				}
			default:
				{
					int i, j = 0;
					if ((c = *pattern) == '$') {
						while ((c = filename[j]) >= '0' && c <= '9') j++;
					} else if (c == 'X') {
						while (((c = filename[j]) >= '0' && c <= '9') || (c >= 'A' && c <= 'F') || (c >= 'a' && c <= 'f')) j++;
					} else if (c == 'A') {
						while (((c = filename[j]) >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z')) j++;
					}
					for (i = 1; i <= j; i++) {
						if (FileMatchesToPattern(filename + i, filename_end, pattern + 1, pattern_end)) return 1;
					}
				}
				return 0; /* Not matched or bad pattern. */
			}
			filename++; /* safe because *filename != '\0' */
			pattern++; /* safe because *pattern != '\0' */
		}
	}
	while (*pattern == '\\' && (*(pattern + 1) == '*' || *(pattern + 1) == '@')) pattern += 2;
	return (filename == filename_end && pattern == pattern_end);
}

static int PathDepth(const char *pathname) {
	int i = 0;
	if (pathname) {
		char *ep = strchr(pathname, '\0');
		if (pathname < ep--) {
			if (*ep != '/') i++;
			while (pathname <= ep) if (*ep-- == '/') i += 2;
		}
	}
	return i;
}

static int PathMatchesToPattern(const char *pathname, const char *pattern) {
	if (!pathname || !pattern) return 0;
	/* if pattern doesn't contain '\', I can use strcmp(). */
	if (!strchr(pattern, '\\')) return !strcmp(pathname, pattern);
	if (PathDepth(pathname) != PathDepth(pattern)) return 0;
	while (*pathname && *pattern) {
		const char *pathname_delimiter = strchr(pathname, '/'), *pattern_delimiter = strchr(pattern, '/');
		if (!pathname_delimiter) pathname_delimiter = strchr(pathname, '\0');
		if (!pattern_delimiter) pattern_delimiter = strchr(pattern, '\0');
		if (!FileMatchesToPattern(pathname, pathname_delimiter, pattern, pattern_delimiter)) return 0;
		pathname = *pathname_delimiter ? pathname_delimiter + 1 : pathname_delimiter;
		pattern = *pattern_delimiter ? pattern_delimiter + 1 : pattern_delimiter;
	}
	while (*pattern == '\\' && (*(pattern + 1) == '*' || *(pattern + 1) == '@')) pattern += 2;
	return (!*pathname && !*pattern);
}

/*
 * Check whether the given filename is patterened.
 * Returns nonzero if patterned, zero otherwise.
 */
static int PathContainsPattern(const char *filename) {
	if (filename) {
		char c, d, e;
		while ((c = *filename++) != '\0') {
			if (c != '\\') continue;
			switch (c = *filename++) {
			case '\\':  /* "\\" */
				continue;
			case '0':   /* "\ooo" */
			case '1':
			case '2':
			case '3':
				if ((d = *filename++) >= '0' && d <= '7' && (e = *filename++) >= '0' && e <= '7'
					&& (c != '0' || d != '0' || e != '0')) continue; /* pattern is not \000 */
			}
			return 1;
		}
	}
	return 0;
}

#ifdef NEED_SORTPOLICY

static int sortpolicy_main(int argc, char *argv[]) {
	ReadDomainPolicy(NULL, 0);
	WriteDomainPolicy(1, 0);
	return 0;
}

#endif

#ifdef NEED_SETPROFILE

static int setprofile_main(int argc, char *argv[]) {
	FILE *fp_in, *fp_out;
	int profile = 0;
	int recursive = 0;
	int i, start = 2;
	if (argc > 1 && strcmp(argv[1], "-r") == 0) {
		recursive = 1;
		start = 3;
	}
	if (argc <= start || sscanf(argv[start - 1], "%u", &profile) != 1) {
		fprintf(stderr, "%s [-r] profile domainname [domainname ...]\n", argv[0]);
		return 0;
	}
	for (i = start; i < argc; i++) NormalizeLine(argv[i]);
	{
		const int fd = open("/proc/ccs/policy/.domain_status", O_RDWR);
		if (fd == EOF) {
			fprintf(stderr, "You can't run this daemon for this kernel.\n");
			return 1;
		} else if (write(fd, "", 0) != 0) {
			fprintf(stderr, "You need to register this program to /proc/ccs/policy/manager to run this program.\n");
			return 1;
		}
		close(fd);
	}
	{
		int profile_found = 0;
		if ((fp_in = fopen("/proc/ccs/status", "r")) == NULL) {
			fprintf(stderr, "Can't open policy file.\n");
			exit(1);
		}
		get();
		while (freadline(fp_in)) {
			if (atoi(shared_buffer) == profile) {
				profile_found = 1;
				break;
			}
		}
		put();
		fclose(fp_in);
		if (!profile_found) {
			fprintf(stderr, "Profile %u not defined.\n", profile);
			exit(1);
		}
	}
	if ((fp_in = fopen("/proc/ccs/policy/.domain_status", "r")) == NULL || (fp_out = fopen("/proc/ccs/policy/.domain_status", "w")) == NULL) {
		fprintf(stderr, "Can't open policy file.\n");
		exit(1);
	}
	get();
	while (freadline(fp_in)) {
		char *cp = strchr(shared_buffer, ' ');
		if (!cp) break;
		*cp++ = '\0';
		for (i = start; i < argc; i++) {
			const int len = strlen(argv[i]);
			if (strncmp(cp, argv[i], len)) continue;
			if (!recursive) {
				if (cp[len]) continue;
			} else {
				if (cp[len] && cp[len] != ' ') continue;
			}
			fprintf(fp_out, "%d %s\n", profile, cp);
			printf("%d %s\n", profile, cp);
		}
	}
	put();
	fclose(fp_in); fclose(fp_out);
	return 0;
}

#endif

#ifdef NEED_SETLEVEL

static int setlevel_main(int argc, char *argv[]) {
	static const char *policy_file = "/proc/ccs/status";
	int i, fd;
	char c;
	if (access("/proc/ccs/", F_OK)) {
		fprintf(stderr, "You can't use this command for this kernel.\n");
		return 1;
	}
	if ((fd = open(policy_file, O_RDWR)) == EOF) {
		fprintf(stderr, "Can't open %s\n", policy_file);
		return 1;
	} else if (write(fd, "", 0) != 0) {
		fprintf(stderr, "You need to register this program to /proc/ccs/policy/manager to run this program.\n");
		return 1;
	}
	if (argc > 1) {
		for (i = 1; i < argc; i++) {
			write(fd, argv[i], strlen(argv[i])); write(fd, "\n", 1);
		}
	}
	printf("<<< Access Control Status >>>\n");
	while (read(fd, &c, 1) == 1) putchar(c);
	close(fd);
	return 0;
}

#endif

#ifdef NEED_SAVEPOLICY

static char *ReadFile(const char *filename) {
	char *read_buffer = NULL;
	int fd;
	if ((fd = open(filename, O_RDONLY)) != EOF) {
		int read_buffer_len = 0;
		while (1) {
			char *cp = realloc(read_buffer, read_buffer_len + 4096);
			int len;
			if (!cp) {
				free(read_buffer);
				return NULL;
			}
			read_buffer = cp;
			len = read(fd, read_buffer + read_buffer_len, 4095);
			if (len <= 0) break;
			read_buffer_len += len;
		}
		close(fd);
		read_buffer[read_buffer_len] = '\0';
	}
	return read_buffer;
}

static int IsSameDomainList(void) {
	if (domain_list_count[0] == domain_list_count[1]) {
		int i, j;
		for (i = 0; i < domain_list_count[0]; i++) {
			const char **string_ptr = domain_list[0][i].string_ptr;
			const int string_count = domain_list[0][i].string_count;
			if (string_count == domain_list[1][i].string_count) {
				const char **ptr = domain_list[1][i].string_ptr;
				for (j = 0; j < string_count; j++) {
					/* Faster comparison, for they are SaveName'd and sorted pointers. */
					if (string_ptr[j] != ptr[j]) break;
				}
				if (j == string_count) continue;
			}
			break;
		}
		if (i == domain_list_count[0]) return 1;
	}
	return 0;
}

static int savepolicy_main(int argc, char *argv[]) {
	int remount_root = 0;
	char filename[1024];
	int save_system_policy = 0;
	int save_exception_policy = 0;
	int save_domain_policy = 0;
	int force_save = 0;
	int repeat;
	time_t now = time(NULL);
	struct tm *tm = localtime(&now);
	memset(filename, 0, sizeof(filename));
	if (access("/proc/self/", F_OK)) mount("/proc", "/proc", "proc", 0, NULL);
	if (access("/proc/ccs/policy/", F_OK)) {
		fprintf(stderr, "You can't run this program for this kernel.\n");
		return 0;
	}
	if (argc == 1) {
		force_save = save_system_policy = save_exception_policy = save_domain_policy = 1;
	} else {
		int i;
		for (i = 1; i < argc; i++) {
			char *p = argv[i];
			char *s = strchr(p, 's');
			char *e = strchr(p, 'e');
			char *d = strchr(p, 'd');
			char *a = strchr(p, 'a');
			char *f = strchr(p, 'f');
			if (s || a) save_system_policy = 1;
			if (e || a) save_exception_policy = 1;
			if (d || a) save_domain_policy = 1;
			if (f) force_save = 1;
			if (strcspn(p, "sedaf")) {
				printf("%s [s][e][d][a][f]\n"
					   "s : Save system_policy.\n"
					   "e : Save exception_policy.\n"
					   "d : Save domain_policy.\n"
					   "a : Save all policies.\n"
					   "f : Save even if on-disk policy and on-memory policy are the same.\n\n"
					   "If no options given, this program assumes 'a' and 'f' are given.\n", argv[0]);
				return 0;
			}
		}
	}
	if (chdir("/etc/ccs/")) {
		printf("Directory /etc/ccs/ doesn't exist.\n");
		return 1;
	}
	if (access(".", W_OK) == EOF) {
		if (errno != EROFS || mount("/", "/", "rootfs", MS_REMOUNT, NULL) == EOF) {
			printf("Can't remount for read-write. (%s)\n", strerror(errno));
			return 1;
		}
		remount_root = 1;
	}

	/* Exclude nonexistent policy. */
	if (access("/proc/ccs/policy/system_policy", R_OK)) save_system_policy = 0;
	if (access("/proc/ccs/policy/exception_policy", R_OK)) save_exception_policy = 0;
	if (access("/proc/ccs/policy/domain_policy", R_OK)) save_domain_policy = 0;

	/* Repeat twice so that necessary permissions for this program are included in domain policy. */
	for (repeat = 0; repeat < 2; repeat++) {

		if (save_system_policy) {
			char *new_policy = ReadFile("/proc/ccs/policy/system_policy");
			char *old_policy = ReadFile("system_policy.txt");
			if (new_policy && (force_save || !old_policy || strcmp(new_policy, old_policy))) {
				int fd;
				snprintf(filename, sizeof(filename) - 1, "system_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
				if ((fd = open(filename, O_WRONLY | O_CREAT, 0600)) != EOF) {
					ftruncate(fd, 0);
					write(fd, new_policy, strlen(new_policy));
					close(fd);
					unlink("system_policy.txt");
					symlink(filename, "system_policy.txt");
				} else {
					printf("Can't create %s\n", filename);
				}
			}
			free(old_policy);
			free(new_policy);
		}
		
		if (save_exception_policy) {
			char *new_policy = ReadFile("/proc/ccs/policy/exception_policy");
			char *old_policy = ReadFile("exception_policy.txt");
			if (new_policy && (force_save || !old_policy || strcmp(new_policy, old_policy))) {
				int fd;
				snprintf(filename, sizeof(filename) - 1, "exception_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
				if ((fd = open(filename, O_WRONLY | O_CREAT, 0600)) != EOF) {
					ftruncate(fd, 0);
					write(fd, new_policy, strlen(new_policy));
					close(fd);
					unlink("exception_policy.txt");
					symlink(filename, "exception_policy.txt");
				} else {
					printf("Can't create %s\n", filename);
				}
			}
			free(old_policy);
			free(new_policy);
		}

	}
	
	if (save_domain_policy) {
		ReadDomainPolicy("/proc/ccs/policy/domain_policy", 0);
		for (repeat = 0; repeat < 10; repeat++) {
			//if (repeat) printf("Domain policy has changed while saving domain policy. Retrying.\n");
			if (access("domain_policy.txt", R_OK) == 0) ReadDomainPolicy("domain_policy.txt", 1);
			/* Need to save domain policy? */
			if (force_save || !IsSameDomainList()) {
				int fd;
				snprintf(filename, sizeof(filename) - 1, "domain_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
				if ((fd = open(filename, O_WRONLY | O_CREAT, 0600)) != EOF) {
					ftruncate(fd, 0);
					WriteDomainPolicy(fd, 0);
					close(fd);
					unlink("domain_policy.txt");
					symlink(filename, "domain_policy.txt");
				} else {
					printf("Can't create %s\n", filename);
				}
			}
			/* Has domain policy changed while saving domain policy? */
			ClearDomainPolicy(0);
			ReadDomainPolicy("/proc/ccs/policy/domain_policy", 0);
			if (IsSameDomainList()) break;
			ClearDomainPolicy(1);
		}
		ClearDomainPolicy(0);
		ClearDomainPolicy(1);
	}
	if (remount_root) mount("/", "/", "rootfs", MS_REMOUNT | MS_RDONLY, NULL);
	return 0;
}

#endif

#ifdef NEED_PATHMATCH

static int need_separator = 0;

static void print_path(const char *dir, const char *file, const char *trailer) {
	if (need_separator) putchar(' ');
	need_separator = 1;
	fprintf_encoded(stdout, dir);
	fprintf_encoded(stdout, file);
	fprintf_encoded(stdout, trailer);
}

static char *current_part = NULL;

static int scandir_filter(const struct dirent *buf) {
	char buffer[1024];
	char c;
	char *dp = buffer;
	const char *cp = buf->d_name;
	if (buf->d_type == DT_DIR) {
		if (strcmp(cp, ".") == 0 || strcmp(cp, "..") == 0) return 0;
	}
	if (strlen(cp) > 255) return 0;
	while ((c = *cp++) != '\0') {
		if (c == '\\') {
			*dp++ = '\\';
			*dp++ = '\\';
		} else if (c > ' ' && c < 127) {
			*dp++ = c;
		} else {
			*dp++ = '\\';
			*dp++ = ((c >> 6) + '0');
			*dp++ = (((c >> 3) & 7) + '0');
			*dp++ = ((c & 7) + '0');
		}
	}
	*dp = '\0';
	//printf("Compare: %s %s\n", buffer, current_part);
	if (FileMatchesToPattern(buffer, dp, current_part, strchr(current_part, '\0'))) return 1;
	return 0;
}

static int target_is_dir = 0; 
static int target_depth = 0;
static char **target_part = NULL;

static void ScanDir(const char *path, int depth) {
	struct dirent **namelist;
	int i, n;
	current_part = target_part[depth];
	//printf("Scan: %d %s\n", depth, current_part);
	if ((n = scandir(path, &namelist, scandir_filter, 0)) >= 0) {	
		for (i = 0; i < n; i++) {
			const char *cp = namelist[i]->d_name;
			const unsigned char type = namelist[i]->d_type;
			if (depth < target_depth - 1) {
				if (type == DT_DIR) {
					const int len = strlen(path) + strlen(cp) + 4;
					char *child_path = malloc(len);
					if (!child_path) OutOfMemory();
					snprintf(child_path, len - 1, "%s%s/", path, cp);
					//printf("Check: %s\n", child_path);
					ScanDir(child_path, depth + 1);
					free(child_path);
				}
			} else if (target_is_dir) {
				if (type == DT_DIR) {
					print_path(path, cp, "/");
				}
			} else if (type != DT_DIR) {
				print_path(path, cp, "");
			}
			free((void *) namelist[i]);
		}
		free((void *) namelist);
	}
}

static void do_pathmatch_main(char *target) {
	if (strcmp(target, "/") == 0) {
		printf("/\n");
	} else if (target[0] != '/') {
		putchar('\n');
	} else {
		char *cp;
		int i;
		target_is_dir = (*(strchr(target, '\0') - 1) == '/');
		target_depth = 0;
		cp = target + 1;
		for (i = 1; ; i++) {
			char c = target[i];
			if (c == '/' || c == '\0') {
				target[i] = '\0';
				target_part = (char **) realloc(target_part, (target_depth + 1) * sizeof(char *));
				if (target + i != cp) target_part[target_depth++] = cp;
				cp = target + i + 1;
				if (!c) break;
			}
		}
		//for (i = 0; i < target_depth; i++) printf("%d %s\n", i, target_part[i]);
		need_separator = 0;
		ScanDir("/", 0);
		putchar('\n');
	}
}

static int pathmatch_main(int argc, char *argv[]) {
	if (argc > 1) {
		int i;
		for (i = 1; i < argc; i++) do_pathmatch_main(argv[i]);
	} else {
		get();
		while (freadline(stdin)) do_pathmatch_main(shared_buffer);
		put();
	}
	return 0;
}

#endif

#ifdef NEED_LOADPOLICY

static int loadpolicy_main(int argc, char *argv[]) {
	int load_system_policy = 0;
	int load_exception_policy = 0;
	int load_domain_policy = 0;
	int refresh_policy = 0;
	if (access("/proc/ccs/policy/", F_OK)) {
		fprintf(stderr, "You can't run this program for this kernel.\n");
		return 0;
	}
	if (argc == 1) {
		goto usage;
	} else {
		int i;
		for (i = 1; i < argc; i++) {
			char *p = argv[i];
			char *s = strchr(p, 's');
			char *e = strchr(p, 'e');
			char *d = strchr(p, 'd');
			char *a = strchr(p, 'a');
			char *f = strchr(p, 'f');
			if (s || a) load_system_policy = 1;
			if (e || a) load_exception_policy = 1;
			if (d || a) load_domain_policy = 1;
			if (f) refresh_policy = 1;
			if (strcspn(p, "sedaf")) {
			usage: ;
				printf("%s [s][e][d][a][f]\n"
					   "s : Load system_policy.\n"
					   "e : Load exception_policy.\n"
					   "d : Load domain_policy.\n"
					   "a : Load all policies.\n"
					   "f : Delete on-memory policy before loading on-disk policy.\n\n", argv[0]);
				printf("Note that 'accept mode' might be unable to append permissions for this program if 'f' is used "
					   "because 'f' might delete 'domain for this program' and 'execute permission for parent domain'.\n");
				return 0;
			}
		}
	}
	if (chdir("/etc/ccs/")) {
		printf("Directory /etc/ccs/ doesn't exist.\n");
		return 1;
	}

	if (load_system_policy) {
		FILE *file_fp, *proc_fp;
		if ((file_fp = fopen("system_policy.txt", "r")) == NULL) {
			fprintf(stderr, "Can't open system_policy.txt\n");
			goto out_system;
		}
		if ((proc_fp = fopen("/proc/ccs/policy/system_policy", "w")) == NULL) {
			fprintf(stderr, "Can't open /proc/ccs/policy/system_policy\n");
			fclose(file_fp);
			goto out_system;
		}
		if (refresh_policy) {
			FILE *proc_clear_fp = fopen("/proc/ccs/policy/system_policy", "r");
			if (!proc_clear_fp) {
				fprintf(stderr, "Can't open /proc/ccs/policy/system_policy\n");
				fclose(file_fp);
				fclose(proc_fp);
				goto out_system;
			}
			get();
			while (freadline(proc_clear_fp)) {
				if (shared_buffer[0]) fprintf(proc_fp, "delete %s\n", shared_buffer);
			}
			put();
			fclose(proc_clear_fp);
			fflush(proc_fp);
		}
		get();
		while (freadline(file_fp)) {
			if (shared_buffer[0]) fprintf(proc_fp, "%s\n", shared_buffer);
		}
		put();
		fclose(proc_fp);
		fclose(file_fp);
	}
 out_system: ;
	
	if (load_exception_policy) {
		FILE *file_fp, *proc_fp;
		if ((file_fp = fopen("exception_policy.txt", "r")) == NULL) {
			fprintf(stderr, "Can't open exception_policy.txt\n");
			goto out_exception;
		}
		if ((proc_fp = fopen("/proc/ccs/policy/exception_policy", "w")) == NULL) {
			fprintf(stderr, "Can't open /proc/ccs/policy/exception_policy\n");
			fclose(file_fp);
			goto out_exception;
		}
		if (refresh_policy) {
			FILE *proc_clear_fp = fopen("/proc/ccs/policy/exception_policy", "r");
			if (!proc_clear_fp) {
				fprintf(stderr, "Can't open /proc/ccs/policy/exception_policy\n");
				fclose(file_fp);
				fclose(proc_fp);
				goto out_exception;
			}
			get();
			while (freadline(proc_clear_fp)) {
				if (shared_buffer[0]) fprintf(proc_fp, "delete %s\n", shared_buffer);
			}
			put();
			fclose(proc_clear_fp);
			fflush(proc_fp);
		}
		get();
		while (freadline(file_fp)) {
			char *cp = strchr(shared_buffer, '\n');
			if (cp) *cp = '\0';
			else if (!feof(file_fp)) break;
			NormalizeLine(shared_buffer);
			if (shared_buffer[0]) fprintf(proc_fp, "%s\n", shared_buffer);
		}
		put();
		fclose(proc_fp);
		fclose(file_fp);
	}
 out_exception: ;

	if (load_domain_policy) {
		int new_index;
		FILE *proc_fp = fopen("/proc/ccs/policy/domain_policy", "w");
		if (!proc_fp) {
			fprintf(stderr, "Can't open /proc/ccs/policy/domain_policy\n");
			goto out_domain;
		}
		ReadDomainPolicy("domain_policy.txt", 0);
		ReadDomainPolicy("/proc/ccs/policy/domain_policy", 1);
		if (domain_list_count[0] == 0) {
			fprintf(stderr, "Can't open domain_policy.txt\n");
			fclose(proc_fp);
			goto out_domain;
		}
		for (new_index = 0; new_index < domain_list_count[0]; new_index++) {
			const char *domainname = domain_list[0][new_index].domainname;
			const char **new_string_ptr = domain_list[0][new_index].string_ptr;
			const int new_string_count = domain_list[0][new_index].string_count;
			const int old_index = FindDomain(domainname, 1);
			int i, j;
			if (refresh_policy && old_index >= 0) {
				/* Old policy for this domain found. */
				const char **old_string_ptr = domain_list[1][old_index].string_ptr;
				const int old_string_count = domain_list[1][old_index].string_count;
				fprintf(proc_fp, "select %s\n", domainname);
				for (j = 0; j < old_string_count; j++) {
					for (i = 0; i < new_string_count; i++) {
						if (new_string_ptr[i] == old_string_ptr[j]) break;
					}
					/* Delete this entry from old policy if not found in new policy. */
					if (i == new_string_count) fprintf(proc_fp, "delete %s\n", old_string_ptr[j]);
				}
			} else {
				/* Old policy for this domain not found or Append to old policy. */
				fprintf(proc_fp, "%s\n", domainname);
			}
			for (i = 0; i < new_string_count; i++) fprintf(proc_fp, "%s\n", new_string_ptr[i]);
			if (old_index >= 0) domain_list[1][old_index].domainname = ""; /* Don't delete this domain later. */
		}
		if (refresh_policy) {
			int old_index;
			/* Delete all domains that are not defined in new policy. */
			for (old_index = 0; old_index < domain_list_count[1]; old_index++) {
				if (domain_list[1][old_index].domainname[0]) fprintf(proc_fp, "delete %s\n", domain_list[1][old_index].domainname);
			}
		}
		fclose(proc_fp);
	}
 out_domain: ;

	return 0;
}

#endif

#ifdef NEED_LDWATCH

static int ldwatch_main(int argc, char *argv[]) {
	PATHNAME_ENTRY *entry_list = NULL;
	int entry_list_count = 0;
	FILE *fp_policy;
	if (argc > 1 && strcmp(argv[1], "--help") == 0) {
		printf("Usage: %s file_to_exclude1 [file_to_exclude2 [...]]\n\n", argv[0]);
		printf("This program automatically registers files shown by 'ldconfig -NXp' as globally readable files.\n");
		printf("This program registers all files shown by 'ldconfig -NXp' by default, but you can specify files that you don't want to register by command line.\n");
		printf("For example, if you invoke\n");
		printf("  %s /lib/libcustom-1.0.0.so /lib/libcustom.so.1\n", argv[0]);
		printf("then, /lib/libcustom-1.0.0.so and /lib/libcustom.so.1 will be excluded from the result of 'ldconfig -NXp'.\n\n");
		printf("Start this program in one window, then update packages in another window.\n");
		printf("After you finished updating, wait for several seconds and terminate this program with 'Ctrl-C'.\n");
		return 0;
	}
	{
		const int fd = open("/proc/ccs/policy/exception_policy", O_RDWR);
		if (fd == EOF) {
			fprintf(stderr, "You can't run this daemon for this kernel.\n");
			return 1;
		} else if (write(fd, "", 0) != 0) {
			fprintf(stderr, "You need to register this program to /proc/ccs/policy/manager to run this program.\n");
			return 1;
		}
		close(fd);
	}
	if ((fp_policy = fopen("/proc/ccs/policy/exception_policy", "w")) == NULL) {
		fprintf(stderr, "Can't open policy file.\n");
		exit(1);
	}
	while (1) {
		struct stat64 buf;
		static time_t last_modified = 0;
		int i;
		if (stat64("/etc/ld.so.cache", &buf) == 0 && buf.st_mtime != last_modified) {
			FILE *fp_ldconfig;
			if ((fp_ldconfig = popen("/sbin/ldconfig -NXp", "r")) != NULL) {
				char buffer[16384];
				last_modified = buf.st_mtime;
				while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp_ldconfig)) {
					char *cp, *pathname, *real_pathname;
					if ((cp = strchr(buffer, '\n')) == NULL) continue;
					*cp = '\0';
					cp = strrchr(buffer, ' ');
					if (!cp || *++cp != '/') continue;
					// Check for duplicated entries.
					if ((real_pathname = realpath(cp, NULL)) == NULL) continue;
					for (i = 0; i < entry_list_count; i++) {
						if (strcmp(entry_list[i].real_pathname, real_pathname) == 0) break;
					}
					if (i < entry_list_count) {
						free(real_pathname);
						continue;
					}
					// Exclude if listed by command line.
					for (i = 1; i < argc; i++) {
						if (strcmp(argv[i], real_pathname) == 0 || strcmp(argv[i], cp) == 0) break;
					}
					if (i < argc) {
						printf("Skipped %s : %s\n", cp, real_pathname);
						free(real_pathname);
						continue;
					}
					// Add an entry.
					pathname = strdup(cp);
					entry_list = (PATHNAME_ENTRY *) realloc(entry_list, (entry_list_count + 1) * sizeof(PATHNAME_ENTRY));
					entry_list[entry_list_count].pathname = pathname;
					entry_list[entry_list_count++].real_pathname = real_pathname;
					printf("Added %s : %s\n", pathname, real_pathname);
					fprintf(fp_policy, "allow_read ");
					fprintf_encoded(fp_policy, real_pathname);
					fprintf(fp_policy, "\n");
					fflush(fp_policy);
				}
				pclose(fp_ldconfig);
			}
			printf("Monitoring %d files.\n", entry_list_count);
		}
		// Check entries for update.
		for (i = 0; i < entry_list_count; i++) {
			PATHNAME_ENTRY *ptr = &entry_list[i];
			char *real_pathname = realpath(ptr->pathname, NULL);
			if (real_pathname && strcmp(ptr->real_pathname, real_pathname)) {
				printf("Changed %s : %s -> %s\n", ptr->pathname, ptr->real_pathname, real_pathname);
				fprintf(fp_policy, "allow_read ");
				fprintf_encoded(fp_policy, real_pathname);
				fprintf(fp_policy, "\n");
				fflush(fp_policy);
				free(ptr->real_pathname); ptr->real_pathname = real_pathname; real_pathname = NULL;
			}
			free(real_pathname);
		}
		sleep(1);
	}
	fclose(fp_policy);
	return 0;
}

#endif

#ifdef NEED_FINDTEMP

static int decode(const char *ascii, char *bin) {
	char c, d, e;
	while ((c = *bin++ = *ascii++) != '\0') {
		if (c == '\\') {
			c = *ascii++;
			switch (c) {
			case '\\':      /* "\\" */
				continue;
			case '0':       /* "\ooo" */
			case '1':
			case '2':
			case '3':
				if ((d = *ascii++) >= '0' && d <= '7' && (e = *ascii++) >= '0' && e <= '7') {
					const unsigned char f =
						(((unsigned char) (c - '0')) << 6) +
						(((unsigned char) (d - '0')) << 3) +
						(((unsigned char) (e - '0')));
					if (f && (f <= ' ' || f >= 127)) {
						*(bin - 1) = f;
						continue; /* pattern is not \000 */
					}
				}
			}
			return 0;
		} else if (c <= ' ' || c >= 127) {
			return 0;
		}
	}
	return 1;
}

static int findtemp_main(int argc, char *argv[]) {
	const char **pattern_list = NULL;
	int pattern_list_count = 0;
	int i, j;
	char buffer[16384], buffer2[sizeof(buffer)];
	if (argc > 1) {
		if (strcmp(argv[1], "--all")) {
			printf("%s < domain_policy\n\n", argv[0]);
			return 0;
		}
	}
	while (memset(buffer, 0, sizeof(buffer)), fscanf(stdin, "%16380s", buffer) == 1) {
		if (buffer[0] != '/') continue;
		{
			struct stat64 buf;
			if (!decode(buffer, buffer2)) continue;
			if (lstat64(buffer2, &buf) == 0) continue;
		}
		for (i = 0; i < pattern_list_count; i++) {
			if (strcmp(pattern_list[i], buffer) == 0) break;
		}
		if (i < pattern_list_count) continue;
		if ((pattern_list = (const char **) realloc(pattern_list, sizeof(const char *) * (pattern_list_count + 1))) == NULL ||
			(pattern_list[pattern_list_count++] = strdup(buffer)) == NULL) {
			fprintf(stderr, "Out of memory.\n");
			exit(1);
		}
	}
	for (i = 0; i < pattern_list_count; i++) {
		for (j = i + 1; j < pattern_list_count; j++) {
			if (strcmp(pattern_list[i], pattern_list[j]) > 0) {
				const char *tmp = pattern_list[i]; pattern_list[i] = pattern_list[j]; pattern_list[j] = tmp;
			}
		}
	}
	for (i = 0; i < pattern_list_count; i++) printf("%s\n", pattern_list[i]);
	return 0;
}

#endif

#ifdef NEED_EDITPOLICY

static const char *policy_file = DOMAIN_POLICY_FILE;
static const char *list_caption = NULL;
static char *current_domain = NULL;

static int current_screen = SCREEN_DOMAIN_LIST;

// List for generic policy.
static char **generic_acl_list = NULL;
static int generic_acl_list_count = 0;

static char **domain_keepers = NULL;
static int domain_keepers_count = 0;
static char **initializers = NULL;
static int initializers_count = 0;

///////////////////////////  ACL HANDLER  //////////////////////////////

static int IsDomainKeeper(const int index) {
	int i;
	const char *domainname = domain_list[0][index].domainname;
	for (i = 0; i < domain_keepers_count; i++) {
		if (strcmp(domainname, domain_keepers[i]) == 0) return 1;
	}
	return 0;
}

static int IsInitializer(const char *filename) {
	if (filename) {
		int i;
		for (i = 0; i < initializers_count; i++) {
			if (strcmp(filename, initializers[i]) == 0) return 1;
		}
	}
	return 0;
}

static int IsInitializerDomain(const int index) {
	char *cp = strchr(domain_list[0][index].domainname, ' ');
	if (cp) return IsInitializer(cp + 1);
	return 0;
}

static int IsVirtualDomain(const int index) {
	return domain_list[0][index].is_virtual;
}

static const char *DomainName(const int index) {
	return domain_list[0][index].domainname;
}

///////////////////////////  UTILITY FUNCTIONS  //////////////////////////////

static int offline_mode = 0;
static int persistent_fd = EOF;

static void SendFD(char *data, int *fd) {
	struct msghdr msg;
	struct iovec iov = { data, strlen(data) };
	char cmsg_buf[CMSG_SPACE(sizeof(int))];
	struct cmsghdr *cmsg = (struct cmsghdr *) cmsg_buf; 
	memset(&msg, 0, sizeof(msg));
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	msg.msg_control = cmsg_buf;
	msg.msg_controllen = sizeof(cmsg_buf);
	cmsg->cmsg_level = SOL_SOCKET;
	cmsg->cmsg_type = SCM_RIGHTS;
	msg.msg_controllen = cmsg->cmsg_len = CMSG_LEN(sizeof(int));
	memmove(CMSG_DATA(cmsg), fd, sizeof(int));
	sendmsg(persistent_fd, &msg, 0);
	close(*fd);
}

static FILE *open_read(const char *filename) {
	if (offline_mode) {
		char request[1024];
		int fd[2];
		FILE *fp;
		if (socketpair(PF_UNIX, SOCK_STREAM, 0, fd)) {
			fprintf(stderr, "socketpair()\n");
			exit(1);
		}
		if (shutdown(fd[0], SHUT_WR) || (fp = fdopen(fd[0], "r")) == NULL) {
			close(fd[1]); close(fd[0]);
			exit(1);
		}
		memset(request, 0, sizeof(request));
		snprintf(request, sizeof(request) - 1, "GET %s", filename);
		SendFD(request, &fd[1]);
		return fp;
	} else {
		return fopen(filename, "r");
	}
}

static int open_write(const char *filename) {
	if (offline_mode) {
		char request[1024];
		int fd[2];
		if (socketpair(PF_UNIX, SOCK_STREAM, 0, fd)) {
			fprintf(stderr, "socketpair()\n");
			exit(1);
		}
		if (shutdown(fd[0], SHUT_RD)) {
			close(fd[1]); close(fd[0]);
			exit(1);
		}
		memset(request, 0, sizeof(request));
		snprintf(request, sizeof(request) - 1, "POST %s", filename);
		SendFD(request, &fd[1]);
		return fd[0];
	} else {
		return open(filename, O_WRONLY);
	}
}

static void ReadGenericPolicy(void) {
	FILE *fp;
	int i, j;
	while (generic_acl_list_count) free(generic_acl_list[--generic_acl_list_count]);
	if ((fp = open_read(policy_file)) != NULL) {
		if (current_screen == SCREEN_ACL_LIST) {
			int flag = 0;
			get();
			while (freadline(fp)) {
				if (IsDomainDef(shared_buffer)) {
					flag = strcmp(shared_buffer, current_domain) == 0 ? 1 : 0;
				} else if (flag && shared_buffer[0]) {
					if ((generic_acl_list = (char **) realloc(generic_acl_list, (generic_acl_list_count + 1) * sizeof(char *))) == NULL
						|| (generic_acl_list[generic_acl_list_count++] = strdup(shared_buffer)) == NULL) OutOfMemory();
				}
			}
			put();
			for (i = 0; i < generic_acl_list_count; i++) {
				for (j = i + 1; j < generic_acl_list_count; j++) {
					char *a = generic_acl_list[i], *b = generic_acl_list[j];
					if (*a && *b && strcmp(a + 1, b + 1) > 0) {
						generic_acl_list[i] = b; generic_acl_list[j] = a;
					}
				}
			}
		} else {
			get();
			while (freadline(fp)) {
				if ((generic_acl_list = (char **) realloc(generic_acl_list, (generic_acl_list_count + 1) * sizeof(char *))) == NULL
					|| (generic_acl_list[generic_acl_list_count++] = strdup(shared_buffer)) == NULL) OutOfMemory();
			}
			put();
			for (i = 0; i < generic_acl_list_count; i++) {
				for (j = i + 1; j < generic_acl_list_count; j++) {
					char *a = generic_acl_list[i], *b = generic_acl_list[j];
					if (strcmp(a, b) > 0) {
						generic_acl_list[i] = b; generic_acl_list[j] = a;
					}
				}
			}
		}
		fclose(fp);
	}
}

static void ReadDomainAndExceptionPolicy(void) {
	FILE *fp;
	int i, j;
	ClearDomainPolicy(0);
	while (initializers_count) free(initializers[--initializers_count]);
	while (domain_keepers_count) free(domain_keepers[--domain_keepers_count]);
	FindOrAssignNewDomain(ROOT_NAME, 0);

	// Load initializer list, domain_keeper list.
	if ((fp = open_read(EXCEPTION_POLICY_FILE)) != NULL) {
		get();
		while (freadline(fp)) {
			if (strncmp(shared_buffer, "initializer ", 12) == 0) {
				RemoveHeader(shared_buffer, 12);
				if (!IsCorrectPath(shared_buffer, 1, -1, -1)) continue;
				if ((initializers = (char **) realloc(initializers, (initializers_count + 1) * sizeof(char *))) == NULL
					|| (initializers[initializers_count++] = strdup(shared_buffer)) == NULL) OutOfMemory();
			} else if (strncmp(shared_buffer, "keep_domain ", 12) == 0) {
				RemoveHeader(shared_buffer, 12);
				if (!IsCorrectDomain(shared_buffer)) continue;
				if ((domain_keepers = (char **) realloc(domain_keepers, (domain_keepers_count + 1) * sizeof(char *))) == NULL
					|| (domain_keepers[domain_keepers_count++] = strdup(shared_buffer)) == NULL) OutOfMemory();
			}
		}
		put();
		fclose(fp);
	}

	// Load all domain list.
	if ((fp = open_read(DOMAIN_POLICY_FILE)) != NULL) {
		int index = EOF;
		get();
		while (freadline(fp)) {
			char *cp;
			if (IsDomainDef(shared_buffer)) {
				index = FindOrAssignNewDomain(shared_buffer, 0);
			} else if (index >= 0 && (atoi(shared_buffer) & 1) == 1 && (cp = strchr(shared_buffer, ' ')) != NULL &&
					   IsCorrectPath(cp + 1, 1, -1, -1) && IsInitializer(cp + 1)) {
				int virtual_index;
				static char *buffer2 = NULL;
				if (!buffer2 && (buffer2 = malloc(shared_buffer_len)) == NULL) OutOfMemory();
				memset(buffer2, 0, shared_buffer_len);
				snprintf(buffer2, shared_buffer_len - 1, "%s %s", domain_list[0][index].domainname, cp + 1);
				NormalizeLine(buffer2);
				if ((virtual_index = FindOrAssignNewDomain(buffer2, 0)) == EOF) {
					fprintf(stderr, "ERROR: Can't create domain for '%s'.\n", buffer2);
					exit(127);
				}
				domain_list[0][virtual_index].is_virtual = 1;
			}
		}
		put();
		fclose(fp);
	}
	// Clear is_virtual flag for initialized domains.
	for (i = 0; i < domain_list_count[0]; i++) {
		char *cp;
		if (!domain_list[0][i].is_virtual) continue;
		if ((cp = strchr(domain_list[0][i].domainname, ' ')) != NULL && strchr(cp + 1, ' ') == NULL) domain_list[0][i].is_virtual = 0;
	}
	// Sort by domain name.
	for (i = 0; i < domain_list_count[0]; i++) {
		for (j = i + 1; j < domain_list_count[0]; j++) {
			if (strcmp(domain_list[0][i].domainname, domain_list[0][j].domainname) > 0) {
				DOMAIN_INFO tmp = domain_list[0][i]; domain_list[0][i] = domain_list[0][j]; domain_list[0][j] = tmp;
			}
		}
	}
}

////////////////////////////  UI HANDLER  ////////////////////////////

static int getch2(void) {
	static int c0 = 0, c1 = 0, c2 = 0, c3 = 0, len = 0;
	if (len > 0) { c0 = c1; c1 = c2; c2 = c3; len--; return c0; }
	c0 = getch(); if (c0 != 0x1B) return c0;
	c1 = getch(); if (c1 != '[') { len = 1; return c0; }
	c2 = getch(); if (c2 != '5' && c2 != '6') { len = 2; return c0; }
	c3 = getch(); if (c3 != '~') { len = 3; return c0; }
	return c2 == '5' ? KEY_PPAGE : KEY_NPAGE;
}

static void ShowCurrent(void);

static int window_width = 0, window_height = 0;
static int current_y[MAXSCREEN], current_item_index[MAXSCREEN], list_item_count[MAXSCREEN];

static struct {
	int current_y[MAXSCREEN];
	int current_item_index[MAXSCREEN];
	char search_buffer[4096];
} history;

static const int header_lines = 3;
static int body_lines = 0;

static void ShowList(void) {
	const int offset = current_item_index[current_screen];
	int i;
	if (current_screen == SCREEN_DOMAIN_LIST) list_item_count[SCREEN_DOMAIN_LIST] = domain_list_count[0];
	else list_item_count[current_screen] = generic_acl_list_count;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	if (current_screen == SCREEN_DOMAIN_LIST) mvprintw(0, 0, "<<< Domain Transition Editor >>>      %d domain%c", list_item_count[SCREEN_DOMAIN_LIST], list_item_count[SCREEN_DOMAIN_LIST] > 1 ? 's' : ' ');
	else mvprintw(0, 0, "<<< %s Editor >>>      %d entr%s", list_caption, list_item_count[current_screen], list_item_count[current_screen] > 1 ? "ies" : "y");
	if (current_screen == SCREEN_DOMAIN_LIST) mvprintw(1, 0, "Commands = Q:quit A:append D:delete R:refresh Enter:edit @:search Tab:switch");
	else if (current_screen == SCREEN_ACL_LIST) mvprintw(1, 0, "Commands = Q:quit A:append D:delete R:refresh Enter:list");
	else mvprintw(1, 0, "Commands = Q:quit A:append D:delete R:refresh Tab:switch");
	if (current_screen == SCREEN_ACL_LIST) mvprintw(2, 0, "%s", current_domain);
	for (i = 0; i < body_lines; i++) {
		const int index = offset + i;
		if (index >= list_item_count[current_screen]) break;
		if (current_screen == SCREEN_DOMAIN_LIST) {
			const char *sp, *cp;
			mvprintw(header_lines + i, 0, "%5d: %c%c ", index, IsDomainKeeper(index) ? '#' : ' ', IsInitializerDomain(index) ? '*' : ' ');
			sp = DomainName(index);
			while ((cp = strchr(sp, ' ')) != NULL) { printw("    "); sp = cp + 1; }
			printw("%s", sp);
			if (IsVirtualDomain(index)) {
				get();
				memset(shared_buffer, 0, shared_buffer_len);
				snprintf(shared_buffer, shared_buffer_len - 1, ROOT_NAME "%s", strrchr(DomainName(index), ' '));
				printw(" ( -> %d )", FindDomain(shared_buffer, 0));
				put();
			}
		} else {
			mvprintw(header_lines + i, 0, "%5d: %s", index, generic_acl_list[index]);
		}
		clrtoeol();
	}
	ShowCurrent();
}

static void ResizeWindow(void) {
	getmaxyx(stdscr, window_height, window_width);
	body_lines = window_height - header_lines;
	if (body_lines <= current_y[current_screen]) current_y[current_screen] = body_lines - 1;
	if (current_y[current_screen] < 0) current_y[current_screen] = 0;
}

static void UpArrowKey(void) {
	if (current_y[current_screen] > 0) {
		current_y[current_screen]--;
		ShowCurrent();
	} else if (current_item_index[current_screen] > 0) {
		current_item_index[current_screen]--;
		ShowList();
	}
}

static void DownArrowKey(void) {
	if (current_y[current_screen] < body_lines - 1) {
		if (current_item_index[current_screen] + current_y[current_screen] < list_item_count[current_screen] - 1) {
			current_y[current_screen]++;
			ShowCurrent();
		}
	} else if (current_item_index[current_screen] + current_y[current_screen] < list_item_count[current_screen] - 1) {
		current_item_index[current_screen]++;
		ShowList();
	}
}

static void PageUpKey(void) {
	if (current_item_index[current_screen] + current_y[current_screen] > body_lines) {
		current_item_index[current_screen] -= body_lines;
		if (current_item_index[current_screen] < 0) current_item_index[current_screen] = 0;
		ShowList();
	} else if (current_item_index[current_screen] + current_y[current_screen] > 0) {
		current_item_index[current_screen] = 0;
		current_y[current_screen] = 0;
		ShowList();
	}
}

static void PageDownKey(void) {
	if (list_item_count[current_screen] - current_item_index[current_screen] > body_lines) {
		current_item_index[current_screen] += body_lines;
		if (current_item_index[current_screen] + current_y[current_screen] > list_item_count[current_screen] - 1) current_y[current_screen] = list_item_count[current_screen] - 1 - current_item_index[current_screen];
		ShowList();
	} else if (current_item_index[current_screen] + current_y[current_screen] < list_item_count[current_screen] - 1) {
		current_y[current_screen] = list_item_count[current_screen] - current_item_index[current_screen] - 1;
		ShowCurrent();
	}
}

static int GetCurrent(void) {
	if (list_item_count[current_screen] == 0) return EOF;
	if (current_item_index[current_screen] + current_y[current_screen] < 0 || current_item_index[current_screen] + current_y[current_screen] >= list_item_count[current_screen]) {
		fprintf(stderr, "ERROR: current_item_index=%d current_y=%d\n", current_item_index[current_screen], current_y[current_screen]);
		exit(127);
	}
	return current_item_index[current_screen] + current_y[current_screen];
}

static void ShowCurrent(void) {
	if (current_screen == SCREEN_DOMAIN_LIST) {
		const int current = GetCurrent();
		get();
		memset(shared_buffer, 0, shared_buffer_len);
		snprintf(shared_buffer, shared_buffer_len - 1, "%s", DomainName(current));
		if (window_width < shared_buffer_len) shared_buffer[window_width] = '\0';
		move(2, 0);
		clrtoeol();
		printw("%s", shared_buffer);
		put();
	}
	move(header_lines + current_y[current_screen], 0);
	refresh();
}

static void LoadCursorPos(const int current_screen) {
	current_item_index[current_screen] = history.current_item_index[current_screen];
	current_y[current_screen] = history.current_y[current_screen];
}

static void AdjustCursorPos(const int item_count) {
	if (item_count == 0) {
		current_item_index[current_screen] = current_y[current_screen] = 0;
	} else {
		while (current_item_index[current_screen] + current_y[current_screen] >= item_count) {
			if (current_y[current_screen] > 0) current_y[current_screen]--;
			else if (current_item_index[current_screen] > 0) current_item_index[current_screen]--;
		}
	}
}

static void SaveCursorPos(const int current_screen) {
	history.current_item_index[current_screen] = current_item_index[current_screen];
	history.current_y[current_screen] = current_y[current_screen];
}

static int GenericListLoop(void) {
	if (current_screen == SCREEN_SYSTEM_LIST) {
		policy_file = SYSTEM_POLICY_FILE;
		list_caption = "System Policy";
	} else if (current_screen == SCREEN_EXCEPTION_LIST) {
		policy_file = EXCEPTION_POLICY_FILE;
		list_caption = "Exception Policy";
	} else {
		policy_file = DOMAIN_POLICY_FILE;
		list_caption = "Domain Policy";
	}
	LoadCursorPos(current_screen);
 start:
	ReadGenericPolicy();
	AdjustCursorPos(generic_acl_list_count);
	ShowList();
	while (1) {
		const int c = getch2();
		SaveCursorPos(current_screen);
		if (c == 'q' || c == 'Q') return MAXSCREEN;
		if (current_screen == SCREEN_ACL_LIST) {
			if (c == '\r') return SCREEN_DOMAIN_LIST;
		} else {
			if (c == '\t') {
				if (current_screen == SCREEN_SYSTEM_LIST) return SCREEN_EXCEPTION_LIST;
				else if (current_screen == SCREEN_EXCEPTION_LIST) return SCREEN_DOMAIN_LIST;
			}
		}
		if (c == ERR) continue; // Ignore invalid key.
		switch(c) {
		case KEY_RESIZE:
			ResizeWindow();
			ShowList();
			break;
		case KEY_UP:
			UpArrowKey();
			break;
		case KEY_DOWN:
			DownArrowKey();
			break;
		case KEY_PPAGE:
			PageUpKey();
			break;
		case KEY_NPAGE:
			PageDownKey();
			break;
		case 'd':
		case 'D':
			{
				const int index = GetCurrent();
				if (index >= 0 && index < generic_acl_list_count) {
					const int fd = open_write(policy_file);
					if (fd != EOF) {
						if (current_screen == SCREEN_ACL_LIST) {
							write(fd, "select ", 7); write(fd, current_domain, strlen(current_domain)); write(fd, "\n", 1);
						}
						write(fd, "delete ", 7); write(fd, generic_acl_list[index], strlen(generic_acl_list[index])); write(fd, "\n", 1); close(fd);
						close(fd);
					}
					goto start;
				}
			}
			break;
		case 'a':
		case 'A':
			{
				get();
				mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
				memset(shared_buffer, 0, shared_buffer_len);
				scrollok(stdscr, 1); getnstr(shared_buffer, shared_buffer_len - 1); scrollok(stdscr, 0); noecho();
				NormalizeLine(shared_buffer);
				if (shared_buffer[0]) {
					const int fd = open_write(policy_file);
					if (fd != EOF) {
						if (current_screen == SCREEN_ACL_LIST) {
							write(fd, "select ", 7); write(fd, current_domain, strlen(current_domain)); write(fd, "\n", 1);
						}
						write(fd, shared_buffer, strlen(shared_buffer)); write(fd, "\n", 1); close(fd);
					}
				}
				put();
				goto start;
			}
			break;
		case 'r':
		case 'R':
			goto start;
		}
	}
}

static int DomainListLoop(void) {
	LoadCursorPos(current_screen);
 start: ;
	ReadDomainAndExceptionPolicy();
	AdjustCursorPos(domain_list_count[0]);
	ShowList();
	while (1) {
		const int c = getch2();
		SaveCursorPos(current_screen);
		if (c == 'q' || c == 'Q') return MAXSCREEN;
		if (c == '\t') return SCREEN_SYSTEM_LIST;
		if (c == ERR) continue; // Ignore invalid key.
		switch (c) {
		case KEY_RESIZE:
			ResizeWindow();
			ShowList();
			break;
		case KEY_UP:
			UpArrowKey();
			break;
		case KEY_DOWN:
			DownArrowKey();
			break;
		case KEY_PPAGE:
			PageUpKey();
			break;
		case KEY_NPAGE:
			PageDownKey();
			break;
		case '@':
			{
				int i;
			input_path:
				mvprintw(window_height - 1, 0, "Search> "); clrtoeol(); refresh(); echo();
				memset(history.search_buffer, 0, sizeof(history.search_buffer));
				scrollok(stdscr, 1); getnstr(history.search_buffer, sizeof(history.search_buffer) - 1); scrollok(stdscr, 0); noecho();
				NormalizeLine(history.search_buffer);
				if (history.search_buffer[0]) {
					for (i = 0; i < list_item_count[current_screen]; i++) {
						const char *cp = DomainName(i);
						if (strchr(history.search_buffer, '/')) {
							if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
						} else {
							if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
						}
						if (strcmp(cp, history.search_buffer)) continue;
						while (i < current_y[current_screen] + current_item_index[current_screen]) {
							if (current_y[current_screen] > 0) current_y[current_screen]--;
							else current_item_index[current_screen]--;
						}
						while (i > current_y[current_screen] + current_item_index[current_screen]) {
							if (current_y[current_screen] < body_lines - 1) current_y[current_screen]++;
							else current_item_index[current_screen]++;
						}
						break;
					}
				}
				ShowList();
			}
			break;
		case KEY_LEFT:
			{
				int i;
				if (!history.search_buffer[0]) goto input_path;
				for (i = GetCurrent() - 1; i >= 0; i--) {
					const char *cp = DomainName(i);
					if (strchr(history.search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, history.search_buffer)) continue;
					while (i < current_y[current_screen] + current_item_index[current_screen]) {
						if (current_y[current_screen] > 0) current_y[current_screen]--;
						else current_item_index[current_screen]--;
					}
					ShowList();
					break;
				}
			}
			break;
		case KEY_RIGHT:
			{
				int i;
				if (!history.search_buffer[0]) goto input_path;
				for (i = GetCurrent() + 1; i < list_item_count[current_screen]; i++) {
					const char *cp = DomainName(i);
					if (strchr(history.search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, history.search_buffer)) continue;
					while (i > current_y[current_screen] + current_item_index[current_screen]) {
						if (current_y[current_screen] < body_lines - 1) current_y[current_screen]++;
						else current_item_index[current_screen]++;
					}
					ShowList();
					break;
				}
			}
			break;
		case 'a':
		case 'A':
			{
				const int current = GetCurrent();
				if (!IsVirtualDomain(current)) {
					get();
					mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
					memset(shared_buffer, 0, shared_buffer_len);
					scrollok(stdscr, 1); getnstr(shared_buffer, shared_buffer_len - 1); scrollok(stdscr, 0); noecho();
					NormalizeLine(shared_buffer);
					if (IsCorrectPath(shared_buffer, 1, -1, -1) && !IsInitializer(shared_buffer)) {
						const int fd = open_write(DOMAIN_POLICY_FILE);
						if (fd != EOF) {
							const char *domainname = DomainName(current);
							write(fd, domainname, strlen(domainname)); write(fd, " ", 1); write(fd, shared_buffer, strlen(shared_buffer)); write(fd, "\n", 1);
							close(fd);
						}
					}
					put();
					goto start;
				}
			}
			break;
		case 'd':
		case 'D':
			{
				const int current = GetCurrent();
				if (current > 0 && !IsVirtualDomain(current)) { // Never delete ROOT_NAME
					const char *key = DomainName(current);
					int c;
					mvprintw(header_lines, 0, "Delete '%s' and its descendant domains? (Y/N):", key);
					clrtoeol();
					refresh();
					do {
						c = getch2();
					} while (!(c == 'Y' || c == 'y' || c == 'N' || c == 'n' || c == EOF));
					ResizeWindow();
					if (c == 'Y' || c == 'y') {
						const int fd = open_write(DOMAIN_POLICY_FILE);
						if (fd != EOF) {
							const int key_len = strlen(key);
							int index;
							for (index = 0; index < domain_list_count[0]; index++) {
								const char *cp = DomainName(index);
								if (strncmp(cp, key, key_len) || (cp[key_len] != '\0' && cp[key_len] != ' ')) continue;
								write(fd, "delete ", 7); write(fd, cp, strlen(cp)); write(fd, "\n", 1);
							}
							close(fd);
						}						
						goto start;
					}
					ShowList();
				}
			}
			break;
		case 'r':
		case 'R':
			goto start;
		case '\r':
			{
				const int current = GetCurrent();
				if (IsVirtualDomain(current)) {
					int redirect_index;
					get();
					memset(shared_buffer, 0, shared_buffer_len);
					snprintf(shared_buffer, shared_buffer_len - 1, ROOT_NAME "%s", strrchr(DomainName(current), ' '));
					redirect_index = FindDomain(shared_buffer, 0);
					put();
					if (redirect_index != EOF) {
						current_item_index[current_screen] = redirect_index - current_y[current_screen];
						while (current_item_index[current_screen] < 0) {
							current_item_index[current_screen]++; current_y[current_screen]--;
						}
						ShowList();
					}
				} else {
					free(current_domain);
					if ((current_domain = strdup(DomainName(current))) == NULL) OutOfMemory();
					return SCREEN_ACL_LIST;
				}
			}
			break;
		}
	}
}

static void policy_daemon(void);

static int editpolicy_main(int argc, char *argv[]) {
	memset(&history, 0, sizeof(history));
	memset(current_y, 0, sizeof(current_y));
	memset(current_item_index, 0, sizeof(current_item_index));
	memset(list_item_count, 0, sizeof(list_item_count));
	if (argc > 1) {
		if (strcmp(argv[1], "s") == 0) current_screen = SCREEN_SYSTEM_LIST;
		else if (strcmp(argv[1], "e") == 0) current_screen = SCREEN_EXCEPTION_LIST;
		else if (strcmp(argv[1], "d") == 0) current_screen = SCREEN_DOMAIN_LIST;
		else {
			printf("Usage: %s [s|e|d]\n", argv[0]);
			return 1;
		}
	}
	{
		char *cp = strrchr(argv[0], '/');
		if (!cp) cp = argv[0];
		else cp++;
		if (strcmp(cp, "editpolicy_offline") == 0) offline_mode = 1;
	}
	if (offline_mode) {
		int fd[2];
		if (chdir("/etc/ccs/")) {
			fprintf(stderr, "/etc/ccs/ doesn't exist.\n");
			return 1;
		}
		if (socketpair(PF_UNIX, SOCK_STREAM, 0, fd)) {
			fprintf(stderr, "socketpair()\n");
			exit(1);
		}
		switch (fork()) {
		case 0:
			close(fd[0]);
			persistent_fd = fd[1];
			policy_daemon();
			_exit(0);
		case -1:
			fprintf(stderr, "fork()\n");
			exit(1);
		}
		close(fd[1]);
		persistent_fd = fd[0];
		{
			int fd0, fd1, len;
			char buffer[1024];
			if ((fd0 = open("system_policy.txt", O_RDONLY)) != EOF) {
				fd1 = open_write(SYSTEM_POLICY_FILE);
				while ((len = read(fd0, buffer, sizeof(buffer))) > 0) write(fd1, buffer, len);
				close(fd1); close(fd0);
			}
			if ((fd0 = open("exception_policy.txt", O_RDONLY)) != EOF) {
				fd1 = open_write(EXCEPTION_POLICY_FILE);
				while ((len = read(fd0, buffer, sizeof(buffer))) > 0) write(fd1, buffer, len);
				close(fd1); close(fd0);
			}
			if ((fd0 = open("domain_policy.txt", O_RDONLY)) != EOF) {
				fd1 = open_write(DOMAIN_POLICY_FILE);
				while ((len = read(fd0, buffer, sizeof(buffer))) > 0) write(fd1, buffer, len);
				close(fd1); close(fd0);
			}
		}
	} else {
		if (chdir("/proc/ccs/policy/")) {
			fprintf(stderr, "You can't use this editor for this kernel.\n");
			return 1;
		}
		{
			const int fd1 = open(SYSTEM_POLICY_FILE, O_RDWR), fd2 = open(EXCEPTION_POLICY_FILE, O_RDWR), fd3 = open(DOMAIN_POLICY_FILE, O_RDWR);
			if ((fd1 != EOF && write(fd1, "", 0) != 0) || (fd2 != EOF && write(fd2, "", 0) != 0) || (fd3 != EOF && write(fd3, "", 0) != 0)) {
				fprintf(stderr, "You need to register this program to /proc/ccs/policy/manager to run this program.\n");
				return 1;
			}
			close(fd1); close(fd2); close(fd3);
		}
	}
	initscr();
	cbreak();
	noecho();
	nonl();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);
	getmaxyx(stdscr, window_height, window_width);
	while (current_screen < MAXSCREEN) {
		if (!offline_mode) {
			if (current_screen == SCREEN_DOMAIN_LIST && access(DOMAIN_POLICY_FILE, F_OK)) current_screen = SCREEN_SYSTEM_LIST;
			else if (current_screen == SCREEN_SYSTEM_LIST && access(SYSTEM_POLICY_FILE, F_OK)) current_screen = SCREEN_EXCEPTION_LIST;
			else if (current_screen == SCREEN_EXCEPTION_LIST && access(EXCEPTION_POLICY_FILE, F_OK)) current_screen = SCREEN_DOMAIN_LIST;
		}
		ResizeWindow();
		if (current_screen == SCREEN_DOMAIN_LIST) current_screen = DomainListLoop();
		else current_screen = GenericListLoop();
	}
	clear();
	move(0, 0);
	refresh();
	endwin();
	if (offline_mode) {
		int fd, len;
		FILE *fp;
		time_t now = time(NULL);
		struct tm *tm = localtime(&now);
		char filename[1024], buffer[1024];
		memset(filename, 0, sizeof(filename));
		snprintf(filename, sizeof(filename) - 1, "system_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
		if ((fd = open(filename, O_WRONLY | O_CREAT, 0600)) != EOF) {
			if ((fp = open_read(SYSTEM_POLICY_FILE)) != NULL) {
				while ((len = fread(buffer, 1, sizeof(buffer), fp)) > 0) write(fd, buffer, len);
				close(fd); fclose(fp);
				unlink("system_policy.txt");
				symlink(filename, "system_policy.txt");
			}
		}
		snprintf(filename, sizeof(filename) - 1, "exception_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
		if ((fd = open(filename, O_WRONLY | O_CREAT, 0600)) != EOF) {
			if ((fp = open_read(EXCEPTION_POLICY_FILE)) != NULL) {
				while ((len = fread(buffer, 1, sizeof(buffer), fp)) > 0) write(fd, buffer, len);
				close(fd); fclose(fp);
				unlink("exception_policy.txt");
				symlink(filename, "exception_policy.txt");
			}
		}
		snprintf(filename, sizeof(filename) - 1, "domain_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
		if ((fd = open(filename, O_WRONLY | O_CREAT, 0600)) != EOF) {
			if ((fp = open_read(DOMAIN_POLICY_FILE)) != NULL) {
				while ((len = fread(buffer, 1, sizeof(buffer), fp)) > 0) write(fd, buffer, len);
				close(fd); fclose(fp);
				unlink("domain_policy.txt");
				symlink(filename, "domain_policy.txt");
			}
		}
	}
	return 0;
}

static void policy_daemon(void) {
	get();
	while (1) {
		static const char **exception_list = NULL, **system_list = NULL;
		static int exception_list_count = 0, system_list_count = 0;
		FILE *fp;
		{
			struct msghdr msg;
			struct iovec iov = { shared_buffer, shared_buffer_len - 1 };
			char cmsg_buf[CMSG_SPACE(sizeof(int))];
			struct cmsghdr *cmsg = (struct cmsghdr *) cmsg_buf;
			memset(&msg, 0, sizeof(msg));
			msg.msg_iov = &iov;
			msg.msg_iovlen = 1;
			msg.msg_control = cmsg_buf;
			msg.msg_controllen = sizeof(cmsg_buf);
			memset(shared_buffer, 0, shared_buffer_len);
			errno = 0;
			if (recvmsg(persistent_fd, &msg, 0) > 0 &&
				(cmsg = CMSG_FIRSTHDR(&msg)) != NULL &&
				cmsg->cmsg_level == SOL_SOCKET &&
				cmsg->cmsg_type == SCM_RIGHTS &&
				cmsg->cmsg_len == CMSG_LEN(sizeof(int))) {
				const int fd = * (int *) CMSG_DATA(cmsg);
				if ((fp = fdopen(fd, "w+")) == NULL) {
					close(fd);
					continue;
				}
			} else {
				break;
			}
		}
		if (strncmp(shared_buffer, "POST ", 5) == 0) {
			if (strcmp(shared_buffer + 5, "domain_policy") == 0) {
				int index = EOF;
				while (freadline(fp)) {
					int is_delete = 0, is_select = 0;
					if (strncmp(shared_buffer, "delete ", 7) == 0) {
						is_delete = 1;
						RemoveHeader(shared_buffer, 7);
					} else if (strncmp(shared_buffer, "select ", 7) == 0) {
						is_select = 1;
						RemoveHeader(shared_buffer, 7);
					}
					if (IsDomainDef(shared_buffer)) {
						if (is_delete) {
							index = FindDomain(shared_buffer, 0);
							if (index > 0) DeleteDomain(index, 0);
							index = EOF;
						} else if (is_select) {
							index = FindDomain(shared_buffer, 0);
						} else {
							index = FindOrAssignNewDomain(shared_buffer, 0);
						}
					} else if (index >= 0 && shared_buffer[0]) {
						unsigned int profile;
						if (sscanf(shared_buffer, "use_profile %u", &profile) == 1) {
							domain_list[0][index].profile = (unsigned char) profile;
						} else if (is_delete) {
							DelStringEntry(shared_buffer, index, 0);
						} else {
							AddStringEntry(shared_buffer, index, 0);
						}
					}
				}
			} else if (strcmp(shared_buffer + 5, "exception_policy") == 0) {
				while (freadline(fp)) {
					if (!shared_buffer[0]) continue;
					if (strncmp(shared_buffer, "delete ", 7) == 0) {
						int i;
						RemoveHeader(shared_buffer, 7);
						for (i = 0; i < exception_list_count; i++) {
							if (strcmp(exception_list[i], shared_buffer)) continue;
							for (exception_list_count--; i < exception_list_count; i++) exception_list[i] = exception_list[i + 1];
							break;
						}
					} else {
						if ((exception_list = (const char **) realloc(exception_list, (exception_list_count + 1) * sizeof(char *))) == NULL
							|| (exception_list[exception_list_count++] = SaveName(shared_buffer)) == NULL) OutOfMemory();
					}
				}
			} else if (strcmp(shared_buffer + 5, "system_policy") == 0) {
				while (freadline(fp)) {
					if (!shared_buffer[0]) continue;
					if (strncmp(shared_buffer, "delete ", 7) == 0) {
						int i;
						RemoveHeader(shared_buffer, 7);
						for (i = 0; i < system_list_count; i++) {
							if (strcmp(system_list[i], shared_buffer)) continue;
							for (system_list_count--; i < system_list_count; i++) system_list[i] = system_list[i + 1];
							break;
						}
					} else {
						if ((system_list = (const char **) realloc(system_list, (system_list_count + 1) * sizeof(char *))) == NULL
							|| (system_list[system_list_count++] = SaveName(shared_buffer)) == NULL) OutOfMemory();
					}
				}
			}
		} else if (strncmp(shared_buffer, "GET ", 4) == 0) {
			if (strcmp(shared_buffer + 4, "domain_policy") == 0) {
				int i, j;
				for (i = 0; i < domain_list_count[0]; i++) {
					const char **string_ptr = domain_list[0][i].string_ptr;
					const int string_count = domain_list[0][i].string_count;
					fprintf(fp, "%s\nuse_profile %u\n\n", domain_list[0][i].domainname, domain_list[0][i].profile);
					for (j = 0; j < string_count; j++) {
						fprintf(fp, "%s\n", string_ptr[j]);
					}
					fprintf(fp, "\n");
				}
			} else if (strcmp(shared_buffer + 4, "exception_policy") == 0) {
				int i;
				for (i = 0; i < exception_list_count; i++) fprintf(fp, "%s\n", exception_list[i]);
			} else if (strcmp(shared_buffer + 4, "system_policy") == 0) {
				int i;
				for (i = 0; i < system_list_count; i++) fprintf(fp, "%s\n", system_list[i]);
			}
		}
		fclose(fp);
	}
	put();
	_exit(0);
}

#endif

#ifdef NEED_CHECKPOLICY

static unsigned int line = 0, errors = 0, warnings = 0;

static char *FindConditionPart(char *data) {
	char *cp = strstr(data, " if "), *cp2;
	if (cp) {
		while ((cp2 = strstr(cp + 4, " if ")) != NULL) cp = cp2;
		*cp++ = '\0';
	}
	return cp;
}

static int parse_ulong(unsigned long *result, const char **str) {
	const char *cp = *str;
	char *ep;
	int base = 10;
	if (*cp == '0') {
		char c = * (cp + 1);
		if (c == 'x' || c == 'X') {
			base = 16; cp += 2;
		} else if (c >= '0' && c <= '7') {
			base = 8; cp++;
		}
	}
	*result = strtoul(cp, &ep, base);
	if (cp == ep) return 0;
	*str = ep;
	return (base == 16 ? VALUE_TYPE_HEXADECIMAL : (base == 8 ? VALUE_TYPE_OCTAL : VALUE_TYPE_DECIMAL));
}

static struct {
	const char *keyword;
	const int keyword_len; /* strlen(keyword) */
} condition_control_keyword[] = {
	{ "task.uid",           8 },
	{ "task.euid",          9 },
	{ "task.suid",          9 },
	{ "task.fsuid",        10 },
	{ "task.gid",           8 },
	{ "task.egid",          9 },
	{ "task.sgid",          9 },
	{ "task.fsgid",        10 },
	{ "task.pid",           8 },
	{ "task.ppid",          9 },
	{ "path1.uid",          9 },
	{ "path1.gid",          9 },
	{ "path1.ino",          9 },
	{ "path1.parent.uid",  16 },
	{ "path1.parent.gid",  16 },
	{ "path1.parent.ino",  16 },
	{ "path2.parent.uid",  16 },
	{ "path2.parent.gid",  16 },
	{ "path2.parent.ino",  16 },
	{ NULL, 0 }
};

static int CheckCondition(const char *condition) {
	const char *start = condition;
	int left, right;
	unsigned long left_min = 0, left_max = 0, right_min = 0, right_max = 0;
	if (strncmp(condition, "if ", 3)) goto out;
	condition += 3;
	while (*condition) {
		if (*condition == ' ') condition++;
		for (left = 0; condition_control_keyword[left].keyword; left++) {
			if (strncmp(condition, condition_control_keyword[left].keyword, condition_control_keyword[left].keyword_len) == 0) {
				condition += condition_control_keyword[left].keyword_len;
				break;
			}
		}
		if (!condition_control_keyword[left].keyword) {
			if (!parse_ulong(&left_min, &condition)) goto out;
			if (*condition == '-') {
				condition++;
				if (!parse_ulong(&left_max, &condition) || left_min > left_max) goto out;
			}
		}
		if (strncmp(condition, "!=", 2) == 0) condition += 2;
		else if (*condition == '=') condition++;
		else goto out;
		for (right = 0; condition_control_keyword[right].keyword; right++) {
			if (strncmp(condition, condition_control_keyword[right].keyword, condition_control_keyword[right].keyword_len) == 0) {
				condition += condition_control_keyword[right].keyword_len;
				break;
			}
		}
		if (!condition_control_keyword[right].keyword) {
			if (!parse_ulong(&right_min, &condition)) goto out;
			if (*condition == '-') {
				condition++;
				if (!parse_ulong(&right_max, &condition) || right_min > right_max) goto out;
			}
		}
	}
	return 1;
 out:
	printf("%u: ERROR: '%s' is a illegal condition.\n", line, start); errors++;
	return 0;
}

static void CheckPortPolicy(char *data) {
	unsigned int from, to;
	char *cp;
	if ((cp = FindConditionPart(data)) != NULL && !CheckCondition(cp)) return;
	if (strchr(data, ' ')) goto out;
	if (sscanf(data, "TCP/%u-%u", &from, &to) == 2) {
		if (from <= to && to < 65536) return;
	} else if (sscanf(data, "TCP/%u", &from) == 1) {
		if (from < 65536) return;
	} else if (sscanf(data, "UDP/%u-%u", &from, &to) == 2) {
		if (from <= to && to < 65536) return;
	} else if (sscanf(data, "UDP/%u", &from) == 1) {
		if (from < 65536) return;
	} else {
		printf("%u: ERROR: Too few parameters.\n", line); errors++;
		return;
	}
 out:
	printf("%u: ERROR: '%s' is a bad port number.\n", line, data); errors++;
}

static void CheckCapabilityPolicy(char *data) {
	static const char *capability_keywords[] = {
		"inet_tcp_create", "inet_tcp_listen", "inet_tcp_connect", "use_inet_udp", "use_inet_ip", "use_route", "use_packet",
		"SYS_MOUNT", "SYS_UMOUNT", "SYS_REBOOT", "SYS_CHROOT", "SYS_KILL", "SYS_VHANGUP", "SYS_TIME", "SYS_NICE", "SYS_SETHOSTNAME",
		"use_kernel_module", "create_fifo", "create_block_dev", "create_char_dev", "create_unix_socket",
		"SYS_LINK", "SYS_SYMLINK", "SYS_RENAME", "SYS_UNLINK", "SYS_CHMOD", "SYS_CHOWN", "SYS_IOCTL", "SYS_KEXEC_LOAD", NULL
	};
	int i;
	char *cp;
	if ((cp = FindConditionPart(data)) != NULL && !CheckCondition(cp)) return;
	for (i = 0; capability_keywords[i]; i++) {
		if (strcmp(data, capability_keywords[i]) == 0) return;
	}
	printf("%u: ERROR: '%s' is a bad capability name.\n", line, data); errors++;
}

static void CheckSignalPolicy(char *data) {
	int sig;
    char *cp;
	if ((cp = FindConditionPart(data)) != NULL && !CheckCondition(cp)) return;
	cp = strchr(data, ' ');
	if (!cp) {
		printf("%u: ERROR: Too few parameters.\n", line); errors++;
		return;
	}
	*cp++ = '\0';
	if (sscanf(data, "%d", &sig) != 1) {
		printf("%u: ERROR: '%s' is a bad signal number.\n", line, data); errors++;
	}
	if (!IsCorrectDomain(cp)) {
		printf("%u: ERROR: '%s' is a bad domainname.\n", line, cp); errors++;
	}
}

static void CheckArgv0Policy(char *data) {
	char *argv0 = strchr(data, ' ');
	char *cp;
	if (!argv0) {
		printf("%u: ERROR: Too few parameters.\n", line); errors++;
		return;
	}
	*argv0++ = '\0';
	if ((cp = FindConditionPart(argv0)) != NULL && !CheckCondition(cp)) return;
	if (!IsCorrectPath(data, 1, 0, -1)) {
		printf("%u: ERROR: '%s' is a bad pathname.\n", line, data); errors++;
	}
	if (!IsCorrectPath(argv0, -1, 0, -1) || strchr(argv0, '/')) {
		printf("%u: ERROR: '%s' is a bad argv[0] name.\n", line, data); errors++;
	}
}

static void CheckNetworkPolicy(char *data) {
	int sock_type, operation, is_ipv6;
	u16 min_address[8], max_address[8];
	unsigned int min_port, max_port;
	int count;
	char *cp1 = NULL, *cp2 = NULL;
	if ((cp1 = FindConditionPart(data)) != NULL && !CheckCondition(cp1)) return;
	if ((cp1 = strchr(data, ' ')) == NULL) goto out; cp1++;
	if (strncmp(data, "TCP ", 4) == 0) sock_type = SOCK_STREAM;
	else if (strncmp(data, "UDP ", 4) == 0) sock_type = SOCK_DGRAM;
	else if (strncmp(data, "RAW ", 4) == 0) sock_type = SOCK_RAW;
	else goto out;
	if ((cp2 = strchr(cp1, ' ')) == NULL) goto out; cp2++;
	if (strncmp(cp1, "bind ", 5) == 0) {
		operation = (sock_type == SOCK_STREAM) ? NETWORK_ACL_TCP_BIND : (sock_type == SOCK_DGRAM) ? NETWORK_ACL_UDP_BIND : NETWORK_ACL_RAW_BIND;
	} else if (strncmp(cp1, "connect ", 8) == 0) {
		operation = (sock_type == SOCK_STREAM) ? NETWORK_ACL_TCP_CONNECT : (sock_type == SOCK_DGRAM) ? NETWORK_ACL_UDP_CONNECT : NETWORK_ACL_RAW_CONNECT;
	} else if (sock_type == SOCK_STREAM && strncmp(cp1, "listen ", 7) == 0) {
		operation = NETWORK_ACL_TCP_LISTEN;
	} else if (sock_type == SOCK_STREAM && strncmp(cp1, "accept ", 7) == 0) {
		operation = NETWORK_ACL_TCP_ACCEPT;
	} else {
		goto out;
	}
	if ((cp1 = strchr(cp2, ' ')) == NULL) goto out; cp1++;
	if ((count = sscanf(cp2, "%hx:%hx:%hx:%hx:%hx:%hx:%hx:%hx-%hx:%hx:%hx:%hx:%hx:%hx:%hx:%hx",
						&min_address[0], &min_address[1], &min_address[2], &min_address[3],
						&min_address[4], &min_address[5], &min_address[6], &min_address[7],
						&max_address[0], &max_address[1], &max_address[2], &max_address[3],
						&max_address[4], &max_address[5], &max_address[6], &max_address[7])) == 8 || count == 16) {
		int i;
		for (i = 0; i < 8; i++) {
			min_address[i] = htons(min_address[i]);
			max_address[i] = htons(max_address[i]);
		}
		if (count == 8) memmove(max_address, min_address, sizeof(min_address));
		is_ipv6 = 1;
	} else if ((count = sscanf(cp2, "%hu.%hu.%hu.%hu-%hu.%hu.%hu.%hu",
							   &min_address[0], &min_address[1], &min_address[2], &min_address[3],
 							   &max_address[0], &max_address[1], &max_address[2], &max_address[3])) == 4 || count == 8) {
		u32 ip = htonl((((u8) min_address[0]) << 24) + (((u8) min_address[1]) << 16) + (((u8) min_address[2]) << 8) + (u8) min_address[3]);
		* (u32 *) min_address = ip;
		if (count == 8) ip = htonl((((u8) max_address[0]) << 24) + (((u8) max_address[1]) << 16) + (((u8) max_address[2]) << 8) + (u8) max_address[3]);
		* (u32 *) max_address = ip;
		is_ipv6 = 0;
	} else {
		goto out;
	}
	if (strchr(cp1, ' ')) goto out;
	if ((count = sscanf(cp1, "%u-%u", &min_port, &max_port)) == 1 || count == 2) {
		if (count == 1) max_port = min_port;
		if (min_port <= max_port && max_port < 65536) return;
	}
 out: ;
	printf("%u: ERROR: Bad network address.\n", line); errors++;
}

static void CheckFilePolicy(char *data) {
	static struct {
		const char *keyword;
		const int paths;
	} acl_type_array[] = {
		{ "create",   1 },
		{ "unlink",   1 },
		{ "mkdir",    1 },
		{ "rmdir",    1 },
		{ "mkfifo",   1 },
		{ "mksock",   1 },
		{ "mkblock",  1 },
		{ "mkchar",   1 },
		{ "truncate", 1 },
		{ "symlink",  1 },
		{ "link",     2 },
		{ "rename",   2 },
		{ "rewrite",  1 },
		{ NULL, 0 }
	};
	char *filename = strchr(data, ' ');
	char *cp;
	unsigned int perm;
	if (!filename) {
		printf("%u: ERROR: Unknown command '%s'\n", line, data); errors++;
		return;
	}
	*filename++ = '\0';
	if ((cp = FindConditionPart(filename)) != NULL && !CheckCondition(cp)) return;
	if (sscanf(data, "%u", &perm) == 1 && perm > 0 && perm <= 7) {
		if (strendswith(filename, "/")) {
			if ((perm & 2) == 0) {
				printf("%u: WARNING: Directory '%s' without write permission will be ignored.\n", line, filename); warnings++;
			}
		} else if ((perm & 1) == 1 && PathContainsPattern(filename)) {
			printf("%u: WARNING: Dropping execute permission for '%s'\n", line, filename); warnings++;
		}
		if (!IsCorrectPath(filename, 0, 0, 0)) goto out;
		return;
	}
	if (strncmp(data, "allow_", 6) == 0) {
		int type;
		for (type = 0; acl_type_array[type].keyword; type++) {
			if (strcmp(data + 6, acl_type_array[type].keyword)) continue;
			if (acl_type_array[type].paths == 2) {
				cp = strchr(filename, ' ');
				if (!cp || !IsCorrectPath(cp + 1, 0, 0, 0)) break;
				*cp = '\0';
			}
			if (!IsCorrectPath(filename, 0, 0, 0)) break;
			return;
		}
		if (!acl_type_array[type].keyword) goto out2;
	out:
		printf("%u: ERROR: '%s' is a bad pathname.\n", line, filename); errors++;
		return;
	}
 out2:
	printf("%u: ERROR: Invalid permission '%s %s'\n", line, data, filename); errors++;
}

static void CheckMountPolicy(char *data) {
	char *cp, *cp2;
	const char *fs, *dev, *dir;
	unsigned int enable = 0, disable = 0;
	cp2 = data; if ((cp = strchr(cp2, ' ')) == NULL) goto out; *cp = '\0'; dev = cp2;
	cp2 = cp + 1; if ((cp = strchr(cp2, ' ')) == NULL) goto out; *cp = '\0'; dir = cp2;
	cp2 = cp + 1;
	if ((cp = strchr(cp2, ' ')) != NULL) {
		char *sp = cp + 1;
		*cp = '\0';
		while ((cp = strsep(&sp, " ,")) != NULL) {
			if (strcmp(cp, "rw") == 0)          disable |= 1;
			else if (strcmp(cp, "ro") == 0)     enable  |= 1;
			else if (strcmp(cp, "suid") == 0)   disable |= 2;
			else if (strcmp(cp, "nosuid") == 0) enable  |= 2;
			else if (strcmp(cp, "dev") == 0)    disable |= 4;
			else if (strcmp(cp, "nodev") == 0)  enable  |= 4;
			else if (strcmp(cp, "exec") == 0)   disable |= 8;
			else if (strcmp(cp, "noexec") == 0) enable  |= 8;
			else if (strcmp(cp, "atime") == 0)      disable |= 1024;
			else if (strcmp(cp, "noatime") == 0)    enable  |= 1024;
			else if (strcmp(cp, "diratime") == 0)   disable |= 2048;
			else if (strcmp(cp, "nodiratime") == 0) enable  |= 2048;
			else if (strcmp(cp, "norecurse") == 0)  disable |= 16384;
			else if (strcmp(cp, "recurse") == 0)    enable  |= 16384;
		}
	}
	fs = cp2;
	if (enable & disable) {
		printf("%u: ERROR: Conflicting mount options.\n", line); errors++;
	}
	if (!IsCorrectPath(dev, 0, 0, 0)) {
		printf("%u: ERROR: '%s' is a bad device name.\n", line, dir); errors++;
	}
	if (!IsCorrectPath(dir, 1, 0, 1)) {
		printf("%u: ERROR: '%s' is a bad mount point.\n", line, dir); errors++;
	}
	return;
 out:
	printf("%u: ERROR: Too few parameters.\n", line); errors++;
}

static void CheckReservedPortPolicy(char *data) {
	unsigned int from, to;
	if (strchr(data, ' ')) goto out;
	if (sscanf(data, "%u-%u", &from, &to) == 2) {
		if (from <= to && to < 65536) return;
	} else if (sscanf(data, "%u", &from) == 1) {
		if (from < 65536) return;
	} else {
		printf("%u: ERROR: Too few parameters.\n", line); errors++;
		return;
	}
 out:
	printf("%u: ERROR: '%s' is a bad port number.\n", line, data); errors++;
}

static int checkpolicy_main(int argc, char *argv[]) {
	int policy_type = POLICY_TYPE_UNKNOWN;
	if (argc > 1) {
		switch (argv[1][0]) {
		case 's':
			policy_type = POLICY_TYPE_SYSTEM_POLICY;
			break;
		case 'e':
			policy_type = POLICY_TYPE_EXCEPTION_POLICY;
			break;
		case 'd':
			policy_type = POLICY_TYPE_DOMAIN_POLICY;
			break;
		}
	}
	if (policy_type == POLICY_TYPE_UNKNOWN) {
		fprintf(stderr, "%s s|e|d < policy_to_check\n", argv[0]);
		return 0;
	}
	get();
	while (memset(shared_buffer, 0, shared_buffer_len), fgets(shared_buffer, shared_buffer_len - 1, stdin)) {
		static int domain = EOF;
		int is_select = 0, is_delete = 0;
		char *cp = strchr(shared_buffer, '\n');
		line++;
		if (!cp) {
			printf("%u: ERROR: Line too long.\n", line); errors++;
			break;
		}
		*cp = '\0';
		{
			int c;
			for (c = 1; c < 256; c++) {
				if (c == '\t' || c == '\r' || (c >= ' ' && c < 127)) continue;
				if (strchr(shared_buffer, c)) {
					printf("%u: WARNING: Line contains illegal character (\\%03o).\n", line, c); warnings++;
					break;
				}
			}
		}
		NormalizeLine(shared_buffer);
		if (!shared_buffer[0]) continue;
		switch (policy_type) {
		case POLICY_TYPE_DOMAIN_POLICY:
			if (strncmp(shared_buffer, KEYWORD_DELETE, KEYWORD_DELETE_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_DELETE_LEN);
				is_delete = 1;
			} else if (strncmp(shared_buffer, KEYWORD_SELECT, KEYWORD_SELECT_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_SELECT_LEN);
				is_select = 1;
			}
			if (IsDomainDef(shared_buffer)) {
				if (!IsCorrectDomain(shared_buffer) || strlen(shared_buffer) >= PAGE_SIZE) {
					printf("%u: ERROR: '%s' is a bad domainname.\n", line, shared_buffer); errors++;
				} else {
					if (is_delete) domain = EOF;
					else domain = 0;
				}
			} else if (is_select) {
				printf("%u: ERROR: Command 'select' is valid for selecting domains only.\n", line); errors++;
			} else if (domain == EOF) {
				printf("%u: WARNING: '%s' is unprocessed because domain is not selected.\n", line, shared_buffer); warnings++;
			} else if (strncmp(shared_buffer, KEYWORD_USE_PROFILE, KEYWORD_USE_PROFILE_LEN) == 0) {
				unsigned int profile;
				RemoveHeader(shared_buffer, KEYWORD_USE_PROFILE_LEN);
				if (sscanf(shared_buffer, "%u", &profile) != 1 || profile >= 256) {
					printf("%u: ERROR: '%s' is a bad profile.\n", line, shared_buffer); errors++;
				}
			} else if (strncmp(shared_buffer, KEYWORD_ALLOW_CAPABILITY, KEYWORD_ALLOW_CAPABILITY_LEN) == 0) {
				CheckCapabilityPolicy(shared_buffer + KEYWORD_ALLOW_CAPABILITY_LEN);
			} else if (strncmp(shared_buffer, KEYWORD_ALLOW_BIND, KEYWORD_ALLOW_BIND_LEN) == 0) {
				CheckPortPolicy(shared_buffer + KEYWORD_ALLOW_BIND_LEN);
			} else if (strncmp(shared_buffer, KEYWORD_ALLOW_CONNECT, KEYWORD_ALLOW_CONNECT_LEN) == 0) {
				CheckPortPolicy(shared_buffer + KEYWORD_ALLOW_CONNECT_LEN);
			} else if (strncmp(shared_buffer, KEYWORD_ALLOW_NETWORK, KEYWORD_ALLOW_NETWORK_LEN) == 0) {
				CheckNetworkPolicy(shared_buffer + KEYWORD_ALLOW_NETWORK_LEN);
			} else if (strncmp(shared_buffer, KEYWORD_ALLOW_SIGNAL, KEYWORD_ALLOW_SIGNAL_LEN) == 0) {
				CheckSignalPolicy(shared_buffer + KEYWORD_ALLOW_SIGNAL_LEN);
			} else if (strncmp(shared_buffer, KEYWORD_ALLOW_ARGV0, KEYWORD_ALLOW_ARGV0_LEN) == 0) {
				CheckArgv0Policy(shared_buffer + KEYWORD_ALLOW_ARGV0_LEN);
			} else {
				CheckFilePolicy(shared_buffer);
			}
			break;
		case POLICY_TYPE_EXCEPTION_POLICY:
			if (strncmp(shared_buffer, KEYWORD_DELETE, KEYWORD_DELETE_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_DELETE_LEN);
			}
			if (strncmp(shared_buffer, KEYWORD_DOMAIN_KEEPER, KEYWORD_DOMAIN_KEEPER_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_DOMAIN_KEEPER_LEN);
				if (!IsCorrectDomain(shared_buffer)) {
					printf("%u: ERROR: '%s' is a bad domainname.\n", line, shared_buffer); errors++;
				}
			} else if (strncmp(shared_buffer, KEYWORD_ALLOW_READ, KEYWORD_ALLOW_READ_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_ALLOW_READ_LEN);
				if (!IsCorrectPath(shared_buffer, 1, -1, -1)) {
					printf("%u: ERROR: '%s' is a bad pathname.\n", line, shared_buffer); errors++;
				}
			} else if (strncmp(shared_buffer, KEYWORD_INITIALIZER, KEYWORD_INITIALIZER_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_INITIALIZER_LEN);
				if (!IsCorrectPath(shared_buffer, 1, -1, -1)) {
					printf("%u: ERROR: '%s' is a bad pathname.\n", line, shared_buffer); errors++;
				}
			} else if (strncmp(shared_buffer, KEYWORD_ALIAS, KEYWORD_ALIAS_LEN) == 0) {
				char *cp;
				RemoveHeader(shared_buffer, KEYWORD_ALIAS_LEN);
				if ((cp = strchr(shared_buffer, ' ')) == NULL) {
					printf("%u: ERROR: Too few parameters.\n", line); errors++;
				} else {
					*cp++ = '\0';
					if (!IsCorrectPath(shared_buffer, 1, -1, -1)) {
						printf("%u: ERROR: '%s' is a bad pathname.\n", line, shared_buffer); errors++;
					}
					if (!IsCorrectPath(cp, 1, -1, -1)) {
						printf("%u: ERROR: '%s' is a bad pathname.\n", line, cp); errors++;
					}
				}
			} else if (strncmp(shared_buffer, KEYWORD_AGGREGATOR, KEYWORD_AGGREGATOR_LEN) == 0) {
				char *cp;
				RemoveHeader(shared_buffer, KEYWORD_AGGREGATOR_LEN);
				if ((cp = strchr(shared_buffer, ' ')) == NULL) {
					printf("%u: ERROR: Too few parameters.\n", line); errors++;
				} else {
					*cp++ = '\0';
					if (!IsCorrectPath(shared_buffer, 1, 0, -1)) {
						printf("%u: ERROR: '%s' is a bad pattern.\n", line, shared_buffer); errors++;
					}
					if (!IsCorrectPath(cp, 1, -1, -1)) {
						printf("%u: ERROR: '%s' is a bad pathname.\n", line, cp); errors++;
					}
				}
			} else if (strncmp(shared_buffer, KEYWORD_FILE_PATTERN, KEYWORD_FILE_PATTERN_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_FILE_PATTERN_LEN);
				if (!IsCorrectPath(shared_buffer, 0, 1, 0)) {
					printf("%u: ERROR: '%s' is a bad pattern.\n", line, shared_buffer); errors++;
				}
			} else if (strncmp(shared_buffer, KEYWORD_DENY_REWRITE, KEYWORD_DENY_REWRITE_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_DENY_REWRITE_LEN);
				if (!IsCorrectPath(shared_buffer, 0, 0, 0)) {
					printf("%u: ERROR: '%s' is a bad pattern.\n", line, shared_buffer); errors++;
				}
			} else {
				printf("%u: ERROR: Unknown command '%s'.\n", line, shared_buffer); errors++;
			}
			break;
		case POLICY_TYPE_SYSTEM_POLICY:
			if (strncmp(shared_buffer, KEYWORD_DELETE, KEYWORD_DELETE_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_DELETE_LEN);
			}
			if (strncmp(shared_buffer, KEYWORD_ALLOW_MOUNT, KEYWORD_ALLOW_MOUNT_LEN) == 0) {
				CheckMountPolicy(shared_buffer + KEYWORD_ALLOW_MOUNT_LEN);
			} else if (strncmp(shared_buffer, KEYWORD_DENY_UNMOUNT, KEYWORD_DENY_UNMOUNT_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_DENY_UNMOUNT_LEN);
				if (!IsCorrectPath(shared_buffer, 1, 0, 1)) {
					printf("%u: ERROR: '%s' is a bad pattern.\n", line, shared_buffer); errors++;
				}
			} else if (strncmp(shared_buffer, KEYWORD_ALLOW_CHROOT, KEYWORD_ALLOW_CHROOT_LEN) == 0) {
				RemoveHeader(shared_buffer, KEYWORD_ALLOW_CHROOT_LEN);
				if (!IsCorrectPath(shared_buffer, 1, 0, 1)) {
					printf("%u: ERROR: '%s' is a bad pattern.\n", line, shared_buffer); errors++;
				}
			} else if (strncmp(shared_buffer, KEYWORD_DENY_AUTOBIND, KEYWORD_DENY_AUTOBIND_LEN) == 0) {
				CheckReservedPortPolicy(shared_buffer + KEYWORD_DENY_AUTOBIND_LEN);
			} else {
				printf("%u: ERROR: Unknown command '%s'.\n", line, shared_buffer); errors++;
			}
			break;
		}
	}
	put();
	printf("Total:   %u Line%s   %u Error%s   %u Warning%s\n", line, line > 1 ? "s" : "", errors, errors > 1 ? "s" : "", warnings, warnings > 1 ? "s" : "");
	return (errors ? 2 : (warnings ? 1 : 0));
}

#endif

#ifdef NEED_CCSTREE

static pid_t GetPPID(const pid_t pid) {
	char buffer[1024];
	FILE *fp;
	pid_t ppid = 1;
	memset(buffer, 0, sizeof(buffer));
	snprintf(buffer, sizeof(buffer) - 1, "/proc/%u/status", pid);
	if ((fp = fopen(buffer, "r")) != NULL) {
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp)) {
			if (sscanf(buffer, "PPid: %u", &ppid) == 1) break;
		}
		fclose(fp);
	}
	return ppid;
}

static char *GetName(const pid_t pid) {
	char buffer[1024];
	FILE *fp;
	memset(buffer, 0, sizeof(buffer));
	snprintf(buffer, sizeof(buffer) - 1, "/proc/%u/status", pid);
	if ((fp = fopen(buffer, "r")) != NULL) {
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp)) {
			if (strncmp(buffer, "Name:", 5) == 0) {
				char *cp = buffer + 5;
				while (*cp == ' ' || *cp == '\t') cp++;
				memmove(buffer, cp, strlen(cp) + 1);
				if ((cp = strchr(buffer, '\n')) != NULL) *cp = '\0';
				break;
			}
		}
		fclose(fp);
		if (buffer[0]) return strdup(buffer);
	}
	return NULL;
}

static int status_fd = EOF;

static const char *ReadInfo(const pid_t pid, int *profile) {
	char *cp; /* caller must use get()/put(). */
	memset(shared_buffer, 0, shared_buffer_len);
	snprintf(shared_buffer, shared_buffer_len - 1, "%d\n", pid);
	write(status_fd, shared_buffer, strlen(shared_buffer));
	memset(shared_buffer, 0, shared_buffer_len);
	read(status_fd, shared_buffer, shared_buffer_len - 1);
	if ((cp = strchr(shared_buffer, ' ')) != NULL) {
		*profile = atoi(cp + 1);
		if ((cp = strchr(cp + 1, ' ')) != NULL) {
			return cp + 1;
		}
	}
	*profile = -1;
	return "<UNKNOWN>";
}

static TASK_ENTRY *task_list = NULL;
static int task_list_len = 0;

static void Dump(const pid_t pid, const int depth) {
	int i;
	for (i = 0; i < task_list_len; i++) {
		const char *info;
		char *name;
		int j, profile;
		if (pid != task_list[i].pid) continue;
		name = GetName(pid);
		get();
		info = ReadInfo(pid, &profile);
		printf("%3d", profile);
		for (j = 0; j < depth - 1; j++) printf("    ");
		for (; j < depth; j++) printf("  +-");
		printf(" %s (%u) %s\n", name, pid, info);
		put();
		free(name);
		task_list[i].done = 1;
	}
	for (i = 0; i < task_list_len; i++) {
		if (pid != task_list[i].ppid) continue;
		Dump(task_list[i].pid, depth + 1);
	}
}

static void DumpUnprocessed(void) {
	int i;
	for (i = 0; i < task_list_len; i++) {
		const char *info;
		char *name;
		int profile;
		const pid_t pid = task_list[i].pid;
		if (task_list[i].done) continue;
		name = GetName(task_list[i].pid);
		get();
		info = ReadInfo(pid, &profile);
		printf("%3d %s (%u) %s\n", profile, name, pid, info);
		put();
		free(name);
		task_list[i].done = 1;
	}
}
		
static int ccstree_main(int argc, char *argv[]) {
	static const char *policy_file = "/proc/ccs/info/.process_status";
	static int show_all = 0;
	if (access("/proc/ccs/", F_OK)) {
		fprintf(stderr, "You can't use this command for this kernel.\n");
		return 1;
	}
	if (argc > 1) {
		if (strcmp(argv[1], "-a") == 0) {
			show_all = 1;
		} else {
			fprintf(stderr, "Usage; %s [-a]\n", argv[0]);
			return 0;
		}
	}
	if ((status_fd = open(policy_file, O_RDWR)) == EOF) {
		fprintf(stderr, "Can't open %s\n", policy_file);
		return 1;
	}
	{
		struct dirent **namelist;
		int i, n = scandir("/proc/", &namelist, 0, 0);
		for (i = 0; i < n; i++) {
			pid_t pid;
			if (sscanf(namelist[i]->d_name, "%u", &pid) == 1) {
				char buffer[128], test[16];
				memset(buffer, 0, sizeof(buffer));
				snprintf(buffer, sizeof(buffer) - 1, "/proc/%u/exe", pid);
				if (show_all || readlink(buffer, test, sizeof(test)) > 0) {
					task_list = (TASK_ENTRY *) realloc(task_list, (task_list_len + 1) * sizeof(TASK_ENTRY));
					task_list[task_list_len].pid = pid;
					task_list[task_list_len].ppid = GetPPID(pid);
					task_list[task_list_len].done = 0;
					task_list_len++;
				}
			}
			free((void *) namelist[i]);
		}
		if (n >= 0) free((void *) namelist);
	}
	Dump(1, 0);
	DumpUnprocessed();
	close(status_fd);
	return 0;
}

#endif

#ifdef NEED_CCSQUERYD

static int ccsqueryd_main(int argc, char *argv[]) {
	const int query_fd = open("/proc/ccs/policy/query", O_RDWR);
	const int domain_policy_fd = open("/proc/ccs/policy/domain_policy", O_WRONLY);
	struct termios tp, tp0;
	if (argc > 1) {
		printf("Usage: %s\n\n", argv[0]);
		printf("This program is used for granting access requests manually.\n");
		printf("This program shows access requests that are about to rejected by the kernel's decision.\n");
		printf("If you answer before the kernel's decision taken effect, your decision will take effect.\n");
		printf("You can use this program to respond to accidental access requests triggered by non-routine tasks (such as restarting daemons after updating).\n");
		printf("To terminate this program, use 'Ctrl-C'.\n");
		return 0;
	}
	if (query_fd == EOF) {
		fprintf(stderr, "You can't run this utility for this kernel.\n");
		return 1;
	} else if (write(query_fd, "", 0) != 0) {
		fprintf(stderr, "You need to register this program to /proc/ccs/policy/manager to run this program.\n");
		return 1;
	}
	write(query_fd, "\n", 1);
	tcgetattr(0, &tp0); tp = tp0;
	tp.c_lflag &= ~(ICANON | ECHO);
	tp.c_cc[VTIME] = 0;
	tp.c_cc[VMIN] = 1;
	while (1) {
		static int first = 1;
		static unsigned int prev_serial = 0;
		static const int buffer_len = 16384;
		static char *buffer = NULL, *prev_buffer = NULL;
		fd_set rfds;
		unsigned int serial;
		char *cp;
		if (!buffer && (buffer = malloc(buffer_len)) == NULL) break;
		if (!prev_buffer) {
			if ((prev_buffer = malloc(buffer_len)) == NULL) break;
			memset(prev_buffer, 0, buffer_len);
		}
		// Wait for query.
		FD_ZERO(&rfds);
		FD_SET(query_fd, &rfds);
		select(query_fd + 1, &rfds, NULL, NULL, NULL);
		if (!FD_ISSET(query_fd, &rfds)) continue;
		
		// Read query.
		memset(buffer, 0, buffer_len);
		if (read(query_fd, buffer, buffer_len - 1) <= 0) continue;
		if ((cp = strchr(buffer, '\n')) == NULL) continue;
		*cp = '\0';
		
		// Get query number.
		if (sscanf(buffer, "Q%u", &serial) != 1) continue;
		memmove(buffer, cp + 1, strlen(cp + 1) + 1);
		
		if (!first && prev_serial == serial) {
			sleep(1);
			write(query_fd, "\n", 1);
			continue;
		}
		first = 0;
		prev_serial = serial;

		// Is this domain query?
		if (strncmp(buffer, "<kernel>", 8) == 0 && (buffer[8] == '\0' || buffer[8] == ' ') && (cp = strchr(buffer, '\n')) != NULL) {
			int c = 0;
			// Check for same domain.
			*cp = '\0';
			if (strcmp(buffer, prev_buffer)) {
				printf("----------------------------------------\n");
				printf("%s\n", buffer);
				memmove(prev_buffer, buffer, strlen(buffer) + 1);
			}
			printf("%s", cp + 1);
			printf("Accept? ('Y'es/Yes and 'A'ppend to policy/'N'o):"); fflush(stdout);
			tcsetattr(0, TCSANOW, &tp);
			while (1) {
				struct timeval tv;
				tv.tv_sec = 1; tv.tv_usec = 0;
				FD_ZERO(&rfds);
				FD_SET(0, &rfds);
				select(1, &rfds, NULL, NULL, &tv);
				if (FD_ISSET(0, &rfds)) {
					c = getc(stdin);
					if (c == EOF || c == 'Y' || c == 'y' || c == 'N' || c == 'n' || c == 'A' || c == 'a') break;
				}
				write(query_fd, "\n", 1);
			}
			tcsetattr(0, TCSANOW, &tp0);
			if (c == EOF) break;
			printf("%c\n", c);
			
			// Append to domain policy.
			if (c == 'A' || c == 'a') {
				*cp = '\n';
				write(domain_policy_fd, buffer, strlen(buffer) + 1);
				write(domain_policy_fd, "\n", 1);
			}
			
			// Write answer.
			snprintf(buffer, buffer_len - 1, "A%u=%u\n", serial, c == 'Y' || c == 'y' || c == 'A' || c == 'a' ? 1 : 2);
			write(query_fd, buffer, strlen(buffer));
		} else {
			int c;
			printf("----------------------------------------\n");
			prev_buffer[0] = '\0';
			printf("%s", buffer);
			printf("Accept? ('Y'es/'N'o):"); fflush(stdout);
			tcsetattr(0, TCSANOW, &tp);
			while (1) {
				struct timeval tv;
				tv.tv_sec = 1; tv.tv_usec = 0;
				FD_ZERO(&rfds);
				FD_SET(0, &rfds);
				select(1, &rfds, NULL, NULL, &tv);
				if (FD_ISSET(0, &rfds)) {
					c = getc(stdin);
					if (c == EOF || c == 'Y' || c == 'y' || c == 'N' || c == 'n') break;
				}
				write(query_fd, "\n", 1);
			}
			tcsetattr(0, TCSANOW, &tp0);
			if (c == EOF) break;
			printf("%c\n", c);
			
			// Write answer.
			snprintf(buffer, buffer_len - 1, "A%u=%u\n", serial, c == 'Y' || c == 'y' ? 1 : 2);
			write(query_fd, buffer, strlen(buffer));
		}
	}
	return 0;
}

#endif

#ifdef NEED_CCSAUDITD

static const char *procfile_path[MAX_FILES] = {
	"/proc/ccs/info/grant_log",
	"/proc/ccs/info/reject_log"
};

static int ccsauditd_main(int argc, char *argv[]) {
	int i, fd_in[MAX_FILES], fd_out[MAX_FILES];
	const char *logfile_path[2] = { "/dev/null", "/dev/null" };
	if (access("/proc/ccs/policy/", F_OK)) {
		fprintf(stderr, "You can't run this daemon for this kernel.\n");
		return 0;
	}
	if (argc < 3) {
		fprintf(stderr, "%s grant_log_file reject_log_file\n" "  These files may /dev/null, if needn't to be saved.\n", argv[0]);
		return 0;
	}
	logfile_path[0] = argv[1]; logfile_path[1] = argv[2];
	{ // Get exclusive lock.
		int fd = open(argv[0], O_RDONLY); if (flock(fd, LOCK_EX | LOCK_NB) == EOF) return 0;
	}
	umask(0);
	for (i = 0; i < MAX_FILES; i++) {
		if ((fd_out[i] = open(logfile_path[i], O_WRONLY | O_CREAT | O_APPEND, 0600)) == EOF) {
			fprintf(stderr, "Can't open %s for writing.\n", logfile_path[i]);
			return 1;
		}
	}
	switch(fork()) {
	case 0:
		break;
	case -1:
		fprintf(stderr, "Can't fork()\n");
		return 1;
	default:
		return 0;
	}
	if (setsid() == EOF) {
		fprintf(stderr, "Can't setsid()\n");
		return 1;
	}
	switch(fork()) {
	case 0:
		break;
	case -1:
		fprintf(stderr, "Can't fork()\n");
		return 1;
	default:
		return 0;
	}
	if (chdir("/")) {
		fprintf(stderr, "Can't chdir()\n");
		return 1;
	}
	close(0); close(1); close(2);
	openlog("ccs-auditd", 0,  LOG_USER);
	for (i = 0; i < MAX_FILES; i++) {
		if ((fd_in[i] = open(procfile_path[i], O_RDONLY)) == EOF) {
			syslog(LOG_WARNING, "Can't open %s for reading.\n", procfile_path[i]);
			return 1;
		}
	}
	syslog(LOG_WARNING, "Started.\n");
	while (1) {
		static const int buffer_len = 16384;
		static char *buffer = NULL;
		char timestamp[128];
		fd_set rfds;
		if (!buffer && (buffer = malloc(buffer_len)) == NULL) break;
		FD_ZERO(&rfds);
		for (i = 0; i < MAX_FILES; i++) FD_SET(fd_in[i], &rfds);
		// Wait for data.
		if (select(FD_SETSIZE, &rfds, NULL, NULL, NULL) == EOF) break;
		for (i = 0; i < MAX_FILES; i++) {
			time_t stamp;
			char *cp;
			int len;
			if (!FD_ISSET(fd_in[i], &rfds)) continue;
			memset(buffer, 0, buffer_len);
			if (read(fd_in[i], buffer, buffer_len - 1) < 0) continue;
			memset(timestamp, 0, sizeof(timestamp));
			if (sscanf(buffer, "#timestamp=%lu", &stamp) == 1 && (cp = strchr(buffer, ' ')) != NULL) {
				struct tm *tm = localtime(&stamp);
				snprintf(timestamp, sizeof(timestamp) - 1, "#%04d-%02d-%02d %02d:%02d:%02d#", tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
				memmove(buffer, cp, strlen(cp) + 1);
			}
			// Open destination file.
			if (access(logfile_path[i], F_OK)) {
				close(fd_out[i]);
				if ((fd_out[i] = open(logfile_path[i], O_WRONLY | O_CREAT | O_APPEND, 0600)) == EOF) {
					syslog(LOG_WARNING, "Can't open %s for writing.\n", logfile_path[i]);
					goto out;
				}
			}
			len = strlen(timestamp);
			write(fd_out[i], timestamp, len);
			len = strlen(buffer);
			write(fd_out[i], buffer, len);
			write(fd_out[i], "\n", 1);
			fsync(fd_out[i]);
		}
	}
 out: ;
	syslog(LOG_WARNING, "Terminated.\n");
	closelog();
	return 0;
}

#endif

#ifdef NEED_PATTERNIZE

static int patternize_main(int argc, char *argv[]) {
	get();
	while (freadline(stdin)) {
		char *sp = shared_buffer, *cp;
		int first = 1;
		int check_second = 0;
		int disabled = 0;
		while ((cp = strsep(&sp, " ")) != NULL) {
		check:
			if (first) {
				unsigned int perm;
				if (sscanf(cp, "%u", &perm) == 1 && (perm & 1) == 1) {
					/* Is this entry for a non-directory? */
					check_second = 1;
				} else if (strcmp(cp, "<kernel>") == 0 || strcmp(cp, "use_profile") == 0
						   || strcmp(cp, "allow_capability") == 0 || strcmp(cp, "allow_signal") == 0 ||
						   strcmp(cp, "allow_bind") == 0 || strcmp(cp, "allow_connect") == 0 || strcmp(cp, "allow_network") == 0) {
					/* This entry is not pathname related permission. I don't convert. */
					disabled = 1;
				}
			} else if (disabled) {
				// Nothing to do.
			} else if (check_second) {
				check_second = 0;
				if (*cp && * (strchr(cp, '\0') - 1) != '/') {
					/* This entry is for a non-directory. I don't convert. */
					disabled = 1;
				}
				goto check;
			} else if (strcmp(cp, "if") == 0) {
				/* Don't convert after condition part. */
				disabled = 1;
			} else if (!PathContainsPattern(cp)) {
				int i;
				for (i = 1; i < argc; i++) {
					if (PathMatchesToPattern(cp, argv[i])) {
						cp = argv[i]; break;
					}
				}
			}
			if (!first) putchar(' ');
			first = 0;
			printf("%s", cp);
		}
		putchar('\n');
	}
	put();
	return 0;
}

#endif

int main(int argc, char *argv[]) {
	const char *argv0 = argv[0];
	if (!argv0) {
		fprintf(stderr, "Function not specified.\n");
		return 1;
	}
	if (strrchr(argv0, '/')) argv0 = strrchr(argv0, '/') + 1;
#ifdef NEED_SORTPOLICY
	if (strcmp(argv0, "sortpolicy") == 0) return sortpolicy_main(argc, argv);
#endif
#ifdef NEED_SETPROFILE
	if (strcmp(argv0, "setprofile") == 0) return setprofile_main(argc, argv);
#endif
#ifdef NEED_SETLEVEL
	if (strcmp(argv0, "setlevel") == 0) return setlevel_main(argc, argv);
#endif
#ifdef NEED_SAVEPOLICY
	if (strcmp(argv0, "savepolicy") == 0) return savepolicy_main(argc, argv);
#endif
#ifdef NEED_PATHMATCH
	if (strcmp(argv0, "pathmatch") == 0) return pathmatch_main(argc, argv);
#endif
#ifdef NEED_LOADPOLICY
	if (strcmp(argv0, "loadpolicy") == 0) return loadpolicy_main(argc, argv);
#endif
#ifdef NEED_LDWATCH
	if (strcmp(argv0, "ld-watch") == 0) return ldwatch_main(argc, argv);
#endif
#ifdef NEED_FINDTEMP
	if (strcmp(argv0, "findtemp") == 0) return findtemp_main(argc, argv);
#endif
#ifdef NEED_EDITPOLICY
	if (strcmp(argv0, "editpolicy") == 0 || strcmp(argv0, "editpolicy_offline") == 0) return editpolicy_main(argc, argv);
#endif
#ifdef NEED_CHECKPOLICY
	if (strcmp(argv0, "checkpolicy") == 0) return checkpolicy_main(argc, argv);
#endif
#ifdef NEED_CCSTREE
	if (strcmp(argv0, "ccstree") == 0) return ccstree_main(argc, argv);
#endif
#ifdef NEED_CCSQUERYD
	if (strcmp(argv0, "ccs-queryd") == 0) return ccsqueryd_main(argc, argv);
#endif
#ifdef NEED_CCSAUDITD
	if (strcmp(argv0, "ccs-auditd") == 0) return ccsauditd_main(argc, argv);
#endif
#ifdef NEED_PATTERNIZE
	if (strcmp(argv0, "patternize") == 0) return patternize_main(argc, argv);
#endif
	/*
	 * Unlike busybox, I don't use argv[1] if argv[0] is the name of this program
	 * because it is dangerous to allow updating policies via unchecked argv[1].
	 * You should use either "symbolic links with 'alias' directive" or "hard links".
	 */
	fprintf(stderr, "Function %s not implemented.\n", argv0);
	return 1;
}
