/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"
#include <tchar.h>
#include <ime.h>


/******************************************************************************
*                                                                             *
* ja:IME制御関数                                                              *
*                                                                             *
******************************************************************************/
typedef LRESULT (WINAPI *SendIMEMessageEx_t) (HWND, LPARAM);


static HINSTANCE hLib = NULL;
static SendIMEMessageEx_t lpSendIMEMessageExW = NULL;
static SendIMEMessageEx_t lpSendIMEMessageExA = NULL;


#ifdef UNICODE
# define lpSendIMEMessageEx lpSendIMEMessageExW
#else /* not UNICODE */
# define lpSendIMEMessageEx lpSendIMEMessageExA
#endif /* not UNICODE */


/*  ja:IMEを初期化する                                                      */
VOID
InitIme (VOID)
{
  if (!hLib)
    hLib = LoadLibrary (_T("USER32.DLL"));
  if (hLib)
    {
      lpSendIMEMessageExW = (SendIMEMessageEx_t)
                                    GetProcAddress (hLib, "SendIMEMessageExW");
      lpSendIMEMessageExA = (SendIMEMessageEx_t)
                                    GetProcAddress (hLib, "SendIMEMessageExA");
    }
}


/*  ja:IMEを終了する                                                        */
VOID
ExitIme (VOID)
{
  if (hLib)
    {
      FreeLibrary (hLib);
      hLib = NULL;
    }
  lpSendIMEMessageExW = lpSendIMEMessageExA = NULL;
}


/*  ja:IMEの位置を設定する
    hWnd,ウインドウ
       x,x座標
       y,y座標
     RET,TRUE:正常終了,FALSE:エラー                                         */
BOOL
SetImePos (HWND      hWnd,
           const int x,
           const int y)
{
  BOOL fResult;

  fResult = !lpSendIMEMessageEx;
  if (!fResult)
    {
      HGLOBAL hGlobal;

      hGlobal = GlobalAlloc (GMEM_MOVEABLE | GMEM_SHARE, sizeof (IMESTRUCT));
      if (hGlobal)
        {
          LPIMESTRUCT lpIme;

          lpIme = GlobalLock (hGlobal);
          if (lpIme)
            {
              lpIme->fnc = IME_SETCONVERSIONWINDOW;
              lpIme->wParam = y < 0 ? MCW_DEFAULT : MCW_WINDOW;
              lpIme->lParam1 = MAKELPARAM (x, y);
              GlobalUnlock (hGlobal);
              fResult = lpSendIMEMessageEx (hWnd, (LPARAM)hGlobal);
            }
          GlobalFree (hGlobal);
        }
    }
  return fResult;
}


/*  ja:IMEのフォントを設定する
         hWnd,ウインドウ
    lpLogFont,フォント情報
          RET,TRUE:正常終了,FALSE:エラー                                    */
BOOL
SetImeFontW (HWND       hWnd,
             LPLOGFONTW lpLogFont)
{
  BOOL fResult;

  fResult = !lpSendIMEMessageExW;
  if (!fResult)
    {
      HGLOBAL hGlobalFont;

      hGlobalFont = GlobalAlloc
                            (GMEM_MOVEABLE | GMEM_SHARE, sizeof (LOGFONTW));
      if (hGlobalFont)
        {
          LPLOGFONTW lpFont;

          lpFont = GlobalLock (hGlobalFont);
          if (lpFont)
            {
              HGLOBAL hGlobalIme;

              *lpFont = *lpLogFont;
              GlobalUnlock (hGlobalFont);
              hGlobalIme = GlobalAlloc
                            (GMEM_MOVEABLE | GMEM_SHARE, sizeof (IMESTRUCT));
              if (hGlobalIme)
                {
                  LPIMESTRUCT lpIme;

                  lpIme = GlobalLock (hGlobalIme);
                  if (lpIme)
                    {
                      lpIme->fnc = IME_SETCONVERSIONFONTEX;
                      lpIme->lParam1 = (LPARAM)hGlobalFont;
                      GlobalUnlock (hGlobalIme);
                      fResult = lpSendIMEMessageExW (hWnd, (LPARAM)hGlobalIme);
                    }
                  GlobalFree (hGlobalIme);
                }
            }
          GlobalFree (hGlobalFont);
        }
    }
  return fResult;
}


/*  ja:IMEのフォントを設定する
         hWnd,ウインドウ
    lpLogFont,フォント情報
          RET,TRUE:正常終了,FALSE:エラー                                    */
BOOL
SetImeFontA (HWND       hWnd,
             LPLOGFONTA lpLogFont)
{
  BOOL fResult;

  fResult = !lpSendIMEMessageExW;
  if (!fResult)
    {
      HGLOBAL hGlobalFont;

      hGlobalFont = GlobalAlloc
                            (GMEM_MOVEABLE | GMEM_SHARE, sizeof (LOGFONTA));
      if (hGlobalFont)
        {
          LPLOGFONTA lpFont;

          lpFont = GlobalLock (hGlobalFont);
          if (lpFont)
            {
              HGLOBAL hGlobalIme;

              *lpFont = *lpLogFont;
              GlobalUnlock (hGlobalFont);
              hGlobalIme = GlobalAlloc
                            (GMEM_MOVEABLE | GMEM_SHARE, sizeof (IMESTRUCT));
              if (hGlobalIme)
                {
                  LPIMESTRUCT lpIme;

                  lpIme = GlobalLock (hGlobalIme);
                  if (lpIme)
                    {
                      lpIme->fnc = IME_SETCONVERSIONFONTEX;
                      lpIme->lParam1 = (LPARAM)hGlobalFont;
                      GlobalUnlock (hGlobalIme);
                      fResult = lpSendIMEMessageExA (hWnd, (LPARAM)hGlobalIme);
                    }
                  GlobalFree (hGlobalIme);
                }
            }
          GlobalFree (hGlobalFont);
        }
    }
  return fResult;
}
