/*
    Text maid
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "charset.h"
#include "command.h"
#include "edit.h"
#include "find.h"
#include "general.h"
#include "menu.h"
#include "sigfile.h"
#include "misc/charuty.h"
#include "misc/misc.h"
#include "orz/orzmdi.h"


/******************************************************************************
*                                                                             *
* ja:一般関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:キャレットを消す
    tmaid,ウインドウ情報                                                    */
static void
hide_caret (TmaidWindow *tmaid)
{
  g_source_remove (timer_id);
  timer_id = 0;
  if (caret)
    {
      caret = FALSE;
      gtk_widget_queue_draw_area (tmaid->drawing,
        (edit_get_align_pos (tmaid, tmaid->cursor.x, tmaid->cursor.y, FALSE)
                                        - tmaid->top.x) * tmaid->font_width,
        (tmaid->cursor.y - tmaid->top.y) * tmaid->font_height,
                                        tmaid->font_width, tmaid->font_height);
      gdk_window_process_updates (gtk_widget_get_window (tmaid->drawing),
                                                                        TRUE);
    }
}


/*  ja:キャレットの描画
     tmaid,ウインドウ情報
    cursor,古いキャレット位置                                               */
void
draw_caret (TmaidWindow *tmaid,
            GdkPoint    *cursor)
{
  gboolean visible;
  gchar *text;
  gint x, y;
  guint context_id;
  GdkRectangle rc;
#ifndef G_OS_WIN32
  GtkWidget *widget;
#endif /* not G_OS_WIN32 */

  if (cursor && timer_id != 0
            && (tmaid->cursor.x != cursor->x || tmaid->cursor.y != cursor->y))
    {
      /* ja:タイマ再設定 */
      g_source_remove (timer_id);
      timer_id = 0;
      /* ja:古いキャレットを消去 */
      gtk_widget_queue_draw_area (tmaid->drawing,
                    (edit_get_align_pos (tmaid, cursor->x, cursor->y, FALSE)
                                        - tmaid->top.x) * tmaid->font_width,
                    (cursor->y - tmaid->top.y) * tmaid->font_height,
                                        tmaid->font_width, tmaid->font_height);
    }
  /* ja:新しいキャレットを描画 */
  if (timer_id == 0)
    caret = TRUE;
  x = edit_get_align_pos (tmaid, tmaid->cursor.x, tmaid->cursor.y, FALSE);
  y = tmaid->cursor.y;
  rc.x = (x - tmaid->top.x) * tmaid->font_width;
  rc.y = (y - tmaid->top.y) * tmaid->font_height;
  rc.width = tmaid->font_width;
  rc.height = tmaid->font_height;
  gtk_widget_queue_draw_area (tmaid->drawing, rc.x, rc.y, rc.width, rc.height);
  if (timer_id == 0)
    timer_id = g_timeout_add (500, signal_timeout, NULL);
  text = g_strdup_printf (_("L%d,C%d"), y + 1, x + 1);
  gtk_label_set_text (GTK_LABEL (label_cursor), text);
  g_free (text);
  text = tmaid->select.x >= 0
                    ? g_strdup_printf (_("Selected(L%d,C%d)-(L%d,C%d)"),
                        tmaid->select.y + 1, tmaid->select.x + 1, y + 1, x + 1)
                    : g_strdup ("");
  context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (status), "Status");
  gtk_statusbar_pop (GTK_STATUSBAR (status), context_id);
  gtk_statusbar_push (GTK_STATUSBAR (status), context_id, text);
  g_free (text);
#ifndef G_OS_WIN32
  gdk_window_get_origin (gtk_widget_get_window (tmaid->drawing), &x, &y);
  widget = gtk_widget_get_parent (tmaid->preedit);
  visible = GTK_WIDGET_VISIBLE (widget);
  if (!visible)
    gtk_window_resize (GTK_WINDOW (widget), 1, 1);
  gtk_widget_hide (widget);
  gtk_window_move (GTK_WINDOW (widget), rc.x + x, rc.y + y);
  if (visible)
    gtk_widget_show (widget);
#endif /* not G_OS_WIN32 */
  rc.width = 0;
  gtk_im_context_set_cursor_location (tmaid->im_context, &rc);
  gdk_window_process_updates (gtk_widget_get_window (tmaid->drawing), TRUE);
}


/*  ja:メニューを設定する
    tmaid,ウインドウ情報                                                    */
void
set_menu_bar (TmaidWindow *tmaid)
{
  if (tmaid)
    {
      gboolean created;

      created = orz_mdi_get_created (ORZ_MDI (mdi),
                            orz_mdi_get_page_from_data (ORZ_MDI (mdi), tmaid));
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/file/close"), TRUE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/file/save"), TRUE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/file/saveas"), TRUE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/file/reload"), !created);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/file/print"), TRUE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/file/properties"), TRUE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/edit"), TRUE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/edit/undo"), tmaid->undo != NULL);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/edit/redo"), tmaid->redo != NULL);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/edit/cut"), tmaid->select.x >= 0);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/edit/copy"), tmaid->select.x >= 0);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/edit/delete"), tmaid->select.x >= 0);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/search"), TRUE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/search/next"), find_text[0] != NULL);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                    "/search/previous"), find_text[0] != NULL);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "save")), TRUE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "properties")), TRUE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "print")), TRUE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "cut")), tmaid->select.x >= 0);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "copy")), tmaid->select.x >= 0);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "paste")), TRUE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "delete")), tmaid->select.x >= 0);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "undo")), tmaid->undo != NULL);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "redo")), tmaid->redo != NULL);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "find")), TRUE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                            "replace")), TRUE);
    }
  else
    {
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/file/close"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/file/save"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/file/saveas"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/file/reload"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/file/print"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/file/properties"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/edit"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/search"), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "save")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "properties")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "print")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "cut")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "copy")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "paste")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "delete")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "undo")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "redo")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "find")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "replace")), FALSE);
    }
}


/*  ja:指定範囲を再描画する
    tmaid,ウインドウ情報
    start,開始
      end,終了                                                              */
void
clear_sel (TmaidWindow *tmaid,
           GdkPoint    *start,
           GdkPoint    *end)
{
  gint st, ed, x, y, width, height;

  if (start->y == end->y)
    {
      /* ja:同じ行 */
      st = edit_get_align_pos (tmaid, MIN (start->x, end->x), start->y, FALSE);
      ed = edit_get_align_pos (tmaid, MAX (start->x, end->x), start->y, TRUE);
      x = (st - tmaid->top.x) * tmaid->font_width;
      y = (start->y - tmaid->top.y) * tmaid->font_height;
      width = (ed - st + 1) * tmaid->font_width;
      height = tmaid->font_height;
    }
  else
    {
      /* ja:違う行 */
      GtkAllocation allocation;

      gtk_widget_get_allocation (tmaid->drawing, &allocation);
      x = 0;
      y = (MIN (start->y, end->y) - tmaid->top.y) * tmaid->font_height;
      width = allocation.width;
      height = (ABS (start->y - end->y) + 1) * tmaid->font_height;
    }
  gtk_widget_queue_draw_area (tmaid->drawing, x, y, width, height);
  gdk_window_process_updates (gtk_widget_get_window (tmaid->drawing), TRUE);
}


/*  ja:移動する
    tmaid,ウインドウ情報
    top,古い左上の座標                                                      */
void
move_edit_window (TmaidWindow *tmaid,
                  GdkPoint    *top)
{
  if (tmaid->top.x != top->x || tmaid->top.y != top->y)
    {
      GtkAllocation allocation;

      gtk_widget_get_allocation (tmaid->drawing, &allocation);
      misc_scroll_window (tmaid->drawing,
                                (top->x - tmaid->top.x) * tmaid->font_width,
                                (top->y - tmaid->top.y) * tmaid->font_height,
                                0, 0,
                                allocation.width,
                                allocation.height);
    }
}


/*  ja:キャレット位置にデータを挿入/上書きする
     tmaid,ウインドウ情報
      text,データを格納するポインタ
    length,データのバイト数
     caret,TRUE:移動する,FALSE:移動しない
    select,TRUE:選択する,FALSE:選択しない
       RET,逆の操作を行うDOING構造体,NULL:エラー                            */
TmaidHistory *
edit_operation (TmaidWindow    *tmaid,
                const gchar    *text,
                const gint      length,
                const gboolean  caret,
                const gboolean  select)
{
  gint sx, sy, max;
  GdkPoint top;
  GtkAllocation allocation;
  TmaidHistory *d;

  hide_caret (tmaid);
  tmaid->cursor.x = edit_get_align_pos (tmaid,
                                    tmaid->cursor.x, tmaid->cursor.y, FALSE);
  top = tmaid->top;
  gtk_widget_get_allocation (tmaid->drawing, &allocation);
  sx = MAX (allocation.width / tmaid->font_width, 1);
  sy = MAX (allocation.height / tmaid->font_height, 1);
  d = g_malloc (sizeof (TmaidHistory));
  if (tmaid->select.x < 0)
    {
      d->text = NULL;
      d->length = 0;
      d->caret = FALSE;
    }
  else
    {
      /* ja:削除 */
      gint i, delete;
      LineBuffer *p;

      d->length = edit_get_sel_bytes (tmaid, &tmaid->cursor, &tmaid->select);
      d->text = g_malloc (d->length * sizeof (gchar));
      edit_cpy_sel_mem (tmaid, &tmaid->cursor, &tmaid->select, d->text);
      delete = edit_del_sel_mem (tmaid, &tmaid->cursor, &tmaid->select);
      tmaid->max -= delete;/* ja:行数減少 */
      /* ja:選択範囲解除とキャレット移動 */
      if (tmaid->select.y < tmaid->cursor.y
                                        || (tmaid->select.y == tmaid->cursor.y
                                        && tmaid->select.x < tmaid->cursor.x))
        {
          tmaid->cursor = tmaid->select;
          d->caret = TRUE;
        }
      else
        {
          tmaid->cursor = tmaid->cursor;
          d->caret = FALSE;
        }
      tmaid->select.x = -1;
      /* ja:削除した行の削除した桁より右を初期化(右マージンのために1つ多い) */
      i = (tmaid->cursor.x - tmaid->top.x - 1) * tmaid->font_width;
      gtk_widget_queue_draw_area (tmaid->drawing,
                    i, (tmaid->cursor.y - tmaid->top.y) * tmaid->font_height,
                    allocation.width - i, tmaid->font_height);
      /* ja:減った行数分だけスクロール */
      misc_scroll_window (tmaid->drawing, 0, -delete * tmaid->font_height,
        0,
        (tmaid->cursor.y - tmaid->top.y + 1) * tmaid->font_height,
        allocation.width,
        MAX (allocation.height
            - (tmaid->cursor.y - tmaid->top.y + 1) * tmaid->font_height, 0));
        /* ja:削除を終えた位置から右マージンの影響のある行数だけ初期化 */
      for (i = 0,
        p = edit_get_line_buf (&tmaid->start, &tmaid->off, tmaid->cursor.y);
                                                        p; i++, p = p->next)
        if (!p->margin)
          break;
      if (i > 0)
        gtk_widget_queue_draw_area (tmaid->drawing,
                0, (tmaid->cursor.y - tmaid->top.y + 1) * tmaid->font_height,
                    allocation.width, i * tmaid->font_height);
    }
  if (length <= 0 || !text)
    {
      d->cursor = tmaid->cursor;
      d->select.x = -1;
    }
  else
    {
      /* ja:挿入 */
      gint i, j, put;
      GdkPoint cursor_old, cursor_new;
      LineBuffer *p, *q;

      tmaid->cursor.x = edit_get_align_pos (tmaid,
                                    tmaid->cursor.x, tmaid->cursor.y, FALSE);
      put = edit_put_mem (tmaid, &tmaid->cursor, &cursor_new, text, length);
      tmaid->max += put;/* ja:行数増加 */
      cursor_old = tmaid->cursor;
      if (caret)
        {
          /* ja:キャレット移動 */
          d->cursor = cursor_new;
          d->select = cursor_old;
          tmaid->cursor = cursor_new;
        }
      else
        {
          /* ja:キャレット移動なし */
          d->cursor = cursor_old;
          d->select = cursor_new;
        }
      if (select)
        {
          /* ja:選択する */
          tmaid->select = d->select;
          gtk_selection_owner_set (tmaid->drawing, GDK_SELECTION_PRIMARY,
                                                            GDK_CURRENT_TIME);
        }
      /* ja:挿入した行の挿入した桁より右を初期化 */
      gtk_widget_queue_draw_area (tmaid->drawing,
                            (cursor_old.x - tmaid->top.x) * tmaid->font_width,
                            (cursor_old.y - tmaid->top.y) * tmaid->font_height,
                            allocation.width, tmaid->font_height);
      /* ja:増えた行数分だけスクロール */
      misc_scroll_window (tmaid->drawing, 0, put * tmaid->font_height,
            0,
            (cursor_old.y - tmaid->top.y + 1) * tmaid->font_height,
            allocation.width,
            MAX (allocation.height
                - (cursor_old.y - tmaid->top.y + 1) * tmaid->font_height, 0));
      /* ja:挿入を終えた位置から右マージンの影響のある行数だけ初期化 */
      p = q = edit_get_line_buf (&tmaid->start, &tmaid->off, cursor_new.y);
      for (i = cursor_new.y, j = 0;
                                p->prev && p->prev->margin && cursor_old.y < i;
                                                        i--, j++, p = p->prev);
      while (q->next && q->margin)
        j++, q = q->next;
      if (j > 0)
        gtk_widget_queue_draw_area (tmaid->drawing,
                                0, (i - tmaid->top.y + 1) * tmaid->font_height,
                                allocation.width, j * tmaid->font_height);
    }
  if (tmaid->cursor.x < tmaid->top.x)
    tmaid->top.x = tmaid->cursor.x;
  else if (tmaid->cursor.x - sx + 1 > tmaid->top.x)
    tmaid->top.x = tmaid->cursor.x - sx + 1;
  if (tmaid->cursor.y < tmaid->top.y)
    tmaid->top.y = tmaid->cursor.y;
  else if (tmaid->cursor.y - sy + 1 > tmaid->top.y)
    tmaid->top.y = tmaid->cursor.y - sy + 1;
  if (tmaid->top.y > tmaid->max - sy)
    tmaid->top.y = MAX (tmaid->max - sy, 0);
  max = edit_get_width_max (tmaid);
  if (tmaid->top.x > max - sx + 1)
    tmaid->top.x = MAX (max - sx + 1, 0);
  move_edit_window (tmaid, &top);
  misc_set_scroll_bar (tmaid->hscroll,
                            G_CALLBACK (signal_value_changed_hscroll), tmaid,
                                            0, max + 1, sx, tmaid->top.x);
  misc_set_scroll_bar (tmaid->vscroll,
                            G_CALLBACK (signal_value_changed_vscroll), tmaid,
                                            0, tmaid->max, sy, tmaid->top.y);
  draw_caret (tmaid, NULL);
  return d;
}


/*  ja:リストから削除する
      d,リストの先頭
    RET,削除したリストの数                                                  */
gint
delete_list (TmaidHistory **d)
{
  gint count = 0;
  TmaidHistory *d0, *d1;

  for (d0 = *d; d0; d0 = d1)
    {
      g_free (d0->text);
      d1 = d0->next;
      g_free (d0);
      count++;
    }
  *d = NULL;
  return count;
}


/*  ja:右マージンで折り返す
    tmaid,ウインドウ情報                                                 */
void
modify_margin (TmaidWindow *tmaid)
{
    gint data_pos, screen_pos;
    LineBuffer *p, *q;

  /* ja:ラインバッファの先頭へ移動 */
  while (tmaid->start->prev)
    tmaid->start = tmaid->start->prev;
  tmaid->off = 0;
  p = tmaid->start;
  if (tmaid->ft.limit)
    /* ja:右マージンによる改行あり */
    while (p)
      {
        data_pos = screen_pos = 0;
        while (data_pos < p->length)
          if (p->text[data_pos] == '\t')
            {
              screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
              if (tmaid->ft.margin < screen_pos)
                break;
              data_pos++;
            }
          else if (data_pos + charset_length (p->text[data_pos]) <= p->length)
            {
              screen_pos += charset_width (tmaid->layout, 
                                        p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
              if (tmaid->ft.margin < screen_pos)
                break;
              data_pos += charset_length (p->text[data_pos]);
            }
          else
            {
              screen_pos++;
              if (tmaid->ft.margin < screen_pos)
                break;
              data_pos++;
            }
        if (tmaid->ft.margin < screen_pos)
          {
            /* ja:右マージンを超えているとき、擬似改行する */
            q = g_malloc (sizeof (LineBuffer));
            q->length = p->length - data_pos;
            q->margin = p->margin;
            q->text = g_malloc (q->length * sizeof (gchar));
            q->prev = p;
            q->next = p->next;
            p->next = q;
            if (q->next)
              q->next->prev = q;
            g_memmove (q->text, p->text + data_pos,
                                                q->length * sizeof (gchar));
            p->length = data_pos;
            p->margin = TRUE;
            p->text = g_realloc (p->text, p->length * sizeof (gchar));
            p = q;
            tmaid->max++;
          }
        else if (screen_pos < tmaid->ft.margin && p->margin && p->next)
          {
            /* ja:右マージンを下回り擬似改行のとき、次の行とあわせる */
            data_pos = p->length;
            q = p->next;
            if (q->next)
              q->next->prev = p;
            p->next = q->next;
            p->length += q->length;
            p->margin = q->margin;
            p->text = g_realloc (p->text, p->length * sizeof (gchar));
            g_memmove (p->text + data_pos, q->text,
                                                q->length * sizeof (gchar));
            g_free (q->text);
            g_free (q);
            tmaid->max--;
          }
        else
          {
            p=p->next;
          }
      }
  else
    /* ja:右マージンによる改行なし */
    while (p)
      if (p->margin)
        {
          /* ja:擬似改行のとき、次の行とあわせる */
          data_pos = p->length;
          q = p->next;
          if (q->next)
            q->next->prev = p;
          p->next = q->next;
          p->length += q->length;
          p->margin = q->margin;
          p->text = g_realloc (p->text, p->length * sizeof (gchar));
          g_memmove (p->text + data_pos, q->text, q->length * sizeof (gchar));
          g_free (q->text);
          g_free (q);
          tmaid->max--;
        }
      else
        {
          p = p->next;
        }
}


/*  ja:ファイルの保存を問い合わせる
    tmaid,TXTウインドウ情報
      RET,TRUE:ファイルは閉じる,FALSE:ファイルを閉じない                    */
gboolean
prompt_close (TmaidWindow *tmaid)
{
  gint page;

  page = orz_mdi_get_page_from_data (ORZ_MDI (mdi), tmaid);
  if (orz_mdi_get_edited (ORZ_MDI (mdi), page))
    {
      gchar *utf8str;
      gint result;
      GtkWidget *dialog;

      utf8str = g_filename_display_name
                                    (orz_mdi_get_file (ORZ_MDI (mdi), page));
      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_INFO, GTK_BUTTONS_NONE,
                            _("File %s was edited.\nSave?"), utf8str);
      g_free (utf8str);
      gtk_dialog_add_buttons (GTK_DIALOG (dialog),
#ifdef G_OS_WIN32
                                        GTK_STOCK_YES, GTK_RESPONSE_YES,
                                        GTK_STOCK_NO, GTK_RESPONSE_NO,
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
#else /* not G_OS_WIN32 */
                                        GTK_STOCK_NO, GTK_RESPONSE_NO,
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                        GTK_STOCK_YES, GTK_RESPONSE_YES,
#endif /* not G_OS_WIN32 */
                                        NULL);
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      result = gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      switch (result)
        {
          case GTK_RESPONSE_YES:
            {
              gint page_num;

              page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
              if (page != page_num)
                gtk_notebook_set_current_page (GTK_NOTEBOOK (mdi), page);
              else
                page_num = -1;
              command_save (NULL, NULL);
              if (page_num >= 0)
                gtk_notebook_set_current_page (GTK_NOTEBOOK (mdi), page_num);
              if (!orz_mdi_get_edited (ORZ_MDI (mdi), page))
                break;
            }
            return FALSE;
          case GTK_RESPONSE_NO:
            break;
          default:
            return FALSE;
        }
    }
  return TRUE;
}


/*  ja:キャラクターセットを置換する
    charlist,置換するキャラクターセット
       ft_id,ファイルタイプのID(-1:追加なし)                                */
void
charlist_renewal_all (const gchar *charlist,
                      const gint   ft_id)
{
  gchar *charset;
  gint i;

  for (i = 0; i < ftnum; i++)
    {
      charset = ftype[i].ft_id == ft_id
                                ? charuty_append (ftype[i].charset, charlist)
                                : charuty_rename (ftype[i].charset, charlist);
      g_free (ftype[i].charset);
      ftype[i].charset = charset;
    }
  for (i = orz_mdi_get_n_pages (ORZ_MDI (mdi)) - 1; i >= 0; i--)
    {
      TmaidWindow *tmaid;

      tmaid = orz_mdi_get_data (ORZ_MDI (mdi), i);
      charset = charuty_rename (tmaid->ft.charset, charlist);
      g_free (tmaid->ft.charset);
      tmaid->ft.charset = charset;
    }
}


/******************************************************************************
*                                                                             *
* ja:高レベル関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:編集履歴操作
    tmaid,ウインドウ情報
     kind,TRUE:redo,FALSE:undo                                              */
void
history_operation (TmaidWindow    *tmaid,
                   const gboolean  kind)
{
  GdkPoint cursor, select;
  TmaidHistory *d0, *d1;

  if (kind)
    {
      d0 = tmaid->redo;
      tmaid->redo = d0->next;
    }
  else
    {
      d0 = tmaid->undo;
      tmaid->undo = d0->next;
    }
  cursor = tmaid->cursor;
  select = tmaid->select;
  tmaid->cursor = d0->cursor;
  tmaid->select = d0->select;
  if (select.x >= 0)
    clear_sel (tmaid, &select, &cursor);
  d1 = edit_operation (tmaid, d0->text, d0->length, d0->caret, TRUE);
  if (kind)
    {
      d1->next = tmaid->undo;
      tmaid->undo = d1;
    }
  else
    {
      d1->next = tmaid->redo;
      tmaid->redo = d1;
    }
  g_free (d0->text);
  g_free (d0);
  set_menu_bar (tmaid);
}


/*  ja:右マージンで改行する
    tmaid,ウインドウ情報                                                    */
void
margin_operation (TmaidWindow *tmaid)
{
  gboolean limit;
  GtkAllocation allocation;
  LineBuffer *p;

  gtk_widget_get_allocation (tmaid->drawing, &allocation);
  delete_list (&tmaid->undo);
  delete_list (&tmaid->redo);
  limit = tmaid->ft.limit;
  tmaid->ft.limit = TRUE;
  modify_margin (tmaid);
  for (p = tmaid->start; p->prev; p = p->prev);
  while (p)
    {
      p->margin = FALSE;
      p = p->next;
    }
  tmaid->ft.limit = limit;
  orz_mdi_set_edited (ORZ_MDI (mdi),
                    orz_mdi_get_page_from_data (ORZ_MDI (mdi), tmaid), TRUE);
  tmaid->cursor.x = tmaid->cursor.y = tmaid->top.x = tmaid->top.y = 0;
  tmaid->select.x = -1;
  misc_set_scroll_bar (tmaid->hscroll,
            G_CALLBACK (signal_value_changed_hscroll), tmaid,
            0, edit_get_width_max (tmaid) + 1,
            MAX (allocation.width / tmaid->font_width, 1), tmaid->top.x);
  misc_set_scroll_bar (tmaid->vscroll,
            G_CALLBACK (signal_value_changed_vscroll), tmaid,
            0, tmaid->max,
            MAX (allocation.height / tmaid->font_height, 1), tmaid->top.y);
  set_menu_bar (tmaid);
  gtk_widget_queue_draw (tmaid->drawing);
}


/*  ja:タブをスペースに変換する
    tmaid,ウインドウ情報                                                    */
void
tab_operation (TmaidWindow *tmaid)
{
  gint data_pos, screen_pos;
  LineBuffer *p;

  delete_list (&tmaid->undo);
  delete_list (&tmaid->redo);
  for (p = tmaid->start; p->prev; p = p->prev);
  while (p)
    {
      data_pos = screen_pos = 0;
      while (data_pos < p->length)
        if (p->text[data_pos] == '\t')
          {
            gint space;

            space = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab
                                                                - screen_pos;
            if (space > 1)
              {
                p->text = g_realloc (p->text, (p->length + space - 1)
                                                            * sizeof (gchar));
                g_memmove (p->text + data_pos + space,
                                p->text + data_pos + 1,
                                (p->length - data_pos - 1) * sizeof (gchar));
                p->length += space - 1;
              }
            g_memset (p->text + data_pos, ' ', space * sizeof (gchar));
            screen_pos += space;
            data_pos += space;
          }
        else if (data_pos + charset_length (p->text[data_pos]) <= p->length)
          {
            screen_pos += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
            data_pos+=charset_length(p->text[data_pos]);
          }
        else
          {
            screen_pos++;
            data_pos++;
          }
      p = p->next;
    }
  orz_mdi_set_edited (ORZ_MDI (mdi),
                    orz_mdi_get_page_from_data (ORZ_MDI (mdi), tmaid), TRUE);
  tmaid->select.x = -1;
  set_menu_bar (tmaid);
  gtk_widget_queue_draw (tmaid->drawing);
}
