/*
    Text maid
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "charset.h"


/******************************************************************************
*                                                                             *
* ja:キャラクターセット関数群                                                 *
*                                                                             *
******************************************************************************/
gsize charset_leng[256] = {
        /*  0 1 2 3 4 5 6 7 8 9 A B C D E F 0 1 2 3 4 5 6 7 8 9 A B C D E F */
    /* 0 */ 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    /* 2 */ 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    /* 4 */ 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    /* 6 */ 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    /* 8 */ 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    /* A */ 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    /* C */ 2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
    /* E */ 3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,5,5,5,5,6,6,1,1};
guchar charset_space[] = {3, 0xE3, 0x80, 0x80, 0};


/*  ja:フォントの幅と高さを取得する
      layout,レイアウト
       width,幅
      height,高さ
    font_buf,バッファ                                                       */
void
charset_size (PangoLayout *layout,
              gint        *width,
              gint        *height,
              guint8      *font_buf)
{
  int w, h;
  gint i;

  if (width)
    *width = 1;
  if (height)
    *height = 1;
  if (font_buf)
    g_memset (font_buf, 0, sizeof(guint8) * 65536);
  for (i = 0; i < 128; i++)
    if (charset_length (i) == 1 && g_unichar_isprint ((gunichar)i))
      {
        pango_layout_set_text (layout, (const char *)&i, 1);
        pango_layout_get_pixel_size (layout, &w, &h);
        if (width && *width < w)
          *width = w;
        if (height && *height < h)
          *height = h;
      }
}


/*  ja:表示上の半角単位の文字幅を求める
      layout,レイアウト
        text,文字
       width,幅
    font_buf,バッファ
         RET,半角単位                                                       */
gint
charset_width (PangoLayout *layout,
               const gchar *text,
               const gint   width,
               guint8      *font_buf)
{
  int w;
  gint result;
  gsize length;
  gunichar c;

  length = charset_length (*text);
  c = g_utf8_get_char_validated (text, length);
  if (!g_unichar_validate (c))
    return 1;
  if ((guint32)c < 65536)
    {
      if (length <= 1)
        font_buf[c] = 1;
      if (font_buf[c])
        return font_buf[c];
    }
  pango_layout_set_text (layout, text, length);
  pango_layout_get_pixel_size (layout, &w, NULL);
  result = MAX ((w + width - 1) / width, 1);
  if ((guint32)c < 65536)
    font_buf[c] = result;
  return result;
}


/*  ja:文字の種類を返す
    text,文字
     RET,文字の種類                                                         */
gint
charset_block (const gchar *text)
{
  gint i;
  gunichar c;
  struct
  {
    gunichar min,max;
    guint type;
  } type_range[] = {
        {0x000000, 0x00007f, CS_BASIC_LATIN},
        {0x000080, 0x0000ff, CS_LATIN_1_SUPPLEMENT},
        {0x000100, 0x00017f, CS_LATIN_EXTENDED_A},
        {0x000180, 0x00024f, CS_LATIN_EXTENDED_B},
        {0x000250, 0x0002af, CS_IPA_EXTENSIONS},
        {0x0002b0, 0x0002ff, CS_SPACING_MODIFIER_LETTERS},
        {0x000300, 0x00036f, CS_COMBINING_DIACRITICAL_MARKS},
        {0x000370, 0x0003ff, CS_GREEK_AND_COPTIC},
        {0x000400, 0x0004ff, CS_CYRILLIC},
        {0x000500, 0x00052f, CS_CYRILLIC_SUPPLEMENTARY},
        {0x000530, 0x00058f, CS_ARMENIAN},
        {0x000590, 0x0005ff, CS_HEBREW},
        {0x000600, 0x0006ff, CS_ARABIC},
        {0x000700, 0x00074f, CS_SYRIAC},
        {0x000780, 0x0007bf, CS_THAANA},
        {0x000900, 0x00097f, CS_DEVANAGARI},
        {0x000980, 0x0009ff, CS_BENGALI},
        {0x000a00, 0x000a7f, CS_GURMUKHI},
        {0x000a80, 0x000aff, CS_GUJARATI},
        {0x000b00, 0x000b7f, CS_ORIYA},
        {0x000b80, 0x000bff, CS_TAMIL},
        {0x000c00, 0x000c7f, CS_TELUGU},
        {0x000c80, 0x000cff, CS_KANNADA},
        {0x000d00, 0x000d7f, CS_MALAYALAM},
        {0x000d80, 0x000dff, CS_SINHALA},
        {0x000e00, 0x000e7f, CS_THAI},
        {0x000e80, 0x000eff, CS_LAO},
        {0x000f00, 0x000fff, CS_TIBETAN},
        {0x001000, 0x00109f, CS_MYANMAR},
        {0x0010a0, 0x0010ff, CS_GEORGIAN},
        {0x001100, 0x0011ff, CS_HANGUL_JAMO},
        {0x001200, 0x00137f, CS_ETHIOPIC},
        {0x0013a0, 0x0013ff, CS_CHEROKEE},
        {0x001400, 0x00167f, CS_UNIFIED_CANADIAN_ABORIGINAL_SYLLABICS},
        {0x001680, 0x00169f, CS_OGHAM},
        {0x0016a0, 0x0016ff, CS_RUNIC},
        {0x001700, 0x00171f, CS_TAGALOG},
        {0x001720, 0x00173f, CS_HANUNOO},
        {0x001740, 0x00175f, CS_BUHID},
        {0x001760, 0x00177f, CS_TAGBANWA},
        {0x001780, 0x0017ff, CS_KHMER},
        {0x001800, 0x0018af, CS_MONGOLIAN},
        {0x001900, 0x00194f, CS_LIMBU},
        {0x001950, 0x00197f, CS_TAI_LE},
        {0x0019e0, 0x0019ff, CS_KHMER_SYMBOLS},
        {0x001d00, 0x001d7f, CS_PHONETIC_EXTENSIONS},
        {0x001e00, 0x001eff, CS_LATIN_EXTENDED_ADDITIONAL},
        {0x001f00, 0x001fff, CS_GREEK_EXTENDED},
        {0x002000, 0x00206f, CS_GENERAL_PUNCTUATION},
        {0x002070, 0x00209f, CS_SUPERSCRIPTS_AND_SUBSCRIPTS},
        {0x0020a0, 0x0020cf, CS_CURRENCY_SYMBOLS},
        {0x0020d0, 0x0020ff, CS_COMBINING_DIACRITICAL_MARKS_FOR_SYMBOLS},
        {0x002100, 0x00214f, CS_LETTERLIKE_SYMBOLS},
        {0x002150, 0x00218f, CS_NUMBER_FORMS},
        {0x002190, 0x0021ff, CS_ARROWS},
        {0x002200, 0x0022ff, CS_MATHEMATICAL_OPERATORS},
        {0x002300, 0x0023ff, CS_MISCELLANEOUS_TECHNICAL},
        {0x002400, 0x00243f, CS_CONTROL_PICTURES},
        {0x002440, 0x00245f, CS_OPTICAL_CHARACTER_RECOGNITION},
        {0x002460, 0x0024ff, CS_ENCLOSED_ALPHANUMERICS},
        {0x002500, 0x00257f, CS_BOX_DRAWING},
        {0x002580, 0x00259f, CS_BLOCK_ELEMENTS},
        {0x0025a0, 0x0025ff, CS_GEOMETRIC_SHAPES},
        {0x002600, 0x0026ff, CS_MISCELLANEOUS_SYMBOLS},
        {0x002700, 0x0027bf, CS_DINGBATS},
        {0x0027c0, 0x0027ef, CS_MISCELLANEOUS_MATHEMATICAL_SYMBOLS_A},
        {0x0027f0, 0x0027ff, CS_SUPPLEMENTAL_ARROWS_A},
        {0x002800, 0x0028ff, CS_BRAILLE_PATTERNS},
        {0x002900, 0x00297f, CS_SUPPLEMENTAL_ARROWS_B},
        {0x002980, 0x0029ff, CS_MISCELLANEOUS_MATHEMATICAL_SYMBOLS_B},
        {0x002a00, 0x002aff, CS_SUPPLEMENTAL_MATHEMATICAL_OPERATORS},
        {0x002b00, 0x002bff, CS_MISCELLANEOUS_SYMBOLS_AND_ARROWS},
        {0x002e80, 0x002eff, CS_CJK_RADICALS_SUPPLEMENT},
        {0x002f00, 0x002fdf, CS_KANGXI_RADICALS},
        {0x002ff0, 0x002fff, CS_IDEOGRAPHIC_DESCRIPTION_CHARACTERS},
        {0x003000, 0x00303f, CS_CJK_SYMBOLS_AND_PUNCTUATION},
        {0x003040, 0x00309f, CS_HIRAGANA},
        {0x0030a0, 0x0030ff, CS_KATAKANA},
        {0x003100, 0x00312f, CS_BOPOMOFO},
        {0x003130, 0x00318f, CS_HANGUL_COMPATIBILITY_JAMO},
        {0x003190, 0x00319f, CS_KANBUN},
        {0x0031a0, 0x0031bf, CS_BOPOMOFO_EXTENDED},
        {0x0031f0, 0x0031ff, CS_KATAKANA_PHONETIC_EXTENSIONS},
        {0x003200, 0x0032ff, CS_ENCLOSED_CJK_LETTERS_AND_MONTHS},
        {0x003300, 0x0033ff, CS_CJK_COMPATIBILITY},
        {0x003400, 0x004dbf, CS_CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A},
        {0x004dc0, 0x004dff, CS_YIJING_HEXAGRAM_SYMBOLS},
        {0x004e00, 0x009fff, CS_CJK_UNIFIED_IDEOGRAPHS},
        {0x00a000, 0x00a48f, CS_YI_SYLLABLES},
        {0x00a490, 0x00a4cf, CS_YI_RADICALS},
        {0x00ac00, 0x00d7af, CS_HANGUL_SYLLABLES},
        {0x00d800, 0x00db7f, CS_HIGH_SURROGATES},
        {0x00db80, 0x00dbff, CS_HIGH_PRIVATE_USE_SURROGATES},
        {0x00dc00, 0x00dfff, CS_LOW_SURROGATES},
        {0x00e000, 0x00f8ff, CS_PRIVATE_USE_AREA},
        {0x00f900, 0x00faff, CS_CJK_COMPATIBILITY_IDEOGRAPHS},
        {0x00fb00, 0x00fb4f, CS_ALPHABETIC_PRESENTATION_FORMS},
        {0x00fb50, 0x00fdff, CS_ARABIC_PRESENTATION_FORMS_A},
        {0x00fe00, 0x00fe0f, CS_VARIATION_SELECTORS},
        {0x00fe20, 0x00fe2f, CS_COMBINING_HALF_MARKS},
        {0x00fe30, 0x00fe4f, CS_CJK_COMPATIBILITY_FORMS},
        {0x00fe50, 0x00fe6f, CS_SMALL_FORM_VARIANTS},
        {0x00fe70, 0x00feff, CS_ARABIC_PRESENTATION_FORMS_B},
        {0x00ff00, 0x00ffef, CS_HALFWIDTH_AND_FULLWIDTH_FORMS},
        {0x00fff0, 0x00ffff, CS_SPECIALS},
        {0x010000, 0x01007f, CS_LINEAR_B_SYLLABARY},
        {0x010080, 0x0100ff, CS_LINEAR_B_IDEOGRAMS},
        {0x010100, 0x01013f, CS_AEGEAN_NUMBERS},
        {0x010300, 0x01032f, CS_OLD_ITALIC},
        {0x010330, 0x01034f, CS_GOTHIC},
        {0x010380, 0x01039f, CS_UGARITIC},
        {0x010400, 0x01044f, CS_DESERET},
        {0x010450, 0x01047f, CS_SHAVIAN},
        {0x010480, 0x0104af, CS_OSMANYA},
        {0x010800, 0x01083f, CS_CYPRIOT_SYLLABARY},
        {0x01d000, 0x01d0ff, CS_BYZANTINE_MUSICAL_SYMBOLS},
        {0x01d100, 0x01d1ff, CS_MUSICAL_SYMBOLS},
        {0x01d300, 0x01d35f, CS_TAI_XUAN_JING_SYMBOLS},
        {0x01d400, 0x01d7ff, CS_MATHEMATICAL_ALPHANUMERIC_SYMBOLS},
        {0x020000, 0x02a6df, CS_CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B},
        {0x02f800, 0x02fa1f, CS_CJK_COMPATIBILITY_IDEOGRAPHS_SUPPLEMENT},
        {0x0e0000, 0x0e007f, CS_TAGS},
        {0x0e0100, 0x0e01ef, CS_VARIATION_SELECTORS_SUPPLEMENT},
        {0x0f0000, 0x0fffff, CS_SUPPLEMENTARY_PRIVATE_USE_AREA_A},
        {0x100000, 0x10ffff, CS_SUPPLEMENTARY_PRIVATE_USE_AREA_B},
        {0,        0,        0}};

  c = g_utf8_get_char (text);
  for (i = 0; type_range[i].type; i++)
    if (type_range[i].min <= c && c <= type_range[i].max)
      return type_range[i].type;
  return 0;
}
