/*
    Text maid
    copyright (c) 1998-2008 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "abort.h"
#include "charset.h"
#include "edit.h"
#include "find.h"
#include "general.h"
#include "sigfile.h"
#include "misc/misc.h"


gboolean find_arrow = TRUE;                 /* ja:検索の設定 */
gboolean find_ignorecase = FALSE;           /* ja:検索の設定 */
gchar *find_text[FIND_TEXT];                /* ja:検索する文字列 */


/******************************************************************************
*                                                                             *
* ja:検索関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:文字の種類を求める
      type,文字の属性を格納するバッファ
      text,文字列
    length,データサイズ                                                     */
static void
make_char_type (gint        *type,
                const gchar *text,
                const gint   length)
{
  gint i = 0, j;

  while (i < length)
    if (i + charset_length (text[i]) <= length)
      {
        for (j = 0; j < charset_length (text[i]); j++)
          type[i + j] = j + 1;
        i += charset_length (text[i]);
      }
    else
      {
        type[i++] = 1;
      }
}


/*  ja:検索
         tmaid,TXTウインドウ情報
         start,検索範囲
           end,検索範囲(x=-1ならば選択範囲なし)
          find,検索文字列
         arrow,TRUE:下へ,FALSE:上へ
    ignorecase,TRUE:大文字小文字を区別する,FALSE:大文字小文字を区別しない
           RET,TRUE:文字列あり,FALSE:文字列なし                             */
gboolean
find_operation (TmaidWindow    *tmaid,
                GdkPoint       *start,
                GdkPoint       *end,
                const gchar    *find,
                const gboolean  arrow,
                const gboolean  ignorecase)
{
  gboolean result;              /* ja:比較の結果,TRUE:同じ,FALSE:異なる */
  gchar *text;
  gint i, max = 0;
  gint length0, length1 = 0;    /* ja:文字数(バイト単位) */
  gint *buf = NULL, *type = NULL;
  LineBuffer *p, *q;
  GdkPoint st, ed, data;

  st = *start;
  ed = *end;
  length0 = g_strlen (find);
  for (i = 0; i < 256; i++)
    if (charset_length (i) > max)
      max = charset_length (i);
  text = g_malloc ((length0 + max) * sizeof (gchar));
  buf = g_malloc ((length0 + max) * sizeof (gint));
  if (ed.x >= 0)
    {
      /* ja:検索結果による選択のときには選択を無効 */
      i = edit_get_sel_bytes (tmaid, &st, &ed);
      if (i <= length0)
        {
          edit_cpy_sel_mem (tmaid, &st, &ed, text);
          text[i] = '\0';
          if (ignorecase)   /* ja:大文字/小文字を区別する */
            result = g_strcmp (find, text) == 0;
          else              /* ja:大文字/小文字を区別しない */
            result = g_ascii_strcasecmp (find, text) == 0;
          if (result)
            ed.x = -1;
        }
    }
  if (ed.x < 0)
    {
      if (arrow)
        {
          ed.x = edit_get_width (tmaid, tmaid->max - 1);
          ed.y = tmaid->max - 1;/* ja:下へ */
        }
      else
        {
          ed.x = ed.y = 0;/* ja:上へ */
        }
    }
  ed.x = edit_get_data_pos (tmaid, ed.x, ed.y, FALSE);
  st.x = edit_get_data_pos (tmaid, st.x, st.y, FALSE);
  p = edit_get_line_buf (&tmaid->start, &tmaid->off, st.y);
  /* ja:同じ文字数で比較するために文字数を数える */
  q = p;
  data = st;
  result = FALSE;
  if (st.y < ed.y || (st.y == ed.y && st.x < ed.x))
    {
      /* ja:上から下 */
      while (st.y < ed.y || (st.y == ed.y && st.x < ed.x))
        {
          /* ja:バッファの最後に文字加える */
          while (length1 < length0
                    && (data.y < ed.y || (data.y == ed.y && data.x < ed.x)))
            {
              if (data.x < q->length)
                {
                  /* ja:文字 */
                  if (data.x + charset_length (q->text[data.x]) <= q->length)
                    {
                      g_memmove (text + length1, q->text + data.x,
                                            charset_length (q->text[data.x])
                                                            * sizeof (gchar));
                      for (i = 0; i < charset_length (q->text[data.x]); i++)
                        buf[length1 + i] = charset_length (q->text[data.x]);
                      length1 += charset_length (q->text[data.x]);
                      data.x += charset_length (q->text[data.x]);
                    }
                  else
                    {
                      text[length1] = q->text[data.x];
                      buf[length1] = 1;
                      length1++;
                      data.x++;
                    }
                }
              else
                {
                  /* ja:改行 */
                  while (gtk_events_pending ())
                    gtk_main_iteration ();
                  if (!abort_break)
                    goto loop;
                  if (!q->margin)
                    {
                      text[length1] = '\n';
                      buf[length1] = 1;
                      length1++;
                    }
                  data.x = 0;
                  data.y++;
                  q = q->next;
                }
            }
          text[length1] = '\0';
          /* ja:比較 */
          if (ignorecase)   /* ja:大文字/小文字を区別する */
            result = g_strcmp (find, text) == 0;
          else              /* ja:大文字/小文字を区別しない */
            result = g_ascii_strcasecmp (find, text) == 0;
          if (result)
            break;/* ja:一致するとき */
          /* ja:バッファの先頭の文字を破棄する */
          if (st.x < p->length)
            {
              st.x += buf[0];
            }
          else
            {
              st.x = p->margin ? buf[0] : 0;
              st.y++;
              p = p->next;
            }
          length1 -= buf[0];
          g_memmove (text, text + buf[0], length1 * sizeof (gchar));
          g_memmove (buf, buf + buf[0], length1 * sizeof (gint));
        }
    }
  else
    {
      /* ja:下から上 */
      if (p->length > 0)
        {
          type = g_malloc (p->length * sizeof (gint));
          make_char_type (type, p->text, p->length);
        }
      while (ed.y < st.y || (st.y == ed.y && ed.x < st.x))
        {
          /* ja:バッファの最初に文字加える */
          while (length1 < length0
                    && (ed.y < data.y || (data.y == ed.y && ed.x < data.x)))
            {
              if (0 < data.x)
                {
                  /* ja:文字 */
                  g_memmove (text + type[data.x - 1], text,
                                                    length1 * sizeof (gchar));
                  g_memmove (buf + type[data.x - 1], buf,
                                                    length1 * sizeof (gint));
                  g_memmove (text, q->text + data.x - type[data.x - 1],
                                            type[data.x - 1] * sizeof (gchar));
                  for (i = 0; i < type[data.x - 1]; i++)
                    buf[i] = i + 1;
                  length1 += type[data.x - 1];
                  data.x -= type[data.x - 1];
                }
              else
                {
                  /* ja:改行 */
                  while (gtk_events_pending ())
                    gtk_main_iteration ();
                  if (!abort_break)
                    goto loop;
                  q = q->prev;
                  type = g_realloc (type, q->length * sizeof (gint));
                  make_char_type (type, q->text, q->length);
                  if (!q->margin)
                    {
                      g_memmove (text + 1, text, length1 * sizeof (gchar));
                      g_memmove (buf + 1, buf, length1 * sizeof (gint));
                      text[0] = '\n';
                      buf[0] = 1;
                      length1++;
                    }
                  data.x = q->length;
                  data.y--;
                }
            }
          text[length1] = '\0';
          /* ja:比較 */
          if (ignorecase)   /* ja:大文字/小文字を区別する */
            result = g_strcmp (find, text) == 0;
          else              /* ja:大文字/小文字を区別しない */
            result = g_ascii_strcasecmp (find, text) == 0;
          if (result)
            break;/* ja:一致するとき */
          /* ja:バッファの最後の文字を破棄する */
          if (st.x > 0)
            {
              st.x -= buf[length1 - 1];
            }
          else
            {
              p = p->prev;
              st.x = p->length - (p->margin ? buf[length1 - 1] : 0);
              st.y--;
            }
          length1 -= buf[length1 - 1];
        }
    }
  loop:
  g_free (type);
  /* ja:終了処理 */
  if (result)
    {
      gint sx, sy;
      GdkPoint cursor, select, top;

      cursor = tmaid->cursor;
      select = tmaid->select;
      top = tmaid->top;
      if (tmaid->select.x >= 0)
        {
          /* ja:選択範囲を解除 */
          tmaid->select.x = -1;
          clear_sel (tmaid, &select, &tmaid->cursor);
        }
      if (p->margin && p->length <= st.x)
        {
          /* ja:右マージンのときには次の行の先頭 */
          st.x = 0;
          st.y++;
        }
      if (q->margin && q->length <= data.x)
        {
          /* ja:右マージンのときには次の行の先頭 */
          data.x = 0;
          data.y++;
        }
      tmaid->cursor.x = edit_get_screen_pos (tmaid, data.x, data.y);
      tmaid->cursor.y = data.y;
      tmaid->select.x = edit_get_screen_pos (tmaid, st.x, st.y);
      tmaid->select.y = st.y;
      sx = MAX (tmaid->drawing->allocation.width / tmaid->font_width, 1);
      sy = MAX (tmaid->drawing->allocation.height / tmaid->font_height, 1);
      if (tmaid->cursor.x < tmaid->top.x)
        tmaid->top.x = tmaid->cursor.x;
      else if (tmaid->cursor.x - sx + 1 > tmaid->top.x)
        tmaid->top.x = tmaid->cursor.x - sx + 1;
      if (tmaid->cursor.y < tmaid->top.y)
        tmaid->top.y = tmaid->cursor.y;
      else if (tmaid->cursor.y - sy + 1 > tmaid->top.y)
        tmaid->top.y = tmaid->cursor.y - sy + 1;
      move_edit_window (tmaid, &top);
      misc_set_scroll_bar (tmaid->hscroll,
                        G_CALLBACK (signal_value_changed_hscroll), tmaid,
                        0, edit_get_width_max (tmaid) + 1, sx, tmaid->top.x);
      misc_set_scroll_bar (tmaid->vscroll,
                        G_CALLBACK (signal_value_changed_vscroll), tmaid,
                        0, tmaid->max, sy, tmaid->top.y);
      clear_sel (tmaid, &tmaid->select, &tmaid->cursor);
      draw_caret (tmaid, &cursor);
      gtk_selection_owner_set (tmaid->drawing, GDK_SELECTION_PRIMARY,
                                                            GDK_CURRENT_TIME);
      if (select.x < 0)
        set_menu_bar (tmaid);
    }
  return result;
}
