/*
    Text maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "charset.h"
#include "edit.h"
#include "file.h"
#include "general.h"
#include "jump.h"
#include "sigfile.h"
#include "sigmain.h"
#include "misc/argument.h"
#include "misc/misc.h"
#include "orz/orzhistory.h"
#include "orz/orzmdi.h"


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(テキスト)                                        *
*                                                                             *
******************************************************************************/
void
signal_style_set (GtkWidget *widget,
                  GtkStyle  *style,
                  gpointer   user_data)
{
  gint i;

  /* ja:システム色 */
  style = gtk_widget_get_style (widget);
  system_color[0] = style->text[0];
  system_color[1] = style->base[0];
  system_color[2] = style->dark[0];
  system_color[3] = style->dark[0];
  system_color[4] = style->dark[0];
  system_color[5] = style->dark[0];
  system_color[6] = style->dark[0];
  system_color[7] = style->mid[0];
  system_color[8] = style->fg[3];
  system_color[9] = style->bg[3];
  if (system_font)
    pango_font_description_free (system_font);
  /* ja:システムフォント */
  system_font = pango_font_description_copy
                                    (pango_context_get_font_description
                                    (gtk_widget_get_pango_context (widget)));
  for (i = orz_mdi_get_n_pages (ORZ_MDI (mdi)) - 1; i >= 0; i--)
    {
      TmaidWindow *tmaid;

      tmaid = orz_mdi_get_data (ORZ_MDI (mdi), i);
      if (tmaid->ft.syscol || !tmaid->ft.font_name)
        {
          if (!tmaid->ft.font_name)
            {
              /* ja:フォント変更 */
              g_object_unref (tmaid->layout);
              gtk_widget_modify_font (tmaid->drawing, system_font);
              tmaid->layout = gtk_widget_create_pango_layout
                                                        (tmaid->drawing, NULL);
              charset_size (tmaid->layout,
                    &tmaid->font_width, &tmaid->font_height, tmaid->font_buf);
              /* ja:スクロールバー */
              tmaid->cursor.x = tmaid->cursor.y
                                            = tmaid->top.x = tmaid->top.y = 0;
              modify_margin (tmaid);
              misc_set_scroll_bar (tmaid->hscroll,
                    G_CALLBACK (signal_value_changed_hscroll), tmaid,
                    0, edit_get_width_max (tmaid) + 1,
                    MAX (tmaid->drawing->allocation.width / tmaid->font_width,
                                                            1), tmaid->top.x);
              misc_set_scroll_bar (tmaid->vscroll,
                    G_CALLBACK (signal_value_changed_vscroll), tmaid,
                    0, tmaid->max,
                    MAX (tmaid->drawing->allocation.height
                                    / tmaid->font_height, 1), tmaid->top.y);
            }
          gtk_widget_draw (tmaid->drawing, NULL);/* ja:再描画 */
        }
    }
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(メニュー)                                        *
*                                                                             *
******************************************************************************/
void
signal_activate_menu_history (GtkWidget   *widget,
                              const gchar *file)
{
  file_open_edit (file);
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(MDI)                                             *
*                                                                             *
******************************************************************************/
void
signal_close_window (GtkWidget   *widget,
                     guint        page_num,
                     TmaidWindow *tmaid)
{
  if (prompt_close (tmaid))
    gtk_notebook_remove_page (GTK_NOTEBOOK (mdi), page_num);
}


void
signal_switch_window (GtkWidget   *widget,
                      guint        page_num,
                      TmaidWindow *tmaid)
{
  /* ja:メニュー表示 */
  set_menu_bar (tmaid);
}


void
signal_destroy_mdi (GtkWidget *widget,
                    gpointer   user_data)
{
  gtk_timeout_remove (timer_id);
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(セレクション)                                    *
*                                                                             *
******************************************************************************/
void
signal_drag_data_received (GtkWidget        *widget,
                           GdkDragContext   *context,
                           gint              x,
                           gint              y,
                           GtkSelectionData *selection_data,
                           guint             info,
                           guint             time)
{
  gchar *file, **files;
  gint i;

  files = g_strsplit (selection_data->data, "\r\n", 0);
  for (i = 0; files[i] != NULL; i++)
    {
      file = g_filename_from_uri (files[i], NULL, NULL);
      if (file)
        {
          file_open_edit (file);
          g_free (file);
        }
    }
  g_strfreev (files);
}


gboolean
signal_selection_clear (GtkWidget         *widget,
                        GdkEventSelection *event,
                        gpointer           user_data)
{
  if (event->selection == GDK_SELECTION_CLIPBOARD)
    {
      /* ja:クリップボード */
      g_free (clipboard_text);
      clipboard_text = NULL;
    }
  return TRUE;
}


void
signal_selection_get (GtkWidget        *widget,
                      GtkSelectionData *data,
                      guint             info,
                      guint             time,
                      TmaidWindow      *tmaid)
{
  if (data->selection==GDK_SELECTION_PRIMARY && tmaid && tmaid->select.x >= 0)
    /* ja:プライマリセレクション */
    {
      gchar *text;
      gint length = 0;

      length = edit_get_sel_bytes (tmaid, &tmaid->select, &tmaid->cursor);
      text = g_malloc ((length + 1) * sizeof (gchar));
      edit_cpy_sel_mem (tmaid, &tmaid->select, &tmaid->cursor, text);
      text[length] = '\0';
      gtk_selection_data_set_text (data, text, length);
      g_free (text);
    }
  else if (data->selection == GDK_SELECTION_CLIPBOARD && clipboard_text)
    {
      /* ja:クリップボード */
      gtk_selection_data_set_text (data,
                                    clipboard_text, g_strlen (clipboard_text));
    }
}


void
signal_selection_received (GtkWidget        *widget,
                           GtkSelectionData *data,
                           guint             time,
                           gpointer          user_data)
{
  if (data->selection == GDK_SELECTION_PRIMARY
                                || data->selection == GDK_SELECTION_CLIPBOARD)
    {
      gchar *text = NULL;

      /* ja:プライマリセレクション/クリップボード */
      if (data->length >= 0)
        {
          if (data->type == GDK_SELECTION_TYPE_ATOM)
            {
              gint i,count;
              GdkAtom *atoms;

              atoms = (GdkAtom *)data->data;
              count = data->length / sizeof (GdkAtom);
              for (i = 0; i < count; i++)
                if (atoms[i] == atom_utf8)
                  break;
              if (i >= count)
                for (i = 0; i < count; i++)
                  if (atoms[i] == atom_compound || atoms[i] == atom_text)
                    break;
              if (i >= count)
                for (i = 0; i < count; i++)
                  if (atoms[i] == GDK_TARGET_STRING)
                    break;
              if (i < count)
                gtk_selection_convert (widget, data->selection, atoms[i],
                                                            GDK_CURRENT_TIME);
            }
          else
            {
              text = gtk_selection_data_get_text (data);
            }
        }
      if (text)
        {
          gint page_num;
          GdkPoint select;
          TmaidWindow *tmaid;
          TmaidHistory *d;

          page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
          tmaid = orz_mdi_get_data (ORZ_MDI (mdi), page_num);
          select = tmaid->select;
          d = edit_operation (tmaid, text, g_strlen (text),
                            data->selection == GDK_SELECTION_CLIPBOARD, FALSE);
          d->next = tmaid->undo;
          tmaid->undo = d;
          if (delete_list (&tmaid->redo) > 0 || !d->next || select.x >= 0)
            set_menu_bar (tmaid);
          orz_mdi_set_edited (ORZ_MDI (mdi), page_num, TRUE);
          g_free (text);
        }
    }
}


void
signal_selection_recv (GObject   *object,
                       gchar    **argv,
                       gpointer   user_data)
{
  gboolean *arg_files;
  gint i, array[2], def[2];
  ArgumentBool arg_n, arg_x;
  ArgumentValue arg_h, arg_l;
  ArgumentArray arg_s;
  ArgumentList arg_opts[] = {
{"history", 'h',  "NUM",          N_("History"),        ARGUMENT_TYPE_VALUE, &arg_h},
{"line",    'l',  "NUM",          N_("Caret Position"), ARGUMENT_TYPE_VALUE, &arg_l},
{"new",     'n',  "ON/OFF",       N_("New File"),       ARGUMENT_TYPE_BOOL,  &arg_n},
{"size",    's',  "WIDTH,HEIGHT", N_("Window Size"),    ARGUMENT_TYPE_ARRAY, &arg_s},
{"second",  'x',  "ON/OFF",       N_("Open Window"),    ARGUMENT_TYPE_BOOL,  &arg_x},
{NULL,      '\0', NULL,           NULL,                 0,                   NULL}};

  /* ja:プロセス間通信 */
  /* ja:ファイルの履歴 */
  arg_h.val = orz_history_get_num (ORZ_HISTORY (history));
  arg_h.def = ORZ_HISTORY_NUM_DEF;
  /* ja:初期のキャレット位置 */
  arg_l.val = 0;
  arg_l.def = 0;
  /* ja:新規ファイル */
  arg_n.flag = newfile;
  arg_n.def = FALSE;
  /* ja:ウインドウサイズ */
  array[0] = def_width;
  array[1] = def_height;
  def[0] = gdk_screen_width () * 2 / 3;
  def[1] = gdk_screen_height () * 2 / 3;
  arg_s.leng = 2;
  arg_s.array = array;
  arg_s.def = def;
  /* ja:新規ウインドウ */
  arg_x.flag = second;
  arg_x.def = FALSE;
  /* ja:コマンドラインの解析 */
  arg_files = arg_analyse (argv, arg_opts);
  orz_history_set_num (ORZ_HISTORY (history), arg_h.val);
  newfile = arg_n.flag;
  def_width = array[0];
  def_height = array[1];
  second = arg_x.flag;
  for (i = 1; argv[i]; i++)
    if (arg_files[i])
      {
        TmaidWindow *tmaid;

        tmaid = file_open_edit (argv[i]);
        if (arg_l.val != 0)   /* ja:指定行に移動 */
          jump_operation (tmaid, arg_l.val);
      }
  g_free (arg_files);
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(メイン)                                          *
*                                                                             *
******************************************************************************/
gboolean
signal_delete (GtkWidget *widget,
               GdkEvent  *event,
               gpointer   user_data)
{
  gint page;

  while ((page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi))) >= 0)
    {
      if (!prompt_close ((TmaidWindow *)
                                    orz_mdi_get_data (ORZ_MDI (mdi), page)))
        return TRUE;
      gtk_notebook_remove_page (GTK_NOTEBOOK (mdi), page);
    }
  return FALSE;
}
