/* fepoch.c */

#include	"tazel.h"

/************************************************************************************************/
// This function convert double TLE-Epoch (epoch) into string "YYYY-MM-DD hh:mm:ss.sss" in UTC.
// The result string is stored into *buf.
/************************************************************************************************/
extern void tleepoch2str(dbl epoch,char *buf){

char	d_frac[BUFSIZ];					/* Buffer for Decimal Fraction							*/
struct tm	tm_epoch;					/* TLE-Epoch Time [UTC]									*/
dbl		yy;								/* 19YY or 20YY											*/
dbl		msec;							/* Time less than 1.0 [msec]							*/

/* Integer Part ( FORMAT = YYDDD.SSSSSSSS ) */
	yy = epoch/1000.0;
	if( 70<=yy )	/* 1970-1999 */
		tm_epoch.tm_year=(int)(epoch/1000.0)+1900-1900;		// YYYY-1900
	else			/* 2000-2069 */
		tm_epoch.tm_year=(int)(epoch/1000.0)+2000-1900;		// YYYY-1900
	tm_epoch.tm_mon		=0;		/* January		*/			//    0-  11
	tm_epoch.tm_mday	=(int)(fmod(epoch,1000.0));			//    1-  31
	tm_epoch.tm_hour	=0;		/* 00:			*/			//    0-  23
	tm_epoch.tm_min		=0;		/* 00:00:		*/			//    0-  59
	tm_epoch.tm_sec		=(int)(fmod(epoch,1.0)*24*60*60);	//    0-  61
	tm_epoch.tm_isdst	=0;
	mktime(&tm_epoch);			/* Normalize it	*/
#ifdef MinGW
	strftime(buf,BUFSIZ,  "%Y-%m-%d %H:%M:%S",&tm_epoch);
#else
	strftime(buf,BUFSIZ,"%04Y-%m-%d %H:%M:%S",&tm_epoch);
#endif

/* Decimal Fraction */
	msec = fmod( fmod(epoch,1.0)*24*60*60 , 1.0 );
	sprintf(d_frac,"%.3f",msec);
	strcat(buf,&d_frac[1]);

}

/************************************************************************************************/
// This function convert double TLE-Epoch stored in TLE into struct TM in UTC.
// The result is returned by value.  If TLE is invalid, "1970-01-01 00:00:00" will be set.
/************************************************************************************************/
extern struct tm tleepoch2tm(tle_t tle){

struct tm	tm_epoch={70,0,1,0,0,0,0};	/* TLE-Epoch Time [UTC]									*/
dbl		yy;								/* 19YY or 20YY											*/

/* Convert */
	if( tle.catno<0 )					/* Invalid												*/
		;
	else{								/* Valid												*/
		yy=tle.epoch/1000.0;
		if( 70<=yy )					/* 1970-1999	*/
			tm_epoch.tm_year=(int)(tle.epoch/1000.0)+1900-1900;		// YYYY-1900
		else							/* 2000-2069	*/
			tm_epoch.tm_year=(int)(tle.epoch/1000.0)+2000-1900;		// YYYY-1900
		tm_epoch.tm_mon		=0;			/* January		*/			//    0-  11
		tm_epoch.tm_mday	=(int)(fmod(tle.epoch,1000.0));			//    1-  31
		tm_epoch.tm_hour	=0;			/* 00:			*/			//    0-  23
		tm_epoch.tm_min		=0;			/* 00:00:		*/			//    0-  59
		tm_epoch.tm_sec		=(int)(fmod(tle.epoch,1.0)*24*60*60);	//    0-  61
		tm_epoch.tm_isdst	=0;
		mktime(&tm_epoch);				/* Normalize it	*/
	}

/* Return */
	return(tm_epoch);

}

/************************************************************************************************/
// This function convert UNIX Time (u_time[sec]&u_msec[msec]) into TLE-Epoch Time (e_time).
// TLE-Epoch Time format is (dbl)YYDDD.DDDDDDDD.
/************************************************************************************************/
extern dbl unix2tleepoch(time_t u_time,dbl u_msec){

struct tm	*p_tm;						/* Pointer to Struct TM									*/
dbl		e_time=0.0;						/* TLE-Epoch Time										*/

/* Convert */
	p_tm=gmtime(&u_time);				/* UNIX Time -> Struct TM								*/

	e_time += (p_tm->tm_year%100)*1000.0;
	e_time += (p_tm->tm_yday)+1;
	e_time += (p_tm->tm_hour)/(24.0          );
	e_time += (p_tm->tm_min )/(24.0*60.0     );
	e_time += (p_tm->tm_sec )/(24.0*60.0*60.0);
	e_time += u_msec/(24.0*60.0*60.0);

/* Return */
	return(e_time);

}

/************************************************************************************************/
// This function calculate difference between TLE-Epoch (epoch) and UNIX time (t_abs).
// It returns difference [ UNIX Time - TLE-Epoch ] in days.
/************************************************************************************************/
extern dbl diff_epoch2utc(dbl epoch,time_t t_abs,time_t t_off){

struct tm	tm_epoch;					/* TLE-Epoch Time [UTC]									*/
dbl		yy;								/* 19YY or 20YY											*/
int		t_diff;							/* [ UNIX Time - TLE-Epoch ] in sec						*/

/* Calculate */
	yy=epoch/1000.0;
	if( 70<=yy )	/* 1970-1999 */
		tm_epoch.tm_year=(int)(epoch/1000.0)+1900-1900;		// YYYY-1900
	else			/* 2000-2069 */
		tm_epoch.tm_year=(int)(epoch/1000.0)+2000-1900;		// YYYY-1900
	tm_epoch.tm_mon		=0;		/* January		*/			//    0-  11
	tm_epoch.tm_mday	=(int)(fmod(epoch,1000.0));			//    1-  31
	tm_epoch.tm_hour	=0;		/* 00:			*/			//    0-  23
	tm_epoch.tm_min		=0;		/* 00:00:		*/			//    0-  59
	tm_epoch.tm_sec		=(int)(fmod(epoch,1.0)*24*60*60);	//    0-  61
	tm_epoch.tm_isdst	=0;
	t_diff=t_abs-(mktime(&tm_epoch)+t_off);

/* Return */
	return(t_diff/(24.0*60.0*60.0));

}

/****************************************************************************/
// This function returns the Julian Day of Year (yyyy) & Day 0.0.
/****************************************************************************/
dbl jd_year(int yyyy){

dbl		jd,jd1970,offs;
int		i;

/* Check Year Range */
	if( yyyy<1970 || 2069<yyyy ){
		fprintf(stderr,"Oops: jd_yyyy() - Unsupported Year Rage [%d]\n",yyyy);
		exit(1);
	}

/* Set jd1970 [Day] */
	jd1970=2440587.5;			/* Julian Day @ 1970-01-01 00:00:00			*/

/* Set offs [Day] */
	offs=0.0;
	for(i=1970;i<yyyy;i++){
		if( ( i%4==0 && i%100!=0 ) || i%400==0 )		/*** Leap Year ***/
			offs+=366.0;
		else
			offs+=365.0;
	}

/* Calculate jd [Day] */
	jd=jd1970+offs-1.0;			/* -1.0 for Day 0.0 ( yyyy-01-00 )			*/

/* Return */
	return(jd);

}

/****************************************************************************/
// This function returns the Julian Day of TLE Epoch Time (epoch).
// TLE Epoch Time (epoch) format is (dbl)YYDDD.DDDDDDDD.
/****************************************************************************/
dbl jd_epoch(dbl epoch){

dbl		year,day;

/* Get Year/Day */
	year=rint(epoch/1000.0);
	day	=epoch-year*1000.0;
	if(70<=year)		/* (19)70-(19)99 */
		year=year+1900;
	else				/* (20)00-(20)69 */
		year=year+2000;

/* Return */
	return(jd_year(year)+day);

}
