/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of Yuichi Watanabe nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <core/assert.h>
#include <core/cpu.h>
#include <core/list.h>
#include <core/initfunc.h>
#include <core/mm.h>
#include <core/panic.h>
#include <core/param.h>
#include <core/printf.h>
#include <core/string.h>
#include <core/vm.h>
#include <core/vm_config.h>
#include "vm.h"

/* #define VMCONFIG_DEBUG */
#ifdef VMCONFIG_DEBUG
#define VMCONFIG_DBG(...)					\
	do {							\
		printf("VMCONFIG: " __VA_ARGS__);		\
	} while (0)
#else
#define VMCONFIG_DBG(...)
#endif

struct vm_config vm0_config;
struct vm_config_list vm_config_list;
apic_id_t shell_apic_id = 0xffffffff;

static void
parse_cpu(struct vm_config *vm_config)
{
	struct cpu_config *cpu_config;
	char param_name[VM_NAME_BUF_SIZE + 4];
	u32 apic_id;
	int i;
	vmmerr_t err;

	ASSERT(sizeof(u32) == sizeof(apic_id_t));

	snprintf(param_name, VM_NAME_BUF_SIZE + 4, "%s.cpu",
		 vm_config->name);
	i = 0;
	while ((err = param_get_u32(param_name, i++, &apic_id))
	       != VMMERR_NOTEXIST) {
		if (err != VMMERR_SUCCESS) {
			printf("Failed to get %s(%d) paramater. err 0x%x\n",
			       param_name, i - 1, err);
			continue;
		}
		if (apic_id == get_bsp_apic_id()) {
			continue;
		}
		cpu_config = alloc(sizeof(struct cpu_config));
		if (cpu_config == NULL) {
			panic("Can't alloc cpu_config");
		}
		memset(cpu_config, 0, sizeof(struct cpu_config));
		cpu_config->apic_id = apic_id;
		LIST2_ADD(vm_config->cpu_list, cpu_list, cpu_config);
		VMCONFIG_DBG("vm %s cpu 0x%x\n",
			     vm_config->name, cpu_config->apic_id);
	}
}

static void
parse_mem(struct vm_config *vm_config)
{
	struct mem_config *mem_config;
	char param_name[VM_NAME_BUF_SIZE + 4];
	phys_t start, end;
	int i;
	vmmerr_t err;

	snprintf(param_name, VM_NAME_BUF_SIZE + 4, "%s.mem",
		 vm_config->name);
	i = 0;
	while ((err = param_get_u64_range(param_name, i++,
					  &start, &end))
	       != VMMERR_NOTEXIST) {
		if (err != VMMERR_SUCCESS) {
			printf("Failed to get %s(%d) paramater. err 0x%x\n",
			       param_name, i - 1,  err);
			continue;
		}
		mem_config = alloc(sizeof(struct mem_config));
		if (mem_config == NULL) {
			panic("Can't alloc mem_config");
		}
		memset(mem_config, 0, sizeof(struct mem_config));
		mem_config->start = start;
		mem_config->end = end;
		LIST2_ADD(vm_config->mem_list, mem_list, mem_config);
		VMCONFIG_DBG("vm %s mem 0x%llx-0x%llx\n",
			     vm_config->name,
			     mem_config->start, mem_config->end);
	}
}

static void
parse_pci(struct vm_config *vm_config)
{
	struct pci_config *pci_config;
	char param_name[VM_NAME_BUF_SIZE + 4];
	u8 bus_no, dev_no, func_no;
	int i;
	vmmerr_t err;

	snprintf(param_name, VM_NAME_BUF_SIZE + 4, "%s.pci",
		 vm_config->name);
	i = 0;
	while ((err = param_get_bdf(param_name, i++,
				    &bus_no, &dev_no, &func_no))
	       != VMMERR_NOTEXIST) {
		if (err != VMMERR_SUCCESS) {
			printf("Failed to get %s(%d) paramater. err 0x%x\n",
			       param_name, i - 1,  err);
			continue;
		}
		pci_config = alloc(sizeof(struct pci_config));
		if (pci_config == NULL) {
			panic("Can't alloc pci_config");
		}
		memset(pci_config, 0, sizeof(struct pci_config));
		pci_config->bus_no = bus_no;
		pci_config->device_no = dev_no;
		pci_config->func_no = func_no;
		LIST2_ADD(vm_config->pci_list, pci_list, pci_config);
		VMCONFIG_DBG("vm %s pci %02x:%02x.%01x\n",
			     vm_config->name,
			     pci_config->bus_no, pci_config->device_no,
			     pci_config->func_no);
	}
}

static void
parse_vm(void)
{
	struct vm_config *vm_config;
	char name[VM_NAME_BUF_SIZE];
	vmmerr_t err;
	int i = 0;

	ASSERT(sizeof(u32) == sizeof(apic_id_t));

	LIST2_HEAD_INIT(vm_config_list.head, vm_config_list);

	snprintf(vm0_config.name, VM_NAME_BUF_SIZE, VM0_NAME);
	LIST2_ADD(vm_config_list.head, vm_config_list, &vm0_config);
	LIST2_HEAD_INIT(vm0_config.cpu_list, cpu_list);
	LIST2_HEAD_INIT(vm0_config.mem_list, mem_list);
	LIST2_HEAD_INIT(vm0_config.pci_list, pci_list);

	while ((err = param_get_str("vm", i++, name, VM_NAME_BUF_SIZE))
	       != VMMERR_NOTEXIST) {
		if (err != VMMERR_SUCCESS) {
			printf("Failed to get vm(%d) paramater. err 0x%x\n",
			       i - 1, err);
			continue;
		}
		if (strcmp(name, VM0_NAME) == 0) {
			continue;
		}
		vm_config = alloc(sizeof(struct vm_config));
		if (vm_config == NULL) {
			panic("Can't alloc vm_config");
		}
		memset(vm_config, 0, sizeof(struct vm_config));
		snprintf(vm_config->name, VM_NAME_BUF_SIZE, name);
		LIST2_HEAD_INIT(vm_config->cpu_list, cpu_list);
		LIST2_HEAD_INIT(vm_config->mem_list, mem_list);
		LIST2_HEAD_INIT(vm_config->pci_list, pci_list);
		LIST2_ADD(vm_config_list.head, vm_config_list, vm_config);
		VMCONFIG_DBG("vm %s\n", vm_config->name);

		parse_cpu(vm_config);
		parse_mem(vm_config);
		parse_pci(vm_config);
	}
}

static void
parse_vm0_boot(void)
{
	vmmerr_t err;
	u8 boot_drive;

	err = param_get_str(VM0_NAME ".boot_int18", 0, NULL, 0);
	if (err == VMMERR_SUCCESS) {
		vm0_config.boot_int18 = true;
		VMCONFIG_DBG("%s\n", VM0_NAME ".boot_int18");
	} else if (err != VMMERR_NOTEXIST) {
		printf("Failed to get %s paramater. err 0x%x\n",
		       VM0_NAME ".boot_int18", err);
	}
	err = param_get_u8(VM0_NAME ".boot_drive", 0, &boot_drive);
	if (err == VMMERR_SUCCESS) {
		vm0_config.boot_drive = boot_drive;
		VMCONFIG_DBG("%s = 0x%x\n", VM0_NAME ".boot_drive",
			     boot_drive);
	} else if (err != VMMERR_NOTEXIST) {
		printf("Failed to get %s paramater. err 0x%x\n",
		       VM0_NAME ".boot_drive", err);
	}

	err = param_get_str(VM0_NAME ".boot_partition", 0, NULL, 0);
	if (err == VMMERR_SUCCESS) {
		vm0_config.boot_partition = true;
		VMCONFIG_DBG("%s\n", VM0_NAME ".boot_partition");
	} else if (err != VMMERR_NOTEXIST) {
		printf("Failed to get %s paramater. err 0x%x\n",
		       VM0_NAME ".boot_int18", err);
	}
}

static void
parse_shell(void)
{
	vmmerr_t err;

	err = param_get_u32("shell", 0, &shell_apic_id);
	if (err == VMMERR_SUCCESS) {
		printf("Failed to get shell paramater. err 0x%x\n", err);
	}
}

static void
vm_config_parse(void)
{
	VMCONFIG_DBG("Parsing parameters.\n");
	parse_vm();
	parse_vm0_boot();
	parse_shell();
	VMCONFIG_DBG("Finished.\n");
}

INITFUNC("global9", vm_config_parse);
