/*
 * Copyright (c) 2014 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of Yuichi Watanabe nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <core.h>
#include <core/byteorder.h>
#include <core/fw_cfg.h>
#include <core/initfunc.h>
#include <core/io.h>
#include <core/printf.h>
#include <core/spinlock.h>
#include <core/string.h>
#include <core/types.h>
#include <core/vm.h>
#include <core/vmmerr.h>
#include <io.h>

/* #define FWCFG_DEBUG */
#include <core/printf.h>
#ifdef FWCFG_DEBUG
#define FWCFG_DBG(...)						\
	do {							\
		printf("FWCFG: " __VA_ARGS__);			\
	} while (0)
#else
#define FWCFG_DBG(...)
#endif

#define FW_CFG_CTRL		0x0510
#define FW_CFG_DATA		0x0511

#define FW_CFG_SIGNATURE	0x0000
#define FW_CFG_FILE_DIR		0x0019
#define FW_CFG_FILE_START	0xc000

#define FW_CFG_FILE_MAX_NUM	5
#define FW_CFG_MAX_NUM		(FW_CFG_FILE_MAX_NUM + 2)

struct cfg_file_info {
	u32	size; /* big endian */
	u16	select; /* big endian */
	u16	reserved;
	char	name[56];
} __attribute__ ((packed));

struct cfg_file_dir {
	u32			count; /* big endian */
	struct cfg_file_info	file[FW_CFG_FILE_MAX_NUM];
} __attribute__ ((packed));

struct fw_cfg {
	u16	select;
	u32	size;
	char	*buffer;
};

struct vm_fw_cfg {
	spinlock_t		lock;
	struct cfg_file_dir	dir;
	struct fw_cfg		cfg[FW_CFG_MAX_NUM];
	u32			cfg_count;
	struct fw_cfg		*cur_fw_cfg;
	u32			cur_address;
};

static drvdata_hdl_t	fw_cfg_handle;

static u64 irq0_override = 1;
static u64 screen_and_debug = 0;
static char system_states[] = {128, 0, 0, 0, 0, 128};

static vmmerr_t
register_fw_cfg(struct vm_fw_cfg *vm_fw_cfg, u16 select, char *buffer,
		u32 size)
{
	struct fw_cfg		*fw_cfg;

	if (vm_fw_cfg->cfg_count >= FW_CFG_MAX_NUM) {
		return VMMERR_NOMEM;
	}
	fw_cfg = vm_fw_cfg->cfg + vm_fw_cfg->cfg_count;
	fw_cfg->select = select;
	fw_cfg->buffer = buffer;
	fw_cfg->size = size;
	vm_fw_cfg->cfg_count++;

	return VMMERR_SUCCESS;
}

vmmerr_t
register_fw_cfg_file(char *name, char *buffer, u32 size)
{
	struct vm_fw_cfg	*vm_fw_cfg;
	struct cfg_file_info	*file_info;
	u16			select;
	u32			count;
	vmmerr_t err;

	if (vm_get_id() == 0) {
		return VMMERR_NOSUP;
	}

	vm_fw_cfg = vm_get_driver_data(fw_cfg_handle);
	spinlock_lock(&vm_fw_cfg->lock);

	count = be32_to_cpu(vm_fw_cfg->dir.count);
	if (count >= FW_CFG_FILE_MAX_NUM) {
		spinlock_unlock(&vm_fw_cfg->lock);
		return VMMERR_NOMEM;
	}
	select = count + FW_CFG_FILE_START;
	err = register_fw_cfg(vm_fw_cfg, select, buffer, size);
	if (err != VMMERR_SUCCESS) {
		spinlock_unlock(&vm_fw_cfg->lock);
		return err;
	}
	file_info = vm_fw_cfg->dir.file + count;
	file_info->size = cpu_to_be32(size);
	file_info->select = cpu_to_be16(select);
	strncpy(file_info->name, name, sizeof(file_info->name));
	vm_fw_cfg->dir.count = cpu_to_be32(count + 1);

	FWCFG_DBG("register file. count %d, select 0x%x, size %d, name %s\n",
		  be32_to_cpu(vm_fw_cfg->dir.count), select, size, name);
	spinlock_unlock(&vm_fw_cfg->lock);
	return VMMERR_SUCCESS;
}

static void
fw_cfg_select(u16 select)
{
	struct vm_fw_cfg	*vm_fw_cfg;
	struct fw_cfg		*fw_cfg;
	int			i;

	vm_fw_cfg = vm_get_driver_data(fw_cfg_handle);
	spinlock_lock(&vm_fw_cfg->lock);

	vm_fw_cfg->cur_fw_cfg = NULL;
	vm_fw_cfg->cur_address = 0;
	for (i = 0; i < vm_fw_cfg->cfg_count; i++) {
		fw_cfg = vm_fw_cfg->cfg + i;
		if (fw_cfg->select == select) {
			vm_fw_cfg->cur_fw_cfg = fw_cfg;
			break;
		}
	}
#ifdef FWCFG_DEBUG
	if (vm_fw_cfg->cur_fw_cfg) {
		FWCFG_DBG("found. select 0x%x\n", select);
	} else {
		FWCFG_DBG("not found. select 0x%x\n", select);
	}
#endif
	spinlock_unlock(&vm_fw_cfg->lock);
}

static ioact_t
fw_cfg_ctrl_handler(iotype_t type, u32 port, void *data)
{

	switch(type) {
	case IOTYPE_OUTW:
		fw_cfg_select(*(u16 *)data);
		break;
	default:
		do_io_nothing(type, port, data);
	}
	return IOACT_CONT;
}

static u8
fw_cfg_read(void)
{
	struct vm_fw_cfg	*vm_fw_cfg;
	struct fw_cfg		*fw_cfg;
	u8			data;

	vm_fw_cfg = vm_get_driver_data(fw_cfg_handle);
	spinlock_lock(&vm_fw_cfg->lock);

	if (vm_fw_cfg->cur_fw_cfg == NULL) {
		spinlock_unlock(&vm_fw_cfg->lock);
		return 0;
	}

	fw_cfg = vm_fw_cfg->cur_fw_cfg;
	if (vm_fw_cfg->cur_address >= fw_cfg->size) {
		spinlock_unlock(&vm_fw_cfg->lock);
		return 0;
	}
	data = fw_cfg->buffer[vm_fw_cfg->cur_address];
	vm_fw_cfg->cur_address++;

	FWCFG_DBG("read 0x%x\n", data);

	spinlock_unlock(&vm_fw_cfg->lock);
	return data;
}

static ioact_t
fw_cfg_data_handler(iotype_t type, u32 port, void *data)
{

	switch(type) {
	case IOTYPE_INB:
		*(u8 *)data = fw_cfg_read();
		break;
	default:
		do_io_nothing(type, port, data);
		break;
	}
	return IOACT_CONT;
}

static void
register_files(void)
{
	vmmerr_t err;

	err = register_fw_cfg_file("etc/irq0-override",
				   (char *)&irq0_override,
				   sizeof(irq0_override));
	if (err != VMMERR_SUCCESS) {
		printf("Failed to register etc/irq0-override. %d\n", err);
	}

	err = register_fw_cfg_file("etc/screen-and-debug",
				   (char *)&screen_and_debug,
				   sizeof(screen_and_debug));
	if (err != VMMERR_SUCCESS) {
		printf("Failed to register etc/screen-and-debug. %d\n", err);
	}

	err = register_fw_cfg_file("etc/system-states",
				   system_states,
				   sizeof(system_states));
	if (err != VMMERR_SUCCESS) {
		printf("Failed to register etc/system_states. %d\n", err);
	}
}

static void
fw_cfg_setup(void)
{
	struct vm_fw_cfg *vm_fw_cfg;
	vmmerr_t err;

	if (vm_get_id() == 0) {
		return;
	}

	vm_fw_cfg = vm_get_driver_data(fw_cfg_handle);
	spinlock_init(&vm_fw_cfg->lock);

	err = register_fw_cfg(vm_fw_cfg,
			FW_CFG_SIGNATURE,
			"VMMV",
			4);
	if (err != VMMERR_SUCCESS) {
		printf("Failed to register fw-cfg signature. %d\n", err);
	}
	err = register_fw_cfg(vm_fw_cfg,
			      FW_CFG_FILE_DIR,
			      (char *)&vm_fw_cfg->dir,
			      sizeof(vm_fw_cfg->dir));
	if (err != VMMERR_SUCCESS) {
		printf("Failed to register fw-cfg directory. %d\n", err);
	}

	register_files();

	set_iofunc(FW_CFG_CTRL, fw_cfg_ctrl_handler);
	set_iofunc(FW_CFG_DATA, fw_cfg_data_handler);
}

static void
fw_cfg_init(void)
{
	fw_cfg_handle = vm_alloc_driver_data(sizeof(struct vm_fw_cfg));
}

FRMWK_VMINIT(fw_cfg_setup);
DRIVER_PREINIT(fw_cfg_init);
