/*
 * Copyright (c) 2011 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.ex.unit.util;

import static jp.terasoluna.fw.ex.unit.common.PropertyKeys.CONTEXT_FILE;
import static jp.terasoluna.fw.ex.unit.common.PropertyKeys.METAINF_DIR;
import static jp.terasoluna.fw.ex.unit.common.PropertyKeys.WEBAPP_PATH;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import javax.sql.DataSource;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathFactory;

import jp.terasoluna.fw.ex.unit.common.DefaultProperties;
import jp.terasoluna.fw.ex.unit.common.PropertyKeys;
import jp.terasoluna.fw.ex.unit.exception.UTRuntimeException;

import org.springframework.jdbc.datasource.DriverManagerDataSource;
import org.springframework.mock.jndi.SimpleNamingContextBuilder;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * JNDIbN[eBeBNXłB
 * 
 */
public class JndiUtils {
    private static final String RESOURCE = "Resource";
    private static final String CONTEXT_RESOURCE_EXP = "//Context/" + RESOURCE;

    /**
     * context.xmlResourcevf{@link NamedNodeMap}̃Xg𐶐܂B
     * 
     * <pre>
     * Resourcevf`Ȃꍇ͋̃Xgԋp܂B
     * </pre>
     * 
     * @param contextPath
     *            context.xml̃pX(NXpX)
     * @return Resourcevf{@link NamedNodeMap}̃Xg
     */
    protected static List<NamedNodeMap> getDataSourceAttributesList(
            String contextPath) {
        try {
            DocumentBuilderFactory domFactory = DocumentBuilderFactory
                    .newInstance();
            domFactory.setNamespaceAware(true);
            DocumentBuilder builder = domFactory.newDocumentBuilder();
            ClassLoader loader = Thread.currentThread().getContextClassLoader();
            InputStream input = loader.getResourceAsStream(contextPath);
            Document doc = null;
            if (input != null) {
                doc = builder.parse(input);
            } else {
                // NXpXɑ݂Ȃꍇ
                doc = builder.parse(contextPath);
            }
            XPathFactory factory = XPathFactory.newInstance();
            XPath xpath = factory.newXPath();
            XPathExpression expr = xpath.compile(CONTEXT_RESOURCE_EXP);
            Node node = (Node) expr.evaluate(doc, XPathConstants.NODE);
            List<NamedNodeMap> result = new ArrayList<NamedNodeMap>();

            while (node != null) {
                if (RESOURCE.equals(node.getNodeName())) {
                    result.add(node.getAttributes());
                }
                node = node.getNextSibling();
            }

            return result;
        } catch (Exception e) {
            throw new UTRuntimeException(e);
        }
    }

    /**
     * Resourcevfnamelɑ΂f[^\[Xԋp܂B
     * 
     * <pre>
     * {NXŃANeBx[gf[^\[X̂ݎ擾ł܂B
     * nameɑ΂f[^\[XANeBx[gĂȂꍇAnullԋp܂B
     * </pre>
     * 
     * @param name
     *            Resourcename
     * @return f[^\[X
     */
    public static DataSource getDataSource(String name) {
        Hashtable<?, ?> table = ReflectionUtils.<Hashtable<?, ?>> getField(
                SimpleNamingContextBuilder.getCurrentContextBuilder(),
                "boundObjects");
        return (DataSource) table.get(name);
    }

    /**
     * <pre>
     * u{@link PropertyKeys#WEBAPP_PATH}/{@link PropertyKeys#METAINF_DIR}/{@link PropertyKeys#CONTEXT_FILE}vǂݍ
     * JNDIANeBx[g܂B
     * </pre>
     */
    public static void activateDefaultNamingContext() {
        String contextPath = String.format("%s/%s/%s",
                DefaultProperties.getValue(WEBAPP_PATH),
                DefaultProperties.getValue(METAINF_DIR),
                DefaultProperties.getValue(CONTEXT_FILE));
        activateNamingContext(contextPath);
    }

    /**
     * @param name
     *            Resource̖O
     * @param driverClassName
     *            JDBChCoNXFQCN
     * @param username
     *            DB[U
     * @param password
     *            DB̃pX[h
     * @param url
     *            DBURL
     */
    public static void activateNamingContext(String name,
            String driverClassName, String username, String password, String url) {
        DriverManagerDataSource ds = new DriverManagerDataSource(url, username,
                password);
        ds.setDriverClassName(driverClassName);
        activateNamingContext(name, ds);
    }

    /**
     * NodeMap̑l擾܂B
     * 
     * <pre>
     * l擾łȂꍇnullԋp܂B
     * </pre>
     * 
     * @param attributes
     *            NodeMap
     * @param name
     *            
     * @return l
     */
    private static String getNodeValue(NamedNodeMap attributes, String name) {
        Node node = attributes.getNamedItem(name);
        if (node == null) {
            return null;
        } else {
            return node.getNodeValue();
        }
    }

    /**
     * context.xmlJNDIANeBx[g܂B
     * 
     * @param contextPath
     *            context.xml̃pX(NXpX)
     */
    public static void activateNamingContext(String contextPath) {
        List<NamedNodeMap> attributesList = getDataSourceAttributesList(contextPath);
        for (NamedNodeMap attributes : attributesList) {
            String name = getNodeValue(attributes, "name");
            String driverClassName = getNodeValue(attributes, "driverClassName");
            String username = getNodeValue(attributes, "username");
            String password = getNodeValue(attributes, "password");
            String url = getNodeValue(attributes, "url");
            activateNamingContext(name, driverClassName, username, password,
                    url);
        }
    }

    /**
     * JNDIANeBx[găf[^\[Xo^܂B
     * 
     * <pre>
     * f[^\[Xjava:comp/env/n܂ꍇ͂̃f[^\[X
     * java:comp/env/菜f[^\[XŃf[^\[Xo^܂B
     * f[^\[Xjava:comp/env/n܂Ȃꍇ͂̃f[^\[X
     * java:comp/env/擪ɕtf[^\[XŃf[^\[Xo^܂B
     * 
     * łJNDIANeBx[gĂꍇ́AJNDIT[rXɒǉo^܂B
     * </pre>
     * 
     * @param name
     *            f[^\[X
     * @param ds
     *            f[^\[X
     */
    public static void activateNamingContext(String name, DataSource ds) {
        try {
            SimpleNamingContextBuilder builder = SimpleNamingContextBuilder
                    .getCurrentContextBuilder();
            if (builder == null) {
                builder = SimpleNamingContextBuilder
                        .emptyActivatedContextBuilder();
            }
            builder.bind(name, ds);
            String prefixForTomcat = "java:comp/env/";

            if (name.startsWith(prefixForTomcat)) {
                // XtomcatpprefixĂꍇ
                builder.bind(name.substring(prefixForTomcat.length()), ds);
            } else {
                // XtomcatpprefixĂȂꍇ
                builder.bind(prefixForTomcat + name, ds);
            }

        } catch (Exception e) {
            throw new UTRuntimeException(e);
        }
    }
}
