/* 
 * Copyright 2007 Tatooine Project <http://tatooine.sourceforge.jp/> 
 *  
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *  
 *     http://www.apache.org/licenses/LICENSE-2.0
 *  
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.tatooine.gtx;

import java.io.PrintStream;
import java.io.Serializable;
import java.util.List;

import jp.sf.tatooine.gtx.node.Node;

/**
 * GtxTemplate.
 *
 * @author  Tooru Noda
 * @version 1.0 2007/09/07
 * @since   JDK5.0 Tiger
 */
public class GtxTemplate implements Serializable {

	private static final long serialVersionUID = 1L;

	/** ルート要素 */
	private List<Node> _topLevelList;
	
	/** コンテキスト変数. */
	private GtxContext _context = null;
	
	/** 空白を無視するか否か. */
	private boolean _ignoreSpace = false;

	/**
	 * 空白を無視するかどうかを設定する.
	 * 
	 * @param ignoreSpace	空白を無視するかどうか
	 */
	protected final void setIgnoreSpace(boolean ignoreSpace) {
		_ignoreSpace = ignoreSpace;
	}
	
	/**
	 * コンテキストを設定する.
	 * 
	 * @param context	コンテキスト
	 */
	protected final void setContext(GtxContext context) {
		_context = context;
	}

	/**
	 * XMLドキュメントのトップレベル要素を設定する.
	 * 
	 * @param topLevelList	トップレベル要素
	 */
	protected void setTopLevelList(List<Node> topLevelList) {
		_topLevelList = topLevelList;
	}
	
	/**
	 * テンプレートを評価する.
	 * 
	 * @param out	PrintStream
	 * @throws GtxSyntaxException	構文エラー
	 */
	public void evaluate(PrintStream out) throws GtxSyntaxException {
		evaluate(out, new GtxContext());
	}
	
	/**
	 * テンプレートを評価する.
	 * 
	 * @param out	PrintStream
	 * @param context	コンテキスト
	 * @throws GtxSyntaxException	構文エラー
	 */
	public void evaluate(PrintStream out, GtxContext context) 
			throws GtxSyntaxException {
		
		if (_context != null) {
			context.setParent(_context);
		}
		StringBuilder builder = new StringBuilder();
		evaluate(builder, context);
		out.println(builder.toString());
	}
	
	/**
	 * テンプレートを評価する.
	 * 
	 * @param builder	結果出力用バッファ
	 * @throws GtxSyntaxException	構文エラー
	 */
	public void evaluate(StringBuilder builder) throws GtxSyntaxException {
		evaluate(builder, new GtxContext());
	}
	
	/**
	 * テンプレートを評価する.
	 * 
	 * @param builder	結果出力用バッファ
	 * @param context	コンテキスト
	 * @throws GtxSyntaxException	構文エラー
	 */
	public void evaluate(StringBuilder builder, GtxContext context) 
			throws GtxSyntaxException {
		
		if (_context != null) {
			context.setParent(_context);
		}
		for (Node node : _topLevelList) {
			node.evaluate(builder, context);
			if (!_ignoreSpace) {
				builder.append("\n");
			}
		}
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		for (Node node : _topLevelList) {
			builder.append(node.toString()).append("\n");
		}
		return builder.toString();
	}
	
}
