/* 
 * Copyright 2007 Tatooine Project <http://tatooine.sourceforge.jp/> 
 *  
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *  
 *     http://www.apache.org/licenses/LICENSE-2.0
 *  
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.tatooine.gtx;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

/**
 * GtxContext.
 * <p>
 * コンテキストパラメータを保持する.
 * 
 * @author  Tooru Noda
 * @version 1.0 2007/04/11
 * @since   JDK5.0 Tiger
 */
public class GtxContext extends HashMap<String, Object> {
	
	/** serialVersionUID. */
	private static final long serialVersionUID = 1L;
	
	/** 上位コンテキスト. */
	private GtxContext _parent = null;
	
	/** 属性の動的変更用. */
	private Map<String, Tag> _tagMap = new HashMap<String, Tag>();
	
	/**
	 * 上位階層のコンテキストを設定する.
	 * 
	 * @param parent	上位階層のコンテキスト
	 */
	public final void setParent(GtxContext parent) {
		_parent = parent;
	}

	/**
	 * 設定されたコンテキストパラメータを検索する.
	 * 
	 * @param 	key	コンテキストキー
	 * @return	コンテキストパラメータ
	 */
	protected final Object search(String key) {
		if (containsKey(key)) {
			return get(key);
		}
		if (_parent != null) {
			return _parent.search(key);
		}
		return null;
	}
	
	/**
	 * 階層で管理された<code>Map</code>を平坦な１つの<code>Map</code>に変換する.
	 * 
	 * @return	変換後の<code>Map</code>
	 */
	public final Map<String, Object> flatten() {
		Map<String, Object> map;
		if (_parent != null) {
			map = _parent.flatten();
		}
		else {
			map = new HashMap<String, Object>();
		}
		map.putAll(this);
		
		return map;
	}
	
	/**
	 * id属性値によって{@link Tag}オブジェクトとテンプレート上のidタグを関連付ける.
	 * 
	 * @param id	id属性値
	 * @param tag	タグ
	 */
	public final void bindById(String id, Tag tag) {
		if (StringUtils.isEmpty(id)) {
			throw new IllegalArgumentException("idが正しく指定されていません。");
		}
		_tagMap.put(id, tag);
	}
	
	/**
	 * 指定されたidを持つ{@link Tag}オブジェクトを取得する.
	 * 
	 * @param id	id属性値
	 * @return	タグオブジェクト
	 */
	public final Tag getTag(String id) {
		if (_tagMap.containsKey(id)) {
			return _tagMap.get(id);
		}
		if (_parent != null) {
			return _parent.getTag(id);
		}
		return null;
	}
	
}
