package jp.sourceforge.talisman.hermes.ui.swing;

/*
 * $Id: HermesContextBuilder.java 198 2009-05-30 14:45:26Z tama3 $
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;

import javax.swing.Action;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;

import jp.sourceforge.talisman.hermes.HermesContext;
import jp.sourceforge.talisman.hermes.HermesContextExporter;
import jp.sourceforge.talisman.hermes.HermesContextParser;
import jp.sourceforge.talisman.hermes.InvalidHermesConfigException;
import jp.sourceforge.talisman.hermes.RepositoryScope;
import jp.sourceforge.talisman.hermes.UpdatingLibraryCheckPolicy;
import jp.sourceforge.talisman.hermes.maven.Artifact;
import jp.sourceforge.talisman.hermes.maven.DependencyScope;
import jp.sourceforge.talisman.hermes.maven.Repository;

/**
 * GUI interface for building hermes.xml.
 * 
 * @author Haruaki Tamada
 * @version $Revision: 198 $
 */
public class HermesContextBuilder extends JPanel{
    private static final long serialVersionUID = -1139978069104398535L;

    private HermesContext context;

    private JComboBox repositoryScopeCombo;
    private JComboBox updatingLibraryCheckPolicyCombo;
    private JCheckBox[] checks = new JCheckBox[DependencyScope.values().length];
    private JTextField destination = new JTextField();
    private DefaultListModel repositoryListModel = new DefaultListModel();
    private JList repositoryList = new JList(repositoryListModel);
    private DefaultTableModel dependencyTableModel = new NotEditableDefaultTableModel(new String[] { "Group Id", "Artifact Id", "Version", "Scope"}, 0);
    private JTable dependencyTable = new JTable(dependencyTableModel);
    private DefaultTableModel ignoreTableModel = new NotEditableDefaultTableModel(new String[] { "Group Id", "Artifact Id", }, 0);
    private JTable ignoreTable = new JTable(ignoreTableModel);
    private JTextArea contextStringArea;
    private CurrentDirectoryHandler handler;
    private transient PropertyChangeListener listener;

    public HermesContextBuilder(String[] args){
        super(new BorderLayout());
        context = findContext(args);
        handler = new CurrentDirectoryHandler();
        handler.addFileFilter(new FileFilter(){
            @Override
            public boolean accept(File f){
                return f.isDirectory() || f.getName().endsWith(".xml");
            }

            @Override
            public String getDescription(){
                return "Xml File (*.xml)";
            }
        });

        initListener();
        initLayout();
        updateFields(context);
        setPreferredSize(new Dimension(800, 500));
        setSize(getPreferredSize());
    }

    /**
     * set context.
     */
    public void setContext(HermesContext context){
        this.context.removePropertyChangeListener(listener);
        this.context = context;
        context.addPropertyChangeListener(listener);
        updateFields(context);
    }

    /**
     * returns context.
     */
    public HermesContext getContext(){
        return context;
    }

    void updateFields(HermesContext context){
        this.context = context;
        repositoryScopeCombo.setSelectedItem(context.getRepositoryScope());

        updatingLibraryCheckPolicyCombo.setSelectedItem(context.getPolicy());

        destination.setText(context.getDestination());

        repositoryListModel.clear();
        for(Repository repo: context.getRepositories()){
            repositoryListModel.addElement(repo.getName());
        }

        dependencyTableModel.setRowCount(0);
        for(Artifact artifact: context.getDependencies()){
            dependencyTableModel.addRow(new Object[] { 
                artifact.getGroupId(), artifact.getArtifactId(), artifact.getVersion(), artifact.getScope(), 
            });
        }

        ignoreTableModel.setRowCount(0);
        for(Artifact artifact: context.getIgnores()){
            ignoreTableModel.addRow(new Object[] {
                artifact.getGroupId(), artifact.getArtifactId()
            });
        }
    }

    private HermesContext findContext(String[] args){
        HermesContext context = null;
        if(args.length > 0 && args[0].endsWith(".xml")){
            try{
                context = new HermesContextParser().parse(new File(args[0]));
            } catch(IOException e){
                JOptionPane.showMessageDialog(this, e.getLocalizedMessage());
            } catch(InvalidHermesConfigException e){
                JOptionPane.showMessageDialog(this, e.getLocalizedMessage());
            }
        }
        if(context == null){
            context = new HermesContext();
        }
        return context;
    }

    private void initListener(){
        listener = new PropertyChangeListener(){
            public void propertyChange(PropertyChangeEvent evt){
                updateFields((HermesContext)evt.getSource());
            }
        };
        context.addPropertyChangeListener(listener);
    }

    private void initLayout(){
        JTabbedPane tab = new JTabbedPane();
        JPanel south = new JPanel(new FlowLayout(FlowLayout.CENTER));
        south.add(new JButton(new LoadAction(this, handler)));
        south.add(new JButton(new SaveAction(this, handler)));

        tab.addTab("default view", createDefaultView());
        tab.addTab("xml view", createXmlView());
        tab.addChangeListener(new ChangeListener(){
            public void stateChanged(ChangeEvent e){
                tabChanged((JTabbedPane)e.getSource());
            }
        });

        add(tab, BorderLayout.CENTER);
        add(south, BorderLayout.SOUTH);
    }

    private void tabChanged(JTabbedPane tab){
        String string = contextStringArea.getText().trim();
        if(tab.getSelectedIndex() == 1){
            try{
                StringWriter out = new StringWriter();
                new HermesContextExporter().export(new PrintWriter(out), context);
                contextStringArea.setText(out.toString());
                out.close();
            } catch(IOException e){
                throw new InternalError(e.getMessage());
            }
        }
        else if(string.length() > 0){
            try{
                ByteArrayInputStream in = new ByteArrayInputStream(string.getBytes());
                HermesContextParser parser = new HermesContextParser();
                HermesContext context = parser.parse(in);
                if(parser.hasExceptions()){
                    StringBuilder sb = new StringBuilder();
                    String ln = System.getProperty("line.separator");
                    for(Exception e: parser){
                        sb.append(e.getLocalizedMessage()).append(ln);
                    }
                    JOptionPane.showMessageDialog(this, new String(sb));
                }
                setContext(context);
            } catch(IOException e){
                JOptionPane.showMessageDialog(this, e.getLocalizedMessage());
            } catch(InvalidHermesConfigException e){
                JOptionPane.showMessageDialog(this, e.getLocalizedMessage());
            }
        }
    }

    private Component createDefaultView(){
        JPanel panel = new JPanel(new GridBagLayout());
        repositoryScopeCombo = new JComboBox();
        for(RepositoryScope scope: RepositoryScope.values()){
            repositoryScopeCombo.addItem(scope);
        }
        repositoryScopeCombo.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                getContext().setRepositoryScope((RepositoryScope)repositoryScopeCombo.getSelectedItem());
            }
        });
        repositoryScopeCombo.setBorder(new TitledBorder("Repository Scope"));

        updatingLibraryCheckPolicyCombo = new JComboBox();
        for(UpdatingLibraryCheckPolicy scope: UpdatingLibraryCheckPolicy.values()){
            updatingLibraryCheckPolicyCombo.addItem(scope);
        }
        updatingLibraryCheckPolicyCombo.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                getContext().setPolicy((UpdatingLibraryCheckPolicy)updatingLibraryCheckPolicyCombo.getSelectedItem());
            }
        });
        updatingLibraryCheckPolicyCombo.setBorder(new TitledBorder("Updating Libracy Check Policy"));

        DependencyScope[] scopes = DependencyScope.values();
        JPanel dependencyScopePanel = new JPanel(new GridLayout(1, checks.length));
        for(int i = 0; i < scopes.length; i++){
            checks[i] = new JCheckBox(new DependencyScopeUpdatingAction(this, scopes[i]));
            checks[i].setSelected(getContext().isInclude(scopes[i]));
            dependencyScopePanel.add(checks[i]);
        }
        dependencyScopePanel.setBorder(new TitledBorder("Dependency Scope"));

        JPanel destinationPanel = new JPanel(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.weightx = 1d;
        gbc1.fill = GridBagConstraints.HORIZONTAL;
        destinationPanel.add(destination, gbc1);
        gbc1.weightx = 0d;
        gbc1.fill = GridBagConstraints.NONE;
        gbc1.gridx = 1;
        destinationPanel.add(new JButton(new SetDestinationAction(this, handler)));
        destination.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                String value = destination.getText();
                getContext().setDestination(value);
            }
        });
        destinationPanel.setBorder(new TitledBorder("Destination"));

        Action editRepositoryAction = new EditRepositoryAction(this, repositoryList.getSelectionModel());
        JPanel repositoryListPanel = new JPanel(new BorderLayout());
        JPanel repositoryListButtonPanel = new JPanel(new GridLayout(1, 3));
        repositoryListButtonPanel.add(new JButton(new AddRepositoryAction(this)));
        repositoryListButtonPanel.add(new JButton(editRepositoryAction));
        repositoryListButtonPanel.add(new JButton(new RemoveRepositoryAction(this, repositoryList.getSelectionModel())));
        repositoryList.addMouseListener(new ActionDelegateMouseListener(editRepositoryAction));

        repositoryListPanel.add(new JScrollPane(repositoryList), BorderLayout.CENTER);
        repositoryListPanel.add(repositoryListButtonPanel, BorderLayout.SOUTH);
        repositoryListPanel.setBorder(new TitledBorder("Repository List"));

        Action editDependencyAction = new EditDependencyAction(this, dependencyTable.getSelectionModel());
        JPanel dependencyListPanel = new JPanel(new BorderLayout());
        JPanel dependencyTableButtonPanel = new JPanel(new GridLayout(1, 3));
        dependencyTableButtonPanel.add(new JButton(new AddDependencyAction(this)));
        dependencyTableButtonPanel.add(new JButton(editDependencyAction));
        dependencyTableButtonPanel.add(new JButton(new RemoveDependencyAction(this, dependencyTable.getSelectionModel())));
        dependencyTable.setRowSorter(new TableRowSorter<TableModel>(dependencyTableModel));
        dependencyTable.addMouseListener(new ActionDelegateMouseListener(editDependencyAction));

        dependencyListPanel.add(new JScrollPane(dependencyTable), BorderLayout.CENTER);
        dependencyListPanel.add(dependencyTableButtonPanel, BorderLayout.SOUTH);
        dependencyListPanel.setBorder(new TitledBorder("Dependency List"));

        Action editIgnoreAction = new EditDependencyAction(this, ignoreTable.getSelectionModel(), false);
        JPanel ignoreListPanel = new JPanel(new BorderLayout());
        JPanel ignoreTableButtonPanel = new JPanel(new GridLayout(1, 3));
        ignoreTableButtonPanel.add(new JButton(new AddDependencyAction(this, false)));
        ignoreTableButtonPanel.add(new JButton(editIgnoreAction));
        ignoreTableButtonPanel.add(new JButton(new RemoveDependencyAction(this, ignoreTable.getSelectionModel(), false)));
        ignoreTable.setRowSorter(new TableRowSorter<TableModel>(ignoreTableModel));
        ignoreTable.addMouseListener(new ActionDelegateMouseListener(editIgnoreAction));

        ignoreListPanel.add(new JScrollPane(ignoreTable), BorderLayout.CENTER);
        ignoreListPanel.add(ignoreTableButtonPanel, BorderLayout.SOUTH);
        ignoreListPanel.setBorder(new TitledBorder("Ignore List"));

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridheight = 1;
        gbc.gridwidth = 2;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1d;

        panel.add(repositoryScopeCombo, gbc);
        gbc.gridy = 1;
        panel.add(updatingLibraryCheckPolicyCombo, gbc);
        gbc.gridy = 2;
        panel.add(dependencyScopePanel, gbc);
        gbc.gridy = 3;
        panel.add(destinationPanel, gbc);

        gbc.gridheight = 2;
        gbc.gridwidth = 1;
        gbc.gridx = 0;
        gbc.gridy = 4;
        gbc.weightx = 0d;
        gbc.weighty = 1d;
        gbc.fill = GridBagConstraints.BOTH;
        panel.add(repositoryListPanel, gbc);

        gbc.gridx = 1;
        gbc.gridy = 4;
        gbc.gridheight = 1;
        gbc.weightx = 1d;
        gbc.weighty = 0.5d;
        panel.add(dependencyListPanel, gbc);

        gbc.gridy = 5;
        panel.add(ignoreListPanel, gbc);

        return panel;
    }

    private Component createXmlView(){
        contextStringArea = new JTextArea();

        return new JScrollPane(contextStringArea);
    }

    public static void main(String[] args){
        JFrame f = new JFrame("Hermes Context Builder");
        f.add(new HermesContextBuilder(args), BorderLayout.CENTER);
        f.pack();
        f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        f.setVisible(true);
    }
}
