package jp.sourceforge.talisman.fmv.ui.swing;

/*
 * $Id: MemoryGraphPane.java 148 2008-07-15 07:34:55Z tama3 $
 */

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.geom.GeneralPath;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import javax.swing.border.TitledBorder;
import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;

import jp.sourceforge.talisman.fmv.MemoryState;
import jp.sourceforge.talisman.fmv.MemoryStateNotifyListener;
import jp.sourceforge.talisman.fmv.RunningListener;
import jp.sourceforge.talisman.fmv.TimeredMemoryStateNotifier;
import jp.sourceforge.talisman.fmv.ui.FmvGraphSetting;
import jp.sourceforge.talisman.fmv.ui.PopupShowAction;
import jp.sourceforge.talisman.i18n.MessageManager;
import jp.sourceforge.talisman.i18n.Messages;
import jp.sourceforge.talisman.i18n.ResourceNotFoundException;

/**
 * Memory transition graph in Swing.
 * 
 * @author Haruaki Tamada
 * @version $Revision: 148 $ 
 */
public class MemoryGraphPane extends JPanel implements MessageManager{
    private static final long serialVersionUID = 6262828285348015917L;

    private TimeredMemoryStateNotifier notifier;
    private FmvGraphSetting settings;
    private ViewerPanel viewerPanel;
    private Box buttonPanel;
    private JPanel detailPanel;
    private JLabel totalMemoryLabel;
    private JLabel freeMemoryLabel;
    private JLabel usageLabel;
    private Messages messages;
    private List<MemoryState> memoryList = new ArrayList<MemoryState>();
    private long max = 0L;

    public MemoryGraphPane() throws ResourceNotFoundException{
        this(new FmvGraphSetting());
    }

    public MemoryGraphPane(Messages messages){
        this(new FmvGraphSetting(), messages);
    }

    public MemoryGraphPane(FmvGraphSetting settings) throws ResourceNotFoundException{
        this(settings, new Messages("resources.fmv"));
    }

    public MemoryGraphPane(FmvGraphSetting settings, Messages messages){
        this.settings = settings;
        this.messages = messages;
        notifier = new TimeredMemoryStateNotifier();
        notifier.addNotifyListener(new MemoryStateNotifyListener(){
            public void currentMemoryStateNotified(MemoryState state){
                updateView(state);
            }
        });

        initLayouts();
        setGraphSettings(settings);
        addAncestorListener(new AncestorListener(){
            public void ancestorAdded(AncestorEvent event){
                notifier.start();
            }

            public void ancestorMoved(AncestorEvent event){
            }

            public void ancestorRemoved(AncestorEvent event){
                notifier.stop();
            }
        });
    }

    public Messages getMessages(){
        return messages;
    }

    public void addMemoryStateNotifyListener(MemoryStateNotifyListener listener){
        notifier.addNotifyListener(listener);
    }

    public void removeMemoryStateNotifyListener(MemoryStateNotifyListener listener){
        notifier.removeNotifyListener(listener);
    }

    public FmvGraphSetting getGraphSettings(){
        return settings;
    }

    public void setGraphSettings(FmvGraphSetting settings){
        this.settings = settings;
        if(settings.isShowConfigurationButton()){
            add(buttonPanel, BorderLayout.SOUTH);
        }
        else{
            remove(buttonPanel);
        }
        if(settings.isShowDetails()){
            add(detailPanel, BorderLayout.NORTH);
        }
        else{
            remove(detailPanel);
        }
        notifier.setInterval(settings.getInterval());

        viewerPanel.setMinimumSize(settings.getMinimumSize());
        viewerPanel.setPreferredSize(settings.getMinimumSize());
        Window window = SwingUtilities.windowForComponent(this);
        if(window != null){
            window.pack();
        }
    }

    private void initLayouts(){
        viewerPanel = new ViewerPanel();
        buttonPanel = Box.createHorizontalBox();
        detailPanel = new JPanel(new GridLayout(1, 3));
        detailPanel.add(freeMemoryLabel = new JLabel());
        detailPanel.add(totalMemoryLabel = new JLabel());
        detailPanel.add(usageLabel = new JLabel());

        setLayout(new BorderLayout());
        add(viewerPanel, BorderLayout.CENTER);

        if(settings.isShowConfigurationButton()){
            add(buttonPanel, BorderLayout.SOUTH);
        }
        if(settings.isShowDetails()){
            add(detailPanel, BorderLayout.NORTH);
        }
        freeMemoryLabel.setBorder(new TitledBorder(messages.get("freememory.label")));
        totalMemoryLabel.setBorder(new TitledBorder(messages.get("totalmemory.label")));
        usageLabel.setBorder(new TitledBorder(messages.get("useratio.label")));

        Action configAction = new ConfigAction(this, this);
        Action gcAction = new GcAction(this, this);
        Action startStopAction = new StartStopAction(notifier, this);

        buttonPanel.add(Box.createHorizontalGlue());
        buttonPanel.add(new JButton(configAction));
        buttonPanel.add(Box.createHorizontalGlue());
        buttonPanel.add(new JButton(gcAction));
        buttonPanel.add(Box.createHorizontalGlue());
        buttonPanel.add(new JButton(startStopAction));
        buttonPanel.add(Box.createHorizontalGlue());

        JPopupMenu popup = new JPopupMenu();
        popup.add(new JMenuItem(configAction));
        popup.add(new JMenuItem(gcAction));
        popup.add(new JMenuItem(startStopAction));
        viewerPanel.addMouseListener(new PopupShowAction(popup));
    }

    private void updateView(MemoryState state){
        memoryList.add(state);
        System.out.println(state);
        while(memoryList.size() > settings.getHistoryCount()){
            memoryList.remove(0);
        }
        if(max < state.getTotalMemory()){
            max = state.getTotalMemory();
        }

        freeMemoryLabel.setText(Long.toString(state.getFreeMemory()));
        totalMemoryLabel.setText(Long.toString(state.getTotalMemory()));
        usageLabel.setText(String.format("%g %%", state.getUsedMemoryRatio() * 100));
        viewerPanel.repaint();
    }

    private class ViewerPanel extends JComponent{
        private static final long serialVersionUID = 4434644949345620306L;

        public void paintComponent(Graphics graphics){
            super.paintComponent(graphics);
            Graphics2D g = (Graphics2D)graphics;
            Dimension dim = getSize();

            Rectangle2D rect = new Rectangle2D.Double(0d, 0d, dim.getWidth(), dim.getHeight());
            g.setColor(settings.getBackgroundColor());
            g.fill(rect);
            if(settings.isShowSubGrid()){
                g.setColor(settings.getSubGridColor());
                drawGrid(g, dim, 16);
            }
            if(settings.isShowGrid()){
                g.setColor(settings.getGridColor());
                drawGrid(g, dim, 8);
            }
            g.setColor(settings.getStringColor());
            if(settings.isShowGraphLabel()){
                drawGraphLabel(g, dim);
            }
            g.setColor(settings.getForegroundColor());

            if(memoryList.size() > 0){
                synchronized(memoryList){
                    float dw = (float)(dim.getWidth() / settings.getStepCount());
                    GeneralPath path = new GeneralPath();
                    int index = memoryList.size() - settings.getStepCount();
                    if(index < 0){
                        index = 0;
                    }
                    MemoryState ms = memoryList.get(index);

                    if(settings.isFill()){
                        path.moveTo(0, (float)dim.getHeight());
                        path.lineTo(0f, (float)((dim.getHeight() / max) * ms.getFreeMemory()));
                    }
                    else{
                        path.moveTo(0f, (float)((dim.getHeight() / max) * ms.getFreeMemory()));
                    }
                    for(int i = 1; i < settings.getStepCount(); i++){
                        if((index + i) >= memoryList.size()){
                            break;
                        }
                        ms = memoryList.get(index + i);
                        path.lineTo(0f + i * dw, (float)((dim.getHeight() / max) * ms.getFreeMemory()));
                    }
                    if(settings.isFill()){
                        path.lineTo(0f + dw * (memoryList.size() - 1), (float)dim.getHeight());
                        g.fill(path);
                    }
                    else{
                        g.draw(path);
                    }
                }
            }
        }

        private void drawGraphLabel(Graphics2D g, Dimension d){
            double dw = ((d.getWidth() - 1) / 8d);
            double dh = ((d.getHeight() - 1) / 8d);
            Font font = getFont();
            FontMetrics metircs = g.getFontMetrics(font);

            g.drawString("  0%", (float)(dw * 4), (float)(dh * 8));
            g.drawString(" 25%", (float)(dw * 4), (float)(dh * 6));
            g.drawString(" 50%", (float)(dw * 4), (float)(dh * 4));
            g.drawString(" 75%", (float)(dw * 4), (float)(dh * 2));
            g.drawString("100%", (float)(dw * 4), (float)(dh * 0) + metircs.getHeight());
        }

        private void drawGrid(Graphics2D g, Dimension d, int val){
            double dw = (d.getWidth() - 1) / val;
            double dh = (d.getHeight() - 1) / val;
            for(int i = 0; i <= val; i++){
                g.draw(new Line2D.Double(0, dh * i, d.getWidth(), dh * i));
                g.draw(new Line2D.Double(dw * i, 0, dw * i, d.getHeight()));
            }
        }
    }

    private abstract static class FmvAction extends AbstractAction implements MessageManager{
        private MessageManager mm;

        public FmvAction(MessageManager mm, String label){
            super(
                mm.getMessages().get(label + ".label"),
                mm.getMessages().getIcon(label + ".icon")
            );
            this.mm = mm;
        }

        public Messages getMessages(){
            return mm.getMessages();
        }
    }

    private static class StartStopAction extends FmvAction{
        private static final long serialVersionUID = -3567505632132908195L;

        private TimeredMemoryStateNotifier notifier;

        public StartStopAction(TimeredMemoryStateNotifier notifier, MessageManager mm){
            super(mm, "startstop");
            this.notifier = notifier;
            notifier.addRunningListener(new RunningListener(){
                public void timerStarted(){
                    putValue(NAME, getMessages().get("stop.label"));
                    putValue(SMALL_ICON, getMessages().getIcon("stop.icon"));
                }

                public void timerStopped(){
                    putValue(NAME, getMessages().get("start.label"));
                    putValue(SMALL_ICON, getMessages().getIcon("start.icon"));
                }
            });
        }

        public void actionPerformed(ActionEvent e){
            if(notifier.isRunning()){
                notifier.stop();
            }
            else{
                notifier.start();
            }
        }
    }

    private static class ConfigAction extends FmvAction{
        private static final long serialVersionUID = -844806456041681421L;

        private MemoryGraphPane memoryGraphPane;

        public ConfigAction(MemoryGraphPane memoryGraphPane, MessageManager mm){
            super(mm, "config");
            this.memoryGraphPane = memoryGraphPane;
        }

        public FmvGraphSetting getSetting(FmvGraphSetting fgs){
            FmvGraphSettingPane panel = new FmvGraphSettingPane(this, fgs);

            int confirmType = JOptionPane.showConfirmDialog(
                memoryGraphPane, panel, getMessages().get("config.title"),
                JOptionPane.OK_CANCEL_OPTION, JOptionPane.INFORMATION_MESSAGE
            );
            FmvGraphSetting newFgs = null;
            if(confirmType == JOptionPane.OK_OPTION){
                newFgs = panel.getGraphSetting();
            }

            return newFgs;
        }

        public void actionPerformed(ActionEvent e){
            boolean running = memoryGraphPane.notifier.isRunning();
            memoryGraphPane.notifier.stop();
            FmvGraphSetting fgs = getSetting(memoryGraphPane.getGraphSettings());
            if(fgs != null){
                memoryGraphPane.setGraphSettings(fgs);
            }
            if(running){
                memoryGraphPane.notifier.start();
            }
        }
    }

    private static class GcAction extends FmvAction{
        private static final long serialVersionUID = 6363441797189312138L;

        private MemoryGraphPane memoryGraphPane;

        public GcAction(MemoryGraphPane memoryGraphPane, MessageManager mm){
            super(mm, "gc");
            this.memoryGraphPane = memoryGraphPane;
        }

        public void actionPerformed(ActionEvent e){
            
            int type = JOptionPane.showConfirmDialog(
                memoryGraphPane,
                memoryGraphPane.getMessages().get("message.exec-gc"),
                memoryGraphPane.getMessages().get("title.exec-gc"),
                JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.QUESTION_MESSAGE
            );
            if(type == JOptionPane.OK_OPTION){
                System.gc();
            }
        }
    }

    public static void main(String[] args) throws ResourceNotFoundException{
        JFrame f = new JFrame();
        f.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
        f.add(new MemoryGraphPane());
        f.pack();
        f.setVisible(true);
    }
}
