package jp.sourceforge.talisman.i18n;

/*
 * $Id: MessagesWithDefaultLocaleTest.java 140 2008-07-15 04:01:32Z tama3 $
 */

import java.util.Locale;

import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;


/**
 * Test class of Messages.
 * 
 * @author Haruaki Tamada
 * @version $Revision: 140 $
 */
public class MessagesWithDefaultLocaleTest{
    private Messages messages;

    /**
     * set default locale as Japan.
     */
    @BeforeClass
    public static void initializeDefaultLocale(){
        Locale.setDefault(Locale.JAPAN);
    }

    /**
     * build instance of Messages class.
     */
    @Before
    public void buildMessages(){
        try{
            messages = new Messages("resources.test");
        } catch(ResourceNotFoundException e){
        }
    }

    /**
     * ResourceNotFoundException will be thrown because resource file is not exist.
     */
    @Test(expected=ResourceNotFoundException.class)
    public void loadFail() throws Exception{
        new Messages("test");
    }

    /**
     * NullPointerException will be thrown.
     */
    @Test(expected=NullPointerException.class)
    public void throwNullPointerException() throws Exception{
        messages.setLocale(null);
    }

    @Test
    public void escapeTest(){
        Assert.assertEquals("$${notescape}", messages.get("test.escape1"));
        Assert.assertEquals("$notescape", messages.get("test.escape2"));
        Assert.assertEquals("${notescape", messages.get("test.escape3"));
        Assert.assertEquals("$notescape}", messages.get("test.escape4"));
    }

    @Test
    public void nullTest() throws Exception{
        Assert.assertFalse(messages.hasValue("missing.key"));
        Assert.assertNull(messages.get("missing.key"));
        Assert.assertNull(messages.format("missing.key", "arg1", "arg2"));

        Assert.assertTrue(messages.hasValue("test1"));
        Assert.assertEquals("default value", messages.get("missing.key", "default value"));
        Assert.assertEquals(0, messages.getArray("missing.key").length);
    }

    @Test
    public void testIcon() throws Exception{
        Assert.assertNotNull(messages.getIcon("test.icon1"));
        Assert.assertNull(messages.getIcon("test.icon2"));
        Assert.assertNull(messages.getIcon("test.icon3"));
    }

    @Test
    public void arrayTest3() throws Exception{
        String[] array = messages.getArray("test.array1");
        Assert.assertNotNull(array);
        Assert.assertEquals(3, array.length);
        Assert.assertEquals("あ", array[0]);
        Assert.assertEquals("い", array[1]);
        Assert.assertEquals("う", array[2]);
    }

    @Test
    public void arrayTest4() throws Exception{
        String[] array = messages.getArray("test.array2");
        Assert.assertNotNull(array);
        Assert.assertEquals(3, array.length);
        Assert.assertEquals("あ", array[0]);
        Assert.assertEquals("い", array[1]);
        Assert.assertEquals("う", array[2]);
    }

    @Test
    public void testJapanese(){
        Assert.assertEquals("テスト1", messages.get("test1"));
        Assert.assertEquals("テスト2", messages.get("test2"));
        Assert.assertEquals("テスト3", messages.get("test3"));
        Assert.assertEquals("test4", messages.get("test4"));

        Assert.assertEquals("日本語 foo bar",
                            messages.format("test.template", "foo", "bar"));
        Assert.assertEquals("default テスト1 テスト2",
                            messages.get("test.replace"));
    }

    @Test
    public void testEnglish(){
        messages.setLocale(Locale.ENGLISH);
        Assert.assertEquals("test1", messages.get("test1"));
        Assert.assertEquals("test2", messages.get("test2"));
        Assert.assertEquals("test3", messages.get("test3"));
        Assert.assertEquals("test4", messages.get("test4"));

        Assert.assertEquals("english foo bar",
                            messages.format("test.template", "foo", "bar"));
        Assert.assertEquals("default test1 test2",
                            messages.get("test.replace"));
    }

    /**
     * resource.test_de resource is exists, however, no entries are defined.
     * Threfore, resource.test resources is selected.
     */
    @Test
    public void testGerman(){
        messages.setLocale(Locale.GERMAN);
        Assert.assertEquals("test1 (default locale)", messages.get("test1"));
        Assert.assertEquals("test2 (default locale)", messages.get("test2"));
        Assert.assertEquals("test3 (default locale)", messages.get("test3"));
        Assert.assertEquals("test4", messages.get("test4"));

        Assert.assertEquals("default foo bar",
                            messages.format("test.template", "foo", "bar"));
        Assert.assertEquals("default test1 (default locale) test2 (default locale)",
                            messages.get("test.replace"));
    }

    /**
     * resource.test_fr resource is not exists.
     * Threfore, default locale (Japan) is selected.
     */
    @Test
    public void testFrench(){
        messages.setLocale(Locale.FRENCH);
        Assert.assertEquals("テスト1", messages.get("test1"));
        Assert.assertEquals("テスト2", messages.get("test2"));
        Assert.assertEquals("テスト3", messages.get("test3"));
        Assert.assertEquals("test4", messages.get("test4"));

        Assert.assertEquals("日本語 foo bar",
                            messages.format("test.template", "foo", "bar"));
        Assert.assertEquals("default テスト1 テスト2",
                            messages.get("test.replace"));
    }
}
