/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2003 The T-Struts Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE T-STRUTS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE T-STRUTS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the T-Struts Project.
 */
package jp.ossc.tstruts.config;

import java.lang.reflect.Array;

import org.apache.commons.beanutils.ConvertUtils;

/**
 * UserProfilePropertypݒNXB<p>
 * 
 * @version $Name: release-1_1-1_0 $
 * @author N.Kurosawa
 * @since 1.0
 */
public class UserProfilePropertyConfig {

	private String name = null;
	private String type = null;
	private boolean indexed = false;
	private int size = 0;
	private Class typeClass = null;
	private Class indexedComponentClass = null;

	private String defaultValueString = null;
	private String defaultValueKey = null;
	private boolean configured = false;

	public UserProfilePropertyConfig() {
	}
	
	public UserProfilePropertyConfig(
		String name, String type, int size)
	throws ClassNotFoundException {
		setName(name);
		setType(type);
		setSize(size);
	}

	public UserProfilePropertyConfig(
		String name, String type, int size, String defaultString)
	throws ClassNotFoundException {
		this(name, type, size);
		setDefault(defaultString);
	}

	public UserProfilePropertyConfig(
		String name, String type, int size, String defaultString, String defaultKey)
	throws ClassNotFoundException {
		this(name, type, size, defaultString);
		setDefaultFromResource(defaultKey);
	}

	public void setName(String name) {
		if (configured) {
			throw new IllegalStateException("Configuration is frozen");
		}
		this.name = name;
	}
	
	public String getName() {
		return name;
	}

	public void setType(String type) throws ClassNotFoundException {
		if (configured) {
			throw new IllegalStateException("Configuration is frozen");
		}
		this.type = type;
	}
	
	public String getType() {
		return type;
	}

	public Class getTypeClass() throws ClassNotFoundException {
		if (typeClass != null)
			return typeClass;

		String baseType = getType(); 
		
		if (baseType.endsWith("[]")) {
			baseType = baseType.substring(0, baseType.length() - 2);
			indexed = true;
		}
		
		Class baseClass = null;
		if (baseType.equals("boolean"))
			baseClass = Boolean.TYPE;
		else if (baseType.equals("byte"))
			baseClass = Byte.TYPE;
		else if (baseType.equals("char"))
			baseClass = Character.TYPE;
		else if (baseType.equals("double"))
			baseClass = Double.TYPE;
		else if (baseType.equals("float"))
			baseClass = Float.TYPE;
		else if (baseType.equals("int"))
			baseClass = Integer.TYPE;
		else if (baseType.equals("long"))
			baseClass = Long.TYPE;
		else if (baseType.equals("short"))
			baseClass = Short.TYPE;
		else if (baseType.equals("String"))
			baseClass = String.class;
		else {
			ClassLoader classLoader =
				Thread.currentThread().getContextClassLoader();
			if (classLoader == null) {
				classLoader = this.getClass().getClassLoader();
			}
			baseClass = classLoader.loadClass(baseType);
		}
		
		if (indexed) {
			indexedComponentClass = baseClass;
			typeClass = Array.newInstance(indexedComponentClass, 0).getClass();
		} else {
			indexedComponentClass = null;
			typeClass = baseClass;
		}
		return typeClass;
	}

	public Class getIndexedComponentTypeClass() throws ClassNotFoundException {
		getTypeClass();
		return indexedComponentClass;
	}

	public void setDefault(String value) {
		if (configured) {
			throw new IllegalStateException("Configuration is frozen");
		}
		this.defaultValueString = value;
	}
	
	public String getDefault() {
		return defaultValueString;
	}
	
	public void setDefaultFromResource(String key) {
		if (configured) {
			throw new IllegalStateException("Configuration is frozen");
		}
		this.defaultValueKey = key;
	}
	
	public String getDefaultFromResource() {
		return defaultValueKey;
	}

	public Object convertInitialValue(String stringValue)
	throws ClassNotFoundException {
		if (isIndexed()) {
			Object array = Array.newInstance(indexedComponentClass, size);
			if (stringValue != null) {
				Object values = ConvertUtils.convert(stringValue, getTypeClass());
				System.arraycopy(values, 0, array, 0,
					Math.min(Array.getLength(values), Array.getLength(array)));
			}
			return array;
		}
		if (stringValue != null)
			return ConvertUtils.convert(stringValue, getTypeClass());
		Class type = getTypeClass();
		if (!type.isPrimitive())
			return null;
		if (type == Boolean.TYPE)
			return new Boolean(false);
		else if (type == Byte.TYPE)
			return new Byte((byte)0);
		else if (type == Character.TYPE)
			return new Character((char)0);
		else if (type == Double.TYPE)
			return new Double(0D);
		else if (type == Float.TYPE)
			return new Float(0F);
		else if (type == Integer.TYPE)
			return new Integer(0);
		else if (type == Long.TYPE)
			return new Long(0);
		else if (type == Short.TYPE)
			return new Short((short)0);
		return null;
	}

	public Object convert(String value)
	throws ClassNotFoundException {
		if (isIndexed()) {
			getTypeClass();
			return ConvertUtils.convert(value, indexedComponentClass);
		}
		return ConvertUtils.convert(value, getTypeClass());
	}

	public int getSize() {
		if (!indexed)
			return 0;
		return size;
	}

	public void setSize(int size) {
		if (configured) {
			throw new IllegalStateException("Configuration is frozen");
		}
		this.size = size;
	}

	public boolean isIndexed() {
		return indexed;
	}
	
	public void freeze() {
		configured = true;
	}
}
